/******************************************************************************
**  COPYRIGHT 2007 Marvell Inernational Ltd.
**  All Rights Reserved
**
**  COPYRIGHT (C) 2006 Intel Corporation.
**
**  This software as well as the software described in it is furnished under
**  license and may only be used or copied in accordance with the terms of the
**  license. The information in this file is furnished for informational use
**  only, is subject to change without notice, and should not be construed as
**  a commitment by Intel Corporation. Intel Corporation assumes no
**  responsibility or liability for any errors or inaccuracies that may appear
**  in this document or any software that may be provided in association with
**  this document.
**  Except as permitted by such license, no part of this document may be
**  reproduced, stored in a retrieval system, or transmitted in any form or by
**  any means without the express written consent of Intel Corporation.
**
**  FILENAME:	boot.c
**
**  PURPOSE: 	Low level mDoc access routines
**
******************************************************************************/
#include "msys.h"
#include "msys_custom.h"
#include "misc.h"

/*****************************************************************************
*	Global variables
******************************************************************************/
//static FLWord   gHibCoreAddress;
//static FLWord   gHibContRegAreaOffset;
//static FLWord   gHibDataPortAreaOffset;
//static FLWord   gHibConfigRegAreaOffset;
//static FLByte  *DeviceBaseAddress;
P_MDOC_ATA_REG_T    pMdocAtaReg;
P_MDOC_CONFIG_REG_T pMdocCfgReg;
volatile FLWord     *pMdocDataPort;


/*****************************************************************************
*	Help Functions
******************************************************************************/
void mDOC_SetupATARegisters( unsigned int sectorOffset, unsigned char partitionNum );

#if M_SYS_DEBUG_PRINT_ON

#define MSG_BUF_SIZE (4*1024)
unsigned char msgBufStart[MSG_BUF_SIZE];
unsigned char *msgBufPtr;

//-----------------------------------------------------------------------------
// mSysDebug()
//             print debugging info to internal buffer
//
//-----------------------------------------------------------------------------
void mSysDebug( unsigned int zone, char *msg )
{
  static int size = -1;
  int buffLeft;

  if( size == -1 )
  {
    memset(msgBufStart, 0, sizeof(msgBufStart));
    msgBufPtr = msgBufStart;
  }

  size = strlen(msg);
  buffLeft = MSG_BUF_SIZE - (msgBufPtr-msgBufStart);

  size = (buffLeft >= size) ? size : buffLeft;

  if( size )
  {
    memcpy(msgBufPtr, msg, size );
    buffLeft -= size;
    msgBufPtr += size;
  }

}
#endif

/******************************************************************************
 *                                                                            *
 *                             r e a d y		                              *
 *                                                                            *
 *  Wait until particular bit pattern appears in specified DOCH register      *
 *                                                                            *
 *  Parameters :                                                              *
 *      pAtaRegAddress       : ATA register                                   *
 *      mask                 : bits we are interested in                      *
 *      on_bits              : bits we are waiting to become '1'              *
 *      millisec             : timeout value in milliseconds                  *
 *                                                                            *
 *  Returns :                                                                 *
 *      DOCH_OK in success, otherwise respective error code.                  *
 *                                                                            *
 ******************************************************************************/
static DOCH_Error ready ( FLByte  *pAtaRegAddress,
                          FLByte  mask,
                          FLByte  on_bits,
                          FLDword millisec )
{
    FLSDword startTime, endTime, elapsedTime;
	FLByte status;

    startTime = GetOSCR0();

    do
    {
        /*Wait for BUSY bit to clear*/
        status = *pAtaRegAddress;
        if ((status & mask) == on_bits)
        {
            return DOCH_OK;
        }
		/*If an error was detected - return the error code*/
		/***********************************************
		*	We might want to know the error code....
		************************************************/
		else if((status & DOCH_ERROR) == DOCH_ERROR )
		{
			return DOCH_ERROR;
		}

        endTime = GetOSCR0();

        if(endTime < startTime)
            endTime = ((0xFFFFFFFF-startTime) + endTime);

        elapsedTime = OSCR0IntervalInMilli(startTime, endTime);  // measure wait time

    }while( elapsedTime <= millisec );

	DBG_PRINT_ERR( FLZONE_ATA, "\r\nready(): Timed Out\r\n" );

    return DOCH_TimedOut;
}

/******************************************************************************
 *                                                                            *
 *                  h a l _ b l k _ r e a d _ n o r                           *
 *                                                                            *
 *  Read 'sectors' sectors from DOCH										  *
 *                                                                            *
 *  Parameters :                                                              *
 *      dest                 : buffer to read to                              *
 *      sectors              : number of sectors to read					  *
 *                                                                            *
 *  Returns :                                                                 *
 *      always zero (success)                                                 *
 *                                                                            *
 ******************************************************************************/
FLSNative hal_blk_read_nor ( FLByte    *pDest,
							 FLSNative sectors )
{
   register FLWord tmp;
   register FLByte *pEnd;
   register FLWord *pWend, *pWdest;

   if( (FLDword)pDest & 0x1 )
   {
      //------------------------------------------------
      // The data is read from a 16 bit wide FIFO, w/o
      // byte read support
      //------------------------------------------------
      pEnd = pDest + (sectors << DOCH_SECTOR_SIZE_BITS);

      //Monahans is little endian
      //-------------------------
      while( pDest < pEnd )
      {
         tmp = *pMdocDataPort;
         *pDest++ = (FLByte)tmp;
         *pDest++ = (FLByte)(tmp>>8);
      }
   }
   else
   {  //------------------------
      // mainstream aligned case
      //------------------------
      pWdest = (FLWord *)pDest;
      pWend  = pWdest + (sectors << (DOCH_SECTOR_SIZE_BITS-1));
      while (pWdest < pWend )
        *pWdest++ = *pMdocDataPort;
   }

   return 0;
}

/******************************************************************************
 *                                                                            *
 *                             i o _ i n p u t                                *
 *                                                                            *
 *  Read sectors from DOCH device.                                            *
 *                                                                            *
 *  Parameters :                                                              *
 *      DeviceBaseAddress            : Physical address of device             *
 *		Num_Of_Sectors_To_Copy : Number of sectors to read					  *
 *      buf                    : Buffer to read to                            *
 *                                                                            *
 *  Returns :                                                                 *
 *      DOCH_OK in success, otherwise respective error code.                  *
 *                                                                            *
 ******************************************************************************/
static DOCH_Error ioInput( unsigned int  numOfSectorsToCopy,
 				    unsigned char *pBuf                )
{

  DOCH_Error rc = DOCH_OK;
  FLNative secPerformed;
  FLNative offset = 0;

  /* Check status to see if command has passed succefully */
  (rc = ready((FLByte*)&(pMdocAtaReg->COMMAND_STATUS_REG),
                        (DOCH_READY | DOCH_DRQ | DOCH_BUSY | DOCH_ERROR),
                        (DOCH_READY | DOCH_DRQ), DOCH_LONG_WAIT));

	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_input(): ATA not Ready (before data XFER 1) \r\n");
		DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
		return rc;
	}

	/* Start the copy procedure */
    for(secPerformed=1; secPerformed <= numOfSectorsToCopy; secPerformed++)
	{
		/*Perform data xfer*/
        DOCHBLK_READ_MDOC((pBuf + offset), 1 );

		/* Check DRQ ready and not BSY before next data xfer */
		/* (don`t perform after LAST sector was transfered) */
        if(secPerformed < numOfSectorsToCopy)
        {
          (rc = ready((FLByte*)&(pMdocAtaReg->COMMAND_STATUS_REG),
                                (DOCH_READY | DOCH_DRQ |DOCH_BUSY | DOCH_ERROR),
                                (DOCH_READY | DOCH_DRQ), DOCH_LONG_WAIT));
			if(rc != DOCH_OK)
			{
				DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_input(): ATA not Ready (before data XFER 2) \r\n");
				DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
				return rc;
			}
		}

		offset += DOCH_SECTOR_SIZE;
	}

  return DOCH_OK;
}

/*****************************************************************************
*	Secure boot API functions
******************************************************************************/
static Boolean mDOC_Identified( void )
{
  FLWord   chipID1 = 0; // init these to zero in case read fails...
  FLWord   chipID2 = 0;

  chipID1 = pMdocCfgReg->CHIPID1_REG;
  chipID2 = pMdocCfgReg->CHIPID2_REG;

  /*Check that Chip IDs match and identify H3  device*/
  if( (chipID1 + chipID2) == 0xFFFF)
  {
    switch( chipID1 )
    {
        case 0x4833:  /* mDoc H3 */
        case 0x4834:  /* mDoc H4 */
        case 0x4835:  /* mDoc H5 */

             return M_SYS_TRUE;
             break;
        default:
             break;
    }
  }

  return M_SYS_FALSE;
}

/*----------------------------------------------------------------------*/
/*       m D o c _ H 3 _ D e t e c t									*/
/*                                                                      */
/* Detects a device of mDOC H3 in the system and configures some static */
/* variables according to the detected window size.   					*/
/* Those static variables would be used in further accesses to the 		*/
/* device. 																*/
/*                                                                      */
/* Parameters: None                                                     */
/*                                                                      */
/* Returns:                                                             */
/*        Boolean   : M_SYS_TRUE on success, M_SYS_FALSE on failure     */
/*----------------------------------------------------------------------*/
static Boolean mDOC_Detect(void)
{
  // NOTE: pMdocDataPort is a bi-directional FIFO

  // Try to Identify device on 8KB window
  //-------------------------------------
  pMdocCfgReg   = (P_MDOC_CONFIG_REG_T)(MDOC_START_ADDRESS + HIB_CORE_ADDRESS_8KB + DOCH_CONFIG_REG_AREA_8KB_OFFSET);
  pMdocAtaReg   = (P_MDOC_ATA_REG_T)   (MDOC_START_ADDRESS + HIB_CORE_ADDRESS_8KB  + DOCH_CONT_REG_AREA_8KB_OFFSET);
  pMdocDataPort = (volatile FLWord *)(MDOC_START_ADDRESS + HIB_CORE_ADDRESS_8KB  + DOCH_DATA_PORT_AREA_8KB_OFFSET);

  if( mDOC_Identified() )
  {
      DBG_PRINT_FLOW(FLZONE_ATA, "Memory window is 8KB");
      return M_SYS_TRUE;
  }

  // Try to Identify device on 128KB window
  //---------------------------------------
  pMdocCfgReg   = (P_MDOC_CONFIG_REG_T)(MDOC_START_ADDRESS + HIB_CORE_ADDRESS_128KB + DOCH_CONFIG_REG_AREA_128KB_OFFSET);
  pMdocAtaReg   = (P_MDOC_ATA_REG_T)   (MDOC_START_ADDRESS + HIB_CORE_ADDRESS_128KB + DOCH_CONT_REG_AREA_128KB_OFFSET);
  pMdocDataPort = (volatile FLWord *)(MDOC_START_ADDRESS + HIB_CORE_ADDRESS_128KB + DOCH_DATA_PORT_AREA_128KB_OFFSET);

  if( mDOC_Identified() )
  {
     DBG_PRINT_FLOW(FLZONE_ATA, "Memory window is 128KB");
     return M_SYS_TRUE;
  }

  DBG_PRINT_ERR(FLZONE_ATA, "\r\nUnable to detect 8/128 Kb window M-SYS device\r\n");

  return M_SYS_FALSE;
}

/*----------------------------------------------------------------------*/
/*       m D o c _ H 3 _ I n i t i a l i z e							*/
/* 																		*/
/* Prepare the mDOC H3 device for reading.								*/
/* Currently, assumes that code runs after device's HW reset, so all    */
/* device's registers are configured to their proper default values     */
/*                                                                      */
/* Parameters: None                                                     */
/*                                                                      */
/* Returns:                                                             */
/*      always M_SYS_TRUE (success)             						*/
/*----------------------------------------------------------------------*/
Boolean mDOC_Initialize(void)
{
    ChipSelectMsys();  // set-up chip select for M-sys
    return (mDOC_Detect() == M_SYS_TRUE );
}


/*----------------------------------------------------------------------*/
/* mDOC_ReadSectors                                                     */
/*                                                                      */
/* Load data from mDOC H3 device into the system RAM                    */
/*                                                                      */
/* Parameters:                                                          */
/*      sectorOffset          : First sector to read from the device    */
/*      numBytesToCopy        : Number of bytes to read                 */
/*      targetBuffer          : Resides in RAM, and serves as the       */
/*                               destination buffer for the data that   */
/*                               is read from the device.               */
/*                                                                      */
/* Returns:                                                             */
/*      Boolean   : TURE on success, otherwise FALSE                    */
/*----------------------------------------------------------------------*/
Boolean mDOC_ReadSectors( unsigned int sectorOffset,
                           unsigned int numBytesToCopy,
                            unsigned char *pTargetBuffer,
                             unsigned char partitionNum     )
{
  FLSNative  status;
  DOCH_Error rc = DOCH_OK;
  FLSNative  sectorsThisCycle;
  unsigned char shimBuffer[DOCH_SECTOR_SIZE];
  unsigned int numOfSectorsToCopy;

  numOfSectorsToCopy = mDOCDrvComputeSectorCount( numBytesToCopy );
  pMdocCfgReg->OPERATION_MODE_REG = (HIB_ADDR_SHIFT | HIB_NON_PIPE_ACCESS);

  while ( numOfSectorsToCopy > 0 )
  {
    /*Wait for busy bit to clear*/
    status = pMdocAtaReg->COMMAND_STATUS_REG;
    if ((status & DOCH_BUSY) == DOCH_BUSY)
    {
       if ((rc = ready((FLByte*)&(pMdocAtaReg->ALT_CONTROL_STATUS_REG), DOCH_BUSY, 0,
                         DOCH_LONG_WAIT)) != DOCH_OK)
       {
         DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_input(): ATA not Ready (before command) \r\n");
         DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
         return M_SYS_FALSE;
       }
    }

    if (numOfSectorsToCopy > DOCH_MAX_SECTORS)
    {
        /* Read exactly 256 sectors */
        sectorsThisCycle = DOCH_MAX_SECTORS;
        pMdocAtaReg->SECTOR_COUNT_REG = (FLWord)0;
    }
    else
    {
        sectorsThisCycle = numOfSectorsToCopy;
        pMdocAtaReg->SECTOR_COUNT_REG = (FLWord)sectorsThisCycle;
    }

    mDOC_SetupATARegisters( sectorOffset, partitionNum );

    /*Write ATA command*/
    pMdocAtaReg->COMMAND_STATUS_REG = (FLWord)DOCH_VSCMD_READ_PARTITION;
    Delay_us(10);   // this delay seems to ensure reliable results...

    /* Read data from device */
    rc = ioInput(sectorsThisCycle, pTargetBuffer);

    if (rc != DOCH_OK)
    {
       return M_SYS_FALSE;
    }

    /*For next possible cycle*/
    /*-----------------------*/
    sectorOffset       += sectorsThisCycle;
    numOfSectorsToCopy -= sectorsThisCycle;
    pTargetBuffer      += (sectorsThisCycle * DOCH_SECTOR_SIZE);
    numBytesToCopy     -= (sectorsThisCycle * DOCH_SECTOR_SIZE);
  }

  // Copy any last partial sector into shim buffer,
  // and copy only requested bytes to the target buffer
  //---------------------------------------------------
  if( numBytesToCopy )
  {
    /*Wait for busy bit to clear*/
    status = pMdocAtaReg->COMMAND_STATUS_REG;
    if ((status & DOCH_BUSY) == DOCH_BUSY)
    {
       if ((rc = ready((FLByte*)&(pMdocAtaReg->ALT_CONTROL_STATUS_REG), DOCH_BUSY, 0,
                         DOCH_LONG_WAIT)) != DOCH_OK)
       {
         DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_input(): ATA not Ready (before command) \r\n");
         DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
         return M_SYS_FALSE;
       }
    }

    pMdocAtaReg->SECTOR_COUNT_REG = (FLWord)1;
    mDOC_SetupATARegisters( sectorOffset, partitionNum );

    /*Write ATA command*/
    pMdocAtaReg->COMMAND_STATUS_REG = (FLWord)DOCH_VSCMD_READ_PARTITION;
    Delay_us(10);   // this delay seems to ensure reliable results...

    /* Read data from device */
    rc = ioInput(1, shimBuffer);
    if (rc != DOCH_OK)
    {
       return M_SYS_FALSE;
    }

    // Now copy the last bytes to the target buffer
    //---------------------------------------------
    memcpy( pTargetBuffer, shimBuffer, numBytesToCopy );
  }


  return M_SYS_TRUE;
}

//==================================================================================
//  mDOCDrvComputeSectorCount()
//
//  Compute the number of DOC sectors equivalent to the byte count
//
//==================================================================================
UINT_T mDOCDrvComputeSectorCount( UINT_T byteCount )
{
    // DOCH_SECTOR_SIZE must be a power of 2
    //----------------------------------------------------
    return ((byteCount >> DOCH_SECTOR_SIZE_BITS));
}


//===================================================================
//===================================================================
//===================================================================

/******************************************************************************
 *                                                                            *
 *                  h a l _ b l k _ w r i t e _ n o r                         *
 *                                                                            *
 *  write 'sectors' sectors to DOCH                                           *
 *                                                                            *
 *  Parameters :                                                              *
 *      base                 : pointer to the base of DOCH register set       *
 *      pSrc                 : buffer to write from                           *
 *      numSectors           : number of sectors to write                     *
 *                                                                            *
 *  Returns :                                                                 *
 *      always zero (success)                                                 *
 *                                                                            *
 ******************************************************************************/
FLSNative hal_blk_write_nor ( FLByte *pSrc, FLSNative numSectors )
{
   //FLSNative       count = numSectors<<DOCH_SECTOR_SIZE_BITS;
   //register int    i;
   register FLWord tmp;
   register FLByte *pEnd;
   register FLWord *pWend, *pWsrc;

   if( (FLDword)pSrc & 0x1 ) /* rare case: unaligned source buffer */
   {
#if 0
      for (i = 0; i < (int)count; i+=2)
      {
        tmp = (FLWord)pSrc[i] + ((FLWord)pSrc[i+1]<<8);
        *pMdocDataPort = tmp;
      }
#endif

      pEnd = pSrc + (numSectors << DOCH_SECTOR_SIZE_BITS);

      //Monahans is little endian
      //-------------------------
      while( pSrc < pEnd )
      {
          tmp  = (FLWord)*pSrc++;
          tmp |= (FLWord)((*pSrc++) << 8);
          *pMdocDataPort = tmp;
      }
   }
   else /* mainstream case */
   {
       //------------------------
       // mainstream aligned case
       //------------------------
       pWsrc = (FLWord *)pSrc;
       pWend  = pWsrc + (numSectors << (DOCH_SECTOR_SIZE_BITS-1));
       while (pWsrc < pWend )
         *pMdocDataPort = *pWsrc++;
   }

   return 0;
}


/******************************************************************************
 *                                                                            *
 *                             i o _ o u t p u t                              *
 *                                                                            *
 *  Write sectors to DOCH device.                                             *
 *                                                                            *
 *  Parameters :                                                              *
 *      pdev                 : device to act on                               *
 *      regs                 : DOCH_Registers                                 *
 *      pSrc                 : source buffer for write                        *
 *      numOfSectorsToWrite  : # of sectors to write                          *
 *                                                                            *
 *  Returns :                                                                 *
 *      DOCH_OK in success, otherwise respective error code.                  *
 *                                                                            *
 ******************************************************************************/
static DOCH_Error ioOutput ( unsigned int  numOfSectorsToWrite,
                               unsigned char *pSrc              )
{
    DOCH_Error  rc = DOCH_OK;
    FLNative    secPerformed;
    FLNative    offset = 0;
    FLSNative   status;

  // Check status to see if command has passed successfully
  //--------------------------------------------------------
  (rc = ready((FLByte*)&(pMdocAtaReg->COMMAND_STATUS_REG),
                        (DOCH_READY | DOCH_DRQ | DOCH_BUSY | DOCH_ERROR),
                        (DOCH_READY | DOCH_DRQ), DOCH_LONG_WAIT));

  if(rc != DOCH_OK)
  {
     DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_output(): ATA not Ready (before data XFER 1) \r\n");
     DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
     return rc;
  }

  // Do PIO write operation
  //-----------------------
  for(secPerformed=1; secPerformed <= numOfSectorsToWrite; secPerformed++)
  {
      /*Perform data xfer*/
      DOCHBLK_WRITE_MDOC((pSrc + offset), 1);

      /* Check DRQ ready and not BSY before next data xfer */
      /* (don`t perform after LAST sector was transfered) */
      if(secPerformed < numOfSectorsToWrite )
      {
          (rc = ready((FLByte*)&(pMdocAtaReg->COMMAND_STATUS_REG),
                                (DOCH_READY | DOCH_DRQ |DOCH_BUSY | DOCH_ERROR),
                                (DOCH_READY | DOCH_DRQ), DOCH_LONG_WAIT));
          if(rc != DOCH_OK)
          {
              DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_output(): ATA not Ready\r\n");
              DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
              return rc;
          }
      }

      // bump Src pointer by one sector
      //-------------------------------
      offset += DOCH_SECTOR_SIZE;
  }

  return rc;
}

/*----------------------------------------------------------------------*/
/* mDOC_WriteSectors                                                    */
/*                                                                      */
/* Write sectors to  mDOC H3 device from system RAM                     */
/*                                                                      */
/* Parameters:                                                          */
/*      sectorOffset          : First sector to write                   */
/*      numOfSectorsToCopy    : Number of sectors to write              */
/*      pSourceBuffer         : Resides in RAM, and serves as the       */
/*                              source buffer for the data that         */
/*                              is written to the device.               */
/*                                                                      */
/* Returns:                                                             */
/*      Boolean   : TURE on success, otherwise FALSE                    */
/*----------------------------------------------------------------------*/
Boolean mDOC_WriteSectors( unsigned int sectorOffset,
                            unsigned int numBytesToWrite,
                             unsigned char *pSourceBuffer,
                              unsigned char partitionNum )
{
  FLSNative  status;
  DOCH_Error rc = DOCH_OK;
  FLSNative  sectorsThisCycle;
  unsigned char shimBuffer[DOCH_SECTOR_SIZE];
  unsigned int numOfSectorsToWrite;

  numOfSectorsToWrite = mDOCDrvComputeSectorCount( numBytesToWrite );
  pMdocCfgReg->OPERATION_MODE_REG = (HIB_ADDR_SHIFT | HIB_NON_PIPE_ACCESS);

  while( numOfSectorsToWrite > 0 )
  {
    /*Wait for busy bit to clear*/
    status = pMdocAtaReg->COMMAND_STATUS_REG;
    if ((status & DOCH_BUSY) == DOCH_BUSY)
    {
       if ((rc = ready((FLByte*)&(pMdocAtaReg->ALT_CONTROL_STATUS_REG), DOCH_BUSY, 0,
                         DOCH_LONG_WAIT)) != DOCH_OK)
       {
          DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_input(): ATA not Ready (before command) \r\n");
          DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
          return M_SYS_FALSE;
       }
    }

    if (numOfSectorsToWrite > DOCH_MAX_SECTORS)
    {
        /* Write exactly 256 sectors */
        sectorsThisCycle = DOCH_MAX_SECTORS;
        pMdocAtaReg->SECTOR_COUNT_REG = (FLWord)0;
    }
    else
    {
        sectorsThisCycle = numOfSectorsToWrite;
        pMdocAtaReg->SECTOR_COUNT_REG = (FLWord)sectorsThisCycle;
    }

    mDOC_SetupATARegisters( sectorOffset, partitionNum );

    /*Write ATA command*/
    pMdocAtaReg->COMMAND_STATUS_REG = (FLWord)DOCH_VSCMD_WRITE_PARTITION;
    Delay_us(10);   // this delay seems to ensure reliable results...

    /* Write data to device */
    rc = ioOutput(sectorsThisCycle, pSourceBuffer);

    if (rc != DOCH_OK)
    {
       return M_SYS_FALSE;
    }

    /*For next possible cycle*/
    /*-----------------------*/
    sectorOffset        += sectorsThisCycle;
    numOfSectorsToWrite -= sectorsThisCycle;
    pSourceBuffer       += (sectorsThisCycle * DOCH_SECTOR_SIZE);
    numBytesToWrite     -= (sectorsThisCycle * DOCH_SECTOR_SIZE);
  }

  // If there's a partial sector to write at the end...
  //---------------------------------------------------
  if( numBytesToWrite )
  {
    // Copy from source buffer to shim buffer...
    // This prevents any memory fault from the source
    // buffer since the mDOC routine
    // writes a minimum of 512 bytes (a NAND sector)
    //------------------------------------------------
    memset(shimBuffer, 0xFF, sizeof(shimBuffer));
    memcpy(shimBuffer, pSourceBuffer, numBytesToWrite );

    /*Wait for busy bit to clear*/
    status = pMdocAtaReg->COMMAND_STATUS_REG;
    if ((status & DOCH_BUSY) == DOCH_BUSY)
    {
      if ((rc = ready((FLByte*)&(pMdocAtaReg->ALT_CONTROL_STATUS_REG), DOCH_BUSY, 0,
                         DOCH_LONG_WAIT)) != DOCH_OK)
      {
         DBG_PRINT_ERR(FLZONE_ATA, "\r\nio_input(): ATA not Ready (before command) \r\n");
         DBG_PRINT_ERR_PRM(FLZONE_ATA, (FLTXT("rc = 0x%x \r\n"),rc));
         return M_SYS_FALSE;
      }
    }

    pMdocAtaReg->SECTOR_COUNT_REG = (FLWord)1;
    mDOC_SetupATARegisters( sectorOffset, partitionNum );

    /*Write ATA command*/
    pMdocAtaReg->COMMAND_STATUS_REG = (FLWord)DOCH_VSCMD_WRITE_PARTITION;
    Delay_us(10);   // this delay seems to ensure reliable results...

    /* Write data to device */
    rc = ioOutput(1, shimBuffer);

    if (rc != DOCH_OK)
    {
       return M_SYS_FALSE;
    }
  }

#if 0
  // check operation's status
  //-------------------------
  status = pMdocAtaReg->COMMAND_STATUS_REG;
  if((status & (DOCH_BUSY | DOCH_READY | DOCH_ERROR)) != DOCH_READY)
  {
      if((status & DOCH_BUSY) == DOCH_BUSY)
          return DOCH_ATABusyNotCleared;
      else if((status & DOCH_READY) != DOCH_READY)
          return DOCH_ATANotReady;
      else if((status & DOCH_ERROR) == DOCH_ERROR)
          return DOCH_ATAErrorDetected;
  }
#endif

  return M_SYS_TRUE;
}


/*----------------------------------------------------------------------*/
/* mDOC_SetupATARegisters                                               */
/*                                                                      */
/* Common Code Reduction                                                */
/*                                                                      */
/* Parameters:                                                          */
/*      sectorOffset          : First sector to write                   */
/*      partitionNum          : mDOC partition                          */
/*                                                                      */
/*----------------------------------------------------------------------*/
void mDOC_SetupATARegisters( unsigned int sectorOffset, unsigned char partitionNum )
{
  pMdocAtaReg->SECTOR_OFFSET_REG  = (FLWord)sectorOffset;
  pMdocAtaReg->CYLINDER_LOW_REG   = (FLWord)(sectorOffset >>  8);
  pMdocAtaReg->CYLINDER_HIGH_REG  = (FLWord)(sectorOffset >> 16);
  pMdocAtaReg->ERROR_FEATURES_REG = (FLWord)partitionNum;
  pMdocAtaReg->DRIVE_HEAD_REG     = (FLWord)(((sectorOffset >> 24) & 0x0f ) | DOCH_LBA);
}


#if 0
UINT8_T  m_sys_buf[512];
#define DEAD_SECTOR_TRACK_COUNT 512
unsigned int deadSector[DEAD_SECTOR_TRACK_COUNT];
unsigned int deadCount = 0;

//--------------------------------------------------------
//  Find non-zero data in a memory/sector block
//
//--------------------------------------------------------
UINT8_T memfind( unsigned char *a, unsigned long bytes )
{
   unsigned char *limit;

   limit = a + bytes;

   while( a < limit )
   {
      if(*a)
        return 1;

      a++;
   }

   return 0;
}


void mDOC_test( void )
{
  unsigned int found, sector_number;
  unsigned char partition, mSysPassed;

  // TEST M_SYS DEVICE -> hunt for non-zero
  // sectors
  //-----------------------------------------
  for( partition = 0; partition < 6; partition++)
  {
    found = 0;
    deadCount = 0;
    memset( m_sys_buf,0,sizeof(m_sys_buf));

    // find first non-zero sector ...
    // 128MB / (512 B/sector) = 262144 sectors
    //--------------------------------------
    for( sector_number = 0; sector_number < 262144; sector_number++ )
    {
      mSysPassed = mDOC_Copy (sector_number, 1, m_sys_buf, partition );

      if( sector_number == 0 )
        found = sector_number;

      if( (partition == 3) && (sector_number == 398) )
        found = sector_number;

      if( !mSysPassed )
      {
        if( deadCount < DEAD_SECTOR_TRACK_COUNT )
          deadSector[deadCount++] = sector_number;
      }
      else if( memfind(m_sys_buf, sizeof(m_sys_buf)))
      {
         found = sector_number;
      }
    }
  }
}
#endif
