/******************************************************************************
**  COPYRIGHT  2007 Marvell Inernational Ltd.
**  All Rights Reserved
******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  Copyright (C), 1995-2006, msystems Ltd. All rights reserved.              */
/*                                                                            */
/*  Redistribution and use in source and binary forms, with or without        */
/*  modification, are permitted provided that the following conditions are    */
/*  met:                                                                      */
/*  1. Redistributions of source code must retain the above copyright notice, */
/*     this list of conditions and the following disclaimer.                  */
/*  2. Redistributions in binary form must reproduce the above copyright      */
/*     notice, this list of conditions and the following disclaimer in the    */
/*     documentation and/or other materials provided with the distribution.   */
/*  3. Neither the name of msystems nor the names of its contributors may be  */
/*     used to endorse or promote products derived from this software without */
/*     specific prior written permission.                                     */
/*                                                                            */
/*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED */
/*  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR             */
/*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT      */
/*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,     */
/*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED  */
/*  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR    */
/*  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    */
/*  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      */
/*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        */
/*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              */
/*                                                                            */
/******************************************************************************/
/*
 * $Log:   V:/PVCSDB/DiskOnChip/archives/Test for 7.x/src/H3/doch_api.c-arc  $
 *
 *    Rev 1.47   Aug 09 2006 17:26:50   Polina.Marimont
 * initial for DOC Driver 1.0
 *
 */
#include "msys.h"
#include "msys_custom.h"
#include "misc.h"

#include "flsystem.h"
#include "flchkdef.h"
#include "flsystyp.h"
#include "flcommon.h"
#include "flsysfun.h"
#include "doch_func.h"
#include "doch_ata.h"
#include "flstdcmp.h"
//#include "hib.h"
#include "part_inf.h"
#include "dochtl.h"
#ifdef FL_MIGRATION_VERSION
#include "docsys.h"
#endif /* FL_MIGRATION_VERSION */

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************/
/*
 *	Global variables
 */
/******************************************************************************/

/* NOTE: Global variables with "boolean like" behavior are set to a unique pattern (DOCH_GLOBAL_BOOL_PATTERN)
		 to indicate them as "TRUE" */

FLDword gDochVerifyWrite[DOCH_MAX_SOCKETS];	/*Indicates to perform verify operation on every written sector to the specified socket*/
FLDword gDochAtaDebug	 = 0;		/*Indicates to retrieve ATA debug buffer upon completion of ATA command */

FLDword gSdkInitDone = 0;			/*Indicates that DochSDKInit() was completed */

FLDword gSdkDOCAddressObtained = 0;	/*Indicates that DOC address was obtained (used by DOCHConfigHW) */

FLDword gConfigHWDefaults[DOCH_NUM_OF_DCONFIGHW_ITEMS];
FLDword gConfigHWInitDone = 0;		/*Indicates that gConfigHWDefaults[] was set to defaults */

FLDword gIPLChunkOffsetInSectors = 0;	/*Offset of current IPL chunk to write*/

FLDword gSDKUpdatesDiskAttributes = 0;	/*Indication that first 0x40 bytes of disk user attributes should be written as well*/

FLDword gIsDMAEnabled = 0;				/*DMA is Enabled/Disabled*/

FLDword gAutoDPDByHost = 0;				/*Host enters device to DPD automatically after every command*/

FLBoolean gDochIrqEnabled[DOCH_MAX_SOCKETS]; /* holds TRUE, when interrupt enable for socket, FALSE otherwise */

extern FLDword gAccessLayerType;

extern FLDword gATANoTimeout;

#ifdef DOCH_DMA_CONFIG
extern FLDword gDMAChannelOpen;
#endif /*DOCH_DMA_CONFIG*/

extern DOCH_DpdSettings gDpdSettings;

extern DOCH_Socket sockets [DOCH_MAX_SOCKETS];

#ifdef CHECK_POWER_ON_EVERY_COMMAND
extern FLBoolean gDeviceTurnedOff;
#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

extern FLDword gUseShortWaitOnBusy;

/******************************************************************************/
/*
 *	Extern routines
 */
/******************************************************************************/

extern DOCH_Error io_input(DOCH_Socket *pdev, FLSNative  devNum, DOCH_Registers* regs, void *buf, FLNative secNum);
extern DOCH_Error io_output(DOCH_Socket *pdev, FLSNative  devNum, DOCH_Registers* regs, void *buf, FLNative secNum);
extern DOCH_Error get_out_registers(DOCH_Socket* pdev, FLSNative  devNum, DOCH_Registers* out_regs);

void DOCH_SetBits(FLDword* var, FLDword mask, FLByte offset, FLDword val)
{
	FLDword temp = *var;


	temp &= ~(mask);
	temp |= (val << offset);

	*var = temp;
}

extern DOCH_Error dochSetMutex(FLByte socketNum,
							   FLBoolean state,
							   FLByte  partition);

/******************************************************************************/
/*
 *	Static routines
 */
/******************************************************************************/
static DOCH_Error performSectorsOperation(DOCH_Command cmd,
										  FLByte devHeadOptions,
										  DOCH_Socket* pdev,
									      IOreq* ioreq,
										  DOCH_Registers* out_regs,
										  FLSNative		sectorsToPerfrom,
									      FLSNative *	sectorNumber,
									      FLSNative *	totalNumOfSectors,
									      FLSNative * sectorsPerformed,
									      FLSNative * buf_offset,
									      FLByte* buf_ptr);


#ifdef DOCH_BIG_ENDIAN
static FLWord  be_FLWord   (FLByte * v);
static FLDword be_FLDword  (FLByte * v);
#endif /*DOCH_BIG_ENDIAN*/

DOCH_Error DOCHGetParitionUserAttributesSingleFloor(IOreq* ioreq);

/******************************************************************************/
/*
 *	Static Buffers
 */
/******************************************************************************/
static DOCH_PartitionFormatInfo		partFormatInfo;
static DOCH_DeviceInfo				diskOnChipDeviceInfo;
static DOCH_DeviceInfo				diskOnChipDeviceInfo2;
static DOCH_DeviceUserAttr			devUserAttr;
static DOCH_PartitionInfo			partitionInfoTemp;
static DOCH_PartitionFormatInfoAPI	partInfoTemp;
static DOCH_PartitionProtectionAPI	protAPI;
static FLByte						currentAttributes[DOCH_SECTOR_SIZE];
static FLByte						verifyBuf[DOCH_SECTOR_SIZE];
static FLDword wipeSectorsDev0[((2*DOCH_SECTOR_SIZE) / sizeof(FLDword))];
static FLDword wipeSectorsDev1[((2*DOCH_SECTOR_SIZE) / sizeof(FLDword))];


/******************************************************************************/
/*
 *	Local MACROs
 */
/******************************************************************************/
#ifdef DOCH_DMA_CONFIG
#define DOCH_OPEN_DMA_CHANNEL											\
	{																	\
		DMA_Params_S dmaParams;											\
																		\
		if(pdev->bUseDMA && (gDMAChannelOpen == 0))						\
		{																\
			dmaParams.bOpType = DOCH_DMA_OPEN_CHANNEL;					\
			DOCH_DMA_CONFIG(&dmaParams);								\
			if(dmaParams.fDmaStatus != 0)								\
			{															\
				DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): DOCH_DMA_OPEN_CHANNEL Failed\r\n");	\
				return DOCH_GeneralFailure;								\
			}															\
																		\
			gDMAChannelOpen = DOCH_GLOBAL_BOOL_PATTERN;					\
																		\
			DBG_PRINT_FLOW(FLZONE_ATA, "\r\nDMA Channel opened\r\n");	\
		}																\
	}
#endif /*DOCH_DMA_CONFIG*/

#define DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum)	\
		addr_vals.bSecNum  = (FLByte)  sectorNumber;							\
		addr_vals.bCylLow  = (FLByte) (sectorNumber >>  8);						\
		addr_vals.bCylHi   = (FLByte) (sectorNumber >> 16);						\
		addr_vals.bDevHead = (FLByte) (((sectorNumber >> 24) & 0x0f) | DOCH_LBA | (deviceNum<<4));


#define DOCH_PARTITION_ON_DEV0 \
	((!pdev->sSpanData.secondFloorActive) || (partNum < pdev->sSpanData.bLastPartitionOnDev0) ||			\
		((partNum == pdev->sSpanData.bLastPartitionOnDev0) && (!pdev->sSpanData.bLastDev0PartSpanned)))

#define DOCH_PARTITION_ON_DEV1 \
	(((pdev->sSpanData.secondFloorActive) && (partNum > pdev->sSpanData.bLastPartitionOnDev0))				\
		&& (pdev->wNumOfDevices > 1))

#define DOCH_PARTITION_IS_SPANNED \
	(((pdev->sSpanData.bLastDev0PartSpanned) && (partNum == pdev->sSpanData.numOfSpannedPartitionOnDev0))	\
		&& (pdev->wNumOfDevices > 1))

/*----------------------------------------------------------------------*/
/*			p e r f o r m S e c t o r O p e r a t i o n					*/
/*                                                                      */
/* Performs sector(s) operation					                        */
/*                                                                      */
/* Parameters:                                                          */
/*      cmd					: ATA command to perform					*/
/*		pdev				: Socket to perform on						*/
/*		ioreq				: IOreq										*/
/*		totalNumOfSectors	: Total num of sectors to perform			*/
/*		sectorsPerformed	: Num of sectors actually performed			*/
/*                                                                      */
/* Returns:                                                             */
/*      device head with proper bit					                	*/
/*----------------------------------------------------------------------*/
static
DOCH_Error performSectorsOperation(DOCH_Command cmd,
								   FLByte devHeadOptions,
								   DOCH_Socket* pdev,
								   IOreq* ioreq,
								   DOCH_Registers* out_regs,
								   FLSNative   sectorsToPerfrom,
								   FLSNative * sectorNumber,
								   FLSNative * totalNumOfSectors,
								   FLSNative * sectorsPerformed,
								   FLSNative * buf_offset,
								   FLByte* buf_ptr)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	Addressing_Values_s addr_vals;

	/*Perform 1 operation cycle*/
	/*=========================*/

	/*Set ATA addressing registers*/
	/*---------------------------*/
	DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, (*sectorNumber), (pdev->bAtaDevNum));

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	if(sectorsToPerfrom < DOCH_MAX_SECTORS)
		in_regs.bSectorCount = sectorsToPerfrom;
	else
		in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

	in_regs.bSectorNumber = addr_vals.bSecNum;
	in_regs.bCylLow = addr_vals.bCylLow;
	in_regs.bCylHigh = addr_vals.bCylHi;
	in_regs.bDriveHead = addr_vals.bDevHead | devHeadOptions;
	in_regs.bCommandStatus = cmd;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						out_regs,
						(void*)(&buf_ptr[*buf_offset]),
						sectorsToPerfrom);

	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "performSectorsOperation(): ATA Error: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x "), rc));
		DBG_PRINT_ERR(FLZONE_API, "\r\n");

		return rc;
	}

	/*Update feedback values*/
	/*----------------------*/
	*sectorsPerformed += out_regs->bSectorCount;

	/*For next possible cycle*/
	/*-----------------------*/
	*sectorNumber += sectorsToPerfrom;
	*totalNumOfSectors -= sectorsToPerfrom;
	*buf_offset += sectorsToPerfrom * DOCH_SECTOR_SIZE;

	return DOCH_OK;
}

#ifdef DOCH_BIG_ENDIAN
/******************************************************************************
 *                                                                            *
 *                            be _ s h o r t                                  *
 *                            be _ l o n g                                    *
 *                                                                            *
 *  Little-to-CPU-native integer conversion routines for big-endian CPUs.     *
 *                                                                            *
 *  Parameters :                                                              *
 *      v                    : correctly aligned little endian value          *
 *  Returns :                                                                 *
 *      value in CPU native format                                            *
 *                                                                            *
 ******************************************************************************/

static
FLWord be_FLWord  (FLByte * v)
{
    register FLWord x = *((FLWord *)v);
    return ((x >> 8) | (x << 8));
}

static
FLDword be_FLDword (FLByte * v)
{
    register FLDword x = *((FLDword *)v);
    return ((x >> 24) | ((x >> 8) & 0x0000ff00) |
                        ((x << 8) & 0x00ff0000) | (x << 24));
}
#endif /*DOCH_BIG_ENDIAN*/

/******************************************************************************/
/*
 *	DOCH SDK API Routines
 */
/******************************************************************************/

/*----------------------------------------------------------------------*/
/*						b d C a l l D O C H								*/
/*                                                                      */
/* Performs needed general operations before calling actual function	*/
/* Calls appropriate DOCH function										*/
/*                                                                      */
/* Parameters:                                                          */
/*        functionNo	: DOCH SDK function to run	                    */
/*        ioreq			: DOCH ioreq2 structure	                        */
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error bdCallDOCH(FLSNative functionNo, IOreq FAR2 *ioreq)
{
	DOCH_Error rc;

#ifdef CHECK_POWER_ON_EVERY_COMMAND
	/* gDeviceTurnedOff will be updated by API routines*/
	gDeviceTurnedOff = FALSE;
#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

   /* Take mutex for current device */
   rc = dochSetMutex(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
					 DOCH_ON,
					 DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq));
   if(rc != DOCH_OK)
   {
		DBG_PRINT_ERR(FLZONE_API, "bdCallDOCH(): dochSetMutex failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
		return rc;
   }

   /*Check if requested operation is wither Read or Write which are performance critical
     (All other operations are directed from within the switch case)
     -------------------------------------------------------------*/
	if(functionNo == SDK_READ_PARTITION_SECTORS)
	{
		rc = DOCHReadPartitionSectors(ioreq);
	}
	else if (functionNo == SDK_WRITE_PARTITION_SECTORS)
	{
		rc = DOCHWritePartitionSectors(ioreq);
	}
	else
	{
	  /*Branch to appropriate DOCH routine according to function#*/
	  /*---------------------------------------------------------*/
	  switch (functionNo)
	  {
			/*Get Info*/
		  case SDK_IDENTIFY_DISKONCHIP_DEVICE:
			  rc = DOCHIdentifyDiskOnChipDevice(ioreq);
			  break;

		  case SDK_GET_RESET_STATUS:
			  rc = DOCHGetResetStatus(ioreq);
			  break;

		  case SDK_NOTIFY_RESET:
			  rc = DOCHNotifyReset(ioreq);
			  break;

		  case SDK_NOTIFY_PLATFORM_RESUMED:
			  rc = DOCHNotifyPlatformResumed(ioreq);
			  break;

		  case SDK_GET_PARTITION_INFO:
			  rc = DOCHPartitionInfo(ioreq);
			  break;

		  case SDK_SET_DISK_USER_ATTR:
			  rc = DOCHSetDiskUserAttributes(ioreq);
			  break;

		  case SDK_GET_DISK_USER_ATTR:
			  rc = DOCHGetDiskUserAttributes(ioreq);
			  break;

		  case SDK_GET_CONFIG_DATA:
			  rc = DOCHGetConfigurationData(ioreq);
			  break;

		  case SDK_SET_CONFIG_DATA:
			  rc = DOCHSetConfigurationData(ioreq);
			  break;

			/*Media*/
		  case SDK_SET_DEFAULT_PARTITION:
			  rc = DOCHSetDefaultPartition(ioreq);
			  break;

		  case SDK_SET_DATA_XFER_MODE:
			  rc = DOCHSetDataTransferMode(ioreq);
			  break;

		  case SDK_OPTIMIZE_MEDIA:
			  rc = DOCHOptimizeMedia(ioreq);
			  break;

		  case SDK_DELETE_PARTITIONS:
			  rc = DOCHDeletePartitions(ioreq);
			  break;

		  case SDK_ADD_PARTITION:
			  rc = DOCHAddPartition(ioreq);
			  break;

		  case SDK_UNFORMAT_DEVICE:
			  rc = DOCHUnformatDevice(ioreq);
			  break;

		  case SDK_WRITE_IPL:
			  rc = DOCHWriteIPL(ioreq);
			  break;

		  case SDK_READ_IPL:
			  rc = DOCHReadIPL(ioreq);
			  break;

			/*IO*/
		  case SDK_WRITE_AND_LOCK:
			  rc = DOCHWriteAndLock(ioreq);
			  break;

		  case SDK_WIPE_SECTORS:
			  rc = DOCHWipeSectors(ioreq);
			  break;

		  case SDK_FREE_SECTORS:
			  rc = DOCHFreeSectors(ioreq);
			  break;

		  case SDK_PREPARE_FOR_WRITE:
			  rc = DOCHPrepareForWrite(ioreq);
			  break;

#if 0
			/*Flexi-Flash*/
		  case SDK_WRITE_FLEXI_FAST:
			  rc = DOCHWriteFlexiFast(ioreq);
			  break;

		  case SDK_WRITE_FLEXI_NORMAL:
			  rc = DOCHWriteFlexiNormal(ioreq);
			  break;

		  case SDK_REWRITE_FLEXI_NORMAL:
			  rc = DOCHReWriteFlexiNormal(ioreq);
			  break;
#endif /*0*/

			/*Protection*/
		  case SDK_ACCESS_PART_WITH_PWD:
			  rc = DOCHAccessPartWithPwd(ioreq);
			  break;

		  case SDK_DISABLE_PART_ACCESS:
			  rc = DOCHDisablePartAccess(ioreq);
			  break;

		  case SDK_SET_PART_PROTECTION:
			  rc = DOCHSetParitionProtection(ioreq);
			  break;

#if 0
			/*PKI*/
		  case SDK_PKI_HOST:
			  rc = DOCHSendHostPublicKey(ioreq);
			  break;

		  case SDK_PKI_DOCH:
			  rc = DOCHReceiveDochPublicKey(ioreq);
			  break;

		  case SDK_PKI_VERIFY_HOST_KEY:
			  rc = DOCHVerifyHostKey(ioreq);
			  break;
#endif /*0*/

			/*Custom Parameters*/
		  case SDK_GET_CUSTOM_PARAM:
			  rc = DOCHGetCustomParameter(ioreq);
			  break;

		  case SDK_SET_CUSTOM_PARAM:
			  rc = DOCHSetCustomParameter(ioreq);
			  break;

#if 0
			/*Atomic Read/Write Sequence*/
		  case SDK_ATOMIC_WRITE_SEQ:
			  rc = DOCHManageAtomicWriteSeq(ioreq);
			  break;

			/*Algorithms*/
		  case SDK_REPORT_SUPP_ALG:
			  rc = DOCHReportSupportedAlgorithms(ioreq);
			  break;

		  case SDK_REPORT_ALG_CAPAB:
			  rc = DOCHGetAlgorithmCapabilities(ioreq);
			  break;

		  case SDK_SET_ALG_MODE:
			  rc = DOCHSetAlgorithmMode(ioreq);
			  break;

			/*Hash*/
		  case SDK_AUTO_HASH_CTRL:
			  rc = DOCHAutoHashControl(ioreq);
			  break;

		  case SDK_READ_CALC_HASH:
			  rc = DOCHReadCalculatedHash(ioreq);
			  break;

		  case SDK_WRITE_CALC_HASH:
			  rc = DOCHWriteCalculatedHash(ioreq);
			  break;

		  case SDK_READ_ORIG_HASH:
			  rc = DOCHReadOriginalHash(ioreq);
			  break;

		  case SDK_WRITE_GIVEN_HASH:
			  rc = DOCHWriteGivenHash(ioreq);
			  break;

		  case SDK_START_HASH_STREAM_CALC:
			  rc = DOCHStartHashStreamCalculation(ioreq);
			  break;

		  case SDK_READ_HASH_STREAM_CALC:
			  rc = DOCHReadStopHashStreamCalc(ioreq);
			  break;

		  case SDK_RETURN_RAND_NUMS:
			  rc = DOCHReturnRandomNumbers(ioreq);
			  break;

		  case SDK_SET_HASH_KEY:
			  rc = DOCHSetHashKey(ioreq);
			  break;

#endif /*0*/

			/*General*/
		  case SDK_SET_PART_USER_ATTR:
			  rc = DOCHSetParitionUserAttributes(ioreq);
			  break;

		  case SDK_GET_PART_USER_ATTR:
			  rc = DOCHGetParitionUserAttributes(ioreq);
			  break;

		  case SDK_SET_POWER_MODE:
			  rc = DOCHSetPowerMode(ioreq);
			  break;

		  case SDK_GET_POWER_MODE:
			  rc = DOCHGetPowerMode(ioreq);
			  break;

		  case SDK_HW_CONFIG:
			  rc = DOCHConfigHW(ioreq);
			  break;

		  case SDK_RECOVER_FROM_POWER_LOSS:
			  rc = DOCHRecoverFromPowerLoss(ioreq);
			  break;

		  case SDK_CLEAR_ATA_INTERRUPT:
			  rc = DOCHClearIRQ(ioreq);
			  break;

		  case SDK_GET_PHYSICAL_ADDRESS:
			  rc = DOCHGetPhysicalAddress(ioreq);
			  break;

		  default:
			  rc = DOCH_FeatureNotSupported;
			  break;
		}
	}

#ifdef DOCH_AUTO_DPD_BY_HOST
	if(gAutoDPDByHost == DOCH_GLOBAL_BOOL_PATTERN)
	{
		FLDword tries;
		DOCH_Socket*  pdev;
		FLDword i;
		FLByte status;
		DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

		/*Protect against access time set to 0*/
		if(gDochAccessNanosec == 0)
			return DOCH_BadParameter;
		tries = (DOCH_SHORT_WAIT * (1000000UL / gDochAccessNanosec));

		/*Check if socket is registered*/
		if(pdev != NULL)
		{
			/*Enter dev0 to DPD mode*/
			/*----------------------*/
			/*Set device*/
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG, 0);

			/*Set device to inactive mode*/
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_SECTOR_CNT_REG,	(DOCH_PM_SET_NONE | DOCH_PM_INACTIVE_MODE));

			/*Invoke command*/
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_FEATURES_REG,	DOCH_SET_POWER_MODE);
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_COMMAND_REG,	DOCH_VSCMD_EXT_DEVICE_CTRL);
            Delay_us(10);   // this delay seems to ensure reliable results...

			/*Wait for Dev0 to become ready*/
			for(i=0; i<tries; i++)
			{
				status = DOCHREAD_ATA_REG(pdev->bRegBase, DOCH_STATUS_REG);
				if((status & (DOCH_READY | DOCH_BUSY)) == DOCH_READY)
					goto doch_Dev0_DPD_OK;
			}

doch_Dev0_DPD_OK:

			if(pdev->wNumOfDevices > 1)
			{
				/*Enter dev1 to DPD mode*/
				/*----------------------*/
				/*Set device*/
				DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG, DOCH_DEVICE);

				/*Set device to inactive mode*/
				DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_SECTOR_CNT_REG,	(DOCH_PM_SET_NONE | DOCH_PM_INACTIVE_MODE));

				/*Invoke command*/
				DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_FEATURES_REG,	DOCH_SET_POWER_MODE);
				DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_COMMAND_REG,	DOCH_VSCMD_EXT_DEVICE_CTRL);
                Delay_us(10);   // this delay seems to ensure reliable results...

				/*Wait for Dev1 to become ready*/
				for(i=0; i<tries; i++)
				{
					status = DOCHREAD_ATA_REG(pdev->bRegBase, DOCH_STATUS_REG);
					if((status & (DOCH_READY | DOCH_BUSY)) == DOCH_READY)
						goto doch_Dev1_DPD_OK;
				}

doch_Dev1_DPD_OK:
				/*Do Nothing*/;
			}
		}
	}
#endif /*DOCH_AUTO_DPD_BY_HOST*/

  /* Free mutex for current device */
   dochSetMutex(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
			    DOCH_OFF,
			    DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq));


#ifdef CHECK_POWER_ON_EVERY_COMMAND
		/*In case Power Fail was detected, return DOCH_DeviceTurnedOff*/
		if(gDeviceTurnedOff)
		{
			return DOCH_DeviceTurnedOff;
		}
#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

	return rc;
}

/*----------------------------------------------------------------------*/
/*         D O C H S e t D e f a u l t P a r t i t i o n				*/
/*                                                                      */
/* Set default partition for Standard-ATA commands                      */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*                        Partition number (zero based)		            */
/*      irCount			: SET_DEFAULT_PARTITION_TEMP                    */
/*						  SET_DEFAULT_PARTITION_PERM                    */
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_SET_DEFAULT_PARTITION									*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetDefaultPartition(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	FLByte dev_head_opt;
	DOCH_Socket*  pdev;
	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	if(ioreq->irCount == SET_DEFAULT_PARTITION_PERM)
		dev_head_opt = PERM_DEFAULT_PART_OP;
	else
		dev_head_opt = 0;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_DEFAULT_PARTITION;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = ((pdev->bAtaDevNum * DOCH_DEVICE) | (dev_head_opt));
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetDefaultPartition(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*         D O C H S e t D a t a T r a n s f e r M o d e				*/
/*                                                                      */
/* Set Data transfer mode						                        */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*		irCount			: Requested data transfer mode:					*/
/*								DOCH_DATA_XFER_MODE_SINGLE				*/
/*								DOCH_DATA_XFER_MODE_MULT				*/
/*		irLength		: In case of DOCH_DATA_XFER_MODE_MULT,			*/
/*						  specifies DRQ	block size						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_SET_DATA_XFER_MODE										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetDataTransferModeSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_TransferMode requstedXferMode = (DOCH_TransferMode)(ioreq->irCount);
	FLByte requestedDrqSize = (FLByte)(ioreq->irLength);
	FLSNative  socketNum = DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq);
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, socketNum);

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_DATA_XFER_MODE;
	in_regs.bSectorCount   = (FLByte)(ioreq->irCount);
	if(ioreq->irCount == DOCH_DATA_XFER_MODE_MULT)
		in_regs.bSectorNumber = (FLByte)(ioreq->irLength);
	else
		in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( socketNum,
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
	{
		/*If operation succeeded - update transfer mode in SDK as well*/
		/*------------------------------------------------------------*/
		rc = doch_setTransferMode(socketNum, pdev->bAtaDevNum, requstedXferMode, requestedDrqSize);

		return rc;
	}
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetDataTransferMode(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
		return DOCH_GeneralFailure;
	}

}

DOCH_Error DOCHSetDataTransferMode(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket* pdev;
	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	pdev->bAtaDevNum = 0;
	rc = DOCHSetDataTransferModeSingleFloor(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHSetDataTransferModeSingleFloor(ioreq);
		pdev->bAtaDevNum = 0;
	}

	return rc;
}

/*----------------------------------------------------------------------*/
/*       D O C H I d e n t i f y D i s k O n C h i p D e v i c e		*/
/*                                                                      */
/* Returns general information about the Device							*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*      irCount         : DOCH_IDENTIFY_FROM_FLASH						*/
/*						  DOCH_IDENTIFY_EXISTANCE						*/
/*      irData          : Address of DOCH_DeviceInfo struct				*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_IDENTIFY_DISKONCHIP_DEVICE								*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error   : 0 on success, otherwise failed                 */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHIdentifyDiskOnChipDeviceSingleFloor(IOreq* ioreq)
{
	DOCH_Error       rc;
	DOCH_Socket      *pdev;
	DOCH_DeviceInfo  *localDiskOnChipDeviceInfo = (DOCH_DeviceInfo*)(ioreq->irData);
	FLSNative        devNum;
	DOCH_Registers   in_regs;
	DOCH_Registers   out_regs;
    static FLSNative failcount = 0;
	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	devNum = pdev->bAtaDevNum;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_IDENTIFY_DISKONCHIP_DEVICE;
	in_regs.bSectorCount   = 0;
	in_regs.bSectorNumber  = 0;
	in_regs.bCylLow        = 0;
	in_regs.bCylHigh       = 0;
	in_regs.bDriveHead     = ((pdev->bAtaDevNum * DOCH_DEVICE) | ((FLByte)ioreq->irCount));
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						devNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						sizeof(DOCH_DeviceInfo) / DOCH_SECTOR_SIZE);


#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	localDiskOnChipDeviceInfo->wVersion = be_FLWord((FLByte*)&localDiskOnChipDeviceInfo->wVersion);

	localDiskOnChipDeviceInfo->dwCommandFlagsOrStatuses	= be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwCommandFlagsOrStatuses);
	localDiskOnChipDeviceInfo->dwDiskAttributes1		= be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwDiskAttributes1);
	localDiskOnChipDeviceInfo->dwRservedGateKeeperAttr	= be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwRservedGateKeeperAttr);

	localDiskOnChipDeviceInfo->wTotalNumOfPartitions	= be_FLWord((FLByte*)&localDiskOnChipDeviceInfo->wTotalNumOfPartitions);
	localDiskOnChipDeviceInfo->wDefaultPartitionNumber	= be_FLWord((FLByte*)&localDiskOnChipDeviceInfo->wDefaultPartitionNumber);

	localDiskOnChipDeviceInfo->dwUnformattedCapacity	= be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwUnformattedCapacity);

	localDiskOnChipDeviceInfo->dwReservedConfigPartitionSize = be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwReservedConfigPartitionSize);
	localDiskOnChipDeviceInfo->dwUnitSize				= be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwUnitSize);
	localDiskOnChipDeviceInfo->dwConfigurationPartitionExistsSign = be_FLDword((FLByte*)&localDiskOnChipDeviceInfo->dwConfigurationPartitionExistsSign);
#endif /*DOCH_BIG_ENDIAN*/

    /*Update device structure*/
    /*=======================*/
    if(rc == DOCH_OK)
    {
		if((localDiskOnChipDeviceInfo->dwUnformattedCapacity != 0) &&
            (localDiskOnChipDeviceInfo->wVersionCompatability == 0)

#ifndef DOCH_DONT_CHECK_EXISTANCE_SIGN
			&& (localDiskOnChipDeviceInfo->dwConfigurationPartitionExistsSign == DOCH_CONFIGURATION_EXISTS_SIGN)
#endif /*DOCH_DONT_CHECK_EXISTANCE_SIGN*/

			)
        {
            rc = update_device_info(pdev, localDiskOnChipDeviceInfo, devNum);

			#ifdef CHECK_POWER_ON_EVERY_COMMAND
			{
				IOreq ioreq2;

				/*Make sure next calls to DOCHGetResetStatus() will reflect actual reset*/
				tffsset(&ioreq2, 0, sizeof(ioreq2));
				ioreq2.irHandle = ioreq->irHandle;
				DOCHGetResetStatus(&ioreq2);
			}
			#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

            return DOCH_OK;
        }
        DBG_PRINT_ERR(FLZONE_API, "ERROR: DOCHIdentifyDiskOnChipDevice(): read incorrect data\r\n");
    }
    if(devNum == 0)
    {
        DBG_PRINT_ERR(FLZONE_API, "ERROR: DOCHIdentifyDiskOnChipDevice(): failed with status: ");
        DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
    }
    else
    {
		/*Prevent invoking Dev1 if it was not detected!!!*/
		tffsset(&pdev->sSpanData, 0, sizeof(pdev->sSpanData));

        /*DBG_PRINT_ERR(FLZONE_API, "DOCHIdentifyDiskOnChipDevice(): Dev1 not found\r\n");*/
    }

    return DOCH_GeneralFailure;
}

DOCH_Error DOCHIdentifyDiskOnChipDevice(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	DOCH_DeviceInfo* tempDiskOnChipDeviceInfo;
	FLByte requestedDevice;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	requestedDevice = pdev->bAtaDevNum;

	/*Identify Dev0*/
	tempDiskOnChipDeviceInfo = (DOCH_DeviceInfo*)(ioreq->irData);
	pdev->bAtaDevNum = 0;

	/*Identify command should have a short timeout*/
	gUseShortWaitOnBusy = DOCH_LONG_IDENTIFY_TIMEOUT;
	rc = DOCHIdentifyDiskOnChipDeviceSingleFloor(ioreq);
	gUseShortWaitOnBusy = 0;

	pdev->bAtaDevNum = requestedDevice;
	if(rc != DOCH_OK)
		return rc;

	/*If only 1 device is present, return*/
	if(pdev->wNumOfDevices > 1)
	{
		/*Identify Dev1*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irData = &diskOnChipDeviceInfo2;

		pdev->bAtaDevNum = 1;
		/*Identify command should have a short timeout*/
		gUseShortWaitOnBusy = DOCH_LONG_IDENTIFY_TIMEOUT;
		rc = DOCHIdentifyDiskOnChipDeviceSingleFloor(&ioreq2);
		gUseShortWaitOnBusy = 0;
		pdev->bAtaDevNum = requestedDevice;
		if(rc != DOCH_OK)
			return rc;

		/*Combine data*/
		tempDiskOnChipDeviceInfo->dwUnformattedCapacity += diskOnChipDeviceInfo2.dwUnformattedCapacity;

		tempDiskOnChipDeviceInfo->wTotalNumOfPartitions += diskOnChipDeviceInfo2.wTotalNumOfPartitions;

		if(diskOnChipDeviceInfo2.wTotalNumOfPartitions > 1)
			tempDiskOnChipDeviceInfo->wTotalNumOfPartitions -= 2;
		else if(diskOnChipDeviceInfo2.wTotalNumOfPartitions == 1)
			tempDiskOnChipDeviceInfo->wTotalNumOfPartitions -= 1;
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*            D O C H G e t R e s e t S t a t u s						*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_GET_RESET_STATUS										*/
/*																		*/
/* Returns:                                                             */
/*        irCount	      : 0	=	No reset since last query			*/
/*							1	=	The device has been reset since last*/
/*									query								*/
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetResetStatus(IOreq* ioreq)
{
	DOCH_Error rc;
	FLSNative  devNum;

	DOCH_Registers in_regs;
	DOCH_Registers out_regs;

	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						   FALSE,
						   0);

	devNum = pdev->bAtaDevNum;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_RESET_STATUS;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						devNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Update device structure*/
	/*=======================*/
	if(rc == DOCH_OK)
	{
		/*Return reset status*/
		ioreq->irCount = out_regs.bSectorCount;

		return DOCH_OK;
	}
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetResetStatus(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*            D O C H N o t i f y R e s e t								*/
/*																		*/
/*	Notify the Device on an upcoming reset initiated by the Host.		*/
/*	Host should NOT perform actual reset before this command completes	*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_NOTIFY_RESET											*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error        : 0 on success, otherwise failed            */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHNotifyResetSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;

	DOCH_Registers in_regs;
	DOCH_Registers out_regs;

	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_NOTIFY_RESET;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						0,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Update device structure*/
	/*=======================*/
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHNotifyReset(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}

	return DOCH_OK;
}

DOCH_Error DOCHNotifyReset(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket* pdev;
	FLByte origDevNum;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	origDevNum = pdev->bAtaDevNum;

	pdev->bAtaDevNum = 0;
	rc = DOCHNotifyResetSingleFloor(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHNotifyResetSingleFloor(ioreq);
		pdev->bAtaDevNum = origDevNum;
	}

	return rc;
}

/*----------------------------------------------------------------------*/
/*          D O C H N o t i f y P l a t f o r m R e s u m e d			*/
/*																		*/
/*	Notify the Device that there was a reset,							*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_NOTIFY_PLATFORM_RESUMED								*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error        : 0 on success, otherwise failed            */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHNotifyPlatformResumedSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;

	DOCH_Registers in_regs;
	DOCH_Registers out_regs;

	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_NOTIFY_PLATFORM_RESUMED;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						0,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Update device structure*/
	/*=======================*/
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHNotifyPlatformResumed(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}

    /*Restore DPD settings */
    /*=====================*/
    tffsset(&ioreq2, 0, sizeof(ioreq2));

    ioreq2.irFlags  = (DOCH_PM_SET_BOTH_MODES | DOCH_PM_WORK_MODE);
    ioreq2.irCount  = (gDpdSettings.activeMode | gDpdSettings.inActiveMode);
	ioreq2.irLength = gDpdSettings.timeOut;

	rc = DOCHSetPowerMode(&ioreq2);
	if(rc != DOCH_OK)
		return rc;

	/*Restore transfer mode */
	/*======================*/
	tffsset(&ioreq2, 0, sizeof(ioreq));

	ioreq2.irCount  = pdev->device[pdev->bAtaDevNum].dataTransferMode;
	ioreq2.irLength = pdev->device[pdev->bAtaDevNum].dwMulti_Current;

	rc = DOCHSetDataTransferModeSingleFloor(&ioreq2);
	if(rc != DOCH_OK)
		return rc;


	return DOCH_OK;
}

DOCH_Error DOCHNotifyPlatformResumed(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket* pdev;
	FLByte origDevNum;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	origDevNum = pdev->bAtaDevNum;

	pdev->bAtaDevNum = 0;
	rc = DOCHNotifyPlatformResumedSingleFloor(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHNotifyPlatformResumedSingleFloor(ioreq);
		pdev->bAtaDevNum = origDevNum;
	}

	return rc;
}

/*----------------------------------------------------------------------*/
/*	    f l D O C H S e t D i s k U s e r A t t r i b u t e s			*/
/*                                                                      */
/* Set disk user attributes					.					        */
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*  irData           : Pointer to 1 sector of data						*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_SET_DISK_USER_ATTR										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetDiskUserAttributes(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if(!gSDKUpdatesDiskAttributes)
	{
		/*First retrieve current attributes sector
		  (for protecting first 0x40 bytes that are SDK specific*/
		/*------------------------------------------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irData = currentAttributes;
		DOCHGetDiskUserAttributes(&ioreq2);

		/*Use currentAttributes bytes 0x41..0x200 to set requested*/
		/*--------------------------------------------------------*/
		tffscpy(ioreq->irData, currentAttributes, DOCH_PART_INFO_SDK_RESERVED_BYTES);
	}

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_DISK_USER_ATTR;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(1));

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetDiskUserAttributes(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*	    f l D O C H G e t D i s k U s e r A t t r i b u t e s			*/
/*                                                                      */
/* Get disk user attributes					.					        */
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*  irData           : Pointer to 1 sector of data						*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_GET_DISK_USER_ATTR										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetDiskUserAttributes(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_DISK_USER_ATTR;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(1));

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetDiskUserAttributes(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*	    f l D O C H G e t C o n f i g u r a t i o n D a t a				*/
/*                                                                      */
/* Read the device configuration data. No authentication is required	*/
/* for calling this command, but the fields containing keys in the		*/
/* configuration data will be masked by ETFFS using zeros. 				*/
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*  irData           : Pointer to 1..3 sector(s) of data.				*/
/*  irCount			 : Sector offset in the configuration data to start	*/
/*					   reading from.									*/
/*  irLength		 : Number of sectors of configuration data to read.	*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_GET_CONFIGURATION_DATA									*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error        : 0 on success, otherwise failed            */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetConfigurationData(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_CONFIGURATION_DATA;
	in_regs.bSectorCount = (FLByte)ioreq->irLength;
	in_regs.bSectorNumber = (FLByte)ioreq->irCount;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(ioreq->irLength));

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetConfigurationData(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*	    f l D O C H S e t C o n f i g u r a t i o n D a t a				*/
/*                                                                      */
/* Save the device configuration data. Calling this command will		*/
/* succeed only if there are no partitions present on the media			*/
/* (including partition 0).												*/
/* You should use this command in the process of duplicating DiskOnChip	*/
/* device to make sure all configuration data of the original device	*/
/* have been duplicated to the new device.						        */
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*  irData           : Pointer to 1..3 sector(s) of data.				*/
/*  irCount			 : Sector offset in the configuration data to start	*/
/*					   writing from.									*/
/*  irLength		 : Number of sectors of configuration data to write.*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_SET_CONFIGURATION_DATA									*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error        : 0 on success, otherwise failed            */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetConfigurationData(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_CONFIGURATION_DATA;
	in_regs.bSectorCount = (FLByte)ioreq->irLength;
	in_regs.bSectorNumber = (FLByte)ioreq->irCount;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(ioreq->irLength));

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetConfigurationData(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*             D O C H P a r t i t i o n I n f o						*/
/*                                                                      */
/* Get information about a specific partition.                          */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*                        Partition number (zero based)		            */
/*      irData			: Address of user buffer to read partition      */
/*                        information into.                             */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_GET_PARTITION_INFO										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHPartitionInfoSingleFloor(IOreq *ioreq)
{
	DOCH_Error  rc;
	DOCH_Socket *pdev;

#ifdef DOCH_BIG_ENDIAN
	DOCH_PartitionInfo *partInfo = (DOCH_PartitionInfo*)(ioreq->irData);
	DOCH_PartitionInfo *partInfoToBig = partInfo;
#endif /*DOCH_BIG_ENDIAN*/

	DOCH_Registers in_regs;
	DOCH_Registers out_regs;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;


	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_PARTITION_INFO;
	in_regs.bSectorCount   = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber  = 0;
	in_regs.bCylLow        = 0;
	in_regs.bCylHigh       = 0;
	in_regs.bDriveHead     = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_PartitionInfo) / DOCH_SECTOR_SIZE));

#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	partInfoToBig->dwCommandFlagsOrStatuses = be_FLDword((FLByte*)&partInfoToBig->dwCommandFlagsOrStatuses);
	partInfoToBig->partitionAttributes1		= be_FLDword((FLByte*)&partInfoToBig->partitionAttributes1);
	partInfoToBig->partitionAttributes2		= be_FLDword((FLByte*)&partInfoToBig->partitionAttributes2);
	partInfoToBig->nPartitionSize			= be_FLDword((FLByte*)&partInfoToBig->nPartitionSize);
	partInfoToBig->nFastAreaSize			= be_FLDword((FLByte*)&partInfoToBig->nFastAreaSize);

	partInfoToBig->wFastAreaFactor			= be_FLWord((FLByte*)&partInfoToBig->wFastAreaFactor);

	partInfoToBig->wPartitionStartSector	= be_FLWord((FLByte*)&partInfoToBig->wPartitionStartSector);
	partInfoToBig->wCurrentSectorsPerTrack	= be_FLWord((FLByte*)&partInfoToBig->wCurrentSectorsPerTrack);
	partInfoToBig->wDefaultSectorsPerTrack	= be_FLWord((FLByte*)&partInfoToBig->wDefaultSectorsPerTrack);
	partInfoToBig->wCurrentCylinders		= be_FLWord((FLByte*)&partInfoToBig->wCurrentCylinders);
	partInfoToBig->wDefaultCylinders		= be_FLWord((FLByte*)&partInfoToBig->wDefaultCylinders);
	partInfoToBig->wCurrentHeads			= be_FLWord((FLByte*)&partInfoToBig->wCurrentHeads);
	partInfoToBig->wDefaultHeads			= be_FLWord((FLByte*)&partInfoToBig->wDefaultHeads);

	partInfoToBig->wFastAreaSectorsInErasableUnit	= be_FLWord((FLByte*)&partInfoToBig->wFastAreaSectorsInErasableUnit);
	partInfoToBig->wNormalAreaSectorsInErasableUnit	= be_FLWord((FLByte*)&partInfoToBig->wNormalAreaSectorsInErasableUnit);
	partInfoToBig->wRecommendedSectorsPerCluster	= be_FLWord((FLByte*)&partInfoToBig->wRecommendedSectorsPerCluster);
	partInfoToBig->wFastMaxRelatedSectors			= be_FLWord((FLByte*)&partInfoToBig->wFastMaxRelatedSectors);
	partInfoToBig->wNormalMaxRelatedSectors			= be_FLWord((FLByte*)&partInfoToBig->wNormalMaxRelatedSectors);
#endif /*DOCH_BIG_ENDIAN*/

    /*Update pdev fields*/
#if 0
    pdev->device[pdev->bAtaDevNum].dwMulti_Current =
        ((partInfo->dwCommandFlagsOrStatuses & DOCH_DCFSB_CURRENT_MULTI_SECTOR) >> DOCH_DCFSO_CURRENT_MULTI_SECTOR);
#endif /* 0 */

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHPartitionInfo(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHPartitionInfo(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	DOCH_PartitionInfo* partInfo;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		rc = DOCHPartitionInfoSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1 - return info*/
	/*===================================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHPartitionInfoSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*Retrieve first chunk info*/
		/*-------------------------*/
		partInfo = (DOCH_PartitionInfo*)(ioreq->irData);
		rc = DOCHPartitionInfoSingleFloor(ioreq);
		if(rc != DOCH_OK)
			return rc;

		/*Retrieve 2nd chunk info*/
		/*-----------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irData = &partitionInfoTemp;

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHPartitionInfoSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;

		/*Combine info*/
		/*------------*/
		partInfo->nPartitionSize += partitionInfoTemp.nPartitionSize;
	}

	return DOCH_OK;
}


/*----------------------------------------------------------------------*/
/*            D O C H D e l e t e P a r t i t i o n s					*/
/*                                                                      */
/* Delete range of partitions											*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*		irFlags			: DOCH_NORMAL_OPERATION                         */
/*						  DOCH_IMMEDIATE_OPERATION						*/
/*						  DOCH_FAST_PARTITOIN_DELETE					*/
/*						  DOCH_COMPLETE_PARTITION_DELETE				*/
/*      irCount         : First partition to delete                     */
/*      irLength        : Last partition to delete						*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_DELETE_PARTITIONS										*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHDeletePartitionsSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	FLByte action_type = 0;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if((ioreq->irFlags & DOCH_IMMEDIATE_OPERATION) == DOCH_IMMEDIATE_OPERATION)
		action_type |= ERASE_IMMEDIATE;

	if((ioreq->irFlags & DOCH_COMPLETE_PARTITION_DELETE) == DOCH_COMPLETE_PARTITION_DELETE)
		action_type |= DELETE_PARTITIONS_COMPLETE;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError	= DOCH_DELETE_PARTITIONS;
	in_regs.bSectorCount	= (FLByte)(ioreq->irCount);
	in_regs.bSectorNumber	= (FLByte)(ioreq->irLength);
	in_regs.bCylLow			= 0;
	in_regs.bCylHigh		= 'D';
	in_regs.bDriveHead		= ((pdev->bAtaDevNum * DOCH_DEVICE) | action_type);
	in_regs.bCommandStatus	= DOCH_VSCMD_PARTITION_MANAGEMENT;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
	{
		pdev->wTotalNumOfPartitions = 0;
		tffsset(&ioreq2, 0, sizeof(ioreq2));
		DOCH_SET_SOCKET_TO_IOREQ_HANDLE(&ioreq2, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
		ioreq2.irData = &diskOnChipDeviceInfo;
		rc = DOCHIdentifyDiskOnChipDevice(&ioreq2);
		if(rc != DOCH_OK)
			return rc;

		pdev->wTotalNumOfPartitions = diskOnChipDeviceInfo.wTotalNumOfPartitions;

		return DOCH_OK;
	}
	else
	{

		DBG_PRINT_ERR(FLZONE_API, "DOCHDeletePartitions(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHDeletePartitions(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0 - return info*/
	/*===================================================*/
	if(!pdev->sSpanData.secondFloorActive)
	{
		rc = DOCHDeletePartitionsSingleFloor(ioreq);
		return rc;
	}
	else
	{
		/*Protect against illegal operation*/
		if(ioreq->irLength < (pdev->wTotalNumOfPartitions-1))
			return DOCH_GeneralFailure;

		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		/*Do not delete partitions from Dev0 if they do not reside on it!*/
		if(ioreq->irCount <= pdev->sSpanData.bLastPartitionOnDev0)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

			/*If last partition to delete resides on Dev1, erase irCount and on on Dev0 as well*/
			if(ioreq->irLength >= pdev->sSpanData.bLastPartitionOnDev0)
				ioreq2.irLength = 0xFF;

			rc = DOCHDeletePartitionsSingleFloor(&ioreq2);
			if(rc != DOCH_OK)
				return rc;

			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

			/*Preserve IPL on Dev1 if it was preserved on Dev0*/
			if(ioreq->irCount == 0)
				ioreq2.irCount = 0;
			else
				ioreq2.irCount = 1;

			ioreq2.irLength = 0xFF;

			pdev->bAtaDevNum = 1;
			rc = DOCHDeletePartitionsSingleFloor(&ioreq2);
			pdev->bAtaDevNum = 0;

			if(rc != DOCH_OK)
				return rc;
		}
		else
		{
			ioreq2.irCount = ((ioreq->irCount - pdev->sSpanData.bLastPartitionOnDev0) + 1);
			if(ioreq->irLength != 0xFF)
				ioreq2.irLength = ((ioreq->irLength - pdev->sSpanData.bLastPartitionOnDev0) + 1);

			pdev->bAtaDevNum = 1;
			rc = DOCHDeletePartitionsSingleFloor(&ioreq2);
			pdev->bAtaDevNum = 0;

			if(rc != DOCH_OK)
				return rc;

			/*Still spanned!*/
			return DOCH_OK;
		}
	}

	/*Update span info structure*/
	/*--------------------------*/
	/*Retrieve current attributes*/
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	ioreq2.irData = &devUserAttr;
	rc = DOCHGetDiskUserAttributes(&ioreq2);
	if(rc != DOCH_OK)
		return rc;

	/*Set span info*/
	devUserAttr.sdkAttributes.sSpanInfo.bLastDev0PartSpanned		= FALSE;
	devUserAttr.sdkAttributes.sSpanInfo.secondFloorActive			= FALSE;
	devUserAttr.sdkAttributes.sSpanInfo.dwSpanSector				= 0;
	devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0		=
	devUserAttr.sdkAttributes.sSpanInfo.numOfSpannedPartitionOnDev0	= 0;

	/*Set attributes back to the device*/
	gSDKUpdatesDiskAttributes = DOCH_GLOBAL_BOOL_PATTERN;
	rc = DOCHSetDiskUserAttributes(&ioreq2);
	gSDKUpdatesDiskAttributes = 0;
	if(rc != DOCH_OK)
		return rc;

	/*Update pdev as well*/
	tffscpy(&pdev->sSpanData, &devUserAttr.sdkAttributes.sSpanInfo, sizeof(pdev->sSpanData));

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*	            D O C H A d d P a r t i t i o n							*/
/*                                                                      */
/* Adds a partition														*/
/* The new partition is created on existing un-formatted disk capacity	*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*      irData			: Pointer to DOCH_PartitionFormatInfoAPI struct */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_ADD_PARTITION											*/
/*																		*/
/* Returns:                                                             */
/*        irCount	: Number of the created partition                   */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHAddPartitionSingleFloor( IOreq  *ioreq)
{
	DOCH_Error                  rc;
	IOreq                       ioreq2;
	DOCH_Registers              in_regs;
	DOCH_Registers              out_regs;
	DOCH_PartitionFormatInfoAPI *partInfo = (DOCH_PartitionFormatInfoAPI*)(ioreq->irData);
	DOCH_Socket                 *pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/* Clear partFormatInfo */
	/*----------------------*/
	tffsset(&partFormatInfo, 0, sizeof(DOCH_PartitionFormatInfo));

	/* Format DOCH_PartitionFormatInfoAPI to DOCH_PartitionFormatInfo */
	/*----------------------------------------------------------------*/
	DOCH_SetBits(&partFormatInfo.dwCommandFlagsOrStatuses,
				 DOCH_CFSB_FAST_AREA_SIZE_TYPE,
				 DOCH_CFSO_FAST_AREA_SIZE_TYPE,
				 partInfo->dwFastAreaSizeType);

	DOCH_SetBits(&partFormatInfo.partitionAttributes1,
				 DOCH_PA1B_PARTITION_TYPE,
				 DOCH_PA1O_PARTITION_TYPE,
				 partInfo->dwPartitionType);
	DOCH_SetBits(&partFormatInfo.partitionAttributes1,
				 DOCH_PA1B_PERFORMANCE_CTRL,
				 DOCH_PA1O_PERFORMANCE_CTRL,
				 partInfo->dwPerformanceControl);
	DOCH_SetBits(&partFormatInfo.partitionAttributes1,
				 DOCH_PA1B_PAGE_SIZE_EXP,
				 DOCH_PA1O_PAGE_SIZE_EXP,
				 partInfo->dwPageSizeExp);
	DOCH_SetBits(&partFormatInfo.partitionAttributes1,
				 DOCH_PA1B_HASH_TYPE,
				 DOCH_PA1O_HASH_TYPE,
				 partInfo->dwHashType);

	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_PROTECTION_TYPE,
				 DOCH_PA2O_PROTECTION_TYPE,
				 partInfo->dwProtectionType);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_USER_MODE,
				 DOCH_PA2O_USER_MODE,
				 partInfo->dwUserAccessMode);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_GUEST_MODE,
				 DOCH_PA2O_GUEST_MODE,
				 partInfo->dwGuestAccessMode);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_MASTER_CTRL,
				 DOCH_PA2O_MASTER_CTRL,
				 partInfo->dwMasterControl);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_ENCRYPT_TYPE,
				 DOCH_PA2O_ENCRYPT_TYPE,
				 partInfo->dwEncryptionType);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_OTP_BIT,
				 DOCH_PA2O_OTP_BIT,
				 partInfo->dwOtpEnabled);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_LOCK_CTRL,
				 DOCH_PA2O_LOCK_CTRL,
				 partInfo->dwLockControl);
	DOCH_SetBits(&partFormatInfo.partitionAttributes2,
				 DOCH_PA2B_MAX_AUTH_ATTEMPTS,
				 DOCH_PA2O_MAX_AUTH_ATTEMPTS,
				 partInfo->dwMaxNumOfAuthAttempts);

	partFormatInfo.nPartitionSize							= partInfo->nPartitionSize;
	partFormatInfo.nFastAreaSize							= partInfo->nFastAreaSize;
	partFormatInfo.wFastAreaFactor							= partInfo->wFastAreaFactor;
	tffscpy(partFormatInfo.bPasskey, &partInfo->bPasskey, sizeof(partFormatInfo.bPasskey));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
 	partFormatInfo.dwCommandFlagsOrStatuses	= be_FLDword((FLByte*)&partFormatInfo.dwCommandFlagsOrStatuses);
	partFormatInfo.partitionAttributes1	    = be_FLDword((FLByte*)&partFormatInfo.partitionAttributes1);
	partFormatInfo.partitionAttributes2	    = be_FLDword((FLByte*)&partFormatInfo.partitionAttributes2);
	partFormatInfo.nPartitionSize			= be_FLDword((FLByte*)&partFormatInfo.nPartitionSize);
	partFormatInfo.nFastAreaSize			= be_FLDword((FLByte*)&partFormatInfo.nFastAreaSize);
	partFormatInfo.wFastAreaFactor			= be_FLWord ((FLByte*)&partFormatInfo.wFastAreaFactor);
	#endif /*DOCH_BIG_ENDIAN*/

	/* Update ATA register values */
	/*----------------------------*/
	in_regs.bFeaturesError = DOCH_ADD_PARTITION;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/* Activate ATA command */
	/*----------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    &partFormatInfo,
						sizeof(DOCH_PartitionFormatInfo) / DOCH_SECTOR_SIZE);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHAddPartition(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}

	pdev->wTotalNumOfPartitions = 0;
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	DOCH_SET_SOCKET_TO_IOREQ_HANDLE(&ioreq2, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	ioreq2.irData = &diskOnChipDeviceInfo;
	rc = DOCHIdentifyDiskOnChipDevice(&ioreq2);
	if(rc != DOCH_OK)
		return rc;

	pdev->wTotalNumOfPartitions = diskOnChipDeviceInfo.wTotalNumOfPartitions;

	/* Output */
	/*--------*/
	ioreq->irCount = out_regs.bSectorCount;

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHAddPartition(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHAddPartition(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_PartitionFormatInfoAPI* partInfo = (DOCH_PartitionFormatInfoAPI*)(ioreq->irData);
	DOCH_Socket* pdev;
	FLDword reqSize = partInfo->nPartitionSize;
	FLDword secondChunkSize = 0;
	FLDword tempPartNum = 0;
	FLBoolean dev0MightBeFull = FALSE;
	FLByte tempLastPartOnDev0 = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If 2nd floor is already active - perform on device1*/
	/*===================================================*/
	if((pdev->sSpanData.secondFloorActive) && (pdev->wNumOfDevices > 1))
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHAddPartitionSingleFloor(ioreq);
		pdev->bAtaDevNum = 0;

		/*Update returned partition number*/
		ioreq->irCount += (pdev->sSpanData.bLastPartitionOnDev0);

		if(pdev->sSpanData.bLastDev0PartSpanned)
			ioreq->irCount -= 1;

		return rc;
	}

	/*If requsted size = 0, Add to both Dev0 and Dev1*/
	/*===============================================*/
	if(reqSize == 0)
	{
		/*Dev0*/
		tffsset(&ioreq2, 0, sizeof(ioreq2));
		ioreq2.irData = ioreq->irData;
		rc = DOCHAddPartitionSingleFloor(&ioreq2);
		if(rc != DOCH_OK)
		{
			/* If operation failed with insufficient space,
			   it means last partition that was added filled Dev0 entirely*/
			dev0MightBeFull = TRUE;
			tempLastPartOnDev0 = devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0;

			/*return rc*/;
		}

		if(!dev0MightBeFull)
		{
			/*If created partition was IPL:*/
			/*1. Delete all partition on dev1*/
			/*2. Create on Dev1 as well*/
			if((pdev->wNumOfDevices > 1) && (ioreq2.irCount == 0))
			{
				tffsset(&ioreq2, 0, sizeof(ioreq2));
				ioreq2.irData = ioreq->irData;
				ioreq2.irLength = 0xFF;
				pdev->bAtaDevNum = 1;
				rc = DOCHDeletePartitions(&ioreq2);
				pdev->bAtaDevNum = 0;
				if(rc != DOCH_OK)
					return rc;

				tffsset(&ioreq2, 0, sizeof(ioreq2));
				ioreq2.irData = ioreq->irData;
				pdev->bAtaDevNum = 1;
				rc = DOCHAddPartitionSingleFloor(&ioreq2);
				pdev->bAtaDevNum = 0;
				if(rc != DOCH_OK)
					return rc;
			}

			/*Make sure returned partition nuymber is correct*/
			devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0 = (FLByte)ioreq2.irCount;
			ioreq->irCount = ioreq2.irCount;
		}

		if(pdev->wNumOfDevices > 1)
		{
			if(!dev0MightBeFull)
			{
				/*Get actual size of the newly added partition*/
				tempPartNum = ioreq2.irCount;
				tffsset(&ioreq2, 0, sizeof(ioreq2));
				tffsset(&partitionInfoTemp, 0, sizeof(partitionInfoTemp));
				DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, tempPartNum);
				ioreq2.irData = &partitionInfoTemp;
				rc = DOCHPartitionInfoSingleFloor(&ioreq2);
				if(rc != DOCH_OK)
					return rc;
			}

			/*Add User partition to Dev1 with size = 0*/
			tffsset(&ioreq2, 0, sizeof(ioreq2));
			ioreq2.irData = ioreq->irData;
			pdev->bAtaDevNum = 1;
			rc = DOCHAddPartitionSingleFloor(&ioreq2);
			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;

			/*Update span info structure*/
			/*--------------------------*/
			/*Retrieve current attributes*/
			tffsset(&ioreq2, 0, sizeof(ioreq2));
			ioreq2.irData = &devUserAttr;
			rc = DOCHGetDiskUserAttributes(&ioreq2);
			if(rc != DOCH_OK)
				return rc;

			/*Set span info*/
			devUserAttr.sdkAttributes.sSpanInfo.bLastDev0PartSpanned		= (!dev0MightBeFull);/*TRUE;*/
			devUserAttr.sdkAttributes.sSpanInfo.secondFloorActive			= TRUE;

			if(!dev0MightBeFull)
			{
				devUserAttr.sdkAttributes.sSpanInfo.dwSpanSector				= partitionInfoTemp.nPartitionSize;
				devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0		=
				devUserAttr.sdkAttributes.sSpanInfo.numOfSpannedPartitionOnDev0	= (FLByte)(tempPartNum);
			}
			else
			{
				devUserAttr.sdkAttributes.sSpanInfo.dwSpanSector				= 0;
				devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0 = tempLastPartOnDev0;
			}

			/*Set attributes back to the device*/
			gSDKUpdatesDiskAttributes = DOCH_GLOBAL_BOOL_PATTERN;
			rc = DOCHSetDiskUserAttributes(&ioreq2);
			gSDKUpdatesDiskAttributes = 0;
			if(rc != DOCH_OK)
				return rc;

			/*Update pdev as well*/
			tffscpy(&pdev->sSpanData, &devUserAttr.sdkAttributes.sSpanInfo, sizeof(pdev->sSpanData));
		}

		return DOCH_OK;
	}

	/*Try adding to Dev0*/
	/*==================*/
	rc = DOCHAddPartitionSingleFloor(ioreq);
	if(rc == DOCH_OK)
	{
		devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0 = (FLByte)(ioreq->irCount);

		/*If created partition was IPL:*/
		/*1. Delete all partition on dev1*/
		/*2. Create on Dev1 as well*/
		if((pdev->wNumOfDevices > 1) && (ioreq->irCount == 0))
		{
			tffsset(&ioreq2, 0, sizeof(ioreq2));
			ioreq2.irData = ioreq->irData;
			ioreq2.irLength = 0xFF;
			pdev->bAtaDevNum = 1;
			rc = DOCHDeletePartitions(&ioreq2);
			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;

			tffsset(&ioreq2, 0, sizeof(ioreq2));
			ioreq2.irData = ioreq->irData;
			pdev->bAtaDevNum = 1;
			rc = DOCHAddPartitionSingleFloor(&ioreq2);
			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;
		}

		return DOCH_OK;
	}
	else if(pdev->wNumOfDevices > 1)
	{
		/*If Adding entire size on Dev0 failed*/
		/*====================================*/
		/*Try adding to Dev0 with size=0*/
		tffsset(&ioreq2, 0, sizeof(ioreq2));
		tffscpy(&partInfoTemp, ioreq->irData, sizeof(partInfoTemp));
		partInfoTemp.nPartitionSize = 0;
		ioreq2.irData = &partInfoTemp;
		rc = DOCHAddPartitionSingleFloor(&ioreq2);
		if(rc != DOCH_OK)
		{
			/* If operation failed with insufficient space,
			   it means last partition that was added filled Dev0 entirely*/

			dev0MightBeFull = TRUE;
			tempLastPartOnDev0 = devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0;

			/*return rc;*/
		}

		if(!dev0MightBeFull)
		{
			/*Make sure returned partition number is correct*/
			ioreq->irCount = ioreq2.irCount;

			/*Get actual size of the newly added partition*/
			tempPartNum = ioreq2.irCount;
			tffsset(&ioreq2, 0, sizeof(ioreq2));
			tffsset(&partitionInfoTemp, 0, sizeof(partitionInfoTemp));
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, tempPartNum);
			ioreq2.irData = &partitionInfoTemp;
			rc = DOCHPartitionInfoSingleFloor(&ioreq2);
			if(rc != DOCH_OK)
				return rc;

			secondChunkSize = (reqSize - partitionInfoTemp.nPartitionSize);
		}
		else
		{
			secondChunkSize = reqSize;
		}

		/*Try adding to Dev1 with size=secondChunkSize*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		tffscpy(&partInfoTemp, ioreq->irData, sizeof(partInfoTemp));
		partInfoTemp.nPartitionSize = secondChunkSize;
		ioreq2.irData = &partInfoTemp;

		pdev->bAtaDevNum = 1;
		rc = DOCHAddPartitionSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;

		/*Update span info structure*/
		/*--------------------------*/
		/*Retrieve current attributes*/
		tffsset(&ioreq2, 0, sizeof(ioreq2));
		ioreq2.irData = &devUserAttr;
		rc = DOCHGetDiskUserAttributes(&ioreq2);
		if(rc != DOCH_OK)
			return rc;

		/*Set span info*/
		devUserAttr.sdkAttributes.sSpanInfo.bLastDev0PartSpanned		= TRUE;
		devUserAttr.sdkAttributes.sSpanInfo.secondFloorActive			= TRUE;

		if(!dev0MightBeFull)
		{
			devUserAttr.sdkAttributes.sSpanInfo.dwSpanSector				= partitionInfoTemp.nPartitionSize;
			devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0		=
			devUserAttr.sdkAttributes.sSpanInfo.numOfSpannedPartitionOnDev0	= (FLByte)(tempPartNum);
		}
		else
		{
			devUserAttr.sdkAttributes.sSpanInfo.dwSpanSector				= 0;
			devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0 = tempLastPartOnDev0;
		}

		/*Set attributes back to the device*/
		gSDKUpdatesDiskAttributes = DOCH_GLOBAL_BOOL_PATTERN;
		rc = DOCHSetDiskUserAttributes(&ioreq2);
		gSDKUpdatesDiskAttributes = 0;
		if(rc != DOCH_OK)
			return rc;

		/*Update pdev as well*/
		tffscpy(&pdev->sSpanData, &devUserAttr.sdkAttributes.sSpanInfo, sizeof(pdev->sSpanData));
	}
	else
	{
		return DOCH_GeneralFailure;
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*				D O C H U n f o r m a t D e v i c e						*/
/*                                                                      */
/* Unformats DOCH device.												*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_DELETE_PARTITIONS										*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHUnformatSingleDevice(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;

	/*1. Delete all partitions */
	/*=========================*/
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	DOCH_SET_SOCKET_TO_IOREQ_HANDLE(&ioreq2, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	ioreq2.irCount	= 0;
	ioreq2.irLength = 0xFF;
	ioreq2.irFlags	= DOCH_COMPLETE_PARTITION_DELETE;

	rc = DOCHDeletePartitions(&ioreq2);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHUnformatDevice(): DOCHDeletePartitions() failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

	/*2. Delete Disk Attributes */
	/*==========================*/
	tffsset(&currentAttributes, 0, sizeof(currentAttributes));
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	DOCH_SET_SOCKET_TO_IOREQ_HANDLE(&ioreq2, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	ioreq2.irData = currentAttributes;

	gSDKUpdatesDiskAttributes = DOCH_GLOBAL_BOOL_PATTERN;
	rc = DOCHSetDiskUserAttributes(&ioreq2);
	gSDKUpdatesDiskAttributes = 0;
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHUnformatDevice(): DOCHSetDiskUserAttributes() failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

	return DOCH_OK;
}

DOCH_Error DOCHUnformatDevice(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;
	DOCH_Socket* pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	rc = DOCHUnformatSingleDevice(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHUnformatSingleDevice(ioreq);
		pdev->bAtaDevNum = 0;

		if(rc != DOCH_OK)
			return rc;
	}

	/*Update span info structure*/
	/*--------------------------*/
	/*Retrieve current attributes*/
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	ioreq2.irData = &devUserAttr;
	rc = DOCHGetDiskUserAttributes(&ioreq2);
	if(rc != DOCH_OK)
		return rc;

	/*Set span info*/
	devUserAttr.sdkAttributes.sSpanInfo.bLastDev0PartSpanned		= FALSE;
	devUserAttr.sdkAttributes.sSpanInfo.secondFloorActive			= FALSE;
	devUserAttr.sdkAttributes.sSpanInfo.dwSpanSector				= 0;
	devUserAttr.sdkAttributes.sSpanInfo.bLastPartitionOnDev0		=
	devUserAttr.sdkAttributes.sSpanInfo.numOfSpannedPartitionOnDev0	= 0;

	/*Set attributes back to the device*/
	gSDKUpdatesDiskAttributes = DOCH_GLOBAL_BOOL_PATTERN;
	rc = DOCHSetDiskUserAttributes(&ioreq2);
	gSDKUpdatesDiskAttributes = 0;
	if(rc != DOCH_OK)
		return rc;

	/*Update pdev as well*/
	tffscpy(&pdev->sSpanData, &devUserAttr.sdkAttributes.sSpanInfo, sizeof(pdev->sSpanData));

	return DOCH_OK;
}


/*----------------------------------------------------------------------*/
/*	           D O C H W r i t e I P L									*/
/*                                                                      */
/* Writes IPL partition.												*/
/* Sets IPL mode and size.												*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*		irFlags			: DOCH_IPL_MODE_NORMAL_RAM                      */
/*						  DOCH_IPL_MODE_PAGED_RAM						*/
/*						  DOCH_IPL_MODE_VIRTUAL_RAM						*/
/*						  DOCH_IPL_MODE_CS_DELAY						*/
/*						  DOCH_IPL_MODE_ADDRESS_SHIFT_IN_AFFECT			*/
/*						  DOCH_IPL_MODE_ACTIVE_SWAP_BYTES				*/
/*						  DOCH_IPL_MODE_8KB_WINDOW						*/
/*						  DOCH_IPL_WRITE_FIRST_CHUNK					*/
/*		irCount			: Max IPL size						            */
/*						  Up to 32KB  for Normal Mode	                */
/*						  Up to 128KB for Virtual Mode	                */
/*						  Up to 256KB	for Paged Mode		            */
/*		irLength		: Current chunk length (in sectors)				*/
/*		irData			: Pointer to user buffer (length defined by		*/
/*						  irLength)										*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/*			DOCH_VSCMD_WRITE_PARTITION									*/
/* ATA sub-command:														*/
/*			DOCH_SET_CUSTOM_PARAM										*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWriteIPLSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;
	IOreq ioreq3;
	FLByte i;
	FLBoolean firstChunk = ((ioreq->irFlags & DOCH_IPL_WRITE_FIRST_CHUNK) == DOCH_IPL_WRITE_FIRST_CHUNK);
	DOCH_Socket*  pdev;

	FLSDword	total_num_of_sectors;
	FLSDword	sectorNumber;
	FLSNative	sectors_written_so_far;
	FLByte*		buf_ptr;
	FLSNative	buf_offset;

	FLByte		deviceNum = 0;
	FLNative    deviceNo = 0;
	FLBoolean	enable = FALSE;

	Addressing_Values_s addr_vals;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;

	FLByte operationRetries = 0;
	FLByte* verBufOrig;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if(firstChunk && (pdev->bAtaDevNum == 0))
	{
		/*1. Set IPL BOOT Mode */
		tffsset(&ioreq2, 0, sizeof(ioreq2));
		ioreq2.irCount		= DOCH_CP_BOOT_MODE;
		ioreq2.irSectorNo	= (FLWord)(ioreq->irFlags);
		ioreq2.irFlags		= DOCH_CUSTOM_PARAM_DEFAULT;
        rc = DOCHSetCustomParameter(&ioreq2);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHWriteIPL(): DOCHSetCustomParameter(Boot Mode) failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			return rc;
		}

		/*2. Set IPL MAX Size */
		tffsset(&ioreq2, 0, sizeof(ioreq2));
		ioreq2.irCount		= DOCH_CP_XIP_MAX_SIZE;
		ioreq2.irSectorNo	= ioreq->irCount;
		ioreq2.irFlags		= DOCH_CUSTOM_PARAM_DEFAULT;
        rc = DOCHSetCustomParameter(&ioreq2);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHWriteIPL(): DOCHSetCustomParameter(XIP Size) failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			return rc;
		}

		gIPLChunkOffsetInSectors = 0;
	}

	/*3. Write chunk of data */
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	ioreq2.irData		 = ioreq->irData;
	ioreq2.irSectorNo	 = gIPLChunkOffsetInSectors;
	ioreq2.irSectorCount = ioreq->irLength;

	/*Sector arithmetic*/
	/*-----------------*/
	total_num_of_sectors	= ioreq2.irSectorCount;
	sectorNumber			= ioreq2.irSectorNo;
	sectors_written_so_far	= 0;

	/*Buffer to write from*/
	/*--------------------*/
	buf_ptr = (FLByte*)ioreq2.irData;
	buf_offset = 0;

	deviceNum = pdev->bAtaDevNum;

	/*Sanity Check*/
	/*------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*Enable interrupts*/
	/*-----------------*/
	pdev->bUseInterrupt = TRUE;

	enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

	for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
	{
		if(enable && (gAccessLayerType != DOCH_AL_SIM))
		{
			pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
		}
		else
		{
			pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
		}
		DOCHWRITE_ATA_REG ( pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
	}

	/*Set DMA and Burst modes according to request*/
	/*--------------------------------------------*/
	pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
					   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
					   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

	pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

	/*Perform write cycles in DOCH_MAX_SECTORS quantities*/
	/*---------------------------------------------------*/
	while(total_num_of_sectors > 0)
	{
		FLSNative  sectors_this_cycle;

		if(total_num_of_sectors > DOCH_MAX_SECTORS)
			sectors_this_cycle = DOCH_MAX_SECTORS;
		else
			sectors_this_cycle = total_num_of_sectors;

dochWriteRetryPointDev1Only:

#ifdef CHECK_POWER_ON_EVERY_COMMAND
		/* Check if device was reset */
		rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
		if(rc != DOCH_OK)
			return rc;
#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

		/*Perform 1 operation cycle*/
		/*=========================*/

		/*Set ATA addressing registers*/
		/*---------------------------*/
		DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

		/*Update ATA register values*/
		/*--------------------------*/
		in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
		if(sectors_this_cycle < DOCH_MAX_SECTORS)
			in_regs.bSectorCount = sectors_this_cycle;
		else
			in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

		in_regs.bSectorNumber = addr_vals.bSecNum;
		in_regs.bCylLow = addr_vals.bCylLow;
		in_regs.bCylHigh = addr_vals.bCylHi;
		in_regs.bDriveHead = addr_vals.bDevHead;
		in_regs.bCommandStatus = DOCH_VSCMD_WRITE_PARTITION;

		/*Activate ATA command*/
		/*--------------------*/
		#ifdef DOCH_DMA_CONFIG
		DOCH_OPEN_DMA_CHANNEL
		#endif /*DOCH_DMA_CONFIG*/

		/*Perform ATA command*/
		rc = io_output (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

		if(rc != DOCH_OK)
		{
			rc = get_out_registers(pdev, deviceNum, &out_regs);

			if(rc != DOCH_OK)
			{
				DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): failed with status: ");
				DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

				if( ((rc & DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) == DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) &&
					(operationRetries++ < 4)
					)
					goto dochWriteRetryPointDev1Only;

				/*Disable interrupts*/
				/*------------------*/
				pdev->bUseInterrupt = FALSE;
				dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

				/*Cancel DMA and Burst modes*/
				/*--------------------------*/
				pdev->bUseDMA	= FALSE;
				pdev->bUseBurst = FALSE;

				pdev->bAtaDevNum = 0;

				if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
					return DOCH_ProtectionFault;
				else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
					return DOCH_PartitionLimitExceeded;
				else
					return DOCH_GeneralFailure;
			}
		}

		/*Update feedback values*/
		/*----------------------*/
		sectors_written_so_far += sectors_this_cycle;

		/*For next possible cycle*/
		/*-----------------------*/
		sectorNumber += sectors_this_cycle;
		total_num_of_sectors -= sectors_this_cycle;
		buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

		#ifdef CHECK_POWER_ON_EVERY_COMMAND
		/* Check if device was reset */
		rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
		if(rc != DOCH_OK)
			return rc;
		#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
	}

	/*Disable interrupts*/
	/*------------------*/
	pdev->bUseInterrupt = FALSE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

	/*Cancel DMA and Burst modes*/
	/*--------------------------*/
	pdev->bUseDMA	= FALSE;
	pdev->bUseBurst = FALSE;


	/*If verify was define, perform verify operation with same parameters*/
	/*-------------------------------------------------------------------*/
	if(gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2)] == DOCH_GLOBAL_BOOL_PATTERN)
	{
		tffsset(&ioreq3, 0, sizeof(ioreq3));
		tffscpy(&ioreq3.irHandle, &ioreq2.irHandle, sizeof(ioreq3.irHandle));
		ioreq3.irData = verifyBuf;
		verBufOrig = (FLByte*)(ioreq2.irData);

		for(i=0; i<ioreq2.irSectorCount; i++)
		{
			ioreq3.irSectorCount = 1;
			ioreq3.irSectorNo	 = (ioreq2.irSectorNo + i);
			rc = DOCHReadPartitionSectors(&ioreq3);
			if(rc != DOCH_OK)
			{
				DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): DOCHReadPartitionSectors() failed with status: ");
				DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

				return rc;
			}

			if(tffscmp(verifyBuf, verBufOrig, DOCH_SECTOR_SIZE))
			{
				DBG_PRINT_ERR(FLZONE_API, "\r\nDOCHWritePartitionSectors(): Sector ");
				DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("#%d "), i));
				DBG_PRINT_ERR(FLZONE_API, "Verify failed ! \r\n");

				return DOCH_WriteFault;
			}

			verBufOrig += DOCH_SECTOR_SIZE;
		}
	}

	/*Return actual number of written sectors*/
	/*---------------------------------------*/
	ioreq2.irSectorCount = sectors_written_so_far;

	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHWriteIPL(): DOCHSetCustomParameter(Boot Mode) failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

	if((pdev->bAtaDevNum == 1) || (pdev->wNumOfDevices == 1))
		gIPLChunkOffsetInSectors += ioreq->irLength;

	return rc;
}

DOCH_Error DOCHWriteIPL(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket* pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	pdev->bAtaDevNum = 0;
	rc = DOCHWriteIPLSingleFloor(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHWriteIPLSingleFloor(ioreq);
		pdev->bAtaDevNum = 0;
	}

	return rc;
}

/*----------------------------------------------------------------------*/
/*	           f l D O C H R e a d I P L								*/
/*                                                                      */
/* Read specified sectors from IPL partition.							*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*		irCount			: First sector to read	                        */
/*		irLength		: Number of sectors to read						*/
/*		irData			: Pointer to user buffer (length defined by		*/
/*						  irLength)										*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/*			DOCH_VSCMD_WRITE_PARTITION									*/
/* ATA sub-command:														*/
/*			DOCH_SET_CUSTOM_PARAM										*/
/*																		*/
/* Returns:                                                             */
/*		irFlags		: DOCH_IPL_MODE_NORMAL_RAM							*/
/*					  DOCH_IPL_MODE_PAGED_RAM							*/
/*					  DOCH_IPL_MODE_VIRTUAL_RAM							*/
/*					  DOCH_IPL_MODE_CS_DELAY							*/
/*					  DOCH_IPL_MODE_ADDRESS_SHIFT_IN_AFFECT				*/
/*					  DOCH_IPL_MODE_ACTIVE_SWAP_BYTES					*/
/*					  DOCH_IPL_MODE_8KB_WINDOW							*/
/*      DOCH_Error  : 0 on success, otherwise failed					*/
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReadIPL(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;

	tffsset(&ioreq2, 0, sizeof(ioreq2));
	DOCH_SET_SOCKET_TO_IOREQ_HANDLE(&ioreq2, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	ioreq2.irData				= ioreq->irData;
	ioreq2.irSectorNo			= ioreq->irCount;
	ioreq2.irSectorCount		= ioreq->irLength;

	/*1. Retrieve IPL data */
	rc = DOCHReadPartitionSectors(&ioreq2);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReadIPL(): DOCHReadPartitionSectors() failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

	/*2. Retrieve IPL Mode */
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	ioreq2.irCount = DOCH_CP_BOOT_MODE;
	ioreq2.irFlags = DOCH_CUSTOM_PARAM_DEFAULT;
	rc = DOCHGetCustomParameter(&ioreq2);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReadIPL(): DOCHGetCustomParameter() failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

	ioreq->irFlags = ioreq2.irLength;

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*		  D O C H R e a d P a r t i t i o n	S e c t o r s				*/
/*                                                                      */
/* Reads sectors by sector no from a specific partition                 */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)		        */
/*        irData          : Address of user buffer to read into         */
/*        irSectorNo      : First sector no. to read.                   */
/*        irSectorCount   : Number of consecutive sectors to read       */
/*        irFlags	      : DOCH_USE_DMA								*/
/*							DOCH_USE_BURST								*/
/*							Note: 4 methods of data transfer are		*/
/*							available:									*/
/*								Normal transfer							*/
/*								DMA    transfer							*/
/*								Burst  transfer							*/
/*								DMA & Burst transfer					*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_READ_PARTITION									*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors NOT read (in case of an	*/
/*							error - this number may not be accurate)	*/
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReadPartitionSectors(IOreq* ioreq)
{
	DOCH_Error          rc = DOCH_OK;
	IOreq               ioreq2;
	DOCH_Socket         *pdev;
	FLByte              partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	FLDword             sectorsToReadFromFirstChunk;
	FLDword             sectorsToReadFromSecondChunk;
	FLByte              *buf;
	DOCH_Registers      in_regs;
	DOCH_Registers      out_regs;
	Addressing_Values_s addr_vals;
	FLByte		        deviceNum = 0;
	FLNative            deviceNo  = 0;
	FLBoolean	        enable    = FALSE;
	FLSDword            total_num_of_sectors;
	FLSDword            sectorNumber;
	FLSNative           sectors_read_so_far;
	FLByte              *buf_ptr;
	FLSNative           buf_offset;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	if(pdev == NULL)
		return DOCH_AdapterNotFound;

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		/*Sector aritmethic*/
		/*-----------------*/
		total_num_of_sectors = ioreq->irSectorCount;
		sectorNumber = ioreq->irSectorNo;
		sectors_read_so_far = 0;

		/*Output buffer*/
		/*-------------*/
		buf_ptr = (FLByte*)ioreq->irData;
		buf_offset = 0;

		deviceNum = pdev->bAtaDevNum;

		/*Check if Partition exceeds number of existing partitions*/
		/*--------------------------------------------------------*/
		if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
			return DOCH_PartitionNotFound;

		/*If a negative number of sectors was requested - return error*/
		/*------------------------------------------------------------*/
		if(total_num_of_sectors < 0)
			return DOCH_BadParameter;

		/*Set DMA and Burst modes according to request*/
		/*--------------------------------------------*/
		pdev->bUseDMA	= (((ioreq->irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
						   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
						   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

		pdev->bUseBurst = ((ioreq->irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

		/*Enable interrupts if DMA is enabled*/
		/*-----------------------------------*/
		if(pdev->bUseDMA)
		{
			pdev->bUseInterrupt = TRUE;

			enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

			for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
			{
				if(enable && (gAccessLayerType != DOCH_AL_SIM))
				{
					pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
				}
				else
				{
					pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
				}
				DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
			}
		}

		/*Read sectors in DOCH_MAX_SECTORS quantities*/
		/*-------------------------------------------*/
		while(total_num_of_sectors > 0)
		{
			FLSNative  sectors_this_cycle;

			if(total_num_of_sectors > DOCH_MAX_SECTORS)
				sectors_this_cycle = DOCH_MAX_SECTORS;
			else
				sectors_this_cycle = total_num_of_sectors;

	#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, TRUE);
			if(rc != DOCH_OK)
				return rc;
	#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

			/*Perform 1 operation cycle*/
			/*=========================*/

			/*Set ATA addressing registers*/
			/*---------------------------*/
			DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

			/*Update ATA register values*/
			/*--------------------------*/
			in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
			if(sectors_this_cycle < DOCH_MAX_SECTORS)
				in_regs.bSectorCount = sectors_this_cycle;
			else
				in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

			in_regs.bSectorNumber = addr_vals.bSecNum;
			in_regs.bCylLow = addr_vals.bCylLow;
			in_regs.bCylHigh = addr_vals.bCylHi;
			in_regs.bDriveHead = addr_vals.bDevHead;
			in_regs.bCommandStatus = DOCH_VSCMD_READ_PARTITION;

			/*Activate ATA command*/
			/*--------------------*/
			#ifdef DOCH_DMA_CONFIG
			DOCH_OPEN_DMA_CHANNEL
			#endif /*DOCH_DMA_CONFIG*/

			/*Perform ATA command*/
			rc = io_input (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

			if(rc != DOCH_OK)
			{
				if(rc == DOCH_TimedOut)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): io_input returned DOCH_TimedOut! \r\n ");

					/*Disable interrupts*/
					/*------------------*/
					if(pdev->bUseDMA)
					{
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
					}

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					#ifdef CHECK_POWER_ON_EVERY_COMMAND
					/* Check if device was reset */
					rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
					if(rc != DOCH_OK)
						return rc;
					#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

					return rc;
				}

				rc = get_out_registers(pdev, deviceNum, &out_regs);

				if(rc != DOCH_OK)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): failed with status: ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

					/*Disable interrupts*/
					/*------------------*/
					if(pdev->bUseDMA)
					{
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
					}

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
						return DOCH_ProtectionFault;
					else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
						return DOCH_PartitionLimitExceeded;
					else
						return DOCH_GeneralFailure;
				}
			}

			/*Update feedback values*/
			/*----------------------*/
			sectors_read_so_far += sectors_this_cycle;

			/*For next possible cycle*/
			/*-----------------------*/
			sectorNumber += sectors_this_cycle;
			total_num_of_sectors -= sectors_this_cycle;
			buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

			#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
			if(rc != DOCH_OK)
				return rc;
			#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
		}

 		/*Disable interrupts*/
		/*------------------*/
		if(pdev->bUseDMA)
		{
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
		}

		/*Cancel DMA and Burst modes*/
		/*--------------------------*/
		pdev->bUseDMA	= FALSE;
		pdev->bUseBurst = FALSE;


		/*Return actual number of read sectors*/
		/*------------------------------------*/
		ioreq->irSectorCount = sectors_read_so_far;

		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;

		/*Sector aritmethic*/
		/*-----------------*/
		total_num_of_sectors = ioreq2.irSectorCount;
		sectorNumber = ioreq2.irSectorNo;
		sectors_read_so_far = 0;

		/*Output buffer*/
		/*-------------*/
		buf_ptr = (FLByte*)ioreq2.irData;
		buf_offset = 0;

		deviceNum = pdev->bAtaDevNum;

		/*Check if Partition exceeds number of existing partitions*/
		/*--------------------------------------------------------*/
		if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		{
			pdev->bAtaDevNum = 0;
			return DOCH_PartitionNotFound;
		}

		/*If a negative number of sectors was requested - return error*/
		/*------------------------------------------------------------*/
		if(total_num_of_sectors < 0)
		{
			pdev->bAtaDevNum = 0;
			return DOCH_BadParameter;
		}

		/*Set DMA and Burst modes according to request*/
		/*--------------------------------------------*/
		pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
						   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
						   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

		pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

		/*Enable interrupts if DMA is enabled*/
		/*-----------------------------------*/
		if(pdev->bUseDMA)
		{
			pdev->bUseInterrupt = TRUE;

			enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

			for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
			{
				if(enable && (gAccessLayerType != DOCH_AL_SIM))
				{
					pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
				}
				else
				{
					pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
				}
				DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
			}
		}

		/*Read sectors in DOCH_MAX_SECTORS quantities*/
		/*-------------------------------------------*/
		while(total_num_of_sectors > 0)
		{
			FLSNative  sectors_this_cycle;

			if(total_num_of_sectors > DOCH_MAX_SECTORS)
				sectors_this_cycle = DOCH_MAX_SECTORS;
			else
				sectors_this_cycle = total_num_of_sectors;

	#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
			if(rc != DOCH_OK)
			{
				pdev->bAtaDevNum = 0;
				return rc;
			}
	#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

			/*Perform 1 operation cycle*/
			/*=========================*/

			/*Set ATA addressing registers*/
			/*---------------------------*/
			DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

			/*Update ATA register values*/
			/*--------------------------*/
			in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
			if(sectors_this_cycle < DOCH_MAX_SECTORS)
				in_regs.bSectorCount = sectors_this_cycle;
			else
				in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

			in_regs.bSectorNumber = addr_vals.bSecNum;
			in_regs.bCylLow = addr_vals.bCylLow;
			in_regs.bCylHigh = addr_vals.bCylHi;
			in_regs.bDriveHead = addr_vals.bDevHead;
			in_regs.bCommandStatus = DOCH_VSCMD_READ_PARTITION;

			/*Activate ATA command*/
			/*--------------------*/
			#ifdef DOCH_DMA_CONFIG
			DOCH_OPEN_DMA_CHANNEL
			#endif /*DOCH_DMA_CONFIG*/

			/*Perform ATA command*/
			rc = io_input (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

			if(rc != DOCH_OK)
			{
				if(rc == DOCH_TimedOut)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): io_input returned DOCH_TimedOut! \r\n ");

					/*Disable interrupts*/
					/*------------------*/
					if(pdev->bUseDMA)
					{
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
					}

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					#ifdef CHECK_POWER_ON_EVERY_COMMAND
					/* Check if device was reset */
					rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
					if(rc != DOCH_OK)
						return rc;
					#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

					return rc;
				}

				rc = get_out_registers(pdev, deviceNum, &out_regs);

				if(rc != DOCH_OK)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): failed with status: ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

					/*Disable interrupts*/
					/*------------------*/
					if(pdev->bUseDMA)
					{
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);
					}

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					pdev->bAtaDevNum = 0;

					if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
						return DOCH_ProtectionFault;
					else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
						return DOCH_PartitionLimitExceeded;
					else
						return DOCH_GeneralFailure;
				}
			}

			/*Update feedback values*/
			/*----------------------*/
			sectors_read_so_far += sectors_this_cycle;

			/*For next possible cycle*/
			/*-----------------------*/
			sectorNumber += sectors_this_cycle;
			total_num_of_sectors -= sectors_this_cycle;
			buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

			#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
			if(rc != DOCH_OK)
			{
				pdev->bAtaDevNum = 0;
				return rc;
			}
			#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
		}

 		/*Disable interrupts*/
		/*------------------*/
		if(pdev->bUseDMA)
		{
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);
		}

		/*Cancel DMA and Burst modes*/
		/*--------------------------*/
		pdev->bUseDMA	= FALSE;
		pdev->bUseBurst = FALSE;


		/*Return actual number of read sectors*/
		/*------------------------------------*/
		ioreq2.irSectorCount = sectors_read_so_far;

		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*irLength	=> First sector to read*/
		/*irCount	=> Number of sectors to read*/
		if( ((FLDword)((ioreq->irLength + ioreq->irCount))) > pdev->sSpanData.dwSpanSector)
		{
			if(((FLDword)ioreq->irLength) >= pdev->sSpanData.dwSpanSector)
			{
				sectorsToReadFromFirstChunk  = 0;
				sectorsToReadFromSecondChunk = ioreq->irCount;

			}
			else
			{
				sectorsToReadFromFirstChunk  = (pdev->sSpanData.dwSpanSector - ioreq->irLength);
				sectorsToReadFromSecondChunk = (ioreq->irCount - sectorsToReadFromFirstChunk);
			}
		}
		else
		{
			sectorsToReadFromFirstChunk  = ioreq->irCount;
			sectorsToReadFromSecondChunk = 0;
		}

		/*Read first chunk from Dev0*/
		/*--------------------------*/
		if(sectorsToReadFromFirstChunk > 0)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
			ioreq2.irCount = sectorsToReadFromFirstChunk;

			/*Sector aritmethic*/
			/*-----------------*/
			total_num_of_sectors = ioreq2.irSectorCount;
			sectorNumber = ioreq2.irSectorNo;
			sectors_read_so_far = 0;

			/*Output buffer*/
			/*-------------*/
			buf_ptr = (FLByte*)ioreq2.irData;
			buf_offset = 0;

			deviceNum = pdev->bAtaDevNum;

			/*Check if Partition exceeds number of existing partitions*/
			/*--------------------------------------------------------*/
			if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
			{
				pdev->bAtaDevNum = 0;
				return DOCH_PartitionNotFound;
			}

			/*If a negative number of sectors was requested - return error*/
			/*------------------------------------------------------------*/
			if(total_num_of_sectors < 0)
			{
				pdev->bAtaDevNum = 0;
				return DOCH_BadParameter;
			}

			/*Set DMA and Burst modes according to request*/
			/*--------------------------------------------*/
			pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
							   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
							   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

			pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

			/*Enable interrupts if DMA is enabled*/
			/*-----------------------------------*/
			if(pdev->bUseDMA)
			{
				pdev->bUseInterrupt = TRUE;

				enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

				for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
				{
					if(enable && (gAccessLayerType != DOCH_AL_SIM))
					{
						pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
					}
					else
					{
						pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
					}
					DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
				}
			}

			/*Read sectors in DOCH_MAX_SECTORS quantities*/
			/*-------------------------------------------*/
			while(total_num_of_sectors > 0)
			{
				FLSNative  sectors_this_cycle;

				if(total_num_of_sectors > DOCH_MAX_SECTORS)
					sectors_this_cycle = DOCH_MAX_SECTORS;
				else
					sectors_this_cycle = total_num_of_sectors;

		#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
		#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

				/*Perform 1 operation cycle*/
				/*=========================*/

				/*Set ATA addressing registers*/
				/*---------------------------*/
				DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

				/*Update ATA register values*/
				/*--------------------------*/
				in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
				if(sectors_this_cycle < DOCH_MAX_SECTORS)
					in_regs.bSectorCount = sectors_this_cycle;
				else
					in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

				in_regs.bSectorNumber = addr_vals.bSecNum;
				in_regs.bCylLow = addr_vals.bCylLow;
				in_regs.bCylHigh = addr_vals.bCylHi;
				in_regs.bDriveHead = addr_vals.bDevHead;
				in_regs.bCommandStatus = DOCH_VSCMD_READ_PARTITION;

				/*Activate ATA command*/
				/*--------------------*/
				#ifdef DOCH_DMA_CONFIG
				DOCH_OPEN_DMA_CHANNEL
				#endif /*DOCH_DMA_CONFIG*/

				/*Perform ATA command*/
				rc = io_input (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

				if(rc != DOCH_OK)
				{
					if(rc == DOCH_TimedOut)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): io_input returned DOCH_TimedOut! \r\n ");

						/*Disable interrupts*/
						/*------------------*/
						if(pdev->bUseDMA)
						{
							pdev->bUseInterrupt = FALSE;
							dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
						}

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						#ifdef CHECK_POWER_ON_EVERY_COMMAND
						/* Check if device was reset */
						rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
						if(rc != DOCH_OK)
							return rc;
						#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

						return rc;
					}

					rc = get_out_registers(pdev, deviceNum, &out_regs);

					if(rc != DOCH_OK)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): failed with status: ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

						/*Disable interrupts*/
						/*------------------*/
						if(pdev->bUseDMA)
						{
							pdev->bUseInterrupt = FALSE;
							dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);
						}

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						pdev->bAtaDevNum = 0;

						if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
							return DOCH_ProtectionFault;
						else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
							return DOCH_PartitionLimitExceeded;
						else
							return DOCH_GeneralFailure;
					}
				}

				/*Update feedback values*/
				/*----------------------*/
				sectors_read_so_far += sectors_this_cycle;

				/*For next possible cycle*/
				/*-----------------------*/
				sectorNumber += sectors_this_cycle;
				total_num_of_sectors -= sectors_this_cycle;
				buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

				#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
				#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
			}

 			/*Disable interrupts*/
			/*------------------*/
			if(pdev->bUseDMA)
			{
				pdev->bUseInterrupt = FALSE;
				dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);
			}

			/*Cancel DMA and Burst modes*/
			/*--------------------------*/
			pdev->bUseDMA	= FALSE;
			pdev->bUseBurst = FALSE;


			/*Return actual number of read sectors*/
			/*------------------------------------*/
			ioreq2.irSectorCount = sectors_read_so_far;

			if(rc != DOCH_OK)
				return rc;
		}

		/*Read 2nd chunk from Dev1*/
		/*------------------------*/
		if(sectorsToReadFromSecondChunk > 0)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
			ioreq2.irCount  = sectorsToReadFromSecondChunk;

			if(sectorsToReadFromFirstChunk > 0)
				ioreq2.irLength = 0;
			else
			{
				ioreq2.irLength = (ioreq->irLength - pdev->sSpanData.dwSpanSector);
			}

			buf = (FLByte*)ioreq->irData;
			ioreq2.irData = buf + (sectorsToReadFromFirstChunk<<DOCH_SECTOR_SIZE_BITS);
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

			pdev->bAtaDevNum = 1;

			/*Sector aritmethic*/
			/*-----------------*/
			total_num_of_sectors = ioreq2.irSectorCount;
			sectorNumber = ioreq2.irSectorNo;
			sectors_read_so_far = 0;

			/*Output buffer*/
			/*-------------*/
			buf_ptr = (FLByte*)ioreq2.irData;
			buf_offset = 0;

			deviceNum = pdev->bAtaDevNum;

			/*Check if Partition exceeds number of existing partitions*/
			/*--------------------------------------------------------*/
			if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
			{
				pdev->bAtaDevNum = 0;
				return DOCH_PartitionNotFound;
			}

			/*If a negative number of sectors was requested - return error*/
			/*------------------------------------------------------------*/
			if(total_num_of_sectors < 0)
			{
				pdev->bAtaDevNum = 0;
				return DOCH_BadParameter;
			}

			/*Set DMA and Burst modes according to request*/
			/*--------------------------------------------*/
			pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
							   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
							   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

			pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

			/*Enable interrupts if DMA is enabled*/
			/*-----------------------------------*/
			if(pdev->bUseDMA)
			{
				pdev->bUseInterrupt = TRUE;

				enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

				for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
				{
					if(enable && (gAccessLayerType != DOCH_AL_SIM))
					{
						pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
					}
					else
					{
						pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
					}
                    DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
				}
			}

			/*Read sectors in DOCH_MAX_SECTORS quantities*/
			/*-------------------------------------------*/
			while(total_num_of_sectors > 0)
			{
				FLSNative  sectors_this_cycle;

				if(total_num_of_sectors > DOCH_MAX_SECTORS)
					sectors_this_cycle = DOCH_MAX_SECTORS;
				else
					sectors_this_cycle = total_num_of_sectors;

		#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
		#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

				/*Perform 1 operation cycle*/
				/*=========================*/

				/*Set ATA addressing registers*/
				/*---------------------------*/
				DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

				/*Update ATA register values*/
				/*--------------------------*/
				in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
				if(sectors_this_cycle < DOCH_MAX_SECTORS)
					in_regs.bSectorCount = sectors_this_cycle;
				else
					in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

				in_regs.bSectorNumber = addr_vals.bSecNum;
				in_regs.bCylLow = addr_vals.bCylLow;
				in_regs.bCylHigh = addr_vals.bCylHi;
				in_regs.bDriveHead = addr_vals.bDevHead;
				in_regs.bCommandStatus = DOCH_VSCMD_READ_PARTITION;

				/*Activate ATA command*/
				/*--------------------*/
				#ifdef DOCH_DMA_CONFIG
				DOCH_OPEN_DMA_CHANNEL
				#endif /*DOCH_DMA_CONFIG*/

				/*Perform ATA command*/
				rc = io_input (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

				if(rc != DOCH_OK)
				{
					if(rc == DOCH_TimedOut)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): io_input returned DOCH_TimedOut! \r\n ");

						/*Disable interrupts*/
						/*------------------*/
						if(pdev->bUseDMA)
						{
							pdev->bUseInterrupt = FALSE;
							dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
						}

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						#ifdef CHECK_POWER_ON_EVERY_COMMAND
						/* Check if device was reset */
						rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
						if(rc != DOCH_OK)
							return rc;
						#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

						return rc;
					}

					rc = get_out_registers(pdev, deviceNum, &out_regs);

					if(rc != DOCH_OK)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHReadPartitionSectors(): failed with status: ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

						/*Disable interrupts*/
						/*------------------*/
						if(pdev->bUseDMA)
						{
							pdev->bUseInterrupt = FALSE;
							dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);
						}

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						pdev->bAtaDevNum = 0;

						if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
							return DOCH_ProtectionFault;
						else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
							return DOCH_PartitionLimitExceeded;
						else
							return DOCH_GeneralFailure;
					}
				}

				/*Update feedback values*/
				/*----------------------*/
				sectors_read_so_far += sectors_this_cycle;

				/*For next possible cycle*/
				/*-----------------------*/
				sectorNumber += sectors_this_cycle;
				total_num_of_sectors -= sectors_this_cycle;
				buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

				#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
				#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
			}

 			/*Disable interrupts*/
			/*------------------*/
			if(pdev->bUseDMA)
			{
				pdev->bUseInterrupt = FALSE;
				dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);
			}

			/*Cancel DMA and Burst modes*/
			/*--------------------------*/
			pdev->bUseDMA	= FALSE;
			pdev->bUseBurst = FALSE;


			/*Return actual number of read sectors*/
			/*------------------------------------*/
			ioreq2.irSectorCount = sectors_read_so_far;

			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;
		}
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*        D O C H W r i t e P a r t i t i o n S e c t o r s				*/
/*                                                                      */
/* Writes sectors by sector no to a specific partition                  */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)		        */
/*        irData          : Address of user buffer to write from        */
/*        irSectorNo      : First sector no. to write                   */
/*        irSectorCount   : Number of consecutive sectors to write		*/
/*        irFlags	      : DOCH_USE_DMA								*/
/*							DOCH_USE_BURST								*/
/*							Note: 4 methods of data transfer are		*/
/*							available:									*/
/*								Normal transfer							*/
/*								DMA    transfer							*/
/*								Burst  transfer							*/
/*								DMA & Burst transfer					*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_WRITE_PARTITION									*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors NOT read (in case of an	*/
/*							error - this number may not be accurate)	*/
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWritePartitionSectors(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	IOreq ioreq3;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	FLDword sectorsToWriteToFirstChunk;
	FLDword sectorsToWriteToSecondChunk;
	FLByte* buf;

	FLByte		deviceNum = 0;
	FLNative    deviceNo = 0;
	FLBoolean	enable = FALSE;
	Addressing_Values_s addr_vals;
	register int i;

	DOCH_Registers in_regs;
	DOCH_Registers out_regs;

	FLSDword	total_num_of_sectors;
	FLSDword	sectorNumber;
	FLSNative	sectors_written_so_far;
	FLByte*		buf_ptr;
	FLSNative	buf_offset;

	FLByte operationRetries = 0;
	FLByte* verBufOrig;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	if(pdev == NULL)
		return DOCH_AdapterNotFound;

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		/*Sector arithmetic*/
		/*-----------------*/
		total_num_of_sectors	= ioreq->irSectorCount;
		sectorNumber			= ioreq->irSectorNo;
		sectors_written_so_far	= 0;

		/*Buffer to write from*/
		/*--------------------*/
		buf_ptr = (FLByte*)ioreq->irData;
		buf_offset = 0;

		deviceNum = pdev->bAtaDevNum;

		/*Sanity Check*/
		/*------------*/
		if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
			return DOCH_PartitionNotFound;

		/*Enable interrupts*/
		/*-----------------*/
		pdev->bUseInterrupt = TRUE;

		enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

		for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
		{
			if(enable && (gAccessLayerType != DOCH_AL_SIM))
			{
				pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
			}
			else
			{
				pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
			}
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
		}

		/*Set DMA and Burst modes according to request*/
		/*--------------------------------------------*/
		pdev->bUseDMA	= (((ioreq->irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
						   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
						   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

		pdev->bUseBurst = ((ioreq->irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

		/*Perform write cycles in DOCH_MAX_SECTORS quantities*/
		/*---------------------------------------------------*/
		while(total_num_of_sectors > 0)
		{
			FLSNative  sectors_this_cycle;

			if(total_num_of_sectors > DOCH_MAX_SECTORS)
				sectors_this_cycle = DOCH_MAX_SECTORS;
			else
				sectors_this_cycle = total_num_of_sectors;

	dochWriteRetryPointDev0Only:

	#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, TRUE);
			if(rc != DOCH_OK)
				return rc;
	#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

			/*Perform 1 operation cycle*/
			/*=========================*/

			/*Set ATA addressing registers*/
			/*---------------------------*/
			DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

			/*Update ATA register values*/
			/*--------------------------*/
			in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
			if(sectors_this_cycle < DOCH_MAX_SECTORS)
				in_regs.bSectorCount = sectors_this_cycle;
			else
				in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

			in_regs.bSectorNumber = addr_vals.bSecNum;
			in_regs.bCylLow = addr_vals.bCylLow;
			in_regs.bCylHigh = addr_vals.bCylHi;
			in_regs.bDriveHead = addr_vals.bDevHead;
			in_regs.bCommandStatus = DOCH_VSCMD_WRITE_PARTITION;

			/*Activate ATA command*/
			/*--------------------*/
			#ifdef DOCH_DMA_CONFIG
			DOCH_OPEN_DMA_CHANNEL
			#endif /*DOCH_DMA_CONFIG*/

			/*Perform ATA command*/
			rc = io_output (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

			if(rc != DOCH_OK)
			{
				if(rc == DOCH_TimedOut)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): io_output returned DOCH_TimedOut! \r\n ");

					/*Disable interrupts*/
					/*------------------*/
					if(pdev->bUseDMA)
					{
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
					}

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					#ifdef CHECK_POWER_ON_EVERY_COMMAND
					/* Check if device was reset */
					rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
					if(rc != DOCH_OK)
						return rc;
					#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

					return rc;
				}

				rc = get_out_registers(pdev, deviceNum, &out_regs);

				if(rc != DOCH_OK)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): failed with status: ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

					if( ((rc & DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) == DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) &&
						(operationRetries++ < 4)
						)
						goto dochWriteRetryPointDev0Only;

					/*Disable interrupts*/
					/*------------------*/
					pdev->bUseInterrupt = FALSE;
					dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					pdev->bAtaDevNum = 0;

					if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
						return DOCH_ProtectionFault;
					else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
						return DOCH_PartitionLimitExceeded;
					else
						return DOCH_GeneralFailure;
				}
			}

			/*Update feedback values*/
			/*----------------------*/
			sectors_written_so_far += sectors_this_cycle;

			/*For next possible cycle*/
			/*-----------------------*/
			sectorNumber += sectors_this_cycle;
			total_num_of_sectors -= sectors_this_cycle;
			buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

			#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
			if(rc != DOCH_OK)
				return rc;
			#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
		}

		/*Disable interrupts*/
		/*------------------*/
		pdev->bUseInterrupt = FALSE;
		dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

		/*Cancel DMA and Burst modes*/
		/*--------------------------*/
		pdev->bUseDMA	= FALSE;
		pdev->bUseBurst = FALSE;


		/*If verify was define, perform verify operation with same parameters*/
		/*-------------------------------------------------------------------*/
		if(gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)] == DOCH_GLOBAL_BOOL_PATTERN)
		{
			tffsset(&ioreq2, 0, sizeof(ioreq2));
			tffscpy(&ioreq2.irHandle, &ioreq->irHandle, sizeof(ioreq2.irHandle));
			ioreq2.irData = verifyBuf;
			verBufOrig = (FLByte*)(ioreq->irData);

			for(i=0; i<ioreq->irSectorCount; i++)
			{
				ioreq2.irSectorCount = 1;
				ioreq2.irSectorNo	 = (ioreq->irSectorNo + i);
				rc = DOCHReadPartitionSectors(&ioreq2);
				if(rc != DOCH_OK)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): DOCHReadPartitionSectors() failed with status: ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

					return rc;
				}

				if(tffscmp(verifyBuf, verBufOrig, DOCH_SECTOR_SIZE))
				{
					DBG_PRINT_ERR(FLZONE_API, "\r\nDOCHWritePartitionSectors(): Sector ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("#%d "), i));
					DBG_PRINT_ERR(FLZONE_API, "Verify failed ! \r\n");

					return DOCH_WriteFault;
				}

				verBufOrig += DOCH_SECTOR_SIZE;
			}
		}

		/*Return actual number of written sectors*/
		/*---------------------------------------*/
		ioreq->irSectorCount = sectors_written_so_far;

		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;

		/*Sector arithmetic*/
		/*-----------------*/
		total_num_of_sectors	= ioreq2.irSectorCount;
		sectorNumber			= ioreq2.irSectorNo;
		sectors_written_so_far	= 0;

		/*Buffer to write from*/
		/*--------------------*/
		buf_ptr = (FLByte*)ioreq2.irData;
		buf_offset = 0;

		deviceNum = pdev->bAtaDevNum;

		/*Sanity Check*/
		/*------------*/
		if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		{
			pdev->bAtaDevNum = 0;
			return DOCH_PartitionNotFound;
		}

		/*Enable interrupts*/
		/*-----------------*/
		pdev->bUseInterrupt = TRUE;

		enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

		for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
		{
			if(enable && (gAccessLayerType != DOCH_AL_SIM))
			{
				pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
			}
			else
			{
				pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
			}
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
		}

		/*Set DMA and Burst modes according to request*/
		/*--------------------------------------------*/
		pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
						   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
						   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

		pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

		/*Perform write cycles in DOCH_MAX_SECTORS quantities*/
		/*---------------------------------------------------*/
		while(total_num_of_sectors > 0)
		{
			FLSNative  sectors_this_cycle;

			if(total_num_of_sectors > DOCH_MAX_SECTORS)
				sectors_this_cycle = DOCH_MAX_SECTORS;
			else
				sectors_this_cycle = total_num_of_sectors;

	dochWriteRetryPointDev1Only:

	#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
			if(rc != DOCH_OK)
			{
				pdev->bAtaDevNum = 0;
				return rc;
			}
	#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

			/*Perform 1 operation cycle*/
			/*=========================*/

			/*Set ATA addressing registers*/
			/*---------------------------*/
			DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

			/*Update ATA register values*/
			/*--------------------------*/
			in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
			if(sectors_this_cycle < DOCH_MAX_SECTORS)
				in_regs.bSectorCount = sectors_this_cycle;
			else
				in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

			in_regs.bSectorNumber = addr_vals.bSecNum;
			in_regs.bCylLow = addr_vals.bCylLow;
			in_regs.bCylHigh = addr_vals.bCylHi;
			in_regs.bDriveHead = addr_vals.bDevHead;
			in_regs.bCommandStatus = DOCH_VSCMD_WRITE_PARTITION;

			/*Activate ATA command*/
			/*--------------------*/
			#ifdef DOCH_DMA_CONFIG
			DOCH_OPEN_DMA_CHANNEL
			#endif /*DOCH_DMA_CONFIG*/

			/*Perform ATA command*/
			rc = io_output (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

			if(rc != DOCH_OK)
			{
				if(rc == DOCH_TimedOut)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): io_output returned DOCH_TimedOut! \r\n ");

					/*Disable interrupts*/
					/*------------------*/
					if(pdev->bUseDMA)
					{
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
					}

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					#ifdef CHECK_POWER_ON_EVERY_COMMAND
					/* Check if device was reset */
					rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
					if(rc != DOCH_OK)
						return rc;
					#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

					return rc;
				}

				rc = get_out_registers(pdev, deviceNum, &out_regs);

				if(rc != DOCH_OK)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): failed with status: ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

					if( ((rc & DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) == DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) &&
						(operationRetries++ < 4)
						)
						goto dochWriteRetryPointDev1Only;

					/*Disable interrupts*/
					/*------------------*/
					pdev->bUseInterrupt = FALSE;
					dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

					/*Cancel DMA and Burst modes*/
					/*--------------------------*/
					pdev->bUseDMA	= FALSE;
					pdev->bUseBurst = FALSE;

					pdev->bAtaDevNum = 0;

					if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
						return DOCH_ProtectionFault;
					else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
						return DOCH_PartitionLimitExceeded;
					else
						return DOCH_GeneralFailure;
				}
			}

			/*Update feedback values*/
			/*----------------------*/
			sectors_written_so_far += sectors_this_cycle;

			/*For next possible cycle*/
			/*-----------------------*/
			sectorNumber += sectors_this_cycle;
			total_num_of_sectors -= sectors_this_cycle;
			buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

			#ifdef CHECK_POWER_ON_EVERY_COMMAND
			/* Check if device was reset */
			rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
			if(rc != DOCH_OK)
			{
				pdev->bAtaDevNum = 0;
				return rc;
			}
			#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
		}

		/*Disable interrupts*/
		/*------------------*/
		pdev->bUseInterrupt = FALSE;
		dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

		/*Cancel DMA and Burst modes*/
		/*--------------------------*/
		pdev->bUseDMA	= FALSE;
		pdev->bUseBurst = FALSE;


		/*If verify was define, perform verify operation with same parameters*/
		/*-------------------------------------------------------------------*/
		if(gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2)] == DOCH_GLOBAL_BOOL_PATTERN)
		{
			tffsset(&ioreq3, 0, sizeof(ioreq3));
			tffscpy(&ioreq3.irHandle, &ioreq2.irHandle, sizeof(ioreq3.irHandle));
			ioreq3.irData = verifyBuf;
			verBufOrig = (FLByte*)(ioreq2.irData);

			for(i=0; i<ioreq2.irSectorCount; i++)
			{
				ioreq3.irSectorCount = 1;
				ioreq3.irSectorNo	 = (ioreq2.irSectorNo + i);
				rc = DOCHReadPartitionSectors(&ioreq3);
				if(rc != DOCH_OK)
				{
					DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): DOCHReadPartitionSectors() failed with status: ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

					pdev->bAtaDevNum = 0;

					return rc;
				}

				if(tffscmp(verifyBuf, verBufOrig, DOCH_SECTOR_SIZE))
				{
					DBG_PRINT_ERR(FLZONE_API, "\r\nDOCHWritePartitionSectors(): Sector ");
					DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("#%d "), i));
					DBG_PRINT_ERR(FLZONE_API, "Verify failed ! \r\n");

					pdev->bAtaDevNum = 0;

					return DOCH_WriteFault;
				}

				verBufOrig += DOCH_SECTOR_SIZE;
			}
		}

		/*Return actual number of written sectors*/
		/*---------------------------------------*/
		ioreq2.irSectorCount = sectors_written_so_far;

		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*irLength	=> First sector to read*/
		/*irCount	=> Number of sectors to read*/
		if( ((FLDword)((ioreq->irLength + ioreq->irCount))) > pdev->sSpanData.dwSpanSector)
		{
			if(((FLDword)ioreq->irLength) >= pdev->sSpanData.dwSpanSector)
			{
				sectorsToWriteToFirstChunk = 0;
				sectorsToWriteToSecondChunk = ioreq->irCount;

			}
			else
			{
				sectorsToWriteToFirstChunk  = (pdev->sSpanData.dwSpanSector - ioreq->irLength);
				sectorsToWriteToSecondChunk = (ioreq->irCount - sectorsToWriteToFirstChunk);
			}
		}
		else
		{
			sectorsToWriteToFirstChunk  = ioreq->irCount;
			sectorsToWriteToSecondChunk = 0;
		}

		/*Write first chunk to Dev0*/
		/*-------------------------*/
		if(sectorsToWriteToFirstChunk > 0)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
			ioreq2.irCount = sectorsToWriteToFirstChunk;

			/*Sector arithmetic*/
			/*-----------------*/
			total_num_of_sectors	= ioreq2.irSectorCount;
			sectorNumber			= ioreq2.irSectorNo;
			sectors_written_so_far	= 0;

			/*Buffer to write from*/
			/*--------------------*/
			buf_ptr = (FLByte*)ioreq2.irData;
			buf_offset = 0;

			deviceNum = pdev->bAtaDevNum;

			/*Sanity Check*/
			/*------------*/
			if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
			{
				pdev->bAtaDevNum = 0;
				return DOCH_PartitionNotFound;
			}

			/*Enable interrupts*/
			/*-----------------*/
			pdev->bUseInterrupt = TRUE;

			enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

			for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
			{
				if(enable && (gAccessLayerType != DOCH_AL_SIM))
				{
					pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
				}
				else
				{
					pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
				}
				DOCHWRITE_ATA_REG (pdev->bRegBase,  DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
			}

			/*Set DMA and Burst modes according to request*/
			/*--------------------------------------------*/
			pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
							   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
							   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

			pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

			/*Perform write cycles in DOCH_MAX_SECTORS quantities*/
			/*---------------------------------------------------*/
			while(total_num_of_sectors > 0)
			{
				FLSNative  sectors_this_cycle;

				if(total_num_of_sectors > DOCH_MAX_SECTORS)
					sectors_this_cycle = DOCH_MAX_SECTORS;
				else
					sectors_this_cycle = total_num_of_sectors;

		dochWriteRetryPointDev0Chunk:

		#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
		#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

				/*Perform 1 operation cycle*/
				/*=========================*/

				/*Set ATA addressing registers*/
				/*---------------------------*/
				DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

				/*Update ATA register values*/
				/*--------------------------*/
				in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
				if(sectors_this_cycle < DOCH_MAX_SECTORS)
					in_regs.bSectorCount = sectors_this_cycle;
				else
					in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

				in_regs.bSectorNumber = addr_vals.bSecNum;
				in_regs.bCylLow = addr_vals.bCylLow;
				in_regs.bCylHigh = addr_vals.bCylHi;
				in_regs.bDriveHead = addr_vals.bDevHead;
				in_regs.bCommandStatus = DOCH_VSCMD_WRITE_PARTITION;

				/*Activate ATA command*/
				/*--------------------*/
				#ifdef DOCH_DMA_CONFIG
				DOCH_OPEN_DMA_CHANNEL
				#endif /*DOCH_DMA_CONFIG*/

				/*Perform ATA command*/
				rc = io_output (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

				if(rc != DOCH_OK)
				{
					if(rc == DOCH_TimedOut)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): io_output returned DOCH_TimedOut! \r\n ");

						/*Disable interrupts*/
						/*------------------*/
						if(pdev->bUseDMA)
						{
							pdev->bUseInterrupt = FALSE;
							dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
						}

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						#ifdef CHECK_POWER_ON_EVERY_COMMAND
						/* Check if device was reset */
						rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
						if(rc != DOCH_OK)
							return rc;
						#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

						return rc;
					}

					rc = get_out_registers(pdev, deviceNum, &out_regs);

					if(rc != DOCH_OK)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): failed with status: ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

						if( ((rc & DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) == DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) &&
							(operationRetries++ < 4)
							)
							goto dochWriteRetryPointDev0Chunk;

						/*Disable interrupts*/
						/*------------------*/
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						pdev->bAtaDevNum = 0;

						if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
							return DOCH_ProtectionFault;
						else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
							return DOCH_PartitionLimitExceeded;
						else
							return DOCH_GeneralFailure;
					}
				}

				/*Update feedback values*/
				/*----------------------*/
				sectors_written_so_far += sectors_this_cycle;

				/*For next possible cycle*/
				/*-----------------------*/
				sectorNumber += sectors_this_cycle;
				total_num_of_sectors -= sectors_this_cycle;
				buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

				#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
				#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
			}

			/*Disable interrupts*/
			/*------------------*/
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

			/*Cancel DMA and Burst modes*/
			/*--------------------------*/
			pdev->bUseDMA	= FALSE;
			pdev->bUseBurst = FALSE;


			/*If verify was define, perform verify operation with same parameters*/
			/*-------------------------------------------------------------------*/
			if(gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2)] == DOCH_GLOBAL_BOOL_PATTERN)
			{
				tffsset(&ioreq3, 0, sizeof(ioreq3));
				tffscpy(&ioreq3.irHandle, &ioreq2.irHandle, sizeof(ioreq3.irHandle));
				ioreq3.irData = verifyBuf;
				verBufOrig = (FLByte*)(ioreq2.irData);

				for(i=0; i<ioreq2.irSectorCount; i++)
				{
					ioreq3.irSectorCount = 1;
					ioreq3.irSectorNo	 = (ioreq2.irSectorNo + i);
					rc = DOCHReadPartitionSectors(&ioreq3);
					if(rc != DOCH_OK)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): DOCHReadPartitionSectors() failed with status: ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

						pdev->bAtaDevNum = 0;

						return rc;
					}

					if(tffscmp(verifyBuf, verBufOrig, DOCH_SECTOR_SIZE))
					{
						DBG_PRINT_ERR(FLZONE_API, "\r\nDOCHWritePartitionSectors(): Sector ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("#%d "), i));
						DBG_PRINT_ERR(FLZONE_API, "Verify failed ! \r\n");

						pdev->bAtaDevNum = 0;

						return DOCH_WriteFault;
					}

					verBufOrig += DOCH_SECTOR_SIZE;
				}
			}

			/*Return actual number of written sectors*/
			/*---------------------------------------*/
			ioreq2.irSectorCount = sectors_written_so_far;

			if(rc != DOCH_OK)
			{
				pdev->bAtaDevNum = 0;
				return rc;
			}
		}

		/*Write 2nd chunk to Dev1*/
		/*-----------------------*/
		if(sectorsToWriteToSecondChunk > 0)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
			ioreq2.irCount  = sectorsToWriteToSecondChunk;

			/*Fix firs sector to read*/
			if(sectorsToWriteToFirstChunk > 0)
				ioreq2.irLength = 0;
			else
			{
				ioreq2.irLength = (ioreq->irLength - pdev->sSpanData.dwSpanSector);
			}

			buf = (FLByte*)ioreq->irData;
			ioreq2.irData = buf + (sectorsToWriteToFirstChunk<<DOCH_SECTOR_SIZE_BITS);
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

			pdev->bAtaDevNum = 1;

			/*Sector arithmetic*/
			/*-----------------*/
			total_num_of_sectors	= ioreq2.irSectorCount;
			sectorNumber			= ioreq2.irSectorNo;
			sectors_written_so_far	= 0;

			/*Buffer to write from*/
			/*--------------------*/
			buf_ptr = (FLByte*)ioreq2.irData;
			buf_offset = 0;

			deviceNum = pdev->bAtaDevNum;

			/*Sanity Check*/
			/*------------*/
			if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
			{
				pdev->bAtaDevNum = 0;
				return DOCH_PartitionNotFound;
			}

			/*Enable interrupts*/
			/*-----------------*/
			pdev->bUseInterrupt = TRUE;

			enable = DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2));

			for(deviceNo = 0 ; deviceNo < pdev->wNumOfDevices ; deviceNo++)
			{
				if(enable && (gAccessLayerType != DOCH_AL_SIM))
				{
					pdev->device[deviceNo].flags |= DOCH_FLAGSB_USE_INTERRUPT;
				}
				else
				{
					pdev->device[deviceNo].flags &= ~DOCH_FLAGSB_USE_INTERRUPT;
				}
				DOCHWRITE_ATA_REG (pdev->bRegBase,DOCH_CONTROL_REG, ((enable) ? 0 : DOCH_ATA_NIEN ));
			}

			/*Set DMA and Burst modes according to request*/
			/*--------------------------------------------*/
			pdev->bUseDMA	= (((ioreq2.irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
							   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
							   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

			pdev->bUseBurst = ((ioreq2.irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

			/*Perform write cycles in DOCH_MAX_SECTORS quantities*/
			/*---------------------------------------------------*/
			while(total_num_of_sectors > 0)
			{
				FLSNative  sectors_this_cycle;

				if(total_num_of_sectors > DOCH_MAX_SECTORS)
					sectors_this_cycle = DOCH_MAX_SECTORS;
				else
					sectors_this_cycle = total_num_of_sectors;

		dochWriteRetryPointDev1Chunk:

		#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, TRUE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
		#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

				/*Perform 1 operation cycle*/
				/*=========================*/

				/*Set ATA addressing registers*/
				/*---------------------------*/
				DOCH_SET_ATA_ADDRESS_VALUES(addr_vals, sectorNumber, deviceNum);

				/*Update ATA register values*/
				/*--------------------------*/
				in_regs.bFeaturesError = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(&ioreq2);
				if(sectors_this_cycle < DOCH_MAX_SECTORS)
					in_regs.bSectorCount = sectors_this_cycle;
				else
					in_regs.bSectorCount = 0;	/*DOCH_MAX_SECTORS*/

				in_regs.bSectorNumber = addr_vals.bSecNum;
				in_regs.bCylLow = addr_vals.bCylLow;
				in_regs.bCylHigh = addr_vals.bCylHi;
				in_regs.bDriveHead = addr_vals.bDevHead;
				in_regs.bCommandStatus = DOCH_VSCMD_WRITE_PARTITION;

				/*Activate ATA command*/
				/*--------------------*/
				#ifdef DOCH_DMA_CONFIG
				DOCH_OPEN_DMA_CHANNEL
				#endif /*DOCH_DMA_CONFIG*/

				/*Perform ATA command*/
				rc = io_output (pdev, deviceNum, &in_regs, (void*)(&buf_ptr[buf_offset]), sectors_this_cycle);

				if(rc != DOCH_OK)
				{
					if(rc == DOCH_TimedOut)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): io_output returned DOCH_TimedOut! \r\n ");

						/*Disable interrupts*/
						/*------------------*/
						if(pdev->bUseDMA)
						{
							pdev->bUseInterrupt = FALSE;
							dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);
						}

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						#ifdef CHECK_POWER_ON_EVERY_COMMAND
						/* Check if device was reset */
						rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), deviceNum, FALSE);
						if(rc != DOCH_OK)
							return rc;
						#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

						return rc;
					}

					rc = get_out_registers(pdev, deviceNum, &out_regs);

					if(rc != DOCH_OK)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): failed with status: ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

						if( ((rc & DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) == DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) &&
							(operationRetries++ < 4)
							)
							goto dochWriteRetryPointDev1Chunk;

						/*Disable interrupts*/
						/*------------------*/
						pdev->bUseInterrupt = FALSE;
						dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

						/*Cancel DMA and Burst modes*/
						/*--------------------------*/
						pdev->bUseDMA	= FALSE;
						pdev->bUseBurst = FALSE;

						pdev->bAtaDevNum = 0;

						if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
							return DOCH_ProtectionFault;
						else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
							return DOCH_PartitionLimitExceeded;
						else
							return DOCH_GeneralFailure;
					}
				}

				/*Update feedback values*/
				/*----------------------*/
				sectors_written_so_far += sectors_this_cycle;

				/*For next possible cycle*/
				/*-----------------------*/
				sectorNumber += sectors_this_cycle;
				total_num_of_sectors -= sectors_this_cycle;
				buf_offset += sectors_this_cycle * DOCH_SECTOR_SIZE;

				#ifdef CHECK_POWER_ON_EVERY_COMMAND
				/* Check if device was reset */
				rc = dochCheckPFSymptom(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), deviceNum, FALSE);
				if(rc != DOCH_OK)
				{
					pdev->bAtaDevNum = 0;
					return rc;
				}
				#endif /*CHECK_POWER_ON_EVERY_COMMAND*/
			}

			/*Disable interrupts*/
			/*------------------*/
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2), FALSE, 0);

			/*Cancel DMA and Burst modes*/
			/*--------------------------*/
			pdev->bUseDMA	= FALSE;
			pdev->bUseBurst = FALSE;


			/*If verify was define, perform verify operation with same parameters*/
			/*-------------------------------------------------------------------*/
			if(gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(&ioreq2)] == DOCH_GLOBAL_BOOL_PATTERN)
			{
				tffsset(&ioreq3, 0, sizeof(ioreq3));
				tffscpy(&ioreq3.irHandle, &ioreq2.irHandle, sizeof(ioreq3.irHandle));
				ioreq3.irData = verifyBuf;
				verBufOrig = (FLByte*)(ioreq2.irData);

				for(i=0; i<ioreq2.irSectorCount; i++)
				{
					ioreq3.irSectorCount = 1;
					ioreq3.irSectorNo	 = (ioreq2.irSectorNo + i);
					rc = DOCHReadPartitionSectors(&ioreq3);
					if(rc != DOCH_OK)
					{
						DBG_PRINT_ERR(FLZONE_API, "DOCHWritePartitionSectors(): DOCHReadPartitionSectors() failed with status: ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

						pdev->bAtaDevNum = 0;

						return rc;
					}

					if(tffscmp(verifyBuf, verBufOrig, DOCH_SECTOR_SIZE))
					{
						DBG_PRINT_ERR(FLZONE_API, "\r\nDOCHWritePartitionSectors(): Sector ");
						DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("#%d "), i));
						DBG_PRINT_ERR(FLZONE_API, "Verify failed ! \r\n");

						pdev->bAtaDevNum = 0;

						return DOCH_WriteFault;
					}

					verBufOrig += DOCH_SECTOR_SIZE;
				}
			}

			/*Return actual number of written sectors*/
			/*---------------------------------------*/
			ioreq2.irSectorCount = sectors_written_so_far;

			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;
		}
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*            D O C H W r i t e A n d L o c k							*/
/*                                                                      */
/* Writes sectors by sector no to a specific partition                  */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irData          : Address of user buffer to write from        */
/*        irSectorNo      : First sector no. to write                   */
/*        irSectorCount   : Number of consecutive sectors to write      */
/*        irFlags	      : DOCH_USE_DMA								*/
/*							DOCH_USE_BURST								*/
/*							Note: 4 methods of data transfer are		*/
/*							available:									*/
/*								Normal transfer							*/
/*								DMA    transfer							*/
/*								Burst  transfer							*/
/*								DMA & Burst transfer					*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_WRITE_PARTITION									*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors NOT read (in case of an	*/
/*							error - this number may not be accurate)	*/
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWriteAndLockSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket * pdev;
	IOreq ioreq2;
	DOCH_Registers out_regs;

	/*Sector arithmetic*/
	/*-----------------*/
	FLSDword total_num_of_sectors = ioreq->irSectorCount;
	FLSDword sectorNumber = ioreq->irSectorNo;
	FLSNative  sectors_written_so_far = 0;

	/*Buffer to write from*/
	/*--------------------*/
	FLByte* buf_ptr = (FLByte*)ioreq->irData;
	FLSNative  buf_offset = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*If a negative number of sectors was requested - return error*/
	/*------------------------------------------------------------*/
	if(total_num_of_sectors < 0)
		return DOCH_BadParameter;

	/*Enable interrupts*/
	/*-----------------*/
	pdev->bUseInterrupt = TRUE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						   DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)),
						   0);


	/*Set DMA and Burst modes according to request*/
	/*--------------------------------------------*/
	pdev->bUseDMA	= (((ioreq->irFlags & DOCH_USE_DMA)	 == DOCH_USE_DMA) &&
					   (gIsDMAEnabled == DOCH_GLOBAL_BOOL_PATTERN) &&
					   (DOCH_DMA_CHECK_BUFFER(buf_ptr, total_num_of_sectors)));

	pdev->bUseBurst = ((ioreq->irFlags & DOCH_USE_BURST) == DOCH_USE_BURST);

	/*Write sectors in DOCH_MAX_SECTORS quantities*/
	/*--------------------------------------------*/
	while(total_num_of_sectors > 0)
	{
		FLSNative  sectors_this_cycle;

		if(total_num_of_sectors > DOCH_MAX_SECTORS)
			sectors_this_cycle = DOCH_MAX_SECTORS;
		else
			sectors_this_cycle = total_num_of_sectors;

		rc = performSectorsOperation(DOCH_VSCMD_WRITE_PARTITION,
									 0,
								 	 pdev,
									 ioreq,
									 &out_regs,
									 sectors_this_cycle,
									 (FLSNative*)&sectorNumber,
									 (FLSNative*)&total_num_of_sectors,
									 &sectors_written_so_far,
									 &buf_offset,
									 buf_ptr);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHWriteAndLock(): failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			/*Disable interrupts*/
			/*------------------*/
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

			/*Cancel DMA and Burst modes*/
			/*--------------------------*/
			pdev->bUseDMA	= FALSE;
			pdev->bUseBurst = FALSE;

			if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
				return DOCH_ProtectionFault;
			else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
				return DOCH_PartitionLimitExceeded;
			else
				return DOCH_GeneralFailure;
		}
	}

	/*Disable interrupts*/
	/*------------------*/
	pdev->bUseInterrupt = FALSE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

	/*Cancel DMA and Burst modes*/
	/*--------------------------*/
	pdev->bUseDMA	= FALSE;
	pdev->bUseBurst = FALSE;

	/* Now lock the partition as OTP */
	/*-------------------------------*/
	tffsset(&protAPI, 0, sizeof(DOCH_PartitionProtectionAPI));
	tffsset(&ioreq2, 0, sizeof(ioreq2));
	tffscpy(&ioreq2.irHandle, &ioreq->irHandle, sizeof(ioreq2.irHandle));
	ioreq2.irData = &protAPI;
	ioreq2.irFlags = DOCH_LOCK_AS_OTP;
	rc = DOCHSetParitionProtection(&ioreq2);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHWriteAndLock(): DOCHSetParitionProtection failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

	/*Return actual number of written sectors*/
	/*---------------------------------------*/
	ioreq->irSectorCount = sectors_written_so_far;

	return DOCH_OK;
}

DOCH_Error DOCHWriteAndLock(IOreq *ioreq)
{
	DOCH_Error       rc = DOCH_OK;
	IOreq            ioreq2;
	DOCH_Socket      *pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	FLDword          sectorsToWriteToFirstChunk;
	FLDword          sectorsToWriteToSecondChunk;
	FLByte           *buf;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		rc = DOCHWriteAndLockSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHWriteAndLockSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*irLength	=> First sector to read*/
		/*irCount	=> Number of sectors to read*/
		if( ((FLDword)((ioreq->irLength + ioreq->irCount))) > pdev->sSpanData.dwSpanSector)
		{
			if(((FLDword)ioreq->irLength) >= pdev->sSpanData.dwSpanSector)
			{
				sectorsToWriteToFirstChunk = 0;
				sectorsToWriteToSecondChunk = ioreq->irCount;

			}
			else
			{
				sectorsToWriteToFirstChunk  = (pdev->sSpanData.dwSpanSector - ioreq->irLength);
				sectorsToWriteToSecondChunk = (ioreq->irCount - sectorsToWriteToFirstChunk);
			}
		}
		else
		{
			sectorsToWriteToFirstChunk  = ioreq->irCount;
			sectorsToWriteToSecondChunk = 0;
		}

		/*Write first chunk to Dev0*/
		/*-------------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irCount = sectorsToWriteToFirstChunk;
		rc = DOCHWriteAndLockSingleFloor(&ioreq2);
		if(rc != DOCH_OK)
			return rc;

		/*Write 2nd chunk to Dev1*/
		/*-----------------------*/
		if(sectorsToWriteToSecondChunk > 0)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
			ioreq2.irCount  = sectorsToWriteToSecondChunk;
			ioreq2.irLength = 0;
			buf = (FLByte*)ioreq->irData;
			ioreq2.irData = buf + (sectorsToWriteToFirstChunk<<DOCH_SECTOR_SIZE_BITS);
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

			pdev->bAtaDevNum = 1;
			rc = DOCHWriteAndLockSingleFloor(&ioreq2);
			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;
		}
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*             D O C H F r e e S e c t o r s							*/
/*                                                                      */
/* Marks absolute sectors by sector no. as free to be written           */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irSectorNo      : First sector no. to free	                */
/*        irSectorCount   : Number of consecutive sectors to free       */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_ERASE_PARTITION_SECTORS							*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors actually deleted			*/
/*----------------------------------------------------------------------*/
DOCH_Error DOCHFreeSectorsSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket * pdev;
	DOCH_Registers out_regs;

	/*Sectors arithmetic*/
	/*------------------*/
	FLSDword total_num_of_sectors = ioreq->irSectorCount;
	FLSDword sectorNumber = ioreq->irSectorNo;
	FLSNative  sectors_freed_so_far = 0;

	/*Dummy buffer for use of performSectorsOperation()*/
	/*-------------------------------------------------*/
	FLByte* buf_ptr = NULL;
	FLSNative  buf_offset = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*If a negative number of sectors was requested - return error*/
	/*------------------------------------------------------------*/
	if(total_num_of_sectors < 0)
		return DOCH_BadParameter;

	/*Free sectors in DOCH_MAX_SECTORS quantities*/
	/*--------------------------------------------*/
	while(total_num_of_sectors > 0)
	{
		FLSNative  sectors_this_cycle;

		if(total_num_of_sectors > DOCH_MAX_SECTORS)
			sectors_this_cycle = DOCH_MAX_SECTORS;
		else
			sectors_this_cycle = total_num_of_sectors;

		rc = performSectorsOperation(DOCH_VSCMD_ERASE_PARTITION_SECTORS,
									 0,
								 	 pdev,
									 ioreq,
									 &out_regs,
									 sectors_this_cycle,
									 (FLSNative*)&sectorNumber,
									 (FLSNative*)&total_num_of_sectors,
									 &sectors_freed_so_far,
									 &buf_offset,
									 buf_ptr);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHFreeSectors(): failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
				return DOCH_ProtectionFault;
			else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
				return DOCH_PartitionLimitExceeded;
			else
				return DOCH_GeneralFailure;
		}
	}

	/*Return actual number of freed sectors*/
	/*-------------------------------------*/
	ioreq->irSectorCount = sectors_freed_so_far;

	return DOCH_OK;
}

DOCH_Error DOCHFreeSectors(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	FLDword sectorsToDeleteFromFirstChunk;
	FLDword sectorsToDeleteFromSecondChunk;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		rc = DOCHFreeSectorsSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHFreeSectorsSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*irLength	=> First sector to read*/
		/*irCount	=> Number of sectors to read*/
		if( ((FLDword)((ioreq->irLength + ioreq->irCount))) > pdev->sSpanData.dwSpanSector)
		{
			if(((FLDword)ioreq->irLength) >= pdev->sSpanData.dwSpanSector)
			{
				sectorsToDeleteFromFirstChunk = 0;
				sectorsToDeleteFromSecondChunk = ioreq->irCount;

			}
			else
			{
				sectorsToDeleteFromFirstChunk  = (pdev->sSpanData.dwSpanSector - ioreq->irLength);
				sectorsToDeleteFromSecondChunk = (ioreq->irCount - sectorsToDeleteFromFirstChunk);
			}
		}
		else
		{
			sectorsToDeleteFromFirstChunk  = ioreq->irCount;
			sectorsToDeleteFromSecondChunk = 0;
		}

		/*Free first chunk from Dev0*/
		/*--------------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irCount = sectorsToDeleteFromFirstChunk;
		rc = DOCHFreeSectorsSingleFloor(&ioreq2);
		if(rc != DOCH_OK)
			return rc;

		/*Free 2nd chunk from Dev1*/
		/*------------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irCount = sectorsToDeleteFromSecondChunk;
		ioreq2.irLength = 0;
		DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

		pdev->bAtaDevNum = 1;
		rc = DOCHFreeSectorsSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;
	}

	return DOCH_OK;
}


/*----------------------------------------------------------------------*/
/*         D O C H W i p e S e c t o r s								*/
/*                                                                      */
/* Securely erase data in sectors										*/
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irSectorCount   : Number of sectors to transfer (1..2).		*/
/*        irData		  : 1..2 sectors with following layout:			*/
/*								- 1st 8bytes - reserved	as zeroes		*/
/*								- (FLDWord) : First sector to delete	*/
/*								  (FLDWord) : # of sectors to delete	*/
/*								- (FLDWord) : Second sector to delete	*/
/*								  (FLDWord) : # of sectors to delete	*/
/*								- ...									*/
/*								- (FLDWord) : Nth sector to delete		*/
/*								  (FLDWord) : # of sectors to delete	*/
/*							* Last transaction should have both start	*/
/*							  and length fields set to 0.				*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_SECURE_ERASE											*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWipeSectorsSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/* Update ATA register values */
	/*----------------------------*/
	in_regs.bFeaturesError = DOCH_SECURE_ERASE;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = (FLByte)(ioreq->irSectorCount);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/* Activate ATA command */
	/*----------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						ioreq->irSectorCount);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHWipeSectors(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
			return DOCH_PartitionLimitExceeded;
		else
			return DOCH_GeneralFailure;
	}

	return DOCH_OK;
}

DOCH_Error DOCHWipeSectors(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	FLByte  dev0TableIndex, dev1TableIndex;
	FLBoolean dev1WipeRequested = FALSE;
	DOCH_Socket* pdev;
	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if( (!pdev->sSpanData.secondFloorActive) || (partNum < pdev->sSpanData.bLastPartitionOnDev0) ||
		((partNum == pdev->sSpanData.bLastPartitionOnDev0) && (!pdev->sSpanData.bLastDev0PartSpanned)) )
	{
		rc = DOCHWipeSectorsSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(((pdev->sSpanData.secondFloorActive) && (partNum > pdev->sSpanData.bLastPartitionOnDev0))
		&& (pdev->wNumOfDevices > 1))
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHWipeSectorsSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(((pdev->sSpanData.bLastDev0PartSpanned) && (partNum == pdev->sSpanData.numOfSpannedPartitionOnDev0))
		&& (pdev->wNumOfDevices > 1))
	{
		tffscpy(wipeSectorsDev0, ioreq->irData, (ioreq->irCount * DOCH_SECTOR_SIZE));
		tffsset(wipeSectorsDev1, 0			  ,	sizeof(wipeSectorsDev1));
		dev0TableIndex = 2;
		dev1TableIndex = 2;

		/*Split requested sectors to delete to Dev0/Dev1*/
		/*----------------------------------------------*/

		/*Copy all entries from that belong to Dev1 to "wipeSectorsDev1"*/
		while((wipeSectorsDev0[dev0TableIndex] != 0) || (wipeSectorsDev0[dev0TableIndex+1] != 0))
		{
			if(wipeSectorsDev0[dev0TableIndex] > pdev->sSpanData.dwSpanSector)
			{
				wipeSectorsDev1[dev1TableIndex]   = (wipeSectorsDev0[dev0TableIndex] - pdev->sSpanData.dwSpanSector);
				wipeSectorsDev1[dev1TableIndex+1] = wipeSectorsDev0[dev0TableIndex+1];

				wipeSectorsDev0[dev0TableIndex]	  = 0;
				wipeSectorsDev0[dev0TableIndex+1] = 0;

				dev1TableIndex += 2;
				dev1WipeRequested = TRUE;
			}
			else if ((wipeSectorsDev0[dev0TableIndex] + wipeSectorsDev0[dev0TableIndex+1]) > pdev->sSpanData.dwSpanSector)
			{
				wipeSectorsDev1[dev1TableIndex]   = 0;
				wipeSectorsDev1[dev1TableIndex+1] = wipeSectorsDev0[dev0TableIndex+1] -
					(pdev->sSpanData.dwSpanSector - wipeSectorsDev0[dev0TableIndex]);

				wipeSectorsDev0[dev0TableIndex+1] -= wipeSectorsDev1[dev1TableIndex+1];

				dev1TableIndex += 2;
				dev1WipeRequested = TRUE;
			}

			dev0TableIndex += 2;
		}

		/*Wipe sectors on Dev0*/
		/*--------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		ioreq2.irData = wipeSectorsDev0;
		rc = DOCHWipeSectorsSingleFloor(&ioreq2);
		if(rc != DOCH_OK)
			return rc;

		/*Wipe Sectors on Dev1*/
		/*--------------------*/
		if(dev1WipeRequested)
		{
			tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));
			ioreq2.irData = wipeSectorsDev1;

			pdev->bAtaDevNum = 1;
			rc = DOCHWipeSectorsSingleFloor(&ioreq2);
			pdev->bAtaDevNum = 0;
			if(rc != DOCH_OK)
				return rc;
		}
	}

	return DOCH_OK;

}

/*----------------------------------------------------------------------*/
/*          D O C H P r e p a r e F o r W r i t e						*/
/*                                                                      */
/*	Signal to DOCH that these sectors are going to be over-written in	*/
/*	following write command												*/
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irSectorNo      : First sector no. to write                   */
/*        irSectorCount   : Number of consecutive sectors to write      */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_OPTIMIZE_PARTITION_SECTORS						*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHPrepareForWrite(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket * pdev;
	DOCH_Registers out_regs;

	/*Sectors arithmetic*/
	/*------------------*/
	FLSDword total_num_of_sectors = ioreq->irSectorCount;
	FLSDword sectorNumber = ioreq->irSectorNo;
	FLSNative  sectorsPreparedSoFar = 0;

	/*Buffer to "write" from*/
	/*----------------------*/
	FLByte* buf_ptr = (FLByte*)ioreq->irData;
	FLSNative  buf_offset = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*If a negative number of sectors was requested - return error*/
	/*------------------------------------------------------------*/
	if(total_num_of_sectors < 0)
		return DOCH_BadParameter;

	rc = performSectorsOperation(DOCH_VSCMD_OPTIMIZE_PARTITION_SECTORS,
								 PREPARE_FOR_WRITE_OP,
								 pdev,
								 ioreq,
								 &out_regs,
								 ioreq->irSectorCount,
								 (FLSNative*)&sectorNumber,
								 (FLSNative*)&total_num_of_sectors,
								 &sectorsPreparedSoFar,
								 &buf_offset,
								 buf_ptr);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHPrepareForWrite(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
			return DOCH_PartitionLimitExceeded;
		else
			return DOCH_GeneralFailure;
	}
	/*If not enough free space - defragment*/
	/*-------------------------------------*/
	if(ioreq->irSectorCount > out_regs.bSectorCount)
	{
		IOreq ioreq2;
		ioreq2.irHandle = ioreq->irHandle;
		rc = DOCHOptimizeMedia(&ioreq2);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHPrepareForWrite(): DOCHOptimizeMedia failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			return rc;
		}

	}

	return DOCH_OK;
}

#if 0
/*	Flexi-Flash	*/

/*----------------------------------------------------------------------*/
/*	        D O C H W r i t e F l e x i F a s t							*/
/*                                                                      */
/* Flexi-Flash Write Fast							                    */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irData          : Address of user buffer to write from        */
/*        irSectorNo      : First sector no. to write                   */
/*        irSectorCount   : Number of consecutive sectors to write      */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_WRITE_FLEXI										*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors actually written          */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWriteFlexiFast(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket * pdev;
	DOCH_Registers out_regs;
	FLByte operationRetries = 0;

	/*Sectors arithmetic*/
	/*------------------*/
	FLSDword total_num_of_sectors = ioreq->irSectorCount;
	FLSDword sectorNumber = ioreq->irSectorNo;
	FLSNative  sectors_written_so_far = 0;

	/*Buffer to write from*/
	/*--------------------*/
	FLByte* buf_ptr = (FLByte*)ioreq->irData;
	FLSNative  buf_offset = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*If a negative number of sectors was requested - return error*/
	/*------------------------------------------------------------*/
	if(total_num_of_sectors < 0)
		return DOCH_BadParameter;

	/*Enable interrupts*/
	/*-----------------*/
	pdev->bUseInterrupt = TRUE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						   DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)),
						   0);


	/*Write sectors in DOCH_MAX_SECTORS quantities*/
	/*--------------------------------------------*/
	while(total_num_of_sectors > 0)
	{
		FLSNative  sectors_this_cycle;

		if(total_num_of_sectors > DOCH_MAX_SECTORS)
			sectors_this_cycle = DOCH_MAX_SECTORS;
		else
			sectors_this_cycle = total_num_of_sectors;

dochWriteFlexiRetryPoint:

		rc = performSectorsOperation(DOCH_VSCMD_WRITE_FLEXI,
									 0,
								 	 pdev,
									 ioreq,
									 &out_regs,
									 sectors_this_cycle,
									 (FLSNative*)&sectorNumber,
									 (FLSNative*)&total_num_of_sectors,
									 &sectors_written_so_far,
									 &buf_offset,
									 buf_ptr);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHWriteFlexiFast(): failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			if( ((rc & DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) == DOCH_ATA_ERROR_ADDRESS_MARK_NOT_FOUND) &&
				(operationRetries++ < 4)
				)
				goto dochWriteFlexiRetryPoint;

			/*Disable interrupts*/
			/*------------------*/
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

			if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
				return DOCH_ProtectionFault;
			else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
				return DOCH_PartitionLimitExceeded;
			else
				return DOCH_GeneralFailure;
		}
	}

	/*Disable interrupts*/
	/*------------------*/
	pdev->bUseInterrupt = FALSE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

	/*Return actual number of written sectors*/
	/*---------------------------------------*/
	ioreq->irSectorCount = sectors_written_so_far;

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*	        D O C H W r i t e F l e x i N o r m a l 					*/
/*                                                                      */
/* Flexi-Flash Write Normal							                    */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irData          : Address of user buffer to write from        */
/*        irSectorNo      : First sector no. to write                   */
/*        irSectorCount   : Number of consecutive sectors to write      */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_WRITE_FLEXI										*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors actually written          */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWriteFlexiNormal(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket * pdev;
	DOCH_Registers out_regs;

	/*Sectors arithmetic*/
	/*------------------*/
	FLSDword total_num_of_sectors = ioreq->irSectorCount;
	FLSDword sectorNumber = ioreq->irSectorNo;
	FLSNative  sectors_written_so_far = 0;

	/*Buffer to write from*/
	/*--------------------*/
	FLByte* buf_ptr = (FLByte*)ioreq->irData;
	FLSNative  buf_offset = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*If a negative number of sectors was requested - return error*/
	/*------------------------------------------------------------*/
	if(total_num_of_sectors < 0)
		return DOCH_BadParameter;

	/*Enable interrupts*/
	/*-----------------*/
	pdev->bUseInterrupt = TRUE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						   DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)),
						   0);


	/*Write sectors in DOCH_MAX_SECTORS quantities*/
	/*--------------------------------------------*/
	while(total_num_of_sectors > 0)
	{
		FLSNative  sectors_this_cycle;

		if(total_num_of_sectors > DOCH_MAX_SECTORS)
			sectors_this_cycle = DOCH_MAX_SECTORS;
		else
			sectors_this_cycle = total_num_of_sectors;

		rc = performSectorsOperation(DOCH_VSCMD_WRITE_FLEXI,
									 WRITE_NORMAL_OP,
								 	 pdev,
									 ioreq,
									 &out_regs,
									 sectors_this_cycle,
									 (FLSNative*)&sectorNumber,
									 (FLSNative*)&total_num_of_sectors,
									 &sectors_written_so_far,
									 &buf_offset,
									 buf_ptr);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHWriteFlexiNormal(): failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			/*Disable interrupts*/
			/*------------------*/
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

			if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
				return DOCH_ProtectionFault;
			else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
				return DOCH_PartitionLimitExceeded;
			else
				return DOCH_GeneralFailure;
		}
	}

	/*Disable interrupts*/
	/*------------------*/
	pdev->bUseInterrupt = FALSE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

	/*Return actual number of written sectors*/
	/*---------------------------------------*/
	ioreq->irSectorCount = sectors_written_so_far;

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*	        D O C H R e W r i t e F l e x i N o r m a l					*/
/*                                                                      */
/* Flexi-Flash Re-Write Normal							                */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*                          Partition number (zero based)	            */
/*        irData          : Address of user buffer to write from        */
/*        irSectorNo      : First sector no. to write                   */
/*        irSectorCount   : Number of consecutive sectors to write      */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_OPTIMIZE_PARTITION_SECTORS						*/
/* ATA sub-command:														*/
/*			None														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*        irSectorCount   : Number of sectors actually re-written       */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReWriteFlexiNormal(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket * pdev;
	DOCH_Registers out_regs;

	/*Sectors arithmetic*/
	/*------------------*/
	FLSDword total_num_of_sectors = ioreq->irSectorCount;
	FLSDword sectorNumber = ioreq->irSectorNo;
	FLSNative  sectors_written_so_far = 0;

	/*Buffer to rewrite from*/
	/*----------------------*/
	FLByte* buf_ptr = (FLByte*)ioreq->irData;
	FLSNative  buf_offset = 0;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*If a negative number of sectors was requested - return error*/
	/*------------------------------------------------------------*/
	if(total_num_of_sectors < 0)
		return DOCH_BadParameter;

	/*Enable interrupts*/
	/*-----------------*/
	pdev->bUseInterrupt = TRUE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						   DOCH_IRQ_RB_INIT(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)),
						   0);


	/*Write sectors in DOCH_MAX_SECTORS quantities*/
	/*--------------------------------------------*/
	while(total_num_of_sectors > 0)
	{
		FLSNative  sectors_this_cycle;

		if(total_num_of_sectors > DOCH_MAX_SECTORS)
			sectors_this_cycle = DOCH_MAX_SECTORS;
		else
			sectors_this_cycle = total_num_of_sectors;

		rc = performSectorsOperation(DOCH_VSCMD_OPTIMIZE_PARTITION_SECTORS,
									 REWRITE_NORMAL_OP,
								 	 pdev,
									 ioreq,
									 &out_regs,
									 sectors_this_cycle,
									 (FLSNative*)&sectorNumber,
									 (FLSNative*)&total_num_of_sectors,
									 &sectors_written_so_far,
									 &buf_offset,
									 buf_ptr);
		if(rc != DOCH_OK)
		{
			DBG_PRINT_ERR(FLZONE_API, "DOCHReWriteNormal(): failed with status: ");
			DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

			/*Disable interrupts*/
			/*------------------*/
			pdev->bUseInterrupt = FALSE;
			dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

			if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
				return DOCH_ProtectionFault;
			else if((rc & DOCH_ATA_ERROR_ID_NOT_FOUND) == DOCH_ATA_ERROR_ID_NOT_FOUND)
				return DOCH_PartitionLimitExceeded;
			else
				return DOCH_GeneralFailure;
		}
	}

	/*Disable interrupts*/
	/*------------------*/
	pdev->bUseInterrupt = FALSE;
	dochEnableATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq), FALSE, 0);

	/*Return actual number of written sectors*/
	/*---------------------------------------*/
	ioreq->irSectorCount = sectors_written_so_far;

	return DOCH_OK;
}


/*----------------------------------------------------------------------*/
/*       D O C H M a n a g e A t o m i c W r i t e S e q				*/
/*                                                                      */
/* Manage atomic write sequence per partition		                    */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*                        Partition number (zero based)					*/
/*		irCount			: DOCH_START_ATOMIC_WRITE						*/
/*						  DOCH_FINISH_ATOMIC_WRITE						*/
/*						  DOCH_ABORT_ATOMIC_WRITE						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_ATOMIC_WRITE_SEQUENCE									*/
/*                                                                      */
/* Returns:                                                             */
/*        irLength        : Maximal possible length						*/
/*        irFlags         : Maximal possible length without performance	*/
/*							loss.										*/
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHManageAtomicWriteSeq(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_ATOMIC_WRITE_SEQUENCE;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = (FLByte)(ioreq->irCount);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Return values*/
	/*-------------*/
	ioreq->irLength = out_regs.bSectorCount + (out_regs.bSectorNumber << 8);
	ioreq->irFlags  = out_regs.bCylLow		+ (out_regs.bCylHigh << 8);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHManageAtomicWriteSeq(): failed with status: ");
        DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}

}

#endif /*0*/


/*----------------------------------------------------------------------*/
/*          D O C H A c c e s s P a r t W i t h P w d					*/
/*                                                                      */
/* Insert the protection key in order to remove the protection of the   */
/* specified partition or the entire media								*/
/* Partition could be authenticated both by user and master password	*/
/* the last successful authentication supersedes the previous one.		*/
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*                     Partition number (zero based)					*/
/*						0xF - for authenticating the entire device		*/
/*	irFlags			 : Authentication Type								*/
/*					 : DOCH_ACCESS_USER_PASSWORD						*/
/*					   DOCH_ACCESS_MASTER_PASSWORD						*/
/*					   DOCH_ACCESS_GATE_KEEPER							*/
/*  irData           : Pointer to an DOCH_PartitionAcessPassword struct */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_ACCESS_CONTROL									*/
/* ATA sub-command:														*/
/*			DOCH_EN_ACCESS_WPWD											*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHAccessPartWithPwdSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_PartitionAcessPassword* partPwdToBig = (DOCH_PartitionAcessPassword*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;
	FLByte sectorCountValue = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	if(sectorCountValue == 0xF)
		sectorCountValue = 0xFF;

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_EN_ACCESS_WPWD;
	in_regs.bSectorCount = sectorCountValue;
	in_regs.bSectorNumber = (FLByte)(ioreq->irFlags);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_ACCESS_CONTROL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_PartitionAcessPassword) / DOCH_SECTOR_SIZE));

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHAccessPartWithPwd(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHAccessPartWithPwd(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		rc = DOCHAccessPartWithPwdSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHAccessPartWithPwdSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*Set to chunk on Dev0*/
		/*--------------------*/
		rc = DOCHAccessPartWithPwdSingleFloor(ioreq);
		if(rc != DOCH_OK)
			return rc;

		/*Set to chunk on Dev0*/
		/*--------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

		pdev->bAtaDevNum = 1;
		rc = DOCHAccessPartWithPwdSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*          D O C H D i s a b l e P a r t A c c e s s					*/
/*                                                                      */
/* Remove the protection key making the partition protected again       */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle        : Socket number (zero based)                  */
/*		                    Partition number (zero based)				*/
/*								0xF - for disabling access to the entire*/
/*									  media								*/
/*		  irFlags		 : Authentication to withdraw					*/
/*						   DOCH_ACCESS_USER_PASSWORD					*/
/*						   DOCH_ACCESS_MASTER_PASSWORD					*/
/*						   DOCH_ACCESS_GATE_KEEPER						*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_ACCESS_CONTROL									*/
/* ATA sub-command:														*/
/*			DOCH_DISABLE_ACCESS											*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHDisablePartAccessSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;
	FLByte sectorCountValue = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	if(sectorCountValue == 0xF)
		sectorCountValue = 0xFF;

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_DISABLE_ACCESS;
	in_regs.bSectorCount = sectorCountValue;
	in_regs.bSectorNumber = (FLByte)(ioreq->irFlags);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_ACCESS_CONTROL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHDisablePartAccess(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHDisablePartAccess(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		rc = DOCHDisablePartAccessSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHDisablePartAccessSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*Set to chunk on Dev0*/
		/*--------------------*/
		rc = DOCHDisablePartAccessSingleFloor(ioreq);
		if(rc != DOCH_OK)
			return rc;

		/*Set to chunk on Dev0*/
		/*--------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

		pdev->bAtaDevNum = 1;
		rc = DOCHDisablePartAccessSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;
	}

	return DOCH_OK;
}

#if 0

/*----------------------------------------------------------------------*/
/*					D O C HS e n d H o s t P K e y 						*/
/*                                                                      */
/* Host sends its public key for this partition & random challenge      */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*	                      Partition number (zero based)					*/
/*							0xF - for disabling access to the entire	*/
/*								  media									*/
/*		irFlags			: DOCH_ACCESS_USER_PASSWORD						*/
/*						  DOCH_ACCESS_MASTER_PASSWORD					*/
/*						  DOCH_ACCESS_GATE_KEEPER						*/
/*		irData			: Pointer to DOCH_HostPublicKey	structure		*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_ACCESS_CONTROL									*/
/* ATA sub-command:														*/
/*			DOCH_TX_HOST_PUBLICKEY										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSendHostPublicKey(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_HostPublicKey* hostPublicKeyToBig = (DOCH_HostPublicKey*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;
	FLByte sectorCountValue = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	if(sectorCountValue == 0xF)
		sectorCountValue = 0xFF;

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_TX_HOST_PUBLICKEY;
	in_regs.bSectorCount  = sectorCountValue;
	in_regs.bSectorNumber = (FLByte)(ioreq->irFlags);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_ACCESS_CONTROL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_HostPublicKey) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	hostPublicKeyToBig->wVersion = be_FLWord((FLByte*)&hostPublicKeyToBig->wVersion);
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSendHostPublicKey(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C HR e c e i v e D o c h P u b l i c K e y 				*/
/*                                                                      */
/* DOCH sends its public key, host's random challenge encrypted by DOCH */
/* private key, and DOCH's random challenge to host						*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*	                      Partition number (zero based)					*/
/*							0xF - for disabling access to the entire	*/
/*								  media									*/
/*		irFlags			: DOCH_ACCESS_USER_PASSWORD						*/
/*						  DOCH_ACCESS_MASTER_PASSWORD					*/
/*						  DOCH_ACCESS_GATE_KEEPER						*/
/*		irData			: Pointer to DOCH_DochPublicKey structure		*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_ACCESS_CONTROL									*/
/* ATA sub-command:														*/
/*			DOCH_RX_DOCH_PUBLICKEY										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReceiveDochPublicKey(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_DochPublicKey* dochPublicKeyToBig = (DOCH_DochPublicKey*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;
	FLByte sectorCountValue = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	if(sectorCountValue == 0xF)
		sectorCountValue = 0xFF;

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_RX_DOCH_PUBLICKEY;
	in_regs.bSectorCount  = sectorCountValue;
	in_regs.bSectorNumber = (FLByte)(ioreq->irFlags);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_ACCESS_CONTROL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_DochPublicKey) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	dochPublicKeyToBig->wVersion = be_FLWord((FLByte*)&dochPublicKeyToBig->wVersion);
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReceiveDochPublicKey(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H V e r i f y H o s t K e y							*/
/*                                                                      */
/* Host sends DOCH's random challenge encrypted by host's private key   */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*	                      Partition number (zero based)					*/
/*							0xF - for disabling access to the entire	*/
/*								  media									*/
/*		irFlags			: DOCH_ACCESS_USER_PASSWORD						*/
/*						  DOCH_ACCESS_MASTER_PASSWORD					*/
/*						  DOCH_ACCESS_GATE_KEEPER						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_ACCESS_CONTROL									*/
/* ATA sub-command:														*/
/*			DOCH_VERIFY_HOST_KEY										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHVerifyHostKey(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_VerifyHostKey* verifyHostKeyToBig = (DOCH_VerifyHostKey*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;
	FLByte sectorCountValue = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	if(sectorCountValue == 0xF)
		sectorCountValue = 0xFF;

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_VERIFY_HOST_KEY;
	in_regs.bSectorCount  = sectorCountValue;
	in_regs.bSectorNumber = (FLByte)(ioreq->irFlags);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_ACCESS_CONTROL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_VerifyHostKey) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	verifyHostKeyToBig->wVersion = be_FLWord((FLByte*)&verifyHostKeyToBig->wVersion);
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHVerifyHostKey(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

#endif /*0*/

/*----------------------------------------------------------------------*/
/*    D O C H S e t P a r i t i o n P r o t e c t i o n					*/
/*                                                                      */
/* Set partition properties (attributes/passkey)	.                   */
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*                     Partition number (zero based)					*/
/*  irData           : Pointer to DOCH_PartitionProtectionAPI           */
/*  irFlags          :  DOCH_PASSKEY_VALID						        */
/*						DOCH_ATTRIBUTES_VALID							*/
/*						DOCH_LOCK_AS_OTP								*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_SET_PARTITION_PROTECTION								*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetParitionProtectionSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_PartitionProtectionAPI* partProperties = (DOCH_PartitionProtectionAPI*)(ioreq->irData);
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	tffsset(&partFormatInfo, 0, sizeof(DOCH_PartitionFormatInfo));

	if((ioreq->irFlags & DOCH_ATTRIBUTES_VALID) == DOCH_ATTRIBUTES_VALID)
	{
		/*Format DOCH_PartitionProtectionAPI structure to DOCH_PartitionFormatInfo*/
		/*------------------------------------------------------------------------*/

		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_PROTECTION_TYPE,
					 DOCH_PA2O_PROTECTION_TYPE,
					 partProperties->dwProtectionType);
		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_USER_MODE,
					 DOCH_PA2O_USER_MODE,
					 partProperties->dwUserAccessMode);
		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_GUEST_MODE,
					 DOCH_PA2O_GUEST_MODE,
					 partProperties->dwGuestAccessMode);
		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_MASTER_CTRL,
					 DOCH_PA2O_MASTER_CTRL,
					 partProperties->dwMasterControl);
		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_ENCRYPT_TYPE,
					 DOCH_PA2O_ENCRYPT_TYPE,
					 partProperties->dwEncryptionType);
		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_LOCK_CTRL,
					 DOCH_PA2O_LOCK_CTRL,
					 partProperties->dwLockControl);
		DOCH_SetBits(&partFormatInfo.partitionAttributes2,
					 DOCH_PA2B_MAX_AUTH_ATTEMPTS,
					 DOCH_PA2O_MAX_AUTH_ATTEMPTS,
					 partProperties->dwMaxNumOfAuthAttempts);
	}
	else
		partFormatInfo.partitionAttributes2 = 0;

	if((ioreq->irFlags & DOCH_PASSKEY_VALID) == DOCH_PASSKEY_VALID)
		tffscpy(&partFormatInfo.bPasskey[0], &partProperties->bPasskey[0], sizeof(partFormatInfo.bPasskey));
	else
		tffsset(&partFormatInfo.bPasskey[0], 0, sizeof(partFormatInfo.bPasskey));

	partFormatInfo.dwValidity = (ioreq->irFlags & (DOCH_PASSKEY_VALID | DOCH_ATTRIBUTES_VALID | DOCH_LOCK_AS_OTP));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	partFormatInfo.dwValidity				= be_FLDword((FLByte*)&partFormatInfo.dwValidity);
	partFormatInfo.dwCommandFlagsOrStatuses = be_FLDword((FLByte*)&partFormatInfo.dwCommandFlagsOrStatuses);
	partFormatInfo.partitionAttributes1		= be_FLDword((FLByte*)&partFormatInfo.partitionAttributes1);
	partFormatInfo.partitionAttributes2		= be_FLDword((FLByte*)&partFormatInfo.partitionAttributes2);
	partFormatInfo.nPartitionSize			= be_FLDword((FLByte*)&partFormatInfo.nPartitionSize);
	partFormatInfo.nFastAreaSize			= be_FLDword((FLByte*)&partFormatInfo.nFastAreaSize);

	partFormatInfo.wFastAreaFactor = be_FLWord((FLByte*)&partFormatInfo.wFastAreaFactor);
	#endif /*DOCH_BIG_ENDIAN*/

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_PARTITION_PROTECTION;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    &partFormatInfo,
						(sizeof(DOCH_PartitionFormatInfo) / DOCH_SECTOR_SIZE));

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetParitionProtection(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHSetParitionProtection(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If partition resides ENTIRELY on Dev0*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV0)
	{
		rc = DOCHSetParitionProtectionSingleFloor(ioreq);
		return rc;
	}

	/*If partition resides ENTIRELY on Dev1*/
	/*=====================================*/
	if(DOCH_PARTITION_ON_DEV1)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHSetParitionProtectionSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		return rc;
	}

	/*If partition is spanned - collect both infos and combine*/
	/*========================================================*/
	if(DOCH_PARTITION_IS_SPANNED)
	{
		/*Set to chunk on Dev0*/
		/*--------------------*/
		rc = DOCHSetParitionProtectionSingleFloor(ioreq);
		if(rc != DOCH_OK)
			return rc;

		/*Set to chunk on Dev0*/
		/*--------------------*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.numOfSpannedPartitionOnDev0 + 1));

		pdev->bAtaDevNum = 1;
		rc = DOCHSetParitionProtectionSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;
	}

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*    D O C H S e t P a r i t i o n U s e r A t t r i b u t e s			*/
/*                                                                      */
/* Set partition user attributes					.                   */
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*                     Partition number (zero based)					*/
/*  irData           : Pointer to DOCH_PartitionUserAttr		        */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_SET_PARTITION_USER_ATTR								*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetParitionUserAttributesSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;
#ifdef DOCH_BIG_ENDIAN
	DOCH_PartitionUserAttr* partAttrToBig = (DOCH_PartitionUserAttr*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*First retrieve current attributes sector
	  (for protecting first 0x40 bytes that are SDK specific*/
	/*------------------------------------------------------*/
	tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
	ioreq2.irData = currentAttributes;
	DOCHGetParitionUserAttributesSingleFloor(&ioreq2);

	/*Use currentAttributes bytes 0x41..0x200 to set requested*/
	/*--------------------------------------------------------*/
	tffscpy(&currentAttributes[DOCH_PART_INFO_SDK_RESERVED_BYTES],
			(((FLByte*)ioreq->irData) + DOCH_PART_INFO_SDK_RESERVED_BYTES),
			(DOCH_SECTOR_SIZE - DOCH_PART_INFO_SDK_RESERVED_BYTES));

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_PARTITION_USER_ATTR;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    currentAttributes,
						(sizeof(DOCH_PartitionUserAttr) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetParitionUserAttributes(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHSetParitionUserAttributes(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Partition resides on Dev0*/
	/*-------------------------*/
	if((partNum <= pdev->sSpanData.bLastPartitionOnDev0) ||
		(!pdev->sSpanData.secondFloorActive))
	{
		rc = DOCHSetParitionUserAttributesSingleFloor(ioreq);
	}
	/*Partition resides on Dev1*/
	/*-------------------------*/
	else if(partNum <= pdev->wTotalNumOfPartitions)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHSetParitionUserAttributesSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
		if(rc != DOCH_OK)
			return rc;
	}
	/*Partition not present*/
	/*---------------------*/
	else
		return DOCH_PartitionNotFound;

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*    D O C H G e t P a r i t i o n U s e r A t t r i b u t e s			*/
/*                                                                      */
/* Get partition user attributes					.                   */
/*                                                                      */
/* Parameters:                                                          */
/*  irHandle         : Socket number (zero based)						*/
/*                     Partition number (zero based)					*/
/*  irData           : Pointer to DOCH_PartitionUserAttr		        */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_PARTITION_MANAGEMENT								*/
/* ATA sub-command:														*/
/*			DOCH_GET_PARTITION_USER_ATTR								*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetParitionUserAttributesSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_PartitionUserAttr* partAttrToBig = (DOCH_PartitionUserAttr*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Check if Partition exceeds number of existing partitions*/
	/*--------------------------------------------------------*/
	if(DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq) >= pdev->device[pdev->bAtaDevNum].wNumOfPartitions)
		return DOCH_PartitionNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_PARTITION_USER_ATTR;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_PARTITION_MANAGEMENT;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_PartitionUserAttr) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetParitionUserAttributes(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		if((rc & DOCH_ATA_ERROR_TRACK_0_NOT_FOUND) == DOCH_ATA_ERROR_TRACK_0_NOT_FOUND)
			return DOCH_ProtectionFault;
		else
			return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHGetParitionUserAttributes(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;
	DOCH_Socket* pdev;
	FLByte partNum = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*Partition resides on Dev0*/
	/*-------------------------*/
	if((partNum <= pdev->sSpanData.bLastPartitionOnDev0) ||
		(!pdev->sSpanData.secondFloorActive))
	{
		rc = DOCHGetParitionUserAttributesSingleFloor(ioreq);
	}
	/*Partition resides on Dev1*/
	/*-------------------------*/
	else if(partNum <= pdev->wTotalNumOfPartitions)
	{
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));

		if(pdev->sSpanData.bLastDev0PartSpanned)
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0 + 1));
		else
			DOCH_SET_PARTITION_TO_IOREQ_HANDLE(&ioreq2, (partNum - pdev->sSpanData.bLastPartitionOnDev0));

		pdev->bAtaDevNum = 1;
		rc = DOCHGetParitionUserAttributesSingleFloor(&ioreq2);
		pdev->bAtaDevNum = 0;
	}
	/*Partition not present*/
	/*---------------------*/
	else
		return DOCH_PartitionNotFound;

	return rc;
}

/*----------------------------------------------------------------------*/
/*            D O C H O p t i m i z e M e d i a						    */
/*                                                                      */
/* Performs garbage collection.										    */
/*                                                                      */
/* Parameters:                                                          */
/*        irHandle  : Socket number (zero based)						*/
/*                    Partition number (zero based)						*/
/*																		*/
/*        irLength  : DOCH_OPTIMIZE_BY_PREPARE_FOR_WRITE				*/
/*						based on information collected during previous	*/
/*						PREPARE FOR WRITE commands						*/
/*					  DOCH_OPTIMIZE_DEFAULT								*/
/*						performs default optimization of the whole		*/
/*						partition or media								*/
/*		  irFlags	: DOCH_NORMAL_OPERATION								*/
/*					  DOCH_IMMEDIATE_OPERATION							*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_OPTIMIZE_MEDIA											*/
/*																		*/
/* Returns:                                                             */
/*        irLength  : Actual number of sectors available for writes     */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHOptimizeMedia(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	FLByte action_type;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if(ioreq->irFlags == DOCH_IMMEDIATE_OPERATION)
		action_type = ERASE_IMMEDIATE;
	else
		action_type = ERASE_NORMAL;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_OPTIMIZE_MEDIA;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = (FLByte)(ioreq->irLength);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = action_type;
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Actual number of sectors available for writes*/
	ioreq->irSectorCount = out_regs.bSectorNumber +
						  ((FLSDword)out_regs.bCylLow << 8) +
						  ((FLSDword)out_regs.bCylHigh << 16) +
						  ((FLSDword)out_regs.bDriveHead << 24) ;

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHOptimizeMedia(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H G e t C u s t o m P a r a m e t e r					*/
/*                                                                      */
/* Returns 16-bit custom DOCH parameter				                    */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*                        Partition number (zero based)					*/
/*		irCount			: Parameter# (see DOCH_CustomParams)			*/
/*		irFlags			: DOCH_CUSTOM_PARAM_DEFAULT - default value		*/
/*						  DOCH_CUSTOM_PARAM_TEMP - current (saved) value*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_GET_CUSTOM_PARAM										*/
/*                                                                      */
/* Returns:                                                             */
/*		  irSectorNo	  : Custom parameter value (16-Bit)				*/
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetCustomParameter(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	FLByte setType;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if(ioreq->irFlags == DOCH_CUSTOM_PARAM_TEMP)
		setType = CUSTOM_PARAM_TEMP;
	else if(ioreq->irFlags == DOCH_CUSTOM_PARAM_DEFAULT)
		setType = CUSTOM_PARAM_DEFAULT;
	else
		return DOCH_BadParameter;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_CUSTOM_PARAM;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = (FLByte)(ioreq->irCount & 0x00FF);
	in_regs.bDriveHead = ((FLByte)(((ioreq->irCount & 0x0F00) >> 8)) | setType);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Return retrieved custom parameter value*/
	/*---------------------------------------*/
	ioreq->irSectorNo = (out_regs.bCylLow << 8) + out_regs.bSectorNumber;

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetCustomParameter(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H S e t C u s t o m P a r a m e t e r					*/
/*                                                                      */
/* Sets 16-bit custom DOCH parameter				                    */
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: Custom Parameter Number (DOCH_CustomParams)	*/
/*		irSectorNo		: Custom Parameter Value (16-Bit)				*/
/*		irFlags			: DOCH_CUSTOM_PARAM_DEFAULT - default value		*/
/*						  DOCH_CUSTOM_PARAM_TEMP - current (saved) value*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_SET_CUSTOM_PARAM										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetCustomParameterSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	FLByte setType;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if(ioreq->irFlags == DOCH_CUSTOM_PARAM_TEMP)
		setType = CUSTOM_PARAM_TEMP;
	else if(ioreq->irFlags == DOCH_CUSTOM_PARAM_DEFAULT)
		setType = CUSTOM_PARAM_DEFAULT;
	else
		return DOCH_BadParameter;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_CUSTOM_PARAM;
	in_regs.bSectorCount = DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq);
	in_regs.bSectorNumber = (FLByte)(ioreq->irSectorNo & 0x00FF);
	in_regs.bCylLow = (FLByte)((ioreq->irSectorNo & 0xFF00) >> 8);
	in_regs.bCylHigh = (FLByte)(ioreq->irCount & 0x00FF);
	in_regs.bDriveHead = (((FLByte)(((ioreq->irCount & 0x0F00) >> 8)) | setType) | (pdev->bAtaDevNum * DOCH_DEVICE));
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetCustomParameter(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHSetCustomParameter(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket* pdev;
	FLByte origDevNum;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	origDevNum = pdev->bAtaDevNum;

	pdev->bAtaDevNum = 0;
	rc = DOCHSetCustomParameterSingleFloor(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHSetCustomParameterSingleFloor(ioreq);
		pdev->bAtaDevNum = origDevNum;
	}

	return rc;
}

#if 0
/*----------------------------------------------------------------------*/
/*			D O C H A u t o H a s h C o n t r o l						*/
/*                                                                      */
/* Enable/Disable Auto Hash Read/Write									*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: AUTO_HASH_ENABLE								*/
/*						  AUTO_HASH_DISABLE								*/
/*		irLength		: AUTO_HASH_READ								*/
/*						  AUTO_HASH_WRITE								*/
/*      irFlags			: DOCH_HASH_NO_ENCRYPTION                       */
/*						  DOCH_HASH_ENCRYPT_BY_KEY						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_AUTO_HASH_CONTROL										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHAutoHashControl(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_AUTO_HASH_CONTROL;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = (FLByte)((ioreq->irCount) | (ioreq->irLength<<1) | (ioreq->irFlags<<2));
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHAutoHashControl(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H R e a d C a l c u l a t e d H a s h					*/
/*                                                                      */
/* Command for Paged partitions											*/
/* Transfers to host digest(s) calculated from data on the flash.		*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_READ_CALCULATED_HASH								*/
/* ATA sub-command:														*/
/*			None														*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReadCalculatedHash(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = 0;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_READ_CALCULATED_HASH;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReadCalculatedHash(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H W r i t e C a l c u l a t e d H a s h				*/
/*                                                                      */
/* Command for Paged partitions											*/
/* No data is transfered. Digest(s) calculated from data in the flash	*/
/* are stored to the dedicated area on flash.							*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*      irCount			: DOCH_HASH_NO_ENCRYPTION                       */
/*						  DOCH_HASH_ENCRYPT_BY_KEY						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_WRITE_CALCULATED_HASH							*/
/* ATA sub-command:														*/
/*			None														*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWriteCalculatedHash(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = 0;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	if(ioreq->irCount == DOCH_HASH_ENCRYPT_BY_KEY)
		in_regs.bDriveHead = HASH_BY_KEY;
	else
		in_regs.bDriveHead = HASH_AS_IS;
	in_regs.bCommandStatus = DOCH_VSCMD_WRITE_CALCULATED_HASH;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHWriteCalculatedHash(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H R e a d O r i g i n a l H a s h						*/
/*                                                                      */
/* Command for Paged partitions											*/
/* Transfers to host digest(s) stored previously on dedicated area on	*/
/* flash																*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*      irCount			: DOCH_HASH_NO_ENCRYPTION                       */
/*						  DOCH_HASH_ENCRYPT_BY_KEY						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_READ_ORIGINAL_HASH								*/
/* ATA sub-command:														*/
/*			None														*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReadOriginalHash(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Socket*  pdev;
	DOCH_Registers out_regs;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = 0;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	if(ioreq->irCount == DOCH_HASH_ENCRYPT_BY_KEY)
		in_regs.bDriveHead = HASH_BY_KEY;
	else
		in_regs.bDriveHead = HASH_AS_IS;
	in_regs.bCommandStatus = DOCH_VSCMD_READ_ORIGINAL_HASH;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReadOriginalHash(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*				D O C H W r i t e G i v e n H a s h						*/
/*                                                                      */
/* Command for Paged partitions											*/
/* Transfers from host digest(s) and stores them to the dedicated area 	*/
/* on flash																*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*      irCount			: DOCH_HASH_NO_ENCRYPTION                       */
/*						  DOCH_HASH_ENCRYPT_BY_KEY						*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_WRITE_GIVEN_HASH									*/
/* ATA sub-command:														*/
/*			None														*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHWriteGivenHash(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = 0;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	if(ioreq->irCount == DOCH_HASH_ENCRYPT_BY_KEY)
		in_regs.bDriveHead = HASH_BY_KEY;
	else
		in_regs.bDriveHead = HASH_AS_IS;
	in_regs.bCommandStatus = DOCH_VSCMD_WRITE_GIVEN_HASH;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHWriteGivenHash(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*		D O C H R e p o r t S u p p o r t e d A l g o r i t h m s		*/
/*                                                                      */
/* Report supported algorithms											*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*      irData	        : Pointer to DOCH_supportedAlgorithms structure	*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_REPORT_SUPPORTED_ALGORITHMS							*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReportSupportedAlgorithms(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_supportedAlgorithms* suppAlgToBig = (DOCH_supportedAlgorithms*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_REPORT_SUPPORTED_ALGORITHMS;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_supportedAlgorithms) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	suppAlgToBig->wVersion = be_FLWord((FLByte*)&suppAlgToBig->wVersion);

	suppAlgToBig->wSymmetricAlgorithmsSupported = be_FLDword((FLByte*)&suppAlgToBig->wSymmetricAlgorithmsSupported);
	suppAlgToBig->wAsymmetricAlgorithmsSupported= be_FLDword((FLByte*)&suppAlgToBig->wAsymmetricAlgorithmsSupported);
	suppAlgToBig->wHashSchemesSupported			= be_FLDword((FLByte*)&suppAlgToBig->wHashSchemesSupported);
	suppAlgToBig->wRNGSchemesSupported			= be_FLDword((FLByte*)&suppAlgToBig->wRNGSchemesSupported);
	suppAlgToBig->wDecompressionSchemesSupported= be_FLDword((FLByte*)&suppAlgToBig->wDecompressionSchemesSupported);
	suppAlgToBig->wCompressionSchemesSupported  = be_FLDword((FLByte*)&suppAlgToBig->wCompressionSchemesSupported);
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReportSupportedAlgorithms(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*		D O C H G e t A l g o r i t h m C a p a b i l i t i e s			*/
/*                                                                      */
/* Get algorithms capabilities											*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: Algorithm Category							*/
/*		irLength		: Algorithm # in the Category					*/
/*      irData	        : Pointer to DOCH_AlgorithmCapabilities			*/
/*						  structure										*/
/*																		*/
/*																		*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_GET_ALGORITHM_CAPABILITIES								*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetAlgorithmCapabilities(IOreq* ioreq)
{
	DOCH_Error rc;
#ifdef DOCH_BIG_ENDIAN
	DOCH_AlgorithmCapabilities* algCapToBig = (DOCH_AlgorithmCapabilities*)(ioreq->irData);
#endif /*DOCH_BIG_ENDIAN*/
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_ALGORITHM_CAPABILITIES;
	in_regs.bSectorCount = (FLByte)(ioreq->irCount);
	in_regs.bSectorNumber = (FLByte)(ioreq->irLength);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						(sizeof(DOCH_AlgorithmCapabilities) / DOCH_SECTOR_SIZE));

	#ifdef DOCH_BIG_ENDIAN
	/*In case of a BIG ENDIAN host CPU, reorder bytes in 16 and 32 Bit variables*/
	algCapToBig->wVersion				 = be_FLWord((FLByte*)&algCapToBig->wVersion);
	algCapToBig->wNumOfKeySizesSupported = be_FLWord((FLByte*)&algCapToBig->wNumOfKeySizesSupported);
	algCapToBig->wNumOfModesSupported	 = be_FLWord((FLByte*)&algCapToBig->wNumOfModesSupported);

	algCapToBig->keysAttributes->dwSize
		= be_FLDword((FLByte*)&algCapToBig->keysAttributes->dwSize);
	algCapToBig->keysAttributes->dwCalcTime
		= be_FLDword((FLByte*)&algCapToBig->keysAttributes->dwCalcTime);
	algCapToBig->modesAttributes->wKeySizes
		= be_FLWord((FLByte*)&algCapToBig->modesAttributes->wKeySizes);
	#endif /*DOCH_BIG_ENDIAN*/

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetAlgorithmCapabilities(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*				D O C H S e t A l g o r i t h m M o d e					*/
/*                                                                      */
/* Set algorithms mode													*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: bits 0..7   - Algorithm Category				*/
/*						  bits 8..15  - Algorithm # in category			*/
/*						  bits 16..24 - Mode index						*/
/*						  bits 24..31 - Key size index					*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_SET_ALGORITHM_MODE										*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetAlgorithmMode(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_ALGORITHM_MODE;
	in_regs.bSectorCount = (FLByte)(ioreq->irCount & 0x000F);
	in_regs.bSectorNumber = (FLByte)((ioreq->irCount & 0x00F0) >> 8);
	in_regs.bCylLow = (FLByte)((ioreq->irCount & 0x0F00) >> 16);
	in_regs.bCylHigh = (FLByte)((ioreq->irCount & 0xF000) >> 24);
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetAlgorithmMode(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*		D O C H S t a r t H a s h S t r e a m C a l c u l a t i o n		*/
/*                                                                      */
/* Restart/Continue hash stream calculation								*/
/* Several streams can be supported at the same time, but only one		*/
/* should be active at any moment.										*/
/* Host should issue Stop command to the current stream, before			*/
/* starting or resuming a new stream.									*/
/*																		*/
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: bits 0..7    - Algorithm Category (hash)		*/
/*						  bits 8..15   - Algorithm # in category		*/
/*						  bits 16..23  - Stream #						*/
/*		irLength		: DOCH_HASH_REINITIALIZE						*/
/*						  DOCH_HASH_CONTINUE							*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_START_HASH_STREAM_CALC									*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHStartHashStreamCalculation(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_START_HASH_STREAM_CALC;
	in_regs.bSectorCount = (FLByte)(ioreq->irCount & 0x000F);
	in_regs.bSectorNumber = (FLByte)((ioreq->irCount & 0x00F0) >> 8);
	in_regs.bCylLow = (FLByte)((ioreq->irCount & 0x0F00) >> 16);
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (FLByte)(ioreq->irLength);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHStartHashStreamCalculation(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H R e a d S t o p H a s h S t r e a m C a l c			*/
/*                                                                      */
/* Read/Stop hash stream calculation									*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: DOCH_HASH_CONTINUE_ACCUM						*/
/*						  DOCH_HASH_STOP_ACCUM							*/
/*		irLength		: DOCH_HASH_DONT_RETURN_DATA					*/
/*						  DOCH_HASH_RETURN_ACCUM_DIGEST					*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_READ_STOP_HASH_STREAM_CALC								*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReadStopHashStreamCalc(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_READ_STOP_HASH_STREAM_CALC;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = ((FLByte)(ioreq->irCount) | (FLByte)(ioreq->irLength));
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReadStopHashStreamCalc(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*			D O C H R e t u r n R a n d om N u m b e r s				*/
/*                                                                      */
/* Return random numbers												*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: bits 0..7   - Algorithm Category				*/
/*						  bits 8..15  - Algorithm # in category			*/
/*		irLength		: # of 32-Bit random numbers to generate		*/
/*						  (max is 64, e.g full sector)					*/
/*		irData			: Pointer to 1 sector of data which will hold	*/
/*						  the number of requested random numbers		*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_RETURN_RANDOM_NUMBERS									*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHReturnRandomNumbers(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_RETURN_RANDOM_NUMBERS;
	in_regs.bSectorCount = (FLByte)(ioreq->irCount & 0x000F);
	in_regs.bSectorNumber = (FLByte)((ioreq->irCount & 0x00F0) >> 8);
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (FLByte)(ioreq->irLength);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    ioreq->irData,
						DOCH_SECTOR_SIZE);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHReturnRandomNumbers(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

/*----------------------------------------------------------------------*/
/*						D O C H S e t H a s h K e y						*/
/*                                                                      */
/* Set hash key															*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*		irCount			: DOCH_HASH_DECRYPTION_KEY						*/
/*						  DOCH_HASH_ENCRYPTION_KEY						*/
/*		irLength		: DOCH_HASH_USE_DOCH_PUBLIC_KEY					*/
/*						  DOCH_HASH_USE_HOST_PUBLIC_KEY					*/
/*						  DOCH_HASH_USE_PARTITION_PUBLIC_KEY			*/
/*		irFlags			: DOCH_HASH_USE_DOCH_PRIVATE_KEY				*/
/*						  DOCH_HASH_USE_HOST_PRIVATE_KEY				*/
/*		irData			: 1 Sector of data in case of any "receive"		*/
/*						  option.										*/
/*																		*/
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_SECURITY_CTRL								*/
/* ATA sub-command:														*/
/*			DOCH_SET_KEYS												*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetHashKey(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_KEYS;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;

	in_regs.bDriveHead = ((FLByte)(ioreq->irCount) |
						  ((FLByte)(ioreq->irLength & 0x03) << 2) |
						  ((FLByte)(ioreq->irFlags  & 0x03) << 4));

	in_regs.bCommandStatus = DOCH_VSCMD_EXT_SECURITY_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetHashKey(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

#endif /*0*/


/*----------------------------------------------------------------------*/
/*				f l D O C H G e t P o w e r M o d e						*/
/*                                                                      */
/* Retrieve device power mode											*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle		: Socket number (zero based)		            */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_SET_POWER_MODE											*/
/*																		*/
/* Returns:                                                             */
/*      irFlags			:								                */
/*						  DOCH_PM_WORK_MODE								*/
/*						  DOCH_PM_INACTIVE_MODE							*/
/*      irCount			:												*/
/*			Bits 0..1	: Work Mode										*/
/*						  DOCH_WM_NORMAL				                */
/*						  DOCH_WM_LOW_FREQ				                */
/*						  DOCH_WM_NORMAL_AND_AUTO_STBY	                */
/*						  DOCH_WM_LOW_FREQ_AND_AUTO_STBY	            */
/*			Bit  4		: Inactive Mode									*/
/*						  DOCH_IM_IDLE					                */
/*						  DOCH_IM_DPD									*/
/*						  DOCH_IM_IDLE_2_DPD							*/
/*      irLength		: Timeout for transfer from Active mode to      */
/*						  Inactive mode (in mSec, 100ms resolution)		*/
/*      DOCH_Error		: 0 on success, otherwise failed                */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetPowerMode(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_GET_POWER_MODE;
	in_regs.bSectorCount = 0;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = 0;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = (pdev->bAtaDevNum * DOCH_DEVICE);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHGetPowerMode(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}

	/*Set output values*/
	ioreq->irFlags	= out_regs.bSectorCount;
	ioreq->irCount	= out_regs.bSectorNumber;
	ioreq->irLength = (out_regs.bCylLow + (out_regs.bCylHigh<<8));

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*	            D O C H S e t P o w e r M o d e							*/
/*                                                                      */
/* Set device power mode												*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle		: Socket number (zero based)		            */
/*      irFlags			:								                */
/*			Bits 0..1 - which mode(s) should be set						*/
/*						  DOCH_PM_SET_WORK_MODE			                */
/*						  DOCH_PM_SET_INACTIVE_MODE						*/
/*						  DOCH_PM_SET_BOTH_MODES						*/
/*						  DOCH_PM_SET_NONE								*/
/*			Bit  2	- which mode device should pass to after set        */
/*						  DOCH_PM_WORK_MODE				                */
/*						  DOCH_PM_INACTIVE_MODE				            */
/*			Bit  4										                */
/*						  DOCH_PM_SAVE_DEFAULT							*/
/*						  (Save new settings as default settings )      */
/*      irCount			:												*/
/*			Bits 0..1	: Work Mode to be set							*/
/*						  DOCH_WM_NORMAL				                */
/*						  DOCH_WM_LOW_FREQ				                */
/*						  DOCH_WM_NORMAL_AND_AUTO_STBY	                */
/*						  DOCH_WM_LOW_FREQ_AND_AUTO_STBY	            */
/*			Bit  4		: Inactive Mode	 to be set      				*/
/*						  DOCH_IM_IDLE					                */
/*						  DOCH_IM_DPD									*/
/*						  DOCH_IM_IDLE_2_DPD							*/
/*      irLength		: Timeout from Active to Inactive mode          */
/*                        (1 millisecond resolution) or from Idle to DPD*/
/*                        mode (100 milliseconds resolution) -          */
/*                        up to active inactive configuration			*/
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_SET_POWER_MODE											*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHSetPowerModeSingleFloor(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;
	FLByte setType;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	if((ioreq->irFlags & DOCH_PM_SAVE_DEFAULT) == DOCH_PM_SAVE_DEFAULT)
		setType = DRIVE_HEAD_OPT1;
	else
		setType = 0;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_SET_POWER_MODE;
	in_regs.bSectorCount = (FLByte)(ioreq->irFlags & 0x0000000F);
	in_regs.bSectorNumber = (FLByte)ioreq->irCount;
	in_regs.bCylLow = (FLByte)(ioreq->irLength & 0x000000FF);
	in_regs.bCylHigh = (FLByte)((ioreq->irLength & 0x0000FF00) >> 8);
	in_regs.bDriveHead = ((pdev->bAtaDevNum * DOCH_DEVICE) | setType);
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	if(rc == DOCH_OK)
	{
		/*Update DPD settings*/
		if( (ioreq->irFlags&DOCH_PM_SET_WORK_MODE) || (ioreq->irFlags&DOCH_PM_SET_BOTH_MODES) )
		{
			gDpdSettings.activeMode		= (ioreq->irCount & 0x0000000F);
		}
		if( (ioreq->irFlags&DOCH_PM_SET_INACTIVE_MODE) || (ioreq->irFlags&DOCH_PM_SET_BOTH_MODES) )
		{
			gDpdSettings.inActiveMode	= (ioreq->irCount & 0x000000F0);
		}

		gDpdSettings.timeOut		= ioreq->irLength;

		/*If HAL is SPI, disable DPD*/
		if(gAccessLayerType == DOCH_AL_SPI)
			gDpdSettings.activeMode = DOCH_WM_NORMAL;

		return DOCH_OK;
	}
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHSetPowerMode(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}

DOCH_Error DOCHSetPowerMode(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	DOCH_Socket* pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	pdev->bAtaDevNum = 0;
	rc = DOCHSetPowerModeSingleFloor(ioreq);
	if(rc != DOCH_OK)
		return rc;

	if(pdev->wNumOfDevices > 1)
	{
		pdev->bAtaDevNum = 1;
		rc = DOCHSetPowerModeSingleFloor(ioreq);
		pdev->bAtaDevNum = 0;
	}

	return rc;
}

#if 0 /* save this api because the command defined in spec */
/*----------------------------------------------------------------------*/
/*            D O C H C a l i b r a t e C l o c k						*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)                    */
/*      irCount			: Desired SYS ICMU clock, in MHZ                */
/*      irLength		: Desired FLASH ICMU clock, in MHZ              */
/*                                                                      */
/* ATA command:															*/
/*			DOCH_VSCMD_EXT_DEVICE_CTRL									*/
/* ATA sub-command:														*/
/*			DOCH_CALIBRATE_CLOCK										*/
/*																		*/
/* Returns:                                                             */
/*      irCount			: SYS ICMU correction factor					*/
/*      irLength		: FLASH ICMU correction factor					*/
/*      DOCH_Error		: 0 on success, otherwise failed                */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHCalibrateClock(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Registers in_regs;
	DOCH_Registers out_regs;
	DOCH_Socket*  pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Update ATA register values*/
	/*--------------------------*/
	in_regs.bFeaturesError = DOCH_CALIBRATE_CLOCK;
	in_regs.bSectorCount = (FLByte)ioreq->irCount;
	in_regs.bSectorNumber = 0;
	in_regs.bCylLow = (FLByte)ioreq->irLength;
	in_regs.bCylHigh = 0;
	in_regs.bDriveHead = 0;
	in_regs.bCommandStatus = DOCH_VSCMD_EXT_DEVICE_CTRL;

	/*Activate ATA command*/
	/*--------------------*/
	rc = doch_command ( DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
						pdev->bAtaDevNum,
						&in_regs,
						&out_regs,
					    NULL,
						0);

	/*Output*/
	/*------*/
	ioreq->irCount = out_regs.bSectorCount  + (out_regs.bSectorNumber << 8);
	ioreq->irLength= out_regs.bCylLow		+ (out_regs.bCylHigh << 8);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHCalibrateClock(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return DOCH_GeneralFailure;
	}
}
#endif /*0*/ /* save this api because the command defined in spec */

/******************************************************************************/
/*
 *	General (Control/Configuration)
 */
/******************************************************************************/

void setConfigHWDefaults()
{
	gConfigHWDefaults[DOCH_BURST_WRITE_MODE_CTRL] = DOCH_BURST_WRITE_MODE_DEFAULT;

	gConfigHWDefaults[DOCH_BURST_READ_MODE_CTRL]  = DOCH_BURST_READ_MODE_DEFAULT;

	gConfigHWDefaults[DOCH_IPL_CTRL] = DOCH_IPL_CTRL_DEFAULT;

	gConfigHWDefaults[DOCH_WARM_BOOT_CTRL] = DOCH_WARM_BOOT_CTRL_DEFAULT;

	gConfigHWDefaults[DOCH_POWER_DOWN] = DOCH_POWER_DOWN_DEFAULT;

	gConfigHWDefaults[DOCH_DMA_CTRL] = DOCH_DMA_CTRL_DEFAULT;

	gConfigHWDefaults[DOCH_DMA_NEGATION_CTRL] = DOCH_DMA_NEGATION_CTRL_DEFAULT;

	gConfigHWDefaults[DOCH_SLOCK] = DOCH_SLOCK_DEFAULT;

	gConfigHWDefaults[DOCH_ENDIAN_CTRL] = DOCH_ENDIAN_CTRL_DEFAULT;

	gConfigHWDefaults[DOCH_OPERATION_MODE_CTRL] = DOCH_OPERATION_MODE_CTRL_DEFAULT;

	gConfigHWInitDone = DOCH_GLOBAL_BOOL_PATTERN;
}

/*----------------------------------------------------------------------*/
/*		          D O C H H w C o n f i g								*/
/*                                                                      */
/* Control HW registers													*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*                        Partition number (zero based)					*/
/*      irFlags			: HW Configuration type			                */
/*						  (See DOCH_HwConfigType for values)            */
/*		irLength		: HW configuration type dependant               */
/*                                                                      */
/* ATA command:															*/
/*			NONE														*/
/* ATA sub-command:														*/
/*			NONE														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHConfigHW(IOreq* ioreq)
{
	DOCH_Error rc = DOCH_OK;
	IOreq ioreq2;

	#ifdef DOCH_USE_FUNC
	DOCH_Socket* pdev;
	#endif /*DOCH_USE_FUNC*/
#if !defined(FL_MIGRATION_VERSION) || defined(DOCH_USE_FUNC)
	FLByte socketNum = DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq);
#endif /*FL_MIGRATION_VERSION*/
	FLByte devNum = 0;

	#ifdef DOCH_DMA_CONFIG
	/*DMA_Params_S dmaParams;*/
	#endif /*DOCH_DMA_CONFIG*/

	#ifdef DOCH_USE_FUNC
	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	#endif /*DOCH_USE_FUNC*/

	/* Set default values if not set yet (e.g. DochSDKInit() was not called yet!) */
	if(gConfigHWInitDone != DOCH_GLOBAL_BOOL_PATTERN)
		setConfigHWDefaults();

	gConfigHWDefaults[ioreq->irFlags] = ioreq->irLength;

	/*If init was not done yet, only keep requested value*/
	if(gSdkDOCAddressObtained != DOCH_GLOBAL_BOOL_PATTERN)
	{
		return DOCH_OK;
	}

setConfigItemToDevice:

#if defined(FL_MIGRATION_VERSION) && !defined(DOCH_USE_FUNC)
	DOCHWRITE_ATA_REG (0, DOCH_DRIVE_HEAD_REG, devNum);
#else /*FL_MIGRATION_VERSION*/
	DOCHWRITE_ATA_REG ((&sockets[socketNum])->bRegBase, DOCH_DRIVE_HEAD_REG, devNum);
#endif /*FL_MIGRATION_VERSION*/

	switch((DOCH_HwConfigType)(ioreq->irFlags))
	{
	case DOCH_BURST_WRITE_MODE_CTRL:
		rc = doch_setConfigReg16 (ioreq,
								  HIB_BURST_WRITE_MODE_CTRL_REG,
								  ioreq->irLength);
		break;

	case DOCH_BURST_READ_MODE_CTRL:
		rc = doch_setConfigReg16 (ioreq,
								  HIB_BURST_READ_MODE_CTRL_REG,
								  ioreq->irLength);
		break;

	case DOCH_IPL_CTRL:
		if(ioreq->irLength & DOCH_IPL_WRITE_ENABLE)
		{
			FLWord wTmpConfigVal = doch_getConfigReg16(ioreq, HIB_IPL_CONTROL_REG);
			if((wTmpConfigVal & DOCH_IPL_WRITE_READY) != DOCH_IPL_WRITE_READY)
			{
				rc = DOCH_ATANotReady;
				break;
			}
		}
		rc = doch_setConfigReg16 (ioreq,
								  HIB_IPL_CONTROL_REG,
								  ioreq->irLength);
		break;

	case  DOCH_WARM_BOOT_CTRL:
		rc = doch_setConfigReg16 (ioreq,
								  HIB_WARM_BOOT_REG,
								  ioreq->irLength);
		break;

	case DOCH_POWER_DOWN:
		rc = doch_setConfigReg16 (ioreq,
								  HIB_POWER_DOWN_REG,
								  ioreq->irLength);
		break;

	case DOCH_DMA_CTRL:
		rc = doch_setConfigReg16 (ioreq,
								  HIB_DMA_CTRL_REG,
								  ioreq->irLength);
		break;

	case DOCH_DMA_ENABLE:
		if(ioreq->irLength)
		{
#ifdef DOCH_DMA_CONFIG
			gIsDMAEnabled = DOCH_GLOBAL_BOOL_PATTERN;
#endif /*DOCH_DMA_CONFIG*/
		}
		else
		{
			gIsDMAEnabled = 0;
		}
		break;

	case DOCH_DMA_NEGATION_CTRL:
		rc = doch_setConfigReg16 (ioreq,
								  HIB_DMA_NEGATION_REG,
								  ioreq->irLength);
		break;

	case DOCH_SLOCK:
		rc = doch_setConfigReg16(ioreq,
								 HIB_SW_LOCK_REG,
								 ioreq->irLength);
		if(rc != DOCH_OK)
			break;

		/* If SLOCK was requested, set by ATA command also*/
		if(ioreq->irLength == TRUE)
		{
			tffsset(&ioreq2, 0, sizeof(ioreq2));
			ioreq2.irCount	  = DOCH_CP_SLOCK_CTRL;
			ioreq2.irSectorNo = ioreq->irLength;
			ioreq2.irFlags	  = DOCH_CUSTOM_PARAM_TEMP;
            rc = DOCHSetCustomParameter(&ioreq2);
			if(rc != DOCH_OK)
				break;
		}
		break;

	case DOCH_ENDIAN_CTRL:
		rc = doch_setConfigReg16(ioreq,
								 HIB_ENDIAN_CTRL_REG,
								 ioreq->irLength);
		break;

	case DOCH_OPERATION_MODE_CTRL:
		rc = doch_setConfigReg16(ioreq,
								 HIB_OPERATION_MODE_REG,
								 ioreq->irLength);
		break;

	case DOCH_POWER_MODE_CTRL:
		rc = doch_setConfigReg16(ioreq,
								 HIB_POWER_MODE_REG,
								 ioreq->irLength);
		break;

	default:
		return DOCH_FeatureNotSupported;
	}

	/* If Dev1 exists, set config Item as well*/
	if(devNum == 0)
	{
		devNum = 1;
		goto setConfigItemToDevice;
	}

	/*Set device back to Dev0*/
#if defined(FL_MIGRATION_VERSION) && !defined(DOCH_USE_FUNC)
	DOCHWRITE_ATA_REG (0, DOCH_DRIVE_HEAD_REG, 0);
#else /*FL_MIGRATION_VERSION*/
	DOCHWRITE_ATA_REG ((&sockets[socketNum])->bRegBase, DOCH_DRIVE_HEAD_REG, 0);
#endif /*FL_MIGRATION_VERSION*/



	return rc;
}

/*----------------------------------------------------------------------*/
/*	         D O C H R e c o v e r F r o m P o w e r L o s s			*/
/*                                                                      */
/* Set last known values of DOCH control registers back to device after	*/
/* power loss was detected.												*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*                                                                      */
/* ATA command:															*/
/*			NONE														*/
/* ATA sub-command:														*/
/*			NONE														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHRecoverFromPowerLoss(IOreq* ioreq)
{
	DOCH_Error rc;
	DOCH_Socket* pdev;

	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/* Re-set all control register values to device */
	rc = DOCH_init_config_regs(ioreq);
	if(rc != DOCH_OK)
		return rc;

	/* Re-set all control register values to dev1 */
	if(pdev->wNumOfDevices > 1)
	{
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG, 1);
		rc = DOCH_init_config_regs(ioreq);
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG, 0);
	}

#ifdef CHECK_POWER_ON_EVERY_COMMAND
	{
		IOreq ioreq2;

		/*Make sure next calls to DOCHGetResetStatus() will reflect actual reset*/
		tffscpy(&ioreq2, ioreq, sizeof(ioreq2));
		DOCHGetResetStatus(&ioreq2);
	}
#endif /*CHECK_POWER_ON_EVERY_COMMAND*/

	return rc;
}

/*----------------------------------------------------------------------*/
/*		     f l D O C H A t a R e s e t								*/
/*                                                                      */
/* Perform ATA reset operation for a specific socket					*/
/* Effects both Device0 and Device1										*/
/*																		*/
/* Parameters:                                                          */
/*      socketNum		: Socket number (zero based)		            */
/*                                                                      */
/* ATA command:															*/
/*			NONE														*/
/* ATA sub-command:														*/
/*			NONE														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error flDOCHAtaReset(FLByte socketNum)
{
	DOCH_Error rc;
	IOreq ioreq;
	DOCH_Socket* pdev;
	FLDword tries;
	FLDword i = 0;
	FLByte status;
	DOCH_get_socket(pdev, socketNum);

	DBG_PRINT_ERR(FLZONE_API, "\r\n***********************************\r\n");
	DBG_PRINT_ERR(FLZONE_API, "\r\n* Performing flDOCHAtaReset() !!! *\r\n");
	DBG_PRINT_ERR(FLZONE_API, "\r\n***********************************\r\n");

	/*Protect against access time set to 0*/
	if(gDochAccessNanosec == 0)
		return DOCH_BadParameter;
	tries = (DOCH_SHORT_WAIT * (1000000UL / gDochAccessNanosec));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	tffsset(&ioreq, 0, sizeof(ioreq));
	DOCH_SET_SOCKET_TO_IOREQ_HANDLE(&ioreq, socketNum);

	/*Reset the device using "Set alert level" command*/
	/*------------------------------------------------*/
	/*Write ATA registers*/
	DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG,		0);		/*Dev0*/
	DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_FEATURES_REG,		0x7F);	/*Set Alert Level*/
	DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_SECTOR_CNT_REG,		1);		/*"And Reboot"*/
	DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_SECTOR_NO_REG,		0xFF);	/*Invalid alert level*/
	DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_COMMAND_REG,		0xFC);	/*Extended Device Contrl*/
    Delay_us(10);   // this delay seems to ensure reliable results...

	/*Wait for device to become ready*/
	for(i=0; i<tries; i++)
	{
		/* Exit virtual RAM mode */
		DOCHWRITE_CTRL_REG (pdev->bRegBase, HIB_CHIPID2_REG, 0);

		/*Make sure Dev0 is addressed*/
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG,		0);

		/* Re-set all control register values to device */
		rc = DOCH_init_config_regs(&ioreq);

		status = DOCHREAD_ATA_REG(pdev->bRegBase, DOCH_STATUS_REG);
		if((status & (DOCH_READY | DOCH_BUSY)) == DOCH_READY)
			goto doch_reset_Dev0_returnOK;
	}

	/*If timed out - reset ATA only*/
	rc = doch_reset(socketNum, 0);
	if(rc != DOCH_OK)
		return rc;

doch_reset_Dev0_returnOK:

	if(pdev->wNumOfDevices > 1)
	{
		/*Write ATA registers*/
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG,		DOCH_DEVICE);	/*Dev1*/
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_FEATURES_REG,		0x7F);	/*Set Alert Level*/
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_SECTOR_CNT_REG,		1);		/*"And Reboot"*/
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_SECTOR_NO_REG,		0xFF);	/*Invalid alert level*/
		DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_COMMAND_REG,		0xFC);	/*Extended Device Contrl*/
        Delay_us(10);   // this delay seems to ensure reliable results...

		/*Wait for device to become ready*/
		for(i=0; i<tries; i++)
		{
			/* Exit virtual RAM mode */
			DOCHWRITE_CTRL_REG (pdev->bRegBase, HIB_CHIPID2_REG, 0);

			/*Make sure Dev1 is addressed*/
			DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG,	DOCH_DEVICE);

			/* Re-set all control register values to device */
			rc = DOCH_init_config_regs(&ioreq);

			status = DOCHREAD_ATA_REG(pdev->bRegBase, DOCH_STATUS_REG);
			if((status & (DOCH_READY | DOCH_BUSY)) == DOCH_READY)
				goto doch_reset_Dev1_returnOK;
		}

		/*If timed out - reset ATA only*/
		rc = doch_reset(socketNum, 1);
		if(rc != DOCH_OK)
			return rc;
	}

doch_reset_Dev1_returnOK:

	/*Set device head register back to Dev0*/
	DOCHWRITE_ATA_REG (pdev->bRegBase, DOCH_DRIVE_HEAD_REG, 0);	/*Dev0*/

    /*Restore DPD settings */
    /*=====================*/
	for(i=0; i<pdev->wNumOfDevices; i++)
	{
		tffsset(&ioreq, 0, sizeof(ioreq));

		ioreq.irFlags  = (DOCH_PM_SET_BOTH_MODES | DOCH_PM_WORK_MODE);
		ioreq.irCount  = (gDpdSettings.activeMode | gDpdSettings.inActiveMode);
		ioreq.irLength = gDpdSettings.timeOut;

		pdev->bAtaDevNum = (FLByte)i;
		rc = DOCHSetPowerModeSingleFloor(&ioreq);
		if(rc != DOCH_OK)
			return rc;
	}
	pdev->bAtaDevNum = 0;

	/*Restore transfer mode */
	/*======================*/
	for(i=0; i<pdev->wNumOfDevices; i++)
	{
		tffsset(&ioreq, 0, sizeof(ioreq));

		ioreq.irCount  = pdev->device[i].dataTransferMode;
		ioreq.irLength = pdev->device[i].dwMulti_Current;

		pdev->bAtaDevNum = (FLByte)i;
		rc = DOCHSetDataTransferModeSingleFloor(&ioreq);
		if(rc != DOCH_OK)
			return rc;
	}
	pdev->bAtaDevNum = 0;


	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*				f l	D O C H S e t E n v V a r							*/
/*                                                                      */
/* Set environment variable.											*/
/* Available environment variables are listed in DOCH_EnVars.			*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*                        Partition number (zero based, if relevant)	*/
/*      irFlags			: Environment variable			                */
/*																		*/
/*		irLength		: Value to set					                */
/*                                                                      */
/* ATA command:															*/
/*			NONE														*/
/* ATA sub-command:														*/
/*			NONE														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error flDOCHSetEnvVar(IOreq* ioreq)
{
	DOCH_Error rc;
	IOreq ioreq2;

	switch((DOCH_EnVars)(ioreq->irFlags))
	{
	case DOCH_ENV_VERIFY_WRITE:
			if(ioreq->irLength != 0)
				gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)] = DOCH_GLOBAL_BOOL_PATTERN;
			else
				gDochVerifyWrite[DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq)] = 0;
		return DOCH_OK;

	case DOCH_ENV_ATA_DEBUG:
			if(ioreq->irLength != 0)
				gDochAtaDebug = DOCH_GLOBAL_BOOL_PATTERN;
			else
				gDochAtaDebug = 0;
		return DOCH_OK;

    case DOCH_ENV_BLOCK_SPI:
            if(gSdkInitDone != DOCH_GLOBAL_BOOL_PATTERN)
            {
                /* We do not have a window address yet - can be added in the future. */
                DBG_PRINT_ERR(FLZONE_API, "Can not set SPI BLOCKING mode before calling init routine.\r\n");
                return DOCH_ATANotReady;
            }
            /* Take mutex for current device */
            rc = dochSetMutex(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
                                DOCH_ON,
                                DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq));
            if(rc != DOCH_OK)
            {
                    DBG_PRINT_ERR(FLZONE_API, "bdCallDOCH(): dochSetMutex failed with status: ");
                    DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
                    return rc;
            }

            /* SDK is already initialized */
            tffsset(&ioreq2, 0, sizeof(ioreq2));
            ioreq2.irCount = DOCH_CP_WORK_MODES;
            ioreq2.irFlags = DOCH_CUSTOM_PARAM_TEMP;

            /*Retreive current value (retrieved in irSectorNo)*/
            rc = DOCHGetCustomParameter(&ioreq2);
            if(rc != DOCH_OK)
            {
                DBG_PRINT_ERR(FLZONE_API, "flDOCHSetEnvVar(): failed with status: ");
                DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
            }
            else
            {
                /*Modify value*/
                if(ioreq->irLength != 0)
                {
                    ioreq2.irSectorNo |= DOCH_WORK_MODES_BLOCK_SPI;
                }
                else
                {
                    ioreq2.irSectorNo &= ~(DOCH_WORK_MODES_BLOCK_SPI);
                }

                /*Set modified value*/

                rc = DOCHSetCustomParameter(&ioreq2);
                if(rc != DOCH_OK)
                {
                    DBG_PRINT_ERR(FLZONE_API, "flDOCHSetEnvVar(): failed with status: ");
                    DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
                }
            }
            /* Free mutex for current device */
            dochSetMutex(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq),
                        DOCH_OFF,
                        DOCH_GET_PARTITION_FROM_IOREQ_HANDLE(ioreq));

            return rc;

	case DOCH_ENV_NO_ATA_TIMEOUT:
			if(ioreq->irLength != 0)
			{
				gATANoTimeout = DOCH_GLOBAL_BOOL_PATTERN;
			}
			else
			{
				gATANoTimeout = 0;
			}

			return DOCH_OK;

	case DOCH_ENV_AUTO_DPD_BY_HOST:
		if(ioreq->irLength != 0)
			gAutoDPDByHost = DOCH_GLOBAL_BOOL_PATTERN;
		else
			gAutoDPDByHost = 0;

		return DOCH_OK;

	default:
		break;
	}

	return DOCH_FeatureNotSupported;
}

/*----------------------------------------------------------------------*/
/*			D O C H _ C l e a n I R Q _ I n t e r r u p t				*/
/*                                                                      */
/* Clear the ATA interrupt.												*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*																		*/
/* ATA command:															*/
/*			NONE														*/
/* ATA sub-command:														*/
/*			NONE														*/
/*																		*/
/* Returns:                                                             */
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHClearIRQ(IOreq* ioreq)
{
	DOCH_Error rc = clearATAInterrupt(DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));
	return rc;
}

/*----------------------------------------------------------------------*/
/*		   f l D O C H G e t P h y s i c a l A d d r e s s				*/
/*                                                                      */
/* Retrieve physical address of the socket.								*/
/*                                                                      */
/* Parameters:                                                          */
/*      irHandle        : Socket number (zero based)					*/
/*																		*/
/* ATA command:															*/
/*			NONE														*/
/* ATA sub-command:														*/
/*			NONE														*/
/*																		*/
/* Returns:                                                             */
/*        irCount	  : Socket Physical Address							*/
/*        DOCH_Error  : 0 on success, otherwise failed                  */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHGetPhysicalAddress(IOreq* ioreq)
{
	DOCH_Socket* pdev;
	DOCH_get_socket(pdev, DOCH_GET_SOCKET_FROM_IOREQ_HANDLE(ioreq));

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	ioreq->irCount = pdev->dwPhysicalAddress;

	return DOCH_OK;
}

/******************************************************************************/
/*
 *	Pass-thru routine
 */
/******************************************************************************/

/*----------------------------------------------------------------------*/
/*				D O C H A t a P a s s T h r o u g h						*/
/*                                                                      */
/* Parameters:                                                          */
/*      socketNum		: Socket to perform operation on				*/
/*      passThruOP		: "0" - No Data									*/
/*						  "1" - Data IN						            */
/*						  "2" - Data OUT						        */
/*      in_regs			: ATA Input registers values					*/
/*      out_regs		: ATA Output registers values					*/
/*		secNum			: Number of sectors to perform					*/
/*		userBuff		: Pointer to user buffer						*/
/*						  (for data in/out commands)					*/
/*		useInterrupt	: Use interrupt when waiting on ATA busy		*/
/*																		*/
/* ATA command:															*/
/*			user defined (in_regs->bCommandStatus)						*/
/* ATA sub-command:														*/
/*			None														*/
/*                                                                      */
/* Returns:                                                             */
/*        DOCH_Error      : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/
DOCH_Error DOCHAtaPassThrough(FLSNative		socketNum,
							  FLSNative		passThruOP,
							  DOCH_Registers *in_regs,
							  DOCH_Registers *out_regs,
							  FLNative		secNum,
							  void*			userBuff,
							  FLBoolean		useInterrupt)
{
	DOCH_Error		 rc;
	DOCH_PassThru_Op ptOP   = (DOCH_PassThru_Op)passThruOP;
	FLSNative		 devNum = ((in_regs->bDriveHead & DOCH_DEVICE) == DOCH_DEVICE);
	DOCH_Socket*	 pdev;

	DOCH_get_socket(pdev, socketNum);

	/*If socket is not registered, return error*/
	if(pdev == NULL)
		return DOCH_DiskNotFound;

	/*Enable interrupts*/
	/*-----------------*/
	pdev->bUseInterrupt = useInterrupt;
	dochEnableATAInterrupt(socketNum,
						   (DOCH_IRQ_RB_INIT(socketNum) && useInterrupt),
						   0);


	/*Perform ATA command with registers values set above*/
	/*---------------------------------------------------*/
	rc = doch_ata_passthru (socketNum, devNum, ptOP, in_regs, out_regs, userBuff, secNum);

	/*Disable interrupts*/
	/*------------------*/
	pdev->bUseInterrupt = FALSE;
	dochEnableATAInterrupt(socketNum, FALSE, 0);

	if(rc == DOCH_OK)
		return DOCH_OK;
	else
	{
		DBG_PRINT_ERR(FLZONE_API, "DOCHAtaPassThrough(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));

		return rc;
	}

}

/*----------------------------------------------------------------------*/
/*                   D o c h S D K I n i t                              */
/*                                                                      */
/* Initializes the DOCH system, sockets and timers.	                    */
/*                                                                      */
/* Parameters:                                                          */
/*      None                                                            */
/*                                                                      */
/* Returns:                                                             */
/*      DOCH_Error        : 0 on success, otherwise failed              */
/*----------------------------------------------------------------------*/

DOCH_Error DochSDKInit(void)
{
    DOCH_Error rc = DOCH_OK;

	/*Set global variable values */
	tffsset(gDochVerifyWrite, 0, sizeof(gDochVerifyWrite));
	tffsset(gDochIrqEnabled,  0, sizeof(gDochIrqEnabled));

	gAccessLayerType   = DOCH_AL_NOR;
	gATANoTimeout      = 0;
	gAutoDPDByHost     = 0;
	gDochAccessNanosec = DOCH_ACCESS_NANOSEC;

	/*Protect against access time set to 0*/
	if(gDochAccessNanosec == 0)
		return DOCH_BadParameter;

	/*Set DPD settings to defaults*/
    gDpdSettings.activeMode		= DOCH_DPD_DEFAULT_ACTIVE_MODE;
	gDpdSettings.inActiveMode	= DOCH_DPD_DEFAULT_INACTIVE_MODE;
	gDpdSettings.timeOut		= DOCH_DPD_DEFAULT_DPD_TIMEOUT;

	/* Set ConfigHW items to defaults */
	if(gConfigHWInitDone != DOCH_GLOBAL_BOOL_PATTERN)
		setConfigHWDefaults();

#ifdef DOCH_DMA_CONFIG
	/* DMA channel was not opened yet... */
	gDMAChannelOpen = 0;
#endif /*DOCH_DMA_CONFIG*/

	/* Init system related */
    DOCH_SYS_FUNC_INIT;

	/* Register DOCH components */
	rc = (DOCH_Error)flRegisterDochParams();
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "flRegisterDochParams(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
		return rc;
	}

	/* Enable interrupts (if requested) */
	/*rc = dochEnableATAInterrupt(0, DOCH_IRQ_RB_INIT(0), 0);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "dochEnableATAInterrupt(): failed with status: ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x \r\n"), rc));
		return rc;
	}
	*/

	/*Set global variable*/
	gSdkInitDone = DOCH_GLOBAL_BOOL_PATTERN;

	return DOCH_OK;
}

/*----------------------------------------------------------------------*/
/*                     D o c h S D K E x i t                            */
/*                                                                      */
/* If the application ever exits, DochSDKExit should be called before	*/
/* exit.																*/
/* DochSDKExit flushes all buffers, closes all open files, powers down	*/
/* the sockets and removes the interval timer.                          */
/*                                                                      */
/* Parameters:                                                          */
/*      None                                                            */
/*                                                                      */
/* Returns:                                                             */
/*      Nothing                                                         */
/*----------------------------------------------------------------------*/
void DochSDKExit(void)
{
	#ifdef DOCH_DMA_CONFIG
	DMA_Params_S dmaParams;
	#endif /*DOCH_DMA_CONFIG*/

	doch_release_socket(0);

	#ifdef DOCH_DMA_CONFIG
	/*Free the DMA channel*/
	dmaParams.bOpType = DOCH_DMA_FREE_CHANNEL;
	DOCH_DMA_CONFIG(&dmaParams);
	if(dmaParams.fDmaStatus != 0)
	{
		DBG_PRINT_ERR(FLZONE_API, "DochSDKExit(): DOCH_DMA_FREE_CHANNEL Failed\r\n");
	}
	#endif /*DOCH_DMA_CONFIG*/

	DOCH_SYS_FUNC_RELEASE;

	gConfigHWInitDone = 0;
    gSdkInitDone      = 0;
}

/*----------------------------------------------------------------------*/
/*                 D o c h R e g i s t e r S o c k e t  				*/
/*                                                                      */
/* Register a DOCH socket												*/
/*                                                                      */
/* Parameters:                                                          */
/*      dwAddress		: Address were DOCH socket is located	        */
/*                                                                      */
/* Returns:                                                             */
/*      DOCH_Error        : 0 on success, otherwise failure             */
/*----------------------------------------------------------------------*/
DOCH_Error DochRegisterSocket(FLDword dwAddress)
{
	DOCH_Error      rc;
	DOCH_InitSocket initSocket;

	tffsset(&initSocket, 0, sizeof(initSocket));
	initSocket.nDeviceAddress = dwAddress;

	rc = doch_init_socket(0,initSocket);
	if(rc != DOCH_OK)
	{
		DBG_PRINT_ERR(FLZONE_API, "DochSDKInit(): doch_init_socket Failed on socket #");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("%d "), 0));
		DBG_PRINT_ERR(FLZONE_API, "with status : ");
		DBG_PRINT_ERR_PRM(FLZONE_API, (FLTXT("0x%x "), rc));

		return DOCH_AdapterNotFound;
	}

	return DOCH_OK;
}

#ifdef __cplusplus
}
#endif
