/******************************************************************************
**  COPYRIGHT  2007 Marvell Inernational Ltd.
**  All Rights Reserved
**
**  COPYRIGHT (C) 2006, Intel Corporation.
**
**  This software as well as the software described in it is furnished under
**  license and may only be used or copied in accordance with the terms of the
**  license. The information in this file is furnished for informational use
**  only, is subject to change without notice, and should not be construed as
**  a commitment by Intel Corporation. Intel Corporation assumes no
**  responsibility or liability for any errors or inaccuracies that may appear
**  in this document or any software that may be provided in association with
**  this document.
**  Except as permitted by such license, no part of this document may be
**  reproduced, stored in a retrieval system, or transmitted in any form or by
**  any means without the express written consent of Intel Corporation.
**
**  FILENAME:	MSYS_INTERFACE.c
**
**  PURPOSE: 	Interface layer between bootrom/DKB/OBM and m-sys DOC devices
**
******************************************************************************/
#include "typedef.h"
#include "errors.h"
#include "msys.h"
#include "misc.h"

//====================================================================
// mDOC_GetTIM
//
//     Retrieves the TIM data block from the mDOC device
//
//====================================================================
UINT_T mDOC_GetTIM( UINT8_T *pAddress, UINT_T timByteCount )
{
  #define MDOC_TIM_SECTOR_OFFSET 0   // For now, we hard code these values
  #define MDOC_TIM_PARTITION 2

  // Read the TIM from fixed location (partition 2, sector 0)
  //-------------------------------------------------------------------
  if( mDOC_ReadSectors( MDOC_TIM_SECTOR_OFFSET, timByteCount, pAddress, MDOC_TIM_PARTITION ) == M_SYS_TRUE )
    return NoError;
  else
    return ReadError;

}


//=======================================================================================
// mDOC_GetImage
//          Retrieve an image from the mDOC device
//
//=======================================================================================
UINT_T mDOC_GetImage( UINT8_T *pTargetAddress,
                       UINT_T sectorOffset,
                        UINT_T byteCount,
                         UINT8_T partition )
{
  if( mDOC_ReadSectors( sectorOffset, byteCount, pTargetAddress, partition ) == M_SYS_TRUE )
    return NoError;
  else
    return ReadError;
}

//=======================================================================================
// mDOC_WriteImage
//          Write an image to the mDOC device
//
//=======================================================================================
UINT_T mDOC_WriteImage( UINT8_T *pSourceAddress,
                         UINT_T sectorOffset,
                          UINT_T byteCount,
                           UINT8_T partition )
{
  if( mDOC_WriteSectors( sectorOffset, byteCount, pSourceAddress, partition ) == M_SYS_TRUE )
    return NoError;
  else
    return WriteError;
}

//=======================================================================================
// mDOC_ReadOtp
//          Read from the OTP area of the mDOC device.
//          The OTP area is treated as a "logical byte addressable" region
//          for consistency between differing devices.
//
//          otpAddress is logical zero-based address in the OTP region of the device
//
//          We assume that this is a sufficiently small region so as not to overflow
//          a 32 bit address...
//
//=======================================================================================
#define MDOC_OTP_PARTITION 1
#define MDOC_MAX_OTP_SECTOR 29 /* 15k area of 512 byte sectors -> sectors 0->29 */
UINT_T mDOC_ReadOtp( UINT8_T *pTargetBuffer,
                       UINT_T otpAddress,
                         UINT_T byteCount )
{
  UINT_T retVal;
  UINT_T sectorOffset;

  // This is the offset within the shim buffer "equivalent" to the OTP address
  //--------------------------------------------------------------------------
  UINT_T sBuffLogicalOffset;

  UINT_T bytesRead;
  unsigned char shimBuffer[DOCH_SECTOR_SIZE];	// limit to single sector

  while( byteCount )
  {
    sectorOffset = otpAddress / DOCH_SECTOR_SIZE;

	if( sectorOffset > MDOC_MAX_OTP_SECTOR )
	  return ReadError;

    retVal = mDOC_ReadSectors( sectorOffset, sizeof(shimBuffer), shimBuffer, MDOC_OTP_PARTITION );

    if( retVal != M_SYS_TRUE )
      return ReadError;

    sBuffLogicalOffset = (otpAddress -(sectorOffset * DOCH_SECTOR_SIZE));

    bytesRead = sizeof(shimBuffer) - sBuffLogicalOffset;
	bytesRead = (bytesRead >= byteCount) ? byteCount : bytesRead;
    memcpy( pTargetBuffer, shimBuffer + sBuffLogicalOffset, bytesRead);
    pTargetBuffer += bytesRead;
    otpAddress    += bytesRead;
    byteCount     -= bytesRead;
  }

  return NoError;
}
