/*
 * Copyright 2004-2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident   "@(#)utadem.c	1.18    05/12/13 SMI"

#ifdef _SCCSID
static char *_SCCSid = "@(#)utadem.c	1.18     05/12/13 SMI";
#endif  /* defined _SCCSID */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/major.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>

#include "utadem.h"

/*
 * utadem - Audio Device Emulation
 *
 * Overview:
 * ---------
 *
 * This is a simple pseudo-audio device.  It tastes like an OSS device, but
 * really spits data up to a daemon that deals with it as it sees fit.
 *
 * This driver takes full advantage of OSS's DSP_CAP_* values.  We don't
 * support DSP_CAP_REALTIME, DSP_CAP_TRIGGER, DSP_CAP_MMAP, or DSP_CAP_BIND.
 * This makes things pretty simple.  If we *had* to support some of those, we
 * could.
 *
 *
 * Buffering:
 * ----------
 *
 * The buffering for this driver is very simple, for now.  Each session has
 * an array[UTA_MAX_NUM_FRAGMENTS] of buffers. The buffer size is dynamically
 * calculated based on  the number of channels, sampling rate and sample
 * format. The application can override the default fragment size and number
 * of fragments values by calling SETFRAGMENT ioctl.
 *
 *
 * Emulation:
 * ----------
 *
 * To a user application, this driver tastes just like an OSS audio device,
 * albeit a limited one.  To the daemon, it tastes completely different.
 *
 * Abnormalities for the user side:
 *
 * - On open(), the filename of the device node is examined.  If the device
 * is named according to the regex /ut.*-[0-9]+/, the driver will attempt to
 * connect to the uta_session with the id corresponding to the final integer
 * of the filename.  If this session does not exist, or the device node is
 * not properly named, the open() will fail.  If the open() succeeds,
 * private_data field of the struct file is pointed at a uta_connect, and
 * thereby a uta_session.  This process occurs for both dsp and mixer
 * devices.
 *
 * Abnormalities for the daemon side:
 *
 * - The daemon can get information about the major and minor numbers of the
 * utadem device from the file /proc/driver/utadem.
 *
 * - On open(), the filename of the device node is examined, as in the user
 * case.  If the session does not exist, or the filename is not formatted as
 * expected, the open will succeed if the caller has CAP_SYS_ADMIN, but the
 * uta_connect will not point to a uta_session.
 *
 * - The daemon can request the UTADEM_VERSION of the driver by calling the
 * UTAIO_GETVERSION ioctl() on the device file descriptor, with an unsigned
 * long pointer as argument.  The UTADEM_VER_MAJOR() and UTADEM_VER_MINOR()
 * macros can be used on the result.
 *
 * - The daemon can initiate a session by calling the UTAIO_NEWSESSION
 * ioctl(), which will return the new session id, or -ve for failure.
 * UTAIO_NEWSESSION takes as an argument the uid of the user for whom to
 * account this session.
 *
 * - The daemon can set the session's capabilities (sample rate, data
 * formats, etc.) by calling the UTAIO_SETDCAPS ioctl(), which takes a
 * struct utadem_dcaps structure as argument.  Once SETDCAPS has been
 * called, the driver will only notify the daemon of settings changes
 * within those capabilities.  Any user-requested setting that falls outside
 * those capabilities (e.g. sample rate) will fail without notifying the
 * daemon.
 *
 * - The daemon should then wait for messages from the driver, via the
 * UTAIO_GETMSG ioctl().  This ioctl() will block until a message arrives,
 * unless the O_NONBLOCK flag has been set on the file descriptor.  Each
 * call to UTAIO_GETMSG will retrieve a single utadem_msg structure.  Each
 * message from the driver is the result of some operation from the user
 * (e.g. start playback, reset, set format).
 *
 *
 * Message Protocol:
 * ----------------
 *
 * - UTADEM_MSG_JOINED: A user has joined the session whose session-id is
 * specified by the argument.
 *
 * - UTADEM_MSG_LEFT: The user has left the session whose session-id is
 * specified by the argument.
 *
 * - UTADEM_MSG_PLAY: The driver is indicating that there is data available
 * for the daemon to consume via read().  Once playback has started, the
 * daemon should periodically call read() to fetch the played data from the
 * user.  As always, read() may return short, and that is not an error.  if
 * read() returns 0, there may have been a buffer underrun, and the daemon
 * should stop playback until another UTADEM_MSG_PLAY is received.  The
 * driver may deliver more than one UTADEM_MSG_PLAY.  If the daemon is
 * already playing, the message should be ignored.
 *
 * - UTADEM_MSG_RECORD: The driver is indicating that the user is waiting
 * for data to consume.  The daemon can provide this data via write().  Once
 * recording has started, the daemon should periodically call write() to
 * keep to provide new data to the user.  As always, write() may return
 * short, and that is not an error.  If write() returns 0, there may have
 * been a buffer overrun and the daemon should stop recording until another
 * UTADEM_MSG_RECORD is received.  The driver may deliver more than one
 * UTADEM_MSG_RECORD.  If the daemon is already RECORDING, the message
 * should be ignored.
 *
 * - UTADEM_MSG_SETFORMAT: The user has requested a specific audio sample
 * format.  The driver will only send UTADEM_MSG_SETFORMAT if the user has
 * requested a format specified in the daemon's passed-in capabilities
 * (see UTAIO_SETDCAPS).
 *
 * - UTADEM_MSG_SETRATE: The user has requested a specific audio sample
 * rate.  The driver will only send UTADEM_MSG_SETRATE if the user has
 * requested a rate within the range in the daemon's passed-in capabilities
 * (see UTAIO_SETDCAPS).
 *
 * - UTADEM_MSG_SETCHANNELS: The user has requested a specific number of
 * audio channels.  The driver will only send UTADEM_MSG_SETCHANNELS if the
 * user has requested a channel count specified in the daemon's passed-in
 * capabilities (see UTAIO_SETDCAPS).
 *
 * - UTADEM_MSG_MIXER: The user has changed the level of a mixer device.
 * The channel number of the mixer device is encoded in the 16
 * least-significant bits of the argument.  The value is encoded in the 16
 * most-significant bits of the argument.  Stereo channels have two values
 * encoded in the 16 bits - the left channel is the least significant byte,
 * and the right channel is the most significant byte.  The argument comes
 * from user-space as a signed int, so the most-significant bit of each
 * value must be off.
 *
 * - UTADEM_MSG_SETINSRC: The user has selected an input source.  The
 * argument of this message is the bitmask of input sources selected.
 *
 * - UTADEM_MSG_SETOUTSRC: The user has selected an output source.  The
 * argument of this message is the bitmask of output sources selected.
 *
 * - UTADEM_MSG_RESET: The user or the driver has requested an immediate
 * reset of the pseudo-device.  Upon receiving this message, the daemon is
 * expected to immediately stop playback and recording.  If the message
 * argument field is non-zero, the daemon must acknowledge the reset by
 * calling the UTAIO_ACKRESET ioctl().
 *
 * - UTADEM_MSG_SYNC: The user has advised the driver that there is going to
 * be a pause in playback.  The daemon is expected to immediately stop
 * recording, but not playback.  Once recording is stopped, the daemon must
 * acknowledge the sync by calling the UTAIO_ACKSYNC ioctl() if the message
 * argument field is non-zero.  The daemon should continue playback as
 * normal until a 0-length read.
 *
 * - UTADEM_MSG_SETFRAGSIZE: The user has requested a specific fragment 
 * size.  The driver will always send UTADEM_MSG_SETFRAGSIZE. 
 *
 * - If the user disconnects during playback or record, the daemon will get
 * a UTADEM_MSG_RESET with an argument of 0.
 *
 *
 * Locking:
 * --------
 * - The list of active sessions is protected by sesslist_lock.  This lock
 * is held when searching, adding to, or removing from the session_list.
 *
 * - The sessbits array is protected by sessbits_lock.  This lock is held
 * when dealing with the sessbits structures.
 *
 * - Each uta_session is refcounted.  When the refcount hits 0, the session
 * is released.  The refcounts are protected by the global sesslist_lock.
 * The refcount is incremented before the session is returned from the list
 * search (sesslist_lock is held), and the lock is taken before decrementing
 * the refcount, to eliminate any races with joiners.
 *
 * - Each uta_session has a flags field which is an array of atomic
 * bit-flags.  These flags indicate the connectedness of the user and daemon
 * ends of the pipeline.  Also, there is a user-lock bit.  This flag
 * indicates that a user has connected, and that the connection is being
 * established.  Establishing the connection may block, so a spinlock is not
 * a viable choice.  We want racey connections (two users connecting at the
 * same time) to allow one caller and fail the other, so a semaphore is
 * sub-optimal.
 *
 * - Each uta_session has a msg_lock field, which is a spinlock protecting
 * the per-session messages list.  This lock must be held whenever the
 * messages list is examined or modified.
 *
 * - Each uta_session has a reset_sem.  When a SNDCTL_DSP_RESET,
 * SNDCTL_DSP_POST, or SNDCTL_DSP_SYN has been issued by the user, the
 * calling process blocks on reset_sem until the daemon acknowledges that
 * the operation has been completed.
 *
 * - Each uta_io has a waitqueue.  The user blocks on these waitqueues when
 * waiting to move forward with IO.  The daemon wakes up these waitqueues
 * when it completes IOs.  The daemon does not block.  If the buffer has not
 * been transferred to the daemon when the daemon needs it, it is an xrun.
 * Ownership is controlled by an atomic_t field of the uta_buffer structure.
 *
 * - The main fops entry points have some weird re-entrancy requirements.
 * The user code path is single threaded - all the user fops helpers do
 * down_write() on the conn->syscall_rwsem.  The daemon side is re-entrant
 * between entry points, but each entry may only be active once at any given
 * time, except ioctl().  Both the read and write entry points may be active
 * at the same time, without affecting anything else, except a few specific
 * ioctl() blocks.  There is a conn->read_sem and a conn->write_sem which
 * make callers of read() and write() single-thread.  Further, each ioctl
 * block that may operate in parallel with other code does
 * down_read(conn->syscall_lock), while ioctl blocks that need exclusive
 * access do down_write().  Callers can be inside read, write, and several
 * ioctl blocks at the same time.  This makes a multi-threaded daemon easier
 * - no polling of UTAIO_GETMSG needed.
 *
 *
 * Utadem ioctl()s
 * ------------------------
 *
 * - UTAIO_GETVERSION: This ioctl call returns the UTADEM_VERSION of the
 * driver in the unsigned long pointer arg.
 *
 * - UTAIO_NEWSESSION: This ioctl call creates and attaches to a new
 * uta_session.  The return value of this call is the new session ID, or
 * negative on error.
 *
 * - UTAIO_GETMSG: This ioctl is called to retrieve the next control
 * message from the driver.  If no message is currently available, the
 * caller will be blocked until a message arrives, unless the O_NONBLOCK is
 * set.  The argument is a pointer to a struct utadem_msg, which will be
 * populated by the driver.
 *
 * - UTAIO_SETDCAPS: This ioctl call is used by the daemon to set the
 * capabilities of the driver (channels, formats, rate, etc).  The argument
 * is a pointer to a struct utadem_dcaps.
 *
 * - UTAIO_SETMIXER: This ioctl call is used by the daemon to set up the state
 * of the driver's mixer configuration.  The argument is a pointer to an
 * array of size UTADEM_MIXER_NDEVS of struct utadem_mixdev.
 *
 * - UTAIO_ACKRESET: This ioctl is called when the daemon has received a
 * UTADEM_MSG_RESET message with a non-zero argument.  After stopping the
 * record and playback threads, the daemon should call this.
 *
 * - UTAIO_ACKSYNC: This ioctl is called when the daemon has received a
 * UTADEM_MSG_SYNC message.  After stopping the record thread, the daemon
 * should call this.
 *
 * - UTAIO_UNDERRUN/UTAIO_OVERRUN: These ioctls let the daemon indicate an
 * underrun or overrun condition to the driver.  If the daemon was expecting
 * to read data but got a 0-length read(), it is an underrun.  If the daemon
 * was expecting to be able to write data, but got a 0-length write(), it is
 * an overrun.
 *
 */

/*
 * Things for a later version:
 *
 * - do multi-client mixing
 * - mmap()
 * - user-defined fragment management
 * - triggers
 */

// FIXME: ----------------------------------------------
// UTADEM_MSG_BLKSIZE?
// diconnected msg
// AUDIT:, LOCK: and FIXME: notes
// only send one play/record msg?  UTAIO_PLAYING(1) etc?
// use max_per_user - require sess->owner field or a new hash of
// uid=>count
// sndstat? /dev/audio and dsp16?
// must support ulaw to be OSS
// sane default buffering
// need to cleanup anything else on unload?

/* constants */

#define	UTA_MAX_NUM_FRAGMENTS	64
#define	UTA_MIN_NUM_FRAGMENTS	2
#define	UTA_DEF_NUM_FRAGMENTS	10
#define	UTA_MAX_BUF_LEN_SEC	1  // 1 sec worth of data

/* debugging stuff */
static int debug;
#define	DPRINTF(lvl, fmt, args...)	do { \
	if (debug && debug >= lvl) \
		printk(KERN_DEBUG "utadem debug: " fmt, ##args); \
} while (0)


struct uta_buffer {
	int offset;		/* offset into this buf */
	int bytes;		/* size of data in this buf */
	atomic_t owner;		/* who owns this buf - daemon or user? */
	unsigned char data[0];  /* This should be the last member */
};

struct uta_io {
	int user;		/* which buffer the user is on */
	int daemon;		/* which buffer the daemon is on */
	int fragment_size;  	/* fragment size currently being used */
	int num_fragments;  	/* No of fragments currently used */
	wait_queue_head_t wait; /* for IO blocking */
	struct uta_buffer *fragments[UTA_MAX_NUM_FRAGMENTS];
};

struct uta_msg {
	uint32_t msgtype;
	uint32_t arg;
	struct list_head list;
};

struct uta_mixdev {
	uint8_t channels;	/* # of audio channels on this device */
	uint16_t level;		/* the current value of this channel */
	char isrecsrc:1;	/* is it a recording source? */
	char isoutsrc:1;	/* is it a playback source? */
};

struct uta_session {
	int id;			/* the session ID */
	uid_t owner;		/* the uid of the owning user */
	unsigned char mode;	/* user's open() mode */
	unsigned long flags;	/* who is connected, etc */
	int refcount;		/* when this is 0, the session can be freed */
	struct list_head list;	/* global session list_head */
	struct uta_io ios[2];	/* I/O streams */
	struct list_head messages;	/* protocol messages */
	spinlock_t msg_lock;	/* protect the messages */
	struct semaphore msg_sem;	/* producer/consumer sync */
	struct semaphore reset_sem;	/* sync device resets */

	int dformats;		/* daemon's supported  formats (bitmask) */
	uint32_t dchannels;	/* daemon's supported channels (bitmask) */
	int drate_min;		/* daemon's min supported rate (inclusive) */
	int drate_max;		/* daemon's max supported rate (inclusive) */
	int format;		/* current format */
	int channels;		/* current # of channels */
	int rate;		/* current sample rate */

	int recsrc;		/* bitmask of recording sources */
	int outsrc;		/* bitmask of playback sources */
	struct uta_mixdev mixer[UTADEM_MIXER_NDEVS]; /* mixer state */
	int mix_modcount;	/* count of mixer modifications */

	int underruns;		/* statistics */
	int overruns;
};

/* I/O indices */
#define	IO_RECORD	0
#define	IO_PLAY		1

#define	PLAYBACK	1
#define	RECORD		2

/* mode bits */
#define	MODE_RECORD	(1 << IO_RECORD)
#define	MODE_PLAY	(1 << IO_PLAY)

struct uta_connect {
	struct uta_session *session;
	int role;		/* am I the user or the daemon? */
	struct semaphore read_sem;	/* one read() user at a time */
	struct semaphore write_sem;	/* one write() user at a time */
	struct rw_semaphore syscall_rwsem; /* block non-safe re-entrances */
};

/* role values */
#define	UTA_NONE		-1
#define	UTA_DAEMON		0
#define	UTA_USER		1
#define	UTA_MIXER		2

/* the global list of active sessions (struct uta_session) */
static LIST_HEAD(session_list);
static spinlock_t sesslist_lock = SPIN_LOCK_UNLOCKED;

/* the global session bitmask */
static int sessbits_count; /* count of ulongs in the array */
static unsigned long *session_bits;
static spinlock_t sessbits_lock = SPIN_LOCK_UNLOCKED;

/* module parameters */
static int max_sessions = 2048;
// static int max_per_user = 64;  /* not used at this time */

/* the global fragment buffer values */
static int fragment_buf_size;
static int total_fragments;


/* functions */
static int new_sessionid(void);
static void release_sessionid(int sessid);
static int do_newsession(struct uta_connect *conn, uid_t owner);
static int do_getmsg(struct uta_session *sess, struct utadem_msg *umsg,
    int nonblock);
static int do_setdcaps(struct uta_session *sess, struct utadem_dcaps *udcaps);
static int do_setmixer(struct uta_session *sess, struct utadem_mixdev *mixdevs);
static int do_ackreset(struct uta_session *sess, int whichio);
static int do_getxspace(struct uta_session *sess, int idx,
    struct audio_buf_info *uinfo);
static struct uta_session *find_session(int sessid);
static int join_session_dsp(struct uta_connect *conn, int sessid, int record,
    int play);
static int join_session_mix(struct uta_connect *conn, int sessid);
static void end_session(struct uta_session *sess);
static int set_fragment_buf_size(int size, int seqlen_sec);
static int alloc_ios(struct uta_session *sess, int whichio);
static void free_ios(struct uta_session *sess);
static void wakeup_ios(struct uta_session *sess);
static int xfer_buf_to_daemon(struct uta_session *sess, int io_idx);
static int xfer_buf_to_user(struct uta_session *sess, int io_idx);
static void free_all_msgs(struct uta_session *sess);
static int device_reset(struct uta_session *sess, int ack);
static int device_sync(struct uta_session *sess, int ack);
static int send_msg(struct uta_session *sess, int msgid, int arg);

/* fops helpers */
static int user_read(struct file *file, char *ubuf, size_t size);
static int user_write(struct file *file, const char *ubuf, size_t size);
static int daemon_read(struct file *file, char *ubuf, size_t size);
static int daemon_write(struct file *file, const char *ubuf, size_t size);
static unsigned int user_poll(struct file *file,
    struct poll_table_struct *wait);
static int user_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);
static int daemon_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);

static int oss_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);
static int mix_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);
static int mix_read_ioctl(struct uta_session *sess, int channel, int *arg);
static int mix_write_ioctl(struct uta_session *sess, int channel, int *arg);

/* this is called with the session unlisted or with sesslist_lock held */
#define	ref_session(sess)	do { \
	sess->refcount++; \
	DPRINTF(3, "sess %d refcount++ = %d\n", sess->id, sess->refcount); \
} while (0)
/* need to lock outside end_session to avoid a race with join_session_dsp */
#define	unref_session(sess)	do { \
	spin_lock(&sesslist_lock); \
	sess->refcount--; \
	DPRINTF(3, "sess %d refcount-- = %d\n", sess->id, sess->refcount); \
	if (sess->refcount == 0) \
		end_session(sess); \
	spin_unlock(&sesslist_lock); \
} while (0)

/* some helpers for session state management - lockless is good */
#define	BIT_USER		0 /* user is connected */
#define	BIT_DAEMON		1 /* daemon is connected */
#define	BIT_ULOCK		2 /* a user connection is imminent */
#define	BIT_RESET		3 /* user wants a reset to happen */
#define	BIT_MIXER		4 /* mixer is connected */
#define	has_daemon(s)		test_bit(BIT_DAEMON, &(s)->flags)
#define	has_user(s)		test_bit(BIT_USER, &(s)->flags)
#define	is_resetting(s)		test_bit(BIT_RESET, &(s)->flags)
#define	set_has_daemon(s, v)	set_flag((s), BIT_DAEMON, (v))
#define	set_has_user(s, v)	set_flag((s), BIT_USER, (v))
#define	set_has_mixer(s, v)	set_flag((s), BIT_MIXER, (v))
#define	set_user_lock(s, v)	set_flag((s), BIT_ULOCK, (v))
#define	set_resetting(s, v)	set_flag((s), BIT_RESET, (v))
static inline int
set_flag(struct uta_session *sess, int bit, int val)
{
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	if (val)
		return (test_and_set_bit(bit, &sess->flags));
	else
		return (test_and_clear_bit(bit, &sess->flags));
}

/* find and allocate a new session ID */
static int
new_sessionid(void)
{
	int idx;
	int base = 0;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	spin_lock(&sessbits_lock);
	for (idx = 0; idx < sessbits_count; idx++) {
		if (session_bits[idx] != ~0UL) {
			int bit;
			bit = ffz(session_bits[idx]);
			DPRINTF(3, "setting sessbit %d:%d\n", idx, bit);
			__set_bit(bit, &session_bits[idx]);
			spin_unlock(&sessbits_lock);
			return (base + bit);
		}

		base += BITS_PER_LONG;
	}
	spin_unlock(&sessbits_lock);

	return (-1);
}

/* deallocate a sessionid */
static void
release_sessionid(int sessid)
{
	int idx;
	int bit;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sessid);

	BUG_ON(sessid >= max_sessions);

	idx = sessid / BITS_PER_LONG;
	bit = sessid % BITS_PER_LONG;

	DPRINTF(3, "clearing sessbit %d:%d\n", idx, bit);
	spin_lock(&sessbits_lock);
	clear_bit(bit, &session_bits[idx]);
	spin_unlock(&sessbits_lock);
}

/*
 * All the daemon-side ioctl helpers are protected from racing with each
 * other and read/write by daemon_ioctl, if needed.  User-side ioctl helpers
 * are protected from re-entrance by user_ioctl().
 */

static int
do_newsession(struct uta_connect *conn, uid_t owner)
{
	struct uta_session *sess;
	int sessid;
	int i;

	DPRINTF(2, "entering %s(owner=%d)\n", __FUNCTION__, owner);

	if (conn->session)
		return (-EBUSY);

	/* allocate a session ID */
	sessid = new_sessionid();
	if (sessid < 0) {
		DPRINTF(1, "can't get a new session ID\n");
		return (-EBUSY);
	}
	DPRINTF(3, "new sessionid = %d\n", sessid);

	/* allocate session struct */
	sess = kmalloc(sizeof (*sess), GFP_KERNEL);
	if (!sess) {
		DPRINTF(1, "can't allocate session\n");
		release_sessionid(sessid);
		return (-ENOMEM);
	}

	/* init the session */
	sess->id = sessid;
	sess->owner = owner;
	sess->mode = 0;
	sess->refcount = 0;
	sess->flags = 0;
	set_has_daemon(sess, 1);
	for (i = 0; i < 2; i++) {
		struct uta_io *io;
		int j;

		io = &sess->ios[i];
		io->user = io->daemon = io->fragment_size = 0;
		io->num_fragments = 0;
		init_waitqueue_head(&io->wait);
		for (j = 0; j < UTA_MAX_NUM_FRAGMENTS; j++)
			io->fragments[j] = NULL;
	}
	INIT_LIST_HEAD(&sess->messages);
	spin_lock_init(&sess->msg_lock);
	init_MUTEX_LOCKED(&sess->msg_sem);
	init_MUTEX_LOCKED(&sess->reset_sem);
	sess->dformats = 0;
	sess->dchannels = 0;
	sess->drate_min = sess->drate_max = 0;
	sess->format = 0;
	sess->channels = 0;
	sess->rate = 0;
	for (i = 0; i < UTADEM_MIXER_NDEVS; i++)
		sess->mixer[i].channels = 0;
	sess->mix_modcount = 0;
	sess->recsrc = sess->outsrc = 0;
	DPRINTF(3, "session %d created for user %d\n", sess->id, owner);

	/* increment the refcount */
	ref_session(sess);

	/* lastly, enlist the session */
	spin_lock(&sesslist_lock);
	list_add_tail(&sess->list, &session_list);
	spin_unlock(&sesslist_lock);

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_DAEMON;

	return (sessid);
}

static int
do_getmsg(struct uta_session *sess, struct utadem_msg *umsg, int nonblock)
{
	struct uta_msg *msg;
	struct utadem_msg tmp_umsg;
	struct list_head *item;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* get the msg_sem to see if there are any messages */
	if (nonblock) {
		if (down_trylock(&sess->msg_sem))
			return (-EAGAIN);
	} else {
		if (down_interruptible(&sess->msg_sem))
			return (-ERESTARTSYS);
	}

	spin_lock(&sess->msg_lock);
	item = sess->messages.next;
	list_del(item);
	spin_unlock(&sess->msg_lock);

	msg = list_entry(item, struct uta_msg, list);
	tmp_umsg.version = UTADEM_VERSION;
	tmp_umsg.msgtype = msg->msgtype;
	tmp_umsg.arg = msg->arg;

	kfree(msg);

	return (copy_to_user(umsg, &tmp_umsg, sizeof (tmp_umsg)));
}

static int
do_setdcaps(struct uta_session *sess, struct utadem_dcaps *udcaps)
{
	struct utadem_dcaps tmp_dcaps;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (copy_from_user(&tmp_dcaps, udcaps, sizeof (tmp_dcaps)))
		return (-EFAULT);

	/* copy the fields we need to store */
	// FIXME: what if we call this again later - does not reset
	sess->dformats = tmp_dcaps.formats;
	sess->dchannels = tmp_dcaps.channels;
	sess->drate_min = tmp_dcaps.rate_min;
	sess->drate_max = tmp_dcaps.rate_max;
	sess->format = tmp_dcaps.def_format;
	sess->channels = tmp_dcaps.def_channels;
	sess->rate = tmp_dcaps.def_rate;

	return (0);
}

static int
do_setmixer(struct uta_session *sess, struct utadem_mixdev *mixdevs)
{
	struct utadem_mixdev tmp_mixdev;
	int i;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	// FIXME: what if we call this again later - does not reset
	for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
		if (copy_from_user(&tmp_mixdev, mixdevs+i, sizeof (tmp_mixdev)))
			return (-EFAULT);
		sess->mixer[i].channels = tmp_mixdev.channels;
		if (tmp_mixdev.channels) {
			sess->mixer[i].level = tmp_mixdev.level;
			sess->mixer[i].isrecsrc = tmp_mixdev.insrc;
			if (tmp_mixdev.insrc)
				sess->recsrc |= 1<<i;
			sess->mixer[i].isoutsrc = tmp_mixdev.outsrc;
			if (tmp_mixdev.outsrc)
				sess->outsrc |= 1<<i;
		}
	}

	return (0);
}

/*
 * Note:
 * This does not race with anything.  By the time the daemon gets here, it
 * has received a request from the user to be here.  The user has blocked on
 * sess->reset_sem, or is racing there.  Therefore, we can muckety-muck all
 * we like.  Just be quick about it.
 */
static int
do_ackreset(struct uta_session *sess, int whichio)
{
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!is_resetting(sess))
		return (-EINVAL);

	/* reset all the ios */
	alloc_ios(sess, whichio);

	/* done resetting */
	set_resetting(sess, 0);

	/* wake up the user */
	up(&sess->reset_sem);

	return (0);
}


static int
do_getxspace(struct uta_session *sess, int idx, struct audio_buf_info *uinfo)
{
	struct audio_buf_info info;
	struct uta_io *io;
	int j, count = 0;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	// LOCK: sess->io_lock
	io = &sess->ios[idx];

	// Check how many fragment buffers are available for write
	int role = (idx == IO_RECORD) ? UTA_DAEMON : UTA_USER;
	for (j = 0; j < io->num_fragments; j++) {
		struct uta_buffer *buf = io->fragments[j];
		if (!buf)
			break;
		if (atomic_read(&buf->owner) == role)
			count++;
	}

	info.fragments = count;
	info.fragstotal = io->num_fragments;
	info.fragsize = io->fragment_size;

	// TBD check for partial fragments
	info.bytes =  count * io->fragment_size;

	DPRINTF(2, "Getxspace idx %d fragments %d fragsize %d bytes %d \n",
		idx, info.fragments, info.fragsize, info.bytes);
	return (copy_to_user(uinfo, &info, sizeof (info)));
}

/* this returns the session already ref'ed */
static struct uta_session *
find_session(int sessid)
{
	struct list_head *pos;
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	spin_lock(&sesslist_lock);
	list_for_each(pos, &session_list) {
		struct uta_session *sess;

		sess = list_entry(pos, struct uta_session, list);
		if (sess->id == sessid && has_daemon(sess)) {
			DPRINTF(3, "found session %d\n", sessid);
			ref_session(sess);
			spin_unlock(&sesslist_lock);
			return (sess);
		}
	}
	spin_unlock(&sesslist_lock);

	return (NULL);
}

static int
join_session_dsp(struct uta_connect *conn, int sessid, int record, int play)
{
	struct uta_session *sess;
	int ret;
	int whichio;

	DPRINTF(2, "entering %s(sessid=%d, record=%d, play=%d)\n",
	    __FUNCTION__, sessid, !!record, !!play);

	/* find the session */
	sess = find_session(sessid);
	if (!sess) {
		DPRINTF(3, "can't find session %d\n", sessid);
		ret = -ENODEV;
		goto out_err;
	}

	/* one user at a time - set this to indicate an unready user */
	if (set_user_lock(sess, 1)) {
		DPRINTF(3, "session %d already has a user\n", sessid);
		ret = -EBUSY;
		goto out_unref;
	}

	/* set up the user side of the session - we want to do this last */
	sess->mode = (record ? MODE_RECORD : 0) | (play ? MODE_PLAY : 0);
	whichio = (record ? RECORD : 0) | (play ? PLAYBACK : 0);

	/*
	 * Set the default fragment size and num of fragments
	 * The application can override the default fragment size and
	 * number of fragments values by calling SETFRAGMENT ioctl.
	 */
	int seqlen_sec = (sess->channels * sess->rate);
	if (sess->format == AFMT_S16_LE)
		seqlen_sec *= 2;

	total_fragments = UTA_DEF_NUM_FRAGMENTS;
	fragment_buf_size = ((seqlen_sec * UTA_MAX_BUF_LEN_SEC) /
			total_fragments);
	DPRINTF(4, "fragments=%d fragment_size=%d\n", total_fragments,
		fragment_buf_size);

	ret = alloc_ios(sess, whichio);
	if (ret < 0) {
		goto out_unlock;
	}
	sess->underruns = sess->overruns = 0;

	/* leave the session ref'ed - it is now ready */
	set_has_user(sess, 1);

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_USER;

	/* tell the daemon that we joined the session */
	send_msg(sess, UTADEM_MSG_JOINED, sessid);
	send_msg(sess, UTADEM_MSG_SETFRAGSIZE, fragment_buf_size);

	return (0);

out_unlock:
	set_user_lock(sess, 0);
out_unref:
	unref_session(sess);
out_err:
	return (ret);
}

static int
join_session_mix(struct uta_connect *conn, int sessid)
{
	struct uta_session *sess;
	int ret;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sessid);

	/* find the session */
	sess = find_session(sessid);
	if (!sess) {
		DPRINTF(3, "can't find session %d\n", sessid);
		ret = -ENODEV;
		goto out_err;
	}

	/* one mixer at a time */
	if (set_has_mixer(sess, 1)) {
		DPRINTF(3, "session %d already has a mixer\n", sessid);
		ret = -EBUSY;
		goto out_unref;
	}

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_MIXER;

	return (0);

out_unref:
	unref_session(sess);
out_err:
	return (ret);
}

/*
 * do the business of deallocating and delisting a session
 */
static void
end_session(struct uta_session *sess)
{
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	list_del(&sess->list);
	release_sessionid(sess->id);
	free_ios(sess);
	free_all_msgs(sess);

	kfree(sess);
}

static void
init_one_buf(struct uta_buffer *buf, int mode)
{
//	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	if (mode == MODE_RECORD) {
		/* record buffers start out owned by the daemon */
		atomic_set(&buf->owner, UTA_DAEMON);
	} else if (mode == MODE_PLAY) {
		/* play buffers start out owned by the user */
		atomic_set(&buf->owner, UTA_USER);
	} else {
		return;
	}
	buf->offset = 0;
	buf->bytes = 0;
	memset(buf->data, 0, sizeof (buf->data));
}

static void
init_one_io(struct uta_io *io)
{
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	io->user = io->daemon = 0;
	init_waitqueue_head(&io->wait);
}

static int
set_fragment_buf_size(int size, int seqlen_sec)
{
	/*
	 * Assume 16bit precision so that 5ms corresponds to 80 bytes. 
	 * For buffer size request of 128 bytes or less assign 80 bytes.
	 * For buffer size request of 256 bytes asssign 160 bytes.
	 * For buffer size request of 512 bytes assign 480 bytes.
	 * for buffer size request of 1024 bytes or greater assign requested 
	 * size.
         */
	if (size < (seqlen_sec / 100))
		size = seqlen_sec / 200;
	else if (size < (seqlen_sec * 3 / 100))
		size = seqlen_sec / 100;
	else if (size < (seqlen_sec / 20))
		size = seqlen_sec * 3 / 100;

	return size;
}

/* initialize buffers for a session */
static int
alloc_ios(struct uta_session *sess, int whichio)
{
	int r = 0;
	int p = 0;
	int seqlen_sec, max_bufsz;
	int tmp_buf_size;
	DPRINTF(2, "entering %s(sessid=%d, whichio=%d)\n",
		__FUNCTION__, sess->id, whichio);

	free_ios(sess);

	seqlen_sec = (sess->channels * sess->rate);
	if (sess->format == AFMT_S16_LE)
		seqlen_sec *= 2;

	tmp_buf_size = set_fragment_buf_size(fragment_buf_size, 
						  seqlen_sec);
	if (tmp_buf_size != fragment_buf_size) {
		send_msg(sess, UTADEM_MSG_SETFRAGSIZE, tmp_buf_size);
	}
	fragment_buf_size = tmp_buf_size;

	if (total_fragments < UTA_MIN_NUM_FRAGMENTS)
		total_fragments = UTA_MIN_NUM_FRAGMENTS;
	else if (total_fragments > UTA_MAX_NUM_FRAGMENTS)
		total_fragments = UTA_MAX_NUM_FRAGMENTS;

	max_bufsz = seqlen_sec * UTA_MAX_BUF_LEN_SEC;
	if ((total_fragments * fragment_buf_size) > max_bufsz)
		total_fragments = max_bufsz / fragment_buf_size;

	DPRINTF(4, "total_fragments=%d fragment_buf_size=%d\n",
	    total_fragments, fragment_buf_size);

	if ((sess->mode & MODE_RECORD) && ((whichio & RECORD) == RECORD)) {
		struct uta_io *io = &sess->ios[IO_RECORD];

		io->num_fragments = total_fragments;
		io->fragment_size = fragment_buf_size;
		for (r = 0; r < io->num_fragments; r++) {
			struct uta_buffer *buf;

			buf = kmalloc(sizeof (struct uta_buffer) +
					io->fragment_size, GFP_KERNEL);
			if (!buf)
				goto recbuf_err;
			init_one_buf(buf, MODE_RECORD);
			io->fragments[r] = buf;
		}
		init_one_io(io);
	}

	if ((sess->mode & MODE_PLAY) && ((whichio & PLAYBACK) == PLAYBACK)) {
		struct uta_io *io = &sess->ios[IO_PLAY];

		io->num_fragments = total_fragments;
		io->fragment_size = fragment_buf_size;
		for (p = 0; p < io->num_fragments; p++) {
			struct uta_buffer *buf;

			buf = kmalloc(sizeof (struct uta_buffer) +
					io->fragment_size, GFP_KERNEL);
			if (!buf)
				goto playbuf_err;
			init_one_buf(buf, MODE_PLAY);
			io->fragments[p] = buf;
		}
		init_one_io(io);
	}

	return (0);

playbuf_err:
	DPRINTF(2, "%s playbuf_err", __FUNCTION__);
	for (; p > 0; p--) {
		kfree(sess->ios[IO_PLAY].fragments[p-1]);
	}

recbuf_err:
	DPRINTF(2, "%s recbuf_err", __FUNCTION__);
	for (; r > 0; r--) {
		kfree(sess->ios[IO_RECORD].fragments[r-1]);
	}

	/* if we got here, something failed to alloc */
	return (-ENOMEM);
}

static void
free_one_io(struct uta_io *io)
{
	int i;
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	for (i = 0; i < io->num_fragments; i++) {
		if (io->fragments[i]) {
			kfree(io->fragments[i]);
			io->fragments[i] = NULL;
		}
	}
}

static void
free_ios(struct uta_session *sess)
{
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	if (sess->mode & MODE_RECORD)
		free_one_io(&sess->ios[IO_RECORD]);
	if (sess->mode & MODE_PLAY)
		free_one_io(&sess->ios[IO_PLAY]);
}

static void
wakeup_ios(struct uta_session *sess)
{
	int i;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	for (i = 0; i < 2; i++)
		wake_up_all(&sess->ios[i].wait);
}

/* return true if the buffer was transferred, otherwise 0 */
static int
xfer_buf_to_daemon(struct uta_session *sess, int io_idx)
{
	struct uta_io *io = &sess->ios[io_idx];
	struct uta_buffer *buf = io->fragments[io->user];

	DPRINTF(2, "entering %s(sessid=%d, io_idx=%d, io->user=%d)\n",
		__FUNCTION__, sess->id, io_idx, io->user);

	/* don't send empty buffers */
	if (!buf || (io_idx == IO_PLAY && buf->offset == 0))
		return (0);

	/* give it to the daemon */
	buf->offset = 0;
	atomic_set(&buf->owner, UTA_DAEMON);

	/* move on to the next buffer */
	io->user++;
	if (io->user == io->num_fragments)
		io->user = 0;

	return (1);
}

/* return true if the buffer was transferred, otherwise 0 */
static int
xfer_buf_to_user(struct uta_session *sess, int io_idx)
{
	struct uta_io *io = &sess->ios[io_idx];
	struct uta_buffer *buf = io->fragments[io->daemon];

	DPRINTF(2, "entering %s(sessid=%d, io_idx=%d)\n", __FUNCTION__,
	    sess->id, io_idx);

	/* don't send empty buffers */
	if (!buf || (io_idx == IO_RECORD && buf->offset == 0))
		return (0);

	/* give it to the user */
	buf->offset = 0;
	atomic_set(&buf->owner, UTA_USER);
	wake_up_interruptible(&io->wait);

	/* move on to the next buffer */
	io->daemon++;
	if (io->daemon == io->num_fragments)
		io->daemon = 0;

	return (1);
}

static void
free_all_msgs(struct uta_session *sess)
{
	struct list_head *tmp, *tmp2;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	spin_lock(&sess->msg_lock);
	list_for_each_safe(tmp, tmp2, &sess->messages) {
		struct uta_msg *msg;
		msg = list_entry(tmp, struct uta_msg, list);
		list_del(tmp);
		kfree(msg);
	}
	spin_unlock(&sess->msg_lock);
}

static int
device_reset(struct uta_session *sess, int ack)
{
	int ret;

	DPRINTF(2, "entering %s(sessid=%d, ack=%d)\n", __FUNCTION__,
	    sess->id, ack);

	if (!has_daemon(sess))
		return (0);

	if (ack)
		set_resetting(sess, 1);
	ret = send_msg(sess, UTADEM_MSG_RESET, ack);
	if (ack)
		down(&sess->reset_sem);

	return (ret);
}

static int
device_sync(struct uta_session *sess, int ack)
{
	int ret;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* pad the current buffer and send it to the daemon */
	if ((sess->mode & MODE_PLAY) && xfer_buf_to_daemon(sess, IO_PLAY))
		send_msg(sess, UTADEM_MSG_PLAY, 0);

	if (ack)
		set_resetting(sess, 1);
	ret = send_msg(sess, UTADEM_MSG_SYNC, ack);
	if (ack)
		down(&sess->reset_sem);

	return (ret);
}

// FIXME: this can fail - fix callers or use a static messaging scheme
static int
send_msg(struct uta_session *sess, int msgid, int arg)
{
	struct uta_msg *msg;

	DPRINTF(2, "entering %s(sessid=%d, msgid=%d, arg=%d)\n",
	    __FUNCTION__, sess->id, msgid, arg);

	/*
	 * There is a slight race here - the daemon could leave after this
	 * but before we enlist the message.  We'll clean up the message
	 * when the session dies, so don't worry about it. Worst case is one
	 * kmalloc()ed message struct that sits around for a while.
	 */
	if (!has_daemon(sess))
		return (-ENXIO);

	msg = kmalloc(sizeof (*msg), GFP_KERNEL);
	if (!msg) {
		DPRINTF(1, "can't allocate message\n");
		return (-ENOMEM);
	}

	msg->msgtype = msgid;
	msg->arg = arg;

	spin_lock(&sess->msg_lock);
	list_add_tail(&msg->list, &sess->messages);
	spin_unlock(&sess->msg_lock);

	up(&sess->msg_sem);

	return (0);
}

/*
 * These four functions [{user,daemon}_{read,write}()] are protected from
 * re-entrancy by the fact that only one user and one daemon are connected
 * to the session, and the conn->read_sem and conn->write_sem are
 * held by the caller of these functions.
 */

static int
user_read(struct file *file, char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;
	int completed = 0;
	int available = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	// LOCK: sess->io_lock
	/* get the next buffer to read from */
	do {
		io = &sess->ios[IO_RECORD];
		buf = io->fragments[io->user];
		if (!buf) {
			ret = -ENOMEM;
			break;
		}

		/* tell daemon to start recording */
		send_msg(sess, UTADEM_MSG_RECORD, 0);

		add_wait_queue(&io->wait, &wait);
		set_current_state(TASK_INTERRUPTIBLE);
		while (atomic_read(&buf->owner) != UTA_USER) {
			/* user asked not to block */
			if (file->f_flags & O_NONBLOCK) {
				ret = -EAGAIN;
				break;
			}
			/* a signal can wake us up */
			if (signal_pending(current)) {
				ret = -ERESTARTSYS;
				break;
			}
			/* if the daemon dies, we'll be woken up */
			if (!has_daemon(sess)) {
				ret = -ENXIO;
				break;
			}
			schedule();
		}
		current->state = TASK_RUNNING;
		remove_wait_queue(&io->wait, &wait);

		if (ret < 0) {
			/*
			 * For non blocking reads, return the amount
			 * of data read succesfully.
			 */
			if ((ret == -EAGAIN) && (completed > 0))
				ret = completed;
			break;
		}

		/*
		 * if we get here, we have data to read
		 */

		/* clamp the size */
		if (size - completed > buf->bytes) {
			available = buf->bytes;
		}
		else
			available = size - completed;

		/* read it */
		if (copy_to_user(&ubuf[completed], buf->data + buf->offset,
			available)) {
			ret = -EFAULT;
			break;
		}
		buf->offset += available;
		buf->bytes -= available;

		/* if we emptied the buffer... */
		if (buf->bytes == 0)
			xfer_buf_to_daemon(sess, IO_RECORD);
		completed += available;
		ret = completed;
	} while (completed < size);

	up_write(&conn->syscall_rwsem);
	DPRINTF(4, "user read %d tried %d\n", completed, size);
	return (ret);
}

static int
user_write(struct file *file, const char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;
	int completed = 0;
	int available = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	ret = 0;
	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	// LOCK: sess->io_lock
	/* get the next buffer to write into */
	do {
		io = &sess->ios[IO_PLAY];
		buf = io->fragments[io->user];
		if (!buf) {
			ret = -ENOMEM;
			break;
		}

		add_wait_queue(&io->wait, &wait);
		set_current_state(TASK_INTERRUPTIBLE);
		while (atomic_read(&buf->owner) != UTA_USER) {
			/* user asked not to block */
			if (file->f_flags & O_NONBLOCK) {
				ret = -EAGAIN;
				break;
			}
			/* a signal can wake us up */
			if (signal_pending(current)) {
				ret = -ERESTARTSYS;
				break;
			}
			/* if the daemon dies, we'll be woken up */
			if (!has_daemon(sess)) {
				ret = -ENXIO;
				break;
			}
			schedule();
		}
		current->state = TASK_RUNNING;
		remove_wait_queue(&io->wait, &wait);

		if (ret < 0) {
			/*
			 * For non blocking writes, return the amount
			 * of data written succesfully.
			 */
			if ((ret == -EAGAIN) && (completed > 0))
				ret = completed;
			break;
		}

		/*
		 * if we get here, we have space to write
		 */

		/* clamp the size */
		if ((buf->bytes + (size - completed)) > io->fragment_size)
			available = io->fragment_size - buf->bytes;
		else
			available = (size - completed);

		/* write it */
		if (copy_from_user(buf->data + buf->offset, &ubuf[completed],
			available)) {
			ret = -EFAULT;
			break;
		}
		buf->offset += available;
		buf->bytes += available;

		/* if we filled the buffer... */
		if (buf->bytes == io->fragment_size) {
			/* transfer ownership */
			xfer_buf_to_daemon(sess, IO_PLAY);
			/* tell the daemon to start playing */
			send_msg(sess, UTADEM_MSG_PLAY, 0);
		}
		completed += available;
		ret = completed;
	} while (completed < size);

	up_write(&conn->syscall_rwsem);
	DPRINTF(4, "user wrote %d tried %d\n", completed, size);
	return (ret);
}

static int
daemon_read(struct file *file, char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	int ret;
	int completed = 0;
	int available = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	ret = 0;
	/* block hazardous operations */
	down_read(&conn->syscall_rwsem);

	/*
	 * NOTE: don't check has_user() here - the user may have
	 * SYNCed and left
	 */

	// LOCK: sess->io_lock
	do {
		/* get the next buffer to play from */
		io = &sess->ios[IO_PLAY];
		buf = io->fragments[io->daemon];

		/* if the buffer is not ready for us to play, it's an error */
		if (!buf || atomic_read(&buf->owner) != UTA_DAEMON) {
			ret = 0;
			goto out;
		}

		/* clamp the size to this block */
		if (size - completed > buf->bytes)
			available = buf->bytes;
		else
			available = size - completed;

		/* we have data to read */
		if (copy_to_user(&ubuf[completed], buf->data +
			buf->offset, available)) {
			ret = -EFAULT;
			goto out;
		}
		buf->offset += available;
		buf->bytes -= available;

		/* if we emptied the buffer... */
		if (buf->bytes == 0)
			xfer_buf_to_user(sess, IO_PLAY);
		completed += available;
	} while (completed < size);

out:
	up_read(&conn->syscall_rwsem);
	return (completed);
}

static int
daemon_write(struct file *file, const char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	int ret;
	int completed = 0;
	int available = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* block hazardous operations */
	down_read(&conn->syscall_rwsem);

	if (!has_user(sess)) {
		ret = 0;
		goto out;
	}

	// LOCK: sess->io_lock
	/* get the next buffer to record into */
	do {
		io = &sess->ios[IO_RECORD];
		buf = io->fragments[io->daemon];

		/* if the buffer is not ready for us to fill, it's an error */
		if (!buf || atomic_read(&buf->owner) != UTA_DAEMON) {
			ret = 0;
			goto out;
		}

		/* clamp the size to this block */
		if ((buf->bytes + size - completed) > io->fragment_size)
			available = io->fragment_size - buf->bytes;
		else
			available = size - completed;

		/* write as much as we can */
		if (copy_from_user(buf->data + buf->offset,
			&ubuf[completed], available)) {
			ret = -EFAULT;
			goto out;
		}
		buf->offset += available;
		buf->bytes += available;

		/* if we filled the buffer... */
		if (buf->bytes == io->fragment_size)
			xfer_buf_to_user(sess, IO_RECORD);
		completed += available;
	} while (completed < size);

out:
	up_read(&conn->syscall_rwsem);
	return (completed);
}

static unsigned int
user_poll(struct file *file, struct poll_table_struct *wait)
{
	unsigned int mask = 0;
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	if (file->f_mode & FMODE_READ) {
		// LOCK: sess->io_lock
		/* tell daemon to start recording */
		send_msg(sess, UTADEM_MSG_RECORD, 0);
		io = &sess->ios[IO_RECORD];
		poll_wait(file, &io->wait, wait);
		buf = io->fragments[io->user];
		if (!buf) {
			mask = POLLHUP;
			goto out;
		}
		/* check if we own the buffer */
		if (atomic_read(&buf->owner) == UTA_USER)
			mask |= POLLIN | POLLRDNORM;
	}

	if (file->f_mode & FMODE_WRITE) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_PLAY];
		poll_wait(file, &io->wait, wait);
		buf = io->fragments[io->user];
		if (!buf) {
			mask = POLLHUP;
			goto out;
		}
		/* check if we own the buffer */
		if (atomic_read(&buf->owner) == UTA_USER)
			mask |= POLLOUT | POLLWRNORM;
	}

	if (!has_daemon(sess))
		mask = POLLHUP;
out:
	up_write(&conn->syscall_rwsem);
	return (mask);
}

/*
 * all the user ioctl() helper functions are open to re-entrancy - protect
 * them with conn->syscall_rwsem, if needed.
 */
static int
user_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	ret = -ENOTTY;
	if (conn->role == UTA_USER)
		ret = oss_ioctl(inode, file, cmd, arg);
	else if (conn->role == UTA_MIXER)
		ret = mix_ioctl(inode, file, cmd, arg);

	up_write(&conn->syscall_rwsem);
	return (ret);
}

/*
 * This function is re-entrant - protect each block internally with read or
 * write locking on conn->syscall_rwsem!
 */
static int
daemon_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* these don't require the session to exist */
	switch (cmd) {
	case UTAIO_GETVERSION:
		DPRINTF(2, " - UTAIO_GETVERSION\n");
		return (put_user(UTADEM_VERSION, (unsigned long *)arg));

	case UTAIO_NEWSESSION:
		DPRINTF(2, " - UTAIO_NEWSESSION\n");
		if (!capable(CAP_SYS_ADMIN))
			return (-EPERM);
		down_write(&conn->syscall_rwsem);
		ret = do_newsession(conn, current->uid);
		up_write(&conn->syscall_rwsem);
		return (ret);
	}

	if (!sess)
		return (-ENXIO);

	/* these all require the session */
	switch (cmd) {
	case UTAIO_GETMSG:
		DPRINTF(2, " - UTAIO_GETMSG\n");
		down_read(&conn->syscall_rwsem);
		ret = do_getmsg(sess, (struct utadem_msg *)arg,
			    file->f_flags & O_NONBLOCK);
		up_read(&conn->syscall_rwsem);
		return (ret);

	case UTAIO_SETDCAPS:
		DPRINTF(2, " - UTAIO_SETDCAPS\n");
		down_read(&conn->syscall_rwsem);
		ret = do_setdcaps(sess, (struct utadem_dcaps *)arg);
		up_read(&conn->syscall_rwsem);
		return (ret);

	case UTAIO_SETMIXER:
		DPRINTF(2, " - UTAIO_SETMIXER\n");
		down_read(&conn->syscall_rwsem);
		ret = do_setmixer(sess, (struct utadem_mixdev *)arg);
		up_read(&conn->syscall_rwsem);
		return (ret);

	case UTAIO_ACKRESET:
		DPRINTF(2, " - UTAIO_ACKRESET\n");
		down_write(&conn->syscall_rwsem);
		ret = do_ackreset(sess, PLAYBACK | RECORD);
		up_write(&conn->syscall_rwsem);
		return (ret);

	case UTAIO_ACKSYNC:
		DPRINTF(2, " - UTAIO_ACKSYNC\n");
		down_write(&conn->syscall_rwsem);
		ret = do_ackreset(sess, PLAYBACK | RECORD);
		up_write(&conn->syscall_rwsem);
		return (ret);

	case UTAIO_UNDERRUN:
		DPRINTF(2, " - UTAIO_UNDERRUN\n");
		down_read(&conn->syscall_rwsem);
		// LOCK: sess->state_lock
		sess->underruns++;
		up_read(&conn->syscall_rwsem);
		return (0);

	case UTAIO_OVERRUN:
		DPRINTF(2, " - UTAIO_OVERRUN\n");
		down_read(&conn->syscall_rwsem);
		// LOCK: sess->state_lock
		sess->overruns++;
		up_read(&conn->syscall_rwsem);
		return (0);

	default:
		break;
	}

	DPRINTF(4, "daemon_ioctl - unknown ioctl: 0x%08x (%d)\n", cmd, cmd);
	return (-ENOTTY);
}

static int
oss_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	int val;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!sess || !has_daemon(sess))
		return (-ENXIO);

	switch (cmd) {
	case OSS_GETVERSION:
		DPRINTF(4, " - OSS_GETVERSION\n");
		return (put_user(SOUND_VERSION, (int *)arg));

	case SNDCTL_DSP_NONBLOCK:
		DPRINTF(4, " - SNDCTL_DSP_NONBLOCK\n");
		file->f_flags |= O_NONBLOCK;
		return (0);

	case SNDCTL_DSP_GETCAPS:
		DPRINTF(4, " - SNDCTL_DSP_GETCAPS\n");
		return (put_user(DSP_CAP_BATCH | DSP_CAP_DUPLEX, (int *)arg));

	case SNDCTL_DSP_SETDUPLEX:
		DPRINTF(4, " - SNDCTL_DSP_SETDUPLEX\n");
		if (file->f_mode & FMODE_READ &&
		    file->f_mode & FMODE_WRITE)
			return (0);
		else
			return (-EINVAL);

	case SNDCTL_DSP_RESET:
		DPRINTF(4, " - SNDCTL_DSP_RESET\n");
		return (device_reset(sess, 1));

	case SNDCTL_DSP_SYNC:
		DPRINTF(4, " - SNDCTL_DSP_SYNC\n");
		return (device_sync(sess, 1));

	case SNDCTL_DSP_POST:
		DPRINTF(4, " - SNDCTL_DSP_POST\n");
		return (device_sync(sess, 1));

	case SNDCTL_DSP_GETFMTS:
		DPRINTF(4, " - SNDCTL_DSP_GETFMTS\n");
		// LOCK: sess->state_lock
		return (put_user(sess->dformats, (int *)arg));

	case SNDCTL_DSP_SETFMT:
		DPRINTF(4, " - SNDCTL_DSP_SETFMT \n");
		if (get_user(val, (int *)arg))
			return (-EFAULT);
		DPRINTF(4, " - SNDCTL_DSP_SETFMT %d\n", val);
		if (val != AFMT_QUERY) {
			// LOCK: sess->state_lock
			if (sess->dformats & val) {
				sess->format = val;
				send_msg(sess, UTADEM_MSG_SETFORMAT, val);
				/* OSS says to stop on this ioctl() */
				device_reset(sess, 1);
			}
		}
		return (put_user(sess->format, (int *)arg));

	case SOUND_PCM_READ_BITS:
		DPRINTF(4, " - SOUND_PCM_READ_BITS\n");
		// LOCK: sess->state_lock
		switch (sess->format) {
		case AFMT_MU_LAW:
		case AFMT_A_LAW:
		case AFMT_U8:
		case AFMT_S8:
			return (put_user(8, (int *)arg));
		case AFMT_S16_LE:
		case AFMT_S16_BE:
		case AFMT_U16_LE:
		case AFMT_U16_BE:
			return (put_user(16, (int *)arg));
		default:
			return (-EINVAL);
		}

	case SOUND_PCM_READ_CHANNELS:
		DPRINTF(4, " - SOUND_PCM_READ_CHANNELS\n");
		// LOCK: sess->state_lock
		return (put_user(sess->channels, (int *)arg));

	case SNDCTL_DSP_CHANNELS:
		DPRINTF(4, " - SNDCTL_DSP_CHANNELS\n");
		if (get_user(val, (int *)arg))
			return (-EFAULT);
		if (val <= 0)
			return (-EINVAL);
		DPRINTF(4, " - SNDCTL_DSP_CHANNELS %d\n", val);
		// LOCK: sess->state_lock
		if (sess->dchannels & (1UL<<(val-1))) {
			sess->channels = val;
			send_msg(sess, UTADEM_MSG_SETCHANNELS, val);
			/* OSS says to stop on this ioctl() */
			device_reset(sess, 1);
		}
		return (put_user(sess->channels, (int *)arg));

	case SNDCTL_DSP_STEREO:
		if (get_user(val, (int *)arg))
			return (-EFAULT);
		if (val)
			val = 2;
		else
			val = 1;
		// LOCK: sess->state_lock
		DPRINTF(4, " - SNDCTL_DSP_STEREO %d\n", val);
		if (sess->dchannels & (1UL<<(val-1))) {
			sess->channels = val;
			send_msg(sess, UTADEM_MSG_SETCHANNELS, val);
			/* OSS says to stop on this ioctl() */
			device_reset(sess, 1);
		}
		return (put_user((sess->channels == 2) ? 1 : 0, (int *)arg));

	case SOUND_PCM_READ_RATE:
		DPRINTF(4, " - SOUND_PCM_READ_RATE\n");
		return (put_user(sess->rate, (int *)arg));

	case SNDCTL_DSP_SPEED:
		DPRINTF(4, " - SNDCTL_DSP_SPEED\n");
		if (get_user(val, (int *)arg))
			return (-EFAULT);
		if (val <= 0)
			return (-EINVAL);
		// LOCK: sess->state_lock
		if (val >= sess->drate_min &&
		    val <= sess->drate_max) {
			sess->rate = val;
			send_msg(sess, UTADEM_MSG_SETRATE, val);
			/* OSS says to stop on this ioctl() */
			device_reset(sess, 1);
		}
		DPRINTF(4, " - SNDCTL_DSP_SPEED %d f %d c%d \n",
			val, sess->format, sess->channels);
		return (put_user(sess->rate, (int *)arg));

	case SNDCTL_DSP_GETBLKSIZE:
		DPRINTF(4, " - SNDCTL_DSP_GETBLKSIZE\n");
		if (sess->mode == MODE_RECORD) {
 			return (put_user(sess->ios[IO_RECORD].fragment_size,
 				(int *)arg));
 		} else {
 			return (put_user(sess->ios[IO_PLAY].fragment_size,
 				(int *)arg));
 		}

	case SNDCTL_DSP_SETFRAGMENT:
		DPRINTF(4, " - SNDCTL_DSP_SETFRAGMENT \n");
		if (get_user(val, (int *)arg))
			return (-EFAULT);

		/*
		 * Set fragment size and num of fragments.
		 * The argument (val) is an int encoded as 0xMMMMSSSS
		 * in hex. Where MMMM determines the maximum num of
		 * fragments and 2^SSSS gives thefragment size
		 */
		total_fragments = (val >> 16) & 0xffff;
		fragment_buf_size = 1 << (val & 0xffff);
		int seqlen_sec = (sess->channels * sess->rate);
		if (sess->format == AFMT_S16_LE)
			seqlen_sec *= 2;

		fragment_buf_size = set_fragment_buf_size(fragment_buf_size, 
							  seqlen_sec);
		DPRINTF(4, " - SNDCTL_DSP_SETFRAGMENT val=%d"
			" total fragments=%d fragment size=%d\n",
			val, total_fragments, fragment_buf_size);
		send_msg(sess, UTADEM_MSG_SETFRAGSIZE, fragment_buf_size);
		return (device_reset(sess, 1));

	case SNDCTL_DSP_GETISPACE:
		DPRINTF(4, " - SNDCTL_DSP_GETISPACE\n");
		if (!(file->f_mode & FMODE_READ))
			return (-EINVAL);
		else
			return (do_getxspace(sess, IO_RECORD,
			    (struct audio_buf_info *)arg));

	case SNDCTL_DSP_GETOSPACE:
		DPRINTF(4, " - SNDCTL_DSP_GETOSPACE\n");
		if (!(file->f_mode & FMODE_WRITE))
			return (-EINVAL);
		else
			return (do_getxspace(sess, IO_PLAY,
			    (struct audio_buf_info *)arg));

	case SNDCTL_DSP_GETODELAY:
		if (!(file->f_mode & FMODE_WRITE))
			return (-EINVAL);
		else {
			// LOCK: sess->io_lock
			// FIXME: this assumes two buffers
			struct uta_io *io = &sess->ios[IO_PLAY];
			val = io->fragments[io->user]->offset;
			val += io->fragment_size -
				io->fragments[io->daemon]->offset;
		DPRINTF(4, " - SNDCTL_DSP_GETODELAY %d\n", val);
			return (put_user(val, (int *)arg));
		}

	// locking on all these
	case SNDCTL_DSP_GETIPTR:
		DPRINTF(4, " - SNDCTL_DSP_GETIPTR\n");
		return (-ENOTTY);
//		return (put_user(8000, (int *)arg));
	case SNDCTL_DSP_GETOPTR:
		DPRINTF(4, " - sndCTL_DSP_GETOPTR\n");
		return (-ENOTTY);

//		return (put_user(8000, (int *)arg));
	// case SNDCTL_DSP_GETERROR:
	//	DPRINTF(4, " - SNDCTL_DSP_GETERROR\n");
	//	return (-ENOTTY);
	case SNDCTL_DSP_SUBDIVIDE:
		DPRINTF(4, " - SNDCTL_DSP_SUBDIVIDE\n");
		return (-ENOTTY);
	case SNDCTL_DSP_PROFILE:
		DPRINTF(4, " - SNDCTL_DSP_PROFILE\n");
		return (-ENOTTY);
	default:
		break;
	}

	DPRINTF(4, "oss_ioctl - unknown ioctl: 0x%08x (%d)\n", cmd, cmd);
	return (-ENOTTY);
}

static int
mix_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	int val;
	int i;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!sess || !has_daemon(sess))
		return (-ENXIO);

	switch (cmd) {
	case SOUND_MIXER_READ_DEVMASK:
		DPRINTF(4, " - SOUND_MIXER_READ_DEVMASK\n");
		val = 0;
		for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
			if (sess->mixer[i].channels)
				val |= 1<<i;
		}
		return (put_user(val, (int *)arg));

	case SOUND_MIXER_READ_RECMASK:
		DPRINTF(4, " - SOUND_MIXER_READ_RECMASK\n");
		val = 0;
		for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
			if (sess->mixer[i].isrecsrc)
				val |= 1<<i;
		}
		return (put_user(val, (int *)arg));

#ifndef SOUND_MIXER_READ_OUTMASK
#define	SOUND_MIXER_READ_OUTMASK	MIXER_READ(SOUND_MIXER_OUTMASK)
#endif
	case SOUND_MIXER_READ_OUTMASK:
		DPRINTF(4, " - SOUND_MIXER_READ_OUTMASK\n");
		val = 0;
		for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
			if (sess->mixer[i].isoutsrc)
				val |= 1<<i;
		}
		return (put_user(val, (int *)arg));

	case SOUND_MIXER_READ_STEREODEVS:
		DPRINTF(4, " - SOUND_MIXER_READ_STEREODEVS\n");
		val = 0;
		for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
			if (sess->mixer[i].channels == 2)
				val |= 1<<i;
		}
		return (put_user(val, (int *)arg));

	case SOUND_MIXER_READ_CAPS:
		DPRINTF(4, " - SOUND_MIXER_READ_CAPS\n");
		return (put_user(SOUND_CAP_EXCL_INPUT, (int *)arg));

	case SOUND_MIXER_READ_RECSRC:
		DPRINTF(4, " - SOUND_MIXER_READ_RECSRC\n");
		return (put_user(sess->recsrc, (int *)arg));

	case SOUND_MIXER_WRITE_RECSRC: {
		int recmask = 0;
		DPRINTF(4, " - SOUND_MIXER_WRITE_RECSRC\n");
		for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
			if (sess->mixer[i].isrecsrc)
				recmask |= 1<<i;
		}
		if (get_user(val, (int *)arg))
			return (-EFAULT);
		if ((recmask & val) != val)
			return (-EINVAL);
		if (sess->recsrc != val) {
			sess->recsrc = val;
			send_msg(sess, UTADEM_MSG_SETINSRC, val);
		}
		return (0);
	}

#ifndef SOUND_MIXER_READ_OUTSRC
#define	SOUND_MIXER_READ_OUTSRC		MIXER_READ(SOUND_MIXER_OUTSRC)
#endif
	case SOUND_MIXER_READ_OUTSRC:
		DPRINTF(4, " - SOUND_MIXER_READ_OUTSRC\n");
		return (put_user(sess->outsrc, (int *)arg));

#ifndef SOUND_MIXER_WRITE_OUTSRC
#define	SOUND_MIXER_WRITE_OUTSRC	MIXER_WRITE(SOUND_MIXER_OUTSRC)
#endif
	case SOUND_MIXER_WRITE_OUTSRC: {
		int outmask = 0;
		DPRINTF(4, " - SOUND_MIXER_WRITE_OUTSRC\n");
		for (i = 0; i < UTADEM_MIXER_NDEVS; i++) {
			if (sess->mixer[i].isoutsrc)
				outmask |= 1<<i;
		}
		if (get_user(val, (int *)arg))
			return (-EFAULT);
		if ((outmask & val) != val)
			return (-EINVAL);
		if (sess->outsrc != val) {
			sess->outsrc = val;
			send_msg(sess, UTADEM_MSG_SETOUTSRC, val);
		}
		return (0);
	}

	case SOUND_MIXER_INFO: {
		mixer_info mi;
		DPRINTF(4, " - SOUND_MIXER_INFO\n");
		strncpy(mi.id, "utadem", sizeof (mi.id));
		strncpy(mi.name, "utadem", sizeof (mi.name));
		mi.modify_counter = sess->mix_modcount;
		return (copy_to_user((mixer_info *)arg, &mi, sizeof (mi)));
	}

	case SOUND_OLD_MIXER_INFO: {
		_old_mixer_info mi;
		DPRINTF(4, " - SOUND_OLD_MIXER_INFO\n");
		strncpy(mi.id, "utadem", sizeof (mi.id));
		strncpy(mi.name, "utadem", sizeof (mi.name));
		return (copy_to_user((_old_mixer_info *)arg, &mi, sizeof (mi)));
	}

	default:
		if (_IOC_TYPE(cmd) != 'M' || _SIOC_SIZE(cmd) != sizeof (int))
			break;
		if (_SIOC_DIR(cmd) == _SIOC_READ) {
			DPRINTF(4, " - MIXER_READ(%d)\n", _IOC_NR(cmd));
			return (mix_read_ioctl(sess, _IOC_NR(cmd), (int *)arg));
		} else {
			DPRINTF(4, " - MIXER_WRITE(%d)\n", _IOC_NR(cmd));
			return (mix_write_ioctl(sess, _IOC_NR(cmd),
				(int *)arg));
		}
	}

	DPRINTF(4, "mix_ioctl - unknown ioctl: 0x%08x (%d)\n", cmd, cmd);
	return (-ENOTTY);
}

static int
mix_read_ioctl(struct uta_session *sess, int mixdev, int *arg)
{
	DPRINTF(2, "entering %s(sessid=%d, mixdev=%d)\n", __FUNCTION__,
	    sess->id, mixdev);

	if (mixdev < 0 || mixdev >= UTADEM_MIXER_NDEVS ||
		(!sess->mixer[mixdev].channels))
		return (-EINVAL);
	return (put_user(sess->mixer[mixdev].level, arg));
}

static int
mix_write_ioctl(struct uta_session *sess, int mixdev, int *arg)
{
	int val;

	DPRINTF(2, "entering %s(sessid=%d, mixdev=%d)\n", __FUNCTION__,
	    sess->id, mixdev);

	if (mixdev < 0 || mixdev >= UTADEM_MIXER_NDEVS ||
		!(sess->mixer[mixdev].channels))
		return (-EINVAL);
	if (get_user(val, arg))
		return (-EFAULT);
	if (val < 0 || (val & 0xff) > 100 || ((val >> 8) & 0xff) > 100)
		return (-EINVAL);

	/* looks ok - save the value and inform the daemon */
	if (sess->mixer[mixdev].level != val) {
		sess->mixer[mixdev].level = val;
		sess->mix_modcount++;
		send_msg(sess, UTADEM_MSG_MIXER,
		    mixdev | ((val & 0xffff) << 16));
	}

	return (0);
}

/*
 * sscanf() doesn't seem to work quite right
 * format for node names: ut.*-[0-9]+
 */
static int
gross_nodename_hack(struct file *file)
{
	const char *p0;
	const char *p1;
	int ret;
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	p0 = file->f_dentry->d_name.name;
	p1 = p0 + strlen(p0) - 1;

	/* must start with ut */
	if (strncmp(p0, "ut", 2))
		return (-1);

	/* must end in a number... */
	if (!isdigit(*p1--))
		return (-1);
	while (isdigit(*p1))
		p1--;

	/* ...preceded by a dash */
	if (*p1 != '-')
		return (-1);

	ret = (int)simple_strtol(p1+1, NULL, 10);

	return (ret);
}


/*
 * the fops functions
 *
 * security: if we get to any of these functions, the FS layer has confirmed
 * the permissions on the device node.
 */

static ssize_t
utadsp_read(struct file *file, char *ubuf, size_t size, loff_t *ppos)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (ppos != &file->f_pos)
		return (-ESPIPE);
	if (size == 0)
		return (0);

	conn = file->private_data;
	/* block other callers of read() */
	if (down_interruptible(&conn->read_sem))
		return (-ERESTARTSYS);
	sess = conn->session;

	if (conn->role == UTA_USER)
		ret = user_read(file, ubuf, size);
	else if (conn->role == UTA_DAEMON)
		ret = daemon_read(file, ubuf, size);
	else
		ret = -ENXIO;

	up(&conn->read_sem);
	return (ret);
}

static ssize_t
utadsp_write(struct file *file, const char *ubuf, size_t size, loff_t *ppos)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (ppos != &file->f_pos)
		return (-ESPIPE);
	if (size == 0)
		return (0);

	conn = file->private_data;
	/* block other callers of write() */
	if (down_interruptible(&conn->write_sem))
		return (-ERESTARTSYS);
	sess = conn->session;

	if (conn->role == UTA_USER)
		ret = user_write(file, ubuf, size);
	else if (conn->role == UTA_DAEMON)
		ret = daemon_write(file, ubuf, size);
	else
		ret = -ENXIO;

	up(&conn->write_sem);
	return (ret);
}

static unsigned int
utadsp_poll(struct file *file, struct poll_table_struct *wait)
{
	struct uta_connect *conn;
	unsigned int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	conn = file->private_data;

	if (conn->role == UTA_USER)
		ret = user_poll(file, wait);
	else
		ret = POLLERR;

	return (ret);
}

static int
utadem_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	int ret;
	struct uta_connect *conn = file->private_data;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (conn->role == UTA_USER || conn->role == UTA_MIXER)
		ret = user_ioctl(inode, file, cmd, arg);
	else
		ret = daemon_ioctl(inode, file, cmd, arg);

	return (ret);
}

static int
utadsp_open(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	int sessid;
	int r;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* first */
	MOD_INC_USE_COUNT;

	/* allocate our connection */
	conn = kmalloc(sizeof (*conn), GFP_KERNEL);
	if (!conn) {
		DPRINTF(1, "can't allocate connection\n");
		r = -ENOMEM;
		goto out_dec;
	}
	conn->session = NULL;
	conn->role = UTA_NONE;
	init_MUTEX(&conn->read_sem);
	init_MUTEX(&conn->write_sem);
	init_rwsem(&conn->syscall_rwsem);
	file->private_data = conn;

	/* check for magic device names */
	sessid = gross_nodename_hack(file);
	if (sessid < 0) {
		DPRINTF(3, "no magic session ID\n");
		if (!capable(CAP_SYS_ADMIN)) {
			r = -ENODEV;
			goto out_free;
		}
		/* allow root to make new sessions */
		return (0);
	}
	DPRINTF(3, "session ID seems to be %d\n", sessid);

	/* join the session */
	r = join_session_dsp(file->private_data, sessid,
	    file->f_mode & FMODE_READ, file->f_mode & FMODE_WRITE);
	if (r < 0)
		goto out_free;

	return (0);

out_free:
	kfree(conn);
	file->private_data = NULL;
out_dec:
	MOD_DEC_USE_COUNT;

	return (r);
}

static int
utadsp_release(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	struct uta_session *sess;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	conn = file->private_data;
	sess = conn->session;

	if (conn->role == UTA_DAEMON) {
		DPRINTF(3, "disconnecting DAEMON from session %d\n", sess->id);
		/* notify the user of the bad news */
		set_has_daemon(sess, 0);

		/*
		 * Careful:
		 * If there is a user, he's done with alloc_ios().  Make sure
		 * he's not asleep or about to sleep on a buffer. If there
		 * is not a user, any connecting user will never get to
		 * down() any fragment, since has_daemon() is now 0.
		 */
		if (has_user(sess))
			wakeup_ios(sess);

		/* any queued messages are moot */
		free_all_msgs(sess);

		/* you can't get any more reset, wake sleepers or racers */
		up(&sess->reset_sem);

		// AUDIT: make sure the user is woken up from any block!
	} else if (conn->role == UTA_USER) {
		DPRINTF(3, "disconnecting USER from session %d\n", sess->id);
		/* tell the daemon, first */
		set_has_user(sess, 0);
		/* sync, no ACK required */
		device_sync(sess, 0);
		/* tell the daemon we're gone */
		send_msg(sess, UTADEM_MSG_LEFT, sess->id);
		/* another user can join, now */
		set_user_lock(sess, 0);
		// AUDIT: anything else needed here?
	}

	/* drop our reference */
	if (sess)
		unref_session(sess);

	/* done with the connection, now */
	kfree(file->private_data);
	file->private_data = NULL;

	/* last */
	MOD_DEC_USE_COUNT;

	return (0);
}

static struct file_operations utadsp_fops = {
	owner:		THIS_MODULE,
	llseek:		no_llseek,
	read:		utadsp_read,
	write:		utadsp_write,
	poll:		utadsp_poll,
	ioctl:		utadem_ioctl,
	open:		utadsp_open,
	release:	utadsp_release,
};

static int
utamix_open(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	int sessid;
	int r;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* first */
	MOD_INC_USE_COUNT;

	/* allocate our connection */
	conn = kmalloc(sizeof (*conn), GFP_KERNEL);
	if (!conn) {
		DPRINTF(1, "can't allocate connection\n");
		r = -ENOMEM;
		goto out_dec;
	}
	conn->session = NULL;
	conn->role = UTA_NONE;
	init_MUTEX(&conn->read_sem);
	init_MUTEX(&conn->write_sem);
	init_rwsem(&conn->syscall_rwsem);
	file->private_data = conn;

	/* check for magic device names */
	sessid = gross_nodename_hack(file);
	if (sessid < 0) {
		DPRINTF(3, "no magic session ID\n");
		r = -ENODEV;
		goto out_free;
	}
	DPRINTF(3, "session ID seems to be %d\n", sessid);

	/* join the session */
	r = join_session_mix(file->private_data, sessid);
	if (r < 0)
		goto out_free;

	return (0);

out_free:
	kfree(conn);
	file->private_data = NULL;
out_dec:
	MOD_DEC_USE_COUNT;

	return (r);
}

static int
utamix_release(struct inode *inode, struct file *file)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (conn->role == UTA_MIXER)
		set_has_mixer(sess, 0);

	/* drop our reference */
	if (sess)
		unref_session(sess);

	/* done with the connection, now */
	kfree(file->private_data);
	file->private_data = NULL;

	/* last */
	MOD_DEC_USE_COUNT;

	return (0);
}

static struct file_operations utamix_fops = {
	owner:		THIS_MODULE,
	llseek:		no_llseek,
	ioctl:		utadem_ioctl,
	open:		utamix_open,
	release:	utamix_release,
};

/* the device numbers allocated by sound_core */
static int dsp_dev;
static int mix_dev;

static int utadem_proc_read(char *buf, char **start, off_t offset,
    int len, int *eof, void *data)
{
	int dlen;
	DPRINTF(2, "entering %s\n", __FUNCTION__);

	dlen = sprintf(buf, "major: %d\n", SOUND_MAJOR);
	dlen += sprintf(buf+dlen, "dsp: %d\n", dsp_dev);
	dlen += sprintf(buf+dlen, "mixer: %d\n", mix_dev);

	/* trying to read a bad offset? */
	if (offset >= dlen) {
		*eof = 1;
		return (0);
	}

	/* did we get everything? */
	if (len >= (dlen-offset)) {
		*eof = 1;
	}

	*start = buf + offset;
	dlen -= offset;

	return ((len > dlen) ? dlen : len);
}

static int __init
utadem_init(void)
{
	int ret = 0;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* set up the session-tracking stuff */
	sessbits_count = max_sessions/BITS_PER_LONG;
	sessbits_count += (max_sessions % BITS_PER_LONG) ? 1 : 0;

	session_bits = kmalloc(sessbits_count * sizeof (*session_bits),
	    GFP_KERNEL);
	if (!session_bits) {
		printk(KERN_ERR "utadem: can't allocate session_bits\n");
		ret = -ENOMEM;
		goto out_sessbits_failed;
	}
	memset(session_bits, 0, sessbits_count * sizeof (*session_bits));

	/* set any bits beyond max_sessions, so we can ignore them later */
	if (max_sessions % BITS_PER_LONG) {
		int r = max_sessions % BITS_PER_LONG;
		int i;
		unsigned long *last_word = &session_bits[sessbits_count - 1];

		for (i = r; i < BITS_PER_LONG; i++) {
			__set_bit(i, last_word);
		}
	}

	/* hook into sound_core */
	dsp_dev = register_sound_dsp(&utadsp_fops, -1);
	if (dsp_dev < 0) {
		printk(KERN_ERR "utadem: can't register dsp device\n");
		ret = dsp_dev;
		goto out_dsp_dev_failed;
	}
	DPRINTF(3, "registered dsp_dev %d\n", dsp_dev);

	mix_dev = register_sound_mixer(&utamix_fops, -1);
	if (mix_dev < 0) {
		printk(KERN_ERR "utadem: can't register mixer device\n");
		ret = mix_dev;
		goto out_mix_dev_failed;
	}
	DPRINTF(3, "registered mix_dev %d\n", mix_dev);

	/* register a file in /proc/driver */
	if (!create_proc_read_entry("driver/utadem", 0, 0,
	    utadem_proc_read, NULL)) {
		printk(KERN_ERR "utadem: can't create /proc/driver/utadem\n");
		ret = -ENOMEM;
		goto out_proc_failed;
	}

	printk(KERN_INFO "utadem: Audio Device Emulation version %d.%d\n",
		UTADEM_VER_MAJOR(UTADEM_VERSION),
		UTADEM_VER_MINOR(UTADEM_VERSION));

	return (0);

out_proc_failed:
	unregister_sound_mixer(mix_dev);
out_mix_dev_failed:
	unregister_sound_dsp(dsp_dev);
out_dsp_dev_failed:
	kfree(session_bits);
out_sessbits_failed:
	return (ret);
}

static void __exit
utadem_cleanup_module(void)
{
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	remove_proc_entry("driver/utadem", NULL);
	unregister_sound_mixer(mix_dev);
	unregister_sound_dsp(dsp_dev);
	kfree(session_bits);
}

module_init(utadem_init);
module_exit(utadem_cleanup_module);

MODULE_AUTHOR("Sun Ray Engineering <srss-feedback-ext@Sun.COM>");
MODULE_DESCRIPTION("Audio device emulation");
MODULE_LICENSE("GPL");

MODULE_PARM(max_sessions, "i");
MODULE_PARM_DESC(max_sessions, "Maximum number of sessions");

/* Not used at this time */
// MODULE_PARM(max_per_user, "i");
// MODULE_PARM_DESC(max_per_user, "Maximum number of sessions per user");

MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug,
	"Debugging level (0=off, 1=errors, 2=basic, 3=full, 4=verbose)");

EXPORT_NO_SYMBOLS;
