/*
 * Copyright (c) 2004, 2011, Oracle and/or its affiliates. All rights reserved.
 */

#pragma ident	"@(#)utio.c	1.18	11/04/13 Oracle"

#ifdef _SCCSID
static char __attribute__ ((unused)) *_SCCSid = "@(#)utio.c	1.18	11/04/13 Oracle";
#endif	/* defined _SCCSID */

#define UTIO_DRIVER_VERSION "1.34"

/*
 * When the utio driver is loaded, utio_init() is called:
 *   - Sets up to handle 2048 sessions (devices).
 *   - Calls register_chrdev() to get a dynamically assigned major number
 *     associated with the following utio driver file op entry points:
 *		llseek:		no llseek handler
 *		read:		utio_read
 *		write:		utio_write
 *		poll:		utio_poll
 *		ioctl:		utio_ioctl
 *		compat_ioctl:	utio_compat_ioctl
 *		open:		utio_open
 *		release:	utio_release
 *   - Registers /proc/driver/utio
 *       utio_proc_read() handles reads of /proc/driver/utio.  It will display:
 *       major: <major number>
 *       version: <driver version, i.e. UTIO_DRIVER_VERSION>
 *
 * Normal sequence of calls made by the daemon (ttydrv.c tty_utio_open():
 *   - Open the clone/master device: master_fd = open("/dev/utio", O_RDWR):
 *       handled by utio_open()
 *   - Get the utio major number:
 *       dev_major = ioctl(master_fd, UTIO_GETMAJOR)
 *       handled by utio_ioctl() which calls daemon_ioctl()
 *       Currently, the daemon does not cache dev_major so it repeats this call
 *   - Get an unused device ID (utio driver calls it a session ID):
 *       utio_devid = ioctl(master_fd, UTIO_GETDEVID)
 *       Handled by utio_ioctl() which calls daemon_ioctl().
 *   - utio_devid is appended to "serial:a%=" to create the device name.
 *       For example, if utio_devid == 0, the device name is "serial:a%=0".
 *   - The daemon does a mknod of .../devices/serial:a%=0.
 *   - Open the slave device:
 *       slave_fd = open("serial:a%=0")
 *       handled by utio_open()
 *   - /dev/term/a is linked to .../devices/serial:a%=0
 *   - The sequence above is repeated for additional slave devices.
 *   - close(slave_fd) is called to close a slave device.
 *       Handled by utio_release().
 *   - close(master_fd) is called to close the clone/master device.
 *       Handled by utio_release().
 *
 * Overview of the utio driver calls shown above:
 *   - Open the clone/master device: master_fd = open("/dev/utio", O_RDWR)
 *     Handled by utio_open(struct file *file):
 *       kmalloc() and initialize a "connection" (struct uta_connect).
 *       The device name, "utio", is not a slave device so no session is
 *       created.
 *       file->private_data = addr of the connection we just set up.
 *   - dev_major = ioctl(master_fd, UTIO_GETMAJOR)
 *       Handled by utio_ioctl() which calls daemon_ioctl().
 *       Returns the major number found in the clone/master device's inode.
 *       [not sure who does the clone device mknod and where it gets
 *       the major number].
 *   - utio_devid = ioctl(master_fd, UTIO_GETDEVID)
 *       Handled by utio_ioctl() which calls daemon_ioctl():
 *         Calls do_newsession() to get an unused session ID and create
 *         the session (slave device).
 *       The daemon uses this session ID to create the slave device name.
 *       For example, if the session ID returned was "0", the daemon would
 *       create the slave device "serial:a%=0".
 *   - slave_fd = open("serial:a%=0")
 *       Handled by utio_open():
 *         kmalloc() and initialize a "connection".
 *         Find the session for this slave device which was created by the
 *         UTIO_GETDEVID ioctl in the previous step, based on the slave
 *         device name.  The "0" in the device name specifies session ID 0.
 *         Set connection->session = the session we just created.
 *         Send a UTIO_MSG_OPEN to the daemon.
 *         Wait for an ACK.
 *
 * The basic utio data structure layout at this point is:
 * 
 *   master_fd is associated with /dev/utio.
 *     Let's call its struct file master_file.
 *   Its connection (struct uta_connection) pointer is in
 *   master_file->private_data.
 *   master_file->private_data->session points to the slave device's
 *   session (struct uta_session).
 *
 *   slave_fd is associated with .../devices/serial:a%=0
 *     Let's call its struct file slave_file.
 *   Its connection pointer is in slave_file->private_data.
 *   slave_file->private_data->session points to the slave device's
 *   session (struct uta_session).
 *
 *   The session has two connections:
 *     - conn->role == UTA_USER: the user side is the connection created when
 *       the slave device (e.g. .../devices/serial:a%=0) was opened.
 *     - conn->role == UTA_DAEMON: the daemon or master side is the connection
 *       created when the clone device (/dev/utio) was opened.
 *
 * Close (release) file op:
 *   - Handled by utio_release():
 *   - close(slave_fd) is called to close a slave device:
 *       Send UTIO_MSG_CLOSE to the daemon.
 *       Wait for an ACK
 *       Free the slave device connection.
 *   - close(master_fd) is called to close the clone/master device:
 *       Free the slave device session.
 *       Free the clone/master device connection.
 *
 * System call entry serialization:
 *    utio_open() is reentrant and does not serialize entry.
 *    utio_release() clears file->private_data holding conn_lock spinlock.
 *    utio_read() serializes entry by obtaining connection->read_sem.
 *      Allows one read() system call at a time.  All other system calls
 *      can occur concurrently with the read().
 *    utio_write() serializes entry by obtaining connection->write_sem.
 *      Allows one write() system call at a time.  All other system calls
 *      can occur concurrently with the write().
 *    entry to the remaining system calls is serialized by obtaining
 *      the read/write semaphore connection->syscall_rwsem.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/fs.h>
#include <linux/smp_lock.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <linux/ioctl.h>
#include <linux/termios.h>
#include <linux/ppdev.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>

#include "utio.h"

/* get major number at load time */
#define	UTIO_MAJOR	0

/* max sessions/connections */
#define	MAX_SESSIONS	2048

/* IO Buffer size */
#define	UTA_DEF_BLKSIZE	8192
/* No. of IO buffers */
#define	UTA_DEF_NBLKS	2

/* TIMEOUT for UTIO_MSG_OPEN in seconds */
#define	UTIO_OPEN_TIMEOUT	(60*HZ)

char *utio_driver_version = UTIO_DRIVER_VERSION;

/* debugging stuff */
/* Debugging level (0=off, 1=errors, 2=basic, 3=full, 4=verbose) */
static int debug=0;
#define	DPRINTF(lvl, fmt, args...)	do { \
	if (debug && debug >= lvl) \
		printk(KERN_DEBUG "utio debug: " fmt, ##args); \
} while (0)


struct uta_buffer {
	int offset;		/* offset into this buf */
	int bytes;		/* size of data in this buf */
	atomic_t owner;		/* who owns this buf - daemon or user? */
	unsigned char data[UTA_DEF_BLKSIZE];
};

struct uta_io {
	int user;		/* which buffer the user is on */
	int daemon;		/* which buffer the daemon is on */
	wait_queue_head_t wait; /* for IO blocking */
	struct uta_buffer *bufs[UTA_DEF_NBLKS];
};

struct uta_msg {
	utio_pmsg_t pmsg;
	struct list_head list;
};

#ifdef	PMSG_SEQ_DEBUGGING
static uint32_t	next_seq = 0;
#endif	/* PMSG_SEQ_DEBUGGING */

struct uta_session {
	int id;			/* the session ID */
	uid_t owner;		/* the uid of the owning user */
	unsigned char mode;	/* user's open() mode */
	unsigned long flags;	/* who is connected, etc */
	int refcount;		/* when this is 0, the session can be freed */
	struct list_head list;	/* global session list_head */
	struct uta_io ios[2];	/* I/O streams */
	struct list_head messages_to_master;	/* protocol messages */
	spinlock_t msg_lock_master;		/* protect the messages */
	struct list_head messages_to_slave;	/* protocol messages */
	spinlock_t msg_lock_slave;		/* protect the messages */
	struct semaphore msg_sem_master;	/* producer/consumer sync */
	struct semaphore msg_sem_slave;		/* producer/consumer sync */
	wait_queue_head_t daemon_wait;		/* for daemon_poll */
};

/* I/O indices */
#define	IO_READ		0
#define	IO_WRITE	1

/* mode bits */
#define	MODE_READ	(1 << IO_READ)
#define	MODE_WRITE	(1 << IO_WRITE)

/* Info about the "connection" to a device */
struct uta_connect {
	struct list_head list;	/* global connection list_head */
	int refcount;		/* when 0, the connection can be freed */
	wait_queue_head_t release_wait;		/* wait for free conn */
	struct uta_session *session;
	int role;				/* user or daemon? */
	struct semaphore read_sem;		/* one read() user at a time */
	struct semaphore write_sem;		/* one write() user at a time */
	struct rw_semaphore syscall_rwsem;	/* block unsafe re-entrances */
};

/* role values */
#define	UTA_NONE		-1
#define	UTA_DAEMON		0
#define	UTA_USER		1

/* protects access to file->private_data and conn->refcount */
DEFINE_SPINLOCK(conn_lock);

/*
 * Increment the reference count for a session.
 * Called with the session unlisted or with sesslist_lock held.
 */
#define	ref_session(sess)				\
do {							\
	sess->refcount++;				\
	DPRINTF(2, "ref sess %d refcount is now %d\n",	\
		sess->id, sess->refcount);		\
} while (0)

/*
 * Decrement the reference count for a session.
 * If the reference count goes to zero, call end_session().
 * Obtains sesslist_lock to protect decrement and for call to end_session().
 */
#define	unref_session(sess)					\
do {								\
	unsigned long flgs;					\
								\
	spin_lock_irqsave(&sesslist_lock, flgs);		\
	--sess->refcount;					\
	DPRINTF(2, "unref sess %d refcount is now %d\n",	\
		sess->id, sess->refcount);			\
	if (sess->refcount == 0)				\
		end_session(sess);				\
	spin_unlock_irqrestore(&sesslist_lock, flgs);		\
} while (0)

/*
 * Get the struct uta_connect pointed to by file->private_data.
 * utio_rlease() does the same sequence manually while clearing
 * file->private_data with conn_lock spinlock held.
 * No references to file->private_data after releasing conn_lock
 * as utio_release() may clear it.
 *
 * Acquire conn_lock spinlock.
 * If file->private_data is NULL, return() with the specified error.
 *   Would happen if utio_rlease() is in the process of closing the file.
 * Increment conn->refcount so that this connection will not go away
 * until we are done with it.
 *
 */
#define	GET_CONNECTION(file, conn, conn_err)				\
do {									\
	unsigned long flgs;						\
									\
	spin_lock_irqsave(&conn_lock, flgs);				\
									\
	conn = file->private_data;					\
	if (!conn) {							\
		spin_unlock_irqrestore(&conn_lock, flgs);		\
		DPRINTF(1, "%s(): error: no connection\n",		\
			__FUNCTION__);					\
		DPRINTF(1, "%s(): exiting\n", __FUNCTION__);		\
		return (conn_err);					\
	}								\
									\
	/*								\
	 * Make sure our connection doesn't go away if we sleep		\
	 * obtaining syscall_rwsem.					\
	 * We will call RELEASE_CONNECTION(conn) when we no longer need	\
	 * to dereference file->private_data.				\
	 * conn->refcount is protected by conn_lock.			\
	 */								\
	conn->refcount++;						\
									\
	spin_unlock_irqrestore(&conn_lock, flgs);			\
} while (0)

/*
 * We no longer need the connection.
 * Decrement conn->refcount.  If it goes to zero, call
 * wake_up(&conn->release_wait) to wake up potential waiter in
 * utio_release().
 */
#define	RELEASE_CONNECTION(conn)					\
do {									\
	unsigned long flgs;						\
									\
	spin_lock_irqsave(&conn_lock, flgs);				\
	/* conn->refcount is protected by conn_lock */			\
	conn->refcount--;						\
	BUG_ON(conn->refcount < 0);					\
	if (conn->refcount == 0)					\
		wake_up(&conn->release_wait);				\
	spin_unlock_irqrestore(&conn_lock, flgs);			\
} while (0)

#define NO_RELEASE_FLG	0
#define RELEASE_FLG	1

/*
 * Set sess = the struct uta_session pointed to by conn->session.
 * If rel_flg, call RELEASE_CONNECTION(conn) on error.
 * If conn->session is NULL, return (-ENODEV).
 * Verify that the session is connected.  If it isn't, return -ENODEV
 */
#define	GET_SESSION(conn, sess, rel_flg)				\
do {									\
	sess = conn->session;						\
	if (!sess) {							\
		DPRINTF(1, "%s(): error: no session\n",			\
			__FUNCTION__);					\
		DPRINTF(1, "%s(): exiting\n", __FUNCTION__);		\
		if (rel_flg)						\
			RELEASE_CONNECTION(conn);			\
		return (-ENODEV);					\
	}								\
									\
	if (is_disc(sess)) {						\
		DPRINTF(1, "%s(): error: no connection\n",		\
			__FUNCTION__);					\
		DPRINTF(1, "%s(): exiting\n", __FUNCTION__);		\
		if (rel_flg)						\
			RELEASE_CONNECTION(conn);			\
		return (-ENODEV);					\
	}								\
} while (0)

/*
 * Same as GET_SESSION() except we are called holding semaphore sem.
 * Release the semaphore using sem_up before returning on error.
 */
#define	GET_SESSION_LOCKED(conn, sess, sem_up, sem, rel_flg)		\
do {									\
	sess = conn->session;						\
	if (!sess) {							\
		DPRINTF(1, "%s(): error: no session\n",			\
			__FUNCTION__);					\
		DPRINTF(1, "%s(): exiting\n", __FUNCTION__);		\
		if (rel_flg)						\
			RELEASE_CONNECTION(conn);			\
		sem_up(&sem);						\
		return (-ENODEV);					\
	}								\
									\
	if (is_disc(sess)) {						\
		DPRINTF(1, "%s(): error: no connection\n",		\
			__FUNCTION__);					\
		DPRINTF(1, "%s(): exiting\n", __FUNCTION__);		\
		if (rel_flg)						\
			RELEASE_CONNECTION(conn);			\
		sem_up(&sem);						\
		return (-ENODEV);					\
	}								\
} while (0)

/* some helpers for session state management - lockless is good */
#define	BIT_USER		0 /* user is connected */
#define	BIT_DAEMON		1 /* daemon is connected */
#define	BIT_ULOCK		2 /* a user connection is imminent */
#define	BIT_DISC		3 /* Daemon is forced to disconnect */
#define	has_daemon(s)		test_bit(BIT_DAEMON, &(s)->flags)
#define	has_user(s)		test_bit(BIT_USER, &(s)->flags)
#define	is_disc(s)		test_bit(BIT_DISC, &(s)->flags)
#define	set_has_daemon(s, v)	set_flag((s), BIT_DAEMON, (v))
#define	set_has_user(s, v)	set_flag((s), BIT_USER, (v))
#define	set_user_lock(s, v)	set_flag((s), BIT_ULOCK, (v))
#define	set_is_disc(s, v)	set_flag((s), BIT_DISC, (v))

/* set WLD_DEBUGGING to a non-zero value to enable debugging */
#define WLD_DEBUGGING		0

/*
 * set WLD_DEBUGGING_CRASH to a non-zero value to cause a crash if the
 * debugging code detects an error.
 */
#define WLD_DEBUGGING_CRASH	0

#if WLD_DEBUGGING

/*
 * Define two circular buffers of debugging structures:
 *     wld_buf[]: entry for events that affect a wait list
 *     wld_init_buf[]: entry for each init of a wait queue head
 */
typedef struct wld {
	unsigned long sequence;		/* unique sequence number */
	int cpu_id;			/* CPU event occurred on */
	unsigned int loc;		/* id of location event occurred at */
	void *data1;			/* caller data */
	void *data2;			/* caller data */
} wld_t;

/* wld_lock protects wld_seq and wld_cur */
DEFINE_SPINLOCK(wld_lock);
unsigned long wld_seq = 1;      	/* event sequence number */

#define	WLD_BUF_SIZE	1000
wld_t wld_buf[WLD_BUF_SIZE];    	/* wait/wakeup events */
unsigned int wld_cur = 0;       	/* last used index into wld_buf[] */

#if WLD_DEBUGGING_CRASH
#define WLD_FILL_ENTRY(loc, data1, data2)			\
do {								\
	wld_fill_entry(loc, (void *)data1, (void *)data2);	\
	if (loc >= 0x80000000)					\
		BUG();						\
} while (0)
#else
#define WLD_FILL_ENTRY(loc, data1, data2)			\
	wld_fill_entry(loc, (void *)data1, (void *)data2)
#endif /* WLD_DEBUGGING_CRASH */

void
wld_fill_entry(unsigned int loc, void *data1, void *data2)
{
	unsigned long flgs;
	unsigned long our_index;

	spin_lock_irqsave(&wld_lock, flgs);
	/* wld_cur and wld_seq are protected by wld_lock */
	wld_cur++;
	if (wld_cur >= WLD_BUF_SIZE) {
		wld_cur = 0;
	}
	our_index = wld_cur;
	wld_buf[our_index].sequence = wld_seq++;
	mb();
	spin_unlock_irqrestore(&wld_lock, flgs);

	wld_buf[our_index].cpu_id = smp_processor_id();
	wld_buf[our_index].loc = loc;
	wld_buf[our_index].data1 = data1;
	wld_buf[our_index].data2 = data2;

	return;
}

#else
#define WLD_FILL_ENTRY(loc, data1, data2)
#endif /* WLD_DEBUGGING */

static inline int
set_flag(struct uta_session *sess, int bit, int val)
{
	if (val)
		return (test_and_set_bit(bit, &sess->flags));
	else
		return (test_and_clear_bit(bit, &sess->flags));
}

/* the global list of active sessions (struct uta_session) */
static LIST_HEAD(session_list);
/* protects session_list */
static DEFINE_SPINLOCK(sesslist_lock);

/* the global list of active connections (struct uta_connection) */
static LIST_HEAD(connection_list);
/* protects connection_list */
static DEFINE_SPINLOCK(connlist_lock);

/* the global session bitmask */
static int sessbits_count; /* count of ulongs in the array */
static unsigned long *session_bits;
static DEFINE_SPINLOCK(sessbits_lock);

/* dynamic utio major number */
static unsigned int utio_major = UTIO_MAJOR;

/* functions */
static int new_sessionid(void);
static int do_newsession(struct uta_connect *conn, uid_t owner);
static struct uta_session *find_session(int sessid);
static int alloc_ios(struct uta_session *sess);
static void wakeup_ios(struct uta_session *sess);
static int slave_open(struct uta_connect *conn, int sessid, int readfrom,
		int writeto);

static void free_all_msgs(struct uta_session *sess);
static void free_ios(struct uta_session *sess);
static void release_sessionid(int sessid);
static void end_session(struct uta_session *sess);

static int xfer_buf_to_daemon(struct uta_session *sess, int io_idx);
static int xfer_buf_to_user(struct uta_session *sess, int io_idx);

static int send_msg_to_master(struct uta_session *sess, utio_pmsg_t *pmsg);
static int send_msg_to_slave(struct uta_session *sess, utio_pmsg_t *pmsg, int flagkptr);
static int do_getmsg_from_slave(struct uta_session *sess, utio_pmsg_t *umsg,
				int nonblock, uid_t owner, int flagkptr);
static int do_getmsg_from_master(struct uta_session *sess, utio_pmsg_t *umsg,
				uid_t owner, int msgtype);


/* fops helpers */
static int daemon_read(struct uta_connect *conn, char *ubuf, size_t size);
static int daemon_write(struct uta_connect *conn, const char *ubuf, size_t size);
static int daemon_ioctl(struct inode *inode, struct file *file,
    struct uta_connect *conn, unsigned int cmd, unsigned long arg);
static unsigned int daemon_poll(struct file *file, struct uta_connect *conn,
    struct poll_table_struct *wait);

static int user_read(struct file *file, struct uta_connect *conn, char *ubuf, size_t size);
static int user_write(struct file *file, struct uta_connect *conn, const char *ubuf, size_t size);
static unsigned int user_poll(struct file *file, struct uta_connect *conn,
    struct poll_table_struct *wait);
static int user_ioctl(struct inode *inode, struct uta_connect *conn,
    unsigned int cmd, unsigned long arg);
static int seriald_ioctl(struct inode *inode, struct uta_connect *conn,
    unsigned int cmd, unsigned long arg);
static int paralleld_ioctl(struct inode *inode, struct uta_connect *conn,
    unsigned int cmd, unsigned long arg);


/*
 * Find the nodename/session ID that is part of the filename with the
 * format filename%=N where N is a numeric value.
 *   Return -1 if the filename does not end with a numeric value.
 *   Otherwise, return the numeric value.
 *
 * Must be reentrant as it is called from utio_open().
 */
static int
gross_nodename_hack(struct file *file)
{
	char *devid;
	char *p0;
	char *p1;
	int ret;

	DPRINTF(2, "entering %s(): device name is %s\n",
		__FUNCTION__, file->f_dentry->d_name.name);

	/*
	 * The format of the nodename is
	 * <device name>%=<unique device id>
	 */
	p0 = (char *) file->f_dentry->d_name.name;
	p1 = p0 + strlen(p0) - 1;

	/* must end in a number... */
	if (!isdigit(*p1--)) {
		DPRINTF(2, "%s: device name does not end with numeric id\n", __FUNCTION__);
		return (-1);
	}

	while (isdigit(*p1))
		p1--;
	devid = p1+1;

	/*  preceded by '%=' */
	if ((*p1-- != '=') || (*p1-- != '%')) {
		DPRINTF(2, "%s: filename does not end with %%=N\n", __FUNCTION__);
		return (-1);
	}

	ret = (int) simple_strtol(devid, NULL, 10);

	DPRINTF(2, "exiting %s(): device id = %d\n", __FUNCTION__, ret);
	return (ret);
}

/*
 * We got here from a read of /proc/driver/utio.
 * Return the following string in buf:
 *   major: <major number>
 * where <major number> is the utio major number.
 *
 * utio_init() does the create_proc_read_entry() which adds us as the read
 * handler.
 */
static int
utio_proc_read(char *buf, char **start, off_t offset,
			int len, int *eof, void *data)
{
	int dlen;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	dlen = snprintf(buf, len, "major: %u\nversion: %s\n",
			utio_major,
			UTIO_DRIVER_VERSION);

	/* trying to read a bad offset? */
	if (offset >= dlen) {
		*eof = 1;
		DPRINTF(1, "%s: bad offset\n", __FUNCTION__);
		return (0);
	}

	/* did we get everything? */
	if (len >= (dlen-offset)) {
		*eof = 1;
	}

	*start = buf + offset;
	dlen -= offset;

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return ((len > dlen) ? dlen : len);
}

/*
 * Called by daemon_ioctl() and compat_daemon_ioctl() with:
 *   down_read(&conn->syscall_rwsem)
 */
static int
do_getmsg_from_slave(struct uta_session *sess, utio_pmsg_t *umsg,
		int nonblock, uid_t owner, int flagkptr)
{
	struct uta_msg *msg;
	struct list_head *item;
	unsigned long flgs;
	int ret = 0;

	WLD_FILL_ENTRY(0x100, sess->id, NULL);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__,
		sess->id);

	/* get the msg_sem_master to see if there are any messages */
	if (nonblock) {
		WLD_FILL_ENTRY(0x101, sess->id, &sess->msg_sem_master);
		if (down_trylock(&sess->msg_sem_master)) {
			WLD_FILL_ENTRY(0x102, sess->id, NULL);
			DPRINTF(2, "%s: down_trylock() failed, returning EAGAIN\n", __FUNCTION__);
			return (-EAGAIN);
		}
	} else {
		WLD_FILL_ENTRY(0x103, sess->id, &sess->msg_sem_master);
		if (down_interruptible(&sess->msg_sem_master)) {
			/* caller is not written to handle ERESTARTSYS */
			WLD_FILL_ENTRY(0x80000104, sess->id, NULL);
			DPRINTF(1, "%s: down_interruptible() interrupted\n",					__FUNCTION__);
			return (-EINTR);
		}
	}

	if (is_disc(sess)) {
		// we are asked to disconnect
		WLD_FILL_ENTRY(0x80000105, sess->id, NULL);
		DPRINTF(1, "%s: we are asked to disconnect\n", __FUNCTION__);
		return (-ENXIO);
	}

	spin_lock_irqsave(&sess->msg_lock_master, flgs);
	if (!list_empty(&sess->messages_to_master)) {
		item = sess->messages_to_master.next;
		list_del(item);
	}
	else {
		item = NULL;
		DPRINTF(2, "%s: list empty, returning EAGAIN\n", __FUNCTION__);
		ret = -EAGAIN;
	}
	spin_unlock_irqrestore(&sess->msg_lock_master, flgs);

	if (item) {
		WLD_FILL_ENTRY(0x106, sess->id, NULL);
		msg = list_entry(item, struct uta_msg, list);
		if (flagkptr)
			memmove(umsg, msg, sizeof (utio_pmsg_t));
		else
			ret = copy_to_user(umsg, msg, sizeof (utio_pmsg_t));
		kfree(msg);
	}
	else {
		WLD_FILL_ENTRY(0x80000107, sess->id, NULL);
		DPRINTF(2, "%s: messages_to_master is empty\n", __FUNCTION__);
	}

	WLD_FILL_ENTRY(0x108, sess->id, ret);
	DPRINTF(2, "exiting %s(): sess id %d ret %d\n",
		__FUNCTION__, sess->id, ret);
	return (ret);
}

/*
 * Timeout handler used by do_getmsg_from_master() to wake up from
 * down_interruptible(&sess->msg_sem_slave) if too much time passes
 * waiting for a response from the daemon to a UTIO_MSG_OPEN message.
 *
 * Wake up waiters on the msg_sema_slave semphore.
 */
static void
timeout_waiting_for_master(unsigned long msg_sema_slave) {
	WLD_FILL_ENTRY(0x200, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	up((struct semaphore *) msg_sema_slave);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}

/*
 * Get (wait if necessary) for a message from the daemon/master.
 */
static int
do_getmsg_from_master(struct uta_session *sess, utio_pmsg_t *umsg,
		uid_t owner, int msgtype)
{
	struct uta_msg *msg;
	struct list_head *item;
	struct timer_list timer;
	unsigned long flgs;
	int ret;

	WLD_FILL_ENTRY(0x300, sess->id, NULL);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__, sess->id);

	if (!has_daemon(sess)) {
		WLD_FILL_ENTRY(0x80000301, sess->id, NULL);
		DPRINTF(1, "exiting %s: Session doesn't have daemon 1\n",
								__FUNCTION__);
		return (-ENXIO);
	}

	WLD_FILL_ENTRY(0x302, sess->id, &sess->msg_sem_slave);
	/* get the msg_sem_slave to see if there are any messages */
	if (msgtype == UTIO_MSG_OPEN) {
		WLD_FILL_ENTRY(0x303, sess->id, &sess->msg_sem_slave);
		DPRINTF(2, "%s: UTIO_MSG_OPEN\n", __FUNCTION__);
		init_timer(&timer);
		timer.expires = jiffies + UTIO_OPEN_TIMEOUT;
		timer.function = timeout_waiting_for_master;
		timer.data = (unsigned long) &sess->msg_sem_slave;
		add_timer(&timer);

		if ((ret = down_interruptible(&sess->msg_sem_slave))) {
			/*
			 * Interrupted.
			 * utio_open() is not written to allow ERESTARTSYS
			 */
			WLD_FILL_ENTRY(0x80000304, sess->id, &sess->daemon_wait);
			(void) del_timer_sync(&timer);
			set_is_disc(sess, 1);
			wake_up_interruptible(&sess->daemon_wait);
			up(&sess->msg_sem_master);
			DPRINTF(1, "exiting %s: down_interruptible interrupted\n",
								__FUNCTION__);
			return (-EINTR);
		} else if (!del_timer_sync(&timer)) {
			/* Timed out.  Notify the user of the bad news */
			WLD_FILL_ENTRY(0x80000305, sess->id, &sess->daemon_wait);
			set_is_disc(sess, 1);
			wake_up_interruptible(&sess->daemon_wait);
			up(&sess->msg_sem_master);
			DPRINTF(1, "exiting %s: Timed out\n", __FUNCTION__);
			return (-ETIMEDOUT);
		}
	} else if (down_interruptible(&sess->msg_sem_slave)) {
		/*
		 * Interrupted.
		 * utio_open() is not written to allow ERESTARTSYS
		 */
		WLD_FILL_ENTRY(0x80000306, sess->id, &sess->daemon_wait);
		set_is_disc(sess, 1);
		wake_up_interruptible(&sess->daemon_wait);
		up(&sess->msg_sem_master);
		DPRINTF(1, "exiting %s: down_interruptible interrupted\n",
								__FUNCTION__);
		return (-EINTR);
	}

	/* We might be woken up if daemon dies */
	if (!has_daemon(sess)) {
		WLD_FILL_ENTRY(0x10000307, sess->id, NULL);
		DPRINTF(1, "exiting %s: error: session doesn't have daemon 2\n",
								__FUNCTION__);
		return (-ENXIO);
	}

	spin_lock_irqsave(&sess->msg_lock_slave, flgs);
	if (!list_empty(&sess->messages_to_slave)) {
		item = sess->messages_to_slave.next;
		list_del(item);
	}
	else {
		/* should never happen? */
		item = NULL;
	}
	spin_unlock_irqrestore(&sess->msg_lock_slave, flgs);

	if (item == NULL) {
		WLD_FILL_ENTRY(0x80000308, sess->id, NULL);
		DPRINTF(1, "exiting %s: messages_to_slave is empty, returning EAGAIN\n",
				__FUNCTION__);
		return (-EAGAIN);
	}

	msg = list_entry(item, struct uta_msg, list);
	memmove(umsg, &(msg->pmsg), sizeof (utio_pmsg_t));
	kfree(msg);

	WLD_FILL_ENTRY(0x309, sess->id, NULL);
	DPRINTF(2, "exiting %s(): sess id %d, ret 0\n", __FUNCTION__, sess->id);
	return (0);
}


/*
 * Find the session with ID sessid on session_list that has a daemon
 * connection (clone device) associated with it.
 * If found, call ref_session(sess) and return a pointer to the session
 * Else return NULL
 *  
 * Called by slave_open().
 * Must be reentrant because we get here through utio_open().
 */
static struct uta_session *
find_session(int sessid)
{
	struct list_head *pos;
	unsigned long flgs;

	WLD_FILL_ENTRY(0x400, sessid, NULL);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__, sessid);

	spin_lock_irqsave(&sesslist_lock, flgs);
	list_for_each(pos, &session_list) {
		struct uta_session *sess;

		sess = list_entry(pos, struct uta_session, list);
		if (sess->id == sessid && has_daemon(sess)) {
			ref_session(sess);
			spin_unlock_irqrestore(&sesslist_lock, flgs);
			WLD_FILL_ENTRY(0x401, sessid, NULL);
			DPRINTF(3, "exiting %s: found session %d\n",
						__FUNCTION__, sessid);
			return (sess);
		}
	}
	spin_unlock_irqrestore(&sesslist_lock, flgs);

	WLD_FILL_ENTRY(0x80000402, sessid, NULL);
	DPRINTF(1, "exiting %s(): error: session %d not found\n",
		__FUNCTION__, sessid);
	return (NULL);
}

/*
 * Search session_bits[] for an unused session ID.
 * If found, return session ID
 * Else return -1
 *
 * Called by do_newsession().
 */
static int
new_sessionid(void)
{
	unsigned long flgs;
	int idx;
	int base = 0;

	WLD_FILL_ENTRY(0x500, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	spin_lock_irqsave(&sessbits_lock, flgs);
	for (idx = 0; idx < sessbits_count; idx++) {
		if (session_bits[idx] != ~0UL) {
			int bit;
			/* ffz - find first zero in word. */
			bit = ffz(session_bits[idx]);
			DPRINTF(3, "%s: setting sessbit %d:%d\n",
						__FUNCTION__, idx, bit);
			__set_bit(bit, &session_bits[idx]);
			spin_unlock_irqrestore(&sessbits_lock, flgs);
			WLD_FILL_ENTRY(0x501, (base + bit), NULL);
			return (base + bit);
		}

		base += BITS_PER_LONG;
	}
	spin_unlock_irqrestore(&sessbits_lock, flgs);

	WLD_FILL_ENTRY(0x80000502, NULL, NULL);
	DPRINTF(1, "exiting %s(): error: no available session IDs\n",
		__FUNCTION__);
	return (-1);
}


/*
 * Deallocate a session ID by clearing its bit in session_bits[].
 * Detects if the session ID is already free.
 */
static void
release_sessionid(int sessid)
{
	unsigned long flgs;
	int idx;
	int bit;

	WLD_FILL_ENTRY(0x600, sessid, NULL);
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sessid);

	BUG_ON(sessid >= MAX_SESSIONS);

	idx = sessid / BITS_PER_LONG;
	bit = sessid % BITS_PER_LONG;

	DPRINTF(3, "%s: clearing sessbit %d:%d\n", __FUNCTION__, idx, bit);
	/*
	 * Obtain sessbits_lock because new_sessionid() uses __set_bit()
	 * to set the session ID bit in sessbits_lock.
	 */
	spin_lock_irqsave(&sessbits_lock, flgs);
	if (!__test_and_clear_bit(bit, &session_bits[idx])) {
		/* session ID bit was already cleared */
		DPRINTF(1, "%s(): error: expected session ID to be set\n",
			__FUNCTION__);
		BUG();
	}
	spin_unlock_irqrestore(&sessbits_lock, flgs);

	WLD_FILL_ENTRY(0x601, sessid, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}

/*
 * All the daemon-side ioctl helpers are protected from racing with each
 * other and read/write by daemon_ioctl, if needed.  User-side ioctl helpers
 * are protected from re-entrance by user_ioctl().
 */

/*
 * Called by daemon_ioctl(UTIO_GETDEVID) after acquiring
 * down_write(&conn->syscall_rwsem).
 *
 * Create a new session:
 *   Get an unused session ID.
 *     Return -EBUSY if none are available.
 *   kmalloc() a struct uta_session
 *     Return -ENOMEM if kmalloc() fails.
 *   Initialize/fill in the UTA_DAEMON session structure.
 *   Return the session ID.  The daemon will use it to create/open a
 *   slave device.
 */
static int
do_newsession(struct uta_connect *conn, uid_t owner)
{
	struct uta_session *sess;
	unsigned long flgs;
	int sessid;
	int i;

	WLD_FILL_ENTRY(0x700, NULL, NULL);
	DPRINTF(2, "entering %s(owner=%d)\n", __FUNCTION__, owner);

	if (conn->session) {
		WLD_FILL_ENTRY(0x80000701, NULL, NULL);
		DPRINTF(1, "%s: conn->session (%p) not NULL\n",
				__FUNCTION__, conn->session);
		return (-EBUSY);
	}

	/* allocate a session ID */
	sessid = new_sessionid();
	if (sessid < 0) {
		WLD_FILL_ENTRY(0x80000702, sessid, NULL);
		DPRINTF(1, "%s: can't get a new session ID\n", __FUNCTION__);
		return (-EBUSY);
	}
	DPRINTF(3, "%s: new sessionid = %d\n", __FUNCTION__, sessid);
	WLD_FILL_ENTRY(0x703, sessid, NULL);

	/* allocate session struct */
	sess = kmalloc(sizeof (*sess), GFP_KERNEL);
	if (!sess) {
		WLD_FILL_ENTRY(0x80000704, sess, NULL);
		DPRINTF(1, "%s: can't allocate session struct\n", __FUNCTION__);
		release_sessionid(sessid);
		return (-ENOMEM);
	}

	/* init the session */
	sess->id = sessid;
	sess->owner = owner;
	sess->mode = 0;
	sess->refcount = 0;
	sess->flags = 0;
	set_has_daemon(sess, 1);
	WLD_FILL_ENTRY(0x705, sessid, NULL);
	for (i = 0; i < 2; i++) {
		struct uta_io *io;
		int j;

		io = &sess->ios[i];
		io->user = io->daemon = 0;
		init_waitqueue_head(&io->wait);
		for (j = 0; j < UTA_DEF_NBLKS; j++)
			io->bufs[j] = NULL;
	}
	init_waitqueue_head(&sess->daemon_wait);
	INIT_LIST_HEAD(&sess->messages_to_master);
	spin_lock_init(&sess->msg_lock_master);
	init_MUTEX_LOCKED(&sess->msg_sem_master);

	DPRINTF(3, "%s: session %d created for user %d\n",
					__FUNCTION__, sess->id, owner);

	 /*
	  * Increment the refcount for this session.
	  * It isn't visible yet so we don't need to hold sesslist_lock.
	  */
	ref_session(sess);
	WLD_FILL_ENTRY(0x706, sessid, sess->refcount);

	/* lastly, enlist the session */
	spin_lock_irqsave(&sesslist_lock, flgs);
	list_add_tail(&sess->list, &session_list);
	spin_unlock_irqrestore(&sesslist_lock, flgs);

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_DAEMON;

	WLD_FILL_ENTRY(0x707, sessid, NULL);
	DPRINTF(2, "exiting %s(): sess id %d\n", __FUNCTION__, sessid);
	return (sessid);
}


/*
 * Called by unref_session(sess) when the session's refcount is decremented
 * to 0.
 * sesslist_lock is already held.
 *
 * Remove the session from the global list.
 * Release the session ID so that it can be re-used.
 * Free the session's buffers and any remaining messages.
 * Free the session structure.
 */
static void
end_session(struct uta_session *sess)
{
	WLD_FILL_ENTRY(0x800, sess->id, NULL);
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	list_del(&sess->list);
	release_sessionid(sess->id);
	free_ios(sess);
	free_all_msgs(sess);

	kfree(sess);

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}


/*
 * Called from alloc_ios() to initialize a newly kmalloc()'d uta_buffer.
 * Must be reentrant as we get here from utio_open().
 */
static void
init_one_buf(struct uta_buffer *buf, int mode)
{
	WLD_FILL_ENTRY(0x900, buf, NULL);
	DPRINTF(2, "entering %s(mode=%d)\n", __FUNCTION__, mode);

	if (mode == MODE_READ) {
		/* read buffers start out owned by the daemon */
		atomic_set(&buf->owner, UTA_DAEMON);
	} else if (mode == MODE_WRITE) {
		/* write buffers start out owned by the user */
		atomic_set(&buf->owner, UTA_USER);
	} else {
		DPRINTF(1, "%s: unknown mode %d\n", __FUNCTION__, mode);
		return;
	}
	buf->offset = 0;
	buf->bytes = 0;
	memset(buf->data, 0, sizeof (buf->data));

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}


/*
 * Called from alloc_ios() to initialize a struct uta_io.
 * Must be reentrant as we get here from utio_open().
 */
static void
init_one_io(struct uta_io *io)
{
	WLD_FILL_ENTRY(0xa00, io, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	io->user = io->daemon = 0;
	init_waitqueue_head(&io->wait);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}


/*
 * Called by free_ios() for each of a session's struct uta_io.
 * Called by alloc_ios() to free buffers that are not needed.
 *
 * Must be reentrant as we could get here from utio_open() through alloc_ios().
 *
 * Must check that buffers are allocated before freeing them because we may
 * have been called with a struct uta_io that does not have any buffers
 * associated with it.
 */
static void
free_one_io(struct uta_io *io)
{
	int i;
	struct uta_buffer *tmp;

	WLD_FILL_ENTRY(0xb00, io, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	for (i = 0; i < UTA_DEF_NBLKS; i++) {
		if (io->bufs[i]) {
			tmp = io->bufs[i];
			io->bufs[i] = NULL;
			kfree(tmp);
		}
	}

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}

/*
 * Called by end_session() to free all of the session's struct uta_io.
 * The work is done by free_one_io().
 */
static void
free_ios(struct uta_session *sess)
{
	WLD_FILL_ENTRY(0xc00, sess->id, NULL);
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	free_one_io(&sess->ios[IO_READ]);
	free_one_io(&sess->ios[IO_WRITE]);

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}

/*
 * Called by utio_release() to wake up potenial waiter on sess->ios[i].wait
 * in:
 *    user_read() and user_write().
 *    poll_wait() in user_poll() or daemon_poll().
 */
static void
wakeup_ios(struct uta_session *sess)
{
	int i;

	WLD_FILL_ENTRY(0xd00, sess->id, NULL);
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	for (i = 0; i < 2; i++) {
		WLD_FILL_ENTRY(0xd01, sess->id, &sess->ios[i].wait);
		wake_up_all(&sess->ios[i].wait);
	}

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}


/*
 * Called by slave_open() to allocate and initialize a buffer set for a session.
 * Must be reentrant as we get here from utio_open().
 *
 * If MODE_READ
 *   kmalloc() sess->ios[IO_READ] buffers if they have not been previously
 *   allocated.
 *     if error, return -ENOMEM
 *   call init_one_io() to initialize the sess->ios[IO_READ] buffers.
 *   free sess->ios[IO_WRITE] buffers if they exist.
 * If MODE_WRITE
 *   kmalloc() sess->ios[IO_WRITE] buffers if they have not been previously
 *   allocated.
 *     if error, return -ENOMEM
 *   call init_one_io() to initialize the sess->ios[IO_WRITE] buffers.
 *   free sess->ios[IO_READ] buffers if they exist.
 */
static int
alloc_ios(struct uta_session *sess)
{
	int r = 0;
	int p = 0;

	WLD_FILL_ENTRY(0xe00, sess->id, NULL);
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	if (sess->mode & MODE_READ) {
		struct uta_io *io = &sess->ios[IO_READ];

		for (r = 0; r < UTA_DEF_NBLKS; r++) {
			struct uta_buffer *buf;

			if (io->bufs[r] == NULL) {
				buf = kmalloc(sizeof (*buf), GFP_KERNEL);
				if (!buf)
					goto readbuf_err;
				io->bufs[r] = buf;
			}
			init_one_buf(io->bufs[r], MODE_READ);
		}
		init_one_io(io);
	} else {
		free_one_io(&sess->ios[IO_READ]);
	}

	if (sess->mode & MODE_WRITE) {
		struct uta_io *io = &sess->ios[IO_WRITE];

		for (p = 0; p < UTA_DEF_NBLKS; p++) {
			struct uta_buffer *buf;

			if (io->bufs[p] == NULL) {
				buf = kmalloc(sizeof (*buf), GFP_KERNEL);
				if (!buf)
					goto writebuf_err;
				io->bufs[p] = buf;
			}
			init_one_buf(io->bufs[p], MODE_WRITE);
		}
		init_one_io(io);
	} else {
		free_one_io(&sess->ios[IO_WRITE]);
	}

	WLD_FILL_ENTRY(0xe01, sess->id, NULL);
	DPRINTF(2, "exiting %s()=0\n", __FUNCTION__);
	return (0);

writebuf_err:
	while (--p >= 0) {
		DPRINTF(1, "%s: freeing write buf[%d]\n", __FUNCTION__, p);
		kfree(sess->ios[IO_WRITE].bufs[p]);
	}

readbuf_err:
	while (--r >= 0) {
		DPRINTF(1, "%s: freeing read buf[%d]\n", __FUNCTION__, r);
		kfree(sess->ios[IO_READ].bufs[r]);
	}
 
	/* if we got here, something failed to alloc */
	WLD_FILL_ENTRY(0x80000e02, sess->id, NULL);
	DPRINTF(1, "some alloc failed, exiting %s()\n", __FUNCTION__);
	return (-ENOMEM);
}


/*
 * Send a message to the master (daemon).
 * Called from several places including slave_open().
 * Must be reentrant because we can get here from utio_open().
 *
 * If this session does not have a daemon:
 *   return -ENXIO
 * kmalloc() a struct uta_msg
 *   return -ENOMEM if error
 * Copy pmsg into message->pmsg
 * Add the message to the session's messages_to_master list
 * Wake up the daemon if it is sleeping on this session's daemon_wait queue.
 * Wake up daemon waiting on this session's msg_sem_master semaphore.
 */
static int
send_msg_to_master(struct uta_session *sess, utio_pmsg_t *pmsg)
{
	struct uta_msg *msg;
	unsigned long flgs;

	WLD_FILL_ENTRY(0xf00, sess->id, NULL);
#ifdef	PMSG_SEQ_DEBUGGING
	DPRINTF(2, "entering %s(sess=%p, sessid=%d, msgid=%d, cmd=%d, "
		"datasize=%d, seq=%d)\n",
		__FUNCTION__, sess, sess->id, pmsg->msgtype, pmsg->cmd,
		pmsg->datasize, pmsg->seq);
#else	/* ifndef PMSG_SEQ_DEBUGGING */
	DPRINTF(2, "entering %s(sess=%p, sessid=%d, msgid=%d, cmd=%d, "
		"datasize=%d)\n",
		__FUNCTION__, sess, sess->id, pmsg->msgtype, pmsg->cmd,
		pmsg->datasize);
#endif	/* PMSG_SEQ_DEBUGGING */

	if (!has_daemon(sess)) {
		WLD_FILL_ENTRY(0x80000f01, sess->id, NULL);
		DPRINTF(1, "%s: no has daemon\n", __FUNCTION__);
		return (-ENXIO);
	}

	msg = kmalloc(sizeof (*msg), GFP_KERNEL);
	if (!msg) {
		WLD_FILL_ENTRY(0x80000f02, sess->id, NULL);
		DPRINTF(1, "%s: can't allocate message\n", __FUNCTION__);
		return (-ENOMEM);
	}

	memmove(&msg->pmsg, pmsg, sizeof (utio_pmsg_t));

	spin_lock_irqsave(&sess->msg_lock_master, flgs);
	list_add_tail(&msg->list, &sess->messages_to_master);
	spin_unlock_irqrestore(&sess->msg_lock_master, flgs);

	WLD_FILL_ENTRY(0xf03, sess->id, &sess->daemon_wait);
	wake_up_interruptible(&sess->daemon_wait);
	up(&sess->msg_sem_master);
 
	WLD_FILL_ENTRY(0xf04, sess->id, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (0);
}

/*
 * Called from daemon_ioctl() and compat_daemon_ioctl() to send a message
 * to a slave device session.
 *
 * If this session does not have a slave device
 *   return ENXIO.
 * kmalloc a struct uta_msg
 *   return -ENOMEM on error.
 * If flagkptr, pmsg is from the daemon.  Copy pmsg to message->pmsg.
 * Else use copy_from_user() to copy pmsg to message->pmsg.
 *   return -EFAULT on error.
 * Add message to session's message list
 * Wake up waiters on this session's msg_sem_slave semaphore.
 */
static int
send_msg_to_slave(struct uta_session *sess, utio_pmsg_t *pmsg, int flagkptr)
{
	struct uta_msg *msg;
	unsigned long flgs;

	WLD_FILL_ENTRY(0x1000, sess->id, NULL);
#ifdef	PMSG_SEQ_DEBUGGING
	DPRINTF(2, "entering %s(sess=%p, sessid=%d, msgid=%d, cmd=%d, "
		"datasize=%d, seq=%d)\n",
		__FUNCTION__, sess, sess->id, pmsg->msgtype, pmsg->cmd,
		pmsg->datasize, pmsg->seq);
#else	/* ifndef PMSG_SEQ_DEBUGGING */
	DPRINTF(2, "entering %s(sess=%p, sessid=%d, msgid=%d, cmd=%d, "
		"datasize=%d)\n",
		__FUNCTION__, sess, sess->id, pmsg->msgtype, pmsg->cmd,
		pmsg->datasize);
#endif	/* PMSG_SEQ_DEBUGGING */

	if (!has_daemon(sess)) {
		WLD_FILL_ENTRY(0x80001001, sess->id, NULL);
		DPRINTF(1, "%s: session does not have a daemon\n", __FUNCTION__);
		return (-ENXIO);
	}

	msg = kmalloc(sizeof (*msg), GFP_KERNEL);
	if (!msg) {
		WLD_FILL_ENTRY(0x80001002, sess->id, NULL);
		DPRINTF(1, "%s: can't allocate message\n", __FUNCTION__);
		return (-ENOMEM);
	}

	if (flagkptr)
	    memmove(&msg->pmsg, pmsg, sizeof (utio_pmsg_t));
	else {
	    if (copy_from_user(&msg->pmsg, pmsg, sizeof (utio_pmsg_t))) {
		    kfree(msg);
		    WLD_FILL_ENTRY(0x80001003, sess->id, NULL);
		    DPRINTF(1, "%s: copy_from_user failed\n", __FUNCTION__);
		    return (-EFAULT);
	    }
	}
	spin_lock_irqsave(&sess->msg_lock_slave, flgs);
	list_add_tail(&msg->list, &sess->messages_to_slave);
	spin_unlock_irqrestore(&sess->msg_lock_slave, flgs);

	up(&sess->msg_sem_slave);

	WLD_FILL_ENTRY(0x1004, sess->id, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (0);
}

/*
 * Called by end_session() and utio_release() to free messages that may
 * remain on the session's messages_to_master and messages_to_slave lists.
 * Remaining messages must be freed because we are about to destroy/free
 * the session.
 *
 * end_session() calls us while holding the sesslist_lock spinlock.
 * We obtain the sess->msg_lock_master and sess->msg_lock_slave spinlocks.
 */
static void
free_all_msgs(struct uta_session *sess)
{
	struct list_head *tmp, *tmp2;
	unsigned long flgs;

	WLD_FILL_ENTRY(0x2000, sess->id, NULL);
	DPRINTF(2, "entering %s(): sessid=%d\n", __FUNCTION__, sess->id);

	spin_lock_irqsave(&sess->msg_lock_master, flgs);
	list_for_each_safe(tmp, tmp2, &sess->messages_to_master) {
		struct uta_msg *msg;
		msg = list_entry(tmp, struct uta_msg, list);
		list_del(tmp);
		kfree(msg);
	}
	spin_unlock_irqrestore(&sess->msg_lock_master, flgs);

	if (!has_user(sess)) {
		WLD_FILL_ENTRY(0x2001, sess->id, NULL);
		DPRINTF(1, "%s: !has_user(sess)\n", __FUNCTION__);
		return;
	}

	WLD_FILL_ENTRY(0x2002, sess->id, NULL);
	spin_lock_irqsave(&sess->msg_lock_slave, flgs);
	list_for_each_safe(tmp, tmp2, &sess->messages_to_slave) {
		struct uta_msg *msg;
		msg = list_entry(tmp, struct uta_msg, list);
		list_del(tmp);
		kfree(msg);
	}
	spin_unlock_irqrestore(&sess->msg_lock_slave, flgs);

	WLD_FILL_ENTRY(0x2003, sess->id, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}


/*
 * Called by user_read() to give a buffer back to the daemon when it has
 * been emptied.
 * Called by user_write() to transfer a buffer with messages to the daemon.
 *
 * Notify the daemon that a buffer has been transferred by waking poll_wait()
 * waiter in daemon_poll(). 
 *
 * Return true if the buffer was transferred, otherwise 0.
 */
static int
xfer_buf_to_daemon(struct uta_session *sess, int io_idx)
{
	struct uta_io *io = &sess->ios[io_idx];
	struct uta_buffer *buf = io->bufs[io->user];

	WLD_FILL_ENTRY(0x3000, sess->id, NULL);
	DPRINTF(2, "entering %s(): sessid=%d, io_idx=%d\n", __FUNCTION__,
		sess->id, io_idx);

	/* don't send empty buffers */
	if (io_idx == IO_WRITE && buf->offset == 0) {
		WLD_FILL_ENTRY(0x80003001, sess->id, NULL);
		DPRINTF(2, "%s: empty buffers\n", __FUNCTION__);
		return (0);
	}

	/* give it to the daemon */
	buf->offset = 0;
	atomic_set(&buf->owner, UTA_DAEMON);
	WLD_FILL_ENTRY(0x3002, sess->id, &io->wait);
	wake_up_interruptible(&io->wait);

	/* move on to the next buffer */
	io->user++;
	if (io->user == UTA_DEF_NBLKS)
		io->user = 0;

	WLD_FILL_ENTRY(0x3003, sess->id, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (1);
}


/*
 * Called by daemon_read() to give a buffer back to the user when it has
 * been emptied.
 * Called by daemon_write() to transfer a buffer with messages to the user.
 *
 * Notify the user that a buffer has been transferred by waking waiter
 * in user_read() or user_write().
 *
 * Return true if the buffer was transferred, otherwise 0.
 */
static int
xfer_buf_to_user(struct uta_session *sess, int io_idx)
{
	struct uta_io *io;
	struct uta_buffer *buf;

	WLD_FILL_ENTRY(0x4000, sess->id, io_idx);
	DPRINTF(2, "entering %s(): sessid=%d, io_idx=%d\n", __FUNCTION__,
		sess->id, io_idx);

	io = &sess->ios[io_idx];
	buf = io->bufs[io->daemon];

	/* don't send empty buffers */
	if (io_idx == IO_READ && buf->offset == 0) {
		WLD_FILL_ENTRY(0x80004001, sess->id, io_idx);
		DPRINTF(2, "%s: empty buffers\n", __FUNCTION__);
		return (0);
	}

	/* give it to the user */
	buf->offset = 0;
	atomic_set(&buf->owner, UTA_USER);
	WLD_FILL_ENTRY(0x4002, sess->id, &io->wait);
	wake_up_interruptible(&io->wait);

	/* move on to the next buffer */
	io->daemon++;
	if (io->daemon == UTA_DEF_NBLKS)
		io->daemon = 0;

	WLD_FILL_ENTRY(0x4003, sess->id, &io->wait);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (1);
}


/*
 * These four functions [{user,daemon}_{read,write}()] are protected from
 * re-entrancy by the fact that only one user and one daemon are connected
 * to the session, and the conn->read_sem and conn->write_sem are
 * held by the caller of these functions.
 */

/*
 * utio_read() called us after GET_CONNECTION() and GET_SESSION()
 * and with conn->read_sem held.
 *
 * Return -ENXIO if the session does not have a daemon.
 * Send a UTIO_MSG_WRITE message to the daemon/master.
 * If nothing to read
 *   If file has been opened O_NONBLOCK
 *     Return -EAGAIN
 *   Else sleep on sess->ios[IO_READ]
 *   If a signal woke us up
 *     Return -EINTR
 *   If the daemon has died
 *     Return -ENXIO
 *   Copy the data given to us by the daemon into the user's buffer
 *     Return -EFAULT if an error occurred copying the data
 *   If we emptied the buffer
 *     call xfer_buf_to_daemon() to give the buffer back to the daemon
 *   Return the number of bytes read.
 */
static int
user_read(struct file *file, struct uta_connect *conn, char *ubuf, size_t size)
{
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	utio_pmsg_t pmsg;
	DEFINE_WAIT(wait);
	int ret = 0;
	int bytes_read;

	WLD_FILL_ENTRY(0x5000, sess->id, conn);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__, sess->id);

	if (!has_daemon(sess)) {
		WLD_FILL_ENTRY(0x80005001, sess->id, NULL);
		DPRINTF(1, "%s(): exiting. sess id %d has no daemon\n",
			__FUNCTION__, sess->id);
		return (-ENXIO);
	}

	DPRINTF(2, "%s: sessid=%d\n", __FUNCTION__, sess->id);

	// LOCK: sess->io_lock
	/* get the next buffer to read from */
	io = &sess->ios[IO_READ];
	buf = io->bufs[io->user];

	/* tell daemon to start reading from the device */
	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_WRITE;
	pmsg.cmd = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));
	pmsg.uid = CURRENT_UID();
	pmsg.datasize = buf->bytes;
#ifdef	PMSG_SEQ_DEBUGGING
	pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */
	WLD_FILL_ENTRY(0x5002, sess->id, NULL);
	ret = send_msg_to_master(sess, &pmsg);
	if (ret) {
		WLD_FILL_ENTRY(0x80005003, sess->id, ret);
		DPRINTF(1, "%s: error return %d from send_msg_to_master()\n",
			__FUNCTION__, ret);
		DPRINTF(1, "exiting %s(), sess->id=%d\n", __FUNCTION__,
			sess->id);
		return (ret);
	}

	WLD_FILL_ENTRY(0x5004, sess->id, NULL);
	DPRINTF(2, "%s: sent msg to master\n", __FUNCTION__);

	/*
	 * If the buffer owner is not set to UTA_USER, the current
	 * will go to sleep. And this process should be rescheduled
	 * only after getting the interrupt. This process will be
	 * woken up explicitly by the wake_up_interrupt() call.
	 */
	while (atomic_read(&buf->owner) != UTA_USER) {
		WLD_FILL_ENTRY(0x5005, sess->id, NULL);
		DPRINTF(2, "%s: Not UTA_USER\n", __FUNCTION__);
		/* user asked not to block */
		if (file->f_flags & O_NONBLOCK) {
			DPRINTF(2, "%s: nonblocking, returning EAGAIN\n", __FUNCTION__);
			ret = -EAGAIN;
			break;
		}
		WLD_FILL_ENTRY(0x5006, sess->id, &io->wait);
		prepare_to_wait(&io->wait, &wait, TASK_INTERRUPTIBLE);
		if (atomic_read(&buf->owner) != UTA_USER) {
			DPRINTF(2, "%s: calling schedule()\n", __FUNCTION__);
			WLD_FILL_ENTRY(0x5007, sess->id, &io->wait);
			schedule();
		}
		/* a signal can wake us up */
		if (signal_pending(current)) {
			WLD_FILL_ENTRY(0x80005008, sess->id, &io->wait);
			DPRINTF(2, "%s: signal pending\n", __FUNCTION__);
			ret = -EINTR;
			break;
		}
		/* if the daemon dies, we'll be woken up */
		if (!has_daemon(sess)) {
			WLD_FILL_ENTRY(0x80005009, sess->id, &io->wait);
			DPRINTF(1, "%s: no daemon\n", __FUNCTION__);
			ret = -ENXIO;
			break;
		}
	}
	WLD_FILL_ENTRY(0x500a, sess->id, &io->wait);
	DPRINTF(2, "%s: finish_wait\n", __FUNCTION__);
	finish_wait(&io->wait, &wait);

	if (ret < 0) {
		WLD_FILL_ENTRY(0x8000500b, sess->id, ret);
		DPRINTF(1, "failed, exiting %s()\n", __FUNCTION__);
		return (ret);
	}

	/*
	 * if we get here, we have data to read
	 */

	/* clamp the size */
	if (size > buf->bytes)
		size = buf->bytes;

	/* read it */
	WLD_FILL_ENTRY(0x500c, sess->id, NULL);
	DPRINTF(2, "%s: reading data\n", __FUNCTION__);
	if (copy_to_user(ubuf, buf->data + buf->offset, size)) {
		WLD_FILL_ENTRY(0x8000500d, sess->id, NULL);
		DPRINTF(1, "%s: copy_to_user failed\n", __FUNCTION__);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes -= size;
	bytes_read = size;

	/* if we emptied the buffer... */
	if (buf->bytes == 0) {
		WLD_FILL_ENTRY(0x500e, sess->id, NULL);
		xfer_buf_to_daemon(sess, IO_READ);
	}

	WLD_FILL_ENTRY(0x500f, sess->id, bytes_read);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (bytes_read);
}

/*
 * utio_write() called us after GET_CONNECTION() and GET_SESSION()
 * and with conn->write_sem held.
 *
 * Return -ENXIO if the session does not have a daemon.
 * Send a UTIO_MSG_WRITE message to the daemon/master.
 * If there is no room in the buffer to write
 *   If file has been opened O_NONBLOCK
 *     Return -EAGAIN
 *   Else sleep on sess->ios[IO_WRITE]
 *   If a signal woke us up
 *     Return -EINTR
 *   If the daemon has died
 *     Return -ENXIO
 *   Copy the data given to us by the user into the daemon's buffer
 *     Return -EFAULT if an error occurred copying the data
 *   Call xfer_buf_to_daemon() to give the buffer to the daemon
 *   Call send_msg_to_master() to send a UTIO_MSG_READ message to the daemon
 *   Return the number of bytes written.
 */
static int
user_write(struct file *file, struct uta_connect *conn, const char *ubuf,
	   size_t size)
{
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	utio_pmsg_t pmsg;
	DEFINE_WAIT(wait);
	int ret = 0;
	int bytes_written;

	WLD_FILL_ENTRY(0x6000, sess->id, conn);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__, sess->id);

	if (!has_daemon(sess)) {
		WLD_FILL_ENTRY(0x80006001, sess->id, NULL);
		DPRINTF(1, "%s(): exiting. sess id %d has no daemon\n",
			__FUNCTION__, sess->id);
		return (-ENXIO);
	}

	DPRINTF(2, "%s: sessid=%d\n", __FUNCTION__, sess->id);

	// LOCK: sess->io_lock
	/* get the next buffer to write into */
	io = &sess->ios[IO_WRITE];
	buf = io->bufs[io->user];

	DPRINTF(2, "%s: atomic_read\n", __FUNCTION__);
	while (atomic_read(&buf->owner) != UTA_USER) {
		WLD_FILL_ENTRY(0x6002, sess->id, &io->wait);
		DPRINTF(2, "%s: not UTA_USER\n", __FUNCTION__);
		/* user asked not to block */
		if (file->f_flags & O_NONBLOCK) {
			DPRINTF(2, "%s: nonblocking, returning EAGAIN\n", __FUNCTION__);
			WLD_FILL_ENTRY(0x6003, sess->id, &io->wait);
			ret = -EAGAIN;
			break;
		}
		prepare_to_wait(&io->wait, &wait, TASK_INTERRUPTIBLE);
		if (atomic_read(&buf->owner) != UTA_USER) {
			WLD_FILL_ENTRY(0x6004, sess->id, &io->wait);
			schedule();
		}
		/* a signal can wake us up */
		if (signal_pending(current)) {
			WLD_FILL_ENTRY(0x80006005, sess->id, &io->wait);
			DPRINTF(2, "%s: signal pending\n", __FUNCTION__);
			ret = -EINTR;
			break;
		}
		/* if the daemon dies, we'll be woken up */
		if (!has_daemon(sess)) {
			WLD_FILL_ENTRY(0x80006006, sess->id, &io->wait);
			DPRINTF(1, "%s: no daemon\n", __FUNCTION__);
			ret = -ENXIO;
			break;
		}
	}
	WLD_FILL_ENTRY(0x6007, sess->id, &io->wait);
	DPRINTF(2, "%s: finish_wait\n", __FUNCTION__);
	finish_wait(&io->wait, &wait);

	if (ret < 0) {
		WLD_FILL_ENTRY(0x80006008, sess->id, NULL);
		DPRINTF(1, "failed, exiting %s()\n", __FUNCTION__);
		return (ret);
	}

	/*
	 * if we get here, we have space to write
	 */

	/* clamp the size */
	if ((buf->bytes + size) > UTA_DEF_BLKSIZE)
		size = UTA_DEF_BLKSIZE - buf->bytes;

	/* write it */
	DPRINTF(2, "%s: writing data\n", __FUNCTION__);
	if (copy_from_user(buf->data + buf->offset, ubuf, size)) {
		WLD_FILL_ENTRY(0x80006009, sess->id, NULL);
		DPRINTF(1, "copy_from_user failed, exiting %s()\n", __FUNCTION__);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes += size;
	bytes_written = size;

	/* transfer ownership */
	xfer_buf_to_daemon(sess, IO_WRITE);
	/* tell the daemon to start reading the data from the kernel */
	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_READ;
	pmsg.cmd = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));
	pmsg.uid = CURRENT_UID();
	pmsg.datasize = buf->bytes;
#ifdef	PMSG_SEQ_DEBUGGING
	pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */
	WLD_FILL_ENTRY(0x600a, sess->id, NULL);
	DPRINTF(2, "%s: sending msg to master\n", __FUNCTION__);
	ret = send_msg_to_master(sess, &pmsg);
	if (ret) {
		WLD_FILL_ENTRY(0x8000600b, sess->id, ret);
		DPRINTF(1, "%s: error return %d from send_msg_to_master()\n",
			__FUNCTION__, ret);
		DPRINTF(1, "exiting %s(), sess->id=%d\n",
			__FUNCTION__, sess->id);
		return (ret);
	}

	WLD_FILL_ENTRY(0x600c, sess->id, bytes_written);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (bytes_written);
}

/*
 * utio_read() called us after GET_CONNECTION() and GET_SESSION()
 * and with conn->read_sem held.
 *
 * Copy data to the user's buffer.
 *   Return -EFAULT if an error occurs copying the data.
 * Return the number of bytes read.
 */
static int
daemon_read(struct uta_connect *conn, char *ubuf, size_t size)
{
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	int ret;

	WLD_FILL_ENTRY(0x7000, sess->id, conn);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__, sess->id);

	/*
	 * NOTE: don't check has_user() here - the user may have
	 * SYNCed and left
	 */

	// LOCK: sess->io_lock
	/* get the next buffer to write from */
	io = &sess->ios[IO_WRITE];
	buf = io->bufs[io->daemon];

	/* if the buffer is not ready for us to write, it's an error */
	if (!buf || (atomic_read(&buf->owner) != UTA_DAEMON)) {
		DPRINTF(1, "buffer not ready, exiting %s()\n", __FUNCTION__);
		return (0);
	}

	/* clamp the size to this block */
	if (size > buf->bytes)
		size = buf->bytes;

	/* we have data to read */
	if (copy_to_user(ubuf, buf->data + buf->offset, size)) {
		DPRINTF(1, "copy_to_user failed, exiting %s()\n", __FUNCTION__);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes -= size;
	ret = size;

	/* if we emptied the buffer... */
	if (buf->bytes == 0)
		xfer_buf_to_user(sess, IO_WRITE);

	DPRINTF(2, "exiting %s(): sess id %d\n", __FUNCTION__, sess->id);

	return (ret);
}

/*
 * utio_write() called us after GET_CONNECTION() and GET_SESSION()
 * and with conn->write_sem held.
 *
 * Return 0 (no bytes written) if the user half of the session has gone away.
 * Return 0 (no bytes written) if there are no buffers ready to be written to.
 * Copy data from the user's buffer.
 *   Return -EFAULT if an error occurs copying the data.
 * Transfer ownership of the buffer to the user half of the session.
 * Return the number of bytes written.
 */
static int
daemon_write(struct uta_connect *conn, const char *ubuf, size_t size)
{
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	int ret;

	WLD_FILL_ENTRY(0x8000, sess->id, conn);
	DPRINTF(2, "entering %s(): sess id %d\n", __FUNCTION__, sess->id);

	if (!has_user(sess)) {
		DPRINTF(1, "%s(): exiting, has_user not set for sess id %d\n",
			__FUNCTION__, sess->id);
		return (0);
	}

	// LOCK: sess->io_lock
	/* get the next buffer to read into */
	io = &sess->ios[IO_READ];
	buf = io->bufs[io->daemon];

	/* if the buffer is not ready for us to fill, it's an error */
	if (!buf || (atomic_read(&buf->owner) != UTA_DAEMON)) {
		DPRINTF(1, "buffer not ready, exiting %s()\n", __FUNCTION__);
		return (0);
	}

	/* clamp the size to this block */
	if ((buf->bytes + size) > UTA_DEF_BLKSIZE)
		size = UTA_DEF_BLKSIZE - buf->bytes;

	/* write as much as we can */
	if (copy_from_user(buf->data + buf->offset, ubuf, size)) {
		DPRINTF(1, "copy_from_user failed, exiting %s()\n", __FUNCTION__);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes  += size;
	ret = size;

	/* transfer ownership */
	xfer_buf_to_user(sess, IO_READ);

	DPRINTF(2, "exiting %s(): sess id %d\n", __FUNCTION__, sess->id);

	return (size);
}


/*
 * utio_poll() called us after GET_CONNECTION() and GET_SESSION()
 * and with conn->syscall_rwsem held.
 *
 * If file's mode includes FMODE_READ
 *   poll_wait() on sess->ios[IO_READ].wait
 *   If there is data available to read
 *     Add POLLIN and POLLRDNORM to the return events mask
 * If file's mode includes FMODE_WRITE
 *   If there is data available to write
 *     Add POLLOUT and POLLWRNORM to the return events mask
 * If the daemon has died
 *   set the return events mask to POLLHUP
 * Return the return events mask
 */
static unsigned int
user_poll(struct file *file, struct uta_connect *conn,
	  struct poll_table_struct *wait)
{
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	unsigned int mask = 0;

	WLD_FILL_ENTRY(0x9000, sess->id, conn);
	DPRINTF(2, "entering %s(): sess id %d wait %p\n",
		__FUNCTION__, sess->id, wait);

	if (file->f_mode & FMODE_READ) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_READ];
		WLD_FILL_ENTRY(0x9001, wait, &io->wait);
		poll_wait(file, &io->wait, wait);
		buf = io->bufs[io->user];

		/* check if we own the buffer */
		if (atomic_read(&buf->owner) == UTA_USER)
			mask |= POLLIN | POLLRDNORM;
	}

	if (file->f_mode & FMODE_WRITE) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_WRITE];
		WLD_FILL_ENTRY(0x9002, wait, &io->wait);
		poll_wait(file, &io->wait, wait);
		buf = io->bufs[io->user];

		/* check if we own the buffer */
		if (atomic_read(&buf->owner) == UTA_USER)
			mask |= POLLOUT | POLLWRNORM;
	}

	if (!has_daemon(sess))
		mask = POLLHUP;

	WLD_FILL_ENTRY(0x9003, sess->id, mask);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (mask);
}

/*
 * utio_poll() called us after GET_CONNECTION() and GET_SESSION()
 * and with conn->syscall_rwsem held.
 *
 * poll_wait() on sess->daemon_wait wait queue.  We will be woken up if
 * an event needs the attention of the daemon.
 * If the messages_to_master list of messages is not empty
 *    Add POLLPRI to the return events mask
 * If there is a user half of the session:
 *   If file's mode includes FMODE_WRITE
 *     poll_wait() on sess->ios[IO_READ] to wait for a buffer to read.
 *     Add POLLOUT and POLLWRNORM to the return events mask
 * Return the return events mask
 */
static unsigned int
daemon_poll(struct file *file, struct uta_connect *conn,
	    struct poll_table_struct *wait)
{
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	unsigned long flgs;
	unsigned int mask = 0;

	WLD_FILL_ENTRY(0xa000, sess->id, conn);
	DPRINTF(2, "entering %s(): sess id %d wait %p\n",
		__FUNCTION__, sess->id, wait);

	WLD_FILL_ENTRY(0xa001, wait, &sess->daemon_wait);
	poll_wait(file, &sess->daemon_wait, wait);

	spin_lock_irqsave(&sess->msg_lock_master, flgs);
	if (!list_empty(&sess->messages_to_master)) {
		mask |= POLLPRI;
	}
	spin_unlock_irqrestore(&sess->msg_lock_master, flgs);

	/*
	 * If there is no user, means IO buffers not
	 * initialized yet, so return the mask
	 * for the above poll_wait
	 */
	if (!has_user(sess)) {
		WLD_FILL_ENTRY(0xa002, sess->id, mask);
		DPRINTF(2, "%s: exiting, sess id %d has no user\n",
			__FUNCTION__, sess->id);
		return (mask);
	}

	DPRINTF(2, "%s: User has been connected to session\n", __FUNCTION__);
	if (file->f_mode & FMODE_WRITE) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_READ];
		buf = io->bufs[io->daemon];

		WLD_FILL_ENTRY(0xa003, wait, &io->wait);
		poll_wait(file, &io->wait, wait);

		if (buf && (atomic_read(&buf->owner) == UTA_DAEMON))
			mask |= POLLOUT | POLLWRNORM;
	}

	WLD_FILL_ENTRY(0xa004, sess->id, mask);
	DPRINTF(2, "exiting %s(): sess id %d\n", __FUNCTION__, sess->id);

	return (mask);
}


/*
 * Called by user_ioctl() with down_write(&conn->syscall_rwsem) held.
 */
static int
seriald_ioctl(struct inode *inode, struct uta_connect *conn, unsigned int cmd,
	    unsigned long arg)
{
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	int ret;
	uid_t uid;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* user_ioctl() called us after acquiring syscall_rwsem */

	GET_SESSION(conn, sess, NO_RELEASE_FLG);

	if (!has_daemon(sess)) {
		DPRINTF(1, "%s(): exiting. sess id %d has no daemon\n",
			__FUNCTION__, sess->id);
		return (-ENXIO);
	}

	DPRINTF(2, "%s(): sess id %d\n", __FUNCTION__, sess->id);

	uid = CURRENT_UID();

	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_IOCTL;
	pmsg.uid = uid;
	pmsg.datasize = 0;
#ifdef	PMSG_SEQ_DEBUGGING
	pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */
	memset(&pmsg.args, 0, sizeof (pmsg.args));

	switch (cmd) {
		case TCXONC:		// IOW
		case TCFLSH:		// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.tios,
				(int *)arg, sizeof (int)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (int);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}
			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		case TCSETS:		// IOW
		case TCSETSW:		// IOW
		case TCSETSF:		// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.tios,
				(struct termios *)arg,
				sizeof (struct termios)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (struct termios);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		case TCSETA: 		// IOW
		case TCSETAW:		// IOW
		case TCSETAF:		// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.tio, (struct termio *)
					arg, sizeof (struct termio)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (struct termio);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		// case TIOCSPPS:
		case TCSBRK:  		// IO but behaving like IOW
		case TCSBRKP: 		// IOW
		case TIOCSSOFTCAR:	// IOW
		case TIOCMSET:		// IOW
		case TIOCMBIS:		// IOW
		case TIOCMBIC:		// IOW
			DPRINTF(3, "ioctl type is 0x%x", cmd);
			get_user(pmsg.args.intval, (int *)arg);
			DPRINTF(3, " arg=%ld\n", arg);

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (int);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		case TIOCSWINSZ:	// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.twsz,
				(struct winsize *)arg,
				sizeof (struct winsize)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (struct winsize);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		// case TIOCGPPS:
		// case TIOCGPPSEV:
		case TIOCMGET:		// IOR
		case TIOCGSOFTCAR:	// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (put_user(pmsg.args.intval,
					(int *)arg));
			} else {
				return (-1);
			}
		case TIOCGWINSZ:	// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct winsize *)
					arg, &pmsg.args.twsz,
					sizeof (struct winsize)));
			} else {
				return (-1);
			}
		case TCGETS:		// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct termios *)
					arg, &pmsg.args.tios,
					sizeof (struct termios)));
			} else {
				return (-1);
			}
		case TCGETA:		// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct termio *)
					arg, &pmsg.args.tio,
					sizeof (struct termio)));
			} else {
				return (-1);
			}
	}

	DPRINTF(4, " - unknown ioctl type: 0x%x (%d)\n", cmd, cmd);
	DPRINTF(2, "exiting %s(): sess id %d\n", __FUNCTION__, sess->id);
	return (-ENOTTY);
}

/*
 * Called by user_ioctl() with down_write(&conn->syscall_rwsem) held.
 */
static int
paralleld_ioctl(struct inode *inode, struct uta_connect *conn, unsigned int cmd,
	    unsigned long arg)
{
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	uid_t uid;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/*
	 * user_ioctl() calls us after acquiring syscall_rwsem
	 * compat_paralleld_ioctl() calls us with lock_kernel() held
	 */

	GET_SESSION(conn, sess, NO_RELEASE_FLG);

	if (!has_daemon(sess)) {
		DPRINTF(1, "%s(): exiting. sess id %d has no daemon\n",
			__FUNCTION__, sess->id);
		return (-ENXIO);
	}

	DPRINTF(2, "%s(): sess id %d\n", __FUNCTION__, sess->id);
	
        uid = CURRENT_UID();

	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_IOCTL;
	pmsg.uid = uid;
	pmsg.datasize = 0;
#ifdef	PMSG_SEQ_DEBUGGING
	pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */
	memset(&pmsg.args, 0, sizeof (pmsg.args));

	switch (cmd) {
		/* Get the current mode */
		case PPGETMODE:
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (put_user(pmsg.args.intval,
					(int *)arg));
			} else {
				return (-1);
			}

		/*
		 * Sets which IEEE 1284 protocol to use for
		 * the read and write calls
		 */
		case PPSETMODE:
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			get_user(pmsg.args.intval, (int *)arg);

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (int);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (put_user(pmsg.args.intval,
					(int *)arg));
			} else {
				return (-1);
			}

		/*
		 * Retrieves the time-out value for read and
		 * write calls
		 */
		case PPGETTIME:
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct timeval *)
					arg, &pmsg.args.tval,
					sizeof (struct timeval)));
			} else {
				return (-1);
			}
	}

	DPRINTF(4, " - unknown ioctl type: 0x%x (%d)\n", cmd, cmd);
	return (-ENOTTY);
}

#if defined(CONFIG_COMPAT) || defined(CONFIG_SPARC64) || defined(CONFIG_X86_64) || defined(CONFIG_PPC64)

/*
 * Called by utio_compat_ioctl() to handle a COMPAT_PPGETTIME ioctl.
 * utio_compat_ioctl() acquired the kernel lock.
 *
 * If called for an ioctl() other than COMPAT_PPGETTIME, call paralleld_ioctl().
 *
 * COMPAT_PPGETTIME:
 *   send UTIO_MSG_IOCTL to the master/daemon
 *   call do_getmsg_from_master() to wait for the response
 *   copy the reply to the user's buffer
 *   return 0 on success
 *   return -1 or negative errno on error.
 */
static int
compat_paralleld_ioctl(unsigned int cmd, unsigned long arg, struct file *file)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	uid_t uid;
	compat_utio_pmsg_t pmsg32;
	int ret;
	struct inode *inode = file->f_dentry->d_inode;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* utio_compat_ioctl() calls us with lock_kernel() held */

	GET_CONNECTION(file, conn, -ENODEV);

	/* only one ioctl needs conversion */
	if (cmd != COMPAT_PPGETTIME) {
	    DPRINTF(2, "%s(): calling paralleld_ioctl()\n", __FUNCTION__);
	    ret = paralleld_ioctl(inode, conn, cmd, arg);
	    DPRINTF(2, "%s(): exiting, ret = %d\n", __FUNCTION__, ret);
	    RELEASE_CONNECTION(conn);
	    return (ret);
	}

	/*
	 * Retrieves the time-out value for read and write calls
	 */

	GET_SESSION(conn, sess, RELEASE_FLG);

	if (!has_daemon(sess)) {
		DPRINTF(1, "exiting %s: Session doesn't have daemon\n",
			__FUNCTION__);
		RELEASE_CONNECTION(conn);
		return (-ENXIO);
	}

        uid = CURRENT_UID();

	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_IOCTL;
	pmsg.uid = uid;
	pmsg.datasize = 0;
#ifdef	PMSG_SEQ_DEBUGGING
	pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */
	memset(&pmsg.args, 0, sizeof (pmsg.args));

	pmsg.cmd = PPGETTIME;
	DPRINTF(3, "ioctl type is 0x%x\n", pmsg.cmd);

	/* Check the return value of send_msg_to_master */
	if ((ret = send_msg_to_master(sess, &pmsg))) {
		DPRINTF(1, "%s(): exiting, ret %d from send_msg_to_master()\n",
			__FUNCTION__, ret);
		RELEASE_CONNECTION(conn);
		return (ret);
	}

	memset(&pmsg, 0, sizeof (pmsg));
	ret = do_getmsg_from_master(sess, &pmsg,
			uid, UTIO_MSG_IOCTL);
	if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
	    (pmsg.response == UTIO_MSG_ACK))
	{
		COPY_UTIO_PMSG_TVAL(pmsg32.args, pmsg.args)
		ret = copy_to_user((struct compat_timeval *)
			arg, &pmsg32.args.tval,
			sizeof (struct compat_timeval));
		DPRINTF(2, "%s(): exiting, ret = %d\n", __FUNCTION__, ret);
		RELEASE_CONNECTION(conn);
		return (ret);
	} else {
		DPRINTF(1, "%s(): exiting, ret = -1\n", __FUNCTION__);
		RELEASE_CONNECTION(conn);
		return (-1);
	}
}
#endif	/* CONFIG_COMPAT */


/*
 * Called by utio_ioctl() after GET_CONNECTION()
 * It did not acquire syscall_rwsem.
 *
 * Acquires down_write(&conn->syscall_rwsem) 
 */
static int
user_ioctl(struct inode *inode, struct uta_connect *conn, unsigned int cmd,
    unsigned long arg)
{
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	/* will check session if/when we use it */

	ret = -ENOTTY;

	if (MINOR(inode->i_rdev) == UTSERIALD_MINOR) {
		ret = seriald_ioctl(inode, conn, cmd, arg);
	} else if (MINOR(inode->i_rdev) == UTPARALLELD_MINOR) {
		ret = paralleld_ioctl(inode, conn, cmd, arg);
	} else {
		DPRINTF(1, "%s: unknown minor\n", __FUNCTION__);
		up_write(&conn->syscall_rwsem);
		return (-ENXIO);
	}

	up_write(&conn->syscall_rwsem);

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (ret);
}

/*
 * utio_ioctl() called GET_CONNECTION()
 * but did not acquire syscall_rwsem
 * compat_daemon_ioctl() called us with lock_kernel()
 *
 * All the daemon ioctl() helper functions are open to re-entrancy - protect
 * them with conn->syscall_rwsem, if needed.
 */
static int
daemon_ioctl(struct inode *inode, struct file *file, struct uta_connect *conn,
	     unsigned int cmd, unsigned long arg)
{
	struct uta_session *sess;
	int ret;

	WLD_FILL_ENTRY(0x30000, cmd, conn);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* these don't require the session to exist */
	switch (cmd) {
	case UTIO_GETVERSION:
		DPRINTF(2, " - UTIO_GETVERSION %d\n", UTIO_VERSION);
		ret = put_user(UTIO_VERSION, (unsigned long *)arg);
		WLD_FILL_ENTRY(0x30001, UTIO_VERSION, ret);
		return ret;

	case UTIO_GETMAJOR:
		DPRINTF(2, " - UTIO_GETMAJOR %d\n", MAJOR(inode->i_rdev));
		WLD_FILL_ENTRY(0x30002, MAJOR(inode->i_rdev), conn);
		return (MAJOR(inode->i_rdev));

	case UTIO_GETDEVID:
		DPRINTF(2, " - UTIO_GETDEVID\n");

		/*
		 * utio_ioctl() called GET_CONNECTION()
		 * but did not acquire conn->syscall_rwsem.
		 */
		down_write(&conn->syscall_rwsem);
		ret = do_newsession(conn, CURRENT_UID());
		up_write(&conn->syscall_rwsem);
		WLD_FILL_ENTRY(0x30003, ret, conn);
		DPRINTF(2, "exiting %s(): device id %d\n", __FUNCTION__, ret);
		return (ret);
	}

	/*
	 * utio_ioctl() called GET_CONNECTION()
	 * but did not acquire conn->syscall_rwsem.
	 */
	down_read(&conn->syscall_rwsem);
	GET_SESSION_LOCKED(conn, sess, up_read, conn->syscall_rwsem,
			   NO_RELEASE_FLG);
	WLD_FILL_ENTRY(0x30004, sess->id, conn);

	/* these all require the session */
	switch (cmd) {
	case UTIO_GETMSG:
		DPRINTF(3, "- UTIO_GETMSG, sess id %d\n", sess->id);
		ret = do_getmsg_from_slave(sess, (utio_pmsg_t *)arg,
			    file->f_flags & O_NONBLOCK, CURRENT_UID(), 0);
		DPRINTF(2, "exiting %s() UTIO_GETMSG\n", __FUNCTION__);
		up_read(&conn->syscall_rwsem);
		WLD_FILL_ENTRY(0x30005, sess->id, ret);
		return (ret);

	case UTIO_PUTMSG:
		DPRINTF(3, "- UTIO_PUTMSG, sess id %d\n", sess->id);
		ret = send_msg_to_slave(sess, (utio_pmsg_t *)arg, 0);
		if (ret) {
			DPRINTF(2, "%s(): error return %d from send_msg_to_slave()\n",
				__FUNCTION__, ret);
			DPRINTF(2, "exiting %s() UTIO_PUTMSG\n", __FUNCTION__);
			return (ret);
		}
		up_read(&conn->syscall_rwsem);
		WLD_FILL_ENTRY(0x30006, sess->id, ret);
		DPRINTF(2, "exiting %s() UTIO_PUTMSG\n", __FUNCTION__);
		return (0);
	default:
		up_read(&conn->syscall_rwsem);
		break;
	}

	DPRINTF(4, " - unknown ioctl: 0x%08x (%d) sess id %d\n",
		cmd, cmd, sess->id);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);

	WLD_FILL_ENTRY(0x80030007, sess->id, cmd);
	return (-ENOTTY);
}

#if defined(CONFIG_COMPAT) || defined(CONFIG_SPARC64) || defined(CONFIG_X86_64) || defined(CONFIG_PPC64)

/*
 * Called by utio_compat_ioctl() with the kernel lock held.
 *
 * This function converts 32-bit ioctl arguments to/from 64-bit arguments.
 */
static int
compat_daemon_ioctl(unsigned int cmd, unsigned long arg, struct file *file)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	utio_pmsg_t		parm64;
	compat_utio_pmsg_t	parm32;
	int ret = -ENOTTY;
	struct inode *inode = file->f_dentry->d_inode;

	/* utio_compat_ioctl calls us with lock_kernel() held */

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	GET_CONNECTION(file, conn, -ENODEV);

	/* only convert ioctls that need it */
	if (cmd == COMPAT_UTIO_GETMSG || cmd == COMPAT_UTIO_PUTMSG)
	{
		/* these all require the session */
		down_read(&conn->syscall_rwsem);
		GET_SESSION_LOCKED(conn, sess, up_read, conn->syscall_rwsem,
				   RELEASE_FLG);

		if (copy_from_user(&parm32, (struct compat_utio_pmsg_t *)arg, sizeof(compat_utio_pmsg_t))) {
		    up_read(&conn->syscall_rwsem);
		    RELEASE_CONNECTION(conn);
		    return -EFAULT;
		}
		DPRINTF_UTIO_PMSG(3, parm32, "parm32 after copy from user");
		COPY_UTIO_PMSG_T(parm64, parm32);
		DPRINTF_UTIO_PMSG(3, parm64, "parm64 after conversion");

		/* now do the actual ioctl with copied arguments */
		/* this is a modified copy of code from daemon_ioctl() above */
		/* modifications are use of parm64 instead of arg and pass flagkptr=1 */
		DPRINTF(2, "%s() ", __FUNCTION__);
		switch (cmd) {
		case COMPAT_UTIO_GETMSG:
			DPRINTF(3, "- UTIO_GETMSG\n");
			ret = do_getmsg_from_slave(sess, &parm64,
				    file->f_flags & O_NONBLOCK, CURRENT_UID(), 1);
			if (ret < 0) {
				up_read(&conn->syscall_rwsem);
				RELEASE_CONNECTION(conn);
				return (ret);
			}
			break;
		case COMPAT_UTIO_PUTMSG:
			DPRINTF(3, "- UTIO_PUTMSG\n");
			ret = send_msg_to_slave(sess, &parm64, 1);
			if (ret) {
				DPRINTF(2, "%s(): error return %d from send_msg_to_slave()\n",
					__FUNCTION__, ret);
				DPRINTF(2, "exiting %s()\n", __FUNCTION__);
				return (ret);
			}
			ret = 0;
			break;
		}

		DPRINTF_UTIO_PMSG(2, parm64, "parm64 after ioctl code");
		COPY_UTIO_PMSG_T(parm32, parm64);
		DPRINTF_UTIO_PMSG(2, parm32, "parm32 before copy to user");
		if (copy_to_user((struct compat_utio_pmsg_t *)arg, &parm32, sizeof(compat_utio_pmsg_t))) {
		    up_read(&conn->syscall_rwsem);
		    RELEASE_CONNECTION(conn);
		    return -EFAULT;
		}
		up_read(&conn->syscall_rwsem);
		RELEASE_CONNECTION(conn);
		return (ret);
	}

	/* all other ioctls do not need conversion */
	DPRINTF(2, "%s(): calling daemon_ioctl()\n", __FUNCTION__);
	ret = daemon_ioctl(inode, file, conn, cmd, arg);
	RELEASE_CONNECTION(conn);
	DPRINTF(2, "exiting %s(): ret == %d\n", __FUNCTION__, ret);
	return (ret);
}
#endif	/* CONFIG_COMPAT */


/*
 * Called by utio_open() to set up the user half of a connection.
 * Must be reentrant.
 *
 * Find the session with session ID sessid.  It was created by the UTIO_GETDEVID
 * daemon ioctl.
 *   Return -ENODEV if it doesn't exist.
 * Lock the session so that there is only one user side.
 *   Return -EBUSY if there already was a user side.
 * Call alloc_ios() to set up buffers and user metadata.
 * set conn->session = the address of the session structure (struct uta_session)
 * This connection is the user side to the session so set conn->role = UTA_USER
 * Send a UTIO_MSG_OPEN message to the master/daemon.
 * Call do_getmsg_from_master() to wait for the reply.
 * Return 0 on success.
 * Return -errno on error.
 */
static int
slave_open(struct uta_connect *conn, int sessid, int readfrom, int writeto)
{
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	uid_t uid;
	int ret;

	WLD_FILL_ENTRY(0xb000, sessid, conn);
	DPRINTF(2, "entering %s(conn=%p, sessid=%d, readfrom=%d, writeto=%d)\n",
		__FUNCTION__, conn, sessid, !!readfrom, !!writeto);

	/* find the session */
	sess = find_session(sessid);
	if (!sess) {
		WLD_FILL_ENTRY(0x8000b001, sessid, sess);
		DPRINTF(1, "%s: can't find session %d\n", __FUNCTION__, sessid);
		return (-ENODEV);
	}

	/* one user at a time - set this to indicate an unready user */
	if (set_user_lock(sess, 1)) {
		WLD_FILL_ENTRY(0x8000b002, sessid, sess);
		DPRINTF(1, "%s: session %d already has a user\n",
			__FUNCTION__, sessid);
		unref_session(sess);
		return (-EBUSY);
	}

	/* set up the user side of the session - we want to do this last */
	sess->mode = (readfrom ? MODE_READ : 0) | (writeto ? MODE_WRITE : 0);
	ret = alloc_ios(sess);
	if (ret < 0) {
		WLD_FILL_ENTRY(0x8000b003, sessid, sess);
		DPRINTF(1, "%s(): error return from alloc_ios(), ret == %d\n",
			__FUNCTION__, ret);
		goto out_unlock;
	}

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_USER;
	INIT_LIST_HEAD(&sess->messages_to_slave);
	spin_lock_init(&sess->msg_lock_slave);
	init_MUTEX_LOCKED(&sess->msg_sem_slave);

	/* leave the session ref'ed - it is now ready */
	set_has_user(sess, 1);
	WLD_FILL_ENTRY(0xb004, sessid, sess);

        uid = CURRENT_UID();

	/* tell the daemon that we have joined the session */
	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_OPEN;
	pmsg.cmd = sessid;
	memset(&pmsg.args, 0, sizeof (pmsg.args));
	pmsg.uid = uid;
	pmsg.datasize = 0;
#ifdef	PMSG_SEQ_DEBUGGING
	pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */

	/* Check the return value of send_msg_to_master */
	WLD_FILL_ENTRY(0xb005, sessid, sess);
	if ((ret = send_msg_to_master(sess, &pmsg))) {
		WLD_FILL_ENTRY(0x8000b006, sessid, ret);
		DPRINTF(1, "%s(): error return from send_msg_to_master(), ret == %d\n",
			__FUNCTION__, ret);
		goto out_unref_user_sess;
	}

	memset(&pmsg, 0, sizeof (pmsg));
	WLD_FILL_ENTRY(0xb007, sessid, sess);
	ret = do_getmsg_from_master(sess, &pmsg, uid,
				    UTIO_MSG_OPEN);
	if (is_disc(sess)) {
		WLD_FILL_ENTRY(0x8000b008, sessid, NULL);
		DPRINTF(1, "%s(): sess id %d disconnected\n",
			__FUNCTION__, sess->id);
		ret = -ENXIO;
		goto out_unref_user_sess;
	}
	if (!ret) {
		if ((pmsg.msgtype == UTIO_MSG_OPEN) &&
		    (pmsg.response == UTIO_MSG_ACK)) {
			WLD_FILL_ENTRY(0xb009, sessid, NULL);
			DPRINTF(3, "%s: ACK received for UTIO_MSG_OPEN\n", __FUNCTION__);
			return (0);
		} else {
			WLD_FILL_ENTRY(0x8000b00a, sessid, NULL);
			DPRINTF(1, "%s: No ACK received from daemon\n", __FUNCTION__);
			ret = -ENODEV;
		}
	}

out_unref_user_sess:
	DPRINTF(1, "%s(): clearing has_user for sess id %d\n",
		__FUNCTION__, sess->id);
	set_has_user(sess, 0);
out_unlock:
	DPRINTF(1, "%s(): clearing user_lock for sess id %d\n",
		__FUNCTION__, sess->id);
	if (!set_user_lock(sess, 0)) {
		/* user_lock was set above.  Should still be set */
		DPRINTF(1, "%s(): error: expected user lock to be set\n",
			__FUNCTION__);
		BUG();
	}
	unref_session(sess);

	WLD_FILL_ENTRY(0x8000b00b, sessid, ret);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (ret);
}


/*
 * the fops functions
 *
 * security: if we get to any of these functions, the FS layer has confirmed
 * the permissions on the device node.
 */

/*
 * Acquires conn->read_sem
 * If conn->role == UTA_USER
 *   call user_read()
 * conn->role == UTA_DAEMON
 *   call daemon_read()
 */
static ssize_t
utio_read(struct file *file, char *ubuf, size_t size, loff_t *ppos)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	int ret;

	WLD_FILL_ENTRY(0xc000, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	GET_CONNECTION(file, conn, -ENODEV);

	/* block other callers of read() */
	if (down_interruptible(&conn->read_sem)) {
		/* At this point it is OK to restart the system call */
		WLD_FILL_ENTRY(0x8000c001, conn, NULL);
		RELEASE_CONNECTION(conn);
		return (-ERESTARTSYS);
	}

	GET_SESSION_LOCKED(conn, sess, up, conn->read_sem, RELEASE_FLG);

	if (*ppos != file->f_pos) {
		up(&conn->read_sem);
		WLD_FILL_ENTRY(0x8000c002, conn, sess->id);
		RELEASE_CONNECTION(conn);
		return (-ESPIPE);
	}
	if (size == 0) {
		up(&conn->read_sem);
		WLD_FILL_ENTRY(0x8000c003, conn, sess);
		RELEASE_CONNECTION(conn);
		return (0);
	}

	if (conn->role == UTA_USER)
		ret = user_read(file, conn, ubuf, size);
	else if (conn->role == UTA_DAEMON)
		ret = daemon_read(conn, ubuf, size);
	else
		ret = (-ENXIO);

	WLD_FILL_ENTRY(0xc004, conn, sess->id);
	up(&conn->read_sem);
	RELEASE_CONNECTION(conn);

	if (ret) {
		DPRINTF(1, "%s() exiting, ret == %d\n", __FUNCTION__, ret);
	}
	else {
		DPRINTF(2, "%s() exiting, ret == 0\n", __FUNCTION__);
	}
	return (ret);
}

/*
 * Acquires conn->write_sem
 * If conn->role == UTA_USER
 *   call user_write()
 * else if conn->role == UTA_DAEMON
 *   call daemon_write()
 * else
 *   write() was made to the clone device after it was opened but
 *   before ioctl(UTIO_GETDEVID) has called do_newsession().
 *   return (-ENXIO)
 */
static ssize_t
utio_write(struct file *file, const char *ubuf, size_t size, loff_t *ppos)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	int ret;

	WLD_FILL_ENTRY(0xd000, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	GET_CONNECTION(file, conn, -ENODEV);

	/* block other callers of write() */
	if (down_interruptible(&conn->write_sem)) {
		/* At this point it is OK to restart the system call */
		WLD_FILL_ENTRY(0x8000d001, conn, NULL);
		RELEASE_CONNECTION(conn);
		return (-ERESTARTSYS);
	}

	GET_SESSION_LOCKED(conn, sess, up, conn->write_sem, RELEASE_FLG);

	if (*ppos != file->f_pos) {
		up(&conn->write_sem);
		WLD_FILL_ENTRY(0x8000d002, conn, sess->id);
		RELEASE_CONNECTION(conn);
		return (-ESPIPE);
	}
	if (size == 0) {
		up(&conn->write_sem);
		WLD_FILL_ENTRY(0x8000d003, conn, sess->id);
		RELEASE_CONNECTION(conn);
		return (0);
	}

	if (conn->role == UTA_USER)
		ret = user_write(file, conn, ubuf, size);
	else if (conn->role == UTA_DAEMON)
		ret = daemon_write(conn, ubuf, size);
	else
		ret = (-ENXIO);

	up(&conn->write_sem);

	WLD_FILL_ENTRY(0xd004, NULL, NULL);
	RELEASE_CONNECTION(conn);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (ret);
}


/*
 * If conn->role == UTA_USER
 *   down_write(&conn->syscall_rwsem)
 *   call user_poll()
 * else if conn->role == UTA_DAEMON
 *   down_read(&conn->syscall_rwsem)
 *   call daemon_poll()
 * else
 *   do nothing.  Poll was made to the clone device after it was opened but
 *   before ioctl(UTIO_GETDEVID) has called do_newsession().
 */
static unsigned int
utio_poll(struct file *file, struct poll_table_struct *wait)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	unsigned int ret = 0;

	WLD_FILL_ENTRY(0xe000, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	GET_CONNECTION(file, conn, POLLHUP);

	if (conn->role == UTA_USER) {
		down_write(&conn->syscall_rwsem);
		GET_SESSION_LOCKED(conn, sess, up_write, conn->syscall_rwsem,
				   RELEASE_FLG);
		ret = user_poll(file, conn, wait);
		up_write(&conn->syscall_rwsem);
	} else if (conn->role == UTA_DAEMON) {
		down_read(&conn->syscall_rwsem);
		GET_SESSION_LOCKED(conn, sess, up_read, conn->syscall_rwsem,
				   RELEASE_FLG);
		ret = daemon_poll(file, conn, wait);
		up_read(&conn->syscall_rwsem);
	}
	else {
		/* conn->role == UTA_NONE */
	}

	WLD_FILL_ENTRY(0xe001, NULL, NULL);
	DPRINTF(2, "exiting %s(), ret = %d\n", __FUNCTION__, ret);
	RELEASE_CONNECTION(conn);
	return (ret);
}

/*
 * if conn->role == UTA_USER
 *   call user_ioctl()
 * else
 *   either:
 *     - conn->role == UTA_DAEMON
 *     - conn->role == UTA_NONE
 *       An ioctl() made to the clone device (/dev/utio) before a session
 *       has been established.  The ioctl is probably UTIO_GETVERSION,
 *       UTIO_GETMAJOR (both do not require a session) or UTIO_GETDEVID.
 *       UTIO_GETDEVID is used to set up a new session and
 *       set conn->role = UTA_DAEMON.
 */
static int
utio_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	int ret;
	struct uta_connect *conn;

	WLD_FILL_ENTRY(0xf000, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	GET_CONNECTION(file, conn, -ENODEV);

	/*
	 * syscall_rwsem locks are acquired
	 * inside user_ioctl() or daemon_ioctl(),
	 * so no locks are needed here.
	 */
	if (conn->role == UTA_USER)
		ret = user_ioctl(inode, conn, cmd, arg);
	else
		ret = daemon_ioctl(inode, file, conn, cmd, arg);

	WLD_FILL_ENTRY(0xf001, NULL, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	RELEASE_CONNECTION(conn);
	return (ret);
}

/* Acquires the kernel lock */
static long
utio_compat_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	lock_kernel();

	switch (cmd) {

#if defined(CONFIG_COMPAT) || defined(CONFIG_SPARC64) || defined(CONFIG_X86_64) || defined(CONFIG_PPC64)

	case COMPAT_UTIO_GETMSG:	/* fall through */
	case COMPAT_UTIO_PUTMSG:
		ret = compat_daemon_ioctl(cmd, arg, file);
		break;
	case COMPAT_PPGETTIME:
		ret = compat_paralleld_ioctl(cmd, arg, file);
		break;

#endif	/* CONFIG_COMPAT */

	default:
		ret = utio_ioctl(file->f_dentry->d_inode, file, cmd, arg);
		break;
	}
	unlock_kernel();

	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (ret);
}

/*
 * Is reentrant.
 *
 * Normally one of two special files is being opened:
 *   clone device (/dev/utio):
 *     kmalloc() and initialize a "conn->role = UTA_NONE" connection
 *     set file->private_data = a pointer to the initialized connection
 *     don't set up a session.  That will be done by an ioctl(UTIO_GETDEVID)
 *   slave device (e.g. .../devices/serial:a%=0)
 *     get the session ID from the slave device name.
 *     call slave_open() to allocate and set up the session and set
 *     conn->role = UTA_USER.
 *     set file->private_data = a pointer to the initialized connection
 */
static int
utio_open(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	unsigned long flgs;
	int sessid;
	int r;

	WLD_FILL_ENTRY(0x10000, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* allocate our connection */
	conn = kmalloc(sizeof (*conn), GFP_KERNEL);
	if (!conn) {
		WLD_FILL_ENTRY(0x80010001, NULL, NULL);
		DPRINTF(1, "exiting %s(): can't allocate connection\n",
			__FUNCTION__);
		return (-ENOMEM);
	}

	conn->session = NULL;
	conn->role = UTA_NONE;
	conn->refcount = 1;
	init_MUTEX(&conn->read_sem);
	init_MUTEX(&conn->write_sem);
	init_rwsem(&conn->syscall_rwsem);
	init_waitqueue_head(&conn->release_wait);
	WLD_FILL_ENTRY(0x10002, conn, NULL);

	/* check for magic device names */
	sessid = gross_nodename_hack(file);
	if (sessid < 0) {
		WLD_FILL_ENTRY(0x10003, conn, NULL);
		DPRINTF(2, "exiting %s(): no device ID\n", __FUNCTION__);

		/*
		 * This check is not required now as permission
		 * on /dev/utio will take care of this
		 * if (!capable(CAP_SYS_ADMIN)) {
		 * 	DPRINTF(1, "not capable\n");
		 *	kfree(conn);
		 *	file->private_data = NULL;
		 *	return -ENODEV;
		 * }
		 */

		/*
		 * Assume that the clone device (/dev/utio) was opened.
		 * file->private_data points to the new UTA_NONE conn we
		 * allocated and initialized above.  This place holder
		 * conn is used by daemon UTIO_GETMAJOR and UTIO_GETDEVID
		 * ioctls to set up a session for a device.
		 * A session isn't opened on the clone device and isn't
		 * needed by those ioctls.
		 */
		spin_lock_irqsave(&connlist_lock, flgs);
		list_add_tail(&conn->list, &connection_list);
		file->private_data = conn;
		spin_unlock_irqrestore(&connlist_lock, flgs);
		return (0);
	}
	DPRINTF(3, "%s: session ID seems to be %d\n", __FUNCTION__, sessid);

	/* join the session */
	WLD_FILL_ENTRY(0x10004, conn, sessid);
	r = slave_open(conn, sessid,
	    file->f_mode & FMODE_READ, file->f_mode & FMODE_WRITE);
	if (r < 0) {
		WLD_FILL_ENTRY(0x80010005, sessid, r);
		kfree(conn);
		DPRINTF(1, "exiting %s(), error return from slave_open()\n", __FUNCTION__);
		return (r);
	}

	spin_lock_irqsave(&connlist_lock, flgs);
	list_add_tail(&conn->list, &connection_list);
	file->private_data = conn;
	spin_unlock_irqrestore(&connlist_lock, flgs);
	WLD_FILL_ENTRY(0x10006, conn, sessid);
	DPRINTF(2, "exiting %s(), opened file %s\n",
		__FUNCTION__, file->f_dentry->d_name.name);
	return (0);
}


/*
 * Acquire conn_lock spinlock.
 * return -ENODEV if file->private_data == NULL.
 *   prevents close() race.  Only one will get through.
 * set file->private_data = NULL.
 *   prevents new system calls for this file from entering the driver.
 *   GET_CONNECTION() will see that file->private_data == NULL and return.
 * release conn_lock spinlock
 * if we are closing a "conn->role == UTA_DAEMON" connection (clone device)
 * that has a session associated with it (conn->session != NULL):
 *   clear the "sesson has daemon" flag.
 *   free any messages that remain on the session's messages_to_master and
 *   messages_to_slave lists.  Remaining messages must be freed because we
 *   are about to destroy/free the daemon side of the connection.
 *   if the session has a user side (the close for the slave device preceeds
 *   the close of the clone device):
 *     wake up waiters on sess->ios[].wait in user_read() and user_write()
 *   call unref_session().  If ours was the last reference, unref_session()
 *   will call end_session().
 * else if we are closing a "conn->role == UTA_USER" connection (slave device)
 * that has a session associated with it (conn->session != NULL):
 *   send a UTIO_MSG_CLOSE message to the daemon/master.
 *   don't wait for a reply.
 *   set "session doesn't have user" flag.     
 *   clear session user lock so that the slave device can be opened again
 *   and inherit the session.
 *   call unref_session().  If ours was the last reference, unref_session()
 *   will call end_session().  This would happen if the daemon connection
 *   (clone device) has been closed.
 * wait for the connection reference count to become 0.  Time out if it takes
 * too long.  The reference count would be non-zero at this point if:
 * a system call was entered on this connection prior to us clearing
 * file->private_data above.  Normally, the system call will complete
 * and the reference count will go to zero.  RELEASE_CONNECTION() will
 * wake us up when the reference count goes to zero.
 * Special cases where the reference count will not go to zero:
 *   - we may have been called by kernel syscall cleanup code when a process
 *     is killed while sleeping in a utio system call for the file we are
 *     trying to release.  That process did not have the chance to do a
 *     RELEASE_CONNECTION(conn) so the reference count will not be decremented.
 *   - a bug has caused conn->refcount to get out of sync.
 * In both cases, we will time out waiting on the reference count and continue
 * with the release.
 * Remove the connection from the global list of connections.
 * kfree() the connection data structure.
 */ 
static int
utio_release(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	int ret;
	unsigned long flgs;

	WLD_FILL_ENTRY(0x20000, NULL, NULL);
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/*
	 * The following sequence is based on GET_CONNECTION().
	 * The difference is we set file->private_data = NULL before
	 * releasing conn_lock.
	 */
	spin_lock_irqsave(&conn_lock, flgs);
	conn = file->private_data;
	if (!conn) {
		WLD_FILL_ENTRY(0x80020001, NULL, NULL);
		spin_unlock_irqrestore(&conn_lock, flgs);
		DPRINTF(1, "%s(): error: no connection\n",
			__FUNCTION__);
		DPRINTF(1, "%s(): exiting\n", __FUNCTION__);
		return (-ENODEV);
	}
	/* prevent any new syscalls (including close()) for this file */
	file->private_data = NULL;
	spin_unlock_irqrestore(&conn_lock, flgs);

	sess = conn->session;

	if (sess && conn->role == UTA_DAEMON) {
		WLD_FILL_ENTRY(0x20002, sess->id, NULL);
		DPRINTF(2, "%s: disconnecting DAEMON from session %d\n",
						__FUNCTION__, sess->id);
		/* notify the user of the bad news */
		if (!set_has_daemon(sess, 0)) {
			/* "has daemon" condition was not previously set */
			DPRINTF(1, "%s(): error: expected session to have a daemon\n",
				__FUNCTION__);
			BUG();
		}

		/* any queued messages are moot */
		free_all_msgs(sess);

		/*
		 * Careful:
		 * If there is a user, he's done with alloc_ios().  Make sure
		 * he's not asleep or about to sleep on a buffer. If there
		 * is not a user, any connecting user will never get to
		 * down() any bufs, since has_daemon() is now 0.
		 */
		if (has_user(sess)) {
			wakeup_ios(sess);
			/*
			 * you can't get any more ACKs/NACKs,
			 * wake sleepers or racers
			 */
			WLD_FILL_ENTRY(0x20003, sess->id, &sess->msg_sem_slave);
			up(&sess->msg_sem_slave);
		}

		/* drop our reference */
		unref_session(sess);
		// AUDIT: make sure the user is woken up from any block!
	} else if (sess && conn->role == UTA_USER) {
		uid_t uid = CURRENT_UID();

		WLD_FILL_ENTRY(0x20004, sess->id, NULL);
		DPRINTF(2, "%s: disconnecting USER from session %d\n",
					__FUNCTION__, sess->id);
		/* sync, no ACK required */
		/* device_sync(sess, 0); */
		/* tell the daemon we're gone */
		pmsg.magic = UTIO_MAGIC;
		pmsg.version = UTIO_VERSION;
		pmsg.msgtype = UTIO_MSG_CLOSE;
		pmsg.cmd = sess->id;
		memset(&pmsg.args, 0, sizeof (pmsg.args));
		pmsg.uid = uid;
		pmsg.datasize = 0;
#ifdef	PMSG_SEQ_DEBUGGING
		pmsg.seq = next_seq++;
#endif	/* PMSG_SEQ_DEBUGGING */

		ret = send_msg_to_master(sess, &pmsg);
		if (!ret) {
			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					uid, UTIO_MSG_CLOSE);
			if (!ret && (pmsg.msgtype == UTIO_MSG_CLOSE) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				WLD_FILL_ENTRY(0x20005, sess->id, conn);
				DPRINTF(3, "%s: ACK received for UTIO_MSG_CLOSE\n",
					__FUNCTION__);
			} else {
				// XXX should never happen
				// XXX should we re-issue close again??
				WLD_FILL_ENTRY(0x80020006, sess->id, conn);
				DPRINTF(1, "%s: No ACK for UTIO_MSG_CLOSE\n",
					__FUNCTION__);
			}
		} else {
			/*
			 * Error return from send_msg_to_master().  We could
			 * get here if the daemon has gone away (-ENXIO) or
			 * kmalloc() couldn't allocate memory for a message
			 * (-ENOMEM).  In any case, this user session is going
			 * away.
			 */
			WLD_FILL_ENTRY(0x80020007, sess->id, ret);
			DPRINTF(1, "%s: ret = %d from send_msg_to_master()\n",
				__FUNCTION__, ret);
		}

		/* tell the daemon, first */
		if (!set_has_user(sess, 0)) {
			/* "has user" condition was not previously set */
			DPRINTF(1, "%s(): error: expected has_user to be set\n",
				__FUNCTION__);
			BUG();
		}

		/*
		 * wake up any waiters on sess->ios[].wait, especially the
		 * daemon doing a poll_wait() in daemon_poll() on
		 * sess->ios[IO_READ].wait.
		 * We may call init_waitqueue_head(&sess->ios[IO_READ].wait)
		 * in init_one_io() if we re-use this session, which
		 * would corrupt the poll_wait() list if the daemon is sleeping
		 * in a poll_wait().
		 */
		wakeup_ios(sess);

		/* wait for sess->ios[IO_READ].wait list to be empty */
		set_current_state(TASK_INTERRUPTIBLE);
		while (!list_empty(&sess->ios[IO_READ].wait.task_list)) {
			/* schedule a wakeup in 1/3 of a second */
			(void)schedule_timeout(HZ/3);
			set_current_state(TASK_INTERRUPTIBLE);
		}
		__set_current_state(TASK_RUNNING);

		/* another user can join, now */
		if (!set_user_lock(sess, 0)) {
			/* user_lock should be set on active sessions */
			DPRINTF(1, "%s(): error: expected user lock to be set\n",
				__FUNCTION__);
			BUG();
		}

		/* drop our reference */
		unref_session(sess);
		// AUDIT: anything else needed here?
	}
	else {
		/* no session */
		WLD_FILL_ENTRY(0x20008, conn, NULL);
	}

	/* done with the connection */
	spin_lock_irqsave(&conn_lock, flgs);
	conn->refcount--;
	spin_unlock_irqrestore(&conn_lock, flgs);
	WLD_FILL_ENTRY(0x20009, conn, conn->refcount);
	DPRINTF(2, "%s(): conn->refcount == %d\n",
		__FUNCTION__, conn->refcount);

	/*
	 * Special cases:
	 *   1) we may have been called by kernel syscall cleanup code
	 *      when a process is killed while sleeping in a utio system
	 *      call for the file we are trying to release.  That process
	 *      did not have the chance to do a RELEASE_CONNECTION(conn).
	 *   2) a bug has caused conn->refcount to get out of sync.
	 * In either case, we will time out waiting and continue.
	 */

	/* wait for active syscalls to complete */
	wait_event_timeout(conn->release_wait, conn->refcount == 0, 5 * HZ);

	if (conn->refcount != 0) {
		WLD_FILL_ENTRY(0x8002000a, conn, conn->refcount);
		DPRINTF(1, "%s(): error, conn->refcount == %d. Expected 0.\n",
			__FUNCTION__, conn->refcount);
	}

	spin_lock_irqsave(&connlist_lock, flgs);
	list_del(&conn->list);
	spin_unlock_irqrestore(&connlist_lock, flgs);
	kfree(conn);

	WLD_FILL_ENTRY(0x2000b, conn, NULL);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	return (0);
}


/*
 * utio - Sun Ray pseudo I/O driver
 */

struct file_operations utio_fops = {
	owner:		THIS_MODULE,
	llseek:		no_llseek,
	read:		utio_read,
	write:		utio_write,
	poll:		utio_poll,
	ioctl:		utio_ioctl,
	compat_ioctl:	utio_compat_ioctl,
	open:		utio_open,
	release:	utio_release,
};


/*
 * Currently there is no benefit to make use of the __init
 * and __exit macros for utio_init and utio_exit functions
 * as these don't have any effect on the loadable modules.
 * But may be in future kernel versions, these macros might
 * have some effect.
 */

/*
 * Called at module load.
 *
 * Initialize the session ID data structure.
 * Register the driver and its file ops with the kernel.
 *   This also gets a dynamically assigned major number.
 * Create /proc/driver/utio and register utio_proc_read() as the handler
 * for reads.
 */
static int __init
utio_init(void)
{
	int result;
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* set up the session-tracking stuff */
	sessbits_count = MAX_SESSIONS/BITS_PER_LONG;
	sessbits_count += (MAX_SESSIONS % BITS_PER_LONG) ? 1 : 0;

	session_bits = kmalloc(sessbits_count * sizeof (*session_bits),
				GFP_KERNEL);

	if (!session_bits) {
		printk(KERN_ERR "utio: can't allocate session_bits\n");
		return (-ENOMEM);
	}

	memset(session_bits, 0, sessbits_count * sizeof (*session_bits));

	/* set any bits beyond max_sessions, so we can ignore them later */
	if (MAX_SESSIONS % BITS_PER_LONG) {
		int r = MAX_SESSIONS % BITS_PER_LONG;
		int i;
		unsigned long *last_word = &session_bits[sessbits_count - 1];

		for (i = r; i < BITS_PER_LONG; i++) {
			__set_bit(i, last_word);
		}
	}

	/* register the device */
	result = register_chrdev(utio_major, "utio", &utio_fops);
	if (result < 0) {
		printk(KERN_ERR "utio: can't register utio device\n");
		kfree(session_bits);
		session_bits = (unsigned long *)NULL;
		return (result);
	}

	/* dynamic major number */
	if (utio_major == 0) utio_major = result;

	DPRINTF(3, "%s: registered utio dev %u\n", __FUNCTION__, utio_major);
	DPRINTF(3, "%s: Create /dev/utio  using mknod /dev/utio c %u 0\n",
		__FUNCTION__, utio_major);

	/* register a file in /proc/driver */
	if (!create_proc_read_entry("driver/utio", 0, 0,
				    utio_proc_read, NULL)) {
	    printk(KERN_ERR "utio: can't create /proc/driver/utio\n");
	    unregister_chrdev(utio_major, "utio");
	    kfree(session_bits);
	    session_bits = (unsigned long *)NULL;
	    return (-ENOMEM);
	}

	printk(KERN_INFO "utio(%s): initialization complete. Utio major %u\n",
		UTIO_DRIVER_VERSION,
		utio_major);
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
	/* init successful */
	return (0);
}

/*
 * Called when the driver is unloaded
 *
 * Panic if there are active connections or sessions.
 * De-activate /proc/driver/utio.
 * Unregister the driver.
 * Free the session bits data structure.
 */
static void __exit
utio_exit(void)
{
	unsigned long flgs;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* panic if there are active connections or sessions */
	spin_lock_irqsave(&connlist_lock, flgs);
	if (!list_empty(&connection_list)) {
		spin_unlock_irqrestore(&connlist_lock, flgs);
		printk(KERN_ERR "utio: error: attempting to exit with active connections\n");
		BUG();
	}
	spin_unlock_irqrestore(&connlist_lock, flgs);
	spin_lock_irqsave(&sesslist_lock, flgs);
	if (!list_empty(&session_list)) {
		spin_unlock_irqrestore(&sesslist_lock, flgs);
		printk(KERN_ERR "utio: error: attempting to exit with active sessions\n");
		BUG();
	}
	spin_unlock_irqrestore(&sesslist_lock, flgs);

	/*
	 * XXX unregister the device
	 * XXX free all the allocated memory
	 */
	remove_proc_entry("driver/utio", NULL);
	unregister_chrdev(utio_major, "utio");
	if (session_bits)
		kfree(session_bits);

	printk(KERN_INFO "utio: exit\n");
	DPRINTF(2, "exiting %s()\n", __FUNCTION__);
}


module_init(utio_init);
module_exit(utio_exit);

MODULE_AUTHOR("Sun Ray Engineering <srss-feedback-ext@Sun.COM>");
MODULE_DESCRIPTION("utio Sun Ray pseudo I/O driver v1.18");
MODULE_LICENSE("GPL");

/*
 * The device name that's supported by this module. In
 * future, this parameter may be used by the kernel to
 * load this module automatically, but is currently used
 * only for documention purposes.
 */
MODULE_SUPPORTED_DEVICE("utio");

module_param(debug, int, 0);
