/*
 * Copyright 2008 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)utio.c	1.10	08/01/14 SMI"

#ifdef _SCCSID
static char __attribute__ ((unused)) *_SCCSid = "@(#)utio.c	1.10	08/01/14 SMI";
#endif	/* defined _SCCSID */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/fs.h>
#include <linux/smp_lock.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <linux/ioctl.h>
#include <linux/termios.h>
#include <linux/ppdev.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>

#include "utio.h"


/* get major number at load time */
#define	UTIO_MAJOR	0

/* max sessions/connections */
#define	MAX_SESSIONS	2048

/* IO Buffer size */
#define	UTA_DEF_BLKSIZE	8192
/* No. of IO buffers */
#define	UTA_DEF_NBLKS	2

/* TIMEOUT for UTIO_MSG_OPEN in seconds */
#define	UTIO_OPEN_TIMEOUT	(60*HZ)

/* debugging stuff */
/* Debugging level (0=off, 1=errors, 2=basic, 3=full, 4=verbose) */
static int debug;
#define	DPRINTF(lvl, fmt, args...)	do { \
	if (debug && debug >= lvl) \
		printk(KERN_DEBUG "utio debug: " fmt, ##args); \
} while (0)


struct uta_buffer {
	int offset;		/* offset into this buf */
	int bytes;		/* size of data in this buf */
	atomic_t owner;		/* who owns this buf - daemon or user? */
	unsigned char data[UTA_DEF_BLKSIZE];
};

struct uta_io {
	int user;		/* which buffer the user is on */
	int daemon;		/* which buffer the daemon is on */
	wait_queue_head_t wait; /* for IO blocking */
	struct uta_buffer *bufs[UTA_DEF_NBLKS];
};

struct uta_msg {
	utio_pmsg_t pmsg;
	struct list_head list;
};


struct uta_session {
	int id;			/* the session ID */
	uid_t owner;		/* the uid of the owning user */
	unsigned char mode;	/* user's open() mode */
	unsigned long flags;	/* who is connected, etc */
	int refcount;		/* when this is 0, the session can be freed */
	struct list_head list;	/* global session list_head */
	struct uta_io ios[2];	/* I/O streams */
	struct list_head messages_to_master;	/* protocol messages */
	spinlock_t msg_lock_master;		/* protect the messages */
	struct list_head messages_to_slave;	/* protocol messages */
	spinlock_t msg_lock_slave;		/* protect the messages */
	struct semaphore msg_sem_master;	/* producer/consumer sync */
	struct semaphore msg_sem_slave;		/* producer/consumer sync */
	wait_queue_head_t daemon_wait;		/* for daemon_poll */
};

/* I/O indices */
#define	IO_READ		0
#define	IO_WRITE	1

/* mode bits */
#define	MODE_READ	(1 << IO_READ)
#define	MODE_WRITE	(1 << IO_WRITE)

struct uta_connect {
	struct uta_session *session;
	int role;				/* user or daemon? */
	struct semaphore read_sem;		/* one read() user at a time */
	struct semaphore write_sem;		/* one write() user at a time */
	struct rw_semaphore syscall_rwsem;	/* block unsafe re-entrances */
};

/* role values */
#define	UTA_NONE		-1
#define	UTA_DAEMON		0
#define	UTA_USER		1

/* this is called with the session unlisted or with sesslist_lock held */
#define	ref_session(sess)	do { \
	sess->refcount++; \
	DPRINTF(3, "sess %d refcount++ = %d\n", sess->id, sess->refcount); \
} while (0)

#define	unref_session(sess)	do { \
	spin_lock(&sesslist_lock); \
	sess->refcount--; \
	DPRINTF(3, "sess %d refcount-- = %d\n", sess->id, sess->refcount); \
	if (sess->refcount == 0) \
		end_session(sess); \
	spin_unlock(&sesslist_lock); \
} while (0)

/* some helpers for session state management - lockless is good */
#define	BIT_USER		0 /* user is connected */
#define	BIT_DAEMON		1 /* daemon is connected */
#define	BIT_ULOCK		2 /* a user connection is imminent */
#define	BIT_DISC		3 /* Daemon is forced to disconnect */
#define	has_daemon(s)		test_bit(BIT_DAEMON, &(s)->flags)
#define	has_user(s)		test_bit(BIT_USER, &(s)->flags)
#define	is_disc(s)		test_bit(BIT_DISC, &(s)->flags)
#define	set_has_daemon(s, v)	set_flag((s), BIT_DAEMON, (v))
#define	set_has_user(s, v)	set_flag((s), BIT_USER, (v))
#define	set_user_lock(s, v)	set_flag((s), BIT_ULOCK, (v))
#define	set_is_disc(s, v)	set_flag((s), BIT_DISC, (v))

static inline int
set_flag(struct uta_session *sess, int bit, int val)
{
	if (val)
		return (test_and_set_bit(bit, &sess->flags));
	else
		return (test_and_clear_bit(bit, &sess->flags));
}

/* the global list of active sessions (struct uta_session) */
static LIST_HEAD(session_list);
static spinlock_t sesslist_lock = SPIN_LOCK_UNLOCKED;

/* the global session bitmask */
static int sessbits_count; /* count of ulongs in the array */
static unsigned long *session_bits;
static spinlock_t sessbits_lock = SPIN_LOCK_UNLOCKED;

/* dynamic utio major number */
static unsigned int utio_major = UTIO_MAJOR;

/* functions */
static int new_sessionid(void);
static int do_newsession(struct uta_connect *conn, uid_t owner);
static struct uta_session *find_session(int sessid);
static int alloc_ios(struct uta_session *sess);
static void wakeup_ios(struct uta_session *sess);
static int slave_open(struct uta_connect *conn, int sessid, int readfrom,
		int writeto);

static void free_all_msgs(struct uta_session *sess);
static void free_ios(struct uta_session *sess);
static void release_sessionid(int sessid);
static void disc_session(struct uta_connect *conn);
static void end_session(struct uta_session *sess);

static int xfer_buf_to_daemon(struct uta_session *sess, int io_idx);
static int xfer_buf_to_user(struct uta_session *sess, int io_idx);

static int send_msg_to_master(struct uta_session *sess, utio_pmsg_t *pmsg);
static int send_msg_to_slave(struct uta_session *sess, utio_pmsg_t *pmsg, int flagkptr);
static int do_getmsg_from_slave(struct uta_session *sess, utio_pmsg_t *umsg,
				int nonblock, uid_t owner, int flagkptr);
static int do_getmsg_from_master(struct uta_session *sess, utio_pmsg_t *umsg,
				uid_t owner, int msgtype);


/* fops helpers */
static int daemon_read(struct file *file, char *ubuf, size_t size);
static int daemon_write(struct file *file, const char *ubuf, size_t size);
static int daemon_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);
static unsigned int daemon_poll(struct file *file,
    struct poll_table_struct *wait);

static int user_read(struct file *file, char *ubuf, size_t size);
static int user_write(struct file *file, const char *ubuf, size_t size);
static unsigned int user_poll(struct file *file,
    struct poll_table_struct *wait);
static int user_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);
static int seriald_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);
static int paralleld_ioctl(struct inode *inode, struct file *file,
    unsigned int cmd, unsigned long arg);


static int
gross_nodename_hack(struct file *file)
{
	char *devid;
	char *p0;
	char *p1;
	int ret;

	/*
	 * The format of the nodename is
	 * <device name>%=<unique device id>
	 */
	p0 = (char *) file->f_dentry->d_name.name;
	p1 = p0 + strlen(p0) - 1;

	/* must end in a number... */
	if (!isdigit(*p1--))
		return (-1);

	while (isdigit(*p1))
		p1--;
	devid = p1+1;

	/*  preceded by '%=' */
	if ((*p1-- != '=') || (*p1-- != '%'))
		return (-1);

	ret = (int) simple_strtol(devid, NULL, 10);
	return (ret);
}

/*
 * will print the following to the file /proc/driver/utio:
 * major: <major number>
 */
static int utio_proc_read(char *buf, char **start, off_t offset,
			int len, int *eof, void *data)
{
	int dlen;

	dlen = sprintf(buf, "major: %d\n", utio_major);

	/* trying to read a bad offset? */
	if (offset >= dlen) {
		*eof = 1;
		return (0);
	}

	/* did we get everything? */
	if (len >= (dlen-offset)) {
		*eof = 1;
	}

	*start = buf + offset;
	dlen -= offset;

	return ((len > dlen) ? dlen : len);
}

static int
do_getmsg_from_slave(struct uta_session *sess, utio_pmsg_t *umsg,
		int nonblock, uid_t owner, int flagkptr)
{
	struct uta_msg *msg;
	struct list_head *item;
	int ret = 0;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* get the msg_sem_master to see if there are any messages */
	if (nonblock) {
		if (down_trylock(&sess->msg_sem_master))
			return (-EAGAIN);
	} else {
		if (down_interruptible(&sess->msg_sem_master))
			return (-ERESTARTSYS);
	}

	if (is_disc(sess)) {
		// we are asked to disconnect
		DPRINTF(3, "we are asked to disconnect\n");
		return (-1);
	}

	spin_lock(&sess->msg_lock_master);
	item = sess->messages_to_master.next;
	list_del(item);
	spin_unlock(&sess->msg_lock_master);

	msg = list_entry(item, struct uta_msg, list);
	if (flagkptr)
	    memmove(umsg, msg, sizeof (utio_pmsg_t));
	else
	    ret = copy_to_user(umsg, msg, sizeof (utio_pmsg_t));
	kfree(msg);

	return (ret);
}

/* timer function */
static void
timeout_waiting_for_master(unsigned long msg_sema_slave) {
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	up((struct semaphore *) msg_sema_slave);
}

static int
do_getmsg_from_master(struct uta_session *sess, utio_pmsg_t *umsg,
		uid_t owner, int msgtype)
{
	struct uta_msg *msg;
	struct list_head *item;
	struct timer_list timer;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!has_daemon(sess)) {
		return (-ENXIO);
	}

	/* get the msg_sem_slave to see if there are any messages */
	if (msgtype == UTIO_MSG_OPEN) {
		init_timer(&timer);
		timer.expires = jiffies + UTIO_OPEN_TIMEOUT;
		timer.function = timeout_waiting_for_master;
		timer.data = (unsigned long) &sess->msg_sem_slave;
		add_timer(&timer);

		if ((ret = down_interruptible(&sess->msg_sem_slave))) {
			return (-ERESTARTSYS);
#ifdef CONFIG_SMP
		} else if (!del_timer_sync(&timer)) {
#else  /* CONFIG_SMP */
		} else if (!del_timer(&timer)) {
#endif /* CONFIG_SMP */
			/* Timed out.  */
			/* notify the user of the bad news */
			set_is_disc(sess, 1);
			wake_up_interruptible(&sess->daemon_wait);
			up(&sess->msg_sem_master);
			return (-ETIMEDOUT);
		}
	} else if (down_interruptible(&sess->msg_sem_slave)) {
		return (-ERESTARTSYS);
	}

	/* We might be woken up if daemon dies */
	if (!has_daemon(sess)) {
		return (-ENXIO);
	}

	spin_lock(&sess->msg_lock_slave);
	item = sess->messages_to_slave.next;
	list_del(item);
	spin_unlock(&sess->msg_lock_slave);

	msg = list_entry(item, struct uta_msg, list);
	memmove(umsg, msg, sizeof (utio_pmsg_t));
	kfree(msg);

	return (0);
}


/* this returns the session already ref'ed */
static struct uta_session *
find_session(int sessid)
{
	struct list_head *pos;

	spin_lock(&sesslist_lock);
	list_for_each(pos, &session_list) {
		struct uta_session *sess;

		sess = list_entry(pos, struct uta_session, list);
		if (sess->id == sessid && has_daemon(sess)) {
			DPRINTF(3, "found session %d\n", sessid);
			ref_session(sess);
			spin_unlock(&sesslist_lock);
			return (sess);
		}
	}
	spin_unlock(&sesslist_lock);

	return (NULL);
}

/* find and allocate a new session ID */
static int
new_sessionid(void)
{
	int idx;
	int base = 0;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	spin_lock(&sessbits_lock);
	for (idx = 0; idx < sessbits_count; idx++) {
		if (session_bits[idx] != ~0UL) {
			int bit;
			/* ffz - find first zero in word. */
			bit = ffz(session_bits[idx]);
			DPRINTF(3, "setting sessbit %d:%d\n", idx, bit);
			__set_bit(bit, &session_bits[idx]);
			spin_unlock(&sessbits_lock);
			return (base + bit);
		}

		base += BITS_PER_LONG;
	}
	spin_unlock(&sessbits_lock);

	return (-1);
}


/* deallocate a sessionid */
static void
release_sessionid(int sessid)
{
	int idx;
	int bit;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sessid);

	BUG_ON(sessid >= MAX_SESSIONS);

	idx = sessid / BITS_PER_LONG;
	bit = sessid % BITS_PER_LONG;

	DPRINTF(3, "clearing sessbit %d:%d\n", idx, bit);
	spin_lock(&sessbits_lock);
	clear_bit(bit, &session_bits[idx]);
	spin_unlock(&sessbits_lock);
}

/*
 * All the daemon-side ioctl helpers are protected from racing with each
 * other and read/write by daemon_ioctl, if needed.  User-side ioctl helpers
 * are protected from re-entrance by user_ioctl().
 */

static int
do_newsession(struct uta_connect *conn, uid_t owner)
{
	struct uta_session *sess;
	int sessid;
	int i;

	DPRINTF(2, "entering %s(owner=%d)\n", __FUNCTION__, owner);

	if (conn->session)
		return (-EBUSY);

	/* allocate a session ID */
	sessid = new_sessionid();
	if (sessid < 0) {
		DPRINTF(1, "can't get a new session ID\n");
		return (-EBUSY);
	}
	DPRINTF(3, "new sessionid = %d\n", sessid);

	/* allocate session struct */
	sess = kmalloc(sizeof (*sess), GFP_KERNEL);
	if (!sess) {
		DPRINTF(1, "can't allocate session\n");
		release_sessionid(sessid);
		return (-ENOMEM);
	}

	/* init the session */
	sess->id = sessid;
	sess->owner = owner;
	sess->mode = 0;
	sess->refcount = 0;
	sess->flags = 0;
	set_has_daemon(sess, 1);
	for (i = 0; i < 2; i++) {
		struct uta_io *io;
		int j;

		io = &sess->ios[i];
		io->user = io->daemon = 0;
		init_waitqueue_head(&io->wait);
		for (j = 0; j < UTA_DEF_NBLKS; j++)
			io->bufs[j] = NULL;
	}
	init_waitqueue_head(&sess->daemon_wait);
	DPRINTF(3, "init_wait_queue -daemon_wait\n");
	INIT_LIST_HEAD(&sess->messages_to_master);
	spin_lock_init(&sess->msg_lock_master);
	init_MUTEX_LOCKED(&sess->msg_sem_master);

	DPRINTF(3, "session %d created for user %d\n", sess->id, owner);

	/* increment the refcount */
	ref_session(sess);

	/* lastly, enlist the session */
	spin_lock(&sesslist_lock);
	list_add_tail(&sess->list, &session_list);
	spin_unlock(&sesslist_lock);

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_DAEMON;

	return (sessid);
}


static void
disc_session(struct uta_connect *conn)
{
	struct uta_session *sess = conn->session;

	DPRINTF(2, "entering %s\n", __FUNCTION__);

	/* notify the user of the bad news */
	set_has_daemon(sess, 0);
	if (has_user(sess))
		wakeup_ios(sess);

	/* any queued messages are moot */
	free_all_msgs(sess);

	/* you can't get any more ACKs/NACKs, wake sleepers or racers */
	up(&sess->msg_sem_slave);

	/* drop our reference */
	if (sess)
		unref_session(sess);

	/* done with the connection, now */
	kfree(conn);
}



/*
 * do the business of deallocating and delisting a session
 */
static void
end_session(struct uta_session *sess)
{
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	list_del(&sess->list);
	release_sessionid(sess->id);
	free_ios(sess);
	free_all_msgs(sess);

	kfree(sess);
}


static void
init_one_buf(struct uta_buffer *buf, int mode)
{
	if (mode == MODE_READ) {
		/* read buffers start out owned by the daemon */
		atomic_set(&buf->owner, UTA_DAEMON);
	} else if (mode == MODE_WRITE) {
		/* write buffers start out owned by the user */
		atomic_set(&buf->owner, UTA_USER);
	} else {
		return;
	}
	buf->offset = 0;
	buf->bytes = 0;
	memset(buf->data, 0, sizeof (buf->data));
}


static void
init_one_io(struct uta_io *io)
{
	io->user = io->daemon = 0;
	init_waitqueue_head(&io->wait);
}


static void
free_one_io(struct uta_io *io)
{
	int i;
	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	for (i = 0; i < UTA_DEF_NBLKS; i++) {
		if (io->bufs[i]) {
			kfree(io->bufs[i]);
			io->bufs[i] = NULL;
		}
	}
}

static void
free_ios(struct uta_session *sess)
{
	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	free_one_io(&sess->ios[IO_READ]);
	free_one_io(&sess->ios[IO_WRITE]);
}

static void
wakeup_ios(struct uta_session *sess)
{
	int i;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	for (i = 0; i < 2; i++)
		wake_up_all(&sess->ios[i].wait);
}


/* initialize buffers for a session */
static int
alloc_ios(struct uta_session *sess)
{
	int r = 0;
	int p = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	if (sess->mode & MODE_READ) {
		struct uta_io *io = &sess->ios[IO_READ];

		for (r = 0; r < UTA_DEF_NBLKS; r++) {
			struct uta_buffer *buf;

			if (io->bufs[r] == NULL) {
				buf = kmalloc(sizeof (*buf), GFP_KERNEL);
				if (!buf)
					goto readbuf_err;
				io->bufs[r] = buf;
			}
			init_one_buf(io->bufs[r], MODE_READ);
		}
		init_one_io(io);
	} else {
		free_one_io(&sess->ios[IO_READ]);
	}

	if (sess->mode & MODE_WRITE) {
		struct uta_io *io = &sess->ios[IO_WRITE];

		for (p = 0; p < UTA_DEF_NBLKS; p++) {
			struct uta_buffer *buf;

			if (io->bufs[p] == NULL) {
				buf = kmalloc(sizeof (*buf), GFP_KERNEL);
				if (!buf)
					goto writebuf_err;
				io->bufs[p] = buf;
			}
			init_one_buf(io->bufs[p], MODE_WRITE);
		}
		init_one_io(io);
	} else {
		free_one_io(&sess->ios[IO_WRITE]);
	}

	return (0);

writebuf_err:
	while (--p >= 0) {
		kfree(sess->ios[IO_WRITE].bufs[p]);
	}

readbuf_err:
	while (--r >= 0) {
		kfree(sess->ios[IO_READ].bufs[r]);
	}

	/* if we got here, something failed to alloc */
	return (-ENOMEM);
}



static int
send_msg_to_master(struct uta_session *sess, utio_pmsg_t *pmsg)
{
	struct uta_msg *msg;

	DPRINTF(2, "entering %s(sessid=%d, msgid=%d, cmd=%d, datasize=%d)\n",
		__FUNCTION__, sess->id, pmsg->msgtype, pmsg->cmd,
		pmsg->datasize);

	if (!has_daemon(sess))
		return (-ENXIO);

	msg = kmalloc(sizeof (*msg), GFP_KERNEL);
	if (!msg) {
		DPRINTF(1, "can't allocate message\n");
		return (-ENOMEM);
	}

	memmove(&msg->pmsg, pmsg, sizeof (utio_pmsg_t));

	spin_lock(&sess->msg_lock_master);
	list_add_tail(&msg->list, &sess->messages_to_master);
	spin_unlock(&sess->msg_lock_master);

	wake_up_interruptible(&sess->daemon_wait);
	up(&sess->msg_sem_master);

	return (0);
}

static int
send_msg_to_slave(struct uta_session *sess, utio_pmsg_t *pmsg, int flagkptr)
{
	struct uta_msg *msg;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	if (!has_user(sess)) {
		return (-ENXIO);
	}

	msg = kmalloc(sizeof (*msg), GFP_KERNEL);
	if (!msg) {
		DPRINTF(1, "can't allocate message\n");
		return (-ENOMEM);
	}

	if (flagkptr)
	    memmove(&msg->pmsg, pmsg, sizeof (utio_pmsg_t));
	else {
	    if (copy_from_user(&msg->pmsg, pmsg, sizeof (utio_pmsg_t))) {
		    kfree(msg);
		    return (-EFAULT);
	    }
	}
	spin_lock(&sess->msg_lock_slave);
	list_add_tail(&msg->list, &sess->messages_to_slave);
	spin_unlock(&sess->msg_lock_slave);

	up(&sess->msg_sem_slave);
	return (0);
}

static void
free_all_msgs(struct uta_session *sess)
{
	struct list_head *tmp, *tmp2;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);
	if (!sess) {
	    return;
	}

	spin_lock(&sess->msg_lock_master);
	list_for_each_safe(tmp, tmp2, &sess->messages_to_master) {
		struct uta_msg *msg;
		msg = list_entry(tmp, struct uta_msg, list);
		list_del(tmp);
		kfree(msg);
	}
	spin_unlock(&sess->msg_lock_master);

	if (!has_user(sess)) {
		return;
	}

	spin_lock(&sess->msg_lock_slave);
	list_for_each_safe(tmp, tmp2, &sess->messages_to_slave) {
		struct uta_msg *msg;
		msg = list_entry(tmp, struct uta_msg, list);
		list_del(tmp);
		kfree(msg);
	}
	spin_unlock(&sess->msg_lock_slave);
}


/* return true if the buffer was transferred, otherwise 0 */
static int
xfer_buf_to_daemon(struct uta_session *sess, int io_idx)
{
	struct uta_io *io = &sess->ios[io_idx];
	struct uta_buffer *buf = io->bufs[io->user];

	DPRINTF(2, "entering %s(sessid=%d, io_idx=%d)\n", __FUNCTION__,
		sess->id, io_idx);

	/* don't send empty buffers */
	if (io_idx == IO_WRITE && buf->offset == 0)
		return (0);

	/* give it to the daemon */
	buf->offset = 0;
	atomic_set(&buf->owner, UTA_DAEMON);
	wake_up_interruptible(&io->wait);

	/* move on to the next buffer */
	io->user++;
	if (io->user == UTA_DEF_NBLKS)
		io->user = 0;

	return (1);
}


/* return true if the buffer was transferred, otherwise 0 */
static int
xfer_buf_to_user(struct uta_session *sess, int io_idx)
{
	struct uta_io *io = &sess->ios[io_idx];
	struct uta_buffer *buf = io->bufs[io->daemon];

	DPRINTF(2, "entering %s(sessid=%d, io_idx=%d)\n", __FUNCTION__,
		sess->id, io_idx);

	/* don't send empty buffers */
	if (io_idx == IO_READ && buf->offset == 0)
		return (0);

	/* give it to the user */
	buf->offset = 0;
	atomic_set(&buf->owner, UTA_USER);
	wake_up_interruptible(&io->wait);

	/* move on to the next buffer */
	io->daemon++;
	if (io->daemon == UTA_DEF_NBLKS)
		io->daemon = 0;

	return (1);
}


/*
 * These four functions [{user,daemon}_{read,write}()] are protected from
 * re-entrancy by the fact that only one user and one daemon are connected
 * to the session, and the conn->read_sem and conn->write_sem are
 * held by the caller of these functions.
 */

static int
user_read(struct file *file, char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	utio_pmsg_t pmsg;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	// LOCK: sess->io_lock
	/* get the next buffer to read from */
	io = &sess->ios[IO_READ];
	buf = io->bufs[io->user];

	/* tell daemon to start reading from the device */
	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_WRITE;
	pmsg.cmd = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));
	pmsg.uid = current->uid;
	pmsg.datasize = buf->bytes;
	send_msg_to_master(sess, &pmsg);

	add_wait_queue(&io->wait, &wait);

	/*
	 * If the buffer owner is not set to UTA_USER, the current
	 * will go to sleep. And this process should be rescheduled
	 * only after getting the interrupt. This process will be
	 * woken up explicitly by the wake_up_interrupt() call.
	 */
	set_current_state(TASK_INTERRUPTIBLE);
	while (atomic_read(&buf->owner) != UTA_USER) {
		/* user asked not to block */
		if (file->f_flags & O_NONBLOCK) {
			ret = -EAGAIN;
			break;
		}
		/* a signal can wake us up */
		if (signal_pending(current)) {
			ret = -ERESTARTSYS;
			break;
		}
		/* if the daemon dies, we'll be woken up */
		if (!has_daemon(sess)) {
			ret = -ENXIO;
			break;
		}
		schedule();
	}
	current->state = TASK_RUNNING;
	remove_wait_queue(&io->wait, &wait);

	if (ret < 0) {
		up_write(&conn->syscall_rwsem);
		return (ret);
	}

	/*
	 * if we get here, we have data to read
	 */

	/* clamp the size */
	if (size > buf->bytes)
		size = buf->bytes;

	/* read it */
	if (copy_to_user(ubuf, buf->data + buf->offset, size)) {
		up_write(&conn->syscall_rwsem);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes -= size;
	ret = size;

	/* if we emptied the buffer... */
	if (buf->bytes == 0)
		xfer_buf_to_daemon(sess, IO_READ);

	up_write(&conn->syscall_rwsem);
	return (ret);
}

static int
user_write(struct file *file, const char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	utio_pmsg_t pmsg;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	// LOCK: sess->io_lock
	/* get the next buffer to write into */
	io = &sess->ios[IO_WRITE];
	buf = io->bufs[io->user];

	add_wait_queue(&io->wait, &wait);
	set_current_state(TASK_INTERRUPTIBLE);
	while (atomic_read(&buf->owner) != UTA_USER) {
		/* user asked not to block */
		if (file->f_flags & O_NONBLOCK) {
			ret = -EAGAIN;
			break;
		}
		/* a signal can wake us up */
		if (signal_pending(current)) {
			ret = -ERESTARTSYS;
			break;
		}
		/* if the daemon dies, we'll be woken up */
		if (!has_daemon(sess)) {
			ret = -ENXIO;
			break;
		}
		schedule();
	}
	current->state = TASK_RUNNING;
	remove_wait_queue(&io->wait, &wait);

	if (ret < 0) {
		up_write(&conn->syscall_rwsem);
		return (ret);
	}

	/*
	 * if we get here, we have space to write
	 */

	/* clamp the size */
	if ((buf->bytes + size) > UTA_DEF_BLKSIZE)
		size = UTA_DEF_BLKSIZE - buf->bytes;

	/* write it */
	if (copy_from_user(buf->data + buf->offset, ubuf, size)) {
		up_write(&conn->syscall_rwsem);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes += size;
	ret = size;

	/* transfer ownership */
	xfer_buf_to_daemon(sess, IO_WRITE);
	/* tell the daemon to start reading the data from the kernel */
	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_READ;
	pmsg.cmd = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));
	pmsg.uid = current->uid;
	pmsg.datasize = buf->bytes;
	send_msg_to_master(sess, &pmsg);

	up_write(&conn->syscall_rwsem);
	return (ret);
}

static int
daemon_read(struct file *file, char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	int ret;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* If the disc bit is set, no operations permitted */
	if (is_disc(sess)) {
		disc_session(file->private_data);
		file->private_data = NULL;
		return (-EBADF);
	}

	/* block hazardous operations */
	down_read(&conn->syscall_rwsem);

	/*
	 * NOTE: don't check has_user() here - the user may have
	 * SYNCed and left
	 */

	// LOCK: sess->io_lock
	/* get the next buffer to write from */
	io = &sess->ios[IO_WRITE];
	buf = io->bufs[io->daemon];

	/* if the buffer is not ready for us to write, it's an error */
	if (!buf || (atomic_read(&buf->owner) != UTA_DAEMON)) {
		up_read(&conn->syscall_rwsem);
		return (0);
	}

	/* clamp the size to this block */
	if (size > buf->bytes)
		size = buf->bytes;

	/* we have data to read */
	if (copy_to_user(ubuf, buf->data + buf->offset, size)) {
		up_read(&conn->syscall_rwsem);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes -= size;
	ret = size;

	/* if we emptied the buffer... */
	if (buf->bytes == 0)
		xfer_buf_to_user(sess, IO_WRITE);

	up_read(&conn->syscall_rwsem);
	return (ret);
}

static int
daemon_write(struct file *file, const char *ubuf, size_t size)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;
	int ret;

	DPRINTF(2, "entering %s(sessid=%d)\n", __FUNCTION__, sess->id);

	/* If the disc bit is set, no operations permitted */
	if (is_disc(sess)) {
		disc_session(file->private_data);
		file->private_data = NULL;
		return (-EBADF);
	}

	/* block hazardous operations */
	down_read(&conn->syscall_rwsem);

	if (!has_user(sess)) {
		up_read(&conn->syscall_rwsem);
		return (0);
	}

	// LOCK: sess->io_lock
	/* get the next buffer to read into */
	io = &sess->ios[IO_READ];
	buf = io->bufs[io->daemon];

	/* if the buffer is not ready for us to fill, it's an error */
	if (!buf || (atomic_read(&buf->owner) != UTA_DAEMON)) {
		up_read(&conn->syscall_rwsem);
		return (0);
	}

	/* clamp the size to this block */
	if ((buf->bytes + size) > UTA_DEF_BLKSIZE)
		size = UTA_DEF_BLKSIZE - buf->bytes;

	/* write as much as we can */
	if (copy_from_user(buf->data + buf->offset, ubuf, size)) {
		up_read(&conn->syscall_rwsem);
		return (-EFAULT);
	}

	buf->offset += size;
	buf->bytes  += size;
	ret = size;

	/* transfer ownership */
	xfer_buf_to_user(sess, IO_READ);
	up_read(&conn->syscall_rwsem);
	return (size);
}


static unsigned int
user_poll(struct file *file, struct poll_table_struct *wait)
{
	unsigned int mask = 0;
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	struct uta_io *io;
	struct uta_buffer *buf;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	if (file->f_mode & FMODE_READ) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_READ];
		poll_wait(file, &io->wait, wait);
		buf = io->bufs[io->user];

		/* check if we own the buffer */
		if (atomic_read(&buf->owner) == UTA_USER)
			mask |= POLLIN | POLLRDNORM;
	}

	if (file->f_mode & FMODE_WRITE) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_WRITE];
		poll_wait(file, &io->wait, wait);
		buf = io->bufs[io->user];

		/* check if we own the buffer */
		if (atomic_read(&buf->owner) == UTA_USER)
			mask |= POLLOUT | POLLWRNORM;
	}

	if (!has_daemon(sess))
		mask = POLLHUP;

	up_write(&conn->syscall_rwsem);
	return (mask);
}

static unsigned int
daemon_poll(struct file *file, struct poll_table_struct *wait)
{
	unsigned int mask = 0;
	struct uta_connect *conn;
	struct uta_session *sess;
	struct uta_io *io;
	struct uta_buffer *buf;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	conn = file->private_data;
	sess = conn->session;

	/* one syscall at a time to the user */
	down_read(&conn->syscall_rwsem);

	if (!sess) {
		DPRINTF(3, "%s: No session...\n", __FUNCTION__);
		up_read(&conn->syscall_rwsem);
		return (POLLERR);
	}

	/* If the disc bit is set, no operations permitted */
	if (is_disc(sess)) {
		disc_session(file->private_data);
		file->private_data = NULL;
		return (POLLERR);
	}

	poll_wait(file, &sess->daemon_wait, wait);

	spin_lock(&sess->msg_lock_master);
	if (!list_empty(&sess->messages_to_master)) {
		mask |= POLLPRI;
	}
	spin_unlock(&sess->msg_lock_master);

	/*
	 * If there is no user, means IO buffers not
	 * initialized yet, so return the mask
	 * for the above poll_wait
	 */
	if (!has_user(sess)) {
		up_read(&conn->syscall_rwsem);
		return (mask);
	}

	DPRINTF(3, "%s: User has been connected to session\n", __FUNCTION__);
	if (file->f_mode & FMODE_WRITE) {
		// LOCK: sess->io_lock
		io = &sess->ios[IO_READ];
		buf = io->bufs[io->daemon];

		poll_wait(file, &io->wait, wait);

		if (buf && (atomic_read(&buf->owner) == UTA_DAEMON))
			mask |= POLLOUT | POLLWRNORM;
	}

	up_read(&conn->syscall_rwsem);
	return (mask);
}


static int
seriald_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	utio_pmsg_t pmsg;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!sess || !has_daemon(sess))
		return (-ENXIO);

	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_IOCTL;
	pmsg.uid = current->uid;
	pmsg.datasize = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));

	switch (cmd) {
		case TCXONC:		// IO
		case TCFLSH:		// IO
			DPRINTF(3, "ioctl type is 0x%x", cmd);
			pmsg.cmd = cmd;
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}
			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		case TCSETS:		// IOW
		case TCSETSW:		// IOW
		case TCSETSF:		// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.tios,
				(struct termios *)arg,
				sizeof (struct termios)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (struct termios);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		case TCSETA: 		// IOW
		case TCSETAW:		// IOW
		case TCSETAF:		// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.tio, (struct termio *)
					arg, sizeof (struct termio)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (struct termio);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		// case TIOCSPPS:
		case TCSBRK:  		// IO but behaving like IOW
		case TCSBRKP: 		// IOW
		case TIOCSSOFTCAR:	// IOW
		case TIOCMSET:		// IOW
		case TIOCMBIS:		// IOW
		case TIOCMBIC:		// IOW
			DPRINTF(3, "ioctl type is 0x%x", cmd);
			get_user(pmsg.args.intval, (int *)arg);
			DPRINTF(3, " arg=%ld\n", arg);

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (int);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		case TIOCSWINSZ:	// IOW
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			if (copy_from_user(&pmsg.args.twsz,
				(struct winsize *)arg,
				sizeof (struct winsize)) != 0) {
				return (-EFAULT);
			}

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (struct winsize);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (0);
			} else {
				return (-1);
			}
		// case TIOCGPPS:
		// case TIOCGPPSEV:
		case TIOCMGET:		// IOR
		case TIOCGSOFTCAR:	// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (put_user(pmsg.args.intval,
					(int *)arg));
			} else {
				return (-1);
			}
		case TIOCGWINSZ:	// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct winsize *)
					arg, &pmsg.args.twsz,
					sizeof (struct winsize)));
			} else {
				return (-1);
			}
		case TCGETS:		// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct termios *)
					arg, &pmsg.args.tios,
					sizeof (struct termios)));
			} else {
				return (-1);
			}
		case TCGETA:		// IOR
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct termio *)
					arg, &pmsg.args.tio,
					sizeof (struct termio)));
			} else {
				return (-1);
			}
	}

	DPRINTF(4, " - unknown ioctl type: 0x%x (%d)\n", cmd, cmd);
	return (-ENOTTY);
}

static int
paralleld_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	utio_pmsg_t pmsg;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!sess || !has_daemon(sess))
		return (-ENXIO);

	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_IOCTL;
	pmsg.uid = current->uid;
	pmsg.datasize = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));

	switch (cmd) {
		/* Get the current mode */
		case PPGETMODE:
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (put_user(pmsg.args.intval,
					(int *)arg));
			} else {
				return (-1);
			}

		/*
		 * Sets which IEEE 1284 protocol to use for
		 * the read and write calls
		 */
		case PPSETMODE:
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			get_user(pmsg.args.intval, (int *)arg);

			pmsg.cmd = cmd;
			pmsg.datasize = sizeof (int);
			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (put_user(pmsg.args.intval,
					(int *)arg));
			} else {
				return (-1);
			}

		/*
		 * Retrieves the time-out value for read and
		 * write calls
		 */
		case PPGETTIME:
			DPRINTF(3, "ioctl type is 0x%x\n", cmd);
			pmsg.cmd = cmd;

			/* Check the return value of send_msg_to_master */
			if ((ret = send_msg_to_master(sess, &pmsg))) {
				return (ret);
			}

			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_IOCTL);
			if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				return (copy_to_user((struct timeval *)
					arg, &pmsg.args.tval,
					sizeof (struct timeval)));
			} else {
				return (-1);
			}
	}

	DPRINTF(4, " - unknown ioctl type: 0x%x (%d)\n", cmd, cmd);
	return (-ENOTTY);
}

#if defined(CONFIG_COMPAT) || defined(CONFIG_SPARC64) || defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
static int
compat_paralleld_ioctl(unsigned int cmd, unsigned long arg, struct file *file)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	compat_utio_pmsg_t pmsg32;
	int ret;
	struct inode *inode = file->f_dentry->d_inode;

	/* only one ioctl needs conversion */
	if (cmd != COMPAT_PPGETTIME)
	    return (paralleld_ioctl(inode, file, cmd, arg));

	/*
	 * Retrieves the time-out value for read and
	 * write calls
	 */
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_IOCTL;
	pmsg.uid = current->uid;
	pmsg.datasize = 0;
	memset(&pmsg.args, 0, sizeof (pmsg.args));

	pmsg.cmd = PPGETTIME;
	DPRINTF(3, "ioctl type is 0x%x\n", pmsg.cmd);

	conn = file->private_data;
	sess = conn->session;
	if (!sess || !has_daemon(sess))
		return (-ENXIO);

	/* Check the return value of send_msg_to_master */
	if ((ret = send_msg_to_master(sess, &pmsg))) {
		return (ret);
	}

	memset(&pmsg, 0, sizeof (pmsg));
	ret = do_getmsg_from_master(sess, &pmsg,
			current->uid, UTIO_MSG_IOCTL);
	if (!ret && (pmsg.msgtype == UTIO_MSG_IOCTL) &&
	    (pmsg.response == UTIO_MSG_ACK))
	{
		COPY_UTIO_PMSG_TVAL(pmsg32.args, pmsg.args)
		return (copy_to_user((struct compat_timeval *)
			arg, &pmsg32.args.tval,
			sizeof (struct compat_timeval)));
	} else {
		return (-1);
	}
}
#endif	/* CONFIG_COMPAT */


/*
 * all the user ioctl() helper functions are open to re-entrancy - protect
 * them with conn->syscall_rwsem, if needed.
 */
static int
user_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	int ret;

	/* one syscall at a time to the user */
	down_write(&conn->syscall_rwsem);

	ret = -ENOTTY;

	if (MINOR(inode->i_rdev) == UTSERIALD_MINOR) {
		ret = seriald_ioctl(inode, file, cmd, arg);
	} else if (MINOR(inode->i_rdev) == UTPARALLELD_MINOR) {
		ret = paralleld_ioctl(inode, file, cmd, arg);
	} else {
		return (-ENXIO);
	}

	up_write(&conn->syscall_rwsem);
	return (ret);
}

/*
 * This function is re-entrant - protect each block internally
 * with read or write locking on conn->syscall_rwsem!
 */
static int
daemon_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* these don't require the session to exist */
	switch (cmd) {
	case UTIO_GETVERSION:
		DPRINTF(3, " - UTIO_GETVERSION\n");
		return (put_user(UTIO_VERSION, (unsigned long *)arg));

	case UTIO_GETMAJOR:
		DPRINTF(3, " - UTIO_GETMAJOR\n");
		return (MAJOR(inode->i_rdev));

	case UTIO_GETDEVID:
		DPRINTF(3, " - UTIO_GETDEVID\n");

		down_write(&conn->syscall_rwsem);
		ret = do_newsession(conn, current->uid);
		up_write(&conn->syscall_rwsem);
		return (ret);
	}

	if (!sess)
		return (-ENXIO);

	/* If the disc bit is set, no operations permitted */
	if (is_disc(sess)) {
		disc_session(file->private_data);
		file->private_data = NULL;
		return (-ENXIO);
	}

	/* these all require the session */
	switch (cmd) {
	case UTIO_GETMSG:
		DPRINTF(3, "- UTIO_GETMSG\n");
		down_read(&conn->syscall_rwsem);
		ret = do_getmsg_from_slave(sess, (utio_pmsg_t *)arg,
			    file->f_flags & O_NONBLOCK, current->uid, 0);
		if (ret < 0) {
			if (is_disc(sess)) {
				disc_session(file->private_data);
				file->private_data = NULL;
				return (-ENXIO);
			}
		}
		up_read(&conn->syscall_rwsem);
		return (ret);

	case UTIO_PUTMSG:
		DPRINTF(3, "- UTIO_PUTMSG\n");
		down_read(&conn->syscall_rwsem);
		send_msg_to_slave(sess, (utio_pmsg_t *)arg, 0);
		up_read(&conn->syscall_rwsem);
		return (0);
	default:
		break;
	}

	DPRINTF(4, " - unknown ioctl: 0x%08x (%d)\n", cmd, cmd);
	return (-ENOTTY);

}

#if defined(CONFIG_COMPAT) || defined(CONFIG_SPARC64) || defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
/* This function converts 32-bit ioctl arguments to/from 64-bit arguments. */
static int
compat_daemon_ioctl(unsigned int cmd, unsigned long arg, struct file *file)
{
	struct uta_connect *conn = file->private_data;
	struct uta_session *sess = conn->session;
	utio_pmsg_t		parm64;
	compat_utio_pmsg_t	parm32;
	int ret = -ENOTTY;
	struct inode *inode = file->f_dentry->d_inode;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* only convert ioctls that need it */
	if (cmd == COMPAT_UTIO_GETMSG || cmd == COMPAT_UTIO_PUTMSG)
	{
		/* these all require the session */
		if (!sess)
			return (-ENXIO);

		/* If the disc bit is set, no operations permitted */
		if (is_disc(sess)) {
			disc_session(file->private_data);
			file->private_data = NULL;
			return (-ENXIO);
		}

		if (copy_from_user(&parm32, (struct compat_utio_pmsg_t *)arg, sizeof(compat_utio_pmsg_t)))
		    return -EFAULT;
		DPRINTF_UTIO_PMSG(2, parm32, "parm32 after copy from user");
		COPY_UTIO_PMSG_T(parm64, parm32);
		DPRINTF_UTIO_PMSG(2, parm64, "parm64 after conversion");

		/* now do the actual ioctl with copied arguments */
		/* this is a modified copy of code from daemon_ioctl() above */
		/* modifications are use of parm64 instead of arg and pass flagkptr=1 */
		switch (cmd) {
		case COMPAT_UTIO_GETMSG:
			DPRINTF(3, "- UTIO_GETMSG\n");
			down_read(&conn->syscall_rwsem);
			ret = do_getmsg_from_slave(sess, &parm64,
				    file->f_flags & O_NONBLOCK, current->uid, 1);
			if (ret < 0) {
				if (is_disc(sess)) {
					disc_session(file->private_data);
					file->private_data = NULL;
					ret = -ENXIO;
					break;
				}
			}
			up_read(&conn->syscall_rwsem);
			break;
		case COMPAT_UTIO_PUTMSG:
			DPRINTF(3, "- UTIO_PUTMSG\n");
			down_read(&conn->syscall_rwsem);
			send_msg_to_slave(sess, &parm64, 1);
			up_read(&conn->syscall_rwsem);
			ret = 0;
			break;
		}

		DPRINTF_UTIO_PMSG(2, parm64, "parm64 after ioctl code");
		COPY_UTIO_PMSG_T(parm32, parm64);
		DPRINTF_UTIO_PMSG(2, parm32, "parm32 before copy to user");
		if (copy_to_user((struct compat_utio_pmsg_t *)arg, &parm32, sizeof(compat_utio_pmsg_t)))
		    return -EFAULT;
		return (ret);
	}

	/* all other ioctls do not need conversion */
	return(daemon_ioctl(inode, file, cmd, arg));
}
#endif	/* CONFIG_COMPAT */

static int
slave_open(struct uta_connect *conn, int sessid, int readfrom, int writeto)
{
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	int ret;

	DPRINTF(2, "entering %s(sessid=%d, readfrom=%d, writeto=%d)\n",
		__FUNCTION__, sessid, !!readfrom, !!writeto);

	/* find the session */
	sess = find_session(sessid);
	if (!sess) {
		DPRINTF(3, "can't find session %d\n", sessid);
		return (-ENODEV);
	}

	/* one user at a time - set this to indicate an unready user */
	if (set_user_lock(sess, 1)) {
		DPRINTF(3, "session %d already has a user\n", sessid);
		unref_session(sess);
		return (-EBUSY);
	}

	/* set up the user side of the session - we want to do this last */
	sess->mode = (readfrom ? MODE_READ : 0) | (writeto ? MODE_WRITE : 0);
	ret = alloc_ios(sess);
	if (ret < 0) {
		goto out_unlock;
	}

	/* init the connection */
	conn->session = sess;
	conn->role = UTA_USER;
	INIT_LIST_HEAD(&sess->messages_to_slave);
	spin_lock_init(&sess->msg_lock_slave);
	init_MUTEX_LOCKED(&sess->msg_sem_slave);

	/* leave the session ref'ed - it is now ready */
	set_has_user(sess, 1);

	/* tell the daemon that we have joined the session */
	pmsg.magic = UTIO_MAGIC;
	pmsg.version = UTIO_VERSION;
	pmsg.msgtype = UTIO_MSG_OPEN;
	pmsg.cmd = sessid;
	memset(&pmsg.args, 0, sizeof (pmsg.args));
	pmsg.uid = current->uid;
	pmsg.datasize = 0;

	/* Check the return value of send_msg_to_master */
	if ((ret = send_msg_to_master(sess, &pmsg))) {
		goto out_unref_user_sess;
	}

	memset(&pmsg, 0, sizeof (pmsg));
	ret = do_getmsg_from_master(sess, &pmsg, current->uid,
				    UTIO_MSG_OPEN);
	if (is_disc(sess)) {
		DPRINTF(3, "slave open failed\n");
		ret = -ENXIO;
		goto out_unref_user_sess;
	}
	if (!ret) {
		if ((pmsg.msgtype == UTIO_MSG_OPEN) &&
		    (pmsg.response == UTIO_MSG_ACK)) {
			DPRINTF(3, "ACK received for UTIO_MSG_OPEN\n");
			return (0);
		} else {
			DPRINTF(3, "No ACK received from daemon\n");
			ret = -ENODEV;
		}
	}

out_unref_user_sess:
	set_has_user(sess, 0);
out_unlock:
	set_user_lock(sess, 0);
	unref_session(sess);
	return (ret);
}


/*
 * the fops functions
 *
 * security: if we get to any of these functions, the FS layer has confirmed
 * the permissions on the device node.
 */

static ssize_t
utio_read(struct file *file, char *ubuf, size_t size, loff_t *ppos)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!file->private_data) {
		DPRINTF(3, "No connection...\n");
		return (-ENXIO);
	}

	if (*ppos != file->f_pos)
		return (-ESPIPE);
	if (size == 0)
		return (0);

	conn = file->private_data;
	/* block other callers of read() */
	if (down_interruptible(&conn->read_sem))
		return (-ERESTARTSYS);
	sess = conn->session;

	if (conn->role == UTA_USER)
		ret = user_read(file, ubuf, size);
	else if (conn->role == UTA_DAEMON)
		ret = daemon_read(file, ubuf, size);
	else
		ret = (-ENXIO);

	up(&conn->read_sem);
	return (ret);
}

static ssize_t
utio_write(struct file *file, const char *ubuf, size_t size, loff_t *ppos)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!file->private_data) {
		DPRINTF(3, "No connection...\n");
		return (-ENXIO);
	}

	if (*ppos != file->f_pos)
		return (-ESPIPE);
	if (size == 0)
		return (0);

	conn = file->private_data;
	/* block other callers of write() */
	if (down_interruptible(&conn->write_sem))
		return (-ERESTARTSYS);
	sess = conn->session;

	if (conn->role == UTA_USER)
		ret = user_write(file, ubuf, size);
	else if (conn->role == UTA_DAEMON)
		ret = daemon_write(file, ubuf, size);
	else
		ret = (-ENXIO);

	up(&conn->write_sem);
	return (ret);
}


static unsigned int
utio_poll(struct file *file, struct poll_table_struct *wait)
{
	struct uta_connect *conn;
	unsigned int ret = 0;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!file->private_data) {
		DPRINTF(3, "No connection...\n");
		return (POLLHUP);
	}

	conn = file->private_data;

	/*
	 * syscall_rwsem locks are acquired
	 * inside user_poll() or daemon_poll(),
	 * so no locks are needed here.
	 */
	if (conn->role == UTA_USER) {
		ret = user_poll(file, wait);
	} else if (conn->role == UTA_DAEMON) {
		ret = daemon_poll(file, wait);
	}
	return (ret);
}

static int
utio_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg)
{
	int ret;
	struct uta_connect *conn;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!file->private_data) {
		DPRINTF(3, "No connection...\n");
		return (-ENODEV);
	}

	conn = file->private_data;

	/*
	 * syscall_rwsem locks are acquired
	 * inside user_ioctl() or daemon_ioctl(),
	 * so no locks are needed here.
	 */
	if (conn->role == UTA_USER)
		ret = user_ioctl(inode, file, cmd, arg);
	else
		ret = daemon_ioctl(inode, file, cmd, arg);

	return (ret);
}

static long
utio_compat_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	lock_kernel();
	switch (cmd) {

#if defined(CONFIG_COMPAT) || defined(CONFIG_SPARC64) || defined(CONFIG_X86_64) || defined(CONFIG_PPC64)

	case COMPAT_UTIO_GETMSG:	/* fall through */
	case COMPAT_UTIO_PUTMSG:
		ret = compat_daemon_ioctl(cmd, arg, file);
		break;
	case COMPAT_PPGETTIME:
		ret = compat_paralleld_ioctl(cmd, arg, file);
		break;

#endif	/* CONFIG_COMPAT */

	default:
		ret = utio_ioctl(file->f_dentry->d_inode, file, cmd, arg);
		break;
	}
	unlock_kernel();

	return (ret);
}

static int
utio_open(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	int sessid;
	int r;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* allocate our connection */
	conn = kmalloc(sizeof (*conn), GFP_KERNEL);
	if (!conn) {
		DPRINTF(1, "can't allocate connection\n");
		return (-ENOMEM);
	}

	conn->session = NULL;
	conn->role = UTA_NONE;
	init_MUTEX(&conn->read_sem);
	init_MUTEX(&conn->write_sem);
	init_rwsem(&conn->syscall_rwsem);
	file->private_data = conn;

	/* check for magic device names */
	sessid = gross_nodename_hack(file);
	if (sessid < 0) {
		DPRINTF(3, "no device ID\n");

		/*
		 * This check is not required now as permission
		 * on /dev/utio will take care of this
		 * if (!capable(CAP_SYS_ADMIN)) {
		 * 	DPRINTF(1, "not capable\n");
		 *	kfree(conn);
		 *	file->private_data = NULL;
		 *	return -ENODEV;
		 * }
		 */

		/* allow root to make new sessions */
		return (0);
	}
	DPRINTF(3, "session ID seems to be %d\n", sessid);

	/* join the session */
	r = slave_open(file->private_data, sessid,
	    file->f_mode & FMODE_READ, file->f_mode & FMODE_WRITE);
	if (r < 0) {
		kfree(conn);
		file->private_data = NULL;
		return (r);
	}
	return (0);
}


static int
utio_release(struct inode *inode, struct file *file)
{
	struct uta_connect *conn;
	struct uta_session *sess;
	utio_pmsg_t pmsg;
	int ret;

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	if (!file->private_data) {
		DPRINTF(3, "No connection...\n");
		return (0);
	}

	conn = file->private_data;
	sess = conn->session;

	if (conn->role == UTA_DAEMON) {
		DPRINTF(3, "disconnecting DAEMON from session %d\n", sess->id);
		/* notify the user of the bad news */
		set_has_daemon(sess, 0);

		/* any queued messages are moot */
		free_all_msgs(sess);

		/*
		 * Careful:
		 * If there is a user, he's done with alloc_ios().  Make sure
		 * he's not asleep or about to sleep on a buffer. If there
		 * is not a user, any connecting user will never get to
		 * down() any bufs, since has_daemon() is now 0.
		 */
		if (has_user(sess)) {
			wakeup_ios(sess);
			/*
			 * you can't get any more ACKs/NACKs,
			 * wake sleepers or racers
			 */
			up(&sess->msg_sem_slave);
		}
		// AUDIT: make sure the user is woken up from any block!
	} else if (conn->role == UTA_USER) {
		DPRINTF(3, "disconnecting USER from session %d\n",
			sess->id);
		/* sync, no ACK required */
		/* device_sync(sess, 0); */
		/* tell the daemon we're gone */
		pmsg.magic = UTIO_MAGIC;
		pmsg.version = UTIO_VERSION;
		pmsg.msgtype = UTIO_MSG_CLOSE;
		pmsg.cmd = sess->id;
		memset(&pmsg.args, 0, sizeof (pmsg.args));
		pmsg.uid = current->uid;
		pmsg.datasize = 0;

		ret = send_msg_to_master(sess, &pmsg);
		if (!ret) {
			memset(&pmsg, 0, sizeof (pmsg));
			ret = do_getmsg_from_master(sess, &pmsg,
					current->uid, UTIO_MSG_CLOSE);
			if (!ret && (pmsg.msgtype == UTIO_MSG_CLOSE) &&
			    (pmsg.response == UTIO_MSG_ACK)) {
				DPRINTF(3, "ACK received for UTIO_MSG_CLOSE\n");
			} else {
				// XXX should never happen
				// XXX should we re-issue close again??
				DPRINTF(3, "No ACK for UTIO_MSG_CLOSE\n");
			}
		} else {
			// XXX failed to send close to master
			// XXX should we retry or wait ??
		}

		/* tell the daemon, first */
		set_has_user(sess, 0);

		/* another user can join, now */
		set_user_lock(sess, 0);
		// AUDIT: anything else needed here?
	}

	/* drop our reference */
	if (sess)
		unref_session(sess);

	/* done with the connection, now */
	kfree(file->private_data);
	file->private_data = NULL;

	return (0);
}


/*
 * utio - Sun Ray pseudo I/O driver
 */

struct file_operations utio_fops = {
	owner:		THIS_MODULE,
	llseek:		no_llseek,
	read:		utio_read,
	write:		utio_write,
	poll:		utio_poll,
	ioctl:		utio_ioctl,
	compat_ioctl:	utio_compat_ioctl,
	open:		utio_open,
	release:	utio_release,
};


/*
 * Currently there is no benefit to make use of the __init
 * and __exit macros for utio_init and utio_exit functions
 * as these don't have any effect on the loadable modules.
 * But may be in future kernel versions, these macros might
 * have some effect.
 */

static int __init
utio_init(void)
{
	int result;
	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/* set up the session-tracking stuff */
	sessbits_count = MAX_SESSIONS/BITS_PER_LONG;
	sessbits_count += (MAX_SESSIONS % BITS_PER_LONG) ? 1 : 0;

	session_bits = kmalloc(sessbits_count * sizeof (*session_bits),
				GFP_KERNEL);

	if (!session_bits) {
		printk(KERN_ERR "utio: can't allocate session_bits\n");
		return (-ENOMEM);
	}

	memset(session_bits, 0, sessbits_count * sizeof (*session_bits));

	/* set any bits beyond max_sessions, so we can ignore them later */
	if (MAX_SESSIONS % BITS_PER_LONG) {
		int r = MAX_SESSIONS % BITS_PER_LONG;
		int i;
		unsigned long *last_word = &session_bits[sessbits_count - 1];

		for (i = r; i < BITS_PER_LONG; i++) {
			__set_bit(i, last_word);
		}
	}

	/* register the device */
	result = register_chrdev(utio_major, "utio", &utio_fops);
	if (result < 0) {
		printk(KERN_ERR "utio: can't register utio device\n");
		kfree(session_bits);
		return (result);
	}

	/* dynamic major number */
	if (utio_major == 0) utio_major = result;

	DPRINTF(3, "registered utio dev %d\n", utio_major);
	DPRINTF(3, "Create /dev/utio  using mknod /dev/utio c %d 0\n",
		utio_major);

	/* register a file in /proc/driver */
	if (!create_proc_read_entry("driver/utio", 0, 0,
				    utio_proc_read, NULL)) {
	    printk(KERN_ERR "utio: can't create /proc/driver/utio\n");
	    unregister_chrdev(utio_major, "utio");
	    kfree(session_bits);
	    return (-ENOMEM);
	}

	/* init successful */
	return (0);
}

static void __exit
utio_exit(void)
{

	DPRINTF(2, "entering %s()\n", __FUNCTION__);

	/*
	 * XXX unregister the device
	 * XXX free all the allocated memory
	 */
	remove_proc_entry("driver/utio", NULL);
	unregister_chrdev(utio_major, "utio");
	kfree(session_bits);
}


module_init(utio_init);
module_exit(utio_exit);

MODULE_AUTHOR("Sun Ray Engineering <srss-feedback-ext@Sun.COM>");
MODULE_DESCRIPTION("utio Sun Ray pseudo I/O driver v1.10");
MODULE_LICENSE("GPL");

/*
 * The device name that's supported by this module. In
 * future, this parameter may be used by the kernel to
 * load this module automatically, but is currently used
 * only for documention purposes.
 */
MODULE_SUPPORTED_DEVICE("utio");

module_param(debug, int, 0);
