/*
 * ident	"@(#)utdisk.c	1.8	09/05/15 SMI"
 *
 * Copyright 2009 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#ifdef _SCCSID
static char *_SCCSid = "@(#)utdisk.c	1.8	09/05/15 SMI";
#endif	/* defined _SCCSID */

/*
 * Sun Ray mass storage loopback driver
 * Provides a block device interface to mass storage devices on Sun Ray
 */

/* includes */
#include <linux/module.h>		/* module macros */
#include <linux/blkdev.h>		/* blk_ functions */
#include <linux/hdreg.h>		/* HDIO */
#include <linux/proc_fs.h>		/* create_proc_read_entry */
#include <linux/init.h>			/* __init macro */
#include <linux/fd.h>			/* FDEJECT */
#include <linux/cdrom.h>		/* CDROMEJECT */
#include <linux/smp_lock.h>		/* lock_kernel */
#include <scsi/scsi.h>			/* command values*/
#include <scsi/scsi_ioctl.h>		/* SCSI_IOCTL_SEND_COMMAND */
#include <scsi/scsi_cmnd.h>		/* MAX_COMMAND_SIZE */
#include "utdiskctl.h"			/* UTDISK_DISK_DRV_NAME */
#include "utdisk.h"

/* macro and constant definitions */

/* some constants to fake disk geometry */
#define	UTDISK_DEFAULT_HEADS	32
#define	UTDISK_DEFAULT_SECTORS	8

#define UT_MAX_SCSI_DATA		PAGE_SIZE
#define UT_SCSI_COMMAND_SIZE		MAX_COMMAND_SIZE
#define UT_SCSI_START_STOP_LOAD_EJECT	0x02

/* typedefs, structs and unions */

/* static function prototypes */

static int utdisk_init(void);
static void utdisk_exit(void);
static int utdisk_open(struct inode *inodep, struct file *fp);
static int utdisk_release(struct inode *inodep, struct file *fp);
static int utdisk_media_changed(struct gendisk *gdp);
static int utdisk_revalidate_disk(struct gendisk *gdp);
static int utdisk_getgeo(struct block_device *bdevp, struct hd_geometry *geo);
static int utdisk_ioctl(struct inode *inodep, struct file *fp,
			unsigned int cmd, unsigned long arg);
static long utdisk_compat_ioctl(struct file *fp,
			unsigned int cmd, unsigned long arg);
static int utdisk_make_request(request_queue_t *q, struct bio *bio);
static int utdisk_scsi_command(utdisk_state_t *dsp, struct block_device *bdevp,
			unsigned long arg, int flag32ptr);

/* global variables */

/* static global variables */

static major_t		utdisk_major;			/* driver major # */

/* block device operations */
static struct block_device_operations utdisk_bd_ops = {
	.owner =		THIS_MODULE,
	.open =			utdisk_open,
	.ioctl =		utdisk_ioctl,
	.compat_ioctl =		utdisk_compat_ioctl,
	.release =		utdisk_release,
	.media_changed =	utdisk_media_changed,
	.revalidate_disk =	utdisk_revalidate_disk,
	.getgeo =		utdisk_getgeo,
};

/* 32bit to 64bit translation */

#ifdef	_LP64

struct hd_geometry32 {
	unsigned char	heads;
	unsigned char	sectors;
	unsigned short	cylinders;
	u32		start;
};

#endif	/* _LP64 */


/* static function definitions */

/*
 * will print the following to the file /proc/driver/utdisk:
 * major: <major number>
 */
static int
utdisk_proc_read(char *buf, char **start, off_t offset,
		 int len, int *eof, void *data)
{
	int dlen;

	dlen = snprintf(buf, len, "major: %d\n", utdisk_major);

	/* trying to read a bad offset? */
	if (offset >= dlen) {
		*eof = 1;
		return (0);
	}

	/* did we get everything? */
	if (len >= (dlen-offset)) {
		*eof = 1;
	}

	*start = buf + offset;
	dlen -= offset;

	return ((len > dlen) ? dlen : len);
}


/* module entry points */

/* module initialization */
static int __init
utdisk_init(void)
{
	utprintf("%s\n", __FUNCTION__);

	/* init globals */
	utdisk_major = 0;

	if ((utdisk_major = register_blkdev(utdisk_major, UTDISK_DISK_DRV_NAME))
		<= 0) {
		utdisk_major = 0;
		return (-EIO);
	}
	utprintf("%s: major = %d\n", __FUNCTION__, utdisk_major);

	/* register a file in /proc/driver */
	if (!create_proc_read_entry("driver/utdisk", 0, 0,
				    utdisk_proc_read, NULL)) {
		printk(KERN_ERR "utdisk: can't create /proc/driver/utdisk\n");
		unregister_blkdev(utdisk_major, UTDISK_DISK_DRV_NAME);
		return (-ENOMEM);
	}
		
	/* publish parameters needed by utdiskctl */
	utstk_set_disk_major(utdisk_major);
	utstk_set_disk_makereq(&utdisk_make_request);
	utstk_set_disk_bdops(&utdisk_bd_ops);

	return (0);
}


/* module exit */
static void __exit
utdisk_exit(void)
{
	utprintf("%s\n", __FUNCTION__);

	if (utdisk_major != 0) {
		unregister_blkdev(utdisk_major, UTDISK_DISK_DRV_NAME);
		utdisk_major = 0;
		utstk_set_disk_major(utdisk_major);
		utstk_set_disk_makereq(NULL);
		utstk_set_disk_bdops(NULL);
	}

	utprintf("%s DONE\n", __FUNCTION__);
}


/* device entry points */

/* open */
static int
utdisk_open(struct inode *inodep, struct file *fp)
{
	struct gendisk	*gdp;
	utdisk_state_t	*dsp;
	minor_t		minnum;

	if ((inodep == NULL)
		|| (inodep->i_bdev == NULL)
		|| ((gdp = inodep->i_bdev->bd_disk) == NULL)
		|| ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		return (-EINVAL);
	}

	minnum = iminor(inodep);

	utprintf("%s: minor=%d\n", __FUNCTION__, minnum);

	if (down_interruptible(&(dsp->lock)) == 0) {
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			utprintf("%s: stale device %d\n", __FUNCTION__, minnum);
			up(&(dsp->lock));
			return (-ENXIO);
		}
		/* check write protect status */
		if ((dsp->flags & UTDISK_STATE_WRPROT) &&
			(fp->f_mode & FMODE_WRITE)) {
			utprintf("%s: write protected\n", __FUNCTION__);
			up(&(dsp->lock));
			return (-EROFS);
		}
		dsp->opens++;
		up(&(dsp->lock));
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-EIO);
	}
	check_disk_change(inodep->i_bdev);
	utprintf("%s: minor=%d [%s] usage=%d\n",
		__FUNCTION__, minnum,
		((char *)(fp->f_dentry->d_name.name)
				? (char *)(fp->f_dentry->d_name.name) : "null"),
		dsp->opens);

	return 0;	/* success */
}


/* release */
static int
utdisk_release(struct inode *inodep, struct file *fp)
{
	struct gendisk	*gdp;
	utdisk_state_t	*dsp;
	minor_t		minnum;

	if ((inodep == NULL)
		|| (inodep->i_bdev == NULL)
		|| ((gdp = inodep->i_bdev->bd_disk) == NULL)
		|| ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		return (-EINVAL);
	}

	minnum = iminor(inodep);

	utprintf("%s: minor=%d usage=%d\n", __FUNCTION__, minnum, dsp->opens);
	if (down_interruptible(&(dsp->lock)) == 0) {
		if (dsp->opens != 0) {
			dsp->opens--;
		}
		up(&(dsp->lock));
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			/* try to clean up stale state */
			utstk_free_state(dsp);
		}
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-EIO);
	}

	return 0;	/* success */
}


/* ioctl */
static int
utdisk_ioctl(struct inode *inodep, struct file *fp, unsigned int cmd,
	unsigned long arg)
{
	struct block_device	*bdevp;
	struct gendisk		*gdp;
	utdisk_state_t		*dsp;
	struct hd_geometry	geo;
#ifdef	_LP64
	struct hd_geometry32	geo32;
#endif	/* _LP64 */
	int			flag32ptr;
	char			*copy_ptr;
	int			copy_len;
	int			ret;

	flag32ptr = (cmd & UTDISK_32BIT_PTRS);
	cmd &= ~UTDISK_32BIT_PTRS;
	utprintf("%s: cmd = %x, flag32=%x\n", __FUNCTION__, cmd, flag32ptr);

	if ((inodep == NULL)
		|| ((bdevp = inodep->i_bdev) == NULL)
		|| ((gdp = bdevp->bd_disk) == NULL)
		|| ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		return (-EINVAL);
	}
	utprintf("%s: dsp = %p ctlmin = %x\n", __FUNCTION__, dsp, dsp->ctlmin);

	if (down_interruptible(&(dsp->lock)) == 0) {
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			utprintf("%s: stale device %d\n", __FUNCTION__,
								dsp->ctlmin);
			up(&(dsp->lock));
			return (-ENXIO);
		}
		up(&(dsp->lock));
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-EIO);
	}

	switch (cmd) {
	case HDIO_GETGEO:
		if ((ret = utdisk_getgeo(bdevp, &geo)) != 0) {
			return (ret);
		}
#ifdef	_LP64
		/* convert to 32 bit */
		geo32.heads = geo.heads;
		geo32.sectors = geo.sectors;
		geo32.cylinders = geo.cylinders;
		geo32.start = geo.start;
		if (flag32ptr) {
			copy_ptr = (char *)&geo32;
			copy_len = sizeof (geo32);
		} else {
			copy_ptr = (char *)&geo;
			copy_len = sizeof (geo);
		}
#else
		copy_ptr = (char *)&geo;
		copy_len = sizeof (geo);
#endif	/* _LP64 */
		if (copy_to_user((void *)arg, copy_ptr, copy_len) != 0) {
			utprintf("%s: copy_out failed\n", __FUNCTION__);
			return (-EFAULT);
		}

		utprintf("%s cyl=%x\n", __FUNCTION__, geo.cylinders);
		return (0);
	case CDROMEJECT:
	case FDEJECT:
		utstk_eject_media(dsp, bdevp);
		return (0);
	case SCSI_IOCTL_SEND_COMMAND:
		return (utdisk_scsi_command(dsp, bdevp, arg, flag32ptr));
	default:
		utprintf("%s: cmd=%x unknown\n", __FUNCTION__, cmd);
		break;
	}

	return (-ENOTTY);	/* unknown command */
}


/*
 * ioctl called by 32bit user process into 64bit kernel
 */
static long
utdisk_compat_ioctl(struct file *fp, unsigned int cmd, unsigned long arg)
{
	int	ret;

	utprintf("%s: cmd = %x\n", __FUNCTION__, cmd);

	lock_kernel();		/* Big Kernel Lock */
	ret = utdisk_ioctl(fp->f_dentry->d_inode, fp, (cmd | UTDISK_32BIT_PTRS),
			arg);
	unlock_kernel();

	return (long)(ret);
}


/*
 * process SCSI_IOCTL_SEND_COMMAND ioctl
 */
static int
utdisk_scsi_command(utdisk_state_t *dsp, struct block_device *bdevp,
		unsigned long arg, int flag32ptr)
{
	struct sdata {
		unsigned int	inlen;
		unsigned int	outlen;
		unsigned char	cmd[UT_SCSI_COMMAND_SIZE];
	} scsi_cmd;
	struct sdata __user	*in_data;
	unsigned char		opcode;
	ut_uscsi_cmd_t		uscsi_cmd;
	unsigned int 		cmdlen;
	unsigned int 		buflen;
	int 			ret;
	int			senselen;

	utprintf("%s: caller %d, owner %d\n", __FUNCTION__,
				current->uid, dsp->owner);
	if ((current->uid != 0) && (dsp->owner != current->uid)) {
		utprintf("%s: caller %d not owner %d\n", __FUNCTION__,
				current->uid, dsp->owner);
		return (-EACCES);
	}

        utprintf("%s\n", __FUNCTION__);
	if (copy_from_user(&scsi_cmd, (void *)arg, sizeof (scsi_cmd)) != 0) {
		utprintf("%s: copy_from_user failed\n", __FUNCTION__);
		return (-EFAULT);
	}
	utprintf("%s: %d %d cmd[0]: %x cmd[4]: %x \n", __FUNCTION__,
		scsi_cmd.inlen, scsi_cmd.outlen, scsi_cmd.cmd[0],
		scsi_cmd.cmd[4]);
	if (scsi_cmd.inlen > UT_MAX_SCSI_DATA) {
		return (-EINVAL);
	}
	if (scsi_cmd.outlen > UT_MAX_SCSI_DATA) {
		return (-EINVAL);
	}
	opcode = scsi_cmd.cmd[0];
	cmdlen = COMMAND_SIZE(opcode);
	if (cmdlen > UT_SCSI_COMMAND_SIZE) {
		utprintf("%s: Command length too large (%x) for opcode %x\n",
						__FUNCTION__, cmdlen, opcode);
		return (-EINVAL);
	}
	/*
	 * If we find an eject request send IOCTL to do the ejection
	 * so that the same code is called in utstoraged for ejection
	 * on Solaris and Linux.
	 */
	if ((opcode == START_STOP) &&
		(scsi_cmd.cmd[4] == UT_SCSI_START_STOP_LOAD_EJECT)) {
		utstk_eject_media(dsp, bdevp);
		return (0);
	}
	buflen = (scsi_cmd.inlen > scsi_cmd.outlen ? scsi_cmd.inlen
							: scsi_cmd.outlen);
	if (buflen > 0) {
		utprintf("%s: Buffer (%x) required cmd %x not implemented\n",
			__FUNCTION__, buflen, opcode);
		return (-EINVAL);
	}
	memset(&uscsi_cmd, 0, sizeof(ut_uscsi_cmd_t));
	uscsi_cmd.uscsi_flags = UT_USCSI_READ | UT_USCSI_RQENABLE;
	uscsi_cmd.uscsi_timeout = UT_SCSI_TIMEOUT;
	uscsi_cmd.uscsi_cdb = scsi_cmd.cmd;
	uscsi_cmd.uscsi_cdblen = cmdlen;
	uscsi_cmd.uscsi_rqlen = UTS_DEFAULT_SENSEBUFSIZE;
	ret = utstk_uscsi_cmd(dsp, bdevp, &uscsi_cmd);
	if (uscsi_cmd.uscsi_rqbuf != NULL) {
		senselen = uscsi_cmd.uscsi_rqlen - uscsi_cmd.uscsi_rqresid;
		if (senselen > cmdlen) {
			senselen = cmdlen;
		}
		in_data = (struct sdata *)arg;
		if (copy_to_user(in_data->cmd, uscsi_cmd.uscsi_rqbuf,
							senselen) != 0) {
			utprintf("%s: copy out failed for rqbuf\n",
							__FUNCTION__);
		}
		vfree(uscsi_cmd.uscsi_rqbuf);
		/*
		 * See http://lxr.linux.no/source/drivers/scsi/scsi_lib.c#L267
		 * which sets return value in similar way.
		 */
		ret |= DRIVER_ERROR << 24;
	}

	return (ret);	
}


/*
 * check whether media has changed
 * return 1 if media has changed, 0 if no change
 */
static int
utdisk_media_changed(struct gendisk *gdp)
{
	utdisk_state_t		*dsp;
	int ret = 0;

        utprintf("%s\n", __FUNCTION__);
	if ((gdp == NULL) || ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		return (-EINVAL);
	}

	if (down_interruptible(&(dsp->lock)) == 0) {
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			utprintf("%s: stale device %d\n", __FUNCTION__,
								dsp->ctlmin);
			up(&(dsp->lock));
			return (-ENXIO);
		}
		if ((dsp->flags & UTDISK_STATE_CHANGED) != 0) {
			ret = 1;
			dsp->flags &= ~UTDISK_STATE_CHANGED;
			dsp->flags |= UTDISK_STATE_READY;
        		utprintf("%s disk changed\n", __FUNCTION__);
		}
		up(&(dsp->lock));
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-EIO);
	}

        return (ret);
}


/*
 * revalidate partition info
 * return 0 on success, or error code
 */
static int
utdisk_revalidate_disk(struct gendisk *gdp)
{
	utdisk_state_t	*dsp;

        utprintf("%s\n", __FUNCTION__);

	if ((gdp == NULL) || ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		return (-EINVAL);
	}

	if (down_interruptible(&(dsp->lock)) == 0) {
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			utprintf("%s: stale device %d\n", __FUNCTION__,
								dsp->ctlmin);
			up(&(dsp->lock));
			return (-ENXIO);
		}
		up(&(dsp->lock));
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-EIO);
	}
        /*
         * only need to set capacity (in 512 Byte sectors)
         * todo: query disk to see whether anything's changed
         */
	set_capacity(gdp, dsp->nblocks);

        utprintf("%s done\n", __FUNCTION__);

        return (0);
}


/*
 * get disk geometry
 * return 0 on success, or error code
 */
static int
utdisk_getgeo(struct block_device *bdevp, struct hd_geometry *geo)
{
	struct gendisk	*gdp;
	utdisk_state_t	*dsp;

        utprintf("%s\n", __FUNCTION__);

	if ((bdevp == NULL) || ((gdp = bdevp->bd_disk) == NULL)
		|| ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		return (-EINVAL);
	}

	if (down_interruptible(&(dsp->lock)) == 0) {
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			utprintf("%s: stale device %d\n", __FUNCTION__,
								dsp->ctlmin);
			up(&(dsp->lock));
			return (-ENXIO);
		}
		up(&(dsp->lock));
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-EIO);
	}
	if (dsp->nblocks != 0) {
		geo->heads = UTDISK_DEFAULT_HEADS;
		geo->sectors = UTDISK_DEFAULT_SECTORS;
		geo->cylinders = (dsp->nblocks /
			(UTDISK_DEFAULT_HEADS * UTDISK_DEFAULT_SECTORS));
		geo->start = get_start_sect(bdevp);
	} else {
		geo->heads = 0;
		geo->sectors = 0;
		geo->cylinders = 0;
		geo->start = 0;
		utstk_check_media(dsp, bdevp);
	}

        utprintf("%s done\n", __FUNCTION__);

        return (0);
}


/*
 * make a transfer request
 * In our case, we don't make a transfer request
 * We don't gain the performance benefit offered by request queues,
 * so simply pass the biop onto the controller
 *
 * Always return 0
 * never return non-zero since we are not a stacking driver
 * On error, call bio_endio(..., -error), but still return 0
 */
static int
utdisk_make_request(request_queue_t *qp, struct bio *biop)
{
	struct block_device	*bdevp;
	struct gendisk		*gdp;
	utdisk_state_t		*dsp;

	utprintf("%s bio: %p\n", __FUNCTION__, biop);

	/* sanity check */
	if ((biop == NULL)
		|| ((bdevp = biop->bi_bdev) == NULL)
		|| ((gdp = bdevp->bd_disk) == NULL)
		|| ((dsp = gdp->private_data) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad argument\n", __FUNCTION__);
		bio_endio(biop, biop->bi_size, (-EINVAL));
		return (0);
	}
	/* is transfer size a multiple of hardware block size */
	if ((biop->bi_size % UTDISK_HARDSECT_SIZE_BYTES) != 0) {
		utprintf("%s: length %x not a block size multiple\n",
			__FUNCTION__, biop->bi_size);
		bio_endio(biop, biop->bi_size, (-EINVAL));
		return (0);
	}
	if (down_interruptible(&(dsp->lock)) == 0) {
		if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
			utprintf("%s: stale device %d\n", __FUNCTION__,
								dsp->ctlmin);
			up(&(dsp->lock));
			bio_endio(biop, biop->bi_size, (-ENXIO));
			return (0);
		}
		up(&(dsp->lock));
	} else {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		bio_endio(biop, biop->bi_size, (-EIO));
		return (0);
	}

	utprintf("%s: bio=%p, disk=%d, sect=%lx, bytes=%x, rw=%lx\n",
		__FUNCTION__, biop, dsp->ctlmin,
		(unsigned long)(biop->bi_sector), biop->bi_size, bio_rw(biop));

	/*
	 * create kernel space mappings of high memory buffers
	 * they get released in bio_endio()
	 */
	blk_queue_bounce(qp, &biop);

	/* pass it on to controller */
	if (utstk_add_bio(biop, dsp) != 0) {
		utprintf("%s: [%d] add buf failed\n", __FUNCTION__,
								dsp->ctlmin);
		bio_endio(biop, biop->bi_size, (-EIO));
		return (0);
	}

	return (0);
}


module_init(utdisk_init);
module_exit(utdisk_exit);

MODULE_LICENSE("GPL");
