/*
 * ident	"@(#)utdiskctl.c	1.12	09/05/15 SMI"
 *
 * Copyright 2009 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#ifdef _SCCSID
static char *_SCCSid = "@(#)utdiskctl.c	1.12	09/05/15 SMI";
#endif	/* defined _SCCSID */

/*
 * Sun Ray mass storage device controller
 * This driver watches for hotplug & other events from the Sun Ray storage
 * service in userland and accordingly initializes and frees state for utdisk.
 * It is the broker for data transfer and ioctl requests between utdisk and
 * utstoraged.
 */

/* includes */

#include <linux/module.h>		/* module macros */
#include <linux/init.h>			/* module entry-point macros */
#include <linux/fs.h>			/* device entry points */
#include <linux/proc_fs.h>		/* create_proc_read_entry */
#include <linux/smp_lock.h>		/* lock_kernel */
#include <asm/uaccess.h>		/* userland access */
#include "utdisk.h"
#include "utdiskctl.h"

/* macro and constant definitions */

/* state space allocation parameters */
#ifdef	UTDEBUG
#define	UTDISK_MAXMINOR		0x6
#define	UTDISK_ALLOC_CHUNK	0x4
#else
#define	UTDISK_MAXMINOR		((1 << MINORBITS) / UTDISK_MINORS_PER_DISK)
#define	UTDISK_ALLOC_CHUNK	0x400
#endif	/* UTDEBUG */

/* ioctl request state flags */
#define UTDISK_IOCTL_NEW        0x1
#define UTDISK_IOCTL_SENT       0x2
#define UTDISK_IOCTL_DONE       0x4
#define UTDISK_IOCTL_INTR       0x8

/* type of request for bio */
#define UTSTK_TYPE_DATA		1
#define UTSTK_TYPE_IOCTL	2
#define UTSTK_TYPE_USCSI	3

/* static function prototypes */

static int	utstk_init(void);
static void	utstk_exit(void);
static void	utstk_cleanup(void);
static int	utstk_open(struct inode *inode, struct file *fp);
static int	utstk_release(struct inode *inode, struct file *fp);
static ssize_t	utstk_file_read(struct file *fp, char *bufp,
				size_t count, loff_t *ppos);
static ssize_t	utstk_file_write(struct file *fp, const char *bufp,
				size_t count, loff_t *ppos);
static int	utstk_ioctl(struct inode *inode, struct file *fp,
				unsigned int command, unsigned long arg);
static long	utstk_compat_ioctl(struct file *fp,
				unsigned int command, unsigned long arg);
static int	utstk_get_ctl_major(unsigned long arg, int flag32ptr);
static int	utstk_get_disk_major(unsigned long arg, int flag32ptr);
static int	utstk_hotplug(unsigned long arg, int flag32ptr);
static int	utstk_unplug(unsigned long arg, int flag32ptr);
static int	utstk_media_change(unsigned long arg, int flag32ptr);
static int	utstk_disk_ready(unsigned long arg, int flag32ptr);
static int	utstk_get_mesg(utdisk_state_t *dsp, unsigned long arg,
							int flag32ptr);
static int	utstk_get_cdb(utdisk_state_t *dsp, unsigned long arg,
							int flag32ptr);
static int	utstk_tx_response(utdisk_state_t *dsp, utdc_tx_t *tx,
							int flag32ptr);
static utdisk_state_t *	utstk_new_state(void);
static utdisk_state_t *	utstk_acquire_state(minor_t num, uint_t mode);
static void	utstk_release_state(utdisk_state_t *dsp);
static int	utstk_set_state(minor_t mnum, utdisk_state_t *dsp);
static int	utstk_do_ioctl(utdisk_state_t *dsp, struct block_device *bdevp,
				int cmd, void *arg, int type);

/* typedefs, structs and unions */

/* char device operations */
static struct file_operations utdc_fops = {
	.read		= utstk_file_read,
	.write		= utstk_file_write,
	.open		= utstk_open,
	.release	= utstk_release,
	.ioctl		= utstk_ioctl,
	.compat_ioctl	= utstk_compat_ioctl,
	.owner		= THIS_MODULE,
};

/* minor number list header */
typedef struct utdc_mlist_s {
	minor_t		next_num;	/* next available instance number */
	utdisk_state_t	**state;	/* array of state pointers */
	minor_t		*freenums;	/* free list, grows same as *state */
	minor_t		free_head;	/* next free num is taken from here */
	minor_t		free_tail;	/* free num is inserted here */
	uint_t		size;		/* size of store & free list */
} utdc_mlist_t;



/*
 * ioctl message to carry utdisk ioctl requests
 *
 */
typedef struct utdc_ioctl_s {
	int             cmd;            /* ioctl command */
	void            *arg;           /* ioctl specific baggage */
	utdisk_state_t  *dsp;           /* utdisk state pointer */
	unsigned int    flags;          /* UTDISK_IOCTL_ states */
} utdc_ioctl_t;

/*
 * data structure stored in bi_private field of cloned bio
 */
typedef struct utdc_bio_descriptor_s {
	int			type;
	struct bio		*bio_orig;
	wait_queue_head_t       ioctl_wq;
	utdc_ioctl_t		ioctl_req;
} utdc_bio_descriptor_t;

/* minor number store manipulation */
static int	utstk_minor_list_init(utdc_mlist_t *mp);
static void	utstk_minor_list_fini(utdc_mlist_t *mp);
static minor_t	utstk_minor_get_num(utdc_mlist_t *mlist);
static int	utstk_minor_free_num(utdc_mlist_t *mlist, minor_t num);
static minor_t	utstk_minor_get_free_head(utdc_mlist_t *mlist);
static void	utstk_minor_invalidate_all(void);

static int	utstk_construct_bio_descriptor(struct bio *biop,
			utdisk_state_t *dsp, int type);

/* global variables */

/* static global variables */

static major_t		utdc_major;		/* our major number */
static major_t		utdisk_major;		/* utdisk major number */
static struct semaphore	glock;			/* to access globals */
static struct cdev	*mcdevp;		/* master cdev */
static utdisk_state_t	mds;			/* master's state */
static make_request_fn  *utdisk_make_request_p;	/* make request function */
static struct block_device_operations *utdisk_bd_ops;	/* bd ops */
static utdc_mlist_t	mlist;			/* minor number pool */


/* static function definitions */

/*
 * will print the following to the file /proc/driver/utdisk:
 * major: <major number>
 */
static int
utdiskctl_proc_read(char *buf, char **start, off_t offset,
		    int len, int *eof, void *data)
{
	int dlen;

	dlen = snprintf(buf, len, "major: %d\n", utdc_major);

	/* trying to read a bad offset? */
	if (offset >= dlen) {
		*eof = 1;
		return (0);
	}

	/* did we get everything? */
	if (len >= (dlen-offset)) {
		*eof = 1;
	}

	*start = buf + offset;
	dlen -= offset;

	return ((len > dlen) ? dlen : len);
}


/* init */
static int
utstk_init(void)
{
	dev_t	dev;

	utprintf("%s\n", __FUNCTION__);

	/* init globals first */
	utdc_major = 0;
	utdisk_major = 0;
	sema_init(&glock, 1);			/* one resource */
	mcdevp = NULL;
	memset(&mds, 0, sizeof (mds));
	mds.ctlmin = UTDISK_MASTER_INSTANCE;
#ifdef	UTDEBUG
	mds.selfp = &mds;
#endif	/* UTDEBUG */
	sema_init(&(mds.lock), 1);
	utdisk_make_request_p = NULL;
	utdisk_bd_ops = NULL;
	if (utstk_minor_list_init(&mlist) != 0) {
		utstk_cleanup();
		return (1);
	}

	/*
	 * request major number & full range of minors we'll need
	 * Note: This call does not allocate memory for minor numbers
	 * It just marks a number range as being in use
	 */
	if (alloc_chrdev_region(&dev, UTDISK_MASTER_INSTANCE, UTDISK_MAXMINOR,
		UTDISK_CTL_DRV_NAME) != 0) {
		utprintf("%s: alloc_chrdev failed\n", __FUNCTION__);
		utstk_cleanup();
		return (1);
	}

	utdc_major = MAJOR(dev);	/* remember major */
	mcdevp = cdev_alloc();		/* no need cdev_init if cdev_alloc'd */
	if (mcdevp == NULL) {
		utprintf("%s: cdev_alloc failed\n", __FUNCTION__);
		utstk_cleanup();
		return (1);
	}
	mcdevp->owner = THIS_MODULE;
	mcdevp->ops = &utdc_fops;
	if (cdev_add(mcdevp, dev, 1) != 0) {
		utprintf("%s: cdev_add failed\n", __FUNCTION__);
		kobject_put(&(mcdevp->kobj));
		mcdevp = NULL;
		utstk_cleanup();
	}

	/* register a file in /proc/driver */
	if (create_proc_read_entry("driver/utdiskctl", 0, 0,
				     utdiskctl_proc_read, NULL) == 0) {
		printk(KERN_ERR "utdiskctl: can't create /proc/driver/utio\n");
		utstk_cleanup();
		return (-ENOMEM);
	}

	utprintf("%s: major=%d\n", __FUNCTION__, utdc_major);

	return (0);
}


/* exit */
static void
utstk_exit(void)
{
	utprintf("%s\n", __FUNCTION__);

	utstk_cleanup();
}


/* clean up everything */
static void
utstk_cleanup(void)
{
	utprintf("%s\n", __FUNCTION__);

	if (mcdevp != NULL) {
		cdev_del(mcdevp);
		mcdevp = NULL;
	}

	if (utdc_major != 0) {
		unregister_chrdev_region(MKDEV(utdc_major,
						UTDISK_MASTER_INSTANCE),
					UTDISK_MAXMINOR);
		utdc_major = 0;
	}

	utstk_minor_list_fini(&mlist);
	utdisk_make_request_p = NULL;
	utdisk_bd_ops = NULL;
}


/* device entry points */

/* open */
static int
utstk_open(struct inode *inode, struct file *fp)
{
	utdisk_state_t	*dsp;
	minor_t		mnum = iminor(inode);

	utprintf("%s: minor=%d\n", __FUNCTION__, mnum);

	/* must be opened exclusively */
	if ((fp == NULL) || ((fp->f_flags & O_EXCL) == 0)) {
		return (-EINVAL);
	}
	dsp = utstk_acquire_state(mnum, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s: minor=%d: no state\n", __FUNCTION__, mnum);
		return (-ENXIO);
	}
	/* enforce exclusive open */
	if (dsp->flags & UTDISK_STATE_OPEN) {
		utstk_release_state(dsp);
		utprintf("%s: minor=%d in use\n", __FUNCTION__, mnum);
		return (-EBUSY);
	}
	dsp->flags |= UTDISK_STATE_OPEN;
	fp->private_data = dsp;		/* save reference to dsp */
	utstk_release_state(dsp);

	return (0);
}


/* close */
static int
utstk_release(struct inode *inode, struct file *fp)
{
	utdisk_state_t	*dsp;
	minor_t		mnum = iminor(inode);

	utprintf("%s: minor=%d\n", __FUNCTION__, mnum);
	/* don't check stale state, device may have just been unplugged */
	dsp = utstk_acquire_state(mnum, UTDISK_STATE_FORCE);
	if (dsp == NULL) {
		utprintf("%s: minor=%d: no state\n", __FUNCTION__, mnum);
		return (-ENXIO);
	}
	dsp->flags &= ~UTDISK_STATE_OPEN;
	if (mnum != UTDISK_MASTER_INSTANCE) {
		dsp->flags |= UTDISK_STATE_STALE;
	}
	fp->private_data = NULL;
	utstk_release_state(dsp);

	/* master's state shall be cleaned up when module is unloaded */
	if (mnum != UTDISK_MASTER_INSTANCE) {
		utstk_free_state(dsp);
	} else {
		/* utstoraged is dying, mark everything stale */
		utstk_minor_invalidate_all();
	}

	return (0);
}


/* read data from application buffer to write to device */
static ssize_t
utstk_file_read(struct file *fp, char *bufp, size_t count, loff_t *ppos)
{
	utdisk_state_t	*dsp;
	struct bio	*biop;
	struct bio_vec	*bvecp;
	minor_t		mnum;
	unsigned char	*src;
	unsigned long	len;
	int		rw;
	int		ret = 0;

	if ((fp == NULL)
		|| ((dsp = (utdisk_state_t *)(fp->private_data)) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad file pointer\n", __FUNCTION__);
		return (-EINVAL);
	}
	mnum = dsp->ctlmin;

	utprintf("%s: minor=%d\n", __FUNCTION__, mnum);

	/* I/O not allowed for master */
	if (mnum == UTDISK_MASTER_INSTANCE) {
		return (-ENXIO);
	}

	dsp = utstk_acquire_state(mnum, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s[%d] no state\n", __FUNCTION__, mnum);
		return (-ENXIO);
	}
	if ((biop = dsp->bio_headp) == NULL) {
		utstk_release_state(dsp);
		utprintf("%s[%d]: request queue empty\n", __FUNCTION__, mnum);
		return (-EINVAL);
	}

	rw = bio_rw(biop);
	if (rw != WRITE) {
		/* this cannot be */
		utstk_release_state(dsp);
		utprintf("%s[%d]: request queue mismatch\n", __FUNCTION__,
				mnum);
		return (-EINVAL);
	}
	utprintf("%s[%d]: bio=%p, bufp=%p, count=%zx\n", __FUNCTION__, mnum,
			biop, bufp, count);
	/*
	 * we cannot assume that I/O requests from our userland driver will
	 * align nicely with bio segments. We are prepared to dish out data
	 * upto a partial segment and continue where we left off at the next
	 * read request.
	 * We are allowed to update bi_idx to keep track of progress, but there
	 * is no mention anywhere about tinkering with bio_vec offsets.
	 * Since we are operating on a cloned bio, we freely update bv_offset
	 * to reflect our current position
	 * Note that (count <= sum_of(bvecp->bv_len)) is always true
	 */
	while (count > 0) {
		bvecp = bio_iovec_idx(biop, biop->bi_idx);
		src = page_address(bvecp->bv_page) + bvecp->bv_offset;
		if (count < bvecp->bv_len) {
			len = count;
		} else {
			len = bvecp->bv_len;
		}
		utprintf("%s[%d]: bufp=%p, src=%p, count=%zx, len=%lx, "
			"boff=%x, blen=%x\n", __FUNCTION__, mnum, bufp, src,
			count, len, bvecp->bv_offset, bvecp->bv_len);
		if (copy_to_user(bufp, src, len) != 0) {
			utprintf("%s: copy out failed\n", __FUNCTION__);
			ret = -EFAULT;
			break;
		}
		bufp += len;			/* reposition offset */
		bvecp->bv_offset += len;
		count -= len;
		bvecp->bv_len -= len;
		if (bvecp->bv_len == 0) {
			biop->bi_idx++;		/* move down to next segment */
		}
	}

	utstk_release_state(dsp);

	return (ret);
}


/* write to application buffer, data read from device */
static ssize_t
utstk_file_write(struct file *fp, const char *bufp, size_t count, loff_t *ppos)
{
	utdisk_state_t	*dsp;
	struct bio	*biop;
	struct bio_vec	*bvecp;
	minor_t		mnum;
	unsigned char	*dest;
	unsigned long	len;
	int		rw;
	int		ret = 0;

	if ((fp == NULL)
		|| ((dsp = (utdisk_state_t *)(fp->private_data)) == NULL)
		|| (utvalidate_dsp(dsp) == 0)) {
		utprintf("%s: bad file pointer\n", __FUNCTION__);
		return (-EINVAL);
	}
	mnum = dsp->ctlmin;

	utprintf("%s: minor=%d\n", __FUNCTION__, mnum);

	/* I/O not allowed for master */
	if (mnum == UTDISK_MASTER_INSTANCE) {
		return (-ENXIO);
	}

	dsp = utstk_acquire_state(mnum, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s[%d] no state\n", __FUNCTION__, mnum);
		return (-ENXIO);
	}
	if ((biop = dsp->bio_headp) == NULL) {
		utstk_release_state(dsp);
		utprintf("%s[%d]: request queue empty\n", __FUNCTION__, mnum);
		return (-EINVAL);
	}

	rw = bio_rw(biop);
	if ((rw != READ) && (rw != READA)) {
		/* this cannot be */
		utstk_release_state(dsp);
		utprintf("%s[%d]: request queue mismatch\n", __FUNCTION__,
				mnum);
		return (-EINVAL);
	}
	utprintf("%s[%d]: bio=%p, bufp=%p, count=%zx\n", __FUNCTION__, mnum,
			biop, bufp, count);
	/*
	 * note on partial segments in utstk_file_read() applies here too
	 */
	while (count > 0) {
		bvecp = bio_iovec_idx(biop, biop->bi_idx);
		dest = page_address(bvecp->bv_page) + bvecp->bv_offset;
		if (count < bvecp->bv_len) {
			len = count;
		} else {
			len = bvecp->bv_len;
		}
		utprintf("%s[%d]: bufp=%p, dest=%p, count=%zx, len=%lx, "
			"boff=%x, blen=%x\n", __FUNCTION__, mnum, bufp, dest,
			count, len, bvecp->bv_offset, bvecp->bv_len);
		if (copy_from_user(dest, bufp, len) != 0) {
			utprintf("%s: copy out failed\n", __FUNCTION__);
			ret = -EFAULT;
			break;
		}
		bufp += len;			/* reposition offset */
		bvecp->bv_offset += len;
		count -= len;
		bvecp->bv_len -= len;
		if (bvecp->bv_len == 0) {
			biop->bi_idx++;		/* move down to next segment */
		}
	}

	utstk_release_state(dsp);

	return (ret);
}


/*
 * ioctl called by 32bit user process into 64bit kernel
 * Tags all ioctls so later code knows pointers are 32-bits
 */
static long
utstk_compat_ioctl(struct file *fp, unsigned int command, unsigned long arg)
{
	int	ret;

	utprintf("%s: cmd=%x\n", __FUNCTION__, command);
	lock_kernel();			/* Big Kernel Lock */
	ret = utstk_ioctl(fp->f_dentry->d_inode, fp,
					(command | UTDISK_32BIT_PTRS), arg);
	unlock_kernel();

	return (long)(ret);
}


/* ioctl */
static int
utstk_ioctl(struct inode *inode, struct file *fp, unsigned int command,
		unsigned long arg)
{
	utdisk_state_t	*dsp;
	minor_t		mnum = iminor(inode);
	int		ret = 0;
	int		flag32ptr;

	flag32ptr = (command & UTDISK_32BIT_PTRS);
	command &= ~UTDISK_32BIT_PTRS;
	utprintf("%s: minor=%d, command=%x\n", __FUNCTION__, mnum, command);

	/* Note: most ioctls are allowed for master only */
	if ((command != UTDISK_GET_MESG) && (mnum != UTDISK_MASTER_INSTANCE)) {
		return (-ENOTTY);
	}

	dsp = utstk_acquire_state(mnum, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s: minor=%d: no state\n", __FUNCTION__, mnum);
		return (-ENXIO);
	}
	if (command != UTDISK_GET_MESG) {
		/* no need to hold onto lock for other commands */
		utstk_release_state(dsp);
	} else if (mnum == UTDISK_MASTER_INSTANCE) {
		/* cannot perform I/O on master */
		utstk_release_state(dsp);
		return (-ENOTTY);
	}

	switch(command) {
	case	UTDISK_GET_CTL_MAJOR:
		ret = utstk_get_ctl_major(arg, flag32ptr);
		break;

	case	UTDISK_GET_DISK_MAJOR:
		ret = utstk_get_disk_major(arg, flag32ptr);
		break;

	case	UTDISK_HOTPLUG:
		ret = utstk_hotplug(arg, flag32ptr);
		break;

	case	UTDISK_UNPLUG:
		ret = utstk_unplug(arg, flag32ptr);
		break;

	case	UTDISK_MEDIA_CHANGE:
		ret = utstk_media_change(arg, flag32ptr);
		break;

	case	UTDISK_GET_CDB:
		ret = utstk_get_cdb(dsp, arg, flag32ptr);
		break;

	case	UTDISK_GET_MESG:
		ret = utstk_get_mesg(dsp, arg, flag32ptr);
		utstk_release_state(dsp);
		break;

	case	UTDISK_DISK_READY:
		ret = utstk_disk_ready(arg, flag32ptr);
		break;

	default:
		ret = (-ENOTTY);
		break;
	}

	return (ret);
}


/*
 * send utdiskctl major number to userland
 * return 0 on success, negative value on error
 */
static int
utstk_get_ctl_major(unsigned long arg, int flag32ptr)
{
	major_t	cmajor;
	int	ret = 0;

	utprintf("%s\n", __FUNCTION__);
	down(&glock);
	cmajor = utdc_major;
	up(&glock);
	if (copy_to_user((void *)arg, &cmajor, sizeof (cmajor)) != 0) {
		utprintf("%s: copy out failed\n", __FUNCTION__);
		ret = -EFAULT;
	}


	return (ret);
}


/*
 * send utdisk major number to userland
 * return 0 on success, negative value on error
 */
static int
utstk_get_disk_major(unsigned long arg, int flag32ptr)
{
	major_t	dmajor;
	int	ret = 0;

	utprintf("%s\n", __FUNCTION__);
	down(&glock);
	dmajor = utdisk_major;
	up(&glock);
	if (copy_to_user((void *)arg, &dmajor, sizeof (dmajor)) != 0) {
		utprintf("%s: copy out failed\n", __FUNCTION__);
		ret = -EFAULT;
	}

	return (ret);
}


/*
 * create new state for hotplugged disk
 * send back updated diskinfo structure with new controller minor number
 * Note that we do not inform system of new disk yet because the add_disk()
 * call generates I/O and our userland driver is not yet ready. We shall
 * call add_disk() later when we're sure we're ready to handle I/O.
 *
 * return 0 on success, negative value on error
 */
static int
utstk_hotplug(unsigned long arg, int flag32ptr)
{
	utdc_diskinfo_t	dinfo;
	utdisk_state_t	*dsp;
	uint64_t	dsize;

	utprintf("%s\n", __FUNCTION__);

	/* download device parameters */
	if (copy_from_user(&dinfo, (void *)arg, sizeof (dinfo)) != 0) {
		utprintf("%s: copy in failed\n", __FUNCTION__);
		return (-EFAULT);
	}
#ifdef	UTDEBUG
	/* show device parameters */
	utprintf("%s: block_size = %x\n", __FUNCTION__, dinfo.block_size);
	utprintf("%s: nblock = %x\n", __FUNCTION__, dinfo.nblock);
	utprintf("%s: ctlmin = %d\n", __FUNCTION__, dinfo.ctlmin);
	utprintf("%s: owner = %d\n", __FUNCTION__, dinfo.owner);
	utprintf("%s: media_state = %x\n", __FUNCTION__, dinfo.media_state);
#endif	/* UTDEBUG */

	/* create state for the new device */
	dsp = utstk_new_state();
	if (dsp == NULL) {
		utprintf("%s: create state failed\n", __FUNCTION__);
		return (-ENOMEM);
	}
	/* set hardware block size */
	blk_queue_hardsect_size(dsp->rqp, dinfo.block_size);

	dinfo.ctlmin = dsp->ctlmin;
	if (dinfo.block_size != UTDISK_HARDSECT_SIZE_BYTES) {
		/* convert to UTDISK_HARDSECT_SIZE_BYTES blocks */
		dsize = (uint64_t)(dinfo.nblock) * (uint64_t)(dinfo.block_size);
		dsp->nblocks = dsize / UTDISK_HARDSECT_SIZE_BYTES;
	} else {
		dsp->nblocks = dinfo.nblock;
	}
	dsp->owner = dinfo.owner;

	if (dinfo.media_state & UT_MEDIASTATE_WRPROT) {
		dsp->flags |= UTDISK_STATE_WRPROT;
	}

	/* capacity in 512 Byte sectors */
	set_capacity(dsp->gdiskp, dsp->nblocks);

	if (copy_to_user((void *)arg, &dinfo, sizeof (dinfo)) != 0) {
		utprintf("%s: copy out failed\n", __FUNCTION__);
		return (-EFAULT);
	}

	return (0);
}


/*
 * A disk has been unplugged
 * invalidate disk state, free if unreferenced
 * return 0 on success, negative value on error
 */
static int
utstk_unplug(unsigned long arg, int flag32ptr)
{
	utdisk_state_t	*dsp;
	utdc_diskinfo_t	dinfo;

	utprintf("%s\n", __FUNCTION__);

	/* download device parameters */
	if (copy_from_user(&dinfo, (void *)arg, sizeof (dinfo)) != 0) {
		utprintf("%s: copy in failed\n", __FUNCTION__);
		return (-EFAULT);
	}
#ifdef	UTDEBUG
	/* show device parameters */
	utprintf("%s: block_size = %x\n", __FUNCTION__, dinfo.block_size);
	utprintf("%s: nblocks = %x\n", __FUNCTION__, dinfo.nblock);
	utprintf("%s: ctlmin = %d\n", __FUNCTION__, dinfo.ctlmin);
#endif	/* UTDEBUG */

	/* master cannot be unplugged */
	if (dinfo.ctlmin == UTDISK_MASTER_INSTANCE) {
		utprintf("%s: bad arg ctlmin=%d\n", __FUNCTION__, dinfo.ctlmin);
		return (-EINVAL);
	}

	dsp = utstk_acquire_state(dinfo.ctlmin, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s: no state for ctlmin=%d\n", __FUNCTION__,
								dinfo.ctlmin);
		return (-ENXIO);
	}
	dsp->flags |= UTDISK_STATE_STALE;
	if ((dsp->flags & UTDISK_STATE_OPEN) || (dsp->opens != 0)) {
		wake_up_all(&(dsp->io_wq));
	}
	utstk_release_state(dsp);
	utstk_free_state(dsp);

	return (0);
}


/*
 * update local state with changed media parameters
 * return 0 on success, negative value on error
 */
static int
utstk_media_change(unsigned long arg, int flag32ptr)
{
	utdisk_state_t	*dsp;
	utdc_diskinfo_t	dinfo;
	uint64_t	dsize;

	utprintf("%s\n", __FUNCTION__);

	/* download device parameters */
	if (copy_from_user(&dinfo, (void *)arg, sizeof (dinfo)) != 0) {
		utprintf("%s: copy in failed\n", __FUNCTION__);
		return (-EFAULT);
	}
#ifdef	UTDEBUG
	/* show device parameters */
	utprintf("%s: block_size = %x\n", __FUNCTION__, dinfo.block_size);
	utprintf("%s: nblocks = %x\n", __FUNCTION__, dinfo.nblock);
	utprintf("%s: ctlmin = %d\n", __FUNCTION__, dinfo.ctlmin);
	utprintf("%s: media_state = %x\n", __FUNCTION__, dinfo.media_state);
#endif	/* UTDEBUG */

	/* no I/O on master */
	if (dinfo.ctlmin == UTDISK_MASTER_INSTANCE) {
		utprintf("%s: bad arg ctlmin=%d\n", __FUNCTION__, dinfo.ctlmin);
		return (-EINVAL);
	}

	dsp = utstk_acquire_state(dinfo.ctlmin, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s: no state for ctlmin=%d\n", __FUNCTION__,
								dinfo.ctlmin);
		return (-ENXIO);
	}
	if (dinfo.block_size != UTDISK_HARDSECT_SIZE_BYTES) {
		/* convert to UTDISK_HARDSECT_SIZE_BYTES blocks */
		dsize = (uint64_t)(dinfo.nblock) * (uint64_t)(dinfo.block_size);
		dsp->nblocks = dsize / UTDISK_HARDSECT_SIZE_BYTES;
	} else {
		dsp->nblocks = dinfo.nblock;
	}
	dsp->flags |= UTDISK_STATE_CHANGED;
	if (dinfo.media_state & UT_MEDIASTATE_WRPROT) {
		dsp->flags |= UTDISK_STATE_WRPROT;
	} else {
		dsp->flags &= ~UTDISK_STATE_WRPROT;
	}

	utstk_release_state(dsp);

	return (0);
}


/* service is ready to process I/O requests, add disk to system list */
static int
utstk_disk_ready(unsigned long arg, int flag32ptr)
{
	utdisk_state_t	*dsp;
	minor_t		ctlmin;

	utprintf("%s\n", __FUNCTION__);

	/* download device parameters */
	if (copy_from_user(&ctlmin, (void *)arg, sizeof (ctlmin)) != 0) {
		utprintf("%s: copy in failed\n", __FUNCTION__);
		return (-EFAULT);
	}

	/* no I/O on master */
	if (ctlmin == UTDISK_MASTER_INSTANCE) {
		utprintf("%s: bad arg ctlmin=%d\n", __FUNCTION__, ctlmin);
		return (-EINVAL);
	}

	dsp = utstk_acquire_state(ctlmin, UTDISK_STATE_VALIDATE);
	if (dsp == NULL) {
		utprintf("%s: no state for ctlmin=%d\n", __FUNCTION__, ctlmin);
		return (-ENXIO);
	}
	if ((dsp->flags & UTDISK_STATE_READY) == 0) {
		dsp->flags |= UTDISK_STATE_READY;
		utstk_release_state(dsp);
		utprintf("%s calling add_disk: %p ctlmin = %x\n", __FUNCTION__, dsp->gdiskp, ctlmin);
		add_disk(dsp->gdiskp);
	} else {
		utstk_release_state(dsp);
	}
	utprintf("%s dsp = %p ctlmin = %x\n", __FUNCTION__, dsp, ctlmin);

	return (0);
}

/*
 * returns uscsi_cdb for pending ioctl buffer
 * This ioctl is issued for vendor specific cdb lengths > pre-allocated cdblens
 */
static int
utstk_get_cdb(utdisk_state_t *dsp, unsigned long arg, int flag32ptr)
{
	struct bio 		*biop;
	utdc_bio_descriptor_t	*desc;
	ut_uscsi_cmd_t		*scmd;
	utdc_ioctl_t		*iocp;
	int			ret = 0;

	utprintf("%s: dspbio=%p\n", __FUNCTION__, dsp->bio_headp);
	if ((biop = dsp->bio_headp) == NULL) {
		/* no bio to work with, good */
		return (0);
	}
	desc = biop->bi_private;
	utprintf("%s: desc=%p\n", __FUNCTION__, desc);
	iocp = &(desc->ioctl_req);
	if (iocp == NULL) {
		utprintf("%s[%d]: bad iocp\n", __FUNCTION__, dsp->ctlmin);
		ret = -EINVAL;
	}
	if ((ret == 0) && (arg) && (desc->type == UTSTK_TYPE_IOCTL) &&
		(scmd = iocp->arg)) {
		/*
		 * Caller should have alloc'd buf size == cdblen
		 * It should not come here with less or memory gets corrupted
		 */
		if (copy_to_user((void *)arg,
					scmd->uscsi_cdb,
					scmd->uscsi_cdblen) != 0) {
			utprintf("%s[%d]: copy out failed iocp\n", __FUNCTION__, dsp->ctlmin);

			ret = -EFAULT;
		}
	} else {
		ret = -EINVAL;
	}

	return (ret);
}

/*
 * send up data transfer mesg
 * sleep if there are no pending messages
 * Note that returning error causes service thread in userland to terminate
 * Note also that dsp->lock is held at this point
 * return 0 on success, negative value on error
 */
static int
utstk_get_mesg(utdisk_state_t *dsp, unsigned long arg, int flag32ptr)
{
	utdc_tx_t		tx;
	minor_t			ctlmin;
	wait_queue_head_t	*wqheadp;
	struct bio		*biop;
	utdc_bio_descriptor_t	*desc;
	utdc_ioctl_t		*iocp;
	ut_uscsi_cmd_t		*scmd;
	int			rw;
	int			ret;
	char			*copy_ptr;
	int			copy_len;
#ifdef	_LP64
	utdc_tx32_t		tx32;
	ut_uscsi_cmd32_t	uscsi32;
#endif	/* _LP64 */
	DEFINE_WAIT(iowait);

	ctlmin = dsp->ctlmin;			/* to reacquire lock later */

	/* download transfer parameters */
#ifdef	_LP64
	if (flag32ptr) {
		copy_ptr = (char *)&tx32;
		copy_len = sizeof (tx32);
	} else {
		copy_ptr = (char *)&tx;
		copy_len = sizeof (tx);
	}
#else	/* pointer sizes are same */
	copy_ptr = (char *)&tx;
	copy_len = sizeof (tx);
#endif	/* _LP64 */

	if (copy_from_user(copy_ptr, (void *)arg, copy_len) != 0) {
		utprintf("%s: copy in failed\n", __FUNCTION__);
		return (-EFAULT);
	}
#ifdef	_LP64
	if (flag32ptr) {
		/* transfer into 64bit structure */
		tx.offset = tx32.offset;
		tx.nblock = tx32.nblock;
	    	tx.cdb = (char *)(uint64_t)(tx32.cdb);
	    	tx.uscsi = (char *)(uint64_t)(tx32.uscsi);
		tx.cdblen = tx32.cdblen;
		tx.type = tx32.type;
		tx.cmd = tx32.cmd;
		tx.resid = tx32.resid;
		tx.ret = tx32.ret;
	}
#endif	/* _LP64 */

#ifdef	UTDEBUG
	/* show transfer results */
	utprintf("%s[%d]: type = %x\n", __FUNCTION__, ctlmin, tx.type);
	utprintf("%s[%d]: offset = %x\n", __FUNCTION__, ctlmin, tx.offset);
	utprintf("%s[%d]: nblocks = %x\n", __FUNCTION__, ctlmin, tx.nblock);
	utprintf("%s[%d]: cmd = %x\n", __FUNCTION__, ctlmin, tx.cmd);
	utprintf("%s[%d]: resid = %x\n", __FUNCTION__, ctlmin, tx.resid);
	utprintf("%s[%d]: cdb = %p\n", __FUNCTION__, ctlmin, tx.cdb);
	utprintf("%s[%d]: cdblen = %x\n", __FUNCTION__, ctlmin, tx.cdblen);
	utprintf("%s[%d]: uscsi = %p\n", __FUNCTION__, ctlmin, tx.uscsi);
	utprintf("%s[%d]: ret = %x\n", __FUNCTION__, ctlmin, tx.ret);
#endif	/* UTDEBUG */

	ret = utstk_tx_response(dsp, &tx, flag32ptr);
	if (ret != 0) {
		return (ret);			/* error */
	}

	tx.cmd = 0;				/* reset */
	wqheadp = &(dsp->io_wq);		/* we may lose dsp midway */

	while (dsp != NULL) {
		if ((biop = dsp->bio_headp) == NULL) {
			/* no new transfer requests, sleep */
			utprintf("%s: bio queue empty\n", __FUNCTION__);
			prepare_to_wait(wqheadp, &iowait, TASK_INTERRUPTIBLE);
			utstk_release_state(dsp);
			io_schedule();
			finish_wait(wqheadp, &iowait);
			dsp = utstk_acquire_state(ctlmin,
							UTDISK_STATE_VALIDATE);
			if (signal_pending(current)) {
				utprintf("%s: interrupted\n", __FUNCTION__);
				return (-EINTR);
			}
		} else {
			/* check whether this is an interrupted ioctl */
			desc = biop->bi_private;
			if (desc == NULL) {
				return (0);
			}
			iocp = &(desc->ioctl_req);
			if (desc->type != UTSTK_TYPE_DATA) {
				if (iocp == NULL) {
					return (0);
				}
				if (iocp->flags & UTDISK_IOCTL_INTR) {
					/* this bio is stale, move on */
					dsp->bio_headp = biop->bi_next;
					if (dsp->bio_headp == NULL) {
						/* that was the last one */
						dsp->bio_tailp = NULL;
					}
					/* release our clone */
					biop->bi_private = NULL;
					bio_put(biop);

					/* release stuff created by ioctl */
					biop = desc->bio_orig;
					if (biop != NULL) {
						bio_put(biop);
					}
					vfree(desc);
					/* go back to waiting */
					continue;
				} else {
					break;
				}
			} else {
				break;
			}
		}
	}
	if (dsp == NULL) {
		/* most likely device unplugged */
		utprintf("%s: device [%d] unplugged\n", __FUNCTION__, ctlmin);
		return (-ENXIO);
	}
	biop = desc->bio_orig;
	if (biop == NULL) {
		utprintf("%s[%d]: bad struct bio\n", __FUNCTION__, ctlmin);
		return (0);
	}
	/*
	 * Determine whether this bio corresponds to data
	 * request or an ioctl
	 */
	if (desc->type == UTSTK_TYPE_DATA) {
		rw = bio_rw(biop);	/* bio_data_dir() masks out READA */
		if ((rw == READ) || (rw == READA)) {
			tx.type = UTDC_TX_READ;
		} else {
			tx.type = UTDC_TX_WRITE;
		}
		tx.nblock = biop->bi_size / UTDISK_HARDSECT_SIZE_BYTES;
		tx.offset = biop->bi_sector;
		if ((tx.offset + tx.nblock) >= get_capacity(dsp->gdiskp)) {
		/* adjust overflow, residue shall be set when tx complete */
			tx.nblock = get_capacity(dsp->gdiskp) - tx.offset;
		}
	} else {
		if (!(iocp->flags & UTDISK_IOCTL_NEW)) {
			utprintf("%s[%d]: old ioctl io=%p f=%x\n",
				__FUNCTION__, ctlmin, iocp, iocp->flags);
			return (-EIO);
		}
		/*
		 * dsp->bio_headp is a cloned bio which we own.
		 * Its bi_private field always has a utdc_bio_descriptor_t
		 * (desc). desc->bio_orig belongs to us only if
		 * desc->type != UTSTK_TYPE_DATA. In this case
		 * desc->ioctl_req is an utdc_ioctl_t.
		 * If desc_type == UTSTK_TYPE_DATA then utdc_ioctl_t
		 * is in the utdc_bio_descriptor_t data structure but is not
		 * used.
		 */

		tx.cmd = iocp->cmd;
		if (desc->type == UTSTK_TYPE_IOCTL) {
			utprintf("%s[%d]: found ioctl request\n",
				__FUNCTION__, ctlmin);
			tx.type = UTDC_TX_IOCTL;
		} else if (desc->type == UTSTK_TYPE_USCSI) {
			utprintf("%s[%d]: found uscsi command\n",
			__FUNCTION__, ctlmin);

			tx.type = UTDC_TX_USCSI;
			scmd = iocp->arg;
			/*
		 	 * Is pre-allocated cdblen sufficient?
			 * lun_thread can determine it by inspecting
			 * uscsi_cmd->cdblen, if insufficient, it can alloc a
			 * new CDB and issue UTDISK_GET_CDB
			 */
			if ((tx.cdblen >= scmd->uscsi_cdblen) && (tx.cdb)) {
				utprintf("%s[%d]: uscsi CDB: dcdb=%p scdb=%p len=%d\n",
					__FUNCTION__, ctlmin, tx.cdb,
					scmd->uscsi_cdb, scmd->uscsi_cdblen);
				ret = copy_to_user((void *)tx.cdb,
							scmd->uscsi_cdb,
							scmd->uscsi_cdblen);
			}
			/* send back scsi cmd */
			if ((ret == 0) && (tx.uscsi)) {
#ifdef	_LP64
				if (flag32ptr) {
					/* transfer into 32bit structure */
					uscsi32.uscsi_flags = scmd->uscsi_flags;
					uscsi32.uscsi_status
						= scmd->uscsi_status;
					uscsi32.uscsi_timeout
						= scmd->uscsi_timeout;
					/*
					 * Set ptr fields to zero to avoid
					 * compiler warnings
					 * These fields are reset by utstoraged
					 */
					uscsi32.uscsi_cdb = 0;
					uscsi32.uscsi_bufaddr = 0;
					uscsi32.uscsi_buflen
						= scmd->uscsi_buflen;
					uscsi32.uscsi_resid = scmd->uscsi_resid;
					uscsi32.uscsi_cdblen
						= scmd->uscsi_cdblen;
					uscsi32.uscsi_rqlen = scmd->uscsi_rqlen;
					uscsi32.uscsi_rqstatus
						= scmd->uscsi_rqstatus;
					uscsi32.uscsi_rqresid
						= scmd->uscsi_rqresid;
					uscsi32.uscsi_rqbuf = 0;
					uscsi32.uscsi_reserved_5 = 0;
					copy_ptr = (char *)&uscsi32;
					copy_len = sizeof (uscsi32);
				} else {
					copy_ptr = (char *)scmd;
					copy_len = sizeof (ut_uscsi_cmd_t);
				}
#else	/* pointer sizes are same */
				copy_ptr = (char *)scmd;
				copy_len = sizeof (ut_uscsi_cmd_t);
#endif	/* _LP64 */
				ret = copy_to_user((void *)tx.uscsi,
							copy_ptr, copy_len);

			}
		} else {
			utprintf("%s: unexpected type: %x\n",
					__FUNCTION__, desc->type);
			return (-ENXIO);
		}
	}
#ifdef	UTDEBUG
	/* show transfer request parameters */
	utprintf("%s[%d]: bio = %p\n", __FUNCTION__, ctlmin, biop->bi_private);
	utprintf("%s[%d]: tx.type = %x\n", __FUNCTION__, ctlmin, tx.type);
	utprintf("%s[%d]: tx.offset = %x\n", __FUNCTION__, ctlmin, tx.offset);
	utprintf("%s[%d]: tx.nblocks = %x\n", __FUNCTION__, ctlmin, tx.nblock);
	utprintf("%s[%d]: tx.cmd = %x\n", __FUNCTION__, ctlmin, tx.cmd);
	utprintf("%s[%d]: tx.resid = %x\n", __FUNCTION__, ctlmin, tx.resid);
	utprintf("%s[%d]: tx.cdb = %p\n", __FUNCTION__, ctlmin, tx.cdb);
	utprintf("%s[%d]: tx.cdblen = %x\n", __FUNCTION__, ctlmin, tx.cdblen);
	utprintf("%s[%d]: tx.uscsi = %p\n", __FUNCTION__, ctlmin, tx.uscsi);
	utprintf("%s[%d]: tx.ret = %x\n", __FUNCTION__, ctlmin, tx.ret);
#endif	/* UTDEBUG */

	/* upload transfer parameters */

#ifdef	_LP64
	if (flag32ptr) {
		copy_ptr = (char *)&tx32;
		copy_len = sizeof (tx32);
		/* transfer into 32bit structure */
		tx32.offset = tx.offset;
		tx32.nblock = tx.nblock;
	    	tx32.cdb = (uint32_t)((uint64_t)(tx.cdb));
	    	tx32.uscsi = (uint32_t)((uint64_t)(tx.uscsi));
		tx32.cdblen = tx.cdblen;
		tx32.type = tx.type;
		tx32.cmd = tx.cmd;
		tx32.resid = tx.resid;
		tx32.ret = tx.ret;
	} else {
		copy_ptr = (char *)&tx;
		copy_len = sizeof (tx);
	}
#else	/* pointer sizes are same */
	copy_ptr = (char *)&tx;
	copy_len = sizeof (tx);
#endif	/* _LP64 */

	if (ret == 0) {
		ret = copy_to_user((void *)arg, copy_ptr, copy_len);
	}
	if (ret !=  0) {
		utprintf("%s: copy outfailed\n", __FUNCTION__);
		return (-EFAULT);
	} else {
		if ((tx.type == UTDC_TX_IOCTL) || (tx.type == UTDC_TX_USCSI)) {
			iocp->flags = UTDISK_IOCTL_SENT;
		}
	}

	return (0);
}


/*
 * data transfer complete, set residue and flags and release pending bio
 * If an error is detected, it means something corrupted our data structures
 * all we do then is stop the user thread servicing the current device
 * Note that dsp->lock is held at this point
 * return 0 on success, or error value
 */
static int
utstk_tx_response(utdisk_state_t *dsp, utdc_tx_t *tx, int flag32ptr)
{
	struct bio		*biop;
	struct bio		*real_biop;
	utdc_bio_descriptor_t	*desc;
	wait_queue_head_t	*wqp;
	ut_uscsi_cmd_t		*scmd;
	utdc_ioctl_t		*iocp;
	void			*rqbuf;
	int			rqlen;
	int			rw;
	int			type;
	int			ret;
	char			*copy_ptr;
	int			copy_len;
#ifdef	_LP64
	ut_uscsi_cmd32_t	uscsi32;
#endif	/* _LP64 */

	utprintf("%s: dspbio=%p\n", __FUNCTION__, dsp->bio_headp);
	if ((biop = dsp->bio_headp) == NULL) {
		/* no bio to work with, good */
		return (0);
	}
	if (tx->cmd == UTDC_TX_NONE) {
		/* no response, means no bios waiting for response */
		return (0);
	}
	desc = biop->bi_private;
	utprintf("%s: desc=%p\n", __FUNCTION__, desc);
	wqp = &(desc->ioctl_wq);
	type = desc->type;
	if ((real_biop = desc->bio_orig) == NULL) {
		utprintf("%s[%d]: bad bio\n", __FUNCTION__, dsp->ctlmin);
		return (-EINVAL);
	}

	ret = 0;
	iocp = NULL;
	if (type == UTSTK_TYPE_DATA) {
		/* validate bio with tx */
		rw = bio_rw(biop);
		if (rw == WRITE) {
			if (tx->cmd != UTDC_TX_WRITE) {
				utprintf("%s[%d]: bio sequence error\n", __FUNCTION__,
								dsp->ctlmin);
				ret = -EINVAL;
			}
		} else if (tx->cmd != UTDC_TX_READ) {
			utprintf("%s[%d]: bio sequence error\n", __FUNCTION__,
								dsp->ctlmin);
			ret = -EINVAL;
		}
	} else {
		iocp = &(desc->ioctl_req);
		if (iocp == NULL) {
			utprintf("%s[%d]: NULL descriptor\n",
					__FUNCTION__, dsp->ctlmin);
			return (-EINVAL);
		}
		if (iocp->flags & UTDISK_IOCTL_INTR) {
			/* interrupted ioctl, send back for removal */
			return (0);
		}
		if (type == UTSTK_TYPE_IOCTL) {
			if (tx->cmd == iocp->cmd) {
				utprintf("%s[%d]: cmd=%d\n",
					__FUNCTION__, dsp->ctlmin, iocp->cmd);
			} else {
				utprintf("%s[%d]: sequence error tx=%d, io=%d\n",
						__FUNCTION__, dsp->ctlmin,
						tx->cmd, iocp->cmd);
				ret = -EINVAL;
			}
		} else if (type == UTSTK_TYPE_USCSI) {
			scmd = iocp->arg;
#ifdef	_LP64
			if (flag32ptr) {
				copy_ptr = (char *)&uscsi32;
				copy_len = sizeof (uscsi32);
			} else {
				copy_ptr = (char *)scmd;
				copy_len = sizeof (ut_uscsi_cmd_t);
			}
#else	/* pointer sizes are same */
			copy_ptr = (char *)scmd;
			copy_len = sizeof (ut_uscsi_cmd_t);
#endif	/* _LP64 */
			if (copy_from_user(copy_ptr, (void *)tx->uscsi,
							copy_len) != 0) {

				utprintf("%s: copy in failed for uscsi cmd\n",
							__FUNCTION__);
				ret = -EFAULT;
			}
#ifdef	_LP64
			if (flag32ptr) {
				/* transfer into 64bit structure */
				scmd->uscsi_flags = uscsi32.uscsi_flags;
				scmd->uscsi_status = uscsi32.uscsi_status;
				scmd->uscsi_timeout = uscsi32.uscsi_timeout;
				scmd->uscsi_cdb = (char *)(uint64_t)(uscsi32.uscsi_cdb);
				scmd->uscsi_bufaddr
					= (char *)(uint64_t)(uscsi32.uscsi_bufaddr);
				scmd->uscsi_buflen = uscsi32.uscsi_buflen;
				scmd->uscsi_resid = uscsi32.uscsi_resid;
				scmd->uscsi_cdblen = uscsi32.uscsi_cdblen;
				scmd->uscsi_rqlen = uscsi32.uscsi_rqlen;
				scmd->uscsi_rqstatus = uscsi32.uscsi_rqstatus;
				scmd->uscsi_rqresid = uscsi32.uscsi_rqresid;
				scmd->uscsi_rqbuf
					= (char *)(uint64_t)(uscsi32.uscsi_rqbuf);
				scmd->uscsi_reserved_5
					= (void *)(uint64_t)(uscsi32.uscsi_reserved_5);
			}
#endif	/* _LP64 */

			/*
			 * sense data
			 * scmd->uscsi_rqbuf was NULL when scmd was
			 * passed  up to the daemon.
			 * if UT_USCSI_RQENABLE was set and uscsi_rqlen > 0
			 * the daemon would have malloc'ed an rqbuf.
			 * So now, scmd->uscsi_rqbuf could be != NULL
			 * (uscsi_rqlen - uscsi->rqresid) will tell us
			 * whether any valid rq data exists. If it
			 * does, malloc new buf to receive sense data
			 */
			rqbuf = scmd->uscsi_rqbuf;
			rqlen = scmd->uscsi_rqlen - scmd->uscsi_rqresid;
			if ((rqbuf) && (rqlen)) {
				scmd->uscsi_rqbuf = vmalloc(rqlen);
				if (scmd->uscsi_rqbuf == NULL) {
					utprintf("%s: rqbuf alloc\n",
						__FUNCTION__);
					ret = -ENOMEM;
				} else  {
					if (copy_from_user(scmd->uscsi_rqbuf, (void *)rqbuf, rqlen) != 0) {
						utprintf("%s: copy in failed for rqbuf\n", __FUNCTION__);
						ret = -EFAULT;
					}
				}
			}
		} else {
			utprintf("%s[%d]: wrong type: %x\n",
			__FUNCTION__, dsp->ctlmin, type);
			ret = -EINVAL;
		}
	}

#ifdef	UTDEBUG
	utprintf("%s: bio=%p, dspbio=%p, head=%p, tail=%p\n", __FUNCTION__,
		real_biop, biop, dsp->bio_headp, dsp->bio_tailp);
	utprintf("%s: bsize=%x, resid=%x\n", __FUNCTION__, real_biop->bi_size,
								tx->resid);
#endif	/* UTDEBUG */

	/* we are now done with this bio */
	dsp->bio_headp = biop->bi_next;
	biop->bi_private = NULL;
	bio_put(biop);				/* release our clone */

	if (dsp->bio_headp == NULL) {
		utprintf("%s[%d]: tail=%p\n", __FUNCTION__,
					dsp->ctlmin, dsp->bio_tailp);
		dsp->bio_tailp = NULL;		/* that was the last one */
	}

	if (type == UTSTK_TYPE_DATA) {
		vfree(desc);		/* we hold the last reference to desc */
		/* set result of transaction */
		if (tx->ret == UTDC_TX_ERROR) {
			utprintf("%s[%d]: IO error on bio=%p, size=%d, resid=%d\n",
				__FUNCTION__, dsp->ctlmin, real_biop,
				real_biop->bi_size, tx->resid);
			bio_endio(real_biop, real_biop->bi_size, (-EIO));
		} else {
			/* success */
			bio_endio(real_biop, (real_biop->bi_size - tx->resid), 0);
		}
	} else {
		/*
		 * don't free desc
		 * the thread waiting for ioctl to complete will free it
		 */
		if (iocp == NULL) {
			utprintf("%s[%d]: iocp unexpectedly NULL\n", __FUNCTION__, dsp->ctlmin);
			return (ret);
		}
		iocp->flags = UTDISK_IOCTL_DONE;
		utprintf("%s[%d]: wake up do_ioctl\n", __FUNCTION__, dsp->ctlmin);
		wake_up_interruptible_sync(wqp);
	}

	return (ret);
}


/*
 * malloc & setup disk state structure
 * return state ptr if successful, NULL on error
 */
static utdisk_state_t *
utstk_new_state(void)
{
	utdisk_state_t	*dsp;
	struct gendisk	*gdp;
	dev_t		dev;

	/* utdisk parameters should be ready */
	if ((utdisk_major == 0) || (utdisk_make_request_p == NULL)
		|| (utdisk_bd_ops == NULL)) {
		utprintf("%s: utdisk not ready\n", __FUNCTION__);
		return (NULL);
	}
	dsp = vmalloc(sizeof (utdisk_state_t));
	if (dsp == NULL) {
		utprintf("%s: insufficient memory\n", __FUNCTION__);
		return (NULL);
	}
	memset(dsp, 0, (sizeof (utdisk_state_t)));
	dsp->ctlmin = utstk_minor_get_num(&mlist);
	if (dsp->ctlmin == 0) {
		utstk_free_state(dsp);
		return (NULL);
	}
	sema_init(&(dsp->lock), 1);		/* one resource */
	/* setup request queue */
	dsp->rqp = blk_alloc_queue(GFP_KERNEL);
	if (dsp->rqp == NULL) {
		utprintf("%s: blk_alloc_queue failed\n", __FUNCTION__);
		utstk_free_state(dsp);
		return (NULL);
	}

	blk_queue_make_request(dsp->rqp, utdisk_make_request_p);

	/* setup gendisk, but don't add to kernel's gendisk list */
	dsp->gdiskp = alloc_disk(UTDISK_MINORS_PER_DISK);
	if (dsp->gdiskp == NULL) {
		utprintf("%s: alloc_disk failed\n", __FUNCTION__);
		utstk_free_state(dsp);
		return (NULL);
	}
	gdp = dsp->gdiskp;
	gdp->major = utdisk_major;
	/*
	 * partition minor numbers start with ctlmin & go on upto
	 * (ctlmin + UTDISK_MINORS_PER_DISK - 1)
	 */
	gdp->first_minor = ((dsp->ctlmin - 1) * UTDISK_MINORS_PER_DISK) + 1;
	gdp->fops = utdisk_bd_ops;
	gdp->queue = dsp->rqp;
	/* alloc_disk returns zero'd memory, no garbage when OR'ing */
	gdp->flags |= GENHD_FL_REMOVABLE;
	snprintf(gdp->disk_name, sizeof (gdp->disk_name), "%s%d",
				UTDISK_DISK_DRV_NAME, dsp->ctlmin);
	/* save pointer to local descriptor */
	gdp->private_data = dsp;
	utprintf("%s: gdp = %p dsp = %p first_minor = %x\n", __FUNCTION__,
			gdp, dsp, gdp->first_minor);

	/* setup cdev for controller */
	dsp->cdevp = cdev_alloc();
	if (dsp->cdevp == NULL) {
		utprintf("%s: cdev_alloc failed\n", __FUNCTION__);
		utstk_free_state(dsp);
		return (NULL);
	}
	dsp->cdevp->owner = THIS_MODULE;
	dsp->cdevp->ops = &utdc_fops;
	dev = MKDEV(utdc_major, dsp->ctlmin);
	if (cdev_add(dsp->cdevp, dev, 1) != 0) {
		utprintf("%s: cdev_add failed\n", __FUNCTION__);
		kobject_put(&(dsp->cdevp->kobj));
		dsp->cdevp = NULL;
		utstk_free_state(dsp);
		return (NULL);
	}

	/* setup wait queue */
	init_waitqueue_head(&(dsp->io_wq));

#ifdef	UTDEBUG
	dsp->selfp = dsp;
#endif	/* UTDEBUG */

	if (utstk_set_state(dsp->ctlmin, dsp) != 0) {
		utprintf("%s: set state failed\n", __FUNCTION__);
		utstk_free_state(dsp);
		return (NULL);
	}

	return (dsp);
}


/*
 * set state pointer in state store
 * return 0 on success, 1 on error
 */
static int
utstk_set_state(minor_t mnum, utdisk_state_t *dsp)
{
	int	ret;

	down(&(glock));
	if ((mnum == UTDISK_MASTER_INSTANCE) || (mnum >= mlist.next_num)) {
		utprintf("%s: unused minor number [%d]\n", __FUNCTION__, mnum);
		ret = 1;
	} else if ((dsp != NULL) && (mlist.state[mnum] != NULL)) {
		utprintf("%s: minor number busy [%d]\n", __FUNCTION__, mnum);
		ret = 1;
	} else {
		mlist.state[mnum] = dsp;
		if (dsp != NULL) {
			/* brand new state ready for use */
			dsp->flags |= UTDISK_STATE_IN_USE;
		}
		ret = 0;
	}
	up(&glock);

	return (ret);
}


/*
 * get state lock
 * if validate requested, check whether state is stale
 * return state pointer if successful
 * return NULL otherwise
 */
static utdisk_state_t *
utstk_acquire_state(minor_t num, uint_t	mode)
{
	utdisk_state_t	*dsp;

	if (down_interruptible(&(glock)) != 0) {
		utprintf("%s: locking interrupted [%d]\n", __FUNCTION__, num);
		return (NULL);
	}
	if (num == UTDISK_MASTER_INSTANCE) {
		dsp = &mds;
	} else {
		if (num >= mlist.next_num) {
			utprintf("%s: unused minor number [%d]\n",
							__FUNCTION__, num);
			dsp = NULL;
		} else {
			dsp = mlist.state[num];
		}
	}

	if (dsp != NULL) {
		if (utvalidate_dsp(dsp) == 0) {
			utprintf("%s: corrupted state [%d]\n",
							__FUNCTION__, num);
			dsp = NULL;
		} else if ((mode == UTDISK_STATE_VALIDATE)
			&& (dsp->flags & UTDISK_STATE_STALE)) {
			utprintf("%s: state stale [%d]\n", __FUNCTION__, num);
			dsp = NULL;
		} else if (down_interruptible(&(dsp->lock)) != 0) {
			utprintf("%s: locking interrupted [%d]\n",
							__FUNCTION__, num);
			dsp = NULL;
		}
	}
	up(&glock);

	return (dsp);
}


/* release state lock */
static void
utstk_release_state(utdisk_state_t *dsp)
{
	utprintf("%s: release state [%p]\n", __FUNCTION__, dsp);
	if (dsp != NULL) {
		up(&(dsp->lock));
	}
}


/***************************************************************
 * minor number & soft state space management:
 *
 * alloc chunks at a time. Do not free anything because if we
 * have consumed some number of minor numbers, it means it is very likely that
 * that many devices can be plugged in again. These are 2 arrays of ints, so
 * no danger of hogging memory. These arrays' resources are the least of our
 * problems when devices start reaching the system limit.
 * Maintain a free list so minor numbers can be reused.
 * The state space store can be indexed directly by controller minor number
 * The free list is a ring fifo.
 * Minor numbers considered active may not really be active. There may be some
 * whose devices are unplugged, but still have active drivers
 * (both utdisk & utdiskctl) who are not yet aware of the unplug event
 * Only the master gets notified of unplug. The minor threads know it when
 * they try to access their state.
 *
 ***************************************************************/

/*
 * init minor list head
 * Returns 0 on success, 1 on error
 */
static int
utstk_minor_list_init(utdc_mlist_t *mp)
{
	uint_t	alloc_size = UTDISK_ALLOC_CHUNK;

	/*
	 * minor number == 0 implies list is empty. 0 is used only for
	 * the master controller which has it's own seperate state
	 */
	mp->free_head = 0;
	mp->free_tail = 0;
	mp->next_num = 1;

	/* alloc soft state store */
	mp->state = vmalloc(sizeof (utdisk_state_t *) * alloc_size);
	mp->freenums = vmalloc(sizeof (minor_t) * alloc_size);
	if ((mp->state == NULL) || (mp->freenums == NULL)) {
		/* alloc failed, clean up & return error */
		if (mp->state != NULL) {
			vfree(mp->state);
		}
		if (mp->freenums != NULL) {
			vfree(mp->freenums);
		}
		mp->size = 0;
		mp->freenums = NULL;
		mp->state = NULL;
		utprintf("%s: malloc failed\n", __FUNCTION__);
		return (1);
	}
	/* else, allocs succeeded */
	mp->size = alloc_size;
	memset(mp->state, 0, (sizeof (utdisk_state_t *) * alloc_size));
	memset(mp->freenums, 0, (sizeof (minor_t) * alloc_size));

	return (0);
}


/*
 * release resources from minor list head
 */
static void
utstk_minor_list_fini(utdc_mlist_t *mp)
{
	mp->next_num = 0;
	if (mp->state != NULL) {
		vfree(mp->state);
		mp->state = NULL;
	}
	if (mp->freenums != NULL) {
		vfree(mp->freenums);
		mp->freenums = NULL;
	}
	mp->size = 0;
	mp->free_head = 0;
	mp->free_tail = 0;
}


/*
 * get next available minor number from given store
 * if there are free numbers, extract & return freelist head
 * if there are no free numbers return next_num & increment it for next use
 * if we run out of numbers alloc more
 * return 0 for error, minor number on success
 * no danger of zero being reused because minor number 0 is master controller
 */
static minor_t
utstk_minor_get_num(utdc_mlist_t *mlist)
{
	utdisk_state_t	**store;	/* new state store */
	minor_t		*flist;		/* new freelist */
	minor_t		num = 0;
	uint_t		alloc_size;

	down(&glock);
	num = utstk_minor_get_free_head(mlist);
	if (num == 0) {
		/* freelist empty, get one from store */
		num = mlist->next_num;
		if (num >= mlist->size) {
			/* store empty, try to extend store size */
			alloc_size = mlist->size + UTDISK_ALLOC_CHUNK;
			store = vmalloc(sizeof (utdisk_state_t *) * alloc_size);
			flist = vmalloc(sizeof (minor_t) * alloc_size);
			if ((store == NULL) || (flist == NULL)) {
				/* alloc failed, clean up & return error */
				if (store != NULL) {
					vfree(store);
				}
				if (flist != NULL) {
					vfree(flist);
				}
				num = 0;	/* return error */
			} else {
				memset(store, 0, (sizeof (utdisk_state_t *)
								* alloc_size));
				memset(flist, 0,
					(sizeof (minor_t) * alloc_size));
				/*
				 * move over to new store, expensive operation
				 * hopefully rare if UTDISK_ALLOC_CHUNK is big
				 */
				memcpy(store, mlist->state,
					(sizeof (utdisk_state_t *)
								* mlist->size));
				/*
				 * no need to copy empty freelist
				 * simply reset pointers
				 */
				mlist->free_head = 0;
				mlist->free_tail = 0;
				vfree(mlist->state);
				vfree(mlist->freenums);
				mlist->size = alloc_size;
				mlist->state = store;
				mlist->freenums = flist;
			}
		}
		/* inc next_num, do not roll over if we ever reach limit */
		if ((num) && (mlist->next_num < (UTDISK_MAXMINOR - 1))) {
			mlist->next_num++;
		}
	}
	if ((num >= UTDISK_MAXMINOR) || (mlist->state[num] != NULL)) {
		/* we reach this condition only if we run out of minor nums */
		num = 0;
	}
	utprintf("%s: n=%x, h=%x, t=%x, x=%x, s=%x\n", __FUNCTION__, num,
		mlist->free_head, mlist->free_tail,
		mlist->next_num, mlist->size);

	up(&glock);

	return (num);
}


/*
 * minor number is no longer in use, add to tail of free list
 * free_head == free_tail usually implies list empty
 * free_head == free_tail && [free_head] != 0 implies list full
 * But we can never have more free numbers than list size, overflow never occurs
 *
 * returns 0 on success, 1 on error
 */
static int
utstk_minor_free_num(utdc_mlist_t *mlist, minor_t num)
{
	int	newtail;

	down(&glock);
	if ((num == 0) || (num >= mlist->size)) {
		/* master's minor num never gets into common free list */
		if (num) {
			utprintf("%s: unused minor %d\n", __FUNCTION__, (int)num);
		}
		up(&glock);
		return (1);
	}

	newtail = mlist->free_tail + 1;
	if (newtail >= mlist->size) {
		newtail = 0;
	}
	utprintf("%s: mp=%p, n=%x, h=%x, t=%x, x=%x, s=%x, nt=%x\n",
		__FUNCTION__, mlist, num, mlist->free_head, mlist->free_tail,
		mlist->next_num, mlist->size, newtail);

	mlist->freenums[mlist->free_tail] = num;
	mlist->free_tail = newtail;
	if (mlist->free_tail == mlist->free_head) {
		/* cannot be full, 1 slot is always available for master */
		utprintf("%s: list %p full\n", __FUNCTION__, mlist);
	}
	up(&glock);

	return (0);
}


/*
 * remove minor number from freelist head and increment pointers
 * glock is held by caller
 * return 0 if freelist empty
 */
static minor_t
utstk_minor_get_free_head(utdc_mlist_t *mlist)
{
	minor_t	num = 0;

	utprintf("%s: h=%x, t=%x, x=%x, s=%x\n", __FUNCTION__, mlist->free_head,
		mlist->free_tail, mlist->next_num, mlist->size);

	if (mlist->free_head != mlist->free_tail) {
		num = mlist->freenums[mlist->free_head];
		mlist->freenums[mlist->free_head] = 0;		/* clean slot */
		if (++(mlist->free_head) >= mlist->size) {
			/* start over in ring */
			mlist->free_head = 0;
		}
	}

	return (num);
}


/*
 * This is done only when utstk_close is called by the master
 * mark all active controller state as stale
 * individual controller threads will clean up disk state
 */
static void
utstk_minor_invalidate_all(void)
{
	minor_t		num;
	utdisk_state_t	*dsp;

	utprintf("%s\n", __FUNCTION__);

	down(&glock);
	/* assumes all minors are > UTDISK_MASTER_INSTANCE */
	for (num = (UTDISK_MASTER_INSTANCE + 1); num < mlist.next_num; ++num) {
		if (((dsp = mlist.state[num]) != NULL) &&
			(dsp->flags & UTDISK_STATE_IN_USE)) {
			utprintf("%s: ctl=%d\n", __FUNCTION__, num);
			down(&dsp->lock);
			dsp->flags |= UTDISK_STATE_STALE;
			wake_up_all(&(dsp->io_wq));
			up(&dsp->lock);
		}
	}
	up(&glock);
}


/*
 * utdiskctl's ioctl proxy for utdisk
 * It passes an ioctl type message and waits until utstoraged executes it
 *
 * returns error value (0 if no error)
 */
static int
utstk_do_ioctl(utdisk_state_t *dsp, struct block_device *bdevp, int cmd,
		void *arg, int type)
{
	struct bio		*biop;
	minor_t			ctlmin;
	int			ret = 0;
	wait_queue_head_t	*wqheadp;
	utdc_bio_descriptor_t	*desc;
	ut_uscsi_cmd_t 		*scmd;
	struct bio_vec		*bvecp;
	char			*uscsi_bufaddr;
	int			uscsi_buflen;
	unsigned char		*dest;
	unsigned char		*src;
	int			reading;
	int			interrupted;
	DEFINE_WAIT(ioctlwait);

	utprintf("%s: cmd = %d %p\n", __FUNCTION__, cmd, dsp);

	if (down_interruptible(&(dsp->lock)) != 0) {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (-ENXIO);
	}
	ctlmin = dsp->ctlmin;	/* to reacquire lock later */
	biop = bio_alloc(GFP_KERNEL, 1);
	if (biop == NULL) {
		utprintf("do ioctl: (%d)=%x: no buf\n",
					dsp->ctlmin, cmd);
		up(&(dsp->lock));
		return (-ENXIO);
	}
	biop->bi_bdev = bdevp;
	uscsi_bufaddr = NULL;
	uscsi_buflen = 0;
	reading = 0;
	if (cmd == UTDC_TX_USCSI) {
		scmd = arg;
		if (scmd->uscsi_bufaddr != NULL) {
			bvecp = bio_iovec_idx(biop, biop->bi_idx);
			bvecp->bv_page = alloc_page(GFP_KERNEL);
			bvecp->bv_len = PAGE_SIZE;
			bvecp->bv_offset = 0;
			uscsi_bufaddr = scmd->uscsi_bufaddr;
			uscsi_buflen = scmd->uscsi_buflen;
			reading = scmd->uscsi_flags & UT_USCSI_READ;
			if (!reading) {
				dest = page_address(bvecp->bv_page);
				if (copy_from_user(dest, uscsi_bufaddr, uscsi_buflen) != 0) {
					utprintf("%s: copy in failed for uscsi data\n", __FUNCTION__);
					free_page((unsigned long)bvecp->bv_page);
					bio_put(biop);
					up(&(dsp->lock));
					return (-ENXIO);
				}
			}
		}
	}

	if (utstk_construct_bio_descriptor(biop, dsp, type) != 0) {
		utprintf("%s: %x: add bio failed\n",
				__FUNCTION__, cmd);
		desc = NULL;
		ret = -ENXIO;
	} else {
		desc = dsp->bio_tailp->bi_private;
		desc->ioctl_req.cmd = cmd;
		desc->ioctl_req.arg = arg;
		desc->ioctl_req.dsp = dsp;
		desc->ioctl_req.flags = UTDISK_IOCTL_NEW;
		wake_up_interruptible_sync(&(dsp->io_wq));

		wqheadp = &(desc->ioctl_wq);
		while ((dsp != NULL) &&
			(desc->ioctl_req.flags != UTDISK_IOCTL_DONE))  {
			utprintf("%s: %x: waiting biop=%p\n",
				__FUNCTION__, cmd, biop);
			prepare_to_wait(wqheadp, &ioctlwait,
							TASK_INTERRUPTIBLE);
			utstk_release_state(dsp);
			schedule();
			finish_wait(wqheadp, &ioctlwait);
			interrupted = 0;
			if (signal_pending(current)) {
				utprintf("%s: interrupted\n", __FUNCTION__);
				interrupted = 1;
			}
			dsp = utstk_acquire_state(ctlmin,
							UTDISK_STATE_VALIDATE);
			if (interrupted) {
				/*
				 * descriptor and bio have been queued for work
				 */
				if (dsp == NULL) {
					/* acquire dsp->lock failed */
					vfree(desc);
					bio_put(biop);
					return (-EINTR);
				}
				if ((desc->ioctl_req.flags
						== UTDISK_IOCTL_NEW) ||
					(desc->ioctl_req.flags
						== UTDISK_IOCTL_SENT)) {
					/* ask control thread to clean up */
					desc->ioctl_req.flags
							= UTDISK_IOCTL_INTR;
				} else if (desc->ioctl_req.flags
						== UTDISK_IOCTL_DONE) {
					vfree(desc);
					bio_put(biop);
				}
				utstk_release_state(dsp);
				return (-EINTR);
			}
		}
		if  (dsp == NULL) {
			utprintf("%s: device [%d] unplugged\n",
							__FUNCTION__, ctlmin);
			ret = -ENXIO;
		}
	}
	/* check whether hotdesk happened while we were sleeping */
	if ((ret == 0) &&
		(((volatile utdisk_state_t *)dsp)->flags
							& UTDISK_STATE_STALE)) {
		ret = -ENXIO;
	}
	utstk_release_state(dsp);		/* release lock */

	if (desc) {
		vfree(desc);
	}
	if (reading) {
		bvecp = bio_iovec_idx(biop, biop->bi_idx);
		if (bvecp == NULL) {
			utprintf("%s: bio_vec NULL\n", __FUNCTION__);
		} else {
			src = page_address(bvecp->bv_page);
			if (copy_to_user(uscsi_bufaddr, src, uscsi_buflen)
									!= 0) {
				utprintf("%s: copy out failed for uscsi data\n",
						__FUNCTION__);
				ret = -ENXIO;
			}
			free_page((unsigned long)bvecp->bv_page);
		}
	}
	bio_put(biop);		/* free bio created for ioctl */

	return (ret);
}


/* public function definitions */

/*
 * set utdisk's major number for use by utdiskctl
 */
void
utstk_set_disk_major(major_t num)
{
	down(&glock);
	utdisk_major = num;
	up(&glock);
	utprintf("%s: major = %d\n", __FUNCTION__, utdisk_major);
}


/*
 * set utdisk's make_request function for use by utdiskctl
 */
void
utstk_set_disk_makereq(make_request_fn *mrfp)
{
	utprintf("%s\n", __FUNCTION__);
	down(&glock);
	utdisk_make_request_p = mrfp;
	up(&glock);
}


/*
 * set utdisk's block devices operations structure for reference by utdiskctl
 */
void
utstk_set_disk_bdops(struct block_device_operations *bdopsp)
{
	utprintf("%s\n", __FUNCTION__);
	down(&glock);
	utdisk_bd_ops = bdopsp;
	up(&glock);
}


/*
 * Ask service daemon to check for presence of media
 */
void
utstk_check_media(utdisk_state_t *dsp, struct block_device *bdevp)
{
	utprintf("%s\n", __FUNCTION__);
	utstk_do_ioctl(dsp, bdevp, UTDC_TX_IOCTL_STATE, NULL, UTSTK_TYPE_IOCTL);
}


/*
 * eject media from a removable media device
 */
void
utstk_eject_media(utdisk_state_t *dsp, struct block_device *bdevp)
{
	utprintf("%s\n", __FUNCTION__);
	utstk_do_ioctl(dsp, bdevp, UTDC_TX_IOCTL_EJECT, NULL, UTSTK_TYPE_IOCTL);
}


/*
 * execute a user defined SCSI command
 * return 0 on success or error value
 */
int
utstk_uscsi_cmd(utdisk_state_t *dsp, struct block_device *bdevp,
		ut_uscsi_cmd_t *cmd)
{
	return utstk_do_ioctl(dsp, bdevp, UTDC_TX_USCSI, (void *)cmd,
							UTSTK_TYPE_USCSI);
}


/*
 * add bio request to disk's bio queue
 * We don't want to mess with the fields of the original bio
 * so we clone it and keep a reference to the real bio in the clone
 * We use the clone's next pointer to build our list of bios
 *
 * The trail is as follows
 * dsp_biop = dsp->bio_headp is our clone
 * desc = dsp_biop->bi_private is the bio descriptor
 * real_biop = desc->bio_orig is the original bio handed to us
 *
 * This routine is called with dsp->lock held
 * Return 0 on success, 1 on error
 */
static int
utstk_construct_bio_descriptor(struct bio *biop, utdisk_state_t *dsp, int type)
{
	struct bio		*dsp_biop;
	utdc_bio_descriptor_t	*desc;

	if ((biop == NULL) || (dsp == NULL)) {
		return (1);
	}
	if ((dsp->flags & UTDISK_STATE_STALE) != 0) {
		utprintf("%s: stale device %d\n", __FUNCTION__, dsp->ctlmin);
		return (1);
	}
	utprintf("%s: %p\n", __FUNCTION__, dsp);
	if ((dsp_biop = bio_clone(biop, GFP_NOIO)) == NULL) {
		utprintf("%s: clone failed\n", __FUNCTION__);
		return (1);
	}

#ifdef	UTDEBUG
	utprintf("%s: bio=%p, dspbio=%p, head=%p, tail=%p\n", __FUNCTION__,
		biop, dsp_biop, dsp->bio_headp, dsp->bio_tailp);
#endif	/* UTDEBUG */

	/* bio_clone leaves bi_idx == 0, we set it to what we got */
	dsp_biop->bi_idx = biop->bi_idx;
	desc = vmalloc(sizeof (utdc_bio_descriptor_t));
	if (desc == NULL) {
		utprintf("%s: insufficient memory\n", __FUNCTION__);
		bio_put(dsp_biop);
		return (1);
	}
	utprintf("%s: create utstk descriptor %p\n", __FUNCTION__, desc);
	desc->type = type;
	desc->bio_orig = biop;
	init_waitqueue_head(&(desc->ioctl_wq));
	dsp_biop->bi_private = desc;
	dsp_biop->bi_next = NULL;
	if (dsp->bio_headp == NULL) {
		/* this is the first bio */
		dsp->bio_headp = dsp_biop;
	} else if (dsp->bio_tailp == NULL) {
		/* extremely unpleasant situation */
		utprintf("%s: Corrupted bio list\n", __FUNCTION__);
		vfree(desc);
		bio_put(dsp_biop);
		return (1);
	} else {
		dsp->bio_tailp->bi_next = dsp_biop;
	}
	dsp->bio_tailp = dsp_biop;

	return (0);
}


/*
 * add a bio to the disk's request queue
 */
int
utstk_add_bio(struct bio *biop, utdisk_state_t *dsp)
{
	int ret;

	if (down_interruptible(&(dsp->lock)) != 0) {
		utprintf("%s: locking interrupted\n", __FUNCTION__);
		return (1);
	}
	ret = utstk_construct_bio_descriptor(biop, dsp, UTSTK_TYPE_DATA);
	if (ret == 0) {
		wake_up_interruptible_sync(&(dsp->io_wq));
	}
	up(&(dsp->lock));
	return ret;
}


/*
 * free disk state
 */
void
utstk_free_state(utdisk_state_t *dsp)
{
	struct bio		*biop;
	struct bio		*real_biop;
	utdc_bio_descriptor_t	*desc;
	utdc_ioctl_t		*iocp;

	if (dsp == NULL) {
		return;
	}
	down(&(dsp->lock));

	/*
	 * release all pending I/O requests
	 */
	while ((biop = dsp->bio_headp) != NULL) {
		/* get forward reference before destroying our cloned bio */
		dsp->bio_headp = biop->bi_next;
		if (dsp->bio_headp == NULL) {
			utprintf("%s[%d]: tail=%p\n", __FUNCTION__,
					dsp->ctlmin, dsp->bio_tailp);
			dsp->bio_tailp = NULL;
		}
		desc = biop->bi_private;	/* get descriptor */
		if (desc != NULL) {
			real_biop = desc->bio_orig;	/* get real bio */
		} else {
			real_biop = NULL;
		}
		biop->bi_private = NULL;
		if (desc != NULL) {
			if (desc->type == UTSTK_TYPE_DATA) {
				/* no one is waiting, destroy */
				vfree(desc);
			} else if (((iocp = &(desc->ioctl_req)) != NULL) &&
					(iocp->flags & UTDISK_IOCTL_INTR)) {
				/* no one waiting for interrupted ioctl */
				if (real_biop != NULL) {
					/*
					 * this bio was created by
					 * now defunct application ioctl thread
					 */
					bio_put(real_biop);
					real_biop = NULL;
				}
				vfree(desc);
			}
		}
		utprintf("%s[%d]: releasing clone bio %p, next=%p\n",
				__FUNCTION__, dsp->ctlmin, biop,
				dsp->bio_headp);
		bio_put(biop);			/* destroy our clone */
		if (real_biop != NULL) {
			utprintf("%s[%d]: releasing original bio %p\n",
				__FUNCTION__, dsp->ctlmin, real_biop);
			bio_endio(real_biop, real_biop->bi_size, (-ENXIO));
		}
	}
	if ((dsp->flags & UTDISK_STATE_OPEN) || (dsp->opens != 0)) {
		/* still more opens */
		up(&(dsp->lock));
		return;
	}
	up(&(dsp->lock));	/* because we're going to get glock */

	/* remove this state from state store */
	if ((dsp->flags & UTDISK_STATE_IN_USE) != 0) {
		utstk_set_state(dsp->ctlmin, NULL);
	}

	if (dsp->gdiskp != NULL) {
		if ((dsp->flags & UTDISK_STATE_READY) != 0) {
			del_gendisk(dsp->gdiskp);
			dsp->flags &= ~UTDISK_STATE_READY;
		}
		put_disk(dsp->gdiskp);
	}
	if (dsp->rqp != NULL) {
		blk_cleanup_queue(dsp->rqp);
	}
	if (dsp->cdevp != NULL) {
		cdev_del(dsp->cdevp);
	}
	utstk_minor_free_num(&mlist, dsp->ctlmin);

	memset(dsp, 0, sizeof (utdisk_state_t));	/* paranoia */
	vfree(dsp);

	return;
}

EXPORT_SYMBOL(utstk_set_disk_major);
EXPORT_SYMBOL(utstk_set_disk_makereq);
EXPORT_SYMBOL(utstk_set_disk_bdops);
EXPORT_SYMBOL(utstk_add_bio);
EXPORT_SYMBOL(utstk_free_state);
EXPORT_SYMBOL(utstk_check_media);
EXPORT_SYMBOL(utstk_eject_media);
EXPORT_SYMBOL(utstk_uscsi_cmd);

module_init(utstk_init);
module_exit(utstk_exit);

MODULE_LICENSE("GPL");
