
#include <math.h>
#include <stdio.h>
#include <stdlib.h>


#define Q_EQ     33000      // the reference point of a queue. QCN aims to keep 
                            // the queue occupancy at this reference level under congestion.
#define W        2          // the control parameter in calculating the congestion level variable Fb.
#define GD       0.0078125  // the control gain parameter which determines the 
                            // level of rate decrease given a Fb < 0 signals.
#define BC_LIMIT   150000   // the parameter which determines the byte-counter time-out threshold.
#define TIMER_PERIOD 0.015  // the parameter which determines the timer time-out threshold.
#define R_AI      5000000   // the parameter which determines the rate increase amount in AI stage.
#define R_HAI     50000000  // the parameter which determines the rate increase amount in HAI stage.
#define FAST_RECOVERY_TH  5 // the threshold which determines when a RL will exit fast 
                            // recovery (FR) stage, set to 5.
#define MIN_RATE       10000000   
                            // the minimum rate of a rate limiter, set to 10Mbps.
#define MIN_DEC_FACTOR   0.5 
                            // the minimum rate decrease factor, set to 0.5.



#define C 1000000000 //link capacity 
#define SWITCH_MAC_ADDRESS 0;
#define MAX_RL   32        //the maximum number of rate limiters
#define INACTIVE 0
#define ACTIVE 1

//
// a packet frame which arrives at a congestion node or at its destination.
struct Incoming_Frame {
        //an incoming frame can be tagged with the field of its flow id.
        int SA;            //the source MAC address of the feedback control frame.
        int DA;            //the destination MAC address of the feedback control frame.
        int flowid;        
        int size;          //frame size
};

//Rate Limiter
struct Rate_Limiter {
        int state;         // state of the rate limiter i: active or inactive.
        int flowid;        // the flow id that is associated with the rate limiter i.
        double crate;      // the current rate of the rate limiter i.
        double trate;      // the target rate of the rate limiter i.
        int tx_bcount;     // number of bytes left before increasing the state of the byte counter.
        int si_count;      // the stage of the byte counter that the rate limiter, i, is in.
        int timer;         // the timer of the rate limiter
        int timer_scount;  // the stage of the timer that the rate limiter, i, is in.
        int qlen;          // the queue length of the rate limiter queue
};


//a feedback control frame which sends the congestion information, Fb, back to the traffic source; 
//this packet frame can be sent either from any intermediate reflection point.
struct Feedback_Frame {
        int SA;            // the source MAC address of the feedback control frame.
        int DA;            // the destination MAC address of the feedback control frame.
        int flowid;        // the flow id of the feedback control frame.
        int fb;            // the congestion control information, Fb, of the feedback control frame.
};


class RP {
    public:
        RP();
        void receive_FbFrame(Feedback_Frame);
        int  get_rate_limiter_index (int);
        void self_increase(int rlidx);
        void set_timer(int,double);
        void transmit(int rlidx, int transmit_len); 
        void timer_expired(int rlidx);

    private:

        int min_dec_factor; // the minimum decrease factor, a single step of 
                            // decrease should not exceed this value.
        Rate_Limiter RL[MAX_RL];
};

// QCN Reaction Point:
RP::RP() 
{
        //initialization
        for (int i = 0; i < MAX_RL; i++) {
            RL[i].state = INACTIVE;
            RL[i].flowid = 1;
            RL[i].crate = C;
            RL[i].trate = C;
            RL[i].tx_bcount = BC_LIMIT;
            RL[i].si_count = 0;
            RL[i].timer_scount = 0;
        }
}

int RP::get_rate_limiter_index ( int flowid ) 
{
        //simple version
        return flowid;
}

void RP::receive_FbFrame(Feedback_Frame FBFrame) 
{
        //obtain the rate limiter index that is associated with a flowid
        //if no match, return the index of the next available rate limiter
        int rlidx = get_rate_limiter_index(FBFrame.flowid);

        if (RL[rlidx].state == INACTIVE) {
	    if (FBFrame.fb != 0) {
                //initialize new rate limiter 
                RL[rlidx].state = ACTIVE;
                RL[rlidx].flowid = FBFrame.flowid;
                RL[rlidx].crate = C;
                RL[rlidx].trate = C;
                RL[rlidx].si_count = 0;
		RL[rlidx].tx_bcount = BC_LIMIT;
            } else {
                //ignore FBFrame
                return;
	    }
        }
 
 
        if (FBFrame.fb != 0) {
            // use the current rate as the next target rate.
            // in the first cycle of fast recovery, 
            // the Fb < 0  signal would not reset the target rate. 
	    if (RL[rlidx].si_count != 0) {
                RL[rlidx].trate = RL[rlidx].crate;
                RL[rlidx].tx_bcount = BC_LIMIT;
	    }
          
            // set the stage counter
            RL[rlidx].si_count = 0;
            RL[rlidx].timer_scount = 0;
 
                  
            // update the current rate, multiplicative decrease
            double dec_factor = (1 - GD * FBFrame.fb);
            if (dec_factor < MIN_DEC_FACTOR) {
                dec_factor = MIN_DEC_FACTOR;
	    }

            RL[rlidx].crate = RL[rlidx].crate * dec_factor;
            if (RL[rlidx].crate < MIN_RATE) {
                RL[rlidx].crate = MIN_RATE;
	    }

 
            //reset the timer
            set_timer(rlidx, TIMER_PERIOD); 
        }
}

void RP::self_increase(int rlidx) 
{
        int to_count = (RL[rlidx].si_count < RL[rlidx].timer_scount) ? 
                        RL[rlidx].si_count:RL[rlidx].timer_scount;
           
        // if in the active probing stages, increase the target rate
	int Ri = 0;
        if (RL[rlidx].si_count > FAST_RECOVERY_TH || 
            RL[rlidx].timer_scount > FAST_RECOVERY_TH) {
            if (RL[rlidx].si_count > FAST_RECOVERY_TH &&
                RL[rlidx].timer_scount > FAST_RECOVERY_TH) {
                //hyperactive increase
                Ri = R_HAI * (to_count - FAST_RECOVERY_TH);
            } else {
                //active increase
                Ri = R_AI;
            } 
        } else {
                Ri = 0;
	}           
           
 
        //at the end of the first cycle of recovery
        if ((RL[rlidx].si_count == 1 || RL[rlidx].timer_scount == 1) && 
	     RL[rlidx].trate > 10* RL[rlidx].crate) {
            RL[rlidx].trate = RL[rlidx].trate/8;
        } else {
            RL[rlidx].trate = RL[rlidx].trate + Ri;
	}
 
        RL[rlidx].crate = (RL[rlidx].trate + RL[rlidx].crate)/2;
           
        //saturate rate at C
        if (RL[rlidx].crate > C) {
            RL[rlidx].crate = C;
        }
}
 
void RP::transmit(int rlidx, int transmit_len)  
{
        //release the rate limiter when its rate has reached C
        //and its associated queue is empty
        if (RL[rlidx].crate == C && RL[rlidx].qlen == 0) {
            RL[rlidx].state = INACTIVE;
            RL[rlidx].flowid = -1;
            RL[rlidx].crate = C;
            RL[rlidx].trate = C;
            RL[rlidx].tx_bcount = BC_LIMIT;
            RL[rlidx].si_count = 0;
            RL[rlidx].timer = INACTIVE; 
	} else {
	    RL[rlidx].tx_bcount -= transmit_len;
            //if a negative FBframe has not been received after transmitting
            //BC_LIMIT bytes, trigger self_increase;margin of randomness 30%
            if (RL[rlidx].tx_bcount < 0) {
	        int expire_thresh = 0;
		if (RL[rlidx].si_count < FAST_RECOVERY_TH) {
		    expire_thresh = int ((rand()*0.3/RAND_MAX+0.85)*BC_LIMIT);
		} else {
		    expire_thresh = int ((rand()*0.3/RAND_MAX+0.85)*BC_LIMIT/2);
		}
                RL[rlidx].si_count++;
                RL[rlidx].tx_bcount = expire_thresh;
                self_increase(rlidx);
            }
        }
}


 /*  Timers */
void RP::timer_expired(int rlidx)
{
        if (RL[rlidx].state == ACTIVE ) {
            RL[rlidx].timer_scount++;;
            self_increase(rlidx);
 
            //reset the  timer
            //margin of randomness 30%
   	    double expire_period = 0;
            if (RL[rlidx].timer_scount < FAST_RECOVERY_TH) {
                   expire_period = ((rand()*0.3/RAND_MAX+0.85)*TIMER_PERIOD);
            } else {
                   expire_period = ((rand()*0.3/RAND_MAX+0.85)*TIMER_PERIOD/2);
	    }
            set_timer(rlidx, expire_period);           
        }
}


class CP {
    public:
        CP();
        void receive (Incoming_Frame);
        int  Mark_Table(int);  
        void forward(Feedback_Frame *); 
    private:
        int qlen;     //current queue length (in pages). incremented upon packet arrivals 
                      //and decremented upon packet departures.
        int qlen_old; //queue length (in pages) at last sample.
        int time_to_mark;

};

////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////                                                        ////////////////////////
////////////////////            QCN Congestion Point                        ////////////////////////
////////////////////                                                        ////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////
CP::CP()
{
        qlen = 0;
        qlen_old = 0;
	time_to_mark = Mark_Table(0);
}

//assuming 6 bits of quantization
int CP::Mark_Table(int qntz_Fb) 
{
        switch (qntz_Fb/8){
            case 0: return 150000;
            case 1: return 75000;
            case 2: return 50000;
            case 3: return 37500;
            case 4: return 30000;
            case 5: return 25000;
	    case 6: return 21500;
	    case 7: return 18500;
	   
            default: return 18500;
        }
}

 
void CP::receive(Incoming_Frame IncomingFrame)  
{
        int Fb;         //feedback value which indicates the level of congestion.
        
	//calculate Fb value
        Fb = (Q_EQ - qlen) - W * (qlen - qlen_old);
        if (Fb < -Q_EQ * (2 * W + 1)) {
            Fb = -Q_EQ * (2 * W + 1);
        } else if (Fb > 0) {
            Fb = 0;
	}


        //the maximum value of Fb determines the number of bits that Fb uses.
        //uniform quantization of Fb, qntz_Fb, uses most significant bits of Fb.
        //note that now qntz_Fb has positive values.
        int qntz_Fb;    //quantized negative Fb (-Fb) value; assuming quantize into 6 bits
        qntz_Fb = Fb*63/(Q_EQ*(2*W+1));
 
        //sampling probability is a function of Fb
        int generate_fb_frame = 0;
        time_to_mark -= IncomingFrame.size; 
        if (time_to_mark < 0) {
            //generate a feedback frame if Fb is negative
	    if (Fb < 0) {
                generate_fb_frame = 1;
                qlen_old = qlen;
		int next_period = Mark_Table(qntz_Fb); //Mark Table is described below.
                time_to_mark = int ((rand()*0.3/RAND_MAX+0.85)*next_period);
            }
        }

        if (generate_fb_frame) {
            Feedback_Frame *FBFrame = new Feedback_Frame; 				  
            FBFrame->DA = IncomingFrame.SA;
            FBFrame->SA = SWITCH_MAC_ADDRESS;
            FBFrame->flowid = IncomingFrame.flowid;
            FBFrame->fb = qntz_Fb;
            forward(FBFrame);
	}
}
 
 

       

