/* 
 * pmcom.c - Execute commands on Livingston Portmaster servers.
 *
 * General information
 * -------------------
 * This is a utility that serves the purpose of executing commands remotely
 * on a Livingston Portmaster server. It takes care of logging on, issuing the
 * commands (from a commandfile, 'stdin' or from the commandline) and then 
 * logging off. The commandfile (and stdin) can use '#' as start of comments, 
 * leading and trailing whitespace are filtered away. This is what a command-
 * file may look like:
 *
 *   $ cat cmdfile
 *   # pmcom script to show status of our Livingston Portmaster servers
 *
 *   show ether0
 *   show all
 *   show netstat
 *   show global
 *   show memory
 *   show netconns
 *   $
 *
 * The usage of 'pmcom' is like this:
 *
 *   $ ./pmcom
 *   usage: pmcom [-vch] server [commandfile | - | commands]
 *   $
 *
 * where 'commandfile' is the name of a commandfile ('-' = stdin), 'commands'
 * are normal commandline arguments (except that each argument is a command to
 * send to the server which means that you may have to quote some of them, you
 * must also specify the option '-c' to indicate that the arguments are indeed 
 * commands) and '-v' means to output the commands given in the session as 
 * well as the results of them (the default is to only show the results and 
 * not the commands; error messages are output to 'stderr' in any case).
 *
 * This is how to send the commands using a commandfile:
 * 
 *   $ ./pmcom -v pm1 cmdfile
 *   < the resulting output >
 *   $
 *
 * and by using commandline arguments:
 *
 *   $ ./pmcom -vc pm1 'reset s7' 'sh ses' ver
 *   < the resulting output >
 *   $
 *
 * One thing to remember in the commandfile (or in the commandline arguments)
 * is to forget about the "quit" command since 'pmcom' will automatically 
 * issue that command at the end of the commandfile (or end of the commandline-
 * arguments). This utility could for instance be used in a 'cron' script 
 * to check status regularly or to setup other lengthy things such as filters.
 *
 * This utility should be protected *very* carefully, preferably only available
 * to the administrators that handle the Portmaster server since anyone who 
 * can run it may give *any* command to the server (including changing setup, 
 * reset ports etc.). Please think twice on where to put it and when setting 
 * the protection mode (something like protection mode 110 ("--x--x---") or
 * 100 ("--x------") with the appropriate user/group ownership sounds 
 * reasonable, remember that the password used to logon to the server can be
 * found in the executable and that utilities such as 'trace' (SunOS 4.x),
 * 'truss' (Solaris 2.x), 'strace' (Linux) may be used to find it). 
 * An ANSI-C compiler is required to compile the program. The program is 
 * completely free, do what you like with it.
 *
 * How it actually works
 * ---------------------
 * It starts a 'telnet' session to the Portmaster server, issues the commands 
 * to login, the commands from the commandfile and then logs out. The 
 * information is passed to the parent process through pipes (the parent 
 * process has a pipe through which it can send commands to the 'stdin' of the 
 * 'telnet process, and another pipe through which it receives the output from 
 * the 'stdout' (and 'stderr') of the 'telnet' process).
 *
 * Configuration options
 * ---------------------
 * Before building the program you should check out (and modify if necessary) 
 * the values of the constants to match your own configuration (good default 
 * values are already provided for SunOS 4.x, Solaris 2.x, DNIX and Linux).
 *
 * PASSWORD - a string, the default password to use for user '!root' if
 *            the hostname is not defined in the 'struct pwtable'.
 * TELNET_EXEC - a string, the path to the executable 'telnet'.
 * TELNET_CLOSE - a string, the leading char's in the string that 'telnet' 
 *                sends when a connection is closed.
 * TELNET_PORT - a string, the port to use when starting 'telnet'.
 * TELNET_PROMPT - a string, the prompt of telnet.
 * HAVE_SYS_WAIT, HAVE_WAIT - #define only one of these to indicate which
 *                            headerfile you have (<sys/wait.h> _or_ <wait.h>,
 *                            in that order), if none of them is #define-d a 
 *                            generic definition be used.
 *
 * Building the utility
 * --------------------
 * - Using SunOS 4.x:
 *   $ gcc -DSUNOS4 pmcom.c -o pmcom
 *   $ chmod 111 pmcom
 *
 * - Using Solaris 2.x:
 *   $ gcc -DSOLARIS pmcom.c -o pmcom
 *   $ chmod 111 pmcom
 *
 * - Using DNIX:
 *   $ cc -DDNIX pmcom.c -o pmcom
 *   $ chmod 111 pmcom
 *
 * - Using Linux:
 *   $ gcc -DLINUX pmcom.c -o pmcom
 *   $ chmod 111 pmcom
 *
 * - Using HP-UX 9.0x
 *   $ cc -DHPUX -D_INCLUDE_POSIX_SOURCE pmcom.c -o pmcom -O -Aa
 *   $ chmod 111 pmcom
 *
 * Version   Date   Comments
 * -------  ------  ----------------------------------------------------------
 *   1.0    960113  First release, copy lots from 'pmwho.c' v1.4.
 *                  Bugs: 1. Only one password is used.
 *                        2. If "dup" or "execl" fails (in the child) the
 *                           'stderr' is closed.
 *
 *   1.1    960404  Fixed bug #1 above. Can also take input from commandline
 *                  arguments and have #defines for compile on HP systems.
 *                  Signal-handling bug fixed.
 *
 *   1.2    980728  Handle the detection of a ">" and a "$hostname>" prompt (we
 *                  can ignore the trailing blank) properly without running into
 *                  problems when the input contains a ">" character elsewhere.
 *
 * Comments on how to improve this simple utility are very much welcome.
 *
 * /jp, Johan Persson, jp@abc.se
 *
 */

/* configuration constants - make your individual settings here */

#define PASSWORD        "password" /* the default password for logging in */

/* don't forget to change the login name below (about line 405) */

/* 
 * a password table structure with the purpose of being able to have different
 * passwords on different servers. If you only want to have one single password
 * you do not have to change anything, it will work as expected by default 
 * (by using the password PASSWORD as defined by the constant a few lines up).
 *
 * If you do however want to be able to have different passwords, then you must
 * enter the appropriate values below (I have added an example for the test-
 * server by the name of "pm1", uncomment and change as appropriate). Don't
 * forget that the constant PASSWORD is still the default if the hostname is
 * not found in the table.
 *
 */
   
struct {
  char hostname[32];
  char password[32];
} pwtable[] = {
  /* { "pm1", "topsecret" } */
};

#if defined(SUNOS4) || defined(SOLARIS)
#define TELNET_EXEC     "/usr/ucb/telnet"
#define TELNET_CLOSE    "Connection closed"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
#define HAVE_SYS_WAIT
#elif defined(DNIX) || defined(HPUX)
#define TELNET_EXEC     "/usr/bin/telnet"
#define TELNET_CLOSE    "Trying"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
#define HAVE_SYS_WAIT
#elif defined(LINUX)
#define TELNET_EXEC     "/usr/bin/telnet"
#define TELNET_CLOSE    "Connection closed"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
#define HAVE_SYS_WAIT
#else /* generic entry */
#define TELNET_EXEC     "/usr/ucb/telnet"
#define TELNET_CLOSE    "Connection closed"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
/* #define HAVE_SYS_WAIT */
/* #define HAVE_WAIT */
#endif

/* header files */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#if defined(HPUX)
#include <unistd.h>
#define NSIG _NSIG
#endif

#include <sys/types.h>

#if defined(HAVE_SYS_WAIT)
#include <sys/wait.h>
#elif defined(HAVE_WAIT)
#include <wait.h>
#else
extern pid_t wait(int *status);
#endif

#include <errno.h>
#include <signal.h>

/* program constants and global vars */
char  *progname;
char  *hostname;
char  *filename;
pid_t pid;
int   got_alarm;
int   verbose;
int   cmdline;
int   cmdc;
char  **cmdv;

#define LINELEN  256
#define PM_PROMPT ((char*)-1)

/* program declarations */
void main(int argc, char *argv[]);
void usage(void);
void pmcom(FILE *fp);
int  do_line(char *t);
int  do_read(int fd, char *text1, char *text2, int show, int timeout);
int  do_write(int fd, char *text, int timeout);
void sigalrm(int signo);
void sigfunc(int signo);
void err_msg(const char *format, ...);
void err_exit(const char *format, ...);
void stop_telnet(void);
int  is_pm_prompt(char *p, int len);

/* the program */

/* main procedure, find out program name and the hostname to use */
void main(int argc, char *argv[])
{
  char *t;
  int n, ch;
  FILE *fp;

  extern int optind;

  t = strrchr(argv[0], '/');
  progname = (t != NULL) ? t+1 : argv[0];
  hostname = NULL;
  pid = -1;
  verbose = cmdline = 0;

  while ((ch = getopt(argc, argv, "cvh")) != -1) {
    switch (ch) {
      case 'v':
        verbose = 1;
	break;
      case 'c':
	cmdline = 1;
	break;
      case 'h':
	/* fall-through */
      default:
	usage();
    }
  }

  if (optind >= argc) usage();
  hostname = argv[optind++];
  if (optind >= argc) usage();

  /* setup signals properly */
  for(n = 1; n < NSIG; n++) signal(n, sigfunc);
#if defined(SIGCHLD)
  signal(SIGCHLD, SIG_IGN);
#endif
#if defined(SIGCLD)
  signal(SIGCLD, SIG_IGN);
#endif

  if (cmdline) {
    cmdv = &argv[optind];
    cmdc = argc - optind;
    pmcom(NULL);
  }
  else if (strcmp(argv[optind], "-") == 0) {
    filename = "(stdin)";
    pmcom(stdin);
  }
  else {
    filename = argv[optind];
    fp = fopen(filename, "r");
    if (!fp) {
      err_exit("%s: cannot open commandfile '%s'\n", progname, filename);
    }
    pmcom(fp);
    fclose(fp);
  }

  fputc('\n', stdout);
  exit(0);
}

/* show usage info */
void usage(void)
{
  fprintf(stderr, "usage: %s [-vch] server [commandfile | - | commands]\n", progname);
  exit(0);
}

/* retrieve the information from the host by logging on, giving the command */
/* "show session" and logging out from 'hostname'                           */
void pmcom(FILE *fp)
{
  int fdpipe[2][2], fdin, fdout, fderr, err, n;

  got_alarm = 0;

  /* we need pipes to communicate between the programs */
  if (pipe(fdpipe[0]) < 0) {
    err_msg("%s: pipe failed: errno=%d\n", progname, errno);
    return;
  }
  if (pipe(fdpipe[1]) < 0) {
    err_msg("%s: pipe failed: errno=%d\n", progname, errno);
    close(fdpipe[0][0]);
    close(fdpipe[0][1]);
    return;
  }

  switch(pid = fork()) {
    case 0: /* child */
      /* child:stdin */
      close(0);
      if (dup(fdpipe[0][0]) < 0) {
	err_exit("%s: dup failed: errno=%d\n", progname, errno);
      }

      /* close first pipe */
      close(fdpipe[0][0]);
      close(fdpipe[0][1]);

      /* child:stdout */
      close(1);
      if (dup(fdpipe[1][1]) < 0) {
	err_exit("%s: dup failed: errno=%d\n", progname, errno);
      }

      /* child:stderr */
      if ((fderr = dup(2)) < 0) {
	err_exit("%s: dup failed: errno=%d\n", progname, errno);
      }

      close(2);
      if (dup(fdpipe[1][1]) < 0) {
	err = errno;
	dup(fderr);
	err_exit("%s: dup failed: errno=%d\n", progname, err);
      }

      /* close second pipe */
      close(fdpipe[1][0]);
      close(fdpipe[1][1]);

      /* exec TELNET application */
      execl(TELNET_EXEC, "telnet", hostname, TELNET_PORT, (char*)NULL);

      /* if we're still here the TELNET_EXEC could not be exec'd */
      err = errno;
      close(2);
      dup(fderr);
      err_exit("%s: execl(%s) failed: errno=%d\n", progname, TELNET_EXEC, err);
      break;

    case -1: /* error */
      err_exit("%s: fork failed: errno=%d\n", progname, errno);
      return;

    default: /* parent */
      /* close the childs end of the pipes */
      close(fdpipe[0][0]);
      close(fdpipe[1][1]);
      break;
  }

  /* now communicate with the 'telnet' process */
  fdin = fdpipe[1][0];
  fdout = fdpipe[0][1];

  for(;;) {
    n = do_read(fdin, "login: ", TELNET_PROMPT, 0, 15);
    if (n != 1) {
      n = -1; /* TELNET_PROMPT == connection failed */
      break;
    }
    /* replace portmaster login name below and don't use !root unless 
    you like 'living on the edge'.*/
    if ((n = do_write(fdout, "portmaster-login-name\n", 5)) < 0) break;

    if ((n = do_read(fdin, "Password: ", NULL, 0, 10)) < 0) break;

    {
      char *pw = PASSWORD;
      int k;

      for(k = 0; k < sizeof(pwtable)/sizeof(pwtable[0]); k++) {
	if (strcmp(pwtable[k].hostname, hostname) == 0) {
	  pw = pwtable[k].password;
	  break;
	}
      }

      if ((n = do_write(fdout, pw, 5)) < 0) break;
    }

    if ((n = do_write(fdout, "\n", 5)) < 0) break;

    if ((n = do_read(fdin, PM_PROMPT, NULL, 0, 10)) < 0) break;

    if (cmdline) {
      char buf[BUFSIZ];
      int m;

      for(m = 0; m < cmdc; m++) {
	strcpy(buf, cmdv[m]);
	if (do_line(buf) == 0) continue;
	if ((n = do_write(fdout, buf, 5)) < 0) break;
	if ((n = do_write(fdout, "\n", 5)) < 0) break;
      
	for(;;) {
	  n = do_read(fdin, "-- Press Return for More -- ", PM_PROMPT, 1, 20);
	  if ((n < 0) || (n == 2)) break;
	  if ((n = do_write(fdout, "\n", 5)) < 0) break;
	}
	if (n < 0) break;
      }
    }
    else {
      char buf[BUFSIZ];

      while (fgets(buf, sizeof(buf)-1, fp)) {
	if (do_line(buf) == 0) continue;
	if ((n = do_write(fdout, buf, 5)) < 0) break;
	if ((n = do_write(fdout, "\n", 5)) < 0) break;
      
	for(;;) {
	  n = do_read(fdin, "-- Press Return for More -- ", PM_PROMPT, 1, 20);
	  if ((n < 0) || (n == 2)) break;
	  if ((n = do_write(fdout, "\n", 5)) < 0) break;
	}
	if (n < 0) break;
      }
    }

    if ((n = do_write(fdout, "quit\n", 5)) < 0) break;
    
    if ((n = do_read(fdin, TELNET_CLOSE, NULL, 0, 20)) < 0) break;

    break;
  }

  close(fdin);
  close(fdout);

  if ((n < 0) && (got_alarm == 0)) {
    err_msg("%s: connection to host '%s' failed\n", progname, hostname);
  }

  stop_telnet();
}

/* prepare a line that is to be sent to the portmaster server as a command */
int do_line(char *t)
{
  char *p;
  char buf[BUFSIZ];
  int n;

  if ((p = strchr(t, '\n')) != NULL) *p = '\0';
  if ((p = strchr(t, '\r')) != NULL) *p = '\0';

  strcpy(buf, t);
  if ((p = strchr(buf, '#')) != NULL) *p = '\0';
  if ((n = strlen(buf)) == 0) return n;

  while (--n >= 0) {
    if ((buf[n] != ' ') && (buf[n] != '\t')) break;
    buf[n] = '\0';
  }

  for(p = buf; (*p == ' ') || (*p == '\t'); p++) ;

  strcpy(t, p);

  return strlen(t);
}

/* read information from 'telnet', stop reading upon errors, on a timeout */
/* and when the beginning of a line equals to 'text1' (and then return 1) */
/* or when the beginning of a line equals to 'text2' (and then return 2), */
/* if 'show' is non zero we display the information to the screen         */
/* (using 'stdout').                                                      */
int do_read(int fd, char *text1, char *text2, int show, int timeout)
{
  char line[LINELEN+1];
  int n, len1, len2, count = 0, m, err;

  /* setup alarm (so we won't hang forever upon problems) */
  signal(SIGALRM, sigalrm);
  alarm(timeout);
  
  len1 = (text1 != PM_PROMPT) ? strlen(text1) : 0;
  len2 = ((text2 != PM_PROMPT) && (text2 != NULL)) ? strlen(text2) : 0;

  /* start reading from 'telnet' */
  for(;;) {
    n = 0;
    for(;;) {
      if (n == LINELEN) {
	alarm(0); /* disable alarm */
	stop_telnet();
	err_exit("%s: too long line!\n", progname);
      }
      m = read(fd, &line[n], 1);
      if (m != 1) {
	err = errno;
	alarm(0); /* disable alarm */
	return(-1);
      }
      if (verbose) fputc(0xff & (int)line[n], stdout);
      if (line[n] == '\r') continue;
      if (text1 == PM_PROMPT) {
	if (is_pm_prompt(line, n+1)) {
	  /* we found the keyword we were searching for */
	  alarm(0); /* disable alarm */
	  return(1);
	}
      }
      else if (n >= len1-1) {
	if (strncmp(line, text1, len1) == 0) {
	  /* we found the keyword we were searching for */
	  alarm(0); /* disable alarm */
	  return(1);
	}
      }
      if (text2 == PM_PROMPT) {
	if (is_pm_prompt(line, n+1)) {
	  /* we found the keyword we were searching for */
	  alarm(0); /* disable alarm */
	  return(2);
	}
      }
      else if ((text2 != NULL) && (n >= len2-1)) {
	if (strncmp(line, text2, len2) == 0) {
	  /* we found the keyword we were searching for */
	  alarm(0); /* disable alarm */
	  return(2);
	}
      }
      if (line[n] == '\n') break;
      n++;
    }
    if (show && !verbose) {
      line[++n] = '\0';
      if (++count > 1) {
	/* the very first line of information is the remains of  */
	/* a previously issued command and should be ignored.    */

	fputs(line, stdout);
	fflush(stdout);
      }
    }
  }
}

/* write a command to the 'telnet' process */
int do_write(int fd, char *text, int timeout)
{
  int err, len, n;

  /* setup alarm (so we won't hang forever upon problems) */
  signal(SIGALRM, sigalrm);
  alarm(timeout);

  len = strlen(text);
  n = write(fd, text, len);
  if (n != len) {
    err = errno;
    alarm(0); /* disable alarm */
    return(-1);
  }
  else {
    alarm(0); /* disable alarm */
    return(0);
  }
}

/* our timeout procedure, used to abort malfunctioning connections */
void sigalrm(int signo)
{
  got_alarm = 1;
  err_msg("%s: timeout on connection to host '%s'\n", progname, hostname);
}

/* handle the reception of signals */
void sigfunc(int signo)
{
#if defined(SIGPIPE)
  if (signo != SIGPIPE)
#endif
  {
    err_msg("%s: received signal #%d during connection to host '%s' -- aborting\n", progname, signo, hostname);
  }

  stop_telnet();
  exit(1);
}

/* print error text */
void err_msg(const char *format, ...)
{
  va_list ap;

  /* show error message */
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  fflush(stderr);
}

/* print error text and exit gracefully */
void err_exit(const char *format, ...)
{
  va_list ap;

  /* show error message */
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  fflush(stderr);

  /* exit with error */
  exit(1);
}

/* try to terminate the telnet-child that we started */
void stop_telnet(void)
{
  pid_t p;

  if ((pid != -1) && (pid != 0)) {
    if (kill(pid, 0) >= 0) {
      kill(pid, SIGTERM);
      kill(pid, SIGKILL);

      if (kill(pid, 0) >= 0) {
	for(p = wait(0); (p != -1) && (p != pid); p = wait(0)) ; /* zombies */
      }
    }
    pid = -1;
  }
}

/* test if we have a portmaster prompt, a portmaster prompt looks like ">" or "hostname>" (actually it may even contain a blank after that but we just ignore it) */
int is_pm_prompt(char *p, int len)
{
  int n, found = 0;

  for(n = 0; n < len; n++) {
    if (p[n] == '>') {
      found = 1;
      break;
    }
    else if ((p[n] <= ' ') && (p[n] != '\r') && (p[n] != '\n')) break;
  }
  return found;
}

