/*
**  File		: chkmodems.c
**  Written By		: Paul White, pwhite@livingston.com
**  Description         :
**	This is an AS IS script for Livingston Customers
**	to use in the case of modems entering the DOWN/ADMIN
**	state when they shouldn't have.  This script will
**	monitor up to 10 Portmasters and will check for any
**	modems which enter the DOWN/ADMIN state.  It will
**	then try and reset them back to READY.
**
** THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
** KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
** PURPOSE AND NONINFRINGEMENT.  IN NO EVEN SHALL PAUL WHITE
** OR LIVINGSTON ENTERPRISES BE LIABLE FOR ANY CLAIM, DAMAGES,
** OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
** OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
** THIS SOFTWARE OR THE USE OF OTHER DEALINGS IN THIS SOFTWARE
**
** All source questions, comments, and suggestions should be
** e-mailed to pwhite@livingston.com however an answer is not
** promised.
*/

#include <sys/errno.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <stdio.h>
#include <netinet/in.h>

#ifdef SYSV
#include <sys/inet.h>
#endif

#define DEF_CHECK_TIME		10		/* Seconds to pause between checks */
#define DEF_PM_LIST		"pms.lst"	/* Filename containing PM List */
#define TIMEOUT			30		/* Read/Write Timeout in seconds */
#define MAXLEN			256		/* Max Line Length */
#define MATCH_MAX		5		/* Max Matches for DoRead() */
#define MATCH_LEN		30		/* Match Length for DoRead() */
#define MAX_MODEMS		61		/* Max # of modems in a PM3 */
#define MAX_PMS			10		/* Max # of PM's at once */

#ifdef ULTRIX
#define INADDR_ANY		0L

struct in_addr
{
	u_long s_addr;
};

struct sockaddr_in
{
	short sin_family;
	u_short sin_port;
	struct in_addr sin_addr;
	char sin_zero[8];
};
#endif

/* Structure to hold Portmaster information */
struct Portmaster
{
	char used;
	int fd;
	char hostname[60];
	int port;
	char ipaddr[60];
	char username[60];
	char password[60];
	char mdown[MAX_MODEMS];
} pm[MAX_PMS];

int	TimedOut, CHECK_TIME;
char	tmdown[MAX_MODEMS], PM_LIST[MAXLEN+1];
char	LastLine[MAXLEN+1], PrevLine[MAXLEN+1];

void	GotSigAlarm(int signo);
void	Usage();
int	DoRead(int fd, int timeout, int matches, char txt[MATCH_MAX][MATCH_LEN]);
int	DoWrite(int fd, char *txt1, char *txt2);
int	LoadPMs();

void Usage()
{
	printf("Usage: chkmodems [-t seconds] [-l pms.lst]\n");
}

void main(int argc, char **argv[])
{
	char line[MAXLEN+1], s1[MAXLEN+1], s2[MAXLEN+1], txt[MATCH_MAX][MATCH_LEN];
	int len, i, ret, n, pn, tmp;
	unsigned long maddr;
	struct sockaddr_in pmaddr;
	long t;

	/* Set default settings */
	strcpy(PM_LIST, DEF_PM_LIST);
	CHECK_TIME = DEF_CHECK_TIME;
	/* Proccess command line arguemnts */
	if (argc > 1)
	{
		if (argc != 3 && argc != 5)
		{
			Usage();
			exit(0);
		}
		for(i = 1;i < argc;i++)
		{
			strcpy(s1, (char *)argv[i]);
			if (s1[0] == '-')
			{
				if (s1[1] == 't') 		/* Received -t seconds */
				{
					CHECK_TIME = atoi(argv[i+1]);
					i++;
					continue;
				}
				else if (s1[1] == 'l')	/* Received -l pms.lst */
				{
					strcpy(PM_LIST, (char *)argv[i+1]);
					i++;
					continue;
				}
				else
				{
					Usage();
					exit(0);
				}
			}
			else
			{
				Usage();
				exit(0);
			}
		}
	}
	/* Set the function for an Alarm Signal */
	signal(SIGALRM, GotSigAlarm);

	/* Load the PM_LIST file */
	printf("Loading Portmasters from %s...", PM_LIST);
	fflush(stdout);
	ret = LoadPMs();
	if (ret == -1)
	{
		printf("Failed, aborting.\n", PM_LIST);
		exit(0);
	}
	printf("Completed.\n");
	/* List portmasters we are going to connect to */
	printf("Connecting to the following Portmasters:\n");
	for(pn = 0;pn < MAX_PMS;pn++)
		if (pm[pn].used == 1)
		{
			sprintf(s1, "[%s]:[%s]", pm[pn].username, pm[pn].password);
			printf("\t%-20s  %-20s  %s\n", pm[pn].hostname, s1, pm[pn].ipaddr);
			if ( (pm[pn].fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
			{
				printf("pm[%d]: Could not create a socket, deleting.\n", pn);
				pm[pn].used = 0;
			}
		}
	printf("\n");
	/* Make TCP connection to the Portmasters */
	for(pn = 0;pn < MAX_PMS;pn++)
	{
		if (pm[pn].used != 1)
			continue;
		maddr = inet_addr(pm[pn].ipaddr);
		pmaddr.sin_family = AF_INET;
		pmaddr.sin_addr.s_addr = maddr;
		pmaddr.sin_port = htons(pm[pn].port);
		printf("Attempting to connect to %s on port %d: ", pm[pn].hostname, pm[pn].port);
		fflush(stdout);
		if (connect(pm[pn].fd, (struct sockaddr *)&pmaddr, sizeof(pmaddr)) < 0)
		{
			printf("Connection Failed! Portmaster deleted.\n");
			pm[pn].used = 0;
			continue;
		}
		printf("Connected\n");
		/* Wait for login prompt */
		printf("\tWaiting for Login Prompt    : ");
		fflush(stdout);
		strcpy(txt[0], "login:");
		ret = DoRead(pm[pn].fd, TIMEOUT, 1, txt);
		if (ret != 1)
		{
			printf("DoRead() Failed! Portmaster deleted. \n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		/* Send the username */
		printf("Received  Sending Login......");
		fflush(stdout);
		ret = DoWrite(pm[pn].fd, pm[pn].username, "\n");
		if (ret != 1)
		{
			printf("DoWrite() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		printf("Sent.\n");
		/* Wait for the Password Prompt */
		printf("\tWaiting for Password Prompt : ");
		fflush(stdout);
		strcpy(txt[0], "Password:");
		ret = DoRead(pm[pn].fd, TIMEOUT, 1, txt);
		if (ret != 1)
		{
			printf("DoRead() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		/* Send the Password */
		printf("Received  Sending Password...");
		fflush(stdout);
		ret = DoWrite(pm[pn].fd, pm[pn].password, "\n");
		if (ret != 1)
		{
			printf("DoWrite() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		printf("Sent.\n");
		/* Wait for the Command prompt */
		printf("\tWaiting for Prompt          : ");
		fflush(stdout);
		strcpy(txt[0], ">");
		strcpy(txt[1], "Invalid");
		ret = DoRead(pm[pn].fd, TIMEOUT, 2, txt);
		if (ret == 2)
		{
			printf("Invalid Login! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		if (ret != 1)
		{
			printf("DoRead() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		printf("Received: \"%s\"\n", LastLine);
		fflush(stdout);
		/* Check ComOS version to make sure this is a PM3 */
		printf("\tChecking ComOS Version      :");
		fflush(stdout);
		strcpy(txt[0], "System");
		ret = DoWrite(pm[pn].fd, "version", "\n");
		if (ret != 1)
		{
			printf("DoWrite() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		ret = DoRead(pm[pn].fd, TIMEOUT, 1, txt);
		if (ret != 1)
		{
			printf("DoRead() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		printf("%s ", PrevLine);
		if (!strstr(PrevLine, "PM-3"))
		{
			/* Was not a PM3 */
			printf(" (NOT A PM3! Deleted)\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		printf("\n");
		GetPrompt(pm[pn].fd);
		/* Get all current DOWNed modems.  Will not try and reset these */
		printf("\tChecking for ADMIN modems   ");
		fflush(stdout);
		ret = DoWrite(pm[pn].fd, "show modems", "\n");
		if (ret != 1)
		{
			printf("DoWrite() Failed! Portmaster deleted.\n");
			close(pm[pn].fd);
			pm[pn].used = 0;
			continue;
		}
		printf(": ");
		fflush(stdout);
		ret = 0;
		strcpy(txt[0], "ADMIN");
		strcpy(txt[1], "DOWN");
		strcpy(txt[2], ">");
		tmp = 0;
		while(ret != 3 && ret != -5)
		{
			ret = DoRead(pm[pn].fd, (TIMEOUT * 2), 3, txt);
			if (ret < 0)
			{
				printf("DoRead() Failed! Portmaster deleted.\n");
				close(pm[pn].fd);
				pm[pn].used = 0;
				ret = -5;
				break;
			}
			if (ret == 1 || ret == 2)
			{
				sscanf(LastLine, "%s %s", s1, s2);
				strcpy(s2, &s1[1]);
				i = atoi(s2);
				pm[pn].mdown[i] = 1;
				tmp = 1;
			}
		}
		if (ret == -5)
			continue;
		n = 0;
		for(i = 0;i < MAX_MODEMS;i++)
			if (pm[pn].mdown[i])
			{
				if (n)
					printf(", ");
				else
					n = 1;
				printf("M%d", i);
			}
		if (tmp != 1)
			printf("None\n");
		else
			printf(" are in ADMIN\n");
	}
	/* Start to check the box every CHECK_TIME seconds */
	printf("Watching Box(es), Will Check every %d seconds...\n", CHECK_TIME);
	for(;;)
	{
		sleep(CHECK_TIME);
		time(&t);
		strcpy(s1, (char *)ctime(&t));
		s1[strlen(s1)-1] = '\0';
		printf("Checking [%s]\n", s1);
		for(pn = 0;pn < MAX_PMS;pn++)
		{
			if (pm[pn].used != 1)
				continue;
			for(i = 0;i < MAX_MODEMS;i++)
				tmdown[i] = 0;
			printf("\t%-20s", pm[pn].hostname);
			fflush(stdout);
			ret = DoWrite(pm[pn].fd, "show modems", "\n");
			if (ret != 1)
			{
				printf(" DoWrite() Failed, skipping.\n");
				continue;		
			}
			printf(".... ");
			fflush(stdout);
			ret = 0;
			strcpy(txt[0], "ADMIN");
			strcpy(txt[1], "DOWN");
			strcpy(txt[2], ">");
			while(ret != 3)
			{
				ret = DoRead(pm[pn].fd, (TIMEOUT * 2), 3, txt);
				if (ret < 0)
				{
					printf("DoRead() Failed, skipping.\n");
					continue;
				}
				if (ret == 1 || ret == 2)
				{
					/* This modem is in the DOWN state */
					sscanf(LastLine, "%s %s", s1, s2);
					strcpy(s2, &s1[1]);
					i = atoi(s2);
					tmdown[i] = 1;
				}
			}
			n = 0;
			for(i = 0;i < MAX_MODEMS;i++)
			{
				if (pm[pn].mdown[i] != tmdown[i])
				{
					if (n != 1)
					{
						printf("Results:\n");
						n = 1;
					}
					if (tmdown[i] == 0)
					{
						/* Modem was in DOWN, but now in READY */
						pm[pn].mdown[i] = 0;
						printf("\t\t\t\tM%d is now READY\n", i);
						continue;
					}
					if (tmdown[i] == 1)
					{
						/* Modem went into DOWN/ADMIN state.  Try and
						   reset it */
						pm[pn].mdown[i] = 1;
						printf("\t\t\t\tM%d is now in ADMIN, Resetting: ", i);
						fflush(stdout);
						ret = ResetModem(pn, i);
						if (ret == -1)
							printf("Failed!\n");
						else
							printf("Successful\n");
						continue;
					}
				}
			}
			/* Everything is OK, no changes */
			if (n != 1)
				printf("All OK\n");
		}
	}
	for(i = 0;i < MAX_PMS;i++)
		close(pm[i].fd);
	exit(0);
}

/* Reset a Modem */
int ResetModem(int pn, int m)
{
	char s1[MAXLEN+1], s2[MAXLEN+2], txt[MATCH_MAX][MATCH_LEN];
	int ret, i;

	tmdown[m] = 0;
	sprintf(s1, "set m%d on", m);
	ret = DoWrite(pm[pn].fd, s1, "\n");
	if (ret != 1)
		return -1;
	ret = GetPrompt(pm[pn].fd);
	if (ret != 1)
		return -1;
	sleep(5);
	ret = DoWrite(pm[pn].fd, "show modems", "\n");
	if (ret != 1)
		return -1;
	ret = 0;
	strcpy(txt[0], "ADMIN");
	strcpy(txt[1], "DOWN");
	strcpy(txt[2], ">");
	while(ret != 3)
	{
		ret = DoRead(pm[pn].fd, (TIMEOUT * 2), 3, txt);
		if (ret == 1 || ret == 2)
		{
			sscanf(LastLine, "%s %s", s1, s2);
			strcpy(s2, &s1[1]);
			i = atoi(s2);
			tmdown[i] = 1;
		}
	}
	if (tmdown[m] == 1)
		return -1;
	else
	{
		pm[pn].mdown[m] = 0;
		return 1;
	}
}

/* Received the Alarm Signal */
void GotSigAlarm(int signo)
{
	TimedOut = 1;
}

/* Read in until we receive the prompt */
int GetPrompt(int fd)
{
	char txt[MATCH_MAX][MATCH_LEN];
	int ret;

	strcpy(txt[0], ">");
	ret = DoRead(fd, TIMEOUT, 1, txt);
	return ret;
}

/* Read in text from FD until we receive a match in txt[][] */
int DoRead(int fd, int timeout, int matches, char txt[MATCH_MAX][MATCH_LEN])
{
	char line[MAXLEN+1];
	int i, len, pos;

	TimedOut = 0;
	alarm(timeout);

	for(;;)
	{	
		line[0] = '\0';
		pos = -1;
		len = 1;
		while(len > 0 && line[pos] != '\n' && line[pos] != '\r')
		{
			if (TimedOut)
				break;
			pos++;
			len = read(fd, &line[pos], 1);
			line[pos+1] = '\0';
			if (line[pos] == '\r')
			{
				line[pos] = '\0';
				strcpy(PrevLine, LastLine);
				strcpy(LastLine, line);
				line[0] = '\0';
				pos = -1;
				continue;
			}
			if (len < 0)
			{
				alarm(0);
				return -1;
			}
			if (strstr(line, "-- Press Return for More --"))
			{
				DoWrite(fd, "\n", "");
				line[0] = '\0';
				pos = -1;
				continue;
			}
			for(i = 0;i < matches;i++)
			{
				if (strstr(line, txt[i]))
				{
					alarm(0);
					line[pos+1] = '\0';
					strcpy(PrevLine, LastLine);
					strcpy(LastLine, line);
					return (i + 1);
				}
			}
		}
	}
	return 0;
}

/* Write txt1 and txt2 to fd */
int DoWrite(int fd, char *txt1, char *txt2)
{
	int ret, len;

	len = strlen(txt1);
	ret = write(fd, txt1, len);
	if (ret != len)
		return -1;
	if (strcmp(txt2, ""))
	{
		len = strlen(txt2);
		ret = write(fd, txt2, len);
		if (ret != len)
			return -1;
	}
	return 1;
}

/* Load the portmasters from PM LIST and do DNS look-up */
int LoadPMs()
{
	FILE *f1;
	struct hostent *hp;
	long maddr, xbyte;
	char s[MAXLEN+1], s1[MAXLEN+1], s2[MAXLEN+1], s3[MAXLEN+1], s4[MAXLEN+1];
	char hostname[MAXLEN+1], ipaddr[MAXLEN+1];
	int addr_byte[4], i, len, ln, pnum, ipchk;

	f1 = fopen(PM_LIST, "rt");
	if (!f1)
		return -1;
	ln = 0;
	pnum = 0;
	for(i = 0;i < MAX_PMS;i++)
		pm[i].used = 0;
	while(fgets(s, 120, f1) != NULL)
	{
		ln++;
		if (s[0] == '#')
			continue;
		if (s[strlen(s)-1] == '\r' || s[strlen(s)-1] == '\n')
			s[strlen(s)-1] = '\0';
		if (!strcmp(s, ""))
			continue;
		if (sscanf(s, "%s %s %s %s", s1, s2, s3, s4) != 4)
		{
			printf("Parse Error: line %d\n", ln);
			return -1;
		}
		if (s1[0] == '\"')
		{
			strcpy(s, &s1[1]);
			strcpy(s1, s);
			if (s1[strlen(s1)-1] == '\"')
				s1[strlen(s1)-1] = '\0';
		}
		if (s2[0] == '\"')
		{
			strcpy(s, &s2[1]);
			strcpy(s2, s);
			if (s2[strlen(s2)-1] == '\"')
				s2[strlen(s2)-1] = '\0';
		}
		if (s3[0] == '\"')
		{
			strcpy(s, &s3[1]);
			strcpy(s3, s);
			if (s3[strlen(s3)-1] == '\"')
				s3[strlen(s3)-1] = '\0';
		}
		if (s4[0] == '\"')
		{
			strcpy(s, &s4[1]);
			strcpy(s4, s);
			if (s4[strlen(s4)-1] == '\"')
				s4[strlen(s4)-1] = '\0';
		}
		strcpy(hostname, s1);
		len = strlen(hostname);
		ipchk = 0;
		for(i = 0;i < len;i++)
			if (!isdigit(hostname[i]) && hostname[i] != '.')
			{
				ipchk = 1;
				break;
			}
		if (ipchk == 0)
			strcpy(ipaddr, hostname);
		else
		{
			hp = gethostbyname(hostname);
			if (hp == 0)
			{
				printf("Invalid IP Address/Hostname: line %d\n", ln);
				return -1;
			}
			maddr = (long)ntohl(*(long *)hp->h_addr);
			for(i = 0;i < 4;i++)
			{
				xbyte = maddr >> (i*8);
				xbyte = xbyte & (long)0x000000FF;
				addr_byte[i] = xbyte;
			}
			sprintf(ipaddr, "%u.%u.%u.%u", addr_byte[3], addr_byte[2], addr_byte[1],
				addr_byte[0]);
		}
		pm[pnum].used = 1;
		strcpy(pm[pnum].hostname, hostname);
		strcpy(pm[pnum].ipaddr, ipaddr);
		pm[pnum].port = atoi(s2);
		strcpy(pm[pnum].username, s3);
		strcpy(pm[pnum].password, s4);
		for(i = 0;i < MAX_MODEMS;i++)
			pm[pnum].mdown[i] = 0;
		pnum++;
	}
	fclose(f1);
	return 1;
}
