/***********************************************************************

RADIUS
Remote Authentication Dial In User Service

Lucent Technologies Remote Access
4464 Willow Road
Pleasanton, CA   94588

Copyright 1992-1999 Lucent Technologies Inc.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

   * Redistributions in binary form must reproduce the above
     copyright notice, this list of conditions and the following
     disclaimer in the documentation and/or other materials provided
     with the distribution.

   * All advertising materials mentioning features or use of this
     software must display the following acknowledgement:

	  This product includes software developed by Lucent
	  Technologies and its contributors.

   * Neither the name of the copyright holder nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

This software is provided by the copyright holders and contributors
``as is'' and any express or implied warranties, including, but not
limited to, the implied warranties of merchantability and fitness for a
particular purpose are disclaimed. In no event shall the copyright
holder or contributors be liable for any direct, indirect, incidental,
special, exemplary, or consequential damages (including, but not
limited to, procurement of substitute goods or services; loss of use,
data, or profits; or business interruption) however caused and on any
theory of liability, whether in contract, strict liability, or tort
(including negligence or otherwise) arising in any way out of the use
of this software, even if advised of the possibility of such damage.

************************************************************************/


static char sccsid[] =
"$Id: users.c,v 1.28 1999/06/23 23:40:44 cdr Exp $ Copyright 1992-1999 Lucent Technologies Inc";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<errno.h>
#include	<string.h>

int		db_index;

#include	"radius.h"
#include	"users.h"

#define		MAXBUF	1024

extern char		*progname;
extern int		debug_flag;
extern int		errno;
extern int		radius_dbm;
extern char		*radius_dir;

static	void fieldcpy();
int  userparse();

#define FIND_MODE_NAME	0
#define FIND_MODE_REPLY	1
#define FIND_MODE_SKIP	2
#define FIND_MODE_FLUSH	3

/*************************************************************************
 *
 *	Function: user_find
 *
 *	Purpose: Find the named user in the database.  Create the
 *		 set of attribute-value pairs to check and reply with
 *		 for this user from the database.
 *
 *************************************************************************/

int
user_find(req_name, auth_name, check_pairs, reply_pairs, user_desc)
char		*req_name;
char		*auth_name;
VALUE_PAIR	**check_pairs;
VALUE_PAIR	**reply_pairs;
USER_DESC	*user_desc;
{
	VALUE_PAIR	*check_first;
	VALUE_PAIR	*reply_first;
	char		*ptr;
	char		buffer[MAXBUF];
	datum		contentd;
	datum		named;
	int		mode;
	int		req_namelen;
	int		user_auth_name();
	time_t		time();
	void		pairfree();

	/* 
	 * Check for valid input, zero length names not permitted 
	 */

	mode = FIND_MODE_NAME;

	ptr=req_name;
	while (*ptr != '\0') {
		if (*ptr == ' ' || *ptr == '\t') {
#ifdef SPACECHOP
			*ptr = '\0';
#else
			log_err("user_find: space in username \"%s\" rejected\n",req_name);
			return(-1);
#endif
		} else {
			ptr++;
		}
	}

	req_namelen=strlen(req_name);

	if (req_namelen < 1) {
		log_err("user_find: zero length username rejected\n");
		return(-1);
	}

	check_first = (VALUE_PAIR *)NULL;
	reply_first = (VALUE_PAIR *)NULL;

	if (radius_dbm != 0) {
		for (;;) {
			if (db_index == -1) {
				named.dptr = req_name;
				named.dsize = strlen(req_name);
			} else if (db_index == 0) {
				sprintf(buffer, "DEFAULT");
				named.dptr = buffer;
				named.dsize = strlen(buffer);
			} else {
				sprintf(buffer, "DEFAULT%d", db_index);
				named.dptr = buffer;
				named.dsize = strlen(buffer);
			}
			db_index++;

#ifdef NDBM
			contentd = dbm_fetch(user_desc, named);
#else /* not NDBM */
			contentd = fetch(named);
#endif /* NDBM */
			if(contentd.dsize == 0) {
				if (db_index == 0) {
					/*
					 * the request id failed
					 * lets try the defaults
					 */
					continue;
				}
				return(-1);
			}

			/*
			 * Parse the check values
			 */
			if (contentd.dsize > MAXBUF) {
				log_err("user_find: user record for user %s is too big, %d exceeds %d\n", req_name,contentd.dsize,MAXBUF);
				return(-1);
			}
			memcpy(buffer,contentd.dptr,contentd.dsize);
			buffer[contentd.dsize] = '\0';
			ptr = buffer;

			if(userparse(ptr, &check_first) != 0) {
				log_err("user_find: unable to parse check-items in dbm entry for user %s\n", req_name);
				pairfree(check_first,"user_find");
				return(-1);
			}

			/*
			 * set authentication name
			 */
			if (user_auth_name( req_name,
					auth_name,
					check_first) != 0) {
				pairfree(check_first,"user_find");
				check_first = (VALUE_PAIR *)NULL;
				continue;
			}
			break;
		}

		while(*ptr != '\n' && *ptr != '\0') {
			ptr++;
		}
		if(*ptr == '\0') {	/* no reply-items */
			*check_pairs = check_first;
			*reply_pairs = (VALUE_PAIR *)NULL;
			return(0);
		}
		ptr++;
		/*
		 * Parse the reply values
		 */
		if(userparse(ptr, &reply_first) != 0) {
			log_err("user_find: unable to parse reply-items in dbm entry for user %s\n", req_name);
			pairfree(check_first,"user_find");
			pairfree(reply_first,"user_find");
			return(-1);
		}
	} else {
	    while(fgets(buffer, sizeof(buffer),
				(FILE *)user_desc) != (char *)NULL) {
		if(mode == FIND_MODE_NAME) {
			/*
			 * Find the entry starting with the users name
			 */
			if((strncmp(buffer, req_name, req_namelen) == 0
					&& (buffer[req_namelen] == ' '
					|| buffer[req_namelen] == '\t'))
					|| strncmp(buffer, "DEFAULT", 7) == 0) {
				if(strncmp(buffer, "DEFAULT", 7) == 0) {
					ptr = &buffer[7];
					/*
					 * advance pointer to next white space
					 */
					while (isspace(*ptr) == 0) {
						ptr++;
					}
				}
				else {
					ptr = &buffer[req_namelen];
				}
				/*
				 * Parse the check values
				 */
				if(userparse(ptr, &check_first) != 0) {
					log_err("user_find: unable to parse check-items for user %s\n", req_name);
					pairfree(check_first,"user_find");
					return(-1);
				}
				/*
				 * set authentication name
				 */
				if (user_auth_name( req_name,
						auth_name,
						check_first) != 0) {
					pairfree(check_first,"user_find");
					check_first = (VALUE_PAIR *)NULL;
					continue;
				}
				mode = FIND_MODE_REPLY;
			}
		}
		else {
			if(*buffer == ' ' || *buffer == '\t') {
				/*
				 * Parse the reply values
				 */
				if(userparse(buffer, &reply_first) != 0) {
					log_err("user_find: unable to parse reply-items for user %s\n", req_name);
					pairfree(check_first,"user_find");
					pairfree(reply_first,"user_find");
					return(-1);
				}
			}
			else {
				/* We are done */
				*check_pairs = check_first;
				*reply_pairs = reply_first;
				return(0);
			}
		}
	}
	}
	/* Update the callers pointers */
	if(reply_first != (VALUE_PAIR *)NULL) {
		*check_pairs = check_first;
		*reply_pairs = reply_first;
		return(0);
	}
	return(-1);
}

/*************************************************************************
 *
 *	Function: user_auth_name
 *
 *	Purpose: Set authentication name, stripping pre/suffix
 *
 *************************************************************************/

int
user_auth_name(rname, auth_name, check_first)
char		*rname;
char		*auth_name;
VALUE_PAIR	*check_first;
{
	VALUE_PAIR	*fix;
	VALUE_PAIR	*get_attribute();
	int		req_len;
	int		len;
	char		namebuf[AUTH_STRING_LEN+2];
	char		*req_name;

	req_len = strlen(rname);
	req_name = namebuf;
	if (req_len > AUTH_STRING_LEN) {
		req_len = AUTH_STRING_LEN;
		req_name[req_len] = '\0';
	}
	strncpy(req_name, rname, req_len);
	if ((fix = get_attribute(check_first, PW_PREFIX))
						!= (VALUE_PAIR *)NULL) {
		len = strlen(fix->strvalue);
		if (req_len <= len || (strncmp(req_name,
						fix->strvalue, len) != 0)) {
			return(-1);
		}
		/*
		 * strip prefix from request name
		 */
		req_name += len;
		req_len -= len;
	}
	if ((fix = get_attribute(check_first, PW_SUFFIX))
			!= (VALUE_PAIR *)NULL) {
		len = strlen(fix->strvalue);
		if (req_len <= len || (strncmp(&req_name[req_len - len],
						fix->strvalue, len) != 0)) {
			return(-1);
		}
		/*
		 * strip suffix from request name
		 */
		req_len -= len;
	}
	strncpy(auth_name, req_name, req_len);
	auth_name[req_len] = '\0';
	return(0);
}

#define PARSE_MODE_NAME		0
#define PARSE_MODE_EQUAL	1
#define PARSE_MODE_VALUE	2
#define PARSE_MODE_INVALID	3

/*************************************************************************
 *
 *	Function: userparse
 *
 *	Purpose: Parses the buffer to extract the attribute-value pairs.
 *
 *************************************************************************/

int
userparse(buffer, first_pair)
char		*buffer;
VALUE_PAIR	**first_pair;
{
	int		mode;
	int		atoi();
	char		attrstr[64];
	char		valstr[256];
	DICT_ATTR	*attr = NULL;
	DICT_ATTR	*dict_attrfind();
	DICT_VALUE	*dval;
	DICT_VALUE	*dict_valfind();
	VALUE_PAIR	*pair;
	VALUE_PAIR	*link;
	VALUE_PAIR	*pairalloc();
	UINT4		ipstr2long();
	UINT4		get_ipaddr();
	int		user_gettime();
	struct tm	*tm;
	time_t		timeval;
	time_t		time();
#ifdef TIMELOCAL
	time_t		timelocal();
#else
	time_t		mktime();
#endif

	mode = PARSE_MODE_NAME;
	while(*buffer != '\n' && *buffer != '\0') {

		if(*buffer == ' ' || *buffer == '\t' || *buffer == ',') {
			buffer++;
			continue;
		}

		switch(mode) {

		case PARSE_MODE_NAME:
			/* Attribute Name */
			fieldcpy(attrstr, &buffer);
			if((attr = dict_attrfind(attrstr)) ==
						(DICT_ATTR *)NULL) {
				return(-1);
			}
			mode = PARSE_MODE_EQUAL;
			break;

		case PARSE_MODE_EQUAL:
			/* Equal sign */
			if(*buffer == '=') {
				mode = PARSE_MODE_VALUE;
				buffer++;
			}
			else {
				return(-1);
			}
			break;

		case PARSE_MODE_VALUE:
			/* Value */
			fieldcpy(valstr, &buffer);

			pair = pairalloc("userparse");

			strcpy(pair->name, attr->name);
			pair->attribute = attr->value;
			pair->type = attr->type;
			pair->vendor = attr->vendor;
			pair->vsattribute = attr->vsvalue;

			switch(pair->type) {

			case PW_TYPE_STRING:
				strcpy(pair->strvalue, valstr);
				pair->lvalue = strlen(valstr);
				break;

			case PW_TYPE_INTEGER:
				if(isdigit(*valstr)) {
					pair->lvalue = atoi(valstr);
				}
				else if((dval = dict_valfind(valstr)) ==
							(DICT_VALUE *)NULL) {
					free(pair);
					return(-1);
				}
				else {
					pair->lvalue = dval->value;
				}
				break;

			case PW_TYPE_IPADDR:
				pair->lvalue = get_ipaddr(valstr);
				break;

			case PW_TYPE_DATE:
                                timeval = time(0);
                                tm = localtime(&timeval);
				if (user_gettime(valstr, tm) < 0) {
					pair->lvalue = 0;
					log_err("invalid expiration format \"%s\" rejected\n",valstr);
				} else {
#ifdef TIMELOCAL
				pair->lvalue = (UINT4)timelocal(tm);
#else /* TIMELOCAL */
				pair->lvalue = (UINT4)mktime(tm);
#endif /* TIMELOCAL */
				}
				break;

			default:
				free(pair);
				return(-1);
			}
			pair->next = (VALUE_PAIR *)NULL;
			if(*first_pair == (VALUE_PAIR *)NULL) {
				*first_pair = pair;
			}
			else {
				link = *first_pair;
				while(link->next != (VALUE_PAIR *)NULL) {
					link = link->next;
				}
				link->next = pair;
			}
			mode = PARSE_MODE_NAME;
			break;

		default:
			mode = PARSE_MODE_NAME;
			break;
		}
	}
	return(0);
}

/*************************************************************************
 *
 *	Function: fieldcpy
 *
 *	Purpose: Copy a data field from the buffer.  Advance the buffer
 *		 past the data field.
 *
 *************************************************************************/

static	void
fieldcpy(string, uptr)
char	*string;
char	**uptr;
{
	char	*ptr;

	ptr = *uptr;
	if(*ptr == '"') {
		ptr++;
		while(*ptr != '"' && *ptr != '\0' && *ptr != '\n') {
			*string++ = *ptr++;
		}
		*string = '\0';
		if(*ptr == '"') {
			ptr++;
		}
		*uptr = ptr;
		return;
	}

	while(*ptr != ' ' && *ptr != '\t' && *ptr != '\0' && *ptr != '\n' &&
						*ptr != '=' && *ptr != ',') {
			*string++ = *ptr++;
	}
	*string = '\0';
	*uptr = ptr;
	return;
}

#ifdef PASSCHANGE
/*************************************************************************
 *
 *	Function: user_update
 *
 *	Purpose: Updates a user in the database.  Replaces the original
 *		 entry with the name, the list of check items, and the
 *		 list of reply items which are supplied.
 *
 *************************************************************************/

int
user_update(name, user_check, user_reply)
char		*name;
VALUE_PAIR	*user_check;
VALUE_PAIR	*user_reply;
{
	FILE		*oldfd;
	FILE		*userfd;
	char		buffer[256];
	char		oldfile[256];
	char		newfile[256];
	int		namelen;
	int		mode;
	void		fprint_attr_val();

	sprintf(oldfile, "%s/%s", radius_dir, RADIUS_USERS);
	sprintf(newfile, "%s/%s", radius_dir, RADIUS_HOLD);

	/* Open the old user file */
	if((oldfd = fopen(oldfile, "r")) == (FILE *)NULL) {
		log_err("user_update: could not read users file %s\n",oldfile);
		return(-1);
	}

	/* Open the new user file */
	if((userfd = fopen(newfile, "w")) == (FILE *)NULL) {
		log_err("user_update: could not write to %s\n", newfile);
		fclose(oldfd);
		return(-1);
	}

	mode = FIND_MODE_NAME;
	namelen = strlen(name);

	/* Copy the old to the new, only recreating the changed user */
	while(fgets(buffer, sizeof(buffer), oldfd) != (char *)NULL) {
		if(mode == FIND_MODE_NAME) {
			if((strncmp(buffer, name, namelen) == 0 &&
		 	 (buffer[namelen] == ' ' || buffer[namelen] == '\t'))) {

				/* Write our new information */
				fprintf(userfd, "%s\t", name);
				while(user_check != (VALUE_PAIR *)NULL) {
					fprint_attr_val(userfd, user_check);
					if(user_check->next !=
							(VALUE_PAIR *)NULL) {
						fprintf(userfd, ", ");
					}
					user_check = user_check->next;
				}
				fprintf(userfd, "\n\t");
				while(user_reply != (VALUE_PAIR *)NULL) {
					fprint_attr_val(userfd, user_reply);
					if(user_reply->next !=
							(VALUE_PAIR *)NULL) {
						fprintf(userfd, ",\n\t");
					}
					user_reply = user_reply->next;
				}
				fprintf(userfd, "\n");
				mode = FIND_MODE_SKIP;
			}
			else {
				fputs(buffer, userfd);
			}
		}
		else if(mode == FIND_MODE_SKIP) {
			if(*buffer != ' ' && *buffer != '\t') {
				fputs(buffer, userfd);
				mode = FIND_MODE_FLUSH;
			}
		}
		else {
			fputs(buffer, userfd);
		}
	}
	fclose(oldfd);
	fclose(userfd);
	if (rename(newfile,oldfile) != 0) {
		log_err("user_update: unable to rename %s to %s after updating user %s; error %d\n",newfile,oldfile,name,errno);
		return(-1);
	}
	return(0);
}

#endif	/* PASSCHANGE */

/*************************************************************************
 *
 *	Function: user_gettime
 *
 *	Purpose: Turns printable string into correct tm struct entries
 *
 *************************************************************************/

static char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

int
user_gettime(valstr, tm)
char		*valstr;
struct tm	*tm;
{
	int	i;
	int	atoi();

	/* Get the month */
	for(i = 0;i < 12;i++) {
		if(strncmp(months[i], valstr, 3) == 0) {
			tm->tm_mon = i;
			i = 13;
		}
	}

	/* Get the Day */
	tm->tm_mday = atoi(&valstr[4]);

	/* Now the year */
	tm->tm_year = atoi(&valstr[7]) - 1900;

	/* Midnight */
	tm->tm_sec = 0;
	tm->tm_min = 0;
	tm->tm_hour = 0;
	
	/* if date makes no sense return failure */
	if (i == 12 || tm->tm_mday < 1 || tm->tm_mday > 31 ||
	    tm->tm_year < 70) {
		return(-1);
	} else {
		return(0);
	}
}

/*************************************************************************
 *
 *	Function: user_open
 *
 *	Purpose: open the users file
 *
 *************************************************************************/

USER_DESC *
user_open(void)
{
	USER_DESC	*user_desc;
	char		buffer[1024];
	extern FILE	*user_open_flat(char *);

	sprintf(buffer, "%s/%s", radius_dir, RADIUS_USERS);

	if (radius_dbm == 0) {
		return( (USER_DESC *)user_open_flat(buffer) );
	}

# ifdef NDBM
	if((user_desc = dbm_open(buffer, O_RDONLY, 0)) == (USER_DESC *)NULL)
# else /* not NDBM */
	if(dbminit(buffer) != 0)
# endif /* NDBM */
	{
		log_err("user_open: could not read user dbm file %s\n", buffer);
		return((USER_DESC *)NULL);
	}
	db_index = -1;
	return(user_desc);
}

FILE *
user_open_flat(file_name)
char	*file_name;
{
	FILE	*user_desc;

	/*
	 * Open the user table
	 */
	if((user_desc = fopen(file_name, "r")) == (FILE *)NULL) {
		log_err("user_open: could not read user file %s\n", file_name);
		return((FILE *)NULL);
	}
	return(user_desc);
}

/*************************************************************************
 *
 *	Function: user_close
 *
 *	Purpose: close the users file
 *
 *************************************************************************/

void
user_close(user_desc)
USER_DESC	*user_desc;
{
	if (radius_dbm != 0) {
# ifdef NDBM
		dbm_close(user_desc);
# else  /* not NDBM */
		dbmclose();
# endif /* NDBM */
	} else {
		fclose((FILE *)user_desc);
	}
}
