/*
 *	PMNETD
 *	PortMaster NETdata Daemon
 *
 *	Lucent Technologies Remote Access
 *	4464 Willow Road
 *	Pleasanton, CA   94588
 *
 *	Copyright 1998 Lucent Technologies, Inc. All Rights Reserved.
 *  Portions Copyright (c) 1999 Greg Cronau
 *
 *	This software is provided under license from Lucent
 *	Technologies, Inc., the terms and conditions of which are set
 *	forth in a Software License Agreement that is contained in an
 *	End User Agreement contained in the product packaging, and
 *	electronically on the Lucent Remote Access ftp site. This
 *	software may only be used in conjunction with Lucent (or Lucent
 *	authorized) PortMaster(R) products.  Lucent makes no warranties
 *	to any licensee concerning the applicability of the software to
 *	licensee's specific requirements or the suitability of the
 *	software for any intended use.  Licensee shall not remove,
 *	modify or alter any copyright and/or other proprietary rights
 *	notice and must faithfully reproduce all such notices on any
 *	copies or modifications to this software that it makes.
 *
 *	Lucent Technologies, Inc. makes no representations about the
 *	suitability of this software for any purpose.  It is provided
 *	"as is" without express or implied warranty.
 */

/*
 * 09/30/99 Greg Cronau, PM Technologies & Simplix Inc.  gregc@pm-tech.com
 *
 * Substantial portions of this code were re-written to allow binary
 * transparency. This code will now work with fax modems.
 *
 * Code was reformatted into something I could deal with. Tabs are 4.
 * My formating style is a bit eccentric. Deal with it.
 *
 * I also fixed a host of other small bugs, memory leaks, and bad malloc()
 * usage. Added proper use of non-blocking I/O, and fixed a great deal
 * of just plain bad coding.
 *
 * Major portions of this code were also rewritten to either a.) simplify
 * the code, b.) optimize the code, or c.) just clean it up because it was
 * so bad I couldn't stand to look at it. The basic structure of this
 * program is the same as the original, but I have gone over each function
 * line by line to clean things up, and practically every line of code
 * was changed in some way.
 *
 * A significant amount of error checking and debugging messages were
 * also added.
 *
 * Switch for external devices file was also changed from "-d" to "-f"
 * and "-d" switch is now used for additional debugging info.
 */


/************************************************************************
1998/09/18

		PMNETD: PortMaster Netdata Daemon

The pmnetd program was written to replace the in.pmd program and is
likely to work as a substitute in almost all situations.  This program
is not supported by Lucent Remote Access but is provided for the
convenience of our customers.

If you port pmnetd to any other systems or fix any bugs in it and would
like to share those with us, please send email to
pmnetd-feedback@livingston.com.  If you make changes in your copy of
pmnetd.c please update the version string that the -v flag prints.

pmnetd provides a gateway between a pseudo-tty and a Lucent
PortMaster(R) serial port, usually connected to a modem or printer.

---------------------------------------------------------------------------
	Running pmnetd:

By default, pmnetd runs in the background.

You can use the following command line options with pmnetd:

	-v					Returns version information
	-d [{N}]			More copious debugging information. (N defaults to 3)
	-x					Program stays in foreground and dumps all
						debug information to stdout
	-f {device file}	Specifies the device file
						(default is /etc/pmnet/devices)
	-l [{log file}]		Logs debug messages to the file specified
						(default with just "-l" is /var/log/pmnetd.log)

Definitions of new debug bits:

0x0001	-	Logs information about data flushed when pty and net socket
			closings are detected.
0x0002	-	Logs count of characters read and written to each fd.
0x0004	-	Logs a marker when each read() or write() operation begins.
			(This was used mainly to insure that non-blocking I/O was
			 implemented and working correctly.)
0x0008	-	Logs data sent and received with DoRead() and DoWrite() calls.
			(These are only used during secure login phase.)
0x0010	-	Logs data from all read()'s to either net sockets or ptys.
0x0020	-	Logs data from all write()'s to either net sockets or ptys.
0x0040	-	Logs data from all read()'s and write()'s from/to net sockets.
0x0080	-	Logs data from all read()'s and write()'s from/to ptys.
0x8000	-	Suppress logging of non-printing characters.

The values 10,20,40,80 log a line of data that consists of the fd, a "-->"
or "<--" indicating read or write, the count of bytes read or written, and
up to the first 35 or so characters of the data. The format is similiar to
the one used by hylafax.

Note: Above information is above and beyond the original basic information
that is logged whenever the "-l" switch is used to enable logging, or the
"-x" switch is used. So even if no "-d" switch is used, you will still
get initialization information, open and close info, and all error messages
to the logfile or stdout.

---------------------------------------------------------------------------
	Configuring pmnetd:

The /etc/pmnet/devices file has the following format:

#Device		Pin?	Sec?	PortMaster		Port	Username	Password

/dev/ttyr1	no		no		192.168.5.10	6001
/dev/ttyr2	no		no		192.168.5.10	6002
/dev/ttyr3	yes		no		192.168.5.10	6003
/dev/ttyra	no		no		testing.edu.com	1234
/dev/ttyrb	yes		no		1.1.1.1			1234
/dev/ttyrc	no		no		2.2.2.2
/dev/ttyr4	yes		yes		192.168.5.10	23		testu		testp
/dev/ttyr5	no		no		dial.edu.com	7000

"Device" column - Place the full path to the pseudo-tty device to map.

"Pin" column - Enter either "yes" or "no" to inform pmnetd to keep this
connection always active ("pinned up"), or to connect to the device
only when a tty open is detected and disconnect when a tty close is
detected.

"Sec" column - Enter either "yes" or "no" to inform pmnetd whether this
is an "outbound" security port on the PortMaster or not.  If you enter
"yes", pmnetd does the following:
1. Expects a login prompt when it connects to the PortMaster 
   at the specified port, and sends the login specified
2. Expects a password prompt, and sends the password specified
3. Expects a message from the PortMaster that it has established a 
   connection to a port

If Sec is set to "yes" then Pin should be set to "yes" as well.
For more information, see technical notes on "Outbound Security."
(Gregc: *WHAT* technical notes???)

"PortMaster" column - Enter the IP address or hostname of the PortMaster.

"Port" column - Enter the port to connect to.  If "Sec" is set to no, enter
the  Netdata TCP port set on the serial port of the PortMaster.  If Sec
is set to "yes", enter the port number 23.

The "Username" and "Password" fields should be filled in when using the
outbound security option.

---------------------------------------------------------------------------
	Compiling pmnetd:

To compile pmnetd.c, define one of the following OS_ defines, depending
on your operating system.  Here is an example for BSDI using gcc:

  gcc -DOS_BSD -Wall -O2 -g -o pmnetd pmnetd.c

[Note: Gregc: I highly reccomend you use gcc or at least an ansi
 compliant compiler. I've introduced a few ansi'isms into the code.]

The following defines are available, although some have not been tested:

Define		Tested?				OS Versions
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
OS_BSD		TESTED: WORKS		BSDI 3.1 and 4.0
OS_BSD		TESTED: WORKS		FreeBSD 2.2.7
OS_BSD		TESTED: WORKS		NetBSD 0.9a [2]
OS_LINUX	TESTED: WORKS		Linux 2.0
OS_SUNOS	TESTED: WORKS		SunOS 4.1.3_U1  (Some problems) [1]
OS_HPUX		UNTESTED
OS_AIX		UNTESTED
OS_MIPS		UNTESTED

Notes (Gregc) 10/6/99:

[1] Sunos version appears to work ok for simple tests using tip.
    However, there appears to be some issue involved with closing and
    re-opening a pty. As long as pmnetd opens the pty(master) side of the
    connection first, everything works fine. But if the client manages to
    open, or re-open the tty(slave) side first, or if the client never
    closes it's side, then pmnetd errors on the open() with a code of EBUSY.
    Apparently, the SunOS pty driver, in ptcopen() in tty_pty.c, first
    checks for the existance of a setup slave side data structure, and
    assumes thet if this structure exists, then the pty is "busy". Newer
    bsd4.4 based systems don't so this. Not sure if this is a bug in the
    old code, the new code, or just a philosophical difference.
    Due to the way that hylafax works, this prevents hylafax from using
    pmnetd, when pmnetd is running on a SunOS system. See my README
    file for more details.

[2] Code compiled cleanly and ran on an ancient version of NetBSD-0.9a
    that I had laying around. No fax server or sophisticated software
    was available on the machine for in-depth testing. However, simple
    testing with tip revealed no immediate problems.

________________________________________________________________________
	Copyright and Trademarks:

Copyright 1998 Lucent Technologies. All rights reserved.

PortMaster, ComOS, and ChoiceNet are registered trademarks of Lucent
Technologies, Inc. RADIUS ABM, PMVision, PMconsole, and IRX are
trademarks of Lucent Technologies, Inc. ProVision is a service mark of
Lucent Technologies, Inc. All other marks are the property of their
respective owners.

	Notices

Lucent Technologies, Inc. makes no representations or warranties
with respect to the contents or use of this publication, and specifically
disclaims any express or implied warranties of merchantability or
fitness for any particular purpose. Further, Lucent Technologies,
Inc. reserves the right to revise this publication and to make changes
to its content, any time, without obligation to notify any person or
entity of such revisions or changes.

	Contacting Lucent Remote Access Technical Support

This software is not supported by Lucent, but you can email suggestions
to pmnetd-feedback@livingston.com

************************************************************************/

#ifdef OS_SUNOS
#define __USE_FIXED_PROTOTYPES__
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <ctype.h>
#include <netdb.h>
#include <errno.h>
#include <strings.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef OS_LINUX
#include <sys/sysmacros.h>
#endif
#if defined(OS_AIX) | defined(OS_LINUX)
#include <dirent.h>
#define direct dirent
#endif
#if defined(OS_HPUX)
#include <termios.h>
#include <sys/bsdtty.h>
#include <sys/ptyio.h>
#endif

#ifdef OS_SUNOS
#include <string.h>
 extern int  gettimeofday(struct timeval *, struct timezone *);
 extern int  ioctl(int, int, ...);      /* caddr_t for 3rd arg. */
 extern void bzero(char *, int);
 extern int  socket(int, int, int);
 extern int  connect(int, struct sockaddr *, int);
 extern int  select(int, fd_set *, fd_set *, fd_set *, struct timeval *);
 extern int  usleep(unsigned int);
#endif

#ifdef OS_BSD
#define MAJOR_TTY	5
#define MAJOR_PTY	6
#endif

#ifdef OS_LINUX
#define MAJOR_TTY   3
#define MAJOR_PTY   2
#endif

#ifdef OS_HPUX
#define MAJOR_TTY	17
#define MAJOR_PTY	16
#endif

#ifdef OS_SUNOS
#define MAJOR_TTY	20
#define MAJOR_PTY	21
#endif

#if !defined(O_NOCTTY)
#define O_NOCTTY	0
#endif

#ifndef min
#define min(a, b)	((a) < (b) ? (a) : (b))
#endif
#ifndef max
#define max(a, b)	((a) > (b) ? (a) : (b))
#endif

#define	VERSION				"2.0"
#define MAXLEN				1024
#define BUF_SIZE			(MAXLEN * 16)
#define DEFAULT_DEVICE_FILE	"/usr/local/etc/pm-devices"
#define DEFAULT_LOG_FILE	"/var/log/pmnetd.log"
#define MATCH_MAX			5
#define MATCH_LEN			80
#define CONNECT_TIME		2	/* # secs to wait between connect trys. */
#define IDLE_TIMEOUT		30	/* # secs to wait until dropping a
								   non-pinned net connection. */

struct global_info
	{
	int  restart;
	int  do_log;
	char *progname;
	FILE *dout;
	time_t last_hup;
	char device_fn[MAXLEN];
	char log_fn[MAXLEN];
	} gi;

typedef struct pm_list
	{
	char ttyname[256];
	char ptyname[256];
	int ptyfd;
	int netfd;
	char ipaddr[32];
	int port;
	int pin;
	int open;
	int security;
	char username[32];
	char password[32];
	time_t lt;
	time_t la;
	char to_buf[BUF_SIZE];
	char from_buf[BUF_SIZE];
	int to_cnt;
	int from_cnt;
	int net_defer;				/* Defer net FD close until buffer drains. */
	int pty_defer;				/* Defer pty FD close until buffer drains. */
	int pty_delay;				/* if > 0 do a delay before reads from pty. */
	struct pm_list *next;
	struct pm_list *prev;
	} PMLIST;

 int Debugf = 0;
 PMLIST *Pmfirst = NULL, *Pmlast = NULL;

 PMLIST *begin_pmlist(void);
 PMLIST *new_pmlist(void);
 PMLIST *add_pm(char *, char *, int, int, int, char *, char *);
 void usage(void);
 void sig_fatal(int);
 void sig_hup(int);
 void Debug(char *);
 void ADebug(char *);
 void ConnectPinnedNETfds(void);
 void ConnectPM(PMLIST *);
 void DisconnectPM(PMLIST *, char *, int, int);
 void cycle_pty(PMLIST *, int);
 void close_pty(PMLIST *, int);
 void free_pmlist(void);
 void PipeLoop(void);
 void flush_to_buf(PMLIST *);
 void flush_from_buf(PMLIST *);
 void ResetGlobalInfo(void);
 void log_data(PMLIST *, int, int, int);
 char *ResolveDNS(char *);
 char *findpty(char *);
 char *dev2pty(dev_t);
 char *Version(void);
 char *fmtstr(char *, int);
 char *fmtdata(char *, int, int);
 int  GeneralSetup(void);
 int  LoadDevices(void);
 int  DoRead(int, int, int, char txt[MATCH_MAX][MATCH_LEN]);
 int  DoWrite(int, char *, char *);
 int  SecureLogin(PMLIST *);
 int  CreatePTYs(void);
 int  getpty(PMLIST *);
 int  set_net_nonblocking(PMLIST *, int);
 int  set_fd_nonblocking(int, int);
 int  cycle_logfile(void);

/* NOTE: Unless specified otherwise, all functions below of type (char *)
   or type (PMLIST *) return NULL if the function failed.
   All functions of type (int) return a value of -1 for failure, and return
   a value of 0 to indicate success. */

void usage()
	{
	printf("Usage: %s [-v] [-d [{n}]] [-x] [-f {device file}] "
										"[-l [{logfile}]]\n", gi.progname);
	printf("  -h  - Print usage message and exit.\n");
	printf("  -v  - Print version and exit.\n");
	printf("  -d  - Set debugging bits to n. (Default = 3)\n");
	printf("  -x  - Do not fork and send log info to stdout.\n");
	printf("  -l  - Send log info to {logfile}.\n");
	printf("Notes: -l and -x switches overide each other.\n");
    printf("Set debugging level to 127 for maximum information.\n");
	exit(0);
	}

int main(argc, argv)
 int argc;
 char *argv[];
	{
	int ret;
	char *e;

	ResetGlobalInfo();
	gi.progname = *argv;
	argc--;
	argv++;
	while(argc)
		{
		if ( **argv != '-' )
			usage();
		switch( *(*argv + 1) )
			{
			case 'v':
				printf("%s", Version());
				exit(0);
			case 'x':
				gi.dout = stdout;
				break;
			case 'd':
				if ( argc > 1 )
					{
					if ( (Debugf = (int)strtol(*(argv + 1), NULL, 0)) > 0 )
						{
						argc--;
						argv++;
						break;
						}
					}
				Debugf = 3;
				break;
			case 'h':
				usage();
			case 'l':
				if ( (argc > 1) && (**(argv + 1) != '-') )
					{
					argc--;
					argv++;
					strcpy(gi.log_fn, *argv);
					}
				gi.do_log = 1;
				break;
			case 'f':
				argc--;
				argv++;
				if ( argc == 0 )
					usage();
				strcpy(gi.device_fn, *argv);
				break;
			default:
				usage();
			}
		argc--;
		argv++;
		}
	if ( gi.dout != stdout )
		{
		if ( (ret = fork()) == -1 )
			{
			fprintf(stderr, "%s: Error: fork failed.\n", gi.progname);
			exit(-1);
			}
		else if ( ret > 0 )
			exit(0);
		}
	signal(SIGHUP, sig_hup);
	signal(SIGINT, sig_fatal);
	signal(SIGQUIT, sig_fatal);
	signal(SIGILL, sig_fatal);
	signal(SIGTRAP, sig_fatal);
	signal(SIGIOT, sig_fatal);
	signal(SIGFPE, sig_fatal);
	signal(SIGTERM, sig_fatal);
	if ( GeneralSetup() == -1 )
		{
		e = "Main: GeneralSetup failed. Exiting...\n";
		if ( gi.dout != NULL )
			Debug(e);
		else
			fprintf(stderr, "%s", e);
		}
	else
		PipeLoop();
	return(-1);
	}

int GeneralSetup()
	{
	char tmp[256];
	PMLIST *pmcur;
	int cnt, ret = -1;

	if ( Pmfirst != NULL )
		free_pmlist();
	if ( (time(NULL) < (gi.last_hup + 60))
	  || (cycle_logfile() != -1) )
		{
		Debug(Version());
		sprintf(tmp, "Debug level = 0x%02X\n", Debugf);
		Debug(tmp);
		sprintf(tmp, "Loading Devices from: %s\n", gi.device_fn);
		Debug(tmp);
		if ( (cnt = LoadDevices()) == -1 )
			Debug("GeneralSetup: LoadDevices failed.\n");
		else if ( cnt == 0 )
			Debug("GeneralSetup: No Devices loaded.\n");
		else
			{
			sprintf(tmp, "Loaded %d devices:\n", cnt);
			Debug(tmp);
			pmcur = Pmfirst;
			while( pmcur != NULL )
				{
				sprintf(tmp, "%s [%s, Port %d]", pmcur->ttyname,
												pmcur->ipaddr, pmcur->port);
				if ( pmcur->pin == 1 )
					strcat(tmp, " Pinned UP.");
				if ( pmcur->security == 1 )
					sprintf(tmp + strlen(tmp), " Security ON: U:%s P:%s",
										pmcur->username, pmcur->password);
				strcat(tmp, "\n");
				Debug(tmp);	
				pmcur = pmcur->next;
				}
			if ( (ret = CreatePTYs()) == -1 )
				Debug("GeneralSetup: CreatePTYs failed.\n");
			else
				ConnectPinnedNETfds();
			}
		}
	gi.restart = 0;
	gi.last_hup = 0;
	return(ret);
	}

void sig_fatal(sig)
 int sig;
	{
	char tmp[256];

	sprintf(tmp, "Exiting on signal: %d\n", sig);
	Debug(tmp);
	free_pmlist();
	exit(1);
	}

void sig_hup(sig)
 int sig;
	{
	sigset_t set;

	sigemptyset(&set);
	sigaddset(&set, sig);
	sigprocmask(SIG_BLOCK, &set, NULL);
	if ( time(NULL) < (gi.last_hup + 60) )
		gi.restart = 1;
	else
		{
		cycle_logfile();
		gi.last_hup = time(NULL);
		}
	sigprocmask(SIG_UNBLOCK, &set, NULL);
	}

void Debug(txt)
 char *txt;
	{
	char tmp[32];
	struct timeval tmv;

	if ( gi.dout != NULL )
		{
		if ( gettimeofday(&tmv, NULL) == -1 )
			{
			time(&tmv.tv_sec);
			tmv.tv_usec = 0L;
			}
		strftime(tmp, 32, "%D %T", localtime(&tmv.tv_sec));
		fprintf(gi.dout, "[%s.%03ld] %s", tmp, tmv.tv_usec / 1000, txt);
		fflush(gi.dout);
		}
	}

void ADebug(txt)
 char *txt;
	{
	if ( gi.dout != NULL )
		{
		fprintf(gi.dout, "%s", txt);
		fflush(gi.dout);
		}
	}

/* Returns -1 for error, 0 for no devices loaded, or 1 though N for
   number of devices loaded. */

int LoadDevices()
	{
	FILE *f;
	char tmp[256], *p;
	int l, ret, num = 0, lnum = 0, pin, security;
	char s[256], s1[32], s2[32], s3[32], s4[32], s5[32], s6[32], s7[32];

	if ( (f = fopen(gi.device_fn, "rt")) == NULL )
		{
		sprintf(tmp, "LoadDevices: Error opening %s for read\n", gi.device_fn);
		Debug(tmp);
		return(-1);
		}
	while( fgets(s, 256, f) != NULL )
		{
		lnum++;
		if ( (strlen(s) < 5) || (*s == '#') || (*s == ';') )
			continue;

		if ( (s[l = (strlen(s) - 1)] == '\r') || (s[l] == '\n') )
			s[l] = '\0';

		ret = sscanf(s, "%s %s %s %s %s %s %s", s1, s2, s3, s4, s5, s6, s7);
		if ( (ret != 5) && (ret != 7) )
			{
			sprintf(tmp, "LoadDevices: Parse error, line %d: Not enough "
											"arguments! Skipping\n", lnum);
			Debug(tmp);
			continue;
			}
		if ( strncmp(s1, "/dev/", 5) )
			{
			sprintf(tmp, "LoadDevices: Parse error, line %d: Device does not"
									" start with /dev/...Skipping\n", lnum);
			Debug(tmp);
			continue;
			}
		if ( (p = ResolveDNS(s4)) == NULL )
			{
			sprintf(tmp, "LoadDevices: Parse error, line %d: Unable to "
								"resolve DNS on %s, skipping\n", lnum, s4);
			Debug(tmp);
			continue;
			}
		pin = (tolower(*s2) == 'y') ? 1 : 0;
		security = (tolower(*s3) == 'y') ? 1 : 0;
		if ( (security == 1) && (ret != 7) )
			{
			sprintf(tmp, "LoadDevices: Parse error, line %d: No username/"
						"password specified for security! Skipping\n", lnum);
			Debug(tmp);
			continue;
			}

		if ( (*s6 == '"') && (s6[strlen(s6) - 1] == '"') )
			{
			strcpy(tmp, s6);
			strcpy(s6, tmp + 1);
			s6[strlen(s6) - 1] = '\0';
			}
		if ( (*s7 == '"') && (s7[strlen(s7) - 1] == '"') )
			{
			strcpy(tmp, s7);
			strcpy(s7, tmp + 1);
			s7[strlen(s7) - 1] = '\0';
			}
		if ( ret == 5 )
			*s6 = (*s7 = '\0');
		if ( add_pm(s1, p, atoi(s5), pin, security, s6, s7) == NULL )
			Debug("LoadDevices: Add_pm failed.\n");
		else
			num++;
		}
	fclose(f);
	return(num);
	}

char *ResolveDNS(hostname)
 char *hostname;
	{
	char tmp[256];
	struct hostent *hp;
	char *p = hostname, *ip = NULL;

	for( ;(*p != '\0') && (isdigit(*p) || (*p == '.'));p++) ;

	if ( *p == '\0' )
		ip = hostname;
	else if ( (hp = gethostbyname(hostname)) != NULL )
		ip = inet_ntoa(*((struct in_addr *)hp->h_addr));
	else
		{
		sprintf("ResolveDNS: gethostbyname failed for: %s\n", hostname);
		Debug(tmp);
		}
	return(ip);
	}

/* Read in text from fd until we receive a match in txt[][] */
/* Returns -1 for error, 0 for timeout, 1 through N for match. */

int DoRead(fd, timeout, matches, txt)
 int fd, timeout, matches;
 char txt[MATCH_MAX][MATCH_LEN];
	{
	fd_set fds;
	struct timeval tv;
	int i, pos, t, ret = -1;
	char line[MAXLEN], tmp[256];

	do	{
		*line = '\0';
		pos = -1;
		do	{
			t = timeout;
			do	{
				if ( t-- <= 0 )
					return(0);
				FD_ZERO(&fds);
				FD_SET(fd, &fds);
				tv.tv_sec = 1;
				tv.tv_usec = 0;
				if ( select(fd + 1, &fds, NULL, NULL, &tv) == -1 )
					{
					if ( errno != EINTR )
						{
						Debug("DoRead: Select failed.\n");
						return(-1);
						}
					}
				} while( ! FD_ISSET(fd, &fds) ) ;
			if ( read(fd, &line[++pos], 1) < 1 )
				{
				sprintf(tmp, "DoRead: Read fd: %d failed.\n", fd);
				Debug(tmp);
				return(-1);
				}
			line[pos + 1] = '\0';
			if ( strstr(line, "-- Press Return for More --") != NULL )
				{
				if ( DoWrite(fd, "\n", "") == -1 )
					{
					sprintf(tmp, "DoRead: Write fd: %d failed.\n", fd);
					Debug(tmp);
					return(-1);
					}
				break;
				}
			for(i = 0;i < matches;i++)
				{
				if ( strstr(line, txt[i]) != NULL )
					{
					ret = i + 1;
					break;
					}
				}
			} while( (ret == -1)
					&& (line[pos] != '\n') && (line[pos] != '\r') ) ;
		if ( Debugf & 0x0008 )
			{
			sprintf(tmp, "DoRead: %d: '%s'\n", fd, fmtstr(line, 30));
			Debug(tmp);
			}
		} while ( ret == -1 );
	return(ret);
	}

int DoWrite(fd, txt1, txt2)
 int fd;
 char *txt1, *txt2;
	{
	char tmp[256], tmp2[256];

	sprintf(tmp, "%s%s", txt1, txt2);
	if ( write(fd, tmp, strlen(tmp)) != strlen(tmp) )
		{
		sprintf(tmp2, "DoWrite: write fd: %d failed.\n", fd);
		Debug(tmp2);
		return(-1);
		}
	if ( Debugf & 0x0008 )
		{
		sprintf(tmp2, "DoWrite: %d: '%s'\n", fd, fmtstr(tmp, 30));
		Debug(tmp2);
		}
	return(0);
	}

void ConnectPinnedNETfds()
	{
	int pinned = 0;
	PMLIST *pmcur = Pmfirst;

	Debug("Creating and connecting PINNED socket connections:\n");
	while ( pmcur != NULL )
		{
		if ( pmcur->pin == 1 )
			{
			pinned = 1;
			ConnectPM(pmcur);
			}
		pmcur = pmcur->next;
		}
	if ( pinned == 0 )
		Debug("No PINNED socket connections were requested.\n");
	}

void ConnectPM(pmcur)
 PMLIST *pmcur;
	{
	int ret;
	char tmp[256];
	struct sockaddr_in pmaddr;

	sprintf(tmp, "Connecting (%s) [%s:%d]: ", pmcur->ttyname,
												pmcur->ipaddr, pmcur->port);
	Debug(tmp);
	if ( pmcur->netfd != -1 )
		{
		sprintf(tmp, "ConnectPM: Already Connected! (%s) Socket fd: %d\n",
											pmcur->ttyname, pmcur->netfd);
		ADebug(tmp);
		}
	else if ( (pmcur->netfd = socket(AF_INET, SOCK_STREAM, 0)) == -1 )
		{
		strcpy(tmp, "Failed!\n");
		ADebug(tmp);
		time(&pmcur->lt);
		}
	else
		{
		sprintf(tmp, "Socket fd: %d: ", pmcur->netfd);
		ADebug(tmp);
		pmaddr.sin_family = AF_INET;
		pmaddr.sin_addr.s_addr = inet_addr(pmcur->ipaddr);
		pmaddr.sin_port = htons(pmcur->port);
		if ( connect(pmcur->netfd, (struct sockaddr *)&pmaddr,
													sizeof(pmaddr)) == -1 )
			{
			strcpy(tmp, "Failed!\n");
			ADebug(tmp);
			DisconnectPM(pmcur, NULL, 0, 0);
			}
		else
			{
			ADebug(" Connected.");
			pmcur->net_defer = 0;
			if ( set_net_nonblocking(pmcur, 1) == -1 )
				{
				Debug("\nConnectPM: SetNetNonblocking failed.\n");
				DisconnectPM(pmcur, NULL, 0, 0);
				}
			else if ( pmcur->security != 1 )
				ADebug("\n");
			else
				{
				ADebug(" Logging on..\n");
				if ( (ret = SecureLogin(pmcur)) == 4 )
					{
					sprintf(tmp, "Login Successful! (%s) [%s:%d]\n",
								pmcur->ttyname, pmcur->ipaddr, pmcur->port);
					Debug(tmp);
					}
				else
					{
					sprintf(tmp, "Login Failed! (%s) [%s:%d]: ",
								pmcur->ttyname, pmcur->ipaddr, pmcur->port);
					switch ( ret )
						{
						case -1:
							sprintf(tmp + strlen(tmp), "errno: %d\n", errno);
							break;
						case 0:
							strcat(tmp, "Timed Out.\n");
							break;
						case 1:
							strcat(tmp, "Received Prompt.\n");
							break;
						case 2:
							strcat(tmp, "Invalid Login.\n");
							break;
						case 3:
							strcat(tmp, "No Available Ports.\n");
							break;
						}
					Debug(tmp);
					DisconnectPM(pmcur, NULL, 0, 0);
					return;
					}
				}
			}
		}
	}

void DisconnectPM(pmcur, logstr, logmode, defer)
 PMLIST *pmcur;
 char *logstr;
 int logmode, defer;
	{
	char tmp[256] = {'\0'};

	if ( pmcur->netfd != -1 )
		{
		if ( logstr != NULL )
			{
			if ( logmode == 0 )
				sprintf(tmp, "Disconnecting (%s) [%s:%d]: %s.",
						pmcur->ttyname, pmcur->ipaddr, pmcur->port, logstr);
			else
				sprintf(tmp, "%s Disconnect: (%s) [%s:%d]: Connection "
								"Closed by foreign Host.", logstr,
								pmcur->ttyname, pmcur->ipaddr, pmcur->port);
			}
		if ( (defer == 1) && (pmcur->to_cnt != 0) )
			{
			if ( logstr != NULL )
				{
				strcat(tmp, " (Deferred)\n");
				Debug(tmp);
				}
			pmcur->net_defer = 1;
			}
		else
			{
			if ( logstr != NULL )
				{
				strcat(tmp, "\n");
				Debug(tmp);
				}
			close(pmcur->netfd);
			pmcur->netfd = -1;
			pmcur->net_defer = 0;
			time(&pmcur->lt);
			flush_to_buf(pmcur);
			}
		}
	}

void cycle_pty(pmcur, defer)
 PMLIST *pmcur;
 int defer;
	{
	char tmp[256];

	if ( pmcur->ptyfd != -1 )
		close_pty(pmcur, defer);
	if ( (pmcur->ptyfd == -1) && (getpty(pmcur) == -1) )
		{
		sprintf(tmp,"Error: (%s): Unable to re-create pty.\n", pmcur->ttyname);
		Debug(tmp);
		}
	}

void close_pty(pmcur, defer)
 PMLIST *pmcur;
 int defer;
	{
	char tmp[256] = {'\0'};

	if ( pmcur->ptyfd != -1 )
		{
		sprintf(tmp, "Closing PTYfd [%d], for (%s) (%s)",
							pmcur->ptyfd, pmcur->ttyname, pmcur->ptyname);
		if ( (defer == 1) && (pmcur->from_cnt != 0) )
			{
			strcat(tmp, " (Deferred)\n");
			Debug(tmp);
			pmcur->pty_defer = 1;
			}
		else
			{
			strcat(tmp, "\n");
			Debug(tmp);
			/* Do a 0 length write to signal a close to the slave side. */
			if ( (pmcur->open == 1) && (write(pmcur->ptyfd, tmp, 0) == -1) )
				{
				sprintf(tmp, "(%s) [%d]: EOF write to pty failed. Errno "
							"= %d\n", pmcur->ptyname, pmcur->ptyfd, errno);
				Debug(tmp);
				}
			close(pmcur->ptyfd);
			pmcur->ptyfd = -1;
			pmcur->open = 0;
			pmcur->pty_defer = 0;
			pmcur->pty_delay = 0;
			flush_from_buf(pmcur);
			}
		}
	}

/* Returns -1 for error, 0 for timeout, 1 through 3 for failures,
   and 4 for success. */

int SecureLogin(pmcur)
 PMLIST *pmcur;
	{
	int ret;
	char txt[MATCH_MAX][MATCH_LEN], tmp[MAXLEN];

	strcpy(txt[0], "ogin: ");
	if ( (ret = set_net_nonblocking(pmcur, 0)) == -1 )
		Debug("SecureLogin: SetNetNonblocking CLR failed.\n");
	else if ( (ret = DoRead(pmcur->netfd, 5, 1, txt)) != 1 )
		Debug("SecureLogin: DoRead #1 failed.\n");
	else if ( (ret = DoWrite(pmcur->netfd, pmcur->username, "\n")) == -1 )
		Debug("SecureLogin: DoWrite #1 failed.\n");
	else
		{
		strcpy(txt[0], "assword: ");
		if ( (ret = DoRead(pmcur->netfd, 5, 1, txt)) != 1 )
			Debug("SecureLogin: DoRead #2 failed.\n");
		else if ( (ret = DoWrite(pmcur->netfd, pmcur->password, "\n")) == -1 )
			Debug("SecureLogin: DoWrite #2 failed.\n");
		else
			{
			strcpy(txt[0], ">");
			strcpy(txt[1], "Invalid");
			strcpy(txt[2], "No available ports");
			strcpy(txt[3], "Session established");
			if ( (ret = DoRead(pmcur->netfd, 5, 4, txt)) < 1 )
				Debug("SecureLogin: DoRead #3 failed.\n");
			else if ( ret == 4 )
				{
				/* TODO: error trap this read. WHY IS THIS DONE??? */
				/* This is an artifact from the old code. I have no
				   idea why this particular piece of voodoo is here.
				   I've left it in primarily from a "if it works, don't
				   fuck with it" point of view. It's probably to drain
				   garbage from the input stream. I don't like it being
				   here because it could block and hang the whole daemon.
				   Most likely it could be removed.
				*/
				read(pmcur->netfd, tmp, sizeof(tmp));
				}
			}
		}
	if ( (ret != -1) && ((ret = set_net_nonblocking(pmcur, 1)) == -1) )
		Debug("SecureLogin: SetNetNonblocking SET failed.\n");
	return(ret);
	}

void free_pmlist()
	{
	PMLIST *pmcur = Pmlast;

	while ( pmcur != NULL )
		{
		if ( pmcur->netfd != -1 )
			DisconnectPM(pmcur, "Restart/Close", 0, 0);
		if ( pmcur->ptyfd != -1 )
			close_pty(pmcur, 0);
		pmcur = Pmlast->prev;
		free(Pmlast);
		Pmlast = pmcur;
		}
	Pmfirst = NULL;
	}

PMLIST *begin_pmlist()
	{
	if ( (Pmfirst = malloc(sizeof(PMLIST))) == NULL )
		Debug("BeginPmlist: malloc failed.\n");
	else
		{
		Pmfirst->next = NULL;
		Pmfirst->prev = NULL;
		}
	Pmlast = Pmfirst;
	return(Pmfirst);
	}

PMLIST *new_pmlist()
	{
	static PMLIST *pmnew;

	if ( Pmfirst == NULL )
		return(begin_pmlist());
	if ( (pmnew = malloc(sizeof(PMLIST))) == NULL )
		Debug("NewPmlist: malloc failed.\n");
	else
		{
		pmnew->next = NULL;
		pmnew->prev = Pmlast;
		Pmlast->next = pmnew;
		Pmlast = pmnew;
		}
	return(pmnew);
	}

PMLIST *add_pm(ttyname, ipaddr, port, pin, security, username, password)
 char *ttyname, *ipaddr;
 int port, pin, security;
 char *username, *password;
	{
	char tmp[256];
	PMLIST *newpm;

	if ( (newpm = new_pmlist()) == NULL )
		{
		sprintf(tmp, "AddPm: NewPmlist (%s) failed.\n", ttyname);
		Debug(tmp);
		}
	else
		{
		strcpy(newpm->ttyname, ttyname);
		strcpy(newpm->ipaddr, ipaddr);
		strcpy(newpm->username, username);
		strcpy(newpm->password, password);
		newpm->port = port;
		newpm->pin = pin;
		newpm->security = security;
		strcpy(newpm->ptyname, "");
		newpm->ptyfd = -1;
		newpm->netfd = -1;
		newpm->to_cnt = 0;
		newpm->from_cnt = 0;
		newpm->lt = 0;
		newpm->la = 0;
		newpm->net_defer = 0;
		newpm->pty_defer = 0;
		newpm->pty_delay = 0;
		newpm->open = 0;
		}
	return(newpm);
	}

int CreatePTYs()
	{
	int ret = 0;
	char tmp[256];
	PMLIST *pl1, *pl2, *pmcur = Pmfirst;

	while ( pmcur != NULL )
		{
		if ( getpty(pmcur) != -1 )
			pmcur = pmcur->next;
		else
			{
			sprintf(tmp, "CreatePTYs: (%s) (%s): Failed! "
							"Removing...\n", pmcur->ttyname, pmcur->ptyname);
			Debug(tmp);

			/* Unlink the pty descriptor out of the list. */
			pl1 = pmcur->prev;
			pl2 = pmcur->next;
			if ( pl1 != NULL )
				pl1->next = pl2;
			if ( pl2 != NULL )
				pl2->prev = pl1;
			if ( pmcur == Pmfirst )
				Pmfirst = pl2;
			if ( pmcur == Pmlast )
				Pmlast = pl1;
			free(pmcur);
			pmcur = pl2;
			}
		}
	if ( Pmfirst == NULL )
		{
		Debug("CreatePTYs: All Ptys failed to create.\n");
		ret = -1;
		}
	return(ret);
	}

int getpty(pmcur)
 PMLIST *pmcur;
	{
	char tmp[256];
	char *ptyname;
	int	ret = -1;
#if (! defined(OS_LINUX)) || defined(OS_HPUX)
	int on = 1;
#endif

	if ( (ptyname = findpty(pmcur->ttyname)) == NULL )
		{
		sprintf(tmp, "Getpty: findpty (%s) failed.\n", pmcur->ttyname);
		Debug(tmp);
		}
	else
		{
		strcpy(pmcur->ptyname, ptyname);
		if ( (pmcur->ptyfd = open(pmcur->ptyname, O_RDWR | O_NOCTTY)) == -1 )
			{
			sprintf(tmp, "Getpty: Open failed: %s, errno: %d\n",
												pmcur->ptyname, errno);
			Debug(tmp);
			}
		else
			{
			/* Set modes */
			if ( (ret = set_fd_nonblocking(pmcur->ptyfd, 1)) == -1 )
				Debug("Getpty: SetFdNonblocking failed.\n");
#ifndef OS_LINUX
			else if ( (ret = ioctl(pmcur->ptyfd, TIOCREMOTE, &on)) == -1 )
				{
				sprintf(tmp, "Getpty: ioctl fd: %d TIOCREMOTE failed.\n",
																pmcur->ptyfd);
				Debug(tmp);
				}
#endif
#ifdef OS_HPUX
			/* Gregc: This ioctl is from the old code. I have no idea if
			   it's needed or not. Someone with access to HPUX will need
			   to test this. */
			else if ( (ret = ioctl(pmcur->ptyfd, TIOCMONITOR, &on)) == -1 )
				{
				sprintf(tmp, "Getpty: ioctl fd: %d TIOCMONITOR failed.\n",
																pmcur->ptyfd);
				Debug(tmp);
				}
#endif
			if ( ret == -1 )
				{
				close(pmcur->ptyfd);
				pmcur->ptyfd = -1;
				}
			else
				{
				pmcur->pty_defer = 0;
				sprintf(tmp, "Created PTYfd [%d], for (%s) (%s)\n",
							pmcur->ptyfd, pmcur->ttyname, pmcur->ptyname);
				Debug(tmp);
				}
			}
		}
	return(ret);
	}

char *findpty(ttyname)
 char *ttyname;
	{
	struct stat	stb;
	dev_t device_type;
	static char device[256];

	/*
	 * The device will come in using the form:
	 * 	"/dev/ttyp1"
	 * this needs to be converted to the "pty" format name
	 * which represents the other side of the psuedo device.
	 * If we don't find it using this simple conversion, then
	 * we will look for the corresponding pty using readdir in /dev
	 */

	if ( strncmp(ttyname, "/dev/tty", 8) == 0 )
		{
		strcpy(device, ttyname);
		device[5] = 'p';
		if ( stat(device, &stb) == 0 )
			return(device);
		}

#if defined(OS_AIX)
	/* Try the AIX form of multiplexed ttys/ptys */
	if ( strncmp(ttyname, "/dev/pty/", 9) == 0 )
		{
		strcpy(device, ttyname);
		device[7] = 'c';
		if ( stat(device, &stb) == 0 )
			return(device);
		}
#endif

	/* Use the more involved process */
	if ( (stat(ttyname, &stb) != -1)
	  && ((stb.st_mode & S_IFMT) == S_IFCHR)
	  && (major(stb.st_rdev) == MAJOR_TTY) )
		{
		device_type = makedev(MAJOR_PTY, minor(stb.st_rdev));
		return(dev2pty(device_type));
		}

	return(NULL);
	}

char *dev2pty(devno)
 dev_t devno;
	{
	DIR *dirp;
	struct stat stb;
	struct direct *dp;
	static char dbuf[256];
	char *dev = NULL, tmpname[256], tmp[256];

	if ( (dirp = opendir("/dev")) == NULL )
		Debug("Dev2pty: Unable to open directory '/dev' for reading.\n");
	else
		{
		while ( (dp = readdir(dirp)) != NULL )
			{
#if defined(OS_LINUX)
			strncpy(tmpname, dp->d_name, dp->d_reclen);
			tmpname[dp->d_reclen] = '\0';
#else
			strncpy(tmpname, dp->d_name, dp->d_namlen);
			tmpname[dp->d_namlen] = '\0';
#endif
			sprintf(dbuf, "/dev/%s", tmpname);
			if ( (stat(dbuf, &stb) == -1)
			  || ((stb.st_mode & S_IFMT) != S_IFCHR) )
				continue;
			if ( stb.st_rdev == devno )
				{
				dev = dbuf;
				break;
				}
			}
		if ( dev == NULL )
			{
			sprintf(tmp, "Dev2pty: No pty for tty minor device#: %d\n",
																minor(devno));
			Debug(tmp);
			}
		closedir(dirp);
		}
	return(dev);
	}

/* TODO:
X  1.) When a tty close is detected, the disconnect from the corresponding
       portmaster socket should be deferred if there is still data in the
       "to" buffer. Once the buffer drains, then close the connection.
       This feels like the right thing to do, but in practice it may not
       actually be necessary. (COMPLETE)

X  2.) Possibly the same thing should be done when a disconnect is detected
       from the portmaster. Let the "from" buffer drain before the pty is
       closed and re-created. Again, this may not be necessary. (COMPLETE)

   3.) Currently, pinned connections are checked for incoming data, even
       if their corresponding tty is closed. In these cases, the received
       data is just tossed. There have been reports, with the old version
       of pmnetd, that an offline printer will cause pmnetd to go into
       "cpu hog" mode. That condition could be caused by a continous stream
       of NULs or other characters from the printer's interface. If that
       occurs, the problem could be fixed by either running the printer
       on an "un-pinned" connection, or by uncommenting the line of code
       below that only sets the select() bit for the netfd when the
       connection is considered to be in an "open" condition. This is
       speculation on my part, as I have no serial printer handy to run
       a test with.

       (On the other hand, the "cpu hog" problem could have been the
       result of all the other bad code that's been fixed. So the problem
       may not actually exist anymore.)
*/

void PipeLoop()
	{
	PMLIST *pmcur;
	struct timeval timeout;
	fd_set readfds, writefds;
	int status, len, alen, nds;
	char *p, tmp[256], btmp[BUF_SIZE];

	while ( 1 )
		{
		if ( gi.restart == 1 )
			{
			Debug("PipeLoop: Received HUP signal, Restarting...\n");
			if ( GeneralSetup() == -1 )
				{
				Debug("PipeLoop: GeneralSetup failed. Exiting...\n");
				sig_fatal(400);
				}
			}

		/* Handle setup for each loop. */
		nds = 0;
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		pmcur = Pmfirst;
		while ( pmcur != NULL )
			{
			/* Note: It may seem like the following 2 if() statements and
			   their contents could be replaced by a call to cycle_pty(),
			   but for various subtle reasons, I want to leave these here.
			   These are meant to speciffically handle deferred close()'s,
			   and catch failed re-open()'s. */
			if ( (pmcur->ptyfd != -1)
			  && (pmcur->pty_defer == 1) && (pmcur->from_cnt == 0) )
				close_pty(pmcur, 0);
			if ( (pmcur->ptyfd == -1) && (getpty(pmcur) == -1) )
				{
				sprintf(tmp, "Error: (%s): Unable to "
										"re-create pty.\n", pmcur->ttyname);
				Debug(tmp);
				}
			if ( pmcur->netfd != -1 )
				{
				if ( (pmcur->net_defer == 1) && (pmcur->to_cnt == 0) )
					DisconnectPM(pmcur, "Deferred disconnect", 0, 0);
				else if ( (pmcur->open == 0)
				  && (pmcur->pin != 1) && (pmcur->to_cnt == 0)
				  && (time(NULL) > (pmcur->la + IDLE_TIMEOUT)) )
					DisconnectPM(pmcur, "Idle Timeout", 0, 0);
				}
			else
				{
				if ( ((pmcur->pin == 1) || (pmcur->to_cnt > 0))
				  && (time(NULL) > (pmcur->lt + CONNECT_TIME)) )
					ConnectPM(pmcur);
				}

			/* Setup select() fd lists. */
			if ( pmcur->ptyfd != -1 )
				{
				nds = max(nds, pmcur->ptyfd);
				FD_SET(pmcur->ptyfd, &readfds);
				if ( pmcur->from_cnt > 0 )
					FD_SET(pmcur->ptyfd, &writefds);
				}
			if ( pmcur->netfd != -1 )
				{
				/* Uncommenting this line might solve "cpu hog" problems. */
				/* if ( pmcur->open == 1 ) */
					{
					nds = max(nds, pmcur->netfd);
					FD_SET(pmcur->netfd, &readfds);
					}
				if ( pmcur->to_cnt > 0 )
					{
					nds = max(nds, pmcur->netfd);
					FD_SET(pmcur->netfd, &writefds);
					}
				}
			pmcur = pmcur->next;
			}

		/* Wait for I/O to occur or 2 second timeout. */
		timeout.tv_sec = 2;
		timeout.tv_usec = 0;
		if ( (status = 
				select(nds + 1, &readfds, &writefds, NULL, &timeout)) == -1)
			{
			if ( errno != EINTR )
				{
				Debug("PipeLoop: Select failed!\n");
				sig_fatal(500);
				}
			continue;
			}

		/* Just a timeout, do another loop. */
		if ( status == 0 )
			continue;

		/* Handle all write()'s first. */
		pmcur = Pmfirst;
		while( pmcur != NULL )
			{
			/* Writes to portmaster. */
			if ( (pmcur->to_cnt > 0) && (pmcur->netfd != -1)
			  && FD_ISSET(pmcur->netfd, &writefds) )
				{
				time(&pmcur->la);
				alen = pmcur->to_cnt;
				if ( Debugf & 0x0004 )
					{
					sprintf(tmp, "Starting write() on netfd: %d\n",
															pmcur->netfd);
					Debug(tmp);
					}
				if ( (len = write(pmcur->netfd, pmcur->to_buf, alen)) > 0 )
					{
					if ( Debugf & 0x0002 )
						{
						sprintf(tmp, "(%s): Wrote %d bytes on net fd: %d\n",
										pmcur->ttyname, len, pmcur->netfd);
						Debug(tmp);
						}
					if ( Debugf & (0x0020 | 0x0040) )
						log_data(pmcur, len, 0, 1);
					if ( len < alen )
						{
						memcpy(btmp, pmcur->to_buf + len, alen - len);
						memcpy(pmcur->to_buf, btmp, alen - len);
						}
					pmcur->to_cnt -= len;
					}
				else
					{
					FD_CLR(pmcur->netfd, &readfds);
					DisconnectPM(pmcur, "Detected", 1, 0);
					/* Also close the pty to signal to the client that
					   the remote device has closed/disconnected/hungup/
					   whatever. Do a deferred close, so the incoming
					   buffer has a chance to drain. */
					close_pty(pmcur, 1);
					}
				}

			/* Writes to pty. */
			if ( (pmcur->from_cnt > 0) && (pmcur->ptyfd != -1)
			  && FD_ISSET(pmcur->ptyfd, &writefds) )
				{
				if ( pmcur->open != 1 )
					{
					sprintf(tmp, "%s not open. Discarding data.\n",
															pmcur->ttyname);
					Debug(tmp);
					flush_from_buf(pmcur);
					}
				else
					{
					time(&pmcur->la);
					alen = pmcur->from_cnt;
					if ( Debugf & 0x0004 )
						{
						sprintf(tmp, "Starting write() on ptyfd: %d\n",
															pmcur->ptyfd);
						Debug(tmp);
						}
					if ( (len = write(pmcur->ptyfd,
												pmcur->from_buf, alen)) > 0 )
						{
						if ( Debugf & 0x0002 )
							{
							sprintf(tmp, "(%s): Wrote %d bytes on pty fd:"
								" %d\n", pmcur->ttyname, len, pmcur->ptyfd);
							Debug(tmp);
							}
						if ( Debugf & (0x0020 | 0x0080) )
							log_data(pmcur, len, 1, 1);
						if ( len < alen )
							{
							memcpy(btmp, pmcur->from_buf + len, alen - len);
							memcpy(pmcur->from_buf, btmp, alen - len);
							}
						pmcur->from_cnt -= len;
						}
					else
						{
						pmcur->open = 0;
						p = "W: Detected TTY Close";
						if ( (pmcur->pin == 0) && (pmcur->netfd != -1) )
							{
							/* Pty closes should be propagated through
							   to a portmaster disconnect so that the pm
							   will properly hang up modems. For this reason,
							   if modems are used, pinned-up mode should
							   probably *not* be used. Close is done in
							   deferred mode so outgoing buffer can drain. */
							DisconnectPM(pmcur, p, 0, 1);
							}
						else
							{
							sprintf(tmp, "(%s) [%d]: %s\n",
											pmcur->ttyname, pmcur->ptyfd, p);
							Debug(tmp);
							}
						FD_CLR(pmcur->ptyfd, &readfds);
						/* This close is necessary in order to clear the
						   end-of-file condition on the pty. */
						cycle_pty(pmcur, 0);
						}
					}
				}
			pmcur = pmcur->next;
			}

		/* Then do all read()'s. */
		pmcur = Pmfirst;
		while( pmcur != NULL )
			{
			/* Reads from portmaster. */
			if ( (pmcur->netfd != -1) && FD_ISSET(pmcur->netfd, &readfds) )
				{
				time(&pmcur->la);
				if ( (alen = BUF_SIZE - pmcur->from_cnt) > 0 )
					{
					if ( Debugf & 0x0004 )
						{
						sprintf(tmp, "Starting read() on netfd: %d\n",
															pmcur->netfd);
						Debug(tmp);
						}
					if ( (len = read(pmcur->netfd,
							pmcur->from_buf + pmcur->from_cnt, alen)) > 0 )
						{
						if ( Debugf & 0x0002 )
							{
							sprintf(tmp, "(%s): Read %d bytes on net fd: %d\n",
										pmcur->ttyname, len, pmcur->netfd);
							Debug(tmp);
							}
						if ( Debugf & (0x0010 | 0x0040) )
							log_data(pmcur, len, 0, 0);
						pmcur->from_cnt += len;
						if ( pmcur->open != 1 )
							{
							if ( Debugf & 0x0001 )
								{
								sprintf(tmp, "%s not open. Discarding "
												"data.\n", pmcur->ttyname);
								Debug(tmp);
								}
							flush_from_buf(pmcur);
							}
						}
					else
						{
						DisconnectPM(pmcur, "Received", 1, 0);
						/* Also close the pty to signal to the client that
						   the remote device has closed/disconnected/hungup/
						   whatever. Do a deferred close, so the incoming
						   buffer has a chance to drain. */
						close_pty(pmcur, 1);
						}
					}
				}

			/* Reads from pty. */
			if ( (pmcur->ptyfd != -1) && FD_ISSET(pmcur->ptyfd, &readfds) )
				{
				time(&pmcur->la);
				if ( pmcur->open != 1 )
					{
					sprintf(tmp, "(%s) [%d]: Detected TTY Open\n",
											pmcur->ttyname, pmcur->ptyfd);
					Debug(tmp);
					pmcur->open = 1;
					pmcur->pty_delay = 20;
					}
				if ( (alen = BUF_SIZE - pmcur->to_cnt) > 0 )
					{
					if ( Debugf & 0x0004 )
						{
						sprintf(tmp, "Starting read() on ptyfd: %d\n",
															pmcur->ptyfd);
						Debug(tmp);
						}

					/* Deep Voodoo Warning!
					   This little Kludge-O-Rama is present to help
					   hylafax along. It's not absolutely required, but
					   it helps prevent a timeout during the initial modem
					   setup. I suspect that the length of the sleep is
					   unimportant, and that any value large enough to force
					   a context switch is all that is required.

					   This hack was discovered by the fact that when
					   the 0x0004 Debug message immediately above this
					   comment is turned on, the problem goes away!!!
					   (ghu save me from timing dependent code...)
					   See the README file for more about this.
					*/
					if ( pmcur->pty_delay > 0 )
						{
						usleep(1000);
						pmcur->pty_delay--;
						}

					if ( (len = read(pmcur->ptyfd,
								pmcur->to_buf + pmcur->to_cnt, alen)) > 0 )
						{
						if ( Debugf & 0x0002 )
							{
							sprintf(tmp, "(%s): Read %d bytes on pty fd: %d\n",
										pmcur->ttyname, len, pmcur->ptyfd);
							Debug(tmp);
							}
						if ( Debugf & (0x0010 | 0x0080) )
							log_data(pmcur, len, 1, 0);
						pmcur->to_cnt += len;
						}
					else
						{
						pmcur->open = 0;
						p = "R: Detected TTY Close";
						if ( (pmcur->pin == 0) && (pmcur->netfd != -1) )
							{
							/* Pty closes should be propagated through
							   to a portmaster disconnect so that the pm
							   will properly hang up modems. For this reason,
							   if modems are used, pinned-up mode should
							   probably *not* be used. Close is done in
							   deferred mode so outgoing buffer can drain. */
							DisconnectPM(pmcur, p, 0, 1);
							}
						else
							{
							sprintf(tmp, "(%s) [%d]: %s\n",
											pmcur->ttyname, pmcur->ptyfd, p);
							Debug(tmp);
							}
						/* This close is necessary in order to clear the
						   end-of-file condition on the pty. */
						cycle_pty(pmcur, 0);
						}
					}
				}
			pmcur = pmcur->next;
			}
		}
	}

void flush_to_buf(pmcur)
 PMLIST *pmcur;
	{
	char tmp[256];

	if ( (pmcur != NULL) && (pmcur->to_cnt > 0) )
		{
		if ( Debugf & 0x0001 )
			{
			sprintf(tmp, "(%s) Flushing TO buffer: %d chars.\n",
											pmcur->ttyname, pmcur->to_cnt);
			Debug(tmp);
			}
		pmcur->to_cnt = 0;
		}
	}

void flush_from_buf(pmcur)
 PMLIST *pmcur;
	{
	char tmp[256];

	if ( (pmcur != NULL) && (pmcur->from_cnt > 0) )
		{
		if ( Debugf & 0x0001 )
			{
			sprintf(tmp, "(%s) Flushing FROM buffer: %d chars.\n",
											pmcur->ttyname, pmcur->from_cnt);
			Debug(tmp);
			}
		pmcur->from_cnt = 0;
		}
	}

void ResetGlobalInfo()
	{
	gi.restart = 0;
	gi.do_log = 0;
	gi.progname = NULL;
	gi.dout = NULL;
	gi.last_hup = 0;
	strcpy(gi.device_fn, DEFAULT_DEVICE_FILE);
	strcpy(gi.log_fn, DEFAULT_LOG_FILE);
	}

char *Version()
	{
	static char ver[256];

	sprintf(ver, "Portmaster netdata daemon. Version: %s\n", VERSION);
	return(ver);
	}

char *fmtstr(str, len)
 char *str;
 int len;
	{
	return(fmtdata(str, strlen(str), len));
	}

char *fmtdata(data, dlen, len)
 char *data;
 int dlen, len;
	{
	int cnt = 0;
	static char tmp[512];
	char *t = tmp, *d = data - 1, *end = data + dlen;

	while ( (++d < end) && (cnt < (len - 3)) )
		{
		if ( (*d < 32) || ((unsigned char)*d > 127) )
			{
			if ( (Debugf & 0x8000) == 0 )
				{
				if ( *d == '\n' )
					strcpy(t, "<nl>");
				else if ( *d == '\r' )
					strcpy(t, "<cr>");
				else if ( (*d < 32) || ((unsigned char)*d > 127) )
					sprintf(t, "<%02X>", (unsigned char)*d);
				t += 4;
				cnt += 4;
				}
			}
		else
			{
			*(t++) = *d;
			cnt++;
			continue;
			}
		}
	if ( cnt >= (len - 3) )
		strcpy(t, "...");
	else
		*t = '\0';
	return(tmp);
	}

int set_net_nonblocking(pmcur, mode)
 PMLIST *pmcur;
 int mode;
	{
	return(set_fd_nonblocking(pmcur->netfd, mode));
	}

int set_fd_nonblocking(fd, mode)
 int fd, mode;
	{
	char tmp[256];
	int flags, ret = -1;

	if ( fd != -1 )
		{
		if ( (flags = fcntl(fd, F_GETFL, 0)) == -1 )
			{
			sprintf(tmp, "SetFdNonblocking: fcntl fd: %d GET failed.\n", fd);
			Debug(tmp);
			}
		else
			{
			if ( mode == 1 )
				flags |= O_NONBLOCK;
			else
				flags &= ~O_NONBLOCK;
			if ( (ret = fcntl(fd, F_SETFL, flags)) == -1 )
				{
				sprintf(tmp, "SetFdNonblocking: fcntl fd: %d "
														"SET failed.\n", fd);
				Debug(tmp);
				}
			}
		}
	return(ret);
	}

void log_data(pmcur, len, netpty, rw)
 PMLIST *pmcur;
 int len, netpty, rw;
	{
	char *buf, tmp[256];

	if ( gi.dout != NULL )
		{
		if ( rw == 0 )
			{
			if ( netpty == 0 )
				buf = pmcur->from_buf + pmcur->from_cnt;
			else
				buf = pmcur->to_buf + pmcur->to_cnt;
			}
		else
			{
			if ( netpty == 0 )
				buf = pmcur->to_buf;
			else
				buf = pmcur->from_buf;
			}
		sprintf(tmp, "%d: %s [%d: %s]\n", (netpty == 0) ? pmcur->netfd :
									pmcur->ptyfd, (rw == 0) ? "-->" : "<--",
									len, fmtdata(buf, len, 36));
		Debug(tmp);
		}
	}

int cycle_logfile()
	{
	int x;

	if ( gi.do_log == 1 )
		{
		if ( (gi.dout != NULL) && (gi.dout != stdout) )
			{
			Debug("Cycling logfile...\n");
			fclose(gi.dout);
			}
		x = access(gi.log_fn, F_OK);
		if ( (gi.dout = fopen(gi.log_fn, "at")) != NULL )
			{
			if ( x == -1 )
				Debug("New log file started.\n");
			}
		else
			{
			fprintf(stderr, "%s: Error: Could not open %s for "
										"append!\n", gi.progname, gi.log_fn);
			return(-1);
			}
		}
	return(0);
	}

