#!/bin/perl
#
# A few subroutines for basic but standard dbm support.  Available functions
# are: create, add, del, query, grep, keys, clean, clobber.
# See the functions themselves for usage.
#
# Note that the associative array %DB is destroyed by these subroutines!
#
# The name of the database _must_ be supplied through the environment
# variable 'DBMNAME'.
#
# If an error occurs, a value of zero is returned, and the error type
# is supplied through the environment variable 'DBMERROR'.  See the
# Nota Bene under &dbm'grep about possible confusion with the return
# code when using &dbm'grep.
#
# Revision History:
#
# I altered it slightly to include a debugging script, so that we can see
# what is happening to the database as it happens.  Marius, 1.14.97
#
# Copyright (C) 1997 Random Communications Inc.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#

package dbm;

$logfile	= '/tmp/testlog';
$debug		= 0;

sub d {
  if ( $debug ) {
    open( LOG, ">>$logfile" ) 	|| die;
    print LOG @_;
    close( LOG );
  }
}

sub open{
# Function: Not for direct use as a library function.  Your code should not
#	call this subroutine.
#
   local($dbmfile) = $ENV{'DBMNAME'};

   if(! -e "$dbmfile.pag"){
      $ENV{'DBMERROR'} = "Dbm open failed: No such database '$dbmfile'.";
      return 0;
   }
   if(!(-w "$dbmfile.pag") && !(-w "$dbmfile.dir")){
      $ENV{'DBMERROR'} = "Dbm open failed: '$dbmfile' is unwriteable.";
      return 0;
   }
   if(!dbmopen(%DB, "$dbmfile", undef)){
      $ENV{'DBMERROR'} = "Dbmopen failed: $!";
      return 0;
   }

   $ENV{'DBMERROR'} = "";
   1;
}

sub close{
# Function: Not for direct use as a library function.  Your code should not
#	call this subroutine.
#
   if(! dbmclose(%DB)){
      $ENV{'DBMERROR'} = "Dbmclose failed: $!";
      return 0;
   }

   $ENV{'DBMERROR'} = "";
   1;
}

sub create{
# Function: Creates a database for use with the other dbm functions.  Simply
#	adding an entry to a nonexistent database will not create the
#	database; rather, it will return an error.
# Usage: &dbm'create;
#
   local($dbmfile) = $ENV{'DBMNAME'};

   if(-e "$dbmfile.pag"){
      $ENV{'DBMERROR'} = "Dbm create failed: Database already exists.";
      return 0;
   }
   if(! dbmopen(%DB, "$dbmfile", "0666")){
      $ENV{'DBMERROR'} = "Dbmopen failed: $!";
      return 0;
   }
   &close || return 1;

   $ENV{'DBMERROR'} = "";
   1;
}

sub Add {
  local($status,$key,$val) = (0, $_[0], $_[1]);

  if(!($key && $val)) {
    $ENV{'DBMERROR'} = "Dbm add failed: Missing key or value.";
    return 0;
  }
  &open || return 0;
  if($DB{$key} eq undef) {
    $DB{$key} = $val;
  } else {
    $DB{$key} .= "|$val";
  }
  &close || return 0;

  $ENV{'DBMERROR'} = "";
  1;
}

sub add{
# Function: Adds an entry to the database.
# Usage: &dbm'add(<key>,<value>);
#
   local($status,$key,$value) = (0, $_[0], $_[1]);

   if(!($key && $value)){
      $ENV{'DBMERROR'} = "Dbm add failed: Missing key or value.";
      return 0;
   }
   &open || return 0;
   $DB{$key} = $value;
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   1;
}

sub del{
# Function: Deletes an entry from the database.
# Usage: &dbm'del(<key>);
#
   local($status,$key) = (0, $_[0]);

   if(!$key){
      $ENV{'DBMERROR'} = "Dbm del failed: Missing key.";
      return 0;
   }
   &open || return 0;
   if(!$DB{$key}){
      $ENV{'DBMERROR'} = "Dbm del failed: No such key '$key'.";
      return 0;
   }
   delete $DB{$key};
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   1;
}

sub Query {
  local($status,$key,$val) = (0, $_[0], undef);

  if(! $key) {
    $ENV{'DBMERROR'} = "Dbm query failed: Missing key.";
    return 0;
  }
  &open || return 0;
  $val = $DB{$key};
  &close || return 0;

  $ENV{'DBMERROR'} = "";
  local(@newtmp) = split(/\|/, $val);
  @newtmp;
}

sub query{
# Function: To return the value for the given key.
# Usage: $scalar_variable = &dbm'query(<key>);
#
   local($status,$key,$value) = (0, $_[0], "");

   if(! $key){
      $ENV{'DBMERROR'} = "Dbm query failed: Missing key.";
      return 0;
   }
   &open || return 0;
   $value = $DB{$key};
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   $value;
}

sub grep{
# Function: To find the keys who's values contain the given match term.
# Returns: An array of the keys who's values contain the given match term.
# Usage: @array_var = &dbm'grep(<match term> [,<case flag> [,<match flag>]]);
#	<Case flag>: If <case flag> is nonzero, the match will be case
#		sensitive.  The match is case insensitive by default.
#	<Match flag>: If <match flag> is nonzero, the compared value must
#		be equal to the match term for a match to occur, instead
#		of merely containing the match term (a subset match).
#		Subset matches are the default match type.
# Nota Bene: If no matches occur, &dbm'grep returns a null/zero value, the
#	same as if an error had occurred.  To check if it's an actual error
#	or if there were simply no matches, check the environment variable
#	'DBMERROR', which will be empty in the case of no matches.
#
   local($term,$case_flag,$match_flag) = ($_[0], $_[1], $_[2]);
   local($key,$value,@keys);

   if(! $term){
      $ENV{'DBMERROR'} = "Dbm grep failed: Missing search term.";
      return 0;
   }
   &open || return 0;
   if( !$case_flag && !$match_flag ){
      # Default matching - Case insensitive subset matching.
      while( ($key,$_) = each %DB ){ /$term/i && ($keys[$#keys+1] = $key); }
   }
   elsif( $case_flag && !$match_flag ){
      # Case sensitive subset matching.
      while( ($key,$_) = each %DB ){ /$term/ && ($keys[$#keys+1] = $key); }
   }
   elsif( !$case_flag && $match_flag ){
      # Case insensitive full field matching.
      while( ($key,$_) = each %DB ){ /^$term$/i && ($keys[$#keys+1] = $key); }
   }
   elsif( $case_flag && $match_flag ){
      # Case sensitive full field matching.
      while( ($key,$_) = each %DB ){ /^$term$/ && ($keys[$#keys+1] = $key); }
   }
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   @keys;
}

sub fgrep{
# Function: To search specific fields of the database values.  Often, the
#	values stored in the databases are not single values, but records
#	containing various information.
# Usage: &dbm'fgrep(<search term>,<search field>,<field separator>,
#		[<case flag>,[<match flag>]]);
#	<search field>: The number of the field to be grep'd, from 0 to
#		(number of fields - 1).
#	<field separator>: The character(s) that separate fields in the
#		database record.
#	<case flag>: see &dbm'grep
#	<match flag>: see &dbm'grep
# Nota Bene: See the nota bene for &dbm'grep.
#
   local($term,$field,$separator,$case_flag,$match_flag) = @_;
   local($key,$value,@keys);

   if(!($term && ($field >= 0) && $separator)){
      $ENV{'DBMERROR'} = "Dbm fgrep failed: Missing required parameter.";
      return 0;
   }
   if(!($field =~ /^\d+$/) || ($field < 0)){
      $ENV{'DBMERROR'} = "Dbm fgrep failed: Invalid field index.";
      return 0;
   }
   &open || return 0;

   # This code could be compacted quite a bit but in the interest of speed
   # I wrote it as follows...
   if(!$case_flag && !$match_flag ){
      # Default matching - Case insensitive subset matching.
      while( ($key,$_) = each %DB ){
         @_ = split(/$separator/,$_);
         ($_[$field] =~ /$term/i) && ($keys[$#keys+1] = $key);
      }
   }
   elsif($case_flag && !$match_flag ){
      # Case sensitive subset matching.
      while( ($key,$_) = each %DB ){
         @_ = split(/$separator/,$_);
         ($_[$field] =~ /$term/) && ($keys[$#keys+1] = $key);
      }
   }
   elsif(!$case_flag && $match_flag ){
      # Case insensitive full field matching.
      while( ($key,$_) = each %DB ){
         @_ = split(/$separator/,$_);
         ($_[$field] =~ /^$term$/i) && ($keys[$#keys+1] = $key);
      }
   }
   elsif( $case_flag && $match_flag ){
      # Case sensitive full field matching.
      while( ($key,$_) = each %DB ){
         @_ = split(/$separator/,$_);
         ($_[$field] =~ /^$term$/) && ($keys[$#keys+1] = $key);
      }
   }
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   @keys;
}

sub keys{
# Function: To return the keys of the dbm database.
# Usage: @array_name = &dbm'keys;
# Nota Bene: This can result in a HUGE array for large databases.  Use
#	with caution.
#
   local(@keys);

   &open || return 0;
   @keys = keys %DB;
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   @keys;
}

sub clean{
# Function: To clean a database, ie. to erase all entries.  The database
#	is not removed, however, and can be reused without recreating it.
# Usage: &dbm'clean;
#
   &open || return 0;
   %DB = {};
   &close || return 0;

   $ENV{'DBMERROR'} = "";
   1;
}

sub clobber{
# Function: To remove a database.  All data is lost and the database is
#	removed.
# Usage: &dbm'clobber;
#
   local($dbmfile) = $ENV{'DBMNAME'};

   if(! -e "$dbmfile.pag"){
      $ENV{'DBMERROR'} = "Dbm clobber failed: No such database '$dbmfile'.";
      return 0;
   }
   if(! unlink("$dbmfile.pag")){
      $ENV{'DMBERROR'} = "Dbm clobber failed: $!";
      return 0;
   }
   if(! unlink("$dbmfile.dir")){
      $ENV{'DMBERROR'} = "Dbm clobber failed: $!";
      return 0;
   }

   $ENV{'DBMERROR'} = "";
   1;
}

1;
