<HTML>
<HEAD>
<TITLE>SRC Modula-3: anim3D/src/Matrix4.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>anim3D/src/Matrix4.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM> Digital Internal Use Only                                                 </EM></BLOCKQUOTE><PRE>
</PRE>                                                                           
       Created on Fri Mar 18 12:05:28 PST 1994 by najork                   

<P>
<P><PRE>MODULE <module><implements><A HREF="Matrix4.i3">Matrix4</A></implements></module>;

IMPORT <A HREF="../../arith/src/Math.i3">Math</A>, <A HREF="Mth.i3">Mth</A>, <A HREF="Point3.i3">Point3</A>;

PROCEDURE <A NAME="Identity"><procedure>Identity</procedure></A> () : T =
  BEGIN
    RETURN Id;
  END Identity;
</PRE>**
PROCEDURE Translate (READONLY M : T; x, y, z : REAL) : T =
  VAR 
    N := T {Row {1.0, 0.0, 0.0, x},
            Row {0.0, 1.0, 0.0, y},
            Row {0.0, 0.0, 1.0, z},
            Row {0.0, 0.0, 0.0, 1.0}};
  BEGIN
    RETURN Multiply (N, M);
  END Translate;
**

<P>
<P><PRE>PROCEDURE <A NAME="Translate"><procedure>Translate</procedure></A> (READONLY M : T; x, y, z : REAL) : T =
   BEGIN
</PRE><BLOCKQUOTE><EM><P>
     &lt;* ASSERT M[3][0] = 0.0 *&gt;
     &lt;* ASSERT M[3][1] = 0.0 *&gt;
     &lt;* ASSERT M[3][2] = 0.0 *&gt;
     &lt;* ASSERT M[3][3] = 1.0 *&gt;
</EM></BLOCKQUOTE><PRE>
     RETURN T {Row{M[0][0], M[0][1], M[0][2], M[0][3] + x},
               Row{M[1][0], M[1][1], M[1][2], M[1][3] + y},
               Row{M[2][0], M[2][1], M[2][2], M[2][3] + z},
               Row{0.0, 0.0, 0.0, 1.0}};
  END Translate;

PROCEDURE <A NAME="Scale"><procedure>Scale</procedure></A> (READONLY M : T; x, y, z : REAL) : T =
  VAR
    N := T {Row {  x, 0.0, 0.0, 0.0},
            Row {0.0,   y, 0.0, 0.0},
            Row {0.0, 0.0,   z, 0.0},
            Row {0.0, 0.0, 0.0, 1.0}};
  BEGIN
    RETURN Multiply (N, M);
  END Scale;

PROCEDURE <A NAME="RotateX"><procedure>RotateX</procedure></A> (READONLY M : T; theta : REAL) : T =
  VAR
    a := Mth.sin (theta);
    b := Mth.cos (theta);
    N := T {Row {1.0, 0.0, 0.0, 0.0},
            Row {0.0,   b,  -a, 0.0},
            Row {0.0,   a,   b, 0.0},
            Row {0.0, 0.0, 0.0, 1.0}};
  BEGIN
    RETURN Multiply (N, M);
  END RotateX;

PROCEDURE <A NAME="RotateY"><procedure>RotateY</procedure></A> (READONLY M : T; theta : REAL) : T =
  VAR
    a := Mth.sin (theta);
    b := Mth.cos (theta);
    N := T {Row {  b, 0.0,   a, 0.0},
            Row {0.0, 1.0, 0.0, 0.0},
            Row { -a, 0.0,   b, 0.0},
            Row {0.0, 0.0, 0.0, 1.0}};
  BEGIN
    RETURN Multiply (N, M);
  END RotateY;

PROCEDURE <A NAME="RotateZ"><procedure>RotateZ</procedure></A> (READONLY M : T; theta : REAL) : T =
  VAR
    a := Mth.sin (theta);
    b := Mth.cos (theta);
    N := T {Row {  b,  -a, 0.0, 0.0},
            Row {  a,   b, 0.0, 0.0},
            Row {0.0, 0.0, 1.0, 0.0},
            Row {0.0, 0.0, 0.0, 1.0}};
  BEGIN
    RETURN Multiply (N, M);
  END RotateZ;

PROCEDURE <A NAME="TransformPoint3"><procedure>TransformPoint3</procedure></A> (READONLY M : T; READONLY p : Point3.T) : Point3.T =
  BEGIN
    RETURN Point3.T {M[0][0] * p.x + M[0][1] * p.y + M[0][2] * p.z + M[0][3],
                     M[1][0] * p.x + M[1][1] * p.y + M[1][2] * p.z + M[1][3],
                     M[2][0] * p.x + M[2][1] * p.y + M[2][2] * p.z + M[2][3]};
  END TransformPoint3;

PROCEDURE <A NAME="Multiply"><procedure>Multiply</procedure></A> (READONLY M, N : T) : T =
  VAR
    P : T;
  BEGIN
    FOR i := 0 TO 3 DO
      FOR j := 0 TO 3 DO
        P[i][j] := 0.0;
        FOR k := 0 TO 3 DO
          P[i][j] := P[i][j] + M[i][k] * N[k][j];
        END;
      END;
    END;
    RETURN P;
  END Multiply;
</PRE> <P>
   In the old version of <CODE>TransformUnitCube</CODE>, I computed the result matrix <CODE>M</CODE> 
   by using trigonometric functions (and I would be very embarrassed to tell 
   just how long it took me to get this function right). This approach was 
   of course motivated by the geometric interpretation on the function 
   (projecting the unit cube through scaling, rotations, and translation 
   onto the cube with corners <CODE>p0</CODE>,<CODE>a0</CODE>,<CODE>b0</CODE>,<CODE>c0</CODE>). 


<P><PRE>PROCEDURE <A NAME="OldTransformUnitCube"><procedure>OldTransformUnitCube</procedure></A> (p0, a0, b0, c0 : Point3.T) : T =
  VAR
    p, a, b, c : Point3.T;
    M     : T;
    N     : T;
    sx, sy, sz : REAL;
    angle1, angle2, angle3 : REAL;
  BEGIN
    M := Identity ();
    M := Translate (M, -p0.x, -p0.y, -p0.z);
    p := TransformPoint3 (M, p0);
    a := TransformPoint3 (M, a0);
    b := TransformPoint3 (M, b0);
    c := TransformPoint3 (M, c0);

    M := Identity ();
    (* We want to rotate vector &quot;a&quot; around the y axis such that it falls into
       the x-y plane. So, we need to find the angle &quot;angle1&quot; between the
       projection of &quot;a&quot; onto the x-z plane and the x axis. *)
    IF a.z = 0.0 THEN
      (* If &quot;a.z&quot; = 0, then &quot;a&quot; is already in the x-y plane. *)
      angle1 := 0.0;
    ELSE
      (* a.z # 0, hence Length ( (a.x, 0, a.z) ) &gt; 0 *)
      angle1 := Mth.asin (a.z / Point3.Length (Point3.T {a.x, 0.0, a.z}));
    END;
    IF a.x &lt; 0.0 THEN
      angle1 := Math.Pi - angle1;
    END;
    M := RotateY (M, angle1);
    p := TransformPoint3 (M, p);
    a := TransformPoint3 (M, a);
    b := TransformPoint3 (M, b);
    c := TransformPoint3 (M, c);

    M := Identity ();
    (* We want to rotate vector &quot;a&quot; around the z axis such that it falls onto
       the x axis. So, we need to find the angle &quot;angle2&quot; between &quot;a&quot; and the
       x axis. Note that the previous rotation moved &quot;a&quot; into the x-y plane,
       hence &quot;a.z&quot; is 0, hence we do not need to project &quot;a&quot; onto any plane.
       Also, note that &quot;a&quot; is guaranteed to have a positive length. *)
    angle2 := - Mth.asin (a.y / Point3.Length (a));
    IF a.x &lt; 0.0 THEN
      angle2 := Math.Pi - angle2;
    END;
    M := RotateZ (M, angle2);
    p := TransformPoint3 (M, p);
    a := TransformPoint3 (M, a);
    b := TransformPoint3 (M, b);
    c := TransformPoint3 (M, c);

    M := Identity ();
    (* At this point, &quot;a&quot; should be lying on the positive half of x axis,
       and &quot;b&quot; and &quot;c&quot; should both be lying in the y-z plane. We want to
       rotate &quot;b&quot; around the x axis so that it lies on the positive half
       of the y axis. *)
    angle3 := -Mth.asin (b.z / Point3.Length (b));
    IF b.y &lt; 0.0 THEN
      angle3 := Math.Pi - angle3;
    END;
    M := RotateX (M, angle3);
    p := TransformPoint3 (M, p);
    a := TransformPoint3 (M, a);
    b := TransformPoint3 (M, b);
    c := TransformPoint3 (M, c);

    sx := Point3.Length (a);
    sy := Point3.Length (b);
    sz := Point3.Length (c);

    (* Construct N *)
    N := Identity ();
    N := Scale (N, sx, sy, sz);
    N := RotateX (N, -angle3);
    N := RotateZ (N, -angle2);
    N := RotateY (N, -angle1);
    N := Translate (N, p0.x, p0.y, p0.z);

    RETURN N;
  END OldTransformUnitCube;

PROCEDURE <A NAME="TransformUnitCube"><procedure>TransformUnitCube</procedure></A> (p, a, b, c : Point3.T) : T =
  BEGIN
    RETURN T {Row {a.x - p.x, b.x - p.x, c.x - p.x, p.x},
              Row {a.y - p.y, b.y - p.y, c.y - p.y, p.y},
              Row {a.z - p.z, b.z - p.z, c.z - p.z, p.z},
              Row {      0.0,       0.0,       0.0, 1.0}};
  END TransformUnitCube;

PROCEDURE <A NAME="UnitSphereMaxSquishFactor"><procedure>UnitSphereMaxSquishFactor</procedure></A> (READONLY M : T) : REAL =

  (* Given a vector v, DecomposeVector returns a unit vector u parallel to v
     and the length l of v. In other words, u = Point3.Scale (v, 1.0) and
     l = Point3.Length (v). *)

  PROCEDURE Iterate (READONLY AAt : T;
                     v            : Point3.T;
                     VAR u        : Point3.T;
                     VAR l        : REAL) =
    BEGIN
      v := TransformPoint3 (AAt, v);
      l := Mth.sqrt (v.x * v.x + v.y * v.y + v.z * v.z);
      u := Point3.T {v.x / l, v.y / l, v.z / l};
    END Iterate;

  CONST
    eps = 0.05;
  VAR
    A, At, AAt    : T;
    v1, v2, v3, v : Point3.T;
    s1, s2, s3, s : REAL;
    delta         : REAL;
    s_prev        : REAL;
  BEGIN
    A := T {Row {M[0][0], M[0][1], M[0][2], 0.0},
            Row {M[1][0], M[1][1], M[1][2], 0.0},
            Row {M[2][0], M[2][1], M[2][2], 0.0},
            Row {    0.0,     0.0,     0.0, 1.0}};
    At := T {Row {M[0][0], M[1][0], M[2][0], 0.0},
             Row {M[0][1], M[1][1], M[2][1], 0.0},
             Row {M[0][2], M[1][2], M[2][2], 0.0},
             Row {    0.0,     0.0,     0.0, 1.0}};
    AAt := Multiply (A, At);

    (*** start with 3 mutually orthogonal unit vectors ***)
    Iterate (AAt, Point3.T {1.0, 0.0, 0.0}, v1, s1);
    Iterate (AAt, Point3.T {0.0, 1.0, 0.0}, v2, s2);
    Iterate (AAt, Point3.T {0.0, 0.0, 1.0}, v3, s3);

    (*** decide which one yielded the largest scaling ***)
    IF s1 &gt;= s2 AND s1 &gt;= s3 THEN
      v := v1;
      s_prev := s1;
    ELSIF s2 &gt;= s1 AND s2 &gt;= s3 THEN
      v := v2;
      s_prev := s2;
    ELSIF s3 &gt;= s1 AND s3 &gt;= s2 THEN
      v := v3;
      s_prev := s3;
    ELSE
      &lt;* ASSERT FALSE *&gt;
    END;

    (*** Iterate until the scale factor approaches a fixed point ***)
    REPEAT
      Iterate (AAt, v, v, s);
      delta := ABS (s / s_prev) - 1.0;
      s_prev := s;
    UNTIL delta &lt; eps;

    RETURN Mth.sqrt (s);
  END UnitSphereMaxSquishFactor;
</PRE> Basic Assertions:
   (1) M has been created by combining rotations, translations, and
       uniform(!) scalings.
   (2) s &gt; 0

<PRE>PROCEDURE <A NAME="Decompose"><procedure>Decompose</procedure></A> ((* in *)  M : T;
                     (* out *) VAR tx, ty, tz, s, angX, angY, angZ : REAL) =
  VAR
    a, b, c: Point3.T;
  BEGIN
    &lt;* ASSERT M[3][0] = 0.0 *&gt;
    &lt;* ASSERT M[3][1] = 0.0 *&gt;
    &lt;* ASSERT M[3][2] = 0.0 *&gt;
    &lt;* ASSERT M[3][3] = 1.0 *&gt;

    (* separate the translation component *)
    tx := M[0][3];
    ty := M[1][3];
    tz := M[2][3];

    (* remove the translation component from M *)
    M[0][3] := 0.0;
    M[1][3] := 0.0;
    M[2][3] := 0.0;

    (* We assumed uniform scaling, which makes it very easy to determine s. *)
    WITH p0 = Point3.T {1.0, 0.0, 0.0},
         p1 = TransformPoint3 (M, p0) DO
      s := Point3.Length (p1);
    END;

    (* Also, for a uniform scaling S, SM = MS for any matrix M.
       So, we can remove S easily. *)
    FOR i := 0 TO 2 DO
      FOR j := 0 TO 2 DO
        M[i][j] := M[i][j] / s;
      END;
    END;

    (* Take three orthogonal unit vectors *)
    a := Point3.T {1.0, 0.0, 0.0};
    b := Point3.T {0.0, 1.0, 0.0};
    c := Point3.T {0.0, 0.0, 1.0};

    (* Apply M to them *)
    a := TransformPoint3 (M, a);
    b := TransformPoint3 (M, b);
    c := TransformPoint3 (M, c);

    (* We want to rotate vector &quot;a&quot; around the z axis such that it falls into
       the x-z plane. So, we need to find the angle &quot;angZ&quot; between the
       projection of &quot;a&quot; onto the x-y plane and the z axis. *)
    IF a.y = 0.0 THEN
      (* If &quot;a.y&quot; = 0, then &quot;a&quot; is already in the x-z plane. *)
      angZ := 0.0;
    ELSE
      (* a.y # 0, hence Length ( (a.x, 0, a.y) ) &gt; 0 *)
      angZ := - Mth.asin (a.y / Point3.Length (Point3.T {a.x, a.y, 0.0}));
    END;
    IF a.x &lt; 0.0 THEN
      angZ := Math.Pi - angZ;
    END;
    WITH N = RotateZ (Id, angZ) DO
      a := TransformPoint3 (N, a);
      b := TransformPoint3 (N, b);
      c := TransformPoint3 (N, c);
    END;

    (* We want to rotate vector &quot;a&quot; around the y axis such that it falls onto
       the x axis. So, we need to find the angle &quot;angY&quot; between &quot;a&quot; and the
       x axis. Note that the previous rotation moved &quot;a&quot; into the x-z plane,
       hence &quot;a.y&quot; is 0, hence we do not need to project &quot;a&quot; onto any plane.
       Also, note that &quot;a&quot; is guaranteed to have a positive length. *)
    angY := Mth.asin (a.z / Point3.Length (a));
    IF a.x &lt; 0.0 THEN
      angY := Math.Pi - angY;
    END;
    WITH N = RotateY (Id, angY) DO
      a := TransformPoint3 (N, a);
      b := TransformPoint3 (N, b);
      c := TransformPoint3 (N, c);
    END;

    (* At this point, &quot;a&quot; should be lying on the positive half of x axis,
       and &quot;b&quot; and &quot;c&quot; should both be lying in the y-z plane. We want to
       rotate &quot;b&quot; around the x axis so that it lies on the positive half
       of the y axis. *)
    angX := - Mth.asin (b.z / Point3.Length (b));
    IF b.y &lt; 0.0 THEN
      angX := Math.Pi - angX;
    END;
    WITH N = RotateX (Id, angX) DO
      a := TransformPoint3 (N, a);
      b := TransformPoint3 (N, b);
      c := TransformPoint3 (N, c);
    END;

    angX := - angX;
    angY := - angY;
    angZ := - angZ;
  END Decompose;
</PRE> Basic Assertions:
   (1) M has been created by combining rotations, translations, and
       uniform(!) scalings.
   (2) s &gt; 0

<PRE>PROCEDURE <A NAME="Decomp"><procedure>Decomp</procedure></A> (M : T; VAR tx, ty, tz, s : REAL) : T RAISES {Error} =
  BEGIN
    &lt;* ASSERT M[3][0] = 0.0 *&gt;
    &lt;* ASSERT M[3][1] = 0.0 *&gt;
    &lt;* ASSERT M[3][2] = 0.0 *&gt;
    &lt;* ASSERT M[3][3] = 1.0 *&gt;

    (* separate the translation component *)
    tx := M[0][3];
    ty := M[1][3];
    tz := M[2][3];

    (* remove the translation component from M *)
    M[0][3] := 0.0;
    M[1][3] := 0.0;
    M[2][3] := 0.0;

    (* We assumed uniform scaling, which makes it very easy to determine s. *)
    WITH p0 = Point3.T {1.0, 0.0, 0.0},
         p1 = TransformPoint3 (M, p0) DO
      s := Point3.Length (p1);
    END;

    (* Also, for a uniform scaling S, SM = MS for any matrix M.
       So, we can remove S easily. *)
    FOR i := 0 TO 2 DO
      FOR j := 0 TO 2 DO
        M[i][j] := M[i][j] / s;
      END;
    END;

    IF NOT Orthonormal (M) THEN
      RAISE Error;
    ELSE
      RETURN M;
    END;
  END Decomp;

PROCEDURE <A NAME="Transpose"><procedure>Transpose</procedure></A> (READONLY M : T) : T =
  BEGIN
    RETURN T {Row {M[0][0], M[1][0], M[2][0], M[3][0]},
              Row {M[0][1], M[1][1], M[2][1], M[3][1]},
              Row {M[0][2], M[1][2], M[2][2], M[3][2]},
              Row {M[0][3], M[1][3], M[2][3], M[3][3]}};
  END Transpose;

PROCEDURE <A NAME="Equal"><procedure>Equal</procedure></A> (READONLY A, B : T) : BOOLEAN =
  CONST
    eps = 0.0005;
  BEGIN
    FOR i := 0 TO 3 DO
      FOR j := 0 TO 3 DO
        IF ABS (A[i][j] - B[i][j]) &gt; eps THEN
          RETURN FALSE;
        END;
      END;
    END;
    RETURN TRUE;
  END Equal;

PROCEDURE <A NAME="Orthonormal"><procedure>Orthonormal</procedure></A> (READONLY U : T) : BOOLEAN =

  PROCEDURE DotProduct (u, v : Row) : REAL =
    BEGIN
      RETURN u[0]*v[0] + u[1]*v[1] + u[2]*v[2] + u[3]*v[3];
    END DotProduct;

  PROCEDURE Zero (x : REAL) : BOOLEAN =
    BEGIN
      RETURN ABS (x) &lt; 0.0001;
    END Zero;

  PROCEDURE One (x : REAL) : BOOLEAN =
    BEGIN
      RETURN Zero (x - 1.0);
    END One;

  BEGIN
    WITH Ut = Transpose (U),
         u0 = SUBARRAY (Ut[0], 0, 4),
         u1 = SUBARRAY (Ut[1], 0, 4),
         u2 = SUBARRAY (Ut[2], 0, 4),
         u3 = SUBARRAY (Ut[3], 0, 4),
         d00 = DotProduct (u0, u0),
         d01 = DotProduct (u0, u1),
         d02 = DotProduct (u0, u2),
         d03 = DotProduct (u0, u3),
         d11 = DotProduct (u1, u1),
         d12 = DotProduct (u1, u2),
         d13 = DotProduct (u1, u3),
         d22 = DotProduct (u2, u2),
         d23 = DotProduct (u2, u3),
         d33 = DotProduct (u3, u3) DO
      RETURN One (d00) AND One (d11) AND One (d22) AND One (d33) AND
             Zero (d01) AND Zero (d02) AND Zero (d03) AND
             Zero (d12) AND Zero (d13) AND Zero (d23);
    END;
  END Orthonormal;

BEGIN
END Matrix4.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
