<HTML>
<HEAD>
<TITLE>SRC Modula-3: etext/src/TextPort.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>etext/src/TextPort.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                                           </EM></BLOCKQUOTE><PRE>
&lt;* PRAGMA LL *&gt;
&lt;* PRAGMA EXPORTED *&gt;

MODULE <module><implements><A HREF="TextPort.i3">TextPort</A></implements></module>;

IMPORT <A HREF="../../geometry/src/Axis.i3">Axis</A>, <A HREF="../../ui/src/vbt/Cursor.i3">Cursor</A>, <A HREF="../../params/src/Env.i3">Env</A>, <A HREF="../../ui/src/vbt/Font.i3">Font</A>, <A HREF="ISOChar.i3">ISOChar</A>, <A HREF="../../ui/src/vbt/KeyboardKey.i3">KeyboardKey</A>, <A HREF="KeyFilter.i3">KeyFilter</A>,
       <A HREF="KeyTrans.i3">KeyTrans</A>, <A HREF="MacModel.i3">MacModel</A>, <A HREF="../../mtext/src/MText.i3">MText</A>, <A HREF="MTextUnit.i3">MTextUnit</A>, <A HREF="../../ui/src/vbt/PaintOp.i3">PaintOp</A>, <A HREF="../../ui/src/vbt/Palette.i3">Palette</A>,
       <A HREF="../../vbtkitutils/src/Pts.i3">Pts</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../rw/src/Common/RdUtils.i3">RdUtils</A>, <A HREF="../../geometry/src/Rect.i3">Rect</A>, <A HREF="../../geometry/src/Region.i3">Region</A>, <A HREF="../../lego/src/ScrollerVBTClass.i3">ScrollerVBTClass</A>, <A HREF="../../rw/src/Common/Stdio.i3">Stdio</A>,
       <A HREF="../../text/src/Text.i3">Text</A>, <A HREF="TextPortClass.i3">TextPortClass</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../ui/src/vbt/VBT.i3">VBT</A>, <A HREF="../../ui/src/vbt/VBTRep.i3">VBTRep</A>, <A HREF="../../vtext/src/VTDef.i3">VTDef</A>, <A HREF="../../vtext/src/VText.i3">VText</A>,
       <A HREF="../../weakref/src/WeakRef.i3">WeakRef</A>, <A HREF="../../rw/src/Common/Wr.i3">Wr</A>;

FROM <A HREF="TextPortClass.i3">TextPortClass</A> IMPORT IRange;

IMPORT <A HREF="EmacsModel.i3">EmacsModel</A>, <A HREF="IvyModel.i3">IvyModel</A>, <A HREF="XtermModel.i3">XtermModel</A>;

IMPORT <A HREF="../../lego/src/VBTutils.i3">VBTutils</A>, <A HREF="../../rw/src/Common/TextWr.i3">TextWr</A>;

CONST
  Backspace = '\010';
  Tab       = '\t';
  Return    = '\n';
  Del       = '\177';
  Primary   = SelectionType.Primary;
  Secondary = SelectionType.Secondary;
  Focus     = TextPortClass.VType.Focus;

REVEAL
  <A NAME="T">T</A> = TextPortClass.T BRANDED OBJECT
        &lt;* LL = v.mu *&gt;
        modifiedP : BOOLEAN;
        linesShown: CARDINAL;
      OVERRIDES
        (* Exported methods *)
        init           := Init;
        filter         := Filter;
        getFont        := GetFont;
        setFont        := SetFont;
        getColorScheme := GetColorScheme;
        setColorScheme := SetColorScheme;
        getModel       := GetModel;
        setModel       := SetModel;
        getReadOnly    := GetReadOnly;
        setReadOnly    := SetReadOnly;
        getKFocus      := GetKFocus;

        (* Callbacks *)
        returnAction := ReturnAction;
        tabAction    := Insert4spaces;
        focus        := IgnoreFocus;
        modified     := IgnoreModification;
        error        := Error;

        (* VBT.T overrides *)
        key       := Key;
        misc      := Misc;
        mouse     := Mouse;
        position  := Position;
        read      := Read;
        redisplay := Redisplay;
        repaint   := Repaint;
        rescreen  := Rescreen;
        reshape   := Reshape;
        shape     := Shape;
        write     := Write;

        (* Exception handlers *)
        rdeoferror := rdeoferror;
        rdfailure  := rdfailure;
        vbterror   := vbterror;
        vterror    := vterror;

        (* Locked methods *)
        getText          := LockedGetText;
        index            := LockedIndex;
        isReplaceMode    := LockedIsReplaceMode;
        length           := LockedLength;
        normalize        := LockedNormalize;
        replace          := LockedReplace;
        unsafeReplace    := UnsafeReplace;
        insert           := LockedInsert;
        unsafeInsert     := UnsafeInsert;
        newlineAndIndent := LockedNewlineAndIndent;
        findSource       := FindSource;
        notFound         := NotFoundProc;

        (* Unlocked methods for callbacks *)
        ULreturnAction := UnlockedReturnAction;
        ULtabAction    := UnlockedTabAction;
        ULfocus        := UnlockedFocus;
        ULmodified     := UnlockedModified;
        ULerror        := UnlockedError;

      END;

VAR debug, UseDiacritical: BOOLEAN;

PROCEDURE <A NAME="Init"><procedure>Init</procedure></A> (v               : T;
                hMargin, vMargin                      := 0.5;
                font                                  := Font.BuiltIn;
                colorScheme     : PaintOp.ColorScheme := NIL;
                wrap                                  := TRUE;
                readOnly                              := FALSE;
                turnMargin                            := 0.5;
                model                                 := Model.Default ): T =
  CONST PRINTABLE = (ISOChar.All - ISOChar.Controls) + SET OF CHAR {'\t'};
  VAR
    vFont   : VText.VFont;
    vOptions: VText.VOptions;
  BEGIN
    TRY
      IF colorScheme = NIL THEN colorScheme := PaintOp.bgFg END;
      vFont := VText.MakeVFont (
                 font := font, printable := PRINTABLE, whiteTabs := TRUE);
      vOptions := VText.MakeVOptions (
                    vFont := vFont, leftMargin := Pts.FromMM (hMargin),
                    rightMargin := Pts.FromMM (hMargin),
                    turnMargin := Pts.FromMM (turnMargin),
                    topMargin := Pts.FromMM (vMargin), leading := 0.0,
                    whiteBlack := colorScheme, whiteStroke := colorScheme,
                    leftOffset := 0.0, wrap := wrap,
                    eob := FALSE, intervalStylePrecedence := NIL);
      v.font := font;
      v.mu := NEW (MUTEX);
      v.vtext := VText.New (MText.New (&quot;&quot;, 256), v, VBT.Domain (v), vOptions);
      v.readOnly := readOnly;
      v.modifiedP := FALSE;
      v.typeinStart := 0;
      v.cur := NEW (TextPortClass.UndoRec);
      LockedSetModel (v, model);
      Register (v);
      VBT.SetCursor (v, Cursor.TextPointer);
      RETURN v
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (&quot;Init&quot;, ec)
    | Rd.EndOfFile =&gt; v.rdeoferror (&quot;Init&quot;)
    | Rd.Failure (ref) =&gt; v.rdfailure (&quot;Init&quot;, ref)
    | Thread.Alerted =&gt;
    END;
    RETURN NIL
  END Init;
</PRE>**************************  Client Interface  **************************

<P><PRE>PROCEDURE <A NAME="GetReadOnly"><procedure>GetReadOnly</procedure></A> (v: T): BOOLEAN =
  BEGIN
    LOCK v.mu DO RETURN v.readOnly END
  END GetReadOnly;

PROCEDURE <A NAME="SetReadOnly"><procedure>SetReadOnly</procedure></A> (v: T; flag: BOOLEAN) =
  BEGIN
    LOCK v.mu DO v.readOnly := flag; FixIntervals (v) END
  END SetReadOnly;
</PRE> UNUSED
PROCEDURE SetWrap (v: T; wrap: BOOLEAN) =
  BEGIN
    LOCK v.mu DO
      IF v.vtext.vOptions.wrap # wrap THEN
        v.vtext.vOptions.wrap := wrap;
        TRY
          VText.ChangeVOptions (v.vtext, v.vtext.vOptions);
          VBT.Mark (v)
        EXCEPT
        <PRE>
      VTDef.Error (ec) =&gt; v.vterror (&quot;SetWrap&quot;, ec)
              Rd.EndOfFile =&gt; v.rdeoferror (&quot;SetWrap&quot;)
              Rd.Failure (ref) =&gt; v.rdfailure (&quot;SetWrap&quot;, ref)
              Thread.Alerted =&gt;
        </PRE>
END
      END
    END
  END SetWrap;


<P><PRE>&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Length"><procedure>Length</procedure></A> (v: T): CARDINAL =
  BEGIN
    LOCK v.mu DO RETURN v.length () END
  END Length;

PROCEDURE <A NAME="LockedLength"><procedure>LockedLength</procedure></A> (v: T): CARDINAL =
  BEGIN
    RETURN MText.Length (v.vtext.mtext)
  END LockedLength;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="GetText"><procedure>GetText</procedure></A> (v    : T;
                   begin: CARDINAL := 0;
                   end  : CARDINAL := LAST (CARDINAL)): TEXT =
  &lt;* LL = VBT.mu *&gt;
  BEGIN
    LOCK v.mu DO RETURN v.getText (begin, end) END
  END GetText;

PROCEDURE <A NAME="LockedGetText"><procedure>LockedGetText</procedure></A> (v: T; begin, end: CARDINAL): TEXT =
  &lt;* LL = v.mu *&gt;
  BEGIN
    RETURN MText.GetText (v.vtext.mtext, begin, end)
  END LockedGetText;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="SetText"><procedure>SetText</procedure></A> (v: T; t: TEXT) =
  &lt;* LL &lt;= VBT.mu *&gt;
  BEGIN
    LOCK v.mu DO
      EVAL v.unsafeReplace (0, LAST (CARDINAL), t);
      TRY
        VText.SetStart (v.vtext, 0, 0);
        VBT.Mark (v)
      EXCEPT
      | VTDef.Error (ec) =&gt; v.vterror (&quot;SetText&quot;, ec)
      | Rd.EndOfFile =&gt; v.rdeoferror (&quot;SetText&quot;)
      | Rd.Failure (ref) =&gt; v.rdfailure (&quot;SetText&quot;, ref)
      | Thread.Alerted =&gt;
      END;
    END
  END SetText;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="PutText"><procedure>PutText</procedure></A> (v: T; t: TEXT) =
  &lt;* LL &lt;= VBT.mu *&gt;
  BEGIN
    LOCK v.mu DO
      EVAL v.unsafeReplace (LAST (CARDINAL), LAST (CARDINAL), t);
      VBT.Mark (v)
    END
  END PutText;

PROCEDURE <A NAME="GetFont"><procedure>GetFont</procedure></A> (v: T): Font.T =
  BEGIN
    LOCK v.mu DO RETURN v.font END
  END GetFont;

PROCEDURE <A NAME="SetFont"><procedure>SetFont</procedure></A> (v: T; font: Font.T) =
  (* By the book, we should call ExplodeVText, ExplodeVOptions, ExplodeVFont,
     MakeVFont, and MakeVOptions before calling ChangeVOptions, but we cheat
     by looking at the implementation and consing only a new VFont. *)
  VAR
    vtext   : VText.T;
    vOptions: VText.VOptions;
    vFont   : VText.VFont;
  BEGIN
    LOCK v.mu DO
      IF font = v.font THEN RETURN END;
      vtext := v.vtext;
      vOptions := vtext.vOptions;
      vFont := vOptions.vFontxxx;
      TRY
        vOptions.vFontxxx :=
          VText.MakeVFont (font := font, printable := vFont.vFont.printable,
                           whiteTabs := vFont.vFont.whiteTabs);
        v.font := font;          (* For convenience only *)
        VText.ChangeVOptions (vtext, vOptions);
        SetFontDimensions (v);
        VBT.NewShape (v);
        VBT.Mark (v)
      EXCEPT
      | VTDef.Error (ec) =&gt; v.vterror (&quot;SetFont&quot;, ec)
      | Rd.EndOfFile =&gt; v.rdeoferror (&quot;SetFont&quot;)
      | Rd.Failure (ref) =&gt; v.rdfailure (&quot;SetFont&quot;, ref)
      | Thread.Alerted =&gt;
      END;
    END
  END SetFont;

PROCEDURE <A NAME="GetColorScheme"><procedure>GetColorScheme</procedure></A> (v: T): PaintOp.ColorScheme  =
  BEGIN
    LOCK v.mu DO
      RETURN v.vtext.vOptions.whiteBlack (* one of several choices *)
    END
  END GetColorScheme;

PROCEDURE <A NAME="SetColorScheme"><procedure>SetColorScheme</procedure></A> (v: T; colorScheme: PaintOp.ColorScheme) =
  CONST name = &quot;SetColorScheme&quot;;
  VAR vOptions: VText.VOptions;
  BEGIN
    LOCK v.mu DO
      TRY
        vOptions := v.vtext.vOptions;
        vOptions.whiteBlack := colorScheme;
        vOptions.whiteStroke := colorScheme;
        VText.ChangeVOptions (v.vtext, vOptions);
        IF v.scrollbar # NIL THEN
          ScrollerVBTClass.Colorize (v.scrollbar, colorScheme)
        END;
        VBT.Mark (v)
      EXCEPT
      | VTDef.Error (ec) =&gt; v.vterror (name, ec)
      | Rd.EndOfFile =&gt; v.rdeoferror (name)
      | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
      | Thread.Alerted =&gt;
      END
    END
  END SetColorScheme;

PROCEDURE <A NAME="GetModel"><procedure>GetModel</procedure></A> (v: T): [Model.Ivy .. Model.Xterm] =
  BEGIN
    LOCK v.mu DO
      TYPECASE v.m OF
      | IvyModel.T =&gt; RETURN Model.Ivy
      | EmacsModel.T =&gt; RETURN Model.Emacs
      | XtermModel.T =&gt; RETURN Model.Xterm
      | MacModel.T =&gt; RETURN Model.Mac
      ELSE                       &lt;* ASSERT FALSE *&gt;
      END
    END
  END GetModel;

PROCEDURE <A NAME="SetModel"><procedure>SetModel</procedure></A> (v: T; model: Model) =
  BEGIN
    LOCK v.mu DO LockedSetModel (v, model) END
  END SetModel;

TYPE
  StandardKeyFilter =
    KeyFilter.T OBJECT OVERRIDES apply := ApplyStandardKeyFilter END;

PROCEDURE <A NAME="LockedSetModel"><procedure>LockedSetModel</procedure></A> (v: T; model: Model) =
  VAR
    cs              := v.vtext.vOptions.whiteBlack;
    f : KeyFilter.T := NEW (StandardKeyFilter);
  BEGIN
    IF v.m # NIL THEN v.m.close () END;
    IF model = Model.Default THEN model := DefaultModel END;
    IF UseDiacritical THEN f := NEW (KeyFilter.Diacritical, next := f) END;
    CASE model OF
    | Model.Ivy =&gt; v.m := NEW (IvyModel.T, v := v).init (cs, f)
    | Model.Xterm =&gt; v.m := NEW (XtermModel.T, v := v).init (cs, f)
    | Model.Emacs =&gt; v.m := NEW (EmacsModel.T, v := v).init (cs, f)
    | Model.Mac =&gt; v.m := NEW (MacModel.T, v := v).init (cs, f)
    ELSE                         &lt;* ASSERT FALSE *&gt;
    END;
    FixIntervals (v)
  END LockedSetModel;

PROCEDURE <A NAME="GetKFocus"><procedure>GetKFocus</procedure></A> (v: T; t: VBT.TimeStamp): BOOLEAN =
  &lt;* LL = v.mu *&gt;
  CONST name = &quot;GetKFocus&quot;;
  BEGIN
    IF NOT v.owns [Focus] THEN
      v.ULfocus (TRUE, t);
      TRY
        VBT.Acquire (v, VBT.KBFocus, t);
        VText.SwitchCaret (v.vtext, VText.OnOffState.On);
        v.owns [Focus] := TRUE
      EXCEPT
      | VBT.Error (ec) =&gt; v.vbterror (name, ec)
      | VTDef.Error (ec) =&gt; v.vterror (name, ec)
      | Rd.EndOfFile =&gt; v.rdeoferror (name)
      | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
      | Thread.Alerted =&gt;
      END
    END;
    RETURN v.owns [Focus]
  END GetKFocus;

TYPE
  WeakRefList = REF RECORD
                      first: WeakRef.T;
                      tail : WeakRefList := NIL
                    END;

VAR mu := NEW (MUTEX);
    ports: WeakRefList := NIL; &lt;* LL = mu *&gt;

PROCEDURE <A NAME="ChangeAllTextPorts"><procedure>ChangeAllTextPorts</procedure></A> (newModel := Model.Default) =
  VAR
    a   : WeakRefList;
    port: T;
  BEGIN
    LOCK mu DO
      a := ports;
      WHILE a # NIL DO
        port := WeakRef.ToRef (a.first);
        IF port # NIL THEN port.setModel (newModel); VBT.Mark (port) END;
        a := a.tail
      END
    END
  END ChangeAllTextPorts;

PROCEDURE <A NAME="Register"><procedure>Register</procedure></A> (v: T) =
  BEGIN
    LOCK mu DO
      ports := NEW (WeakRefList, first := WeakRef.FromRef (v, Cleanup),
                    tail := ports)
    END
  END Register;

PROCEDURE <A NAME="Cleanup"><procedure>Cleanup</procedure></A> (READONLY w: WeakRef.T; &lt;* UNUSED *&gt; r: REFANY) =
  (* Delete &quot;w&quot; from &quot;ports&quot;. *)
  BEGIN
    LOCK mu DO
      IF ports = NIL THEN        (* skip *)
      ELSIF ports.first = w THEN
        ports := ports.tail
      ELSE
        VAR
          a              := ports;
          b: WeakRefList;
        BEGIN
          LOOP
            b := a.tail;
            IF b = NIL THEN RETURN END;
            IF b.first = w THEN a.tail := b.tail; RETURN END;
            a := b
          END
        END
      END
    END
  END Cleanup;

PROCEDURE <A NAME="SetFontDimensions"><procedure>SetFontDimensions</procedure></A> (v: T) =
  &lt;* LL = v.mu *&gt;
  BEGIN
    (* metrics := FontClass.FontMetrics(vbt, v.font); *)
    WITH st = VBT.ScreenTypeOf (v) DO
      IF st # NIL THEN
        WITH bounds = Palette.ResolveFont (
                        st, v.font).metrics.maxBounds,
             box = bounds.boundingBox DO
          v.fontHeight := Rect.VerSize (box);
          v.charWidth := bounds.printWidth
          (* not &quot;Rect.HorSize (box)&quot;, alas *)
        END
      END
    END
  END SetFontDimensions;
</PRE> UNUSED 
PROCEDURE Width (v: T): CARDINAL =
   (* Return the number of characters that will fit on a line, given the current
   size of <CODE>v</CODE>.  If <CODE>v</CODE> has no width (because its window is iconic, for
   example), return 0.  If <CODE>v</CODE>'s font is not fixed-pitch, the result will be
   computed in terms of the width of the widest character in the font. 
  <PRE>This code is wrong. It ignores the margins. See TypeinVBT.Shape and
  NumericVBT.Reshape.
  VAR n := Rect.HorSize (VBT.Domain (v));
  BEGIN
    LOCK v.mu DO
      IF v.charWidth = 0 THEN RETURN 0 ELSE RETURN n DIV v.charWidth END
    END
  END Width;
*)
</PRE> UNUSED
PROCEDURE Height (v: T): CARDINAL =
  BEGIN
    RETURN v.shape (Axis.T.Ver, 0).pref
  END Height;


<P>*************** Focus, selections, etc. ****************

<P><PRE>&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="TryFocus"><procedure>TryFocus</procedure></A> (v: T; t: VBT.TimeStamp): BOOLEAN =
  &lt;* LL &lt; v.mu, LL.sup = VBT.mu *&gt;
  BEGIN
    (* Force all pending redisplays: *)
    VBTRep.Redisplay ();
    IF Rect.IsEmpty (VBT.Domain (v)) THEN
      RETURN FALSE
    ELSE
      LOCK v.mu DO
        IF NOT v.getKFocus (t) THEN
          RETURN FALSE
        ELSIF v.m.selection [Primary].alias = VBT.Source
                AND NOT v.m.takeSelection (VBT.Source, Primary, t)
                OR v.m.selection [Primary].alias = VBT.Target
                     AND NOT v.m.takeSelection (VBT.Target, Primary, t) THEN
          VBT.Release (v, VBT.KBFocus);
          v.owns [Focus] := FALSE;
          RETURN FALSE
        ELSE
          VBT.Mark (v);
          RETURN TRUE
        END
      END
    END
  END TryFocus;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="HasFocus"><procedure>HasFocus</procedure></A> (v: T): BOOLEAN =
  BEGIN
    LOCK v.mu DO RETURN v.owns [Focus] END
  END HasFocus;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Select"><procedure>Select</procedure></A> (v          : T;
                  time       : VBT.TimeStamp;
                  begin      : CARDINAL        := 0;
                  end        : CARDINAL        := LAST (CARDINAL);
                  sel                          := SelectionType.Primary;
                  replaceMode                  := FALSE;
                  caretEnd                     := VText.WhichEnd.Right   ) =
  BEGIN
    LOCK v.mu DO v.m.select (time, begin, end, sel, replaceMode, caretEnd) END
  END Select;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="IsReplaceMode"><procedure>IsReplaceMode</procedure></A> (v: T): BOOLEAN =
  BEGIN
    LOCK v.mu DO RETURN v.isReplaceMode () END
  END IsReplaceMode;

PROCEDURE <A NAME="LockedIsReplaceMode"><procedure>LockedIsReplaceMode</procedure></A> (v: T): BOOLEAN =
  BEGIN
    RETURN NOT v.readOnly AND v.m.selection [Primary].replaceMode
  END LockedIsReplaceMode;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="GetSelection"><procedure>GetSelection</procedure></A> (v: T; sel := SelectionType.Primary): Extent =
  BEGIN
    LOCK v.mu DO RETURN v.m.getSelection (sel) END
  END GetSelection;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="GetSelectedText"><procedure>GetSelectedText</procedure></A> (v: T; sel := SelectionType.Primary): TEXT =
  &lt;* LL.sup = VBT.mu *&gt;
  BEGIN
    LOCK v.mu DO RETURN v.m.getSelectedText (sel) END
  END GetSelectedText;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="PutSelectedText"><procedure>PutSelectedText</procedure></A> (v: T; t: TEXT; sel := SelectionType.Primary) =
  &lt;* LL.sup = VBT.mu *&gt;
  BEGIN
    LOCK v.mu DO v.m.putSelectedText (t, sel) END
  END PutSelectedText;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Index"><procedure>Index</procedure></A> (v: T): CARDINAL =
  BEGIN
    LOCK v.mu DO RETURN v.index () END
  END Index;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Seek"><procedure>Seek</procedure></A> (v: T; n: CARDINAL) =
  BEGIN
    LOCK v.mu DO v.m.seek (n) END
  END Seek;

PROCEDURE <A NAME="LockedIndex"><procedure>LockedIndex</procedure></A> (v: T): CARDINAL =
  BEGIN
    TRY
      RETURN VText.CaretIndex (v.vtext)
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (&quot;Index&quot;, ec); RETURN 0
    END
  END LockedIndex;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="IsVisible"><procedure>IsVisible</procedure></A> (v: T; pos: CARDINAL): BOOLEAN =
  CONST name = &quot;IsVisible&quot;;
  BEGIN
    TRY
      RETURN VText.InRegion (v.vtext, 0, pos)
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (name, ec)
    | Rd.EndOfFile =&gt; v.rdeoferror (name)
    | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
    | Thread.Alerted =&gt;
    END;
    RETURN FALSE
  END IsVisible;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="IsModified"><procedure>IsModified</procedure></A> (v: T): BOOLEAN =
  BEGIN
    RETURN v.modifiedP
  END IsModified;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="SetModified"><procedure>SetModified</procedure></A> (v: T; modified: BOOLEAN) =
  BEGIN
    v.modifiedP := modified
  END SetModified;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="GetVText"><procedure>GetVText</procedure></A> (v: T): VText.T =
  BEGIN
    LOCK v.mu DO RETURN v.vtext END
  END GetVText;
</PRE>**************************  Replace &amp; Insert  ***************************

<P><PRE>&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Replace"><procedure>Replace</procedure></A> (v: T; begin, end: CARDINAL; newText: TEXT) =
  BEGIN
    LOCK v.mu DO EVAL v.unsafeReplace (begin, end, newText) END
  END Replace;

PROCEDURE <A NAME="LockedReplace"><procedure>LockedReplace</procedure></A> (v: T; begin, end: CARDINAL; newText: TEXT): Extent =
  &lt;* LL = v.mu *&gt;
  BEGIN
    IF v.readOnly OR begin &lt; v.typeinStart THEN
      RETURN NotFound
    ELSE
      RETURN v.unsafeReplace (begin, end, newText)
    END
  END LockedReplace;

PROCEDURE <A NAME="UnsafeReplace"><procedure>UnsafeReplace</procedure></A> (v: T; begin, end: CARDINAL; newText: TEXT): Extent =
  &lt;* LL = v.mu *&gt;
  CONST name = &quot;Replace&quot;;
  VAR len := v.length ();
  BEGIN
    begin := MIN (begin, len);
    end := MIN (MAX (begin, end), len);
    IF begin &lt;= end THEN
      TextPortClass.AddToUndo (v, begin, end, newText);
      TRY
        VText.Replace (v.vtext, begin, end, newText);
        IF NOT v.modifiedP THEN v.modifiedP := TRUE; v.ULmodified () END;
        VBT.Mark (v);
        RETURN Extent {begin, begin + Text.Length (newText)}
      EXCEPT
      | VTDef.Error (ec) =&gt; v.vterror (name, ec)
      | Rd.EndOfFile =&gt; v.rdeoferror (name)
      | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
      | Thread.Alerted =&gt;
      END
    END;
    RETURN NotFound
  END UnsafeReplace;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Insert"><procedure>Insert</procedure></A> (v: T; t: TEXT) =
  BEGIN
    IF NOT Text.Empty (t) THEN LOCK v.mu DO v.unsafeInsert (t) END END
  END Insert;

PROCEDURE <A NAME="LockedInsert"><procedure>LockedInsert</procedure></A> (v: T; t: TEXT) =
  BEGIN
    IF NOT v.readOnly THEN v.unsafeInsert (t) END
  END LockedInsert;

PROCEDURE <A NAME="UnsafeInsert"><procedure>UnsafeInsert</procedure></A> (v: T; t: TEXT) =
  VAR
    m   := v.m;
    rec := m.selection [Primary];
  VAR p: CARDINAL;
  BEGIN
    IF v.isReplaceMode () THEN
      m.putSelectedText (t, Primary)
    ELSE
      p := v.index ();
      IF p &lt; v.typeinStart THEN p := v.length (); m.seek (p) END;
      EVAL v.unsafeReplace (p, p, t)
    END;
    p := v.index ();
    m.highlight (rec, IRange {p, p, p});
    rec.anchor.l := p;
    rec.anchor.r := p
  END UnsafeInsert;
</PRE>***********************  Shape of current text  ************************

<P>
<P><PRE>PROCEDURE <A NAME="Shape"><procedure>Shape</procedure></A> (v: T; ax: Axis.T; n: CARDINAL): VBT.SizeRange =
  BEGIN
    IF ax = Axis.T.Ver THEN v.lastNonEmptyWidth := n END;
    RETURN TextPortClass.T.shape(v, ax, n);
  END Shape;

PROCEDURE <A NAME="Reshape"><procedure>Reshape</procedure></A> (v: T; READONLY cd: VBT.ReshapeRec) =
  CONST name = &quot;Reshape&quot;;
  VAR
    newRect                                 := cd.new;
    dividers: ARRAY [0 .. 0] OF VText.Pixels;
  BEGIN
    IF Rect.IsEmpty(newRect) THEN RETURN END;
    VAR
      old := v.lastNonEmptyWidth;
      new := Rect.HorSize(newRect);
    BEGIN
      IF new # old THEN
        VAR
          oldShape := v.shape(Axis.T.Ver, old);
          newShape := v.shape(Axis.T.Ver, new);
        BEGIN
          IF oldShape # newShape THEN VBT.NewShape(v) END
        END
      END
    END;
    LOCK v.mu DO
      IF NOT v.vtext.vOptions.wrap THEN
        newRect.east := LAST(INTEGER) DIV 2
      END;
      TRY
        VText.Move(v.vtext, newRect, cd.saved, dividers);
        VText.Update(v.vtext);
        v.linesShown :=
          1 + VText.WhichLine(v.vtext, 0, cd.new.south);
        (* if it will all fit, normalize to fit *)
        IF Rect.IsEmpty(cd.prev) AND NOT Rect.IsEmpty(cd.new)
             AND VText.LinesBetween(
                   v.vtext, 0, v.length(), v.linesShown)
                   &lt; v.linesShown THEN
          v.normalize(0)         (* Normalize calls
                                    VBT.Mark *)
        ELSE
          VBT.Mark(v)
        END
      EXCEPT
      | VTDef.Error (ec) =&gt; v.vterror(name, ec)
      | Rd.EndOfFile =&gt; v.rdeoferror(name)
      | Rd.Failure (ref) =&gt; v.rdfailure(name, ref)
      | Thread.Alerted =&gt;
      END
    END
  END Reshape;
</PRE> UNUSED
PROCEDURE ShapeInfo (v: T; VAR lineCount, lineLength: INTEGER) =
  (* Return the number of lines in the text and the number of
   characters in the longest line (excluding the newline).  A
   client may use this information in conjunction with
   <CODE>VBTShape.Set</CODE> to shape <CODE>v</CODE> to fit its text.  Having done so,
   it should be prepared to do it again on a rescreen to a screen
   of different density. 
  <PRE>BEGIN
    LOCK v.mu DO
      VAR
        e      := MTextUnit.Extent {0, 0, TRUE};
        length := v.length ();
      BEGIN
        lineCount := 0;
        lineLength := 0;
        IF length = 0 THEN RETURN END;
        WHILE e.right &lt; length DO
          e := MTextUnit.LineExtent (v.vtext.mtext, e.right);
          INC (lineCount);
          lineLength := MAX (lineLength, e.right - e.left - 1)
        END;
        (* adjust for last line: if ends with \n, increment lineCount;
           otherwise, len of last line is right-left, not right-left-1. *)
        IF MText.GetChar (v.vtext.mtext, length - 1) = '\n' THEN
          INC (lineCount);
          lineLength := MAX (lineLength, e.right - e.left - 1)
        ELSE
          lineLength := MAX (lineLength, e.right - e.left)
        END                      (* IF *)
      END                        (* BEGIN *)
    END                          (* LOCK *)
  END ShapeInfo;
*)

PROCEDURE <A NAME="Key"><procedure>Key</procedure></A> (v: T; READONLY cd: VBT.KeyRec) =
  VAR OK: BOOLEAN;
  BEGIN
    LOCK v.mu DO
      OK :=
        cd.wentDown AND v.owns [Focus] AND NOT Rect.IsEmpty (VBT.Domain (v))
    END;
    IF OK THEN v.m.keyfilter.apply (v, cd) END
  END Key;

PROCEDURE <A NAME="ApplyStandardKeyFilter"><procedure>ApplyStandardKeyFilter</procedure></A> (&lt;* UNUSED *&gt; self: StandardKeyFilter;
                                  vbt: VBT.T;
                                  cd : VBT.KeyRec) =
  VAR v: T := vbt;
  BEGIN
    v.filter (cd)
  END ApplyStandardKeyFilter;

PROCEDURE <A NAME="Filter"><procedure>Filter</procedure></A> (v: T; cd: VBT.KeyRec) =
  VAR
    ch: CHAR;
    m        := v.m;
  BEGIN
    IF cd.whatChanged = VBT.NoKey THEN RETURN END;
    LOCK v.mu DO
      v.lastCmdKind := v.thisCmdKind;
      v.thisCmdKind := TextPortClass.CommandKind.OtherCommand;
      ch := KeyTrans.Latin1 (cd.whatChanged);

      IF ch = Return THEN
        IF VBT.Modifier.Shift IN cd.modifiers THEN
          v.insert (&quot;\n&quot;)
        ELSIF VBT.Modifier.Option IN cd.modifiers THEN
          TextPortClass.InsertNewline (v)
        ELSE
          v.ULreturnAction (cd)
        END
      ELSIF ch = Tab THEN
        v.ULtabAction (cd)
      ELSIF KeyboardKey.Left &lt;= cd.whatChanged
              AND cd.whatChanged &lt;= KeyboardKey.Down THEN
        m.arrowKey (cd)
      ELSIF VBT.Modifier.Control IN cd.modifiers THEN
        m.controlChord (ch, cd);
        RETURN
      ELSIF VBT.Modifier.Option IN cd.modifiers THEN
        m.optionChord (ch, cd);
        RETURN
      ELSIF ch = Backspace OR ch = Del THEN
        VAR interval := v.m.selection [Primary].interval;
        BEGIN
          (* Treat an empty interval as if it were not replace-mode *)
          IF v.isReplaceMode () AND interval.left () # interval.right () THEN
            m.putSelectedText (&quot;&quot;)
          ELSE
            EVAL TextPortClass.DeletePrevChar (v)
          END
        END
      ELSIF ch IN ISOChar.Graphics THEN
        v.insert (Text.FromChar (ch))
      ELSE
        (* including NullKey, for untranslatable keys *)
        RETURN
      END;
      v.normalize ()
    END                          (* LOCK *)
  END Filter;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Newline"><procedure>Newline</procedure></A> (v: T) =
  BEGIN
    LOCK v.mu DO IF NOT v.readOnly THEN v.insert (&quot;\n&quot;) END END
  END Newline;

&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="NewlineAndIndent"><procedure>NewlineAndIndent</procedure></A> (v: T) =
  BEGIN
    LOCK v.mu DO v.newlineAndIndent () END
  END NewlineAndIndent;

PROCEDURE <A NAME="LockedNewlineAndIndent"><procedure>LockedNewlineAndIndent</procedure></A> (v: T) =
  BEGIN
    IF v.readOnly THEN RETURN END;
    VAR
      index := v.index ();
      a     := MTextUnit.LineInfo (v.vtext.mtext, index);
    BEGIN
      IF a.leftMargin = a.rightEnd AND index = a.rightEnd
           AND NOT v.isReplaceMode () THEN
        (* We're at the end of an all-blank line. *)
        EVAL v.replace (a.left, a.left, &quot;\n&quot;)
      ELSIF a.leftMargin = a.rightMargin THEN (* line is all blanks *)
        v.insert (&quot;\n&quot;)
      ELSE
        (* Copy all the leading blanks onto the new line. *)
        v.insert (
          &quot;\n&quot; &amp; MText.GetText (v.vtext.mtext, a.left, a.leftMargin))
      END
    END
  END LockedNewlineAndIndent;

PROCEDURE <A NAME="Repaint"><procedure>Repaint</procedure></A> (v: T; READONLY rgn: Region.T) =
  CONST name = &quot;Repaint&quot;;
  BEGIN
    TRY
      LOCK v.mu DO
        VText.Bad (v.vtext, Region.BoundingBox (rgn));
        VText.Update (v.vtext)
      END
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (name, ec)
    | Rd.EndOfFile =&gt; v.rdeoferror (name)
    | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
    | Thread.Alerted =&gt;
    END
  END Repaint;

PROCEDURE <A NAME="FixIntervals"><procedure>FixIntervals</procedure></A> (v: T) =
  &lt;* LL = v.mu *&gt;
  BEGIN
    TRY
      FOR t := Primary TO Secondary DO
        IF v.m.selection [t] # NIL THEN
          TextPortClass.ChangeIntervalOptions (v, v.m.selection [t])
        END
      END
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (&quot;Rescreen&quot;, ec)
    END
  END FixIntervals;

PROCEDURE <A NAME="Rescreen"><procedure>Rescreen</procedure></A> (v: T; READONLY cd: VBT.RescreenRec) =
  BEGIN
    LOCK v.mu DO
      VText.Rescreen (v.vtext, cd);
      SetFontDimensions (v);
      FixIntervals (v);
      VBT.NewShape (v)
    END
  END Rescreen;

PROCEDURE <A NAME="Redisplay"><procedure>Redisplay</procedure></A> (v: T) =
  CONST name = &quot;Redisplay&quot;;
  BEGIN
    TRY
      LOCK v.mu DO
        VText.Update (v.vtext);
        IF v.scrollbar # NIL THEN v.scrollbar.update () END
      END
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (name, ec)
    | Rd.EndOfFile =&gt; v.rdeoferror (name)
    | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
    | Thread.Alerted =&gt;
    END
  END Redisplay;
</PRE>***************************  callback methods  ***************************

<P><PRE>PROCEDURE <A NAME="ReturnAction"><procedure>ReturnAction</procedure></A> (v: T; &lt;* UNUSED *&gt; READONLY event: VBT.KeyRec) =
  BEGIN
    NewlineAndIndent (v)
  END ReturnAction;

PROCEDURE <A NAME="UnlockedReturnAction"><procedure>UnlockedReturnAction</procedure></A> (v: T; READONLY event: VBT.KeyRec) =
  BEGIN
    Thread.Release (v.mu);
    TRY v.returnAction (event) FINALLY Thread.Acquire (v.mu) END
  END UnlockedReturnAction;

PROCEDURE <A NAME="Insert4spaces"><procedure>Insert4spaces</procedure></A> (v: T; &lt;* UNUSED *&gt; READONLY event: VBT.KeyRec) =
  BEGIN
    LOCK v.mu DO v.insert (&quot;    &quot;) END
  END Insert4spaces;

PROCEDURE <A NAME="UnlockedTabAction"><procedure>UnlockedTabAction</procedure></A> (v: T; READONLY event: VBT.KeyRec) =
  BEGIN
    Thread.Release (v.mu);
    TRY v.tabAction (event) FINALLY Thread.Acquire (v.mu) END
  END UnlockedTabAction;
</PRE>************************  Miscellany  ***********************

<P><PRE>&lt;* EXPORTED *&gt;
PROCEDURE <A NAME="Normalize"><procedure>Normalize</procedure></A> (v: T; to: INTEGER := -1) =
  BEGIN
    LOCK v.mu DO v.normalize (to) END
  END Normalize;

PROCEDURE <A NAME="LockedNormalize"><procedure>LockedNormalize</procedure></A> (v: T; to: INTEGER) =
  &lt;* LL = v.mu *&gt;
  CONST name = &quot;Normalize&quot;;
  VAR point: CARDINAL;
  BEGIN
    TRY
      IF to &lt; 0 THEN
        point := v.index ()
      ELSE
        point := MIN (to, v.length ())
      END;
      IF NOT VText.InRegion (v.vtext, 0, point) THEN
        VText.SetStart (v.vtext, 0, point, v.linesShown DIV 2)
      END;
      VBT.Mark (v)
    EXCEPT
    | VTDef.Error (ec) =&gt; v.vterror (name, ec)
    | Rd.EndOfFile =&gt; v.rdeoferror (name)
    | Rd.Failure (ref) =&gt; v.rdfailure (name, ref)
    | Thread.Alerted =&gt;
    END
  END LockedNormalize;

PROCEDURE <A NAME="UnlockedFocus"><procedure>UnlockedFocus</procedure></A> (v: T; gaining: BOOLEAN; time: VBT.TimeStamp) =
  BEGIN
    Thread.Release (v.mu);
    TRY v.focus (gaining, time) FINALLY Thread.Acquire (v.mu) END
  END UnlockedFocus;

PROCEDURE <A NAME="IgnoreFocus"><procedure>IgnoreFocus</procedure></A> (&lt;* UNUSED *&gt; v:     T;
                       &lt;* UNUSED *&gt; gaining: BOOLEAN;
                       &lt;* UNUSED *&gt; time   : VBT.TimeStamp) =
  BEGIN
  END IgnoreFocus;

PROCEDURE <A NAME="UnlockedModified"><procedure>UnlockedModified</procedure></A> (v: T) =
  BEGIN
    Thread.Release (v.mu);
    TRY v.modified () FINALLY Thread.Acquire (v.mu) END
  END UnlockedModified;

PROCEDURE <A NAME="IgnoreModification"><procedure>IgnoreModification</procedure></A> (&lt;* UNUSED *&gt; v: T)  =
  BEGIN
  END IgnoreModification;

PROCEDURE <A NAME="FindSource"><procedure>FindSource</procedure></A> (v   : T;
                      time: VBT.TimeStamp;
                      loc        := TextPortClass.Loc.Next;
                      ignoreCase := TRUE                    ) =
  BEGIN
    TRY
      TextPortClass.FindAndSelect (
        v, v.m.read (VBT.Source, time), time, loc, ignoreCase)
    EXCEPT
    | VBT.Error (ec) =&gt; v.vbterror (&quot;findSource&quot;, ec)
    END
  END FindSource;

PROCEDURE <A NAME="NotFoundProc"><procedure>NotFoundProc</procedure></A> (&lt;* UNUSED *&gt; v: T) =
  BEGIN
    (* SmallIO.PutChar (SmallIO.stderr, '\007') *)
  END NotFoundProc;
</PRE>************************* VBT methods **************************

<P> In this section, we lock v.mu and relay the method-calls to the Model,
   which handles selections, the mouse, etc. 

<P><PRE>VAR
  miscwr    := TextWr.New ();
  debugMisc := FALSE;

PROCEDURE <A NAME="Misc"><procedure>Misc</procedure></A> (v: T; READONLY cd: VBT.MiscRec) =
  BEGIN
    LOCK v.mu DO
      IF debugMisc THEN
        VBTutils.WriteMiscRec (miscwr, cd);
        v.error (TextWr.ToText (miscwr))
      END;
      v.m.misc (cd)
    END
  END Misc;

PROCEDURE <A NAME="Read"><procedure>Read</procedure></A> (v: T; s: VBT.Selection; typecode: CARDINAL): VBT.Value
  RAISES {VBT.Error} =
  &lt;* LL.sup &lt;= VBT.mu *&gt;
  BEGIN
    IF typecode # TYPECODE (TEXT) THEN
      RAISE VBT.Error (VBT.ErrorCode.WrongType)
    ELSE
      LOCK v.mu DO RETURN VBT.FromRef (v.m.read (s, 0)) END
    END
  END Read;

PROCEDURE <A NAME="Write"><procedure>Write</procedure></A> (v       : T;
                 s       : VBT.Selection;
                 value   : VBT.Value;
                 typecode: CARDINAL       ) RAISES {VBT.Error} =
  &lt;* LL.sup &lt;= VBT.mu *&gt;
  BEGIN
    LOCK v.mu DO
      IF typecode # TYPECODE (TEXT) THEN
        RAISE VBT.Error (VBT.ErrorCode.WrongType)
      ELSIF v.readOnly THEN
        RAISE VBT.Error (VBT.ErrorCode.Unwritable)
      ELSE
        TYPECASE value.toRef () OF
        | NULL =&gt; RAISE VBT.Error (VBT.ErrorCode.WrongType)
        | TEXT (t) =&gt; v.m.write (s, 0, t)
        ELSE
          RAISE VBT.Error (VBT.ErrorCode.WrongType)
        END
      END
    END
  END Write;

PROCEDURE <A NAME="Mouse"><procedure>Mouse</procedure></A> (v: T; READONLY cd: VBT.MouseRec) =
  BEGIN
    LOCK v.mu DO
      (* IF v.getKFocus (cd.time) THEN v.m.mouse (cd) END *)
      v.m.mouse (cd)
    END
  END Mouse;

PROCEDURE <A NAME="Position"><procedure>Position</procedure></A> (v: T; READONLY cd: VBT.PositionRec) =
  BEGIN
    LOCK v.mu DO
      IF NOT v.m.dragging THEN     (* skip *)
      ELSIF cd.cp.gone THEN
        VBT.SetCage (v, VBT.GoneCage)
      ELSE
        v.m.position (cd)
      END
    END
  END Position;

PROCEDURE <A NAME="vbterror"><procedure>vbterror</procedure></A> (v: T; msg: TEXT; ec: VBT.ErrorCode) =
  BEGIN
    v.ULerror (msg &amp; &quot;: &quot; &amp; TextPortClass.VBTErrorCodeTexts [ec])
  END vbterror;

PROCEDURE <A NAME="vterror"><procedure>vterror</procedure></A> (v: T; msg: TEXT; ec: VTDef.ErrorCode) =
  BEGIN
    v.ULerror (msg &amp; &quot;: &quot; &amp; VTDef.ErrorCodeTexts [ec])
  END vterror;

PROCEDURE <A NAME="rdfailure"><procedure>rdfailure</procedure></A> (v: T; msg: TEXT; ref: REFANY) =
  BEGIN
    v.ULerror (msg &amp; &quot;: &quot; &amp; RdUtils.FailureText (ref))
  END rdfailure;

PROCEDURE <A NAME="rdeoferror"><procedure>rdeoferror</procedure></A> (v: T; msg: TEXT) =
  BEGIN
    v.ULerror (msg &amp; &quot;: End of file&quot;)
  END rdeoferror;

PROCEDURE <A NAME="UnlockedError"><procedure>UnlockedError</procedure></A> (v: T; msg: TEXT) =
  BEGIN
    Thread.Release (v.mu);
    TRY v.error (msg) FINALLY Thread.Acquire (v.mu) END
  END UnlockedError;

VAR errMu := NEW(MUTEX);

PROCEDURE <A NAME="Error"><procedure>Error</procedure></A> (&lt;* UNUSED *&gt; v: T; msg: TEXT) =
  &lt;* FATAL Wr.Failure, Thread.Alerted *&gt;
  BEGIN
    IF debug THEN
      LOCK errMu DO
        Wr.PutText(Stdio.stderr, msg);
        Wr.PutChar(Stdio.stderr, '\n')
      END
    END
  END Error;
</PRE>************************  Module Initialization  ***********************

<P><PRE>BEGIN
  VAR s: TEXT;
  BEGIN
    s := Env.Get (&quot;TEXTPORTMODEL&quot;);
    IF s # NIL THEN
      IF Text.Equal (s, &quot;ivy&quot;) THEN
        DefaultModel := Model.Ivy
      ELSIF Text.Equal (s, &quot;xterm&quot;) THEN
        DefaultModel := Model.Xterm
      ELSIF Text.Equal (s, &quot;mac&quot;) THEN
        DefaultModel := Model.Mac
      ELSE
        DefaultModel := Model.Emacs
      END
    END;
    debug := Env.Get (&quot;TEXTPORTDEBUG&quot;) # NIL;
    s := Env.Get (&quot;KEYBOARD_MODE&quot;);
    UseDiacritical := s # NIL AND Text.Equal (s, &quot;French&quot;)
  END
END TextPort.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
