<HTML>
<HEAD>
<TITLE>SRC Modula-3: etimer/src/ETimer.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>etimer/src/ETimer.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE> File: ETimer.m3                                             

<P><PRE>UNSAFE (*because of RTHeapRep*)
MODULE <module><implements><A HREF="ETimer.i3">ETimer</A></implements></module>;

IMPORT <A HREF="../../rw/src/Common/Wr.i3">Wr</A>, <A HREF="../../time/src/Common/Time.i3">Time</A>, <A HREF="../../fmtlex/src/Fmt.i3">Fmt</A>, <A HREF="../../runtime/src/common/RTHeapRep.i3">RTHeapRep</A>, <A HREF="../../runtime/src/common/RTHeapDep.i3">RTHeapDep</A>;

REVEAL
  <A NAME="T">T</A> = BRANDED &quot;ETimer.T&quot; REF RECORD
    tag   : TEXT;
    cnt   : INTEGER := 0;
    time  : Time.T := 0.0d+0;
    next  : T := NIL;
  END;

TYPE
  StackIndex = [0..99];

TYPE
  Rec = RECORD
    timer : T;
    start : Time.T;
  END;

CONST
  Second      = 1.0d0;
  Millisecond = Second / 1000.0d0;

VAR
  mu := NEW (MUTEX);  (* protects the following global variables *)
  enabled    : BOOLEAN    := TRUE;
  all_timers : T          := NIL;
  misc       : T          := NIL;
  gc         : T          := NIL;
  tos        : StackIndex := 0;
  stack      : ARRAY StackIndex OF Rec;

PROCEDURE <A NAME="New"><procedure>New</procedure></A> (lab: TEXT): T =
  VAR t := NEW (T, tag := lab);
  BEGIN
    LOCK mu DO
      t.next := all_timers;
      all_timers := t;
    END;
    RETURN t;
  END New;

PROCEDURE <A NAME="Relabel"><procedure>Relabel</procedure></A> (t: T;  lab: TEXT) =
  BEGIN
    t.tag := lab;
  END Relabel;

PROCEDURE <A NAME="Push"><procedure>Push</procedure></A> (t: T) =
  VAR now: Time.T;
  BEGIN
    IF NOT enabled THEN RETURN END;
    LOCK mu DO
      now := Time.Now ();

      (* update the suspended timer *)
      IF (tos &gt; 0) THEN
        WITH top = stack [tos-1], t = top.timer.time DO
          t := t + (now - top.start);
        END;
      END;

      (* start the new timer *)
      WITH top = stack [tos] DO
        top.start := now;
        top.timer := t;
      END;
      INC (t.cnt);

      INC (tos);
    END;
  END Push;

PROCEDURE <A NAME="Pop"><procedure>Pop</procedure></A> () =
  VAR now: Time.T;
  BEGIN
    IF NOT enabled THEN RETURN END;
    LOCK mu DO
      now := Time.Now ();
      DEC (tos);

      (* update the popped timer *)
      WITH top = stack [tos], t = top.timer.time DO
        t := t + (now - top.start);
      END;

      (* restart the suspended timer *)
      IF (tos &gt; 0) THEN
        stack[tos-1].start := now;
      END;
    END;
  END Pop;

PROCEDURE <A NAME="Dump"><procedure>Dump</procedure></A> (wr: Wr.T) =
  &lt;*FATAL ANY*&gt;
  CONST MinPrintable = 5.0d+0 * Millisecond;
  VAR t: T;  now, total: Time.T := 0.0d+0;
  BEGIN
    LOCK mu DO
      (* update the currently running timer, but leave it running *)
      IF (tos &gt; 0) THEN
        now := Time.Now ();
        WITH top = stack [tos-1], t = top.timer.time DO
          t := t + (now - top.start);
          top.start := now;
        END;
      END;

      t := all_timers; (* capture the head of the list *)
    END;

    (* sleazy: don't lock the timers during the output since
       writing may cause allocation which may cause garbage
       collection which will cause a call to Push which will
       cause a deadlock... *)

    (* write the report *)
    Wr.PutText (wr, &quot;\n seconds  #times  operation\n&quot;);
    WHILE (t # NIL) DO
      IF (t.cnt &gt; 0) AND (t.time &gt;= MinPrintable) THEN
        Wr.PutText (wr, FmtTime (t.time));
        IF (t # misc)
          THEN Wr.PutText (wr, Fmt.Pad (Fmt.Int (t.cnt), 8));
          ELSE Wr.PutText (wr, &quot;        &quot;);
        END;
        Wr.PutText (wr, &quot;  &quot;);
        IF (t.tag # NIL) THEN Wr.PutText (wr, t.tag); END;
        Wr.PutText (wr, &quot;\n&quot;);
      END;
      total := total + t.time;
      t := t.next;
    END;
    Wr.PutText (wr, &quot;---------------------------------------------------\n&quot;);
    Wr.PutText (wr, FmtTime (total));
    Wr.PutText (wr, &quot;          TOTAL\n\n&quot;);
  END Dump;

PROCEDURE <A NAME="FmtTime"><procedure>FmtTime</procedure></A> (t: Time.T): TEXT =
  BEGIN
    RETURN Fmt.Pad (Fmt.LongReal (t/Second, Fmt.Style.Fix, 2), 8);
  END FmtTime;

PROCEDURE <A NAME="Elapsed"><procedure>Elapsed</procedure></A> (t: T): LONGREAL =
  VAR now: Time.T;
  BEGIN
    LOCK mu DO
      (* update t if it's running, but leave it running *)
      IF (tos &gt; 0) THEN
        WITH top = stack [tos-1], z = top.timer.time DO
          IF (top.timer = t) THEN
            now := Time.Now ();
            z := z + (now - top.start);
            top.start := now;
          END;
        END;
      END;

      (* finally, return t's accumulated time *)
      RETURN t.time / Second;
    END;
  END Elapsed;

PROCEDURE <A NAME="TotalElapsed"><procedure>TotalElapsed</procedure></A> (): LONGREAL =
  VAR t: T;  total: Time.T := 0.0d+0;
  BEGIN
    LOCK mu DO
      t := all_timers;
      WHILE (t # NIL) DO
        total := total + t.time;
        t := t.next;
      END;
    END;
    RETURN total / Second;
  END TotalElapsed;

PROCEDURE <A NAME="Next"><procedure>Next</procedure></A> (prev: T): T =
  BEGIN
    LOCK mu DO
      IF (prev = NIL)
        THEN RETURN all_timers;
        ELSE RETURN prev.next;
      END;
    END;
  END Next;

TYPE
  GCClosure = RTHeapRep.MonitorClosure OBJECT OVERRIDES
    before := StartGC;
    after  := StopGC;
  END;

PROCEDURE <A NAME="StartGC"><procedure>StartGC</procedure></A> (&lt;*UNUSED*&gt; cl: GCClosure) =
  BEGIN
    Push (gc);
  END StartGC;

PROCEDURE <A NAME="StopGC"><procedure>StopGC</procedure></A> (&lt;*UNUSED*&gt; cl: GCClosure) =
  BEGIN
    Pop ();
  END StopGC;

PROCEDURE <A NAME="Enable"><procedure>Enable</procedure></A> () =
  BEGIN
    IF enabled THEN RETURN END;
    LOCK mu DO
      enabled := TRUE;
      IF NOT RTHeapDep.VM THEN
        RTHeapRep.RegisterMonitor (NEW (GCClosure));
        (* VM = TRUE =&gt; GC times are smeared across many page faults
           so the beginning and ending of collections aren't meaningful
           points to collect timing info. *)
      END;
    END;
  END Enable;

PROCEDURE <A NAME="Reset"><procedure>Reset</procedure></A> (t: T) =
  BEGIN
    LOCK mu DO
      t.cnt := 0;
      t.time := 0.0d+0;
      IF (tos &gt; 0) AND (stack[tos-1].timer = t) THEN
        (* t is running, reset its start time *)
        stack[tos-1].start := Time.Now ();
      END;
    END;
  END Reset;

PROCEDURE <A NAME="ResetAll"><procedure>ResetAll</procedure></A> () =
  VAR t: T;
  BEGIN
    LOCK mu DO
      t := all_timers;
      WHILE (t # NIL) DO
        t.cnt := 0;
        t.time := 0.0d+0;
        t := t.next;
      END;
      IF (tos &gt; 0) THEN
        (* reset the running timer *)
        stack[tos-1].start := Time.Now ();
      END;
    END;
  END ResetAll;

BEGIN
  misc := New (&quot;other&quot;);
  gc   := New (&quot;garbage collection&quot;);
  Push (misc);
  enabled := FALSE;
END ETimer.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
