<HTML>
<HEAD>
<TITLE>SRC Modula-3: formsvbt/src/Macro.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>formsvbt/src/Macro.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                                           </EM></BLOCKQUOTE><PRE>

MODULE <module><implements><A HREF="Macro.i3">Macro</A></implements></module>;
</PRE> In this module, we handle the reading of S-expressions, as well as
   the implementation of macros. 

<P><PRE>IMPORT <A HREF="../../types/src/ASCII.i3">ASCII</A>, <A HREF="../../atom/src/Atom.i3">Atom</A>, <A HREF="../../libm3/derived/AtomRefTbl.i3">AtomRefTbl</A>, <A HREF="../../fmtlex/src/Fmt.i3">Fmt</A>, <A HREF="FormsVBT.i3">FormsVBT</A>, <A HREF="FVRuntime.i3">FVRuntime</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../libm3/derived/RefList.i3">RefList</A>, <A HREF="../../sx/src/Sx.i3">Sx</A>,
       <A HREF="../../text/src/Text.i3">Text</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>;

FROM <A HREF="FVRuntime.i3">FVRuntime</A> IMPORT FVSyntax, ToText;
FROM <A HREF="RefListUtils.i3">RefListUtils</A> IMPORT AssocQ, Equal, NthTail, Pop, Push, SetNth;

REVEAL
  <A NAME="T">T</A> = Public BRANDED OBJECT
        name    : Atom.T;
        formals : RefList.T := NIL;
        expander: Op;            (* compiled object *)
        boa     : BOOLEAN;       (* actuals are not named *)
      OVERRIDES
        apply := Apply
      END;

TYPE ReadMacro = Sx.ReadMacro OBJECT bqLevel: REF CARDINAL END;

VAR                              (* CONST *)
  qAppend     := Atom.FromText (&quot;Append&quot;);
  qCons       := Atom.FromText (&quot;Cons&quot;);
  qLength     := Atom.FromText (&quot;Length&quot;);
  qList       := Atom.FromText (&quot;List&quot;);
  qListStar   := Atom.FromText (&quot;List*&quot;);
  qNth        := Atom.FromText (&quot;Nth&quot;);
  qNthTail    := Atom.FromText (&quot;NthTail&quot;);
  qEqual      := Atom.FromText (&quot;Equal&quot;);
  qIf         := Atom.FromText (&quot;IF&quot;);
  qAnd        := Atom.FromText (&quot;AND&quot;);
  qNot        := Atom.FromText (&quot;NOT&quot;);
  qOr         := Atom.FromText (&quot;OR&quot;);
  qEQ         := Atom.FromText (&quot;=&quot;);
  qGE         := Atom.FromText (&quot;&gt;=&quot;);
  qGT         := Atom.FromText (&quot;&gt;&quot;);
  qLE         := Atom.FromText (&quot;&lt;=&quot;);
  qLT         := Atom.FromText (&quot;&lt;&quot;);
  qNIL        := Atom.FromText (&quot;NIL&quot;);
  qMinus      := Atom.FromText (&quot;-&quot;);
  qPlus       := Atom.FromText (&quot;+&quot;);
  qCat        := Atom.FromText (&quot;Cat&quot;);
  qTextEmpty  := Atom.FromText (&quot;Empty&quot;);
  qTextSub    := Atom.FromText (&quot;Sub&quot;);
  qFromName   := Atom.FromText (&quot;Intern&quot;);
  qSymbolName := Atom.FromText (&quot;SymbolName&quot;);

PROCEDURE <A NAME="Parse"><procedure>Parse</procedure></A> (list: RefList.T): T RAISES {FormsVBT.Error} =
  (* list = (name [BOA] formals bqexp). *)
  VAR
    formals: RefList.T;
    res             := NEW (T);
    n               := RefList.Length (list);
  PROCEDURE err (msg: TEXT; x: REFANY := &quot;&quot;) RAISES {FormsVBT.Error} =
    BEGIN
      RAISE
        FormsVBT.Error (Fmt.F (&quot;Illegal Macro form: %s %s&quot;, msg, ToText (x)))
    END err;
  BEGIN
    res.boa := n = 4 AND list.tail.head = FVRuntime.qBOA;
    IF NOT res.boa AND NOT n = 3 THEN err (&quot;Syntax error&quot;) END;
    TYPECASE Pop (list) OF
    | NULL =&gt; err (&quot;Macro name is NIL&quot;)
    | Atom.T (s) =&gt; res.name := s
    | REFANY (r) =&gt; err (&quot;Macro name isn't a symbol: &quot;, r)
    END;
    IF res.boa THEN list := list.tail END;
    TYPECASE Pop (list) OF
    | RefList.T (x) =&gt; formals := x
    | REFANY (x) =&gt; err (&quot;Bad list of formals: &quot;, x)
    END;
    WHILE formals # NIL DO
      TYPECASE Pop (formals) OF
      | NULL =&gt; err (&quot;Null formal&quot;)
      | Atom.T (s) =&gt;
          IF AssocQ (res.formals, s) # NIL THEN
            err (&quot;Duplicate formal: &quot;, s)
          ELSE
            Push (res.formals, RefList.List2 (s, NoDefault))
          END
      | RefList.T (pair) =&gt;
          IF RefList.Length (pair) # 2 THEN
            err (&quot;Bad formal&quot;, pair)
          ELSE
            TYPECASE pair.head OF
            | Atom.T (s) =&gt;
                IF AssocQ (res.formals, s) # NIL THEN
                  err (&quot;Duplicate formal: &quot;, s)
                ELSE
                  Push (res.formals, RefList.List2 (s, pair.tail.head))
                END
            ELSE
              err (&quot;Bad formal&quot;, pair)
            END
          END
      | REFANY (r) =&gt; err (&quot;Formals must be symbols: &quot;, r)
      END
    END;
    res.formals := RefList.ReverseD (res.formals);
    res.expander := Compile (list.head, res.formals, RefanyTC);
    RETURN res
  END Parse;

CONST RefanyTC = -1;
VAR
  TextTC    := TYPECODE (TEXT);
  ListTC    := TYPECODE (RefList.T);
  IntegerTC := TYPECODE (REF INTEGER);
  RealTC    := TYPECODE (REF REAL);
  NullTC    := TYPECODE (NULL);
  BooleanTC := TYPECODE (REF BOOLEAN);
  SymbolTC  := TYPECODE (Atom.T);

VAR
  NullOp := NEW (
              Op, args := RefList.List1 (NIL), tc := NullTC, eval := EvalQuote);
</PRE><P>
CONST LastTypeIndex = 7;
<P>
TYPE TypeIndex = [0 .. LastTypeIndex];
<P>
VAR TypeCodes: ARRAY TypeIndex OF INTEGER;
<P>
PROCEDURE TypeCodeIndex (tc: INTEGER): TypeIndex =
  BEGIN
    FOR i := FIRST (TypeIndex) TO LAST (TypeIndex) DO
      IF tc = TypeCodes [i] THEN RETURN i END
    END;
    &lt;* ASSERT FALSE *&gt;
    END TypeCodeIndex;
<P>
PROCEDURE InitTypeCodes () =
  PROCEDURE OK (a, b: TypeIndex) =
    BEGIN
      ComparableTypes [a, b] := TRUE;
      ComparableTypes [b, a] := TRUE
    END OK;
  BEGIN
    TypeCodes := ARRAY TypeIndex OF
                   INTEGER {RefanyTC, TextTC, ListTC, IntegerTC, RealTC,
                            NullTC, BooleanTC, SymbolTC};
    FOR i := FIRST (TypeIndex) TO LAST (TypeIndex) DO
      FOR j := FIRST (TypeIndex) TO LAST (TypeIndex) DO
        ComparableTypes [i, j] := i = j
      END
    END;
    WITH ref     = TypeCodeIndex (RefanyTC),
         text    = TypeCodeIndex (TextTC),
         list    = TypeCodeIndex (ListTC),
         integer = TypeCodeIndex (IntegerTC),
         real    = TypeCodeIndex (RealTC),
         null    = TypeCodeIndex (NullTC),
         boolean = TypeCodeIndex (BooleanTC),
         symbol  = TypeCodeIndex (SymbolTC)  DO
      OK (ref, text);
      OK (ref, list);
      OK (ref, null);
      OK (ref, symbol);
      OK (text, null);
      OK (list, null);
    END;
  END InitTypeCodes;
<P>
VAR ComparableTypes: ARRAY TypeIndex, TypeIndex OF BOOLEAN;
<P>
&lt;* UNUSED *&gt; PROCEDURE Comparable (a, b: INTEGER): BOOLEAN =
  BEGIN
    RETURN ComparableTypes [TypeCodeIndex (a), TypeCodeIndex (b)]
  END Comparable;


<P><PRE>VAR VarOps := ARRAY [0 .. 5] OF Op {NIL, ..};

PROCEDURE <A NAME="Compile"><procedure>Compile</procedure></A> (exp: REFANY; formals: RefList.T; tc := RefanyTC): Op
  RAISES {FormsVBT.Error} =
  VAR
    value: REFANY;
    c    : Compiler;
  BEGIN
    TYPECASE exp OF
    | NULL =&gt; Check (tc, NullTC); RETURN NullOp
    | Atom.T (s) =&gt;
        IF s = qNIL THEN Check (tc, NullTC); RETURN NullOp END;
        WITH p = Position (formals, s) DO
          IF p = -1 THEN
            RAISE FormsVBT.Error (&quot;Unbound variable: &quot; &amp; Atom.ToText (s))
          ELSIF p &lt; NUMBER (VarOps) THEN
            RETURN VarOps [p]
          ELSE
            RETURN NEW (Op, tc := p, eval := EvalVar)
          END
        END
    | TEXT =&gt;
        Check (tc, TextTC);
        RETURN
          NEW (Op, args := RefList.List1 (exp), tc := TextTC, eval := EvalQuote)
    | REF INTEGER =&gt;
        Check (tc, IntegerTC);
        RETURN NEW (Op, args := RefList.List1 (exp), tc := IntegerTC,
                    eval := EvalQuote)
    | REF REAL =&gt;
        Check (tc, RealTC);
        RETURN
          NEW (Op, args := RefList.List1 (exp), tc := RealTC, eval := EvalQuote)
    | REF BOOLEAN =&gt;
        Check (tc, BooleanTC);
        RETURN NEW (Op, args := RefList.List1 (exp), tc := BooleanTC,
                    eval := EvalQuote)
    | RefList.T (x) =&gt;
        WITH f    = x.head,
             args = x.tail,
             n    = RefList.Length (args) DO
          TYPECASE f OF
          | Atom.T (s) =&gt;
              IF Ctable.get (s, value) THEN
                c := value;
                Check (tc, c.tc, c.n, n);
                RETURN c.compile (c, args, formals, tc)
              END
          ELSE
          END
        END
    ELSE
    END;
    RAISE FormsVBT.Error (
            &quot;Illegal expression in macro definition:&quot; &amp; ToText (exp))
  END Compile;

TYPE Display = REF ARRAY OF REFANY;

PROCEDURE <A NAME="Fault"><procedure>Fault</procedure></A> (typeName: TEXT; arg: REFANY): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RAISE FormsVBT.Error (
            Fmt.F (&quot;A %s was required here: %s&quot;, typeName, ToText (arg)))
  END Fault;

PROCEDURE <A NAME="Apply"><procedure>Apply</procedure></A> (m: T; actuals: RefList.T): REFANY RAISES {FormsVBT.Error} =
  PROCEDURE err (msg: TEXT; actuals: REFANY := &quot;&quot;) RAISES {FormsVBT.Error} =
    BEGIN
      RAISE FormsVBT.Error (Fmt.F (&quot;Error in call to macro %s: %s %s&quot;,
                                   Atom.ToText (m.name), msg, ToText (actuals)))
    END err;
  VAR
    ac              := RefList.Length (actuals);
    fc              := RefList.Length (m.formals);
    d               := NEW (Display, fc);
    vars: RefList.T := NIL;
    pair: RefList.T;
  BEGIN
    IF ac &gt; fc THEN err (&quot;Too many arguments: &quot;, Sx.FromInt (ac)) END;
    IF m.boa THEN
      FOR i := 0 TO ac - 1 DO d [i] := Pop (actuals) END;
      FOR i := ac TO fc - 1 DO
        pair := RefList.Nth (m.formals, i);
        IF pair.tail.head = NoDefault THEN
          err (&quot;Argument has no default: &quot;, pair.head)
        ELSE
          d [i] := pair.tail.head
        END
      END
    ELSE
      IF ac # fc THEN
        FOR i := 0 TO fc - 1 DO
          pair := RefList.Nth (m.formals, i);
          d [i] := pair.tail.head
        END
      END;
      WHILE actuals # NIL DO
        TYPECASE Pop (actuals) OF
        | NULL =&gt; err (&quot;NIL argument&quot;)
        | RefList.T (y) =&gt;
            IF RefList.Length (y) # 2 THEN
              err (&quot;Illegal argument: &quot;, y)
            ELSE
              WITH p = Position (m.formals, y.head) DO
                IF p = -1 THEN
                  err (&quot;Unknown variable: &quot;, y.head)
                ELSIF RefList.Member (vars, y.head) THEN
                  err (&quot;Argument passed twice: &quot;, y.head)
                ELSE
                  d [p] := y.tail.head;
                  Push (vars, y.head)
                END
              END
            END
        | REFANY (r) =&gt; err (&quot;Illegal argument: &quot;, r)
        END
      END;
      IF fc # ac THEN
        FOR i := 0 TO fc - 1 DO
          IF d [i] = NoDefault THEN
            pair := RefList.Nth (m.formals, i);
            err (&quot;No value was supplied for &quot;, pair.head)
          END
        END
      END
    END;
    RETURN m.expander.eval (m.expander, d)
  END Apply;

PROCEDURE <A NAME="Eval"><procedure>Eval</procedure></A> (op: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN op.eval (op, d)
  END Eval;

TYPE
  Compiler = OBJECT
               tc: INTEGER;      (* the typecode of the result *)
               n : CARDINAL;     (* the number of parameters *)
               compile: CProc    (* the compilation &quot;method&quot; *)
             END;
  CProc =
    PROCEDURE (c: Compiler; args: RefList.T; formals: RefList.T; tc: INTEGER):
      Op RAISES {FormsVBT.Error};
  Test = {GE, GT, LE, LT, EQ};
  ComparisonCompiler = Compiler OBJECT test: Test END;
  Op = OBJECT
         tc                := RefanyTC;
         args: RefList.T;
         eval: PROCEDURE (op: Op; d: Display): REFANY RAISES {FormsVBT.Error}
       END;
  ComparisonOp = Op OBJECT test: Test END;

VAR Ctable := NEW (AtomRefTbl.Default).init (20); (* Maps symbol -&gt; compiler *)

PROCEDURE <A NAME="InitCompilers"><procedure>InitCompilers</procedure></A> () =
  PROCEDURE f (s: Atom.T; tc: INTEGER; n: CARDINAL; compile: CProc) =
    VAR c := NEW (Compiler, tc := tc, n := n, compile := compile);
    BEGIN
      EVAL Ctable.put (s, c)
    END f;
  PROCEDURE g (s: Atom.T; test: Test) =
    BEGIN
      EVAL Ctable.put (
             s,
             NEW (ComparisonCompiler, tc := BooleanTC, n := LAST (CARDINAL),
                  compile := CompileComparison, test := test))
    END g;
  BEGIN
    f (qAnd, BooleanTC, LAST (CARDINAL), CompileAnd);
    f (qAppend, ListTC, 2, CompileAppend);
    f (FVRuntime.qBackquote, RefanyTC, 1, CompileBackquote);
    f (qCat, TextTC, LAST (CARDINAL), CompileCat);
    f (qCons, ListTC, 2, CompileCons);
    f (qFromName, SymbolTC, 1, CompileFromName);
    f (qIf, RefanyTC, 3, CompileIf);
    f (qList, ListTC, LAST (CARDINAL), CompileList);
    f (qEqual, BooleanTC, 2, CompileEqual);
    f (qLength, IntegerTC, 1, CompileLength);
    f (qListStar, ListTC, LAST (CARDINAL), CompileListStar);
    f (qMinus, RefanyTC, LAST (CARDINAL), CompileMinus);
    f (qNot, BooleanTC, 1, CompileNot);
    f (qNth, RefanyTC, 2, CompileNth);
    f (qNthTail, ListTC, 2, CompileNthTail);
    f (qOr, BooleanTC, LAST (CARDINAL), CompileOr);
    f (qPlus, RefanyTC, LAST (CARDINAL), CompilePlus);
    f (FVRuntime.qQuote, RefanyTC, 1, CompileQuote);
    f (qSymbolName, TextTC, 1, CompileSymbolName);
    f (qTextEmpty, BooleanTC, 1, CompileEmpty);
    (* f (qTextEqual, BooleanTC, 2, CompileTextEqual); *)
    (* f (qTextLength, IntegerTC, 1, CompileTextLength); *)
    f (qTextSub, TextTC, 3, CompileSub);
    g (qEQ, Test.EQ);
    g (qGE, Test.GE);
    g (qGT, Test.GT);
    g (qLE, Test.LE);
    g (qLT, Test.LT)
  END InitCompilers;

PROCEDURE <A NAME="Check"><procedure>Check</procedure></A> (TCwanted, TCgonnaGet       : INTEGER;
                 argCountWanted, argCountGot: CARDINAL  := 0)
  RAISES {FormsVBT.Error} =
  BEGIN
    IF argCountWanted # argCountGot AND argCountWanted # LAST (CARDINAL) THEN
      RAISE FormsVBT.Error (
              Fmt.F (&quot;Wrong number of args: %s instead of %s&quot;,
                     Fmt.Int (argCountGot), Fmt.Int (argCountWanted)))
    ELSIF TCwanted # RefanyTC AND TCgonnaGet # NullTC
            AND TCgonnaGet # TCwanted
            AND TCgonnaGet # RefanyTC (* NARROW at runtime *)
      THEN
      RAISE FormsVBT.Error (&quot;Invalid type&quot;)
    END
  END Check;

PROCEDURE <A NAME="CompileQuote"><procedure>CompileQuote</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                     args   : RefList.T;
                        &lt;* UNUSED *&gt; formals: RefList.T;
                                     tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR actualTC := TYPECODE (args.head);
  BEGIN
    Check (tc, actualTC);
    RETURN NEW (Op, args := args, tc := actualTC, eval := EvalQuote)
  END CompileQuote;

PROCEDURE <A NAME="EvalQuote"><procedure>EvalQuote</procedure></A> (x: Op; &lt;* UNUSED *&gt; d: Display): REFANY =
  BEGIN
    RETURN x.args.head
  END EvalQuote;

PROCEDURE <A NAME="CompileCons"><procedure>CompileCons</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                    args   : RefList.T;
                                    formals: RefList.T;
                                    tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, RefanyTC);
    args.tail.head := Compile (args.tail.head, formals, ListTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalCons)
  END CompileCons;

PROCEDURE <A NAME="EvalCons"><procedure>EvalCons</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN
      RefList.Cons (Eval (x.args.head, d), GetList (Eval (x.args.tail.head, d)))
  END EvalCons;

PROCEDURE <A NAME="CompileLength"><procedure>CompileLength</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                          args   : RefList.T;
                                          formals: RefList.T;
                                          tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, ListTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalLength)
  END CompileLength;

PROCEDURE <A NAME="EvalLength"><procedure>EvalLength</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE Eval (x.args.head, d) OF
    | RefList.T (list) =&gt; RETURN Sx.FromInt (RefList.Length (list))
    | TEXT (t) =&gt; RETURN Sx.FromInt (Text.Length (t))
    | REFANY (ref) =&gt; RETURN Fault (&quot;list or text&quot;, ref)
    END
  END EvalLength;

PROCEDURE <A NAME="CompileEqual"><procedure>CompileEqual</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                         args   : RefList.T;
                                         formals: RefList.T;
                                         tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, ListTC);
    args.tail.head := Compile (args.tail.head, formals, ListTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalEqual)
  END CompileEqual;

PROCEDURE <A NAME="EvalEqual"><procedure>EvalEqual</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN
      Sx.FromBool (Equal (Eval (x.args.head, d), Eval (x.args.tail.head, d)))
  END EvalEqual;

PROCEDURE <A NAME="CompileNth"><procedure>CompileNth</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                   args   : RefList.T;
                                   formals: RefList.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, ListTC);
    args.tail.head := Compile (args.tail.head, formals, IntegerTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalNth)
  END CompileNth;

PROCEDURE <A NAME="EvalNth"><procedure>EvalNth</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    list := GetList (Eval (x.args.head, d));
    n    := GetRefCardinal (Eval (x.args.tail.head, d))^;
  BEGIN
    IF n &lt; RefList.Length (list) THEN
      RETURN RefList.Nth (list, n)
    ELSE
      RAISE FormsVBT.Error (
              Fmt.F (&quot;RefList.Nth (..., %s): range error&quot;, Fmt.Int (n)))
    END
  END EvalNth;

PROCEDURE <A NAME="CompileNthTail"><procedure>CompileNthTail</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                       args   : RefList.T;
                                       formals: RefList.T;
                                       tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, ListTC);
    args.tail.head := Compile (args.tail.head, formals, IntegerTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalNthTail)
  END CompileNthTail;

PROCEDURE <A NAME="EvalNthTail"><procedure>EvalNthTail</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    list := GetList (Eval (x.args.head, d));
    n    := GetRefCardinal (Eval (x.args.tail.head, d))^;
  BEGIN
    IF n &lt;= RefList.Length (list) THEN
      RETURN NthTail (list, n)
    ELSE
      RAISE FormsVBT.Error (
              Fmt.F (&quot;RefList.NthTail (..., %s): range error&quot;, Fmt.Int (n)))
    END
  END EvalNthTail;

PROCEDURE <A NAME="CompileList"><procedure>CompileList</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                    args   : RefList.T;
                                    formals: RefList.T;
                                    tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalList);
  BEGIN
    WHILE args # NIL DO
      args.head := Compile (args.head, formals);
      args := args.tail
    END;
    RETURN res
  END CompileList;

PROCEDURE <A NAME="EvalList"><procedure>EvalList</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    res: RefList.T := NIL;
    ops         := x.args;
  BEGIN
    WHILE ops # NIL DO Push (res, Eval (Pop (ops), d)) END;
    RETURN RefList.ReverseD (res)
  END EvalList;

PROCEDURE <A NAME="CompileListStar"><procedure>CompileListStar</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                        args   : RefList.T;
                                        formals: RefList.T;
                                        tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalListStar);
  BEGIN
    WHILE args # NIL DO
      args.head := Compile (args.head, formals);
      args := args.tail
    END;
    RETURN res
  END CompileListStar;

PROCEDURE <A NAME="EvalListStar"><procedure>EvalListStar</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    ops       := x.args;
    op   : Op := Pop (ops);
    first     := RefList.List1 (Eval (op, d));
    last      := first;
  BEGIN
    WHILE ops.tail # NIL DO
      op := Pop (ops);
      Push (last.tail, Eval (op, d));
      last := last.tail
    END;
    op := ops.head;
    last.tail := GetList (Eval (op, d));
    RETURN first
  END EvalListStar;

PROCEDURE <A NAME="CompileAppend"><procedure>CompileAppend</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                      args   : RefList.T;
                                      formals: RefList.T;
                                      tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalAppend);
  BEGIN
    WHILE args # NIL DO
      args.head := Compile (args.head, formals, ListTC);
      args := args.tail
    END;
    RETURN res
  END CompileAppend;

PROCEDURE <A NAME="EvalAppend"><procedure>EvalAppend</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    res : RefList.T := NIL;
    args         := RefList.Reverse (x.args);
  BEGIN
    WHILE args # NIL DO
      res := RefList.Append (GetList (Eval (args.head, d)), res);
      args := args.tail
    END;
    RETURN res
  END EvalAppend;

PROCEDURE <A NAME="CompileIf"><procedure>CompileIf</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                  args   : RefList.T;
                                  formals: RefList.T;
                                  tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    SetNth (args, 0, Compile (RefList.Nth (args, 0), formals, BooleanTC));
    SetNth (args, 1, Compile (RefList.Nth (args, 1), formals, RefanyTC));
    SetNth (args, 2, Compile (RefList.Nth (args, 2), formals, RefanyTC));
    RETURN NEW (Op, args := args, tc := tc, eval := EvalIf)
  END CompileIf;

PROCEDURE <A NAME="EvalIf"><procedure>EvalIf</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    IF GetBoolean (Eval (RefList.Nth (x.args, 0), d)) THEN
      RETURN Eval (RefList.Nth (x.args, 1), d)
    ELSE
      RETURN Eval (RefList.Nth (x.args, 2), d)
    END
  END EvalIf;

PROCEDURE <A NAME="CompileAnd"><procedure>CompileAnd</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                   args   : RefList.T;
                                   formals: RefList.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalAnd);
  BEGIN
    WHILE args # NIL DO
      args.head := Compile (args.head, formals, BooleanTC);
      args := args.tail
    END;
    RETURN res
  END CompileAnd;

PROCEDURE <A NAME="EvalAnd"><procedure>EvalAnd</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    WHILE x.args # NIL DO
      IF NOT GetBoolean (Eval (Pop (x.args), d)) THEN
        RETURN Sx.False
      END
    END;
    RETURN Sx.True
  END EvalAnd;

PROCEDURE <A NAME="CompileOr"><procedure>CompileOr</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                   args   : RefList.T;
                                   formals: RefList.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalOr);
  BEGIN
    WHILE args # NIL DO
      args.head := Compile (args.head, formals, BooleanTC);
      args := args.tail
    END;
    RETURN res
  END CompileOr;

PROCEDURE <A NAME="EvalOr"><procedure>EvalOr</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    WHILE x.args # NIL DO
      IF GetBoolean (Eval (Pop (x.args), d)) THEN RETURN Sx.True END
    END;
    RETURN Sx.False
  END EvalOr;

PROCEDURE <A NAME="CompileNot"><procedure>CompileNot</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                   args   : RefList.T;
                                   formals: RefList.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalNot);
  BEGIN
    args.head := Compile (args.head, formals, BooleanTC);
    RETURN res
  END CompileNot;

PROCEDURE <A NAME="EvalNot"><procedure>EvalNot</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    IF GetBoolean (Eval (x.args.head, d)) THEN
      RETURN Sx.False
    ELSE
      RETURN Sx.True
    END
  END EvalNot;

PROCEDURE <A NAME="CompileBackquote"><procedure>CompileBackquote</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                         args   : RefList.T;
                                         formals: RefList.T;
                            &lt;* UNUSED *&gt; tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    (* There is no EvalBackquote.  Backquoted S-expressions simply expand into
       other S-expressions, which are in turn compiled. *)
    RETURN Compile (Backquote (RefList.Nth (args, 0)), formals)
  END CompileBackquote;

PROCEDURE <A NAME="Backquote"><procedure>Backquote</procedure></A> (exp: REFANY): REFANY RAISES {FormsVBT.Error} =
  (* This returns a Lisp-like S-expression that can be passed to Eval to
     produce a new FormsVBT expression.  The only operators are QUOTE,
     LIST, LIST*, and APPEND. *)
  BEGIN
    TYPECASE exp OF
    | NULL =&gt; RETURN NIL
    | RefList.T (list) =&gt;
        IF list.head = FVRuntime.qComma THEN
          RETURN list.tail.head
        ELSIF list.head = FVRuntime.qBackquote THEN
          RETURN Backquote (Backquote (list.tail.head))
        ELSE
          TYPECASE list.head OF
          | NULL =&gt;
          | RefList.T (sublist) =&gt;
              IF sublist.head = FVRuntime.qCommaAtsign THEN
                RETURN RefList.List3 (qAppend, sublist.tail.head,
                                   Backquote (list.tail))
              END
          ELSE
          END;
          RETURN Combine (Backquote (list.head), Backquote (list.tail))
        END
    ELSE
    END;
    RETURN RefList.List2 (FVRuntime.qQuote, exp)
  END Backquote;

PROCEDURE <A NAME="Combine"><procedure>Combine</procedure></A> (car, cdr: REFANY): REFANY =
  BEGIN
    (* This implementation attempts to recycle cons-cells wherever possible. *)
    TYPECASE car OF
    | NULL =&gt;
        TYPECASE cdr OF
        | NULL =&gt;
            (* (cons NIL NIL) -&gt; (QUOTE (NIL)) *)
            RETURN RefList.List2 (FVRuntime.qQuote, RefList.List1 (NIL))
        | RefList.T (cdr) =&gt;
            IF cdr.head = FVRuntime.qQuote THEN
              (* (cons NIL (QUOTE x)) -&gt; (QUOTE (NIL .  x)) *)
              cdr.tail.head := RefList.Cons (NIL, cdr.tail.head);
              RETURN cdr
            END
        ELSE
        END
    | RefList.T (car) =&gt;
        IF car.head = FVRuntime.qQuote THEN
          TYPECASE cdr OF
          | NULL =&gt;
              (* (cons (QUOTE x) NIL) -&gt; (QUOTE (x)) *)
              car.tail := RefList.List1 (car.tail);
              RETURN car
          | RefList.T (cdr) =&gt;
              IF cdr.head = FVRuntime.qQuote THEN
                (* (cons (QUOTE x) (QUOTE y)) -&gt; (QUOTE (x .  y)) *)
                car.tail.tail := cdr.tail.head;
                cdr.head := car.tail;
                cdr.tail := NIL;
                car.tail := cdr;
                RETURN car
                (* RETURN RefList.List2 ( qQuote, RefList.New (car.tail.head,
                   cdr.tail.head)) *)
              ELSIF cdr.head = qList OR cdr.head = qListStar THEN
                Push (cdr.tail, car);
                RETURN cdr
              END
          ELSE
          END
        ELSE
          TYPECASE cdr OF
          | NULL =&gt;
              (* (cons x NIL) -&gt; (LIST x) *)
              RETURN RefList.List2 (qList, car)
          | RefList.T (cdr) =&gt;
              IF cdr.head = qList OR cdr.head = qListStar THEN
                (* (cons x (LIST .  y)) -&gt; (LIST x .  y) *)
                Push (cdr.tail, car);
                RETURN cdr
              END
          ELSE &lt;* ASSERT FALSE *&gt;
          END
        END
    ELSE
    END;
    RETURN RefList.List3 (qListStar, car, cdr)
  END Combine;
</PRE><P>
PROCEDURE CompileEquals (&lt;* UNUSED *&gt; self   : Compiler;
                                      args   : RefList.T;
                                      formals: RefList.T;
                                      tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    res     := NEW (Op, args := args, tc := tc, eval := EvalEquals);
    op : Op;
  BEGIN
    args.head := Compile (args.head, formals);
    op := args.head;
    args.tail.head := Compile (args.tail.head, formals, op.tc);
    RETURN res
  END CompileEquals;
<P>
PROCEDURE EvalEquals (x: Op; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  VAR
    op1: Op := x.args.head;
    op2: Op := x.args.tail.head;
    a       := Eval (op1, d);
    b       := Eval (op2, d);
  BEGIN
    IF a = b THEN
      RETURN Sx.True
    ELSIF NOT Comparable (op1.tc, op2.tc) THEN
      RAISE FormsVBT.Error (<CODE>Invalid comparison</CODE>)
    ELSIF x.tc = IntegerTC THEN
      RETURN BooleanRefs [GetRefInteger (a)^ = GetRefInteger (b)^]
    ELSIF x.tc = RealTC THEN
      RETURN BooleanRefs [GetRefReal (a)^ = GetRefReal (b)^]
    ELSE
      (* If a and b are non-numeric refs, and we got here, then a # b. 
      <PRE>RETURN Sx.False
    END
  END EvalEquals;
*)

PROCEDURE <A NAME="CompilePlus"><procedure>CompilePlus</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                    args   : RefList.T;
                                    formals: RefList.T;
                       &lt;* UNUSED *&gt; tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    foundType     := FALSE;
    type          := RefanyTC;
    res           := NEW (Op, args := args, eval := EvalPlus);
    op       : Op;
  BEGIN
    IF args = NIL THEN RAISE FormsVBT.Error (&quot;(+) isn't defined.&quot;) END;
    REPEAT
      op := Compile (args.head, formals);
      args.head := op;
      args := args.tail;
      IF foundType THEN
        IF (op.tc = IntegerTC OR op.tc = RealTC) AND op.tc # type THEN
          RAISE FormsVBT.Error (&quot;Invalid argument to +&quot;)
        END
      ELSIF op.tc = IntegerTC OR op.tc = RealTC THEN
        foundType := TRUE;
        type := op.tc
      ELSIF op.tc # RefanyTC THEN
        RAISE FormsVBT.Error (&quot;Invalid argument to +&quot;)
      END
    UNTIL args = NIL;
    res.tc := type;
    RETURN res
  END CompilePlus;

PROCEDURE <A NAME="EvalPlus"><procedure>EvalPlus</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    ops     := x.args;
    op : Op;
  PROCEDURE AddIntegers (isum: INTEGER): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := Pop (ops);
        isum := isum + GetRefInteger (Eval (op, d))^
      END;
      RETURN Sx.FromInt (isum)
    END AddIntegers;
  PROCEDURE AddReals (rsum: REAL): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := Pop (ops);
        rsum := rsum + GetRefReal (Eval (op, d))^
      END;
      RETURN Sx.FromReal (rsum)
    END AddReals;
  BEGIN
    IF x.tc = IntegerTC THEN
      RETURN AddIntegers (0)
    ELSIF x.tc = RealTC THEN
      RETURN AddReals (0.0)
    ELSE
      op := Pop (ops);
      TYPECASE Eval (op, d) OF
      | NULL =&gt; RETURN Fault (&quot;number&quot;, NIL)
      | REF INTEGER (ri) =&gt; RETURN AddIntegers (ri^)
      | REF REAL (rr) =&gt; RETURN AddReals (rr^)
      | REFANY (ref) =&gt; RETURN Fault (&quot;number&quot;, ref)
      END
    END
  END EvalPlus;

PROCEDURE <A NAME="CompileMinus"><procedure>CompileMinus</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                     args   : RefList.T;
                                     formals: RefList.T;
                        &lt;* UNUSED *&gt; tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    foundType         := FALSE;
    type              := RefanyTC;
    res               := NEW (Op, args := args, eval := EvalMinus);
    op       : Op;
  BEGIN
    IF args = NIL THEN RAISE FormsVBT.Error (&quot;(-) isn't defined.&quot;) END;
    REPEAT
      op := Compile (args.head, formals);
      args.head := op;
      args := args.tail;
      IF foundType THEN
        IF (op.tc = IntegerTC OR op.tc = RealTC) AND op.tc # type THEN
          RAISE FormsVBT.Error (&quot;Invalid argument to -&quot;)
        END
      ELSIF op.tc = IntegerTC OR op.tc = RealTC THEN
        foundType := TRUE;
        type := op.tc
      ELSIF op.tc # RefanyTC THEN
        RAISE FormsVBT.Error (&quot;Invalid argument to -&quot;)
      END
    UNTIL args = NIL;
    res.tc := type;
    RETURN res
  END CompileMinus;

PROCEDURE <A NAME="EvalMinus"><procedure>EvalMinus</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    ops     := x.args;
    op : Op;
  PROCEDURE SubIntegers (isum: INTEGER): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := Pop (ops);
        isum := isum - GetRefInteger (op.eval (op, d))^
      END;
      RETURN Sx.FromInt (isum)
    END SubIntegers;
  PROCEDURE SubReals (rsum: REAL): REFANY RAISES {FormsVBT.Error} =
    BEGIN
      WHILE ops # NIL DO
        op := Pop (ops);
        rsum := rsum - GetRefReal (op.eval (op, d))^
      END;
      RETURN Sx.FromReal (rsum)
    END SubReals;
  BEGIN
    IF x.tc = IntegerTC THEN
      RETURN SubIntegers (0)
    ELSIF x.tc = RealTC THEN
      RETURN SubReals (0.0)
    ELSE
      op := Pop (ops);
      TYPECASE op.eval (op, d) OF
      | NULL =&gt; RETURN Fault (&quot;number&quot;, NIL)
      | REF INTEGER (ri) =&gt; RETURN SubIntegers (ri^)
      | REF REAL (rr) =&gt; RETURN SubReals (rr^)
      | REFANY (ref) =&gt; RETURN Fault (&quot;number&quot;, ref)
      END
    END
  END EvalMinus;

PROCEDURE <A NAME="CompileComparison"><procedure>CompileComparison</procedure></A> (             self   : Compiler;
                                          args   : RefList.T;
                                          formals: RefList.T;
                             &lt;* UNUSED *&gt; tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR
    c: ComparisonCompiler := self;
    res := NEW (ComparisonOp, args := args, eval := EvalComparison,
                test := c.test);
  BEGIN
    IF RefList.Length (args) &lt; 2 THEN
      RAISE FormsVBT.Error (&quot;Too few arguments&quot;)
    END;
    WHILE args # NIL DO
      args.head := Compile (args.head, formals);
      args := args.tail
    END;
    RETURN res
  END CompileComparison;

PROCEDURE <A NAME="EvalComparison"><procedure>EvalComparison</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    xc: ComparisonOp := x;
    a, args : RefList.T := NIL;
  PROCEDURE compareIntegers (base: INTEGER): REFANY RAISES {FormsVBT.Error} =
    VAR
      z: BOOLEAN;
      n: INTEGER;
    BEGIN
      WHILE args # NIL DO
        n := GetRefInteger (Pop (args))^;
        CASE xc.test OF
        | Test.EQ =&gt; z := base = n
        | Test.GE =&gt; z := base &gt;= n
        | Test.GT =&gt; z := base &gt; n
        | Test.LE =&gt; z := base &lt;= n
        | Test.LT =&gt; z := base &lt; n
        END;
        IF z THEN base := n ELSE RETURN Sx.False END
      END;
      RETURN Sx.True
    END compareIntegers;
  PROCEDURE compareReals (base: REAL): REFANY RAISES {FormsVBT.Error} =
    VAR
      z: BOOLEAN;
      n: REAL;
    BEGIN
      WHILE args # NIL DO
        n := GetRefReal (Pop (args))^;
        CASE xc.test OF
        | Test.EQ =&gt; z := base = n
        | Test.GE =&gt; z := base &gt;= n
        | Test.GT =&gt; z := base &gt; n
        | Test.LE =&gt; z := base &lt;= n
        | Test.LT =&gt; z := base &lt; n
        END;
        IF z THEN base := n ELSE RETURN Sx.False END
      END;
      RETURN Sx.True
    END compareReals;
  PROCEDURE compareRefsEQ (base: REFANY): REFANY =
    BEGIN
      WHILE args # NIL DO
        IF base # Pop (args) THEN RETURN Sx.False END
      END;
      RETURN Sx.True
    END compareRefsEQ;
  BEGIN
    a := x.args;
    (* Evaluate all the operands. *)
    WHILE a # NIL DO Push (args, Eval (Pop (a), d)) END;
    args := RefList.ReverseD (args);
    CASE xc.test OF
    | Test.EQ =&gt;
        TYPECASE Pop (args) OF
        | NULL =&gt; RETURN compareRefsEQ (NIL)
        | REF INTEGER (ri) =&gt; RETURN compareIntegers (ri^)
        | REF REAL (rr) =&gt; RETURN compareReals (rr^)
        | REFANY (ref) =&gt; RETURN compareRefsEQ (ref)
        END
    ELSE                         (* arithmetic comparison *)
      TYPECASE Pop (args) OF
      | NULL =&gt; RAISE FormsVBT.Error (&quot;Invalid comparison&quot;)
      | REF INTEGER (ri) =&gt; RETURN compareIntegers (ri^)
      | REF REAL (rr) =&gt; RETURN compareReals (rr^)
      ELSE RAISE FormsVBT.Error (&quot;Invalid comparison&quot;)
      END
    END
  END EvalComparison;

PROCEDURE <A NAME="CompileCat"><procedure>CompileCat</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                   args   : RefList.T;
                                   formals: RefList.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  VAR res := NEW (Op, args := args, tc := tc, eval := EvalCat);
  BEGIN
    WHILE args # NIL DO
      args.head := Compile (args.head, formals, tc);
      args := args.tail
    END;
    RETURN res
  END CompileCat;

PROCEDURE <A NAME="EvalCat"><procedure>EvalCat</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  VAR
    res     := &quot;&quot;;
    ops     := x.args;
  BEGIN
    WHILE ops # NIL DO res := res &amp; GetText (Eval (Pop (ops), d)) END;
    RETURN res
  END EvalCat;

PROCEDURE <A NAME="CompileFromName"><procedure>CompileFromName</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                        args   : RefList.T;
                                        formals: RefList.T;
                                        tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, TextTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalFromName)
  END CompileFromName;

PROCEDURE <A NAME="EvalFromName"><procedure>EvalFromName</procedure></A> (x: Op; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  BEGIN
    RETURN Atom.FromText (GetText (Eval (x.args.head, d)))
  END EvalFromName;

PROCEDURE <A NAME="CompileSymbolName"><procedure>CompileSymbolName</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                          args   : RefList.T;
                                          formals: RefList.T;
                                          tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals, SymbolTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalSymbolName)
  END CompileSymbolName;

PROCEDURE <A NAME="EvalSymbolName"><procedure>EvalSymbolName</procedure></A> (x: Op; d: Display): REFANY
  RAISES {FormsVBT.Error} =
  BEGIN
    RETURN Atom.ToText (GetSymbol (Eval (x.args.head, d)))
  END EvalSymbolName;

PROCEDURE <A NAME="CompileEmpty"><procedure>CompileEmpty</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                     args   : RefList.T;
                                     formals: RefList.T;
                                     tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    args.head := Compile (args.head, formals,TextTC);
    RETURN NEW (Op, args := args, tc := tc, eval := EvalEmpty)
  END CompileEmpty;

VAR BooleanRefs := ARRAY BOOLEAN OF Atom.T {Sx.False, Sx.True};

PROCEDURE <A NAME="EvalEmpty"><procedure>EvalEmpty</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN BooleanRefs [Text.Empty (GetText (Eval (x.args.head, d)))]
  END EvalEmpty;

PROCEDURE <A NAME="CompileSub"><procedure>CompileSub</procedure></A> (&lt;* UNUSED *&gt; self   : Compiler;
                                   args   : RefList.T;
                                   formals: RefList.T;
                                   tc     : INTEGER   ): Op
  RAISES {FormsVBT.Error} =
  BEGIN
    SetNth (args, 0, Compile (RefList.Nth (args, 0), formals, tc));
    SetNth (args, 1, Compile (RefList.Nth (args, 1), formals, IntegerTC));
    SetNth (args, 2, Compile (RefList.Nth (args, 2), formals, IntegerTC));
    RETURN NEW (Op, args := args, tc := tc, eval := EvalSub)
  END CompileSub;

PROCEDURE <A NAME="EvalSub"><procedure>EvalSub</procedure></A> (x: Op; d: Display): REFANY RAISES {FormsVBT.Error} =
  BEGIN
    RETURN Text.Sub (GetText (Eval (RefList.Nth (x.args, 0), d)),
                     GetRefCardinal (Eval (RefList.Nth (x.args, 1), d))^,
                     GetRefCardinal (Eval (RefList.Nth (x.args, 2), d))^)
  END EvalSub;

PROCEDURE <A NAME="EvalVar"><procedure>EvalVar</procedure></A> (x: Op; d: Display): REFANY =
  BEGIN
    RETURN d [x.tc]
  END EvalVar;
</PRE> ******** Safe retrieval functions ******* 

<P><PRE>PROCEDURE <A NAME="GetText"><procedure>GetText</procedure></A> (ref: REFANY): TEXT RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF | NULL =&gt; | TEXT (t) =&gt; RETURN t ELSE END;
    RETURN Fault (&quot;text&quot;, ref)
  END GetText;

PROCEDURE <A NAME="GetBoolean"><procedure>GetBoolean</procedure></A> (ref: REFANY): BOOLEAN RAISES {FormsVBT.Error} =
  BEGIN
    IF ref = Sx.True THEN
      RETURN TRUE
    ELSIF ref = Sx.False THEN
      RETURN FALSE
    ELSE
      EVAL Fault (&quot;boolean&quot;, ref);
      &lt;* ASSERT FALSE *&gt;
    END
  END GetBoolean;

PROCEDURE <A NAME="GetRefInteger"><procedure>GetRefInteger</procedure></A> (ref: REFANY): REF INTEGER RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF | NULL =&gt; | REF INTEGER (t) =&gt; RETURN t ELSE END;
    RETURN Fault (&quot;integer&quot;, ref)
  END GetRefInteger;

PROCEDURE <A NAME="GetRefCardinal"><procedure>GetRefCardinal</procedure></A> (ref: REFANY): REF INTEGER RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | NULL =&gt;
    | REF INTEGER (t) =&gt;         (* All Sx-integers are REF INTEGER *)
        IF t^ &gt;= 0 THEN RETURN t END
    ELSE
    END;
    RETURN Fault (&quot;integer&quot;, ref)
  END GetRefCardinal;

PROCEDURE <A NAME="GetRefReal"><procedure>GetRefReal</procedure></A> (ref: REFANY): REF REAL RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF | NULL =&gt; | REF REAL (t) =&gt; RETURN t ELSE END;
    RETURN Fault (&quot;real&quot;, ref)
  END GetRefReal;

PROCEDURE <A NAME="GetList"><procedure>GetList</procedure></A> (ref: REFANY): RefList.T RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF
    | RefList.T (t) =&gt; RETURN t     (* NIL is OK here *)
    ELSE
      RETURN Fault (&quot;list&quot;, ref)
    END
  END GetList;

PROCEDURE <A NAME="GetSymbol"><procedure>GetSymbol</procedure></A> (ref: REFANY): Atom.T RAISES {FormsVBT.Error} =
  BEGIN
    TYPECASE ref OF | NULL =&gt; | Atom.T (t) =&gt; RETURN t ELSE END;
    RETURN Fault (&quot;symbol&quot;, ref)
  END GetSymbol;

PROCEDURE <A NAME="Position"><procedure>Position</procedure></A> (list: RefList.T; item: REFANY): [-1 .. LAST (CARDINAL)] =
  VAR i: CARDINAL := 0;
  BEGIN
    LOOP
      IF list = NIL THEN RETURN -1
      ELSIF RefList.Nth (Pop (list), 0) = item THEN RETURN i
      ELSE INC (i)
      END
    END
  END Position;
</PRE>***************** Syntax for reading/writing %foo, =baz *****************

<P><PRE>PROCEDURE <A NAME="ReadEqual"><procedure>ReadEqual</procedure></A> (&lt;* UNUSED *&gt; rm    : Sx.ReadMacro;
                                  rd    : Rd.T;
                                  syntax: Sx.Syntax     ): RefList.T
  RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    TRY
      IF Rd.GetChar (rd) IN ASCII.Spaces THEN
        RETURN RefList.List1 (qEQ)
      ELSE
        Rd.UnGetChar (rd);
        RETURN RefList.List1 (
                 RefList.List2 (FVRuntime.qValue, Sx.Read (rd, syntax)))
      END
    EXCEPT
    | Rd.Failure =&gt; RAISE Sx.ReadError (&quot;Rd.Failure&quot;) (* FIXME *)
    | Rd.EndOfFile =&gt; RAISE Sx.ReadError (&quot;Premature EOF&quot;)
    END
  END ReadEqual;

PROCEDURE <A NAME="ReadPercent"><procedure>ReadPercent</procedure></A> (&lt;* UNUSED *&gt; rm    : Sx.ReadMacro;
                                    rd    : Rd.T;
                                    syntax: Sx.Syntax     ): RefList.T
  RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    TRY
      RETURN RefList.List1 (RefList.List2 (FVRuntime.qName, Sx.Read (rd, syntax)))
    EXCEPT
    | Rd.EndOfFile =&gt; RAISE Sx.ReadError (&quot;Premature EOF&quot;)
    END
  END ReadPercent;

PROCEDURE <A NAME="ReadQuote"><procedure>ReadQuote</procedure></A> (&lt;* UNUSED *&gt; rm    : Sx.ReadMacro;
                                  rd    : Rd.T;
                                  syntax: Sx.Syntax     ): RefList.T
  RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    TRY
      RETURN RefList.List1 (RefList.List2 (FVRuntime.qQuote, Sx.Read (rd, syntax)))
    EXCEPT
    | Rd.EndOfFile =&gt; RAISE Sx.ReadError (&quot;Premature EOF&quot;)
    END
  END ReadQuote;

PROCEDURE <A NAME="ReadBackquote"><procedure>ReadBackquote</procedure></A> (rm: ReadMacro; rd: Rd.T; syntax: Sx.Syntax):
  RefList.T RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    TRY
      INC (rm.bqLevel^);
      TRY
        RETURN RefList.List1 (
                 RefList.List2 (FVRuntime.qBackquote, Sx.Read (rd, syntax)))
      FINALLY
        DEC (rm.bqLevel^)
      END
    EXCEPT
    | Rd.EndOfFile =&gt; RAISE Sx.ReadError (&quot;Premature EOF&quot;)
    END
  END ReadBackquote;

PROCEDURE <A NAME="ReadComma"><procedure>ReadComma</procedure></A> (rm: ReadMacro; rd: Rd.T; syntax: Sx.Syntax): RefList.T
  RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    TRY
      IF rm.bqLevel^ = 0 THEN
        RAISE Sx.ReadError (&quot;comma not inside backquote&quot;)
      ELSE
        DEC (rm.bqLevel^);
        TRY
          IF Rd.GetChar (rd) = '@' THEN
            RETURN RefList.List1 (RefList.List2 (FVRuntime.qCommaAtsign,
                                                 Sx.Read (rd, syntax)))
          ELSE
            Rd.UnGetChar (rd);
            RETURN RefList.List1 (
                     RefList.List2 (FVRuntime.qComma, Sx.Read (rd, syntax)))
          END
        FINALLY
          INC (rm.bqLevel^)
        END
      END
    EXCEPT
    | Rd.Failure =&gt; RAISE Sx.ReadError (&quot;Rd.Failure&quot;) (* FIXME *)
    | Rd.EndOfFile =&gt; RAISE Sx.ReadError (&quot;Premature EOF&quot;)
    END
  END ReadComma;

PROCEDURE <A NAME="ReadSharp"><procedure>ReadSharp</procedure></A> (&lt;* UNUSED *&gt; rm    : Sx.ReadMacro;
                                  rd    : Rd.T;
                     &lt;* UNUSED *&gt; syntax: Sx.Syntax     ): RefList.T
  RAISES {Sx.ReadError, Thread.Alerted} =
  VAR
    level         := 0;
    c, prev: CHAR;
  BEGIN
    TRY
      c := Rd.GetChar (rd);
      IF c # '|' THEN
        RAISE Sx.ReadError (&quot;Illegal character after #: &quot; &amp; Fmt.Char (c))
      END;
      LOOP
        prev := c;
        c := Rd.GetChar (rd);
        IF c = '#' AND prev = '|' THEN
          IF level = 0 THEN RETURN NIL ELSE DEC (level) END
        ELSIF c = '|' AND prev = '#' THEN
          INC (level)
        END
      END
    EXCEPT
    | Rd.Failure =&gt; RAISE Sx.ReadError (&quot;Rd.Failure&quot;) (* FIXME *)
    | Rd.EndOfFile =&gt; RAISE Sx.ReadError (&quot;Premature EOF&quot;)
    END
  END ReadSharp;

VAR
  NoDefault := NEW (REF CARDINAL); (* Any unique ref will do. *)

PROCEDURE <A NAME="Init"><procedure>Init</procedure></A> () =
  VAR b := NEW (REF CARDINAL);
  BEGIN
    (* Use a special syntax table to handle %name, =value, etc. *)
    FVSyntax := Sx.CopySyntax ();
    b^ := 0;

    Sx.SetReadMacro (
      FVSyntax, '=', NEW (ReadMacro, read := ReadEqual, bqLevel := b));
    Sx.SetReadMacro (
      FVSyntax, '%', NEW (ReadMacro, read := ReadPercent, bqLevel := b));
    Sx.SetReadMacro (
      FVSyntax, '\'', NEW (ReadMacro, read := ReadQuote, bqLevel := b));
    Sx.SetReadMacro (
      FVSyntax, '`', NEW (ReadMacro, read := ReadBackquote, bqLevel := b));
    Sx.SetReadMacro (
      FVSyntax, ',', NEW (ReadMacro, read := ReadComma, bqLevel := b));
    Sx.SetReadMacro (
      FVSyntax, '#', NEW (ReadMacro, read := ReadSharp, bqLevel := b));

    InitCompilers ();
    (* InitTypeCodes (); *)
    FOR i := FIRST (VarOps) TO LAST (VarOps) DO
      VarOps [i] := NEW (Op, tc := i, eval := EvalVar)
    END
  END Init;

BEGIN
END Macro.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
