<HTML>
<HEAD>
<TITLE>SRC Modula-3: lego/src/ListVBT.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>lego/src/ListVBT.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                                           </EM></BLOCKQUOTE><PRE>
</PRE> ListVBT.m3 

<P><PRE>MODULE <module>ListVBT</module> EXPORTS <A HREF="ListVBT.i3"><implements>ListVBT</A></implements>;

IMPORT <A HREF="../../geometry/src/Axis.i3">Axis</A>, <A HREF="../../ui/src/vbt/Font.i3">Font</A>, <A HREF="../../ui/src/split/HVSplit.i3">HVSplit</A>, <A HREF="../../ui/src/vbt/PaintOp.i3">PaintOp</A>, <A HREF="../../ui/src/vbt/Pixmap.i3">Pixmap</A>, <A HREF="../../geometry/src/Point.i3">Point</A>, <A HREF="../../geometry/src/Rect.i3">Rect</A>, <A HREF="../../geometry/src/Region.i3">Region</A>,
       <A HREF="ScrollerVBTClass.i3">ScrollerVBTClass</A>, <A HREF="../../ui/src/split/Split.i3">Split</A>, <A HREF="../../ui/src/split/TextureVBT.i3">TextureVBT</A>, <A HREF="../../ui/src/vbt/VBT.i3">VBT</A>;
</PRE> 
 Types and such 
 

<P><PRE>TYPE

  CellContents = RECORD
      value: REFANY;
      selected: BOOLEAN;
    END;

  Scroller = ScrollerVBTClass.T OBJECT
      list: T;
    OVERRIDES
      scroll := Scroll;
      autoScroll := AutoScroll;
      thumb := Thumb;
    END;

  Bar = TextureVBT.T OBJECT
      size: REAL := 0.25
    OVERRIDES
      shape := BarShape;
    END;

  Contents = VBT.Leaf OBJECT
      (* The VBT wherein the cells get painted. All painting and mouse
         interpretation is done by this class, including all calls of painter
         and selector methods. *)
      list: T;
      haveScreen: BOOLEAN := FALSE;
      height: INTEGER := 1; (* cached result of painter.height *)
      hasFocus: BOOLEAN := FALSE; (* while we have mouse focus *)
    METHODS
      cellForCP(cp: VBT.CursorPosition;
                VAR cage: VBT.Cage): INTEGER := CellForCP;
        (* LL.sup &lt; list *)
        (* Returns cell number for given cursor position, or -1; if
           cp is above or below &quot;contents&quot;, auto-scrolls; in any
           case, assigns to &quot;cage&quot; the minimal cage that would cause
           cellForCP to do something different. *)
      scrollContents(this: INTEGER) := ScrollContents;
        (* LL.sup = list *)
        (* Sets list.firstVisible to boundFirstVisible(this), and does
            scroll/repaints. *)
      boundFirstVisible() := BoundFirstVisible;
        (* LL.sup = list *)
        (* Sets firstVisible to a value that satisfies the firstVisible
           invariant and is as close as possible to the current firstVisible.
           Also calls updateScroller. *)
      moveCells(at: Cell; delta: INTEGER) := MoveCells;
        (* LL.sup = list *)
        (* Fixup the firstVisible invariant, and update the screen for cells
           [ MIN(at, at+delta), ... ).  The pixels at cell positions [at, ...)
           are suitable for cells [at+delta, ...).  Delta might be negative.
           Also calls updateScroller. *)
      paintCells(at:Cell; n: INTEGER; bad: Rect.T) := PaintCells;
        (* LL.sup = list *)
        (* Repaint the cells; if at+n &gt;= nCells, erase the blank space *)
      selectCell(this: Cell) := SelectCell;
        (* LL.sup = list *)
        (* Uses painter.select to adjust the cell's appearance on-screen. *)
    OVERRIDES
      repaint := Repaint;
      reshape := Reshape;
      rescreen := Rescreen;
      redisplay := Redisplay;
      mouse := Mouse;
      position := Position;
    END;

REVEAL

  <A NAME="Private">Private</A> = HVSplit.T BRANDED &quot;ListVBT.Private 1.0&quot; OBJECT
    END;

  <A NAME="T">T</A> = Public BRANDED &quot;ListVBT.T 1.0&quot; OBJECT
      mu: MUTEX;                    (* protects all the fields *)
      vScroll: Scroller;            (* sub-window containg the scroll bar *)
      contents: Contents;           (* sub-window containing the cells *)
      cells: REF ARRAY OF CellContents;
      nCells: CARDINAL := 0;        (* total number of cells *)
      nSelections: CARDINAL := 0;   (* total number of selected cells *)
      firstSelected: Cell := 0;     (* first selected cell, for efficiency;
                                       valid iff nSelections &gt; 0 *)
      firstVisible: Cell := 0;      (* first visible cell *)
        (* Invariant: firstVisible+nVisible&gt;nCells iff nCells&lt;nVisible *)
      nVisible: CARDINAL := 0;      (* screen space for visible cells *)
    METHODS
      getNextSelected(VAR this: Cell) := GetNextSelected;
        (* LL.sup = list;
           PRE: list.nSelections &gt; 0 and there exists a selected cell &gt;= this;
           sets &quot;this&quot; to first selected cell &gt;= this *)
      updateScroller() := UpdateScroller;
        (* LL.sup = list; informs the scroller of current position and size *)
    OVERRIDES
      init := Init;
      setValue := SetValue;
      getValue := GetValue;
      count := Count;
      insertCells := InsertCells;
      removeCells := RemoveCells;
      selectNone := SelectNone;
      selectOnly := SelectOnly;
      select := Select;
      isSelected := IsSelected;
      getAllSelected := GetAllSelected;
      getFirstSelected := GetFirstSelected;
      scrollTo := ScrollTo;
      scrollToShow := ScrollToShow;
      redisplay := TRedisplay;
      reportVisible := ReportVisible;
    END;

  <A NAME="TextPainter">TextPainter</A> = TextPainterPublic BRANDED &quot;ListVBT.TextPainter 1.0&quot; OBJECT
      mu: MUTEX;
      eraseColor, textColor, hiliteColor, hiliteTextColor: PaintOp.T;
      font: Font.T;
      ascent, descent: INTEGER;
    OVERRIDES
      init := TextPainterInit;
      height := TextPainterHeight;
      paint := TextPainterPaint;
      select := TextPainterSelect;
      erase := TextPainterErase;
      setFont := TextPainterSetFont;
    END;

  <A NAME="UniSelector">UniSelector</A> = Selector BRANDED &quot;ListVBT.UniSelector 1.0&quot; OBJECT
      list: T;
    OVERRIDES
      init := UniSelectorInit;
      insideClick := UniSelectorInsideClick;
      outsideClick := UniSelectorOutsideClick;
      insideDrag := UniSelectorInsideDrag;
      outsideDrag := UniSelectorOutsideDrag;
    END;

  <A NAME="MultiSelector">MultiSelector</A> = Selector BRANDED &quot;ListVBT.MultiSelector 1.0&quot; OBJECT
      list: T;
      anchor: Cell := 0;
      prev: Cell := 0;
      adding: BOOLEAN := FALSE;
    OVERRIDES
      init := MultiSelectorInit;
      insideClick := MultiSelectorInsideClick;
      outsideClick := MultiSelectorOutsideClick;
      insideDrag := MultiSelectorInsideDrag;
      outsideDrag := MultiSelectorOutsideDrag;
    END;
</PRE> 
 Implementations of ListVBT methods 
 

<P><PRE>PROCEDURE <A NAME="Init"><procedure>Init</procedure></A> (list: T; colors: PaintOp.ColorQuad): T =
  BEGIN
    EVAL HVSplit.T.init (list, Axis.T.Hor);
    list.mu := NEW (MUTEX);
    LOCK list.mu DO
      list.cells := NEW (REF ARRAY OF CellContents, 100);
      list.vScroll := NEW (Scroller, list := list).init (Axis.T.Ver, colors);
      Split.AddChild (list, list.vScroll);
      Split.AddChild (list, NEW (Bar).init (colors.fg, Pixmap.Solid));
      list.contents := NEW (Contents, list := list);
      Split.AddChild (list, list.contents);
      IF list.painter = NIL THEN
        list.painter :=
          NEW (TextPainter).init (colors.bg, colors.fg, colors.fg, colors.bg)
      END;
      IF list.selector = NIL THEN
        list.selector := NEW (UniSelector).init (list)
      END;
    END;
    RETURN list
  END Init;

PROCEDURE <A NAME="SetValue"><procedure>SetValue</procedure></A>(list: T; this: Cell; value: REFANY) =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      IF (this &gt;= 0) AND (this &lt; list.nCells) THEN
        list.cells[this].value := value;
        list.contents.paintCells(this, 1, Rect.Full);
      END;
    END;
  END SetValue;

PROCEDURE <A NAME="GetValue"><procedure>GetValue</procedure></A>(list: T; this: Cell): REFANY =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      IF this &lt; 0 THEN
        RETURN NIL
      ELSIF this &gt;= list.nCells THEN
        RETURN NIL
      ELSE
        RETURN list.cells[this].value
      END;
    END;
  END GetValue;

PROCEDURE <A NAME="Count"><procedure>Count</procedure></A>(list: T): CARDINAL =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      RETURN list.nCells
    END;
  END Count;

PROCEDURE <A NAME="GetNextSelected"><procedure>GetNextSelected</procedure></A>(list: T; VAR this: Cell) =
  (* LL.sup = list;
     PRE: list.nSelections &gt; 0 and there exists a selected cell &gt;= this *)
  BEGIN
    LOOP
      &lt;* ASSERT(this &lt; list.nCells) *&gt;
      IF list.cells^[this].selected THEN EXIT END;
      INC(this);
    END;
  END GetNextSelected;

PROCEDURE <A NAME="InsertCells"><procedure>InsertCells</procedure></A>(list: T; at: Cell; n: CARDINAL) =
  (* LL.sup &lt; list *)
  VAR first: Cell; oldCells: REF ARRAY OF CellContents;
  BEGIN
    LOCK list.mu DO
      first := MAX(0, MIN(at, list.nCells));
      IF list.firstSelected &gt;= first THEN INC(list.firstSelected, n) END;
      IF n + list.nCells &gt; NUMBER(list.cells^) THEN
        oldCells := list.cells;
        list.cells := NEW(REF ARRAY OF CellContents,
            MAX(n + list.nCells,
                NUMBER(oldCells^) + NUMBER(oldCells^) DIV 2));
        SUBARRAY(list.cells^, 0, NUMBER(oldCells^)) := oldCells^;
      END;
      SUBARRAY(list.cells^, first+n, list.nCells-first) :=
          SUBARRAY(list.cells^, first, list.nCells-first);
      FOR i := first TO first + n - 1 DO
        list.cells^[i] := CellContents{ value := NIL, selected := FALSE };
      END;
      INC(list.nCells, n);
      list.contents.moveCells(first, n);
    END;
  END InsertCells;

PROCEDURE <A NAME="RemoveCells"><procedure>RemoveCells</procedure></A>(list: T; at: Cell; n: CARDINAL) =
  (* LL.sup &lt; list *)
  VAR first, this: Cell; amount: CARDINAL;
  BEGIN
    LOCK list.mu DO
      first := MAX(0, MIN(at, list.nCells));
      amount := MIN(at+n, list.nCells) - first;
      IF amount &gt; 0 THEN
        this := first;
        WHILE (list.nSelections &gt; 0) AND (this &lt; first + amount) DO
          IF list.cells^[this].selected THEN
            list.cells^[this].selected := FALSE;
            DEC(list.nSelections);
          END;
          INC(this);
        END;
        (* Now list.firstSelected might be wrong, either because we just
           deselected it, or because it's beyond first+amount and must be
           relocated. *)
        IF list.nSelections &gt; 0 THEN
          IF list.firstSelected &gt;= first THEN
            this := list.firstSelected;
            list.getNextSelected(this);
            list.firstSelected := this - amount;
          END;
        END;
        SUBARRAY(list.cells^, first, list.nCells-(first+amount)) :=
            SUBARRAY(list.cells^, first+amount, list.nCells-(first+amount));
        DEC(list.nCells, amount);
        list.contents.moveCells(first+amount, -amount);
      END;
    END;
  END RemoveCells;

PROCEDURE <A NAME="SelectNone"><procedure>SelectNone</procedure></A>(list: T) =
  (* LL.sup &lt; list *)
  VAR this: INTEGER;
  BEGIN
    LOCK list.mu DO
      this := list.firstSelected;
      WHILE list.nSelections &gt; 0 DO
        list.getNextSelected(this);
        list.cells^[this].selected := FALSE;
        DEC(list.nSelections);
        list.contents.selectCell(this);
      END;
    END;
  END SelectNone;

PROCEDURE <A NAME="SelectOnly"><procedure>SelectOnly</procedure></A>(list: T; this: Cell) =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO (* optimise the no-op case, to reduce flicker *)
      IF (this &gt;= 0) AND (this &lt; list.nCells) AND
         (list.nSelections = 1) AND list.cells^[this].selected THEN
        RETURN
      END;
    END;
    list.selectNone();
    list.select(this, TRUE);
  END SelectOnly;

PROCEDURE <A NAME="Select"><procedure>Select</procedure></A>(list: T; this: Cell; selected: BOOLEAN) =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      IF (this &gt;= 0) AND (this &lt; list.nCells) THEN
        IF list.cells^[this].selected # selected THEN
          list.cells^[this].selected := selected;
          IF selected THEN
            INC(list.nSelections);
            IF (list.nSelections = 1) OR (this &lt; list.firstSelected) THEN
              list.firstSelected := this;
            END;
          ELSE
            DEC(list.nSelections);
            IF (list.nSelections &gt; 0) AND (this = list.firstSelected) THEN
              list.getNextSelected(list.firstSelected);
            END;
          END;
          list.contents.selectCell(this);
        END;
      END;
    END;
  END Select;

PROCEDURE <A NAME="IsSelected"><procedure>IsSelected</procedure></A>(list: T; this: Cell): BOOLEAN =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      IF this &lt; 0 THEN
        RETURN FALSE
      ELSIF this &gt;= list.nCells THEN
        RETURN FALSE
      ELSE
        RETURN list.cells^[this].selected
      END;
    END;
  END IsSelected;

PROCEDURE <A NAME="GetAllSelected"><procedure>GetAllSelected</procedure></A>(list: T): REF ARRAY OF Cell =
  (* LL.sup &lt; list *)
  VAR sel: REF ARRAY OF Cell; this: Cell;
  BEGIN
    LOCK list.mu DO
      sel := NEW(REF ARRAY OF Cell, list.nSelections);
      this := list.firstSelected;
      FOR i := 0 TO NUMBER(sel^)-1 DO
        list.getNextSelected(this);
        sel^[i] := this;
        INC(this);
      END;
      RETURN sel
    END;
  END GetAllSelected;

PROCEDURE <A NAME="GetFirstSelected"><procedure>GetFirstSelected</procedure></A>(list: T; VAR this: Cell): BOOLEAN =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      IF list.nSelections &gt; 0 THEN
        this := list.firstSelected;
        &lt;* ASSERT(list.cells^[this].selected) *&gt;
        RETURN TRUE
      ELSE
        RETURN FALSE
      END;
    END;
  END GetFirstSelected;

PROCEDURE <A NAME="ScrollTo"><procedure>ScrollTo</procedure></A>(list: T; this: Cell) =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      list.contents.scrollContents(this);
    END;
  END ScrollTo;

PROCEDURE <A NAME="ScrollToShow"><procedure>ScrollToShow</procedure></A>(list: T; this: Cell) =
  (* LL.sup &lt; list *)
  BEGIN
    LOCK list.mu DO
      IF (this &lt; list.firstVisible) OR
          (this &gt;= list.firstVisible + list.nVisible) THEN
        list.contents.scrollContents(this - list.nVisible DIV 2);
      END;
    END;
  END ScrollToShow;

PROCEDURE <A NAME="TRedisplay"><procedure>TRedisplay</procedure></A>(list: T) =
  (* LL.sup = VBT.mu *)
  BEGIN
    HVSplit.T.redisplay(list);
    list.contents.redisplay();
  END TRedisplay;

PROCEDURE <A NAME="ReportVisible"><procedure>ReportVisible</procedure></A>(
   &lt;*UNUSED*&gt;list: T;
   &lt;*UNUSED*&gt;first: Cell;
   &lt;*UNUSED*&gt;num: CARDINAL) =
  BEGIN
  END ReportVisible;

PROCEDURE <A NAME="UpdateScroller"><procedure>UpdateScroller</procedure></A>(list: T) =
  (* LL.sup = list *)
  BEGIN
    IF list.vScroll # NIL THEN
      ScrollerVBTClass.Update(list.vScroll, list.firstVisible,
                         MIN(list.firstVisible+list.nVisible, list.nCells),
                         list.nCells);
    END;
    list.reportVisible(list.firstVisible,
      MIN(list.nVisible, list.nCells - list.firstVisible))
  END UpdateScroller;
</PRE> 
 Implementations of Contents methods 
 

<P><PRE>PROCEDURE <A NAME="Mouse"><procedure>Mouse</procedure></A>(contents: Contents; READONLY cd: VBT.MouseRec) =
  (* LL.sup = VBT.mu *)
  VAR
    this: INTEGER; cage: VBT.Cage;
  BEGIN
    IF (cd.clickType = VBT.ClickType.FirstDown) OR contents.hasFocus THEN
      this := contents.cellForCP(cd.cp, cage);
      IF cd.clickType = VBT.ClickType.LastUp THEN
        VBT.SetCage(contents, VBT.EverywhereCage);
        contents.hasFocus := FALSE;
      ELSE
        VBT.SetCage(contents, cage);
        contents.hasFocus := TRUE;
      END;
      IF this &gt;= 0 THEN
        contents.list.selector.insideClick(cd, this);
      ELSE
        contents.list.selector.outsideClick(cd);
      END;
    END;
  END Mouse;

PROCEDURE <A NAME="Position"><procedure>Position</procedure></A>(contents: Contents; READONLY cd: VBT.PositionRec) =
  (* LL.sup = VBT.mu *)
  VAR
    this: INTEGER; cage: VBT.Cage;
  BEGIN
    IF contents.hasFocus THEN
      this := contents.cellForCP(cd.cp, cage);
      VBT.SetCage(contents, cage);
      IF this &gt;= 0 THEN
        contents.list.selector.insideDrag(cd, this);
      ELSE
        contents.list.selector.outsideDrag(cd);
      END;
    END;
  END Position;

PROCEDURE <A NAME="Redisplay"><procedure>Redisplay</procedure></A>(contents: Contents) =
  (* LL.sup = mu *)
  BEGIN
    WITH list = contents.list DO
      LOCK list.mu DO
        IF contents.haveScreen THEN
          contents.height := list.painter.height(contents)
        ELSE
          contents.height := 1;
        END;
      END;
    END;
    VBT.Leaf.redisplay(contents);
  END Redisplay;

PROCEDURE <A NAME="Reshape"><procedure>Reshape</procedure></A>(contents: Contents; READONLY cd: VBT.ReshapeRec) =
  (* LL.sup = mu.contents *)
  VAR
    wasNormalized: BOOLEAN;
    needsRepaint: Rect.T;
    delta: Point.T;
    oldFirstVisible: Cell;
    firstVisibleDelta: INTEGER;
  BEGIN
    WITH list = contents.list DO
      LOCK list.mu DO
        IF cd.new = Rect.Empty THEN
          list.nVisible := 0;
          firstVisibleDelta := 0;
          list.reportVisible(list.firstVisible, 0)
        ELSE
          wasNormalized := (list.nSelections&gt;0) AND
              (list.firstSelected &gt;= list.firstVisible) AND
              (list.firstSelected &lt; list.firstVisible+list.nVisible);
          list.nVisible := Rect.VerSize(cd.new) DIV contents.height;
          IF wasNormalized AND
              (list.firstSelected &gt;= list.firstVisible+list.nVisible) THEN
            list.firstVisible := list.firstSelected - list.nVisible DIV 2;
          END;
          (* in any case, fix up the firstVisible invariant *)
          oldFirstVisible := list.firstVisible;
          contents.boundFirstVisible();
          firstVisibleDelta :=
              (oldFirstVisible - list.firstVisible) * contents.height;
        END;
      END;
    END;
    (* Salvage old pixels; but RWT's whiteboard says salvage never succeeds. *)
    IF (cd.saved.west &lt;= cd.prev.west) AND
        (cd.saved.east &gt;= cd.prev.east) AND
        (Rect.HorSize(cd.prev) &gt;= Rect.HorSize(cd.new)) THEN
      (* If we don't have full width, we'll repaint the cells anyway *)
      delta := Point.Sub(Rect.NorthWest(cd.new), Rect.NorthWest(cd.prev));
      INC(delta.v, firstVisibleDelta);
      IF delta # Point.Origin THEN
        VBT.Scroll(contents, cd.new, delta);
      END;
      needsRepaint := cd.new;
      needsRepaint.south := needsRepaint.north +
          (cd.saved.north - cd.prev.north) + firstVisibleDelta;
      IF needsRepaint.south &gt; needsRepaint.north THEN
        contents.repaint(Region.FromRect(needsRepaint));
      END;
      needsRepaint := cd.new;
      INC(needsRepaint.north,
          cd.saved.south - cd.prev.north + firstVisibleDelta);
      IF needsRepaint.south &gt; needsRepaint.north THEN
        contents.repaint(Region.FromRect(needsRepaint));
      END;
    ELSE
      contents.repaint(Region.FromRect(cd.new));
    END;
  END Reshape;

PROCEDURE <A NAME="Rescreen"><procedure>Rescreen</procedure></A>(contents: Contents; READONLY cd: VBT.RescreenRec) =
  (* LL.sup = mu.contents *)
  BEGIN
    WITH list = contents.list DO
      LOCK list.mu DO
        IF cd.st = NIL THEN
          contents.haveScreen := FALSE;
          contents.height := 1;
          list.nVisible := 0;
          list.reportVisible(list.firstVisible, 0)
        ELSE
          contents.haveScreen := TRUE;
          contents.height := list.painter.height(contents);
        END;
      END;
    END;
  END Rescreen;

PROCEDURE <A NAME="Repaint"><procedure>Repaint</procedure></A>(contents: Contents; READONLY rgn: Region.T) =
  (* LL.sup = mu.contents *)
  VAR
    domain := VBT.Domain(contents);
    firstHit, lastHit: Cell;
  BEGIN
    WITH list = contents.list DO
      LOCK contents.list.mu DO
        IF rgn.r.north &lt; domain.north THEN
          firstHit := list.firstVisible
        ELSE
          firstHit := (rgn.r.north-domain.north) DIV contents.height +
                       list.firstVisible;
        END;
        IF rgn.r.south &gt; domain.south THEN
          lastHit := list.firstVisible + list.nVisible
        ELSE
          lastHit := (rgn.r.south-domain.north-1) DIV contents.height +
                     list.firstVisible;
        END;
        contents.paintCells(firstHit, lastHit-firstHit+1, rgn.r);
      END;
    END;
  END Repaint;

PROCEDURE <A NAME="CellForCP"><procedure>CellForCP</procedure></A>(contents: Contents; cp: VBT.CursorPosition;
                    VAR cage: VBT.Cage): INTEGER =
  (* LL.sup &lt; list *)
  VAR
    domain := VBT.Domain(contents);
    cellInDomain: INTEGER; (* cell number relative to list.firstVisible *)
    r := domain;
  BEGIN
    WITH list = contents.list DO
      LOCK list.mu DO
        IF cp.gone THEN
          cage := VBT.EmptyCage;
          IF NOT cp.offScreen THEN
            IF cp.pt.v &lt; domain.north THEN
              contents.scrollContents(list.firstVisible-1);
              IF list.nCells &gt; 0 THEN
                RETURN list.firstVisible
              END;
            ELSIF cp.pt.v &gt;= domain.south THEN
              contents.scrollContents(list.firstVisible+1);
              IF list.nCells &gt; 0 THEN
                RETURN MIN(list.nCells, list.firstVisible + list.nVisible)-1
              END;
            END;
          END;
          RETURN -1
        ELSE
          &lt;* ASSERT(cp.pt.v &gt;= domain.north) *&gt;
          cellInDomain := (cp.pt.v - domain.north) DIV contents.height;
          r.north := domain.north + cellInDomain * contents.height;
          r.south := MIN(r.north + contents.height, domain.south);
          cage := VBT.Cage{ r, VBT.InOut{FALSE}, VBT.AllScreens };
          IF list.firstVisible + cellInDomain &gt;= list.nCells THEN
            RETURN -1
          ELSE
            RETURN list.firstVisible + cellInDomain;
          END;
        END;
      END;
    END;
  END CellForCP;

PROCEDURE <A NAME="ScrollContents"><procedure>ScrollContents</procedure></A>(contents: Contents; this: INTEGER) =
  (* LL.sup = list *)
  VAR delta: INTEGER;
  BEGIN
    WITH list = contents.list DO
      delta := list.firstVisible - this;
      list.firstVisible := this;
      contents.moveCells(this, delta);
    END;
  END ScrollContents;

PROCEDURE <A NAME="BoundFirstVisible"><procedure>BoundFirstVisible</procedure></A>(contents: Contents) =
  (* LL.sup = list *)
  BEGIN
    WITH list = contents.list DO
      list.firstVisible :=
          MAX(0, MIN(list.firstVisible, list.nCells-list.nVisible) );
      list.updateScroller();
    END;
  END BoundFirstVisible;

PROCEDURE <A NAME="MoveCells"><procedure>MoveCells</procedure></A>(contents: Contents; at: Cell; delta: INTEGER) =
  (* LL.sup = list *)
        (* Fixup the firstVisible invariant, and update the screen for cells
           [ MIN(at, at+delta), ... ).  The pixels at cell positions [at, ...)
           are suitable for cells [at+delta, ...).  Delta might be negative.
           Also calls updateScroller. *)
  VAR
    oldFirst, adjustment, boundedFirst, boundedDelta: INTEGER;
    boundedAt: Cell;
    domain := VBT.Domain(contents);
    clip: Rect.T;
  BEGIN
    WITH list = contents.list DO
      oldFirst := list.firstVisible;
      contents.boundFirstVisible();
      boundedFirst := list.firstVisible;
      adjustment := oldFirst - boundedFirst;
      boundedDelta := delta + adjustment;
      boundedAt := at - adjustment;
      (* NOTE: at+delta = boundedAt+boundedDelta *)
      IF (adjustment # 0) AND (MIN(boundedAt, at+delta) &gt; boundedFirst) THEN
        (* extra repaint caused by bounding firstVisible *)
        (* repaint [list.firstVisible .. MIN(boundedAt, at+delta) ) *)
        clip := domain;
        clip.south := clip.north +
                (MIN(boundedAt, at+delta)-boundedFirst) * contents.height;
        VBT.Scroll(contents,
                   clip,
                   Point.T{h :=  0, v := adjustment * contents.height}
                   );
        IF adjustment &gt; 0 THEN
          (* repaint newly exposed cells at top *)
          contents.paintCells(boundedFirst, adjustment, Rect.Full);
        END;
      END;
      IF boundedDelta # 0 THEN
        (* repaint [MIN(boundedAt, at+delta) .. ) *)
        clip := domain;
        INC(clip.north,
            (boundedAt+boundedDelta-boundedFirst) * contents.height);
        clip.south := domain.north + list.nVisible * contents.height;
        IF clip.north &lt; clip.south THEN
          (* scroll into [at+delta .. ) *)
          VBT.Scroll(contents,
                     clip,
                     Point.T{h := 0, v := boundedDelta * contents.height}
                     );
        END;
        IF boundedDelta &gt; 0 THEN
          (* repaint [boundedAt .. at+delta) *)
          contents.paintCells(boundedAt, boundedDelta, Rect.Full);
        END;
        IF boundedDelta &lt; 0 THEN
          (* repaint newly exposed cells at bottom *)
          contents.paintCells(
            boundedFirst+list.nVisible+boundedDelta, -boundedDelta, Rect.Full);
        END;
      END;
    END;
  END MoveCells;

PROCEDURE <A NAME="PaintCells"><procedure>PaintCells</procedure></A>(contents: Contents; at: Cell; n: INTEGER; bad: Rect.T) =
  (* LL.sup = list *)
  VAR
    domain := VBT.Domain(contents);
    r := domain;
    start, limit: Cell;
  BEGIN
    WITH list = contents.list DO
      start := MAX(at, list.firstVisible);
      limit := MIN(MIN(at+n, list.firstVisible+list.nVisible), list.nCells);
      FOR this := start TO limit-1 DO
        r.north := domain.north + (this-list.firstVisible) * contents.height;
        r.south := r.north + contents.height;
        list.painter.paint(contents, r, list.cells^[this].value, this,
                                        list.cells^[this].selected,
                                        Rect.Meet (r, bad));
      END;
      IF limit &lt; at+n THEN
        (* erase the rest of the cell positions *)
        r.north := domain.north + (limit-list.firstVisible) * contents.height;
        r.south := domain.north + (at+n-list.firstVisible) * contents.height;
        list.painter.erase(contents, r);
      END;
    END;
  END PaintCells;

PROCEDURE <A NAME="SelectCell"><procedure>SelectCell</procedure></A>(contents: Contents; this: Cell) =
  (* LL.sup = list *)
  VAR r, domain: Rect.T;
  BEGIN
    WITH list = contents.list DO
      domain := VBT.Domain(contents);
      IF domain # Rect.Empty THEN
        IF (this &gt;= list.firstVisible) AND
                                (this &lt; list.firstVisible + list.nVisible) THEN
          r := domain;
          INC(r.north, (this-list.firstVisible) * contents.height);
          r.south := r.north + contents.height;
          list.painter.select(contents, r, list.cells^[this].value, this,
                                           list.cells^[this].selected);
        END;
      END;
    END;
  END SelectCell;
</PRE> 
 Implementations of Scroller methods 
 

<P><PRE>PROCEDURE <A NAME="Scroll"><procedure>Scroll</procedure></A>(scroller: Scroller;
                 &lt;*UNUSED*&gt; READONLY cd: VBT.MouseRec;
                 part: INTEGER;
                 height: INTEGER;
                 towardsEOF: BOOLEAN) =
  (* LL.sup &lt; list *)
  VAR distance: INTEGER;
  BEGIN
    WITH list = scroller.list DO
      LOCK list.mu DO
        distance := MAX(1, (part * list.nVisible) DIV height);
        IF NOT towardsEOF THEN distance := -distance END;
        list.contents.scrollContents(list.firstVisible+distance);
      END;
    END;
  END Scroll;

PROCEDURE <A NAME="AutoScroll"><procedure>AutoScroll</procedure></A> (scroller: Scroller;
                      &lt;*UNUSED*&gt; READONLY cd: VBT.MouseRec;
                      linesToScroll: CARDINAL;
                      towardsEOF: BOOLEAN) =
  (* LL.sup &lt; list *)
  VAR distance: INTEGER;
  BEGIN
    WITH list = scroller.list DO
      LOCK list.mu DO
        distance := linesToScroll;
        IF NOT towardsEOF THEN distance := -distance END;
        list.contents.scrollContents(list.firstVisible+distance);
      END;
    END;
  END AutoScroll;

CONST NearEdge = 13;
    (* Thumbing closer than this to top/bottom of scroll bar is treated as
       being exactly at the top/bottom. *)

PROCEDURE <A NAME="Thumb"><procedure>Thumb</procedure></A> (scroller: Scroller;
                 &lt;*UNUSED*&gt; READONLY cd: VBT.MouseRec;
                 part: INTEGER;
                 height: INTEGER) =
  (* LL.sup &lt; list *)
  VAR position: INTEGER;
  BEGIN
    WITH list = scroller.list DO
      LOCK list.mu DO
        IF part &lt; NearEdge THEN
          position := 0
        ELSIF part + NearEdge &gt; height THEN
          position := list.nCells
        ELSE
          position := (part * list.nCells) DIV height
        END;
        list.contents.scrollContents(position);
      END;
    END;
  END Thumb;
</PRE> 
 Implementation of Bar method 
 

<P><PRE>PROCEDURE <A NAME="BarShape"><procedure>BarShape</procedure></A>(bar: Bar; ax: Axis.T; n: CARDINAL): VBT.SizeRange =
  (* LL.sup = VBT.mu.bar *)
  VAR
    sr: VBT.SizeRange;
  BEGIN
    WITH hv = HVSplit.AxisOf(VBT.Parent(bar)) DO
      IF hv = ax THEN
        sr.lo := ROUND(VBT.MMToPixels(bar, bar.size, hv));
        sr.pref := sr.lo;
        sr.hi := sr.lo + 1;
        RETURN sr
      ELSE
        RETURN TextureVBT.T.shape(bar, ax, n)
      END
    END
  END BarShape;
</PRE> 
 Implementations of TextPainter methods 
 

<P><PRE>CONST
  Leading = 0;
  LMargin = 2;

PROCEDURE <A NAME="TextPainterInit"><procedure>TextPainterInit</procedure></A>(painter: TextPainter;
                          bg: PaintOp.T := PaintOp.Bg;
                          fg: PaintOp.T := PaintOp.Fg;
                          hiliteBg: PaintOp.T := PaintOp.Fg;
                          hiliteFg: PaintOp.T := PaintOp.Bg;
                          font: Font.T := Font.BuiltIn): TextPainter =
  BEGIN
    painter.mu := NEW(MUTEX);
    painter.eraseColor := bg;
    painter.textColor := PaintOp.Pair(PaintOp.Transparent, fg);
    painter.hiliteColor := hiliteBg;
    painter.hiliteTextColor := PaintOp.Pair(PaintOp.Transparent, hiliteFg);
    LOCK painter.mu DO
      painter.font := font;
    END;
    RETURN painter
  END TextPainterInit;

PROCEDURE <A NAME="TextPainterHeight"><procedure>TextPainterHeight</procedure></A>(painter: TextPainter; v: VBT.T): INTEGER =
  (* LL.sup = list *)
  VAR bBox: Rect.T;
  BEGIN
    LOCK painter.mu DO
      bBox := VBT.BoundingBox(v, &quot;X&quot;, painter.font);
      painter.ascent := -bBox.north;
      painter.descent := bBox.south;
    END;
    RETURN Leading + Rect.VerSize(bBox)
  END TextPainterHeight;

PROCEDURE <A NAME="TextPainterPaint"><procedure>TextPainterPaint</procedure></A> (painter : TextPainter;
                            v       : VBT.T;
                            r       : Rect.T;
                            value   : REFANY;
                 &lt;*UNUSED*&gt; index   : CARDINAL;
                            selected: BOOLEAN;
                            bad     : Rect.T       ) =
  (* LL.sup = list *)
  VAR tintColor, textColor: PaintOp.T;
  BEGIN
    IF selected THEN
      tintColor := painter.hiliteColor;
      textColor := painter.hiliteTextColor;
    ELSE
      tintColor := painter.eraseColor;
      textColor := painter.textColor;
    END;
    VBT.PaintTint (v, bad, tintColor);
    IF value # NIL THEN
      LOCK painter.mu DO
        VBT.PaintText (
          v := v,
          pt := Point.T {h := r.west + LMargin, v :=
                         r.south - painter.descent - Leading},
          fnt := painter.font, op := textColor,
          t := NARROW (value, TEXT));
      END;
    END;
  END TextPainterPaint;

PROCEDURE <A NAME="TextPainterSelect"><procedure>TextPainterSelect</procedure></A>(painter: TextPainter; v: VBT.T; r: Rect.T;
                            value: REFANY; index: CARDINAL; selected: BOOLEAN) =
  (* LL.sup = list *)
  BEGIN
    painter.paint(v, r, value, index, selected, r);
  END TextPainterSelect;

PROCEDURE <A NAME="TextPainterErase"><procedure>TextPainterErase</procedure></A>(painter: TextPainter; v: VBT.T; r: Rect.T) =
  (* LL.sup = list *)
  BEGIN
    VBT.PaintTint(v, r, painter.eraseColor);
  END TextPainterErase;

PROCEDURE <A NAME="TextPainterSetFont"><procedure>TextPainterSetFont</procedure></A>(painter: TextPainter; v: VBT.T; font: Font.T) =
  (* LL.sup &lt; v *)
  BEGIN
    LOCK painter.mu DO
      painter.font := font;
      VBT.Mark(v);
    END;
  END TextPainterSetFont;
</PRE> 
 Implementations of UniSelector methods 
 

<P><PRE>PROCEDURE <A NAME="UniSelectorInit"><procedure>UniSelectorInit</procedure></A>(selector: UniSelector; l: T): Selector =
  BEGIN
    selector.list := l;
    RETURN selector
  END UniSelectorInit;

PROCEDURE <A NAME="UniSelectorInsideClick"><procedure>UniSelectorInsideClick</procedure></A> (                    selector: UniSelector;
                                  &lt;*UNUSED*&gt; READONLY cd      : VBT.MouseRec;
                                                      this    : Cell          ) =
  (* LL.sup = VBT.mu *)
  BEGIN
    selector.list.selectOnly (this);
  END UniSelectorInsideClick;

PROCEDURE <A NAME="UniSelectorOutsideClick"><procedure>UniSelectorOutsideClick</procedure></A> (&lt;*UNUSED*&gt;          selector: UniSelector;
                                   &lt;*UNUSED*&gt; READONLY cd      : VBT.MouseRec ) =
  (* LL.sup = VBT.mu *)
  BEGIN
  END UniSelectorOutsideClick;

PROCEDURE <A NAME="UniSelectorInsideDrag"><procedure>UniSelectorInsideDrag</procedure></A> (selector: UniSelector;
                                 &lt;*UNUSED*&gt; READONLY cd  : VBT.PositionRec;
                                                     this: Cell             ) =
  (* LL.sup = VBT.mu *)
  BEGIN
    selector.list.selectOnly (this);
  END UniSelectorInsideDrag;

PROCEDURE <A NAME="UniSelectorOutsideDrag"><procedure>UniSelectorOutsideDrag</procedure></A> (&lt;*UNUSED*&gt; selector: UniSelector;
                                  &lt;*UNUSED*&gt; READONLY cd: VBT.PositionRec) =
  (* LL.sup = VBT.mu *)
  BEGIN
  END UniSelectorOutsideDrag;
</PRE> 
 Implementations of MultiSelector methods 
 

<P><PRE>PROCEDURE <A NAME="MultiSelectorInit"><procedure>MultiSelectorInit</procedure></A>(selector: MultiSelector; l: T): Selector =
  BEGIN
    selector.list := l;
    RETURN selector
  END MultiSelectorInit;

PROCEDURE <A NAME="MultiSelectorInsideClick"><procedure>MultiSelectorInsideClick</procedure></A> (         selector: MultiSelector;
                                    READONLY cd      : VBT.MouseRec;
                                             this    : Cell           ) =
  (* LL.sup = VBT.mu *)
  BEGIN
    IF cd.clickType = VBT.ClickType.FirstDown THEN
      WITH list = selector.list DO
        selector.anchor := this;
        IF VBT.Modifier.Shift IN cd.modifiers THEN
          selector.adding := NOT list.isSelected (this);
        ELSE
          selector.adding := TRUE;
          list.selectNone ();
        END;
        list.select (this, selector.adding);
        selector.prev := this;
      END;
    END;
  END MultiSelectorInsideClick;

PROCEDURE <A NAME="MultiSelectorOutsideClick"><procedure>MultiSelectorOutsideClick</procedure></A> (&lt;*UNUSED*&gt; selector: MultiSelector;
                                     &lt;*UNUSED*&gt; READONLY cd: VBT.MouseRec) =
  (* LL.sup = VBT.mu *)
  BEGIN
  END MultiSelectorOutsideClick;

PROCEDURE <A NAME="MultiSelectorInsideDrag"><procedure>MultiSelectorInsideDrag</procedure></A> (selector: MultiSelector;
                                   &lt;*UNUSED*&gt; READONLY cd  : VBT.PositionRec;
                                                       this: Cell             ) =
  (* LL.sup = VBT.mu *)
  BEGIN
    WITH list = selector.list DO
      (* There are numerous cases; either first or last loop is empty. *)
      FOR i := selector.prev TO MIN (this, selector.anchor) - 1 DO
        (* prev &lt; this and prev &lt; anchor: undo after prev *)
        list.select (i, NOT selector.adding);
      END;
      FOR i := MIN (this, selector.anchor + 1)
          TO MAX (selector.anchor - 1, this) DO
        (* apply between this and anchor, in either order *)
        list.select (i, selector.adding);
      END;
      FOR i := MAX (this, selector.anchor) + 1 TO selector.prev DO
        (* prev &gt; this and prev &gt; anchor: undo up to prev *)
        list.select (i, NOT selector.adding);
      END;
      selector.prev := this;
    END;
  END MultiSelectorInsideDrag;

PROCEDURE <A NAME="MultiSelectorOutsideDrag"><procedure>MultiSelectorOutsideDrag</procedure></A> (&lt;*UNUSED*&gt; selector: MultiSelector;
                                    &lt;*UNUSED*&gt; READONLY cd: VBT.PositionRec) =
  (* LL.sup = VBT.mu *)
  BEGIN
  END MultiSelectorOutsideDrag;

BEGIN
END ListVBT.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
