<HTML>
<HEAD>
<TITLE>SRC Modula-3: m3front/src/misc/CG.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>m3front/src/misc/CG.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                             </EM></BLOCKQUOTE><PRE>
</PRE> File: CG.m3                                                 

<P><PRE>MODULE <module><implements><A HREF="CG.i3">CG</A></implements></module>;

IMPORT <A HREF="../../../text/src/Text.i3">Text</A>, <A HREF="../../../libm3/derived/IntIntTbl.i3">IntIntTbl</A>, <A HREF="../../../libm3/derived/IntRefTbl.i3">IntRefTbl</A>, <A HREF="../../../fmtlex/src/Fmt.i3">Fmt</A>, <A HREF="../../../word/src/Word.i3">Word</A>;
IMPORT <A HREF="Scanner.i3">Scanner</A>, <A HREF="Error.i3">Error</A>, <A HREF="../values/Module.i3">Module</A>, <A HREF="Runtime.i3">Runtime</A>, <A HREF="WebInfo.i3">WebInfo</A>;
IMPORT <A HREF="M3.i3">M3</A>, <A HREF="../../../m3middle/src/M3CG.i3">M3CG</A>, <A HREF="../../../m3middle/src/M3CG_Ops.i3">M3CG_Ops</A>, <A HREF="../../../m3middle/src/M3CG_Check.i3">M3CG_Check</A>;
IMPORT <A HREF="Host.i3">Host</A>, <A HREF="../../../m3middle/src/Target.i3">Target</A>, <A HREF="../../../m3middle/src/TInt.i3">TInt</A>, <A HREF="../../../m3middle/src/TFloat.i3">TFloat</A>, <A HREF="../../../m3middle/src/TWord.i3">TWord</A>, <A HREF="../../../m3middle/src/TargetMap.i3">TargetMap</A>, <A HREF="../../../m3middle/src/M3RT.i3">M3RT</A> (**, RTObject **);

CONST
  Max_init_chars = 256; (* max size of a single init_chars string *)

REVEAL
  <A NAME="Val">Val</A> = BRANDED &quot;CG.Val&quot; REF ValRec;

TYPE
  VKind = {      (* TYPE   VALUE                 *)
    Integer,     (* Int    int                   *)
    Float,       (* Float  float                 *)
    Stacked,     (* any    S0.type               *)
    Direct,      (* any    MEM(ADR(base) + OFFS) *)
    Absolute,    (* Addr   ADR(base) + OFFS      *)
    Indirect,    (* Addr   MEM(base) + OFFS      *)
    Pointer      (* Addr   S0.A + OFFS           *)
  }; (* where OFFS == offset + MEM(bits)         *)

TYPE
  ValRec = RECORD
    kind      : VKind;        (* type of descriptor *)
    type      : Type;         (* type of the value *)
    temp_base : BOOLEAN;      (* TRUE =&gt; base is a temp. *)
    temp_bits : BOOLEAN;      (* TRUE =&gt; bits is a temp. *)
    align     : Alignment;    (* assumed alignment of base address *)
    base      : Var;          (* base address *)
    bits      : Var;          (* non-constant bit offset *)
    offset    : INTEGER;      (* constant bit offset *)
    next      : Val;          (* link for lists *)
    int       : Target.Int;   (* literal integer value *)
    float     : Target.Float; (* literal floating point value *)
  END;

TYPE
  TempWrapper = REF RECORD
    next   : TempWrapper;
    temp   : Var;
    size   : Size;
    align  : Alignment;
    type   : Type;
    in_mem : BOOLEAN;
    block  : INTEGER;
  END;

TYPE
  Node = OBJECT
    next : Node;
    (** file : String.T;**)
    (** line : INTEGER; **)
    o    : Offset;
  METHODS
    dump();
  END;

TYPE
  FloatNode   = Node OBJECT f: Target.Float OVERRIDES dump := DumpFloat END;
  CharsNode   = Node OBJECT t: TEXT  OVERRIDES dump := DumpChars END;
  ProcNode    = Node OBJECT v: Proc OVERRIDES dump := DumpProc END;
  LabelNode   = Node OBJECT v: Label OVERRIDES dump := DumpLabel END;
  VarNode     = Node OBJECT v: Var;  b: Offset OVERRIDES dump := DumpVar END;
  OffsetNode  = Node OBJECT v: Var;  OVERRIDES dump := DumpOffset END;
  CommentNode = Node OBJECT a, b, c, d: TEXT OVERRIDES dump := DumpComment END;
  IntNode     = Node OBJECT s: Size; v: Target.Int OVERRIDES dump := DumpInt END;
  FieldNode   = Node OBJECT n: Name; s: Size; t: TypeUID OVERRIDES dump := DumpField END;

VAR
  cg_wr       : M3CG.T      := NIL;
  cg_check    : M3CG.T      := NIL;
  cg          : M3CG.T      := NIL;
  last_offset : INTEGER     := -2;
  last_file   : TEXT        := NIL;
  last_line   : INTEGER     := -2;
  pending     : Node        := NIL;
  fields      : Node        := NIL;
  in_init     : BOOLEAN     := FALSE;
  init_pc     : INTEGER     := 0;
  init_bits   : Target.Int  := TInt.Zero;
  free_temps  : TempWrapper := NIL;
  busy_temps  : TempWrapper := NIL;
  free_values : Val         := NIL;
  busy_values : Val         := NIL;
  indirects   : IntIntTbl.T := NIL;
  variables   : IntRefTbl.T := NIL;
  procedures  : IntRefTbl.T := NIL;
  block_cnt   : INTEGER     := 0;
  tos         : CARDINAL    := 0;  (* top-of-stack *)
  stack       : ARRAY [0..99] OF ValRec;
</PRE>---------------------------------------------------------------------------

<P><PRE>PROCEDURE <A NAME="Init"><procedure>Init</procedure></A> () =
  BEGIN
    Max_alignment := Target.Alignments [LAST (Target.Alignments)];

    cg_wr := Host.env.init_code_generator ();
    IF (cg_wr = NIL) THEN
      Error.Msg (&quot;unable to create a code generator&quot;);
      RETURN;
    END;
    (** RTObject.PatchMethods (cg_wr); **)

    cg_check := M3CG_Check.New (cg_wr,
                                clean_jumps  := Host.clean_jumps,
                                clean_stores := Host.clean_stores,
                                nested_calls := Host.nested_calls,
                                nested_procs := Host.inline_nested_procs);
    (** RTObject.PatchMethods (cg_check); **)
    cg := cg_check;

    cg.set_error_handler (Error.Msg);

    last_offset := -2;
    last_file   := NIL;
    last_line   := -2;
    pending     := NIL;
    fields      := NIL;
    in_init     := FALSE;
    init_pc     := 0;
    init_bits   := TInt.Zero;
    free_temps  := NIL;
    busy_temps  := NIL;
    free_values := NIL;
    busy_values := NIL;
    indirects   := NIL;
    variables   := NIL;
    procedures  := NIL;
    block_cnt   := 0;
    tos         := 0;
  END Init;
</PRE>----------------------------------------------------------- ID counters ---

<P><PRE>PROCEDURE <A NAME="Next_label"><procedure>Next_label</procedure></A> (n_labels := 1): Label =
  BEGIN
    RETURN cg.next_label (n_labels);
  END Next_label;
</PRE>----------------------------------------------------- compilation units ---

<P><PRE>PROCEDURE <A NAME="Begin_unit"><procedure>Begin_unit</procedure></A> (optimize: INTEGER := 0) =
  BEGIN
    cg.begin_unit (optimize);
  END Begin_unit;

PROCEDURE <A NAME="End_unit"><procedure>End_unit</procedure></A> () =
  BEGIN
    Free_all_values ();
    Free_all_temps ();
    cg.end_unit ();
  END End_unit;

PROCEDURE <A NAME="Import_unit"><procedure>Import_unit</procedure></A> (n: Name) =
  BEGIN
    cg.import_unit (n);
    WebInfo.Import_unit (n);
  END Import_unit;

PROCEDURE <A NAME="Export_unit"><procedure>Export_unit</procedure></A> (n: Name) =
  BEGIN
    cg.export_unit (n);
    WebInfo.Export_unit (n);
  END Export_unit;
</PRE>------------------------------------------------ debugging line numbers ---

<P><PRE>PROCEDURE <A NAME="Gen_location"><procedure>Gen_location</procedure></A> (here: INTEGER) =
  VAR file: TEXT;  save, line: INTEGER;
  BEGIN
    IF (here = last_offset) THEN RETURN END;

    save := Scanner.offset;
    Scanner.offset := here;
    Scanner.LocalHere (file, line);

    IF (last_file = NIL) OR NOT Text.Equal (last_file, file) THEN
      cg.set_source_file (file);
      last_file := file;
    END;

    IF (last_line # line) THEN
      cg.set_source_line (line);
      last_line := line;
    END;

    Scanner.offset := save;
    last_offset := here;
  END Gen_location;
</PRE>------------------------------------------- debugging type declarations ---

<P><PRE>PROCEDURE <A NAME="Declare_typename"><procedure>Declare_typename</procedure></A> (t: TypeUID;  n: Name) =
  BEGIN
    cg.declare_typename (t, n);
  END Declare_typename;

PROCEDURE <A NAME="Declare_array"><procedure>Declare_array</procedure></A> (t: TypeUID;  index, elt: TypeUID;  s: Size) =
  BEGIN
    cg.declare_array (t, index, elt, s);
    WebInfo.Declare_array (t, index, elt, s);
  END Declare_array;

PROCEDURE <A NAME="Declare_open_array"><procedure>Declare_open_array</procedure></A> (t: TypeUID;  elt: TypeUID;  s: Size) =
  BEGIN
    cg.declare_open_array (t, elt, s);
    WebInfo.Declare_open_array (t, elt, s);
  END Declare_open_array;

PROCEDURE <A NAME="Declare_enum"><procedure>Declare_enum</procedure></A> (t: TypeUID;  n_elts: INTEGER;  s: Size) =
  BEGIN
    cg.declare_enum (t, n_elts, s);
    WebInfo.Declare_enum (t, n_elts, s);
  END Declare_enum;

PROCEDURE <A NAME="Declare_enum_elt"><procedure>Declare_enum_elt</procedure></A> (n: Name) =
  BEGIN
    cg.declare_enum_elt (n);
    WebInfo.Declare_enum_elt (n);
  END Declare_enum_elt;

PROCEDURE <A NAME="Declare_packed"><procedure>Declare_packed</procedure></A> (t: TypeUID;  s: Size;  base: TypeUID) =
  BEGIN
    cg.declare_packed (t, s, base);
    WebInfo.Declare_packed (t, s, base);
  END Declare_packed;

PROCEDURE <A NAME="Declare_record"><procedure>Declare_record</procedure></A> (t: TypeUID;  s: Size;  n_fields: INTEGER) =
  BEGIN
    cg.declare_record (t, s, n_fields);
    WebInfo.Declare_record (t, s, n_fields);
  END Declare_record;

PROCEDURE <A NAME="Declare_field"><procedure>Declare_field</procedure></A> (n: Name;  o: Offset;  s: Size;  t: TypeUID) =
  BEGIN
    cg.declare_field (n, o, s, t);
    WebInfo.Declare_field (n, o, s, t);
  END Declare_field;

PROCEDURE <A NAME="Declare_set"><procedure>Declare_set</procedure></A> (t, domain: TypeUID;  s: Size) =
  BEGIN
    cg.declare_set (t, domain, s);
    WebInfo.Declare_set (t, domain, s);
  END Declare_set;

PROCEDURE <A NAME="Declare_subrange"><procedure>Declare_subrange</procedure></A> (t, domain: TypeUID;  READONLY min, max: Target.Int;
                                                 s: Size) =
  BEGIN
    cg.declare_subrange (t, domain, min, max, s);
    WebInfo.Declare_subrange (t, domain, min, max, s);
  END Declare_subrange;

PROCEDURE <A NAME="Declare_pointer"><procedure>Declare_pointer</procedure></A> (t, target: TypeUID;  brand: TEXT;  traced: BOOLEAN)=
  BEGIN
    cg.declare_pointer (t, target, brand, traced);
    WebInfo.Declare_pointer (t, target, brand, traced);
  END Declare_pointer;

PROCEDURE <A NAME="Declare_indirect"><procedure>Declare_indirect</procedure></A> (target: TypeUID): TypeUID =
  VAR x: INTEGER;
  BEGIN
    IF (indirects = NIL) THEN indirects := NewIntTbl () END;
    IF NOT indirects.get (target, x) THEN
      x := Word.Not (target);  (* !! fingerprint HACK !! *)
      cg.declare_indirect (x, target);
      WebInfo.Declare_indirect (x, target);
      EVAL indirects.put (target, x);
    END;
    RETURN x;
  END Declare_indirect;

PROCEDURE <A NAME="Declare_proctype"><procedure>Declare_proctype</procedure></A> (t: TypeUID;  n_formals: INTEGER;
                            result: TypeUID;  n_raises: INTEGER;
                            cc: CallingConvention) =
  BEGIN
    cg.declare_proctype (t, n_formals, result, n_raises, cc);
    WebInfo.Declare_proctype (t, n_formals, result, n_raises);
  END Declare_proctype;

PROCEDURE <A NAME="Declare_formal"><procedure>Declare_formal</procedure></A> (n: Name;  t: TypeUID) =
  BEGIN
    cg.declare_formal (n, t);
    WebInfo.Declare_formal (n, t);
  END Declare_formal;

PROCEDURE <A NAME="Declare_raises"><procedure>Declare_raises</procedure></A> (n: Name) =
  BEGIN
    cg.declare_raises (n);
    WebInfo.Declare_raises (n);
  END Declare_raises;

PROCEDURE <A NAME="Declare_object"><procedure>Declare_object</procedure></A> (t, super: TypeUID;  brand: TEXT;  traced: BOOLEAN;
                           n_fields, n_methods, n_overrides: INTEGER;
                           field_size: Size) =
  BEGIN
    cg.declare_object (t, super, brand, traced,
                       n_fields, n_methods, field_size);
    WebInfo.Declare_object (t, super, brand, traced,
                            n_fields, n_methods, n_overrides, field_size);
  END Declare_object;

PROCEDURE <A NAME="Declare_method"><procedure>Declare_method</procedure></A> (n: Name;  signature: TypeUID;  dfault: M3.Expr) =
  BEGIN
    cg.declare_method (n, signature);
    WebInfo.Declare_method (n, signature, dfault);
  END Declare_method;

PROCEDURE <A NAME="Declare_override"><procedure>Declare_override</procedure></A> (n: Name;  dfault: M3.Expr) =
  BEGIN
    WebInfo.Declare_override (n, dfault);
  END Declare_override;

PROCEDURE <A NAME="Declare_opaque"><procedure>Declare_opaque</procedure></A> (t, super: TypeUID) =
  BEGIN
    cg.declare_opaque (t, super);
    WebInfo.Declare_opaque (t, super);
  END Declare_opaque;

PROCEDURE <A NAME="Reveal_opaque"><procedure>Reveal_opaque</procedure></A> (lhs, rhs: TypeUID) =
  BEGIN
    cg.reveal_opaque (lhs, rhs);
    WebInfo.Reveal_opaque (lhs, rhs);
  END Reveal_opaque;

PROCEDURE <A NAME="Declare_global_field"><procedure>Declare_global_field</procedure></A> (n: Name;  o: Offset;  s: Size;  t: TypeUID) =
  BEGIN
    fields := NEW (FieldNode, next := fields, n := n, o := o, s := s, t := t);
  END Declare_global_field;

PROCEDURE <A NAME="DumpField"><procedure>DumpField</procedure></A> (x: FieldNode) =
  BEGIN
    (* DumpNode (x);  -- no file &amp; line number info *)
    cg.declare_field (x.n, x.o, x.s, x.t);
  END DumpField;

PROCEDURE <A NAME="Emit_global_record"><procedure>Emit_global_record</procedure></A> (s: Size) =
  VAR n := fields;  cnt := 0;  xx: REF ARRAY OF Node;
  BEGIN
    (* build a sorted array of fields *)
    WHILE (n # NIL) DO INC (cnt);  n := n.next END;
    xx := NEW (REF ARRAY OF Node, cnt);
    n := fields;  cnt := 0;
    WHILE (n # NIL) DO xx[cnt] := n;  INC (cnt);  n := n.next;  END;
    SortNodes (xx^);

    (* finally, declare the record *)
    cg.declare_record (-1, s, NUMBER (xx^));
    FOR i := 0 TO LAST (xx^) DO  xx[i].dump () END;
    fields := NIL;
  END Emit_global_record;

PROCEDURE <A NAME="Declare_exception"><procedure>Declare_exception</procedure></A> (n: Name;  arg_type: TypeUID;
                           raise_proc: BOOLEAN;  base: Var;  offset: INTEGER) =
  BEGIN
    cg.declare_exception (n, arg_type, raise_proc, base, ToBytes (offset));
  END Declare_exception;
</PRE>--------------------------------------------------------- runtime hooks ---

<P><PRE>PROCEDURE <A NAME="Set_runtime_hook"><procedure>Set_runtime_hook</procedure></A> (n: Name;  v: Var;  o: Offset) =
  BEGIN
    cg.set_runtime_hook (n, v, AsBytes (o));
  END Set_runtime_hook;

PROCEDURE <A NAME="Get_runtime_hook"><procedure>Get_runtime_hook</procedure></A> (n: Name;  VAR v: Var;  VAR o: Offset) =
  BEGIN
    cg.get_runtime_hook (n, v, o);
    o := o * Target.Byte; (* bytes back to bits... *)
  END Get_runtime_hook;
</PRE>------------------------------------------------- variable declarations ---

<P><PRE>PROCEDURE <A NAME="Import_global"><procedure>Import_global</procedure></A> (n: Name;  s: Size;  a: Alignment;  t: Type;
                         m3t: TypeUID): Var =
  VAR ref: REFANY;  v: Var;
  BEGIN
    IF (variables = NIL) THEN variables := NewNameTbl () END;
    IF variables.get (n, ref) THEN RETURN ref END;
    v := cg.import_global (n, ToVarSize (s, a), FixAlign (a), t, m3t);
    EVAL variables.put (n, v);
    RETURN v;
  END Import_global;

PROCEDURE <A NAME="Declare_segment"><procedure>Declare_segment</procedure></A> (n: Name;  m3t: TypeUID): Var =
  BEGIN
    RETURN cg.declare_segment (n, m3t);
  END Declare_segment;

PROCEDURE <A NAME="Bind_segment"><procedure>Bind_segment</procedure></A> (seg: Var;  s: Size;  a: Alignment;  t: Type;
                        exported, init: BOOLEAN) =
  BEGIN
    cg.bind_segment (seg, ToVarSize (s, a), FixAlign (a), t, exported, init);
    IF (init) THEN
      Begin_init (seg);
      DumpPendingNodes ();
      End_init (seg);
    END;
  END Bind_segment;

PROCEDURE <A NAME="Declare_global"><procedure>Declare_global</procedure></A> (n: Name;  s: Size;  a: Alignment;  t: Type;
                          m3t: TypeUID;  exported, init: BOOLEAN): Var =
  BEGIN
    RETURN cg.declare_global (n, ToVarSize (s, a), FixAlign (a),
                              t, m3t, exported, init);
  END Declare_global;

PROCEDURE <A NAME="Declare_constant"><procedure>Declare_constant</procedure></A> (n: Name;  s: Size;  a: Alignment;  t: Type;
                            m3t: TypeUID;  exported, init: BOOLEAN): Var =
  BEGIN
    RETURN cg.declare_constant (n, ToVarSize (s, a), FixAlign (a),
                                t, m3t, exported, init);
  END Declare_constant;

PROCEDURE <A NAME="Declare_local"><procedure>Declare_local</procedure></A> (n: Name;  s: Size;  a: Alignment;  t: Type;
                         m3t: TypeUID;  in_memory, up_level: BOOLEAN;
                         f: Frequency): Var =
  BEGIN
    RETURN cg.declare_local (n, ToVarSize (s, a), FixAlign (a),
                             t, m3t, in_memory, up_level, f);
  END Declare_local;

PROCEDURE <A NAME="Declare_param"><procedure>Declare_param</procedure></A> (n: Name;  s: Size;  a: Alignment;  t: Type;
                         m3t: TypeUID;  in_memory, up_level: BOOLEAN;
                         f: Frequency): Var =
  BEGIN
    RETURN cg.declare_param (n, ToVarSize (s, a), FixAlign (a),
                             t, m3t, in_memory, up_level, f);
  END Declare_param;
</PRE>----------------------------------------------------------- temporaries ---

<P>
<P><PRE>PROCEDURE <A NAME="Declare_temp"><procedure>Declare_temp</procedure></A> (s: Size;  a: Alignment;  t: Type;
                          in_memory: BOOLEAN): Var =
  VAR w := free_temps;  last_w: TempWrapper := NIL;  tmp: Var;
  BEGIN
    LOOP
      IF (w = NIL) THEN
        (* we need to allocate a fresh one *)
        tmp := cg.declare_temp (ToVarSize (s, a), FixAlign (a), t, in_memory);
        busy_temps := NEW (TempWrapper, size := s, align := a, type := t,
                           in_mem := in_memory, temp := tmp,
                           block := block_cnt, next := busy_temps);
        RETURN tmp;
      ELSIF (w.size = s) AND (w.align = a) AND (w.type = t) AND
        (w.in_mem = in_memory) THEN
        (* we found a match *)
        IF (last_w = NIL)
          THEN free_temps := w.next;
          ELSE last_w.next := w.next;
        END;
        w.next := busy_temps;  busy_temps := w;
        RETURN w.temp;
      ELSE
        (* try the next one *)
        last_w := w;
        w := w.next;
      END;
    END;
  END Declare_temp;

PROCEDURE <A NAME="Free_temp"><procedure>Free_temp</procedure></A> (&lt;*UNUSED*&gt; v: Var) =
  BEGIN
  END Free_temp;

PROCEDURE <A NAME="Free_temps"><procedure>Free_temps</procedure></A> () =
  VAR w := busy_temps;
  BEGIN
    SEmpty (&quot;Free_temps&quot;);
    IF (w # NIL) THEN
      WHILE (w.next # NIL) DO  w := w.next;  END;
      w.next := free_temps;
      free_temps := busy_temps;
      busy_temps := NIL;
    END;
  END Free_temps;
</PRE>*****
PROCEDURE Free_one_temp (v: Var) =
  VAR w := busy_temps;  last_w : TempWrapper := NIL;
  BEGIN
    LOOP
      IF (w = NIL) THEN Error.Msg (<CODE></CODE>);
        (* missing wrapper! 
        <PRE>Err (&quot;missing temp wrapper&quot;);
        cg.free_temp (v);
        RETURN;
      ELSIF (w.temp = v) THEN
        (* we found the match *)
        IF (last_w = NIL)
          THEN busy_temps := w.next;
          ELSE last_w.next := w.next;
        END;
        w.next := free_temps;  free_temps := w;
        RETURN;
      ELSE
        (* try the next one *)
        last_w := w;
        w := w.next;
      END;
    END;
  END Free_one_temp;
*********)

PROCEDURE <A NAME="Free_all_temps"><procedure>Free_all_temps</procedure></A> () =
  VAR w: TempWrapper;
  BEGIN
    Free_temps ();
    &lt;*ASSERT busy_temps = NIL*&gt;
    w := free_temps;
    WHILE (w # NIL) DO
      cg.free_temp (w.temp);
      w := w.next;
    END;
    free_temps := NIL;
  END Free_all_temps;

PROCEDURE <A NAME="Free_block_temps"><procedure>Free_block_temps</procedure></A> (block: INTEGER) =
  VAR w, prev_w: TempWrapper;
  BEGIN
    Free_temps ();
    &lt;*ASSERT busy_temps = NIL*&gt;
    w := free_temps;  prev_w := NIL;
    WHILE (w # NIL) DO
      IF (w.block = block) THEN
        cg.free_temp (w.temp);
        IF (prev_w # NIL)
          THEN  prev_w.next := w.next;
          ELSE  free_temps := w.next;
        END;
      END;
      w := w.next;
    END;
  END Free_block_temps;
</PRE>--------------------------------------------- direct stack manipulation ---

<P><PRE>PROCEDURE <A NAME="Pop"><procedure>Pop</procedure></A> (): Val =
  VAR z: Var;  v: Val;
  BEGIN
    (* get a free value *)
    v := free_values;
    IF (v = NIL)
      THEN v := NEW (Val);
      ELSE free_values := v.next;
    END;

    (* fill it in *)
    WITH x = stack [SCheck (1, &quot;Pop&quot;)] DO
      v^ := x;
    END;
    SPop (1, &quot;Pop&quot;);

    (* mark it as busy *)
    v.next := busy_values;
    busy_values := v;

    (* make sure it's not bound to the M3CG stack *)
    IF (v.kind = VKind.Stacked) THEN
      z := Declare_temp (TargetMap.CG_Size [v.type], TargetMap.CG_Align [v.type],
                         v.type, in_memory := FALSE);
      cg.store (z, 0, v.type);
      v.kind      := VKind.Direct;
      v.temp_base := TRUE;
      v.temp_bits := FALSE;
      v.align     := TargetMap.CG_Align [v.type];
      v.base      := z;
      v.bits      := NIL;
      v.offset    := 0;

    ELSIF (v.kind = VKind.Pointer) THEN
      z := Declare_temp (Target.Address.size, Target.Address.align,
                         Type.Addr, in_memory := FALSE);
      cg.store (z, 0, Type.Addr);

      v.kind      := VKind.Indirect;
      v.type      := Type.Addr;
      v.temp_base := TRUE;
      v.temp_bits := FALSE;
      v.base      := z;
      v.bits      := NIL;
    END;

    RETURN v;
  END Pop;

PROCEDURE <A NAME="Pop_temp"><procedure>Pop_temp</procedure></A> (): Val =
  BEGIN
    Force ();
    RETURN Pop ();
  END Pop_temp;

PROCEDURE <A NAME="Push"><procedure>Push</procedure></A> (v: Val) =
  BEGIN
    WITH x = stack [SCheck (0, &quot;Push&quot;)] DO
      x := v^;
      x.temp_base := FALSE;
      x.temp_bits := FALSE;
      x.next      := NIL;
    END;
    INC (tos);
  END Push;

PROCEDURE <A NAME="Store_temp"><procedure>Store_temp</procedure></A> (v: Val) =
  BEGIN
    &lt;*ASSERT v.kind = VKind.Direct  AND  v.offset = 0 *&gt;
    Store (v.base, 0, TargetMap.CG_Size[v.type], TargetMap.CG_Align[v.type], v.type);
  END Store_temp;

PROCEDURE <A NAME="Free"><procedure>Free</procedure></A> (v: Val) =
  VAR x := busy_values;  last_x: Val := NIL;
  BEGIN
    (* remove 'v' from the busy list *)
    LOOP
      IF (x = NIL) THEN
        Err (&quot;non-busy value freed&quot;);
        EXIT;
      ELSIF (x = v) THEN
        (* we found the match *)
        IF (last_x = NIL)
          THEN busy_values := v.next;
          ELSE last_x.next := v.next;
        END;
        v.next := free_values;  free_values := v;
        EXIT;
      ELSE
        last_x := x;
        x := x.next;
      END;
    END;

    (* finally, free the temps *)
    Release_temps (v^);
  END Free;

PROCEDURE <A NAME="Free_all_values"><procedure>Free_all_values</procedure></A> () =
  BEGIN
    WHILE (busy_values # NIL) DO  Free (busy_values); END;
  END Free_all_values;

PROCEDURE <A NAME="XForce"><procedure>XForce</procedure></A> () =
  (* force the value enough so that we can do a simple indirect load/store *)
  VAR offs: INTEGER;
  BEGIN
    WITH x = stack [SCheck (1, &quot;XForce&quot;)] DO
      IF (x.kind = VKind.Direct) THEN
        Force ();
      ELSIF (x.kind = VKind.Indirect) THEN
        offs := x.offset;  x.offset := 0;
        Force ();
        x.offset := offs;
      END;
    END;
  END XForce;

PROCEDURE <A NAME="Force"><procedure>Force</procedure></A> () =
  BEGIN
    WITH x = stack [SCheck (1, &quot;Force&quot;)] DO

      (* force the value on the stack *)
      CASE (x.kind) OF

      | VKind.Integer =&gt;
          cg.load_integer (x.int);
          x.type := Type.Int;

      | VKind.Float =&gt;
          cg.load_float (x.float);
          x.type := TargetMap.Float_types [TFloat.Prec (x.float)].cg_type;

      | VKind.Stacked =&gt;
          (* value is already on the stack *)

      | VKind.Direct =&gt;
          Force_align (x);
          cg.load (x.base, AsBytes (x.offset), x.type);
          IF (x.bits # NIL) THEN
            Err (&quot;attempt to force a direct bit-level address...&quot;);
          END;

      | VKind.Absolute =&gt;
          Force_align (x);
          cg.load_address (x.base, AsBytes (x.offset));
          Force_LValue (x);

      | VKind.Indirect =&gt;
          Force_align (x);
          cg.load  (x.base, 0, Type.Addr);
          IF (x.offset # 0) THEN cg.add_offset (AsBytes (x.offset)) END;
          Force_LValue (x);

      | VKind.Pointer =&gt;
          Force_align (x);
          IF (x.offset # 0) THEN cg.add_offset (AsBytes (x.offset)) END;
          Force_LValue (x);

      END;

      (* free any temps that we used *)
      Release_temps (x);

      (* finish the descriptor *)
      x.kind      := VKind.Stacked;
      x.type      := TargetMap.CG_Base [x.type];
      x.offset    := 0;
      x.next      := NIL;
      (** x.align     := TargetMap.CG_Align [x.type];
        --- we're not changing the alignment of this value **)
    END;
  END Force;

PROCEDURE <A NAME="Force_align"><procedure>Force_align</procedure></A> (VAR x: ValRec) =
  BEGIN
    x.align := LV_align (x);
    IF (x.align MOD Target.Byte) # 0 THEN
      Err (&quot;address is not byte-aligned&quot;);
    END;
  END Force_align;

PROCEDURE <A NAME="Force_LValue"><procedure>Force_LValue</procedure></A> (VAR x: ValRec) =
  BEGIN
    x.type := Type.Addr;
    IF (x.bits # NIL) THEN
      Err (&quot;attempt to force a bit-level L-value...&quot;);
    END;
  END Force_LValue;

PROCEDURE <A NAME="Release_temps"><procedure>Release_temps</procedure></A> (VAR x: ValRec) =
  BEGIN
    IF (x.temp_base) THEN Free_temp (x.base); END;
    IF (x.temp_bits) THEN Free_temp (x.bits); END;
    x.temp_base := FALSE;
    x.temp_bits := FALSE;
    x.base      := NIL;
    x.bits      := NIL;
  END Release_temps;

PROCEDURE <A NAME="Force1"><procedure>Force1</procedure></A> (tag: TEXT) =
  BEGIN
    Force ();
    SPop (1, tag);
  END Force1;

PROCEDURE <A NAME="Force2"><procedure>Force2</procedure></A> (tag: TEXT;  commute: BOOLEAN): BOOLEAN =
  VAR swapped := Force_pair (commute);
  BEGIN
    SPop (2, tag);
    RETURN swapped;
  END Force2;
</PRE>---------------------------------------- static variable initialization ---

<P><PRE>PROCEDURE <A NAME="Begin_init"><procedure>Begin_init</procedure></A> (v: Var) =
  BEGIN
    cg.begin_init (v);
    in_init := TRUE;
    init_pc := 0;
    init_bits := TInt.Zero;
  END Begin_init;

PROCEDURE <A NAME="End_init"><procedure>End_init</procedure></A> (v: Var) =
  BEGIN
    AdvanceInit (init_pc + Target.Byte - 1); (* flush any pending bits *)
    cg.end_init (v);
    in_init := FALSE;
  END End_init;

PROCEDURE <A NAME="DumpPendingNodes"><procedure>DumpPendingNodes</procedure></A> () =
  VAR n := pending;  cnt := 0;  xx: REF ARRAY OF Node;
  BEGIN
    WHILE (n # NIL) DO INC (cnt);  n := n.next END;
    xx := NEW (REF ARRAY OF Node, cnt);
    n := pending;  cnt := 0;
    WHILE (n # NIL) DO xx[cnt] := n;  INC (cnt);  n := n.next;  END;
    SortNodes (xx^);
    FOR i := 0 TO LAST (xx^) DO  xx[i].dump () END;
    pending := NIL;
  END DumpPendingNodes;

PROCEDURE <A NAME="SortNodes"><procedure>SortNodes</procedure></A> (VAR x: ARRAY OF Node) =
  BEGIN
    QuickSort (x, 0, NUMBER (x));
    InsertionSort (x, 0, NUMBER (x));
  END SortNodes;

PROCEDURE <A NAME="QuickSort"><procedure>QuickSort</procedure></A> (VAR a: ARRAY OF Node;  lo, hi: INTEGER) =
  CONST CutOff = 9;
  VAR i, j: INTEGER;  key, tmp: Node;
  BEGIN
    WHILE (hi - lo &gt; CutOff) DO (* sort a[lo..hi) *)

      (* use median-of-3 to select a key *)
      i := (hi + lo) DIV 2;
      IF (a[lo].o &lt; a[i].o) THEN
        IF (a[i].o &lt; a[hi-1].o) THEN
	  key := a[i];
        ELSIF (a[lo].o &lt; a[hi-1].o) THEN
          key := a[hi-1];  a[hi-1] := a[i];  a[i] := key;
        ELSE
	  key := a[lo];  a[lo] := a[hi-1];  a[hi-1] := a[i];  a[i] := key;
        END;
      ELSE
        IF (a[hi-1].o &lt; a[i].o) THEN
	  key := a[i];  tmp := a[hi-1];  a[hi-1] := a[lo];  a[lo] := tmp;
        ELSIF (a[lo].o &lt; a[hi-1].o) THEN
	  key := a[lo];  a[lo] := a[i];  a[i] := key;
        ELSE
	  key := a[hi-1];  a[hi-1] := a[lo];  a[lo] := a[i];  a[i] := key;
        END;
      END;

      (* partition the array *)
      i := lo+1;  j := hi-2;

      (* find the first hole *)
      WHILE (a[j].o &gt; key.o) DO DEC (j) END;
      tmp := a[j];
      DEC (j);

      LOOP
        IF (i &gt; j) THEN EXIT END;

        WHILE (a[i].o &lt; key.o) DO INC (i) END;
        IF (i &gt; j) THEN EXIT END;
        a[j+1] := a[i];
        INC (i);

        WHILE (a[j].o &gt; key.o) DO DEC (j) END;
        IF (i &gt; j) THEN  IF (j = i-1) THEN  DEC (j)  END;  EXIT  END;
        a[i-1] := a[j];
        DEC (j);
      END;

      (* fill in the last hole *)
      a[j+1] := tmp;
      i := j+2;

      (* then, recursively sort the smaller subfile *)
      IF (i - lo &lt; hi - i)
        THEN  QuickSort (a, lo, i-1);   lo := i;
        ELSE  QuickSort (a, i, hi);     hi := i-1;
      END;

    END; (* WHILE (hi-lo &gt; CutOff) *)
  END QuickSort;

PROCEDURE <A NAME="InsertionSort"><procedure>InsertionSort</procedure></A> (VAR a: ARRAY OF Node;  lo, hi: INTEGER) =
  VAR j: INTEGER;  key: Node;
  BEGIN
    FOR i := lo+1 TO hi-1 DO
      key := a[i];
      j := i-1;
      WHILE (j &gt;= lo) AND (key.o &lt; a[j].o) DO
        a[j+1] := a[j];
        DEC (j);
      END;
      a[j+1] := key;
    END;
  END InsertionSort;

PROCEDURE <A NAME="PushPending"><procedure>PushPending</procedure></A> (n: Node) =
  BEGIN
    (** n.file := last_file; **)
    (** n.line := last_line; **)
    n.next := pending;
    pending := n;
  END PushPending;

PROCEDURE <A NAME="DumpNode"><procedure>DumpNode</procedure></A> (&lt;*UNUSED*&gt; n: Node) =
  BEGIN
    (******
    IF (last_file # n.file) THEN
      cg.set_source_file (n.file);
      last_file := n.file;
    END;
    IF (last_line # n.line) THEN
      cg.set_source_line (n.line);
      last_line := n.line;
    END;
    *******)
  END DumpNode;

PROCEDURE <A NAME="AdvanceInit"><procedure>AdvanceInit</procedure></A> (o: Offset) =
  VAR
    n_bytes := (o - init_pc) DIV Target.Byte;
    base, n_bits, tmp, new_bits: Target.Int;
    b_size: INTEGER;
    t: Type;
  BEGIN
    &lt;*ASSERT n_bytes &gt;= 0*&gt;
    &lt;*ASSERT in_init*&gt;
    WHILE (n_bytes &gt; 0) DO
      IF TInt.EQ (init_bits, TInt.Zero) THEN
        (* no more bits to flush *)
        n_bytes := 0;
        init_pc := (o DIV Target.Byte) * Target.Byte;
      ELSE
        (* send out some number of bytes *)
        EVAL FindInitType (n_bytes, init_pc, t);
        b_size := TargetMap.CG_Bytes[t];
        IF (b_size = Target.Integer.bytes) THEN
          cg.init_int (init_pc DIV Target.Byte, init_bits, t);
          init_bits := TInt.Zero;
        ELSIF Target.Little_endian
          AND TInt.FromInt (b_size * Target.Byte, base)
          AND TInt.FromInt (Target.Integer.size - b_size*Target.Byte, n_bits)
          AND TWord.Extract (init_bits, TInt.Zero, base, tmp)
          AND TWord.Extract (init_bits, base, n_bits, new_bits) THEN
          cg.init_int (init_pc DIV Target.Byte, tmp, t);
          init_bits := new_bits;
        ELSIF (NOT Target.Little_endian)
          AND TInt.FromInt (Target.Integer.size - b_size * Target.Byte, base)
          AND TInt.FromInt (b_size*Target.Byte, n_bits)
          AND TWord.Extract (init_bits, base, n_bits, tmp) THEN
          TWord.Shift (init_bits, n_bits, new_bits);
          cg.init_int (init_pc DIV Target.Byte, tmp, t);
          init_bits := new_bits;
        ELSE
          Err (&quot;unable to convert or initialize bit field value??&quot;);
          &lt;*ASSERT FALSE*&gt;
        END;
        DEC (n_bytes, TargetMap.CG_Bytes[t]);
        INC (init_pc, TargetMap.CG_Size[t]);
      END;
    END;
  END AdvanceInit;

PROCEDURE <A NAME="FindInitType"><procedure>FindInitType</procedure></A> (n_bytes, offset: INTEGER;  VAR t: Type): BOOLEAN =
  BEGIN
    FOR i := LAST (TargetMap.Int_types) TO FIRST (TargetMap.Int_types) BY -1 DO
      IF (TargetMap.Int_types[i].bytes &lt;= n_bytes)
        AND (offset MOD TargetMap.Int_types[i].align = 0) THEN
        t := TargetMap.Int_types[i].cg_type;
        RETURN TRUE;
      END;
    END;
    ErrI (n_bytes, &quot;cg: unable to find suitable target machine type&quot;);
    t := Type.Void;
    RETURN FALSE;
  END FindInitType;

PROCEDURE <A NAME="Init_int"><procedure>Init_int</procedure></A> (o: Offset;  s: Size;  READONLY value: Target.Int) =
  VAR bit_offset: INTEGER;  itype: Type;  base, n_bits, tmp: Target.Int;
  BEGIN
    IF (NOT in_init) THEN
      PushPending (NEW (IntNode, o := o, s := s, v := value));
      RETURN;
    END;

    AdvanceInit (o);
    IF Target.Little_endian
      THEN bit_offset := o - init_pc;
      ELSE bit_offset := Target.Integer.size - (o - init_pc) - s;
    END;

    IF (o = init_pc)
      AND (s &gt;= Target.Byte)
      AND (FindInitType (s DIV Target.Byte, init_pc, itype))
      AND (TargetMap.CG_Size[itype] = s) THEN
      (* simple, aligned integer initialization *)
      cg.init_int (o DIV Target.Byte, value, itype);
    ELSIF TInt.FromInt (bit_offset, base)
      AND TInt.FromInt (s, n_bits)
      AND TWord.Insert (init_bits, value, base, n_bits, tmp) THEN
      init_bits := tmp;
    ELSE
      Err (&quot;unable to stuff bit field value??&quot;);
      &lt;*ASSERT FALSE*&gt;
    END;
  END Init_int;

PROCEDURE <A NAME="Init_intt"><procedure>Init_intt</procedure></A> (o: Offset;  s: Size;  value: INTEGER) =
  VAR val: Target.Int;  b := TInt.FromInt (value, val);
  BEGIN
    IF NOT b THEN ErrI (value, &quot;integer const not representable&quot;) END;
    Init_int (o, s, val);
  END Init_intt;

PROCEDURE <A NAME="DumpInt"><procedure>DumpInt</procedure></A> (x: IntNode) =
  BEGIN
    DumpNode (x);
    Init_int (x.o, x.s, x.v);
  END DumpInt;

PROCEDURE <A NAME="Init_proc"><procedure>Init_proc</procedure></A> (o: Offset;  value: Proc) =
  BEGIN
    IF (in_init) THEN
      AdvanceInit (o);
      &lt;*ASSERT o = init_pc*&gt;
      &lt;*ASSERT o MOD Target.Address.align = 0 *&gt;
      cg.init_proc (AsBytes (o), value);
    ELSE
      PushPending (NEW (ProcNode, o := o, v := value));
    END;
  END Init_proc;

PROCEDURE <A NAME="DumpProc"><procedure>DumpProc</procedure></A> (x: ProcNode) =
  BEGIN
    DumpNode (x);
    Init_proc (x.o, x.v);
  END DumpProc;

PROCEDURE <A NAME="Init_label"><procedure>Init_label</procedure></A> (o: Offset;  value: Label) =
  BEGIN
    IF (in_init) THEN
      AdvanceInit (o);
      &lt;*ASSERT o = init_pc*&gt;
      &lt;*ASSERT o MOD Target.Address.align = 0 *&gt;
      cg.init_label (AsBytes (o), value);
    ELSE
      PushPending (NEW (LabelNode, o := o, v := value));
    END;
  END Init_label;

PROCEDURE <A NAME="DumpLabel"><procedure>DumpLabel</procedure></A> (x: LabelNode) =
  BEGIN
    DumpNode (x);
    Init_label (x.o, x.v);
  END DumpLabel;

PROCEDURE <A NAME="Init_var"><procedure>Init_var</procedure></A> (o: Offset;  value: Var;  bias: Offset) =
  BEGIN
    IF (in_init) THEN
      AdvanceInit (o);
      &lt;*ASSERT o = init_pc*&gt;
      &lt;*ASSERT o MOD Target.Address.align = 0 *&gt;
      &lt;*ASSERT bias MOD Target.Byte = 0*&gt;
      cg.init_var (AsBytes (o), value, AsBytes (bias));
    ELSE
      PushPending (NEW (VarNode, o := o, v := value, b := bias));
    END;
  END Init_var;

PROCEDURE <A NAME="DumpVar"><procedure>DumpVar</procedure></A> (x: VarNode) =
  BEGIN
    DumpNode (x);
    Init_var (x.o, x.v, x.b);
  END DumpVar;

PROCEDURE <A NAME="Init_offset"><procedure>Init_offset</procedure></A> (o: Offset;  value: Var) =
  BEGIN
    IF (in_init) THEN
      AdvanceInit (o);
      &lt;*ASSERT o = init_pc*&gt;
      &lt;*ASSERT o MOD Target.Integer.align = 0 *&gt;
      cg.init_offset (AsBytes (o), value);
    ELSE
      PushPending (NEW (OffsetNode, o := o, v := value));
    END;
  END Init_offset;

PROCEDURE <A NAME="DumpOffset"><procedure>DumpOffset</procedure></A> (x: OffsetNode) =
  BEGIN
    DumpNode (x);
    Init_offset (x.o, x.v);
  END DumpOffset;

PROCEDURE <A NAME="Init_chars"><procedure>Init_chars</procedure></A> (o: Offset;  value: TEXT) =
  VAR len, start: INTEGER;
  BEGIN
    IF (in_init) THEN
      AdvanceInit (o);
      &lt;*ASSERT o = init_pc*&gt;
      &lt;*ASSERT o MOD Target.Char.align = 0 *&gt;
      start := 0;
      len := Text.Length (value);
      WHILE (len - start &gt; Max_init_chars) DO
        cg.init_chars (AsBytes (o), Text.Sub (value, start, Max_init_chars));
        INC (o, Max_init_chars * Target.Char.size);
        INC (start, Max_init_chars);
      END;
      IF (start &lt; len) THEN
        cg.init_chars (AsBytes (o), Text.Sub (value, start));
      END;
    ELSE
      PushPending (NEW (CharsNode, o := o, t := value));
    END;
  END Init_chars;

PROCEDURE <A NAME="DumpChars"><procedure>DumpChars</procedure></A> (x: CharsNode) =
  BEGIN
    DumpNode (x);
    Init_chars (x.o, x.t);
  END DumpChars;

PROCEDURE <A NAME="Init_float"><procedure>Init_float</procedure></A> (o: Offset;  READONLY f: Target.Float) =
  BEGIN
    IF (in_init) THEN
      AdvanceInit (o);
      &lt;*ASSERT o = init_pc*&gt;
      &lt;*ASSERT o MOD Target.Real.align = 0 *&gt;
      cg.init_float (AsBytes (o), f);
    ELSE
      PushPending (NEW (FloatNode, o := o, f := f));
    END;
  END Init_float;

PROCEDURE <A NAME="DumpFloat"><procedure>DumpFloat</procedure></A> (x: FloatNode) =
  BEGIN
    DumpNode (x);
    Init_float (x.o, x.f);
  END DumpFloat;

PROCEDURE <A NAME="EmitText"><procedure>EmitText</procedure></A> (t: TEXT): INTEGER =
  VAR  len, size, align, offset: INTEGER;
  BEGIN
    IF (t = NIL) THEN t := &quot;&quot; END;
    len    := Text.Length (t) + 1;
    size   := len * Target.Char.size;
    (** align  := MAX (Target.Char.align, Target.Integer.align); **)
    align  := Target.Char.align;
    offset := Module.Allocate (size, align, &quot;*string*&quot;);
    PushPending (NEW (CharsNode, o := offset, t := t));
    RETURN offset;
  END EmitText;
</PRE>------------------------------------------------------------ procedures ---

<P><PRE>PROCEDURE <A NAME="Import_procedure"><procedure>Import_procedure</procedure></A> (n: Name;  n_params: INTEGER;  ret_type: Type;
                            cc: CallingConvention;
                            VAR(*OUT*) new: BOOLEAN): Proc =
  VAR ref: REFANY;  p: Proc;
  BEGIN
    IF (procedures = NIL) THEN procedures := NewNameTbl() END;
    IF procedures.get (n, ref) THEN new := FALSE;  RETURN ref END;
    p := cg.import_procedure (n, n_params, ret_type, cc);
    EVAL procedures.put (n, p);
    new := TRUE;
    RETURN p;
  END Import_procedure;

PROCEDURE <A NAME="Declare_procedure"><procedure>Declare_procedure</procedure></A> (n: Name;  n_params: INTEGER;  ret_type: Type;
                             lev: INTEGER;  cc: CallingConvention;
                             exported: BOOLEAN;  parent: Proc): Proc =
  BEGIN
    RETURN cg.declare_procedure (n, n_params, ret_type,
                                 lev, cc, exported, parent);
  END Declare_procedure;

PROCEDURE <A NAME="Begin_procedure"><procedure>Begin_procedure</procedure></A> (p: Proc) =
  BEGIN
    cg.begin_procedure (p);
  END Begin_procedure;

PROCEDURE <A NAME="End_procedure"><procedure>End_procedure</procedure></A> (p: Proc) =
  BEGIN
    Free_all_values ();
    Free_all_temps ();
    cg.end_procedure (p);
  END End_procedure;

PROCEDURE <A NAME="Begin_block"><procedure>Begin_block</procedure></A> () =
  BEGIN
    cg.begin_block ();
    INC (block_cnt);
  END Begin_block;

PROCEDURE <A NAME="End_block"><procedure>End_block</procedure></A> () =
  BEGIN
    Free_block_temps (block_cnt);
    DEC (block_cnt);
    cg.end_block ();
  END End_block;

PROCEDURE <A NAME="Note_procedure_origin"><procedure>Note_procedure_origin</procedure></A> (p: Proc) =
  BEGIN
    cg.note_procedure_origin (p);
  END Note_procedure_origin;
</PRE>------------------------------------------------------------ statements ---

<P><PRE>PROCEDURE <A NAME="Set_label"><procedure>Set_label</procedure></A> (l: Label;  barrier: BOOLEAN := FALSE) =
  BEGIN
    cg.set_label (l, barrier);
  END Set_label;

PROCEDURE <A NAME="Jump"><procedure>Jump</procedure></A> (l: Label) =
  BEGIN
    cg.jump (l);
  END Jump;

PROCEDURE <A NAME="If_true"><procedure>If_true</procedure></A> (l: Label;  f: Frequency) =
  BEGIN
    Force1 (&quot;If_true&quot;);
    cg.if_true (l, f);
  END If_true;

PROCEDURE <A NAME="If_false"><procedure>If_false</procedure></A> (l: Label;  f: Frequency) =
  BEGIN
    Force1 (&quot;If_false&quot;);
    cg.if_false (l, f);
  END If_false;

PROCEDURE <A NAME="If_eq"><procedure>If_eq</procedure></A> (l: Label;  t: ZType;  f: Frequency) =
  BEGIN
    EVAL Force2 (&quot;If_eq&quot;, commute := TRUE);
    cg.if_eq (l, t, f);
  END If_eq;

PROCEDURE <A NAME="If_ne"><procedure>If_ne</procedure></A> (l: Label;  t: ZType;  f: Frequency) =
  BEGIN
    EVAL Force2 (&quot;If_ne&quot;, commute := TRUE);
    cg.if_ne (l, t, f);
  END If_ne;

PROCEDURE <A NAME="If_gt"><procedure>If_gt</procedure></A> (l: Label;  t: ZType;  f: Frequency) =
  BEGIN
    IF Force2 (&quot;If_gt&quot;, commute := TRUE)
      THEN cg.if_lt (l, t, f);
      ELSE cg.if_gt (l, t, f);
    END;
  END If_gt;

PROCEDURE <A NAME="If_ge"><procedure>If_ge</procedure></A> (l: Label;  t: ZType;  f: Frequency) =
  BEGIN
    IF Force2 (&quot;If_ge&quot;, commute := TRUE)
      THEN cg.if_le (l, t, f);
      ELSE cg.if_ge (l, t, f);
    END;
  END If_ge;

PROCEDURE <A NAME="If_lt"><procedure>If_lt</procedure></A> (l: Label;  t: ZType;  f: Frequency) =
  BEGIN
    IF Force2 (&quot;If_lt&quot;, commute := TRUE)
      THEN cg.if_gt (l, t, f);
      ELSE cg.if_lt (l, t, f);
    END;
  END If_lt;

PROCEDURE <A NAME="If_le"><procedure>If_le</procedure></A> (l: Label;  t: ZType;  f: Frequency) =
  BEGIN
    IF Force2 (&quot;If_le&quot;, commute := TRUE)
      THEN cg.if_ge (l, t, f);
      ELSE cg.if_le (l, t, f);
    END;
  END If_le;

PROCEDURE <A NAME="Case_jump"><procedure>Case_jump</procedure></A> (READONLY labels: ARRAY OF Label) =
  BEGIN
    Force1 (&quot;Case_jump&quot;);
    cg.case_jump (labels);
  END Case_jump;

PROCEDURE <A NAME="Exit_proc"><procedure>Exit_proc</procedure></A> (t: Type) =
  BEGIN
    IF (t # Type.Void) THEN  Force1 (&quot;Exit_proc&quot;);  END;
    cg.exit_proc (t);
  END Exit_proc;
</PRE>------------------------------------------------------------ load/store ---

<P><PRE>PROCEDURE <A NAME="Load"><procedure>Load</procedure></A> (v: Var;  o: Offset;  s: Size;  a: Alignment;  t: Type) =
  VAR
    size  := TargetMap.CG_Size [t];
    align := TargetMap.CG_Align [t];
    best_align : Alignment;
    best_size  : Size;
    best_type  : MType;
  BEGIN
    IF (size = s) AND ((a+o) MOD align) = 0 THEN
      (* a simple aligned load *)
      SimpleLoad (v, o, t);

    ELSIF (size &lt; s) THEN
      Err (&quot;load size too large&quot;);
      SimpleLoad (v, o, t);
      Force ();  (* to connect the error message to the bad code *)

    ELSIF (t = Type.Word) OR (t = Type.Int) THEN
      best_type  := FindIntType (t, s, o, a);
      best_size  := TargetMap.CG_Size [best_type];
      best_align := TargetMap.CG_Align [best_type];
      align := (a+o) MOD best_align;
      IF (s = best_size) AND (align = 0) THEN
        (* this is a simple partial word load *)
        SimpleLoad (v, o, best_type);
      ELSE
        (* unaligned, partial load *)
        cg.load (v, AsBytes (o - align), best_type);
        IF Target.Little_endian
          THEN cg.extract_mn (t = Type.Int, align, s);
          ELSE cg.extract_mn (t = Type.Int, best_size - align - s, s);
        END;
        SPush (t);
      END;
    ELSE
      (* unaligned non-integer value *)
      Err (&quot;unaligned load  type=&quot;&amp; Fmt.Int (ORD (t))
          &amp; &quot;  s/o/a=&quot; &amp; Fmt.Int (s) &amp; &quot;/&quot; &amp; Fmt.Int (o) &amp; &quot;/&quot; &amp; Fmt.Int (a));
      SimpleLoad (v, o, t);
      Force ();  (* to connect the error message to the bad code *)
    END;
  END Load;

PROCEDURE <A NAME="SimpleLoad"><procedure>SimpleLoad</procedure></A> (v: Var;  o: Offset;  t: Type) =
  BEGIN
    WITH x = stack [SCheck (0, &quot;SimpleLoad&quot;)] DO
      x.kind      := VKind.Direct;
      x.type      := t;
      x.temp_base := FALSE;
      x.temp_bits := FALSE;
      x.align     := Target.Byte;
      x.base      := v;
      x.bits      := NIL;
      x.offset    := o;
      x.next      := NIL;
    END;
    INC (tos);
  END SimpleLoad;

PROCEDURE <A NAME="Load_addr_of"><procedure>Load_addr_of</procedure></A> (v: Var;  o: Offset;  a: Alignment) =
  BEGIN
    WITH x = stack [SCheck (0, &quot;Load_addr_of&quot;)] DO
      x.kind      := VKind.Absolute;
      x.type      := Type.Addr;
      x.temp_base := FALSE;
      x.temp_bits := FALSE;
      x.align     := FixAlign (a) * Target.Byte;
      x.base      := v;
      x.bits      := NIL;
      x.offset    := o;
      x.next      := NIL;
    END;
    INC (tos);
  END Load_addr_of;

PROCEDURE <A NAME="Load_addr_of_temp"><procedure>Load_addr_of_temp</procedure></A> (v: Var;  o: Offset;  a: Alignment) =
  BEGIN
    Load_addr_of (v, o, a);
    stack[tos-1].temp_base := TRUE;
  END Load_addr_of_temp;

PROCEDURE <A NAME="Load_int"><procedure>Load_int</procedure></A> (v: Var;  o: Offset := 0) =
  BEGIN
    SimpleLoad (v, o, Type.Int);
  END Load_int;

PROCEDURE <A NAME="Load_int_temp"><procedure>Load_int_temp</procedure></A> (v: Var;  o: Offset := 0) =
  BEGIN
    SimpleLoad (v, o, Type.Int);
    stack [tos-1].temp_base := TRUE;
  END Load_int_temp;

PROCEDURE <A NAME="Load_addr"><procedure>Load_addr</procedure></A> (v: Var;  o: Offset) =
  BEGIN
    SimpleLoad (v, o, Type.Addr);
  END Load_addr;

PROCEDURE <A NAME="Load_indirect"><procedure>Load_indirect</procedure></A> (t: Type;  o: Offset;  s: Size) =
  VAR
    size  := TargetMap.CG_Size [t];
    align := TargetMap.CG_Align [t];
    best_align : Alignment;
    best_size  : Size;
    best_type  : MType;
    a: INTEGER;
    base_align : INTEGER;
    bit_offset : INTEGER;
    save_bits  : Var;
    save_temp  : BOOLEAN;
    const_bits : INTEGER;
  BEGIN
    WITH x = stack [SCheck (1, &quot;Load_indirect&quot;)] DO
      IF (x.kind = VKind.Direct) THEN
        (* there's no lazy form of MEM(x) *)
        Force ();
      ELSIF (x.kind = VKind.Indirect) THEN
        (* there's no lazy form of MEM(x) *)
        INC (o, x.offset);  x.offset := 0;
        Force ();
      END;

      IF (x.kind = VKind.Stacked) THEN
        &lt;*ASSERT x.offset = 0*&gt;
        &lt;*ASSERT x.bits = NIL*&gt;
        x.kind := VKind.Pointer;
      END;

      &lt;*ASSERT x.kind = VKind.Pointer
            OR x.kind = VKind.Absolute *&gt;

      INC (x.offset, o);
      a := LV_align (x);

      IF (size = s) AND (a MOD align) = 0 THEN
        (* a simple aligned load *)
        SimpleIndirectLoad (x, t);

      ELSIF (size &lt; s) THEN
        Err (&quot;load_indirect size too large&quot;);
        Force (); (* to connect the error message with the code *)
        SimpleIndirectLoad (x, t);

      ELSIF (t = Type.Word) OR (t = Type.Int) THEN
        base_align := Base_align (x);
        best_type  := FindIntType (t, s, x.offset, base_align);
        best_size  := TargetMap.CG_Size [best_type];
        best_align := TargetMap.CG_Align [best_type];
        bit_offset := x.offset MOD best_align;
        IF (bit_offset = 0) AND (x.bits = NIL) THEN
          (* this is a simple partial word load *)
          SimpleIndirectLoad (x, best_type);
          (** x.type := TargetMap.CG_Base [best_type]; -- nope **)
          IF (s # best_size) THEN
            Force ();
            IF Target.Little_endian
              THEN cg.extract_mn (t = Type.Int, 0, s);
              ELSE cg.extract_mn (t = Type.Int, best_size - s, s);
            END;
          END;
        ELSIF (x.bits = NIL) THEN
          (* partial load with unaligned constant offset *)
          x.offset := x.offset - bit_offset;
          SimpleIndirectLoad (x, best_type);
          Force ();
          IF Target.Little_endian
            THEN cg.extract_mn (t = Type.Int, bit_offset, s);
            ELSE cg.extract_mn (t = Type.Int, best_size - bit_offset - s, s);
          END;
        ELSE
          (* unaligned, partial load with variable offset *)
          IF (best_align &gt; x.align) THEN Err (&quot;unaligned base variable&quot;); END;

          (* hide the bit offset *)
          save_bits := x.bits;       x.bits := NIL;
          save_temp := x.temp_bits;  x.temp_bits := FALSE;

          (* generate the aligned load *)
          const_bits := x.offset MOD best_align;
          DEC (x.offset, const_bits);
          SimpleIndirectLoad (x, best_type);
          Force ();

          (* compute the full bit offset *)
          IF Target.Little_endian THEN
            cg.load (save_bits, 0, Type.Int);
            IF (const_bits # 0) THEN
              Push_int (const_bits);
              cg.add (Type.Int);
            END;
          ELSE (* big endian *)
            Push_int (best_size - const_bits - s);
            cg.load (save_bits, 0, Type.Int);
            cg.subtract (Type.Int);
          END;

          (* extract the needed bits *)
          cg.extract_n (t = Type.Int, s);

          (* restore the hidden bit offset *)
          x.bits := save_bits;
          x.temp_bits := save_temp;
        END;
      ELSE
        (* unaligned non-integer value *)
        Err (&quot;unaligned load_indirect  type=&quot;&amp; Fmt.Int (ORD (t))
            &amp; &quot;  s/a=&quot; &amp; Fmt.Int (s) &amp; &quot;/&quot; &amp; Fmt.Int (a));
        Force ();  (* to connect the error message *)
        SimpleIndirectLoad (x, t);
        Force ();
      END;

    END;
  END Load_indirect;

PROCEDURE <A NAME="SimpleIndirectLoad"><procedure>SimpleIndirectLoad</procedure></A> (VAR x: ValRec;  t: Type) =
  VAR offs: INTEGER;
  BEGIN
    IF (x.kind = VKind.Absolute) THEN
      x.kind := VKind.Direct;
      x.type := t;
    ELSIF (x.kind = VKind.Pointer) OR (x.kind = VKind.Stacked) THEN
      offs := x.offset;  x.offset := 0;
      Force ();
      cg.load_indirect (AsBytes (offs), t);
      x.type  := t;
      x.align := Target.Byte;
      x.kind  := VKind.Stacked;
    ELSE (* ?? *)
      ErrI (ORD (x.kind), &quot;bad mode in SimpleIndirectLoad&quot;);
      Force ();
      cg.load_indirect (AsBytes (x.offset), t);
      x.type  := t;
      x.align := Target.Byte;
      x.kind  := VKind.Stacked;
    END;
  END SimpleIndirectLoad;

PROCEDURE <A NAME="Store"><procedure>Store</procedure></A> (v: Var;  o: Offset;  s: Size;  a: Alignment;  t: Type) =
  VAR
    size  := TargetMap.CG_Size [t];
    align := TargetMap.CG_Align [t];
    best_align : Alignment;
    best_size  : Size;
    best_type  : MType;
  BEGIN
    Force ();  (* materialize the value to be stored *)

    IF (size = s) AND ((a+o) MOD align) = 0 THEN
      (* a simple aligned store *)
      cg.store (v, AsBytes (o), t);
    ELSIF (size &lt; s) THEN
      Err (&quot;store size too large&quot;);
      cg.store (v, AsBytes (o), t);
    ELSIF (t = Type.Word) OR (t = Type.Int) THEN
      best_type  := FindIntType (t, s, o, a);
      best_size  := TargetMap.CG_Size [best_type];
      best_align := TargetMap.CG_Align [best_type];
      align := (a+o) MOD best_align;
      IF (s = best_size) AND (align = 0) THEN
        (* this is a simple partial word store *)
        cg.store (v, AsBytes (o), best_type);
      ELSE
        (* unaligned, partial store *)
        cg.load (v, AsBytes (o - align), best_type);
        cg.swap (t, t);
        IF Target.Little_endian
          THEN cg.insert_mn (align, s);
          ELSE cg.insert_mn (best_size - align - s, s);
        END;
        cg.store (v, AsBytes (o - align), best_type);
      END;
    ELSE
      (* unaligned non-integer value *)
      Err (&quot;unaligned store  type=&quot;&amp; Fmt.Int (ORD (t))
            &amp; &quot;  s/o/a=&quot; &amp; Fmt.Int (s) &amp; &quot;/&quot; &amp; Fmt.Int (o) &amp; &quot;/&quot; &amp; Fmt.Int(a));
      cg.store (v, ToBytes (o), t);
    END;
    SPop (1, &quot;Store&quot;);
  END Store;

PROCEDURE <A NAME="Store_ref"><procedure>Store_ref</procedure></A> (v: Var;  o: Offset := 0) =
  BEGIN
    Store (v, o, Target.Address.size, Target.Address.align, Type.Addr);
  END Store_ref;

PROCEDURE <A NAME="Store_int"><procedure>Store_int</procedure></A> (v: Var;  o: Offset := 0) =
  BEGIN
    Store (v, o, Target.Integer.size, Target.Integer.align, Type.Int);
  END Store_int;

PROCEDURE <A NAME="Store_addr"><procedure>Store_addr</procedure></A> (v: Var;  o: Offset := 0) =
  BEGIN
    Store (v, o, Target.Address.size, Target.Address.align, Type.Addr);
  END Store_addr;

PROCEDURE <A NAME="Store_ref_indirect"><procedure>Store_ref_indirect</procedure></A> (o: Offset;  &lt;*UNUSED*&gt;var: BOOLEAN) =
  BEGIN
    Store_indirect (Type.Addr, o, Target.Address.size);
  END Store_ref_indirect;

PROCEDURE <A NAME="Store_indirect"><procedure>Store_indirect</procedure></A> (t: Type;  o: Offset;  s: Size) =
  VAR
    size  := TargetMap.CG_Size [t];
    align := TargetMap.CG_Align [t];
    best_align : Alignment;
    best_size  : Size;
    best_type  : MType;
    a: INTEGER;
    tmp: Val;
    base_align: INTEGER;
    save_bits : Var     := NIL;
    save_temp : BOOLEAN := FALSE;
    const_bits: INTEGER := 0;
  BEGIN
    Force (); (* materialize the value to be stored *)

    WITH x = stack [SCheck (2, &quot;Store_indirect-x&quot;)],
         y = stack [SCheck (1, &quot;Store_indirect-y&quot;)] DO

      (* normalize the address and the value *)
      IF (x.kind = VKind.Stacked) THEN
        &lt;*ASSERT x.offset = 0*&gt;
        &lt;*ASSERT x.bits = NIL*&gt;
        const_bits := o MOD x.align;
        x.offset := o - const_bits;
        x.kind := VKind.Pointer;

        Force (); (* the rhs *)

      ELSIF (x.kind = VKind.Pointer) THEN
        (* save the bit offset *)
        save_bits := x.bits;  x.bits := NIL;
        save_temp := x.temp_bits;  x.temp_bits := FALSE;
        const_bits := (x.offset + o) MOD x.align;
        x.offset := x.offset + o - const_bits;

        Force (); (* the rhs *)

      ELSIF (x.kind = VKind.Direct) THEN
        EVAL Force_pair (commute := FALSE);  (* force both sides *)

        const_bits := o MOD x.align;
        x.offset := o - const_bits;
        x.kind := VKind.Pointer;

      ELSIF (x.kind = VKind.Absolute) THEN
        INC (x.offset, o);
        Force (); (* the rhs *)

      ELSIF (x.kind = VKind.Indirect) THEN
        (* save the bit offset *)
        save_bits := x.bits;  x.bits := NIL;
        save_temp := x.temp_bits;  x.temp_bits := FALSE;
        const_bits := (x.offset + o) MOD x.align;
        x.offset := x.offset + o - const_bits;

        EVAL Force_pair (commute := FALSE); (* both sides *)
        x.kind := VKind.Pointer;
      END;

      &lt;*ASSERT x.kind = VKind.Pointer
            OR x.kind = VKind.Absolute *&gt;

      (* restore the bit offset *)
      x.bits := save_bits;
      x.temp_bits := save_temp;
      INC (x.offset, const_bits);

      a := LV_align (x);

      IF (size = s) AND (a MOD align) = 0 THEN
        (* a simple aligned store *)
        SimpleIndirectStore (x, t);
      ELSIF (size &lt; s) THEN
        Err (&quot;store_indirect size too large&quot;);
        SimpleIndirectStore (x, t);
      ELSIF (t = Type.Word) OR (t = Type.Int) THEN
        base_align := Base_align (x);
        best_type  := FindIntType (t, s, x.offset, base_align);
        best_size  := TargetMap.CG_Size [best_type];
        best_align := TargetMap.CG_Align [best_type];
        const_bits := x.offset MOD best_align;
        IF (const_bits = 0) AND (s = best_size) AND (x.bits = NIL) THEN
          (* this is a simple partial word store *)
          SimpleIndirectStore (x, best_type);
        ELSIF (const_bits = 0) AND (x.bits = NIL) THEN
          (* this is an aligned, partial word store *)
          Swap ();
          tmp := Pop ();
          Push (tmp);  XForce ();
          SimpleIndirectLoad (stack [SCheck (1,&quot;Store_indirect-3&quot;)],best_type);
          Swap ();
          EVAL Force_pair (commute := FALSE);
          IF Target.Little_endian
            THEN cg.insert_mn (0, s);
            ELSE cg.insert_mn (best_size - s, s);
          END;
          SPop (1, &quot;Store_indirect #1&quot;);
          Push (tmp);  XForce ();
          Swap ();
          SimpleIndirectStore (x, best_type);
          Free (tmp);
        ELSIF (x.bits = NIL) THEN
          (* partial store with unaligned constant offset *)
          x.offset := x.offset DIV best_align * best_align;
          Swap ();
          tmp := Pop ();
          Push (tmp);  XForce ();
          SimpleIndirectLoad (stack [SCheck (1, &quot;Store_indirect-4&quot;)], best_type);
          Swap ();
          EVAL Force_pair (commute := FALSE);
          IF Target.Little_endian
            THEN cg.insert_mn (const_bits, s);
            ELSE cg.insert_mn (best_size - const_bits - s, s);
          END;
          SPop (1, &quot;Store_indirect #2&quot;);
          Push (tmp);  XForce ();
          Swap ();
          SimpleIndirectStore (x, best_type);
          Free (tmp);
        ELSE
          (* unaligned, partial store with variable offset *)
          IF (best_align &gt; x.align) THEN
            Err (&quot;unaligned base variable in store&quot;);
          END;

          (* hide the bit offset *)
          save_bits := x.bits;       x.bits := NIL;
          save_temp := x.temp_bits;  x.temp_bits := FALSE;

          (* generate the aligned load *)
          const_bits := x.offset MOD best_align;
          DEC (x.offset, const_bits);
          Swap ();
          tmp := Pop ();
          Push (tmp);  Force ();
          SimpleIndirectLoad (y, best_type);
          Force ();

          (* stuff the bits *)
          Swap ();
          IF Target.Little_endian THEN
            cg.load (save_bits, 0, Type.Int);
            IF (const_bits # 0) THEN
              Push_int (const_bits);
              cg.add (Type.Int);
            END;
          ELSE (* big endian *)
            Push_int (best_size - const_bits - s);
            cg.load (save_bits, 0, Type.Int);
            cg.subtract (Type.Int);
          END;
          cg.insert_n (s);
          SPop (1, &quot;Store_indirect #3&quot;);

          (* finally, store the result *)
          Push (tmp);  Force ();
          Swap ();
          SimpleIndirectStore (x, best_type);

          Free (tmp);
        END;
      ELSE
        (* unaligned non-integer value *)
        Err (&quot;unaligned store_indirect  type=&quot;&amp; Fmt.Int (ORD (t))
            &amp; &quot;  s/a=&quot; &amp; Fmt.Int (s) &amp; &quot;/&quot; &amp; Fmt.Int (a));
        SimpleIndirectStore (x, t);
      END;

    END;
    SPop (2, &quot;Store_indirect&quot;);
  END Store_indirect;

PROCEDURE <A NAME="SimpleIndirectStore"><procedure>SimpleIndirectStore</procedure></A> (READONLY x: ValRec;  t: MType)=
  BEGIN
    IF (x.kind = VKind.Absolute) THEN
      cg.store (x.base, AsBytes (x.offset), t);
    ELSIF (x.kind = VKind.Pointer) OR (x.kind = VKind.Stacked) THEN
      cg.store_indirect (AsBytes (x.offset), t);
    ELSE (* ?? *)
      ErrI (ORD (x.kind), &quot;bad mode in SimpleIndirectStore&quot;);
      cg.store_indirect (AsBytes (x.offset), t);
    END;
  END SimpleIndirectStore;
</PRE>-------------------------------------------------------------- literals ---

<P><PRE>PROCEDURE <A NAME="Load_nil"><procedure>Load_nil</procedure></A> () =
  BEGIN
    SPush (Type.Addr);
    cg.load_nil ();
    stack [tos-1].align := Target.Address.align;
  END Load_nil;

PROCEDURE <A NAME="Load_byte_address"><procedure>Load_byte_address</procedure></A> (x: INTEGER) =
  BEGIN
    SPush (Type.Addr);
    cg.load_nil ();
    cg.add_offset (x);
    stack [tos-1].align := Target.Byte;
  END Load_byte_address;

PROCEDURE <A NAME="Load_intt"><procedure>Load_intt</procedure></A> (i: INTEGER) =
  VAR val: Target.Int;  b := TInt.FromInt (i, val);
  BEGIN
    IF NOT b THEN ErrI (i, &quot;integer not representable&quot;) END;
    Load_integer (val);
  END Load_intt;

PROCEDURE <A NAME="Load_integer"><procedure>Load_integer</procedure></A> (READONLY i: Target.Int) =
  BEGIN
    SPush (Type.Int);
    WITH x = stack[tos-1] DO
      x.kind := VKind.Integer;
      x.int  := i;
    END;
  END Load_integer;

PROCEDURE <A NAME="Load_float"><procedure>Load_float</procedure></A> (READONLY f: Target.Float) =
  VAR t := TargetMap.Float_types [TFloat.Prec (f)].cg_type;
  BEGIN
    SPush (t);
    WITH x = stack[tos-1] DO
      x.kind  := VKind.Float;
      x.float := f;
    END;
  END Load_float;
</PRE>------------------------------------------------------------ arithmetic ---
   
<P><PRE>PROCEDURE <A NAME="Eq"><procedure>Eq</procedure></A> (t: ZType) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.eq (t);
    SPop (2, &quot;Eq&quot;);
    SPush (Type.Int);
  END Eq;

PROCEDURE <A NAME="Ne"><procedure>Ne</procedure></A> (t: ZType) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.ne (t);
    SPop (2, &quot;Ne&quot;);
    SPush (Type.Int);
  END Ne;

PROCEDURE <A NAME="Gt"><procedure>Gt</procedure></A> (t: ZType) =
  BEGIN
    IF Force_pair (commute := TRUE)
      THEN cg.lt (t);
      ELSE cg.gt (t);
    END;
    SPop (2, &quot;Gt&quot;);
    SPush (Type.Int);
  END Gt;

PROCEDURE <A NAME="Ge"><procedure>Ge</procedure></A> (t: ZType) =
  BEGIN
    IF Force_pair (commute := TRUE)
      THEN cg.le (t);
      ELSE cg.ge (t);
    END;
    SPop (2, &quot;Ge&quot;);
    SPush (Type.Int);
  END Ge;

PROCEDURE <A NAME="Lt"><procedure>Lt</procedure></A> (t: ZType) =
  BEGIN
    IF Force_pair (commute := TRUE)
      THEN cg.gt (t);
      ELSE cg.lt (t);
    END;
    SPop (2, &quot;Lt&quot;);
    SPush (Type.Int);
  END Lt;

PROCEDURE <A NAME="Le"><procedure>Le</procedure></A> (t: ZType) =
  BEGIN
    IF Force_pair (commute := TRUE)
      THEN cg.ge (t);
      ELSE cg.le (t);
    END;
    SPop (2, &quot;Le&quot;);
    SPush (Type.Int);
  END Le;

PROCEDURE <A NAME="Add"><procedure>Add</procedure></A> (t: AType) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.add (t);
    SPop (2, &quot;Add&quot;);
    SPush (t);
  END Add;

PROCEDURE <A NAME="Subtract"><procedure>Subtract</procedure></A> (t: AType) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.subtract (t);
    SPop (2, &quot;Subtract&quot;);
    SPush (t);
  END Subtract;

PROCEDURE <A NAME="Multiply"><procedure>Multiply</procedure></A> (t: AType) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.multiply (t);
    SPop (2, &quot;Multiply&quot;);
    SPush (t);
  END Multiply;

PROCEDURE <A NAME="Divide"><procedure>Divide</procedure></A> (t: RType) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.divide (t);
    SPop (2, &quot;Divide&quot;);
    SPush (t);
  END Divide;

PROCEDURE <A NAME="Negate"><procedure>Negate</procedure></A> (t: AType) =
  BEGIN
    Force ();
    cg.negate (t);
    SPop (1, &quot;Negate&quot;);
    SPush (t);
  END Negate;

PROCEDURE <A NAME="Abs"><procedure>Abs</procedure></A> (t: AType) =
  BEGIN
    Force ();
    cg.abs (t);
    SPop (1, &quot;Abs&quot;);
    SPush (t);
  END Abs;

PROCEDURE <A NAME="Max"><procedure>Max</procedure></A> (t: ZType) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.max (t);
    SPop (2, &quot;Max&quot;);
    SPush (t);
  END Max;

PROCEDURE <A NAME="Min"><procedure>Min</procedure></A> (t: ZType) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.min (t);
    SPop (2, &quot;Min&quot;);
    SPush (t);
  END Min;

PROCEDURE <A NAME="Round"><procedure>Round</procedure></A> (t: RType) =
  BEGIN
    Force ();
    cg.round (t);
    SPop (1, &quot;Round&quot;);
    SPush (Type.Int);
  END Round;

PROCEDURE <A NAME="Trunc"><procedure>Trunc</procedure></A> (t: RType) =
  BEGIN
    Force ();
    cg.trunc (t);
    SPop (1, &quot;Trunc&quot;);
    SPush (Type.Int);
  END Trunc;

PROCEDURE <A NAME="Floor"><procedure>Floor</procedure></A> (t: RType) =
  BEGIN
    Force ();
    cg.floor (t);
    SPop (1, &quot;Floor&quot;);
    SPush (Type.Int);
  END Floor;

PROCEDURE <A NAME="Ceiling"><procedure>Ceiling</procedure></A> (t: RType) =
  BEGIN
    Force ();
    cg.ceiling (t);
    SPop (1, &quot;Ceiling&quot;);
    SPush (Type.Int);
  END Ceiling;

PROCEDURE <A NAME="Cvt_float"><procedure>Cvt_float</procedure></A> (t: AType;  u: RType) =
  BEGIN
    Force ();
    cg.cvt_float (t, u);
    SPop (1, &quot;Cvt_float&quot;);
    SPush (u);
  END Cvt_float;

PROCEDURE <A NAME="Div"><procedure>Div</procedure></A> (t: IType;  a, b: Sign) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.div (t, a, b);
    SPop (2, &quot;Div&quot;);
    SPush (t);
  END Div;

PROCEDURE <A NAME="Mod"><procedure>Mod</procedure></A> (t: IType;  a, b: Sign) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.mod (t, a, b);
    SPop (2, &quot;Mod&quot;);
    SPush (t);
  END Mod;
</PRE>------------------------------------------------------------------ sets ---

<P><PRE>PROCEDURE <A NAME="Set_union"><procedure>Set_union</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.or ();
      SPop (1, &quot;Set_union&quot;);
    ELSE
      cg.set_union (AsBytes (s));
      SPop (3, &quot;Set_union&quot;);
    END;
  END Set_union;

PROCEDURE <A NAME="Set_difference"><procedure>Set_difference</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.not ();
      cg.and ();
      SPop (1, &quot;Set_diff&quot;);
    ELSE
      cg.set_difference (AsBytes (s));
      SPop (3, &quot;Set_diff&quot;);
    END;
  END Set_difference;

PROCEDURE <A NAME="Set_intersection"><procedure>Set_intersection</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.and ();
      SPop (1, &quot;Set_inter&quot;);
    ELSE
      cg.set_intersection (AsBytes (s));
      SPop (3, &quot;Set_inter&quot;);
    END;
  END Set_intersection;

PROCEDURE <A NAME="Set_sym_difference"><procedure>Set_sym_difference</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.xor ();
      SPop (1, &quot;Set_symd&quot;);
    ELSE
      cg.set_sym_difference (AsBytes (s));
      SPop (3, &quot;Set_symd&quot;);
    END;
  END Set_sym_difference;

PROCEDURE <A NAME="Set_member"><procedure>Set_member</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.load_integer (TInt.One);
      cg.swap (Type.Int, Type.Int);
      cg.shift_left ();
      cg.and ();
      cg.load_integer (TInt.Zero);
      cg.ne (Type.Word);
    ELSE
      cg.set_member (AsBytes (s));
    END;
    SPop (2, &quot;Set_member&quot;);
    SPush (Type.Int);
  END Set_member;

PROCEDURE <A NAME="Set_eq"><procedure>Set_eq</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.eq (Type.Word);
    ELSE
      cg.set_eq (AsBytes (s));
    END;
    SPop (2, &quot;Set_eq&quot;);
    SPush (Type.Int);
  END Set_eq;

PROCEDURE <A NAME="Set_ne"><procedure>Set_ne</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.ne (Type.Word);
    ELSE
      cg.set_ne (AsBytes (s));
    END;
    SPop (2, &quot;Set_ne&quot;);
    SPush (Type.Int);
  END Set_ne;

PROCEDURE <A NAME="Set_lt"><procedure>Set_lt</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      &lt;*ASSERT FALSE*&gt;
    ELSE
      cg.set_lt (AsBytes (s));
    END;
    SPop (2, &quot;Set_lt&quot;);
    SPush (Type.Int);
  END Set_lt;

PROCEDURE <A NAME="Set_le"><procedure>Set_le</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.not ();
      cg.and ();
      cg.load_integer (TInt.Zero);
      cg.eq (Type.Word);
    ELSE
      cg.set_le (AsBytes (s));
    END;
    SPop (2, &quot;Set_le&quot;);
    SPush (Type.Int);
  END Set_le;

PROCEDURE <A NAME="Set_gt"><procedure>Set_gt</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      &lt;*ASSERT FALSE*&gt;
    ELSE
      cg.set_gt (AsBytes (s));
    END;
    SPop (2, &quot;Set_gt&quot;);
    SPush (Type.Int);
  END Set_gt;

PROCEDURE <A NAME="Set_ge"><procedure>Set_ge</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.swap (Type.Word, Type.Word);
      cg.not ();
      cg.and ();
      cg.load_integer (TInt.Zero);
      cg.eq (Type.Word);
    ELSE
      cg.set_ge (AsBytes (s));
    END;
    SPop (2, &quot;Set_ge&quot;);
    SPush (Type.Int);
  END Set_ge;

PROCEDURE <A NAME="Set_range"><procedure>Set_range</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.load_integer (TInt.MOne);
      cg.swap (Type.Int, Type.Int);
      cg.shift_left ();
      cg.swap (Type.Int, Type.Int);
      cg.load_integer (TInt.MOne);
      cg.swap (Type.Int, Type.Int);
      Push_int (Target.Integer.size-1);
      cg.swap (Type.Int, Type.Int);
      cg.subtract (Type.Int);
      cg.shift_right ();
      cg.and ();
      cg.and ();
      SPop (3, &quot;Set_range-a&quot;);
      SPush (Type.Int);
    ELSE
      cg.set_range (AsBytes (s));
      SPop (3, &quot;Set_range-b&quot;);
    END;
  END Set_range;

PROCEDURE <A NAME="Set_singleton"><procedure>Set_singleton</procedure></A> (s: Size) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s &lt;= Target.Integer.size) THEN
      cg.load_integer (TInt.One);
      cg.swap (Type.Int, Type.Int);
      cg.shift_left ();
      cg.or ();
      SPop (2, &quot;Set_single-b&quot;);
      SPush (Type.Int);
    ELSE
      cg.set_singleton (AsBytes (s));
      SPop (2, &quot;Set_single-b&quot;);
    END;
  END Set_singleton;
</PRE>------------------------------------------------- Word.T bit operations ---

<P><PRE>PROCEDURE <A NAME="Not"><procedure>Not</procedure></A> () =
  BEGIN
    Force ();
    cg.not ();
    SPop (1, &quot;Not&quot;);
    SPush (Type.Int);
  END Not;

PROCEDURE <A NAME="And"><procedure>And</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.and ();
    SPop (2, &quot;And&quot;);
    SPush (Type.Int);
  END And;

PROCEDURE <A NAME="Or"><procedure>Or</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.or ();
    SPop (2, &quot;Or&quot;);
    SPush (Type.Int);
  END Or;

PROCEDURE <A NAME="Xor"><procedure>Xor</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := TRUE);
    cg.xor ();
    SPop (2, &quot;Xor&quot;);
    SPush (Type.Int);
  END Xor;

PROCEDURE <A NAME="Shift"><procedure>Shift</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.shift ();
    SPop (2, &quot;Shift&quot;);
    SPush (Type.Int);
  END Shift;

PROCEDURE <A NAME="Shift_left"><procedure>Shift_left</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.shift_left ();
    SPop (2, &quot;Shift_left&quot;);
    SPush (Type.Int);
  END Shift_left;

PROCEDURE <A NAME="Shift_right"><procedure>Shift_right</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.shift_right ();
    SPop (2, &quot;Shift_right&quot;);
    SPush (Type.Int);
  END Shift_right;

PROCEDURE <A NAME="Rotate"><procedure>Rotate</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.rotate ();
    SPop (2, &quot;Rotate&quot;);
    SPush (Type.Int);
  END Rotate;

PROCEDURE <A NAME="Rotate_left"><procedure>Rotate_left</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.rotate_left ();
    SPop (2, &quot;Rotate_left&quot;);
    SPush (Type.Int);
  END Rotate_left;

PROCEDURE <A NAME="Rotate_right"><procedure>Rotate_right</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.rotate_right ();
    SPop (2, &quot;Rotate_right&quot;);
    SPush (Type.Int);
  END Rotate_right;

PROCEDURE <A NAME="Extract"><procedure>Extract</procedure></A> (sign: BOOLEAN) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.extract (sign);
    SPop (3, &quot;Extract&quot;);
    SPush (Type.Int);
  END Extract;

PROCEDURE <A NAME="Extract_n"><procedure>Extract_n</procedure></A> (sign: BOOLEAN;  n: INTEGER) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.extract_n (sign, n);
    SPop (2, &quot;Extract_n&quot;);
    SPush (Type.Int);
  END Extract_n;

PROCEDURE <A NAME="Extract_mn"><procedure>Extract_mn</procedure></A> (sign: BOOLEAN;  m, n: INTEGER) =
  BEGIN
    Force ();
    cg.extract_mn (sign, m, n);
    SPop (1, &quot;Extract_mn&quot;);
    SPush (Type.Int);
  END Extract_mn;

PROCEDURE <A NAME="Insert"><procedure>Insert</procedure></A> () =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.insert ();
    SPop (4, &quot;Insert&quot;);
    SPush (Type.Int);
  END Insert;

PROCEDURE <A NAME="Insert_n"><procedure>Insert_n</procedure></A> (n: INTEGER) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.insert_n (n);
    SPop (3, &quot;Insert_n&quot;);
    SPush (Type.Int);
  END Insert_n;

PROCEDURE <A NAME="Insert_mn"><procedure>Insert_mn</procedure></A> (m, n: INTEGER) =
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.insert_mn (m, n);
    SPop (2, &quot;Insert_mn&quot;);
    SPush (Type.Int);
  END Insert_mn;
</PRE>------------------------------------------------ misc. stack/memory ops ---

<P><PRE>PROCEDURE <A NAME="Swap"><procedure>Swap</procedure></A> () =
  VAR tmp: ValRec;
  BEGIN
    WITH xa = stack [SCheck (2, &quot;Swap-a&quot;)],
         xb = stack [SCheck (1, &quot;Swap-b&quot;)] DO

      (* exchange the underlying values *)
      IF ((xa.kind = VKind.Stacked) OR (xa.kind = VKind.Pointer))
        AND ((xb.kind = VKind.Stacked) OR (xb.kind = VKind.Pointer)) THEN
        (* both values are on the stack =&gt; must swap *)
        cg.swap (xa.type, xb.type);
      END;

      (* exchnage the local copies *)
      tmp := xa;  xa := xb;  xb := tmp;
    END;
  END Swap;

PROCEDURE <A NAME="Discard"><procedure>Discard</procedure></A> (t: Type) =
  BEGIN
    SPop (1, &quot;Discard&quot;);
    WITH x = stack [SCheck (0, &quot;Pop&quot;)] DO
      IF (x.kind = VKind.Stacked) OR (x.kind = VKind.Pointer) THEN
        cg.pop (t);
      END;
      Release_temps (x);
    END;
  END Discard;

PROCEDURE <A NAME="Copy_n"><procedure>Copy_n</procedure></A> (s: Size;  overlap: BOOLEAN) =
  VAR t: MType;  z: Size;  a := MIN (SLV_align (2), SLV_align (3));
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (a &lt; Target.Byte) THEN ErrI (a, &quot;unaligned copy_n&quot;) END;

    (* convert the count into a multiple of a machine type's size *)
    IF (s = Target.Byte) THEN
      t := AlignedType (s, Target.Byte);
      z := TargetMap.CG_Size [t];
      &lt;*ASSERT z = Target.Byte*&gt;
    ELSIF (s &lt; Target.Byte) THEN
      IF (Target.Byte MOD s) # 0 THEN ErrI (s, &quot;impossible copy_n size&quot;) END;
      t := AlignedType (s, Target.Byte);
      z := TargetMap.CG_Size [t];
      &lt;*ASSERT z = Target.Byte*&gt;
      Push_int (Target.Byte DIV s);
      cg.div (Type.Int, Sign.Positive, Sign.Positive);
    ELSE (* s &gt; Target.Byte *)
      IF (s MOD Target.Byte) # 0 THEN ErrI (s, &quot;impossible copy_n size&quot;) END;
      t := AlignedType (s, a);
      z := TargetMap.CG_Size [t];
      IF (z &lt; s) THEN
        IF (s MOD z) # 0 THEN ErrI (s, &quot;impossible copy_n size&quot;) END;
        Push_int (s DIV z);
        cg.multiply (Type.Int);
      END;
    END;

    cg.copy_n (t, overlap);
    SPop (3, &quot;Copy_n&quot;);
  END Copy_n;

PROCEDURE <A NAME="Copy"><procedure>Copy</procedure></A> (s: Size;  overlap: BOOLEAN) =
  VAR
    a := MIN (SLV_align (2), SLV_align (1));
    t := AlignedType (s, a);
    z := TargetMap.CG_Size [t];
  BEGIN
    EVAL Force_pair (commute := FALSE);
    IF (s MOD z) # 0 THEN ErrI (s, &quot;impossible copy size&quot;) END;
    cg.copy (s DIV z, t, overlap);
    SPop (2, &quot;Copy&quot;);
  END Copy;

PROCEDURE <A NAME="Zero"><procedure>Zero</procedure></A> (s: Size) =
  VAR
    a := SLV_align (1);
    t := AlignedType (s, a);
    z := TargetMap.CG_Size [t];
  BEGIN
    Force ();
    IF (s MOD z) # 0 THEN ErrI (s, &quot;impossible zero size&quot;) END;
    cg.zero (s DIV z, t);
    SPop (1, &quot;Zero&quot;);
  END Zero;
</PRE>----------------------------------------------------------- conversions ---

<P><PRE>PROCEDURE <A NAME="Loophole"><procedure>Loophole</procedure></A> (from, two: Type) =
  BEGIN
    Force ();
    cg.loophole (from, two);
    SPop (1, &quot;Loophole&quot;);
    SPush (two);
  END Loophole;
</PRE>------------------------------------------------ traps &amp; runtime checks ---

<P><PRE>PROCEDURE <A NAME="Assert_fault"><procedure>Assert_fault</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.AssertFault);
    cg.assert_fault ();
  END Assert_fault;

PROCEDURE <A NAME="Narrow_fault"><procedure>Narrow_fault</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.NarrowFault);
    cg.narrow_fault ();
  END Narrow_fault;

PROCEDURE <A NAME="Return_fault"><procedure>Return_fault</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.ReturnFault);
    cg.return_fault ();
  END Return_fault;

PROCEDURE <A NAME="Case_fault"><procedure>Case_fault</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.CaseFault);
    cg.case_fault ();
  END Case_fault;

PROCEDURE <A NAME="Typecase_fault"><procedure>Typecase_fault</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.TypecaseFault);
    cg.typecase_fault ();
  END Typecase_fault;

PROCEDURE <A NAME="Check_nil"><procedure>Check_nil</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.NilFault);
    Force ();
    cg.check_nil ();
  END Check_nil;

PROCEDURE <A NAME="Check_lo"><procedure>Check_lo</procedure></A> (READONLY i: Target.Int) =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.RangeFault);
    Force ();
    cg.check_lo (i);
  END Check_lo;

PROCEDURE <A NAME="Check_hi"><procedure>Check_hi</procedure></A> (READONLY i: Target.Int) =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.RangeFault);
    Force ();
    cg.check_hi (i);
  END Check_hi;

PROCEDURE <A NAME="Check_range"><procedure>Check_range</procedure></A> (READONLY a, b: Target.Int) =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.RangeFault);
    Force ();
    cg.check_range (a, b);
  END Check_range;

PROCEDURE <A NAME="Check_index"><procedure>Check_index</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.SubscriptFault);
    EVAL Force_pair (commute := FALSE);
    cg.check_index ();
    SPop (1, &quot;Check_index&quot;);
  END Check_index;

PROCEDURE <A NAME="Check_eq"><procedure>Check_eq</procedure></A> () =
  BEGIN
    EVAL Runtime.LookUpProc (Runtime.Hook.ShapeFault);
    EVAL Force_pair (commute := TRUE);
    cg.check_eq ();
    SPop (2, &quot;Check_eq&quot;);
  END Check_eq;

PROCEDURE <A NAME="Check_byte_aligned"><procedure>Check_byte_aligned</procedure></A> () =
  BEGIN
    WITH x = stack [SCheck (1, &quot;Check_byte_aligned&quot;)] DO
      IF (x.align MOD Target.Byte) # 0 THEN
        Err (&quot;unaligned base variable&quot;);
      ELSIF (x.offset MOD Target.Byte) # 0 THEN
        Err (&quot;address's offset is not byte aligned&quot;);
      ELSIF (x.bits # NIL) THEN
        EVAL Runtime.LookUpProc (Runtime.Hook.ShapeFault);
        cg.load (x.bits, 0, Type.Int);
        Push_int (Target.Byte);
        cg.mod (Type.Int, Sign.Unknown, Sign.Positive);
        cg.load_integer (TInt.Zero);
        cg.check_eq ();
        Boost_alignment (Target.Byte);
      END;
    END;
  END Check_byte_aligned;
</PRE>---------------------------------------------------- address arithmetic ---

<P><PRE>PROCEDURE <A NAME="Add_offset"><procedure>Add_offset</procedure></A> (i: INTEGER) =
  BEGIN
    WITH x = stack [SCheck (1, &quot;Add_offset&quot;)] DO
      IF (x.type # Type.Addr) THEN
        Err (&quot;add_offset on non-address&quot;);
        Force ();
      ELSIF (x.kind = VKind.Stacked) THEN
        x.kind := VKind.Pointer;
        x.offset := i;
      ELSIF (x.kind = VKind.Direct) THEN
        Force ();
        x.kind   := VKind.Pointer;
        x.offset := i;
      ELSIF (x.kind = VKind.Absolute) THEN
        INC (x.offset, i);
      ELSIF (x.kind = VKind.Indirect) THEN
        INC (x.offset, i);
      ELSIF (x.kind = VKind.Pointer) THEN
        INC (x.offset, i);
      ELSE
        Err (&quot;add_offset on non-address form&quot;);
        Force ();
      END;
    END;
  END Add_offset;

PROCEDURE <A NAME="Index_bytes"><procedure>Index_bytes</procedure></A> (size: INTEGER) =
  VAR align := SLV_align (2);
  BEGIN
    EVAL Force_pair (commute := FALSE);
    cg.index_address (AsBytes (size));
    SPop (2, &quot;Index_bytes&quot;);
    SPush (Type.Addr);
    stack [SCheck (1, &quot;Index_bytes&quot;)].align := GCD (align, size);
  END Index_bytes;

PROCEDURE <A NAME="Index_bits"><procedure>Index_bits</procedure></A> () =
  VAR index := Pop_temp ();
  BEGIN
    WITH x = stack [SCheck (1, &quot;Index_address&quot;)] DO
      IF (x.bits # NIL) THEN  Err (&quot;index_bits applied twice&quot;);  END;
      IF (x.kind = VKind.Stacked) THEN x.kind := VKind.Pointer; END;
      x.bits := index.base;
      x.temp_bits := TRUE;
    END;
    (*** SPop (1, &quot;Index_address&quot;); ***)
  END Index_bits;

PROCEDURE <A NAME="Boost_alignment"><procedure>Boost_alignment</procedure></A> (a: Alignment) =
  BEGIN
    WITH x = stack [SCheck (1, &quot;Boost_alignment&quot;)] DO
      x.align := MAX (x.align, a);
    END;
  END Boost_alignment;
</PRE>------------------------------------------------------- procedure calls ---

<P><PRE>PROCEDURE <A NAME="Start_call_direct"><procedure>Start_call_direct</procedure></A> (proc: Proc;  lev: INTEGER;  t: Type) =
  BEGIN
    SEmpty (&quot;Start_call_direct&quot;);
    cg.start_call_direct (proc, lev, t);
  END Start_call_direct;

PROCEDURE <A NAME="Call_direct"><procedure>Call_direct</procedure></A> (p: Proc;  t: Type) =
  BEGIN
    SEmpty (&quot;Call_direct&quot;);
    cg.call_direct (p, t);
    PushResult (t);
  END Call_direct;

PROCEDURE <A NAME="Start_call_indirect"><procedure>Start_call_indirect</procedure></A> (t: Type;  cc: CallingConvention) =
  BEGIN
    SEmpty (&quot;Start_call_indirect&quot;);
    cg.start_call_indirect (t, cc);
  END Start_call_indirect;

PROCEDURE <A NAME="Call_indirect"><procedure>Call_indirect</procedure></A> (t: Type;  cc: CallingConvention) =
  BEGIN
    Force ();
    cg.call_indirect (t, cc);
    SPop (1, &quot;Call_indirect&quot;);
    SEmpty (&quot;Call_indirect&quot;);
    PushResult (t);
  END Call_indirect;

PROCEDURE <A NAME="PushResult"><procedure>PushResult</procedure></A> (t: Type) =
  BEGIN
    IF (t # Type.Void) THEN  SPush (t)  END;
  END PushResult;

PROCEDURE <A NAME="Pop_param"><procedure>Pop_param</procedure></A> (t: Type) =
  BEGIN
    Force ();
    cg.pop_param (t);
    SPop (1, &quot;Pop_param&quot;);
    SEmpty (&quot;Pop_param&quot;);
  END Pop_param;

PROCEDURE <A NAME="Pop_struct"><procedure>Pop_struct</procedure></A> (s: Size;  a: Alignment) =
  BEGIN
    Force ();
    cg.pop_struct (ToBytes (s), FixAlign (a));
    SPop (1, &quot;Pop_struct&quot;);
    SEmpty (&quot;Pop_struct&quot;);
  END Pop_struct;

PROCEDURE <A NAME="Pop_static_link"><procedure>Pop_static_link</procedure></A> () =
  BEGIN
    Force ();
    cg.pop_static_link ();
    SPop (1, &quot;Pop_static_link&quot;);
  END Pop_static_link;
</PRE>------------------------------------------- procedure and closure types ---

<P><PRE>PROCEDURE <A NAME="Load_procedure"><procedure>Load_procedure</procedure></A> (p: Proc) =
  BEGIN
    cg.load_procedure (p);
    SPush (Type.Addr);
  END Load_procedure;

PROCEDURE <A NAME="Load_static_link"><procedure>Load_static_link</procedure></A> (p: Proc) =
  BEGIN
    cg.load_static_link (p);
    SPush (Type.Addr);
  END Load_static_link;
</PRE>------------------------------------------------ builtin type operations --

<P><PRE>PROCEDURE <A NAME="Ref_to_typecode"><procedure>Ref_to_typecode</procedure></A> () =
  VAR base: INTEGER;
  BEGIN
    Boost_alignment (Target.Address.align);
    Load_indirect (Type.Int, -Target.Address.pack, Target.Address.size);
    Force ();
    IF Target.Little_endian THEN
      base := M3RT.RH_typecode_offset;
    ELSE
      base := Target.Integer.size
                  - M3RT.RH_typecode_offset
                  - M3RT.RH_typecode_size;
    END;
    cg.extract_mn (FALSE, base, M3RT.RH_typecode_size);
  END Ref_to_typecode;
</PRE>------------------------------------------------------------ open arrays --

<P><PRE>PROCEDURE <A NAME="Open_elt_ptr"><procedure>Open_elt_ptr</procedure></A> (a: Alignment) =
  BEGIN
    Boost_alignment (Target.Address.align);
    Load_indirect (Type.Addr, M3RT.OA_elt_ptr, Target.Address.size);
    (*** Boost_alignment (a); ***)
    WITH x = stack [SCheck (1, &quot;Open_elt_ptr&quot;)] DO
      x.align := a;
    END;
  END Open_elt_ptr;

PROCEDURE <A NAME="Open_size"><procedure>Open_size</procedure></A> (n: INTEGER) =
  BEGIN
    Boost_alignment (Target.Address.align);
    Load_indirect (Type.Int, M3RT.OA_sizes + n * Target.Integer.pack,
                   Target.Integer.size);
  END Open_size;
</PRE>------------------------------------------- procedure and closure types ---

<P><PRE>PROCEDURE <A NAME="If_closure"><procedure>If_closure</procedure></A> (proc: Val;  true, false: Label;  freq: Frequency) =
  VAR skip := Next_label ();
  BEGIN
    IF NOT Target.Aligned_procedures THEN
      Push (proc);
      Force ();
      cg.loophole (Type.Addr, Type.Int);
      Push_int (3);
      cg.and ();
      IF (false # No_label)
        THEN cg.if_true (false, Always - freq);
        ELSE cg.if_true (skip,  Always - freq);
      END;
      SPop (1, &quot;If_closure-unaligned&quot;);
    END;
    Push (proc);
    Boost_alignment (Target.Address.align);
    Force ();
    cg.load_nil ();
    IF (false # No_label)
      THEN cg.if_eq (false, Type.Addr, Always - freq);
      ELSE cg.if_eq (skip, Type.Addr, Always - freq);
    END;
    Push (proc);
    Boost_alignment (Target.Integer.align);
    Load_indirect (Type.Int, M3RT.CL_marker, Target.Integer.size);
    Push_int (M3RT.CL_marker_value);
    IF (true # No_label)
      THEN cg.if_eq (true, Type.Int, freq);
      ELSE cg.if_ne (false, Type.Int, freq);
    END;
    Set_label (skip);
    SPop (2, &quot;If_closure&quot;);
  END If_closure;

PROCEDURE <A NAME="Closure_proc"><procedure>Closure_proc</procedure></A> () =
  BEGIN
    Boost_alignment (Target.Address.align);
    Load_indirect (Type.Addr, M3RT.CL_proc, Target.Address.size);
  END Closure_proc;

PROCEDURE <A NAME="Closure_frame"><procedure>Closure_frame</procedure></A> () =
  BEGIN
    Boost_alignment (Target.Address.align);
    Load_indirect (Type.Addr, M3RT.CL_frame, Target.Address.size);
  END Closure_frame;
</PRE>----------------------------------------------------------------- misc. ---

<P><PRE>PROCEDURE <A NAME="Comment"><procedure>Comment</procedure></A> (o: INTEGER;  a, b, c, d: TEXT := NIL) =
  BEGIN
    IF (o &lt; 0) THEN
      cg.comment (a, b, c, d);
    ELSE
      PushPending (NEW (CommentNode, o := o-1, a:=a, b:=b, c:=c, d:=d));
    END;
  END Comment;

PROCEDURE <A NAME="DumpComment"><procedure>DumpComment</procedure></A> (x: CommentNode) =
  BEGIN
    DumpNode (x);
    cg.comment (x.a, x.b, x.c, x.d);
  END DumpComment;
</PRE>-------------------------------------------------------------- internal ---

<P><PRE>PROCEDURE <A NAME="FixAlign"><procedure>FixAlign</procedure></A> (a: Alignment): Alignment =
  BEGIN
    RETURN MAX (a, Target.Byte) DIV Target.Byte;
  END FixAlign;

PROCEDURE <A NAME="AlignedType"><procedure>AlignedType</procedure></A> (s: Size;  a: Alignment): MType =
  BEGIN
    IF IsAlignedMultiple (s, a, Target.Integer) THEN RETURN Type.Int;   END;
    IF IsAlignedMultiple (s, a, Target.Int_D)   THEN RETURN Type.Int_D; END;
    IF IsAlignedMultiple (s, a, Target.Int_C)   THEN RETURN Type.Int_C; END;
    IF IsAlignedMultiple (s, a, Target.Int_B)   THEN RETURN Type.Int_B; END;
    IF IsAlignedMultiple (s, a, Target.Int_A)   THEN RETURN Type.Int_A; END;
    Err (&quot;unaligned copy or zero:  s/a=&quot; &amp; Fmt.Int (s) &amp; &quot;/&quot; &amp; Fmt.Int (a));
    RETURN Type.Int_A;
  END AlignedType;

PROCEDURE <A NAME="IsAlignedMultiple"><procedure>IsAlignedMultiple</procedure></A> (s: Size;  a: Alignment;
                             READONLY t: Target.Int_type): BOOLEAN =
  BEGIN
    RETURN (s MOD t.size = 0)
       AND ((a = t.align) OR (a MOD t.align = 0));
  END IsAlignedMultiple;

PROCEDURE <A NAME="ToVarSize"><procedure>ToVarSize</procedure></A> (n: INTEGER;  a: Alignment): INTEGER =
  VAR n_bytes := (n + Target.Byte - 1) DIV Target.Byte;
      align   := FixAlign (a);
  BEGIN
    RETURN (n_bytes + align - 1) DIV align * align;
  END ToVarSize;

PROCEDURE <A NAME="ToBytes"><procedure>ToBytes</procedure></A> (n: INTEGER): INTEGER =
  BEGIN
    RETURN  (n + Target.Byte - 1) DIV Target.Byte;
  END ToBytes;

PROCEDURE <A NAME="AsBytes"><procedure>AsBytes</procedure></A> (n: INTEGER): INTEGER =
  VAR x := n DIV Target.Byte;
  BEGIN
    IF (x * Target.Byte # n) THEN ErrI (n, &quot;unaligned offset&quot;) END;
    RETURN  x;
  END AsBytes;

PROCEDURE <A NAME="Push_int"><procedure>Push_int</procedure></A> (i: INTEGER) =
  VAR val: Target.Int;  b := TInt.FromInt (i, val);
  BEGIN
    IF NOT b THEN ErrI (i, &quot;integer not representable&quot;) END;
    cg.load_integer (val);
  END Push_int;

PROCEDURE <A NAME="Force_pair"><procedure>Force_pair</procedure></A> (commute: BOOLEAN): BOOLEAN =
  (* Returns TRUE if the items are stacked in the wrong order *)
  VAR s1 := stack [SCheck (1, &quot;Force_pair&quot;)].kind = VKind.Stacked;
  VAR s2 := stack [SCheck (2, &quot;Force_pair&quot;)].kind = VKind.Stacked;
  BEGIN
    IF s1 AND s2 THEN
      (* both elements are already stacked *)
      RETURN FALSE;
    ELSIF s2 THEN
      (* bottom element is already stacked *)
      Force ();
      RETURN FALSE;
    ELSIF s1 THEN
      Swap ();
      Force ();
      IF commute THEN RETURN TRUE END;
      Swap ();
      RETURN FALSE;
    ELSE (* neither element is stacked *)
      Swap ();
      Force ();
      Swap ();
      Force ();
      RETURN FALSE;
    END;
  END Force_pair;

PROCEDURE <A NAME="SLV_align"><procedure>SLV_align</procedure></A> (n: INTEGER): INTEGER =
  BEGIN
    RETURN LV_align (stack [SCheck (n, &quot;SLV_align&quot;)]);
  END SLV_align;

PROCEDURE <A NAME="LV_align"><procedure>LV_align</procedure></A> (READONLY x: ValRec): INTEGER =
  VAR align := x.align;
  BEGIN
    IF (x.offset # 0) THEN align := GCD (align, x.offset) END;
    IF (x.bits # NIL) THEN align := 1  END;
    RETURN align;
  END LV_align;

PROCEDURE <A NAME="Base_align"><procedure>Base_align</procedure></A> (READONLY x: ValRec): INTEGER =
  (* like LV_align, but ignore the constant offset *)
  BEGIN
    RETURN x.align;
    (***********
    IF (x.bits = NIL)
      THEN RETURN x.align;
      ELSE RETURN 1;
    END;
    ************)
  END Base_align;

PROCEDURE <A NAME="GCD"><procedure>GCD</procedure></A> (a, b: INTEGER): INTEGER =
  VAR c: INTEGER;
  BEGIN
    IF (a &lt; 0) THEN a := -a END;
    IF (b &lt; 0) THEN b := -b END;
    IF (b = 0) THEN RETURN a END;
    LOOP
      c := a MOD b;
      IF (c = 0) THEN RETURN b END;
      a := b; b := c;
    END;
  END GCD;

PROCEDURE <A NAME="FindIntType"><procedure>FindIntType</procedure></A> (t: Type;  s: Size;  o: Offset;  a: Alignment): MType =
  VAR j := -1;
    best_s := TargetMap.CG_Size [t] + 1;
    best_a := TargetMap.CG_Align [t] + 1;
    size   : Size;
    align  : Alignment;
  BEGIN
    FOR i := FIRST (TargetMap.Int_types) TO LAST (TargetMap.Int_types) DO
      size  := TargetMap.Int_types[i].size;
      align := TargetMap.Int_types[i].align;
      IF (TargetMap.CG_Base [TargetMap.Int_types[i].cg_type] = t)
        AND (s &lt;= size) AND (size &lt; best_s)
        AND (align &lt;= best_a)
        AND (a MOD align = 0)
        AND (s + (o MOD align) &lt;= size) THEN
         (* remember this type *)
        j := i;
        best_s := size;
        best_a := align;
      END;
    END;
    IF (j # -1) THEN RETURN TargetMap.Int_types[j].cg_type END;
    Err (&quot;unable to find integer type?  type=&quot;&amp; Fmt.Int (ORD (t))
          &amp; &quot;  s/o/a=&quot; &amp; Fmt.Int (s) &amp; &quot;/&quot; &amp; Fmt.Int (o) &amp; &quot;/&quot; &amp; Fmt.Int (a));
    RETURN t;
  END FindIntType;

PROCEDURE <A NAME="SPush"><procedure>SPush</procedure></A> (t: Type) =
  BEGIN
    WITH x = stack[tos] DO
      x.kind      := VKind.Stacked;
      x.type      := t;
      x.temp_base := FALSE;
      x.temp_bits := FALSE;
      x.align     := Target.Byte;
      x.base      := NIL;
      x.bits      := NIL;
      x.offset    := 0;
      x.int       := TInt.Zero;
      x.float     := TFloat.ZeroR;
      x.next      := NIL;
    END;
    INC (tos);
  END SPush;

PROCEDURE <A NAME="SPop"><procedure>SPop</procedure></A> (n: INTEGER;  tag: TEXT) =
  BEGIN
    IF (tos &lt; n)
      THEN ErrI (n, &quot;SPop: stack underflow in &quot; &amp; tag);  tos := 0;
      ELSE DEC (tos, n);
    END;
  END SPop;

PROCEDURE <A NAME="SCheck"><procedure>SCheck</procedure></A> (n: INTEGER;  tag: TEXT): INTEGER =
  BEGIN
    IF (tos &lt; n)
      THEN ErrI (n, &quot;SCheck: stack underflow in &quot; &amp; tag); RETURN 0;
      ELSE RETURN tos - n;
    END;
  END SCheck;

PROCEDURE <A NAME="Err"><procedure>Err</procedure></A> (msg: TEXT) =
  BEGIN
    msg := &quot;** INTERNAL CG ERROR *** &quot; &amp; msg;
    Error.Msg (msg);
    cg.comment (msg);
  END Err;

PROCEDURE <A NAME="ErrI"><procedure>ErrI</procedure></A> (n: INTEGER;  msg: TEXT) =
  BEGIN
    msg := &quot;** INTERNAL CG ERROR *** &quot; &amp; msg;
    Error.Int (n, msg);
    cg.comment (msg, &quot;: &quot;, Fmt.Int (n));
  END ErrI;

PROCEDURE <A NAME="NewIntTbl"><procedure>NewIntTbl</procedure></A> (): IntIntTbl.T =
  BEGIN
    RETURN NEW (IntIntTbl.Default).init ();
  END NewIntTbl;

PROCEDURE <A NAME="NewNameTbl"><procedure>NewNameTbl</procedure></A> (): IntRefTbl.T =
  BEGIN
    RETURN NEW (IntRefTbl.Default).init ();
  END NewNameTbl;
</PRE>------------------------------------------------------------- debugging ---
*********
*********

<P><PRE>CONST
  Bool = ARRAY BOOLEAN OF TEXT { &quot;F &quot;, &quot;T &quot;};
CONST
  TypeName = ARRAY Type OF TEXT {
    &quot;Addr   &quot;, &quot;Word   &quot;, &quot;Int    &quot;,
    &quot;Reel   &quot;, &quot;LReel  &quot;, &quot;XReel  &quot;,
    &quot;Int_A  &quot;, &quot;Int_B  &quot;, &quot;Int_C  &quot;, &quot;Int_D  &quot;,
    &quot;Word_A &quot;, &quot;Word_B &quot;, &quot;Word_C &quot;, &quot;Word_D &quot;,
    &quot;Struct &quot;, &quot;Void   &quot;
  };
CONST
  VName = ARRAY VKind OF TEXT {
    &quot;Integer  &quot;,
    &quot;Float    &quot;,
    &quot;Stacked  &quot;,
    &quot;Direct   &quot;,
    &quot;Absolute &quot;,
    &quot;Indirect &quot;,
    &quot;Pointer  &quot;
  };

PROCEDURE <A NAME="SDump"><procedure>SDump</procedure></A> (tag: TEXT) =
  VAR msg: TEXT;
  BEGIN
    cg.comment (tag);
    cg.comment (&quot;------------ begin stack dump ------------&quot;);
    FOR i := tos-1 TO 0 BY -1 DO
      WITH x = stack[i] DO
        msg := VName [x.kind];
        msg := msg &amp; TypeName [x.type];
        msg := msg &amp; Bool [x.temp_base];
        msg := msg &amp; Bool [x.temp_bits];
        msg := msg &amp; Fmt.Int (x.align) &amp; &quot; &quot;;
        msg := msg &amp; Fmt.Int (x.offset);
        cg.comment (msg);
      END;
    END;
    cg.comment (&quot;------------- end stack dump -------------&quot;);
  END SDump;

PROCEDURE <A NAME="SEmpty"><procedure>SEmpty</procedure></A> (tag: TEXT) =
  BEGIN
    IF (tos &gt; 0) THEN
      Force ();
      ErrI (tos, &quot;stack not empty, depth&quot;);
      SDump (tag);
    END;
  END SEmpty;

BEGIN
END CG.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
