<HTML>
<HEAD>
<TITLE>SRC Modula-3: m3front/src/misc/Scanner.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>m3front/src/misc/Scanner.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE> File: Scanner.m3                                              

<P><PRE>UNSAFE MODULE <module><implements><A HREF="Scanner.i3">Scanner</A></implements></module>;

IMPORT <A HREF="../../../text/src/Text.i3">Text</A>, <A HREF="../../../os/src/Common/File.i3">File</A>, <A HREF="../../../os/src/Common/OSError.i3">OSError</A>, <A HREF="../../../libm3/derived/TextIntTbl.i3">TextIntTbl</A>;
IMPORT <A HREF="M3.i3">M3</A>, <A HREF="#x1">M3ID</A>, <A HREF="Error.i3">Error</A>, <A HREF="M3String.i3">M3String</A>, <A HREF="Token.i3">Token</A>;
IMPORT <A HREF="../../../m3middle/src/Target.i3">Target</A>, <A HREF="../../../m3middle/src/TInt.i3">TInt</A>, <A HREF="../../../m3middle/src/TWord.i3">TWord</A>, <A HREF="../../../m3middle/src/TFloat.i3">TFloat</A>, <A HREF="Host.i3">Host</A>, <A HREF="../../../m3middle/src/M3Buf.i3">M3Buf</A>;

CONST
  MaxStack  = 40;
  MaxLines  = 100000;
  MaxString = 4095;
  MaxBuffer = 4096;
  UndoPad   = 4;
  EOFChar   = '\000';
  MaxRsrvd  = 250;  (* =&gt; only the first few ids may be reserved *)

TYPE
  (* CharSet = SET OF CHAR; *)
  TK = Token.T;
  IDList = UNTRACED REF RECORD id: M3ID.T;  next: IDList END;
  CharMap = ARRAY CHAR OF BOOLEAN;

VAR (* CONST *)
  WhiteSpace    : CharMap := CharMap { FALSE, .. };
  AlphaNumerics : CharMap := CharMap { FALSE, .. };
  Digits        : CharMap := CharMap { FALSE, .. };
  OctalDigits   : CharMap := CharMap { FALSE, .. };
  HexDigits     : CharMap := CharMap { FALSE, .. };
  CommentAlert  : CharMap := CharMap { FALSE, .. };
  missing       : M3ID.T;
  LINE          : M3ID.T;
  NOWARN        : M3ID.T;
  PRAGMA        : M3ID.T;

TYPE
  InputBufferIndex = [-UndoPad .. MaxBuffer+1];
  InputBuffer = REF ARRAY InputBufferIndex OF File.Byte;
  (* Note: to avoid a range check in GetCh (and ScanComment), we don't
     use &quot;MaxBuffer-1&quot; as the upper bound of the input buffer. *)

TYPE
  StringBufferIndex =  [0..MaxString];
  StringBuffer =  ARRAY StringBufferIndex OF CHAR;

TYPE
  FileState = RECORD
    ch      : CHAR;
    offs    : INTEGER;	(* fileno * MaxLines + lineno *)
    input   : File.T;
    buf     : InputBuffer;
    buf_ptr : InputBufferIndex;
    buf_len : INTEGER;
    sym     : Symbol;
    ignore  : IDList;  (* pragmas to ignore *)
    accepted: INTEGER;
    is_main : BOOLEAN;
  END;

TYPE
  FileNames = REF ARRAY OF TEXT;

VAR (* never reset *)
  nFiles      : INTEGER;
  file_map    := NEW (TextIntTbl.Default).init ();
  files       := NEW (FileNames, 200);
  local_files := NEW (FileNames, 200);
  reserved    : ARRAY [0..MaxRsrvd] OF M3.Value;

VAR (* explicitly reset *)
  input       : File.T;
  input_buf   : InputBuffer;
  input_ptr   : InputBufferIndex;
  input_len   : INTEGER;
  ch          : CHAR;
  ignore      : IDList;
  accepted    : INTEGER := 2;
  tos         : INTEGER;
  stack       : ARRAY [0..MaxStack] OF FileState;
  buf         : StringBuffer;

PROCEDURE <A NAME="Initialize"><procedure>Initialize</procedure></A> () =
  BEGIN
    missing    := M3ID.Add (&quot;&lt;missing id&gt;&quot;);
    LINE       := M3ID.Add (&quot;LINE&quot;);
    NOWARN     := M3ID.Add (&quot;NOWARN&quot;);
    PRAGMA     := M3ID.Add (&quot;PRAGMA&quot;);

    nFiles     := 0;
    tos        := 0;

    WhiteSpace [' ']  := TRUE;
    WhiteSpace ['\n'] := TRUE;
    WhiteSpace ['\t'] := TRUE;
    WhiteSpace ['\r'] := TRUE;
    WhiteSpace ['\f'] := TRUE;

    AlphaNumerics ['_'] := TRUE;
    FOR c := 'a' TO 'z' DO AlphaNumerics [c] := TRUE END;
    FOR c := 'A' TO 'Z' DO AlphaNumerics [c] := TRUE END;
    FOR c := '0' TO '9' DO AlphaNumerics [c] := TRUE END;

    FOR c := '0' TO '9' DO Digits [c] := TRUE END;

    FOR c := '0' TO '7' DO OctalDigits [c] := TRUE END;

    FOR c := '0' TO '9' DO HexDigits [c] := TRUE END;
    FOR c := 'a' TO 'f' DO HexDigits [c] := TRUE END;
    FOR c := 'A' TO 'F' DO HexDigits [c] := TRUE END;

    CommentAlert ['*'] := TRUE;
    CommentAlert ['('] := TRUE;
    CommentAlert [EOFChar] := TRUE;
    CommentAlert ['\n'] := TRUE;
    CommentAlert ['@'] := TRUE;
  END Initialize;

PROCEDURE <A NAME="Reset"><procedure>Reset</procedure></A> () =
  BEGIN
    WHILE (tos &gt; 0) DO
      Pop ();
      Host.CloseFile (input);
    END;
    input_buf  := NIL;
    input_len  := -1; (* not 0 ==&gt; EOF *)
    input_ptr  := 0;
    ch         := ' ';
    ignore     := NIL;
    accepted   := 2;

    (* interface variables *)
    offset     := 0;
    nLines     := 0;
    nPushed    := 0;
    cur.token  := TK.tEOF;
  END Reset;

PROCEDURE <A NAME="Push"><procedure>Push</procedure></A> (name: TEXT;  file: File.T;  is_main: BOOLEAN) =
  BEGIN
    INC (nPushed);
    WITH z = stack[tos] DO
      z.ch      := ch;
      z.offs    := offset;
      z.sym     := cur;
      z.input   := input;
      z.buf     := input_buf;
      z.buf_len := input_len;
      z.buf_ptr := input_ptr;
      z.ignore  := ignore;
      z.accepted:= accepted;
      z.is_main := in_main;
    END;
    INC (tos);
    in_main   := is_main;
    offset    := FileNumber (name) * MaxLines + 1;
    ch        := ' ';
    ignore    := NIL;
    accepted  := 2;
    input     := file;
    input_ptr := 0;
    input_len := -1;  (* not 0 ==&gt; EOF *)
    input_buf := stack[tos].buf;
    IF (input_buf = NIL) THEN
      input_buf := NEW (InputBuffer);
      stack[tos].buf := input_buf;
    END;
    IF (file # NIL) THEN GetToken (); (* prime the input stream *) END;
  END Push;

PROCEDURE <A NAME="Pop"><procedure>Pop</procedure></A> () =
  BEGIN
    DEC (tos);
    WITH z = stack[tos] DO
      ch        := z.ch;
      offset    := z.offs;
      cur       := z.sym;
      input     := z.input;
      input_buf := z.buf;
      input_ptr := z.buf_ptr;
      input_len := z.buf_len;
      ignore    := z.ignore;
      accepted  := z.accepted;
      in_main   := z.is_main;
    END;
  END Pop;

PROCEDURE <A NAME="FileNumber"><procedure>FileNumber</procedure></A> (filename: TEXT): INTEGER =
   (* returns index into files of filename, adding it if it doesn't exist *)
  VAR index := offset DIV MaxLines;
  BEGIN
    (* often we'll hit the current file *)
    IF (index &lt; NUMBER (files^))
      AND (files[index] # NIL)
      AND Text.Equal (files[index], filename) THEN
      RETURN index;
    END;

    IF file_map.get (filename, index) THEN RETURN index; END;

    (* add a new one *)
    IF (nFiles &gt;= NUMBER (files^)) THEN ExpandFiles(); END;
    EVAL file_map.put (filename, nFiles);
    files[nFiles] := filename;
    local_files[nFiles] := NIL;
    INC(nFiles);
    RETURN nFiles-1;
  END FileNumber;

PROCEDURE <A NAME="ExpandFiles"><procedure>ExpandFiles</procedure></A> () =
  VAR
    n := NUMBER (files^);
    new_global := NEW (FileNames, n + n);
    new_local  := NEW (FileNames, n + n);
  BEGIN
    SUBARRAY (new_global^, 0, n) := files^;
    SUBARRAY (new_local^, 0, n)  := local_files^;
    files       := new_global;
    local_files := new_local;
  END ExpandFiles;

PROCEDURE <A NAME="Here"><procedure>Here</procedure></A> (VAR file: TEXT;  VAR line: INTEGER) =
  BEGIN
    file := files [offset DIV MaxLines];
    line := offset MOD MaxLines;
  END Here;

PROCEDURE <A NAME="LocalHere"><procedure>LocalHere</procedure></A> (VAR file: TEXT;  VAR line: INTEGER) =
  VAR fnum := offset DIV MaxLines;
  BEGIN
    IF (local_files[fnum] = NIL) THEN
      local_files[fnum] := Host.FileTail (files[fnum]);
    END;
    file := local_files [fnum];
    line := offset MOD MaxLines;
  END LocalHere;

PROCEDURE <A NAME="SameFile"><procedure>SameFile</procedure></A> (a, b: INTEGER): BOOLEAN =
  BEGIN
    RETURN (a DIV MaxLines) = (b DIV MaxLines);
  END SameFile;

PROCEDURE <A NAME="Match"><procedure>Match</procedure></A> (t: Token.T) =
  BEGIN
    IF (cur.token = t) THEN
      GetToken ();
    ELSE
      DoFail (&quot;missing \'&quot; &amp; M3ID.ToText (Token.name [t]) &amp; &quot;\'&quot;, t);
      IF (cur.token = t) THEN GetToken () END;
    END;
  END Match;

PROCEDURE <A NAME="MatchID"><procedure>MatchID</procedure></A> (): M3ID.T =
  VAR id: M3ID.T;
  BEGIN
    IF (cur.token = TK.tIDENT) THEN
      id := cur.id;
      GetToken ();
    ELSE
      DoFail (&quot;missing identifier&quot;, TK.tIDENT);
      IF (cur.token = TK.tIDENT)
        THEN  id := cur.id;  GetToken ();
        ELSE  id := missing;
      END;
    END;
    RETURN id;
  END MatchID;

PROCEDURE <A NAME="Fail"><procedure>Fail</procedure></A> (msg: TEXT) =
  BEGIN
    DoFail (msg, TK.tEOF);
  END Fail;

PROCEDURE <A NAME="DoFail"><procedure>DoFail</procedure></A> (msg: TEXT;  stop: TK) =
  VAR t: TEXT;  i: INTEGER;
  BEGIN
    IF (accepted &gt; 1) THEN
      t := &quot;syntax error: &quot; &amp; msg;
      CASE cur.token OF
      | TK.tIDENT =&gt;
          Error.ID (cur.id, t);
      | TK.tTEXTCONST =&gt;
          Error.Txt (M3String.ToText (cur.str), t);
      | TK.tREALCONST, TK.tLONGREALCONST, TK.tEXTENDEDCONST =&gt;
          Error.Txt (&quot;&lt;float&gt;&quot;, t);
      | TK.tCARDCONST, TK.tCHARCONST =&gt;
          IF TInt.ToInt (cur.int, i)
            THEN Error.Int (i, t);
            ELSE Error.Txt (&quot;&lt;integer&gt;&quot;, t);
          END;
      ELSE (* no extra info *)
          Error.Msg (t);
      END;
    END;
    (* skip forward to a major token... *)
    WHILE (cur.token # stop) AND NOT (cur.token IN Restart) DO
      GetToken ();
    END;
    accepted := 0;
    IF (cur.token = stop) THEN accepted := 1; END;
  END DoFail;

CONST
  Restart = Token.Set {
    TK.tEOF, TK.tSEMI, TK.tINLINE, TK.tEXTERNAL, TK.tASSERT,
    TK.tUNUSED, TK.tOBSOLETE, TK.tTRACE, TK.tCALLCONV,
    TK.tFATAL,  TK.tBEGIN, TK.tCASE, TK.tCONST,
    TK.tELSE, TK.tELSIF, TK.tEVAL, TK.tEXCEPT, TK.tEXCEPTION,
    TK.tEXIT, TK.tEXPORTS, TK.tFINALLY, TK.tFOR, TK.tFROM,
    TK.tGENERIC, TK.tIF, TK.tIMPORT, TK.tINTERFACE,
    TK.tLOCK, TK.tLOOP, TK.tMODULE, TK.tPROCEDURE,
    TK.tRAISE, TK.tREADONLY, TK.tREPEAT, TK.tRETURN, TK.tREVEAL,
    TK.tTHEN, TK.tTRY, TK.tTYPE, TK.tTYPECASE, TK.tUNSAFE, TK.tUNTIL,
    TK.tVALUE, TK.tVAR, TK.tWHILE, TK.tWITH };

PROCEDURE <A NAME="NoteReserved"><procedure>NoteReserved</procedure></A> (name: M3ID.T;  value: M3.Value) =
  BEGIN
    &lt;* ASSERT M3ID.GetClass (name) = 0 *&gt;
    &lt;* ASSERT reserved[name] = NIL *&gt;
    reserved [name] := value;
  END NoteReserved;

&lt;*INLINE*&gt; PROCEDURE <A NAME="GetCh"><procedure>GetCh</procedure></A> () =
  &lt;*FATAL OSError.E*&gt;
  BEGIN
    LOOP
      IF (input_ptr &lt; input_len) THEN
        ch := VAL (input_buf[input_ptr], CHAR);
        INC (input_ptr);
        RETURN;
      ELSIF (input_len = 0) THEN
        ch := EOFChar;
        RETURN;
      ELSE
        input_len := input.read (SUBARRAY (input_buf^, UndoPad, MaxBuffer));
        input_ptr := 0;
        input_buf[input_len] := ORD('@'); (* =&gt; in CommentAlert *)
        (* loop around and try again *)
      END;
    END;
  END GetCh;

PROCEDURE <A NAME="GetToken"><procedure>GetToken</procedure></A> () =
  VAR len: StringBufferIndex;
  BEGIN
    INC (accepted);
    LOOP
      (* skip white space *)
      WHILE (WhiteSpace[ch]) DO
        IF (ch = '\n') THEN INC (offset);  INC (nLines)  END;
        GetCh ();
      END;
      (* remember where this token starts *)
      cur.offset := offset;

      CASE ch OF

      | 'a'..'z', 'A'..'Z' =&gt;
          (* scan an identifier *)
          len := 0;
          WHILE (AlphaNumerics[ch]) DO
            buf [len] := ch;  INC (len);
	    GetCh ();
          END;
          cur.id    := M3ID.FromStr (buf, len);
          cur.token := TK.tIDENT;
          cur.defn  := NIL;
          VAR i := M3ID.GetClass (cur.id); BEGIN
            IF (ORD (Token.First_Keyword) &lt;= i)
              AND (i &lt;= ORD (Token.Last_Keyword)) THEN
              cur.token := VAL (i, TK);
            END;
          END;
          IF (cur.id &lt;= LAST(reserved)) THEN
            cur.defn := reserved[cur.id];
          END;
          RETURN;

      | '0'..'9' =&gt; ScanNumber ();                             RETURN;
      | '\''     =&gt; ScanChar ();                               RETURN;
      | '\&quot;'     =&gt; ScanText ();                               RETURN;
      | '+'      =&gt; cur.token := TK.tPLUS;       GetCh ();  RETURN;
      | '-'      =&gt; cur.token := TK.tMINUS;      GetCh ();  RETURN;
      | '/'      =&gt; cur.token := TK.tSLASH;      GetCh ();  RETURN;
      | '&amp;'      =&gt; cur.token := TK.tAMPERSAND;  GetCh ();  RETURN;
      | ','      =&gt; cur.token := TK.tCOMMA;      GetCh ();  RETURN;
      | ';'      =&gt; cur.token := TK.tSEMI;       GetCh ();  RETURN;
      | '['      =&gt; cur.token := TK.tLBRACKET;   GetCh ();  RETURN;
      | '{'      =&gt; cur.token := TK.tLBRACE;     GetCh ();  RETURN;
      | '^'      =&gt; cur.token := TK.tARROW;      GetCh ();  RETURN;
      | '#'      =&gt; cur.token := TK.tSHARP;      GetCh ();  RETURN;
      | ')'      =&gt; cur.token := TK.tRPAREN;     GetCh ();  RETURN;
      | ']'      =&gt; cur.token := TK.tRBRACKET;   GetCh ();  RETURN;
      | '}'      =&gt; cur.token := TK.tRBRACE;     GetCh ();  RETURN;
      | '|'      =&gt; cur.token := TK.tBAR;        GetCh ();  RETURN;
      | EOFChar  =&gt; cur.token := TK.tEOF;                   RETURN;

      | '*' =&gt; (* '*&gt;' '*' *)
	    GetCh ();
            IF (ch = '&gt;')
	      THEN  cur.token := TK.tENDPRAGMA;  GetCh ();
              ELSE  cur.token := TK.tASTERISK;
            END;
            RETURN;
      | '=' =&gt; (*  '='  '=&gt;'  *)
            GetCh ();
            IF (ch = '&gt;')
	      THEN  cur.token := TK.tIMPLIES;  GetCh ();
              ELSE  cur.token := TK.tEQUAL;
            END;
            RETURN;
      | ':' =&gt; (*  ':'  ':='  *)
            GetCh ();
            IF (ch = '=')
	      THEN  cur.token := TK.tASSIGN;  GetCh ();
              ELSE  cur.token := TK.tCOLON;
            END;
            RETURN;
      | '.' =&gt; (*  '.'  '..'  *)
            GetCh ();
            IF (ch = '.')
	      THEN  cur.token := TK.tDOTDOT;  GetCh ();
              ELSE  cur.token := TK.tDOT;
            END;
            RETURN;
      | '(' =&gt; (*  '('*'  '('  *)
            GetCh ();
            IF (ch = '*')
	      THEN  ScanComment ();
              ELSE  cur.token := TK.tLPAREN;  RETURN;
            END;
      | '&gt;' =&gt; (*  '&gt;'  '&gt;='  *)
            GetCh ();
            IF (ch = '=')
	      THEN  cur.token := TK.tGREQUAL;  GetCh ();
              ELSE  cur.token := TK.tGREATER;
            END;
            RETURN;
      | '&lt;' =&gt; (*  '&lt;'  '&lt;='  '&lt;:'  '&lt;*' *)
            GetCh ();
            IF    (ch = '=') THEN  cur.token := TK.tLSEQUAL;  GetCh ();
            ELSIF (ch = ':') THEN  cur.token := TK.tSUBTYPE;  GetCh ();
            ELSIF (ch = '*') THEN  ScanPragma ();
            ELSE                   cur.token := TK.tLESS;
            END;
            RETURN;

      ELSE
        Error.Int (ORD (ch), &quot;Illegal character&quot;);
        GetCh ();

      END; (*case*)
    END; (*loop*)
  END GetToken;

PROCEDURE <A NAME="ScanNumber"><procedure>ScanNumber</procedure></A> () =
  VAR
    base: INTEGER;
    val: Target.Int;
    pre: Target.Precision;
    len: StringBufferIndex;
  BEGIN
    (* scan the decimal digits *)
    len := 0;
    WHILE (Digits[ch]) DO
      buf[len] := ch;  INC (len);
      GetCh ();
    END;

    IF (ch = '_') THEN
      (* scan a based integer *)
      IF    NOT TInt.New (SUBARRAY (buf, 0, len), val)
         OR NOT TInt.ToInt (val, base)
         OR (base &lt; 2)
         OR (16 &lt; base) THEN
        Error.Int (base, &quot;illegal base for based literal, 10 used&quot;);
        base := 10;
      END;
      len := 0;
      LOOP
        GetCh ();
        IF NOT (HexDigits[ch]) THEN EXIT END;
        buf [len] := ch;  INC (len);
      END;
      IF (len = 0) OR NOT TWord.New (SUBARRAY (buf, 0, len), base, val) THEN
        Error.Msg (&quot;illegal based integer literal, zero used&quot;);
        val := TInt.Zero;
      END;
      cur.token := TK.tCARDCONST;
      cur.int   := val;

    ELSIF (ch = '.') THEN
      (* scan a floating point number *)
      buf[len] := '.';  INC (len);
      GetCh (); (* eat the '.' *)
      IF (ch = '.') THEN
        (* we saw  &quot;dddd..&quot; *)

	(*****  Rd.UnGetChar (input);  *****)
        DEC (input_ptr);  input_buf[input_ptr] := ORD ('.');

        IF NOT TInt.New (SUBARRAY (buf, 0, len-1), val) THEN
          Error.Msg (&quot;illegal integer literal, zero used&quot;);
          val := TInt.Zero;
        END;
        cur.token := TK.tCARDCONST;
        cur.int   := val;
        RETURN;
      END;

      (* scan the fractional digits *)
      IF NOT (Digits[ch]) THEN
        Error.Msg (&quot;missing digits in real fraction&quot;);
        buf[len] := '0';  INC (len);
      END;
      WHILE (Digits[ch]) DO  buf[len] := ch; INC (len); GetCh ()  END;

      (* check for the exponent *)
      pre := Target.Precision.Short;
      IF (ch = 'e') OR (ch = 'E') THEN
        buf[len] := 'e';  INC (len);
        cur.token := TK.tREALCONST;
        pre := Target.Precision.Short;
      ELSIF (ch = 'd') OR (ch = 'D') THEN
        buf[len] := 'e';  INC (len);
        cur.token := TK.tLONGREALCONST;
        pre := Target.Precision.Long;
      ELSIF (ch = 'x') OR (ch = 'X') THEN
        buf[len] := 'e';  INC (len);
        cur.token := TK.tEXTENDEDCONST;
        pre := Target.Precision.Extended;
      ELSE (* real constant with no exponent *)
        IF NOT TFloat.New (SUBARRAY (buf, 0, len), pre, cur.float) THEN
          Error.Msg (&quot;illegal floating-point literal&quot;);
        END;
        cur.token := TK.tREALCONST;
        RETURN;
      END;
      GetCh (); (* eat the exponent entry char *)

      (* get the exponent sign *)
      IF (ch = '+') THEN
        buf[len] := '+';  INC (len);
        GetCh ();
      ELSIF (ch = '-') THEN
        buf[len] := '-';  INC (len);
        GetCh ();
      ELSE
        buf[len] := '+';
      END;

      (* finally, get the exponent digits *)
      IF NOT (Digits[ch]) THEN
        Error.Msg (&quot;missing digits in real exponent&quot;);
        buf[len] := '0';  INC (len);
      END;
      WHILE (Digits[ch]) DO  buf[len] := ch; INC (len); GetCh ();  END;

      IF NOT TFloat.New (SUBARRAY (buf, 0, len), pre, cur.float) THEN
        Error.Msg (&quot;illegal floating-point literal&quot;);
      END;

    ELSE
      (* already scanned a decimal integer *)
      IF NOT TInt.New (SUBARRAY (buf, 0, len), val) THEN
        Error.Msg (&quot;illegal integer literal, zero used&quot;);
        val := TInt.Zero;
      END;
      cur.token := TK.tCARDCONST;
      cur.int   := val;
    END;

  END ScanNumber;

PROCEDURE <A NAME="ScanChar"><procedure>ScanChar</procedure></A> () =
  VAR val := 0;
  BEGIN
    cur.token := TK.tCHARCONST;
    cur.int   := TInt.Zero;
    GetCh ();
    IF (ch = '\'') THEN
      Error.Msg (&quot;missing character in character literal&quot;);
      GetCh ();
      RETURN;
    ELSIF (ch = '\n') OR (ch = '\r') OR (ch = '\f') THEN
      Error.Msg (&quot;end-of-line encountered in character literal&quot;);
      RETURN;
    ELSIF (ch = '\\') THEN
      GetCh ();
      IF    (ch = 'n')  THEN  val := ORD ('\n');   GetCh ();
      ELSIF (ch = 't')  THEN  val := ORD ('\t');   GetCh ();
      ELSIF (ch = 'r')  THEN  val := ORD ('\r');   GetCh ();
      ELSIF (ch = 'f')  THEN  val := ORD ('\f');   GetCh ();
      ELSIF (ch = '\\') THEN  val := ORD ('\\');   GetCh ();
      ELSIF (ch = '\'') THEN  val := ORD ('\'');   GetCh ();
      ELSIF (ch = '\&quot;') THEN  val := ORD ('\&quot;');   GetCh ();
      ELSIF (OctalDigits[ch]) THEN  val := GetOctalChar ();
      ELSE  Error.Msg (&quot;unknown escape sequence in character literal&quot;);
      END;
    ELSIF (ch = EOFChar) THEN
      Error.Msg (&quot;EOF encountered in character literal&quot;);
      RETURN ;
    ELSE (* a simple character literal *)
      val := ORD (ch);
      GetCh ();
    END;
    IF (ch # '\'')
      THEN Error.Msg (&quot;missing closing quote on character literal&quot;);
      ELSE GetCh ();
    END;
    IF NOT TInt.FromInt (val, cur.int) THEN
      Error.Msg (&quot;illegal character literal&quot;);
    END;
  END ScanChar;

PROCEDURE <A NAME="ScanText"><procedure>ScanText</procedure></A> () =
  VAR i: INTEGER;  mbuf: M3Buf.T := NIL;
  PROCEDURE Stuff (c: CHAR) =
    BEGIN
      IF (i &lt; NUMBER (buf)) THEN
        buf [i] := c;  INC (i);
      ELSIF (i = NUMBER (buf)) THEN
        mbuf := M3Buf.New ();
        M3Buf.PutSub (mbuf, buf);
        M3Buf.PutChar (mbuf, c);
        INC (i);
      ELSE
        M3Buf.PutChar (mbuf, c);
        INC (i);
      END;
    END Stuff;
  BEGIN
    i := 0;
    cur.token := TK.tTEXTCONST;
    GetCh ();
    LOOP
      IF (ch = '\&quot;') THEN
        GetCh ();
        EXIT;
      ELSIF (ch = '\n') OR (ch = '\r') OR (ch = '\f') THEN
        Error.Msg (&quot;end-of-line encountered in text literal&quot;);
        EXIT;
      ELSIF (ch = '\\') THEN
        GetCh ();
        IF    (ch = 'n') THEN  Stuff ('\n');  GetCh ();
        ELSIF (ch = 't') THEN  Stuff ('\t');  GetCh ();
        ELSIF (ch = 'r') THEN  Stuff ('\r');  GetCh ();
        ELSIF (ch = 'f') THEN  Stuff ('\f');  GetCh ();
        ELSIF (ch = '\\') THEN Stuff ('\\');  GetCh ();
        ELSIF (ch = '\'') THEN Stuff ('\'');  GetCh ();
        ELSIF (ch = '\&quot;') THEN Stuff ('\&quot;');  GetCh ();
        ELSIF (OctalDigits[ch]) THEN Stuff (VAL (GetOctalChar (), CHAR));
        ELSE  Error.Msg (&quot;unknown escape sequence in text literal&quot;);
        END;
      ELSIF (ch = EOFChar) THEN
        Error.Msg (&quot;EOF encountered in text literal&quot;);
        EXIT;
      ELSE (* a simple character *)
        Stuff (ch);
        GetCh ();
      END;
    END;

    IF (mbuf = NIL)
      THEN cur.str := M3String.FromStr (buf, i);
      ELSE cur.str := M3String.Add (M3Buf.ToText (mbuf));
    END;
  END ScanText;

PROCEDURE <A NAME="GetOctalChar"><procedure>GetOctalChar</procedure></A> (): INTEGER =
  VAR value: INTEGER;
  BEGIN
    &lt;* ASSERT OctalDigits[ch] *&gt;
    value := ORD (ch) - ORD ('0');
    GetCh ();
    IF  NOT (OctalDigits[ch]) THEN BadOctal (); RETURN value END;
    value := value * 8 + ORD (ch) - ORD ('0');
    GetCh ();
    IF  NOT (OctalDigits[ch]) THEN BadOctal (); RETURN value END;
    value := value * 8 + ORD (ch) - ORD ('0');
    GetCh ();
    RETURN value;
  END GetOctalChar;

PROCEDURE <A NAME="BadOctal"><procedure>BadOctal</procedure></A> () =
  BEGIN
    Error.Msg (&quot;octal character constant must have 3 digits&quot;);
  END BadOctal;

PROCEDURE <A NAME="ScanComment"><procedure>ScanComment</procedure></A> () =
  VAR nest, save: INTEGER; start: INTEGER;
  BEGIN
    start := cur.offset;
    GetCh ();
    nest := 1;
    WHILE (nest &gt; 0) DO
      WHILE (NOT CommentAlert[ch]) DO
        (* INLINE GetCh (); *)
        ch := VAL (input_buf[input_ptr], CHAR);
        INC (input_ptr);
      END;
      IF (ch = '*') THEN
        GetCh ();  IF (ch = ')') THEN DEC (nest); GetCh ();  END;
      ELSIF (ch = '(') THEN
        GetCh ();  IF (ch = '*') THEN INC (nest); GetCh ();  END;
      ELSIF (ch = EOFChar) THEN
        save := offset;
	offset := start;
        Error.Msg (&quot;EOF encountered in comment&quot;);
	offset := save;
        nest := 0;
      ELSIF (ch = '\n') THEN
        INC (offset);  INC (nLines);
        GetCh ();
      ELSE
        GetCh ();
      END;
    END;
  END ScanComment;

PROCEDURE <A NAME="ScanPragma"><procedure>ScanPragma</procedure></A> () =
  VAR nest, save, start, i, lineno, fileno: INTEGER;  ss: IDList;
  BEGIN
    start := cur.offset;
    GetCh();  (* '*' *)

    (* skip white space *)
    WHILE (WhiteSpace[ch]) DO
      IF (ch = '\n') THEN INC (offset);  INC (nLines);  END;
      GetCh();
    END;

    (* scan an identifier *)
    i := 0;
    WHILE (AlphaNumerics[ch]) DO
      buf [i] := ch;  INC (i);
      GetCh ();
    END;
    cur.id    := M3ID.FromStr (buf, i);
    cur.token := VAL (M3ID.GetClass (cur.id), TK);

    IF (Token.First_Pragma&lt;=cur.token) AND (cur.token&lt;=Token.Last_Pragma) THEN
      RETURN;
    END;

    IF (cur.id = LINE) THEN
      GetToken (); (* LINE *)
      IF (cur.token # TK.tCARDCONST) THEN
        Error.Msg (&quot;missing line number on LINE pragma; skipping to \'*&gt;\'&quot;);
        WHILE (cur.token # TK.tENDPRAGMA) AND (cur.token # TK.tEOF) DO
          GetToken ();
        END;
        IF (cur.token = TK.tENDPRAGMA) THEN GetToken () END;
        RETURN;
      END;
      IF NOT TInt.ToInt (cur.int, lineno) THEN
        Error.Msg (&quot;illegal line number, ignored&quot;);
        lineno := offset MOD MaxLines;
      END;
      fileno := offset DIV MaxLines;
      GetToken (); (* CARD &quot;line number&quot; *)
      IF (cur.token = TK.tTEXTCONST) THEN
        fileno := FileNumber (M3String.ToText (cur.str));
        GetToken(); (* TEXT &quot;filename&quot; *)
      END;
      offset := fileno * MaxLines + lineno - 1;
      IF (cur.token # TK.tENDPRAGMA)
        THEN Error.Msg (&quot;missing \'*&gt;\' on LINE pragma&quot;);
        ELSE GetToken (); (* fetch the next one *)
      END;
      RETURN;
    ELSIF (cur.id = NOWARN) THEN
      Error.IgnoreWarning (cur.offset);
      GetToken ();  (* NOWARN *)
      IF (cur.token # TK.tENDPRAGMA)
        THEN Error.Msg (&quot;missing \'*&gt;\' on NOWARN pragma&quot;);
        ELSE GetToken (); (* fetch the next one *)
      END;
      RETURN;
    ELSIF (cur.id = PRAGMA) THEN
      GetToken (); (* PRAGMA *)
      WHILE (cur.token = TK.tIDENT)
      OR ((Token.First_Pragma&lt;=cur.token) AND (cur.token&lt;=Token.Last_Pragma))
      OR ((Token.First_Keyword&lt;=cur.token) AND (cur.token&lt;=Token.Last_Keyword))
      DO
        ignore := NEW (IDList, id := cur.id, next := ignore);
        GetToken ();  (* IDENT *)
        IF (cur.token # TK.tCOMMA) THEN EXIT END;
        GetToken ();  (* COMMA *)
      END;
      IF (cur.token # TK.tENDPRAGMA)
        THEN Error.Msg (&quot;missing \'*&gt;\' on PRAGMA pragma&quot;);
        ELSE GetToken ();  (* fetch the next real token *)
      END;
      RETURN;
    ELSIF Target.FindConvention (M3ID.ToText (cur.id)) # NIL THEN
      cur.token := TK.tCALLCONV;
      RETURN;
    ELSE (* scan and ignore the list *)
      ss := ignore;
      WHILE (ss # NIL) AND (ss.id # cur.id) DO  ss := ss.next  END;
      IF (ss = NIL) THEN
        Error.WarnID (2, cur.id, &quot;unrecognized pragma (ignored)&quot;);
      END;
    END;

    (* scan over and ignore the offending pragma *)
    nest := 1;
    WHILE (nest &gt; 0) DO
      IF (ch = '*') THEN
        GetCh();  IF (ch = '&gt;') THEN DEC(nest); GetCh(); END;
      ELSIF (ch = '&lt;') THEN
        GetCh();  IF (ch = '*') THEN INC(nest); GetCh(); END;
      ELSIF (ch = EOFChar) THEN
        save := offset;
	offset := start;
        Error.Msg (&quot;EOF encountered in pragma&quot;);
	offset := save;
	nest := 0;
      ELSIF (ch = '\n') THEN
        INC (offset);  INC (nLines);
	GetCh();
      ELSE
        GetCh();
      END;
    END;

    GetToken (); (* get the next token *)
  END ScanPragma;

BEGIN
END Scanner.
</PRE>
</inModule>
<HR>
<A NAME="x1">interface M3ID is in:
</A><UL>
<LI><A HREF="../../../m3middle/src/M3ID.i3#0TOP0">m3middle/src/M3ID.i3</A>
<LI><A HREF="../../../m3tools/src/M3ID.i3#0TOP0">m3tools/src/M3ID.i3</A>
</UL>
<P>
<PRE>























</PRE>
</BODY>
</HTML>
