<HTML>
<HEAD>
<TITLE>SRC Modula-3: m3tools/src/M3Scanner.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>m3tools/src/M3Scanner.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                               </EM></BLOCKQUOTE><PRE>
</PRE> File: Scanner.m3                                              

<P><PRE>MODULE <module><implements><A HREF="M3Scanner.i3">M3Scanner</A></implements></module>;

IMPORT <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../fmtlex/src/Fmt.i3">Fmt</A>;
IMPORT <A HREF="M3Token.i3">M3Token</A>, <A HREF="#x1">M3ID</A>;

CONST
  EOFChar   = '\000';

TYPE
  TK = M3Token.T;

VAR (* CONST after intialization *)
  WhiteSpace    := ARRAY CHAR OF BOOLEAN { FALSE, .. };
  AlphaNumerics := ARRAY CHAR OF BOOLEAN { FALSE, .. };
  Digits        := ARRAY CHAR OF BOOLEAN { FALSE, .. };
  OctalDigits   := ARRAY CHAR OF BOOLEAN { FALSE, .. };
  HexDigits     := ARRAY CHAR OF BOOLEAN { FALSE, .. };
  CommentAlert  := ARRAY CHAR OF BOOLEAN { FALSE, .. };
  CharAlert     := ARRAY CHAR OF BOOLEAN { FALSE, .. };

TYPE
  Public = T OBJECT METHODS
    initFromRd  (source        : Rd.T;
                 skip_comments := TRUE;
                 split_pragmas := TRUE): T;
    initFromBuf (buf           : Buf;
                 skip_comments := TRUE;
                 split_pragmas := TRUE): T;
  END;

REVEAL
  <A NAME="Default">Default</A> = Public BRANDED &quot;M3Scanner.Default&quot; OBJECT
      skip_comments   : BOOLEAN  := FALSE;
      split_pragmas   : BOOLEAN  := FALSE;
      line_start      : CARDINAL := 0;
      next_offset     : CARDINAL := 0;
      next_line       : CARDINAL := 0;
      next_line_start : CARDINAL := 0;
    OVERRIDES
      next        := GetToken;
      initFromRd  := InitFromRd;
      initFromBuf := InitFromBuf;
    END;

PROCEDURE <A NAME="InitFromRd"><procedure>InitFromRd</procedure></A> (t: Default;
                      source        : Rd.T;
                      skip_comments : BOOLEAN;
                      split_pragmas : BOOLEAN): T =
  VAR len0, len1: INTEGER;
  BEGIN
    t.skip_comments := skip_comments;
    t.split_pragmas := split_pragmas;
    t.line          := 1;
    t.next_line     := 1;

    (* read the source *)
    TRY
      len0 := Rd.Length (source);
      t.buffer := NEW (Buf, MAX (0, len0) + 1);
      IF (len0 &lt; 0) THEN Err (t, 0, &quot;can't tell how much source there is&quot;) END;
      len1 := Rd.GetSub (source, t.buffer^);
      IF (len0 # len1) THEN Err (t, 0, &quot;couldn't read the entire source&quot;) END;
      t.buffer [len0] := EOFChar;
    EXCEPT
    | Rd.Failure     =&gt; Err (t, 0, &quot;I/O failure reading the source&quot;);
    | Thread.Alerted =&gt; Err (t, 0, &quot;*alerted*&quot;);
    END;
    RETURN t;
  END InitFromRd;

PROCEDURE <A NAME="InitFromBuf"><procedure>InitFromBuf</procedure></A> (t: Default;
                       buf           : Buf;
                       skip_comments : BOOLEAN;
                       split_pragmas : BOOLEAN): T =
  BEGIN
    t.skip_comments := skip_comments;
    t.split_pragmas := split_pragmas;
    t.line          := 1;
    t.next_line     := 1;
    t.buffer        := buf;
    RETURN t;
  END InitFromBuf;

PROCEDURE <A NAME="Err"><procedure>Err</procedure></A> (t: Default;  offset: CARDINAL;  msg: TEXT) =
  BEGIN
    t.token       := TK.Error;
    t.msg         := msg;
    t.length      := offset - t.offset;
    t.next_offset := offset;
  END Err;

PROCEDURE <A NAME="GetToken"><procedure>GetToken</procedure></A> (t: Default) =
  VAR
    buf    := t.buffer;
    offset := t.next_offset;
    ch     := buf [offset];
  BEGIN
    t.line       := t.next_line;
    t.line_start := t.next_line_start;

    LOOP
      (* skip white space *)
      WHILE (WhiteSpace[ch]) DO
        INC (offset);
        IF (ch = '\n') THEN
          INC (t.line);
          INC (t.next_line);
          t.line_start := offset;
          t.next_line_start := offset;
        END;
        ch := buf [offset];
      END;

      (* remember where this token starts *)
      t.offset := offset;
      t.start  := offset;
      t.column := offset - t.line_start;

      CASE ch OF

      | 'a'..'z', 'A'..'Z' =&gt;
          (* scan an identifier *)
          WHILE AlphaNumerics[ch] DO INC (offset); ch := buf[offset]; END;
          t.token := M3ID.Classify (SUBARRAY (buf^, t.offset, offset - t.offset));
          EXIT;

      | '0'..'9' =&gt; ScanNumber (t);                          RETURN;
      | '\''     =&gt; ScanChar (t);                            RETURN;
      | '\&quot;'     =&gt; ScanText (t);                            RETURN;
      | '+'      =&gt; t.token := TK.Plus;       INC (offset);  EXIT;
      | '-'      =&gt; t.token := TK.Minus;      INC (offset);  EXIT;
      | '/'      =&gt; t.token := TK.Slash;      INC (offset);  EXIT;
      | '&amp;'      =&gt; t.token := TK.Ampersand;  INC (offset);  EXIT;
      | ','      =&gt; t.token := TK.Comma;      INC (offset);  EXIT;
      | ';'      =&gt; t.token := TK.Semi;       INC (offset);  EXIT;
      | '['      =&gt; t.token := TK.L_bracket;  INC (offset);  EXIT;
      | '{'      =&gt; t.token := TK.L_brace;    INC (offset);  EXIT;
      | '^'      =&gt; t.token := TK.Arrow;      INC (offset);  EXIT;
      | '#'      =&gt; t.token := TK.Sharp;      INC (offset);  EXIT;
      | ')'      =&gt; t.token := TK.R_paren;    INC (offset);  EXIT;
      | ']'      =&gt; t.token := TK.R_bracket;  INC (offset);  EXIT;
      | '}'      =&gt; t.token := TK.R_brace;    INC (offset);  EXIT;
      | '|'      =&gt; t.token := TK.Bar;        INC (offset);  EXIT;
      | EOFChar  =&gt; t.token := TK.EOF;                       EXIT;

      | '*' =&gt; (* '*&gt;' '*' *)
	    INC (offset);  ch := buf [offset];
            IF (ch = '&gt;')
	      THEN  t.token := TK.End_pragma;  INC (offset);
              ELSE  t.token := TK.Asterisk;
            END;
            EXIT;

      | '=' =&gt; (*  '='  '=&gt;'  *)
            INC (offset);  ch := buf [offset];
            IF (ch = '&gt;')
	      THEN  t.token := TK.Implies;  INC (offset);
              ELSE  t.token := TK.Equal;
            END;
            EXIT;

      | ':' =&gt; (*  ':'  ':='  *)
            INC (offset);  ch := buf [offset];

            IF (ch = '=')
	      THEN  t.token := TK.Assign;  INC (offset);
              ELSE  t.token := TK.Colon;
            END;
            EXIT;

      | '.' =&gt; (*  '.'  '..'  *)
            INC (offset);  ch := buf [offset];
            IF (ch = '.')
	      THEN  t.token := TK.Dot_dot;  INC (offset);
              ELSE  t.token := TK.Dot;
            END;
            EXIT;

      | '(' =&gt; (*  '('*'  '('  *)
            INC (offset);  ch := buf [offset];
            IF    (ch # '*')      THEN  t.token := TK.L_paren; EXIT;
            ELSIF ScanComment (t) THEN  RETURN;
            ELSE  offset := t.offset;  ch := buf [offset];
            END;

      | '&gt;' =&gt; (*  '&gt;'  '&gt;='  *)
            INC (offset);  ch := buf [offset];
            IF (ch = '=')
	      THEN  t.token := TK.Gr_equal;  INC (offset);
              ELSE  t.token := TK.Greater;
            END;
            EXIT;

      | '&lt;' =&gt; (*  '&lt;'  '&lt;='  '&lt;:'  '&lt;*' *)
            INC (offset);  ch := buf [offset];
            IF    (ch = '=') THEN  t.token := TK.Ls_equal; INC (offset);
            ELSIF (ch = ':') THEN  t.token := TK.Subtype;  INC (offset);
            ELSIF (ch # '*') THEN  t.token := TK.Less;
            ELSIF (NOT t.split_pragmas) THEN  ScanPragma (t);  RETURN;
            ELSE  t.token := TK.Begin_pragma;  INC (offset);
            END;
            EXIT;

      ELSE
        INC (offset);
        Err (t, offset, &quot;Illegal character: &quot; &amp; Fmt.Int (ORD (ch)));
        EXIT;

      END; (*case*)
    END; (*loop*)

    (* record the length of the token and where we are *)
    t.length := offset - t.offset;
    t.next_offset := offset;
  END GetToken;

PROCEDURE <A NAME="ScanNumber"><procedure>ScanNumber</procedure></A> (t: Default) =
  VAR
    offset := t.offset;
    ch     := t.buffer [offset];
  BEGIN
    (* scan the decimal digits *)
    WHILE Digits[ch] DO  INC (offset);  ch := t.buffer [offset];  END;

    IF (ch = '_') THEN
      (* scan a based integer *)
      INC (offset);  ch := t.buffer [offset];
      IF HexDigits[ch] THEN
        WHILE HexDigits[ch] DO  INC (offset);  ch := t.buffer [offset];  END;
        t.token := TK.Card_const;
      ELSE
        Err (t, offset, &quot;illegal based integer: missing digits&quot;);
        RETURN;
      END;

    ELSIF (ch = '.') AND (t.buffer [offset+1] = '.') THEN
      (* we saw  &quot;dddd..&quot; *)
      t.token := TK.Card_const;

    ELSIF (ch = '.') THEN
      (* scan a floating point number *)
      INC (offset);  ch := t.buffer [offset];

      (* scan the fractional digits *)
      IF Digits[ch] THEN
        WHILE Digits[ch] DO  INC (offset);  ch := t.buffer [offset];  END;
      ELSE
        Err (t, offset, &quot;missing digits in real fraction&quot;);
        RETURN;
      END;

      (* check for the exponent *)
      IF (ch = 'e') OR (ch = 'E') THEN
        t.token := TK.Real_const;
      ELSIF (ch = 'd') OR (ch = 'D') THEN
        t.token := TK.Longreal_const;
      ELSIF (ch = 'x') OR (ch = 'X') THEN
        t.token := TK.Extended_const;
      ELSE (* real constant with no exponent *)
        t.token := TK.Real_const;
        t.next_offset := offset;
        t.length := offset - t.offset;
        RETURN;
      END;
      INC (offset);  ch := t.buffer [offset]; (* eat the e/d/x *)

      (* get the exponent sign *)
      IF (ch = '+') OR (ch = '-') THEN
        INC (offset);  ch := t.buffer [offset];
      END;

      (* finally, get the exponent digits *)
      IF Digits[ch] THEN
        WHILE Digits[ch] DO  INC (offset);  ch := t.buffer [offset];  END;
      ELSE
        Err (t, offset, &quot;missing digits in real exponent&quot;);
      END;

    ELSE
      (* scanned a simple decimal integer *)
      t.token := TK.Card_const;

    END;

    t.length := offset - t.offset;
    t.next_offset := offset;
  END ScanNumber;

PROCEDURE <A NAME="ScanChar"><procedure>ScanChar</procedure></A> (t: Default) =
  VAR
    offset : CARDINAL := t.offset+1;
    ch     : CHAR     := t.buffer [offset];
  BEGIN
    IF NOT CharAlert [ch] THEN
      (* a simple character literal *)
      INC (offset);  ch := t.buffer[offset];
    ELSIF (ch = '\'') THEN
      INC (offset);
      Err (t, offset, &quot;missing character in character literal&quot;);
      RETURN;
    ELSIF (ch = '\n') OR (ch = '\r') OR (ch = '\f') THEN
      Err (t, offset, &quot;end-of-line encountered in character literal&quot;);
      RETURN;
    ELSIF (ch = '\\') THEN
      INC (offset);
      IF NOT ScanEscape (t, offset) THEN
        Err (t, offset, &quot;unknown escape sequence in character literal&quot;);
        RETURN;
      END;
      ch := t.buffer [offset];
    ELSIF (ch = EOFChar) THEN
      Err (t, offset, &quot;EOF encountered in character literal&quot;);
      RETURN;
    ELSE
      (* a simple character literal *)
      INC (offset);  ch := t.buffer[offset];
    END;

    IF (ch # '\'') THEN
      Err (t, offset, &quot;missing closing quote on character literal&quot;);
      RETURN;
    END;
    INC (offset); (* eat the closing quote *)

    t.token := TK.Char_const;
    t.length := offset - t.offset;
    t.next_offset := offset;
  END ScanChar;

PROCEDURE <A NAME="ScanText"><procedure>ScanText</procedure></A> (t: Default) =
  VAR
    offset : CARDINAL := t.offset+1;
    ch     : CHAR     := t.buffer [offset];
  BEGIN
    LOOP
      IF NOT CharAlert [ch] THEN
        INC (offset);  ch := t.buffer [offset];
      ELSIF (ch = '\&quot;') THEN
        INC (offset);
        EXIT;
      ELSIF (ch = '\n') OR (ch = '\r') OR (ch = '\f') THEN
        Err (t, offset, &quot;end-of-line encountered in text literal&quot;);
        RETURN;
      ELSIF (ch = '\\') THEN
        INC (offset);
        IF NOT ScanEscape (t, offset) THEN
          Err (t, offset, &quot;unknown escape sequence in text literal&quot;);
          RETURN;
        END;
        ch := t.buffer [offset];
      ELSIF (ch = EOFChar) THEN
        Err (t, offset, &quot;EOF encountered in text literal&quot;);
        RETURN;
      ELSE (* a simple character *)
        INC (offset);  ch := t.buffer [offset];
      END;
    END;

    t.token := TK.Text_const;
    t.length := offset - t.offset;
    t.next_offset := offset;
  END ScanText;

PROCEDURE <A NAME="ScanEscape"><procedure>ScanEscape</procedure></A> (t: Default;  VAR offset: CARDINAL): BOOLEAN =
  VAR ch := t.buffer [offset];
  BEGIN
    INC (offset);
    IF (ch = 'n') OR (ch = 't') OR (ch = 'r') OR (ch = 'f')
      OR (ch = '\\') OR (ch = '\'') OR (ch = '\&quot;') THEN
      RETURN TRUE;
    END;
    IF NOT OctalDigits [ch] THEN RETURN FALSE END;
    ch := t.buffer [offset];  INC (offset);
    IF NOT OctalDigits [ch] THEN RETURN FALSE END;
    ch := t.buffer [offset];  INC (offset);
    RETURN OctalDigits [ch];
  END ScanEscape;

PROCEDURE <A NAME="ScanComment"><procedure>ScanComment</procedure></A> (t: Default): BOOLEAN =
  VAR
    nest    := 1;
    start   := t.offset;
    offset  := start + 2; (* skip the opening '(' and '*' *)
    ch      := t.buffer [offset];
    my_line := t.line;
    l_start := t.line_start;
  BEGIN
    WHILE (nest &gt; 0) DO
      (************** UNSAFE VERSION **********************
      VAR p: UNTRACED REF CHAR;
      p := ADR (t.buffer[offset]);
      WHILE NOT CommentAlert[ch] DO INC (p);  ch := p^; END;
      offset := p - ADR (t.buffer[0]);
      ******************************************************)
      WHILE NOT CommentAlert[ch] DO INC (offset);  ch := t.buffer[offset]; END;
      IF (ch = '*') THEN
        INC (offset);  ch := t.buffer [offset];
        IF (ch = ')') THEN
          INC (offset);  ch := t.buffer[offset];
          DEC (nest);
        END;
      ELSIF (ch = '(') THEN
        INC (offset);  ch := t.buffer [offset];
        IF (ch = '*') THEN
          INC (offset);  ch := t.buffer[offset];
          INC (nest);
        END;
      ELSIF (ch = EOFChar) THEN
        t.next_line         := my_line;
        t.next_line_start   := l_start;
        Err (t, offset, &quot;EOF encountered in comment&quot;);
        RETURN TRUE;
      ELSIF (ch = '\n') THEN
        INC (offset);  ch := t.buffer [offset];
        INC (my_line);  l_start := offset;
      ELSE
        INC (offset);  ch := t.buffer [offset];
      END;
    END;

    IF t.skip_comments THEN
      t.offset     := offset;
      t.line       := my_line;
      t.line_start := l_start;
      RETURN FALSE;
    ELSE
      t.token           := TK.Comment;
      t.length          := offset - t.offset;
      t.next_offset     := offset;
      t.next_line       := my_line;
      t.next_line_start := l_start;
      RETURN TRUE;
    END;
  END ScanComment;

PROCEDURE <A NAME="ScanPragma"><procedure>ScanPragma</procedure></A> (t: Default) =
  VAR
    start   := t.offset;
    offset  := start + 2; (* skip the opening '&lt;' and '*' *)
    ch      := t.buffer [offset];
    my_line := t.line;
    l_start := t.line_start;
  BEGIN
    (* scan the entire pragma as a string *)
    WHILE (ch # '*') OR (t.buffer[offset+1] # '&gt;') DO
      IF (ch = EOFChar) THEN
        Err (t, offset, &quot;EOF encountered in pragma&quot;);
        RETURN;
      END;
      INC (offset);
      IF (ch = '\n') THEN  INC (my_line);  l_start := offset;  END;
      ch := t.buffer [offset];
    END;
    INC (offset, 2);

    t.token           := TK.Begin_pragma;
    t.length          := offset - t.offset;
    t.next_offset     := offset;
    t.next_line       := my_line;
    t.next_line_start := l_start;
  END ScanPragma;

BEGIN
  WhiteSpace [' ']  := TRUE;
  WhiteSpace ['\n'] := TRUE;
  WhiteSpace ['\t'] := TRUE;
  WhiteSpace ['\r'] := TRUE;
  WhiteSpace ['\f'] := TRUE;

  AlphaNumerics ['_'] := TRUE;
  FOR c := 'a' TO 'z' DO AlphaNumerics [c] := TRUE END;
  FOR c := 'A' TO 'Z' DO AlphaNumerics [c] := TRUE END;
  FOR c := '0' TO '9' DO AlphaNumerics [c] := TRUE END;

  FOR c := '0' TO '9' DO Digits [c] := TRUE END;
  FOR c := '0' TO '7' DO OctalDigits [c] := TRUE END;
  FOR c := '0' TO '9' DO HexDigits [c] := TRUE END;
  FOR c := 'a' TO 'f' DO HexDigits [c] := TRUE END;
  FOR c := 'A' TO 'F' DO HexDigits [c] := TRUE END;

  CommentAlert ['*']     := TRUE;
  CommentAlert ['(']     := TRUE;
  CommentAlert [EOFChar] := TRUE;
  CommentAlert ['\n']    := TRUE;

  CharAlert ['\''] := TRUE;
  CharAlert ['\&quot;'] := TRUE;
  CharAlert ['\n'] := TRUE;
  CharAlert ['\r'] := TRUE;
  CharAlert ['\f'] := TRUE;
  CharAlert ['\\'] := TRUE;
  CharAlert [EOFChar] := TRUE;
END M3Scanner.
</PRE>
</inModule>
<HR>
<A NAME="x1">interface M3ID is in:
</A><UL>
<LI><A HREF="../../m3middle/src/M3ID.i3#0TOP0">m3middle/src/M3ID.i3</A>
<LI><A HREF="M3ID.i3#0TOP0">m3tools/src/M3ID.i3</A>
</UL>
<P>
<PRE>























</PRE>
</BODY>
</HTML>
