<HTML>
<HEAD>
<TITLE>SRC Modula-3: pickle/src/Pickle.i3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>pickle/src/Pickle.i3</H2></A><HR>
<inInterface>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE> A {\it pickle} is a representation of a Modula-3 value as 
   a stream of bytes.  Writing a value as a pickle and 
   then reading it back produces a value ``equivalent'' to the 
   original value, in the sense that the value's non-ref 
   fields have the same contents and that their <CODE>REF</CODE> and 
   <CODE>OBJECT</CODE> fields are <CODE>=</CODE> in the same places.  In other words, 
   pickles preserve value, shape and sharing.  You can write 
   pickles for values that have cyclic references (such as 
   doubly-linked lists), or that are arbitrary graph structures. 
<P>
   Pickles are used both to communicate values between two
   programs running at the same time and to store a value and 
   reuse it in a program running later.
<P>
   Pickles preserve ordinal values (integers and enumerations)
   exactly,  automatically performing any byte-swapping 
   that may be required. An exception is raised if a pickled 
   ordinal value is too large to be represented in a program 
   reading the pickle.  
<P>
   Pickles preserve floating point values as well as possible
   considering that the two machines may have different
   floating point representations.
<P>
   Pickles do not preserve untraced references.  Any untraced
   references will turn into <CODE>NIL</CODE> if they are pickled,
   regardless of their value.
<P>
   Pickles preserve the names and signatures of procedures,
   but the code for the procedures can be different in
   the program writing the pickle and the program
   reading the pickle.  Object methods are treated
   the same as procedures.
<P>
   Sharing is preserved within pickles, but not between pickles.
<P>
   Pickles are written and read by a single-pass 
   algorithm that uses modest auxiliary tables and does not
   require a seekable stream. 
   <P>
   For example, if you compile and run the following program:
<P>
<PRE>
      MODULE P1 EXPORTS Main;  IMPORT Pickle, IO;
        TYPE T = REF RECORD val: INTEGER END;
      BEGIN
        Pickle.Write(NEW(T, val := 6), IO.OpenWrite(&quot;A.pickle&quot;))
      END P1.
</PRE>
   and then in the same directory, you compile and run the
   following program:
<P>
<PRE>
      MODULE P2 EXPORTS Main; IMPORT Pickle, IO;
        TYPE U = REF RECORD val: INTEGER END;
        VAR v: U := Pickle.Read(IO.OpenRead(&quot;A.pickle&quot;));
      BEGIN
        IO.Put(Fmt.Int(v.val) &amp; &quot;\n&quot;)
      END P2.
</PRE>
   then <CODE>P2</CODE> will read the pickle written by <CODE>P1</CODE> and will
   print <CODE>6</CODE>.  Notice that the type of the pickled reference
   was named differently in <CODE>P1</CODE> and <CODE>P2</CODE>; but this doesn't
   present any problem, since Modula-3
   uses structural equivalence.
<P>
   If you pickle a <CODE>BRANDED REF INTEGER</CODE>, there is no
   guarantee that you will be able to read it back as
   a <CODE>BRANDED REF INTEGER</CODE>, since the implicit brand can
   vary from program to program.  Therefore, you should
   use explicit brands in types that will be pickled.


<P><PRE>INTERFACE <interface><A HREF="Pickle.m3">Pickle</A></interface>;

IMPORT <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../rw/src/Common/Wr.i3">Wr</A>;

EXCEPTION Error(TEXT);

PROCEDURE <A HREF="Pickle.m3#Write">Write</A>(wr: Wr.T; ref: REFANY)
    RAISES {Error, Wr.Failure, Thread.Alerted};
</PRE><BLOCKQUOTE><EM> Trace the data structure reachable via traced references 
   from <CODE>ref</CODE>, convert it into a pickle, and write the pickle to 
   <CODE>wr</CODE>, starting at the current position of <CODE>wr</CODE> and leaving 
   <CODE>wr</CODE> positioned after the last byte written. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="Pickle.m3#Read">Read</A>(rd: Rd.T): REFANY
    RAISES {Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted};
</PRE><BLOCKQUOTE><EM> Read a pickle from <CODE>rd</CODE>, reconstruct a copy of the pickled
   data structure, and return it.  The bytes to be read start at
   <CODE>rd</CODE>'s current position; <CODE>rd</CODE> is left positioned after the
   last byte consumed. </EM></BLOCKQUOTE><PRE>
</PRE> Most people should go no further.  The remainder of the 
   interface allows you to customize the pickling and unpickling
   of selected types.  There are two ways of doing this: you
   can override the methods of objects called {\it pickle writers}
   and {\it pickle readers}, or you can register {\it specials}
   for particular types.  Several examples are presented
   after the end of the interface.
<P>
   \paragraph{Pickle writers and readers.}
   A <CODE>Pickle.Writer</CODE> is an object whose <CODE>write</CODE> method writes
   pickles to the writer stored in its <CODE>wr</CODE> data field.  
   You can override the <CODE>write</CODE> method to produce special
   effects. 
   
<P><PRE>TYPE
  TypeCode = INTEGER;
</PRE><BLOCKQUOTE><EM> Used to represent types of allocated objects. A <CODE>TypeCode</CODE>
   value is valid within a single execution of a single program.
   There are methods in <CODE>Writer</CODE> and <CODE>Reader</CODE> objects for writing
   and reading program-independent representations of a TypeCode.
   </EM></BLOCKQUOTE><PRE>

TYPE
  <A HREF="Pickle.m3#Writer">Writer</A> &lt;: WriterPublic;
  WriterPublic = OBJECT
      wr: Wr.T;
    METHODS
      write(r: REFANY) RAISES
        {Error, Wr.Failure, Thread.Alerted};
      writeType(tc: TypeCode) RAISES
        {Wr.Failure, Thread.Alerted};
      writeInt(i: INTEGER) RAISES
        {Wr.Failure, Thread.Alerted};
    END;
</PRE> If <CODE>w</CODE> is a <CODE>Pickle.Writer</CODE>, then <CODE>w.write(r)</CODE>
   traces the data structure reachable via traced references 
   from <CODE>r</CODE>, converts it into a pickle, and writes the pickle to 
   <CODE>w.wr</CODE>, starting at the current position of <CODE>w.wr</CODE> and 
   leaving <CODE>w.wr</CODE> positioned after the last byte written.
<P>
   The default method <CODE>w.write(r)</CODE> behaves as follows:
<P>
\begin{itemize}
\item If <CODE>r=NIL</CODE>, write a representation of <CODE>NIL</CODE> on <CODE>w.wr</CODE>.
<P>
\item Otherwise if <CODE>r</CODE> has been previously written to the 
      current pickle, write the index of <CODE>r</CODE> in this pickle 
      on <CODE>w.wr</CODE>.
<P>
\item Otherwise, find the <CODE>Special</CODE> (as defined below) for 
      the closest supertype of <CODE>r</CODE>'s allocated type for 
      which a special has been registered. Let <CODE>sp</CODE> be this 
      special (note that there always is such a special, since
      the implementation registers a special for <CODE>REFANY</CODE>). Call
      <CODE>writeType(sp.sc)</CODE>, then write a representation of
      <CODE>TYPECODE(r)</CODE>, then invoke <CODE>sp.write(r, w)</CODE>.   
\end{itemize}
<P>
   The call <CODE>w.writeType(tc)</CODE> writes bytes on w.wr to represent
   the type <CODE>tc</CODE> in a program-independent fashion. These bytes
   are a fingerprint: fingerprints have the property that if
   two Modula-3 types are equivalent, then their fingerprints
   are equal; and with extremely hight probability, if their
   fingerprints are equal then they are equivalent. Note that
   the implementation of this method optimizes so that if a type
   is written to a pickle many times, occurences after the first
   typically occupy only one byte.
<P>
   The call <CODE>w.writeInt(i)</CODE> writes bytes on <CODE>w.wr</CODE> to represent
   the integer <CODE>i</CODE> in a platform-independent fashion. In
   particular, the <CODE>readInt</CODE> method of a subsequent <CODE>Pickle.Reader</CODE>
   can read the integer correctly, even if executed on a platform
   with different byte order or integer size (if possible).
<P>
   A <CODE>Pickle.Writer</CODE> may be serially re-used to pickle multiple
   objects.
<P>
   The call <CODE>Pickle.Write(wr, ref)</CODE> is equivalent to (but more 
   efficient than) the following:
   <P>
<PRE>
      NEW(Pickle.Writer, wr := wr).write(ref)
</PRE>
   A <CODE>Pickle.Reader</CODE> is an object whose <CODE>read</CODE> method reads
   pickles from the reader stored in its <CODE>rd</CODE> data field.  
   You can override the <CODE>read</CODE> method to produce special
   effects. 
   
<P><PRE>TYPE
  <A HREF="Pickle.m3#Reader">Reader</A> &lt;: ReaderPublic;
  RefID = INTEGER;
  ReaderPublic = OBJECT
      rd: Rd.T;
    METHODS
      read(): REFANY RAISES
        {Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted};
      readType(): TypeCode RAISES
        {Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted};
      readInt(): INTEGER RAISES
        {Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted};
      noteRef(ref: REFANY; id: RefID);
    END;
</PRE> If <CODE>r</CODE> is a <CODE>Pickle.Reader</CODE>, then <CODE>r.read()</CODE>
   reads a pickle from <CODE>r.rd</CODE>, reconstructs a copy of 
   the pickled data structure, and returns it.  The bytes 
   to be read start at <CODE>r.rd</CODE>'s current position; 
   <CODE>r.rd</CODE> is left positioned after the last byte consumed.   
<P>
   The default method <CODE>r.read(r)</CODE> behaves as follows:
<P>
\begin{itemize}
\item If <CODE>r.rd</CODE> contains a representation of <CODE>NIL</CODE>, consume those 
      bytes and return <CODE>NIL</CODE>.
<P>
\item Otherwise, if <CODE>r.rd</CODE> contains the index of a previously 
      read value in this pickle, consume those bytes and return
      that value.
<P>
\item Otherwise compute <CODE>sc = readType()</CODE>, and proceed as follows.
      Find the registered special 
      <CODE>sp</CODE> (as defined below) whose <CODE>sp.sc</CODE> field equals <CODE>sc</CODE>.  
      Read the representation of a type code <CODE>ac</CODE> from <CODE>r.rd</CODE>.  
      Let <CODE>id</CODE> be the index that the result of this method will 
      have in the pickle. Invoke <CODE>sp.read(ac, r, id)</CODE>, call 
      <CODE>r.noteRef(result, id)</CODE>, and return the result.   Note that
      in the default case <CODE>sp</CODE> will be <CODE>RootSpecial()</CODE>, defined
      below. 
\end{itemize}
<P>
   In order to implement the second case above, the <CODE>Pickle.Reader</CODE>
   maintains a table mapping indexes to references.  In simple cases
   this table can be maintained entirely by the <CODE>r.read</CODE> method,
   which places the result of the <CODE>sp.read</CODE> invocation into the
   table. But if the value being unpickled will contain an occurence
   of its own reference (the second case above), the reference must
   be placed in the table before unpickling that occurence. To do
   so, the special's <CODE>read</CODE> method must call <CODE>r.noteRef</CODE> before any
   such occurrence.  More exactly, a special's <CODE>read</CODE> method must
   call <CODE>r.noteRef</CODE> before any occurrence of the ref that the <CODE>read</CODE>
   method will return, giving <CODE>noteRef</CODE> the reference and the
   <CODE>refID</CODE> that was a parameter of the <CODE>read</CODE> method. If there is no
   such occurrence, the special need not (but may) call <CODE>r.noteRef</CODE>.
   It is a checked runtime error to call <CODE>r.noteRef</CODE> with a value other
   than the value that will be returned by the special's <CODE>read</CODE> method;
   it is OK to call <CODE>r.noteRef</CODE> multiple times with the same value.
<P>
   The call <CODE>r.readType</CODE> reads bytes from <CODE>r.rd</CODE> that represent
   a type, as written by the <CODE>writeType</CODE> method of a <CODE>Writer</CODE>,
   and converts them into a typecode valid in the current
   execution of the current program. If there is no such type,
   or if the bytes couldn't correspond to ones written by the
   <CODE>writeType</CODE> method, raises <CODE>Pickle.Error</CODE>.
<P>
   The call <CODE>r.readInt</CODE> reads bytes from <CODE>r.rd</CODE> that represent an
   integer, as written by a writer's <CODE>writeInt</CODE> method, and
   converts them into an integer, taking into consideration
   reperesentation differences such as byte order and integer
   length. It raises <CODE>Pickle.Error</CODE> if the integer cannot be represented
   on this machine.
<P>
   A <CODE>Pickle.Reader</CODE> may be serially re-used to unpickle multiple
   objects.
<P>
   The call <CODE>Read(rd)</CODE>  is equivalent to (but more efficient than)
   the following:
<P>
<PRE>
      RETURN NEW(Reader, rd := rd).read();
</PRE>
\paragraph{Specials.}
   Specials provide for customized pickling of specified data 
   types on every call of <CODE>read</CODE> or <CODE>write</CODE> in this process.  A
   client can achieve this effect for individual calls of <CODE>read</CODE>
   or <CODE>write</CODE> by using a sub-class of <CODE>Pickle.Reader</CODE> or
   <CODE>Pickle.Writer</CODE>, and checking for that sub-class (using
   <CODE>TYPECASE</CODE>) inside the special.  The implementation of this
   interface registers a special for the type <CODE>REFANY</CODE>; this is
   special is the value of a call <CODE>NEW(Special)</CODE>, i.e. its
   methods are the default methods of the type <CODE>Special</CODE>.   There
   are three constraints on the methods of a special:
<P>
\begin{itemize}
\item  the methods must leave the <CODE>Rd.T</CODE> or <CODE>Wr.T</CODE> positioned
       after the last byte read or written;
<P>
\item  the <CODE>read</CODE> method must consume the number of bytes written
       by the <CODE>write</CODE> method;
<P>
\item  the <CODE>read</CODE> method must produce a value equivalent to the
       one that was given to the <CODE>write</CODE> method.
\end{itemize}
<P>
  If these rules are violated, the result could be either a
  checked runtime error or an invalid result from reading a
  pickle.
<P>
  There are many ways to program a special.  For example, 
  the <CODE>write</CODE> method could modify the value and then call 
  the root special.  Or the <CODE>write</CODE> method could create a related
  value and call <CODE>writer.write</CODE> or <CODE>Special.write</CODE>. Or it
  could write some data fields individually and call
  <CODE>writer.write</CODE> for selected sub-values of its value.  Or it
  could use mixtures of these techniques. 

<P><PRE>TYPE
  SpecialPublic = OBJECT
      sc: TypeCode;
    METHODS
      write(ref: REFANY; writer: Writer)
        RAISES {Error, Wr.Failure, Thread.Alerted};
      read(reader: Reader; id: RefID): REFANY
        RAISES {Error, Rd.EndOfFile, Rd.Failure,
          Thread.Alerted};
    END;
  <A HREF="Pickle.m3#Special">Special</A> &lt;: SpecialPublic;
</PRE> The special's <CODE>write</CODE> method should write a representation 
   of <CODE>ref</CODE>'s contents on <CODE>wr</CODE>, in a manner analogous to
   the default <CODE>Special.write</CODE> method. The value <CODE>ref</CODE> is 
   not <CODE>NIL</CODE> and its allocated type is a sub-type of <CODE>sc</CODE>.
<P>
   The default <CODE>write</CODE> method calls
   <CODE>writer.writeType(TYPECODE(ref))</CODE>, then writes on <CODE>writer.wr</CODE> the
   bytes of <CODE>ref</CODE>'s untraced data fields, then calls
   <CODE>writer.write(x)</CODE> for each traced value <CODE>x</CODE> directly contained
   in <CODE>ref</CODE>.
<P>
   The special's <CODE>read</CODE> method must return an appropriate value
   read from <CODE>rd</CODE>. If the value might involve a reference to
   itself, the special must pass the value and <CODE>id</CODE> to <CODE>r.noteRef</CODE>
   before the first such occurrence of the value. In other
   situations the special need not (but may) call <CODE>r.noteRef</CODE>.
   See the earlier discussion of the <CODE>r.read</CODE> method.
<P>
   The result of the <CODE>read</CODE> method must be a value acceptable in
   the context where the method is being invoked. For example, if
   the method is being invoked to read the contents of a field
   of a record, the result must be assignable to the declared type
   of the field.  It is always correct to return a value of the
   same type as the value that was written when the pickle was
   created, but a sub-type might be acceptable too.
<P>
   The default <CODE>read</CODE> method reads a typecode <CODE>tc</CODE> by calling
   <CODE>reader.readType()</CODE>, allocates a value whose type-code is <CODE>tc</CODE>,
   passes the allocated value to the reader's <CODE>noteRef</CODE> method,
   reads the bytes of its untraced data fields from <CODE>reader.rd</CODE>, 
   then for each directly contained traced reference 
   calls <CODE>reader.read()</CODE> and assigns the result into the 
   appropriate field of the allocated value; finally it returns 
   the value.
   <P>
   Note that the special's methods can use the <CODE>writeType</CODE> and
   <CODE>readType</CODE> methods of the <CODE>Writer</CODE> or <CODE>Reader</CODE> to record
   relevant type information. For example they could record the
   allocated type of the value being pickled in the same way as
   the default methods do, or they could record more extensive
   type information if available, such as the entire type
   ancestry of the value's type. 

<P><PRE>PROCEDURE <A HREF="Pickle.m3#RegisterSpecial">RegisterSpecial</A>(sp: Special);
</PRE><BLOCKQUOTE><EM> Register <CODE>sp</CODE> as the special for pickling and unpickling 
   objects having type code <CODE>sp.sc</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> After you call <CODE>RegisterSpecial(sp)</CODE>, the special <CODE>sp</CODE> will be
   called from a <CODE>Pickle.Writer</CODE>'s <CODE>write</CODE> method, or a
   <CODE>Pickle.Reader</CODE>'s <CODE>read</CODE> method, to pickle or unpickle objects
   whose type is <CODE>sp.sc</CODE>, or whose type is a sub-type of <CODE>sp.sc</CODE>
   and has no closer registered special.  It is a checked runtime
   error to register a special <CODE>sp</CODE> when there is already a
   special registered with the same <CODE>sp.sc</CODE>.  There is always a
   special registered for <CODE>REFANY</CODE> that special's methods are
   the default methods of the type <CODE>Special</CODE>. 

<P><PRE>END Pickle.
</PRE> \paragraph{Examples.}  For example, suppose you
are writing a pickle that contains many references of
type <CODE>TEXT</CODE>, and you want to modify the pickling process
so that any texts that are <CODE>Text.Equal</CODE> will share storage.
(By default, texts in the pickle will share storage if and only if
they shared storage in the original data structure.) Of course this 
modification will mean that the unpickled value will have more sharing than 
the original value. A simple way to achieve this is to build the following
subtype of <CODE>Pickle.Writer</CODE>:
<P>
<PRE>
      TYPE WriterP = Pickle.Writer OBJECT
        tbl: TxtTxtTbl.T
      METHODS
        init(): WriterP := InitP
      OVERRIDES
        write := WriteP
      END;
      
      PROCEDURE WriteP(self: WriterP; r: REFANY)
        RAISES {Error, Wr.Failure, Thread.Alerted} =
        VAR txt: TEXT;
        BEGIN
          IF ISTYPE(r, TEXT) THEN
            IF self.tbl.get(r, txt) THEN 
              r := txt
            ELSE
              EVAL self.tbl.put(r, r)
            END
          END;
          Pickle.Writer.write(self, r)
        END WriteP;
      
      PROCEDURE InitP(self: WriterP): WriterP =
        BEGIN
          self.tbl := NEW(TxtTxtTbl.T).init()
          RETURN self
        END InitP;
</PRE>
The <CODE>WriterP</CODE> object keeps a table with an entry for
every text that has been encountered in the pickle.
When the method <CODE>WriteP</CODE> encounters
a reference <CODE>r</CODE> of type <CODE>TEXT</CODE>, it checks if the table contains
a text <CODE>txt</CODE> that is <CODE>Text.Equal</CODE> to <CODE>r</CODE>.  (A <CODE>TxtTxtTbl.T</CODE>
considers two texts to be the same
key if they are <CODE>Text.Equal</CODE>.)  If so, <CODE>txt</CODE> is pickled
instead of <CODE>r</CODE>; if not, <CODE>r</CODE> is entered into the table
so that if any text is later encountered that is
<CODE>Text.Equal</CODE> to <CODE>r</CODE>, then <CODE>r</CODE> will be pickled in
its place.  In either case, the subtype method <CODE>WriteP</CODE>
delegates the actual work of writing the bytes into
the pickle to its supertype method
<CODE>Pickle.Writer.write</CODE>.  
<P>
Given the program above, you can pickle a data
structure <CODE>ref</CODE> to the writer <CODE>wr</CODE>, identifying texts 
that have the same contents, with:
<P>
<PRE>
       NEW(WriterP, wr := wr).init()(ref)
</PRE>
For this example, no special processing is necessary when 
reading; you just use an ordinary <CODE>Pickle.Reader</CODE>, or
a normal call to <CODE>Pickle.Read</CODE>. 
<P>
Next we consider a simple example of the use of specials.  
Suppose that you have a type defined by
<P>
<PRE>
      TYPE T = REF RECORD ref: REFANY; tc: CARDINAL END
</PRE>
in which the <CODE>tc</CODE> field represents a Modula-3 typecode.
Since typecodes are address-space specific, it is necessary
to translate the <CODE>tc</CODE> field so that it represents the same
Modula-3 type in the address space reading the pickle as
it did in the address space that wrote the pickle.  This
is achieved by using the procedures <CODE>RTTypeFP.ToFingerPrint</CODE>
and <CODE>RTTypeFP.FromFingerprint</CODE>, which translate between
typecodes and type fingerprints.  A type fingerprint is
a 64-bit hash value that with high probability identifies
a Modula-3 type in an address-space independent way.
A fingerprint is represented as an <CODE>ARRAY [0..7] OF [0..255]</CODE>. 
<P>
Here is one way to register specials for type <CODE>T</CODE>:
<P>
<PRE>
      TYPE TSpecial = Pickle.Special OBJECT
        OVERRIDES
          write := WriteT;
          read := ReadT
        END;
</PRE>
<PRE>
      PROCEDURE WriteT(
          self: TSpecial; 
          ref: REFANY; 
          wr: Pickle.Writer)
        RAISES {Error, Wr.Failure, Thread.Alerted} =
        VAR 
          t: T := ref; 
          fp := RTTypeFP.ToFingerPrint(t.tc); 
        BEGIN
          FOR i := 0 TO 7 DO 
            Wr.PutChar(wr.wr, VAL(fp[i], CHAR)) 
          END;
          wr.write(t.ref)
        END WriteT;
</PRE>
<PRE>
      PROCEDURE ReadT(
          ac: INTEGER; 
          rd: Pickle.Reader; 
          id: RefID): REFANY
        RAISES {Error, Rd.EndOfFile, Rd.Failure, 
                Thread.Alerted} =
        VAR fp: ARRAY [0..7] OF [0..255]; res: T; BEGIN
          FOR i := 0 TO 7 DO 
            fp[i] := ORD(Rd.GetChar(rd.rd)) 
          END;
          res := NEW(t, tc := RTTypeFP.FromFingerprint(fp));
          rd.noteRef(res, id);
          res.ref := rd.read();
          RETURN res
        END ReadT;
</PRE>
<PRE>
      ...
</PRE>
<PRE>
      Pickle.RegisterSpecial(
        NEW(TSpecial, tc := TYPECODE(T)))
</PRE>
It is important to understand the call to <CODE>rd.noteRef</CODE> made by
the special's read method.  The effect of the call is to
make an entry in the pickle reader's table recording that <CODE>res</CODE>
is the reference associated with <CODE>id</CODE>.  This is essential because
the recursive call to <CODE>rd.read</CODE> might encounter additional
instances of <CODE>id</CODE>, if some value <CODE>t</CODE> of type <CODE>T</CODE> is reachable
from its own <CODE>t.ref</CODE> field.
<P>
As a final example, suppose that <CODE>T</CODE> is an
object instead of a <CODE>REF RECORD</CODE>:
<P>
<PRE>
      TYPE T = OBJECT ref: REFANY; tc: CARDINAL END.
</PRE>
The solution above still works for objects whose
allocated type is <CODE>T</CODE>, but it doesn't work for objects
whose allocated type is a subtype of <CODE>T</CODE> that
has additional fields.  Here are new methods for
the special that automatically handle subtypes
(assuming that any additional fields do not themselves require 
non-standard translation):
<P>
<PRE>
      PROCEDURE WriteT(
          self: TSpecial; 
          ref: REFANY; 
          wr: Pickle.Writer)
        RAISES {Error, Wr.Failure, Thread.Alerted} =
        VAR 
          t: T := ref; 
          fp := RTTypeFP.ToFingerPrint(t.tc); 
        BEGIN
          FOR i := 0 TO 7 DO 
            Wr.PutChar(wr.wr, VAL(fp[i], CHAR)) 
          END;
          Special.write(t, wr)
        END WriteT;
</PRE>
<PRE>
      PROCEDURE ReadT(
          ac: INTEGER; 
          rd: Pickle.Reader; 
          id: RefID): REFANY
        RAISES {Error, Rd.EndOfFile, Rd.Failure, 
                Thread.Alerted} =
        VAR 
          fp: ARRAY [0..7] OF [0..255]; 
          res: T; 
        BEGIN
          FOR i := 0 TO 7 DO 
            fp[i] := ORD(Rd.GetChar(rd.rd)) 
          END;
          res := Special.Read(ac, rd, id);
          res.tc := RTTypeFP.FromFingerprint(fp);
          RETURN res
        END ReadT;
</PRE>
In this case the special uses the root special to transmit the whole
object in the usual way.  The write special also sends the
fingerprint, which the read special uses to fix up the typecode. 
Notice that with this strategy, the call to <CODE>rd.noteRef</CODE> is not
required. 

</inInterface>
<PRE>























</PRE>
</BODY>
</HTML>
