<HTML>
<HEAD>
<TITLE>SRC Modula-3: rw/src/Common/AutoFlushWr.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>rw/src/Common/AutoFlushWr.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>

MODULE <module><implements><A HREF="AutoFlushWr.i3">AutoFlushWr</A></implements></module>;

IMPORT <A HREF="Wr.i3">Wr</A>, <A HREF="WrClass.i3">WrClass</A>, <A HREF="../../../time/src/Common/Time.i3">Time</A>, <A HREF="../../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../../os/src/Common/Process.i3">Process</A>;
FROM <A HREF="Wr.i3">Wr</A> IMPORT Failure;
FROM <A HREF="../../../thread/src/Common/Thread.i3">Thread</A> IMPORT Alerted;
</PRE> A writer wr is *inactive* if wr.cur=wr.hi=wr.lo, and *active* otherwise.  
   Notice that if wr is inactive, c(wr) = target(wr) (by WrClass.V2) 
   and therefore can be ignored by the worker thread that does the
   flushing.  Our strategy is to keep a list containing all active
   writers.  The worker thread wakes up periodically and checks all 
   the writers on the list, flushing them if they have gone unflushed 
   for too long.  The delay between wake ups is the 
   equal to minimum delay of any active writer.  When the list 
   is empty, the worker blocks.  

<P><PRE>CONST
  Forever = LAST(LONGREAL);
  Default = 0.1d0;

VAR
  mu := NEW (Thread.Mutex);
  (* mu protects the next four variables *)
  (* LL(wr) &lt; LL(mu) for all wr: T *)
  active := NEW(T);
  minDelay: Time.T := Forever;
  c := NEW (Thread.Condition); (* that active is nonempty *)
  worker: Thread.T := NIL;
</PRE> The comment LL(wr) &lt; LL(mu) means that the locking level of mu 
   is greater than the locking level of any auto-flush writer.
   This means that a thread must not try to acquire a writer lock 
   while it holds mu. 

<P> The first object on the active list is a sentinal; thus the list 
   is empty if active.link = NIL.  The value of minDelay is the minimum 
   delay of any writer on the active list. 

<P><PRE>REVEAL
  <A NAME="T">T</A> = Public BRANDED OBJECT
        child: Wr.T;
        delay: Time.T;
        alarm: Time.T;           (* readonly if onQ is true *)
        onQ  : BOOLEAN;          (* protected by mu *)
        link : T;                (* protected by mu *)
      OVERRIDES
        seek      := Seek;
        length    := Length;
        flush     := Flush;
        close     := Close;
        init      := Init;
      END;
</PRE> For wr of type T, wr.child is the writer to which wr's writing 
   is forwarded.  wr.delay is the inverse of the frequency with which 
   wr will be flushed.  If wr is active, wr.alarm is the next time 
   at which wr should be flushed; otherwise wr.alarm is irrelevant.  
   The boolean wr.onQ is true iff wr is on the active list, and  
   wr.link is the link field for the active list.  The child, delay, 
   and alarm fields are protected by wr's mutex; the onQ and link 
   fields are protected by the global mu. 

<P><PRE>PROCEDURE <A NAME="Init"><procedure>Init</procedure></A> (wr: T; ch: Wr.T; p := -1.0D0): T =
  BEGIN
    WrClass.Lock(wr);
    WrClass.Lock(ch);
    TRY
      wr.buff := ch.buff;
      wr.st := ch.st;
      wr.cur := ch.cur;
      wr.lo := ch.lo;
      wr.hi := ch.hi;
      wr.closed := ch.closed;
      wr.seekable := ch.seekable;
      wr.buffered := TRUE;
      wr.child := ch;
      wr.onQ := FALSE;
      wr.link := NIL;
    FINALLY
      WrClass.Unlock(ch);
      WrClass.Unlock(wr);
    END;
    IF p &lt; 0.0D0 THEN wr.delay := Default ELSE wr.delay := p END;
    RETURN wr
  END Init;

PROCEDURE <A NAME="Seek"><procedure>Seek</procedure></A> (wr: T; n: CARDINAL) RAISES {Failure, Alerted} =
  VAR wasEmpty := FALSE;
  BEGIN
    WrClass.Lock(wr.child);
    TRY
      wr.child.cur := wr.cur;
      wr.child.seek(n);
    FINALLY
      WrClass.Unlock(wr.child);
    END;
    wr.buff := wr.child.buff;
    wr.st := wr.child.st;
    wr.cur := wr.child.cur;
    wr.lo := wr.child.lo;
    wr.hi := wr.child.hi;
    LOCK mu DO
      IF NOT wr.onQ THEN
        wr.onQ := TRUE;
        wasEmpty := active.link = NIL;
        wr.link := active.link;
        active.link := wr;
        wr.alarm := Time.Now() + wr.delay;
        IF wasEmpty THEN
          minDelay := wr.delay;
          IF worker = NIL THEN
            worker := Thread.Fork(NEW(Thread.Closure, apply := Worker))
          END
        ELSIF minDelay &gt; wr.delay THEN
          minDelay := wr.delay;
          Thread.Alert(worker)
        END
      END
    END;
    IF wasEmpty THEN Thread.Signal(c) END
  END Seek;

PROCEDURE <A NAME="Flush"><procedure>Flush</procedure></A> (wr: T) RAISES {Failure, Alerted} =
  BEGIN
    WrClass.Lock(wr.child);
    TRY
      wr.child.cur := wr.cur;
      wr.child.flush();

      wr.buff := wr.child.buff;
      (* Set everything empty. *)
      wr.st := wr.child.st + wr.child.cur - wr.child.lo;
      wr.cur := wr.child.cur;
      wr.lo := wr.child.cur;
      wr.hi := wr.child.cur;
    FINALLY
      WrClass.Unlock(wr.child);
    END;
  END Flush;

PROCEDURE <A NAME="Length"><procedure>Length</procedure></A> (wr: T): CARDINAL RAISES {Failure, Alerted} =
  BEGIN
    WrClass.Lock(wr.child);
    TRY
      wr.child.cur := wr.cur;
      WITH res = wr.child.length() DO
        wr.buff := wr.child.buff;
        wr.st := wr.child.st;
        wr.cur := wr.child.cur;
        wr.lo := wr.child.lo;
        wr.hi := wr.child.hi;
        RETURN res;
      END;
    FINALLY
      WrClass.Unlock(wr.child);
    END;
  END Length;

PROCEDURE <A NAME="Close"><procedure>Close</procedure></A>(wr: T) RAISES {Failure, Alerted} =
  BEGIN
    WrClass.Lock(wr.child);
    TRY
      wr.child.cur := wr.cur;
      wr.child.close();
    FINALLY
      WrClass.Unlock(wr.child);
    END;
    wr.buff := NIL
  END Close;
</PRE> The worker thread delay is the minimum delay of any active writer. 
    Each time it wakes up, it executes the following:
<P>
<PRE>
          Let S be the set of active writers.
          Set minDelay := infinity;
          FOR wr IN S do:
           IF wr's timer has expired THEN
             Flush wr, remove it from the active list
           ELSE
             minDelay := min(minDelay, wr.delay)
           END
         END
</PRE>
    Notice that the active list can be enlarged while S is being 
    processed, since clients may be operating on writers concurrently.  
    If minDelay decreases while the worker is asleep, it is alerted. 
    

<P><PRE>PROCEDURE <A NAME="Worker"><procedure>Worker</procedure></A> (&lt;*UNUSED*&gt;cl: Thread.Closure): REFANY RAISES {} =
  &lt;*FATAL Wr.Failure, Thread.Alerted*&gt;
  VAR
    delay, now: Time.T;
    s:     T;
    wr:    T;
  BEGIN
    LOOP
      LOCK mu DO
        WHILE active.link = NIL DO Thread.Wait(mu, c) END;
        delay := minDelay
      END;
      TRY
        Thread.AlertPause(delay)
      EXCEPT
        Thread.Alerted  =&gt; (*skip*)
      END;
      LOCK mu DO
        s := active;
        minDelay := Forever;
      END;
      now := Time.Now();
      LOOP
        (* s scans through the set called S above. *)
        LOCK mu DO
          IF s.link = NIL THEN EXIT END;
          wr := s.link;
          IF now &gt; wr.alarm THEN
            s.link := wr.link;
            wr.onQ := FALSE;
            wr.link := NIL
          ELSE
            s := wr;
            wr := NIL;
            IF minDelay &gt; s.delay THEN
              minDelay := s.delay
            END
          END
        END;
        IF wr # NIL THEN
          WrClass.Lock(wr);
          TRY
            IF NOT wr.closed THEN Flush(wr) END;
          FINALLY
            WrClass.Unlock(wr);
          END;
        END (* IF *);
      END
    END
  END Worker;
</PRE> Notice that Flush(wr) cannot be called while mu is locked, since 
   this violates the locking level order.  Thus wr is taken out of 
   the lock to be flushed.  

<P><PRE>PROCEDURE <A NAME="Shutdown"><procedure>Shutdown</procedure></A> () =
  VAR s, wr: T;
  &lt;*FATAL Wr.Failure, Thread.Alerted *&gt;
  BEGIN
    LOCK mu DO s := active; END;
    LOOP
      (* s scans through the set called S above. *)
      LOCK mu DO
        IF s.link = NIL THEN EXIT END;
        wr := s.link;
        s.link := wr.link;
        wr.onQ := FALSE;
        wr.link := NIL
      END;
      WrClass.Lock(wr);
      TRY
        IF NOT wr.closed THEN Flush(wr) END;
      FINALLY
        WrClass.Unlock(wr);
      END;
    END
  END Shutdown;

BEGIN Process.RegisterExitor(Shutdown); END AutoFlushWr.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
