<HTML>
<HEAD>
<TITLE>SRC Modula-3: slisp/src/SLisp.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>slisp/src/SLisp.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                                           </EM></BLOCKQUOTE><PRE>

UNSAFE MODULE <module>SLisp</module> EXPORTS <A HREF="SLisp.i3"><implements>SLisp</A></implements>, <A HREF="SLispClass.i3"><implements>SLispClass</A></implements>;

IMPORT <A HREF="../../text/src/Text.i3">Text</A>, <A HREF="../../sx/src/Sx.i3">Sx</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../rw/src/Common/Wr.i3">Wr</A>, <A HREF="../../libm3/derived/RefList.i3">RefList</A>, <A HREF="../../rw/src/Common/IO.i3">IO</A>, <A HREF="../../atom/src/Atom.i3">Atom</A> AS Aatom;
IMPORT <A HREF="../../libm3/derived/IntRefTbl.i3">IntRefTbl</A>, <A HREF="SLispMath.i3">SLispMath</A>, <A HREF="../../rw/src/Common/Stdio.i3">Stdio</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../fmtlex/src/Fmt.i3">Fmt</A>, <A HREF="../../rw/src/Common/TextRd.i3">TextRd</A>, <A HREF="../../rw/src/Common/TextWr.i3">TextWr</A>,
       <A HREF="../../runtime/src/common/RTTypeSRC.i3">RTTypeSRC</A>, <A HREF="../../word/src/Word.i3">Word</A>;

&lt;*FATAL Wr.Failure *&gt;
&lt;*FATAL Thread.Alerted*&gt;
&lt;*FATAL Rd.EndOfFile*&gt;
&lt;*FATAL Sx.PrintError *&gt;

REVEAL
  <A NAME="T">T</A> = TPublic BRANDED OBJECT
         props: RefList.T;
      OVERRIDES
         new := new;
         init := init;
         error := error;

         load := load1;

         defineVar := defineVar;
         defineFun := defineFun;

         checkSymbol := checkSymbol;
         checkList := checkList;
         checkInt := checkInt;
         checkFloat := checkFloat;
         checkString := checkString;

         eval := eval;
           evalSymbol := evalSymbol;
           evalList := evalList;
           evalInt := evalInt;
           evalFloat := evalFloat;
           evalString := evalString;

         sEval := sEval;
         varEval := varEval;
         lookup     := lookup;
        lookupAtom := lookupAtom;
        pushScope  := PushScope;
        popScope   := PopScope;
   END;

PROCEDURE <A NAME="new"><procedure>new</procedure></A> (&lt;* UNUSED *&gt; self: T): T =
  BEGIN
    RETURN NEW (T).init();
  END new;

PROCEDURE <A NAME="init"><procedure>init</procedure></A> (self: T): T =
  BEGIN
    self.props := NIL;

    self.underEval := NEW (List);
    self.topFrame := NEW (Frame, procName := &quot;*top*&quot;, endScope := TRUE);
    self.frame := self.topFrame;
    self.depth := 0;

    self.defineFun (NEW (Builtin, name := &quot;abort&quot;, apply := Abort,
                         minArgs := 0, maxArgs := 0));
    self.defineFun (NEW (Builtin, name := &quot;setq&quot;, apply := Setq,
                         minArgs := 2, maxArgs := 2));
    self.defineFun (NEW (Builtin, name := &quot;quote&quot;, apply := Quote,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;intern&quot;, apply := Intern,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;eval&quot;, apply := EvalBI,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;printname&quot;, apply := Printname,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;cond&quot;, apply := Cond,
                         minArgs := 0, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;if&quot;, apply := If,
                         minArgs := 2, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;while&quot;, apply := While,
                         minArgs := 1, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;listp&quot;, apply := Listp,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;integerp&quot;, apply := Integerp,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;floatp&quot;, apply := Floatp,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;stringp&quot;, apply := Stringp,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;symbolp&quot;, apply := Symbolp,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;defun&quot;, apply := Defun,
                         minArgs := 2, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;apply&quot;, apply := UApply,
                         minArgs := 2, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;defmacro&quot;, apply := Defmacro,
                         minArgs := 2, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;progn&quot;, apply := Progn,
                         minArgs := 0, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;let&quot;, apply := Let,
                         minArgs := 1, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;letstar&quot;, apply := LetStar,
                         minArgs := 1, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;backtrace&quot;, apply := Backtrace,
                         minArgs := 0, maxArgs := 0));
    self.defineFun (NEW (Builtin, name := &quot;load&quot;, apply := Load,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;print&quot;, apply := Print,
                         minArgs := 0, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;cons&quot;, apply := Cons,
                         minArgs := 2, maxArgs := 2));

    self.defineFun (NEW (Builtin, name := &quot;car&quot;, apply := Car,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;cdr&quot;, apply := Cdr,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;caar&quot;, apply := Caar,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;cadr&quot;, apply := Cadr,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;cdar&quot;, apply := Cdar,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;cddr&quot;, apply := Cddr,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;concat&quot;, apply := Concat,
                         minArgs := 0, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;append&quot;, apply := Append,
                         minArgs := 0, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;list&quot;, apply := LIst,
                         minArgs := 0, maxArgs := LAST (INTEGER)));
    self.defineFun (NEW (Builtin, name := &quot;length&quot;, apply := Length,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;get_prop&quot;, apply := GetProp,
                         minArgs := 1, maxArgs := 1));
    self.defineFun (NEW (Builtin, name := &quot;set_prop&quot;, apply := SetProp,
                         minArgs := 2, maxArgs := 2));

    self.defineVar (&quot;t&quot;,      syms.t);
    self.defineVar (&quot;nil&quot;,    NIL);
    self.defineVar (&quot;stdin&quot;,  Stdio.stdin);
    self.defineVar (&quot;stdout&quot;, Stdio.stdout);
    self.defineVar (&quot;stderr&quot;, Stdio.stderr);

    SLispMath.Register(self); (* depends on &quot;t&quot; *)

    RETURN self;
  END init;

PROCEDURE <A NAME="error"><procedure>error</procedure></A> (self: T; msg: Text.T := &quot;&quot;): Sexp RAISES {Error} =
  VAR
    stdin  := self.varEval(&quot;stdin&quot;);
    stdout := self.varEval(&quot;stdout&quot;);
    stderr := self.varEval(&quot;stderr&quot;);
  BEGIN
    Wr.PutText(stderr, &quot;error: &quot;);
    Wr.PutText(stderr, msg);
    Wr.PutText(stderr, &quot;\n&quot;);

    IF stdin # NIL THEN
      INC(self.depth);
      self.underEval := RefList.List1(self.underEval);
      self.defineVar(&quot;stdout&quot;, stderr);
      TRY
        LOOP
          Wr.PutText(stderr, Fmt.Int(self.depth) &amp; &quot;&gt; &quot;);
          Wr.Flush(stderr);
          Write(stderr, self.eval(Read(stdin)));
          Wr.PutText(stderr, &quot;\n&quot;);
          Wr.Flush(stderr);
        END;
      EXCEPT
      | Sx.ReadError =&gt;
          Wr.PutText(stderr, &quot;can't parse input\n&quot;);
          Wr.Flush(stderr);
      | Error =&gt;
          DEC(self.depth);
          self.underEval := self.underEval.tail;
          self.defineVar(&quot;stdout&quot;, stdout);
      END;

      RETURN NIL;
    ELSE
      EVAL Backtrace(NIL, self, NIL);
      RAISE Error;
    END;
  END error;

PROCEDURE <A NAME="lookup"><procedure>lookup</procedure></A> (self: T; s: Symbol; create: LookupMode): Atom =
  VAR
    at   : Atom;
    frame: Frame;
  BEGIN
    IF self.frame # self.topFrame THEN
      frame := self.frame;
      LOOP
        FOR i := 0 TO frame.size - 1 DO
          IF frame.table[i].symbol = s THEN
            RETURN (frame.table[i].atom);
          END;
        END;
        IF frame.endScope THEN EXIT END;
        frame := frame.next;
      END;
    END;
    FOR i := 0 TO self.topFrame.size - 1 DO
      IF self.topFrame.table[i].symbol = s THEN
        RETURN (self.topFrame.table[i].atom);
      END;
    END;
    CASE create OF
    | LookupMode.CreateLocal =&gt;
        at := NEW(Atom);
        Insert(self.frame, s, at);
        RETURN at;
    | LookupMode.CreateGlobal =&gt;
        at := NEW(Atom);
        Insert(self.topFrame, s, at);
        RETURN at;
    | LookupMode.LookupOnly =&gt; RETURN NIL
    END;
  END lookup;

PROCEDURE <A NAME="lookupAtom"><procedure>lookupAtom</procedure></A> (self: T; atom: Atom): Symbol =
  VAR frame: Frame;
  BEGIN
    IF self.frame # self.topFrame THEN
      frame := self.frame;
      LOOP
        FOR i := 0 TO frame.size - 1 DO
          IF frame.table[i].atom = atom THEN
            RETURN (frame.table[i].symbol);
          END;
        END;
        IF frame.endScope THEN EXIT END;
        frame := frame.next;
      END;
    END;
    FOR i := 0 TO self.topFrame.size - 1 DO
      IF self.topFrame.table[i].atom = atom THEN
        RETURN (self.topFrame.table[i].symbol);
      END;
    END;
    RETURN NIL;
  END lookupAtom;

PROCEDURE <A NAME="Insert"><procedure>Insert</procedure></A> (frame: Frame; symbol: Symbol; atom: Atom) =
  BEGIN
    IF frame.table = NIL OR frame.size = NUMBER (frame.table^) THEN
      VAR newTable := NEW (REF ARRAY OF Binding,
                           MAX (frame.size * 2, 5)); BEGIN
        IF frame.table # NIL THEN
          SUBARRAY (newTable^, 0, frame.size) := frame.table^; END;
        frame.table := newTable; END; END;
    frame.table [frame.size] := NEW (Binding, symbol := symbol, atom := atom);
    INC (frame.size);
  END Insert;

PROCEDURE <A NAME="defineVar"><procedure>defineVar</procedure></A> (self: T; name: Text.T; val: Sexp) =
  VAR sym := Aatom.FromText (name);
      at := self.lookup (sym);
  BEGIN
    at.val := val;
  END defineVar;

PROCEDURE <A NAME="defineFun"><procedure>defineFun</procedure></A> (self: T; cl: Builtin) =
  VAR sym := Aatom.FromText (cl.name);
      at := self.lookup (sym);
  BEGIN
    at.builtin := cl;
    at.funDefined := TRUE;
  END defineFun;

PROCEDURE <A NAME="eval"><procedure>eval</procedure></A> (self: T; e: Sexp): Sexp RAISES {Error} =
  BEGIN
    self.underEval.head := e;
    self.evalStack := RefList.Cons(e, self.evalStack);
    TRY
      TYPECASE e OF
      | Integer, Float, String =&gt; RETURN (e);
      | Symbol (sym) =&gt; RETURN self.lookup(sym, LookupMode.CreateLocal).val;
      | List (list) =&gt;
          IF list = NIL OR list.head = NIL THEN
            EVAL self.error(&quot;cannot apply&quot;);
          END;
          RETURN Apply(self, list.head, list.tail);
      | REF REAL (r) =&gt;
          VAR e := NEW(Float);
          BEGIN
            e^ := FLOAT(r^, REAL);
            RETURN (e);
          END;
      | REF EXTENDED (r) =&gt;
          VAR e := NEW(Float);
          BEGIN
            e^ := FLOAT(r^, REAL);
            RETURN (e);
          END;
      ELSE
        RETURN self.error(&quot;wrong type ?&quot;);
      END;
    FINALLY
      self.evalStack := self.evalStack.tail;
    END;
  END eval;

PROCEDURE <A NAME="varEval"><procedure>varEval</procedure></A> (self: T; name: Text.T): Sexp =
  BEGIN
    RETURN self.lookup (Aatom.FromText (name), LookupMode.CreateLocal).val;
  END varEval;

PROCEDURE <A NAME="sEval"><procedure>sEval</procedure></A> (self: T; s: Text.T): Text.T RAISES {Error} =
  VAR rd := TextRd.New (s); &lt;* FATAL Sx.ReadError *&gt;
  BEGIN
    RETURN SxToText(self.eval (Sx.Read (rd, syntax)));
  END sEval;

PROCEDURE <A NAME="checkSymbol"><procedure>checkSymbol</procedure></A> (self: T; e: Sexp): Symbol RAISES {Error} =
  BEGIN
    IF e = NIL OR NOT ISTYPE(e, Symbol) THEN
      RETURN
        self.error(Fmt.F(&quot;\&quot;%s\&quot; should be a symbol&quot;, SxToText(e)));
    ELSE
      RETURN NARROW(e, Symbol);
    END;
  END checkSymbol;

PROCEDURE <A NAME="evalSymbol"><procedure>evalSymbol</procedure></A> (self: T; e: Sexp): Symbol RAISES {Error} =
  BEGIN
    RETURN self.checkSymbol (self.eval (e));
  END evalSymbol;

PROCEDURE <A NAME="checkList"><procedure>checkList</procedure></A> (self: T; e: Sexp): List RAISES {Error} =
  BEGIN
    IF NOT ISTYPE (e, List) THEN
       RETURN self.error(Fmt.F(&quot;\&quot;%s\&quot; should be a list&quot;, SxToText(e)));
    ELSE
      RETURN NARROW (e, List); END;
  END checkList;

PROCEDURE <A NAME="evalList"><procedure>evalList</procedure></A> (self: T; e: Sexp): List RAISES {Error} =
  BEGIN
    RETURN self.checkList (self.eval (e));
  END evalList;

PROCEDURE <A NAME="checkInt"><procedure>checkInt</procedure></A> (self: T; e: Sexp): INTEGER RAISES {Error} =
  BEGIN
    IF e = NIL OR NOT ISTYPE (e, Integer) THEN
       EVAL self.error(Fmt.F(&quot;\&quot;%s\&quot; should be an integer&quot;, SxToText(e)));
       RETURN 0;
    ELSE
      RETURN NARROW (e, Integer)^; END;
  END checkInt;

PROCEDURE <A NAME="evalInt"><procedure>evalInt</procedure></A> (self: T; e: Sexp): INTEGER RAISES {Error} =
  BEGIN
    RETURN self.checkInt (self.eval (e));
  END evalInt;

PROCEDURE <A NAME="checkFloat"><procedure>checkFloat</procedure></A> (self: T; e: Sexp): REAL RAISES {Error} =
  BEGIN
    IF e = NIL OR NOT ISTYPE(e, Float) THEN
      IF ISTYPE(e, REF REAL) THEN
        RETURN FLOAT(NARROW(e, REF REAL)^, REAL);
      ELSIF ISTYPE(e, REF EXTENDED) THEN
        RETURN FLOAT(NARROW(e, REF EXTENDED)^, REAL);
      ELSE
        EVAL self.error(Fmt.F(&quot;\&quot;%s\&quot; should be a float&quot;, SxToText(e)));
        RETURN 0.0; (* get rid of warning *)
      END;
    ELSE
      RETURN NARROW(e, Float)^;
    END;
  END checkFloat;

PROCEDURE <A NAME="evalFloat"><procedure>evalFloat</procedure></A> (self: T; e: Sexp): REAL RAISES {Error} =
  BEGIN
    RETURN self.checkFloat (self.eval (e));
  END evalFloat;

PROCEDURE <A NAME="checkString"><procedure>checkString</procedure></A> (self: T; e: Sexp): String RAISES {Error} =
  BEGIN
    IF e = NIL OR NOT ISTYPE (e, String) THEN
      RETURN self.error(Fmt.F(&quot;\&quot;%s\&quot; should be a string&quot;, SxToText(e)));
    ELSE
      RETURN NARROW (e, String); END;
  END checkString;

PROCEDURE <A NAME="evalString"><procedure>evalString</procedure></A> (self: T; e: Sexp): String RAISES {Error} =
  BEGIN
    RETURN self.checkString (self.eval (e));
  END evalString;
</PRE>---------------------------------------------------------------------------

<P><PRE>PROCEDURE <A NAME="Apply"><procedure>Apply</procedure></A> (self: T; fun: Sexp; args: List): Sexp RAISES {Error} =
  VAR
    atom    : Atom;
    newFrame: Frame;
    funSym  : Symbol;
  BEGIN
    funSym := self.checkSymbol(fun);
    atom := self.lookup(funSym, LookupMode.CreateLocal);
    newFrame := NEW(Frame, next := self.frame, size := 0,
                    procName := Aatom.ToText(funSym), endScope := TRUE);

    IF NOT atom.funDefined THEN
      RETURN self.error(&quot;undefined: &quot; &amp; Aatom.ToText(self.lookupAtom(atom)));
    ELSIF atom.builtin # NIL THEN
      VAR n := RefList.Length(args);
      BEGIN
        IF n &lt; atom.builtin.minArgs THEN
          RETURN self.error(&quot;not enough arguments for: &quot;
                              &amp; Aatom.ToText(self.lookupAtom(atom)));
        ELSIF n &gt; atom.builtin.maxArgs THEN
          RETURN
            self.error(
              &quot;too many arguments for: &quot; &amp; Aatom.ToText(self.lookupAtom(atom)));
        ELSE
          RETURN atom.builtin.apply(self, args);
        END;
      END;
    ELSE
      VAR
        formals           := atom.funFormals;
        formalSym: Symbol;
        actuals           := args;
        body              := atom.funBody;
        res      : Sexp;
        eval              := NOT atom.macro;
      BEGIN

        WHILE formals # NIL DO
          formalSym := self.checkSymbol(formals.head);
          IF formalSym = syms.ampersandRest THEN
            formals := formals.tail;
            formalSym := self.checkSymbol(formals.head);
            VAR
              ll      := Copy(actuals);
              ll_last := ll;
            BEGIN
              IF eval THEN
                WHILE ll_last # NIL DO
                  ll_last.head := self.eval(ll_last.head);
                  ll_last := ll_last.tail;
                END;
              END;
              Insert(newFrame, formalSym, NEW(Atom, val := ll));
            END;
          ELSIF formalSym = syms.ampersandNoEval THEN
            eval := FALSE;
          ELSIF formalSym = syms.ampersandEval THEN
            eval := TRUE;
          ELSE
            IF actuals = NIL THEN
              RETURN self.error(
                       &quot;not enough arguments for call to: &quot; &amp; Aatom.ToText(funSym))
            END;
            IF eval THEN
              Insert(newFrame, formalSym,
                     NEW(Atom, val := self.eval(actuals.head)));
            ELSE
              Insert(newFrame, formalSym, NEW(Atom, val := actuals.head));
            END;
            actuals := actuals.tail;
          END;
          formals := formals.tail;
        END;

        self.frame := newFrame;
        WHILE body # NIL DO
          res := self.eval(body.head);
          body := body.tail;
        END;
        self.frame := self.frame.next;

        IF atom.macro THEN res := self.eval(res); END;
        RETURN res;
      END;
    END;
  END Apply;
</PRE>-------------------------------------------------------------- builtins ---

<P><PRE>PROCEDURE <A NAME="Abort"><procedure>Abort</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; &lt;*UNUSED*&gt; interp: T;
                 &lt;*UNUSED*&gt; args: List): Sexp RAISES {Error} =
  BEGIN
    RAISE Error;
  END Abort;

PROCEDURE <A NAME="Setq"><procedure>Setq</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR sym: Symbol; at: Atom;
  BEGIN
    sym := interp.checkSymbol (args.head);
    at := interp.lookup (sym);
    at.val := interp.eval (args.tail.head);
    RETURN at.val;
  END Setq;

PROCEDURE <A NAME="Quote"><procedure>Quote</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; &lt;*UNUSED*&gt; interp: T;
                 args: List): Sexp =
  BEGIN
    RETURN args.head;
  END Quote;

PROCEDURE <A NAME="EvalBI"><procedure>EvalBI</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR e1 := interp.eval(args.head);
  BEGIN
    RETURN interp.eval(e1);
  END EvalBI;

PROCEDURE <A NAME="Intern"><procedure>Intern</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR sym: Symbol;
  BEGIN
    sym := Aatom.FromText (interp.evalString (args.head));
    Insert (interp.topFrame, sym, NEW (Atom, val := NIL));
    RETURN sym;
  END Intern;

PROCEDURE <A NAME="Printname"><procedure>Printname</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR sym: Symbol;
  BEGIN
    sym := interp.evalSymbol (args.head);
    RETURN Aatom.ToText (sym);
  END Printname;

PROCEDURE <A NAME="Cond"><procedure>Cond</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR res, caseVal: Sexp := NIL; condCase: List;
  BEGIN
    WHILE args # NIL DO
      condCase := interp.checkList (args.head);
      caseVal := interp.eval (condCase.head);
      IF caseVal # NIL THEN
        condCase := condCase.tail;
        WHILE condCase # NIL DO
          res := interp.eval (condCase.head);
          condCase := condCase.tail; END;
        RETURN res; END;
      args := args.tail; END;
    RETURN NIL;
  END Cond;

PROCEDURE <A NAME="If"><procedure>If</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR
    res  : Sexp;
    forms       := args.tail;
  BEGIN
    IF interp.eval(args.head) # NIL THEN
      RETURN interp.eval(forms.head);
    ELSE
      forms := forms.tail;
      WHILE forms # NIL DO
        res := interp.eval(forms.head);
        forms := forms.tail;
      END;
    END;
    RETURN res;
  END If;

PROCEDURE <A NAME="While"><procedure>While</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR
    res  : Sexp;
    cond        := args.head;
    body        := args.tail;
    forms: List;
  BEGIN
    WHILE interp.eval(cond) # NIL DO
      forms := body;
      WHILE forms # NIL DO
        res := interp.eval(forms.head);
        forms := forms.tail;
      END;
    END;
    RETURN res;
  END While;

PROCEDURE <A NAME="Listp"><procedure>Listp</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    WITH a = interp.eval(args.head) DO
      IF NOT ISTYPE(a, List) THEN RETURN NIL; ELSE RETURN syms.t; END;
    END;
  END Listp;

PROCEDURE <A NAME="Integerp"><procedure>Integerp</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    WITH a = interp.eval(args.head) DO
      IF a = NIL OR NOT ISTYPE(a, Integer) THEN
        RETURN NIL;
      ELSE
        RETURN syms.t;
      END;
    END;
  END Integerp;

PROCEDURE <A NAME="Floatp"><procedure>Floatp</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    WITH a = interp.eval(args.head) DO
      IF a = NIL OR NOT ISTYPE(a, Float) THEN
        RETURN NIL;
      ELSE
        RETURN syms.t;
      END;
    END;
  END Floatp;

PROCEDURE <A NAME="Stringp"><procedure>Stringp</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    WITH a = interp.eval(args.head) DO
      IF a = NIL OR NOT ISTYPE(a, String) THEN
        RETURN NIL;
      ELSE
        RETURN syms.t;
      END;
    END;
  END Stringp;

PROCEDURE <A NAME="Symbolp"><procedure>Symbolp</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    WITH a = interp.eval(args.head) DO
      IF a = NIL OR NOT ISTYPE(a, Symbol) THEN
        RETURN NIL;
      ELSE
        RETURN syms.t;
      END;
    END;
  END Symbolp;

PROCEDURE <A NAME="UApply"><procedure>UApply</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR a,l: RefList.T := NIL; f: Symbol;
  BEGIN
    f := interp.evalSymbol (args.head);
    args := args.tail;
    WHILE args.tail # NIL DO
      IF l = NIL THEN
        l := RefList.List2 (syms.quote, interp.eval (args.head));
        a := l;
      ELSE
        l.tail := RefList.List2 (syms.quote, interp.eval (args.head));
        l := l.tail; END;
      args := args.tail; END;
    args := interp.eval (args.head);
    WHILE args # NIL DO
      IF l = NIL THEN
        l := RefList.List2 (syms.quote, args.head);
        a := l;
      ELSE
        l.tail := RefList.List2(syms.quote, args.head);
        l := l.tail; END;
      args := args.tail; END;
    RETURN Apply (interp, f, a);
  END UApply;

PROCEDURE <A NAME="Defun"><procedure>Defun</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR sym: Symbol; at: Atom;
  BEGIN
    sym := interp.checkSymbol (args.head);
    at := interp.lookup (sym);
    at.funDefined := TRUE;
    at.macro := FALSE;
    at.funFormals := interp.checkList (args.tail.head);
    at.funBody := args.tail.tail;
    RETURN sym;
  END Defun;

PROCEDURE <A NAME="Defmacro"><procedure>Defmacro</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR sym: Symbol; at: Atom;
  BEGIN
    sym := interp.checkSymbol (args.head);
    at := interp.lookup (sym);
    at.funDefined := TRUE;
    at.macro := TRUE;
    at.funFormals := interp.checkList (args.tail.head);
    at.funBody := args.tail.tail;
    RETURN sym;
  END Defmacro;

PROCEDURE <A NAME="Progn"><procedure>Progn</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR res: Sexp := NIL;
  BEGIN
    WHILE args # NIL DO
      res := interp.eval (args.head);
      args := args.tail; END;
    RETURN res;
  END Progn;

PROCEDURE <A NAME="LetStar"><procedure>LetStar</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    RETURN Let2(interp, args, TRUE);
  END LetStar;

PROCEDURE <A NAME="Let"><procedure>Let</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    RETURN Let2(interp, args);
  END Let;

PROCEDURE <A NAME="Let2"><procedure>Let2</procedure></A> (interp: T; args: List; letStar := FALSE): Sexp
  RAISES {Error} =
  VAR
    newFrame: Frame;
    bindings: List;
    forms           := args.tail;
    res     : Sexp  := NIL;
  BEGIN
    newFrame := NEW(Frame, next := interp.frame, size := 0);
    IF letStar THEN interp.frame := newFrame END;
    bindings := checkList(interp, args.head);
    WHILE bindings # NIL DO
      TYPECASE bindings.head OF
      | NULL =&gt;
          EVAL interp.error(&quot;first argument of a binding can not be NIL&quot;);
      | List (l) =&gt;
          IF ISTYPE(l.head, Symbol) THEN
            IF l.tail # NIL THEN
              Insert(newFrame, l.head,
                     NEW(Atom, val := interp.eval(l.tail.head)));
            ELSE
              Insert(newFrame, l.head, NEW(Atom, val := NIL));
            END;
          ELSE
            EVAL interp.error(&quot;should be a symbol: &quot; &amp; SxToText(l.head));
          END;
      | Symbol (s) =&gt; Insert(newFrame, s, NEW(Atom, val := NIL));
      ELSE
        EVAL interp.error(&quot;should be a symbol or a list: &quot;
                            &amp; SxToText(bindings.head));
      END;
      bindings := bindings.tail;
    END;
    IF NOT letStar THEN interp.frame := newFrame END;
    WHILE forms # NIL DO
      res := interp.eval(forms.head);
      forms := forms.tail;
    END;
    interp.frame := interp.frame.next;
    RETURN res;
  END Let2;

PROCEDURE <A NAME="Backtrace"><procedure>Backtrace</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T;
                     &lt;*UNUSED*&gt; args: List): Sexp RAISES {Error} =
  VAR frame := interp.frame; stdout := interp.varEval (&quot;stdout&quot;);
  BEGIN
    IF interp.depth &gt; 0 THEN
      Write (stdout, interp.underEval.tail.head);
      Wr.PutText (stdout, &quot;)\n&quot;); END;
    WHILE frame # interp.topFrame DO
      Wr.PutText (stdout, &quot;(&quot; &amp; frame.procName);
      FOR i := 0 TO frame.size - 1 DO
        Wr.PutText (stdout, &quot; &quot;);
        Write (stdout, frame.table [i].atom.val); END;
      Wr.PutText (stdout, &quot;)\n&quot;);
      Wr.Flush (stdout);
      frame := frame.next; END;
    RETURN syms.t;
  END Backtrace;

PROCEDURE <A NAME="load1"><procedure>load1</procedure></A>  (interp: T; name: Text.T): Sexp RAISES {Error} =
  VAR from: Rd.T; res: Sexp := NIL;
  BEGIN
    from := IO.OpenRead (name);
    IF from = NIL THEN
      RETURN interp.error(Fmt.F(&quot;Could not load file: %s&quot;, name))
    END;
    TRY
      LOOP
        res := interp.eval (Read (from)); END;
    EXCEPT
      | Sx.ReadError =&gt; RETURN interp.error(Fmt.F(&quot;Sx error loading file: %s&quot;, name))
      | Rd.EndOfFile =&gt; END;
    RETURN res;
  END load1;

PROCEDURE <A NAME="Load"><procedure>Load</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  BEGIN
    RETURN load1 (interp, interp.evalString (args.head));
  END Load;

PROCEDURE <A NAME="Print"><procedure>Print</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR stdout := interp.varEval (&quot;stdout&quot;); arg := args;
  BEGIN
    WHILE args # NIL DO
      Write (stdout, interp.eval (args.head));
      args := args.tail; END;
    RETURN arg;
  END Print;

PROCEDURE <A NAME="Cons"><procedure>Cons</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  BEGIN
    RETURN RefList.Cons (interp.eval     (args.head),
                      interp.evalList (args.tail.head));
  END Cons;

PROCEDURE <A NAME="Car"><procedure>Car</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR l := interp.evalList(args.head);
  BEGIN
    IF l = NIL THEN
      RETURN interp.error(&quot;\&quot;car\&quot; of empty list&quot;)
    ELSE
      RETURN l.head
    END;
  END Car;

PROCEDURE <A NAME="Cdr"><procedure>Cdr</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR l := interp.evalList(args.head);
  BEGIN
    IF l = NIL THEN
      RETURN interp.error(&quot;\&quot;cdr\&quot; of empty list&quot;)
    ELSE
      RETURN l.tail
    END;
  END Cdr;

PROCEDURE <A NAME="Caar"><procedure>Caar</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR l := interp.evalList(args.head);
  BEGIN
    IF l = NIL THEN
      RETURN interp.error(&quot;can't take \&quot;caar\&quot; of empty list&quot;)
    ELSE
      TYPECASE l.head OF
      | NULL =&gt;
          RETURN interp.error(
                   &quot;can't take \&quot;caar\&quot; of list when first element is nil&quot;)
      | List (first) =&gt; RETURN first.head
      ELSE
        RETURN
          interp.error(
            &quot;can't take \&quot;caar\&quot; of list when first element isn't a list&quot;)
      END;
    END;
  END Caar;

PROCEDURE <A NAME="Cadr"><procedure>Cadr</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR l := interp.evalList(args.head);
  BEGIN
    IF l = NIL THEN
      RETURN interp.error(&quot;can't take \&quot;cadr\&quot; of empty list&quot;)
    ELSIF l.tail = NIL THEN
      RETURN interp.error(&quot;can't take \&quot;cadr\&quot; of too short list&quot;)
    ELSE
      RETURN l.tail.head
    END;
  END Cadr;

PROCEDURE <A NAME="Cdar"><procedure>Cdar</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR l := interp.evalList(args.head);
  BEGIN
    IF l = NIL THEN
      RETURN interp.error(&quot;can't take \&quot;cdar\&quot; of empty list&quot;)
    ELSE
      TYPECASE l.head OF
      | NULL =&gt;
          RETURN interp.error(
                   &quot;can't take \&quot;cdar\&quot; of list when first element is nil&quot;)
      | List (first) =&gt; RETURN first.tail
      ELSE
        RETURN
          interp.error(
            &quot;can't take \&quot;cdar\&quot; of list when first element isn't a list&quot;)
      END;
    END;
  END Cdar;

PROCEDURE <A NAME="Cddr"><procedure>Cddr</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR l := interp.evalList(args.head);
  BEGIN
    IF l = NIL THEN
      RETURN interp.error(&quot;can't take \&quot;cddr\&quot; of empty list&quot;)
    ELSIF l.tail = NIL THEN
      RETURN interp.error(&quot;can't take \&quot;cddr\&quot; of too short list&quot;)
    ELSE
      RETURN l.tail.tail
    END;
  END Cddr;

PROCEDURE <A NAME="Concat"><procedure>Concat</procedure></A> (&lt;*UNUSED*&gt;self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR res := &quot;&quot;;
  BEGIN
    WHILE args # NIL DO
      res := res &amp; interp.evalString (args.head);
      args := args.tail; END;
    RETURN res;
  END Concat;

PROCEDURE <A NAME="Append"><procedure>Append</procedure></A> (&lt;*UNUSED*&gt;self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR res, last, l : RefList.T := NIL;
  BEGIN
    WHILE args # NIL DO
      l := interp.evalList (args.head);
      WHILE l # NIL DO
        IF last = NIL THEN
          last := RefList.List1 (l.head);
          res := last;
        ELSE
          last.tail := RefList.List1 (l.head);
          last := last.tail; END;
        l := l.tail; END;
      args := args.tail; END;
    RETURN res;
  END Append;

PROCEDURE <A NAME="LIst"><procedure>LIst</procedure></A> (&lt;*UNUSED*&gt;self: Builtin; interp: T; args: List): Sexp RAISES {Error} =
  VAR res, last: RefList.T := NIL;
  BEGIN
    WHILE args # NIL DO
      IF last = NIL THEN
        last := RefList.Cons (interp.eval (args.head), NIL);
        res := last;
      ELSE
        last.tail := RefList.Cons (interp.eval (args.head), NIL);
        last := last.tail; END;
      args := args.tail; END;
    RETURN res;
  END LIst;

PROCEDURE <A NAME="Length"><procedure>Length</procedure></A> (&lt;*UNUSED*&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  BEGIN
    TYPECASE interp.eval(args.head) OF
    | NULL =&gt; RETURN Sx.FromInt(0);
    | Symbol, Integer, Float =&gt; RETURN Sx.FromInt(1);
    | String (s) =&gt; RETURN Sx.FromInt(Text.Length(s));
    | List (l) =&gt; RETURN Sx.FromInt(RefList.Length(l));
    ELSE                         &lt;* ASSERT FALSE *&gt;
    END;
  END Length;

PROCEDURE <A NAME="SetProp"><procedure>SetProp</procedure></A> (&lt;* UNUSED *&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR
    key              := interp.eval(args.head);
    value            := interp.eval(args.tail.head);
    props            := interp.props;
    assoc: RefList.T;
  BEGIN
    WHILE props # NIL DO
      assoc := props.head;
      IF assoc.head = key THEN assoc.tail.head := value; RETURN value END;
      props := props.tail;
    END;
    interp.props := RefList.Cons(RefList.List2(key, value), interp.props);
    RETURN value;
  END SetProp;

PROCEDURE <A NAME="GetProp"><procedure>GetProp</procedure></A> (&lt;* UNUSED *&gt; self: Builtin; interp: T; args: List): Sexp
  RAISES {Error} =
  VAR
    key              := interp.eval(args.head);
    props            := interp.props;
    assoc: RefList.T;
  BEGIN
    WHILE props # NIL DO
      assoc := props.head;
      IF assoc.head = key THEN RETURN assoc.tail.head END;
      props := props.tail;
    END;
    RETURN NIL
  END GetProp;
</PRE>---------------------------------------------------------------- syntax ---

<P><PRE>VAR
  quoteParser     := NEW (Sx.ReadMacro, read := QuoteParser);

PROCEDURE <A NAME="QuoteParser"><procedure>QuoteParser</procedure></A> (&lt;*UNUSED*&gt; self: Sx.ReadMacro;
                       rd: Rd.T; syntax: Sx.Syntax): RefList.T
       RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    RETURN RefList.List1 (RefList.List2 (syms.quote, Sx.Read (rd, syntax)));
  END QuoteParser;

VAR
  backQuoteParser := NEW (Sx.ReadMacro, read := BackQuoteParser);
  backQuoteSyntax := Sx.CopySyntax ();

PROCEDURE <A NAME="ApplyBackQuote"><procedure>ApplyBackQuote</procedure></A> (s: Sexp): Sexp =
  BEGIN
    TYPECASE s OF
      | List (sl) =&gt;
           VAR res := NEW (List); BEGIN
             res.head := syms.append;
             ApplyBackQuoteList (sl, res);
             RETURN res; END;
      | Coma (c) =&gt; RETURN c.form;
      ELSE RETURN RefList.List2 (syms.quote, s); END;
  END ApplyBackQuote;

PROCEDURE <A NAME="ApplyBackQuoteList"><procedure>ApplyBackQuoteList</procedure></A> (l: List; rest: List) =
  BEGIN
    IF l = NIL THEN
      rest.tail := NIL;
    ELSE
      rest.tail := NEW (RefList.T);
      rest := rest.tail;
      TYPECASE l.head OF
        | NULL =&gt; rest.head := RefList.List2 (syms.list, NIL);
        | Coma (c) =&gt; rest.head := RefList.List2 (syms.list, c.form);
        | ComaAt (c) =&gt; rest.head := c.form;
        ELSE
          rest.head := RefList.List2 (syms.list,
                                     ApplyBackQuote (l.head)); END;

      ApplyBackQuoteList (l.tail, rest); END;
  END ApplyBackQuoteList;

PROCEDURE <A NAME="BackQuoteParser"><procedure>BackQuoteParser</procedure></A> (&lt;*UNUSED*&gt; self: Sx.ReadMacro;
                           rd: Rd.T; &lt;*UNUSED*&gt;syntax: Sx.Syntax): RefList.T
       RAISES {Sx.ReadError, Thread.Alerted} =
  BEGIN
    RETURN RefList.List1 (ApplyBackQuote (Sx.Read (rd, backQuoteSyntax)));
  END BackQuoteParser;

VAR
  comaParser      := NEW (Sx.ReadMacro, read := ComaParser);

TYPE
  Coma = BRANDED REF RECORD form: REFANY; END;
  ComaAt = BRANDED REF RECORD form: REFANY; END;

PROCEDURE <A NAME="ComaParser"><procedure>ComaParser</procedure></A> (&lt;*UNUSED*&gt; self: Sx.ReadMacro;
                      rd: Rd.T; &lt;*UNUSED*&gt; syntax: Sx.Syntax): RefList.T
       RAISES {Sx.ReadError, Thread.Alerted} =
  &lt;* FATAL Rd.Failure *&gt;
  BEGIN
    IF Rd.GetChar (rd) = '@' THEN
      RETURN RefList.List1 (NEW (ComaAt, form := Sx.Read (rd, backQuoteSyntax)));
    ELSE
      Rd.UnGetChar (rd);
      RETURN RefList.List1 (NEW (Coma, form := Sx.Read (rd, backQuoteSyntax))); END;
  END ComaParser;

VAR
  syntax          := Sx.CopySyntax ();
  syms: RECORD
          ampersandRest, ampersandEval, ampersandNoEval,
          list, append, quote, t, nil: Symbol; END;

PROCEDURE <A NAME="InitSyntax"><procedure>InitSyntax</procedure></A> () =
  BEGIN
    Sx.SetReadMacro (syntax, '\'', quoteParser);
    Sx.SetReadMacro (syntax, '`', backQuoteParser);

    Sx.SetReadMacro (backQuoteSyntax, ',', comaParser);
    Sx.SetReadMacro (backQuoteSyntax, '\'', quoteParser);

    syms.ampersandRest   := Aatom.FromText (&quot;THE_REST&quot;);
    syms.ampersandEval   := Aatom.FromText (&quot;_EVAL&quot;);
    syms.ampersandNoEval := Aatom.FromText (&quot;NO_EVAL&quot;);
    syms.list            := Aatom.FromText (&quot;list&quot;);
    syms.append          := Aatom.FromText (&quot;append&quot;);
    syms.quote           := Aatom.FromText (&quot;quote&quot;);
    syms.t               := Aatom.FromText (&quot;t&quot;);
    syms.nil             := Aatom.FromText (&quot;nil&quot;);
  END InitSyntax;

PROCEDURE <A NAME="Read"><procedure>Read</procedure></A> (rd: Reader): Sexp RAISES {Rd.EndOfFile, Sx.ReadError} =
  BEGIN
    RETURN Sx.Read (rd, syntax);
  END Read;

TYPE
  ReadMacro =
    Sx.ReadMacro OBJECT table: IntRefTbl.T OVERRIDES read := ReadList END;

PROCEDURE <A NAME="ReadList"><procedure>ReadList</procedure></A> (rm: ReadMacro; rd: Rd.T; s: Sx.Syntax): RefList.T
  RAISES {Sx.ReadError, Thread.Alerted} =
  (* Record the starting and ending positions of every list we read, so
     that we can highlight the list if there's an error. *)
  VAR
    start := Rd.Index(rd) - 1;
    form  := Sx.ReadDelimitedList(rd, ')', s);
    end   := Rd.Index(rd);
  BEGIN
    EVAL rm.table.put(start, NEW(Range, start := start, end := end, form := form));
    RETURN RefList.List1(form)
  END ReadList;

PROCEDURE <A NAME="ReadToTable"><procedure>ReadToTable</procedure></A> (rd: Reader; table: IntRefTbl.T): Sexp
  RAISES {Rd.EndOfFile, Sx.ReadError} =
  VAR tSyntax := Sx.CopySyntax(syntax);
  BEGIN
    IF table # NIL THEN
      Sx.SetReadMacro (tSyntax, '(', NEW (ReadMacro, table := table));
    END;
    RETURN Sx.Read(rd, tSyntax);
  END ReadToTable;

PROCEDURE <A NAME="Write"><procedure>Write</procedure></A> (wr: Writer; s: Sexp) =
  &lt;* FATAL Sx.PrintError *&gt;
  BEGIN
     SxPrint (wr, s);
  END Write;

PROCEDURE <A NAME="PushScope"><procedure>PushScope</procedure></A>(interp: T) =
  VAR newFrame := NEW(Frame, next := interp.frame, size := 0);
  BEGIN
    interp.frame := newFrame;
  END PushScope;

PROCEDURE <A NAME="PopScope"><procedure>PopScope</procedure></A>(interp: T) =
  BEGIN
    interp.frame := interp.frame.next;
  END PopScope;
</PRE>---------------------------------------------------------------------------
<PRE>PROCEDURE <A NAME="Copy"><procedure>Copy</procedure></A> (x: RefList.T): RefList.T =
  BEGIN
    RETURN RefList.Cons (x.head, Copy (x.tail));
  END Copy;

PROCEDURE <A NAME="SxToText"><procedure>SxToText</procedure></A>(sx: REFANY): TEXT RAISES {Sx.PrintError} =
  VAR wr: TextWr.T;
  BEGIN
    wr := TextWr.New();
    SxPrint(wr, sx);
    RETURN TextWr.ToText(wr);
  END SxToText;

CONST
  BAR           = '|';
  SQUOTE        = '\'';
  DQUOTE        = '&quot;';
  SLASH         = '\\';
  DIGITS        = SET OF CHAR {'0'.. '9'};
  LETTERS       = SET OF CHAR {'a'.. 'z', 'A'.. 'Z'};
  ALPHANUMERICS = LETTERS + DIGITS;

CONST
  ATOM_CHARS = SET OF
                 CHAR {
                 '!', '#', '$', '%', '&amp;', '*', '+', '-', '.', '/', ':', '&lt;',
                 '=', '&gt;', '?', '@', '[', ']', '^', '_', '{', '}', '~'};
  ID_CHARS = ALPHANUMERICS + SET OF CHAR {'_'};

PROCEDURE <A NAME="SxPrint"><procedure>SxPrint</procedure></A> (wr       : Wr.T;
                   sx       : Sx.T;
                   maxDepth : CARDINAL := LAST(CARDINAL);
                   maxLength: CARDINAL := LAST(CARDINAL)  )
  RAISES {Sx.PrintError} =
  &lt;* FATAL Wr.Failure, Thread.Alerted *&gt;
  CONST
    DEPTH_ELLIPSIS  = &quot;...&quot;;
    LENGTH_ELLIPSIS = &quot;...&quot;;
  BEGIN
    TYPECASE sx OF
    | NULL =&gt; Wr.PutText(wr, &quot;()&quot;)
    | REF INTEGER (r) =&gt; Wr.PutText(wr, Fmt.Int(r^))
    | REF CHAR (r) =&gt;
        Wr.PutChar(wr, SQUOTE);
        SxPrintChar(wr, r^, SQUOTE);
        Wr.PutChar(wr, SQUOTE)
    | REF REAL (r) =&gt;
        (* Wr.PutText (wr, Fmt.Real (r^, modula := TRUE)) *)
        Wr.PutText(wr, Fmt.Real(r^, Fmt.Style.Auto, literal := TRUE))
    | REF LONGREAL (r) =&gt;
        (* Wr.PutText (wr, Fmt.LongReal (r^, modula := TRUE)) *)
        Wr.PutText(wr, Fmt.LongReal(r^, Fmt.Style.Auto, literal := TRUE))
    | REF EXTENDED (r) =&gt;
        (* Wr.PutText (wr, Fmt.Extended (r^, modula := TRUE)) *)
        Wr.PutText(wr, Fmt.Extended(r^, Fmt.Style.Auto, literal := TRUE))
    | TEXT (t) =&gt;
        Wr.PutChar(wr, DQUOTE);
        FOR i := 0 TO Text.Length(t) - 1 DO
          SxPrintChar(wr, Text.GetChar(t, i), DQUOTE)
        END;
        Wr.PutChar(wr, DQUOTE)
    | Aatom.T (a) =&gt;
        VAR name := Aatom.ToText(a);
        BEGIN
          IF NeedsBars(name) THEN
            Wr.PutChar(wr, BAR);
            FOR i := 0 TO Text.Length(name) - 1 DO
              SxPrintChar(wr, Text.GetChar(name, i), BAR)
            END;
            Wr.PutChar(wr, BAR)
          ELSE
            Wr.PutText(wr, name)
          END
        END
    | RefList.T (list) =&gt;
        IF maxDepth = 0 THEN
          Wr.PutText(wr, DEPTH_ELLIPSIS)
        ELSE
          VAR len := maxLength;
          BEGIN
            Wr.PutChar(wr, '(');
            DEC(maxDepth);
            LOOP
              SxPrint(wr, list.head, maxDepth, maxLength);
              list := list.tail;
              IF list = NIL THEN EXIT END;
              Wr.PutChar(wr, ' ');
              IF len = 0 THEN Wr.PutText(wr, LENGTH_ELLIPSIS); EXIT END;
              DEC(len)
            END;
            Wr.PutChar(wr, ')')
          END
        END
    ELSE
      Wr.PutText(wr, Fmt.F(&quot;%s&lt;0x%s&gt;&quot;, RTTypeSRC.TypeName(sx),
                           Fmt.Unsigned(LOOPHOLE(sx, Word.T))));
    END
  END SxPrint;

PROCEDURE <A NAME="SxPrintChar"><procedure>SxPrintChar</procedure></A> (wr: Wr.T; ch: CHAR; delim: CHAR)
  RAISES {Wr.Failure, Thread.Alerted} =
  BEGIN
    IF ch = '\n' THEN
      Wr.PutText (wr, &quot;\\n&quot;)
    ELSIF ch = '\t' THEN
      Wr.PutText (wr, &quot;\\t&quot;)
    ELSIF ch = '\r' THEN
      Wr.PutText (wr, &quot;\\r&quot;)
    ELSIF ch = '\f' THEN
      Wr.PutText (wr, &quot;\\f&quot;)
    ELSIF ch = SLASH THEN
      Wr.PutText (wr, &quot;\\\\&quot;)
    ELSIF ch = delim THEN
      Wr.PutChar (wr, SLASH);
      Wr.PutChar (wr, ch)
    ELSIF ISO_Latin_printing (ch) THEN
      Wr.PutText (wr, Text.FromChar (ch))
    ELSE
      Wr.PutText (wr, Fmt.F (&quot;\\%03s&quot;, Fmt.Int (ORD (ch), 8)))
    END
  END SxPrintChar;

PROCEDURE <A NAME="ISO_Latin_printing"><procedure>ISO_Latin_printing</procedure></A> (ch: CHAR): BOOLEAN =
  BEGIN
    RETURN ' ' &lt;= ch AND ch &lt;= '~' OR '\241' &lt;= ch AND ch &lt;= '\377'
  END ISO_Latin_printing;

PROCEDURE <A NAME="NeedsBars"><procedure>NeedsBars</procedure></A> (t: TEXT): BOOLEAN =
  VAR
    len       := Text.Length (t);
    c  : CHAR;
  BEGIN
    IF len = 0 THEN RETURN TRUE END; (* || *)
    c := Text.GetChar (t, 0);
    IF c IN LETTERS THEN
      FOR i := 1 TO len - 1 DO
        c := Text.GetChar (t, i);
        IF NOT c IN ID_CHARS THEN RETURN TRUE END
      END;
      RETURN FALSE
    ELSIF c IN ATOM_CHARS THEN
      FOR i := 1 TO len - 1 DO
        c := Text.GetChar (t, i);
        IF NOT c IN ATOM_CHARS THEN RETURN TRUE END
      END;
      RETURN FALSE
    ELSE
      RETURN TRUE
    END
  END NeedsBars;
</PRE>---------------------------------------------------------------------------
      
<P><PRE>BEGIN
  InitSyntax ();
END SLisp.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
