<HTML>
<HEAD>
<TITLE>SRC Modula-3: smalldb/src/SmallDB.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>smalldb/src/SmallDB.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM> Implement the stable storage module. Michael B. Jones. 18-Jul-86 </EM></BLOCKQUOTE><PRE>

MODULE <module><implements><A HREF="SmallDB.i3">SmallDB</A></implements></module>;

IMPORT <A HREF="../../os/src/Common/FS.i3">FS</A>, <A HREF="../../rw/src/Common/FileRd.i3">FileRd</A>, <A HREF="../../rw/src/Common/FileWr.i3">FileWr</A>, <A HREF="../../os/src/Common/OSError.i3">OSError</A>, <A HREF="OSSupport.i3">OSSupport</A>;
IMPORT <A HREF="../../atom/src/Atom.i3">Atom</A>, <A HREF="../../libm3/derived/AtomList.i3">AtomList</A>, <A HREF="../../fmtlex/src/Lex.i3">Lex</A>, <A HREF="#x1">FloatMode</A>, <A HREF="../../fmtlex/src/Fmt.i3">Fmt</A>, <A HREF="../../time/src/Common/FmtTime.i3">FmtTime</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../text/src/Text.i3">Text</A>,
       <A HREF="../../rw/src/Common/TextRd.i3">TextRd</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../time/src/Common/Time.i3">Time</A>, <A HREF="../../rw/src/Common/Wr.i3">Wr</A>, <A HREF="../../word/src/Word.i3">Word</A>;

REVEAL
  <A NAME="T">T</A> = Public BRANDED OBJECT
    dirname: TEXT; (* Base directory name (no trailing '/' *)
    version: INTEGER := 0; (* Current snapshot and log version # *)
    logName: TEXT := NIL;
    logWriter: OSSupport.T := NIL; (* Writer to the current logfile *)
    (* statistics *)
    snapshotByteCnt, logByteCnt: CARDINAL := 0;
    logEntries: CARDINAL := 0;
    lastSnapshot, lastLog: Time.T;
    pad: BOOLEAN; (* if TRUE, pad each update to a DiskPageSize boundary *)
    cl: Closure;
  OVERRIDES
    recover := Recover;
    update := Update;
    snapshot := Snapshot;
    close := Close;
    snapshotBytes := SnapshotBytes;
    logBytes := LogBytes;
    status := Status;
  END;

CONST
  DirSeparator = &quot;/&quot;;
  SnapshotPrefix = &quot;Snapshot.&quot;;
  LogfilePrefix = &quot;Logfile.&quot;;
  VersionFile = &quot;Version_Number&quot;;
  NewVersionFile = &quot;New_Version_Number&quot;;

VAR BadUpdateLen, MalformedUpdate: AtomList.T;

&lt;* FATAL Thread.Alerted *&gt;
</PRE> exported procedures 

<P><PRE>PROCEDURE <A NAME="New"><procedure>New</procedure></A>(dir: TEXT; cl: Closure; pad: BOOLEAN := TRUE): T
     RAISES {OSError.E, Failed} =
  (* Returns a &quot;T&quot; for the data that is maintained in files in the directory
     &quot;dir&quot;. Raises an exception if the directory doesn't exist or is
     inaccessible. If the directory exists but contains no backing files,
     creates files corresponding to a NIL object with no updates. *)
  VAR t: T;
      ok: BOOLEAN := FALSE;
  BEGIN
    TRY
      ok := FS.Status(dir).type = FS.DirectoryFileType;
    EXCEPT
    | OSError.E =&gt;
    END;
    IF NOT ok THEN FS.CreateDirectory(dir); END;
    t := NEW(T, dirname := dir, pad := pad, cl := cl);
    t.lastSnapshot := 0.0D0;
    t.lastLog := 0.0D0;
    GetVersion(t);
    IF t.version = 0 THEN
       (* Commit to version 1, so that we have the log file initialized
	correctly (and atomically).  We can't just create the log file,
	since we'd need special care to make the creation atomic - it's
         easier to just use the mechanisms already present in Snapshot. *)
      Snapshot(t, cl.new());
    END;
    RETURN t;
  END New;
</PRE> private procedures 

<P><PRE>PROCEDURE <A NAME="FName"><procedure>FName</procedure></A>(t: T; n1: TEXT): TEXT =
  (* * Build a file name in a stable storage directory. *)
  BEGIN RETURN t.dirname &amp; DirSeparator &amp; n1; END FName;

PROCEDURE <A NAME="VersionName"><procedure>VersionName</procedure></A>(t: T; n1: TEXT; thisversion: INTEGER := 0): TEXT =
  (* * Build a file name in a stable storage directory. *)
  VAR version: INTEGER;
  BEGIN
    IF thisversion = 0 THEN
      version := t.version;
    ELSE
      version := thisversion;
    END;
    RETURN t.dirname &amp; DirSeparator &amp; n1 &amp; Fmt.Int(version);
  END VersionName;

PROCEDURE <A NAME="WriteVersionFile"><procedure>WriteVersionFile</procedure></A>(t: T; new: BOOLEAN) RAISES {OSError.E} =
  (* * Write a Version number file for t. *)
  VAR versionWriter: Wr.T; fname: TEXT;
  BEGIN
    TRY
      IF new THEN
        fname := FName(t, NewVersionFile);
      ELSE
        fname := FName(t, VersionFile);
      END;
      versionWriter := FileWr.Open(fname);
      Wr.PutText(versionWriter, Fmt.Int(t.version) &amp; &quot;\n&quot;);
      Wr.Close(versionWriter);
    EXCEPT
    | Wr.Failure(e) =&gt; RAISE OSError.E(e);
    END;
  END WriteVersionFile;

PROCEDURE <A NAME="DeleteNewVersionFile"><procedure>DeleteNewVersionFile</procedure></A>(t: T) RAISES {OSError.E} =
  (* * Delete a NewVersion number file for t. *)
  VAR fname: TEXT;
  BEGIN
    fname := FName(t, NewVersionFile);
    FS.DeleteFile(fname);
  END DeleteNewVersionFile;

PROCEDURE <A NAME="DeleteSnapshot"><procedure>DeleteSnapshot</procedure></A>(t: T; version: INTEGER) RAISES {OSError.E} =
  (* * Delete a NewVersion number file for t. *)
  VAR fname: TEXT;
  BEGIN
    IF version = 0 THEN RETURN; END; (* Never a version 0 *)
    fname := VersionName(t, SnapshotPrefix, version);
    FS.DeleteFile(fname);
  END DeleteSnapshot;

PROCEDURE <A NAME="DeleteLogfile"><procedure>DeleteLogfile</procedure></A>(t: T; version: INTEGER) RAISES {OSError.E} =
  (* * Delete a log file for t. *)
  VAR fname: TEXT;
  BEGIN
    IF version = 0 THEN RETURN; END; (* Never a version 0 *)
    fname := VersionName(t, LogfilePrefix, version);
    FS.DeleteFile(fname);
  END DeleteLogfile;

PROCEDURE <A NAME="CloseLogfile"><procedure>CloseLogfile</procedure></A>(t: T) RAISES {OSError.E} =
  (* * Close the log file for t. *)
  BEGIN
    IF t.logWriter = NIL THEN RETURN; END;
    TRY
      Wr.Close(t.logWriter);
    EXCEPT
    | Wr.Failure(e) =&gt;
        t.logWriter := NIL;
        RAISE OSError.E(e);
    END;
    t.logWriter := NIL;
  END CloseLogfile;

PROCEDURE <A NAME="OpenLogfile"><procedure>OpenLogfile</procedure></A>(t: T; truncate: BOOLEAN) RAISES {OSError.E} =
  (* * Open the log file for writing at byte number byte. *)
  VAR fname: TEXT;
  BEGIN
    TRY
      CloseLogfile(t); (* Close any open log file *)
    EXCEPT
    | OSError.E =&gt; (* Assume can proceed ok *)
    END;

    fname := VersionName(t, LogfilePrefix);
    t.logName := fname;
    t.logWriter := NEW(OSSupport.T).init(FS.OpenFile(fname, truncate));
    (* force changes to disk ??? *)
  END OpenLogfile;

PROCEDURE <A NAME="CreateFirstVersion"><procedure>CreateFirstVersion</procedure></A>(t: T) RAISES {OSError.E} =
  (* * Initialize a stable storage directory. *)
  BEGIN
    t.version := 0;
    (* No snapshot file for version 0 *)
    WriteVersionFile(t, FALSE);
  END CreateFirstVersion;

PROCEDURE <A NAME="IncrVersion"><procedure>IncrVersion</procedure></A>(t: T) =
  (* * Increment the directory version number. *)
  BEGIN
    REPEAT t.version := t.version + 1; UNTIL t.version # 0;
    (* No snapshot file for version 0 *)
  END IncrVersion;

PROCEDURE <A NAME="CommitToNewVersion"><procedure>CommitToNewVersion</procedure></A>(t: T) RAISES {OSError.E} =
  (* * Take us from the state where we have a NewVersion file to where we
     don't. *)
  BEGIN
    WriteVersionFile(t, FALSE);
    DeleteNewVersionFile(t);
  END CommitToNewVersion;

PROCEDURE <A NAME="GetVersion"><procedure>GetVersion</procedure></A>(t: T) RAISES {OSError.E} =
  (* * Determines the current version number for the directory t. *)
  VAR versionReader: Rd.T;
  VAR version: INTEGER;
  BEGIN
    TRY
      versionReader := FileRd.Open(FName(t, NewVersionFile));
      TRY
        version := Lex.Int(versionReader);
        IF version &lt;= 0 THEN
          RAISE Lex.Error; (* No number! *)
        END;
      FINALLY
        Rd.Close(versionReader);
      END;
      t.version := version;
      CommitToNewVersion(t);
    EXCEPT
    | OSError.E, Lex.Error, Rd.Failure, FloatMode.Trap =&gt;

        TRY
          DeleteNewVersionFile(t); (* Make SURE it's not there *)
        EXCEPT
        | OSError.E =&gt;
        END;

        TRY
          versionReader := FileRd.Open(FName(t, VersionFile));
          TRY
            version := Lex.Int(versionReader);
            IF version &lt;= 0 THEN
              RAISE Lex.Error; (* No number! *)
            END;
          FINALLY
            Rd.Close(versionReader);
          END;
          t.version := version;
        EXCEPT
        | OSError.E, FloatMode.Trap, Lex.Error, Rd.Failure =&gt;
            CreateFirstVersion(t);
        END;
    END;
  END GetVersion;

PROCEDURE <A NAME="Snapshot"><procedure>Snapshot</procedure></A>(t: T; value: REFANY) RAISES {OSError.E} =
  (* Records this value as the current snapshot, and empties the log. *)
  VAR
    snapshotWriter: Wr.T;
    oldversion: INTEGER;
    fname: TEXT;
  BEGIN
    oldversion := t.version;
    IncrVersion(t);
    TRY
      fname := VersionName(t, SnapshotPrefix);
      snapshotWriter := FileWr.Open(fname);
      TRY
        t.cl.snapshot(snapshotWriter, value);
        t.snapshotByteCnt := Wr.Length(snapshotWriter);
        t.lastSnapshot := Time.Now();
      FINALLY
        Wr.Close(snapshotWriter);
      END;
    EXCEPT
    | Wr.Failure(e) =&gt; RAISE OSError.E(e);
    END;

    OpenLogfile(t, TRUE);
    t.logByteCnt := 0;
    t.logEntries := 0;
    WriteVersionFile(t, TRUE);
    CommitToNewVersion(t);
    DeleteSnapshot(t, oldversion);
    DeleteLogfile(t, oldversion);
  END Snapshot;

PROCEDURE <A NAME="Recover"><procedure>Recover</procedure></A>(t: T) : REFANY RAISES {OSError.E, Failed} =
  (* Returns a REFANY which is the value recorded in the current snapshot. *)
  VAR
    snapshotReader: Rd.T;
    snapshot: REFANY;
    fname: TEXT;
  BEGIN
    IF t.version = 0 THEN RETURN NIL END;
    TRY
      fname := VersionName(t, SnapshotPrefix);
      snapshotReader := FileRd.Open(fname);
      TRY
        snapshot := t.cl.recover(snapshotReader);
        t.snapshotByteCnt := Rd.Length(snapshotReader);
      FINALLY
        Rd.Close(snapshotReader);
      END;
    EXCEPT
    | Rd.Failure(e) =&gt; RAISE OSError.E(e);
    END;
    RETURN RecoverUpdates(t, snapshot);
  END Recover;

CONST
  DiskPageSize = 512;
  IntBytes = 4;   (* writing only 32 bits of update length *)
  ByteBits = BITSIZE(CHAR);

PROCEDURE <A NAME="Update"><procedure>Update</procedure></A>(t: T; update: REFANY; forceToDisk: BOOLEAN := TRUE )
    RAISES {OSError.E} =
  (* Records this update in the log file for &quot;t&quot; *)
  VAR
    updateLen: Word.T;
    entryStart, entryEnd: CARDINAL;
  BEGIN

    &lt;* ASSERT t.logWriter # NIL *&gt;
    TRY
      entryStart := Wr.Index(t.logWriter);
      FOR i := 0 TO IntBytes - 1 DO (* Write zero update len *)
        Wr.PutChar(t.logWriter, VAL(0, CHAR)); (* Chars of update len *)
      END;

      t.cl.logUpdate(t.logWriter, update);

      entryEnd := Wr.Index(t.logWriter);
      updateLen := (entryEnd - entryStart) - IntBytes;

      IF forceToDisk THEN (* force update contents before updating length *)
        Wr.Flush(t.logWriter);
        OSSupport.Sync(t.logWriter);
      END;
      Wr.Seek(t.logWriter, entryStart);
      FOR i := 0 TO IntBytes-1 DO (* Write real update len *)
        Wr.PutChar(t.logWriter,
           VAL(Word.Extract(updateLen, i*ByteBits, ByteBits), CHAR));
      END;
      Wr.Seek(t.logWriter, entryEnd);

      (* don't start an entry at the end of page *)
      IF t.pad OR (entryEnd MOD DiskPageSize &gt; DiskPageSize-IntBytes) THEN
        WHILE (Wr.Index(t.logWriter) MOD DiskPageSize) # 0 DO
          Wr.PutChar(t.logWriter, VAL(0, CHAR)); (* Fill to page boundary *)
        END;
      END;

      IF forceToDisk THEN
        Wr.Flush(t.logWriter);
        OSSupport.Sync(t.logWriter);
        (* force changes to disk *)
      END;
      t.logByteCnt := Wr.Index(t.logWriter);
      t.lastLog := Time.Now();
      INC(t.logEntries);
    EXCEPT
    | Wr.Failure(e) =&gt; RAISE OSError.E(e);
    END;
  END Update;

PROCEDURE <A NAME="RecoverUpdates"><procedure>RecoverUpdates</procedure></A>(t: T; state: REFANY): REFANY
    RAISES {OSError.E, Failed} =
  VAR
    logReader: Rd.T;
    updateLen: Word.T;
    fname: TEXT;
    updateText: TEXT;
    pos: CARDINAL;
  BEGIN
    IF t.version = 0 THEN RETURN state; END;
    TRY
      fname := VersionName(t, LogfilePrefix);
      logReader := FileRd.Open(fname);
      TRY
        LOOP
          IF Rd.EOF(logReader) THEN EXIT END;

          updateLen := 0;
          FOR i := 0 TO IntBytes - 1 DO
            TRY
              updateLen := Word.Insert(
                updateLen, ORD(Rd.GetChar(logReader)), i*ByteBits, ByteBits);
            EXCEPT
            | Rd.EndOfFile =&gt; RAISE Failed(MalformedUpdate);
            END;
          END;

          IF updateLen = 0 THEN
            EXIT (* crashed while writing last log entry *)
          END;

          IF (updateLen &lt; 0) THEN RAISE Failed(BadUpdateLen); END;

          updateText := Rd.GetText(logReader, updateLen);
          IF Text.Length(updateText) # updateLen THEN
            RAISE Failed(BadUpdateLen);
          END;

          state := t.cl.readUpdate(TextRd.New(updateText), state);

          t.logByteCnt := Rd.Index(logReader);
          pos := ((t.logByteCnt+DiskPageSize-1) DIV DiskPageSize) * DiskPageSize;
          IF t.pad OR ((pos-t.logByteCnt) &lt; IntBytes) THEN
            Rd.Seek(logReader, pos); (* Seek to page boundary *)
            t.logByteCnt := pos;
          END;
          INC(t.logEntries);
        END;
      FINALLY
        Rd.Close(logReader);
      END;
    EXCEPT
    | Rd.Failure(e) =&gt; RAISE OSError.E(e);
    END;

    (* now reopen the file at the end *)
    OpenLogfile(t, FALSE); (* Prepare for possible appending *)
      (* this code truncates the log file, in case the
         last entry was incompletely written *)

    TRY
      Wr.Seek(t.logWriter, t.logByteCnt);
      OSSupport.Truncate(t.logWriter);
      IF t.pad OR (t.logByteCnt MOD DiskPageSize &gt; DiskPageSize-IntBytes) THEN
        WHILE (Wr.Index(t.logWriter) MOD DiskPageSize) # 0 DO
          Wr.PutChar(t.logWriter, VAL(0, CHAR)); (* Fill to page boundary *)
        END;
      END;
    EXCEPT
    | Wr.Failure(e) =&gt; RAISE OSError.E(e);
    END;
    RETURN state;
  END RecoverUpdates;

PROCEDURE <A NAME="Close"><procedure>Close</procedure></A>(t: T) RAISES {OSError.E} =
  (* Close a stable storage directory in an orderly manner *)
  BEGIN CloseLogfile(t); END Close;

PROCEDURE <A NAME="SnapshotBytes"><procedure>SnapshotBytes</procedure></A>(t: T): CARDINAL =
  BEGIN RETURN t.snapshotByteCnt; END SnapshotBytes;

PROCEDURE <A NAME="LogBytes"><procedure>LogBytes</procedure></A>(t: T): CARDINAL = BEGIN RETURN t.logByteCnt; END LogBytes;

PROCEDURE <A NAME="Status"><procedure>Status</procedure></A>(t: T) : TEXT =
  VAR out: TEXT;
  BEGIN
    out := &quot;   Stable storage status for directory \&quot;&quot; &amp; t.dirname &amp; &quot;\&quot;\n&quot;;
    out := out &amp; &quot;      Snapshot version &quot; &amp; Fmt.Int(t.version)
              &amp; &quot;, size &quot; &amp; Fmt.Int(t.snapshotByteCnt) &amp; &quot; bytes.\n&quot;;
    IF t.lastSnapshot # 0.0D0 THEN
      out := out &amp; &quot;      Last snapshot was written at &quot; &amp;
         FmtTime.Long(t.lastSnapshot) &amp; &quot;\n&quot;;
    ELSE
      out := out &amp; &quot;      No snapshot has been made in this run.\n&quot;;
    END;
    out := out &amp; &quot;      Log has &quot; &amp; Fmt.Int(t.logEntries)
               &amp; &quot; entries, total size &quot;
               &amp; Fmt.Int(t.logByteCnt) &amp; &quot; bytes.\n&quot;;
    IF t.lastLog # 0.0D0 THEN
      out := out &amp; &quot;      Last log entry was written at &quot; &amp;
         FmtTime.Long(t.lastLog) &amp; &quot;\n&quot;;
    ELSE
      out := out &amp; &quot;      No log entries have been written in this run.\n&quot;;
    END;
    RETURN out;
  END Status;

BEGIN
  BadUpdateLen := AtomList.List1(Atom.FromText(&quot;SmallDB.BadUpdateLen&quot;));
  MalformedUpdate := AtomList.List1(Atom.FromText(&quot;SmallDB.MalformedUpdate&quot;));
END SmallDB.
</PRE>
</inModule>
<HR>
<A NAME="x1">interface FloatMode is in:
</A><UL>
<LI><A HREF="../../float/src/DS3100/FloatMode.i3#0TOP0">float/src/DS3100/FloatMode.i3</A>
<LI><A HREF="../../float/src/IEEE-default/FloatMode.i3#0TOP0">float/src/IEEE-default/FloatMode.i3</A>
<LI><A HREF="../../float/src/IRIX5/FloatMode.i3#0TOP0">float/src/IRIX5/FloatMode.i3</A>
<LI><A HREF="../../float/src/SOLsun/FloatMode.i3#0TOP0">float/src/SOLsun/FloatMode.i3</A>
<LI><A HREF="../../float/src/SPARC/FloatMode.i3#0TOP0">float/src/SPARC/FloatMode.i3</A>
<LI><A HREF="../../float/src/SUN386/FloatMode.i3#0TOP0">float/src/SUN386/FloatMode.i3</A>
<LI><A HREF="../../float/src/VAX/FloatMode.i3#0TOP0">float/src/VAX/FloatMode.i3</A>
</UL>
<P>
<PRE>























</PRE>
</BODY>
</HTML>
