<HTML>
<HEAD>
<TITLE>SRC Modula-3: stable/src/LogManager.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>stable/src/LogManager.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                             </EM></BLOCKQUOTE><PRE>
</PRE> Created by Carsten Weich                                    
                                                             

<P> The <CODE>LogManager</CODE> provides an object which methods take a <CODE>Pathname.T</CODE>
   and produce readers and writers for log and checkpoint files.  It is
   used by stable objects.
<P>
   The default log mananger uses the <CODE>nm</CODE> paramter as the name of a
   directory in the filesystem.  In this directory it will put the log and
   checkpointfiles. 

<P><PRE>MODULE <module><implements><A HREF="LogManager.i3">LogManager</A></implements></module>;

IMPORT <A HREF="../../os/src/Common/Pathname.i3">Pathname</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../rw/src/Common/Wr.i3">Wr</A>, <A HREF="../../rw/src/Common/FileWr.i3">FileWr</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../rw/src/Common/FileRd.i3">FileRd</A>, <A HREF="../../os/src/Common/FS.i3">FS</A>, <A HREF="../../os/src/Common/OSError.i3">OSError</A>, <A HREF="../../os/src/POSIX/OSErrorPosix.i3">OSErrorPosix</A>,
       <A HREF="../../atom/src/Atom.i3">Atom</A>, <A HREF="../../libm3/derived/AtomList.i3">AtomList</A>;

IMPORT <A HREF="Log.i3">Log</A>;
FROM <A HREF="Log.i3">Log</A> IMPORT CrashPoint;

REVEAL <A NAME="T">T</A> = Public BRANDED OBJECT END;

REVEAL
  <A NAME="Default">Default</A> = DefaultPublic BRANDED OBJECT
    METHODS
      (* produce the complete filenames: *)
      cpfn    (nm: Pathname.T): Pathname.T RAISES {OSError.E} := Cpfn;
      newcpfn (nm: Pathname.T): Pathname.T RAISES {OSError.E} := NewCpfn;
      oldcpfn (nm: Pathname.T): Pathname.T RAISES {OSError.E} := OldCpfn;
      dummycp (nm: Pathname.T): Pathname.T RAISES {OSError.E} := DummyCp;
      logfn   (nm: Pathname.T): Pathname.T RAISES {OSError.E} := Logfn;

    OVERRIDES
      init := Init;
      reOpenLog       := ReOpenLog;
      beginCheckpoint := BeginCheckpoint;
      endCheckpoint   := EndCheckpoint;
      recover         := Recover;
      recoverable     := Recoverable;
      emptyLog        := EmptyLog;
      dispose         := Dispose;
    END;
</PRE> \subsection{produce filenames}
 

<P><PRE>PROCEDURE <A NAME="Logfn"><procedure>Logfn</procedure></A> (&lt;*UNUSED*&gt; lm: Default; nm: Pathname.T): Pathname.T =
  (* log filename *)
  BEGIN
    RETURN Pathname.Join(nm, &quot;log&quot;, NIL)
  END Logfn;

PROCEDURE <A NAME="Cpfn"><procedure>Cpfn</procedure></A> (&lt;*UNUSED*&gt; lm: Default; nm: Pathname.T): Pathname.T =
  (* checkpoint filename *)
  BEGIN
    RETURN Pathname.Join(nm, &quot;checkpoint&quot;, NIL)
  END Cpfn;

PROCEDURE <A NAME="OldCpfn"><procedure>OldCpfn</procedure></A> (lm: Default; nm: Pathname.T): Pathname.T
  RAISES {OSError.E} =
  (* old (backup) checkpoint filename *)
  BEGIN
    RETURN lm.cpfn(nm) &amp; &quot;.old&quot;
  END OldCpfn;

PROCEDURE <A NAME="NewCpfn"><procedure>NewCpfn</procedure></A> (lm: Default; nm: Pathname.T): Pathname.T
  RAISES {OSError.E} =
  (* new checkpoint filename *)
  BEGIN
    RETURN lm.cpfn(nm) &amp; &quot;.new&quot;
  END NewCpfn;

PROCEDURE <A NAME="DummyCp"><procedure>DummyCp</procedure></A> (lm: Default; nm: Pathname.T): Pathname.T
  RAISES {OSError.E} =
  (* dummy checkpoint filename for bootstrapping *)
  BEGIN
    RETURN lm.cpfn(nm) &amp; &quot;.dummy&quot;
  END DummyCp;
</PRE> \subsection{Public Methods}
 

<P><PRE>PROCEDURE <A NAME="Init"><procedure>Init</procedure></A>(lm: Default): Default =
</PRE><BLOCKQUOTE><EM> No dynamic initialization needed: </EM></BLOCKQUOTE><PRE>
  BEGIN
    RETURN lm;
  END Init;

PROCEDURE <A NAME="ReOpenLog"><procedure>ReOpenLog</procedure></A> (lm: Default; nm: Pathname.T): Wr.T
  RAISES {OSError.E} =
  BEGIN
    RETURN FileWr.OpenAppend(lm.logfn(nm))
  END ReOpenLog;

PROCEDURE <A NAME="BeginCheckpoint"><procedure>BeginCheckpoint</procedure></A> (lm: Default; nm: Pathname.T): Wr.T
  RAISES {OSError.E} =
  BEGIN
    Log.PutText(&quot;BeginCheckpoint: &quot;);
    (**)
    CrashPoint(101);
    TestDir(nm);
    IF TestFile(lm.cpfn(nm)) THEN
      FS.Rename(lm.cpfn(nm), lm.oldcpfn(nm));
      Log.PutText(&quot;renamed checkpointfile to &quot; &amp; lm.oldcpfn(nm));
      Log.Nl();
    ELSE
      TRY
        &lt;*FATAL Thread.Alerted*&gt;
        BEGIN
          Wr.Close(FileWr.Open(lm.dummycp(nm)));
        END
      EXCEPT
        Wr.Failure =&gt;
          RAISE
            OSError.E(AtomList.List1(
                        Atom.FromText(&quot;error creating dummy checkpoint &quot;
                                        &amp; lm.dummycp(nm))));
      END;
      Log.PutText(&quot;created dummy checkpoint &quot; &amp; lm.dummycp(nm));
      Log.Nl();
    END;
    Log.PutText(&quot;new checkpointfile &quot; &amp; lm.newcpfn(nm));
    Log.Nl();
    RETURN FileWr.Open(lm.newcpfn(nm));
  END BeginCheckpoint;

PROCEDURE <A NAME="EndCheckpoint"><procedure>EndCheckpoint</procedure></A> (lm: Default; nm: Pathname.T): Wr.T
  RAISES {OSError.E} =
  BEGIN
    Log.PutText(&quot;EndCheckpoint: &quot;);
    Log.PutText(&quot;renaming checkpoint &quot; &amp; lm.newcpfn(nm));
    Log.Nl();
    (**)
    CrashPoint(103);
    FS.Rename(lm.newcpfn(nm), lm.cpfn(nm));
    (**)
    CrashPoint(104);
    VAR newLog := FileWr.Open(lm.logfn(nm));
    BEGIN
      IF TestFile(lm.dummycp(nm)) THEN
        Log.PutText(&quot;deleting dummy checkpoint&quot; &amp; lm.dummycp(nm));
        Log.Nl();
        (**)
        CrashPoint(105);
        FS.DeleteFile(lm.dummycp(nm));
      ELSE
        Log.PutText(&quot;deleting old checkpoint&quot; &amp; lm.oldcpfn(nm));
        Log.Nl();
        (**)
        CrashPoint(105);
        FS.DeleteFile(lm.oldcpfn(nm));
      END;
      (**)
      CrashPoint(106);
      RETURN newLog;
    END;
  END EndCheckpoint;

PROCEDURE <A NAME="Recover"><procedure>Recover</procedure></A> (lm: Default; nm: Pathname.T; VAR log, checkp: Rd.T)
  RAISES {OSError.E} =
  (* Check possible leftovers from checkpoint-crashes: This procedure
     tolerates absence of logfiles since itself deletes logfiles when
     invalid.  Usually there are always logfiles (with possible size zero)
     since there is no way of deleting logfiles using the default
     logmanager. *)

  BEGIN
    (* (1) renaming to final name is the commit for writing a new
       checkpoint.  If there is a checkpoint with temporary name then it is
       invalid and the log file belongs to the old version. *)
    IF TestFile(lm.newcpfn(nm)) OR NOT TestFile(lm.cpfn(nm)) THEN
      Log.PutText(&quot;found valid backup&quot;);
      Log.Nl();
      FS.Rename(lm.oldcpfn(nm), lm.cpfn(nm)); (* must exist or fatal
                                                 error *)
      IF TestFile(lm.newcpfn(nm)) THEN
        Log.PutText(&quot;removing new checkpoint&quot;);
        Log.Nl();
        FS.DeleteFile(lm.newcpfn(nm))
      END;
      IF NOT TestFile(lm.logfn(nm)) THEN
        log := NIL
      ELSE
        log := FileRd.Open(lm.logfn(nm));
      END;

      (* (2) there is a checkpoint with final name.  Thus it is valid.  If
         there is a backup checkpoint than the log is invalid. *)
    ELSIF TestFile(lm.cpfn(nm)) AND TestFile(lm.oldcpfn(nm)) THEN
      Log.PutText(&quot;found valid new checkpoint&quot;);
      Log.Nl();
      Log.PutText(&quot;deleting backup checkpoint &quot; &amp; lm.oldcpfn(nm));
      FS.DeleteFile(lm.oldcpfn(nm)); (* must exist of fatal error *)
      IF TestFile(lm.logfn(nm)) THEN (* may or may not exist depending on
                                        the crash *)
        Log.PutText(&quot;removing log&quot;);
        Log.Nl();
        FS.DeleteFile(lm.logfn(nm));
      END;
      log := NIL;

      (* (3) Finally if there is no temporary checkpoint there than the
         checkpoint and the log is valid. *)
    ELSE
      IF NOT TestFile(lm.logfn(nm)) THEN
        log := NIL
      ELSE
        log := FileRd.Open(lm.logfn(nm));
      END;
    END;
    checkp := FileRd.Open(lm.cpfn(nm));
  END Recover;

PROCEDURE <A NAME="Recoverable"><procedure>Recoverable</procedure></A> (lm: Default; nm: Pathname.T): BOOLEAN =
  BEGIN
    TRY
      (* is there such a directory? *)
      IF FS.Status(nm).type # FS.DirectoryFileType THEN RETURN FALSE END;

      (* Do not care about to logfile but there must be a version of the
         checkpointfile: *)
      RETURN TestFile(lm.cpfn(nm)) OR TestFile(lm.oldcpfn(nm));

    EXCEPT
      OSError.E =&gt; RETURN FALSE  (* nothing found *)
    END;
  END Recoverable;

PROCEDURE <A NAME="EmptyLog"><procedure>EmptyLog</procedure></A> (lm: Default; nm: Pathname.T): BOOLEAN
  RAISES {OSError.E} =
  BEGIN
    IF NOT lm.recoverable(nm) THEN
      RAISE OSError.E(
              AtomList.List1(
                Atom.FromText(
                  &quot;no checkpointfile for log in &quot; &amp; nm)));
    END;
    IF TestFile(lm.logfn(nm)) THEN
      RETURN FS.Status(nm).size &gt; 0
    ELSE
      RETURN TRUE
    END;
  END EmptyLog;

PROCEDURE <A NAME="Dispose"><procedure>Dispose</procedure></A> (lm: Default; nm: Pathname.T) RAISES {OSError.E} =
  BEGIN
    IF TestFile(nm) THEN
      Log.PutText(&quot;Erasing directory &quot; &amp; nm);
      Log.Nl();
      (**)
      CrashPoint(501);
      IF TestFile(lm.cpfn(nm)) THEN FS.DeleteFile(lm.cpfn(nm)) END;
      IF TestFile(lm.oldcpfn(nm)) THEN FS.DeleteFile(lm.oldcpfn(nm)) END;
      IF TestFile(lm.newcpfn(nm)) THEN FS.DeleteFile(lm.newcpfn(nm)) END;
      (**)
      CrashPoint(502);
      IF TestFile(lm.logfn(nm)) THEN FS.DeleteFile(lm.logfn(nm)) END;
      (**)
      CrashPoint(503);
      IF TestFile(lm.dummycp(nm)) THEN FS.DeleteFile(lm.dummycp(nm)) END;
      FS.DeleteDirectory(nm);
    END
  END Dispose;
</PRE> \subsection{Utilities}
 

<P><PRE>PROCEDURE <A NAME="TestFile"><procedure>TestFile</procedure></A> (fname: Pathname.T): BOOLEAN RAISES {OSError.E} =
  (* Test for existence of a file named 'fname'.  Returns TRUE iff it
     exists and is readable.  Exceptions other than &quot;file not found&quot;# are
     reraised. *)
  CONST Enoent = 2;              (* Unix-errno for file not found *)
  BEGIN
    TRY
      EVAL FS.Status(fname);
      RETURN TRUE;               (* ok, passed the test *)
    EXCEPT
      OSError.E (err) =&gt;
        IF err.head = OSErrorPosix.ErrnoAtom(Enoent) THEN
          RETURN FALSE
        ELSE
          RAISE OSError.E(err)
        END
    END
  END TestFile;

PROCEDURE <A NAME="TestDir"><procedure>TestDir</procedure></A> (dir: Pathname.T) RAISES {OSError.E} =
  (* This procedure assures that &quot;dir&quot; is a valid directory.  If it does
     not exist it creates it. *)
  BEGIN
    TRY
      IF FS.Status(dir).type # FS.DirectoryFileType THEN
        RAISE
          OSError.E(AtomList.List1(
                      Atom.FromText(
                        &quot;name of stable obj is not a directory: &quot; &amp; dir)));
      END
    EXCEPT
      OSError.E =&gt;
        Log.PutText(&quot;creating dir &quot; &amp; dir);
        Log.Nl();
        FS.CreateDirectory(dir)
    END
  END TestDir;

BEGIN
  default:= NEW(Default).init();
END LogManager.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
