<HTML>
<HEAD>
<TITLE>SRC Modula-3: stable/src/StableRep.mg</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>stable/src/StableRep.mg</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                             </EM></BLOCKQUOTE><PRE>
</PRE> Created by Carsten Weich                                    
                                                             

<P> This generic module provides the part of the implementation of stable
   objects that is independent of {\tt Data.T}.


<P><PRE>GENERIC MODULE <genericModule><A HREF="StableRep.ig">StableRep</A></genericModule>(StableData);

IMPORT <A HREF="StableError.i3">StableError</A>, <A HREF="LogManager.i3">LogManager</A>, <A HREF="../../os/src/Common/Pathname.i3">Pathname</A>, <A HREF="../../pickle/src/Pickle.i3">Pickle</A>;
IMPORT <A HREF="../../os/src/Common/OSError.i3">OSError</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../rw/src/Common/Wr.i3">Wr</A>, <A HREF="../../rw/src/Common/RdUtils.i3">RdUtils</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../atom/src/Atom.i3">Atom</A>, <A HREF="../../libm3/derived/AtomList.i3">AtomList</A>;

IMPORT <A HREF="Log.i3">Log</A>;

&lt;*FATAL Thread.Alerted*&gt;
</PRE> \subsection{Initialization Procedures} 

<P><PRE>PROCEDURE <A NAME="ReOpenLog"><procedure>ReOpenLog</procedure></A> (self: StableData.T) =
  BEGIN
    TRY
      IF self.log = NIL THEN
        self.log := self.lm.reOpenLog(self.nm)
      END
    EXCEPT
      OSError.E (err) =&gt;
        StableError.Halt(&quot;cannot reopen log file: &quot;
                           &amp; RdUtils.FailureText(err))
    END
  END ReOpenLog;

PROCEDURE <A NAME="FlushLog"><procedure>FlushLog</procedure></A> (self: Public) RAISES {StableError.E} =
  BEGIN
    TRY
      IF self.log # NIL THEN Wr.Flush(self.log) END
    EXCEPT
      Wr.Failure (err) =&gt; RAISE StableError.E(err);
    END
  END FlushLog;

PROCEDURE <A NAME="FreeLog"><procedure>FreeLog</procedure></A> (self: Public) RAISES {StableError.E} =
  BEGIN
    TRY
      IF self.log # NIL THEN Wr.Close(self.log) END;
      self.log := NIL;
    EXCEPT
      Wr.Failure (err) =&gt; RAISE StableError.E(err);
    END
  END FreeLog;
</PRE> \subsection{Procedure Init}
   Initialize the <CODE>nm</CODE> and <CODE>lm</CODE> fields for the stable object first
   (this is needed for the <CODE>Recover()</CODE> procedure.
<P>
   Recover if possible and set <CODE>recovered</CODE> to TRUE. Call <CODE>Checkpoint</CODE>
   if <CODE>Init</CODE> is called the first time (i.e. if <CODE>recoverable=FALSE</CODE>)
   or if recovered from a crash (i.e. if <CODE>emptyLog=FALSE</CODE>).
   This might be saver than necessary. I assume that the <CODE>Recover</CODE>
   procedure might consume the log file, so I test <CODE>emptyLog</CODE> before
   running <CODE>Recover()</CODE>.
<P>
   Set the <CODE>forceToDisk</CODE> field after a possible recover. This is
   to set the field to a fresh value -- do not leave as it was
   when the stable object was saved the last time.

<PRE>PROCEDURE <A NAME="Init"><procedure>Init</procedure></A> (    self       : Public;
                    nm        : Pathname.T;
                VAR recovered  : BOOLEAN;
                    forceToDisk               := TRUE;
                    lm         : LogManager.T := NIL   ): StableData.T
  RAISES {StableError.E} =
  BEGIN
    IF self.lm # NIL THEN
      RAISE StableError.E(
          AtomList.List1(
            Atom.FromText(
              &quot;attempted to restabilize without intervening dispose&quot;)))
    END;
    TRY
      self.nm := nm;
      IF lm = NIL THEN
        self.lm := LogManager.default
      ELSE
        self.lm := lm
      END;
      VAR emptyLog:= TRUE;
      BEGIN
        IF self.lm.recoverable(nm) THEN
          emptyLog:= self.lm.emptyLog(nm);
          self := Recover(self);
          recovered := TRUE;
        ELSE
          recovered := FALSE
        END;
        IF NOT recovered OR NOT emptyLog THEN Checkpoint(self) END;
        self.forceToDisk := forceToDisk;
        RETURN NARROW(self, StableData.T)
      END
    EXCEPT
      OSError.E (err) =&gt; RAISE StableError.E(err);
    | Wr.Failure (err) =&gt; RAISE StableError.E(err);
    END
  END Init;

PROCEDURE <A NAME="Dispose"><procedure>Dispose</procedure></A> (self: Public) RAISES {StableError.E} =
  BEGIN
    IF (self.lm # NIL) THEN
      TRY
        self.freeLog();
        self.lm.dispose(self.nm);
        self.lm := NIL;
      EXCEPT
        OSError.E (err) =&gt; RAISE StableError.E(err);
      END
    END
  END Dispose;
</PRE> \subsection{Procedure Recover}
  <CODE>t</CODE> already has a <CODE>nm</CODE> value, a log manager and a
     valid <CODE>readCheckpoint</CODE> method.  <CODE>Recover</CODE> will use
     those to recover from <CODE>t.nm</CODE>.


<P><PRE>PROCEDURE <A NAME="Recover"><procedure>Recover</procedure></A> (t: StableData.T): StableData.T
  RAISES {StableError.E, OSError.E} =

  VAR log, cp: Rd.T;
  BEGIN
    t.lm.recover(t.nm, log, cp); (* might open log *)
    TRY
      t := t.readCheckpoint(cp);
      Rd.Close(cp);
    EXCEPT
      Rd.Failure (err) =&gt;
        RAISE StableError.E(
                AtomList.Cons(
                  Atom.FromText(&quot;error reading checkpoint&quot;),
                  err));
    END;
    IF log # NIL THEN
      t.replayLog(log);
      TRY
        Rd.Close(log) (* get rid of the file handle *)
      EXCEPT
        Rd.Failure =&gt; (* well, it's not the end of the world *)
      END
    END;
    RETURN t;
  END Recover;
</PRE> \subsection{Procedure Checkpoint}
   Free the log and use the protocoll described in <CODE>LogManager.i3</CODE>
   to do a checkpoint.


<P><PRE>PROCEDURE <A NAME="Checkpoint"><procedure>Checkpoint</procedure></A> (t: StableData.T)
  RAISES {StableError.E} =
  BEGIN
    TRY
      t.freeLog();
      VAR cp := t.lm.beginCheckpoint(t.nm);
      BEGIN
        Log.CrashPoint(102);
        t.writeCheckpoint(cp);
        Wr.Close(cp);
        t.log := t.lm.endCheckpoint(t.nm);
      END;
    EXCEPT
    | OSError.E (err) =&gt; RAISE StableError.E(err);
    | Wr.Failure (err) =&gt;
        RAISE StableError.E(
                AtomList.Cons(
                  Atom.FromText(&quot;checkpoint error truncating logfile&quot;),
                  err));
    END;
  END Checkpoint;
</PRE> \subsection{Procedures Read- and WriteCheckpoint}
   These are the default procedures for reading and writing a checkpoint.
   They use the pickle package.


<P><PRE>PROCEDURE <A NAME="ReadCheckpoint"><procedure>ReadCheckpoint</procedure></A> (self: Public; cp: Rd.T):
  StableData.T RAISES {StableError.E} =
  BEGIN
    TRY
      TYPECASE Pickle.Read(cp) OF
        StableData.T (d) =&gt; RETURN d;
      ELSE
        RAISE StableError.E(
            AtomList.List1(
              Atom.FromText(
                &quot;Checkpoint in &quot; &amp; self.nm
                  &amp; &quot; does not contain data of the correct type&quot;)))
      END;
    EXCEPT
    | Rd.EndOfFile =&gt;
        RAISE StableError.E(
            AtomList.List1(
              Atom.FromText(
                &quot;unexpected EOF encountered reading checkpointfile&quot;)))
    | Rd.Failure (err) =&gt;
        RAISE StableError.E(
                AtomList.Cons(
                  Atom.FromText(
                    &quot;error reading checkpointfile&quot;), err))
    | Pickle.Error (msg) =&gt;
        RAISE StableError.E(
                AtomList.List1(
                  Atom.FromText(
                    &quot;pickle-error (&quot; &amp; msg
                      &amp; &quot;) reading checkpointfile&quot;)))
    END;
  END ReadCheckpoint;

PROCEDURE <A NAME="WriteCheckpoint"><procedure>WriteCheckpoint</procedure></A> (self: Public; wr: Wr.T)
  RAISES {StableError.E} =
  BEGIN
    TRY
      Pickle.Write(wr, self);
    EXCEPT
    | Wr.Failure (err) =&gt;
        RAISE StableError.E(
                AtomList.Cons(
                  Atom.FromText(
                    &quot;error writing checkpointfile&quot;), err))
    | Pickle.Error (msg) =&gt;
        RAISE StableError.E(
                AtomList.List1(
                  Atom.FromText(
                    &quot;pickle-error (&quot; &amp; msg
                      &amp; &quot;) writing checkpointfile&quot;)))
    END
  END WriteCheckpoint;

BEGIN
END StableRep.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
