<HTML>
<HEAD>
<TITLE>SRC Modula-3: tcp/src/common/ConnMsgRW.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>tcp/src/common/ConnMsgRW.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM> Created on Sat Jan 11 15:49:00 PST 1992 by gnelson </EM></BLOCKQUOTE><PRE>

UNSAFE MODULE <module><implements><A HREF="ConnMsgRW.i3">ConnMsgRW</A></implements></module>;

IMPORT <A HREF="ConnFD.i3">ConnFD</A>, <A HREF="../../../atom/src/Atom.i3">Atom</A>, <A HREF="../../../libm3/derived/AtomList.i3">AtomList</A>;
IMPORT <A HREF="../../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../../rw/src/Common/Wr.i3">Wr</A>, <A HREF="../../../rw/src/Common/MsgRd.i3">MsgRd</A>, <A HREF="../../../rw/src/Common/MsgWr.i3">MsgWr</A>, <A HREF="../../../rw/src/Common/RdClass.i3">RdClass</A>, <A HREF="../../../rw/src/Common/WrClass.i3">WrClass</A>, <A HREF="../../../thread/src/Common/Thread.i3">Thread</A>;
IMPORT <A HREF="../../../uid/src/Common/Swap.i3">Swap</A>;
</PRE> The byte stream is divided into a sequence of fragments.  Each
   fragment has a header that consists of 4 bytes of control information
   and 4 bytes of count, followed by a sequence of data bytes.  The last
   fragment of each message has the <CODE>eom</CODE> word set it its header.  Both
   control word and count are transmitted in little-endian order (that
   is, the bytes of the integer occur in the stream in order of
   increasing significance).
   <P>
   The {\it alignment} of a byte in a sequence is its index modulo
   eight.  The data format maintains the invariants that
<P>
    -- each data byte has the same alignment in its fragment and
       in its message, and
<P>
    -- the alignment of the first byte of each header is zero.
<P>
  This is achieved by inserting up to seven bytes of padding before
  and after each fragment.  Alignment padding is only included for
  fragments that contain one or more data byte.
<P>
  That is, given a sequence of fragments
<P>
  <PRE>
      f[0], ..., f[N-1]
</PRE>
  a message is represented by a contiguous subsequence of fragments,
<P>
  <PRE>
      f[p], ... f[q-1]
</PRE>
  where 
<P>
  <PRE>
      (A i: p &lt;= i &lt; q : f[i].eom = (i = q-1))
        AND (p=0) OR f[p-1].eom
</PRE>
  The bytes of the message are the concatenation of the data bytes of
  the fragments containing the message, minus any padding introduced
  to satisfy the alignment invariants.  For example, the following code
  sets <CODE>m</CODE> to the contents of the message f[p], ...  f[q-1]:
<P>
  <PRE>
      VAR 
           m := &quot;&quot;;
           skip := 0;
         CONST Alignment = 8;    (* at least as big as a fragment header 


<P>  <PRE>|  BEGIN
  |    FOR i := p TO q-1 DO
  |      m := m &amp; Text.Sub(f[i].data, skip, f[i].count);
  |      skip := skip + f[i].count MOD Alignment
  |    END
  |  END

*)

TYPE
  Int32 =  BITS 32 FOR [-16_7FFFFFFF-1..16_7FFFFFFF];
  FragmentHeader = RECORD
    (* N.B. This is in little-endian order on the wire. *)
    eom: Int32;
    nb: Int32;
  END;

CONST HeaderBytes = BYTESIZE(FragmentHeader);
CONST AlignBytes = HeaderBytes;

TYPE
  RdT = MsgRd.T BRANDED OBJECT
    fd: ConnFD.T;
    hdr := FragmentHeader{eom := 1, nb := 0};
    lim: CARDINAL := 0;
  OVERRIDES
    seek := RdSeek;
    close := RdClose;
    nextMsg := RdNextMsg;
    length := Length;
  END;
</PRE> The {\it current fragment} of <CODE>rd</CODE> is the fragment containing
   the data byte rd.buff[rd.sd+sr.cur], if <CODE>rd</CODE> is ready.  Otherwise
   the current fragment is the last fragment that has been processed
   by the reader.  The initial state reflects an artificial fragment
   with size zero that is prepended to the actual sequence of fragments
   read over the TCP stream.
   <P>
   The value <CODE>hdr.eom</CODE> is the end of message bit for the current fragment.
<P>
   The value <CODE>hdr.nb</CODE> is the number of bytes in the current fragment that
   remain to be read from the TCP stream.
   <P>
   The value of <CODE>lim</CODE> is one greater than the buffer index of the last
   valid TCP stream byte in the buffer.  The value of <CODE>lim MOD 8</CODE> is
   always zero.

  
<P><PRE>TYPE
  WrT = MsgWr.T BRANDED OBJECT
    fd: ConnFD.T;
  OVERRIDES
    seek := WrSeek;
    flush := WrFlush;
    close := WrClose;
    nextMsg := WrNextMsg;
  END;

CONST BufferSize = 8192;

VAR
  ProtocolErrorEOF: AtomList.T;
  ProtocolErrorNB: AtomList.T;

PROCEDURE <A NAME="NewRd"><procedure>NewRd</procedure></A>(fd: ConnFD.T) : MsgRd.T =
  BEGIN
    RETURN NEW(RdT, fd := fd,
        buff := NEW(REF ARRAY OF CHAR, BufferSize),
        st := 0,
        lo := 0,
        hi := 0,
        cur := 0,
        intermittent := TRUE,
        seekable := FALSE,
        closed := FALSE)
  END NewRd;

PROCEDURE <A NAME="NewWr"><procedure>NewWr</procedure></A>(fd: ConnFD.T) : MsgWr.T =
  BEGIN
      RETURN NEW(WrT, fd := fd,
        buff := NEW(REF ARRAY OF CHAR, BufferSize),
        st := HeaderBytes,
        lo := 0,
        hi := BufferSize - HeaderBytes,
        cur := 0,
        buffered := TRUE,
        seekable := FALSE,
        closed := FALSE)
  END NewWr;

PROCEDURE <A NAME="RdSeek"><procedure>RdSeek</procedure></A>(rd: RdT; &lt;*UNUSED*&gt; n: CARDINAL;
                          dontBlock: BOOLEAN): RdClass.SeekResult
  RAISES {Rd.Failure, Thread.Alerted} =
  VAR
    nb: CARDINAL;
  BEGIN
    IF dontBlock THEN RETURN RdClass.SeekResult.WouldBlock; END;
    REPEAT
      (* Advance to next sub-buffer and set &quot;nb&quot; to the
         number of data bytes available in it. There
         are two cases: *)
      IF rd.hdr.nb # 0 THEN
        (* read the rest of the current fragment into the buffer *)
        rd.st := 0;
        rd.lo := rd.hi;
        rd.lim := ReadAligned(rd.fd, rd.buff);
        (* fall out into common code below *)
      ELSE
        (* advance to next non-empty fragment *)
        rd.st := Align((rd.hi - rd.lo) + rd.st);
        rd.lo := rd.hi;
        REPEAT
          IF rd.hdr.eom # 0 THEN RETURN RdClass.SeekResult.Eof; END;
          IF rd.lim - rd.st &lt; HeaderBytes THEN
            &lt;* ASSERT rd.st = rd.lim *&gt;
            (* read from stream to get next header *)
            rd.st := 0;
            rd.lim := ReadAligned(rd.fd, rd.buff);
          END;
          (* careful, this is endian dependent *)
          rd.hdr :=
            LOOPHOLE(ADR(rd.buff[rd.st]), UNTRACED REF FragmentHeader)^;
          IF Swap.endian = Swap.Endian.Big THEN
            rd.hdr.nb := Swap.Swap4(rd.hdr.nb);
          END;
          IF rd.hdr.nb &lt; 0 THEN RAISE Rd.Failure(ProtocolErrorNB); END;
          INC(rd.st, HeaderBytes);
        UNTIL rd.hdr.nb # 0;
      END;
      (* Now &quot;rd.st&quot; is the buffer index where the data part of
         the now-current fragment begins.  This could be beyond
         the end of the buffer. *)
      rd.st := rd.st + (rd.lo MOD AlignBytes);
      nb := MIN(rd.hdr.nb, MAX(rd.lim-rd.st, 0));
      INC(rd.hi, nb);
      DEC(rd.hdr.nb, nb);
    UNTIL nb # 0;   (* loop necessary in case hdr is at end of buffer *)
    RETURN RdClass.SeekResult.Ready;
  END RdSeek;

PROCEDURE <A NAME="Align"><procedure>Align</procedure></A>(n: CARDINAL) : CARDINAL =
</PRE><BLOCKQUOTE><EM> Round <CODE>n</CODE> up to the next multiple of 8. </EM></BLOCKQUOTE><PRE>
  BEGIN RETURN n + ((-n) MOD 8) END Align;

PROCEDURE <A NAME="ReadAligned"><procedure>ReadAligned</procedure></A>(
    fd: ConnFD.T;
    buff: REF ARRAY OF CHAR) : INTEGER
  RAISES {Rd.Failure, Thread.Alerted} =
    (*
       &quot;ReadAligned&quot; always returns a non-zero result.
       If we are at the end of the TCP stream, then we
       have encountered a protocol error.  Otherwise, we
       return a postive count of bytes which is
       zero modulo &quot;AlignBytes&quot;.
    *)
  VAR
    len := 0;
    nb: CARDINAL;
    &lt;* FATAL ConnFD.TimedOut *&gt;
  BEGIN
    len := 0;
    REPEAT
      nb := fd.get(SUBARRAY(buff^, len, NUMBER(buff^)-len));
      INC(len, nb);
    UNTIL (len MOD AlignBytes) = 0;
    IF len = 0 THEN RAISE Rd.Failure(ProtocolErrorEOF); END;
    RETURN len;
  END ReadAligned;

PROCEDURE <A NAME="Length"><procedure>Length</procedure></A>(rd: RdT) : INTEGER =
  BEGIN
    IF rd.hdr.eom # 0 THEN
      RETURN rd.lo + rd.hdr.nb;
    ELSE
      RETURN -1;
    END;
  END Length;

PROCEDURE <A NAME="RdClose"><procedure>RdClose</procedure></A>(rd: RdT) RAISES {Rd.Failure} =
  BEGIN
    rd.buff := NIL;
    rd.fd.shutdownIn();
  END RdClose;

PROCEDURE <A NAME="RdNextMsg"><procedure>RdNextMsg</procedure></A>(rd: RdT) : BOOLEAN
    RAISES {Rd.Failure, Thread.Alerted} =
  BEGIN
    (* For now, we are not going to differentiate between end
       of connection and communications failure.  This procedure
       will raise Rd.Failure in either case. *)
    WHILE rd.hdr.nb # 0 OR NOT rd.hdr.eom # 0 DO
      EVAL RdSeek(rd, rd.hi, FALSE);
    END;
    rd.st := Align((rd.hi - rd.lo) + rd.st);
    rd.cur := 0;
    rd.lo := 0;
    rd.hi := 0;
    rd.hdr := FragmentHeader{eom := 0, nb := 0};
    EVAL RdSeek(rd, rd.cur, FALSE);
    RETURN TRUE;
  END RdNextMsg;

PROCEDURE <A NAME="WrSeek"><procedure>WrSeek</procedure></A>(wr: WrT; &lt;*UNUSED*&gt; n: CARDINAL)
  RAISES {Wr.Failure, Thread.Alerted} =
  BEGIN WrFlush(wr) END WrSeek;

PROCEDURE <A NAME="WrFlush"><procedure>WrFlush</procedure></A>(wr: WrT) RAISES {Wr.Failure, Thread.Alerted} =
  BEGIN
    IF wr.cur # wr.lo THEN PutFrag(wr, FALSE); END;
    wr.st := HeaderBytes + (wr.cur MOD AlignBytes);
    wr.lo := wr.cur;
    wr.hi := wr.lo + (NUMBER(wr.buff^) - wr.st);
  END WrFlush;

PROCEDURE <A NAME="WrClose"><procedure>WrClose</procedure></A>(wr: WrT) RAISES {Wr.Failure} =
  BEGIN
    wr.buff := NIL;
    wr.fd.shutdownOut();
  END WrClose;

PROCEDURE <A NAME="WrNextMsg"><procedure>WrNextMsg</procedure></A>(wr: WrT) RAISES {Wr.Failure, Thread.Alerted} =
  BEGIN
    PutFrag(wr, TRUE);
    wr.st := HeaderBytes;
    wr.cur := 0;
    wr.lo := 0;
    wr.hi := NUMBER(wr.buff^) - HeaderBytes;
  END WrNextMsg;

PROCEDURE <A NAME="PutFrag"><procedure>PutFrag</procedure></A>(wr: WrT; eom: BOOLEAN) RAISES {Wr.Failure, Thread.Alerted} =
  VAR len := wr.cur - wr.lo;
  BEGIN
    WITH hdr = LOOPHOLE(ADR(wr.buff[0]), UNTRACED REF FragmentHeader) DO
      hdr^ := FragmentHeader{eom := ORD(eom), nb := len};
      IF Swap.endian = Swap.Endian.Big THEN
        hdr.nb := Swap.Swap4(hdr.nb);
      END;
    END;
    IF len = 0 THEN
      (* output just a header for null fragments *)
      wr.fd.put(SUBARRAY(wr.buff^, 0, HeaderBytes));
    ELSE
      (* otherwise correct for alignment of first byte *)
      wr.fd.put(SUBARRAY(wr.buff^, 0, Align(len+wr.st)));
    END;
  END PutFrag;

BEGIN
  ProtocolErrorEOF :=
    AtomList.Cons(Atom.FromText(&quot;ConnMsgRW.UnexpectedEOF&quot;), NIL);
  ProtocolErrorNB :=
    AtomList.Cons(Atom.FromText(&quot;ConnMsgRW.ProtocolError&quot;), NIL);
END ConnMsgRW.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
