<HTML>
<HEAD>
<TITLE>SRC Modula-3: ui/src/vbt/VBT.i3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>ui/src/vbt/VBT.i3</H2></A><HR>
<inInterface>
<PRE><A HREF="../../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                                           </EM></BLOCKQUOTE><PRE>
</PRE> by Steve Glassman, Mark Manasse and Greg Nelson           
<PRE>&lt;*PRAGMA LL*&gt;
</PRE> A <CODE>VBT.T</CODE> (or simply a <CODE>VBT</CODE>) is the basic window 
   abstraction of the Trestle system.  

<P><PRE>INTERFACE <interface><A HREF="VBT.m3">VBT</A></interface>;

IMPORT <A HREF="../../../word/src/Word.i3">Word</A>, <A HREF="../../../geometry/src/Axis.i3">Axis</A>, <A HREF="../../../geometry/src/Point.i3">Point</A>, <A HREF="../../../geometry/src/Rect.i3">Rect</A>, <A HREF="../../../geometry/src/Region.i3">Region</A>, <A HREF="../../../geometry/src/Trapezoid.i3">Trapezoid</A>,
  <A HREF="../../../geometry/src/Path.i3">Path</A>, <A HREF="Pixmap.i3">Pixmap</A>, <A HREF="Cursor.i3">Cursor</A>, <A HREF="Font.i3">Font</A>, <A HREF="PaintOp.i3">PaintOp</A>, <A HREF="ScrnPixmap.i3">ScrnPixmap</A>;
</PRE> \subsection{The public methods} 

<P> A <CODE>VBT</CODE> is represented as an object with a private prefix and twelve
   public methods, which define the way the <CODE>VBT</CODE> responds to events.
   Here are the type declarations that reveal the public methods, while
   concealing the private prefix: 

<P><PRE>TYPE
  <A HREF="VBTRep.m3#T">T</A> &lt;: Public;
  Public = Prefix OBJECT
    METHODS
      &lt;* LL.sup = mu *&gt;
      mouse(READONLY cd: MouseRec);
      position(READONLY cd: PositionRec);
      redisplay();
      misc(READONLY cd: MiscRec);
      key(READONLY cd: KeyRec);
      discard();
      &lt;* LL.sup = mu.SELF *&gt;
      reshape(READONLY cd: ReshapeRec);
      rescreen(READONLY cd: RescreenRec);
      repaint(READONLY rgn: Region.T);
      shape(ax: Axis.T; n: CARDINAL): SizeRange;
      &lt;* LL.sup &lt;= mu *&gt;
      read(sel: Selection; tc: CARDINAL): Value
        RAISES {Error};
      write(sel: Selection; val: Value; tc: CARDINAL)
        RAISES {Error};
    END;
  <A HREF="VBTRep.i3#Prefix">Prefix</A> &lt;: ROOT;
</PRE> For example, if the user reshapes a window, Trestle will call
   the window's reshape method; if the user exposes some part of the
   window, Trestle will call the window's repaint method.  The
   remainder of the <CODE>VBT</CODE> interface specifies the methods in detail. 
   The pragmas about <CODE>LL</CODE> are explained in the section on locking level, 
   below.
<P>
   You should never call a <CODE>VBT</CODE>'s methods directly.  The <CODE>VBTClass</CODE>
   interface provides wrapper procedures that call the methods
   indirectly.  

<P> \subsection{Screens and domains} 

<P> Every <CODE>VBT</CODE> has a {\it screen} that associates a pixel value with
   each integer lattice point.  We write <CODE>v[p]</CODE> to denote the value
   of the pixel at point <CODE>p</CODE> of the screen of the <CODE>VBT</CODE> <CODE>v</CODE>.  Changing
   the pixel values in a <CODE>VBT</CODE>'s screen is called {\it painting}.
     <P>
   The part of a <CODE>VBT</CODE>'s screen that is visible to the user---or that would
   be visible if other windows weren't in the way---is called the
   {\it domain} of the <CODE>VBT</CODE>: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#Domain">Domain</A>(v: T): Rect.T; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM>  Return the rectangular extent of the visible part of
    <CODE>v</CODE>'s screen. </EM></BLOCKQUOTE><PRE>

 (* The domain is an arbitrary rectangle: it can be empty, the
    coordinate origin can be anywhere inside or outside it, and it does
    not necessarily correspond to the position of the window on the
    physical display screen.

    When &quot;v&quot; is reshaped, &quot;Domain(v)&quot; changes from one rectangle to
    another.  During this transformation Trestle tries to save the old
    screen until the new screen is fully repainted: thus in the midst
    of reshaping, &quot;v[p]&quot; can be useful for some points &quot;p&quot; outside
    &quot;Domain(v)&quot;.  At other times, Trestle keeps track of &quot;v[p]&quot; only
    for points &quot;p&quot; inside &quot;Domain(v)&quot;.

    The pragma &quot;LL.sup &lt; v&quot; is explained in the next section.  *)
</PRE> \subsection{Locking level} 

<P> The global mutex <CODE>mu</CODE> serializes operations that affect the tree
   of <CODE>VBTs</CODE>:  
   \index{LL (Locking Level)@{\protect\tt LL} (Locking Level)}
   
<P><PRE>VAR mu: MUTEX;
</PRE> In addition, every <CODE>VBT</CODE> includes a private mutex that serializes
   operations on the <CODE>VBT</CODE> itself.  The private mutex of a <CODE>VBT</CODE> is
   revealed in the <CODE>VBTClass</CODE> interface, not in this interface.
<P>
   The order in which a thread is allowed to acquire these locks is
   called the ``locking order''.  It is defined by these two rules: 
<P>
   \medskip\bulletitem The global <CODE>mu</CODE> precedes every <CODE>VBT</CODE>.
   <P>
   \medskip\bulletitem Every <CODE>VBT</CODE> precedes its parent.
       <P>
   \medskip\noindent The ``locking level'' of a thread, or <CODE>LL</CODE> for
   short, is the set of locks that the thread has acquired.  The
   expression <CODE>LL.sup</CODE> denotes the maximum of the locks in <CODE>LL</CODE>.  (The
   locking order is partial, but <CODE>LL.sup</CODE> will be defined for any thread
   in a correct program, since threads acquire locks in ascending
   order.)
   <P>
   Each procedure declaration in the Trestle system includes a pragma
   specifying the locking level at which a thread can legally call the
   procedure.  For example, the pragma <CODE>LL.sup &lt; v</CODE> on the <CODE>Domain</CODE>
   procedure allows a thread to call <CODE>Domain</CODE> with no locks, or with
   <CODE>mu</CODE> locked, or with descendants of <CODE>v</CODE> locked, but forbids calling
   it with any other <CODE>VBT</CODE>s locked. 
<P>
   Similarly, each public data field and method of an object has a
   locking level.  In both cases, a locking level pragma applies to
   all the fields or methods between it and the next pragma.  These 
   pragmas may contain the special identifier <CODE>SELF</CODE>, which refers 
   to the object itself.
   <P>
   The locking level for a method is identical to the locking level
   for a procedure: it specifies the locking level at which a thread
   can legally call the method.  For example, whenever the <CODE>mouse</CODE>, 
   <CODE>position</CODE>, <CODE>redisplay</CODE>, <CODE>misc</CODE>, <CODE>key</CODE>, or <CODE>discard</CODE> methods of 
   a <CODE>VBT</CODE> are called, the locking level satisfies <CODE>LL.sup = mu</CODE>.
   <P>
   The locking level for a writable data field is of the form
<P>
<PRE>
      LL &gt;= {mu1, ..., muN}.  
</PRE>
   This specifies that in order to write the field, a thread must hold
   all of the locks <CODE>mu1</CODE> through <CODE>muN</CODE>.  As a consequence, a thread
   can read the field if it holds any of the locks.
   <P>
   (In a locking level pragma, the ordering symbols <CODE>&gt;=</CODE>, <CODE>&lt;=</CODE>, <CODE>&lt;</CODE>,
   and <CODE>&gt;</CODE> are overloaded to denote either set containment or lock
   order, depending on context.  For example, <CODE>LL &gt;= {mu, v}</CODE> indicates
   that the thread has both <CODE>mu</CODE> and <CODE>v</CODE> locked, while <CODE>LL.sup &lt;= mu</CODE>
   indicates that all locks held by the thread precede <CODE>mu</CODE> in the
   locking order.)
<P>
   A data field may also be commented <CODE>CONST</CODE>, meaning that it is 
   readonly after initialization and therefore can be read with no 
   locks at all.
<P>
   There is one more special notation related to locking levels: a <CODE>VBT</CODE>
   <CODE>v</CODE> can hold a ``share'' of the global lock <CODE>mu</CODE>; its share is
   denoted by <CODE>mu.v</CODE>.  This is explained in the section of this
   interface that specifies the <CODE>reshape</CODE> method.
   <P>
   All the procedures in the Trestle system restore the caller's locking
   level when they return.  For example, calling <CODE>Domain(v)</CODE> has no
   net effect on a thread's locking level.  

<P>
<P> \subsection{ScreenTypes} 

<P> Pixel values are integers.   The color associated with a pixel value
   is determined in some manner that depends on the {\it screentype}
   of the <CODE>VBT</CODE>.  A value <CODE>st</CODE> of type <CODE>VBT.ScreenType</CODE> represents a
   screentype: 

<P><PRE>TYPE
  <A HREF="VBTRep.m3#ScreenType">ScreenType</A> &lt;: ScreenTypePublic;
  ScreenTypePublic = OBJECT (*CONST*)
    depth: INTEGER;
    color: BOOLEAN;
    res: ARRAY Axis.T OF REAL
  END;
</PRE> The integer <CODE>st.depth</CODE> is the number of bits per pixel in screens
    of type <CODE>st</CODE>. The boolean <CODE>st.color</CODE> is <CODE>TRUE</CODE> if the pixels are
    colored, <CODE>FALSE</CODE> if they are black and white or gray-scale.  The
    array <CODE>st.res</CODE> gives the horizontal and vertical resolution of the
    screen in pixels per millimeter for desk-top displays, or in
    visually equivalent units for other displays.
   <P>
   The screentype of a newly-allocated <CODE>VBT</CODE> is <CODE>NIL</CODE>; it becomes
   non-<CODE>NIL</CODE> only when the <CODE>VBT</CODE> is connected to a window system.
   <P>
   Here are two procedures for reading the screentype of a <CODE>VBT</CODE> and
   for converting distances to screen coordinates: 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#ScreenTypeOf">ScreenTypeOf</A>(v: T): ScreenType;
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return the screentype of <CODE>v</CODE>. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#MMToPixels">MMToPixels</A>(v: T; mm: REAL; ax: Axis.T)
: REAL; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return the number of pixels that correspond to <CODE>mm</CODE> millimeters on
   <CODE>v</CODE>'s screentype in the axis <CODE>ax</CODE>; or return <CODE>0</CODE> if <CODE>v</CODE>'s screentype
   is <CODE>NIL</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> The <CODE>ScreenType</CODE> interface reveals more details, for example,
   about color maps.  
   
<P> \subsection{Splits and leaves} 

<P> User interfaces are usually constructed from a tree of <CODE>VBTs</CODE> whose
   root is the ``top-level window'' known to the window manager.  <CODE>VBTs</CODE>
   are classified into two main subtypes based on their positions in
   the tree: 

<P><PRE>TYPE
  <A HREF="VBTClass.m3#Split">Split</A> &lt;: T;
  <A HREF="VBT.m3#Leaf">Leaf</A> &lt;: T;

PROCEDURE <A HREF="VBT.m3#Parent">Parent</A>(v: T): Split; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return <CODE>v</CODE>'s parent, or <CODE>NIL</CODE> if <CODE>v</CODE> has no parent.  </EM></BLOCKQUOTE><PRE>
</PRE> A <CODE>Split</CODE> (also called a parent <CODE>VBT</CODE>) divides its screen up
   among its children according to some layout policy that depends on
   the class of split.  Each pixel of the parent screen represents 
   a pixel of one of the child <CODE>VBTs</CODE>, which is said to control that 
   pixel.  For example, overlapping windows are provided by a class of 
   split called a <CODE>ZSplit</CODE>, for which the children are ordered bottom to 
   top, and each pixel <CODE>v[p]</CODE> of the parent domain is controlled by the 
   top-most child whose domain includes <CODE>p</CODE>.
   <P>
   See the <CODE>Split</CODE> interface for common operations on splits (e.g.,
   enumerating children).
<P>
   A <CODE>Leaf</CODE> is a <CODE>VBT</CODE> in which the twelve public methods make the
   <CODE>Leaf</CODE> ignore all events, be indifferent about its shape, and do
   nothing when discarded.  It is provided as a starting point: you
   can define a useful subtype of <CODE>Leaf</CODE> by overriding the methods that
   are relevant to the new class.
   <P>
   Almost all subtypes of <CODE>VBT</CODE> are subtypes of either <CODE>Split</CODE> or
   <CODE>Leaf</CODE>.  

<P>   
<P> \subsection{Timestamps, modifiers, mouse buttons, and cursor positions} 

<P> The following types are used in several of the event methods: 

<P><PRE>TYPE
  TimeStamp = Word.T;

  Modifier =
    {Shift, Lock, Control, Option,
     Mod0, Mod1, Mod2, Mod3,
     MouseL, MouseM, MouseR,
     Mouse0, Mouse1, Mouse2, Mouse3, Mouse4};

  Button = [Modifier.MouseL..Modifier.Mouse4];

  Modifiers = SET OF Modifier;

  ScreenID = INTEGER;

  CursorPosition = RECORD
    pt: Point.T;
    screen: ScreenID;
    gone, offScreen: BOOLEAN;
  END;

CONST
  Buttons = Modifiers{FIRST(Button)..LAST(Button)};
</PRE> Trestle has an internal unsigned clock register that is incremented every
   few milliseconds.  When Trestle reports a mouse or keyboard event to a
   <CODE>VBT</CODE>, it also reports the value of the clock register when the event
   occurred, which is called the {\it timestamp} of the event.
   Timestamps serve as unique identifiers for the associated events.
   Also, the absolute time interval between two events can be computed
   by subtracting their timestamps with <CODE>Word.Minus</CODE> and multiplying by
   <CODE>Trestle.TickTime()</CODE>, which is the absolute interval between clock
   ticks. \index{time interval between events}
<P>
   A few keys on the keyboard are defined to be {\it modifiers}, like <CODE>Shift</CODE>,
   <CODE>Control</CODE>, and <CODE>Option</CODE>.  When Trestle reports a mouse or keyboard event
   to a <CODE>VBT</CODE>, it also reports the set of modifier keys and buttons that
   were down when the event occurred.  Thus the application can distinguish
   shifted mouse clicks from unshifted mouse clicks, for example.
   <P>
   The modifier <CODE>Shift</CODE> is reported if either of the keyboard's shift
   keys is down; similarly for <CODE>Control</CODE> and <CODE>Option</CODE>.  The modifier <CODE>Lock</CODE>
   is reported if the lock key is locked down.  If the keyboard has
   a key labelled <CODE>lock</CODE> but this key does not have mechanical alternate
   action, then the modifier <CODE>Lock</CODE> reflects the simulated state of
   the lock key (that is, alternate presses of the lock key turn the
   modifier on or off).  Trestle does not define whether it reports up
   and down transitions for lock keys while the modifier is set.
<P>
   Some Trestle servers interpret other keys as modifiers: the type
   definition accommodates up to four additional modifiers, <CODE>Mod0</CODE>
   through <CODE>Mod3</CODE>.
<P>
   The mouse buttons are reported as modifiers.  The naming of the first
   three buttons assumes a three-button mouse; in general it is assumed
   that there are at most eight buttons.
<P>
   When Trestle reports a mouse position event to a <CODE>VBT</CODE> <CODE>v</CODE>, it
   also reports a value <CODE>cp</CODE> of type <CODE>CursorPosition</CODE>.  The point
   <CODE>cp.pt</CODE> is the position of the cursor; the integer <CODE>cp.screen</CODE>
   identifies the screen of the window system where the event occurred;
   and <CODE>cp.offScreen</CODE> is <CODE>TRUE</CODE> if the position is on a different screen
   than <CODE>v</CODE>, and <CODE>FALSE</CODE> otherwise.  If <CODE>cp.offScreen</CODE> is <CODE>FALSE</CODE>, then
   <CODE>cp.pt</CODE> is in <CODE>v</CODE>'s coordinate system, otherwise <CODE>cp.pt</CODE> is in the
   coordinate system of <CODE>cp.screen</CODE>.  The boolean <CODE>cp.gone</CODE> is <CODE>TRUE</CODE>
   if <CODE>v</CODE> doesn't control the position <CODE>cp.pt</CODE>, and <CODE>FALSE</CODE> if it does.
   If <CODE>cp.offScreen</CODE> is <CODE>TRUE</CODE>, then so is <CODE>cp.gone</CODE>.  A position is 
   controlled by a <CODE>VBT</CODE> <CODE>w</CODE> if a mouse-click at that position would
   ordinarily be delivered to <CODE>w</CODE>.  All positions controlled by a <CODE>VBT</CODE>
   are in its domain; every pixel in the domain of a split is controlled
   by at most one child of that split.  You should think of the positions
   controlled by a <CODE>VBT</CODE> as the visible positions in its domain. 

<P> \subsection{The mouse method} 

<P> Trestle calls a <CODE>VBT</CODE>'s mouse method to report mouse clicks.  The
   method will be called with <CODE>LL.sup = mu</CODE>, and takes an argument of
   type <CODE>MouseRec</CODE>. 
   
<P><PRE>TYPE MouseRec = RECORD
  whatChanged: Button;
  time: TimeStamp;
  cp: CursorPosition;
  modifiers: Modifiers;
  clickType: ClickType;
  clickCount: INTEGER;
END;

ClickType =
  {FirstDown, OtherDown, OtherUp, LastUp};
</PRE> The method call <CODE>v.mouse(cd)</CODE> indicates that the mouse button
   <CODE>cd.whatChanged</CODE> went down or up at time <CODE>cd.time</CODE> and cursor
   position <CODE>cd.cp</CODE>.  
   <P>
   The field <CODE>cd.clickType</CODE> is <CODE>FirstDown</CODE> if the button went down when
   no other buttons were down, <CODE>OtherDown</CODE> if it went down when some
   other button(s) were already down, <CODE>LastUp</CODE> if it went up when all
   other buttons were up, and <CODE>OtherUp</CODE> if it went up when some other
   button(s) were still down.
   <P>
   The field <CODE>cd.modifiers</CODE> reflects the state of the modifiers
   (either just before or just after the button transition; it is
   not specified which). 
<P>
   If <CODE>cd.clickType</CODE> is <CODE>FirstDown</CODE>, then <CODE>cd.cp.gone</CODE> will be <CODE>FALSE</CODE>.
<P>
   The field <CODE>cd.clickCount</CODE> is the number of preceding transitions 
   of the button that were near in time and space.  For example, 
   <CODE>clickCount=3</CODE> on the final up transition of a double click. 
   Some Trestle implementations have auxilliary interfaces that 
   allow you to set the amount of time and mouse motion allowed. 

<P> \subsection{The mouse focus rule} 

<P> A split relays mouse clicks to whichever child of the split controls the
   pixel at the position of the click---more or less.  If this rule were
   applied blindly, a child could receive a down-click and never receive
   the corresponding up-click, which would make it impossible to program
   many user interfaces that involve dragging.  Therefore the actual rule
   is more complicated.  \index{mouse~focus}
<P>
   Each split <CODE>sp</CODE> contains a variable <CODE>mouseFocus(sp)</CODE>, which records 
   the child of the split that has received a transition of type 
   <CODE>FirstDown</CODE> but not yet received a subsequent transition of type 
   <CODE>LastUp</CODE>.  If there is no such child, <CODE>mouseFocus(sp)</CODE> is <CODE>NIL</CODE>.  
   The split <CODE>sp</CODE> relays the <CODE>MouseRec</CODE> <CODE>cd</CODE> by the ``mouse focus rule'':
<P>
<PRE>
      IF <KBD>some child </KBD>ch<KBD> controls </KBD>cd.cp<KBD></KBD> THEN
        w := ch;
        w.mouse(cd)
      ELSE
        w := NIL
      END;
      IF cd.clickType = ClickType.FirstDown THEN
        mouseFocus(sp) := w
      ELSE
        IF mouseFocus(sp) # NIL AND mouseFocus(sp) # w THEN
          cd.cp.gone := TRUE;
          mouseFocus(sp).mouse(cd)
        END;
        IF cd.clickType = ClickType.LastUp THEN 
          mouseFocus(sp) := NIL 
        END
      END
</PRE>
   The mouse focus is guaranteed to receive all button transitions until
   the last button comes up, no matter where it occurs.  

<P> \subsection{The position method} 

<P> Trestle calls a <CODE>VBT</CODE>'s position method to report cursor positions.  The
   method will be called with <CODE>LL.sup = mu</CODE>, and takes an argument of
   type <CODE>PositionRec</CODE>. 
   
<P><PRE>TYPE PositionRec = RECORD
  cp: CursorPosition;
  time: TimeStamp;
  modifiers: Modifiers;
END;
</PRE> The method call <CODE>v.position(cd)</CODE> indicates that at the time <CODE>cd.time</CODE>
   the cursor position was <CODE>cd.cp</CODE> and the set of modifiers keys that
   were down was <CODE>cd.modifiers</CODE>.
<P>
   The next section explains how to control the delivery of cursor 
   positions.  

<P>
<P> \subsection{Tracking the cursor by setting cages} 

<P> Every <CODE>VBT</CODE> <CODE>v</CODE> contains a field <CODE>cage(v)</CODE>, which represents a set
   of cursor positions.  As long as the cursor's position is inside
   <CODE>v</CODE>'s cage, Trestle won't report the position to <CODE>v</CODE>.  As soon as
   the cursor's position moves outside <CODE>cage(v)</CODE>, Trestle reports the
   position to <CODE>v</CODE>, after first resetting <CODE>v</CODE>'s cage to contain all
   cursor positions.  Resetting the cage inhibits further reporting
   of cursor positions: to continue tracking, the position method must
   set a new cage.  \index{cursor~tracking}
   \index{cages~(for~cursor~tracking)} 

<P><PRE>TYPE
  Cage = RECORD
    rect: Rect.T;
    inOut: InOut;
    screen: ScreenID;
  END;
  InOut = SET OF BOOLEAN;

CONST
  AllScreens: ScreenID = -1;
</PRE> The cage <CODE>cg</CODE> contains the cursor position <CODE>cp</CODE> if
<P>
\medskip\bulletitem <CODE>cp.pt</CODE> is in <CODE>cg.rect</CODE>,
   <P>
\medskip\bulletitem <CODE>cp.gone</CODE> is in <CODE>cg.inOut</CODE>, and 
<P>
\medskip\bulletitem either <CODE>cg.screen = AllScreens</CODE> or <CODE>cg.screen = cp.screen</CODE>.
<P>
\medskip\noindent Trestle imposes the restriction on cages that if <CODE>cg.screen
   = AllScreens</CODE>, then <CODE>cg.rect</CODE> must be <CODE>Rect.Full</CODE> or <CODE>Rect.Empty</CODE>,
   and if <CODE>cg</CODE> contains no cursor positions, then it must be equal as a
   record to <CODE>EmptyCage</CODE> (which is declared below).  For example,
   here are some useful cages: 
   
<P><PRE>CONST
  GoneCage =
    Cage{Rect.Full, InOut{TRUE}, AllScreens};
  InsideCage =
    Cage{Rect.Full, InOut{FALSE}, AllScreens};
  EverywhereCage =
    Cage{Rect.Full, InOut{FALSE, TRUE}, AllScreens};
  EmptyCage =
    Cage{Rect.Empty, InOut{}, AllScreens};
</PRE> <CODE>GoneCage</CODE> contains all cursor positions that are ``gone''; set it
   on a <CODE>VBT</CODE> to wait for the cursor to be over a position controlled
   by the <CODE>VBT</CODE>.  The cage <CODE>InsideCage</CODE> is the complement of <CODE>GoneCage</CODE>:
   it contains all positions that the <CODE>VBT</CODE> controls.  The cage
   <CODE>EverywhereCage</CODE> contains all cursor positions, and <CODE>EmptyCage</CODE>
   contains none.
<P>
   Here is the procedure for setting the cage of a <CODE>VBT</CODE>: 
   
<P><PRE>PROCEDURE <A HREF="VBT.m3#SetCage">SetCage</A>(v: T; READONLY cg: Cage);
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Set <CODE>cage(v)</CODE> to the intersection of <CODE>cage(v)</CODE> with <CODE>cg</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> In the usual case, <CODE>SetCage</CODE> is called from <CODE>v</CODE>'s position method,
   at which point <CODE>v</CODE>'s cage is <CODE>EverywhereCage</CODE> and therefore the
   intersection just comes out to <CODE>cg</CODE>.  In unusual cases, it will be found
   that intersecting the new cage with the old is what is required.
   <P>
   The procedure <CODE>CageFromPosition</CODE> is helpful for tracking the
   cursor continuously.  By setting <CODE>CageFromPosition(cp)</CODE> in 
   response to each cursor position <CODE>cp</CODE>, you can track the cursor as long 
   as it moves within your <CODE>VBT</CODE>.  There are two additional optional
   boolean arguments: setting <CODE>trackOutside</CODE> allows you to 
   track the cursor over the whole screen containing the <CODE>VBT</CODE>;  
   setting <CODE>trackOffScreen</CODE> allows you to track the cursor even 
   onto other screens: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#CageFromPosition">CageFromPosition</A>(
  READONLY cp: CursorPosition;
  trackOutside, trackOffScreen: BOOLEAN := FALSE)
  : Cage; &lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> <CODE>CageFromPosition(cp)</CODE> returns the cage that contains only the position 
   <CODE>cp</CODE>; or <CODE>GoneCage</CODE> if either <CODE>cp.gone</CODE> or <CODE>cp.offScreen</CODE> is <CODE>TRUE</CODE>  
   and the corresponding argument is not. </EM></BLOCKQUOTE><PRE>
</PRE> More precisely, <CODE>CageFromPosition</CODE> is equivalent to:
<P>
<PRE>
      IF NOT cp.gone OR
        trackOutside AND NOT cp.offScreen OR
        trackOffScreen
      THEN
        RETURN <KBD>the cage containing only the position</KBD> cp
      ELSIF cp.offScreen AND trackOutside THEN
        RETURN Cage{Rect.Full, InOut{FALSE,TRUE}, cp.screen}
      ELSE
        RETURN GoneCage
      END
</PRE>
Finally, the following two procedures are occasionally useful: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#Outside">Outside</A>(
  READONLY cp: CursorPosition; READONLY c: Cage)
  : BOOLEAN; &lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> Return whether the position <CODE>cp</CODE> is outside the cage <CODE>cg</CODE>. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#CageFromRect">CageFromRect</A>(READONLY r: Rect.T;
  READONLY cp: CursorPosition): Cage; &lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> Return <CODE>Cage{r, InOut{cp.gone}, cp.screen}</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> The effect of <CODE>SetCage(v, CageFromRect(r, cp))</CODE> is to suspend cursor
   positions as long as the cursor stays inside the rectangle <CODE>r</CODE> and
   has the same value of <CODE>gone</CODE> as <CODE>cp</CODE> does.  This is useful when
   sweeping text selections, for example.  

<P> Splits relay cursor positions to their children.  If several
   of the children are tracking the cursor at the same time, the order in
   which positions are relayed to the different children can be
   important.  The order is determined by the following rule, which
   specifies the way a split <CODE>sp</CODE> forwards a <CODE>PositionRec</CODE> <CODE>cd</CODE> to its
   children (the variable <CODE>current(sp)</CODE> is the child that controls the
   last cursor position seen by <CODE>sp</CODE>):
<P>
<PRE>
      IF <KBD>some child </KBD>ch<KBD> controls </KBD>cd.cp<KBD></KBD> THEN
        w := ch
      ELSE
        w := NIL
      END;
      goneCd := cd;
      goneCd.cp.gone := TRUE;
      IF w # current(sp) THEN 
        Deliver(current(sp), goneCd) 
      END;
      FOR <KBD>all </KBD>ch<KBD> other than </KBD>w<KBD> and </KBD>current(sp)<KBD></KBD> DO
        Deliver(ch, goneCd)
      END;
      IF w # NIL THEN Deliver(w, cd) END;
      current(sp) := w
</PRE>
where
<P>
<PRE>
      Deliver(v, cd) =
        IF Outside(cd.cp, cage(v)) THEN
          cage(v) := EverywhereCage;
          v.position(cd)
        END
</PRE>
   A split maintains its cage to be a subset of the intersection of
   its children's cages, so that it will receive any cursor positions
   that it owes its children. 
   
<P>
<P> \subsection{The key method} 

<P> Trestle calls a <CODE>VBT</CODE>'s <CODE>key</CODE> method to report keystrokes.  The
   method will be called with <CODE>LL.sup = mu</CODE>, and takes an argument of
   type <CODE>KeyRec</CODE>.  \index{key method} 
   
<P><PRE>TYPE
  KeyRec = RECORD
    whatChanged: KeySym;
    time: TimeStamp;
    wentDown: BOOLEAN;
    modifiers: Modifiers;
  END;

  KeySym = INTEGER;

CONST
  NoKey: KeySym = 0;
</PRE> The method call <CODE>v.key(cd)</CODE> indicates that the key <CODE>cd.whatChanged</CODE>
   went up or down at time <CODE>cd.time</CODE>.  The boolean <CODE>cd.wentDown</CODE> is
   true if the key went down; false if it went up.  The set
   <CODE>cd.modifiers</CODE> reflects the state of the modifiers (either just
   before or just after the transition; it is not specified which).
<P>
   A <CODE>KeySym</CODE> represents a symbol on a key of the keyboard.  For
   example, there are separate <CODE>KeySyms</CODE> for upper and lower case
   letters.  The interfaces <CODE>Latin1Key</CODE> and <CODE>KeyboardKey</CODE> specify the
   <CODE>KeySym</CODE> codes for many symbols that occur on standard keyboards.
   These interfaces are shipped with SRC Trestle but are not included
   in the printed version of the reference manual.  The codes are chosen
   to agree with the X Keysym codes (see X Window System, Scheifler
   et al., \cite{XSpec} Appendix E).
     <P>
   If the keyboard, like most keyboards, has two symbols on some of
   the keys, then the <CODE>KeySym</CODE> for the down transition and later up
   transition might be different.  For example, if the user pushes the
   left shift key, then the <CODE>z</CODE>/<CODE>Z</CODE> key, and then releases the keys in
   the same order, Trestle would report these four transitions:
<P>
<PRE>
      <KBD>left shift down</KBD>, modifiers = {} or {Shift}
      Z <KBD>down</KBD>, modifiers = {Shift}
      <KBD>left shift up</KBD>, modifiers = {} or {Shift}
      z <KBD>up</KBD>, modifiers = {}
</PRE>
   Although the same physical <CODE>Z</CODE>/<CODE>z</CODE> key went down and up, the down
   transition is reported for the <CODE>Z</CODE> <CODE>KeySym</CODE> and the up transition
   is reported for the <CODE>z</CODE> <CODE>KeySym</CODE>.
<P>
   The constant <CODE>NoKey</CODE> is simply an unused <CODE>KeySym</CODE> code.  
   <P>
   To get Trestle to deliver keystrokes to a <CODE>VBT</CODE>, you make the <CODE>VBT</CODE>
   the owner of the keyboard focus by calling the procedure
   <CODE>VBT.Acquire</CODE>.  

<P>
<P> \subsection{The redisplay method} 

<P> A typical <CODE>VBT</CODE> has a ``display invariant'' that defines what its
   screen looks like as a function of its state.  When the state changes,
   the display invariant is reestablished by updating the screen.
   \index{redisplay~method} \index{marking~for~redisplay}
   <P>
   When a series of changes are made, each of which invalidates the
   display invariant, it is undesirable to update the screen
   after every change.  For example, if the border width and the
   border texture of a <CODE>BorderedVBT</CODE> both change, it is better not
   to paint the intermediate state.
<P>
   Therefore, Trestle keeps track of a set of <CODE>VBTs</CODE> that have been
   ``marked for redisplay''.  Procedures that invalidate a <CODE>VBT</CODE>'s
   display invariant mark the <CODE>VBT</CODE> instead of updating the screen
   directly.  Trestle automatically schedules a call to the <CODE>redisplay</CODE>
   method of every marked window (unless the window's screentype is
   <CODE>NIL</CODE>).  The method takes no arguments: the call <CODE>v.redisplay()</CODE>
   must reestablish <CODE>v</CODE>'s display invariant.  It will be called with
   <CODE>LL.sup = mu</CODE>.
<P>
   The default redisplay method for a <CODE>Leaf</CODE> calls the reshape method
   with an empty <CODE>saved</CODE> rectangle.
   <P>
   There are several procedures related to redisplay:  
   
<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Mark">Mark</A>(v: T); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Mark <CODE>v</CODE> for redisplay. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#IsMarked">IsMarked</A>(v: T): BOOLEAN; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return <CODE>TRUE</CODE> if <CODE>v</CODE> is marked for redisplay. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#Unmark">Unmark</A>(v: T); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> If <CODE>v</CODE> is marked for redisplay, unmark it. </EM></BLOCKQUOTE><PRE>
</PRE> A marked window is automatically unmarked when it is redisplayed,
   reshaped, or rescreened.  Thus the <CODE>Unmark</CODE> procedure is rarely
   needed.  
    
<P>
<P> \subsection{The reshape method} 

<P> Trestle calls a <CODE>VBT</CODE>'s reshape method to report changes in its
   domain.  The method will be called with <CODE>LL.sup = mu.v</CODE> (as explained
   below), and takes an argument of type <CODE>ReshapeRec</CODE>.  
   \index{reshape method} 
   
<P><PRE>TYPE ReshapeRec = RECORD
  new, prev, saved: Rect.T;
  marked: BOOLEAN
END;
</PRE> The method call <CODE>v.reshape(cd)</CODE> indicates that the domain of <CODE>v</CODE>
   has changed from <CODE>cd.prev</CODE> to <CODE>cd.new</CODE>.  The rectangle <CODE>cd.saved</CODE>
   is the subset of the previous domain that Trestle has preserved for
   the client in case it is of use in painting the new domain.  This
   is the only case in which Trestle tries to save portions of a <CODE>VBT</CODE>'s
   screen outside its domain.  After the reshape method returns, Trestle
   will generally forget the old parts of the screen.  The boolean
   <CODE>cd.marked</CODE> indicates whether <CODE>v</CODE> was marked when it was reshaped;
   in any case, <CODE>v</CODE> is automatically unmarked as it is reshaped.
<P>
   If <CODE>new = Rect.Empty</CODE> then the window is no longer visible (for 
   example, this happens when the window is iconized).  Any background 
   threads that are painting should be stopped, since their efforts 
   are useless. 
<P>
   The default reshape method for a <CODE>Leaf</CODE> calls the <CODE>repaint</CODE> method
   to repaint the whole new domain.
<P>
   When the reshape method is called, <CODE>mu</CODE> is locked, and it will remain
   locked until the method returns.  However, Trestle may lock <CODE>mu</CODE>
   and then reshape, repaint, or rescreen several <CODE>VBTs</CODE> concurrently,
   so you can't assume that an activation of your reshape method
   excludes the activation of another <CODE>VBT</CODE>'s reshape, repaint, or
   rescreen method.
   <P>
   This locking level will be referred to as <CODE>v</CODE>'s share of <CODE>mu</CODE>,
   and written <CODE>mu.v</CODE>.  Holding <CODE>mu</CODE> is logically equivalent to holding
   <CODE>mu.v</CODE> for every <CODE>v</CODE>.  Consequently, <CODE>mu.v &lt; mu</CODE> in the locking
   order.  Holding <CODE>mu.v</CODE> does not suffice to call a procedure that
   requires <CODE>mu</CODE> to be locked; on the other hand you cannot lock <CODE>mu</CODE>
   while holding <CODE>mu.v</CODE>, since this would deadlock.  

<P> \subsection{The rescreen method} 

<P> Trestle calls a <CODE>VBT</CODE>'s rescreen method to report changes to its
   screentype.  The method will be called with <CODE>LL.sup = mu.v</CODE>, and
   takes an argument of type <CODE>RescreenRec</CODE>. \index{rescreen method} 
   
<P><PRE>TYPE RescreenRec = RECORD
  prev: Rect.T;
  st: ScreenType;
  marked: BOOLEAN;
END;
</PRE> The method call <CODE>v.rescreen(cd)</CODE> indicates that the screentype of
   <CODE>v</CODE> has changed to <CODE>cd.st</CODE> and that its domain has changed from
   <CODE>cd.prev</CODE> to <CODE>Rect.Empty</CODE>.  (Typically the <CODE>VBT</CODE> will be reshaped
   to a non-empty domain on the new screentype.)  It is possible that
   <CODE>cd.st=NIL</CODE>.  The boolean <CODE>cd.marked</CODE> indicates whether <CODE>v</CODE> was
   marked when it was rescreened; in any case, <CODE>v</CODE> is automatically
   unmarked as it is rescreened.  <CODE>VBT.Leaf.rescreen</CODE> reshapes <CODE>v</CODE> to 
   empty.  

<P> \subsection{The repaint method} 

<P> Trestle calls a <CODE>VBT</CODE>'s repaint method to report that part of its
   screen has been exposed and must be repainted.  The method
   will be called with <CODE>LL.sup = mu.v</CODE>, and takes an argument of type
   <CODE>Region.T</CODE>.  \index{repaint method}
   <P>
   There are some subtleties if you are scrolling (that is, copying 
   bits from one part of the screen to another) at the same time that 
   Trestle is activating your repaint method.  To explain them we will 
   become more formal and precise.
<P>
   Every <CODE>VBT</CODE> <CODE>v</CODE> has a ``bad region'' <CODE>bad(v)</CODE>.  For each point <CODE>p</CODE>
   that is in <CODE>Domain(v)</CODE> and not in <CODE>bad(v)</CODE>, the pixel <CODE>v[p]</CODE> is
   displayed to the user; that is, if <CODE>vis[p]</CODE> denotes what is actually
   visible at pixel <CODE>p</CODE>, then we have the basic invariant
   \index{bad region} \index{exposed region}
<P>
<PRE>
      vis[p] = v[p] <KBD>for all</KBD> p <KBD>controlled by </KBD>v<KBD> and outside</KBD> bad(v) 
</PRE>
   Trestle can expand <CODE>bad(v)</CODE> at any time, as though
   cosmic rays had damaged the pixels.
<P>
   Whenever <CODE>bad(v)</CODE> contains pixels that are controlled by <CODE>v</CODE>, 
   Trestle will call <CODE>v</CODE>'s repaint method by setting <CODE>exposed(v)</CODE> 
   (the ``exposed region'' of <CODE>v</CODE>) to include all such pixels, and then 
   executing the following code:
<P>
<PRE>
      &lt; bad(v) := <KBD>the set difference</KBD> bad(v) - exposed(v);   
        FOR p <KBD>in</KBD> exposed(v) DO v[p] := vis[p] END &gt;;
      v.repaint(exposed(v));
      exposed(v) := <KBD>the empty set</KBD>
</PRE>
   That is, as a pixel <CODE>p</CODE> is removed from <CODE>bad(v)</CODE> and added to
   <CODE>exposed(v)</CODE>, the screen <CODE>v[p]</CODE> is changed to <CODE>vis[p]</CODE>, so that the
   basic invariant is maintained.  You can imagine that the cosmic ray's
   damage has now reached <CODE>v[p]</CODE>, not just <CODE>vis[p]</CODE>.  The angle brackets
   indicate that the shrinking of <CODE>bad(v)</CODE> and the damaging of <CODE>v[p]</CODE>
   occur atomically, so that the basic invariant is maintained. (In
   particular, the basic invariant is true whenever you call the
   procedure <CODE>VBT.Scroll</CODE>, where you can find more about the
   bad region and the exposed region.)
<P>
   Sometimes it is convenient to do all painting from the repaint
   method; in which case the following procedure is useful:  

<P><PRE>PROCEDURE <A HREF="VBT.m3#ForceRepaint">ForceRepaint</A>(v: T; READONLY rgn: Region.T);
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Set <CODE>bad(v) := Region.Join(rgn, bad(v))</CODE>.  If the resulting
   <CODE>bad(v)</CODE> is non-empty, schedule an activation of <CODE>v</CODE>'s repaint 
   method. </EM></BLOCKQUOTE><PRE>
</PRE> \subsection{About painting in general} 

<P> Trestle's painting procedures all follow the same pattern.
   The arguments to the procedure specify: 
<P>
    \medskip\bulletitem a {\it destination}, which is a set of pixels
      in a <CODE>VBT</CODE>'s screen.  For example, the destination could be a
      rectangle, a trapezoid, a shape bounded by a curved path, or a
      region.
<P>
    \medskip\bulletitem a {\it source}, which is conceptually an
      infinite array of pixels, not necessarily of the same depth as
      those on the screen.  For example, the source could be a texture,
      a text string in some font, an explicit bitmap or image, or the
      <CODE>VBT</CODE>'s screen itself.  
<P>
    \medskip\bulletitem an {\it operation}, which is a function that takes
      a destination pixel value and a source pixel value and produces a 
      destination pixel value.  For example, the operation could be 
      planewise <CODE>XOR</CODE>. 
<P>
   \medskip\noindent The effect of the painting procedure is to apply
   the operation to each pixel in the destination region.  That is,
   if <CODE>v</CODE> is the <CODE>VBT</CODE>, the effect of the painting procedure is to set
   <CODE>v[p] := op(v[p], s[p])</CODE> for each point <CODE>p</CODE> in the destination, where
   <CODE>op</CODE> is the operation, <CODE>v[p]</CODE> is the pixel at point <CODE>p</CODE> of <CODE>v</CODE>'s
   screen, and <CODE>s[p]</CODE> is the source pixel at point <CODE>p</CODE>.
<P>
   Two useful operations are <CODE>PaintOp.Bg</CODE> and <CODE>PaintOp.Fg</CODE>,
   defined by
<P>
<PRE>
      PaintOp.Bg(d, s) = <KBD>the screen's background pixel</KBD>
      PaintOp.Fg(d, s) = <KBD>the screen's foreground pixel</KBD>
</PRE>
   These operations ignore their arguments; they set each destination
   pixel to a constant value, regardless of its previous value or the
   source value.  The actual background and foreground pixels vary from
   screentype to screentype; you can think of <CODE>Bg</CODE> as white and <CODE>Fg</CODE>
   as black (unless you prefer video-reversed screens).
<P>
   Another useful operation is <CODE>PaintOp.Copy</CODE>, defined by
<P>
<PRE>
      PaintOp.Copy(d, s) = s
</PRE>
   For example, <CODE>PaintOp.Copy</CODE> can be used to paint an eight-bit pixmap 
   source on an eight-bit pixmap screen.  It would be an error to use
   <CODE>PaintOp.Copy</CODE> with a one-bit source and an eight-bit screen---the
   system wouldn't crash, but anything could happen to the destination
   pixels.s  
<P>
   For more painting operations, see the <CODE>PaintOp</CODE> interface.  

<P> \subsection{Scrolling (copying one part of the screen to another)} 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Scroll">Scroll</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    READONLY delta: Point.T;
    op: PaintOp.T := PaintOp.Copy); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Translate a rectangle of <CODE>v</CODE>'s screen by <CODE>delta</CODE> and use 
   it as a source for the operation <CODE>op</CODE> applied to each
   destination pixel in the clipping rectangle <CODE>clip</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> The <CODE>Scroll</CODE> procedure uses <CODE>v</CODE>'s screen as source.  It can
   therefore be used to copy pixels from one part of <CODE>v</CODE>'s screen 
   to another.  Any operation can be used for combining the 
   translated pixels with the destination pixels, but the operation 
   defaults to <CODE>PaintOp.Copy</CODE>.
<P>
   The source rectangle can be computed from <CODE>clip</CODE> by subtracting
   <CODE>delta</CODE>.  More precisely, <CODE>Scroll(v, clip, delta, op)</CODE> is equivalent
   to:
<P>
<PRE>
      <KBD>for each pair of points </KBD>p<KBD>, </KBD>q<KBD> such that</KBD>
           p <KBD>is in</KBD> clip, 
           p = q + delta<KBD>, and</KBD>
           q <KBD>is in</KBD> Domain(v)
      <KBD>simultaneously assign</KBD>
           v[p] := op(v[p], v[q]);
           <KBD>if </KBD>q<KBD> is in </KBD>exposed(v)<KBD> and </KBD>p<KBD> is not,</KBD> 
               <KBD>or if </KBD>q<KBD> is in </KBD>bad(v)<KBD></KBD>
           <KBD>then add </KBD>p<KBD> to </KBD>bad(v)<KBD></KBD>
</PRE>
   By ``simultaneously'' it is meant that the pairs <CODE>p</CODE>, <CODE>q</CODE> are
   enumerated in an order so that no destination pixel of an early pair
   corresponds to a source pixel of any later pair.
   \index{bad region} \index{exposed region}
<P>
   Recall the bad region and exposed region <CODE>bad(v)</CODE> and <CODE>exposed(v)</CODE>
   from the description of the repaint method.  
   <P>
   If you do all your painting from within the <CODE>repaint</CODE>, <CODE>reshape</CODE>,
   and <CODE>redisplay</CODE> methods, then you can ignore the subtleties involving
   the <CODE>bad(v)</CODE> and <CODE>exposed(v)</CODE>.  But if you have any asynchronous
   threads that call <CODE>Scroll</CODE>, you have to be careful.  For example,
   suppose you do all your painting from a concurrent worker thread,
   and arrange for your repaint and reshape methods to simply add entries
   to the worker thread's queue recording the painting that must be
   done.  Then you must be careful to avoid the following sequence of events:
<P>
    \medskip\bulletitem The worker thread removes from its work queue
   an item indicating that it must repaint some region <CODE>A</CODE>, and
   determines that the best way to do this is to scroll some other
   region <CODE>B</CODE>.
<P>
    \medskip\bulletitem The <CODE>repaint</CODE> method is activated with exposed
   region <CODE>B</CODE>; it adds <CODE>B</CODE> to the work queue and returns.  As it
   returns, the system sets the <CODE>VBT</CODE>'s bad and exposed regions to be
   empty. (See the description of the <CODE>repaint</CODE> method.)
   <P>
    \medskip\bulletitem The worker thread copies the garbage from <CODE>B</CODE>
    into <CODE>A</CODE>.
<P>
   \medskip Eventually the worker thread will get around to repainting
   <CODE>B</CODE>, but the damage to <CODE>A</CODE> will never be repaired.
<P>
   To avoid this race condition, the repaint method should convey
   the bad region to the worker thread by a separate communication path, 
   rather than simply put it the ordinary work queue.  The worker thread
   can thus avoid using bad bits as the source of scroll operations.
<P>
   Of course it is possible for the scrolling to happen after the
   <CODE>repaint</CODE> method is called but before the method has conveyed the
   bad region to the worker thread.  There is no way to prevent this
   sequence of events, but there is no need to, either: in this case
   the source of the scroll operation will be in the exposed region
   (since the <CODE>repaint</CODE> method has not yet returned), and therefore
   (by the specification above) the call to <CODE>Scroll</CODE> will expand the
   bad region.  This will eventually lead to the repaint method being
   activated a second time, repairing the damage.
<P>
   In short, in order to allow concurrent painting, we do not clear the
   exposed region until the <CODE>repaint</CODE> method returns, and we specify that
   a scroll from a <CODE>q</CODE> in <CODE>bad(v)</CODE> or <CODE>exposed(v)</CODE> to a <CODE>p</CODE> that is
   not in <CODE>bad(v)</CODE> invalidates the destination.  
   <P>
   Notice that a scroll from <CODE>exposed(v)</CODE> to <CODE>exposed(v)</CODE> does not
   invalidate the destination.  This allows the repaint method to paint
   a portion of <CODE>exposed(v)</CODE> and then scroll that portion to other parts
   of <CODE>exposed(v)</CODE>---unusual, but legal. 

<P> \subsection{Painting textures} 

<P> This section describes procedures for texturing rectangles, regions, and trapezoids. 

<P><PRE>PROCEDURE <A HREF="VBT.m3#PaintTexture">PaintTexture</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the rectangle <CODE>clip</CODE> with the texture <CODE>src+delta</CODE> using
   the operation <CODE>op</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> A {\it texture} is an infinite periodic pixmap.  A texture <CODE>txt</CODE>
is represented by a pixmap <CODE>src</CODE> with a finite non-empty rectangular
domain <CODE>Domain(src)</CODE>; the rule is that <CODE>txt</CODE> is the result of tiling
the plane with translates of the pixmap <CODE>src</CODE>.  Using the convenient
procedure <CODE>Rect.Mod</CODE> we can state this rule as: <CODE>txt[p] = src[Rect.Mod(p,
Domain(src))]</CODE>.
 <P>
The texture <CODE>src+delta</CODE> is the translation of the texture <CODE>src</CODE> by
the vector <CODE>delta</CODE>.
   <P>
Putting this all together, <CODE>PaintTexture(v, clip, op, src, delta)</CODE>
is equivalent to:
<P>
<PRE>
      <KBD>for each pair of points </KBD>p<KBD>, </KBD>q<KBD> such that</KBD>
          p <KBD>is in</KBD> clip <KBD>and</KBD>
          p = q + delta
      <KBD>assign</KBD>
          v[p] := op(v[p], src[Rect.Mod(q, Domain(src))]).
</PRE>
   Note that setting <CODE>delta</CODE> to <CODE>Point.Origin</CODE> causes the texture to
   be aligned in an absolute coordinate system independent of the domain
   of the window (which helps to make textures in different windows
   match), while setting it to the northwest corner of <CODE>v</CODE>'s domain
   causes the texture to be aligned in the window's coordinate system
   (which allows a window to be reshaped by scrolling the old domain
   into the new).
<P>
   If <CODE>src</CODE>'s domain is empty, the effect is undefined but limited to 
   the clipping region. 
   <P>
   The default paint operation for PaintTexture is <CODE>BgFg</CODE>, defined by
<P>
<PRE>
      PaintOp.BgFg(d, 0) = <KBD>the screen's background pixel</KBD>
      PaintOp.BgFg(d, 1) = <KBD>the screen's foreground pixel</KBD>
</PRE>
   This paint operation is only appropriate if <CODE>src</CODE> is one-bit deep;
   the effect is to copy the source to the destination, interpreting
   0 as background and 1 as foreground.  

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#PaintTint">PaintTint</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    op: PaintOp.T); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the rectangle <CODE>clip</CODE> with the texture <CODE>Pixmap.Solid</CODE> using
   the operation <CODE>op</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> For example, <CODE>PaintTint(v, clip, PaintOp.Bg)</CODE> paints <CODE>clip</CODE> with
   the background color, and <CODE>PaintTint(v, clip, PaintOp.Fg)</CODE> paints
   <CODE>clip</CODE> with the foreground color.  

<P><PRE>PROCEDURE <A HREF="VBT.m3#PolyTint">PolyTint</A>(
    v: Leaf;
    READONLY clip: ARRAY OF Rect.T;
    op: PaintOp.T); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint each rectangle <CODE>clip[i]</CODE> in order with the texture
   <CODE>Pixmap.Solid</CODE> using the operation <CODE>op</CODE>.  </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#PolyTexture">PolyTexture</A>(
    v: Leaf;
    READONLY clip: ARRAY OF Rect.T;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint each rectangle <CODE>clip[i]</CODE> in order with the texture <CODE>src+delta</CODE> using
   the operation <CODE>op</CODE>. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#PaintRegion">PaintRegion</A>(
    v: Leaf;
    READONLY rgn: Region.T;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T := Pixmap.Solid;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the region <CODE>rgn</CODE> with the texture <CODE>src+delta</CODE> using
   the operation <CODE>op</CODE>. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#PaintTrapezoid">PaintTrapezoid</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    READONLY trap: Trapezoid.T;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T := Pixmap.Solid;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the intersection of <CODE>clip</CODE> and <CODE>trap</CODE> with the texture 
   <CODE>src+delta</CODE> using the operation <CODE>op</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> \subsection{Filling and stroking paths} 

<P> Trestle also supports PostScript-like graphics operations
   \cite{PostScript}: 

<P><PRE>TYPE
  WindingCondition = {Odd, NonZero};
  EndStyle = {Round, Butt, Square};
  JoinStyle = {Round, Bevel, Miter};

PROCEDURE <A HREF="VBT.m3#Fill">Fill</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    path: Path.T;
    wind := WindingCondition.NonZero;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T := Pixmap.Solid;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the intersection of <CODE>clip</CODE> and the region entwined by <CODE>path</CODE> 
   with the texture <CODE>src+delta</CODE> using the operation <CODE>op</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> The point <CODE>p</CODE> is entwined by <CODE>path</CODE> if the winding number of <CODE>path</CODE>
   around <CODE>p</CODE> satisfies the winding condition <CODE>wind</CODE>.  To ensure that
   the winding number is defined even for the points on the path, the
   path is regarded as translated north by $\epsilon$ and west by
   $\epsilon^2$, where $\epsilon$ is infinitesimal.  

<P><PRE>PROCEDURE <A HREF="VBT.m3#Stroke">Stroke</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    path: Path.T;
    width: CARDINAL := 0;
    end := EndStyle.Round;
    join := JoinStyle.Round;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T := Pixmap.Solid;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the intersection of <CODE>clip</CODE> and the stroke determined by <CODE>path</CODE>,
   <CODE>end</CODE>, and <CODE>join</CODE> with the texture <CODE>src+delta</CODE> using the operation
   <CODE>op</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> The exact results of <CODE>Stroke</CODE> are different on different Trestle
   implementations.  The approximate specification is like PostScript:
<P>
   If <CODE>end = Round</CODE> and <CODE>join = Round</CODE>, the path is drawn by a
   circular brush of diameter <CODE>width</CODE> that traverses the path.
<P>
   If <CODE>end = Butt</CODE>, then the ends of unclosed trails in the path are
   stroked by a line segment of length <CODE>width</CODE> centered and
   perpendicular to the path in the neighborhood of the endpoint. If
   <CODE>end = Square</CODE>, the path is extended at the endpoint by a straight
   line segment of length <CODE>width/2</CODE> tangent to the path and a butt end is
   drawn.
<P>
   If <CODE>join = Bevel</CODE>, the joint between two patches is constructed
   by using <CODE>Butt</CODE> endstyles for them and then filling the triangular
   notch that remains. If <CODE>join = Miter</CODE>, then instead of just
   filling the triangular notch, the outer edges of the two lines are
   extended to meet at a point, and the resulting quadrilateral is
   filled. 
 <P>
   If <CODE>width = 0</CODE>, <CODE>join</CODE> is ignored and <CODE>end</CODE> determines whether the
   final endpoint of an open subpath should be drawn: if <CODE>end</CODE> is
   <CODE>Butt</CODE>, the final endpoint is omitted, otherwise it is drawn.
<P>
   If <CODE>join = Miter</CODE>, <CODE>width &gt; 0</CODE>, and the angle formed by the two
   segments meeting at some joint is small, then the tip of the miter may
   extend quite far from the joint point. Trestle implementations are
   free to bevel those joints whose angle is smaller than some
   implementation-dependent {\it miter limit}. The miter limit will be
   made available to clients in some to-be-determined interface.  On X,
   the miter limit is 11 degrees.
<P>
   Finally, there is a convenience procedure for stroking a path 
   containing a single straight line segment: 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Line">Line</A>(
    v: Leaf;
    READONLY clip: Rect.T;
    p, q: Point.T;
    width: CARDINAL := 0;
    end := EndStyle.Round;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T := Pixmap.Solid;
    READONLY delta := Point.Origin); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Like <CODE>Stroke</CODE> applied to the path containing the segment <CODE>(p,q)</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> \subsection{Painting pixmaps} 

<P> The following procedure paints a pixmap without replicating it into an 
   infinite texture: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#PaintPixmap">PaintPixmap</A>(
    v: Leaf;
    READONLY clip: Rect.T := Rect.Full;
    op: PaintOp.T := PaintOp.BgFg;
    src: Pixmap.T;
    READONLY delta: Point.T); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Translate the pixmap <CODE>src</CODE> by <CODE>delta</CODE> and paint it on the screen
of <CODE>v</CODE>, using the operation <CODE>op</CODE> and clipping to the rectangle <CODE>clip</CODE>.
</EM></BLOCKQUOTE><PRE>
</PRE> More precisely, <CODE>PaintPixmap(v, clip, op, src, delta)</CODE> is 
equivalent to
<P>
<PRE>
      <KBD>for each pair of points </KBD>p<KBD>, </KBD>q<KBD> such that</KBD>
          p <KBD>is in</KBD> clip,
          q <KBD>is in</KBD> Domain(src)<KBD>, and</KBD>
          p = q + delta,
      <KBD>assign</KBD>
          v[p] := op(v[p], src[q])
</PRE>
Since a <CODE>Pixmap.T</CODE> is a screen-independent resource, you can't read its domain without specifying the <CODE>VBT</CODE> it is to be used on: 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#PixmapDomain">PixmapDomain</A>(v: T; pix: Pixmap.T): Rect.T;
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return the domain of <CODE>pix</CODE> on the screentype of <CODE>v</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> It is also possible to paint screen-dependent pixmaps: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#PaintScrnPixmap">PaintScrnPixmap</A>(
    v: Leaf;
    READONLY clip: Rect.T := Rect.Full;
    op: PaintOp.T := PaintOp.Copy;
    src: ScrnPixmap.T;
    READONLY delta: Point.T); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM>  Like <CODE>PaintPixmap</CODE>, but with a screen-dependent pixmap instead
of a screen-independent pixmap.  </EM></BLOCKQUOTE><PRE>
</PRE> If <CODE>src</CODE> does not have an appropriate screentype for <CODE>v</CODE>, the effect
   of the procedure is undefined but limited to the clipping
   region.
<P>
   Because Trestle batches painting operations, the pixmap <CODE>src</CODE>
   must be regarded as still in use after <CODE>PaintScrnPixmap</CODE> 
   returns.  If you wish to free the pixmap by calling <CODE>src.free()</CODE>,
   you should first call <CODE>VBT.Sync(v)</CODE>. 

<P> \subsection{Painting text} 

<P> The text painting procedures take an optional array of
displacements, whose entries have the following type: 

<P><PRE>TYPE
  DeltaH = [-512 .. 511];
  Displacement =
    RECORD index: CARDINAL; dh: DeltaH END;
</PRE> A displacement <CODE>d</CODE> causes all characters whose index in the text
   is <CODE>d.index</CODE> or greater to be displaced <CODE>d.dh</CODE> pixels to the right.
   The first character has index <CODE>0</CODE>. The <CODE>d.index</CODE> values in an array
   of displacements must be non-decreasing.  

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#PaintText">PaintText</A>(
    v: Leaf;
    READONLY clip: Rect.T := Rect.Full;
    READONLY pt: Point.T;
    fnt: Font.T := Font.BuiltIn;
    t: TEXT;
    op: PaintOp.T := PaintOp.TransparentFg;
    READONLY dl := ARRAY OF Displacement{});
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Paint the text <CODE>t</CODE> onto the screen of <CODE>v</CODE>, starting
   at position <CODE>pt</CODE>, using the font <CODE>fnt</CODE>, the operation <CODE>op</CODE>,
   and the displacement list <CODE>dl</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> The arguments to <CODE>PaintText</CODE> must satisfy at least one of the
   following two conditions:
<P>
   \medskip\bulletitem the background operation is transparent; that is,
   <CODE>op(p, 0) = p</CODE> for any pixel <CODE>p</CODE>, or
<P>
   \medskip\bulletitem the font is self-clearing (see below) and 
   <CODE>dl</CODE> is empty.  
<P>
   \medskip\noindent If neither condition is true, the effect of <CODE>PaintText</CODE> is
   implementation-dependent, but is confined to the clipping rectangle.
   <P>
   The <CODE>ScrnFont</CODE> interface defines the properties of fonts.
   Here we introduce names for the properties needed to
   explain <CODE>PaintText</CODE>.  If <CODE>f</CODE> is a font and <CODE>ch</CODE> is a character, then 
<P>
       \medskip\bulletitem  <CODE>printWidth(ch, f)</CODE> is the printing width of <CODE>ch</CODE>; 
       that is, the amount to increment the reference point when <CODE>ch</CODE> is 
       printed in font <CODE>f</CODE>;
<P>
       \medskip\bulletitem <CODE>bits(ch, f)</CODE> is the bitmap for <CODE>ch</CODE> in <CODE>f</CODE>, which
        is positioned with <CODE>ch</CODE>'s reference point at the origin;
<P>
       \medskip\bulletitem <CODE>height(ch, f)</CODE> is the height of <CODE>ch</CODE> above
       the baseline; that is, the number of rows of <CODE>bits(ch, f)</CODE> whose
       <CODE>v</CODE>-coordinate is at most zero; and <CODE>depth(ch, fnt)</CODE> is the
       number of rows of <CODE>bits(ch, f)</CODE> whose <CODE>v</CODE>-coordinate exceeds
       zero;
       <P>
       \medskip\bulletitem <CODE>ascent(f)</CODE> and <CODE>descent(f)</CODE> are the logical extent
       of <CODE>f</CODE> above and below the baseline.  Some characters may extend
       higher or lower.
<P>
\medskip\noindent A font is {\it self-clearing} if 
<P>
\medskip\bulletitem
each character's
height and depth equal the font's ascent and descent, and 
<P>
\medskip\bulletitem
each character's
<CODE>printWidth</CODE> equals the width of its bitmap and each character's
reference point is at the west boundary of its bitmap (or each
character's <CODE>printWidth</CODE> equals the negative of the width of its bitmap
and each character's reference point is at the east boundary of its
bitmap).
<P>
\medskip\noindent The call to <CODE>PaintText</CODE> is equivalent to the following loop:
<P>
<PRE>
      rp := pt;
      i := 0;
      LOOP
        IF dl # NIL THEN 
          FOR j := 0 TO HIGH(dl^) DO
            IF dl[j].index = i THEN INC(rp.h, dl[j].dh) END
          END
        END;
        IF i = Text.Length(t) THEN EXIT END;
        PaintPixmap(v, clip, op, bits(t[i], fnt), rp);
        rp.h := rp.h + PrintWidth(t[i], fnt);
        i := i + 1
      END
    

<P> </PRE>
The following two procedures are useful for computing the sizes
   of texts.  Since fonts are screen-independent, they take the
   <CODE>VBT</CODE> whose screentype is to be used:  

<P><PRE>PROCEDURE <A HREF="VBT.m3#BoundingBox">BoundingBox</A>
  (v: Leaf; txt: TEXT; fnt: Font.T): Rect.T;
  &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return the bounding box of the text <CODE>txt</CODE> if it were painted
   at the origin on the screen of <CODE>v</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> More precisely, let <CODE>r</CODE> be the smallest rectangle that contains the
   bounding boxes of the characters of <CODE>txt</CODE> if <CODE>txt</CODE> were painted on
   <CODE>v</CODE> in the font <CODE>fnt</CODE> with <CODE>txt</CODE>'s reference point at the origin.
   Then <CODE>BoundingBox</CODE> returns a rectangle with the same horizontal extent
   as <CODE>r</CODE>, but whose height and depth are the maximum height and depth
   of any character in the font.  

<P><PRE>PROCEDURE <A HREF="VBT.m3#TextWidth">TextWidth</A>
  (v: Leaf; txt: TEXT; fnt: Font.T): INTEGER;
  &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return the sum of the printing widths of the characters in <CODE>txt</CODE>
   in the font <CODE>fnt</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> <CODE>TextWidth</CODE> returns the displacement of the reference point
   that would occur if <CODE>t</CODE> were painted on <CODE>v</CODE> in font <CODE>fnt</CODE>.  It may
   differ from the width of <CODE>BoundingBox(txt, fnt)</CODE>, since the printing
   width of the last character can be different from the width of its
   bounding box, and the reference point for the first character might
   not be at the left edge of <CODE>txt</CODE>'s bounding box. 
   <P>
   You can paint characters out of an array instead of a <CODE>TEXT</CODE>: 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#PaintSub">PaintSub</A>(
    v: Leaf;
    READONLY clip: Rect.T := Rect.Full;
    READONLY pt: Point.T;
    fnt: Font.T := Font.BuiltIn;
    READONLY chars: ARRAY OF CHAR;
    op: PaintOp.T := PaintOp.TransparentFg;
    READONLY dl :=  ARRAY OF Displacement{});
    &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Like <CODE>PaintText</CODE> applied to the characters in <CODE>chars</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> \subsection{Synchronization of painting requests} 

<P> To improve painting performance, Trestle combines painting commands
   into batches, and sends them to the server a batch at a time.  
   <P>
   Most applications can ignore the batching, but the procedures in this
   section can be of use in applications where the timing of paint
   operations is critical.  
   <P>
   For example, when replacing one line of
   text with another in a non-self-clearing font, the old text must
   be erased before the new text is painted.  If the painting command
   that erases the old text happens to fall at the end of a batch, 
   there may be a delay of several milliseconds between the time it
   affects the screen and the time the following paint text command affects
   the screen, which can produce an undesirable flickering effect.  The
   chances of this happening can be greatly reduced by enclosing the 
   two commands in a {\it group}, using the following two procedures:
   \index{paint batch} \index{batch (of painting commands)} 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#BeginGroup">BeginGroup</A>(v: Leaf; sizeHint: INTEGER := 0);
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Begin a group of painting commands. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#EndGroup">EndGroup</A>(v: Leaf); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> End the current group of painting commands. </EM></BLOCKQUOTE><PRE>
</PRE> If a group of painting commands are bracketed by <CODE>BeginGroup</CODE> and 
   <CODE>EndGroup</CODE>, Trestle will try to avoid introducing delays between 
   the commands, such as might otherwise be introduced by batching. 
   Trestle assumes that you will generate the painting commands and 
   the <CODE>EndGroup</CODE> in rapid succession.  
   <P>
   Increasing the value of <CODE>sizeHint</CODE> may improve atomicity, at 
   the cost of throughput.  The maximum 
   useful value of <CODE>sizeHint</CODE> is the total size in bytes of the painting 
   commands in the group, which you can compute using the interface 
   <CODE>PaintPrivate</CODE>. 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Sync">Sync</A>(v: Leaf; wait := TRUE); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Force all painting commands issued to <CODE>v</CODE> prior to the call to be
   executed. If <CODE>wait</CODE> = FALSE then Sync just flushes the output queue
   and returns. Otherwise, Sync waits until it believes the commands
   in the output queue have been completed. </EM></BLOCKQUOTE><PRE>
</PRE> \subsection{Screen capture} 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Capture">Capture</A>(
    v: T;
    READONLY clip: Rect.T;
    VAR (*out*) br: Region.T)
    : ScrnPixmap.T; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return a pixmap containing the part of <CODE>v</CODE>'s screen in
   the rectangle <CODE>rect</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> The screentype of the result will be the same as the screentype of 
   <CODE>v</CODE>.  Because a <CODE>VBT</CODE>'s screen is forgetful, it may be impossible
   to read the requested region.  In this case <CODE>br</CODE> is set to contain
   all positions of pixels that were not copied.  Naturally, Trestle
   makes <CODE>br</CODE> as small as it can.  If none of the bits are available,
   the result may be <CODE>NIL</CODE>. 
   \index{reading the screen} 

<P> \subsection{Controlling the cursor shape} 

<P> Every <CODE>VBT</CODE> <CODE>v</CODE> contains a field <CODE>cursor(v)</CODE>, which is set with the following procedure: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#SetCursor">SetCursor</A>(v: T; cs: Cursor.T);
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Set <CODE>cursor(v)</CODE> to <CODE>cs</CODE>. </EM></BLOCKQUOTE><PRE>
</PRE> A split displays the cursor of its mouse focus, or of its current
   child if its mouse focus is <CODE>NIL</CODE>.  Only if the cursor of the relevant
   child is <CODE>Cursor.DontCare</CODE> or if there is no relevant child does the
   split display its own cursor.
   \index{cursor shape, how to change}
<P>
   To be more precise, the shape of the cursor over the top level 
   window <CODE>v</CODE> is determined by the following recursive procedure:
<P>
<PRE>
      GetCursor(v) =
        IF NOT ISTYPE(v, Split) THEN 
          RETURN cursor(v)
        ELSE
          IF mouseFocus(v) # NIL THEN 
            cs := GetCursor(mouseFocus(v))
          ELSIF current(v) # NIL THEN
            cs := GetCursor(current(v))
          ELSE
            cs := Cursor.DontCare
          END;
          IF cs = Cursor.DontCare THEN
            RETURN cursor(v)
          ELSE
            RETURN cs
          END
        END
 

<P>
<P> </PRE>
\subsection{Selections} 

<P> Trestle maintains an internal table of named selections,
   which initially contains several selections of general use, 
   and which can be extended by users: 

<P>
<P><PRE>TYPE Selection = RECORD sel: CARDINAL END;

PROCEDURE <A HREF="VBT.m3#GetSelection">GetSelection</A>(name: TEXT): Selection;
&lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> Return the selection with the given name, creating it if necessary. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#SelectionName">SelectionName</A>(s: Selection): TEXT;
&lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> Return the name used to create <CODE>s</CODE>, or <CODE>NIL</CODE> if <CODE>s</CODE> is unknown. </EM></BLOCKQUOTE><PRE>

VAR (*CONST*)
  NilSel:  Selection (* := GetSelection(&quot;NilSel&quot;) *);
  Forgery: Selection (* := GetSelection(&quot;Forgery&quot;) *);
  KBFocus: Selection (* := GetSelection(&quot;KBFocus&quot;) *);
  Target:  Selection (* := GetSelection(&quot;Target&quot;) *);
  Source:  Selection (* := GetSelection(&quot;Source&quot;) *);
</PRE> <CODE>NilSel</CODE> and <CODE>Forgery</CODE> are reserved for Trestle's internal use.  
   The owner of <CODE>KBFocus</CODE> (the keyboard focus) is the <CODE>VBT</CODE> that
   receives keystrokes.
   \index{input or keyboard focus} \index{keyboard focus}
<P>
   We offer the following suggestions for the use of target and
   source selections:
   \index{target selection}
   \index{source selection}
<P>
    \medskip\bulletitem The target selection.  If text, this should
      be underlined black or reverse video.  The selection gesture
      should not require modifiers like shift or control.  
<P>
    \medskip\bulletitem The source selection.  If text, this should
      be underlined gray.  The source gesture should be a modified
      version of the gesture for making the target selection.  
   <P>
   \medskip An operation like ``copy'' should replace the
   target selection with the value of the source selection. 

<P>   
<P> The following exception declaration provides for the errors
   that can occur in dealing with selections. 

<P><PRE>EXCEPTION Error(ErrorCode);

TYPE ErrorCode =
  {EventNotCurrent, TimeOut, Uninstalled, Unreadable,
   Unwritable, UnownedSelection, WrongType};
</PRE> Explanation of error codes:
<P>
   \medskip\bulletitem <CODE>EventNotCurrent</CODE>:  Raised by attempts to access a
   selection with an event time that is not current.
<P>
   \medskip\bulletitem <CODE>TimeOut</CODE>:  If you attempt to read or write a selection,
   and the selection owner's method does not return for an unreasonably
   long time, then Trestle stops waiting and raises this exception.
<P>
   \medskip\bulletitem <CODE>Uninstalled</CODE>:  Raised by event-time operations on
   uninstalled <CODE>VBTs</CODE>; that is, on <CODE>VBTs</CODE> none of whose ancestors have
   been connected to a window system by one of the installation
   procedures in the <CODE>Trestle</CODE> interface.
<P>
   \medskip\bulletitem <CODE>Unreadable</CODE>, <CODE>Unwritable</CODE>:  Raised by attempts to read
   an unreadable selection, or write an unwritable selection.
<P>
   \medskip\bulletitem <CODE>UnownedSelection</CODE>:  Raised by attempts to read, write,
   or deliver miscellaneous codes to the owner of an unowned selection.
<P>
   \medskip\bulletitem <CODE>WrongType</CODE>:  Raised by attempts to read or write a
   selection with a type not supported by the selection owner.  

<P>
<P> \subsection{Acquiring and releasing selection ownership} 

<P><PRE>PROCEDURE <A HREF="VBT.m3#Acquire">Acquire</A>(
    v: T;
    s: Selection;
    t: TimeStamp)
  RAISES {Error}; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Make <CODE>v</CODE> the owner of selection <CODE>s</CODE>, provided that <CODE>t</CODE> is the current
   event. </EM></BLOCKQUOTE><PRE>
</PRE> If <CODE>Acquire(v, s, t)</CODE> is successful, the previous owner of the
   selection will receive a miscellaneous code of type <CODE>Lost</CODE> (even
   if the owner is <CODE>v</CODE>).  The window system affected is the one
   to which <CODE>v</CODE> is connected.  The possible error codes are
   <CODE>EventNotCurrent</CODE> and <CODE>Uninstalled</CODE>.  

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Release">Release</A>(v: T; s: Selection);
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> If the current owner of <CODE>s</CODE> is <CODE>v</CODE>, then a <CODE>Lost</CODE> code is queued
   for delivery to <CODE>v</CODE> and the owner of <CODE>s</CODE> becomes <CODE>NIL</CODE> </EM></BLOCKQUOTE><PRE>
</PRE> The window system affected is the one to which <CODE>v</CODE> is connected.
   <CODE>Release</CODE> is a no-op if the current owner is not <CODE>v</CODE> or if <CODE>v</CODE> is
   not installed.  

<P> \subsection{The miscellaneous method} 

<P> Trestle calls a <CODE>VBT</CODE>'s <CODE>misc</CODE> method to deliver miscellaneous
   codes.  The method will be called with <CODE>LL.sup = mu</CODE>, and takes an
   argument of type <CODE>MiscRec</CODE>.  \index{misc method} 
   <P>
   Trestle maintains an internal table of named miscellaneous code
   types, which initially contains several types of general interest,
   and which can be extended by users.  
    
<P><PRE>TYPE MiscRec = RECORD
  type: MiscCodeType;
  detail: MiscCodeDetail;
  time: TimeStamp;
  selection: Selection;
END;

  MiscCodeType = RECORD typ: CARDINAL END;
  MiscCodeDetail = ARRAY [0 .. 1] OF INTEGER;

PROCEDURE <A HREF="VBT.m3#GetMiscCodeType">GetMiscCodeType</A>(name: TEXT): MiscCodeType;
&lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> Return the MiscCodeType with the given name, creating it if
   necessary.  </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#MiscCodeTypeName">MiscCodeTypeName</A>(type: MiscCodeType): TEXT;
&lt;* LL arbitrary *&gt;
</PRE><BLOCKQUOTE><EM> Return the name used to create <CODE>s</CODE>, or <CODE>NIL</CODE> if <CODE>s</CODE> is unknown. </EM></BLOCKQUOTE><PRE>

CONST
   NullDetail = MiscCodeDetail {0, ..};
</PRE> The interface <CODE>MiscDetail</CODE> provides some convenient procedures for
   encoding a REF as an integer for internal miscellaneous codes. 
   
<P><PRE>VAR (*CONST*)
   Deleted: MiscCodeType;
   Disconnected: MiscCodeType;
   TakeSelection: MiscCodeType;
   Lost: MiscCodeType;
   TrestleInternal: MiscCodeType;
   Moved: MiscCodeType;
</PRE> These ``variables'' are really constants for the following codes:
<P>
<PRE>
      GetMiscCodeType(&quot;Deleted&quot;)
      GetMiscCodeType(&quot;Disconnected&quot;)
      GetMiscCodeType(&quot;TakeSelection&quot;)
      GetMiscCodeType(&quot;Lost&quot;)
      GetMiscCodeType(&quot;TrestleInternal&quot;)
      GetMiscCodeType(&quot;Moved&quot;)
</PRE>
   The method call <CODE>v.misc(cd)</CODE> sends <CODE>v</CODE> the misc code relevant
   to <CODE>cd.selection</CODE> as part of the event <CODE>cd.time</CODE>.  The meaning of
   the <CODE>type</CODE> and <CODE>detail</CODE> fields is up to the application, except for
   the following.
<P>
   A <CODE>Deleted</CODE> code is delivered to a top-level window when it is
   explicitly deleted from its server, either by a user command to the window
   manager or under program control.  A <CODE>Disconnected</CODE> code is delivered
   to a top-level window when it is disconnected from its server, either
   because the server crashed or because the network connection was
   lost.  A <CODE>TakeSelection</CODE> code is delivered to a top-level window
   when the user has gestured that it would like the window to acquire
   the indicated selection; most often the keyboard focus.  (The nature
   of the gesture is between the user and the window manager.  Many
   applications also acquire the keyboard focus in response to mouse
   clicks.)  A <CODE>Lost</CODE> code with <CODE>selection = s</CODE> will be delivered to
   a window when it loses ownership of <CODE>s</CODE>.  <CODE>TrestleInternal</CODE> codes
   are reserved for the implementation.  A <CODE>Moved</CODE> code is delivered to a
   top-level window when it is moved in a way that does not cause a
   rescreen or a reshape to be delivered.
<P>
   The timestamp in a <CODE>TakeSelection</CODE> code is the timestamp for the
   current event and is therefore valid for event-time operations.  The
   timestamps in <CODE>Deleted</CODE>, <CODE>Disconnected</CODE>, <CODE>Lost</CODE>, <CODE>Moved</CODE> codes are not.
   The selection field is relevant in <CODE>Lost</CODE> and <CODE>TakeSelection</CODE> codes;
   it is irrelevant in <CODE>Deleted</CODE>, <CODE>Disconnected</CODE>, <CODE>Moved</CODE> codes.
<P>
   

<P>
<P> \subsection{Sending miscellaneous codes} 

<P> You can send a miscellanous code to the owner of a selection by using 
   the following procedure: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#Put">Put</A>(
    v: T;
    s: Selection;
    t: TimeStamp;
    type: MiscCodeType;
    READONLY detail := NullDetail)
  RAISES {Error}; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Create a <CODE>MiscRec</CODE> with the given fields and enqueue it for delivery
  to the owner of selection <CODE>s</CODE>, if <CODE>t</CODE> is the current event-time.
  </EM></BLOCKQUOTE><PRE>
</PRE> The window system affected is the one to which <CODE>v</CODE> is connected.  The
   possible error codes are <CODE>EventNotCurrent</CODE>, <CODE>Uninstalled</CODE>, and
   <CODE>UnownedSelection</CODE>.  If the selection is unowned it is possible that 
   the <CODE>Put</CODE> will be silently ignored. 

<P> \subsection{Circumventing event-time} 

<P> The following procedure offers an escape from the event-time protocol.
   For example, a long-running thread that has no idea what the current 
   event time is can forge a miscellaneous code to itself and use its 
   timestamp to acquire the keyboard focus. (Your users may not like it
   if you do this.) 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Forge">Forge</A>(
    v: T;
    type: MiscCodeType;
    READONLY detail := NullDetail)
  RAISES {Error}; &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Create a <CODE>MiscRec</CODE> with the given <CODE>type</CODE> and <CODE>detail</CODE> fields, with
   selection field <CODE>Forgery</CODE>, and with a newly created timestamp and
   enqueue it for delivery to <CODE>v</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> The timestamp will be valid for event-time operations (provided that
   it is used promptly).  Forging codes that have meaning to the window
   manager (e.g., a <CODE>Deleted</CODE> code) could have unexpected effects if
   they are delivered to installed windows or their descendants.  The
   only possible error code is <CODE>Uninstalled</CODE>.  

<P> \subsection{Communicating selection values} 

<P> When you read the value of a Trestle selection you get a result of 
   type <CODE>Value</CODE>: 
   
<P><PRE>TYPE
  <A HREF="VBT.m3#Value">Value</A> &lt;: Value_Public;
  Value_Public =
    OBJECT METHODS toRef(): REFANY RAISES {Error} END;
</PRE> Call the <CODE>toRef</CODE> method to convert the <CODE>Value</CODE> into a <CODE>REFANY</CODE>.
<P>
  The simplest way to construct a <CODE>Value</CODE> is with the following 
  procedure: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#FromRef">FromRef</A>(r: REFANY): Value;
&lt;* LL.sup &lt;= mu *&gt;
</PRE><BLOCKQUOTE><EM> Return a <CODE>Value</CODE> <CODE>v</CODE> such that <CODE>v.toRef()</CODE> is equal to the result
   of pickling and unpickling <CODE>r</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> On a system without pickles, the value <CODE>r</CODE> must have type <CODE>TEXT</CODE>.
   If <CODE>r</CODE> does not have type <CODE>TEXT</CODE>, any exceptions raised by pickling
   lead to checked run-time errors.
   <P>
   Using <CODE>FromRef</CODE> leads to synchronous transmission of selection
   values---that is, the value is transferred as part of the call to
   <CODE>Read</CODE> or <CODE>Write</CODE>.  To get asynchronous behavior, allocate your own
   <CODE>Values</CODE> and override the <CODE>toRef</CODE> method.  Trestle will transmit
   the <CODE>Value</CODE> to the other application, and only when that application
   calls the <CODE>toRef</CODE> method will your <CODE>toRef</CODE> method be called.
<P>
   The <CODE>toRef</CODE> method in a <CODE>Value</CODE> will be called with <CODE>LL.sup &lt;= mu</CODE>.
   The <CODE>toRef</CODE> method can raise the error <CODE>Unreadable</CODE> if, for example,
   the address space of the selection owner has been destroyed.  It can
   also raise the error <CODE>WrongType</CODE> if the underlying <CODE>REFANY</CODE> cannot be
   represented in the address space calling the method; this can only
   happen with non-<CODE>TEXT</CODE> selections.  
   <P>
   The procedure <CODE>Ready</CODE> tests whether a value is synchronous or 
   asynchronous: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#Ready">Ready</A>(v: Value): BOOLEAN; &lt;* LL.sup &lt;= mu *&gt;
</PRE><BLOCKQUOTE><EM> Return <CODE>TRUE</CODE> if calling <CODE>v.toRef()</CODE> will return quickly; return
   <CODE>FALSE</CODE> if calling <CODE>v.toRef()</CODE> might be slow or block. </EM></BLOCKQUOTE><PRE>
</PRE> Finally, here are the procedures for reading and writing selections: 

<P><PRE>PROCEDURE <A HREF="VBT.m3#Read">Read</A>(
    v: T;
    s: Selection;
    t: TimeStamp;
    tc: INTEGER := -1)
    : Value
  RAISES {Error}; &lt;* LL.sup &lt;= mu *&gt;
</PRE><BLOCKQUOTE><EM> Return the value of selection <CODE>s</CODE> as a reference of type <CODE>tc</CODE>,
  if <CODE>t</CODE> is the current event-time. </EM></BLOCKQUOTE><PRE>
</PRE> If <CODE>tc = -1</CODE>, <CODE>Read</CODE> uses the typecode for <CODE>TEXT</CODE>.  The window system
   affected is the one to which <CODE>v</CODE> is connected.  The <CODE>KBFocus</CODE>
   selection is always unreadable.  If the selection owner's read method
   is erroneous, calling the <CODE>toRef</CODE> method of the returned <CODE>Value</CODE>
   may produce a reference with a typecode other than <CODE>tc</CODE>.  The
   possible error codes are <CODE>EventNotCurrent</CODE>, <CODE>Uninstalled</CODE>,
   <CODE>Unreadable</CODE>, <CODE>WrongType</CODE>, <CODE>TimeOut</CODE>, and <CODE>UnownedSelection</CODE>.  

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#Write">Write</A>(
    v: T;
    s: Selection;
    t: TimeStamp;
    val: Value;
    tc: INTEGER := -1)
  RAISES {Error}; &lt;* LL.sup &lt;= mu *&gt;
</PRE><BLOCKQUOTE><EM> Replace the selection <CODE>s</CODE> with the value <CODE>val</CODE>, which encodes a
   reference with typecode <CODE>tc</CODE>, assuming <CODE>t</CODE> is the current event-time.
   </EM></BLOCKQUOTE><PRE>
</PRE> If <CODE>tc = -1</CODE>, <CODE>Write</CODE> uses the typecode for <CODE>TEXT</CODE>.  The window
   system affected is the one to which <CODE>v</CODE> is connected.  The <CODE>KBFocus</CODE>
   selection is always unwritable.  The possible error codes are
   <CODE>EventNotCurrent</CODE>, <CODE>Uninstalled</CODE>, <CODE>Unwritable</CODE>, <CODE>TimeOut</CODE>, and
   <CODE>WrongType</CODE>.  

<P>
<P> \subsection{The read and write methods} 

<P> Trestle calls a <CODE>VBT</CODE>'s read and write methods to access any
   selections that it owns.  The method will be called with <CODE>LL.sup
   &lt;= mu</CODE> (see below). \index{read method}
<P>
   The signature of the read method is
<P>
<PRE>
      (s: Selection; tc: CARDINAL): Value RAISES {Error}
</PRE>
   Trestle calls <CODE>v.read(s, tc)</CODE> whenever <CODE>v</CODE> is the owner of
   selection <CODE>s</CODE> and some application passes <CODE>s</CODE> and <CODE>tc</CODE> to <CODE>Read</CODE>.
   The method should return the value of the selection, or raise
   <CODE>Error(Unreadable)</CODE> if for some reason the value cannot be delivered,
   or <CODE>Error(WrongType)</CODE> if the selection cannot be converted to the
   requested type.  The methods will be called with <CODE>LL.sup &lt;= mu</CODE>;
   in fact, if the caller of <CODE>Read</CODE> is in the same address space, <CODE>LL</CODE>
   for the method call is the same as <CODE>LL</CODE> for the caller of <CODE>Read</CODE>,
   else <CODE>LL</CODE> for the method call is <CODE>{}</CODE>.
   <P>
   The signature of the write method is      
<P>
<PRE>
      (s: Selection; val: Value; tc: CARDINAL) 
      RAISES {Error} 
</PRE>
   Trestle calls <CODE>v.write(s, val, tc)</CODE> whenever <CODE>v</CODE> is the owner of
   selection <CODE>s</CODE> and some application passes <CODE>s</CODE>, <CODE>val</CODE>, and <CODE>tc</CODE> to
   <CODE>Write</CODE>.  The method should replace the selection with the value
   of <CODE>val</CODE>, or raise the exception with error code <CODE>Unwritable</CODE> if
   for some reason the selection is not writable, or with error code
   <CODE>WrongType</CODE> if the selection cannot be written with the requested
   type.  Trestle does not enforce any consistency between <CODE>tc</CODE> and
   the typecode of the reference <CODE>val.toRef()</CODE>.  For example, if
   <CODE>val.toRef()</CODE> is <CODE>NIL</CODE>, the meaning could be determined by <CODE>tc</CODE>.
   The locking level is the same as for the read method.  \index{write
   method}
<P>
   While a read or write method is active in a descendant of an
   installed window, Trestle will block the delivery to that window
   of any mouse or key events, misc codes, or cursor positions.  If the
   computations are long, it is therefore preferable to do them
   asynchronously, to avoid blocking the user.  

<P>
<P> \subsection{Controlling the shape of a VBT} 

<P> The preferred shape of a <CODE>VBT</CODE> is represented by a pair of 
   records of type <CODE>SizeRange</CODE>, one for each axis: 

<P><PRE>TYPE SizeRange = RECORD lo, pref, hi: CARDINAL END;

CONST DefaultShape =
  SizeRange{lo := 0, pref := 0, hi := 99999};
</PRE> If a <CODE>VBT</CODE>'s preferred shape in the axis <CODE>ax</CODE> is the <CODE>SizeRange</CODE>
   <CODE>sh</CODE>, then the desirable sizes for the <CODE>VBT</CODE> in axis <CODE>ax</CODE>
   range from <CODE>sh.lo</CODE> to <CODE>sh.hi-1</CODE>, and its preferred size is
   <CODE>sh.pref</CODE>.  
<P>
   A <CODE>SizeRange</CODE> <CODE>sh</CODE> is illegal unless <CODE>sh.lo &lt;= sh.pref &lt; sh.hi</CODE>.
   <P>
   When a parent <CODE>VBT</CODE> divides its screen up between its children, it
   tries to satisfy its children's shape requirements, which it finds
   by calling the children's shape method.
   <P>
   The signature of the shape method is      
<P>
<PRE>
      (ax: Axis.T; n: CARDINAL): SizeRange 
</PRE>
   The behavior of the shape method depends on whether <CODE>n</CODE> is zero.
   The call <CODE>v.shape(ax, 0)</CODE> returns the preferred shape for <CODE>v</CODE> in
   the <CODE>ax</CODE> axis, assuming nothing is known about its size in the other
   axis.  If <CODE>n#0</CODE>, the call <CODE>sh := v.shape(ax, n)</CODE> returns the
   preferred shape for <CODE>v</CODE> in the <CODE>ax</CODE> axis assuming that <CODE>v</CODE>'s size
   in the other axis is <CODE>n</CODE>. When the method is called, <CODE>LL.sup = mu.v</CODE>.
<P>
   It is a checked runtime error for a shape method to return
   an illegal size range.  A common error is to return an illegal
   size range with <CODE>sh.lo = sh.hi</CODE>.
   <P>
   The child must not assume that its shape requirement is satisfied, 
   since, for example, the requirements of a split's children can be 
   inconsistent.   
   <P>
   The default <CODE>shape</CODE> method for a <CODE>Leaf</CODE> returns <CODE>DefaultShape</CODE>. 
   <P>
   When the preferred shape of a <CODE>VBT</CODE> changes, you should
   call <CODE>NewShape</CODE>:  

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#NewShape">NewShape</A>(v: T);
&lt;* LL.sup &gt;= mu.v AND LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Notify <CODE>v</CODE>'s parent that its preferred size range may have changed. </EM></BLOCKQUOTE><PRE>
</PRE> Typically, the parent will mark itself, and any change will take effect 
  at the time of the next redisplay.  Notice that the locking level 
  allows <CODE>NewShape</CODE> to be called from a <CODE>reshape</CODE> or <CODE>rescreen</CODE> method;
  it can also be called from a thread that has <CODE>mu</CODE> locked.  

<P>
<P> \subsection{Putting properties on a VBT} 

<P> Associated with each window is a ``property set'', which is a set 
   of non-nil traced references.  \index{property set, of window} 

<P>
<P><PRE>PROCEDURE <A HREF="VBT.m3#PutProp">PutProp</A>(v: T; ref: REFANY); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Add <CODE>ref</CODE> to <CODE>v</CODE>'s property set, replacing any existing reference of 
   the same type as <CODE>ref</CODE>.  This is a checked runtime error if <CODE>ref</CODE> is 
   <CODE>NIL</CODE>. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#GetProp">GetProp</A>(v: T; tc: INTEGER): REFANY;
&lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Return the element of <CODE>v</CODE>'s property set with typecode <CODE>tc</CODE>, or  
   <CODE>NIL</CODE> if no such element exists. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="VBT.m3#RemProp">RemProp</A>(v: T; tc: INTEGER); &lt;* LL.sup &lt; v *&gt;
</PRE><BLOCKQUOTE><EM> Remove the element with typecode <CODE>tc</CODE> from <CODE>v</CODE>'s property set, if one 
   exists. </EM></BLOCKQUOTE><PRE>
</PRE> \subsection{Discarding a VBT} 

<P> It is good form to call <CODE>VBT.Discard(v)</CODE> when <CODE>v</CODE> is about to be 
   garbage-collected:  \index{discard method} 
     
<P><PRE>PROCEDURE <A HREF="VBT.m3#Discard">Discard</A>(v: T); &lt;* LL.sup = mu *&gt;
</PRE><BLOCKQUOTE><EM> Prepare for and call <CODE>v.discard()</CODE>.  </EM></BLOCKQUOTE><PRE>
</PRE> The discard method will be called with <CODE>LL.sup = mu</CODE>, and takes
   no argument.  The method should perform any class-dependent cleanup
   that is needed.  The default discard method is a no-op. 

<P><PRE>END VBT.
</PRE>
</inInterface>
<PRE>























</PRE>
</BODY>
</HTML>
