<HTML>
<HEAD>
<TITLE>SRC Modula-3: vtext/src/VText.m3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>vtext/src/VText.m3</H2></A><HR>
<inModule>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE><BLOCKQUOTE><EM>                                                                           </EM></BLOCKQUOTE><PRE>
</PRE> Visible text: the part of the text editor that maintains a consistent
   display of an editable text. It also provides highlighting,
   scrolling, window-splitting, and the insertion caret. 

<P><PRE>MODULE <module><implements><A HREF="VText.i3">VText</A></implements></module>;

IMPORT <A HREF="../../ui/src/vbt/Font.i3">Font</A>, <A HREF="../../mtext/src/MText.i3">MText</A>, <A HREF="../../geometry/src/Point.i3">Point</A>, <A HREF="../../rw/src/Common/Rd.i3">Rd</A>, <A HREF="../../geometry/src/Rect.i3">Rect</A>, <A HREF="../../thread/src/Common/Thread.i3">Thread</A>, <A HREF="../../ui/src/vbt/VBT.i3">VBT</A>, <A HREF="VT.i3">VT</A>, <A HREF="VTBase.i3">VTBase</A>, <A HREF="VTCaret.i3">VTCaret</A>, <A HREF="VTDef.i3">VTDef</A>,
       <A HREF="VTInterval.i3">VTInterval</A>, <A HREF="VTMarker.i3">VTMarker</A>, <A HREF="VTPounce.i3">VTPounce</A>, <A HREF="VTReal.i3">VTReal</A>, <A HREF="VTView.i3">VTView</A>, <A HREF="VTVirtual.i3">VTVirtual</A>, <A HREF="VTextRegion.i3">VTextRegion</A>;

TYPE
  LineNo = VTDef.LineNo;
  VirtualStart = VTDef.VirtualStart;
</PRE>**********************************************************************
			        Creation 				
**********************************************************************

<P>
<P><PRE>PROCEDURE <A NAME="New"><procedure>New</procedure></A> (         mtext   : MText.T;
                        vbt     : VBT.T;
               READONLY rect    : Rect.T;
               READONLY vOptions: VOptions ): T
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR vtext: T;
  BEGIN
    IF (mtext = NIL) OR (vbt = NIL) OR (vOptions.vFontxxx = NIL) THEN
      RAISE VTDef.Error (ErrorCode.IsNil);
    END;
    vtext := NEW (T);
    vtext.mtext := mtext;       (* old-style *)
    vtext.vbt := vbt;           (* old-style *)
    vtext.font := vOptions.vFontxxx.vFont.font; (* old-style *)
    vtext.vOptions := vOptions; (* semi old-style *)
    vtext.vt := VT.New (mtext);
    vtext.west := rect.west;    (* old-style *)
    vtext.north := rect.north;  (* old-style *)
    vtext.width := rect.east - rect.west; (* old-style *)
    vtext.height := rect.south - rect.north; (* old-style *)
    vtext.regionMax := 0;       (* old-style *)
    vtext.closed := FALSE;
    VTextRegion.SetupRegion (
      vtext, 0, rect.north, rect.south - rect.north, 0);
    vtext.lineSpacing :=
      vtext.region [0].view.vScreenFont.vScreenFont.box.south
        - vtext.region [0].view.vScreenFont.vScreenFont.box.north
        + vtext.leading;
    (* old-style *)
    vtext.lineOffset :=
      -vtext.region [0].view.vScreenFont.vScreenFont.box.north;
    (* old-style *)
    vtext.caretState := vtext.vt.caret.state; (* old-style *)
    vtext.dividersDirty := FALSE;
    RETURN vtext;
  END New;

PROCEDURE <A NAME="ExplodeVText"><procedure>ExplodeVText</procedure></A> (READONLY     vtext   : T;
                        VAR (* OUT*) mtext   : MText.T;
                        VAR (* OUT*) vbt     : VBT.T;
                        VAR (* OUT*) rect    : Rect.T;
                        VAR (* OUT*) vOptions: VOptions ) RAISES {} =
  BEGIN
    mtext := vtext.mtext;
    vbt := vtext.vbt;
    rect := Rect.FromEdges(vtext.west, vtext.west + vtext.width,
                           vtext.north, vtext.north + vtext.height);
    vOptions := vtext.vOptions;
  END ExplodeVText;

PROCEDURE <A NAME="MakeVFont"><procedure>MakeVFont</procedure></A> (         font     : Font.T;
                     READONLY printable: SET OF CHAR;
                              whiteTabs: BOOLEAN      ): VFont
  RAISES {VTDef.Error} =
  BEGIN
    RETURN VTView.MakeVFont(font, printable, whiteTabs);
  END MakeVFont;

PROCEDURE <A NAME="ExplodeVFont"><procedure>ExplodeVFont</procedure></A> (READONLY     vFont    : VFont;
                        VAR (* OUT*) font     : Font.T;
                        VAR (* OUT*) printable: SET OF CHAR;
                        VAR (* OUT*) whiteTabs: BOOLEAN      ) RAISES {} =
  BEGIN
    font := vFont.vFont.font;
    printable := vFont.vFont.printable;
    whiteTabs := vFont.vFont.whiteTabs;
  END ExplodeVFont;

PROCEDURE <A NAME="MakeVOptions"><procedure>MakeVOptions</procedure></A> (vFont: VFont;
                        leftMargin, rightMargin, turnMargin, topMargin,
                          leading: Points;
                        whiteBlack, whiteStroke: ColorScheme;
                        leftOffset             : Points;
                        wrap                   : BOOLEAN;
                        eob                    : BOOLEAN;
                        intervalStylePrecedence:
                            IntervalStylePrecedence := NIL):
  VOptions RAISES {} =
  BEGIN
    RETURN VTView.MakeVOptions(vFont, leftMargin, rightMargin, turnMargin,
                               topMargin, leading, whiteBlack, whiteStroke,
                               leftOffset, wrap, eob,
                               intervalStylePrecedence);
  END MakeVOptions;

PROCEDURE <A NAME="ExplodeVOptions"><procedure>ExplodeVOptions</procedure></A> (READONLY     vOptions: VOptions;
                           VAR (* OUT*) vFont   : VFont;
                           VAR (* OUT*) leftMargin, rightMargin, turnMargin,
                                        topMargin, leading: Points;
                           VAR (* OUT*) whiteBlack, whiteStroke: ColorScheme;
                           VAR (* OUT*) leftOffset: Points;
                           VAR (* OUT*) wrap: BOOLEAN;
                           VAR (* OUT*) eob: BOOLEAN;
                           VAR (* OUT*) intervalStylePrecedence:
                                            IntervalStylePrecedence)
  RAISES {} =
  BEGIN
    vFont := vOptions.vFontxxx;
    leftMargin := vOptions.leftMarginPts;
    rightMargin := vOptions.rightMarginPts;
    turnMargin := vOptions.turnMarginPts;
    topMargin := vOptions.topMarginPts;
    leading := vOptions.leadingPts;
    whiteBlack := vOptions.whiteBlack;
    whiteStroke := vOptions.whiteStroke;
    leftOffset := vOptions.leftOffsetPts;
    wrap := vOptions.wrap;
    eob := vOptions.eob;
    intervalStylePrecedence := vOptions.intervalStylePrecedence;
  END ExplodeVOptions;

PROCEDURE <A NAME="ChangeVOptions"><procedure>ChangeVOptions</procedure></A> (vtext: T; READONLY vOptions: VOptions)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR start: Index;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      vtext.vOptions := vOptions;
      vtext.font := vtext.vOptions.vFontxxx.vFont.font; (* old-style *)
      WITH vo = vtext.vOptions DO
        VTView.SetPixelOptions (vo, vtext.vbt);
        vtext.leftMargin := vo.leftMargin;
        vtext.rightMargin := vo.rightMargin;
        vtext.turnMargin := vo.turnMargin;
        vtext.topMargin := vo.topMargin;
        vtext.leading := vo.leading
      END;
      vtext.left := vtext.west + vtext.leftMargin + vtext.turnMargin;
      vtext.lineSpacing :=
        vtext.region [0].view.vScreenFont.vScreenFont.box.south
          - vtext.region [0].view.vScreenFont.vScreenFont.box.north
          + vtext.leading;
      (* old-style *)
      vtext.lineOffset :=
        -vtext.region [0].view.vScreenFont.vScreenFont.box.north;
      (* old-style *)

      FOR r := 0 TO vtext.regionMax DO
        start := vtext.region [r].view.virtual.start.at;
        VTView.Close (vtext.region [r].view);
        VTextRegion.SetupRegion (
          vtext, r, vtext.region [r].north, vtext.region [r].height, start);
      END;
      vtext.dividersDirty := TRUE;

    END;
  END ChangeVOptions;

PROCEDURE <A NAME="Close"><procedure>Close</procedure></A> (vtext: T) RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed);  END;
      VT.Close (vtext.vt);
      vtext.closed := TRUE;
    END;
  END Close;
</PRE>**********************************************************************
				 Regions				
**********************************************************************

<P><PRE>PROCEDURE <A NAME="SplitRegion"><procedure>SplitRegion</procedure></A> (vtext : T;
                       r     : Region;
                       v     : Pixels;
                       scroll: BOOLEAN  := TRUE)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      VTextRegion.SplitRegion(vtext, r, v, scroll);
    END;
  END SplitRegion;

PROCEDURE <A NAME="MergeRegion"><procedure>MergeRegion</procedure></A> (vtext: T; i, j: Region; scroll: BOOLEAN := TRUE)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      VTextRegion.MergeRegion(vtext, i, j, scroll);
    END;
  END MergeRegion;
</PRE>**********************************************************************
 Moving 
**********************************************************************

<P>
<P><PRE>PROCEDURE <A NAME="Move"><procedure>Move</procedure></A> (         vtext             : T;
                READONLY newRect, savedRect: Rect.T;
                READONLY dividers          : ARRAY OF Pixels;
                         scroll            : BOOLEAN          := TRUE)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      VTextRegion.Move(vtext, newRect, savedRect, dividers, scroll);
    END;
  END Move;

PROCEDURE <A NAME="Rescreen"><procedure>Rescreen</procedure></A> (vtext: T; READONLY cd: VBT.RescreenRec) RAISES {} =
  VAR view := vtext.vt.views;
  BEGIN
    WHILE view # NIL DO VTView.Rescreen (view, cd); view := view.next; END;
    WITH vo = vtext.vOptions DO
      VTView.SetPixelOptions (vo, vtext.vbt);
      vtext.leftMargin := vo.leftMargin;
      vtext.rightMargin := vo.rightMargin;
      vtext.turnMargin := vo.turnMargin;
      vtext.topMargin := vo.topMargin;
      vtext.leading := vo.leading;

      vtext.left := vtext.west + vtext.leftMargin + vtext.turnMargin;
      vtext.lineSpacing :=
        vtext.region [0].view.vScreenFont.vScreenFont.box.south
          - vtext.region [0].view.vScreenFont.vScreenFont.box.north
          + vtext.leading;      (* old-style *)
      vtext.lineOffset :=
        -vtext.region [0].view.vScreenFont.vScreenFont.box.north;
      (* old-style *)
    END
  END Rescreen;
</PRE>**********************************************************************
 Drawing 
**********************************************************************

<P>
<P><PRE>PROCEDURE <A NAME="Update"><procedure>Update</procedure></A> (vtext: T)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    ConcurrentUpdate (vtext);
    Quiesce (vtext);
  END Update;

PROCEDURE <A NAME="ConcurrentUpdate"><procedure>ConcurrentUpdate</procedure></A> (vtext: T)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      VTextRegion.UpdateDividers (vtext);
      VTReal.Update (vtext.vt);
    END;
  END ConcurrentUpdate;

PROCEDURE <A NAME="Quiesce"><procedure>Quiesce</procedure></A> (vtext: T) RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed);  END;
    END;
  END Quiesce;

PROCEDURE <A NAME="Bad"><procedure>Bad</procedure></A> (vtext: T; READONLY where: Rect.T) RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed);  END;
      VTextRegion.Bad (vtext, where);
    END
  END Bad;
</PRE><BLOCKQUOTE><EM>**********************************************************************</EM></BLOCKQUOTE><PRE>
</PRE>			        Editing   				
**********************************************************************

<P><PRE>PROCEDURE <A NAME="Replace"><procedure>Replace</procedure></A> (vtext: T; begin, end: Index; newText: TEXT)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF begin &gt; vtext.vt.length THEN begin := vtext.vt.length; END;
      IF end &gt; vtext.vt.length THEN end := vtext.vt.length; END;
      IF begin &gt; end THEN RAISE VTDef.Error(ErrorCode.IllegalIndex); END;
      VT.Replace(vtext.vt, begin, end, newText);
    END;
  END Replace;

PROCEDURE <A NAME="ReplaceChars"><procedure>ReplaceChars</procedure></A> (         vtext     : T;
                                 begin, end: Index;
                        READONLY str       : ARRAY OF CHAR)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF begin &gt; vtext.vt.length THEN begin := vtext.vt.length; END;
      IF end &gt; vtext.vt.length THEN end := vtext.vt.length; END;
      IF begin &gt; end THEN RAISE VTDef.Error (ErrorCode.IllegalIndex); END;
      VT.ReplaceChars (vtext.vt, begin, end, str);
    END;
  END ReplaceChars;

PROCEDURE <A NAME="ReplaceFile"><procedure>ReplaceFile</procedure></A> (vtext     : T;
                       begin, end: Index;
                       file      : Rd.T;
                       start     : Index   := 0;
                       numChars  : Index   := LAST(Index))
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF begin &gt; vtext.vt.length THEN begin := vtext.vt.length; END;
      IF end &gt; vtext.vt.length THEN end := vtext.vt.length; END;
      IF begin &gt; end THEN RAISE VTDef.Error(ErrorCode.IllegalIndex); END;
      VT.ReplaceFile(vtext.vt, begin, end, file, start, numChars);
    END;
  END ReplaceFile;

PROCEDURE <A NAME="Invalidate"><procedure>Invalidate</procedure></A> (vtext: T; begin, oldEnd, newEnd: Index)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF begin &gt; vtext.vt.length THEN begin := vtext.vt.length; END;
      IF oldEnd &gt; vtext.vt.length THEN oldEnd := vtext.vt.length; END;
      IF (begin &gt; oldEnd) OR (begin &gt; newEnd) THEN
        RAISE VTDef.Error(ErrorCode.IllegalIndex);
      END;
      VT.Invalidate(vtext.vt, begin, oldEnd, newEnd - begin);
    END;
  END Invalidate;
</PRE>**********************************************************************
 The Caret 
**********************************************************************

<P>
<P><PRE>PROCEDURE <A NAME="SwitchCaret"><procedure>SwitchCaret</procedure></A> (vtext: T; state: OnOffState)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      VTCaret.Switch(vtext.vt, state);
      vtext.caretState := state; (* old-style *)
    END;
  END SwitchCaret;

PROCEDURE <A NAME="MoveCaret"><procedure>MoveCaret</procedure></A> (vtext: T; place: Index)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF place &gt; vtext.vt.length THEN place := vtext.vt.length; END;
      VTCaret.Move(vtext.vt, place);
    END;
  END MoveCaret;

PROCEDURE <A NAME="CaretIndex"><procedure>CaretIndex</procedure></A> (vtext: T): Index RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      RETURN vtext.vt.caret.index;
    END;
  END CaretIndex;
</PRE>**********************************************************************
 Intervals 
**********************************************************************

<P>
<P><PRE>PROCEDURE <A NAME="CreateInterval"><procedure>CreateInterval</procedure></A> (         vtext         : T;
                                   indexL, indexR: Index;
                          READONLY options       : IntervalOptions):
  Interval RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF indexL &gt; vtext.vt.length THEN indexL := vtext.vt.length; END;
      IF indexR &gt; vtext.vt.length THEN indexR := vtext.vt.length; END;
      IF indexL &gt; indexR THEN
        RAISE VTDef.Error(ErrorCode.IllegalIndex);
      END;
      RETURN VTInterval.New(vtext.vt, indexL, indexR, options);
    END;
  END CreateInterval;

PROCEDURE <A NAME="ExplodeInterval"><procedure>ExplodeInterval</procedure></A> (READONLY      interval      : Interval;
                           VAR (* OUT *) indexL, indexR: Index;
                           VAR (* OUT *) options       : IntervalOptions;
                           VAR (* OUT *) state         : OnOffState       )
  RAISES {} =
  BEGIN
    VTInterval.ExplodeInterval (interval, indexL, indexR, options, state)
  END ExplodeInterval;

PROCEDURE <A NAME="MakeIntervalOptions"><procedure>MakeIntervalOptions</procedure></A> (style                  : IntervalStyle;
                               whiteBlack, whiteStroke: ColorScheme;
                               leading                : Tint           ):
  IntervalOptions RAISES {} =
  BEGIN
    RETURN VTInterval.MakeOptions(style, whiteBlack, whiteStroke, leading);
  END MakeIntervalOptions;

PROCEDURE <A NAME="ExplodeIntervalOptions"><procedure>ExplodeIntervalOptions</procedure></A> (READONLY intervalOptions: IntervalOptions;
                                  VAR (* OUT*) style: IntervalStyle;
                                  VAR (* OUT*) whiteBlack, whiteStroke: ColorScheme;
                                  VAR (* OUT*) leading: Tint) RAISES {} =
  BEGIN
    style := intervalOptions.style;
    whiteBlack := intervalOptions.whiteBlack;
    whiteStroke := intervalOptions.whiteStroke;
    leading := intervalOptions.leading;
  END ExplodeIntervalOptions;

PROCEDURE <A NAME="SwitchInterval"><procedure>SwitchInterval</procedure></A> (interval: Interval; state: OnOffState)
  RAISES {VTDef.Error} =
  BEGIN
    VTInterval.Switch (interval, state)
  END SwitchInterval;

PROCEDURE <A NAME="MoveInterval"><procedure>MoveInterval</procedure></A> (interval: Interval; indexL, indexR: Index)
  RAISES {VTDef.Error} =
  BEGIN
    VTInterval.Move (interval, indexL, indexR)
  END MoveInterval;

PROCEDURE <A NAME="ChangeIntervalOptions"><procedure>ChangeIntervalOptions</procedure></A> (         interval: Interval;
                                 READONLY options : IntervalOptions)
  RAISES {VTDef.Error} =
  BEGIN
    VTInterval.ChangeOptions (interval, options)
  END ChangeIntervalOptions;

PROCEDURE <A NAME="DeleteInterval"><procedure>DeleteInterval</procedure></A> (interval: Interval) RAISES {VTDef.Error} =
  BEGIN
    VTInterval.Delete (interval)
  END DeleteInterval;
</PRE>**********************************************************************
			     Markers					
**********************************************************************

<P><PRE>PROCEDURE <A NAME="CreateMarker"><procedure>CreateMarker</procedure></A> (vtext: T; at: Index; options: MarkerOptions):
  Marker RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF at &gt; vtext.vt.length THEN at := vtext.vt.length; END;
      RETURN VTMarker.New (vtext.vt, at, options);
    END;
  END CreateMarker;

PROCEDURE <A NAME="ExplodeMarker"><procedure>ExplodeMarker</procedure></A> (READONLY      marker : Marker;
                         VAR (* OUT *) at     : Index;
                         VAR (* OUT *) options: MarkerOptions;
                         VAR (* OUT*)  state  : OnOffState     )
  RAISES {} =
  BEGIN
    at := marker.index;
    options := marker.options;
    state := marker.state;
  END ExplodeMarker;

PROCEDURE <A NAME="MakeMarkerOptions"><procedure>MakeMarkerOptions</procedure></A> (whichEnd   : WhichEnd;
                             top, bottom: BOOLEAN;
                             stroke     : Tint      ): MarkerOptions
  RAISES {} =
  BEGIN
    RETURN VTMarker.MakeOptions (whichEnd, top, bottom, stroke);
  END MakeMarkerOptions;

PROCEDURE <A NAME="ExplodeMarkerOptions"><procedure>ExplodeMarkerOptions</procedure></A> (READONLY markerOptions: MarkerOptions;
                                VAR (* OUT *) whichEnd   : WhichEnd;
                                VAR (* OUT *) top, bottom: BOOLEAN;
                                VAR (* OUT *) stroke     : Tint      )
  RAISES {} =
  BEGIN
    whichEnd := markerOptions.whichEnd;
    top := markerOptions.top;
    bottom := markerOptions.bottom;
    stroke := markerOptions.stroke;
  END ExplodeMarkerOptions;

PROCEDURE <A NAME="SwitchMarker"><procedure>SwitchMarker</procedure></A> (marker: Marker; state: OnOffState)
  RAISES {VTDef.Error} =
  VAR vt: VTDef.T;
  BEGIN
    IF marker = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    vt := marker.vt;
    IF vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vt.mutex DO
      IF vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF marker.vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
      VTMarker.Switch (marker, state);
    END;
  END SwitchMarker;
</PRE><BLOCKQUOTE><EM> Sets marker's state := state. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A NAME="MoveMarker"><procedure>MoveMarker</procedure></A> (marker: Marker; index: Index) RAISES {VTDef.Error} =
  VAR vt: VTDef.T;
  BEGIN
    IF marker = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    vt := marker.vt;
    IF vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    LOCK vt.mutex DO
      IF vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed);  END;
      IF marker.vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
      IF index &gt; marker.vt.length THEN index := marker.vt.length;  END;
      VTMarker.Move (marker, index);
    END;
  END MoveMarker;
</PRE><BLOCKQUOTE><EM> Moves the marker. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A NAME="ChangeMarkerOptions"><procedure>ChangeMarkerOptions</procedure></A> (marker: Marker; options: MarkerOptions)
  RAISES {VTDef.Error} =
  VAR vt: VTDef.T;
  BEGIN
    IF marker = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    vt := marker.vt;
    IF vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vt.mutex DO
      IF vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF marker.vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
      VTMarker.ChangeOptions (marker, options);
    END;
  END ChangeMarkerOptions;
</PRE><BLOCKQUOTE><EM> Re-sets marker's options. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A NAME="DeleteMarker"><procedure>DeleteMarker</procedure></A> (marker: Marker) RAISES {VTDef.Error} =
  VAR vt: VTDef.T;
  BEGIN
    IF marker = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    vt := marker.vt;
    IF vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
    LOCK vt.mutex DO
      IF vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed);  END;
      IF marker.vt = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil);  END;
      VTMarker.Close (marker);
    END;
  END DeleteMarker;
</PRE><BLOCKQUOTE><EM> Sets marker's state = Off and then deletes marker from the set of
   markers associated with the VText. </EM></BLOCKQUOTE><PRE>
</PRE>**********************************************************************
		                Scrolling				
**********************************************************************

<P><PRE>PROCEDURE <A NAME="Scroll"><procedure>Scroll</procedure></A> (vtext: T; r: Region; displacement: INTEGER)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      WITH z_144 = vtext.region [r] DO
        WITH z_145 = z_144.view^ DO
          IF displacement = 0 THEN RETURN; END;
          IF displacement &gt; 0 THEN
            VTVirtual.SetStart (
              z_144.view, MIN (VTBase.Down (
                                 z_144.view, z_145.virtual.start.at,
                                 displacement), z_145.vt.length));
          ELSE
            VTVirtual.SetStart (
              z_144.view, z_145.virtual.start.at, -displacement);
          END;
        END;
      END;
    END;
  END Scroll;

PROCEDURE <A NAME="SetStart"><procedure>SetStart</procedure></A> (vtext  : T;
                    r      : Region;
                    place  : Index;
                    upLines: CARDINAL := 0;
                    force  : BOOLEAN  := FALSE)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error(ErrorCode.IllegalRegion);
      END;
      IF place &gt; vtext.vt.length THEN place := vtext.vt.length; END;
      VTVirtual.SetStart(vtext.region[r].view, place, upLines, force);
    END;
  END SetStart;

PROCEDURE <A NAME="LinesBetween"><procedure>LinesBetween</procedure></A> (vtext     : T;
                        begin, end: Index;
                        max       : CARDINAL;
                        avail     : Pixels      := UseCurrentWidth; ):
  INTEGER RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  (* Compute the number of lines that would be displayed between indexL and
     end, with &quot;avail&quot; horizontal pixels available. 0 if on same line, 1 if
     adjacent, etc. Will quit computing and return &quot;max&quot; if there are more
     lines than that between begin and end. *)
  VAR
    index, by: INTEGER;
    lineCount: INTEGER;
    turned   : BOOLEAN;
    l0, l1   : INTEGER;
    start    : VirtualStart;
    width    : Pixels;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext^.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF avail = UseCurrentWidth THEN
        avail := vtext^.region[0].view.lineWidth;
      END;
      IF begin &gt; vtext.vt.length THEN begin := vtext.vt.length; END;
      IF end &gt; vtext.vt.length THEN end := vtext.vt.length; END;
      IF begin &gt; end THEN RAISE VTDef.Error(ErrorCode.IllegalIndex); END;
      (* See if we can give a simple answer *)
      IF avail = vtext^.region[0].view.lineWidth THEN
        FOR r := 0 TO vtext^.regionMax DO
          IF NOT vtext^.region[r].view.virtual.bodyDirty THEN
            l0 := VTBase.UnsafeLocateLine(vtext^.region[r].view, begin);
            IF l0 &gt;= 0 THEN
              l1 := VTBase.UnsafeLocateLine(vtext^.region[r].view, end);
              IF l1 &gt;= 0 THEN RETURN MIN(MAX(l1 - l0, -1), max); END;
            END;
          END;
        END;
      END;
      (* do it the hard way *)
      VTBase.Up(vtext.region[0].view, avail, begin, 0, start);
      index := start.at;
      lineCount := -1;
      WHILE (index &lt;= end) AND (lineCount &lt; max) DO
        index := VTBase.ComputeLine(
                   vtext^.region[0].view, avail, index, by, turned, width);
        lineCount := lineCount + 1;
      END;
      RETURN lineCount;
    END;
  END LinesBetween;

PROCEDURE <A NAME="ComputeLine"><procedure>ComputeLine</procedure></A> (              vtext : T;
                                     from  : Index;
                       VAR (* OUT *) max   : Index;
                       VAR (* OUT *) turned: BOOLEAN;
                       VAR (* OUT *) width : Pixels    ): Index
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR i, m: INTEGER;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF from &gt; vtext.vt.length THEN from := vtext.vt.length; END;
      i := VTBase.ComputeLine (
             vtext.region [0].view, vtext.region [0].view.lineWidth, from, m,
             turned, width);
      max := m;
      RETURN MIN (i, vtext.vt.length);
    END;
  END ComputeLine;

PROCEDURE <A NAME="UpLines"><procedure>UpLines</procedure></A> (vtext: T; place: Index; n: CARDINAL; r: Region := 0):
  Index RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR start: VirtualStart;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error(ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error(ErrorCode.Closed); END;
      IF place &gt; vtext.vt.length THEN place := vtext.vt.length; END;
      VTBase.Up(vtext.region[r].view, vtext.region[r].view.lineWidth,
                place, n, start);
      RETURN start.at;
    END;
  END UpLines;
</PRE>**********************************************************************
 Locations 
**********************************************************************

<P><PRE>PROCEDURE <A NAME="StartIndex"><procedure>StartIndex</procedure></A> (vtext: T; r: Region): Index RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      WITH z_148 = vtext.region [r] DO
        RETURN z_148.view.virtual.start.at;
      END;
    END;
  END StartIndex;

PROCEDURE <A NAME="LineIndex"><procedure>LineIndex</procedure></A> (vtext: T; r: Region; n: CARDINAL): Index
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      WITH z_149 = vtext.region [r] DO
        WITH z_150 = z_149.view^ DO
          IF z_150.virtual.dirty THEN
            VTVirtual.UpdateView (z_149.view);
          END;
          RETURN MIN (z_150.virtual.line [
                        MIN (n, z_150.virtual.lines)].virtualLine.from,
                      z_150.vt.length);
        END;
      END;
    END;
  END LineIndex;

PROCEDURE <A NAME="CharsInRegion"><procedure>CharsInRegion</procedure></A> (vtext: T; r: Region): CARDINAL
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      WITH z_151 = vtext.region [r] DO
        WITH z_152 = z_151.view^ DO
          IF z_152.virtual.dirty THEN
            VTVirtual.UpdateView (z_151.view);
          END;
          RETURN MIN (z_152.virtual.line [
                        z_152.virtual.lines].virtualLine.from,
                      z_152.vt.length)
                   - z_152.virtual.line [0].virtualLine.from;
        END;
      END;
    END;
  END CharsInRegion;

PROCEDURE <A NAME="Locate"><procedure>Locate</procedure></A> (              vtext: T;
                                r    : Region;
                                place: Index;
                  VAR (* OUT *) h, v : INTEGER )
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR p: Point.T;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      IF place &gt; vtext.vt.length THEN place := vtext.vt.length; END;
      WITH z_153 = vtext.region [r] DO
        IF z_153.view.virtual.dirty THEN
          VTVirtual.UpdateView (z_153.view);
        END;
        VTBase.UnsafeLocatePoint (z_153.view, place, p);
        IF p.v &gt;= 0 THEN
          h := p.h;
          v := p.v + -z_153.view.vScreenFont.vScreenFont.box.north;
        ELSE
          v := p.v;
        END;
      END;
    END;
  END Locate;

PROCEDURE <A NAME="InRegion"><procedure>InRegion</procedure></A> (vtext: T; r: Region; place: Index): BOOLEAN
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR p: Index;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      IF place &gt; vtext.vt.length THEN place := vtext.vt.length; END;
      p := place;
      WITH z_154 = vtext.region [r] DO
        WITH z_155 = z_154.view^ DO
          IF p &lt; z_155.virtual.start.at THEN RETURN FALSE; END;
          IF z_155.virtual.dirty THEN
            VTVirtual.UpdateView (z_154.view);
          END;
          RETURN
            p &lt; z_155.virtual.line [z_155.virtual.lines].virtualLine.from;
        END;
      END;
    END;
  END InRegion;

PROCEDURE <A NAME="WhichLine"><procedure>WhichLine</procedure></A> (vtext: T; r: Region; v: Pixels): CARDINAL
  RAISES {VTDef.Error} =
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      WITH z_156 = vtext.region [r] DO
        RETURN
          MAX (
            MIN ((v - z_156.view.vOptions.topMargin)
                   DIV z_156.view.lineSpacing, z_156.view.nLines - 1), 0);
      END;
    END;
  END WhichLine;

PROCEDURE <A NAME="Pounce"><procedure>Pounce</procedure></A> (              vtext                 : T;
                                r                     : Region;
                                p                     : Point.T;
                                mode                  : SelectionMode;
                  VAR (* OUT *) indexL, indexM, indexR: Index;
                  VAR (* OUT *) cage                  : Rect.T         ):
  WhichEnd RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR
    c         : CHAR;
    iL, iM, iR: INTEGER;
    whichEnd  : WhichEnd;
    lineNo    : LineNo;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      VTPounce.Locate (vtext.region [r].view, p, iL, iR, lineNo, c);
      VTPounce.Extend (vtext.region [r].view, iL, iR, lineNo, c, mode);
      whichEnd :=
        VTPounce.Encage (vtext.region [r].view, p, iL, iM, iR, cage);
      &lt;* ASSERT iR &lt;= vtext.vt.length *&gt; (* past the end of the mtext *)
      indexL := iL;
      indexM := iM;
      indexR := iR;
      RETURN whichEnd;
    END;
  END Pounce;

PROCEDURE <A NAME="PounceLocate"><procedure>PounceLocate</procedure></A> (              vtext         : T;
                                      r             : Region;
                                      p             : Point.T;
                        VAR (* OUT *) indexL, indexR: Index;
                        VAR (* OUT *) lineNumber    : CARDINAL;
                        VAR (* OUT *) c             : CHAR      )
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR
    iL, iR: INTEGER;
    lineNo: LineNo;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      VTPounce.Locate (vtext.region [r].view, p, iL, iR, lineNo, c);
      indexL := iL;
      indexR := iR;
      lineNumber := lineNo;
    END;
  END PounceLocate;

PROCEDURE <A NAME="PounceExtend"><procedure>PounceExtend</procedure></A> (                vtext         : T;
                                        r             : Region;
                        VAR (* INOUT *) indexL, indexR: Index;
                                        lineNumber    : CARDINAL;
                                        c             : CHAR;
                                        mode          : SelectionMode)
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure, Thread.Alerted} =
  VAR iL, iR: INTEGER;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      IF indexL &gt; vtext.vt.length THEN indexL := vtext.vt.length; END;
      IF indexR &gt; vtext.vt.length THEN indexR := vtext.vt.length; END;
      IF indexL &gt; indexR THEN
        RAISE VTDef.Error (ErrorCode.IllegalIndex);
      END;
      iL := indexL;
      iR := indexR;
      VTPounce.Extend (vtext.region [r].view, iL, iR, lineNumber, c, mode);
      &lt;* ASSERT iR &lt;= vtext.vt.length *&gt; (* past the end of the mtext *)
      indexL := iL;
      indexR := iR;
    END;
  END PounceExtend;

PROCEDURE <A NAME="PounceEncage"><procedure>PounceEncage</procedure></A> (              vtext : T;
                                      r     : Region;
                                      p     : Point.T;
                                      indexL: Index;
                        VAR (* OUT *) indexM: Index;
                                      indexR: Index;
                        VAR (* OUT *) cage  : Rect.T   ): WhichEnd
  RAISES {VTDef.Error, Rd.EndOfFile, Rd.Failure,
                Thread.Alerted} =
  VAR
    iM      : INTEGER;
    whichEnd: WhichEnd;
  BEGIN
    IF vtext = NIL THEN RAISE VTDef.Error (ErrorCode.IsNil); END;
    LOCK vtext.vt.mutex DO
      IF vtext.vt.closed THEN RAISE VTDef.Error (ErrorCode.Closed); END;
      IF r &gt; vtext.regionMax THEN
        RAISE VTDef.Error (ErrorCode.IllegalRegion);
      END;
      IF indexL &gt; vtext.vt.length THEN indexL := vtext.vt.length; END;
      IF indexR &gt; vtext.vt.length THEN indexR := vtext.vt.length; END;
      IF indexL &gt; indexR THEN
        RAISE VTDef.Error (ErrorCode.IllegalIndex);
      END;
      whichEnd := VTPounce.Encage (
                    vtext.region [r].view, p, indexL, iM, indexR, cage);
      indexM := iM;
      RETURN whichEnd;
    END;
  END PounceEncage;

BEGIN
END VText.
</PRE>
</inModule>
<PRE>























</PRE>
</BODY>
</HTML>
