<HTML>
<HEAD>
<TITLE>SRC Modula-3: weakref/src/WeakRef.i3</TITLE>
</HEAD>
<BODY>
<A NAME="0TOP0">
<H2>weakref/src/WeakRef.i3</H2></A><HR>
<inInterface>
<PRE><A HREF="../../COPYRIGHT.html">Copyright (C) 1994, Digital Equipment Corp.</A>
</PRE> Most Modula-3 programs simply let the garbage collector deallocate
   storage automatically, but some programs need more control.  
   For example, if a variable allocated in the traced
   heap contains a handle on a resource in the operating system or in
   some other address space, then when the variable is garbage-collected
   it may be important to deallocate the resource.  The <CODE>WeakRef</CODE>
   interface provides this additional control.
   \index{weak reference}
   \index{deallocating resources}
   \index{garbage collection!weak references}
<P>
   A {\it node} is a datum allocated on the traced heap. Thus a node
   is either the referent of a variable of a fixed reference type or
   the data record of a traced object.  Note that a node is not a
   Modula-3 reference, but the allocated storage to which a reference
   can refer.
<P>
   A <CODE>WeakRef.T</CODE> is a data structure that refers to a node 
   without protecting the node from the garbage collector.  If <CODE>w</CODE> is a 
   weak reference, we write <CODE>nd(w)</CODE> to denote the node to which <CODE>w</CODE>
   refers.
<P>
   We say that a weak reference <CODE>w</CODE> {\it dies} at the moment that the
   garbage collector detects that <CODE>nd(w)</CODE> is unreachable.  A precise
   definition of unreachable is given below.  Once a weak reference
   has died, it remains dead forever, even if the node to which
   it refers becomes reachable again.
<P>
   Associated with each weak reference <CODE>w</CODE> is a {\it cleanup procedure}
   <CODE>cp(w)</CODE>.  If the cleanup procedure is not <CODE>NIL</CODE>, the garbage
   collector will schedule a call to it when the weak reference
   dies. 

<P><PRE>INTERFACE <interface><A HREF="WeakRef.m3">WeakRef</A></interface>;

TYPE T =
  RECORD
    byte: ARRAY [0..7] OF BITS 8 FOR [0..255]
  END;
</PRE><BLOCKQUOTE><EM> Please treat this as though it were an opaque type:  the
   only operations allowed are assignment, equality tests,
   and the procedures in this interface.  </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="WeakRef.m3#FromRef">FromRef</A>(r: REFANY; p: CleanUpProc := NIL): T;
</PRE><BLOCKQUOTE><EM> Return a weak reference <CODE>w</CODE> such that  <CODE>nd(w) = r</CODE> and 
   <CODE>cp(w) = p</CODE>.  It is a checked runtime error if <CODE>r</CODE> is <CODE>NIL</CODE>.
   It is illegal to create more than one weak reference with a
   non-nil cleanup to the same node; violations of this rule
   may lead to a checked runtime error, or may cause one
   of the cleanup actions to be omitted.  <CODE>FromRef</CODE> is not
   necessarily functional: it is possible that <CODE>nd(w1) = nd(w2)</CODE>
   but <CODE>w1 # w2</CODE>. </EM></BLOCKQUOTE><PRE>

PROCEDURE <A HREF="WeakRef.m3#ToRef">ToRef</A>(w: T): REFANY;
</PRE><BLOCKQUOTE><EM> Return a reference to <CODE>nd(w)</CODE>, unless <CODE>w</CODE> is dead, in which 
   case return <CODE>NIL</CODE>.  </EM></BLOCKQUOTE><PRE>

TYPE CleanUpProc = PROCEDURE(READONLY w: T; r: REFANY);
</PRE><BLOCKQUOTE><EM> If <CODE>cp(w)</CODE> is not <CODE>NIL</CODE>, then when <CODE>w</CODE> dies, the garbage collector
   will schedule the call <CODE>cp(w)(w, &lt;reference to nd(w)&gt;)</CODE>.  </EM></BLOCKQUOTE><PRE>

END WeakRef.
</PRE> The cleanup procedure will be executed at some point after 
   the weak reference dies.   A cleanup procedure is called with 
   no locks held; it must return promptly to allow other 
   objects to be cleaned up.  
<P>
   The computation <CODE>cp(w)(w, ref)</CODE> is allowed to store <CODE>ref</CODE> in a
   non-local variable, thus making <CODE>nd(w)</CODE> reachable again; the heap
   storage will not have been freed.  This does not change the fact
   that <CODE>w</CODE> is dead.  The cleanup procedure can re-enable cleanup, if
   desired, by creating a new weak reference to <CODE>nd(w)</CODE>.
<P>
   The storage for a node is reclaimed when it is unreachable
   and all weak references to it are dead and all cleanup calls 
   scheduled for it have been completed.
<P>
   Finally we come to the precise definition of ``reachable'':
<P>
   A node is {\it reachable} if it can be reached by a path of traced
   references starting from a current procedure activation record, a global
   variable, or a weakly referenced node with a non-nil
   cleanup {\it other than itself}.
<P>
   Thus a weak reference to a node <CODE>nd</CODE> does not make <CODE>nd</CODE> reachable,
   but if it has a non-nil cleanup, it makes other nodes referenced 
   from <CODE>nd</CODE> reachable.
<P>
   For example, if <CODE>A</CODE> and <CODE>B</CODE> are two nodes that are weakly referenced 
   by weak references with non-nil cleanup procedures, then if <CODE>B</CODE>
   is reachable from <CODE>A</CODE>, then <CODE>B</CODE> is reachable.  But if <CODE>A</CODE> is not
   reachable, then the garbage collector will eventually detect this 
   and schedule the cleanup of <CODE>A</CODE>.  If the cleanup call returns 
   without resurrecting <CODE>A</CODE>, then <CODE>A</CODE>'s storage will be reclaimed, 
   at which point <CODE>B</CODE> will be unreachable, which will lead to its cleanup.
<P>
   If <CODE>A</CODE> and <CODE>B</CODE> are weakly referenced nodes with non-nil cleanups
   that are connected by a cycle of traced references, then both of 
   them are reachable.   As long as the cycle persists, neither will 
   be cleaned up.  This situation represents a storage leak and 
   should be avoided.
<P>
\subsection*{Examples}
<P>
{\bf 1}.\ \  Suppose you want writers of the class <CODE>WrX.T</CODE> to be automatically
flushed and closed if they become unreachable.  Then you could write
code like the following in the <CODE>WrX</CODE> module:
<P>
<PRE>
      MODULE WrX; IMPORT WeakRef, Wr, ...;
      
      PROCEDURE New(...): T =
        VAR res := NEW(T); BEGIN
          (* ... initialize res as a WrX.T ... 

<PRE>|     EVAL WeakRef.FromRef(res, Cleanup);
|     RETURN res
|   END New;
|
| PROCEDURE Cleanup(READONLY self: WeakRef.T; ref: REFANY) =
|   VAR wr: T := ref; BEGIN
|     IF NOT Wr.Closed(wr) THEN
|       Wr.Flush(wr);
|       Wr.Close(wr)
|     END
|   END Cleanup;
|

There is no danger that another thread could close the writer
after the test &quot;NOT Wr.Closed(wr)&quot; and before the call &quot;Wr.Flush(wr)&quot;,
since when &quot;Cleanup&quot; is called, the writer is unreachable.  Therefore
the cleanup method has exclusive access to the writer.

\smallskip

{\bf 2.}\ \  The network object runtime must map wire
representations for network objects into surrogate objects.  To hand
out the same surrogate for the same wire representation, it keeps a
table mapping wire representations to surrogates.  This table contains
weak references, so the table entry itself does not prevent the
surrogate from being collected.  When the surrogate is collected, it
is removed from the table and the server containing that object is
notified that the client no longer has a surrogate for it.

When a weak reference in the table becomes dead, the network object
represented by the dead surrogate might be unmarshaled by the address
space before the surrogate is cleaned up.  In this case the unmarshaling
code resurrects the unreachable surrogate by creating a new weak
reference and inserting it in the table in place of the dead weak
reference.  The cleanup code can tell whether to report in clean by
checking whether there is a new weak reference in the table or not.

Here is a sketch of the code:

| TYPE Surrogate = OBJECT wr: WireRep; ... END;

| VAR
|   mu := NEW(MUTEX);
|   &lt;* LL &gt;= {mu} *&gt;
|   tbl := NEW(WireRepToWeakRefTbl.T);

   The mutex &quot;mu&quot; must be held to read or write &quot;tbl&quot; (that
   is what the &quot;LL&quot; pragma means).

   The table &quot;tbl&quot; maps &quot;WireRep&quot;s to &quot;WeakRef&quot;s that reference
   surrogates.

   The following invariants hold whenever &quot;mu&quot; is not held:

   If &quot;tbl(wrep)&quot; is not dead, then &quot;nd(tbl(wrep))&quot; is the
   surrogate for the network object whose wire representation
   is &quot;wrep&quot;.

   If &quot;tbl(wrep)&quot; is dead, then the surrogate for &quot;wrep&quot; is
   unreachable.

   If &quot;tbl&quot; has no entry for &quot;wrep&quot;, then the address space
   contains no surrogate for &quot;wrep&quot;.

| PROCEDURE Cleanup(READONLY wref: WeakRef.T; ref: REFANY) =
| &lt;* LL = {} *&gt;
|   VAR
|     srg := NARROW(ref, Surrogate);
|     tblVal: WeakRef.T;
|   BEGIN
|     LOCK mu DO
|       IF tbl.get(srg.wr, tblVal) AND wref = tblVal
|       THEN
|         EVAL tbl.delete(srg.wr);
|         ... Report that srg is deleted ...
|       END
|     END
|   END Cleanup;
|
| PROCEDURE WireRepToSrg(wrep: WireRep): Surrogate =
|   VAR wref: WeakRef.T; res: Surrogate; BEGIN
|     LOCK mu DO
|       IF tbl.get(wrep, wref) THEN
|         res := WeakRef.ToRef(wref);
|         IF res # NIL THEN RETURN res END
|       END;
|       res := NewSurrogate(wrep);
|       EVAL tbl.put(wrep, WeakRef.FromRef(res, Cleanup));
|       RETURN res
|     END
|   END WireRepToSrg;

  In the above we assume that &quot;NewSurrogate&quot; creates a new surrogate
  from a wire representation. *)
</PRE>
</inInterface>
<PRE>























</PRE>
</BODY>
</HTML>
