/*******************************************************/
/* rport.c: A program to pass on internet connections  */
/*          Comparable to term `tredir'                */
/*                                                     */
/* Author: Ton Hospel                                  */
/*         ton@linux.cc.kuleuven.ac.be                 */
/*         (1994, 1995)                                */
/*                                                     */
/* Copyright: GNU copyleft                             */
/*******************************************************/

#ifdef HAVE_TERMNET
# include <termnet.h>
#endif /* HAVE_TERMNET */

#include <mymalloc.h>
#include <except.h>

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#if !STDC_HEADERS && HAVE_MEMORY_H
# include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */

#include <errno.h>
#include <signal.h>
#ifndef RETSIGTYPE
# define RETSIGTYPE void
#endif /* RETSIGTYPE */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <fcntl.h>
#ifndef HAVE_NO_UNISTD_H
# include <unistd.h>
#endif /* HAVE_NO_UNISTD_H */

#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif

#define INT_FROM_VOID(x) ((int)(long) (x))
#define VOID_FROM_INT(x) ((void *) (long) (x))

#ifndef _POSIX_SOURCE
extern int read( /* int fd,       char *buf, unsigned int n */);
extern int write(/* int fd, const char *buf, unsigned int n */);
#endif /* _POSIX_SOURC */
extern int close(/* int fd */);
extern int gethostname(/* char *address, int addrlen */);
extern int select(/* int nfds,
                     fd_set *readfds, fd_set *writefds, fd_set *exceptfds,
                     struct timeval *timeout */);
#ifdef    HAVE_NO_MEMCHR_PROTO
extern void *memchr(const void *s, int c, size_t n);
#endif /* HAVE_NO_MEMCHR_PROTO */

#ifdef    HAVE_NO_MEMMOVE
void bcopy(/* char *source, char *target, int n */);
# define memmove(to, from, n)   bcopy(from, to, n)
#endif /* HAVE_NO_MEMMOVE */

#define EEA ErrnoExceptionAction
#define SE  static Exception
SE InputLost    = { "Lost stdin. Bailing out"};
SE FcntlError   = { "Could not execute fcntl()",            0, EEA };
SE HostError    = { "Could not execute gethostname()",      0, EEA };
SE SockError    = { "Could not create socket",              0, EEA };
SE BindError    = { "Could not bind socket",                0, EEA };
SE GetNameError = { "Could not get socket name",            0, EEA };
SE ConnectError = { "Could not connect socket",             0, EEA };
SE ListenError  = { "Could not set backlog",                0, EEA };
SE AcceptError  = { "Could not accept connection",          0, EEA };
SE SelectError  = { "Select() call failed",                 0, EEA };
#undef EEA
#undef SE

#define IOREAD   1
#define IOWRITE  2
#define IOERROR  4

typedef struct wantio *WantIo;
typedef void IoFun(WantIo Id, int Fid, void *Closure);
struct wantio {
    struct wantio *Next, *Previous;
    int            Fid, Type;
    IoFun         *Fun;
    void          *Closure;
};

static struct wantio WantIos = { &WantIos, &WantIos };
static fd_set WantIn, WantOut, WantErr;
static int    MaxFd, WantRedo;
static WantIo NextWantIo;

#define FDSET(a, b)                             \
do {                                            \
/*  fprintf(stderr, "Set on fd %d\n", Fid); */  \
    FD_SET(a, b);                               \
} while(0)

static WantIo AddInput(int Fid, int Type, IoFun *Fun, void *Closure)
{
    WantIo Ptr;

    Ptr = mynew(struct wantio);
    Ptr->Fid  = Fid;
    Ptr->Fun  = Fun;
    Ptr->Type = Type;
    Ptr->Closure = Closure;
    Ptr->Next = WantIos.Next;
    Ptr->Previous = &WantIos;
    Ptr->Next->Previous = Ptr->Previous->Next = Ptr;
    if (!WantRedo) {
        if (Type & IOREAD)  FDSET(Fid, &WantIn);
        if (Type & IOWRITE) FDSET(Fid, &WantOut);
        if (Type & IOERROR) FDSET(Fid, &WantErr);
        if (Fid > MaxFd)    MaxFd = Fid;
    }
    return Ptr;
}

static void RemoveInput(WantIo Ptr)
{
    if (NextWantIo == Ptr) NextWantIo = Ptr->Next;
    Ptr->Previous->Next = Ptr->Next;
    Ptr->Next->Previous = Ptr->Previous;
    myfree(Ptr);
    WantRedo = 1;
}

static void InitInput(void)
{
    WantIos.Next = WantIos.Previous = &WantIos;
    FD_ZERO(&WantIn);
    FD_ZERO(&WantOut);
    FD_ZERO(&WantErr);
    MaxFd = -1;
    WantRedo   = 0;
    NextWantIo = 0;
}

/* Wait is not reentrant due to the NextWantIo variable */
static int Wait(void)
{
    WantIo Ptr;
    int    Type, Fid, rc;
    fd_set in, out, err;

    if (WantRedo) {
        MaxFd = -1;
        FD_ZERO(&WantIn);
        FD_ZERO(&WantOut);
        FD_ZERO(&WantErr);
        for (Ptr = WantIos.Next; Ptr != &WantIos; Ptr = Ptr->Next) {
            Type = Ptr->Type;
            Fid  = Ptr->Fid;
            if (Fid > MaxFd)    MaxFd = Fid;
            if (Type & IOREAD)  FDSET(Fid, &WantIn);
            if (Type & IOWRITE) FDSET(Fid, &WantOut);
            if (Type & IOERROR) FDSET(Fid, &WantErr);
            WantRedo = 0;
        }
    }
    memcpy(&in,  &WantIn,  sizeof(in));
    memcpy(&out, &WantOut, sizeof(out));
    memcpy(&err, &WantErr, sizeof(err));
    do {
        rc = select(MaxFd+1, &in, &out, &err, NULL);
    } while (rc < 0 && errno == EINTR);
    if (rc < 0) Raise(SelectError);

    for (Ptr = WantIos.Next; Ptr != &WantIos; Ptr = NextWantIo) {
        NextWantIo = Ptr->Next;
        Type = Ptr->Type;
        Fid  = Ptr->Fid;
        if (((Type & IOREAD)  && FD_ISSET(Fid, &in))  || 
            ((Type & IOWRITE) && FD_ISSET(Fid, &out)) ||
            ((Type & IOERROR) && FD_ISSET(Fid, &err))) {
            (*Ptr->Fun)(Ptr, Ptr->Fid, Ptr->Closure);
            break;
        }    
    }
    return rc;
}

static void CleanInput(void)
{
    WantIo Ptr;

    while (Ptr = WantIos.Next, Ptr != &WantIos) RemoveInput(Ptr);
}

static size_t ComPos;
static int    IgnoreIn;

extern int h_errno;
static char NoError[]      = "No error";
static char InvalidError[] = "Unknown error number to strherror()";

#ifdef HAVE_H_ERRLIST
extern char *h_errlist[];
extern int   h_nerr;
#else  /* HAVE_H_ERRLIST */
static const char *h_errlist[] = {
    NoError,
    "Authoritive Answer Host not found",
    "Non-Authoritive Host not found, or SERVERFAIL",
    "Non recoverable errors, FORMERR, REFUSED, NOTIMP",
    "Valid host, no address, look for MX record"
};
static int h_nerr = sizeof(h_errlist)/sizeof(*h_errlist);
#endif /* HAVE_H_ERRLIST */

typedef struct _Accept *Accept;
typedef struct _Pass   *Pass;

struct _Pass {
    Accept Parent;
    Pass   Previous, Next;
    int    Hosted, PeerPort, Local, Log;
    size_t HostToLocalLength, LocalToHostLength;
    int    HostToLocalOffset, LocalToHostOffset;
    unsigned long HostToLocalRead, HostToLocalWritten;
    unsigned long LocalToHostRead, LocalToHostWritten;
    WantIo OnHosted, OnLocal;
    char HostToLocal[200], LocalToHost[200];
    struct in_addr  PeerAddr;
    char           *PeerName;
};

struct _Accept {
    Accept        Previous, Next;
    WantIo        Id;
    int           Local, LocalPort, HostPort, Log;
    struct _Pass  Passes;
    struct sockaddr_in Template;
    char         *HostSite;
};

static struct _Accept AcceptBase = { &AcceptBase, &AcceptBase };

static const char *strherrno(void)
{
    if (0 < h_errno && h_errno < h_nerr) return h_errlist[h_errno];
    else if (h_errno == 0)               return NoError;
    else                                 return InvalidError;
}

static void FreePass(Pass pass)
{
    Accept acc;

    acc = pass->Parent;
    fprintf(stderr,
            "Closing connection on %d: %s(%s) %d translated to port %d\n",
            acc->Local, acc->HostSite, inet_ntoa(acc->Template.sin_addr),
            acc->HostPort, acc->LocalPort);
    fprintf(stderr, "  Connection socket %d (host) to %d (local): %s %d\n",
            pass->Hosted, pass->Local, pass->PeerName, pass->PeerPort);
    fprintf(stderr, "           HostToLocal: read %08lu written %08lu\n",
            pass->HostToLocalRead, pass->HostToLocalWritten);
    fprintf(stderr, "           LocalToHost: read %08lu written %08lu\n",
                    pass->LocalToHostRead, pass->LocalToHostWritten);
    if (pass->OnLocal)  RemoveInput(pass->OnLocal);
    if (pass->OnHosted) RemoveInput(pass->OnHosted);
    if (pass->Hosted >= 0) close(pass->Hosted);
    if (pass->Local  >= 0) close(pass->Local);
    if (pass->Log    >= 0) close(pass->Log);
    myfree(pass->PeerName);
    pass->Previous->Next = pass->Next;
    pass->Next->Previous = pass->Previous;
    myfree(pass);
}

static void FreeAcc(Accept acc)
{
    fprintf(stderr,
            "Closing acceptance on %d: %s(%s) %d translated to port %d\n",
            acc->Local, acc->HostSite, inet_ntoa(acc->Template.sin_addr),
            acc->HostPort, acc->LocalPort);
    while (acc->Passes.Next != &acc->Passes) FreePass(acc->Passes.Next);
    RemoveInput(acc->Id);
    close(acc->Local);
    myfree(acc->HostSite);
    if (acc->Log >= 0) close(acc->Log);
    acc->Previous->Next = acc->Next;
    acc->Next->Previous = acc->Previous;
    myfree(acc);
}

static void GotLocal(WantIo Id, int Fid, void *Closure);
static void GotHosted(WantIo Id, int Fid, void *Closure);

static void WriteLocal(WantIo Id, int Fid, void *Closure)
{
    int rc;
    Pass pass;

    pass = Closure;
    rc = write(Fid, pass->HostToLocal+pass->HostToLocalOffset,
               pass->HostToLocalLength);
    if (rc <= 0)
        if (rc == 0) FreePass(pass);
        else fprintf(stderr, "Got error on output: %d, %s\n", rc, strerrno());
    else {
        pass->HostToLocalLength -= rc;
        pass->HostToLocalOffset += rc;
        pass->HostToLocalWritten += rc;
        if (pass->HostToLocalLength == 0) {
            RemoveInput(Id);
            pass->OnHosted = 0;
            pass->OnHosted =
                AddInput(pass->Hosted, IOREAD, GotHosted, Closure);
        }
/*      fprintf(stderr, "wrote output on %d: '%.*s'\n",
                Fid, rc, pass->HostToLocal); */
    }
}

static void WriteHosted(WantIo Id, int Fid, void *Closure)
{
    int rc;
    Pass pass;

    pass = Closure;
    rc = write(Fid, pass->LocalToHost+pass->LocalToHostOffset,
               pass->LocalToHostLength);
    if (rc <= 0)
        if (rc == 0) FreePass(pass);
        else fprintf(stderr, "Got error on output: %d, %s\n", rc, strerrno());
    else {
        pass->LocalToHostLength  -= rc;
        pass->LocalToHostOffset  += rc;
        pass->LocalToHostWritten += rc;
        if (pass->LocalToHostLength == 0) {
            RemoveInput(Id);
            pass->OnLocal = 0;
            pass->OnLocal =
                AddInput(pass->Local, IOREAD, GotLocal, Closure);
        }
/*        fprintf(stderr, "wrote output on %d: '%.*s'\n", 
                  Fid, rc, pass->LocalToHost); */
    }
}

static void GotLocal(WantIo Id, int Fid, void *Closure)
{
    int rc;
    Pass pass;

    pass = Closure;
    rc = read(Fid, pass->LocalToHost, sizeof(pass->LocalToHost));
    if (rc <= 0)
        if (rc == 0) FreePass(pass);
        else switch(errno) {
	  default:
	    fprintf(stderr, "Got error on input: %d, %s\n", rc, strerrno());
	    break;
#ifdef ETIMEDOUT
          case ETIMEDOUT:
#endif /* ETIMEDOUT */
#ifdef ECONNREFUSED
          case ECONNREFUSED:
#endif /* ECONNREFUSED */
#ifdef EHOSTDOWN
          case EHOSTDOWN:
#endif /* EHOSTDOWN */
#ifdef EHOSTUNREACH
          case EHOSTUNREACH:
#endif /* EHOSTUNREACH */
	    fprintf(stderr, "Got bad error on input: %d, %s\n", rc, strerrno());
	    FreePass(pass);
	    break;
	}
    else {
	if (pass->Log >= 0) {
	    write(pass->Log, "Local-> ", 8);
	    write(pass->Log, pass->LocalToHost, (size_t) rc);
	    if (write(pass->Log, "\n", 1) < 0) {
		fprintf(stderr, "Could not write to log file: %s. Closing\n",
		        strerrno());
		close(pass->Log);
		pass->Log = -1;
	    }
	}
        pass->LocalToHostLength = rc;
        pass->LocalToHostRead += rc;
        RemoveInput(Id);
        pass->OnLocal = 0;
        pass->OnLocal = AddInput(pass->Hosted, IOWRITE, WriteHosted, Closure);
        pass->LocalToHostOffset = 0;
/*      fprintf(stderr, "Read input on %d: '%.*s'\n",
                Fid, rc, pass->LocalToHost); */
    }
}

static void GotHosted(WantIo Id, int Fid, void *Closure)
{
    int rc;
    Pass pass;

    pass = Closure;
    rc = read(Fid, pass->HostToLocal, sizeof(pass->HostToLocal));
    if (rc <= 0)
        if (rc == 0) FreePass(pass);
        else switch(errno) {
	  default:
	    fprintf(stderr, "Got error on input: %d, %s\n", rc, strerrno());
	    break;
#ifdef ETIMEDOUT
          case ETIMEDOUT:
#endif /* ETIMEDOUT */
#ifdef ECONNREFUSED
          case ECONNREFUSED:
#endif /* ECONNREFUSED */
#ifdef EHOSTDOWN
          case EHOSTDOWN:
#endif /* EHOSTDOWN */
#ifdef EHOSTUNREACH
          case EHOSTUNREACH:
#endif /* EHOSTUNREACH */
	    fprintf(stderr, "Got bad error on input: %d, %s\n", rc, strerrno());
	    FreePass(pass);
	    break;
	}
    else {
	if (pass->Log >= 0) {
	    write(pass->Log, "Host -> ", 8);
	    write(pass->Log, pass->HostToLocal, (size_t) rc);
	    if (write(pass->Log, "\n", 1) < 0) {
		fprintf(stderr, "Could not write to log file: %s. Closing\n",
		        strerrno());
		close(pass->Log);
		pass->Log = -1;
	    }
	}
        pass->HostToLocalLength = rc;
        pass->HostToLocalRead += rc;
        RemoveInput(Id);
        pass->OnHosted = 0;
        pass->OnHosted = AddInput(pass->Local, IOWRITE, WriteLocal, Closure);
        pass->HostToLocalOffset = 0;
/*      fprintf(stderr, "Read input on %d: '%.*s'\n",
                Fid, rc, pass->HostToLocal); */
    }
}

static void DoneConnect(WantIo Id, int Fid, void *Closure)
{
    Pass pass;

    pass = Closure;
    fprintf(stderr, "Done connect\n");
    RemoveInput(Id);
    pass->OnHosted = 0;
    pass->OnHosted = AddInput(pass->Hosted, IOREAD, GotHosted, Closure);
}

static void GotAttempt(WantIo Id, int Fid, void *Closure)
{
    int local, hosted, len;
    size_t length;
    unsigned long addr;
    Accept acc;
    Pass   pass;
    struct sockaddr_in peer;
    struct hostent *hp;
    char **hosts, *host;

    acc = Closure;
    fprintf(stderr, "Got attempt");

    len = sizeof(peer);
    local = accept(acc->Local, (struct sockaddr *) &peer, &len);
    if (local < 0) Raise(AcceptError);
    WITH_HANDLING {
        hosted = socket(AF_INET, SOCK_STREAM, 0);
        if (hosted < 0) Raise(SockError);
        WITH_HANDLING {
            if (fcntl(hosted, F_SETFL, O_NONBLOCK) < 0) Raise(FcntlError);
            pass = mynew(struct _Pass);
            WITH_HANDLING {
                pass->Next     = &acc->Passes;
                pass->Previous =  acc->Passes.Previous;
                pass->Previous->Next = pass->Next->Previous = pass;
                pass->Parent   = acc;
                pass->Hosted   = hosted;
                pass->Local    = local;
		pass->Log      = acc->Log; acc->Log = -1;
                pass->HostToLocalLength = pass->LocalToHostLength = 0;
                pass->HostToLocalRead = pass->HostToLocalWritten = 0;
                pass->LocalToHostRead = pass->LocalToHostWritten = 0;
                pass->OnHosted = pass->OnLocal = 0;
                pass->PeerPort = ntohs(peer.sin_port);
                /* Next two lines are for alpha. Do they work ? --Ton */
                memcpy(&addr, &peer.sin_addr, sizeof(addr)); /* align long */
                pass->PeerAddr.s_addr = addr;
                hp = gethostbyaddr((char *) &pass->PeerAddr.s_addr,
                                   sizeof(pass->PeerAddr.s_addr), AF_INET);
                if (hp) {
                    if (!strchr(hp->h_name, '.')) {
                        length = strlen(hp->h_name);
                        for (hosts = hp->h_aliases; (host = *hosts) != NULL;
                             hosts++)
                            if (strncmp(host, hp->h_name, length) == 0 &&
                                host[length] == '.') {
                                pass->PeerName = mystrdup(host);
                                goto done;
                            }
                    }
                    pass->PeerName = mystrdup(hp->h_name);
                } else pass->PeerName = mystrdup(inet_ntoa(pass->PeerAddr));
              done:
                WITH_HANDLING {
                    if (connect(hosted, (struct sockaddr *) &acc->Template,
                                sizeof(acc->Template)) >= 0)
                        pass->OnHosted = AddInput(hosted, IOREAD,
                                                  GotHosted, pass);
                    else if (errno != EINPROGRESS) Raise(ConnectError);
                    else pass->OnHosted =
                        AddInput(hosted, IOWRITE, DoneConnect, pass);
                    pass->OnLocal = AddInput(local, IOREAD, GotLocal, pass);
                    fprintf(stderr, " from %s(%s) %d\n",
                            pass->PeerName, inet_ntoa(pass->PeerAddr),
                            pass->PeerPort);
                } ON_EXCEPTION {
                    putc('\n', stderr);
                    myfree(pass->PeerName);
                    ReRaise();
                } END_HANDLING;
            } ON_EXCEPTION {
                pass->Previous->Next = pass->Next;
                pass->Next->Previous = pass->Previous;
                myfree(pass);
                ReRaise();
            } END_HANDLING;
        } ON_EXCEPTION {
            close(hosted);
            ReRaise();
        } END_HANDLING;
    } ON_EXCEPTION {
        close(local);
        ActionException(theException);
        CleanupException(theException);
    } END_HANDLING;
}

static void Connect(const char *Site, int Port, int LocalPort)
{
    struct sockaddr_in server;
    struct hostent    *hp;
    int                local, length;
    unsigned long      addr;
    Accept             acc;

    acc = mynew(struct _Accept);
    WITH_HANDLING {
        local = socket(AF_INET, SOCK_STREAM, 0);
        if (local < 0)  Raise(SockError);
        WITH_HANDLING {
            addr = inet_addr(Site);
            /* We really expect (unsigned long) -1, but alpha returns
               only part. So we should use INADDR_NONE, but often excluded
               from netinet/in.h so just use INADDR_BROADCAST which is the
               same */
       
            if (addr == (unsigned long) INADDR_BROADCAST) {
                hp = gethostbyname(Site);
                if (!hp) Raise1(HostError, strherrno());
                memcpy(&server.sin_addr, hp->h_addr, (size_t) hp->h_length);
                /* Portability trick for alpha (long = 8 bytes) 
                   Hope it doesn't break on other long=8 machines */
                addr = (unsigned long) server.sin_addr.s_addr;
            }
            memcpy(&acc->Template.sin_addr, &addr, sizeof(addr));
            acc->Template.sin_family = AF_INET;
            acc->Template.sin_port = htons(Port);

            /* Name socket using wildcards */
            server.sin_family = AF_INET;
            server.sin_addr.s_addr = INADDR_ANY;
            server.sin_port = htons(LocalPort);
            if (bind(local, (struct sockaddr *) &server, sizeof(server)))
                Raise(BindError);

            length = sizeof(server);
            if (getsockname(local, (struct sockaddr *) &server, &length))
                Raise(GetNameError);
            if (listen(local, 5) < 0) Raise(ListenError);
	    acc->Log   = -1;
            acc->Local = local;
            acc->LocalPort = ntohs(server.sin_port);
            acc->HostPort  = Port;
            acc->Passes.Next = acc->Passes.Previous = &acc->Passes;
            acc->HostSite  = mystrdup(Site);
            WITH_HANDLING {
                acc->Id       = AddInput(local, IOREAD, GotAttempt, acc);
                acc->Next     = &AcceptBase;
                acc->Previous =  AcceptBase.Previous;
                acc->Previous->Next = acc->Next->Previous = acc;
                printf("0 %d\n", acc->LocalPort);
            } ON_EXCEPTION {
                myfree(acc->HostSite);
                ReRaise();
            } END_HANDLING;
        } ON_EXCEPTION {
            close(local);
            ReRaise();
        } END_HANDLING;
    } ON_EXCEPTION {
        myfree(acc);
        ActionException(theException);
        CleanupException(theException);
        printf("1\n");
    } END_HANDLING;
}

static void Kill(int sock)
{
    Accept acc;
    Pass   pass;

    for (acc=AcceptBase.Next; acc != &AcceptBase; acc = acc->Next)
        if (acc->Local == sock) {
            FreeAcc(acc);
            printf("0\n");
            return;
        } else
            for (pass = acc->Passes.Next; pass != &acc->Passes;
                 pass = pass->Next)
                if (pass->Hosted == sock || pass->Local == sock) {
                    FreePass(pass);
                    printf("0\n");
                    return;
                }
    fprintf(stderr, "Could not find socket %d\n", sock);
    printf("1\n");
}

static void Log(int sock, const char *Name)
{
    Accept acc;
    Pass   pass;
    int    fd;

    for (acc=AcceptBase.Next; acc != &AcceptBase; acc = acc->Next)
        if (acc->Local == sock) {
            fd = creat(Name, 0644);
	    if (fd >= 0) {
	        acc->Log = fd;
                printf("0\n");
	        return;
	    }
	    fprintf(stderr, "Could not create %s: %s\n", Name, strerrno());
	    goto fail;
        } else
            for (pass = acc->Passes.Next; pass != &acc->Passes; pass = pass->Next)
                if (pass->Hosted == sock || pass->Local == sock) {
                    fd = creat(Name, 0644);
		    if (fd >= 0) {
		        pass->Log = fd;
                        printf("0\n");
		        return;
		    }
		    fprintf(stderr, "Could not create %s: %s\n",
		            Name, strerrno());
		    goto fail;
                }
  fail:
    printf("1\n");
}

static void UnLog(int sock)
{
    Accept acc;
    Pass   pass;

    for (acc=AcceptBase.Next; acc != &AcceptBase; acc = acc->Next)
        for (pass = acc->Passes.Next; pass != &acc->Passes; pass = pass->Next)
            if (pass->Hosted == sock || pass->Local == sock) {
		if (pass->Log >= 0) {
		    close(pass->Log);
		    pass->Log = -1;
		    return;
		} else
		fprintf(stderr, "Socket %d was not logging.\n", sock);
		goto fail;
            }
  fail:
    printf("1\n");
}


static void PrintStatus(void)
{
    Accept acc;
    Pass   pass;

    for (acc=AcceptBase.Next; acc != &AcceptBase; acc = acc->Next) {
        fprintf(stderr,
                "Translations for %d: %s(%s) %d translated to port %d\n",
                acc->Local, acc->HostSite, inet_ntoa(acc->Template.sin_addr),
                acc->HostPort, acc->LocalPort);
        for (pass = acc->Passes.Next; pass != &acc->Passes;pass = pass->Next) {
            fprintf(stderr,
                    "  Connect: socket %d (host) to %d (local): %s(%s) %d\n",
                    pass->Hosted, pass->Local,
                    pass->PeerName, inet_ntoa(pass->PeerAddr), pass->PeerPort);
            fprintf(stderr, "           HostToLocal: read %08lu written %08lu\n",
                    pass->HostToLocalRead, pass->HostToLocalWritten);
            fprintf(stderr, "           LocalToHost: read %08lu written %08lu\n",
                    pass->LocalToHostRead, pass->LocalToHostWritten);
            fprintf(stderr, "           log: %d\n", pass->Log);
        }
    }
}

#define CONNECTSYNTAX "connect <site> <port> [<localport>]"
#define KILLSYNTAX    "kill <socket>"
#define LOGSYNTAX     "log <socket> <file>"
#define UNLOGSYNTAX   "unlog <socket>"

/* Very naive interpreter. Replace by something better --Ton */
static void DoCommand(const char *Line, size_t length)
{
    char Buf[5][200], *ptr;
    int i, j, rc;

    Buf[0][0] = 0;
    rc = sscanf(Line, "%s%s%s%s%s", Buf[0], Buf[1], Buf[2], Buf[3], Buf[4]);
    if (strcmp(Buf[0], "connect") == 0)
        if (rc < 3 || rc > 4)
            fprintf(stderr, "syntax: " CONNECTSYNTAX "\n");
        else {
            i = strtol(Buf[2], &ptr, 0);
            if (*ptr || ptr == Buf[2])
                fprintf(stderr, "syntax: " CONNECTSYNTAX "\n");
            else {
                if (rc == 4) {
                    j = strtol(Buf[3], &ptr, 0);
                    if (*ptr || ptr == Buf[3])
                        fprintf(stderr, "syntax: " CONNECTSYNTAX "\n");
                    else {
                        Connect(Buf[1], i, j);
                        return;
                    }
                } else {
                    Connect(Buf[1], i, 0);
                    return;
                }
            }
        }
    else if (strcmp(Buf[0], "status") == 0)
        if (rc != 1) fprintf(stderr, "syntax: status\n");
        else {
            PrintStatus();
            printf("0\n");
            return;
        }
    else if (strcmp(Buf[0], "kill") == 0)
        if (rc != 2) fprintf(stderr, "syntax: " KILLSYNTAX "\n");
        else {
            i = strtol(Buf[1], &ptr, 0);
            if (*ptr || ptr == Buf[1])
                fprintf(stderr, "syntax: " KILLSYNTAX "\n");
            else {
                Kill(i);
                return;
            }
        }
    else if (strcmp(Buf[0], "log") == 0)
        if (rc != 3) fprintf(stderr, "syntax: " LOGSYNTAX "\n");
        else {
            i = strtol(Buf[1], &ptr, 0);
            if (*ptr || ptr == Buf[1])
                fprintf(stderr, "syntax: " LOGSYNTAX "\n");
            else {
                Log(i, Buf[2]);
                return;
            }
        }
    else if (strcmp(Buf[0], "unlog") == 0)
        if (rc != 3) fprintf(stderr, "syntax: " UNLOGSYNTAX "\n");
        else {
            i = strtol(Buf[1], &ptr, 0);
            if (*ptr || ptr == Buf[1])
                fprintf(stderr, "syntax: " UNLOGSYNTAX "\n");
            else {
                UnLog(i);
                return;
            }
        }
    else {
        fprintf(stderr, "Valid command are:\n");
        fprintf(stderr, "  " CONNECTSYNTAX "\n");
        fprintf(stderr, "  " KILLSYNTAX "\n");
        fprintf(stderr, "  " LOGSYNTAX "\n");
        fprintf(stderr, "  status\n");
        fprintf(stderr, "  help\n");
        if (strcmp(Buf[0], "help") == 0) {
            printf("0\n");
            return;
        }
    }
/*  fail: */
    printf("1\n");
}

static void GotStdin(WantIo Id, int Fid, void *Closure)
{
    char Buffer[200], *ptr, *Base, *End;
    int rc;

    rc = read(Fid, Buffer+ComPos, sizeof(Buffer)-ComPos);
    if (rc <= 0)
        if (rc == 0) Raise(InputLost);
        else fprintf(stderr, "Error on stdin: %s\n", strerrno());
    else {
        ptr = memchr(Buffer+ComPos, '\n', (size_t) rc);
        ComPos += rc;
        if (ptr) {
            Base = Buffer;
            End  = Buffer+ComPos;
            do {
                if (IgnoreIn) IgnoreIn = 0;
                else {
                    *ptr = 0;
                    DoCommand(Base, (size_t) (ptr-Base));
                }
                Base = ++ptr;
            } while ((ptr = memchr(Base, '\n', (size_t) (End-Base))) != NULL);
            ComPos = End-Base;
            memmove(Buffer, Base, ComPos);
        } else if (IgnoreIn) ComPos = 0;
        if (ComPos == sizeof(Buffer)) {
            fprintf(stderr, "Command overflow. Ignoring upto next new line\n");
            ComPos = 0;
            IgnoreIn = 1;
        }
    }
}

static int Running;

int main(void)
{
    Running = 1;
    ComPos = 0;
    IgnoreIn = 0;
    InitInput();
    WITH_UNWIND {
        AddInput(0, IOREAD, GotStdin, NULL);
        do {
            Wait();
        } while(Running);
    } ON_UNWIND {
        CleanInput();
        while (AcceptBase.Next != &AcceptBase) FreeAcc(AcceptBase.Next);
    } END_UNWIND;
    return 0;
}
