#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include <mymalloc.h>
#include <except.h>
#include <myxlib.h>

#include "GoBoard.h"
#include "SmeBell.h"

#include "analyze.h"
#include "broadcast.h"
#include "connect.h"
#include "events.h"
#include "games.h"
#include "gointer.h"
#include "gospel.h"
#include "match.h"
#include "messages.h"
#include "observe.h"
#include "players.h"
#include "reviews.h"
#include "stats.h"
#include "tell.h"
#include "utils.h"
#include "version.h"
#include "xgospel.h"

#define DEBUGFUN              1
#define DEBUGPENDING          2
#define DEBUGABORTONEXCEPTION 4
#define DEBUGROUNDTRIP        8
#define DEBUGBISON        0x100
#define DEBUGFLEX         0x200

#define MIX        100
#define RESOLUTION 10000

/*
#define VERSIONMESSAGE                                                       \
"\nThe latest version of the program should be available for anonymous ftp " \
"on cc1.kuleuven.ac.be in the directory ANONYMOU.200 as the file "           \
"XGOSPEL.TARZ. Just get the file by name, you don't want to do a dir in "    \
"this place (in case you ever wondered why people invented directories, "    \
"DO try dir :-) ). Don't forget to set the transfermode to binary (cc1 is "  \
"an EBCDIC system). Once you have the file, rename it to xgospel.tar.Z and " \
"proceed as usual.\n"                                                        \
"\nIf you don't want this message anymore, set the X resource "              \
"xgospel*version: %s\n"
*/

#define VERSIONMESSAGE                                                       \
"\nThe latest version of the program should be available for anonymous ftp " \
"on bsdserver.ucsf.edu in the directory Go/clients as the file "             \
"xgospel???.tar.Z . Don't forget to set the transfermode to binary.\n"       \
"If you can handle an ocasional crash, you can also ftp the latest "         \
"development version at linux3.cc.kuleuven.ac.be in the directory "          \
"pub/games/Go.\n"                                                            \
"\nIf you don't want this message anymore, set the X resource "              \
"xgospel*version: %s\n"

/* #define SITE "bsdserver.ucsf.edu" */
/* #define SITE "hellspark.wharton.upenn.edu" */
#define SITE    "igs.nuri.net"

/* We measure the roundtrip time (to the X server) every RETRIP seconds */
#define RETRIP    5
#define NEWOUTPUT 3
/*****************************************************************************/

static Widget Info, Input, Stdout, Localtime, Universaltime, Servertime;
static Widget ErrorBeep, ErrorRaise, OutputBeep, OutputRaise, Overwrite, Main;
static char  *MainFileName;
static String DateFormat;
static unsigned long ClockTime, TripTime, LastTrip, RoundTrip, LastRoundTrip;
static unsigned long OutputTime, CurTime;
static time_t        BaseTime;

static Atom   TripAtom;
static int    DebugRoundTrip;
char         *SgfDateFormat, *CompileTime;
static void IgsCommand(    Widget w,XEvent *event,String *string, Cardinal *n);
static void IgsUserCommand(Widget w,XEvent *event,String *string, Cardinal *n);
static void IgsSendCommand(Widget w,XEvent *event,String *string, Cardinal *n);
static void TripMessage(   Widget w,XEvent *event,String *string, Cardinal *n);
static void DoOutput(      Widget w,XEvent *event,String *string, Cardinal *n);

static XtActionsRec actionTable[] = {
    { (String) "igscommand",  IgsCommand     },
    { (String) "usercommand", IgsUserCommand },
    { (String) "sendcommand", IgsSendCommand },
    { (String) "tripmessage", TripMessage    },
    { (String) "output",      DoOutput       },
};

AppData         appdata;
XtAppContext	app_context;

FILE               *DebugFile;
int                 DebugFun, DebugPending;
Widget              toplevel;
static int	    global_argc;
static char       **global_argv;
char               *MyPassword, *MyName, *UserId;
int                 SetServerTime, Entered;
Boolean             WantStdout;
struct tm           LocalTime, UniversalTime, ServerTime;

/*****************************************************************************/

extern String fallback_resources[];
extern int    RealQuit;
extern int    IgsYYdebug, IgsYY_flex_debug;

extern void   _IgsRestartParse(void);

void IgsPrompt(void)
{
    Output("> ");
}

static void DebugFunCommand(const char *args)
{
    char *ptr;
    int   value;

    while (*args == ' ') args++;
    value = strtol(args, &ptr, 0);
    if (ptr == args) Outputf("DebugFun remains 0x%02x\n", appdata.DebugFun);
    else {
        Outputf("DebugFun changed from 0x%02x to 0x%02x\n",
                appdata.DebugFun, value);
        appdata.DebugFun = value;
        DebugFun         = (value & DEBUGFUN)       != 0;
        DebugPending     = (value & DEBUGPENDING)   != 0;
        AbortOnException = (value & DEBUGABORTONEXCEPTION) ? -1 : 0;
        DebugRoundTrip   = (value & DEBUGROUNDTRIP) != 0;
        IgsYYdebug       = (value & DEBUGBISON)     != 0;
        IgsYY_flex_debug = (value & DEBUGFLEX)      != 0;
    }
}

static void ShowCounters(const char *args)
{
    Outputf("Current time:                                 %4lu\n",CurTime);
    Outputf("Time for next triptest:                       %4lu\n",TripTime);
    Outputf("Time for next counters update:                %4lu\n",ClockTime);
    Outputf("Timer for how soon to bell/raise main widget: %4lu\n",OutputTime);
    Outputf("Last X server round trip:                     %4lu\n",LastRoundTrip);
    Outputf("Moving average X server round trip:           %4lu.%02lu\n",
            RoundTrip/RESOLUTION,
            (100*RoundTrip)/RESOLUTION-100*(RoundTrip/RESOLUTION));
    ShowConnectCounters(NULL);
}

static void ShowCommands(const char *Options)
{
    Output("Xgospel extended commands:\n");
    Outputf("  showcommands\n");
    Outputf("  dumpgames\n");
    Outputf("  dumpplayers\n");
    Outputf("  dumpreviews\n");
    Outputf("  debugfun\n");
    Outputf("  showcounters\n");
    Outputf("  xgospeltell\n");
}

static void IgsCommand(Widget w, XEvent *event, String *string, Cardinal *n)
{
    String          Buffer;
    int             Prompt;

    XtVaGetValues(w, XtNstring, &Buffer, NULL);
/*  BatchAppendText(Info, Buffer, strlen(Buffer), 0); */
    Prompt = 1;
    while (isspace(*Buffer)) Buffer++;
    if      (!strncmp(Buffer, "xgospelcommands",15)) ShowCommands(Buffer+15);
    else if (!strncmp(Buffer, "dumpgames"   , 9)) DumpGames(Buffer+9);
    else if (!strncmp(Buffer, "dumpplayers" ,11)) DumpPlayers(Buffer+11);
    else if (!strncmp(Buffer, "dumpreviews" ,11)) DumpReviews(Buffer+11);
    else if (!strncmp(Buffer, "debugfun"    , 8)) DebugFunCommand(Buffer+8);
    else if (!strncmp(Buffer, "showcounters",12)) ShowCounters(Buffer+12);
    else if (!strncmp(Buffer, "xgospeltell" ,11))
        if (Buffer[11]) TellXgospelUsers(NULL, Buffer+12);
        else TellXgospelUsers(NULL, "");
    else {
        Prompt = 0;
        if      (!strncmp(Buffer, "review", 6))
            ReviewWanted(NULL, Buffer+6, 2);
        else if (!strncmp(Buffer, "sgf ", 4))
            UserCommand(NULL, "%%sgf %s", Buffer);
        else if (!strncmp(Buffer, "bet ", 4))
            UserCommand(NULL, "%%bet %s", Buffer);
        else if (!strncmp(Buffer, "literal ", 8))
            UserCommand(NULL, "%%literal %s", Buffer+8);
        else UserCommand(NULL, "%s", Buffer);
    }
    if (Prompt) IgsPrompt();
    XtVaSetValues(Input, XtNstring, "", NULL);
}

static void IgsUserCommand(Widget w, XEvent *event,
                           String *string, Cardinal *n)
{
    if (*n == 1)
        if (!strncmp(string[0], "review", 6))
            ReviewWanted(NULL, string[0]+6, 1);
        else UserCommand(NULL, "%s", string[0]);
    else WidgetWarning(w, "usercommand() expects exactly 1 argument");
}

static void IgsSendCommand(Widget w, XEvent *event,
                           String *string, Cardinal *n)
{
    if (*n == 1)
        if (!strncmp(string[0], "review", 6))
            ReviewWanted(NULL, string[0]+6, 1);
        else SendCommand(NULL, NULL, "%s", string[0]);
    else WidgetWarning(w, "sendcommand() expects exactly 1 argument");
}

static void DoOutput(Widget w, XEvent *event, String *string, Cardinal *n)
{
    Cardinal i;

    if (*n) Output(ResourceStringToString(string[0]));
    for (i=1; i<*n; i++) {
        Output(" ");
        Output(ResourceStringToString(string[i]));
    }
    Output("\n");
}

size_t Output(const char *Text)
{
    size_t Length;

    if (Entered && !OutputTime) {
        OutputTime = NEWOUTPUT;
        IfBell(OutputBeep);
        IfRaise(OutputRaise, OutputRaise);
    }

    Length = strlen(Text);
    BatchAppendText(Info, Text, Length, 0);
    return Length;
}

#ifndef HAVE_NO_STDARG_H
size_t Outputf(const char *Comment, ...)
#else  /* HAVE_NO_STDARG_H */
size_t Outputf(va_alist)
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    char    Text[2048];
    va_list args;

    if (Entered && !OutputTime) {
        OutputTime = NEWOUTPUT;
        IfBell(OutputBeep);
        IfRaise(OutputRaise, OutputRaise);
    }

#ifndef HAVE_NO_STDARG_H
    va_start(args, Comment);
#else  /* HAVE_NO_STDARG_H */
    const char *Comment;

    va_start(args);
    Comment = va_arg(args, const char *);
#endif /* HAVE_NO_STDARG_H */
    vsprintf(Text, Comment, args);
    va_end(args);
    return Output(Text);
}

static void TestBoard(Widget w, XtPointer clientdata, XtPointer calldata)
{
    OpenAnalyze(0, "", 0, (size_t) appdata.AnalyzeSize, 0,
                appdata.AllowSuicide);
}

/* Very simpleminded writability test. God help you if you are root */
static Boolean TryToWrite(String Name)
{
    DebugFile = fopen(Name, "w");
    if (DebugFile) return True;
    else           return False;
}

/*****************************************************************************/

static time_t StringToTime(const char *Date, const char *Tim)
{
    static const char *Month[] = {
        "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

    struct tm   TmTime;
    char       *Ptr;
    int         i;

    if (strlen(Tim) != 8 || Tim[2] != ':' || Tim[5] != ':') return 0;
    Ptr = strchr(Date, ' ');
    if (!Ptr || Ptr-Date != 3) return 0;
    for (i=0; i<12; i++) if (memcmp(Date, Month[i], 3) == 0) {
        TmTime.tm_mon  = i;
        TmTime.tm_mday = strtol(Ptr, &Ptr, 10);
        TmTime.tm_year = strtol(Ptr, &Ptr, 10)-1900;
        if (*Ptr) return 0;

        TmTime.tm_hour = atoi(&Tim[0]);
        TmTime.tm_min  = atoi(&Tim[3]);
        TmTime.tm_sec  = atoi(&Tim[6]);

        TmTime.tm_isdst = LocalTime.tm_isdst;
        
        return mktime(&TmTime);
    }
    return 0;
}

static const char *DateString(struct tm *date)
{
    static char Buffer[80];
    char       *ptr;
    size_t      Length;

    if (DateFormat) {
        if (strftime(Buffer, sizeof(Buffer), DateFormat, date))
            return Buffer;
        Warning("strftime(.., %d, %s, ..) overflows the buffer\n",
                    sizeof(Buffer), DateFormat);
        myfree(DateFormat);
        DateFormat = NULL;
    }

    /* This code can also be used if your libraries do not have strftime */
    ptr = asctime(date);
    Length = strlen(ptr)-1;
    memcpy(Buffer, ptr, Length);
    Buffer[Length] = 0;
    return Buffer;
}

/* Returns 0 on success, 1 on failure */
static int SendTripMessage(Widget w, long Message)
{
    Display *dpy;
    Window   win;
    XEvent   myEvent;

    dpy = XtDisplay(w);
    win = XtWindow(w);
    myEvent.xclient.type         = ClientMessage;
    myEvent.xclient.display      = dpy;
    myEvent.xclient.window       = win;
    myEvent.xclient.message_type = TripAtom;
    myEvent.xclient.format       = 32;
    myEvent.xclient.data.l[0]    = Message;
    return !XSendEvent(dpy, win, False, NoEventMask, &myEvent);
}

static void TripMessage(Widget w, XEvent *event, String *string, Cardinal *n)
{
    LastTrip  = event->xclient.data.l[0];
    LastRoundTrip = CurTime-LastTrip;
    RoundTrip = ((MIX-1)*RoundTrip+RESOLUTION*LastRoundTrip)/MIX;
    if (DebugRoundTrip) {
        printf("LastRoundTrip = %3lu, RoundTrip = %4lu.%02lu\n",
               LastRoundTrip, RoundTrip/RESOLUTION,
               (100*RoundTrip)/RESOLUTION-100*(RoundTrip/RESOLUTION));
        fflush(stdout);
    }
}

static void HeartBeat(XtPointer closure, XtIntervalId *id)
{
    unsigned long Diff;
    time_t        NewTime;

    if (++UniversalTime.tm_sec < 60) {
        Diff = 1;
        if (++LocalTime.tm_sec >= 60) mktime(&LocalTime);
    } else {
        time(&NewTime);
        LocalTime     = *localtime(&NewTime);
        UniversalTime = *gmtime(   &NewTime);
#ifndef HAVE_NO_DIFFTIME
        Diff = (unsigned long) difftime(NewTime, BaseTime);
#else                           /* HAVE_NO_DIFFTIME */
        Diff = (unsigned long) (NewTime - BaseTime);
#endif                          /* HAVE_NO_DIFFTIME */
        Diff -= CurTime;
    }
    if (Diff) {
        GamesTime(Diff);
        ReviewsTime(Diff);
        AnalyzeTime(Diff);
        ConnectTime(Diff);

        if (SetServerTime && (ServerTime.tm_sec += Diff) >= 60)
            mktime(&ServerTime);
        CurTime += Diff;
        if (CurTime >= TripTime) {
            TripTime = CurTime + RETRIP;
            SendTripMessage(Main, (long) CurTime);
        }
        if (CurTime >= ClockTime) {
            ClockTime = CurTime + appdata.ClockTimeout;
            if (60 % appdata.ClockTimeout == 0)
                ClockTime -= LocalTime.tm_sec % appdata.ClockTimeout;
            if (Localtime)   
                XtVaSetValues(Localtime, XtNstring,
                              (XtArgVal) DateString(&LocalTime), NULL);
            if (Universaltime)
                XtVaSetValues(Universaltime, XtNstring,
                              (XtArgVal) DateString(&UniversalTime), NULL);
            if (SetServerTime && Servertime)
                XtVaSetValues(Servertime, XtNstring,
                              (XtArgVal) DateString(&ServerTime), NULL);
        }
        if (OutputTime)
            if (OutputTime > Diff) OutputTime -= Diff;
            else OutputTime = 0;
    }
    XtAppAddTimeOut((XtAppContext) closure, 1000, HeartBeat, closure);
}

static void InitHeartBeat(XtAppContext appcontext)
{
    TripAtom = GetAtom(Main, "XGOSPEL_TRIP_PROBE");
    time(&BaseTime);
    LocalTime     = *localtime(&BaseTime);
    UniversalTime = *gmtime(   &BaseTime);
    CurTime       = 0;
    SetServerTime = 0;
    RoundTrip     = 0;
    ClockTime = TripTime = LastTrip = OutputTime = 0;
    XtAppAddTimeOut(appcontext, 1000, HeartBeat, (XtPointer) appcontext);
}

/*****************************************************************************/

static void ConvertAnalyzeSize(XtPointer Closure)
{
    size_t Size;

    Size = appdata.AnalyzeSize;
    if (ConvertSize("Analyze size error", &Size, Closure, 1))
        appdata.AnalyzeSize = Size;
}

static void ChangeAnalyzeSize(Widget w,
                              XtPointer clientdata, XtPointer calldata)
{
    Widget        Root;
    char          size[80];
    static char  *Result;
    XtVarArgsList Args;

    Result = NULL;
    sprintf(size, "%d", appdata.AnalyzeSize);
    Args = XtVaCreateArgsList(NULL, "size", size, NULL);
    Root = AskString(toplevel, ConvertAnalyzeSize, (XtPointer) &Result,
                     "Enter size of analyze board",
                     "analyzeSize", &Result, Args, NULL);
    XtFree(Args);
    MyDependsOn(Root, w);
}

typedef struct {
    AppDataPtr appData;
    char      *analyzeSize;
    char      *allowSuicide;
    char      *replayTimeout;
    char      *gamesTimeout;
    char      *whoTimeout;
    char      *reviewsTimeout;
    char      *clockTimeout;
    char      *serverTimeout;
    char      *inactiveTimeout;
    char      *playersUpdateTimeout;
    char      *gamesUpdateTimeout;
    char      *reviewsUpdateTimeout;
    char      *quitTimeout;
    char      *tersePlay;
} SettingsType;

static void ConvertSettings(XtPointer Closure)
{
    SettingsType *Settings;

    Settings = (SettingsType *) Closure;
    ConvertAnalyzeSize((XtPointer) &Settings->analyzeSize);
    ConvertBoolean("Allow suicide error",
                   &Settings->appData->AllowSuicide,
                   (XtPointer) &Settings->allowSuicide, 1);
    ConvertPositive("Replay rate error", &Settings->appData->ReplayTimeout,
                    (XtPointer) &Settings->replayTimeout, 1);
    ConvertPositive("Who rate error", &Settings->appData->WhoTimeout,
                    (XtPointer) &Settings->whoTimeout, 1);
    ConvertPositive("Games rate error", &Settings->appData->GamesTimeout,
                    (XtPointer) &Settings->gamesTimeout, 1);
    ConvertPositive("Reviews rate error", &Settings->appData->ReviewsTimeout,
                    (XtPointer) &Settings->reviewsTimeout, 1);
    if (ConvertPositive("Clock rate error", &Settings->appData->ClockTimeout,
                        (XtPointer) &Settings->clockTimeout, 1))
        ClockTime = CurTime;
    ConvertPositive("Server resend rate error",
                    &Settings->appData->ServerTimeout,
                    (XtPointer) &Settings->serverTimeout, 1);
    ConvertPositive("Disconnect rate error",
                    &Settings->appData->InactiveTimeout,
                    (XtPointer) &Settings->inactiveTimeout, 1);
    ConvertNatural ("Player window update rate error",
                    &Settings->appData->PlayerUpdateTimeout,
                    (XtPointer) &Settings->playersUpdateTimeout, 1);
    ConvertNatural ("Game window update rate error",
                    &Settings->appData->GameUpdateTimeout,
                    (XtPointer) &Settings->gamesUpdateTimeout, 1);
    ConvertNatural ("Review window update rate error",
                    &Settings->appData->ReviewUpdateTimeout,
                    (XtPointer) &Settings->reviewsUpdateTimeout, 1);
    ConvertPositive("quit timeout error", &Settings->appData->QuitTimeout,
                    (XtPointer) &Settings->quitTimeout, 1);
    if (ConvertBoolean("Terse play error",
                       &Settings->appData->TersePlay,
                       (XtPointer) &Settings->tersePlay, 1)) {
        /* Send games/who if was requested --Ton */ ;
    }
}

static void ChangeSettings(Widget w, XtPointer clientdata, XtPointer calldata)
{
    AppDataPtr    appData;
    Widget        Root;
    char          size[40], replayTimeout[40], whoTimeout[40], quitTimeout[40];
    char          gamesTimeout[40], reviewsTimeout[40], serverTimeout[40];
    char          playersUpdateTimeout[40], gamesUpdateTimeout[40];
    char          reviewsUpdateTimeout[40], clockTimeout[40];
    char          inactiveTimeout[40];
    const char   *tersePlay, *allowSuicide;
    XtVarArgsList ArgsAna, ArgsAllowSuicide, ArgsReplay, ArgsWho, ArgsGames;
    XtVarArgsList ArgsReviews, ArgsServer, ArgsPlayersUpdate, ArgsGamesUpdate;
    XtVarArgsList ArgsReviewsUpdate, ArgsQuit, ArgsInactive;
    XtVarArgsList ArgsTerse, ArgsClock;
    /* Allocate is better as soon as we have more contexts --Ton */
    static SettingsType  Settings;

    appData = (AppDataPtr) clientdata;
    Settings.appData        = appData;
    Settings.analyzeSize    = NULL;
    Settings.allowSuicide   = NULL;
    Settings.replayTimeout  = NULL;
    Settings.whoTimeout     = NULL;
    Settings.gamesTimeout   = NULL;
    Settings.reviewsTimeout = NULL;
    Settings.clockTimeout   = NULL;
    Settings.serverTimeout  = NULL;
    Settings.inactiveTimeout      = NULL;
    Settings.playersUpdateTimeout = NULL;
    Settings.gamesUpdateTimeout   = NULL;
    Settings.reviewsUpdateTimeout = NULL;
    Settings.quitTimeout    = NULL;
    Settings.tersePlay      = NULL;

    sprintf(size,            "%d", appData->AnalyzeSize);
    ArgsAna     = XtVaCreateArgsList(NULL, "size",    size, NULL);
    allowSuicide = appData->AllowSuicide != False ? "True" : "False";
    ArgsAllowSuicide = XtVaCreateArgsList(NULL, "boolean", allowSuicide, NULL);
    sprintf(replayTimeout,  "%d", appData->ReplayTimeout);
    ArgsReplay  = XtVaCreateArgsList(NULL, "timeout", replayTimeout, NULL);
    sprintf(whoTimeout,     "%d", appData->WhoTimeout);
    ArgsWho     = XtVaCreateArgsList(NULL, "timeout", whoTimeout, NULL);
    sprintf(gamesTimeout,   "%d", appData->GamesTimeout);
    ArgsGames   = XtVaCreateArgsList(NULL, "timeout", gamesTimeout, NULL);
    sprintf(reviewsTimeout, "%d", appData->ReviewsTimeout);
    ArgsReviews = XtVaCreateArgsList(NULL, "timeout", reviewsTimeout, NULL);
    sprintf(clockTimeout, "%d", appData->ClockTimeout);
    ArgsClock   = XtVaCreateArgsList(NULL, "timeout", clockTimeout, NULL);
    sprintf(serverTimeout,  "%d", appData->ServerTimeout);
    ArgsServer  = XtVaCreateArgsList(NULL, "timeout", serverTimeout, NULL);
    sprintf(inactiveTimeout,   "%d", appData->InactiveTimeout);
    ArgsInactive = XtVaCreateArgsList(NULL, "timeout", inactiveTimeout, NULL);
    sprintf(playersUpdateTimeout, "%d", appData->PlayerUpdateTimeout);
    ArgsPlayersUpdate =
        XtVaCreateArgsList(NULL, "timeout", playersUpdateTimeout, NULL);
    sprintf(gamesUpdateTimeout,   "%d", appData->GameUpdateTimeout);
    ArgsGamesUpdate   =
        XtVaCreateArgsList(NULL, "timeout", gamesUpdateTimeout, NULL);
    sprintf(reviewsUpdateTimeout, "%d", appData->ReviewUpdateTimeout);
    ArgsReviewsUpdate =
        XtVaCreateArgsList(NULL, "timeout", reviewsUpdateTimeout, NULL);
    sprintf(quitTimeout,    "%d", appData->QuitTimeout);
    ArgsQuit  = XtVaCreateArgsList(NULL, "timeout", quitTimeout, NULL);
    tersePlay = appData->TersePlay != False ? "True" : "False";
    ArgsTerse = XtVaCreateArgsList(NULL, "boolean", tersePlay, NULL);

    Root = AskString(toplevel, ConvertSettings, (XtPointer) &Settings,
                     "Enter new settings",
                     "analyzeSize",    &Settings.analyzeSize,    ArgsAna,
                     "allowSuicide",   &Settings.allowSuicide,
                     ArgsAllowSuicide,
                     "replayTimeout",  &Settings.replayTimeout,  ArgsReplay,
                     "whoTimeout",     &Settings.whoTimeout,     ArgsWho,
                     "gamesTimeout",   &Settings.gamesTimeout,   ArgsGames,
                     "reviewsTimeout", &Settings.reviewsTimeout, ArgsReviews,
                     "clockTimeout",   &Settings.clockTimeout,   ArgsClock,
                     "inactiveTimeout",      &Settings.inactiveTimeout,
                     ArgsInactive,
                     "playersUpdateTimeout", &Settings.playersUpdateTimeout,
                     ArgsPlayersUpdate,
                     "gamesUpdateTimeout", &Settings.gamesUpdateTimeout,
                     ArgsGamesUpdate,
                     "reviewsUpdateTimeout", &Settings.reviewsUpdateTimeout,
                     ArgsReviewsUpdate,
                     "serverTimeout",  &Settings.serverTimeout,  ArgsServer,
                     "quitTimeout",    &Settings.quitTimeout,    ArgsQuit,
                     "tersePlay",      &Settings.tersePlay,      ArgsTerse,
                     NULL);

    XtFree(ArgsAna);
    XtFree(ArgsAllowSuicide);
    XtFree(ArgsReplay);
    XtFree(ArgsWho);
    XtFree(ArgsGames);
    XtFree(ArgsReviews);
    XtFree(ArgsServer);
    XtFree(ArgsInactive);
    XtFree(ArgsPlayersUpdate);
    XtFree(ArgsGamesUpdate);
    XtFree(ArgsReviewsUpdate);
    XtFree(ArgsQuit);
    XtFree(ArgsTerse);
    XtFree(ArgsClock);
    MyDependsOn(Root, w);
}

static void ChangeMainFilename(Widget w,
                               XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "mainFilename",
                          "Enter name of main file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

static void SaveMain(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Boolean                   overwrite;
    static char const * const ErrorType = "Main save error";

    if (Info) {
        if (Overwrite) CallReadToggle(Overwrite, (XtPointer) &overwrite, NULL);
        else           overwrite = False;

        SaveWrite(MainFileName, overwrite,
                  ErrorBeep, ErrorRaise, ErrorType,
                  SaveTextFun, (XtPointer) Info);
    } else {
        IfBell(ErrorBeep);
        IfRaise(ErrorRaise, ErrorRaise);
        PopMessage(ErrorType, "Main messages were ignored so there is "
                   "nothing to be saved");
    }
}

static char *MainTitle(const char *Pattern, XtPointer Closure)
{
    return StringToFilename(Pattern,
                            (int) 'N', (char *) Closure,
                            (int) 'v', VERSION,
                            (int) 'V', appdata.Version,
                            (int) 'd', DATE,
                            (int) 'D', CompileTime, /* in GMT format */
                            0);
}

static XrmOptionDescRec options[] = {
    {(char *)"-fg",	    (char *)"*foreground", XrmoptionSepArg, NULL},
    {(char *)"-foreground", (char *)"*foreground", XrmoptionSepArg, NULL},
    {(char *)"-user",       (char *)"*user",       XrmoptionSepArg, NULL},
    {(char *)"-password",   (char *)"*password",   XrmoptionSepArg, NULL},
    {(char *)"-host",       (char *)"*site",       XrmoptionSepArg, NULL},
    {(char *)"-site",       (char *)"*site",       XrmoptionSepArg, NULL},
    {(char *)"-port",       (char *)"*port",       XrmoptionSepArg, NULL},
    {(char *)"-debugFile",  (char *)"*debugFile",  XrmoptionSepArg, NULL},
    {(char *)"-debugFun",   (char *)"*debugFun",   XrmoptionSepArg, NULL},
    {(char *)"-debug",      (char *)"*debug",	   XrmoptionNoArg,
         (XPointer) "True"},
};

static const char *UseMessages[] = {
    "    -user       name         Userid on IGS",
    "    -password   secret       Password on IGS",
    "    -host       site         Connect to site",
    "    -site       site         Connect to site",
    "    -port       port         Connect to port",
    "    -debugfile  name         File to log interaction with IGS",
    "    -debugfun   int          Set program debugging flags",
    "    -debug                   Turn on internal widget debugging",
};

#define XtNuser                  "user"
#define XtCUser                  "User"
#define XtNpassword              "password"
#define XtCPassword              "Password"
#define XtNuseTerm               "useTerm"
#define XtCUseTerm               "UseTerm"
#define XtNdebugFun              "debugFun"
#define XtCDebugFun              "DebugFun"
#define XtNdebugFile             "debugFile"
#define XtCDebugFile             "DebugFile"
#define XtNkibitzSize            "kibitzSize"
#define XtCKibitzSize            "KibitzSize"
#define XtNtellsize              "tellsize"
#define XtCTellsize              "Tellsize"
#define XtNbroadcastSize         "broadcastSize"
#define XtCBroadcastSize         "BroadcastSize"
#define XtNyellsize              "yellsize"
#define XtCYellsize              "Yellsize"
#define XtCKibitzSize            "KibitzSize"
#define XtNmaintainer            "maintainer"
#define XtCMaintainer            "Maintainer"
#define XtNversion               "version"
#define XtCVersion               "Version"
#define XtNdateFormat            "dateFormat"
#define XtNsgfDateFormat         "sgfDateFormat"
#define XtCDateFormat            "DateFormat"
#define XtNreplayTimeout         "replayTimeout"
#define XtNgamesTimeout          "gamesTimeout"
#define XtNreviewsTimeout        "reviewsTimeout"
#define XtNclockTimeout          "clockTimeout"
#define XtNserverTimeout         "serverTimeout"
#define XtNinactiveTimeout       "inactiveTimeout"
#define XtNplayersUpdateTimeout  "playersUpdateTimeout"
#define XtNgamesUpdateTimeout    "gamesUpdateTimeout"
#define XtNreviewsUpdateTimeout  "reviewsUpdateTimeout"
#define XtNwhoTimeout            "whoTimeout"
#define XtNquitTimeout           "quitTimeout"
#define XtNreconnectTimeout      "reconnectTimeout"
#define XtCTimeout               "Timeout"
#define XtNgamesScale            "gamesScale"
#define XtNplayersScale          "playersScale"
#define XTCScale                 "Scale"
#define XtNscrollUnit            "scrollUnit"
#define XtCScrollUnit            "ScrollUnit"
#define XtNsite                  "site"
#define XtCSite                  "Site"
#define XtNport                  "port"
#define XtCPort                  "Port"
#define XtNdirectory             "directory"
#define XtCDirectory             "Directory"
#define XtNanalyzeFilename       "analyzeFilename"
#define XtNsgfFilename           "sgfFilename"
#define XtNpsFilename            "psFilename"
#define XtNkibitzFilename        "kibitzFilename"
#define XtNbroadcastFilename     "broadcastFilename"
#define XtNyellFilename          "yellFilename"
#define XtNigsMessageFilename    "igsMessageFilename"
#define XtNtellFilename          "tellFilename"
#define XtNmessageFilename       "messageFilename"
#define XtNeventsFilename        "eventsFilename"
#define XtNgamesFilename         "gamesFilename"
#define XtNplayersFilename       "playersFilename"
#define XtNmainFilename          "mainFilename"
#define XtCFileFilename          "FileFilename"
#define XtNtellKill              "tellKill"
#define XtNyellKill              "yellKill"
#define XtNbroadcastKill         "broadcastKill"
#define XtNkibitzKill            "kibitzKill"
#define XtCKill                  "Kill"
#define XtNtellPass              "tellPass"
#define XtNyellPass              "yellPass"
#define XtNbroadcastPass         "broadcastPass"
#define XtNkibitzPass            "kibitzPass"
#define XtCPass                  "Pass"
#define XtNfriends               "friends"
#define XtCFriends               "Friends"
#define XtCTextToWidget          "TextToWidget"
#define XtNplayerToWidget        "playerToWidget"
#define XtNgameToWidget          "gameToWidget"
#define XtNreviewToWidget        "reviewToWidget"
#define XtNanalyzeSize           "analyzeSize"
#define XtCAnalyzeSize           "AnalyzeSize"
#define XtNallowSuicide          "allowSuicide"
#define XtCAllowSuicide          "AllowSuicide"
#define XtNversionMessage        "versionMessage"
#define XtCVersionMessage        "VersionMessage"
#define XtNtersePlay             "tersePlay"
#define XtCTersePlay             "TersePlay"

#define offset(field) XtOffset(AppDataPtr, field)

static XtResource resources[] = {
    { (String) XtNuser, (String) XtCUser, XtRString, sizeof(String),
      offset(User), XtRString, NULL },
    { (String) XtNpassword, (String) XtCPassword, XtRString, sizeof(String),
      offset(Password), XtRString, NULL },
    { (String) XtNuseTerm, (String) XtCUseTerm, XtRBoolean, sizeof(Boolean),
      offset(UseTerm), XtRString, (XtPointer) "True" },
    { (String) XtNdebugFile, (String) XtCDebugFile, XtRString, sizeof(String),
      offset(DebugFile), XtRString, NULL },
    { (String) XtNdebugFun, (String) XtCDebugFun, XtRInt, sizeof(int),
      offset(DebugFun), XtRString, (XtPointer) "0" },
    { (String) XtNtellsize, (String) XtCTellsize, XtRInt, sizeof(int),
      offset(TellSize), XtRString, (XtPointer) "128" },
    { (String) XtNkibitzSize, (String) XtCKibitzSize, XtRInt, sizeof(int),
      offset(KibitzSize), XtRString, (XtPointer) "180" },
    { (String) XtNmaintainer, (String) XtCMaintainer, XtRString, sizeof(String),
      offset(Maintainer), XtRString, (XtPointer) "AshaiRey" },
    { (String) XtNversion, (String) XtCVersion, XtRString, sizeof(String),
      offset(Version), XtRString, (XtPointer) VERSION },
    { (String) XtNversionMessage, (String) XtCVersionMessage, XtRString, sizeof(String),
      offset(VersionMessage), XtRString, (XtPointer) VERSIONMESSAGE },
    { (String) XtNdateFormat, (String) XtCDateFormat, XtRString, sizeof(String),
      offset(DateFormat), XtRString, NULL },
    { (String) XtNsgfDateFormat, (String) XtCDateFormat, XtRString, sizeof(String),
      offset(SgfDateFormat), XtRString, (XtPointer) "%Y/%m/%d %H:%M:%S UT" },
    { (String) XtNbroadcastSize, (String) XtCBroadcastSize, XtRInt, sizeof(int),
      offset(BroadcastSize), XtRString, (XtPointer) "128" },
    { (String) XtNyellsize, (String) XtCYellsize, XtRInt, sizeof(int),
      offset(YellSize), XtRString, (XtPointer) "128" },
    { (String) XtNtersePlay, (String) XtCTersePlay, XtRBoolean, sizeof(Boolean),
      offset(TersePlay), XtRString, (XtPointer) "True" },
    { (String) XtNplayersScale, (String) XTCScale, XtRInt, sizeof(int),
      offset(PlayersScale), XtRString, (XtPointer) "20" },
    { (String) XtNgamesScale, (String) XTCScale, XtRInt, sizeof(int),
      offset(GamesScale), XtRString, (XtPointer) "5" },
    { (String) XtNscrollUnit, (String) XtCScrollUnit, XtRInt, sizeof(int),
      offset(ScrollUnit), XtRString, (XtPointer) "20" },
    { (String) XtNserverTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(ServerTimeout), XtRString, (XtPointer) "120" },
    { (String) XtNinactiveTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(InactiveTimeout), XtRString, (XtPointer) "2700" },
    { (String) XtNwhoTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(WhoTimeout), XtRString, (XtPointer) "300" },
    { (String) XtNplayersUpdateTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(PlayerUpdateTimeout), XtRString, (XtPointer) "20" },
    { (String) XtNreplayTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(ReplayTimeout), XtRString, (XtPointer) "1" },
    { (String) XtNgamesTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(GamesTimeout), XtRString, (XtPointer) "300" },
    { (String) XtNgamesUpdateTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(GameUpdateTimeout), XtRString, (XtPointer) "2" },
    { (String) XtNreviewsTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(ReviewsTimeout), XtRString, (XtPointer) "1200" },
    { (String) XtNclockTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(ClockTimeout), XtRString, (XtPointer) "1" },
    { (String) XtNreviewsUpdateTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(ReviewUpdateTimeout), XtRString, (XtPointer) "2" },
    { (String) XtNquitTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(QuitTimeout), XtRString, (XtPointer) "20" },
    { (String) XtNreconnectTimeout, (String) XtCTimeout, XtRInt, sizeof(int),
      offset(ReconnectTimeout), XtRString, (XtPointer) "15" },
    { (String) XtNanalyzeSize, (String) XtCAnalyzeSize, XtRInt, sizeof(int),
      offset(AnalyzeSize), XtRString, (XtPointer) "19" },
    { (String) XtNallowSuicide, (String) XtCAllowSuicide, XtRBoolean, sizeof(Boolean),
      offset(AllowSuicide), XtRString, (XtPointer) "False" },
    { (String) XtNdirectory, (String) XtCDirectory, XtRString, sizeof(String),
      offset(Directory), XtRString, (XtPointer) "" },
    { (String) XtNanalyzeFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(AnalyzeFilename), XtRString, (XtPointer) "%D%N%l%L%U%B%U%b%U%V%U%w%U%W%t%T" },
    { (String) XtNsgfFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(SgfFilename), XtRString, (XtPointer) "%D%B%U%b%U%V%U%w%U%W%t%T" },
    { (String) XtNpsFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(PsFilename),  XtRString, (XtPointer) "%D%B%U%b%U%V%U%w%U%W%t%T" },
    { (String) XtNkibitzFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(KibitzFilename), XtRString, (XtPointer) "%D%B%U%b%U%V%U%w%U%W%t%T" },
    { (String) XtNbroadcastFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(BroadcastFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNyellFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(YellFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNigsMessageFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(IgsMessageFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNeventsFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(EventsFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNtellFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(TellFilename), XtRString, (XtPointer) "%D%N%t%n.%T" },
    { (String) XtNmessageFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(MessageFilename), XtRString, (XtPointer) "%D%N%U%n.%T" },
    { (String) XtNgamesFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(GamesFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNplayersFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(PlayersFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNmainFilename, (String) XtCFileFilename, XtRString, sizeof(String),
      offset(MainFilename), XtRString, (XtPointer) "%D%T" },
    { (String) XtNplayerToWidget, (String) XtCTextToWidget, (String) XtRStringPairList,
      sizeof(StringPairListPtr), offset(PlayerToWidget), XtRString, NULL},
    { (String) XtNgameToWidget, (String) XtCTextToWidget, (String) XtRStringPairList,
      sizeof(StringPairListPtr), offset(GameToWidget), XtRString, NULL},
    { (String) XtNreviewToWidget, (String) XtCTextToWidget, (String) XtRStringPairList,
      sizeof(StringPairListPtr), offset(ReviewToWidget), XtRString, NULL},
    { (String) XtNtellKill, (String) XtCKill, (String) XtRStringList,
      sizeof(StringListPtr), offset(TellKill), XtRString, NULL},
    { (String) XtNtellPass, (String) XtCPass, (String) XtRStringList,
      sizeof(StringListPtr), offset(TellPass), XtRString, NULL},
    { (String) XtNyellKill, (String) XtCKill, (String) XtRStringList,
      sizeof(StringListPtr), offset(YellKill), XtRString, NULL},
    { (String) XtNyellPass, (String) XtCPass, (String) XtRStringList,
      sizeof(StringListPtr), offset(YellPass), XtRString, NULL},
    { (String) XtNbroadcastKill, (String) XtCKill, (String) XtRStringList,
      sizeof(StringListPtr), offset(BroadcastKill), XtRString, NULL},
    { (String) XtNbroadcastPass, (String) XtCPass, (String) XtRStringList,
      sizeof(StringListPtr), offset(BroadcastPass), XtRString, NULL},
    { (String) XtNkibitzKill, (String) XtCKill, (String) XtRStringList,
      sizeof(StringListPtr), offset(KibitzKill), XtRString, NULL},
    { (String) XtNkibitzPass, (String) XtCPass, (String) XtRStringList,
      sizeof(StringListPtr), offset(KibitzPass), XtRString, NULL},
    { (String) XtNfriends, (String) XtCFriends, (String) XtRStringList,
      sizeof(StringListPtr), offset(Friends), XtRString, NULL},
    { (String) XtNsite, (String) XtCSite, XtRString, sizeof(String),
      offset(Site), XtRString, (XtPointer) SITE },
    { (String) XtNport, (String) XtCPort, XtRInt, sizeof(int),
      offset(Port), XtRString, (XtPointer) "6969" },
};
#undef offset

int main(int argc, char **argv)
{
    char    *ptr, Name[200];
    Widget   Buttons, Collect, AnalyzeButton, Save, File;
    Widget   analyzeSize, programSettings, ResTree;
    Window   GroupWindow;
    time_t   Now;
    Cardinal i;
    Arg      args[3];

    ExceptionProgram = argv[0];
    global_argc      = argc;
    global_argv      = argv;

    Now = StringToTime(__DATE__, __TIME__);
    if (Now) {
        ptr = Name;
        strftime(Name, sizeof(Name), "%Y-%m-%d %T", gmtime(&Now));
    } else ptr = (char *) __DATE__ " " __TIME__;
    CompileTime = mystrdup(ptr);

    i = 0;
    NameClassArg(args,
                 "Board", boardWidgetClass,
                 "SmeBell", smeBellObjectClass,
                 NULL); i++;
    toplevel = MyAppInitialize(&app_context, "XGospel",
                               options, XtNumber(options),
                               &argc, argv, fallback_resources, args, i,
                               UseMessages, XtNumber(UseMessages),
                               NULL);

    Main    = XtNameToWidget(toplevel, XtName(toplevel));
    if (!Main) Raise1(FatalException, "Could not find real toplevel");

    XtAppAddActions(app_context, actionTable, XtNumber(actionTable));
    XtGetApplicationResources(toplevel, &appdata, resources,
                              XtNumber(resources), NULL, 0);

    appdata.WantStdout = False;
    DebugFun         = (appdata.DebugFun & DEBUGFUN)       != 0;
    DebugPending     = (appdata.DebugFun & DEBUGPENDING)   != 0;
    AbortOnException = (appdata.DebugFun & DEBUGABORTONEXCEPTION) ? -1 : 0;
    DebugRoundTrip   = (appdata.DebugFun & DEBUGROUNDTRIP) != 0;
    IgsYYdebug       = (appdata.DebugFun & DEBUGBISON)     != 0;
    IgsYY_flex_debug = (appdata.DebugFun & DEBUGFLEX)      != 0;
    SgfDateFormat = DateFormat   = MyPassword = MyName = NULL;
    sprintf(Name, "%.*s@%.*s",
            (int) sizeof(Name)/2-1, GetUserId(),
            (int) sizeof(Name)/2-1, GetHostname());
    UserId = mystrdup(Name);

    if (appdata.DateFormat)
        DateFormat = mystrdup(appdata.DateFormat);
    if (appdata.SgfDateFormat)
        SgfDateFormat = mystrdup(appdata.SgfDateFormat);
    if (appdata.Password && *appdata.Password) {
        MyPassword = mystrdup(appdata.Password);
        ptr = strchr(MyPassword, ' ');
        if (ptr) *ptr = 0;
    }
    if (appdata.User && *appdata.User) {
        MyName = mystrdup(appdata.User);
        ptr = strchr(MyName, ' ');
        if (ptr) *ptr = 0;
    }
    if (appdata.ReplayTimeout  <  1) appdata.ReplayTimeout  =  1;
    if (appdata.WhoTimeout     < 10) appdata.WhoTimeout     = 10;
    if (appdata.GamesTimeout   < 10) appdata.GamesTimeout   = 10;
    if (appdata.ReviewsTimeout < 10) appdata.ReviewsTimeout = 10;
    if (appdata.ClockTimeout   <  1) appdata.ClockTimeout   =  1;
    if (appdata.InactiveTimeout<  1) appdata.InactiveTimeout=  1;

    SetWidgetTitles(Main, MainTitle, (XtPointer) XtName(toplevel));

    DebugFile = NULL;
    if (appdata.DebugFile) {
        SubstitutionRec FileSubs[5];
        String          fn;

        i = 0;
        FileSubs[i].match = 'N';
        FileSubs[i].substitution = MyName ? MyName : (char *) "NoName";
        i++;
        FileSubs[i].match = 'P';
        FileSubs[i].substitution =
            MyPassword ? MyPassword : (char *) "NoPassword";
        i++;
        fn = XtFindFile(appdata.DebugFile, FileSubs, i, TryToWrite);
        if (fn) XtFree(fn);
    }

    time(&Now);
    Buttons = MyNameToWidget(Main, "main");
    if (!Buttons) Raise1(FatalException, "Could not find main widget");

    Collect         = XtNameToWidget(Buttons, "*collect");
    GamesButton     = XtNameToWidget(Buttons, "*gamesButton");
    PlayersButton   = XtNameToWidget(Buttons, "*playersButton");
    ReviewsButton   = XtNameToWidget(Buttons, "*reviewsButton");
    ServerButton    = XtNameToWidget(Buttons, "*messageButton");
    BroadcastButton = XtNameToWidget(Buttons, "*broadcastButton");
    YellButton      = XtNameToWidget(Buttons, "*yellButton");
    AnalyzeButton   = XtNameToWidget(Buttons, "*analyzeButton");
    EventsButton    = XtNameToWidget(Buttons, "*eventsButton");
    Info            = XtNameToWidget(Buttons, "*info");
    Input           = XtNameToWidget(Buttons, "*input");
    OutputBeep      = XtNameToWidget(Buttons, "*beep");
    OutputRaise     = XtNameToWidget(Buttons, "*raise");
    ErrorBeep       = XtNameToWidget(Buttons, "*errorBeep");
    ErrorRaise      = XtNameToWidget(Buttons, "*errorRaise");
    Overwrite       = XtNameToWidget(Buttons, "*overwrite");
    Save = XtNameToWidget(Buttons, "*save");
    if (Save) XtAddCallback(Save, XtNcallback, SaveMain, NULL);
    File = XtNameToWidget(Buttons, "*file");
    if (File) XtAddCallback(File, XtNcallback, ChangeMainFilename,
                            (XtPointer) &MainFileName);
    Stdout = XtNameToWidget(Buttons, "*stdout");
    if (Stdout) {
        CallReadToggle(Stdout, &appdata.WantStdout, NULL);
        XtAddCallback(Stdout, XtNcallback, CallReadToggle,
                      &appdata.WantStdout);
    }
    analyzeSize = XtNameToWidget(Buttons, "*analyzeSize");
    if (analyzeSize)
        XtAddCallback(analyzeSize, XtNcallback, ChangeAnalyzeSize, NULL);
    programSettings = XtNameToWidget(Buttons, "*programSettings");
    if (programSettings) XtAddCallback(programSettings, XtNcallback,
                                       ChangeSettings, (XtPointer) &appdata);
    MainFileName = StringToFilename(appdata.MainFilename,
                                    (int) 'T', "IGSsession",
                                    (int) 't', "_",
                                    0);
    XtAddCallback(Buttons, XtNdestroyCallback, CallFree, MainFileName);

    Localtime     = XtNameToWidget(Buttons, "*localTime");
    if (Localtime)     AddText(Localtime, DateString(localtime(&Now)));
    Universaltime = XtNameToWidget(Buttons, "*universalTime");
    if (Universaltime) AddText(Universaltime, DateString(gmtime(&Now)));
    Servertime    = XtNameToWidget(Buttons, "*serverTime");
    if (Servertime)    AddText(Servertime, "--- --- -- --:--:-- ----");

    XtSetKeyboardFocus(Collect, Input);
    MyRealizeWidget(Main);
    GroupWindow = XtWindow(Main);
    ResTree = XtNameToWidget(toplevel, "resourceTree");
    if (ResTree)
        XtVaSetValues(ResTree, XtNwindowGroup, (XtArgVal) GroupWindow, NULL);

/*
    XtAppAddInput(app_context, 0,
                  (XtPointer) XtInputReadMask, UserThroughPut, NULL);
*/
    InitGospel();
    InitTextBatch();
    InitConnect(toplevel);
    InitEvents(toplevel);
    InitUtils(toplevel);
    InitGames(toplevel);
    InitPlayers(toplevel);
    InitReviews(toplevel);
    InitBroadcast(toplevel);
    InitYell(toplevel);
    InitMessages(toplevel);
    InitObserve(toplevel);
    InitTell(toplevel);
    InitMatch(toplevel);
    InitStats(toplevel);

    InitHeartBeat(app_context);
    if (AnalyzeButton) XtAddCallback(AnalyzeButton,XtNcallback,TestBoard,NULL);

    WITH_UNWIND {
        Conn = Connect(appdata.Site, appdata.Port);
        RealQuit = 0;
        Entered  = 0;
        do {
            IgsYYparse();
            _IgsRestartParse();
        } while (!RealQuit);
    } ON_UNWIND {
        CleanTell();
        CleanObserve();
        CleanYell();
        CleanBroadcast();
        CleanReviews();
        CleanPlayers();
        CleanGames();
        CleanEvents();
        CleanConnect();
        CleanTextBatch();
        CleanGospel();
        if (DebugFile) {
            fflush(DebugFile);
            fclose(DebugFile);
        }
        if (appdata.DebugFile != False) fflush(stdout);
        XtDestroyWidget(toplevel);
        myfree(SgfDateFormat);
        myfree(DateFormat);
        myfree(UserId);
        myfree(MyName);
        myfree(MyPassword);
        /*      XtDestroyApplicationContext(app_context); */
    } END_UNWIND;
    mallocstats();
    return 0;
}
