(* Copyright 1990 Digital Equipment Corporation. *)
(* Distributed only by permission. *)

(* Lectern: a user interface for viewing documents stored as images *)
(* VBT for viewing images *)

(* Last modified on Mon Apr  4 17:51:03 PDT 1994 by birrell   *)

MODULE ImageVBT EXPORTS ImageVBT;

IMPORT
  Axis,
  Images,
  Math,
  PaintOp, Pixmap, Point,
  Rect, Region,
  Thread, Time,
  VBT;

REVEAL T = Public BRANDED OBJECT
    pm: Images.T;                     (* The pixmap to be displayed *)
    bg: PaintOp.T := PaintOp.Bg;      (* painting areas outside the pixmap *)
    delta: Point.T;                   (* Offset to pixmap's origin *)
  OVERRIDES
    init := Init;
    put := Put;
    get := Get;
    getDelta := GetDelta;
    moveTo := MoveTo;
    repaint := Repaint;
    shape := Shape;
  END;

PROCEDURE Init(v: T; pm: Images.T; bg: PaintOp.T): T =
    (* LL = any *)
  BEGIN
    v.delta := Point.T{0,0};
    v.pm := pm;
    v.bg := bg;
    RETURN v
  END Init;

PROCEDURE Put(v: T; pm: Images.T; bg: PaintOp.T) =
    (* LL = VBT.mu *)
  VAR oldBounds := v.pm.domain(v);
  BEGIN
    v.pm := pm;
    v.bg := bg;
    IF NOT Rect.Equal(oldBounds, v.pm.domain(v)) THEN VBT.NewShape(v) END;
    VBT.Mark(v);
  END Put;

PROCEDURE Get(v: T): Images.T =
    (* LL = VBT.mu *)
  BEGIN
    RETURN v.pm
  END Get;

PROCEDURE GetDelta(v: T): Point.T =
    (* LL = VBT.mu *)
  BEGIN
    RETURN v.delta
  END GetDelta;

PROCEDURE MoveTo(v: T; delta: Point.T; pixelRate := 0.0) =
    (* LL = VBT.mu *)
  CONST TimePerStep = 0.017D0; (* ideal time for each step *)
  VAR
    dom := VBT.Domain(v);
    distance := Math.sqrt(FLOAT(Point.DistSquare(delta, Point.Origin),
                                LONGREAL));
    finish := Time.Now() + distance * FLOAT(pixelRate, LONGREAL);
  PROCEDURE MoveBy(amount: Point.T) =
    VAR badR := Region.Difference(Region.FromRect(dom),
                                  Region.FromRect(Rect.Add(dom, amount)));
    BEGIN
      IF amount # Point.Origin THEN
        VBT.Scroll(v, dom, amount);
        v.delta := Point.Add(v.delta, amount);
        v.repaint(badR);
        VBT.Sync(v, FALSE); (* flush to process boundary *)
      END;
    END MoveBy;
  BEGIN
    IF VBT.IsMarked(v) THEN
      VBT.Unmark(v);
      v.delta := delta;
      v.repaint(Region.Full);
    ELSE
      WHILE v.delta # delta DO
        WITH
          stepStart = Time.Now(),
          remainingSteps = MAX(1, FLOOR((finish-stepStart) / TimePerStep)) DO
          MoveBy(Point.Div(Point.Sub(delta, v.delta), remainingSteps));
          Thread.Pause(stepStart+TimePerStep-Time.Now());
        END;
      END;
    END;
  END MoveTo;

PROCEDURE Repaint(v: T; READONLY rgn: Region.T) =
    (* Originally stolen from PixmapVBT.m3 *)
    (* LL = mu.v *)
  VAR
    dom := VBT.Domain(v);
    bounds: Rect.T;
    delta: Point.T;
    rects := Region.ToRects(rgn);
    a: Rect.Partition;
  BEGIN
    IF NOT Rect.IsEmpty(dom) THEN
      bounds := v.pm.domain(v);
      delta := Point.Add(v.delta,
                         Point.Sub(Rect.Middle(dom),
                                   Rect.Middle(bounds)));
      FOR i := 0 TO LAST(rects^) DO
        Rect.Factor(
            Rect.Meet(dom, rects[i]), Rect.Move(bounds, delta), a, 0, 0);
        v.pm.paint(v, a[2], delta);
        a[2] := Rect.Empty;
        VBT.PolyTexture(v, a, v.bg, Pixmap.Solid)
      END;
    END;
  END Repaint;

PROCEDURE Shape(v: T; ax: Axis.T; <*UNUSED*> n: CARDINAL): VBT.SizeRange =
    (* LL = VBT.mu.v *)
  VAR sr: VBT.SizeRange;
  BEGIN
    sr.pref := Rect.Size(ax, v.pm.domain(v));
    sr.lo := sr.pref;
    sr.hi := sr.pref+1;
    RETURN sr
  END Shape;


BEGIN

END ImageVBT.
