/* Copyright (C) 1994, Digital Equipment Corporation           */
/* All rights reserved.                                        */
/* See the file COPYRIGHT for a full description.              */
/***************************************************************/
/* File: trap.c                                                */
/* Last modified on Wed Jan 25 10:01:35 PST 1995 by kalsow     */
/*      modified on Thu May 12 14:37:29 PDT 1994 by wobber     */
/*      modified on Fri May 17 13:09:27 1991 by hisgen         */
/*      modified on Wed Sep  7 10:27:43 PDT 1988 by roberts    */
/* ----------------------------------------------------------- */
/*     The trap package implements a dynamically-nested        */
/* exception handling facility for C and also provides the     */
/* error routine.  User level documentation is avaliable in    */
/* the trap manpage and in the header file.                    */
/***************************************************************/

#include <stdio.h>
#include <setjmp.h>
#include <varargs.h>
#include "SRCstdlib.h"
#include "strlib.h"
#include "trap.h"

#define MAXERRMSG 1024

/***************************************************************/
/* trapblock structure                                         */
/*                                                             */
/*     Traps are stored in a list of trapblocks which are      */
/* dynamically allocated by calls to trap.  Each trap block    */
/* contains (1) the name of the trap, (2) a link to the        */
/* next trap block in the chain, and (3) the setjmp data.      */
/* Note that this mechanism takes account of the internal      */
/* structure of the jmp_buf and assumes that a frame           */
/* pointer lives in the first word.  Moreover, the code        */
/* assumes that the stack pointer grows downward.              */
/***************************************************************/

typedef struct _trapblock{
    string name;
    struct _trapblock *link;
    jmp_buf data;
} trapblock, *trapptr;

/***************************************************************/
/* Package variables                                           */
/***************************************************************/

static trapptr trapchain = NULL;
static jmp_buf trapbuf;

/***************************************************************/
/* Global variables                                            */
/***************************************************************/

string errmsg;

static char errfmt[MAXERRMSG];
static char errbuf[MAXERRMSG];

int *_trapbuf = (int *) trapbuf;

extern int sys_nerr, errno;
#if !defined(__FreeBSD__)
extern char *sys_errlist[];
#endif

/***************************************************************/
/* Local function declarations                                 */
/***************************************************************/

static void preprocess(/* msg */);
static void flushtrap(/* oldfp */);
static void copyjb(/* dst, src */);



/***************************************************************/
/* trap(name) => _trap(name, setjmp(_trapbuf))                 */
/*                                                             */
/*     The trap function installs a new exception with the     */
/* indicated name.  The original call returns FALSE and any    */
/* generated exception returns TRUE.  If there is an existing  */
/* exception at this level, it is overwritten.  If existing    */
/* traps are higher in the control stack, a new entry is used. */
/* This function is implemented as a macro because setjmp must */
/* be called from the current frame.                           */
/***************************************************************/

int _trap(name, sjflag)
string name;
int sjflag;
{
    trapptr tp;

    if (sjflag) return (TRUE);
#if defined(__FreeBSD__)
    flushtrap((trapbuf[0]._jb[0])-1);
#else
    flushtrap(trapbuf[0]-1);
#endif
    tp = trapchain;
    while (tp != NULL && strcmp(tp->name, name) != 0) tp = tp->link;
#if defined(__FreeBSD__)
    if (tp == NULL || (tp->data[0]._jb[0]) > (trapbuf[0]._jb[0])) {
#else
    if (tp == NULL || (tp->data[0]) > (trapbuf[0])) {
#endif
	tp = (trapptr) getmem(sizeof *tp);
	tp->name = scopy(name);
	tp->link = trapchain;
	trapchain = tp;
    }
    copyjb(tp->data, trapbuf);
    return (FALSE);    
}



/***************************************************************/
/* untrap() => _untrap(setjmp(_trapbuf))                       */
/*                                                             */
/*     The untrap function removes all exceptions at or below  */
/* the current control point.  This function is implemented    */
/* as a macro because setjmp must be called from the current   */
/* frame.                                                      */
/***************************************************************/

void _untrap(sjflag)
int sjflag;
{
    flushtrap(trapbuf[0]);
}



/***************************************************************/
/* raise(name);                                                */
/*                                                             */
/*     Raises the specified exception.  An error is generated  */
/* if no exception is found.                                   */
/***************************************************************/

void raise(name)
string name;
{
    trapptr tp;

    tp = trapchain;
    while (tp != NULL && strcmp(tp->name, name) != 0) tp = tp->link;
    if (tp == NULL) error("Undefined exception name %s", name);
#if defined(__FreeBSD__)
    flushtrap((tp->data[0]._jb[0]) - 1);
#else
    flushtrap((tp->data[0]) - 1);
#endif
    longjmp(tp->data, TRUE);
}



/***************************************************************/
/* error(msg, args);                                           */
/*                                                             */
/*     The error routine is used for error reporting througout */
/* the library packages and in most applications.  To the      */
/* user, error looks like printf and, barring any use of the   */
/* trap routine, simply prints a message to stderr of the form */
/*                                                             */
/*                 Error: <error-message>\n                    */
/*                                                             */
/* where <error-message> is composed of the string msg after   */
/* substitution of any % parameters in the standard printf     */
/* style.  After printing the message, error calls exit with   */
/* status 1.                                                   */
/*                                                             */
/*     If an error trap has been set by calling                */
/*                                                             */
/*                 trap("error")                               */
/*                                                             */
/* the message is copied into the global errmsg buffer and     */
/* a raise("error") is simulated.                              */
/***************************************************************/

/* VARARGS1 */
/* void error(msg, arg0) */
void error(va_alist)
va_dcl
{
    string msg;
    trapptr tp;
    int res;
    va_list args;

    va_start(args);
    msg = va_arg(args, string);
    preprocess(msg);
    res = (int) sprintf(errbuf, errfmt, args);
    if (res < 0) abort();
    va_end(args);

    if (strlen(errbuf) >= MAXERRMSG) {
      fprintf(stderr, "Error [BUFFER OVERRUN!]: %s\n", errbuf);
      exit(1);
    }

    errmsg = (string) errbuf;

    tp = trapchain;
    while (tp != NULL && strcmp(tp->name, "error") != 0) tp = tp->link;
    if (tp != NULL) {
#if defined(__FreeBSD__)
	flushtrap((tp->data[0]._jb[0]) - 1);
#else
	flushtrap((tp->data[0]) - 1);
#endif
	longjmp(tp->data, TRUE);
    }
    fprintf(stderr, "Error: %s\n", errbuf);
    exit(1);
}



/***************************************************************/
/* preprocess(msg)                                             */
/*                                                             */
/*     Copies msg into errfmt, replacing any legal occurrences */
/* of %M with the current error message.                       */
/***************************************************************/

static void preprocess(msg)
string msg;
{
    register char *src, *dst, *cp;
    char c;

    src = msg;
    dst = errfmt;
    while ((c = *src++) && dst < &errfmt[MAXERRMSG-4]) {
	switch (c) {
	    case '%':
		if ((c = *src++) == 'M') {
		    if (errno >= sys_nerr)
			cp = "Unknown error";
		    else
			cp = sys_errlist[errno];
		    while ((c = *cp++) && dst < &errfmt[MAXERRMSG-4])
			 *dst++ = c;
		} else {
		    *dst++ = '%';
		    *dst++ = c;
		}
		break;
	    case '\\':
		*dst++ = c;
		*dst++ = *src++;
		break;
	    default:
		*dst++ = c;
	}
    }
    if (dst == &errfmt[MAXERRMSG-4])
	strcpy(dst, "...");
    else
	*dst = '\0';
}



/***************************************************************/
/* flushtrap(oldfp);                                           */
/*                                                             */
/*     Does the work common to untrap and raise and clears all */
/* exception entries at or below oldfp.                        */
/***************************************************************/

static void flushtrap(oldfp)
int oldfp;
{
    trapptr oldptr;

#if defined(__FreeBSD__)
    while ((trapchain != NULL) && ((trapchain->data[0]._jb[0]) <= oldfp)) {
#else
    while ((trapchain != NULL) && ((trapchain->data[0]) <= oldfp)) {
#endif
        oldptr = trapchain;
	trapchain = trapchain->link;
	free((char *) oldptr);
    }
}



/***************************************************************/
/* copyjb(dst, src);                                           */
/*                                                             */
/*     Copies a jmp_buf from src to dst.                       */
/***************************************************************/

static void copyjb(dst, src)
register int *dst, *src;
{
    int count;

    count = sizeof (jmp_buf) / sizeof (int);
    while (count--) *dst++ = *src++;
}
