(* Copyright (C) 1993, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)
(*                                                             *)
(* Last modified on Wed Aug 24 16:15:15 PDT 1994 by isard      *)
(*      modified on Fri Nov 19 09:30:31 PST 1993 by kalsow     *)
(*      modified on Mon Apr 13 09:55:12 PDT 1992 by muller     *)

MODULE ObjFilex86;

IMPORT Text, M3CG, M3ID, M3x86, Wrx86;

FROM M3CG IMPORT Name, ByteSize, Alignment, Type, TypeUID, ByteOffset;

FROM M3x86 IMPORT x86Var, x86Proc, VLoc;

REVEAL T = Public BRANDED "ObjFilex86" OBJECT
        curs          : ARRAY Segment OF INTEGER;
        in_segment    := M3ID.NoID;
        n_params      := 0;
        param_offset  : ByteOffset;
        symno         := 0;
        symtop        := 0;
        sym           : REF ARRAY OF Symbol;
        init_var      : x86Var := NIL;
        init_size     := 0;
        init_sym      := 0;
        relochead     : ARRAY Segment OF ReLocList;
        reloctail     : ARRAY Segment OF ReLocList;
      OVERRIDES
        put := put;
        ranput := ranput;
        get := get;
        cursor := cursor;
        reloc := reloc;
        import_global := import_global;
        declare_segment := declare_segment;
        bind_segment := bind_segment;
        declare_global := declare_global;
        declare_param := declare_param;
        import_procedure := import_procedure;
        declare_procedure := declare_procedure;
        begin_init := begin_init;
        end_init := end_init;
        init_int := init_int;
        init_relocaddr := init_relocaddr;
        init_chars := init_chars;
      END;

TYPE Symbol = RECORD
  name: Name;
  import: BOOLEAN;
  proc := FALSE;
  offset: ByteOffset;
END;

TYPE ReLocList = REF RECORD
  src, dest: ReLoc;
  forward, backward: ReLocList;
END;

PROCEDURE put (t: T; seg: Segment; val, size: INTEGER) =
  BEGIN
    INC(t.curs[seg], size);
    size := val;
  END put;

PROCEDURE ranput (<*UNUSED*> t: T; <*UNUSED*> seg: Segment;
                  <*UNUSED*> offset, val, size: INTEGER) =
  BEGIN
  END ranput;

PROCEDURE get (<*UNUSED*> t: T; <*UNUSED*> seg: Segment;
               <*UNUSED*> offset, size: INTEGER): INTEGER =
  BEGIN
    RETURN 0;
  END get;

PROCEDURE cursor (t: T; seg: Segment): INTEGER =
  BEGIN
    RETURN t.curs[seg];
  END cursor;

PROCEDURE reloc (t: T; src, dest: ReLoc) =
  VAR item := t.reloctail[src.seg];
      newitem: ReLocList;
  BEGIN
    WHILE item # NIL AND src.sym <= item.src.sym AND
          (src.sym # item.src.sym OR src.o < item.src.o) DO
      item := item.backward;
    END;

    IF item = NIL THEN
      t.relochead[src.seg] := NEW(ReLocList, src := src, dest := dest,
                                  forward := t.relochead[src.seg],
                                  backward := NIL);
      newitem := t.relochead[src.seg];
    ELSE
      item.forward := NEW(ReLocList, src := src, dest := dest,
                          forward := item.forward,
                          backward := item);
      newitem := item.forward;
    END;

    IF newitem.forward = NIL THEN
      t.reloctail[src.seg] := newitem;
    ELSE
      newitem.forward.backward := newitem;
    END
  END reloc;

PROCEDURE import_global (t: T; n: Name; <*UNUSED*> s: ByteSize;
                         <*UNUSED*> a: Alignment; <*UNUSED*> type: Type;
                         <*UNUSED*> m3t: TypeUID): ByteOffset =
  BEGIN
    IF t.symno = t.symtop THEN
      expandsym(t);
    END;

    t.sym[t.symno].name := n;
    t.sym[t.symno].import := TRUE;

    INC(t.symno);
    RETURN t.symno - 1;
  END import_global;

PROCEDURE expandsym (t: T) =
  VAR newarray := NEW(REF ARRAY OF Symbol, t.symtop * 2);
  BEGIN
    FOR i := 0 TO t.symtop - 1 DO
      newarray[i] := t.sym[i];
    END;

    t.sym := newarray;

    t.symtop := t.symtop * 2;
  END expandsym;

PROCEDURE declare_segment (t: T; n: Name; <*UNUSED*> m3t: TypeUID):
            ByteOffset =
  BEGIN
    <* ASSERT t.in_segment = M3ID.NoID *>
    t.in_segment := n;

    IF t.symno = t.symtop THEN
      expandsym(t);
    END;

    t.sym[t.symno].name := n;
    t.sym[t.symno].import := FALSE;

    INC(t.symno);
    RETURN t.symno - 1;
  END declare_segment;

PROCEDURE bind_segment (t: T; o: ByteOffset; s: ByteSize; a: Alignment;
                        <*UNUSED*> type: Type;
                        <*UNUSED*> exported: BOOLEAN; inited: BOOLEAN) =
  BEGIN
    <* ASSERT o < t.symno AND t.sym[o].name = t.in_segment *>

    t.in_segment := M3ID.NoID;

    IF t.curs[Segment.data] MOD a # 0 THEN
      INC(t.curs[Segment.data], a - (t.curs[Segment.data] MOD a));
    END;

    t.sym[o].offset := t.curs[Segment.data];

    IF inited THEN
      <* ASSERT t.init_var = NIL *>
      t.init_size := s;
      t.init_sym := o;
    ELSE
      INC(t.curs[Segment.data], s);
    END
  END bind_segment;

PROCEDURE declare_global (t: T; n: Name; s: ByteSize; a: Alignment;
                          <*UNUSED*> type: Type; <*UNUSED*> m3t: TypeUID;
                          <*UNUSED*> exported: BOOLEAN; inited: BOOLEAN):
            ByteOffset =
  BEGIN
    IF t.symno = t.symtop THEN
      expandsym(t);
    END;

    t.sym[t.symno].name := n;
    t.sym[t.symno].import := FALSE;

    IF t.curs[Segment.data] MOD a # 0 THEN
      INC(t.curs[Segment.data], a - (t.curs[Segment.data] MOD a));
    END;

    t.sym[t.symno].offset := t.curs[Segment.data];

    IF inited THEN
      <* ASSERT t.init_var = NIL *>
      t.init_size := s;
      t.init_sym := t.symno;
    ELSE
      INC(t.curs[Segment.data], s);
    END;

    INC(t.symno);
    RETURN t.symno - 1;
  END declare_global;

PROCEDURE declare_param (t: T; v: x86Var; <*UNUSED*> n: Name;
                         s: ByteSize; a: Alignment;
                         <*UNUSED*> type: Type; <*UNUSED*> m3t: TypeUID):
            INTEGER =
  BEGIN
    <* ASSERT t.n_params > 0 *>

    <* ASSERT a <= 4 *>

    IF s MOD 4 # 0 THEN
      INC(s, 4 - (s MOD 4));
    END;

    DEC(t.param_offset, s);

    v.offset := t.param_offset;

    DEC(t.n_params);

    RETURN s;
  END declare_param;

PROCEDURE import_procedure (t: T; n: Name; <*UNUSED*> n_params: INTEGER;
                            <*UNUSED*> type: Type; <*UNUSED*> lang: TEXT):
            ByteOffset =
  BEGIN
    IF t.symno = t.symtop THEN
      expandsym(t);
    END;

    t.sym[t.symno].name := n;
    t.sym[t.symno].import := TRUE;
    t.sym[t.symno].proc := TRUE;

    INC(t.symno);
    RETURN t.symno - 1;
  END import_procedure;

PROCEDURE declare_procedure (t: T; n: Name; <*UNUSED*> n_params: INTEGER;
                             <*UNUSED*> type: Type;
                             <*UNUSED*> exported: BOOLEAN;
                             <*UNUSED*> lev: INTEGER;
                             <*UNUSED*> proc: x86Proc) =
  BEGIN
    IF t.symno = t.symtop THEN
      expandsym(t);
    END;

    t.sym[t.symno].name := n;
    t.sym[t.symno].import := FALSE;
    t.sym[t.symno].proc := TRUE;

    INC(t.symno);
  END declare_procedure;

PROCEDURE begin_init (t: T; v: x86Var) =
  BEGIN
    <* ASSERT t.init_var = NIL *>
    t.init_var := v;
  END begin_init;

PROCEDURE end_init (t: T; v: x86Var) =
  BEGIN
    <* ASSERT t.init_var = v *>

    pad_init(t, t.init_size);
    t.init_var := NIL;
  END end_init;

PROCEDURE init_int (t: T; o: ByteOffset; i, size: INTEGER) =
  BEGIN
    pad_init(t, o);

    <* ASSERT o + size <= t.init_size *>
    put(t, Segment.data, i, size);
  END init_int;

PROCEDURE init_relocaddr (t: T; o: ByteOffset; loc: VLoc;
                          valoff, bias: ByteOffset) =
  VAR seg := Segment.data;
  BEGIN
    pad_init(t, o);

    <* ASSERT o + 4 <= t.init_size *>
    put(t, Segment.data, bias, 4);

    <* ASSERT loc = VLoc.global OR loc = VLoc.import *>
    IF loc = VLoc.import THEN
      seg := Segment.global;
    END;

    reloc(t, ReLoc {Segment.data, t.init_sym, o},
             ReLoc {seg, valoff, bias} );
  END init_relocaddr;

PROCEDURE init_chars(t: T; o: ByteOffset; txt: TEXT) =
  BEGIN
    pad_init(t, o);

    <* ASSERT o + Text.Length(txt) <= t.init_size *>
    FOR i := 0 TO Text.Length(txt) - 1 DO
      put(t, Segment.data, ORD(Text.GetChar(txt, i)), 1);
    END
  END init_chars;

PROCEDURE pad_init (t: T; up_to: ByteOffset) =
  VAR length := up_to - t.curs[Segment.data];
  BEGIN
    IF length = 0 THEN
      RETURN;
    END;

    <* ASSERT length > 0 *>

    FOR i := 0 TO length DO
      put(t, Segment.data, 0, 1);
    END
  END pad_init;

PROCEDURE New (<*UNUSED*>wr: Wrx86.T): T =
  VAR obj := NEW(T);
  BEGIN
    FOR i := FIRST(Segment) TO LAST(Segment) DO
      obj.curs[i] := 0;
      obj.relochead[i] := NIL;
      obj.reloctail[i] := NIL;
    END;

    obj.symtop := 256;
    obj.sym := NEW(REF ARRAY OF Symbol, obj.symtop);

    RETURN obj;
  END New;

(*

(*---------------------------------------------- future update list stuff ---*)

TYPE OffsetList = OBJECT
  val: ByteOffset;
  tag: INTEGER;
  singleton := FALSE;
  ptr: OffsetList;
END;

PROCEDURE in_offset_list (t: T; l : INTEGER; toadd: BOOLEAN): OffsetList =
  VAR ptr := t.offlist_top;
  BEGIN
    WHILE ptr # NIL DO
      IF ptr.tag = l AND NOT (ptr.singleton AND toadd) THEN
        RETURN ptr;
      END;
      ptr := ptr.ptr;
    END;
    RETURN NIL;
  END in_offset_list;

<*UNUSED*> PROCEDURE del_offset_item (t: T; ptr: OffsetList; l: INTEGER) =
  VAR old := ptr;
  BEGIN
    WHILE old # NIL DO
      ptr := old.ptr;
      IF ptr.tag = l THEN
        old.ptr := ptr.ptr;
        RETURN;
      END;
      old := ptr;
    END;
    t.Err("Tried to delete non-existent offset list item");
  END del_offset_item;

PROCEDURE add_offset_item (t: T; l: INTEGER; val: ByteOffset;
                           insert: INTEGER) =
  BEGIN
    t.offlist_top := NEW(OffsetList, val := val, tag := l,
                         ptr := t.offlist_top);
    t.obj.put(Segment.text, val, insert, 4);
    IF insert > 16_FFFF THEN
      t.offlist_top.singleton := TRUE;
    END
  END add_offset_item;

PROCEDURE add_offset_to_thread(t: T; oldend, newend: ByteOffset;
                               insert: INTEGER): ByteOffset =
  VAR offset := newend - oldend;
  BEGIN
    <* ASSERT offset > 0 AND offset < 16_10000 AND
              insert > 0 AND insert < 16_10000 *>
    insert := Word.And(insert, Word.Shift(offset, 16));
    t.obj.put(Segment.text, newend, insert, 4);
    RETURN newend;
  END add_offset_to_thread;

PROCEDURE log_global_var (t: T; var: Operand; reltocurs: INTEGER) =
  VAR listptr: OffsetList;
  BEGIN
    IF var.mvar.o > 16_FFFF THEN
      add_offset_item(t, var.mvar.var.tag, place, var.mvar.o);
      RETURN;
    END;
    listptr := in_offset_list(t, var.mvar.var.tag, TRUE);
    IF listptr = NIL OR (place - listptr.val) > 16_FFFF THEN
      add_offset_item(t, var.mvar.var.tag, place, var.mvar.o);
    ELSE
      listptr.val := add_offset_to_thread(t, listptr.val, place, var.mvar.o);
    END
  END log_global_var;

PROCEDURE log_case_jump (t: T; place: ByteOffset; nlabels: CARDINAL) =
  BEGIN
    add_offset_item(t, -1, place, Word.Shift(nlabels, 16));
  END log_case_jump;
*)


BEGIN
END ObjFilex86.
