(* Copyright (C) 1993, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)
(*                                                             *)
(* Last modified on Mon Aug  8 13:16:20 PDT 1994 by isard      *)
(*      modified on Mon Jun 27 17:05:36 PDT 1994 by kalsow     *)
(*      modified on Tue May 25 14:22:35 PDT 1993 by muller     *)

MODULE Regx86;

IMPORT M3CG;

FROM M3CG IMPORT MType;
FROM M3CG_Ops IMPORT ErrorHandler;

IMPORT M3x86, Codex86, Stackx86;

FROM M3x86 IMPORT Operand, MVar, Regno, NRegs, Force, RegSet;
FROM M3x86 IMPORT OLoc;

REVEAL T = Public BRANDED "Regx86.T" OBJECT
        cg : Codex86.T := NIL;
        vstack : Stackx86.T := NIL;
        Err : ErrorHandler := NIL;
        reguse : ARRAY [1 .. NRegs] OF Register;
      OVERRIDES
        unlock := unlock;
        clearall := clearall;
        rdiscard := rdiscard;
        find := find;
        freereg := freereg;
        set_error_handler := set_error_handler;
      END;

CONST HighPrec: INTEGER = NRegs * 1000;

CONST NoStore = MVar {var := NIL, t := FIRST(MType)};

PROCEDURE unlock(t: T) =
  BEGIN
    FOR i := 1 TO NRegs DO
      t.reguse[i].locked := FALSE;
    END
  END unlock;

PROCEDURE clear(t: T; r: Regno) =
  BEGIN
    <* ASSERT t.reguse[r].stackp = -1 *>
    t.reguse[r].last_store := NoStore;
    t.reguse[r].locked := FALSE;
  END clear;

PROCEDURE storereg (t: T; READONLY op: Operand; r: Regno) =
  VAR type: MType;
  BEGIN
    <* ASSERT op.loc = OLoc.mem *>
    t.vstack.sweep(op);
    t.cg.movOp(op, t.cg.reg[r], type);
    t.vstack.set_mvar(t.reguse[r].stackp, op.mvar, type);
    t.reguse[r].stackp := -1;
    FOR i := 1 TO NRegs DO
      IF i # r AND M3x86.SameVar(t.reguse[i].last_store, op.mvar) THEN
        t.reguse[r].last_store := NoStore;
      END
    END;
    t.reguse[r].last_store := op.mvar;
  END storereg;

PROCEDURE loadreg (t: T; r: Regno; op: Operand) =
  VAR type: MType;
  BEGIN
    t.cg.movOp(t.cg.reg[r], op, type);
    t.reguse[r].stackp := op.stackp;
    IF op.loc = OLoc.mem THEN
      IF op.mvar.var.stack_temp THEN
        t.vstack.free_temp(op.mvar);
        t.reguse[r].last_store := NoStore;
      ELSE
        t.reguse[r].last_store := op.mvar;
      END
    ELSE
      t.reguse[r].last_store := NoStore;
    END;
    t.vstack.set_reg(op.stackp, r, type);
  END loadreg;

PROCEDURE loadphantom (t: T; r: Regno; stackp: INTEGER) =
  BEGIN
    t.reguse[r].stackp := stackp;
    t.vstack.set_reg(stackp, r, t.vstack.get_type(stackp));
  END loadphantom; 

PROCEDURE copyreg(t: T; stackp: INTEGER; to, from: Regno) =
  VAR dummytype: MType;
  BEGIN
    t.reguse[to].stackp := stackp;
    t.reguse[to].last_store := t.reguse[from].last_store;
    t.vstack.set_reg(stackp, to, t.vstack.get_type(t.reguse[from].stackp));
    t.cg.movOp(t.cg.reg[to], t.cg.reg[from], dummytype);
  END copyreg;

PROCEDURE movereg(t: T; to, from: Regno) =
  VAR dummytype: MType;
  BEGIN
    t.reguse[to].stackp := t.reguse[from].stackp;
    t.reguse[from].stackp := -1;
    t.reguse[to].last_store := t.reguse[from].last_store;
    t.vstack.set_reg(t.reguse[to].stackp, to,
                     t.vstack.get_type(t.reguse[from].stackp));
    t.cg.movOp(t.cg.reg[to], t.cg.reg[from], dummytype);
  END movereg;

PROCEDURE swapreg(t: T; to, from: Regno) =
  VAR tempstack := t.reguse[from].stackp;
      tempstore := t.reguse[from].last_store;
      temptype  := t.vstack.get_type(tempstack);
  BEGIN
    t.reguse[from].stackp := t.reguse[to].stackp;
    t.reguse[to].stackp := tempstack;
    t.reguse[from].last_store := t.reguse[to].last_store;
    t.reguse[to].last_store := tempstore;
    t.vstack.set_reg(tempstack, to, temptype);
    t.vstack.set_reg(t.reguse[from].stackp, from,
                      t.vstack.get_type(t.reguse[from].stackp));
    t.cg.swapOp(t.cg.reg[to], t.cg.reg[from]);
  END swapreg;

PROCEDURE clearall (t: T) =
  BEGIN
    FOR i := 1 TO NRegs DO
      clear(t, i);
    END
  END clearall; 

PROCEDURE find (t: T; stackp: INTEGER;
                force: Force := Force.any; set := RegSet {}) =
  (* Find a suitable register to put a stack item in *)
  VAR in, to: Regno;
  BEGIN
    CASE t.vstack.loc(stackp) OF
      OLoc.fstack =>
        t.Err("Tried to put a float in an int register in 'find'");
    | OLoc.mem =>
        in := inreg(t, t.vstack.mvar(stackp), set);
    | OLoc.register =>
        in := t.vstack.reg(stackp);
    | OLoc.imm =>
        in := 0;
    END;

    (* If it isn't in a register yet, and it doesn't have to be, do nothing *)
    IF force = Force.any AND in = 0 THEN
      RETURN;
    END;

    (* If for any reason it isn't in the right register, find the best
       candidate for a register to put it in. *)
    IF (in = 0) OR (force = Force.regset AND (NOT in IN set)) OR
       t.reguse[in].locked OR
       (t.reguse[in].stackp # -1 AND t.reguse[in].stackp # stackp) THEN
      to := pickreg(t, set);

    (* Otherwise, it is in the right place, so leave it there *)
    ELSE
      loadphantom(t, in, stackp);
      t.reguse[in].locked := TRUE;
      RETURN;
    END;

    (* If it doesn't have to be in a register, and there are no
       unused registers, do nothing *)
    IF force = Force.any AND t.reguse[to].stackp # -1 THEN
      RETURN;
    END;

    (* Now we know that we want to put it into 'to' *)

    (* If 'to' is unused, this is easy *)
    IF t.reguse[to].stackp = -1 THEN
      IF in = 0 THEN
        loadreg(t, to, t.vstack.op(stackp));
      ELSE
        IF t.reguse[in].stackp = stackp THEN
          movereg(t, to, in);
        ELSE
          copyreg(t, stackp, to, in);
        END
      END;

    ELSE
    (* Otherwise, see if 'in' is used for something other than stackp. If not,
       swap the registers over. If so, force 'to' out. If there is a free
       register, 'to' will be moved into it, otherwise it will be stored to
       memory *)
      IF in = 0 OR
         (t.reguse[in].stackp # -1 AND t.reguse[in].stackp # stackp) THEN
        forceout(t, to);
        IF in = 0 THEN
          loadreg(t, to, t.vstack.op(stackp));
        ELSE
          copyreg(t, stackp, to, in);
        END
      ELSE
        swapreg(t, to, in);
      END
    END;

    t.reguse[to].locked := TRUE;
  END find;

PROCEDURE freereg (t: T; set := RegSet {}): Regno =
  VAR to: Regno;
  BEGIN
    to := pickreg(t, set);
    IF t.reguse[to].stackp # -1 THEN
      forceout(t, to);
    END;
    t.reguse[to].locked := TRUE;
    RETURN to;
  END freereg;

PROCEDURE forceout(t: T; r: Regno) =
  VAR dead: Regno;
  BEGIN
    dead := finddead(t);
    IF dead = 0 THEN
      t.vstack.get_temp(t.reguse[r].stackp, r);
    ELSE
      movereg(t, dead, r);
    END
  END forceout;

PROCEDURE finddead(t: T): Regno =
  VAR minprec := HighPrec;
      bestreg: Regno := 0;
  BEGIN
    FOR i := 1 TO NRegs DO
      IF (t.reguse[i].stackp = -1) THEN
        WITH prec = precedence(t, i) DO
          IF prec < minprec THEN
            minprec := prec;
            bestreg := i;
          END
        END
      END
    END;
    RETURN bestreg;
  END finddead;

PROCEDURE pickreg (t: T; set: RegSet:= RegSet {}): Regno =
  VAR minprec := HighPrec;
      bestreg: Regno;
  BEGIN
    FOR i := 1 TO NRegs DO
      IF set = RegSet {} OR i IN set THEN
        WITH prec = precedence(t, i) DO
          IF prec < minprec THEN
            minprec := prec;
            bestreg := i;
          END
        END
      END
    END;
    <* ASSERT minprec # HighPrec *>
    RETURN bestreg;
  END pickreg;

PROCEDURE inreg(t: T; READONLY v: MVar; set: RegSet:= RegSet {}): Regno =
  VAR minprec := HighPrec * HighPrec;
      prec := 0;
      bestreg: Regno := 0;
  BEGIN
    FOR i := 1 TO NRegs DO
      IF t.reguse[i].last_store # NoStore AND
         M3x86.SameVar(v, t.reguse[i].last_store) THEN
        prec := precedence(t, i);
        IF (set # RegSet {}) AND (NOT i IN set) THEN
          prec := prec * HighPrec;
        END;
        IF prec < minprec THEN
          minprec := prec;
          bestreg := i;
        END
      END
    END;
    RETURN bestreg;
  END inreg;

PROCEDURE precedence(t: T; r: Regno): INTEGER =
  VAR prec: INTEGER;
  BEGIN
    IF r > 6 THEN
      RETURN HighPrec;
    END;
    IF t.reguse[r].locked THEN
      RETURN HighPrec;
    END;
    IF t.reguse[r].stackp = -1 THEN
      prec := NRegs;
    ELSIF t.reguse[r].last_store = NoStore THEN
      prec := 2 * NRegs;
    ELSE
      prec := 3 * NRegs;
    END;
    prec := prec + (NRegs - r);
    RETURN prec;
  END precedence;

PROCEDURE rdiscard (t: T; depth: INTEGER) =
  BEGIN
    FOR i := 0 TO depth-1 DO
      WITH stack = t.vstack.op(t.vstack.pos(i, "rdiscard")) DO
        IF stack.loc = OLoc.register THEN
          t.reguse[stack.reg].stackp := -1;
        END
      END
    END
  END rdiscard;

(*----------------------------------------------------------------- misc. ---*)

PROCEDURE set_error_handler (t: T; err: ErrorHandler) =
  BEGIN
    t.Err := err;
  END set_error_handler;

(*---------------------------------------------------------------------------*)

PROCEDURE New (cg: Codex86.T; vstack: Stackx86.T): T =
  BEGIN
    RETURN NEW (T,
                vstack := vstack,
                cg := cg);
  END New;

BEGIN
END Regx86.
