(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Thu Dec  8 16:27:11 PST 1994 by kalsow                   *)

MODULE M3MarkUp;

IMPORT Buf, Text, M3Scanner, M3Token;

CONST
  End_Anchor      = "</A>";
  Start_Exporters = "<A HREF=\"/4";
  Start_Interface = "<A HREF=\"/3";
  End_I3          = ".i3\">";
  End_IG          = ".ig\">";
  End_MG          = ".mg\">";
  
TYPE
  TK = M3Token.T;

TYPE
  State = { Idle, GrabUnit, GrabExports, GrabGeneric, GrabImports,
            GrabFromUnit, SkipFromImports, SkipRename
          };

PROCEDURE Get (buf: Buf.T): Insertion =
  VAR n_exports := 0;
  VAR state := State.GrabUnit;
  VAR is_interface := TRUE;
  VAR is_generic := FALSE;
  VAR id : TEXT;
  VAR id_offs: INTEGER;
  VAR unit := "";
  VAR unit_offs: INTEGER;
  VAR head := NEW (Insertion, next := NIL);
  VAR ins := head;
  VAR lex := NEW (M3Scanner.Default).initFromBuf (buf, skip_comments := TRUE,
                                                  split_pragmas := FALSE);
  BEGIN
    (* build a list of insertions *)
    LOOP
      CASE lex.token OF
      | TK.Module    => is_interface := FALSE;
      | TK.Interface => is_interface := TRUE;
      | TK.Generic   => is_generic := TRUE;
      | TK.Ident =>
          CASE state OF
          | State.Idle =>
              (* skip it *)
          | State.GrabUnit =>
              GetID (lex, unit, unit_offs);
              IF is_interface AND NOT is_generic THEN
                Add (ins, unit_offs, Start_Exporters);
                Add (ins, unit_offs, unit);
                Add (ins, unit_offs, End_I3);
                Add (ins, unit_offs + lex.length, End_Anchor);
              END;
              IF is_generic
                THEN state := State.SkipFromImports;
                ELSE state := State.GrabImports;
              END;
          | State.GrabExports =>
              GetID (lex, id, id_offs);
              Add (ins, id_offs, Start_Interface);
              Add (ins, id_offs, id);
              Add (ins, id_offs, End_I3);
              Add (ins, id_offs + lex.length, End_Anchor);
              INC (n_exports);
          | State.GrabGeneric =>
              GetID (lex, id, id_offs);
              Add (ins, id_offs, Start_Interface);
              Add (ins, id_offs, id);
              IF is_interface
                THEN Add (ins, id_offs, End_IG);
                ELSE Add (ins, id_offs, End_MG);
              END;
              Add (ins, id_offs + lex.length, End_Anchor);
              state := State.GrabImports;
          | State.GrabImports =>
              GetID (lex, id, id_offs);
              Add (ins, id_offs, Start_Interface);
              Add (ins, id_offs, id);
              Add (ins, id_offs, End_I3);
              Add (ins, id_offs + lex.length, End_Anchor);
          | State.GrabFromUnit =>
              GetID (lex, id, id_offs);
              Add (ins, id_offs, Start_Interface);
              Add (ins, id_offs, id);
              Add (ins, id_offs, End_I3);
              Add (ins, id_offs + lex.length, End_Anchor);
          | State.SkipRename =>
              (* skip this one *)
              state := State.GrabImports;
          | State.SkipFromImports =>
              (* skip this one *)
          END;
      | TK.Exports =>
          state := State.GrabExports;
      | TK.Semi =>
          IF (state = State.GrabExports)
            OR (state = State.GrabUnit)
            OR (state = State.GrabFromUnit)
            OR (state = State.SkipRename)
            OR (state = State.SkipFromImports) THEN
            state := State.GrabImports;
          ELSIF (n_exports = 0) AND (state = State.GrabImports)
            AND NOT is_generic AND NOT is_interface THEN
            Add (ins, unit_offs, Start_Interface);
            Add (ins, unit_offs, unit);
            Add (ins, unit_offs, End_I3);
            Add (ins, unit_offs + Text.Length (unit), End_Anchor);
            INC (n_exports);
          ELSIF (state = State.SkipRename) THEN
            state := State.GrabImports;
          END;
      | TK.Equal =>
          IF (state = State.GrabExports) OR (state = State.GrabUnit) THEN
            state := State.GrabGeneric;
          ELSIF (state = State.GrabImports) THEN
            state := State.GrabGeneric;
            IF (n_exports = 0) AND NOT is_generic AND NOT is_interface THEN
              Add (ins, unit_offs, Start_Interface);
              Add (ins, unit_offs, unit);
              Add (ins, unit_offs, End_I3);
              Add (ins, unit_offs + Text.Length (unit), End_Anchor);
              INC (n_exports);
            END;
          END;
      | TK.From =>
          state := State.GrabFromUnit;
      | TK.Import =>
          IF (state = State.GrabFromUnit) THEN
            state := State.SkipFromImports;
          ELSE
            state := State.GrabImports;
          END;
      | TK.As =>
          state := State.SkipRename;
      | TK.Comma =>
          IF (state = State.SkipRename) THEN
            state := State.GrabImports;
          END;
      | TK.Error, TK.EOF, TK.Const, TK.Type, TK.Exception, TK.Var,
        TK.Procedure, TK.Reveal, TK.End, TK.Raises, TK.Value,
        TK.Readonly, TK.Begin, TK.Case, TK.Exit, TK.Eval, TK.For,
        TK.If, TK.Lock, TK.Loop, TK.Raise, TK.Repeat, TK.Until,
        TK.Return, TK.Typecase, TK.Try, TK.Finally, TK.Except,
        TK.While, TK.Do, TK.With => EXIT;
      ELSE (* skip it *)
      END;
      lex.next ();
    END;

    RETURN head.next;
  END Get;

PROCEDURE GetID (lex: M3Scanner.T;  VAR id: TEXT;  VAR offset: INTEGER) =
  BEGIN
    offset := lex.offset;
    id := Text.FromChars (SUBARRAY (lex.buffer^, lex.offset, lex.length));
  END GetID;

PROCEDURE Add (VAR x: Insertion;  offs: INTEGER;  txt: TEXT) =
  BEGIN
    x.next := NEW (Insertion, next := NIL, offset := offs, insert := txt);
    x := x.next;
  END Add;

BEGIN
END M3MarkUp.
