(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Mon Mar 28 07:49:25 PST 1994 by kalsow                   *)


MODULE TCPServer;

IMPORT Text, Thread, Time, TCP, IP, ConnFD, Rd, Wr;

REVEAL
  T = Thread.Closure BRANDED OBJECT
    n_workers : CARDINAL              := 0;
    workers   : REF ARRAY OF Thread.T := NIL;
    port      : TCP.Connector         := NIL;
    handler   : RequestHandler        := NIL;
    refresher : Refresher             := NIL;
    timeout   : Time.T                := 0.0d0;
    err_log   : ErrorLogger           := NIL;
  OVERRIDES
    apply := Server;
  END;

TYPE
  TT = Thread.Closure OBJECT
    self: T := NIL;
  OVERRIDES
    apply := Refresh;
  END;

(*---------------------------------------------------- external interface ---*)

PROCEDURE Fork (socket    : CARDINAL;
                n_threads : CARDINAL;
                handler   : RequestHandler;
                refresher : Refresher;
                refresh_interval: Time.T;
                err_log   : ErrorLogger): T =
  VAR t := NEW (T);
  BEGIN
    IF (err_log = NIL) THEN err_log := DumpErr; END;
    t.n_workers := 0;
    t.workers   := NEW (REF ARRAY OF Thread.T, n_threads+1);
    t.handler   := handler;
    t.refresher := refresher;
    t.timeout   := refresh_interval;
    t.err_log   := err_log;

    (* open a TCP connection *)
    TRY
      t.port := TCP.NewConnector (IP.Endpoint {IP.GetHostAddr (), socket});
    EXCEPT IP.Error =>
      err_log ("cannot open TCP connection");
      RETURN NIL;
    END;

    (* fire up the refresh thread *)
    IF (refresher # NIL) AND (refresh_interval > 0.0d0)
      THEN t.workers[0] := Thread.Fork (NEW (TT, self := t));
      ELSE t.workers[0] := NIL;
    END;

    (* fire up the server threads *)
    FOR i := 1 TO n_threads DO  t.workers[i] := Thread.Fork (t);  END;

    RETURN t;
  END Fork;

PROCEDURE Join (t: T) =
  VAR z: Thread.T;
  BEGIN
    IF (t = NIL) THEN RETURN END;
    FOR i := 0 TO LAST (t.workers^) DO
      z := t.workers [i];
      IF (z # NIL) THEN
        EVAL Thread.Join (z);
        t.workers[i] := NIL;
      END;
    END;
    IF (t.port # NIL) THEN
      TCP.CloseConnector (t.port);
      t.port := NIL;
    END;
  END Join;

PROCEDURE Abort (t: T) =
  BEGIN
    Alert (t);
    Join (t);
  END Abort;

(*------------------------------------------------- request server thread ---*)

PROCEDURE Server (closure: Thread.Closure): REFANY =
  CONST Second = 1000.0d0;
  VAR
    self    : T := closure;
    channel : TCP.T;
    len, j, n : INTEGER;
    request : TEXT;
    buf     : ARRAY [0..2047] OF CHAR;
  BEGIN
    TRY
      LOOP
        TRY
          request := ""; (* give the collector a chance while we wait... *)
          channel := TCP.Accept (self.port);
          TRY

            (* read a new-line terminated request *)
            REPEAT
              len := channel.get (buf, 30.0d0 * Second);
              j := 0;  WHILE (j < len) AND (buf[j] # '\n') DO INC (j) END;
              request := request & Text.FromChars (SUBARRAY (buf, 0, j));
            UNTIL (j < len);

            (* process it *)
            request := self.handler (self, request);

            (* send the reply *)
            len := Text.Length (request);
            j := 0;
            WHILE (j < len) DO
              n := MIN (NUMBER (buf), len-j);
              FOR k := 0 TO n-1 DO  buf[k] := Text.GetChar (request, k+j); END;
              channel.put (SUBARRAY (buf, 0, n));
              INC (j, NUMBER (buf));
            END;

          FINALLY
            TCP.Close (channel);
          END;
        EXCEPT
        | ConnFD.TimedOut =>
            self.err_log ("client is non-responsive");
        | IP.Error, Rd.Failure, Wr.Failure =>
            (* bail out ... *)
            self.err_log ("IP error!");
            Alert (self);
            RETURN NIL;
        END;
      END;
    EXCEPT Thread.Alerted =>
      (* bail out... *)
      Alert (self);
    END;
    RETURN NIL;
  END Server;

(*----------------------------------------------- periodic refresh thread ---*)

PROCEDURE Refresh (closure: Thread.Closure): REFANY =
  VAR tt: TT := closure;  self := tt.self;
  BEGIN
    TRY
      LOOP
        Thread.Pause (self.timeout);
        self.refresher (self);
      END;
    EXCEPT Thread.Alerted =>
      (* bail out... *)
      Alert (self);
    END;
    RETURN NIL;
  END Refresh;

(*------------------------------------------------------------------ misc ---*)

PROCEDURE Alert (t: T) =
  VAR z: Thread.T;
  BEGIN
    IF (t = NIL) THEN RETURN END;
    FOR i := 0 TO LAST (t.workers^) DO
      z := t.workers[i];
      IF (z # NIL) THEN Thread.Alert (z); END;
    END;
  END Alert;

PROCEDURE DumpErr (<*UNUSED*> x: TEXT) =
  BEGIN
  END DumpErr;

BEGIN
END TCPServer.
