(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Thu May  6 14:11:08 PDT 1993 by kalsow                   *)


MODULE Main;

IMPORT Text, Rd, Wr, TextWr, TextRd, Thread, LongRealTime;
IMPORT RTMisc, Params, Lex, Convert, OS, List;
IMPORT TxtRefTbl, FileStream, MarkUp, TCPServer, ErrLog;

CONST
  Refresh_interval  = 5 * 60; (*seconds*)
  Default_n_workers = 3;
  My_socket_id      = 3828;

TYPE
  Time_T = LongRealTime.T;

TYPE
  IdList = REF RECORD
    name : TEXT   := NIL;
    next : IdList := NIL;
  END;

TYPE
  DirHandle = REF RECORD
    name       : TEXT;
    last_visit : Time_T;
    next       : DirHandle;
  END;

TYPE
  Unit = REF RECORD
    name  : TEXT           := NIL;
    time  : Time_T := OS.NO_TIME;
    dir   : DirHandle      := NIL;
    next  : Unit           := NIL;
  END;

TYPE
  Lib = REF RECORD
    name  : TEXT           := NIL;
    time  : Time_T := OS.NO_TIME;
    dir   : DirHandle      := NIL;
    next  : Lib            := NIL;
    intfs : IdList         := NIL;
    impls : IdList         := NIL;
  END;

VAR
  server    : TCPServer.T := NIL;
  n_workers : CARDINAL  := Default_n_workers;
  dirs      : DirHandle := NIL;
  test_arg  : TEXT := NIL;
  intfs     := TxtRefTbl.New ();  (* intf name -> intf *)
  impls     := TxtRefTbl.New ();  (* impl name -> impl *)
  libs      := TxtRefTbl.New ();  (* lib name  -> lib  *)
  exporters := TxtRefTbl.New ();  (* intf name -> set of impl ids *)

(*-------------------------------------------------- command line parsing ---*)

PROCEDURE ParseOptions () =
  VAR i := 1;  parm: TEXT;  x: INTEGER;
  BEGIN
    WHILE (i < Params.Count) DO
      parm := Params.Get (i); INC (i);
      IF Text.Equal (parm, "-test") THEN
        test_arg := Params.Get (i);  INC (i);
      ELSIF Text.Equal (parm, "-workers") THEN
        parm := Params.Get (i);  INC (i);
        IF GetInt (parm, x) AND (x > 0)
          THEN n_workers := x;
          ELSE ErrLog.Msg ("bad number of workers specified: ", parm); Abort();
        END;
      ELSIF NOT AddDir (parm) THEN
        ErrLog.Msg ("not a directory: ", parm); Abort ();
      END;
    END;
  END ParseOptions;

PROCEDURE GetInt (txt: TEXT;  VAR x: INTEGER): BOOLEAN =
  BEGIN
    TRY
      x := Lex.Int (TextRd.New (txt));
      RETURN TRUE;
    EXCEPT Rd.Failure, Convert.Failed, Lex.Error =>
      RETURN FALSE;
    END;
  END GetInt;

PROCEDURE AddDir (name: TEXT): BOOLEAN =
  VAR dir := OS.OpenDir (name);
  BEGIN
    IF (dir = NIL) THEN RETURN FALSE END;
    OS.CloseDir (dir);
    dirs := NEW (DirHandle, name := name,
                 last_visit := OS.NO_TIME, next := dirs);
    RETURN TRUE;
  END AddDir;

(*----------------------------------------------- periodic refresh thread ---*)

PROCEDURE Refresh (<*UNUSED*> service: TCPServer.T) RAISES {Thread.Alerted} =
  VAR d := dirs;  now := LongRealTime.Now ();  create: Time_T;
  BEGIN
    WHILE (d # NIL) DO
      create := OS.CreateTime (d.name);
      IF (d.last_visit = OS.NO_TIME) OR (d.last_visit < create) THEN
        UpdateDir (d, now);
      END;
      d := d.next;
    END;
  END Refresh;

PROCEDURE UpdateDir (d: DirHandle;  now: Time_T)
  RAISES {Thread.Alerted} =
  VAR dir := OS.OpenDir (d.name);  file: TEXT;
  BEGIN
    IF (dir = NIL) THEN RETURN END;
    LOOP
      file := OS.ReadDir (dir);
      IF (file = NIL) THEN EXIT END;
      NoteFile (d, file, now);
    END;
    OS.CloseDir (dir);
    d.last_visit := now;
    ErrLog.Msg ("refreshed: ", d.name);
  END UpdateDir;

PROCEDURE NoteFile (d: DirHandle;  file: TEXT;  now: Time_T)
  RAISES {Thread.Alerted} =
  VAR len := Text.Length (file);  suffix: TEXT;
  BEGIN
    IF (len <= 3) THEN RETURN END;
    suffix := Text.Sub (file, len-3, 3);
    file   := Text.Sub (file, 0, len-3);
    IF    Text.Equal (suffix, ".i3") THEN  AddIntf (d, file, now);
    ELSIF Text.Equal (suffix, ".m3") THEN  AddImpl (d, file, now);
    ELSIF Text.Equal (suffix, ".ax") THEN  AddLib  (d, file, now);
    END;
  END NoteFile;

PROCEDURE AddIntf (d: DirHandle;  file: TEXT;  now: Time_T)
  RAISES {Thread.Alerted} =
  BEGIN
    AddUnit (d, file, now, intfs);
  END AddIntf;

PROCEDURE AddImpl (d: DirHandle;  file: TEXT;  now: Time_T)
  RAISES {Thread.Alerted} =
  BEGIN
    AddUnit (d, file, now, impls);
  END AddImpl;

PROCEDURE AddUnit (d: DirHandle;  file: TEXT;  now: Time_T; units: TxtRefTbl.T)
  RAISES {Thread.Alerted} =
  VAR ref: REFANY;  unit, first_unit: Unit := NIL;
  BEGIN
    IF units.in (file, ref) THEN
      (* search for the one from this directory *)
      unit := ref;  first_unit := unit;
      WHILE (unit # NIL) AND (unit.dir # d) DO  unit := unit.next;  END;
    END;

    IF (unit = NIL) THEN
      (* we didn't find an interface for this directory *)
      unit := NEW (Unit, name := file, dir := d);
      IF (first_unit = NIL) THEN
        first_unit := unit;
        EVAL units.put (file, unit);
      ELSE
        unit.next := first_unit.next;
        first_unit.next := unit;
      END;
    END;

    (* update the time we last saw this entry *)
    unit.time := now;
  END AddUnit;

PROCEDURE AddLib (d: DirHandle;  file: TEXT;  now: Time_T)
  RAISES {Thread.Alerted} =
  VAR ref: REFANY;  lib, first_lib: Lib := NIL;
  BEGIN
    IF libs.in (file, ref) THEN
      (* search for the one from this directory *)
      lib := ref;  first_lib := lib;
      WHILE (lib # NIL) AND (lib.dir # d) DO  lib := lib.next;  END;
    END;

    IF (lib = NIL) THEN
      (* we didn't find an interface for this directory *)
      lib := NEW (Lib, name := file, dir := d);
      IF (first_lib = NIL) THEN
        first_lib := lib;
        EVAL libs.put (file, lib);
      ELSE
        lib.next := first_lib.next;
        first_lib.next := lib;
      END;
    END;

    (* update the time we last saw this entry *)
    lib.time := now;

    (* read the link-info file that tells us what's in the library *)
    AddLinkInfo (lib, d.name & "/" & file & ".ax");
  END AddLib;

PROCEDURE AddLinkInfo (lib: Lib;  file: TEXT)
  RAISES {Thread.Alerted} =
  VAR
    rd        : Rd.T;
    line      : TEXT;
    c         : CHAR;
    len       : INTEGER;
    unit      : TEXT   := NIL;
    is_intf   : BOOLEAN;
    name      : TEXT;
    new_intfs : IdList := NIL;
    new_impls : IdList := NIL;
  BEGIN
    TRY
      rd := FileStream.OpenRead (file);
      IF (rd = NIL) THEN RETURN END;
      WHILE NOT Rd.EOF (rd) DO
        line := Rd.GetLine (rd);
        len  := Text.Length (line);
        IF (len >= 2) THEN
          c := Text.GetChar (line, 0);
          IF (c = 'I') THEN
            (* start interface *)
            unit     := Text.Sub (line, 1, len-1);
            new_intfs := NEW (IdList, name := unit, next := new_intfs);
            is_intf  := TRUE;
          ELSIF (c = 'M') AND (Text.GetChar (line, 1) # '3') THEN
            (* start module *)
            unit     := Text.Sub (line, 1, len-1);
            new_impls := NEW (IdList, name := unit, next := new_impls);
            is_intf  := FALSE;
          ELSIF (c = 'A') THEN
            (* note export *)
            IF (unit # NIL) AND (NOT is_intf) THEN
              name := FirstWord (line, 1);
              NoteExporter (unit, name);
            END;
          END;
        END;
      END;
    EXCEPT
    | Rd.EndOfFile => (* shouldn't happen *)
    | Rd.Failure   => ErrLog.Msg ("error while reading: ", file);
    END;

    lib.intfs := new_intfs;
    lib.impls := new_impls;
  END AddLinkInfo;

PROCEDURE FirstWord (t: TEXT;  start: CARDINAL): TEXT =
  VAR len := Text.Length (t);  ch: CHAR;  stop: CARDINAL;
  BEGIN
    WHILE (start < len) DO
      ch := Text.GetChar (t, start);
      IF (ch # ' ') AND (ch # '\n') AND (ch # '\r') AND (ch # '\t') THEN
        EXIT;
      END;
      INC (start);
    END;
    stop := start+1;
    WHILE (stop < len) DO
      ch := Text.GetChar (t, stop);
      IF (ch = ' ') OR (ch = '\n') OR (ch = '\r') OR (ch = '\t') THEN
        EXIT;
      END;
      INC (stop);
    END;
    RETURN Text.Sub (t, start, stop - start);
  END FirstWord;

PROCEDURE NoteExporter (impl, intf: TEXT) =
  VAR ref: REFANY;  ids: IdList;
  BEGIN
    IF exporters.in (intf, ref) THEN
      ids := ref;
      WHILE (ids # NIL) AND NOT Text.Equal (ids.name, impl) DO
        ids := ids.next;
      END;
      IF (ids = NIL) THEN
        ids := NEW (IdList, name := impl, next := ref);
        EVAL exporters.put (intf, ids);
      END;
    ELSE
      ids := NEW (IdList, name := impl);
      EVAL exporters.put (intf, ids);
    END;
  END NoteExporter;

(*--------------------------------------------------- main request server ---*)

PROCEDURE ProcessRequest (<*UNUSED*>service: TCPServer.T;  req: TEXT): TEXT
  RAISES {Thread.Alerted} =
  VAR len := Text.Length (req);  cmd: CHAR;  wr := TextWr.New ();
  BEGIN
    IF (len < 6) THEN
      ErrLog.Msg ("request too short: ", req);
      RETURN "<H3>request too short: " & req & "</H3>\n";
    ELSIF NOT Text.Equal (Text.Sub (req, 0, 5), "GET /") THEN
      ErrLog.Msg ("unknown request: ", req);
      RETURN "<H3>unknown request: " & req & "</H3>\n";
    END;

    cmd := Text.GetChar (req, 5);
    req := Trim (req, 6);

    TRY
      CASE cmd OF
      | '0' =>  GenLibList (wr);
      | '1' =>  GenIntfList (wr);
      | '2' =>  GenImplList (wr);
      | '3' =>  GenIntf (req, wr);
      | '4' =>  GenImpl (req, wr);
      | '5' =>  GenLib  (req, wr);
      | '6' =>  GenOneIntf (req, wr);
      | '7' =>  GenOneImpl (req, wr);
      | '8' =>  GenOneLib  (req, wr);
      ELSE
        req := Text.FromChar (cmd) & req;
        ErrLog.Msg ("unknown request: GET /", req);
        RETURN "<H3>unknown request: GET /" & req & "</H3>\n";
      END;
    EXCEPT
    | Wr.Failure =>
        ErrLog.Msg ("internal text writer failure");
        RETURN  "<H3>** INTERNAL WRITER FAILURE **</H3>\n";
    | Thread.Alerted =>
        TRY Out (wr, "\n<H3>** INTERRUPTED **</H3>\n");
        EXCEPT Wr.Failure => (*skip*)
        END;
    END;

    RETURN TextWr.ToText (wr);
  END ProcessRequest;

PROCEDURE Trim (t: TEXT;  start: CARDINAL): TEXT =
  VAR len := Text.Length (t);  ch: CHAR;
  BEGIN
    WHILE (start < len) DO
      ch := Text.GetChar (t, start);
      IF (ch # ' ') AND (ch # '\n') AND (ch # '\r') AND (ch # '\t') THEN
        EXIT;
      END;
      INC (start);
    END;
    WHILE (start < len) DO
      ch := Text.GetChar (t, len-1);
      IF (ch # ' ') AND (ch # '\n') AND (ch # '\r') AND (ch # '\t') THEN
        EXIT;
      END;
      DEC (len);
    END;
    RETURN Text.Sub (t, start, len - start);
  END Trim;

PROCEDURE GenLibList (wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR all := libs.toValueList ();  lib: Lib;  n: CARDINAL;  l: List.T;
      elts: REF ARRAY OF TEXT;
  BEGIN
    (* first, count the names *)
    l := all;  n := 0;
    WHILE (l # NIL) DO
      lib := all.first;
      IF (lib.next = NIL) THEN INC (n) END;
      l := l.tail;
    END;

    (* then, build a list of them *)
    elts := NEW (REF ARRAY OF TEXT, n);
    l := all;  n := 0;
    WHILE (l # NIL) DO
      lib := l.first;
      IF (lib.next = NIL) THEN elts[n] := lib.name;  INC (n) END;
      l := l.tail;
    END;

    (* finally, generate the output *)
    Out (wr, "<TITLE>Public Modula-3 Libraries</TITLE>\n");
    Out (wr, "<H3>Public Libraries</H3>\n");
    GenDir ("5", elts^, wr);
  END GenLibList;

PROCEDURE GenIntfList (wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR elts := UnitVector (intfs.toValueList ());
  BEGIN
    Out (wr, "<TITLE>Public Modula-3 Interfaces</TITLE>\n");
    Out (wr, "<H3>Public Interfaces</H3>\n");
    GenDir ("3", elts^, wr);
  END GenIntfList;

PROCEDURE GenImplList (wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR elts := UnitVector (impls.toValueList ());
  BEGIN
    Out (wr, "<TITLE>Modula-3 Implementations</TITLE>\n");
    Out (wr, "<H3> Implementations </H3>\n");
    GenDir ("4", elts^, wr);
  END GenImplList;

PROCEDURE UnitVector (all: List.T): REF ARRAY OF TEXT =
  VAR unit: Unit;  n: CARDINAL;  l: List.T;  elts: REF ARRAY OF TEXT;
  BEGIN
    (* first, count the names *)
    l := all;  n := 0;
    WHILE (l # NIL) DO
      unit := all.first;
      IF (unit.next = NIL) THEN INC (n) END;
      l := l.tail;
    END;

    (* allocate space... *)
    elts := NEW (REF ARRAY OF TEXT, n);

    (* and finally, build a list of them *)
    l := all;  n := 0;
    WHILE (l # NIL) DO
      unit := l.first;
      IF (unit.next = NIL) THEN elts[n] := unit.name;  INC (n) END;
      l := l.tail;
    END;

    RETURN elts;
  END UnitVector;

PROCEDURE GenIntf (name: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR unit, last_u: Unit;  n_units: INTEGER;
  BEGIN
    Out (wr, "<TITLE>Modula-3 interface: ", name, "</TITLE>\n");
    Out (wr, "<H3>", name, ".i3:</H3>\n");

    unit := Find (intfs, name, wr);
    IF (unit = NIL) THEN RETURN END;

    (* check for multiple units *)
    ScanChoices (unit, NIL, n_units, last_u);

    IF (n_units = 0) THEN
      Out (wr, "<STRONG> *deleted* </STRONG>\n");
    ELSIF (n_units > 1) THEN
      GenChoices ("6", unit, wr);
    ELSE
      GenExporters (name, wr);
      GenUnit (last_u, ".i3", wr);
    END;
  END GenIntf;

PROCEDURE GenOneIntf (req: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR unit, last_u: Unit;  n_units: INTEGER;  name, dir: TEXT;
  BEGIN
    SplitReq (req, name, dir);
    Out (wr, "<TITLE>Modula-3 interface: ", name, "</TITLE>\n");
    Out (wr, "<H3>", name, ".i3 in ", dir, ":</H3>\n");

    unit := Find (intfs, name, wr);
    IF (unit = NIL) THEN RETURN END;

    (* check for multiple units *)
    ScanChoices (unit, dir, n_units, last_u);

    IF (n_units = 0) THEN
      Out (wr, "<STRONG> *deleted* </STRONG>\n");
    ELSE
      GenExporters (name, wr);
      GenUnit (last_u, ".i3", wr);
    END;

  END GenOneIntf;

PROCEDURE SplitReq (req: TEXT;  VAR name, dir: TEXT) =
  VAR len := Text.Length (req);  j: CARDINAL;
  BEGIN
    j := 0;
    WHILE (j < len) AND Text.GetChar (req, j) # '@' DO INC (j) END;
    IF (j < len) THEN
      name := Text.Sub (req, 0, j);
      dir  := Text.Sub (req, j+1, len - j - 1);
    ELSE
      name := req;
      dir  := "";
    END;
  END SplitReq;

PROCEDURE GenExporters (name: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR ref: REFANY;
  BEGIN
    IF exporters.in (name, ref) THEN
      Out (wr, "<H5>exported by:</H5>\n");
      GenIdList ("4", ref, wr);
    END;
  END GenExporters;

PROCEDURE GenImpl (name: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR unit, last_u: Unit;  n_units: INTEGER;
  BEGIN
    Out (wr, "<TITLE>Modula-3 implementation: ", name, "</TITLE>\n");
    Out (wr, "<H3>", name, ".m3:</H3>\n");

    unit := Find (impls, name, wr);
    IF (unit = NIL) THEN RETURN END;

    (* check for multiple units *)
    ScanChoices (unit, NIL, n_units, last_u);

    IF (n_units = 0) THEN
      Out (wr, "<STRONG> *deleted* </STRONG>\n");
    ELSIF (n_units > 1) THEN
      GenChoices ("7", unit, wr);
    ELSE
      GenUnit (last_u, ".m3", wr);
    END;
  END GenImpl;

PROCEDURE ScanChoices (u: Unit; dir: TEXT; VAR cnt: INTEGER; VAR last: Unit) =
  VAR n := 0;  last_u: Unit := NIL;
  BEGIN
    WHILE (u # NIL) DO
      IF (u.dir # NIL)
        AND (u.dir.last_visit <= u.time)
        AND ((dir = NIL) OR Text.Equal (dir, u.dir.name)) THEN
        (* this one is still current *)
        last_u := u;
        INC (n);
      END;
      u := u.next;
    END;
    cnt  := n;
    last := last_u;
  END ScanChoices;

PROCEDURE GenChoices (cmd: TEXT;  u: Unit;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    Out (wr, "<H4>which instance:</H4>\n");
    Out (wr, "<UL>\n");
    TRY
      WHILE (u # NIL) DO
        IF (u.dir # NIL) AND (u.dir.last_visit <= u.time) THEN
          (* this one is still current *)
          Out (wr, "<LI><A HREF=\"/", cmd, u.name, "@", u.dir.name);
          Out (wr, "\">", u.dir.name, "</A>\n");
        END;
        u := u.next;
      END;
    FINALLY
      Out (wr, "</UL>\n");
    END;
  END GenChoices;

PROCEDURE GenOneImpl (req: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR unit, last_u: Unit;  n_units: INTEGER;  name, dir: TEXT;
  BEGIN
    SplitReq (req, name, dir);
    Out (wr, "<TITLE>Modula-3 implementation: ", name, "</TITLE>\n");
    Out (wr, "<H3>", name, ".m3 in ", dir, ":</H3>\n");

    unit := Find (impls, name, wr);
    IF (unit = NIL) THEN RETURN END;

    (* check for multiple units *)
    ScanChoices (unit, dir, n_units, last_u);

    IF (n_units = 0)
      THEN Out (wr, "<STRONG> *deleted* </STRONG>\n");
      ELSE GenUnit (last_u, ".m3", wr);
    END;
  END GenOneImpl;

PROCEDURE GenUnit (u: Unit;  suffix: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR rd: Rd.T;  file := u.dir.name & "/" & u.name & suffix;
  BEGIN
    TRY rd := FileStream.OpenRead (file);
    EXCEPT Rd.Failure => rd := NIL;
    END;

    IF (rd = NIL) THEN
      Out (wr, "<STRONG> unable to open ", file, " </STRONG>");
      RETURN;
    END;

    TRY
      MarkUp.Annotate (rd, wr);
      Rd.Close (rd);
    EXCEPT
    | Rd.Failure =>
        ErrLog.Msg ("reader failure: ", file);
        Out (wr, "<STRONG> I/O failure reading ", file, " </STRONG>");
    END;
  END GenUnit;

PROCEDURE GenLib (name: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR lib, last_l: Lib;  n_libs: INTEGER;
  BEGIN
    Out (wr, "<TITLE>Modula-3 library: ", name, "</TITLE>\n");
    Out (wr, "<H3>", name, ".a:</H3>\n");

    lib := Find (libs, name, wr);
    IF (lib = NIL) THEN RETURN END;

    (* check for multiple libs *)
    ScanLibChoices (lib, NIL, n_libs, last_l);

    IF (n_libs = 0) THEN
      Out (wr, "<STRONG> *deleted* </STRONG>\n");
    ELSIF (n_libs > 1) THEN
      GenLibChoices ("8", lib, wr);
    ELSE
      GenLibContents (last_l, wr);
    END;
  END GenLib;

PROCEDURE GenOneLib (req: TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR lib, last_l: Lib;  n_libs: INTEGER;  name, dir: TEXT;
  BEGIN
    SplitReq (req, name, dir);
    Out (wr, "<TITLE>Modula-3 library: ", name, "</TITLE>\n");
    Out (wr, "<H3>", name, ".a in ", dir, ":</H3>\n");

    lib := Find (libs, name, wr);
    IF (lib = NIL) THEN RETURN END;

    (* check for multiple libs *)
    ScanLibChoices (lib, dir, n_libs, last_l);

    IF (n_libs = 0)
      THEN Out (wr, "<STRONG> *deleted* </STRONG>\n");
      ELSE GenLibContents (last_l, wr);
    END;
  END GenOneLib;

PROCEDURE ScanLibChoices (l: Lib; dir: TEXT; VAR cnt: INTEGER; VAR last: Lib) =
  VAR n := 0;  last_l: Lib := NIL;
  BEGIN
    WHILE (l # NIL) DO
      IF (l.dir # NIL)
        AND (l.dir.last_visit <= l.time)
        AND ((dir = NIL) OR Text.Equal (dir, l.dir.name)) THEN
        (* this one is still current *)
        last_l := l;
        INC (n);
      END;
      l := l.next;
    END;
    cnt  := n;
    last := last_l;
  END ScanLibChoices;

PROCEDURE GenLibChoices (cmd: TEXT;  l: Lib;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    Out (wr, "<H4>which instance:</H4>\n");
    Out (wr, "<UL>\n");
    TRY
      WHILE (l # NIL) DO
        IF (l.dir # NIL) AND (l.dir.last_visit <= l.time) THEN
          (* this one is still current *)
          Out (wr, "<LI><A HREF=\"/", cmd, l.name, "@", l.dir.name);
          Out (wr, "\">", l.dir.name, "</A>\n");
        END;
        l := l.next;
      END;
    FINALLY
      Out (wr, "</UL>\n");
    END;
  END GenLibChoices;

PROCEDURE GenLibContents (l: Lib;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    Out (wr, "<H4>interfaces:</H4>\n");
    GenIdList ("3", l.intfs, wr);
    Out (wr, "<H4>implementations:</H4>\n");
    GenIdList ("4", l.impls, wr);
  END GenLibContents;

PROCEDURE GenIdList (cmd: TEXT;  ids: IdList;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR x := ids;  n := 0;
  BEGIN
    WHILE (x # NIL) DO  INC (n);  x := x.next;  END;
    IF    (n <= 0) THEN (* skip *)
    ELSIF (n < 40) THEN GenShortIdList (cmd, ids, wr);
    ELSE                GenLongIdList (cmd, ids, wr, n);
    END;
  END GenIdList;

PROCEDURE GenShortIdList (cmd: TEXT;  ids: IdList;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR n := 0;  elts: ARRAY [0..39] OF TEXT;
  BEGIN
    WHILE (ids # NIL) DO  elts[n] := ids.name;  INC (n);  ids := ids.next; END;
    GenDir (cmd, SUBARRAY (elts, 0, n), wr);
  END GenShortIdList;

PROCEDURE GenLongIdList (cmd: TEXT;  ids: IdList;  wr: Wr.T;  n: INTEGER)
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR elts := NEW (REF ARRAY OF TEXT, n);
  BEGIN
    n := 0;
    WHILE (ids # NIL) DO  elts[n] := ids.name;  INC (n);  ids := ids.next; END;
    GenDir (cmd, elts^, wr);
  END GenLongIdList;

(*------------------------------------------------------ HTML directories ---*)

(* In principle an HTML front-end will do a good job rendering
   a list of names in <DIR></DIR> brackets.  In practice "xmosaic"
   doesn't.  The following code is intended to compensate. *)

PROCEDURE GenDir (cmd: TEXT;  VAR names: ARRAY OF TEXT;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure} =
  CONST Dir_width = 78; (* max # characters per line *)
  CONST Max_cols  = 6;  (* max # columns per line *)
  CONST Gap       = 2;  (* inter-column gap *)
  CONST Gap_text  = "  ";
  VAR max_len := 5;  n_cols := 1;  width, n_rows, j: CARDINAL;  nm: TEXT;
  BEGIN
    IF NUMBER (names) <= 0 THEN RETURN END;
    Sort (names);

    (* compute an approriate layout *)
    FOR i := FIRST (names) TO LAST (names) DO
      max_len := MAX (max_len, Text.Length (names[i]));
    END;
    INC (max_len, Gap);
    n_cols := MAX (1, MIN (Dir_width DIV max_len, Max_cols));
    n_rows := (NUMBER (names) + n_cols - 1) DIV n_cols;
    width  := Dir_width DIV n_cols - Gap;

    Out (wr, "<PRE>\n");
    TRY
      FOR row := 0 TO n_rows-1 DO
        FOR col := 0 TO n_cols-1 DO
          j := col * n_rows + row;
          IF (j <= LAST (names)) THEN
            nm := names [j];
            Out (wr, "<A HREF=\"/", cmd, nm, "\">", nm);
            IF (col # n_cols-1) THEN
              (* pad to the next column *)
              FOR x := 1 TO width - Text.Length (nm) DO Out (wr, " "); END;
            END;
            Out (wr, "</A>", Gap_text);
          END;
        END;
        Out (wr, "\n");
      END;
    FINALLY
      Out (wr, "</PRE>");
    END;
  END GenDir;

(*--------------------------------------------------------------- sorting ---*)

TYPE  Elem_T = TEXT;
CONST Elem_Compare = Text.Compare;

PROCEDURE Sort (VAR a: ARRAY OF Elem_T;  cmp := Elem_Compare) =
  BEGIN
    QuickSort (a, 0, NUMBER (a), cmp);
    InsertionSort (a, 0, NUMBER (a), cmp);
  END Sort;

PROCEDURE QuickSort (VAR a: ARRAY OF Elem_T;  lo, hi: INTEGER;
                     cmp := Elem_Compare) =
  CONST CutOff = 9;
  VAR i, j: INTEGER;  key, tmp: Elem_T;
  BEGIN
    WHILE (hi - lo > CutOff) DO (* sort a[lo..hi) *)

      (* use median-of-3 to select a key *)
      i := (hi + lo) DIV 2;
      IF cmp (a[lo], a[i]) < 0 THEN
        IF cmp (a[i], a[hi-1]) < 0 THEN
          key := a[i];
        ELSIF cmp (a[lo], a[hi-1]) < 0 THEN
          key := a[hi-1];  a[hi-1] := a[i];  a[i] := key;
        ELSE
          key := a[lo];  a[lo] := a[hi-1];  a[hi-1] := a[i];  a[i] := key;
        END;
      ELSE (* a[lo] >= a[i] *)
        IF cmp (a[hi-1], a[i]) < 0 THEN
          key := a[i];  tmp := a[hi-1];  a[hi-1] := a[lo];  a[lo] := tmp;
        ELSIF cmp (a[lo], a[hi-1]) < 0 THEN
          key := a[lo];  a[lo] := a[i];  a[i] := key;
        ELSE
          key := a[hi-1];  a[hi-1] := a[lo];  a[lo] := a[i];  a[i] := key;
        END;
      END;

      (* partition the array *)
      i := lo+1;  j := hi-2;

      (* find the first hole *)
      WHILE cmp (a[j], key) > 0 DO DEC (j) END;
      tmp := a[j];
      DEC (j);

      LOOP
        IF (i > j) THEN EXIT END;

        WHILE cmp (a[i], key) < 0 DO INC (i) END;
        IF (i > j) THEN EXIT END;
        a[j+1] := a[i];
        INC (i);

        WHILE cmp (a[j], key) > 0 DO DEC (j) END;
        IF (i > j) THEN  IF (j = i-1) THEN  DEC (j)  END;  EXIT  END;
        a[i-1] := a[j];
        DEC (j);
      END;

      (* fill in the last hole *)
      a[j+1] := tmp;
      i := j+2;

      (* then, recursively sort the smaller subfile *)
      IF (i - lo < hi - i)
        THEN  QuickSort (a, lo, i-1);   lo := i;
        ELSE  QuickSort (a, i, hi);     hi := i-1;
      END;

    END; (* WHILE (hi-lo > CutOff) *)
  END QuickSort;


PROCEDURE InsertionSort (VAR a: ARRAY OF Elem_T;  lo, hi: INTEGER;
                         cmp := Elem_Compare) =
  VAR j: INTEGER;  key: Elem_T;
  BEGIN
    FOR i := lo+1 TO hi-1 DO
      key := a[i];
      j := i-1;
      WHILE (j >= lo) AND cmp (key, a[j]) < 0 DO
        a[j+1] := a[j];
        DEC (j);
      END;
      a[j+1] := key;
    END;
  END InsertionSort;


(*------------------------------------------------------- low-level stuff ---*)

PROCEDURE Find (tbl: TxtRefTbl.T;  name: TEXT;  wr: Wr.T): REFANY
  RAISES {Thread.Alerted, Wr.Failure} =
  VAR ref: REFANY;
  BEGIN
    IF NOT tbl.in (name, ref) THEN
      Out (wr, "<STRONG> *unknown* </STRONG>\n");
      ref := NIL;
    END;
    RETURN ref;
  END Find;

PROCEDURE Out (wr: Wr.T;  a, b, c, d, e: TEXT := NIL)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    IF (a # NIL) THEN Wr.PutText (wr, a) END;
    IF (b # NIL) THEN Wr.PutText (wr, b) END;
    IF (c # NIL) THEN Wr.PutText (wr, c) END;
    IF (d # NIL) THEN Wr.PutText (wr, d) END;
    IF (e # NIL) THEN Wr.PutText (wr, e) END;
  END Out;

PROCEDURE Abort () =
  BEGIN
    TCPServer.Abort (server);
    RTMisc.Exit (-1);
  END Abort;

(*---------------------------------------------------------------------------*)

BEGIN
  ParseOptions ();
  TRY
    Refresh (NIL);
    IF (test_arg # NIL) THEN
      ErrLog.Note (ProcessRequest (NIL, test_arg));
    ELSE
      server := TCPServer.Fork (My_socket_id, n_workers, ProcessRequest,
                                Refresh, Refresh_interval, ErrLog.Note );
      TCPServer.Join (server);
    END;
  EXCEPT Thread.Alerted => (*die*)
  END;
END Main.
