(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Mon Aug 30 08:13:30 PDT 1993 by kalsow                   *)
(*      modified on Mon May  3 12:23:44 PDT 1993 by mcjones                  *)
(*      modified on Tue Feb  9 12:30:09 1993 by gnelson                      *)
(*      modified on Thu Jan  7 11:08:45 PST 1993 by muller                   *)

(* Implementation of the m3tohtml command; see its manpage for details. *)

MODULE MarkUp;

IMPORT Rd, Wr, Thread, M3MarkUp;

CONST
  EscapeSpecials = TRUE;
  EOF = '\000';

TYPE
  State = RECORD
    rd  : Rd.T := NIL;
    wr  : Wr.T := NIL;
    eof : BOOLEAN := FALSE;
    ins : M3MarkUp.Insertion := NIL;
    offset : INTEGER := 0;
  END;

PROCEDURE Annotate (rd: Rd.T;  wr: Wr.T)
  RAISES {Thread.Alerted, Wr.Failure, Rd.Failure} =
  VAR s: State;  here: CARDINAL;
  BEGIN
    s.rd := rd;
    s.wr := wr;
    here := Rd.Index (rd);
    s.ins := M3MarkUp.Get (rd);
    Rd.Seek (rd, here);
    Trans (s);
  END Annotate;

(*------------------------------------------------------- file processing ---*)

PROCEDURE Trans(VAR s: State)
  RAISES {Thread.Alerted, Wr.Failure, Rd.Failure} =
  VAR spec: BOOLEAN;
  BEGIN
    AdvanceToBlankLine (s);
    WHILE NOT s.eof DO
      spec := Prog (s);
      IF (spec) THEN OutT (s, "<MENU><LI><EM>"); END;
      Comment (s);
      IF (spec) THEN OutT (s, "</EM></MENU>"); END;
    END;
    OutT (s, "\n");
  END Trans;

PROCEDURE AdvanceToBlankLine (VAR s: State)
  RAISES {Thread.Alerted, Rd.Failure, Wr.Failure} =
  VAR blank: BOOLEAN; c: CHAR;
  BEGIN
    REPEAT
      blank := TRUE;
      LOOP
        c := GetC (s);
        IF s.eof THEN EXIT END;
        IF c = '\n' THEN EXIT END;
        IF c # ' ' THEN blank := FALSE END
      END
    UNTIL blank OR s.eof;
  END AdvanceToBlankLine;

PROCEDURE Prog (VAR s: State): BOOLEAN
  RAISES {Thread.Alerted, Wr.Failure, Rd.Failure} =
  VAR
    c, d: CHAR;
    vspace := 0;
    hspace := 0;
    empty := TRUE;
    startOfLine := TRUE;

  PROCEDURE Space () RAISES {Thread.Alerted, Wr.Failure} =
    BEGIN
      IF empty THEN (*! OutT (s, "\\par\\medskip ");  !*) END;
      empty := FALSE;
      startOfLine := FALSE;
      IF vspace = 1 THEN
        OutT (s, "\n");
      ELSIF vspace > 1 THEN
        OutT (s, "\n\n");
      END;
      vspace := 0;
      WHILE hspace > 0 DO OutT (s, " "); DEC (hspace); END;
    END Space;

  BEGIN
    OutT (s, "<PRE>");
    TRY
      WHILE NOT s.eof DO
        c := GetC (s);
        CASE c OF
        | '\n' => INC(vspace); hspace := 0; startOfLine := TRUE;
        | ' '  => INC(hspace);
        | '('  => d := GetC (s);
                  IF (d = '*') AND startOfLine AND (hspace = 0) THEN EXIT END;
                  UngetC (s);  Space ();  OutC (s, c);
        | EOF  => EXIT;
        ELSE      Space (); OutC (s, c);
        END;
      END;
      IF (vspace > 0) THEN OutT (s, "\n") END;
    FINALLY
      OutT (s, "</PRE>");
    END;
    RETURN (vspace < 2) AND (NOT empty) AND (NOT s.eof);
  END Prog;

PROCEDURE Comment (VAR s: State)
  RAISES {Thread.Alerted, Wr.Failure, Rd.Failure} =
  CONST CodeEdge = ARRAY BOOLEAN OF TEXT { "<CODE>", "</CODE>" };
  VAR c, d: CHAR; startOfLine := TRUE; afterDisplay := FALSE; in_code := FALSE;
  BEGIN
    WHILE (NOT s.eof) DO
      c := GetC (s);
      CASE c OF
      | '\"' =>
          OutT (s, CodeEdge [in_code]);
          in_code := NOT in_code;

      | '<', '>', '&'
            => OutX (s, c);

      | '|' =>
          IF startOfLine THEN 
            IF NOT afterDisplay THEN OutT (s, "<PRE>\n"); END;
            c := GetC (s);
            IF (c # ' ') THEN UngetC (s); END;
            Display (s);
            c := '\n';
            afterDisplay := TRUE;
          ELSE 
            OutT (s, "|");
          END;

      | '\n' => 
          IF afterDisplay THEN 
            OutT (s, "</PRE>\n");
            afterDisplay := FALSE;
          ELSIF startOfLine THEN
            OutT (s, "<P>\n");
          ELSE
            OutT (s, "\n");
          END;

      | '*' =>
          d := GetC (s);
          IF (d = ')') THEN RETURN; END;
          UngetC (s);
          IF afterDisplay THEN
            OutT (s, "</PRE>\n");
            afterDisplay := FALSE;
          END;
          OutC (s, c);

      ELSE
          IF afterDisplay AND c # ' ' THEN 
            OutT (s, "</PRE>\n");
            afterDisplay := FALSE;
          END;
          OutC (s, c);

      END; (*CASE*)

      startOfLine := (c = '\n') OR (startOfLine AND c = ' ')
    END; (*WHILE*)
  END Comment;

PROCEDURE Display (VAR s: State)
  RAISES {Thread.Alerted, Wr.Failure, Rd.Failure} =
  VAR c: CHAR; 
  BEGIN
    OutT (s, "      ");
    WHILE NOT s.eof DO
      c := GetC (s);
      CASE c OF
      | '<', '>', '&', '"' => OutC (s, c);
      |'\n' => OutC (s, '\n'); RETURN
      | ' ' => OutC (s, ' ');
      | '`' => Undisplay (s);
      ELSE OutC (s, c);
      END;
    END;
  END Display;

PROCEDURE Undisplay (VAR s: State)
  RAISES {Thread.Alerted, Wr.Failure, Rd.Failure} =
  CONST CodeEdge = ARRAY BOOLEAN OF TEXT { "<KBD>", "</KBD>" };
  VAR c: CHAR;  in_code := TRUE;
  BEGIN
    OutT (s, "<KBD>");
    WHILE NOT s.eof DO
      c := GetC (s);
      CASE c OF
      | '<', '>', '&' => OutC (s, c);
      | '\"'          => OutT (s, CodeEdge [in_code]); in_code := NOT in_code;
      | '`'           => OutT (s, "</KBD>"); RETURN;
      ELSE               OutC (s, c);
      END;
    END;
  END Undisplay;

(*--------------------------------------------------------- low-level I/O ---*)

PROCEDURE UngetC (VAR s: State) =
  BEGIN
    DEC (s.offset);
    Rd.UnGetChar (s.rd);
  END UngetC;

PROCEDURE GetC (VAR s: State): CHAR
  RAISES {Thread.Alerted, Rd.Failure, Wr.Failure} =
  <*FATAL Rd.EndOfFile*>
  BEGIN
    WHILE (s.ins # NIL) AND (s.ins.offset <= s.offset) DO
      OutT (s, s.ins.insert);
      s.ins := s.ins.next;
    END;
    IF (s.eof) OR Rd.EOF (s.rd) THEN
      s.eof := TRUE;
      RETURN EOF;
    ELSE
      INC (s.offset);
      RETURN Rd.GetChar (s.rd);
    END;
  END GetC;

PROCEDURE OutT (VAR s: State;  a, b, c: TEXT := NIL)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    IF (a # NIL) THEN Wr.PutText (s.wr, a); END;
    IF (b # NIL) THEN Wr.PutText (s.wr, b); END;
    IF (c # NIL) THEN Wr.PutText (s.wr, c); END;
  END OutT;

PROCEDURE OutC (VAR s: State;  ch: CHAR)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    Wr.PutChar (s.wr, ch);
  END OutC;

PROCEDURE OutX (VAR s: State;  ch: CHAR)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    IF NOT EscapeSpecials THEN OutC (s, ch);
    ELSIF (ch = '<')      THEN OutT (s, "&lt;");
    ELSIF (ch = '>')      THEN OutT (s, "&gt;");
    ELSIF (ch = '&')      THEN OutT (s, "&amp;");
    ELSIF (ch = '"')      THEN OutT (s, "&quot;");
    ELSE                       OutC (s, ch);
    END;
  END OutX;

BEGIN
END MarkUp.
