(* Copyright 1992 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Example.m3 *)
(* Last modified on Thu Feb 24 17:58:10 PST 1994 by wobber *)
(* This module is an example of a hand-coded stub. *)

MODULE Example;

IMPORT NetObj, StubLib, Thread, Rd, Wr;

TYPE
  P = { Get, Put };  R = { OK, Invalid };

(* The enumerated types "P" and "R" define values to be 
   associated with the methods of "T" and with the various results 
   (normal return or exception) of these methods. *)
 
TYPE
  StubT = T OBJECT
  OVERRIDES
    get := SurrogateGet;
    put := SurrogatePut;
  END;

(* The type "StubT" is the surrogate object type for "T".  It provides 
   method overrides that perform remote invocation. *)

CONST StubVersion = StubLib.SystemStubProtocol;

(* This constant will be set by the stub generator to denote the
   stub generator version that created a given stub. *)

PROCEDURE SurrogateGet (t: StubT; key: TEXT) : TEXT
    RAISES {Invalid, NetObj.Error, Thread.Alerted} =
  VAR reuse := FALSE;
      rep: StubLib.DataRep;
      c: StubLib.Conn;
      res: TEXT;
  BEGIN
    TRY
      c := StubLib.StartCall(t, StubVersion);
      TRY
        StubLib.OutInt32(c, ORD(P.Get));
        StubLib.OutRef(c, key);
        rep := StubLib.AwaitResult(c);
        CASE StubLib.InInt32(c, rep) OF
        | ORD(R.OK) =>
            res := StubLib.InRef(c, rep, TYPECODE(TEXT));
            reuse := TRUE;
        | ORD(R.Invalid) =>
            reuse := TRUE;
            RAISE Invalid;
        ELSE
            StubLib.RaiseUnmarshalFailure();
        END;
      FINALLY
        StubLib.EndCall(c, reuse);
      END;
    EXCEPT
    | Rd.Failure(ec) => StubLib.RaiseCommFailure(ec);
    | Wr.Failure(ec) => StubLib.RaiseCommFailure(ec);
    END;
    RETURN res;
  END SurrogateGet;

PROCEDURE SurrogatePut (t: StubT; key: TEXT; value: TEXT)
    RAISES {Invalid, NetObj.Error, Thread.Alerted} =
  VAR reuse := FALSE;
      rep: StubLib.DataRep;
      c: StubLib.Conn;
  BEGIN
    TRY
      c := StubLib.StartCall(t, StubVersion);
      TRY
        StubLib.OutInt32(c, ORD(P.Put));
        StubLib.OutRef(c, key);
        StubLib.OutRef(c, value);
        rep := StubLib.AwaitResult(c);
        CASE StubLib.InInt32(c, rep) OF
        | ORD(R.OK) =>
            reuse := TRUE;
        | ORD(R.Invalid) =>
            reuse := TRUE;
            RAISE Invalid;
        ELSE
            StubLib.RaiseUnmarshalFailure();
        END;
      FINALLY
        StubLib.EndCall(c, reuse);
      END;
    EXCEPT
    | Rd.Failure(ec) => StubLib.RaiseCommFailure(ec);
    | Wr.Failure(ec) => StubLib.RaiseCommFailure(ec);
    END;
  END SurrogatePut;

(* "Invoke" is the server stub dispatcher for "T".  It is called when 
    the network object runtime receives a method invocation for an 
    object of type "T". *)

PROCEDURE Invoke(
    c: StubLib.Conn;
    obj: NetObj.T;
    rep: StubLib.DataRep;
    <*UNUSED*> stubProt: StubLib.StubProtocol)
    RAISES {NetObj.Error, Rd.Failure,
            Wr.Failure, Thread.Alerted} =
  VAR t := NARROW(obj, T);
  BEGIN
    TRY
      CASE StubLib.InInt32(c, rep) OF
      | ORD(P.Get) => GetStub(c, t, rep);
      | ORD(P.Put) => PutStub(c, t, rep);
      ELSE StubLib.RaiseUnmarshalFailure();
      END;
    EXCEPT
    | Invalid =>
        StubLib.StartResult(c);
        StubLib.OutInt32(c, ORD(R.Invalid));
    END;
  END Invoke;

(* There is one server side stub procedure for each method of "T". *)

PROCEDURE GetStub (
    c: StubLib.Conn;
    t: T;
    rep: StubLib.DataRep)
    RAISES {Invalid, NetObj.Error, Rd.Failure,
            Wr.Failure, Thread.Alerted} =
  VAR key, res: TEXT;
  BEGIN
    key := StubLib.InRef(c, rep, TYPECODE(TEXT));
    res := t.get(key);
    StubLib.StartResult(c);
    StubLib.OutInt32(c, ORD(R.OK));
    StubLib.OutRef(c, res);
  END GetStub;

PROCEDURE PutStub (
    c: StubLib.Conn;
    t: T;
    rep: StubLib.DataRep)
    RAISES {Invalid, NetObj.Error, Rd.Failure,
            Wr.Failure, Thread.Alerted} =
  VAR
    key, value: TEXT;
  BEGIN
    key := StubLib.InRef(c, rep, TYPECODE(TEXT));
    value := StubLib.InRef(c, rep, TYPECODE(TEXT));
    t.put(key, value);
    StubLib.StartResult(c);
    StubLib.OutInt32(c, ORD(R.OK));
  END PutStub;

(* All stub code is registered with the network object runtime by 
   the main body of the stub module.  The protocol number is set
   to the stub protocol constant defined above. *)

BEGIN
  StubLib.Register(
    TYPECODE(T), StubVersion,
    TYPECODE(StubT), Invoke);
END Example.
