/* Support routines for building symbol tables in GDB's internal format.
   Copyright 1986, 1987, 1988, 1989, 1990, 1991, 1992
   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or2
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* This module provides subroutines used for creating and adding to
   the symbol table.  These routines are called from various symbol-
   file-reading routines.

   Routines to support specific debugging information formats (stabs,
   DWARF, etc) belong somewhere else. */

#include "defs.h"
#include "bfd.h"
#include "obstack.h"
#include "symtab.h"
#include "symfile.h"		/* Needed for "struct complaint" */
#include "objfiles.h"
#include "complaints.h"
#include <string.h>
#include "gdbtypes.h"            /* Oliver: Needed for "nested functions" */

/* Ask buildsym.h to define the vars it normally declares `extern'.  */
#define	EXTERN	/**/
#include "buildsym.h"		/* Our own declarations */
#undef	EXTERN

/* For cleanup_undefined_types and finish_global_stabs (somewhat
   questionable--see comment where we call them).  */
#include "stabsread.h"

static int
compare_line_numbers PARAMS ((const void *, const void *));

static struct blockvector *
make_blockvector PARAMS ((struct objfile *));


/* Initial sizes of data structures.  These are realloc'd larger if needed,
   and realloc'd down to the size actually used, when completed.  */

#define	INITIAL_CONTEXT_STACK_SIZE	10
#define	INITIAL_LINE_VECTOR_LENGTH	1000


/* Complaints about the symbols we have encountered.  */

/* Oliver */
struct complaint n_m2c_wrong_format_complaint =
{"Wrong Format of %s", 0, 0};

struct complaint innerblock_complaint =
{"inner block not inside outer block in %s", 0, 0};

struct complaint innerblock_anon_complaint =
{"inner block not inside outer block", 0, 0};

struct complaint blockvector_complaint = 
{"block at 0x%lx out of order", 0, 0};


/*
 * Routines added by Oliver.
 * These routines manage Modula 2 Modules.
 */
/*
 * Diese Routine fuegt in das aktuelle subfile
 * ein Module unter der Nummer desc ein
 */
void add_module(name, desc, objfile)
     int desc;
     char *name;
     struct objfile *objfile;
     
{
  if (strchr(name,','))
    {
      /* Free this in end_symtab */
      if (!MODULES(current_subfile))
	{
	  MODULES(current_subfile) = (struct module_entry *)malloc(sizeof(struct module_entry)*OM_MAXMOD);
	  memset(MODULES(current_subfile), 0, sizeof(struct module_entry)*OM_MAXMOD);
	}
      /* erstmal initialisieren */
      current_subfile->nmodules = (desc>current_subfile->nmodules?desc:current_subfile->nmodules);
      MODULES(current_subfile)[desc].number = desc;
      MODULES(current_subfile)[desc].symbol_table = NULL;    
      
      /* Speicher fuer den Namen; den (aber) leider auch nicht auf dem obstack,  */
      MODULES(current_subfile)[desc].name = malloc( strlen(strtok(name,","))+1);
      
      /* obstack_alloc(&objfile->symbol_obstack, strlen(strtok(name,","))+1); */
      strcpy(MODULES (current_subfile) [desc].name, name);
      
    }
  else
    {
      complain(&n_m2c_wrong_format_complaint, name);
    };
};


/* maintain the lists of symbols and blocks */

/* Add a symbol to one of the lists of symbols.  */

void
add_symbol_to_list (symbol, listhead)
     struct symbol *symbol;
     struct pending **listhead;
{
  register struct pending *link;
  
  /* We keep PENDINGSIZE symbols in each link of the list.
     If we don't have a link with room in it, add a new link.  */
  if (*listhead == NULL || (*listhead)->nsyms == PENDINGSIZE)
    {
      if (free_pendings)
	{
	  link = free_pendings;
	  free_pendings = link->next;
	}
      else
	{
	  link = (struct pending *) xmalloc (sizeof (struct pending));
	}

      link->next = *listhead;
      *listhead = link;
      link->nsyms = 0;
    }

  (*listhead)->symbol[(*listhead)->nsyms++] = symbol;
}

/* Find a symbol named NAME on a LIST.  NAME need not be '\0'-terminated;
   LENGTH is the length of the name.  */

struct symbol *
find_symbol_in_list (list, name, length)
     struct pending *list;
     char *name;
     int length;
{
  int j;
  char *pp;

  while (list != NULL)
    {
      for (j = list->nsyms; --j >= 0; )
	{
	  pp = SYMBOL_NAME (list->symbol[j]);
	  if (*pp == *name && strncmp (pp, name, length) == 0 &&
	      pp[length] == '\0')
	    {
	      return (list->symbol[j]);
	    }
	}
      list = list->next;
    }
  return (NULL);
}

/* At end of reading syms, or in case of quit,
   really free as many `struct pending's as we can easily find. */

/* ARGSUSED */
void
really_free_pendings (foo)
     int foo;
{
  struct pending *next, *next1;
#if 0
  struct pending_block *bnext, *bnext1;
#endif

  for (next = free_pendings; next; next = next1)
    {
      next1 = next->next;
      free ((PTR)next);
    }
  free_pendings = NULL;

#if 0 /* Now we make the links in the symbol_obstack, so don't free them.  */
  for (bnext = pending_blocks; bnext; bnext = bnext1)
    {
      bnext1 = bnext->next;
      free ((PTR)bnext);
    }
#endif
  pending_blocks = NULL;

  for (next = file_symbols; next != NULL; next = next1)
    {
      next1 = next->next;
      free ((PTR)next);
    }
  file_symbols = NULL;

  for (next = global_symbols; next != NULL; next = next1)
    {
      next1 = next->next;
      free ((PTR)next);
    }
  global_symbols = NULL;
}

/* Take one of the lists of symbols and make a block from it.
   Keep the order the symbols have in the list (reversed from the input file).
   Put the block on the list of pending blocks.  */

void
finish_block (symbol, listhead, old_blocks, start, end, objfile)
     struct symbol *symbol;
     struct pending **listhead;
     struct pending_block *old_blocks;
     CORE_ADDR start, end;
     struct objfile *objfile;
{
  register struct pending *next, *next1;
  register struct block *block;
  register struct pending_block *pblock;
  struct pending_block *opblock;
  register int i;
  register int j;

  /* Count the length of the list of symbols.  */

  for (next = *listhead, i = 0;
       next;
       i += next->nsyms, next = next->next)
    {
      /*EMPTY*/;
	     }

  block = (struct block *) obstack_alloc (&objfile -> symbol_obstack,
					  (sizeof (struct block) + ((i - 1) * sizeof (struct symbol *))));

  /* Copy the symbols into the block.  */

  BLOCK_NSYMS (block) = i;
  for (next = *listhead; next; next = next->next)
    {
      for (j = next->nsyms - 1; j >= 0; j--)
	{
	  BLOCK_SYM (block, --i) = next->symbol[j];
	}
    }

  BLOCK_START (block) = start;
  BLOCK_END (block) = end;
  /* Superblock filled in when containing block is made */
  BLOCK_SUPERBLOCK (block) = NULL;
  BLOCK_GCC_COMPILED (block) = processing_gcc_compilation;

  /* Put the block in as the value of the symbol that names it.  */

  if (symbol)
    {
      SYMBOL_BLOCK_VALUE (symbol) = block;
      BLOCK_FUNCTION (block) = symbol;
    }
  else
    {
      BLOCK_FUNCTION (block) = NULL;
    }

  /* Now "free" the links of the list, and empty the list.  */

  for (next = *listhead; next; next = next1)
    {
      next1 = next->next;
      next->next = free_pendings;
      free_pendings = next;
    }
  *listhead = NULL;

  /* Install this block as the superblock
     of all blocks made since the start of this scope
     that don't have superblocks yet.  */

  opblock = NULL;
  for (pblock = pending_blocks; pblock != old_blocks; pblock = pblock->next)
    {
      /* Oliver: Skip blocks belonging to functions, since
       * gdb has no support for nested functions
       * This prevents top level functions to be subblocks of STATIC_BLOCK,
       * this is fixed in make blockvector, since I didn't find ANY way to figure
       * out if block is the static block.*/
      if ((BLOCK_SUPERBLOCK (pblock->block) == NULL) && (BLOCK_FUNCTION(pblock->block) == NULL))
	{
#if 1
	  /* Check to be sure the blocks are nested as we receive them. 
	     If the compiler/assembler/linker work, this just burns a small
	     amount of time.  */
	  if (BLOCK_START (pblock->block) < BLOCK_START (block) ||
	      BLOCK_END   (pblock->block) > BLOCK_END   (block))
	    {
	      if (symbol)
		{
		  complain (&innerblock_complaint,
			    SYMBOL_SOURCE_NAME (symbol));
		}
	      else
		{
		  complain (&innerblock_anon_complaint);
		}
	      BLOCK_START (pblock->block) = BLOCK_START (block);
	      BLOCK_END   (pblock->block) = BLOCK_END   (block); 
	    }
#endif
	  BLOCK_SUPERBLOCK (pblock->block) = block; 
	}
      opblock = pblock;
    }
  
  /* Record this block on the list of all blocks in the file.
     Put it after opblock, or at the beginning if opblock is 0.
     This puts the block in the list after all its subblocks.  */

  /* Allocate in the symbol_obstack to save time.
     It wastes a little space.  */
  pblock = (struct pending_block *)
    obstack_alloc (&objfile -> symbol_obstack,
		   sizeof (struct pending_block));
  pblock->block = block;
  if (opblock)
    {
      pblock->next = opblock->next;
      opblock->next = pblock;
    }
  else
    {
      pblock->next = pending_blocks;
      pending_blocks = pblock;
    }
}

static struct blockvector *
make_blockvector (objfile)
     struct objfile *objfile;
{
  register struct pending_block *next;
  register struct blockvector *blockvector;
  register int i;

  /* Count the length of the list of blocks.  */

  for (next = pending_blocks, i = 0; next; next = next->next, i++) {;}

  blockvector = (struct blockvector *)
    obstack_alloc (&objfile -> symbol_obstack,
		   (sizeof (struct blockvector)
		    + (i - 1) * sizeof (struct block *)));

  /* Copy the blocks into the blockvector.
     This is done in reverse order, which happens to put
     the blocks into the proper order (ascending starting address).
     finish_block has hair to insert each block into the list
     after its subblocks in order to make sure this is true.  */

  BLOCKVECTOR_NBLOCKS (blockvector) = i;
  for (next = pending_blocks; next; next = next->next)
    {
      BLOCKVECTOR_BLOCK (blockvector, --i) = next->block;
    }

#if 0 /* Now we make the links in the obstack, so don't free them.  */
  /* Now free the links of the list, and empty the list.  */

  for (next = pending_blocks; next; next = next1)
    {
      next1 = next->next;
      free (next);
    }
#endif
  pending_blocks = NULL;

  /* Oliver: Make functions subblocks of STATIC_BLOCK (see finish_block)*/
  for (i=FIRST_LOCAL_BLOCK; i< BLOCKVECTOR_NBLOCKS(blockvector); i++)
    {
      if (BLOCK_FUNCTION(BLOCKVECTOR_BLOCK(blockvector, i)) != NULL
	  && BLOCK_SUPERBLOCK(BLOCKVECTOR_BLOCK(blockvector,i))== NULL)
	{
	  BLOCK_SUPERBLOCK(BLOCKVECTOR_BLOCK(blockvector,i))=BLOCKVECTOR_BLOCK(blockvector, STATIC_BLOCK);
	}
    }
  

  
#if 1  /* FIXME, shut this off after a while to speed up symbol reading.  */
  /* Some compilers output blocks in the wrong order, but we depend
     on their being in the right order so we can binary search. 
     Check the order and moan about it.  FIXME.  */
  if (BLOCKVECTOR_NBLOCKS (blockvector) > 1)
    {
      for (i = 1; i < BLOCKVECTOR_NBLOCKS (blockvector); i++)
	{
	  if (BLOCK_START(BLOCKVECTOR_BLOCK (blockvector, i-1))
	      > BLOCK_START(BLOCKVECTOR_BLOCK (blockvector, i)))
	    {

	      /* FIXME-32x64: loses if CORE_ADDR doesn't fit in a
		 long.  Possible solutions include a version of
		 complain which takes a callback, a
		 sprintf_address_numeric to match
		 print_address_numeric, or a way to set up a GDB_FILE
		 * which causes sprintf rather than fprintf to be
		 called.  */

	      complain (&blockvector_complaint, 
			(unsigned long) BLOCK_START(BLOCKVECTOR_BLOCK (blockvector, i)));
	    }
	}
    }
#endif

  return (blockvector);
}


/* Start recording information about source code that came from an included
   (or otherwise merged-in) source file with a different name.  NAME is
   the name of the file (cannot be NULL), DIRNAME is the directory in which
   it resides (or NULL if not known).  */

void
start_subfile (name, dirname)
     char *name;
     char *dirname;
{
  register struct subfile *subfile;

  /* See if this subfile is already known as a subfile of the
     current main source file.  */

  for (subfile = subfiles; subfile; subfile = subfile->next)
    {
      if (STREQ (subfile->name, name))
	{
	  current_subfile = subfile;
	  return;
	}
    }

  /* This subfile is not known.  Add an entry for it.
     Make an entry for this subfile in the list of all subfiles
     of the current main source file.  */

  subfile = (struct subfile *) xmalloc (sizeof (struct subfile));
  subfile->next = subfiles;
  subfiles = subfile;
  current_subfile = subfile;

  /* Save its name and compilation directory name */
  subfile->name = (name == NULL) ? NULL : savestring (name, strlen (name));
  subfile->dirname =
    (dirname == NULL) ? NULL : savestring (dirname, strlen (dirname));
  
  /* Initialize line-number recording for this subfile.  */
  subfile->line_vector = NULL;

  /* Default the source language to whatever can be deduced from
     the filename.  If nothing can be deduced (such as for a C/C++
     include file with a ".h" extension), then inherit whatever
     language the previous subfile had.  This kludgery is necessary
     because there is no standard way in some object formats to
     record the source language.  Also, when symtabs are allocated
     we try to deduce a language then as well, but it is too late
     for us to use that information while reading symbols, since
     symtabs aren't allocated until after all the symbols have
     been processed for a given source file. */

  subfile->language = deduce_language_from_filename (subfile->name);
  if (subfile->language == language_unknown &&
      subfile->next != NULL)
    {
      subfile->language = subfile->next->language;
    }

  /* cfront output is a C program, so in most ways it looks like a C
     program.  But to demangle we need to set the language to C++.  We
     can distinguish cfront code by the fact that it has #line
     directives which specify a file name ending in .C.

     So if the filename of this subfile ends in .C, then change the language
     of any pending subfiles from C to C++.  We also accept any other C++
     suffixes accepted by deduce_language_from_filename (in particular,
     some people use .cxx with cfront).  */

  if (subfile->name)
    {
      struct subfile *s;

      if (deduce_language_from_filename (subfile->name) == language_cplus)
	for (s = subfiles; s != NULL; s = s->next)
	  if (s->language == language_c)
	    s->language = language_cplus;
    }

  /* And patch up this file if necessary.  */
  if (subfile->language == language_c
      && subfile->next != NULL
      && subfile->next->language == language_cplus)
    {
      subfile->language = language_cplus;
    }

  /* Oliver added initialization of Modules */
  subfile->nmodules = 0;
  MODULES(subfile) = NULL;
  
}

/* For stabs readers, the first N_SO symbol is assumed to be the source
   file name, and the subfile struct is initialized using that assumption.
   If another N_SO symbol is later seen, immediately following the first
   one, then the first one is assumed to be the directory name and the
   second one is really the source file name.

   So we have to patch up the subfile struct by moving the old name value to
   dirname and remembering the new name.  Some sanity checking is performed
   to ensure that the state of the subfile struct is reasonable and that the
   old name we are assuming to be a directory name actually is (by checking
   for a trailing '/'). */

void
patch_subfile_names (subfile, name)
     struct subfile *subfile;
     char *name;
{
  if (subfile != NULL && subfile->dirname == NULL && subfile->name != NULL
      && subfile->name[strlen(subfile->name)-1] == '/')
    {
      subfile->dirname = subfile->name;
      subfile->name = savestring (name, strlen (name));

      /* Default the source language to whatever can be deduced from
	 the filename.  If nothing can be deduced (such as for a C/C++
	 include file with a ".h" extension), then inherit whatever
	 language the previous subfile had.  This kludgery is necessary
	 because there is no standard way in some object formats to
	 record the source language.  Also, when symtabs are allocated
	 we try to deduce a language then as well, but it is too late
	 for us to use that information while reading symbols, since
	 symtabs aren't allocated until after all the symbols have
	 been processed for a given source file. */

      subfile->language = deduce_language_from_filename (subfile->name);
      if (subfile->language == language_unknown &&
	  subfile->next != NULL)
	{
	  subfile->language = subfile->next->language;
	}
    }
}


/* Handle the N_BINCL and N_EINCL symbol types
   that act like N_SOL for switching source files
   (different subfiles, as we call them) within one object file,
   but using a stack rather than in an arbitrary order.  */

void
push_subfile ()
{
  register struct subfile_stack *tem
    = (struct subfile_stack *) xmalloc (sizeof (struct subfile_stack));

  tem->next = subfile_stack;
  subfile_stack = tem;
  if (current_subfile == NULL || current_subfile->name == NULL)
    {
      abort ();
    }
  tem->name = current_subfile->name;
}

char *
pop_subfile ()
{
  register char *name;
  register struct subfile_stack *link = subfile_stack;

  if (link == NULL)
    {
      abort ();
    }
  name = link->name;
  subfile_stack = link->next;
  free ((PTR)link);
  return (name);
}


/* Add a linetable entry for line number LINE and address PC to the line
   vector for SUBFILE.  */

void
record_line (subfile, line, pc)
     register struct subfile *subfile;
     int line;
     CORE_ADDR pc;
{
  struct linetable_entry *e;
  /* Ignore the dummy line number in libg.o */

  if (line == 0xffff)
    {
      return;
    }

  /* Make sure line vector exists and is big enough.  */
  if (!subfile->line_vector)
    {
      subfile->line_vector_length = INITIAL_LINE_VECTOR_LENGTH;
      subfile->line_vector = (struct linetable *)
	xmalloc (sizeof (struct linetable)
		 + subfile->line_vector_length * sizeof (struct linetable_entry));
      subfile->line_vector->nitems = 0;
    }

  if (subfile->line_vector->nitems + 1 >= subfile->line_vector_length)
    {
      subfile->line_vector_length *= 2;
      subfile->line_vector = (struct linetable *)
	xrealloc ((char *) subfile->line_vector, (sizeof (struct linetable)
						  + subfile->line_vector_length * sizeof (struct linetable_entry)));
    }

  e = subfile->line_vector->item + subfile->line_vector->nitems++;
  e->line = line; e->pc = pc;
}


/* Needed in order to sort line tables from IBM xcoff files.  Sigh!  */

static int
compare_line_numbers (ln1p, ln2p)
     const PTR ln1p;
     const PTR ln2p;
{
  struct linetable_entry *ln1 = (struct linetable_entry *) ln1p;
  struct linetable_entry *ln2 = (struct linetable_entry *) ln2p;

  /* Note: this code does not assume that CORE_ADDRs can fit in ints.
     Please keep it that way.  */
  if (ln1->pc < ln2->pc)
    return -1;

  if (ln1->pc > ln2->pc)
    return 1;

  /* If pc equal, sort by line.  I'm not sure whether this is optimum
     behavior (see comment at struct linetable in symtab.h).  */
  return ln1->line - ln2->line;
}


/* Start a new symtab for a new source file.
   Called, for example, when a stabs symbol of type N_SO is seen, or when
   a DWARF TAG_compile_unit DIE is seen.
   It indicates the start of data for one original source file.  */

void
start_symtab (name, dirname, start_addr)
     char *name;
     char *dirname;
     CORE_ADDR start_addr;
{

  last_source_file = name;
  last_source_start_addr = start_addr;
  file_symbols = NULL;
  global_symbols = NULL;
  within_function = 0;

  /* Context stack is initially empty.  Allocate first one with room for
     10 levels; reuse it forever afterward.  */
  if (context_stack == NULL)
    {
      context_stack_size = INITIAL_CONTEXT_STACK_SIZE;
      context_stack = (struct context_stack *)
	xmalloc (context_stack_size * sizeof (struct context_stack));
    }
  context_stack_depth = 0;

  /* Initialize the list of sub source files with one entry
     for this file (the top-level source file).  */

  subfiles = NULL;
  current_subfile = NULL;
  start_subfile (name, dirname);
}

static int compare_addr(i,j)
     CORE_ADDR *i, *j;
{
  if (*i < *j)
    return -1;
  if( *i > *j)
    return 1;
  return 0;
}


/* Finish the symbol definitions for one main source file,
   close off all the lexical contexts for that file
   (creating struct block's for them), then make the struct symtab
   for that file and put it in the list of all such.

   END_ADDR is the address of the end of the file's text.
   SECTION is the section number (in objfile->section_offsets) of
   the blockvector and linetable.

   Note that it is possible for end_symtab() to return NULL.  In particular,
   for the DWARF case at least, it will return NULL when it finds a
   compilation unit that has exactly one DIE, a TAG_compile_unit DIE.  This
   can happen when we link in an object file that was compiled from an empty
   source file.  Returning NULL is probably not the correct thing to do,
   because then gdb will never know about this empty file (FIXME). */

/* Oliver: I changed the meaning of sort_pending
 * 0 still means don't sort.
 * 1 means sort but don't adjust blocks.
 * 2 means sort AND adjust blocks. (for dbxread and Modula-2)
 */
 
struct symtab *
   end_symtab (end_addr, sort_pending, sort_linevec, objfile, section)
     CORE_ADDR end_addr;
   int sort_pending;
   int sort_linevec;
   struct objfile *objfile;
   int section;
{
  register struct symtab *symtab = NULL;
  register struct blockvector *blockvector;
  register struct subfile *subfile;
  register struct context_stack *cstk;
  struct subfile *nextsub;
#if 0
  int blk_nr; /*this added by Oliver */
#endif
  

  /* Finish the lexical context of the last function in the file;
     pop the context stack.  */

  if (context_stack_depth > 0)
    {
      context_stack_depth--;
      cstk = &context_stack[context_stack_depth];
      /* Make a block for the local symbols within.  */
      finish_block (cstk->name, &local_symbols, cstk->old_blocks,
		    cstk->start_addr, end_addr, objfile);

      if (context_stack_depth > 0)
	{
	  /* This is said to happen with SCO.  The old coffread.c code
	     simply emptied the context stack, so we do the same.  FIXME:
	     Find out why it is happening.  This is not believed to happen
	     in most cases (even for coffread.c); it used to be an abort().  */
	  static struct complaint msg =
	    {"Context stack not empty in end_symtab", 0, 0};
	  complain (&msg);
	  context_stack_depth = 0;
	}
    }

  /* It is unfortunate that in xcoff, pending blocks might not be ordered
     in this stage. Especially, blocks for static functions will show up at
     the end.  We need to sort them, so tools like `find_pc_function' and
     `find_pc_block' can work reliably. */
  /* Oliver: Always sort blocks if reading Modula 2
   * blocks may be nested */
  
  if (sort_pending && pending_blocks) 
    {
      /* FIXME!  Remove this horrid bubble sort and use qsort!!! */
      int swapped;
      do
	{
	  struct pending_block *pb, *pbnext;
	  
	  pb = pending_blocks;
	  pbnext = pb->next;
	  swapped = 0;

	  while (pbnext)
	    {
	      /* swap blocks if unordered! */
	      
	      if (BLOCK_START(pb->block) < BLOCK_START(pbnext->block)) 
		{
		  struct block *tmp = pb->block;
		  pb->block = pbnext->block;
		  pbnext->block = tmp;
		  swapped = 1;
		}
	      pb = pbnext;
	      pbnext = pbnext->next;
	    }
	} while (swapped);
    }

  /* Oliver: Adjust block addresses.
   * If this is EPC Modula-2 file we have to set the end address
   * for the functions 'by hand'.
   */

  if (sort_pending == 2)
    {
      struct pending_block *current_pending;
      struct block *current_block;
      long unsigned int last_end;
      
      int i;
/* Oliver: I implement two alternative algorithms to determine the real end
 * of a block. Define only ONE of the following: */
#define USE_BLOCK_START_AS_BLOCK_END 1
/* #define USE_LINE_NUMBERS_AS_BLOCK_END 1 */
      
#ifdef USE_BLOCK_START_AS_BLOCK_END
      int block_count = 0;
      CORE_ADDR *pcs;
      CORE_ADDR current;
  
      for (current_pending = pending_blocks;
	   current_pending;
	   current_pending = current_pending->next)
	{
	  block_count++;
	}
      pcs = alloca(sizeof(CORE_ADDR)*block_count*2);
      for (current_pending = pending_blocks, i=0;
	   current_pending;
	   current_pending = current_pending -> next)
	{
	  pcs[i++]= BLOCK_START(current_pending->block);
	  pcs[i++]= BLOCK_END(current_pending->block);
	}
      qsort ((char *)pcs, block_count*2, sizeof(CORE_ADDR), compare_addr);
      /* now remove duplicates */
      {
	int from;

	for (i=0, from=1; from<block_count*2; from++)
	  {
	    if (pcs[i] != pcs[from])
	      {
		pcs[++i] = pcs[from];
	      }
	  }
      }
      block_count = i+1;
      
      for (current_pending = pending_blocks;
	   current_pending;
	   current_pending = current_pending -> next)
	{
	  /* I look for the start address of each block and set its end address
	     to the next higher one in pcs. */
	  current = BLOCK_START(current_pending->block);
	  for(i=0;
	      i< block_count-1;
	      i++)
	    {
	      if (pcs[i]==current)
		{
		  BLOCK_END(current_pending->block) = pcs[i+1];
		  break;
		}
	    }
	}
      
      
#endif
      
#ifdef USE_LINE_NUMBERS_AS_BLOCK_END
      /* End of first block (this is the last functions) is the largest known address
       * I know this is wrong, but it's right if we take modules in account, and
       * the end addresses of Modula-2 blocks are wrong anyway.
       */
      last_end = subfiles->line_vector->item[subfiles->line_vector->nitems - 1].pc;
      
      for (current_pending = pending_blocks;
	   current_pending;
	   current_pending = current_pending->next)
	{
	  current_block = current_pending->block;
	  for (i = 0;
	       (i<subfiles->line_vector->nitems) && (subfiles->line_vector->item[i].pc<BLOCK_START(current_block));
	       i++)
	    {
	    }
	  if (i==subfiles->line_vector->nitems)
	    {
	      static struct complaint msg =
	      {
		"Start address %d of function to large", 0, 0
	      };
	      complain (&msg, BLOCK_START(current_block));
	    }
	  else
	    {
	      BLOCK_END(current_block) = last_end;
	      if (i)
		last_end = subfiles->line_vector->item[i-1].pc;
	      else
		if (current_pending->next)
		  {
		    static struct complaint msg =
		    {
		      "Can't determine end address", 0, 0
		    };
		    complain (&msg);
		  }
	      
	    }
	}
#endif
    }

  /* Cleanup any undefined types that have been left hanging around
     (this needs to be done before the finish_blocks so that
     file_symbols is still good).

     Both cleanup_undefined_types and finish_global_stabs are stabs
     specific, but harmless for other symbol readers, since on gdb
     startup or when finished reading stabs, the state is set so these
     are no-ops.  FIXME: Is this handled right in case of QUIT?  Can
     we make this cleaner?  */

  cleanup_undefined_types ();
  finish_global_stabs (objfile);

  if (pending_blocks == NULL
      && file_symbols == NULL
      && global_symbols == NULL)
    {
      /* Ignore symtabs that have no functions with real debugging info */
      blockvector = NULL;
    }
  else
    {
      /* Define the STATIC_BLOCK & GLOBAL_BLOCK, and build the blockvector. */
      finish_block (0, &file_symbols, 0, last_source_start_addr, end_addr,
		    objfile);
      finish_block (0, &global_symbols, 0, last_source_start_addr, end_addr,
		    objfile);
      blockvector = make_blockvector (objfile);
    }

#ifdef PROCESS_LINENUMBER_HOOK
  PROCESS_LINENUMBER_HOOK ();			/* Needed for xcoff. */
#endif

  /* Now create the symtab objects proper, one for each subfile.  */
  /* (The main file is the last one on the chain.)  */

  for (subfile = subfiles; subfile; subfile = nextsub)
    {
      int linetablesize = 0;
      /* If we have blocks of symbols, make a symtab.
	 Otherwise, just ignore this file and any line number info in it.  */
      symtab = NULL;
      if (blockvector)
	{
	  if (subfile->line_vector)
	    {
	      linetablesize = sizeof (struct linetable) +
		subfile->line_vector->nitems * sizeof (struct linetable_entry);
#if 0
	      /* I think this is artifact from before it went on the obstack.
		 I doubt we'll need the memory between now and when we
		 free it later in this function.  */
	      /* First, shrink the linetable to make more memory.  */
	      subfile->line_vector = (struct linetable *)
		xrealloc ((char *) subfile->line_vector, linetablesize);
#endif
	      /* If sort_linevec is false, we might want just check to make
		 sure they are sorted and complain() if not, as a way of
		 tracking down compilers/symbol readers which don't get
		 them sorted right.  */

	      if (sort_linevec)
		qsort (subfile->line_vector->item,
		       subfile->line_vector->nitems,
		       sizeof (struct linetable_entry), compare_line_numbers);
	    }

	  /* Now, allocate a symbol table.  */
	  symtab = allocate_symtab (subfile->name, objfile);

	  /* Fill in its components.  */
	  symtab->blockvector = blockvector;
	  if (subfile->line_vector)
	    {
	      /* Reallocate the line table on the symbol obstack */
	      symtab->linetable = (struct linetable *) 
		obstack_alloc (&objfile -> symbol_obstack, linetablesize);
	      memcpy (symtab->linetable, subfile->line_vector, linetablesize);
	    }
	  else
	    {
	      symtab->linetable = NULL;
	    }
	  symtab->block_line_section = section;
	  if (subfile->dirname)
	    {
	      /* Reallocate the dirname on the symbol obstack */
	      symtab->dirname = (char *)
		obstack_alloc (&objfile -> symbol_obstack,
			       strlen (subfile -> dirname) + 1);
	      strcpy (symtab->dirname, subfile->dirname);
	    }
	  else
	    {
	      symtab->dirname = NULL;
	    }
	  symtab->free_code = free_linetable;
	  symtab->free_ptr = NULL;

	  /* Use whatever language we have been using for this subfile,
	     not the one that was deduced in allocate_symtab from the
	     filename.  We already did our own deducing when we created
	     the subfile, and we may have altered our opinion of what
	     language it is from things we found in the symbols. */
	  symtab->language = subfile->language;

	  /* All symtabs for the main file and the subfiles share a
	     blockvector, so we need to clear primary for everything but
	     the main file.  */

	  symtab->primary = 0;
	}
      /*
       * Oliver: If there's an array of modules in the subfile, add it contents
       * to this symtab.
       */
      if (MODULES(subfile))
	{
	  MODULES(symtab)=(struct module_entry*)realloc(MODULES(subfile), sizeof(struct module_entry)*(subfile->nmodules));
	  symtab->nmodules=subfile->nmodules;
	}
#if 0
      /*
       * Oliver:
       * Die gesamte Tabelle nach Symbolen durchsuchen, die noch
       * in anderen Modulen stehen.
       */
      for(blk_nr = 0; blk_nr < BLOCKVECTOR_NBLOCKS(BLOCKVECTOR(symtab)); blk_nr++)
	{
	  struct block *blk;
	  int sym_nr;

	  blk = (BLOCKVECTOR_BLOCK(BLOCKVECTOR (symtab),blk_nr));
	  
	  for(sym_nr = 0; sym_nr < BLOCK_NSYMS(blk); sym_nr++)
	    if (SYMBOL_CLASS(BLOCK_SYM(blk,sym_nr))==LOC_MOD_NUMBER)
	      {
		/* eine Symbol gefunden */
		char *name;
		int number = SYMBOL_VALUE (BLOCK_SYM(blk,sym_nr));
		struct symtab *symt;
		struct symbol *defined_symbol;
		/* die Definition des symbols suchen */
		defined_symbol = lookup_imported_symbol(SYMBOL_NAME(BLOCK_SYM(blk,sym_nr)), MODULES(symtab)[number]);
		
		
		if (defined_symbol)
		  {
		    /* Jetzt noch die Daten des anderen Symbols kopieren */
		    memcpy(BLOCK_SYM(blk,sym_nr),  defined_symbol, sizeof(struct symbol));
		  }
	      }
	  
	  
	}
#endif
      /* cleanup_imported_types(); */
      
#if 0
      /* Speicher freigeben */
      if (MODULES(subfile) != NULL)
	{
	  free ((PTR) MODULES(subfile));
	}
#endif     
      if (subfile->name != NULL)
	{
	  free ((PTR) subfile->name);
	}
      if (subfile->dirname != NULL)
	{
	  free ((PTR) subfile->dirname);
	}
      if (subfile->line_vector != NULL)
	{
	  free ((PTR) subfile->line_vector);
	}

      nextsub = subfile->next;
      free ((PTR)subfile);
    }

  /* Set this for the main source file.  */
  if (symtab)
    {
      symtab->primary = 1;
    }
  
  last_source_file = NULL;
  current_subfile = NULL;

  return (symtab);
}


/* Push a context block.  Args are an identifying nesting level (checkable
   when you pop it), and the starting PC address of this context.  */

struct context_stack *
push_context (desc, valu)
     int desc;
     CORE_ADDR valu;
{
  register struct context_stack *new;

  if (context_stack_depth == context_stack_size)
    {
      context_stack_size *= 2;
      context_stack = (struct context_stack *)
	xrealloc ((char *) context_stack,
		  (context_stack_size * sizeof (struct context_stack)));
    }

  new = &context_stack[context_stack_depth++];
  new->depth = desc;
  new->locals = local_symbols;
  new->old_blocks = pending_blocks;
  new->start_addr = valu;
  new->name = NULL;

  local_symbols = NULL;

  return (new);
}


/* Compute a small integer hash code for the given name. */

int
hashname (name)
     char *name;
{
  register char *p = name;
  register int total = p[0];
  register int c;

  c = p[1];
  total += c << 2;
  if (c)
    {
      c = p[2];
      total += c << 4;
      if (c)
	{
	  total += p[3] << 6;
	}
    }

  /* Ensure result is positive.  */
  if (total < 0)
    {
      total += (1000 << 6);
    }
  return (total % HASHSIZE);
}


/* Initialize anything that needs initializing when starting to read
   a fresh piece of a symbol file, e.g. reading in the stuff corresponding
   to a psymtab.  */

void
buildsym_init ()
{
  free_pendings = NULL;
  file_symbols = NULL;
  global_symbols = NULL;
  pending_blocks = NULL;
}

/* Initialize anything that needs initializing when a completely new
   symbol file is specified (not just adding some symbols from another
   file, e.g. a shared library).  */

void
buildsym_new_init ()
{
  buildsym_init ();
}

/* Initializer for this module */

void
_initialize_buildsym ()
{
}
