#include "defs.h"
#include <string.h>
#include "expression.h"
/* Oliver included symtab.h */
#include "symtab.h"
#include "parser-defs.h"
#include "value.h"
#include "language.h"
#include "m3-lang.h"
#include "gdbtypes.h"

#define TK_EOF                   '\000'
#define TK_IDENT                 '0'
#define TK_INT                   '1'
#define TK_LE                    '2'
#define TK_GE                    '3'
#define TK_WRONG                 '4'
#define TK_LIKE_FIRST            '5'
#define TK_GDB_HISTORY           '6'
#define TK_REGISTER              '7'
#define TK_GDB_VAR               '8'
#define TK_ASSIGN                '9'

#define TK_ABS                   'a'
#define TK_ADR		         'b'
#define TK_ADRSIZE		 'c'
#define TK_AND		         'd'
#define TK_ARRAY		 'e'
#define TK_BITSIZE		 'f'
#define TK_BYTESIZE	         'g'
#define TK_CEILING		 'h'
#define TK_DIV		         'i'
#define TK_EXTENDED    	         'j'
#define TK_FALSE		 'k'
#define TK_FLOAT		 'm'
#define TK_FLOOR		 'n'
#define TK_IN		         'o'
#define TK_INTEGER		 'p'
#define TK_ISTYPE		 'q'
#define TK_LONGREAL	         's'
#define TK_LOOPHOLE	         't'
#define TK_MAX		         'u'
#define TK_MIN		         'v'
#define TK_MOD		         'w'
#define TK_NARROW		 'x'
#define TK_NEW		         'y'
#define TK_NIL		         'z'
#define TK_NOT		         'A'
#define TK_OR		         'C'
#define TK_ORD		         'D'
#define TK_SUBARRAY	         'E'
#define TK_TRUE		         'F'
#define TK_TRUNC		 'G'
#define TK_TYPECODE	         'H'
#define TK_VAL		         'I'
#define TK_ROUND                 'J'

static struct
  {
    char kind;
    enum exp_opcode op;
    char *string;
    int length;
    LONGEST intval;
  }
cur_tok;

extern char *lexptr;

struct reserved
  {
    char *name;
    char kind;
    enum exp_opcode op;
  };
static struct reserved reserved[] =
{
  {"ABS", TK_ABS},
  {"ADDRESS", TK_WRONG},
  {"ADR", TK_LIKE_FIRST, UNOP_M3_ADR},
  {"ADRSIZE", TK_ADRSIZE},
  {"AND", TK_AND},
  {"ANY", TK_WRONG},
  {"ARRAY", TK_ARRAY},
  {"AS", TK_WRONG},
  {"BEGIN", TK_WRONG},
  {"BITS", TK_WRONG},
  {"BITSIZE", TK_BITSIZE},
  {"BOOLEAN", TK_WRONG},
  {"BRANDED", TK_WRONG},
  {"BY", TK_WRONG},
  {"BYTESIZE", TK_BYTESIZE},
  {"CARDINAL", TK_WRONG},
  {"CASE", TK_WRONG},
  {"CEILING", TK_CEILING},
  {"CHAR", TK_WRONG},
  {"CONST", TK_WRONG},
  {"DEC", TK_WRONG},
  {"DISPOSE", TK_WRONG},
  {"DIV", TK_DIV},
  {"DO", TK_WRONG},
  {"ELSE", TK_WRONG},
  {"ELSIF", TK_WRONG},
  {"END", TK_WRONG},
  {"EVAL", TK_WRONG},
  {"EXCEPT", TK_WRONG},
  {"EXCEPTION", TK_WRONG},
  {"EXIT", TK_WRONG},
  {"EXPORTS", TK_WRONG},
  {"EXTENDED", TK_EXTENDED},
  {"FALSE", TK_FALSE},
  {"FINALLY", TK_WRONG},
  {"FIRST", TK_LIKE_FIRST, UNOP_M3_FIRST},
  {"FLOAT", TK_FLOAT},
  {"FLOOR", TK_FLOOR},
  {"FOR", TK_WRONG},
  {"FROM", TK_WRONG},
  {"GENERIC", TK_WRONG},
  {"IF", TK_WRONG},
  {"IMPORT", TK_WRONG},
  {"IN", TK_IN},
  {"INC", TK_WRONG},
  {"INTEGER", TK_INTEGER},
  {"INTERFACE", TK_WRONG},
  {"ISTYPE", TK_ISTYPE},
  {"LAST", TK_LIKE_FIRST, UNOP_M3_LAST},
  {"LOCK", TK_WRONG},
  {"LONGREAL", TK_LONGREAL},
  {"LOOP", TK_WRONG},
  {"LOOPHOLE", TK_LOOPHOLE},
  {"METHODS", TK_WRONG},
  {"MAX", TK_MAX},
  {"MIN", TK_MIN},
  {"MOD", TK_MOD},
  {"MODULE", TK_WRONG},
  {"MUTEX", TK_WRONG},
  {"NARROW", TK_NARROW},
  {"NEW", TK_NEW},
  {"NIL", TK_NIL},
  {"NOT", TK_NOT},
  {"NULL", TK_WRONG},
  {"NUMBER", TK_LIKE_FIRST, UNOP_M3_NUMBER},
  {"OBJECT", TK_WRONG},
  {"OF", TK_WRONG},
  {"OR", TK_OR},
  {"ORD", TK_ORD},
  {"OVERRIDES", TK_WRONG},
  {"PROCEDURE", TK_WRONG},
  {"RAISE", TK_WRONG},
  {"RAISES", TK_WRONG},
  {"READONLY", TK_WRONG},
  {"REAL", TK_WRONG},
  {"RECORD", TK_WRONG},
  {"REF", TK_WRONG},
  {"REFANY", TK_WRONG},
  {"REPEAT", TK_WRONG},
  {"RETURN", TK_WRONG},
  {"REVEAL", TK_WRONG},
  {"ROOT", TK_WRONG},
  {"ROUND", TK_ROUND},
  {"SET", TK_WRONG},
  {"SUBARRAY", TK_SUBARRAY},
  {"TEXT", TK_WRONG},
  {"THEN", TK_WRONG},
  {"TO", TK_WRONG},
  {"TRUE", TK_TRUE},
  {"TRUNC", TK_TRUNC},
  {"TRY", TK_WRONG},
  {"TYPE", TK_WRONG},
  {"TYPECASE", TK_WRONG},
  {"TYPECODE", TK_TYPECODE},
  {"UNSAFE", TK_WRONG},
  {"UNTIL", TK_WRONG},
  {"UNTRACED", TK_WRONG},
  {"VAL", TK_VAL},
  {"VALUE", TK_WRONG},
  {"VAR", TK_WRONG},
  {"WHILE", TK_WRONG},
  {"WITH", TK_WRONG}};

static void 
recognize_reserved_word ()
{
  int low, high, mid, cmp;

  low = 0;
  high = sizeof (reserved) / sizeof (struct reserved);

  /* cur_tok.string may be in [low .. high[ */

  while (low < high)
    {
      mid = (low + high) / 2;
      cmp = strcmp (reserved[mid].name, cur_tok.string);
      if (cmp == 0)
	{
	  cur_tok.kind = reserved[mid].kind;
	  cur_tok.op = reserved[mid].op;
	  return;
	}
      else if (cmp < 0)
	{
	  low = mid + 1;
	}
      else
	{
	  high = mid;
	}
    }

  cur_tok.kind = TK_IDENT;
  return;
}

static void 
write_exp_text (opcode, str, len)
     enum exp_opcode opcode;
     char *str;
     int len;
{
  struct stoken t;

  write_exp_elt_opcode (opcode);
  t.ptr = str;
  t.length = len;
  write_exp_string (t);
  write_exp_elt_opcode (opcode);
}

static void 
write_exp_var (sym)
     struct symbol *sym;
{
  write_exp_elt_opcode (OP_VAR_VALUE);
  write_exp_elt_block (block_found);
  write_exp_elt_sym (sym);
  write_exp_elt_opcode (OP_VAR_VALUE);
}

/* Oliver: write_exp_type
 * Writes an OP_TYPE construct to the expression 
 */
static void
write_exp_type (type)
     struct type *type;
{
  write_exp_elt_opcode (OP_TYPE);
  write_exp_elt_type (type);
  write_exp_elt_opcode (OP_TYPE);
}

static void 
get_token ()
{
  char *tokstart;
  while (*lexptr == ' ' || *lexptr == '\t')
    {
      lexptr++;
    }

  switch (*(tokstart = lexptr))
    {
    case '\000':
      cur_tok.kind = TK_EOF;
      return;

    case '$':
      {
	int negate = 1;
	tokstart++;
	lexptr++;
	switch (*lexptr)
	  {
	  case '$':
	    tokstart++;
	    lexptr++;
	    negate = -1;
	    /* fall through */
	  case '0':
	  case '1':
	  case '2':
	  case '3':
	  case '4':
	  case '5':
	  case '6':
	  case '7':
	  case '8':
	  case '9':
	    while ('0' <= *lexptr && *lexptr <= '9')
	      {
		lexptr++;
	      }
	    cur_tok.kind = TK_GDB_HISTORY;
	    cur_tok.intval = ((lexptr == tokstart) ? 1 : atoi (tokstart))
	      * negate;
	    return;

	  default:
	    {
	      int len, c;
	      lexptr = tokstart;
	      while (('a' <= *lexptr && *lexptr <= 'z')
		     || ('A' <= *lexptr && *lexptr <= 'Z')
		     || ('0' <= *lexptr && *lexptr <= '9')
		     || *lexptr == '_')
		{
		  lexptr++;
		}
	      len = lexptr - tokstart;
	      if (len == 0)
		{
		  cur_tok.kind = TK_GDB_HISTORY;
		  cur_tok.intval = 0;
		  return;
		}

	      for (c = 0; c < NUM_REGS; c++)
		{
		  if (len == strlen (reg_names[c])
		      && STREQN (tokstart, reg_names[c], len))
		    {
		      cur_tok.kind = TK_REGISTER;
		      cur_tok.intval = c;
		      return;
		    }
		}

	      for (c = 0; c < num_std_regs; c++)
		{
		  if (len == strlen (std_regs[c].name)
		      && STREQN (tokstart, std_regs[c].name, len))
		    {
		      cur_tok.kind = TK_REGISTER;
		      cur_tok.intval = std_regs[c].regnum;
		      return;
		    }
		}

	      cur_tok.kind = TK_GDB_VAR;
	      cur_tok.string = (char *) malloc (lexptr - tokstart + 1);
	      strncpy (cur_tok.string, tokstart, lexptr - tokstart);
	      cur_tok.string[lexptr - tokstart] = '\0';
	      cur_tok.length = lexptr - tokstart;
	      return;
	    }
	  }
      }

    case '<':
      {
	if (*(tokstart + 1) == '=')
	  {
	    cur_tok.kind = TK_LE;
	    lexptr = tokstart + 2;
	  }
	else
	  {
	    cur_tok.kind = '<';
	    lexptr = tokstart + 1;
	  }
	return;
      }

    case '>':
      {
	if (*(tokstart + 1) == '=')
	  {
	    cur_tok.kind = TK_GE;
	    lexptr = tokstart + 2;
	  }
	else
	  {
	    cur_tok.kind = '>';
	    lexptr = tokstart + 1;
	  }
	return;
      }

    case ':':
      {
	if (*(tokstart + 1) == '=')
	  {
	    cur_tok.kind = TK_ASSIGN;
	    lexptr = tokstart + 2;
	  }
	else
	  {
	    cur_tok.kind = ':';
	    lexptr = tokstart + 1;
	  }
	return;
      }

    case '=':
    case '#':
    case '&':
    case '+':
    case '-':
    case '*':
    case '/':
    case '.':
    case ',':
    case '^':
    case '(':
    case ')':
    case '[':
    case ']':
      {
	cur_tok.kind = *tokstart;
	lexptr = tokstart + 1;
	return;
      }

    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      {
	char *c;
	c = tokstart + 1;
	while ('0' <= *c && *c <= '9')
	  {
	    c++;
	  }
	cur_tok.kind = TK_INT;
	sscanf (tokstart, "%ld", &cur_tok.intval);
	if ((*c == '_') && (cur_tok.intval >= 2) && (cur_tok.intval <= 16))
	  {
	    LONGEST base = cur_tok.intval;
	    LONGEST digit;
	    LONGEST val = 0;
	    c++;
	    while (1)
	      {
		if ('0' <= *c && *c <= '9')
		  {
		    digit = *c - '0';
		  }
		else if ('A' <= *c && *c <= 'F')
		  {
		    digit = *c - 'A' + 10;
		  }
		else if ('a' <= *c && *c <= 'f')
		  {
		    digit = *c - 'a' + 10;
		  }
		else
		  {
		    break;
		  }
		if (digit >= base)
		  {
		    break;
		  }
		val = val * base + digit;
		c++;
	      }
	    cur_tok.intval = val;
	  }
	lexptr = c;
	return;
      }

    case '_':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'G':
    case 'H':
    case 'I':
    case 'J':
    case 'K':
    case 'L':
    case 'M':
    case 'N':
    case 'O':
    case 'P':
    case 'Q':
    case 'R':
    case 'S':
    case 'T':
    case 'U':
    case 'V':
    case 'W':
    case 'X':
    case 'Y':
    case 'Z':
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
    case 'g':
    case 'h':
    case 'i':
    case 'j':
    case 'k':
    case 'l':
    case 'm':
    case 'n':
    case 'o':
    case 'p':
    case 'q':
    case 'r':
    case 's':
    case 't':
    case 'u':
    case 'v':
    case 'w':
    case 'x':
    case 'y':
    case 'z':
      {
	lexptr = tokstart + 1;
	while (('a' <= *lexptr && *lexptr <= 'z')
	       || ('A' <= *lexptr && *lexptr <= 'Z')
	       || ('0' <= *lexptr && *lexptr <= '9')
	       || (*lexptr == '_'))

	  {
	    lexptr++;
	  }
	cur_tok.string = (char *) malloc (lexptr - tokstart + 1);
	strncpy (cur_tok.string, tokstart, lexptr - tokstart);
	cur_tok.string[lexptr - tokstart] = '\0';
	cur_tok.length = lexptr - tokstart;

	recognize_reserved_word ();
	return;
      }
      /* Oliver: for a ' interpret the whole quoted Text as one (1) token */
    case '\'':
      {
	for (lexptr = tokstart + 1;
	     '\'' != *lexptr && *lexptr != '\000';
	     lexptr++);
	if (*lexptr == '\'')
	  {
	    /* tokenstart is still on the leading ', but we just
	     * want the quoted text as a the token.
	     */
	    cur_tok.string = (char *) malloc (lexptr - tokstart);
	    strncpy (cur_tok.string, tokstart + 1, lexptr - tokstart - 1);
	    cur_tok.string[lexptr - tokstart - 1] = '\0';
	    cur_tok.length = lexptr - tokstart - 1;
	    lexptr++;
	    recognize_reserved_word ();
	  }
	else
	  error ("unmatched single quote.", lexptr);
	return;
      }

    default:
      {
	error ("can't recognize start of token: %s", lexptr);
      }
    }
}

static int m3_parse_expr ();

static int 
m3_parse_e7 ()
{
  switch (cur_tok.kind)
    {
    case TK_GDB_HISTORY:
      write_exp_elt_opcode (OP_LAST);
      write_exp_elt_longcst ((LONGEST) cur_tok.intval);
      write_exp_elt_opcode (OP_LAST);
      get_token ();
      break;

    case TK_GDB_VAR:
      write_exp_elt_opcode (OP_INTERNALVAR);
      write_exp_elt_intern (lookup_internalvar (cur_tok.string));
      write_exp_elt_opcode (OP_INTERNALVAR);
      get_token ();
      break;

    case TK_REGISTER:
      write_exp_elt_opcode (OP_REGISTER);
      write_exp_elt_longcst ((LONGEST) cur_tok.intval);
      write_exp_elt_opcode (OP_REGISTER);
      get_token ();
      break;

    case TK_INT:
      write_exp_elt_opcode (OP_M3_LONG);
      write_exp_elt_type (builtin_type_m3_integer);
      write_exp_elt_longcst ((LONGEST) (cur_tok.intval));
      write_exp_elt_opcode (OP_M3_LONG);
      get_token ();
      break;

#if 0
    case CHAR:
      EnumExpr;
    case TEXT:
      TextExpr;
    case REAL:
      ReelExpr;
#endif

    case TK_TRUE:
      write_exp_elt_opcode (OP_M3_LONG);
      write_exp_elt_type (builtin_type_m3_boolean);
      write_exp_elt_longcst ((LONGEST) (1));
      write_exp_elt_opcode (OP_M3_LONG);
      get_token ();
      break;

    case TK_FALSE:
      write_exp_elt_opcode (OP_M3_LONG);
      write_exp_elt_type (builtin_type_m3_boolean);
      write_exp_elt_longcst ((LONGEST) (0));
      write_exp_elt_opcode (OP_M3_LONG);
      get_token ();
      break;

    case '(':
      get_token ();
      m3_parse_expr ();
      if (cur_tok.kind != ')')
	{
	  error ("missing closing )");
	}
      get_token ();
      break;

    case TK_LIKE_FIRST:
      {
	int this_op = cur_tok.op;
	get_token ();
	if (cur_tok.kind != '(')
	  {
	    error ("missing opening (");
	  }
	get_token ();
	m3_parse_expr ();
	if (cur_tok.kind != ')')
	  {
	    error ("missing closing )");
	  }
	get_token ();
	write_exp_elt_opcode (this_op);
	return 0;
      }

    case TK_IDENT:
      {
	struct symbol *sym;
	struct type *interfaces;
	struct block *b;
	char *current_unit_name = 0;
	int i;

	/* Rule 1a: is it local symbol ? */
	if ((sym = lookup_symbol (cur_tok.string, expression_context_block,
				  VAR_NAMESPACE, 0, NULL)) != 0
	    && sym->class != LOC_STATIC)
	  {
	    write_exp_var (sym);
	    goto ident_ok;
	  }

	/* Rule 1b: could it be a global name in the current unit ?
	   these are accessible only through the interface record,
	   which happens to be the only symbol in the topmost block. */
	b = expression_context_block;
	while (b && BLOCK_SUPERBLOCK (b))
	  {
	    b = BLOCK_SUPERBLOCK (b);
	  }
	if (b)
	  {
	    sym = BLOCK_SYM (b, 0);
	    current_unit_name = SYMBOL_NAME (sym) + 3;

	    if (find_m3_rec_field (SYMBOL_TYPE (sym), cur_tok.string, 0, 0, 0))
	      {
		write_exp_var (sym);
		write_exp_text (STRUCTOP_M3_STRUCT, cur_tok.string, cur_tok.length);
		goto ident_ok;
	      }
	  }

	/* Rule 1c: could it be a global name in one of the interfaces
	   exported by the current unit ? */
	b = expression_context_block;
	while (b && BLOCK_SUPERBLOCK (b)
	       && BLOCK_SUPERBLOCK (BLOCK_SUPERBLOCK (b)))
	  {
	    b = BLOCK_SUPERBLOCK (b);
	  }
	if (b && BLOCK_SUPERBLOCK (b))
	  {
	    if ((interfaces = find_m3_exported_interfaces (current_unit_name)))
	      {
		for (i = 0; i < TYPE_NFIELDS (interfaces); i++)
		  {
		    if (sym = find_m3_ir ('I', TYPE_FIELD_NAME (interfaces, i)))
		      {
			if (find_m3_rec_field (SYMBOL_TYPE (sym),
					       cur_tok.string, 0, 0, 0))
			  {
			    write_exp_var (sym);
			    write_exp_text (STRUCTOP_M3_STRUCT,
					    cur_tok.string, cur_tok.length);
			    goto ident_ok;
			  }
		      }
		  }
	      }
	    else
	      {
		if (sym = find_m3_ir ('I', current_unit_name))
		  {
		    if (find_m3_rec_field (SYMBOL_TYPE (sym),
					   cur_tok.string, 0, 0, 0))
		      {
			write_exp_var (sym);
			write_exp_text (STRUCTOP_M3_STRUCT,
					cur_tok.string, cur_tok.length);
			goto ident_ok;
		      }
		  }
	      }
	  }

	/* Oliver: Rule 1d: Could it be a Modula-3 type? */
	{
	  struct type *type;

	  type = find_m3_type_named(cur_tok.string);

	  if (type != NULL)
	    {
	      write_exp_type(type);
	      goto ident_ok;
	    }
	}
	
	
	/* Rule 2 and 3 really need to have ".entity" after them, and
	   we need to know entity to resolve rule 2 properly (an entity
	   in an exported interface does not show up in the module 
	   interface record), so we may as well get it now. */

	{
	  char *compilation_unit_name = cur_tok.string;
	  struct symtab *symtab;

	  get_token ();
	  if (cur_tok.kind != '.')
	    {
	      /* Oliver: Changed error message */
 	      error ("can't find identifier: %s", compilation_unit_name); 

	      /* error ("malformed expression"); */
	      return 1;
	    }
	  get_token ();
	  if (cur_tok.kind != TK_IDENT)
	    {
	      error ("malformed expression");
	      return 1;
	    }

	  /* Rule 2: could this be the name of a module ? */
	  if (sym = find_m3_ir ('M', compilation_unit_name))
	    {
	      if (find_m3_rec_field (SYMBOL_TYPE (sym),
				     cur_tok.string, 0, 0, 0))
		{
		  write_exp_var (sym);
		  write_exp_text (STRUCTOP_M3_MODULE,
				  cur_tok.string, cur_tok.length);
		  goto ident_ok;
		}

	      /* may be it is in one of the interfaces exported by that module */
	      if (current_unit_name
		  && (interfaces = find_m3_exported_interfaces (current_unit_name)))
		{
		  for (i = 0; i < TYPE_NFIELDS (interfaces); i++)
		    {
		      if (sym = find_m3_ir ('I', TYPE_FIELD_NAME (interfaces, i)))
			{
			  if (find_m3_rec_field (SYMBOL_TYPE (sym),
						 cur_tok.string, 0, 0, 0))
			    {
			      write_exp_var (sym);
			      write_exp_text (STRUCTOP_M3_STRUCT,
					    cur_tok.string, cur_tok.length);
			      goto ident_ok;
			    }
			}
		    }
		}
	      else
		{
		  if (sym = find_m3_ir ('I', current_unit_name))
		    {
		      if (find_m3_rec_field (SYMBOL_TYPE (sym),
					     cur_tok.string, 0, 0, 0))
			{
			  write_exp_var (sym);
			  write_exp_text (STRUCTOP_M3_STRUCT,
					  cur_tok.string, cur_tok.length);
			  goto ident_ok;
			}
		    }
		}
	    }

	  /* Rule 3: could this be the name of an interface ? */
	  if (sym = find_m3_ir ('I', compilation_unit_name))
	    {
	      if (find_m3_rec_field (SYMBOL_TYPE (sym), cur_tok.string,
				     0, 0, 0))
		{
		  write_exp_var (sym);
		  write_exp_text (STRUCTOP_M3_INTERFACE,
				  cur_tok.string, cur_tok.length);
		  goto ident_ok;
		}
	    }
	}

	/* out of ideas */
	error ("can't find identifier: %s", cur_tok.string);

      ident_ok:
	get_token ();
	break;
      }

    default:
      error ("what is this expression ?");
    }

  while (1)
    {
      switch (cur_tok.kind)
	{
	case '^':
	  write_exp_elt_opcode (UNOP_M3_IND);
	  get_token ();
	  break;

	case '.':
	  {
	    get_token ();
	    /* we cannot ascertain what the type what we have parsed
	       so far is; it may be an object and we need the dynamic type.
	       So we are just going to accept anything that looks ok. */

	    if (cur_tok.kind != TK_IDENT)
	      {
		error ("Field name must be an identifier");
		return 1;
	      }

	    write_exp_text (STRUCTOP_M3_STRUCT, cur_tok.string, cur_tok.length);
	    get_token ();
	    break;
	  }

	case '(':
	  {
	    extern int arglist_len;
	    arglist_len = 0;
	    cur_tok.kind = ',';
	    start_arglist ();
	    while (cur_tok.kind == ',')
	      {
		get_token ();
		m3_parse_expr ();
		arglist_len++;
	      }
	    if (cur_tok.kind != ')')
	      {
		error ("missing ')'");
	      }
	    get_token ();
	    write_exp_elt_opcode (OP_FUNCALL);
	    write_exp_elt_longcst ((LONGEST) end_arglist ());
	    write_exp_elt_opcode (OP_FUNCALL);
	    break;
	  }

	case '[':
	  {
	    struct type *array_type;
	    cur_tok.kind = ',';
	    while (cur_tok.kind == ',')
	      {
		get_token ();
		m3_parse_expr ();
		write_exp_elt_opcode (BINOP_M3_SUBSCRIPT);
	      }

	    if (cur_tok.kind != ']')
	      {
		error ("missing ']'");
	      }
	    get_token ();
	    break;
	  }

	case TK_EOF:
	default:
	  return 0;
	}
    }
}


static int 
m3_parse_e6 ()
{
  int m = 0;

  while (cur_tok.kind == '+' || cur_tok.kind == '-')
    {
      if (cur_tok.kind == '-')
	{
	  m++;
	}
      get_token ();
    }

  if (m3_parse_e7 ())
    {
      return 1;
    }
  if (m % 2 == 1)
    {
      write_exp_elt_opcode (UNOP_M3_NEG);
    }
  return 0;
}

static int 
m3_parse_e5 ()
{
  if (m3_parse_e6 ())
    {
      return 1;
    }
  while (1)
    {
      switch (cur_tok.kind)
	{
	case '*':
	  get_token ();
	  if (m3_parse_e6 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_MULT);
	  break;
	case '/':
	  get_token ();
	  if (m3_parse_e6 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_DIVIDE);
	  break;
	case TK_DIV:
	  get_token ();
	  if (m3_parse_e6 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_DIV);
	  break;
	case TK_MOD:
	  get_token ();
	  if (m3_parse_e6 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_MOD);
	  break;
	default:
	  return 0;
	}
    }
}

static int 
m3_parse_e4 ()
{
  if (m3_parse_e5 ())
    {
      return 1;
    }
  while (1)
    {
      switch (cur_tok.kind)
	{
	case '+':
	  get_token ();
	  if (m3_parse_e5 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_ADD);
	  break;
	case '-':
	  get_token ();
	  if (m3_parse_e5 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_MINUS);
	  break;
	case '&':
	  get_token ();
	  if (m3_parse_e5 ())
	    {
	      return 1;
	    }
	  write_exp_elt_opcode (BINOP_M3_CAT);
	  break;
	default:
	  return 0;
	}
    }
}

static int 
m3_parse_e3 ()
{
  enum exp_opcode op;

  if (m3_parse_e4 ())
    {
      return 1;
    }
  while (1)
    {
      switch (cur_tok.kind)
	{
	case '=':
	  op = BINOP_M3_EQUAL;
	  goto other_arg;
	case '#':
	  op = BINOP_M3_NE;
	  goto other_arg;
	case '<':
	  op = BINOP_M3_LT;
	  goto other_arg;
	case TK_LE:
	  op = BINOP_M3_LE;
	  goto other_arg;
	case '>':
	  op = BINOP_M3_GT;
	  goto other_arg;
	case TK_GE:
	  op = BINOP_M3_GE;
	  goto other_arg;
	case TK_IN:
	  op = BINOP_M3_IN;
	  goto other_arg;

	other_arg:
	  get_token ();
	  if (m3_parse_e4 ())
	    {
	      return (1);
	    }
	  write_exp_elt_opcode (op);
	  break;

	default:
	  return 0;
	}
    }
}

static int 
m3_parse_e2 ()
{
  int n = 0;

  while (cur_tok.kind == TK_NOT)
    {
      n++;
      get_token ();
    }

  if (m3_parse_e3 ())
    {
      return 1;
    }
  if (n % 2 == 1)
    {
      write_exp_elt_opcode (UNOP_M3_NOT);
    }
  return 0;
}

static int 
m3_parse_e1 ()
{
  if (m3_parse_e2 ())
    {
      return 1;
    }
  while (cur_tok.kind == TK_AND)
    {
      get_token ();
      if (m3_parse_e2 ())
	{
	  return 1;
	}
      write_exp_elt_opcode (BINOP_M3_AND);
    }
  return 0;
}

static int 
m3_parse_e0 ()
{
  if (m3_parse_e1 ())
    {
      return 1;
    }
  while (cur_tok.kind == TK_OR)
    {
      get_token ();
      if (m3_parse_e1 ())
	{
	  return 1;
	}
      write_exp_elt_opcode (BINOP_M3_OR);
    }
  return 0;
}

static int 
m3_parse_expr ()
{
  int lhs = 0, rhs = 0;
  lhs = m3_parse_e0 ();
  if (cur_tok.kind == TK_ASSIGN)
    {
      get_token ();
      rhs = m3_parse_e0 ();
      write_exp_elt_opcode (BINOP_ASSIGN);
    }
  if (expout->nelts && expout->elts[0].opcode != OP_TYPE)
    write_exp_elt_opcode (M3_FINAL_TYPE);
  return ((lhs + rhs) != 0);
}

int 
m3_parse ()
{
  get_token ();
  return m3_parse_expr ();
}
