/* Support for printing Modula-3 values for GDB, the GNU debugger.
   Copyright 1994, Digitial Equipement Corporation */

#include "defs.h"
#include "symtab.h"
#include "gdbtypes.h"
#include "expression.h"
#include "value.h"
#include "demangle.h"
#include "valprint.h"
#include "language.h"
#include "target.h"
#include "c-lang.h"

#include "m3-lang.h"


LONGEST
m3_unpack_int (valaddr, bitpos, bitsize)
     char *valaddr;
     int bitpos;
     int bitsize;
{
  LONGEST res; 
  int u;

  if (bitsize <= 8) {		/* char access */
    valaddr += bitpos / 8;
    bitpos = bitpos % 8;
    res = *((char *)valaddr);
    u = 8; }
  else if (bitsize <= 16) {	/* short access */
    short *v = (short *)valaddr;
    v += bitpos / 16;
    bitpos = bitpos % 16;
    res = *v;
    u = 16; }
  else if (bitsize <= 32) {	/* int access */
    int *v = (int *)valaddr;
    v += bitpos / 32;
    bitpos = bitpos % 32;
    res = *v;
    u = 32; }
  else if (bitsize <= TARGET_LONG_BIT) {	/* int access */
    long *v = (long *)valaddr;
    v += bitpos / 64;
    bitpos = bitpos % 64;
    res = *v;
    u = 64; }
  else {
    error ("wrong bitsize in m3_unpack_int: %d", bitsize); }
  
  res = res >> bitpos;
  res = res & ((~0L) >> (sizeof (res) * HOST_CHAR_BIT - bitsize));
  return res;
}

LONGEST
m3_unpack_int2 (val)
     value_ptr val;
{
  return m3_unpack_int (VALUE_CONTENTS (val),  0,
			TYPE_M3_SIZE (VALUE_TYPE (val)));
}


CORE_ADDR
m3_unpack_pointer (valaddr, bitpos)
     char *valaddr;
     int bitpos;
{
  return (CORE_ADDR) m3_unpack_int (valaddr, bitpos, TARGET_PTR_BIT);
}

CORE_ADDR
m3_unpack_pointer2 (val)
     value_ptr val;
{
  return *(CORE_ADDR*) VALUE_CONTENTS (val);
}


double 
m3_unpack_float2 (val)
     value_ptr val;
{
  double res;
  int size = TYPE_LENGTH (VALUE_TYPE (val));

  if (size == 4) {
    res = *(float *) VALUE_CONTENTS (val); }
  else {
    res = *(double*) VALUE_CONTENTS (val); }
  return res;
}


static void 
m3_print_scalar (valaddr, bitpos, bitsize, stream, format)
     char *valaddr;
     int bitpos, bitsize;
     FILE *stream;
     int format;
{
  LONGEST v = m3_unpack_int (valaddr, bitpos, bitsize);

  switch (format) {
    case 'x': fprintf_filtered (stream, "16_%lx", v); break;
    case 'o': fprintf_filtered (stream, "8_%lo", v);  break;
    case 'd':
    default:  fprintf_filtered (stream, "%ld", v);    break; }
}

static void
m3_print_object_1 (valaddr, tc_addr, stream, format)
     char *valaddr;
     CORE_ADDR tc_addr;
     FILE *stream;
     int format;
{
  char name [100];
  struct type *this_obj;
  struct symbol *this_obj_sym;
  int i, data_offset; 
  if (tc_addr == 0) {
    return; }

  this_obj = find_m3_type_from_tc (tc_addr);
  if (TYPE_CODE (this_obj) == TYPE_CODE_M3_ROOT
      || TYPE_CODE (this_obj) == TYPE_CODE_M3_UN_ROOT) {
    return; }

  m3_print_object_1 (valaddr, tc_address_to_parent_tc_address (tc_addr),
		     stream, format);

  data_offset = tc_address_to_dataOffset (tc_addr);

  fputs_filtered ("{", stream);
  for (i = 0; i < TYPE_M3_OBJ_NFIELDS (this_obj); i++) {
    if (i != 0) {
      fputs_filtered (", ", stream);
      wrap_here ("    "); }
    fputs_filtered (TYPE_M3_OBJ_FIELD_NAME (this_obj, i), stream);
    fputs_filtered (" = ", stream);
    m3_val_print2 (TYPE_M3_OBJ_FIELD_TYPE (this_obj, i), 
		   valaddr, 
		   data_offset * TARGET_CHAR_BIT + 
		      TYPE_M3_OBJ_FIELD_BITPOS (this_obj, i),
		   TYPE_M3_OBJ_FIELD_BITSIZE (this_obj, i),
		   stream, format, 0, 0); }
  fputs_filtered ("}", stream);
}

void m3_read_object_fields_bits (valaddr, bitpos, type, tc_addr_res, bits)
     char *valaddr;
     int bitpos;
     struct type *type;
     CORE_ADDR *tc_addr_res;
     char **bits;
{
  int typecode;
  CORE_ADDR bits_addr, typecode_addr, tc_addr;
  int dataSize, dataOffset;

  bits_addr = m3_unpack_int (valaddr, bitpos, TARGET_PTR_BIT);

  if (bits_addr == 0) {
    *bits = 0;
    return; }

  tc_addr = find_m3_heap_tc_addr (bits_addr);
  dataSize = tc_address_to_dataSize (tc_addr);
  *bits = malloc (dataSize);
  target_read_memory (bits_addr, *bits, dataSize);
  if (tc_addr_res != 0) {
    *tc_addr_res = tc_addr; }
}

static void
m3_print_object (valaddr, bitpos, type, stream, format)
     char *valaddr;
     int bitpos;
     struct type *type;
     FILE *stream;
     int format;
{
  char *bits;
  CORE_ADDR tc_addr;

  m3_read_object_fields_bits (valaddr, bitpos, type, &tc_addr, &bits);

  if (bits == 0) {
    fputs_filtered ("NIL", stream); 
    return; }

  m3_print_object_1 (bits, tc_addr, stream, format);
}


/* Print data of type TYPE located at VALADDR (within GDB), which came from
   the inferior at address ADDRESS, onto stdio stream STREAM according to
   FORMAT (a letter or 0 for natural format).  The data at VALADDR is in
   target byte order.
   
   If the data are a string pointer, returns the number of string characters
   printed.
   
   If DEREF_REF is nonzero, then dereference references, otherwise just print
   them like pointers.
   
   The PRETTY parameter controls prettyprinting.  */

static int 
compare (valaddr, bitpos1, bitpos2, bitsize)
     char *valaddr;
     int bitpos1, bitpos2, bitsize;
{
  if ((bitpos1 % 8) != 0 || (bitpos2 % 8) != 0 || (bitsize % 8 != 0)) {
    /* this comparaisons are too hard for now */
    return 0; }
  return memcmp (valaddr + bitpos1/8, valaddr + bitpos2/8, bitsize/8) == 0;
}

extern unsigned int repeat_count_threshold;

int
m3_val_print2 (type, valaddr, bitpos, bitsize, stream, format, deref_ref, toplevel)
     struct type *type;
     char *valaddr;
     int bitpos;
     int bitsize;
     FILE *stream;
     int format;
     int deref_ref;
     int toplevel;
{
  register unsigned int i = 0;		/* Number of characters printed */
  unsigned len;
  struct type *elttype;
  unsigned eltlen;
  LONGEST val;
  unsigned char c;
  CORE_ADDR addr;
  int things_printed = 0;
  int reps, j;

  switch (TYPE_CODE (type))
    {
    case TYPE_CODE_M3_ARRAY: {
      struct type *index = TYPE_M3_ARRAY_INDEX (type);
      struct type *elt   = TYPE_M3_ARRAY_ELEM (type);
      LONGEST lower, upper, n;
      
      fputs_filtered ("{", stream);

      m3_ordinal_bounds (index, &lower, &upper);
      n = upper - lower + 1;

      for (i = things_printed = 0; i < n && things_printed < print_max; i++) {
	if (i != 0) {
	  fputs_filtered (", ", stream);
	  wrap_here ("    "); }

	m3_val_print2 (elt, valaddr, 
		       bitpos + i * TYPE_M3_SIZE (elt), TYPE_M3_SIZE (elt),
		       stream, format, 0, 0);
        things_printed++;
	for (j = i + 1, reps = 1; 
	     j < n &&  compare (valaddr, bitpos + i * TYPE_M3_SIZE (elt),
				bitpos + j * TYPE_M3_SIZE (elt),
				TYPE_M3_SIZE (elt));
	     j++, reps++);
	if (reps > repeat_count_threshold) {
	  fprintf_filtered (stream, " <repeats %d times>", reps);
	  i += reps - 1;
	  things_printed += repeat_count_threshold; }}
	    
      if (i < n) {
	fputs_filtered ("...", stream); }

      fputs_filtered ("}", stream);
      break; }
      
    case TYPE_CODE_M3_OPEN_ARRAY: {
      struct type *elt_type   = TYPE_M3_OPEN_ARRAY_ELEM (type);
      CORE_ADDR elems = m3_unpack_pointer (valaddr, bitpos);
      int      nelems = m3_unpack_int (valaddr + TARGET_PTR_BIT/HOST_CHAR_BIT,
				       bitpos, TARGET_LONG_BIT);
      int      eltsize = 1;

      if (bitpos % HOST_CHAR_BIT != 0) {
	error ("improperly aligned open array"); }

      valaddr += (bitpos / HOST_CHAR_BIT);
      bitpos = 0;

      { struct type *e = elt_type;
	char *nelem_addr = valaddr
	                    + (TARGET_PTR_BIT + TARGET_LONG_BIT)/HOST_CHAR_BIT;
	while (TYPE_CODE (e) == TYPE_CODE_M3_OPEN_ARRAY) {
	  eltsize = eltsize * m3_unpack_int (nelem_addr, 0, TARGET_LONG_BIT);
	  nelem_addr += TARGET_LONG_BIT / HOST_CHAR_BIT;
	  e = TYPE_M3_OPEN_ARRAY_ELEM (e); }
	eltsize = eltsize * TYPE_M3_SIZE (e); }
      if (eltsize % 8 != 0) {
	error ("another improper alignment"); }
      eltsize = eltsize / 8;

      fputs_filtered ("{", stream);
      if (TYPE_CODE (elt_type) == TYPE_CODE_M3_OPEN_ARRAY) {
	for (i = things_printed = 0; 
	     i < nelems && things_printed < print_max; i++) {
	  if (i > 0) {
	    fputs_filtered (",", stream); 
	    wrap_here ("    "); }
	  *(long*)(valaddr + TARGET_PTR_BIT / HOST_CHAR_BIT) = elems + i * eltsize;
	  m3_val_print2 (elt_type, valaddr + TARGET_PTR_BIT / HOST_CHAR_BIT, 
			 0, TYPE_M3_SIZE (elt_type), 
			 stream, format, 0, 0);
	  things_printed++; }}
      else {
	char *a = alloca (eltsize);
	char *previous = alloca (eltsize);
	reps = 0;
	for (i = things_printed = 0; 
	     i < nelems && things_printed < print_max; i++) {
	  target_read_memory (elems, a, eltsize);
	  if (reps > 0 && memcmp (a, previous, eltsize) == 0) {
	    reps++; }
	  else {
	    if (reps > 1) {
	      if (reps > repeat_count_threshold) {
		fprintf_filtered (stream, " <repeats %d times>", reps); }
	      else {
		for (j = 0; j < reps - 1 && things_printed < print_max; j++) {
		  if (things_printed) {
		    fputs_filtered (",", stream); 
		    wrap_here ("    "); }
		  m3_val_print2 (elt_type, previous, 
				 0, TYPE_M3_SIZE (elt_type), 
				 stream, format, 0, 0);
		  things_printed++; }}
	      things_printed += reps; }
	    if (things_printed < print_max) {
	      if (things_printed) {
		fputs_filtered (",", stream); 
		wrap_here ("    "); }
	      m3_val_print2 (elt_type, a, 
			     0, TYPE_M3_SIZE (elt_type), 
			     stream, format, 0, 0);
	      things_printed++; }
	    reps = 1;
	    memcpy (previous, a, eltsize); }
	  elems += eltsize; }
	if (reps > 1) {
	  if (reps > repeat_count_threshold) {
	    fprintf_filtered (stream, " <repeats %d times>", reps); 
	    things_printed += reps - 1; }
	  else {
	    for (j = 0; j < reps - 1 && things_printed < print_max; j++) {
	      if (things_printed) {
		fputs_filtered (",", stream); 
		wrap_here ("    ");  }
	      m3_val_print2 (elt_type, previous, 
			     0, TYPE_M3_SIZE (elt_type),
			     stream, format, 0, 0);
	      things_printed++; }}}}
      if (things_printed < nelems) {
	fputs_filtered ("...", stream); }
      fputs_filtered ("}", stream);
      break; }

    case TYPE_CODE_M3_PACKED: {
      m3_val_print2 (TYPE_M3_TARGET (type), valaddr,
		     bitpos, TYPE_M3_SIZE (type),
		     stream, format, 0, 0);
      break; }
      
    case TYPE_CODE_M3_ENUM: {
      long val = m3_unpack_int (valaddr, bitpos, bitsize);
      fputs_filtered (TYPE_M3_ENUM_VALNAME (type, val), stream);
      break; }
      
    case TYPE_CODE_M3_INDIRECT: {
      CORE_ADDR target_addr = m3_unpack_pointer (valaddr, 0);
      struct type *target = TYPE_M3_INDIRECT_TARGET (type);
      int target_size = TYPE_LENGTH (target);
      char *target_val = alloca (target_size); 
      
      target_read_memory (target_addr, target_val, target_size);
      m3_val_print2 (target, target_val, 
		     0, TYPE_M3_SIZE (target), 
		     stream, format, deref_ref, toplevel);
      break; }
      
    case TYPE_CODE_M3_PROC: {
      m3_print_scalar (valaddr, bitpos, bitsize, stream, format);
      break; }
      
    case TYPE_CODE_M3_RECORD: {
      int nonfirst = 0;

      fputs_filtered ("{", stream);
      for (i = 0; i < TYPE_M3_REC_NFIELDS (type); i++) {
	if (TYPE_M3_REC_FIELD_NAME (type, i)[0] != '_') {
	  if (nonfirst != 0) {
	    fputs_filtered (", ", stream);
	    wrap_here ("    "); }
	  nonfirst = 1;

	  fputs_filtered (TYPE_M3_REC_FIELD_NAME (type, i), stream);
	  fputs_filtered (" = ", stream);
	  m3_val_print2 (TYPE_M3_REC_FIELD_TYPE (type, i), valaddr,
			 bitpos + TYPE_M3_REC_FIELD_BITPOS (type, i),
			 TYPE_M3_SIZE (TYPE_M3_REC_FIELD_TYPE (type, i)),
			 stream, format, 0, 0);  }}
      fputs_filtered ("}", stream);
      break; }
      
    case TYPE_CODE_M3_SET: {
      int n = 0;
      int j;
      LONGEST lower, upper;
      struct type *target = TYPE_M3_SET_TARGET (type);
      int nelems = TYPE_NFIELDS (target);
      int en = (TYPE_CODE (target) == TYPE_CODE_M3_ENUM);
      int ch = (TYPE_CODE (target) == TYPE_CODE_M3_CHAR);
      int chs = (TYPE_CODE (target) == TYPE_CODE_M3_SUBRANGE)
	&& (TYPE_CODE (TYPE_M3_SUBRANGE_TARGET (target)) == TYPE_CODE_M3_CHAR);

      m3_ordinal_bounds (target, &lower, &upper);
      fputs_filtered ("{", stream);
      
      for (i = 0; i < TYPE_LENGTH (type) / sizeof (long); i++) {
	val = m3_unpack_int (valaddr, bitpos, TARGET_LONG_BIT);
	for (j = 0; j < TARGET_LONG_BIT; j++) {
	  LONGEST ord = i * TARGET_LONG_BIT + j + lower;
	  if (val & 1 << j) {
	    if (n > 0) {
	      fputs_filtered (", ", stream); }
	    if (en) {
	      fputs_filtered (TYPE_FIELD_NAME (target, ord), stream); }
	    else if (ch) {
	      fprintf_filtered (stream, "'%c'", ord); }
	    else if (chs) {
	      fprintf_filtered (stream, "'%c'", ord); }
	    else {
	      fprintf_filtered (stream, "%ld", ord); }
	    n++; }}
	valaddr += sizeof (long); }
      
      fputs_filtered ("}", stream);
      
      break; }
      
    case TYPE_CODE_M3_SUBRANGE: {
      struct type *target = TYPE_M3_SUBRANGE_TARGET (type);
      if (TYPE_CODE (target) == TYPE_CODE_M3_ENUM) {
	long val = m3_unpack_int (valaddr, bitpos, TYPE_M3_SIZE (type));
	fputs_filtered (TYPE_M3_ENUM_VALNAME (target, val), stream); }
      else {
	m3_print_scalar (valaddr, bitpos, bitsize, stream, format); }
      break; }

    case TYPE_CODE_M3_ADDRESS:
      m3_print_scalar (valaddr, bitpos, bitsize, stream, 
		       format ? format : 'x');
      break;

    case TYPE_CODE_M3_BOOLEAN:
      if (m3_unpack_int (valaddr, bitpos, bitsize)) {
	fputs_filtered ("TRUE", stream); }
      else {
	fputs_filtered ("FALSE", stream); }
      break;

    case TYPE_CODE_M3_CHAR:
      m3_printchar (m3_unpack_int (valaddr, bitpos, 8), stream);
      break;

    case TYPE_CODE_M3_INTEGER:
    case TYPE_CODE_M3_CARDINAL:
    case TYPE_CODE_M3_NULL:
    case TYPE_CODE_M3_VOID:
      m3_print_scalar (valaddr, bitpos, bitsize, stream, format);
      break;

    case TYPE_CODE_M3_ROOT:
    case TYPE_CODE_M3_UN_ROOT:
    case TYPE_CODE_M3_OBJECT: {
      if (deref_ref && !format) {
	m3_print_object (valaddr, bitpos, type, stream, format); }
      else {
	m3_print_scalar (valaddr, bitpos, bitsize, stream, 
			 format ? format : 'x'); }
      break; }

    case TYPE_CODE_M3_REFANY: {
      m3_print_scalar (valaddr, bitpos, bitsize, stream, 
		       format ? format : 'x');
      break; }

    case TYPE_CODE_M3_POINTER: {
      struct type *target = TYPE_M3_POINTER_TARGET (type);
      if (TYPE_CODE (target) == TYPE_CODE_M3_OPEN_ARRAY
	  && TYPE_CODE (TYPE_M3_OPEN_ARRAY_ELEM (target)) == TYPE_CODE_M3_CHAR) {
	CORE_ADDR chars_addr;
	CORE_ADDR text_value;
	text_value = m3_unpack_pointer (valaddr, bitpos);
	if (text_value == 0) {
	  fputs_filtered ("NIL", stream); }
	else {
	  /* FIXME: &chars_addr is not a char* as target_read_memory wishes */
	  target_read_memory (text_value, &chars_addr, TYPE_M3_SIZE (type));
	  val_print_string (chars_addr, 0, stream); }}
      else {
	m3_print_scalar (valaddr, bitpos, bitsize, stream, 
			 format ? format : 'x'); }

      break; }

    case TYPE_CODE_FLT: {
      if (format) {
	m3_print_scalar (valaddr, bitpos, bitsize, stream, format); }
      else {
	if (bitpos % 8 != 0) {
	  error ("improperly aligned floating point value"); }
	print_floating (valaddr + bitpos / 8, type, stream); }
      break; }

    case TYPE_CODE_UNDEF:
      fprintf_filtered (stream, "<incomplete type> ");
      
      /* For the convenience of the user we print this as an integer also.
	 This happens quite often when a few libraries
	 were not compiled with debug info.
       */
      format = format ? format : output_format;
      {
	struct type *type = builtin_type_int;
	
	if (format)
	  print_scalar_formatted (valaddr, type, format, 0, stream);
	else
	  val_print_type_code_int (type, valaddr, stream);
      }
      
      break;

    default: 
     return 1; }

  fflush (stream);
  return (0);
}

int
m3_val_print (type, valaddr, address, stream, format, deref_ref, recurse,
	     pretty)
     struct type *type;
     char *valaddr;
     CORE_ADDR address;
     FILE *stream;
     int format;
     int deref_ref;
     int recurse;
     enum val_prettyprint pretty;
{
  if (m3_val_print2 (type, valaddr, 0, TYPE_M3_SIZE (type),
		     stream, format, deref_ref, 1)) {
    /* like the value of registers */
    return c_val_print (type, valaddr, address, stream, format, deref_ref,
			recurse, pretty); }
  return 0;
}





