/*
Welcome to the Talisman MPEG Decoder Device Driver for FreeBSD 2.1

By Brian E. Litzinger <brian@easynet.com> 
                      <brian@mpress.com>

This software is Copyright (C) 1995 By Brian E. Litzinger

This software may be used, modified, copied, distributed, and sold, 
in both source and binary form provided that all copyright information
contained herein and these terms are retained, for this and
all subsequent and/or derived works.

Under no circumstances is the author responsible for the proper functioning
of this software, for the continued support of this software, nor
does the author assume any responsbility for damages incurred with its use.

This software is provided "AS-IS".



This device driver contains the core (kernel) level functionality
required to drive the Talisman MPEG Decoder Card.

The Talisman MPEG decoder card is a hardware MPEG Decoder with
NTSC output and Window-in-a-VGA capability.

I.E. It can create a hardware window in your VGA display, and/or output
true NTSC.

It accepts MPEG Systems Streams directly.  No Stream parsing is
required of the host.

The Talisman MPEG Decoder has a number of other features, such as
Video Capture, Still graphics overlay, support for other
MPEG Stream formats (CDDA,VideoCD), which are not implemented by
this driver.  Of course, I hope to add support for all those things
in the future.

To actually do much useful with the card, you need an additional
suite of software, which contains:

    tmmgr   - A management program for configuring the card
    tmplay  - A sample non-X based MPEG Stream Player
    xtmplay - A sample X based MPEG Stream Player (not written yet)


These files will eventually be available in source form from:

    ftp://omt.com/pub/tm/freebsd

Until the files are available at that site you can email me
and I'll send them to you.

OmniMedia, manufacturer of the Talisman card, can be reached 
at +1 408 774 9999.

Brian Litzinger <brian@easynet.com>
                <brian@mpress.com>
*/

/*
 * talisman
 */

/* #define TMDEBUG 1 */

#include "tm.h"

#include "sys/param.h"
#include "sys/systm.h"
#include "sys/ioctl.h"
#include "sys/tty.h"
#include "sys/proc.h"
#include "sys/user.h"
#include "sys/conf.h"
#include "sys/file.h"
#include "sys/uio.h"
#include "sys/kernel.h"
#include "sys/syslog.h"
#include "sys/device.h"
#include "sys/malloc.h"
#include "sys/devconf.h"

#if 0
#include "machine/cpufunc.h"
#endif

#include "i386/isa/isa.h"
#include "i386/isa/isa_device.h"
#include "i386/isa/icu.h"

#include "i386/isa/tm_reg.h"
#include "i386/isa/tm_ioctl.h"
#include "i386/isa/av-vxp201.h"
#include "i386/isa/c3-cl480.h"

#define POLLSLICE 2 /*ms*/	/* default POLLSLICE */
#define CIRC_BUFSIZ 256*1024	/* bytes, must be even */

#define UNIT(dev)	(minor(dev))

struct tm_softc {
    int tm_port;
    int tm_inintr;
    int tm_pollactive;
    caddr_t cb_write_ptr;
    caddr_t cb_read_ptr;
    caddr_t cb_start_ptr;
    caddr_t cb_end_ptr;
    unsigned short tm_baseio;
    unsigned short tm_idxreg;
    unsigned short tm_datreg;
    unsigned short tm_pollslice;
    int sleeperwaitingforlen;
    struct  kern_devconf kdc; /* kernel configuration database info */
} tm_softc[NTM];


int     tm_probe        __P((struct isa_device *));
int     tm_attach       __P((struct isa_device *));
void    tm_intr         __P((int unit));
void    tm_poll         __P((void));
int     tm_open         __P((dev_t dev, int oflags, int devtype,
                             struct proc *p));
int     tm_close        __P((dev_t dev, int fflag, int devtype,
                             struct proc *p));
int     tm_read         __P((dev_t dev, struct uio *uio, int ioflag));
int     tm_write        __P((dev_t dev, struct uio *uio, int ioflag));
int     tm_ioctl        __P((dev_t dev, int cmd, caddr_t data,
                             int fflag, struct proc *p));
int     tm_select       __P((dev_t dev, int rw, struct proc *p));

static void tmpoll(/* struct tm_softc * */);
static int init_cbuf(struct tm_softc *);
static void write_cbuf(struct tm_softc *sc,caddr_t buf, int len);
static int freespacein_cbuf(struct tm_softc *sc);
static int read_cbuf(struct tm_softc *sc, int len);
static int slowzfer(struct tm_softc *sc, char *buf, int len);
static void loadpal(struct tm_softc *sc);
static void Set_playmode(struct tm_softc *sc);
static void Send_CL480(struct tm_softc *sc, unsigned short *buf,int len);

static void Write_VxP201(struct tm_softc *sc, unsigned short reg, unsigned short data);
static void Write_VxP201b(struct tm_softc *sc, unsigned short reg, unsigned char data);
static void Read_VxP201(struct tm_softc *sc, unsigned short reg, unsigned short *data);
static void Read_VxP201b(struct tm_softc *sc, unsigned short reg, unsigned char *data);
static void Write_CL480(struct tm_softc *sc, unsigned char A_MSB_8, unsigned short A_LSB_16, unsigned short D16);
static void Read_CL480(struct tm_softc *sc, unsigned char A_MSB_8, unsigned short A_LSB_16, unsigned short *D16);

static char *TM_copyright="Copyright (C) 1995, Brian Litzinger";
static unsigned char tm_initstr[] = {0x41,0x55,0x52,0x41,0x56,0x31,0xf4,0x0b};

struct isa_driver tmdriver = {
        tm_probe,
        tm_attach,
        "tm", 1
};
 
static struct kern_devconf kdc_tm_template = {
        0, 0, 0,                /* filled in by dev_attach */
        "tm", 0, { MDDT_ISA, 0, "mpg" },
        isa_generic_externalize, 0, 0, ISA_EXTERNALLEN,
        &kdc_isa0,              /* parent */
        0,                      /* parentdata */
        DC_UNCONFIGURED,
        "MPEG Decoder"          /* description */
};

static inline void
tm_registerdev(struct isa_device *id, const char *descr)
{
        struct kern_devconf *kdc = &tm_softc[id->id_unit].kdc;
        char *longdescr;
        *kdc = kdc_tm_template;
        kdc->kdc_unit = id->id_unit;
        kdc->kdc_parentdata = id;
        kdc->kdc_description = descr;
        dev_attach(kdc);
}


/*
 * Probe routine
 */
int
tm_probe(struct isa_device *isa_dev) {
    struct tm_softc *sc = &tm_softc[isa_dev->id_unit];
	     
    tm_registerdev(isa_dev, "MPEG Decoder");

    sc->tm_baseio = sc->tm_idxreg = isa_dev->id_iobase;
    sc->tm_datreg = sc->tm_idxreg+2;


#define TM_NBR_IO_PORTS 4
    return (TM_NBR_IO_PORTS);
}

/*
 * Attach routine
 */
int
tm_attach(struct isa_device *isa_dev) {
	register struct tm_softc *sc = &tm_softc[isa_dev->id_unit];

	sc->cb_start_ptr = (char*)malloc(CIRC_BUFSIZ, M_DEVBUF, M_WAITOK);
	if (sc->cb_start_ptr==(char*)0) {
	    printf("tm: unable to malloc circular buffer\n");
            return(0);
	}
	sc->cb_end_ptr = sc->cb_start_ptr + CIRC_BUFSIZ;
	sc->cb_read_ptr = sc->cb_write_ptr = sc->cb_start_ptr;


	sc->tm_inintr = 0;
	sc->tm_pollactive = 1;
	sc->tm_pollslice  = POLLSLICE;
	sc->sleeperwaitingforlen = 0;
        sc->kdc.kdc_state = DC_IDLE;
  	timeout(tmpoll,(caddr_t)sc,sc->tm_pollslice);    
	printf("\n");
	return(1);
}

/*
 * Open line
 */
int
tm_open(dev, flag, mode, p)
    dev_t dev;
    int flag;
    int mode;
    struct proc *p;
{
    /*
	if (sc->sb_start_ptr==NULL) 
	    return(ENODEV);
    */
    return (0);
}

/*
 * Close line
 */
int
tm_close(dev, flag, mode, p)
    dev_t dev;
    int flag;
    int mode;
    struct proc *p;
{
    return (0);
}

/*
 * Read from line
 */
int
tm_read(dev, uio, flag)
    dev_t dev;
    struct uio *uio;
    int flag;
{
    return(ENODEV);
}

/*
 * Write to line
 */
int
tm_write(dev, uio, flag)
    dev_t dev;
    struct uio *uio;
    int flag;
{
    struct tm_softc *sc = &tm_softc[UNIT(dev)];

    int result;
    int len;
    int i;

    for (i=0; i<uio->uio_iovcnt; i++) {
        len = uio->uio_iov[i].iov_len;

	if (freespacein_cbuf(sc) < len) {
	    sc->sleeperwaitingforlen = len;
            tsleep(sc,PVM, "tm_cbufwait", 0); 
	}
	write_cbuf(sc,uio->uio_iov[i].iov_base,len);
    }
    return(0);
}

int
tm_select(dev, flag, p)
	dev_t dev;
	int flag;
	struct proc *p;
{
    return(ENODEV);
}

static void
tmpoll(sc)
    struct tm_softc *sc;
{
    int wordsxfered;
    int len;
    int s;

    s = spltty();
    if (sc->cb_read_ptr > sc->cb_write_ptr)
	len = sc->cb_end_ptr - sc->cb_read_ptr;
    else
	len = sc->cb_write_ptr - sc->cb_read_ptr;
    len >>= 1;	/* Convert to words */
    while (len>0) {
	wordsxfered = slowzfer(sc,sc->cb_read_ptr,len);
	if (wordsxfered<=0) {
	    splx(s);
	    timeout(tmpoll,(caddr_t)sc,sc->tm_pollslice);
	    if (sc->sleeperwaitingforlen && freespacein_cbuf(sc)>sc->sleeperwaitingforlen) {
		wakeup((caddr_t)sc);
	    }
	    return;
	}
	sc->cb_read_ptr += (wordsxfered<<1);	/* times 2 */
	len -= wordsxfered;
    }
    if (sc->cb_read_ptr >= sc->cb_end_ptr)
	sc->cb_read_ptr = sc->cb_start_ptr;
    splx(s);
    timeout(tmpoll,(caddr_t)sc,sc->tm_pollslice);
    if (sc->sleeperwaitingforlen && freespacein_cbuf(sc)>sc->sleeperwaitingforlen) {
	wakeup((caddr_t)sc);
    }
}

/*
 * Interrupt routine
 */
void
tm_intr(unit)
    int unit;
{
    return;
}

/*
 * Ioctl routine
 */
int
tm_ioctl(dev, cmd, data, flag, p)
    dev_t dev;
    int cmd;
    caddr_t data;
    int flag;
    struct proc *p;
{
    register struct tm_softc *sc = &tm_softc[UNIT(dev)];
    register int error;
    int s;

    switch (cmd) {
        case TM_SETWORDREG: {
	    struct tm_regop_t *ro = (struct tm_regop_t *)data;

	    outw(sc->tm_idxreg,ro->idx);
	    outw(sc->tm_datreg,ro->dat);
	}
	break;

        case TM_GETWORDREG: {
	    struct tm_regop_t *ro = (struct tm_regop_t *)data;

	    outw(sc->tm_idxreg,ro->idx);
	    ro->dat = inw(sc->tm_datreg);
	}
	break;

        case TM_SETBYTEREG: {
	    struct tm_regop_t *ro = (struct tm_regop_t *)data;

	    outw(sc->tm_idxreg,ro->idx);
	    outb(sc->tm_datreg,(unsigned char)ro->dat);
	}
	break;

        case TM_GETBYTEREG: {
	    struct tm_regop_t *ro = (struct tm_regop_t *)data;

	    outw(sc->tm_idxreg,ro->idx);
	    ro->dat = inb(sc->tm_datreg);
	}
	break;

	case TM_RESET: {
	    s = spltty();
	    Write_CL480(sc,DRAM_BUS,CL480_CID_LOC,CL480_RESET_CMD);
	    sc->cb_read_ptr = sc->cb_write_ptr = sc->cb_start_ptr;
	    splx(s);
	};
	break;

	case TM_PLAY: {
	    unsigned short buf = CL480_PLAY_CMD;
	    int s;

	    s = spltty();
	    Send_CL480(sc,&buf,1);
	    Set_playmode(sc);
	    splx(s);
	}
	break;

	case TM_PAUSE: {
	    unsigned short buf = CL480_PAUSE_CMD;
	    Send_CL480(sc,&buf,1);
	    Set_playmode(sc);
	}
	break;

	case TM_PLAYSEQ: {
	    unsigned short buf[] = { CL480_PLAYSEQ_CMD,0,0,0,0 };
	    int s;

	    s = spltty();
	    Send_CL480(sc,buf,5);
	    Set_playmode(sc);
	    splx(s);
	}
	break;

	case TM_INIT: {
	    struct tm_regop_t *ro = (struct tm_regop_t *)data;
	    unsigned char b = (ro->idx>>2)&0xff;
	    unsigned char *p;
	    int i;

            for (i=0,p=tm_initstr;i<sizeof(tm_initstr)/sizeof(unsigned char);i++,p++) {
		outb(0x80,*p);
	    }
	    outb(0x80,b);
            sc->tm_idxreg = ro->idx;
            sc->tm_datreg = ro->idx+2;
	}
	break;

	case TM_LOADPAL: {
	    loadpal(sc);
	}
	break;

	case TM_RESETMPEG: {
	    Write_VxP201b(sc,0x100,0x00);
	    DELAY(50000); /* 50 ms */
	    Write_VxP201b(sc,0x100,0x14); /* *** WARNING *** */
	}
	break;

        case TM_POLLSLICE: {
	    unsigned short *pollslice = (unsigned short *)data;

	    if (*pollslice>0) {
		sc->tm_pollslice = *pollslice;
	    } else {
		return(EINVAL);
	    }
	}
	break;
	default:
	    return (ENOTTY);
    }
    return (0);
}


static int
freespacein_cbuf(sc) /* diff between write and read pointer adjusted for wrap */
    struct tm_softc *sc;
{	
    int spaceleft;

    spaceleft = sc->cb_read_ptr - sc->cb_write_ptr;
    if (spaceleft<=0) spaceleft += CIRC_BUFSIZ;
    return(spaceleft);
}

/*
    write_cbuf

    does not need to check for overrun condition, because callers must make
    sure the free space is available before calling this routine.
*/

static void
write_cbuf(struct tm_softc *sc, caddr_t buf, int len) {
    int s;

    if (sc->cb_write_ptr + len < sc->cb_end_ptr) { /* Do as one operation */
	copyin(buf,sc->cb_write_ptr,len);
	s = spltty();
	sc->cb_write_ptr += len;
	splx(s);
    } else {					   /* Wrap condition */
        int spaceleft;

        spaceleft = sc->cb_end_ptr - sc->cb_write_ptr;	/* fill to end */
        copyin(buf,sc->cb_write_ptr,spaceleft);
        buf += spaceleft;
        len -= spaceleft;
	s = spltty();
        sc->cb_write_ptr = sc->cb_start_ptr;
        splx(s);

        copyin(buf,sc->cb_write_ptr,len);		/* fill remainder */
	s = spltty();
        sc->cb_write_ptr += len;			/* from beginning */
	splx(s);
    }
}

static void
Write_VxP201(struct tm_softc *sc, unsigned short reg, unsigned short data) {
    outw(sc->tm_idxreg,reg);
    outw(sc->tm_datreg,data);
}

static void
Write_VxP201b(struct tm_softc *sc, unsigned short reg, unsigned char data) {
    outw(sc->tm_idxreg,reg);
    outb(sc->tm_datreg,data);
}

static void
Read_VxP201(struct tm_softc *sc, unsigned short reg, unsigned short *data) {
    outw(sc->tm_idxreg,reg);
    *data = inw(sc->tm_datreg);
}

static void
Read_VxP201b(struct tm_softc *sc, unsigned short reg, unsigned char *data) {
    outw(sc->tm_idxreg,reg);
    *data = inb(sc->tm_datreg);
}

static void
Write_CL480(struct tm_softc *sc, unsigned char A_MSB_8, unsigned short A_LSB_16, unsigned short D16) {
    Write_VxP201b(sc,CL480_BASE+HSEL_A_LSB,(unsigned short)A_LSB_16&0xff);
    Write_VxP201b(sc,CL480_BASE+HSEL_A_MB, (unsigned short)A_LSB_16>>8);
    Write_VxP201b(sc,CL480_BASE+HSEL_A_MSB, A_MSB_8);
    Write_VxP201b(sc,CL480_BASE+HSEL_D_LSB,(unsigned short)D16&0xff);
    Write_VxP201b(sc,CL480_BASE+HSEL_D_MSB,(unsigned short)D16>>8);
}

static void
Read_CL480(struct tm_softc *sc, unsigned char A_MSB_8, unsigned short A_LSB_16, unsigned short *D16) {
    unsigned char l,h;
    Write_VxP201b(sc,CL480_BASE+HSEL_A_LSB,(unsigned short)A_LSB_16&0xff);
    Write_VxP201b(sc,CL480_BASE+HSEL_A_MB, (unsigned short)A_LSB_16>>8);
    Write_VxP201b(sc,CL480_BASE+HSEL_A_MSB, A_MSB_8);
    Read_VxP201b(sc,CL480_BASE+HSEL_D_MSB,&h);
    Read_VxP201b(sc,CL480_BASE+HSEL_D_LSB,&l);
    *D16 = (unsigned short) ((l&0xff) | ((h&0xff)<<8));
}


static int
slowzfer(struct tm_softc *sc, caddr_t buf, int len) {
    register int res = 0;
    register int cnt;
    register unsigned short idxreg = sc->tm_idxreg;
    register unsigned short datreg = sc->tm_datreg;

    for (;;) {
	cnt = imin(22,len);
	if (cnt==0) return(res);

	outw(idxreg,0x100);
	if (inw(datreg)&0x01)
	    return(res);

	len -= cnt;
	res += cnt;

	while (cnt--) {
	    outw(idxreg,0xc004);
	    outb(datreg,*buf++);
	    outw(idxreg,0xc005);
	    outb(datreg,*buf++);
	}
    }
    return(res);
}

static void
loadpal(struct tm_softc *sc) {
    unsigned char conf_a;
    int i;

    Read_VxP201b(sc,0x38,&conf_a);
    Write_VxP201b(sc,0x38,conf_a|0x10);

    for (i=0;i<255;i++) {
	Write_VxP201b(sc,0x5d,i);
	Write_VxP201b(sc,0x5e,i);
	Write_VxP201b(sc,0x5f,i);
    }
    Write_VxP201b(sc,0x38,conf_a);
}

static void
Send_CL480(struct tm_softc *sc, unsigned short *p, int len) {
    unsigned short CMD_FIFO_wptr;
    unsigned short CMD_FIFO_eptr;
    unsigned short CMD_FIFO_sptr;

    Read_CL480(sc,DRAM_BUS, CL480_CMFD_WRITE, &CMD_FIFO_wptr);
#if 1
    Read_CL480(sc,DRAM_BUS, CL480_CMFD_EADDR, &CMD_FIFO_eptr);
    Read_CL480(sc,DRAM_BUS, CL480_CMFD_SADDR, &CMD_FIFO_sptr);
#else
    CMD_FIFO_wptr = 0x080;   /* *** WARNING *** */
    CMD_FIFO_eptr = 0x100;
    CMD_FIFO_sptr = 0x080;
#endif
    if (CMD_FIFO_wptr <0x80 || CMD_FIFO_wptr>=0x100) {
	printf("tm0: warning: CMD_FIFO_wptr = %x, fixing\n",CMD_FIFO_wptr);
	CMD_FIFO_wptr &= 0xff;
    }
    while (len--) {
	Write_CL480(sc,DRAM_BUS, CMD_FIFO_wptr++, *p++);
	if (CMD_FIFO_wptr>=CMD_FIFO_eptr)
	    CMD_FIFO_wptr = CMD_FIFO_sptr;
    }
    Write_CL480(sc,DRAM_BUS, CL480_CMFD_WRITE, CMD_FIFO_wptr);
}

static void
Set_playmode(struct tm_softc *sc) {
    Write_VxP201b(sc,CL480_BASE+HSEL_A_LSB,0);
    Write_VxP201b(sc,CL480_BASE+HSEL_A_MB, 0);
    Write_VxP201b(sc,CL480_BASE+HSEL_A_MSB,FIFO_BUS);
}
