# ColorEditor - a general purpose Tk widget Color Editor (based on tcolor.tcl from the Tcl/Tk distribution).
#
# Stephen O. Lidie, Lehigh University Computing Center.  95/03/05
# lusol@Lehigh.EDU
#
# Many thanks to Guy Decoux (decoux@moulon.inra.fr) for doing the initial translation of tcolor.tcl to TkPerl, from which this
# code has been derived.
#
# ColorEditor is implemented as an object with various methods, described below.  First, create your ColorEditor object during
# program initialization (one should be sufficient), and then configure it by specifying a list of Tk widgets to colorize.
# When it's time to use the editor, invoke the `show' method.
#
# ColorEditor allows some customization: you may alter the color attribute menu by adding and/or deleting menu items and/or
# separators, turn the status window on or off, alter the configurator's list of color widgets, or even supply your own custom
# color configurator procedure.
#
# 1) Call the constructor to create the editor object, which in turn returns a blessed reference to the new object:
#
#    use Tk::ColorEditor;
#
#    $cref = $top->ColorEditor($title, @cursor);
#
#       top            - a window reference, usually the result of a MainWindow->new call.  As the default root of a widget
#                        tree, $top and all descendant widgets are configured by the default color configurator procedure. 
#                        (You probably want to change this though; keep reading.)
#       title          - Toplevel title, default = ' '.
#       cursor         - a valid Tk '-cursor' specification (default is 'top_left_arrow').  This cursor is used over all
#                        ColorEditor "hot spots".
#
# 2) Invoke the `configure' method to change editor characteristics:
#
#    $cref->configure([option => value], ..., [option-n => value-n]);
#    
#       option         - 'add_menu_item'    : "SEP", "display_color_editor_status", or a color attribute menu item.
#                        'delete_menu_item' : "SEP", "display_color_editor_status", a color attribute menu item, or menu ordinal.
#                        'color_list'       : a reference to a list of widget references for the color configurator.
#                        'my_configurator'  : a replacement subroutine `configure_application_colors', which is `eval'ed.
#
#    Note that this method expects a list of list, e.g.:
#
#       $cref->configure(['delete_menu_item' => 3],  ['delete_menu_item' => 'disabledforeground'],
#                        ['add_menu_item' => 'SEP'], ['add_menu_item' => 'New color attribute'],
#                        ['color_list' => [$ce, $qu, $f2b2]],
#                        ['color_list' => [$cref->get_widget_tree($f2)]],
#                        ['my_configurator' => 'sub configure_application_colors{print "woof\n";}'],
#                        );
#
# 3) Invoke the `show' method on the editor object, say, by a button or menu press:
#
#    $cref->show;
#
# 4) The `get_widget_tree' method recursively descends the widget hierarchy and returns a list of descendant widgets.  You can
#    supply this list to `configure' to change which widgets the color configurator colorizes.
#
#    $cref->get_widget_tree($parent);
#
#       parent         - root of widget hierarchy.
#
# 5) The `get_color_list' method returns a reference to a list of widgets that are colorized by the configurator.  Typically,
#    you add new widgets to this list and then use it in a subsequent `configure' call to expand your color list.
#
#    $cref->configure(['color_list' => [@{$Filesystem_ref->get_color_list}, @{$cref->get_color_list}]]);
#
#    Note here that the Filesystem object too has a `get_color_list' method.  Indeed, any Class that uses ColorEditor should
#    provide this method since only the object "knows" which inner widgets are suitable for colorizing.
#
# 6) The `delete_color_list' method expects a reference to a list of widgets which are then removed from the current color list.
#
#    $cref->delete_color_list($objtable{$obj}->{'color_list'})

package Tk::ColorEditor;
@ISA = qw(Tk::Composite Tk::Toplevel);
bless(\qw(ColorEditor))->WidgetClass;
 
use Carp;
use English;
sub changeColorSpace; sub hsvToRgb; sub lsearch; sub rgbToHsv;

my $class_list1 = 'Button|Canvas|Checkbutton|Entry|Frame|Label|Listbox|Menu|Menubutton';
my $class_list2 = 'Message|Radiobutton|Scale|Scrollbar|ScrolledListbox|Text|Toplevel';
$CLASS_LIST = "${class_list1}|${class_list2}"; # configure these TkPerl classes


# ColorEditor public methods.


sub new {

    # ColorEditor constructor.

    my($class, $parent, $title, @cursor) = @ARG;

    croak "ColorEditor:  `new' constructor requires at least 2 arguments" if scalar @ARG < 2;

    my $top = $parent->Toplevel(-class => $class);
    $top->withdraw;
    my $highlight = 'foreground';	# current highlight color
    $top->title("$title $highlight Color Editor");

    my $color_space = 'hsb';	# rgb, cmy, hsb
    my $red = 65535;		# color intensity, 0-65535
    my $green = 0;
    my $blue = 0;
    my $color = '#ffff00000000';   # current color string, #RRRRGGGGBBBB
    @cursor = ('top_left_arrow') unless @cursor;
    my(@highlight_list) = qw(TEAR_SEP foreground background SEP activeforeground activebackground SEP selectforeground 
			     selectbackground SEP disabledforeground insertbackground selector sliderforeground);

    # Create the menu bar at the top of the window for the File and Colors menubuttons.

    my $m0 = $top->Frame(-relief => 'raised', -borderwidth => 2);
    $m0->pack(-side => 'top', -fill => 'x');
    my $mf = $m0->Menubutton(-text => 'File', -underline => 0, -cursor => \@cursor, -bd => 1, -relief => 'raised');
    $mf->pack(-side => 'left');
    my $close_command = [sub {shift->withdraw}, $top];
    $mf->command(-label => 'Close', -underline => 0, -command => $close_command, -accelerator => 'Ctrl-w');
    $top->bind('<Control-Key-w>' => $close_command);
    $top->protocol(WM_DELETE_WINDOW => $close_command);

    my $mc = $m0->Menubutton(-text => 'Colors', -underline => 0, -cursor => \@cursor, -bd => 1, -relief => 'raised');
    $mc->pack(-side => 'left');
    my $color_attributes = 'Color Attributes';
    $mc->cascade(-label => $color_attributes, -underline => 6);
    $mc->separator;
    my $color_spaces = 'Color Spaces';
    $mc->cascade(-label => $color_spaces, -underline => 6);
    $mc->separator;
    $mc->command(-label => 'Apply Default Colors', -underline => 6, -command => [sub {
	my($objref) = @ARG;
	foreach (@{$objref->{'highlight_list'}}) {
	    next if $ARG =~ /TEAR_SEP|SEP/;
	    $objref->configure_application_colors($ARG, 'use_widget_default_color');
	}
    }, $top]);

    my $m1 = $mc->cget(-menu);
    my $mcm1 = $m1->Menu;
    $m1->entryconfigure($color_spaces, -menu => $mcm1);
    $mcm1->radiobutton(-label => 'RGB color space', -variable => \$color_space, -value => 'rgb', -underline => 0, 
		       -command => [\&changeColorSpace, $top, 'rgb']);
    $mcm1->radiobutton(-label => 'CMY color space', -variable => \$color_space, -value => 'cmy', -underline => 0, 
		       -command => [\&changeColorSpace, $top, 'cmy']);
    $mcm1->radiobutton(-label => "HSB color space", -variable => \$color_space, -value => 'hsb', -underline => 0, 
		       -command => [\&changeColorSpace, $top, 'hsb']);

    my $mcm2 = $m1->Menu;
    $m1->entryconfigure($color_attributes, -menu => $mcm2);
    foreach (@highlight_list) {
	next if /^TEAR_SEP$/;
	if(/^SEP$/) {
	    $mcm2->separator;
	} else {
	    $mcm2->command(-label => $ARG, -command => [sub {
		my ($w, $title, $h) = @ARG;
		$w->title("$title $h Color Editor");
		$w->{'update'}->configure(-text => "Apply $h Color");
		$w->{'highlight'} = $h;
	    }, $top, $title, $ARG]);
	}
    }

    # Create the Apply button.

    my $bot = $top->Frame(-relief => 'raised', -bd => 2);
    $bot->pack(-side => 'bottom', -fill =>'x');
    my $update = $bot->Button(-text => "Apply $highlight Color", -cursor => \@cursor, -command => [sub {
	my($objref) = @ARG;
	$objref->configure_application_colors($objref->{'highlight'}, $objref->{'color'});
    }, $top]);
    $update->pack(-pady => 1, -padx => '0.25c');

    # Create the listbox that holds all of the color names in rgb.txt, if an rgb.txt file can be found.

    my $middle = $top->Frame(-relief => 'raised', -borderwidth => 2);
    $middle->pack(-side => 'top', -fill => 'both');
    my($i, @a);
    foreach $i ('/usr/local/lib/X11/rgb.txt', '/usr/lib/X11/rgb.txt', '/usr/local/X11R5/lib/X11/rgb.txt',
		'/X11/R5/lib/X11/rgb.txt', '/X11/R4/lib/rgb/rgb.txt', '/usr/openwin/lib/X11/rgb.txt') {
	next if ! open FOO, $i;
	my $middle_left = $middle->Frame;
	$middle_left->pack(-side => 'left', -padx => '0.25c', -pady => '0.25c');
	my $names = $top->Listbox(-width => 20, -height => 12, -relief => 'sunken', -borderwidth => 2, -exportselection => 0);
	$names->bind('<Double-1>' => [sub {
	    my($l, $t) = @ARG;
	    $t->{'Entry'} = $l->get($l->curselection);
	    \&load_named_color($t);
	}, $top]);
	my $scroll = $top->Scrollbar(-orient => 'vertical', -command => ["yview", $names], -relief => 'sunken',
				     -borderwidth => 2, -cursor => \@cursor);
	$names->configure(-yscrollcommand => ["set",$scroll]);
	$names->pack(-in => $middle_left, -side => 'left');
	$scroll->pack(-in => $middle_left, -side => 'right', -fill => 'y');
	while(<FOO>) {
	    chomp;
	    @a = split /\s+/;
	    $names->insert('end', $a[3]) if $#a == 3;
	  }
	close FOO;
	last;
    }

    # Create the three scales for editing the color, and the entry for typing in a color value.

    my $middle_middle = $middle->Frame;
    $middle_middle->pack(-side => 'left', -expand => 1, -fill => 'y');
    my(@middle_middle, @label, @scale);
    $middle_middle[1] = $middle_middle->Frame;
    $middle_middle[2] = $middle_middle->Frame;
    $middle_middle[3] = $middle_middle->Frame;
    $middle_middle[4] = $middle_middle->Frame;
    $middle_middle[1]->pack(-side => 'top', -expand => 1);
    $middle_middle[2]->pack(-side => 'top', -expand => 1);
    $middle_middle[3]->pack(-side => 'top', -expand => 1);
    $middle_middle[4]->pack(-side => 'top', -expand => 1, -fill => 'x');
    my (@label, @scale);
    foreach $i (1..3) {
	$label[$i] = $top->Label(-textvariable => \$Label[$i]);
	$scale[$i] = $top->Scale(-from => 0, -to => 1000, '-length' => '6c', -orient => 'horizontal', -cursor => \@cursor,
				 -command => [\&scale_changed, $top]);
	$scale[$i]->pack(-in => $middle_middle[$i], -side => 'top', -anchor => 'w');
	$label[$i]->pack(-in => $middle_middle[$i], -side => 'top', -anchor => 'w');
    }
    my $nameLabel = $top->Label(-text => "Name:");
    my $name = $top->Entry(-relief => 'sunken', -borderwidth => 2, -textvariable => \$top->{'Entry'}, -width => 10,
			-font => "-Adobe-Courier-Medium-R-Normal-*-120-*-*-*-*-*-*");
    $nameLabel->pack(-in => $middle_middle[4], -side => 'left');
    $name->pack(-in => $middle_middle[4], -side => 'right', -expand => 1, -fill => 'x');
    $name->bind('<Return>' => [sub {
	shift;
	\&load_named_color(shift());
    }, $top]);

    # Create the color display swatch on the right side of the window.

    my $middle_right = $middle->Frame;
    $middle_right->pack(-side => 'left', -pady => '.25c', -padx => '.25c', -anchor => 's');
    my $swatch = $top->Canvas(-width => '2.5c', -height => '5c', -cursor => \@cursor);
    my $swatch_item = $swatch->create('oval', '.5c', '.3c', '2.26c', '4.76c', -fill => $color);
    my $value = $top->Label(-textvariable => \$top->{'color'}, -width => 13,
			    -font => '-Adobe-Courier-Medium-R-Normal-*-120-*-*-*-*-*-*');
    $swatch->pack(-in => $middle_right, -side => 'top', -expand => 1, -fill => 'both');
    $value->pack(-in => $middle_right, -side => 'bottom', -pady => '.25c');

    # Create the status window.

    my $status = $top->Toplevel;
    $status->withdraw;
    $status->geometry('+0+0');
    my $status_l = $status->Label(-width => 50,  -anchor => 'w');
    $status_l->pack(-side => 'top');

    $top->{'highlight_list'} = [@highlight_list];
    $top->{'mcm2'} = $mcm2;
    $top->{'highlight'} = $highlight;
    $top->{'color'} = $color;
    $top->{'updating'} = 0;
    $top->{'color_space'} = $color_space;
    $top->{'swatch'} = $swatch;
    $top->{'swatch_item'} = $swatch_item;
    $top->{'Entry'} = '';
    $top->{'scale'} = [@scale];
    $top->{'red'} = $red;
    $top->{'blue'} = $blue;
    $top->{'green'} = $green;
    $top->{'display_color_editor_status'} = 0;
    $top->{'Status'} = $status;
    $top->{'Status_l'} = $status_l;
    $top->{'gwt_depth'} = 0;
    $top->{'title'} = $title;
    $top->{'update'} = $update;

    changeColorSpace $top, $color_space;

    bless $top, $class;
    $top->{'color_list'} = [$top->get_widget_tree($parent)];
    my $pixmap = $top->Pixmap('-file' => "$Tk::tk_library/ColorEditor.xpm");
    $top->Icon(-image => $pixmap);
    return $top;

} # end new, ColorEditor constructor


sub configure {

    # Process color hooks now.

    my($objref, @hook_list) = @ARG;

	foreach $hook (@hook_list) {
	    if ($hook->[0] eq 'add_menu_item') {
		if ($hook->[1] eq 'SEP') {
		    $objref->{'mcm2'}->separator;
		    push @{$objref->{'highlight_list'}}, $hook->[1];
		} elsif ($hook->[1] eq 'display_color_editor_status') {
		    $objref->{'display_color_editor_status'} = 1;
		} else {
		    $objref->{'mcm2'}->command(-label => $hook->[1], -command => [sub {
			my ($w, $h) = @ARG;
			$w->title("$w->{'title'} $h Color Editor");
			$w->{'update'}->configure(-text => "Apply $h Color");
			$w->{'highlight'} = $h;
		    }, $objref, $hook->[1]]);
		    push @{$objref->{'highlight_list'}}, $hook->[1];
		}
	    } elsif ($hook->[0] eq 'color_list') {
		$objref->{'color_list'} = $hook->[1];
	    } elsif ($hook->[0] eq 'my_configurator') {
		eval $hook->[1];
		print STDERR $EVAL_ERROR if $EVAL_ERROR;
	    } elsif ($hook->[0] eq 'delete_menu_item') {
		if ($hook->[1] eq 'display_color_editor_status') {
		    $objref->{'display_color_editor_status'} = 0;
		} else {
		    $objref->{'mcm2'}->delete($hook->[1]);
		    my $list_ord = $hook->[1] =~ /\d+/ ? $hook->[1] : lsearch($objref->{'highlight_list'}, $hook->[1]);
		    splice(@{$objref->{'highlight_list'}}, $list_ord, 1) if $list_ord != -1;
		}
	    } else {
		print STDERR "Unknown color hook command: \"$hook->[0] => $hook->[1]\"\n";
	    }
	}

} # end configure


sub delete_color_list {

    # Remove widgets from consideration by the color configurator.  $color_list_ref points to widgets added via `configure'.

    my($objref, $color_list_ref) = @ARG;

    my($i, $found, $r1, $r2);
    foreach $r1 (@{$color_list_ref}) {
	$i = -1;
	$found = 0;
	foreach $r2 (@{$objref->{'color_list'}}) {
	    $i++;
	    next if $r1 != $r2;
	    $found = 1;
	    last;
	}
	splice(@{$objref->{'color_list'}}, $i, 1) if $found;
    }

} # end delete_color_list


sub get_color_list {

    # Return a reference to the color configurator's current widget list.

    my($objref) = @ARG;

    return $objref->{'color_list'};

} # end get_color_list


sub get_widget_tree {

    # Return a list of widgets derived from a parent widget and all its descendants.

    my($objref, $widget) = @ARG;

    my(@widget_list) = ($widget->children);
    unshift @widget_list, $widget if $objref->{'gwt_depth'} == 0 and $widget->class =~ /$CLASS_LIST/;

    $objref->{'gwt_depth'}++; # note recursion depth

    foreach $widget (@widget_list) {
	next if $widget->class !~ /$CLASS_LIST/;
	push @widget_list, $objref->get_widget_tree($widget); # get kids
    }

    $objref->{'gwt_depth'}--;
    return @widget_list;

} # end get_widget_tree


sub show {

    my($objref) = @ARG;

    $objref->deiconify;

} # end show


# ColorEditor default configurator procedure - can be redefined by the application.


sub configure_application_colors {

    # Configure all the widgets in $color_list for attribute $type and color $color.  If $color == 'use_widget_default_color'
    # then reset all colors to the Tk defaults.

    my($objref, $type, $color) = @ARG;

    $objref->{'Status'}->title("Configure $type");
    $objref->{'Status'}->deiconify if $objref->{'display_color_editor_status'};
    my($widget, $reset) = (undef, ($color eq 'use_widget_default_color') ? 1 : 0);

    foreach $widget (@{$objref->{'color_list'}}) {
	next if $widget->class !~ /$CLASS_LIST/;
	if ($objref->{'display_color_editor_status'}) {
	    $objref->{'Status_l'}->configure(-text => "WIDGET:  $widget");
	    Tk->update;
	} 
	eval {$color = ($widget->configure("-${type}"))[3]} if $reset;
	eval {$widget->configure("-${type}" => $color)};
    }

    $objref->{'Status'}->withdraw if $objref->{'display_color_editor_status'};

} # end configure_application_colors


# ColorEditor private methods.


sub scale_changed {

    # The procedure below is invoked when one of the scales is adjusted.  It propagates color information from the current
    # scale readings to everywhere else that it is used.

    my($objref) = @ARG;

    return if $objref->{'updating'};
    my ($red, $green, $blue);

    if($objref->{'color_space'} eq 'rgb') {
	$red = int($objref->{'scale'}->[1]->get * 65.535 + 0.5);
	$green = int($objref->{'scale'}->[2]->get * 65.535 + 0.5);
	$blue = int($objref->{'scale'}->[3]->get * 65.535 + 0.5);
    } elsif($objref->{'color_space'} eq 'cmy') {
	$red = int(65535 - $objref->{'scale'}->[1]->get * 65.535 + 0.5);
	$green = int(65535 - $objref->{'scale'}->[2]->get * 65.535 + 0.5);
	$blue = int(65535 - $objref->{'scale'}->[3]->get * 65.535 + 0.5);
    } else {
	($red, $green, $blue) = hsvToRgb($objref->{'scale'}->[1]->get/1000.0, $objref->{'scale'}->[2]->get/1000.0,
					 $objref->{'scale'}->[3]->get/1000.0);
    }
    $objref->{'red'} = $red;
    $objref->{'blue'} = $blue;
    $objref->{'green'} = $green;
    $objref->{'color'} = sprintf("#%04x%04x%04x", $red, $green, $blue);
    $objref->{'swatch'}->itemconfigure($objref->{'swatch_item'}, -fill => $objref->{'color'});
    Tk->idletasks;

} # end scale_changed


sub set_scales {

    my($objref) = @ARG;

    # The procedure below is invoked to update the scales from the current red, green, and blue intensities.  It's invoked
    # after a change in the color space and after a named color value has been loaded.

    $objref->{'updating'} = 1;
    my($red, $blue, $green) = ($objref->{'red'}, $objref->{'blue'}, $objref->{'green'});

    if($objref->{'color_space'} eq 'rgb') {
	$objref->{'scale'}->[1]->set(int($red / 65.535 + 0.5));
	$objref->{'scale'}->[2]->set(int($green / 65.535 + 0.5));
	$objref->{'scale'}->[3]->set(int($blue / 65.535 + 0.5));
    } elsif($objref->{'color_space'} eq 'cmy') {
	$objref->{'scale'}->[1]->set(int((65535 - $red) / 65.535 + 0.5));
	$objref->{'scale'}->[2]->set(int((65535 - $green) / 65.535 + 0.5));
	$objref->{'scale'}->[3]->set(int((65535 - $blue) / 65.535 + 0.5));
    } else {
	my ($s1, $s2, $s3) = rgbToHsv($red, $green, $blue);
	$objref->{'scale'}->[1]->set(int($s1 * 1000.0 + 0.5));
	$objref->{'scale'}->[2]->set(int($s2 * 1000.0 + 0.5));
	$objref->{'scale'}->[3]->set(int($s3 * 1000.0 + 0.5));
    }
    $objref->{'updating'} = 0;

} # end set_scales


sub load_named_color {

    # The procedure below is invoked when a named color has been selected from the listbox or typed into the entry.  It loads
    # the color into the editor.

    my($objref) = @ARG;

    my $name = $objref->{'Entry'};
    return if $name eq '';
    my ($format, $shift);

    if($name !~ /^#/) {
       ($red, $green, $blue) = $objref->{'swatch'}->rgb($name);
   } else {
       my $len = length $name;
       if($len == 4) { $format = "#(.)(.)(.)"; $shift = 12; }
	 elsif($len == 7) { $format = "#(..)(..)(..)"; $shift = 8; }
	   elsif($len == 10) { $format = "#(...)(...)(...)"; $shift = 4; }
	     elsif($len == 13) { $format = "#(....)(....)(....)"; $shift = 0; }
       else { print STDERR "ColorEditor error:  syntax error in color name \"$name\"\n"; return; }
       ($red,$green,$blue) = $name =~ /$format/;
       eval "\$red = 0x$red; \$green = 0x$green; \$blue = 0x$blue;";
       $red = $red << $shift;
       $green = $green << $shift;
       $blue = $blue << $shift;
   }
    $objref->{'red'} = $red;
    $objref->{'blue'} = $blue;
    $objref->{'green'} = $green;
    $objref->{'color'} = sprintf("#%04x%04x%04x", $red, $green, $blue);
    &set_scales($objref);
    $objref->{'swatch'}->itemconfigure($objref->{'swatch_item'}, -fill => $objref->{'color'});

} # end load_named_color


sub changeColorSpace {

    my($objref, $space) = @ARG;

    # The procedure below is invoked when a new color space is selected. It changes the labels on the scales and re-loads the
    # scales with the appropriate values for the current color in the new color space

    if($space eq "rgb") {
	$Label[1] = "Red";
	$Label[2] = "Green";
	$Label[3] = "Blue";
    } elsif($space eq "cmy") {
	$Label[1] = "Cyan";
	$Label[2] = "Magenta";
	$Label[3] = "Yellow";
    } else {
	$Label[1] = "Hue";
	$Label[2] = "Saturation";
	$Label[3] = "Brightness";
    }
    &set_scales($objref);
}


sub rgbToHsv {

    # The procedure below converts an RGB value to HSB.  It takes red, green, and blue components (0-65535) as arguments, and
    # returns a list containing HSB components (floating-point, 0-1) as result.  The code here is a copy of the code on page
    # 615 of "Fundamentals of Interactive Computer Graphics" by Foley and Van Dam.

    my($red, $green, $blue) = @ARG;
    my($max, $min, $sat, $range, $hue, $rc, $gc, $bc);

    $max = ($red > $green) ? (($blue > $red) ? $blue : $red) : (($blue > $green) ? $blue : $green);
    $min = ($red < $green) ? (($blue < $red) ? $blue : $red) : (($blue < $green) ? $blue : $green);
    $range = $max - $min;
    if ($max == 0) {
	$sat = 0;
    } else {
	$sat = $range / $max;
    }
    if ($sat == 0) {
	$hue = 0;
    } else {
	$rc = ($max - $red) / $range;
	$gc = ($max - $green) / $range;
	$bc = ($max - $blue) / $range;
	$hue = ($max == $red)?(0.166667*($bc - $gc)):(($max == $green)?(0.166667*(2 + $rc - $bc)):(0.166667*(4 + $gc - $rc)));
#	$hue = 1 + $hue if $hue < 0; # kludge, but it fixes a long-standing bug in tcolor.tcl!  (SOL, 95/03/20, LUCC)
    }
    return ($hue, $sat, $max/65535);

} # end rgbToHsv


sub hsvToRgb {

    # The procedure below converts an HSB value to RGB.  It takes hue, saturation, and value components (floating-point, 0-1.0)
    # as arguments, and returns a list containing RGB components (integers, 0-65535) as result.  The code here is a copy of the
    # code on page 616 of "Fundamentals of Interactive Computer Graphics" by Foley and Van Dam.

    my($hue, $sat, $value) = @ARG;
    my($v, $i, $f, $p, $q, $t);

    $v = int(65535 * $value);
    return ($v, $v, $v) if $sat == 0;
    $hue *= 6;
    $hue = 0 if $hue >= 6;
    $i = int($hue);
    $f = $hue - $i;
    $p = int(65535 * $value * (1 - $sat));
    $q = int(65535 * $value * (1 - ($sat * $f)));
    $t = int(65535 * $value * (1 - ($sat * (1 - $f))));
    return ($v, $t, $p) if $i == 0;
    return ($q, $v, $p) if $i == 1;
    return ($p, $v, $t) if $i == 2;
    return ($p, $q, $v) if $i == 3;
    return ($t, $p, $v) if $i == 4;
    return ($v, $p, $q) if $i == 5;

} # end hsvToRgb


sub lsearch {

    # Search a list for an entry; return list ordinal, or -1 if not found.

    my($ar, $x) = @ARG;

    my $i;
    for ($i = 0; $i < scalar @$ar; $i++) {
	return $i if ($$ar[$i] eq $x);
    }
    return -1;

} # end lsearch


1;
