# ErrorDialog - a translation of `tkerror' from Tcl/Tk to TkPerl.
#
# Currently TkPerl background errors are sent to stdout/stderr; use this module if you want them in a window.  You can also
# "roll your own" by supplying the routine Tk::BackgroundError.
#
# Stephen O. Lidie, Lehigh University Computing Center.  95/03/02
# lusol@Lehigh.EDU
#
# This is an OO implementation of `tkerror', with a twist:  since there is only one ErrorDialog object, you aren't required to
# invoke the constructor to create it; it will be created automatically when the first background error occurs.  However, in
# order to configure the ErrorDialog object you must call the constructor and create it manually.
#
# The ErrorDialog object essentially consists of two sub-widgets: an ErrorDialog widget to display the background error and a
# Text widget for the traceback information.  If required, you can invoke the `configure' method to change some characteristics
# of these sub-widgets.
#
# 1) Call the constructor to create the ErrorDialog object, which in turn returns a blessed reference to the new object:
#
#    use Tk::ErrorDialog;
#
#    $ED = $top->ErrorDialog;
#
#       top      - a window reference, usually the result of a MainWindow->new call.
#
# 2) Invoke the `configure' method to change ErrorDialog and/or Text characteristics:
#
#    $ED->configure(SubWidget, option => value, ..., option-n => value-n);
#    
#      SubWidget - name of sub-widget to configure: 'ErrorDialog' or 'Text'.
#
#    Configuration options for sub-widget 'Text' are whatever Tk supports; 'ErrorDialog' has two special configuration options:
#
#      -cleanupcode	- a CODE reference if special post-background error processing is required (default is undefined).
#      -appendtraceback - a boolean indicating whether or not to append successive tracebacks (default is 1, do append).	
#

package Tk::ErrorDialog;
@ISA = qw(Tk::Composite Tk::Toplevel);
bless(\qw(ErrorDialog))->WidgetClass;

use Carp;
use English;
use Tk::Dialog;

BEGIN {
    ($ok, $sm, $st) = ('OK', 'Skip Messages', 'Stack trace');
    $ED_OBJECT = undef;
}


sub new {

    # ErrorDialog constructor.

    my($class, $winref) = @ARG;

    my $dr = $winref->Dialog('Error in Perl Script', 'on-the-fly-text', 'error', $ok, $ok, $sm, $st);
    $dr->{'-cleanupcode'} = undef;
    $dr->{'-appendtraceback'} = 1;
    my $t = $winref->Toplevel(-class => 'ErrorTrace');
    $t->minsize(1, 1);
    $t->title('Stack Trace for Error');
    $t->iconname('Stack Trace');
    my $t_ok = $t->Button(-text => 'OK', -command => [sub {shift->withdraw}, $t]);
    my $t_text = $t->Text(-relief => 'sunken', -bd => 2, -setgrid => 'true', -width => 60, -height => 20);
    my $t_scroll = $t->Scrollbar(-relief => 'sunken', -command => ['yview', $t_text]);
    $t_text->configure(-yscrollcommand => ['set', $t_scroll]);
    $t_ok->pack(-side => 'bottom', -padx => '3m', -pady => '2m');
    $t_scroll->pack(-side => 'right', -fill => 'y');
    $t_text->pack(-side => 'left', -expand => 'yes', -fill => 'both');
    $t->withdraw;

    $ED_OBJECT = bless {'ErrorDialog' => $dr, 'Toplevel' => $t, 'Text' => $t_text};
    return $ED_OBJECT;

} # end new, ErrorDialog constructor


sub Tk::BackgroundError {

    # Post a dialog box with the error message and give the user a chance to see a more detailed stack trace.

    my($w, $error, @msgs) = @ARG;

    if (defined $w->grab('current')) {
	$w->grab('release', $w->grab('current'));
    }

    &new('', $w) if not defined $ED_OBJECT;
    my($d, $t) = ($ED_OBJECT->{'ErrorDialog'}, $ED_OBJECT->{'Text'});
    chop $error;
    $d->configure('Message', -text => "Error:  $error");
    my $ans = $d->show;

    if ($ans eq $st) {
	$t->delete('0.0', 'end') if not $d->{'-appendtraceback'};
	$t->insert('end', "\n");
	$t->mark('set', 'ltb', 'end');
	$t->insert('end', "--- Begin Traceback ---\n$error\n");
	for (@msgs) {
	    $t->insert('end', "$ARG\n");
	}
	$t->yview('ltb');
	$ED_OBJECT->{'Toplevel'}->deiconify;
    }

    my $c = $d->{'-cleanupcode'};
    &$c if defined $c;		# execute any cleanup code if it was defined
    $w->break if $ans eq $sm;

} # end Tk::BackgroundError


sub configure {

    # ErrorDialog object public method - configure the error dialog.

    my($w, $widget, %config) = @ARG;
    my $real_w;

    croak "ErrorDialog:  `configure' method was not invoked with a \"Tk::ErrorDialog\" object" unless
      $w->IsErrorDialog;
    croak "ErrorDialog:  `configure' method requires at least 4 arguments" if scalar @ARG < 4;

    if ($widget =~ /ErrorDialog|Text/) {
        $real_w = $w->{$widget};
    } else {
        croak "ErrorDialog:  `configure' sub-widget name must be \"ErrorDialog\" or \"Text\"";
    }

    foreach $attr (keys %config) {
	if ($attr =~ /^-cleanupcode|-appendtraceback$/) {
	    $real_w->{$attr} = $config{$attr};
	} else {
	    $real_w->configure($attr => $config{$attr});
	}
    } # forend all configure attributes

} # end ErrorDialog configure method


1;

