# Dialog - a translation of `tk_dialog' from Tcl/Tk to TkPerl (based on John Stoffel's idea).
#
# Stephen O. Lidie, Lehigh University Computing Center.  94/12/27
# lusol@Lehigh.EDU
#
# This is an OO implementation of `tk_dialog'.  First, create all your Dialog objects during program initialization.  When it's
# time to use a dialog, invoke the `show' method on a dialog object; the method then displays the dialog, waits for a button to
# be invoked, and returns the text label of the selected button.
#
# A Dialog object essentially consists of two sub-widgets: a Label widget for the bitmap and a Label wigdet for the text of the
# dialog.  If required, you can invoke the `configure' method to change any characteristic of these sub-widgets.
#
# 1) Call the constructor to create the dialog object, which in turn returns a blessed reference to the new object:
#
#    use Tk::Dialog;
#
#    $DialogRef = $top->Dialog($title, $text, $bitmap, $default_button, @button_labels);
#
#       top            - a window reference, usually the result of a MainWindow->new call.
#       title          - Title to display in the dialog's decorative frame.
#       text           - Message to display in the dialog widget.
#       bitmap         - Bitmap to display in the dialog ('' signifies no bitmap).
#       default_button - Text label of the button that is to display the default ring (''signifies no default button).
#       button_labels  - One or more strings to display in buttons across the bottom of the dialog box.
#
# 2) Invoke the `show' method on a dialog object:
#
#    $button_label = $DialogRef->show;
#
#       This returns the text label of the selected button.
#
#    (Note:  you can request a global grab by passing the string "-global" to the `show' method.)
#
# 3) Invoke the `configure' method to change dialog characteristics:
#
#    $DialogRef->configure(SubWidget, option => value, ..., option-n => value-n);
#    
#      SubWidget - name of sub-widget to configure: 'Bitmap' for the bitmap, 'Message' for the dialog text.
#
#    (Note:  specifying '' for the configuration option '-bitmap' removes the bitmap from a Dialog object.)
#

package Tk::Dialog;
@ISA = qw(Tk::Composite Tk::Toplevel);
bless(\qw(Dialog))->WidgetClass;

use Carp;
use English;
use Tk;

sub new {

    # Dialog object constructor - create the dialog window.

    my($class, $path, $title, $text, $bitmap, $default_button, @button_labels) = @ARG;

    croak "Dialog:  `new' constructor requires at least 7 arguments" if scalar @ARG < 7;

    # Create the Toplevel window and divide it into top and bottom parts.

    $selected_button = '';
    @pl = (-side => 'top', -fill => 'both');
    ($pad1, $pad2) = ([-padx => '3m', -pady => '3m'], [-padx => '3m', -pady => '2m']);

    my $w = $path->Toplevel(-class => 'Dialog');
    $w->title($title);
    $w->iconname('Dialog');
    $w->protocol('WM_DELETE_WINDOW' => sub {});
    $w->transient($w->parent->toplevel);

    my $w_top = $w->Frame(-relief => 'raised', -borderwidth => 1);
    my $w_bot = $w->Frame(-relief => 'raised', -borderwidth => 1);
    $w_top->pack(@pl);
    $w_bot->pack(@pl);

    # Fill the top part with the (optional) bitmap and message.

    @pl = (-side => 'left');
    if ($bitmap) {
	$w_bitmap = $w_top->Label(-bitmap => $bitmap);
	$w_bitmap->pack(@pl, @$pad1);
    } else {
	$w_bitmap = $w_top->Label(); # an empty label for possible later configuration
    }

    my $w_msg = $w_top->Label(-wraplength => '3i', -justify => 'left', -text => $text,
			   -font => '-Adobe-Times-Medium-R-Normal-*-180-*-*-*-*-*-*');
    $w_msg->pack(-side => 'right', -expand => 1, -fill => 'both', @$pad1);

    # Create a row of buttons at the bottom of the dialog.

    my($w_default_button, $bl) = (undef, '');
    foreach $bl (@button_labels) {
	$w_but = $w_bot->Button(-text => $bl, -command => [sub {$selected_button = $ARG[0]}, $bl]);
	if ($bl eq $default_button) {
	    $w_default_button = $w_bot->Frame(-relief => 'sunken',  -borderwidth => 1);
	    $w_but->raise($w_default_button);
	    $w_default_button->pack(@pl, -expand => 1, @$pad2);
	    $w_but->pack(-in => $w_default_button, -padx => '2m', -pady => '2m');
	    $w->bind('<Return>' => [sub {$ARG[1]->flash; $selected_button = $ARG[2]}, $w_but, $bl]);
	} else {
	    $w_but->pack(@pl, -expand => 1, @$pad2);
	}
    } # forend all buttons

    # Withdraw the window and return a reference to the dialog.

    $w->withdraw;

    $w->{'Message'} = $w_msg;
    $w->{'Bitmap'} = $w_bitmap;
    $w->{'default_button'} = $w_default_button;

    return bless $w, $class;

} # end Dialog constructor


sub show {

    # Dialog object public method - display the dialog.

    my($w, $grab_type) = @ARG;

    croak "Dialog:  `show' method was not invoked with a \"Dialog\" object" unless $w->IsDialog;
    croak "Dialog:  `show' method requires at least 1 argument" if scalar @ARG < 1 ;

    # Update all geometry information, center the dialog in the display and deiconify it, set a grab and claim the focus.

    update('idletasks');
    my $winpar = $w->parent;
    my $winvx = $winpar->vrootx;
    my $winvy = $winpar->vrooty;
    my $winrw = $w->reqwidth;
    my $winrh = $w->reqheight;
    my $winsw = $w->screenwidth;
    my $winsh = $w->screenheight;
    my $x = int($winsw/2 - $winrw/2 - $winvx);
    my $y = int($winsh/2 - $winrh/2 - $winvy);
    $w->geometry("+$x+$y");

    $w->deiconify;
    my $old_focus = Tk->focus();
    if ($grab_type) {
	$w->grab($grab_type);
    } else {
	$w->grab;
    }
    tkwait('visibility', $w);
    if (defined $w->{'default_button'}) {
	$w->{'default_button'}->focus;
    } else {
	$w->focus;
    }

    # Wait for the user to respond, restore the focus and grab, withdraw the dialog and return the label of the selected button.

    tkwait('variable' => \$selected_button);
    $old_focus->focus if defined $old_focus;
    $w->grab('release');
    $w->withdraw;

    return $selected_button;

} # end Dialog show method


sub configure {

    # Dialog object public method - configure the dialog.

    my($w, $widget, %config) = @ARG;
    my $real_w;

    croak "Dialog:  `configure' method was not invoked with a \"Tk::Dialog\" object" unless $w->IsDialog;
    croak "Dialog:  `configure' method requires at least 4 arguments" if scalar @ARG < 4;

    if ($widget =~ /Message|Bitmap/) {
	$real_w = $w->{$widget};
    } else {
	croak "Dialog:  `configure' sub-widget name must be \"Message\" or \"Bitmap\"";
    }

    foreach $attr (keys %config) {
	if ($attr eq -bitmap) {
	    if ($config{$attr} ne '') {
		$real_w->configure($attr => $config{$attr});
		$real_w->pack(@pl, @$pad1);
	    } else {
		Tk::pack('forget', $real_w);
	    }
	} else {
	    $real_w->configure($attr => $config{$attr});
	}
    } # forend all configure attributes

    return $w;

} # end Dialog configure method


1;
